(* Copyright (C) 1992, Digital Equipment Corporation                         *)
(* All rights reserved.                                                      *)
(* See the file COPYRIGHT for a full description.                            *)
(*                                                                           *)
(* Last modified on Tue Aug 11 21:07:58 PDT 1992 by meehan                   *)
(*      modified on Tue Jun 16 12:54:53 PDT 1992 by muller                   *)
(*      modified on Fri Jun 12 19:08:26 1992 by mhb                          *)
(*      modified on Tue Feb 12 13:44:21 PST 1991 by chan                     *)
(*      modified on Wed Dec 26 11:03:59 PST 1990 by brooks                   *)

(* The "AutoRepeat" interface provides support for calling
   a procedure repetitively.  Typically, auto-repeating takes place
   while a key or mouse button is held down, although there
   is no direct relation between "AutoRepeat" and "VBT"s.

   When an auto-repeat object "ar" is started, it forks a ``timer
   thread'' that calls "ar.repeat()" after "firstWait"
   milliseconds, and every "period" milliseconds thereafter.
   However, there is a ``flow control'' mechanism: if the call
   to "ar.repeat()" has not returned by the time the next
   repetition should take place, the timer thread will wait.
   That is, repetitions cannot ``queue up'' more than one deep.

   An auto-repeat is activated by a call to the "start" method,
   terminated by a call to "stop", and resumed by a call
   to "continue".  Both "start" and "stop" may be
   called by the "repeat" method; be warned, however, that a
   call to "start" from "repeat" (or a call to "continue" if
   the auto-repeat is not stopped) may cause a slight glitch
   in the timing of calls to "repeat".

   All locking is handled within "AutoRepeat"; clients can call
   "start", "stop", or "continue" asynchronously and
   concurrently.  *)

INTERFACE AutoRepeat;

TYPE Milliseconds = CARDINAL;

CONST
  DefaultFirstWait: Milliseconds = 500;
  DefaultPeriod   : Milliseconds = 66;

TYPE
  T <: Public;
  Public =
    Private OBJECT
    METHODS
      init (firstWait: Milliseconds := DefaultFirstWait;
            period   : Milliseconds := DefaultPeriod     ): T;
      repeat   ();  (* Client should override this method. *)
      start    ();
      stop     ();
      continue ();
    END;
  Private <: ROOT;

(* The call "ar.init()" initializes "ar" as an "AutoRepeat.T",
   and it returns "ar".

   The call "ar.start()" initiates auto-repeating.  The timer
   will wait "firstWait" milliseconds before calling
   "ar.repeat()" the first time, then "period" milliseconds
   between subsequent calls to "ar.repeat()".  If "ar.start()"
   is called while an auto-repeat timer is already running, the
   previous call is cancelled, and timing starts over from the
   time of the new call to "ar.start()".

   The call "ar.stop()" stops auto-repeating as soon as
   possible.  This procedure is a no-op if "ar" is not running.

   The call "ar.continue()" causes auto-repeating to begin
   immediately, rather than waiting "firstWait" milliseconds as
   in the call to "ar.start()".  Subsequent calls to
   "ar.repeat()" happen every "period" milliseconds, as usual.
   If "ar" is not stopped, then "ar.continue()" will first call
   "ar.stop()". *)

END AutoRepeat.
