/**********************************************************************

			P A L A S M

***********************************************************************

	RE-WRITTEN FOR C BY GIRVIN HERR, LOCKHEED PARL

REVISIONS:
 1.0 -	14 JUN 83   GRH
	INITIAL RELEASE

 2.0 -	10 aug 83   grh
	paltype mod to eliminate nested if - elses & allow for future
	expansion by implementing a table driven algorithm.
	ixlate() now returns all information needed on a pin.

 2.1 -	22 Feb 84   GTK
	Module PALASM:	repaired move_name so device names are properly
			handled

	Module PALDOIT:	modified error messages for product and term errors
			fixed conditional test invoking "too many terms"
			error so 16 terms are handled properly

	File PALS.DEF:	modified to include the non-standard (bit-wise)
			definitions of TRUE and FALSE used by PALASM
*/
#define	VERS "2.1"

/***********************************************************************

	TRANSLATES SYMBOLIC EQUATIONS INTO PAL OBJECT CODE FORMATTED FOR
 DIRECT INPUT TO STANDARD PROM PROGRAMMERS.

calling sequence:
PALASM </S> </S>.. INFILE

WHERE:
	INFILE=		TEXT FILE CONTAINING THE PAL DESIGN SPECIFICATION
			 AS SPECIFIED IN THE MMI PAL MANUAL 


			THE FOLLOWING FILES ARE OUTPUT:

			INFILE.LST = PAL LISTING
			INFILE.HEX = PAL HEX CODE FOR PROGRAMMER (/H)
			INFILE.BIN = PAL CODE FOR PROGRAMMER (/BH OR /BN)


	/S=		OPTIONAL OUTPUT CONTROL SWITCHES:

			/PL = OUTPUT FUSE PLOT CHART TO LISTING
			/S  = OUTPUT SHORT FUSE PLOT CHART TO LISTING
			/PI = OUTPUT PINOUT DIAGRAM TO LISTING
			/M  = OUTPUT HEX PROGRAMMER DATA MAP TO LISTING

			/H  = OUTPUT HEX FORMAT
			/BH = OUTPUT BHLF BINARY FORMAT
			/BN = OUTPUT BNPF BINARY FORMAT



****************************************************************************


	MAIN PROGRAM
*/

#include	libc.h
#include	pals.def
#include	paltokes.inc



	/***   global variables   ***/
FILE	*infile,	/**  ptr to input <pds> file  **/
	*listfile,	/**  ptr to list output file  **/
	*codefile;	/**  ptr to code output file  **/

int	errno;		/**  global file error word  **/

	/***   user specified command line options   ***/
unsigned options;
#include	palsw.inc

int	pal_type,	/**  PAL type key <1..6>  **/
	fuses_blown,	/**  blown fuse count  **/
	nfuses,		/**  max number of TERM fuses for this PAL  **/
	mxterms,	/**  max number of PAL terms  **/
	npins;		/**  PAL package pin count  **/

char	src_line[82],	/**  line buffer for getline() (includes \n & \0)  **/
	*cptr;		/**  ptr to input line  **/
extern
char	symbuf[max_pins][9];	/**  symbol buffer  **/

char	fuses[max_terms][max_inputs],	/**  the fuse array  **/
	pin_phase[max_pins],	      /**  signal inversion array for pins  **/
	cur_phase,			/**  current pin phase  **/
	pterm_used[max_terms];		/**  product line use flags  **/


/**	input data  					PAL20X8   **/
/**							   ^ ^^	  **/
int	pal_inputs;	/**  pal input pin count___________| ||	  **/
char	pal_letter;	/**  pal type letter_________________||	  **/
int	pal_outputs;	/**  pal output pin count_____________|	  **/



	/***   local variables   ***/
static
  char	pal_partn[9],	/**  the whole pal part # for error msgs  **/
	title[82],	/**  pds line 3  **/
	part_no[82],	/**  pds line 2 (user's device part number)  **/
	plot_text[max_inputs];	/**  plot output text line for fuse chars  **/


static
   char	listname[15],	/**  storage for listing filename to open  **/
	codename[15],	/**  storage for code filename to open  **/

	*argp,		/**  ptr to argv[]  **/
	*infileptr;	/**  ptr to argv which contains the input filename  **/

	/***   external functions   ***/

int	getsym(),	/**  put symbol into symbol table  **/
	match(),	/**  return pin # of symbol represented by cptr  **/
	isop(),		/**  return 0 if not operator  **/
	parse(),	/**  return token of symbol, operator or fixed sym  **/
	parse2(),	/**  return token of operator or fixed symbol  **/
	chk_fixed(),	/**  return token of fixed sym. & value in fixedn[] **/
	fixsym(),	/** ???  **/
	hexout();	/**  output data to code file in hex format  **/

char	*errors[] = {"",
                     "Unexpected input",
                     "Premature End of File",
		     "Disk Full"
                     };
#define	nul_error errors[0]
#define unx_input errors[1]
#define pre_eof   errors[2]
#define disk_full errors[3]

#define no_skip FALSE
#define skip	TRUE





/**************************************************************************

	main program start

***************************************************************************/

main(argc, argv)   int argc; char *argv[];{
   int x;
   static int cntr;
   static char c;

   fprintf(stdout, "PALASM  Vers. %s\n", VERS);

	/***	set command argument switches   ***/

   if (argc < 2){
      fprintf(stdout, "\nNeed Input File!\n");
      exit(1);
      }
   options = FALSE;		/***  start with all false  ***/
   for (cntr = 1; cntr < argc; cntr++){
      argp = argv[cntr];
      if (strncmp(argp, "/PL", 3) == 0)       options |= plot_sw;
      else if (strncmp(argp, "/H", 2) == 0)   options |= hex_sw;
      else if (strncmp(argp, "/S", 2) == 0)   options |= short_sw;
      else if (strncmp(argp, "/BH", 3) == 0)  options |= bhlf_sw;
      else if (strncmp(argp, "/BN", 3) == 0)  options |= bnpf_sw;
      else if (strncmp(argp, "/M", 2) == 0)   options |= map_sw;
      else if (strncmp(argp, "/PI", 3) == 0)  options |= pinout_sw;
      else infileptr = argp;
      }

	/***	construct output files   ***/

   move_name(listname, infileptr, "LST");
   if (options & hex_sw)   move_name(codename, infileptr, "HEX");
   else   move_name(codename, infileptr, "BIN");

	/***	open files   ***/

   errno = 0;
   infile = fopen(infileptr, "r");
   if (errno){
      fprintf(stdout, "\nCannot open input file!\n");   exit(1);
      }
   listfile = fopen(listname, "w");
   if (options & (hex_sw | bhlf_sw | bnpf_sw))
      codefile = fopen(codename, "w");
   if (errno){   fprintf(stdout, "\nDisk Directory Full!\n");   exit(1);}

	/***	read first line   ***/

   cptr = getline(src_line);
   if (strncmp(cptr, "PAL", 3) != 0)   ierr(unx_input);

	/***	move pal number to storage   ***/

   for (cntr = 0; cntr < 8 && src_line[cntr] != ' ' &&
			      src_line[cntr] != '\t' &&
			      src_line[cntr] != '\n' ; cntr++)
      pal_partn[cntr] = src_line[cntr];
   pal_partn[cntr] = '\0';

   pal_inputs = atoi(cptr = src_line+3);  /**  fetch pal input count  **/

	/**  skip to pal type letter  **/
   while (isdigit(*cptr))   if (incr(no_skip) < 0)   ierr(pre_eof);
   pal_letter = *cptr++;
   pal_outputs = atoi(cptr);		/* fetch pal output count */

	/***   save user part number data   ***/

   if (*(cptr = getline(src_line)) == '\0')   ierr(pre_eof);
   strcpy(part_no, src_line);

	/***   save device application text   ***/

   if (*(cptr = getline(src_line)) == '\0')   ierr(pre_eof);
   strcpy(title, src_line);

   if (*(cptr = getline(src_line)) == '\0')   ierr(pre_eof);


   if ((pal_type = initlz()) < 0){
      fprintf(stdout, "\nPAL part type %s is incorrect!", pal_partn);
      exit(1);
      }

	/***   fill pin symbol table   ***/

   if (*(cptr = getline(src_line)) == '\0')   ierr(pre_eof);

   for (cntr = 0; cntr < npins; cntr++){
      if (*cptr != '/' && isop(FALSE))
         ierr("Less than 20 pin names in pin list");
      if ((x = getsym(pin_phase, cntr)) < 0){
         fprintf(stdout,"%d:%d\n", cntr, x);
         ierr("Pin list");
         }
      }
   fprintf(stdout,"\nAnalyzing..  ");
   doit();		/***   set fuse map to equations   ***/

	/***   do phantom fuses   ***/
   if (pal_type <= 4 || pal_type == 9)   tweek();

	/***   do plot output if requested   ***/
   if (options & (plot_sw | short_sw))	plot();
   fprintf(stdout,"(%d Fuses Blown)   ", fuses_blown);

	/***   generate map if requested   ***/
   if (options & map_sw)	map();

	/***   output code   ***/
   if (options & hex_sw)	hex();
   else if (options & (bhlf_sw | bnpf_sw))	binr();

	/***   output pinout diagram if requested   ***/
   if (options & pinout_sw)		pinout();

   if (pal_letter != 'H')
      fprintf(stdout,"\n**** WARNING!  Invert PAL data. ****");

	/***   check for write errors   ***/
   if (errno)   fprintf(stdout,"\nDisk Error - %d", errno);

   fprintf(stdout,"\n");
   exit(0);
   }


/*************************************************************************

	functions

**************************************************************************/

/***********************************************************************
	initialize function
   This function initializes variables & matches PAL part number with
pal_type.
	exit -	-1: type error
************************************************************************/
int initlz(){
   register int cnt;
   register char *fptr;
   static char *pals[] = {"10H8","12H6","14H4","16H2","10L8","12L6","14L4",
			  "16L2","16L8","16R8","16R6","16R4","16C1","16X4",
			  "12L10","14L8","16L6","18L4","20L2","20C1","20L10",
			  "20X10","20X8","20X4"};

	/***  initialize fuses array to all unblown  ***/
   fptr = &fuses[0][0];
   for (cnt = 0; cnt < (max_inputs * max_terms); cnt++)   *fptr++ = FALSE;

   fuses_blown = 0;			/*  number of fuses blown  */

/*** 	pal_type is assigned the following values:
-------- 20 PIN ----------     ------ 24 PIN ------
 PAL10H8, PAL10L8	1	PAL12L10	11
 PAL12H6, PAL12L6	2	PAL14L8		12
 PAL14H4, PAL14L4	3	PAL16L6		13
 PAL16H2, PAL16L2	4	PAL18L4		14
 PAL16L8		5	PAL20L2		15
 PAL16R8		6	PAL20C1		16
 PAL16R6		7	PAL20L10	17
 PAL16R4		8	PAL20X10	18
 PAL16C1		9	PAL20X8		19
 PAL16X4		10	PAL20X4		20

	determine pal_type
***/

   for (cnt = 0; cnt < 24; cnt++)
      if (strcmp(&pal_partn[3], pals[cnt]) == 0)   break;

   if (cnt >= 24)   return -1;	/**  not valid pal  **/

   if (cnt >= 4)   cnt -= 4;	/**  make H = L  **/

   if (cnt > 9){
      npins = 24;	/**  set number of pins  **/
      nfuses = 40;	/**  set number of fuses  **/
      mxterms = 80;
      }
   else{
      npins = 20;
      nfuses = 32;
      mxterms = 64;
      }

   return cnt + 1;
   }


/***********************************************************************
	pinout function
   The pinout function prints the pinout of the PAL.
************************************************************************/
pinout(){
   register int i, j, k;
   char pins[max_pins][9];   /* temporary pin array so we can manipulate it */
   static char	top[] =    {"\t\t         ************\\_/************"},
		middle[] = {"\t\t         *                         *"},
		bot[] =    {"\t\t         ***************************"};

	/***   move symbols to temporary work area   ***/
   for (i = 0; i < npins; i++){
      strncpy(&pins[i][0], &symbuf[i][0], 8);
      pins[i][8] = '\0';	/**  insure termination  **/
      }

	/***   right justify pins 1..GND   ***/
   for (i = 0; i < (npins / 2); i++){
      j = strlen(&pins[i][0]);
      if (j < 8){
         for (k = 7; k >= 0; k--){
            if (--j < 0)   pins[i][k] = ' ';
            else pins[i][k] = pins[i][j];
            }
         }
      }

	/***   output header   ***/
   header("Pinout");
   fprintf(listfile, "\t\t\t\t   %s\n%s\n%s",
                      pal_partn, top, middle);

   for (i = 1, j = npins; i < j; i++, j--){
      fprintf(listfile, "\n%s\n\t\t%s - %2d\t\t\t%2d - %s\n%s\n%s",
              middle, &pins[i - 1][0], i, j, &pins[j - 1][0],
              middle, middle);
      }
   fprintf(listfile, "\n%s\n\f", bot);
   }


/*********************************************************
	plot function
   The plot function produces the fuse plot
**********************************************************/
plot(){
   register int cnt;
   char *txt_ptr;
   static int	prod_line,
		prod_grp,
		term;

   fprintf(stdout,"Plotting..  ");

	/***   output header to listing   ***/
   header("Fuse Plot");
   fprintf(listfile, "\
                11 1111 1111 2222 2222 2233\n\
    0123 4567 8901 2345 6789 0123 4567 8901\n\n");

	/***   set up loops   ***/
   for (prod_grp = 0; prod_grp < (mxterms - 7); prod_grp += 8){
      for (term = 0; term < 8; term++){
         prod_line = prod_grp + term;

	/***   set text line from fuses   ***/
         for (cnt = 0, txt_ptr = plot_text; cnt < nfuses; cnt++)
            *txt_ptr++ = (fuses[prod_line][cnt]) ?  '-' : 'X';

	/***   display phantom fuses   ***/
         if (pal_type == 9 || pal_type <= 4)   fantom(prod_line, term);

         if (options & short_sw)
            for (cnt = 0, txt_ptr = plot_text; cnt < nfuses; cnt++, txt_ptr++)
               if (*txt_ptr == '0')   *txt_ptr = ' ';

	/***   now output the fuse term   ***/
         if ((options & plot_sw) ||
             ((options & short_sw) && pterm_used[prod_line])){

            fprintf(listfile, " %2d", prod_line);
            for (cnt = 0; cnt < nfuses; cnt += 4)
               fprintf(listfile," %c%c%c%c",
                 plot_text[cnt], plot_text[cnt +1],
                 plot_text[cnt +2], plot_text[cnt +3]);
            fprintf(listfile,"\n");
            }
         }
      fprintf(listfile, "\n");
      }
   fprintf(listfile, "\n\
Legend:  X : Fuse Not Blown (L,N,0)   - : Fuse Blown   (H,P,1)");
      if ((options & plot_sw) && (pal_type == 9 || pal_type <= 4))
         fprintf(listfile, "\n\
         0 : Phantom Fuse   (L,N,0)   O : Phantom Fuse (H,P,1)");
   fprintf(listfile, "\nNumber of fuses blown = %d\n\n", fuses_blown);
   }


/****************************************************************
	input error outputs src_line[] to stdout & error message to
  both stdout & listfile. src_line has already been output to
  listfile.  Aborts program.
	entry -	p1= ptr to error message
		p2= optional number to output 1st if *p1 == ' '
*****************************************************************/
ierr(err_msg, n)   char *err_msg; int n;{
   int cnt;

	/***   print error line on stdout, list file already has it   ***/
   fprintf(stdout,"\n%s", src_line);

	/***   space over to position arrow   ***/
   for (cnt = cptr - src_line; cnt > 0 && cnt < 80; cnt--){
      fprintf(stdout, " ");
      fprintf(listfile, " ");
      }

	/***   output arrow   ***/
   fprintf(stdout, "^\n");
   fprintf(listfile, "^\n");

	/***   output error message   ***/
   if (*err_msg == ' '){
      fprintf(stdout, "%d%s Error!\n", n, err_msg);
      fprintf(listfile, "%d%s Error!\n", n, err_msg);
      }
   else{
      fprintf(stdout, "%s Error!\n", err_msg);
      fprintf(listfile, "%s Error!\n", err_msg);
      }
   exit(1);
   }



/*****************************************************************
	get a line of input text function
   errno on entry is saved & restored, so that an EOF condition
is not reported as an error on program termination.
	entry-	p1= ptr to char buffer for input text
	exit -	returns line ptr

******************************************************************/
char *getline(lptr)   char *lptr;{
   unsigned i;
   int	errsav, t;

   errsav = errno;		/**  save current errno  **/

   *lptr = '\0';		/**  set up to return null line  **/

   for (i = 0, t = 0; i < 80 && t != '\n'; i++){
      t = agetc(infile);
      if (t < 0)   break;	/**  if EOF then done  **/
      lptr[i] = toupper(t);	/**  convert to upper case  **/
      lptr[i + 1] = '\n';	/**  insure cr if line is truncated  **/
      lptr[i + 2] = '\0';

	/***   echo char   ***/
      if (aputc(t, listfile) < 0){
         fprintf(stdout, "\nWrite error in list file!\n");
         exit(1);
         }
      }
   errno = errsav;		/**  restore errno  **/
   return lptr;
   }


/*************************************************************
	tweek function
   This function tweeks fuses (the programming fuse plot)
**************************************************************/
tweek(){
   register int input_line, prod_line;

   if (pal_type < 4){
      for (prod_line = 0; prod_line < 64; prod_line++){
         fuses[prod_line][14] = TRUE;
         fuses[prod_line][15] = TRUE;
         fuses[prod_line][18] = TRUE;
         fuses[prod_line][19] = TRUE;
         if (pal_type < 3){
            fuses[prod_line][10] = TRUE;
            fuses[prod_line][11] = TRUE;
            fuses[prod_line][22] = TRUE;
            fuses[prod_line][23] = TRUE;
            if (pal_type < 2){
               fuses[prod_line][6] = TRUE;
               fuses[prod_line][7] = TRUE;
               fuses[prod_line][26] = TRUE;
               fuses[prod_line][27] = TRUE;
               }
            }
         }
      for (input_line = 6; input_line < 28; input_line++){
         for (prod_line = 0; prod_line < 57; prod_line += 8){
            fuses[prod_line + 4][input_line] = FALSE;
            fuses[prod_line + 5][input_line] = FALSE;
            fuses[prod_line + 6][input_line] = FALSE;
            fuses[prod_line + 7][input_line] = FALSE;
            }
         if (pal_type < 3){
            for (prod_line = 16; prod_line < 41; prod_line += 8){
               fuses[prod_line + 2][input_line] = FALSE;
               fuses[prod_line + 3][input_line] = FALSE;
               }
            if (pal_type < 2){
               for (prod_line = 0; prod_line < 57; prod_line += 8){
                  fuses[prod_line + 2][input_line] = FALSE;
                  fuses[prod_line + 3][input_line] = FALSE;
                  }
               }
            }
         }
      }
   if (pal_type == 1 || (pal_type == 4 && pal_letter == 'L'))   return;
   for (input_line = 0; input_line < 32; input_line++){
      for (prod_line = 0; prod_line < 8; prod_line++){
         fuses[prod_line][input_line] = (pal_letter != 'L');
         if (pal_letter != 'C')
            fuses[prod_line + 56][input_line] = (pal_letter != 'L');
         }
      if (pal_type > 2){
         for (prod_line = 0; prod_line < 8; prod_line++){
            fuses[prod_line + 8][input_line] = (pal_letter != 'L');
            if (pal_letter != 'C')
               fuses[prod_line + 48][input_line] = (pal_letter != 'L');
            }
         if (pal_type > 3){
            for (prod_line = 0; prod_line < 8; prod_line++){
               fuses[prod_line + 16][input_line] = (pal_letter != 'L');
               if (pal_letter != 'C')
                  fuses[prod_line + 40][input_line] = (pal_letter != 'L');
               }
            }
         }
      }
   }


/**********************************************************************
	funtion fantom
   This function updates plot_text (the printed fuse plot) for high and
low fantom fuses.
***********************************************************************/
fantom(product, term)   int product, term;{
   register int cnt;
   register char *txt_ptr;

	/***   make them all phantom until proven otherwise   ***/
   for (cnt = 0, txt_ptr = plot_text; cnt < nfuses; txt_ptr++, cnt++){
      if (*txt_ptr == '-')   *txt_ptr = 'O';
      if (*txt_ptr == 'X')   *txt_ptr = '0';
      }

   if (((pal_type == 4 || pal_type == 9) && (product < 24 || product >= 40)) ||
       (pal_type == 3 && (product < 16 || product >= 44)) ||
       (pal_type == 2 && (product < 8  || product >= 52)) ||
       (pal_type <= 3 && term > 3) ||
       (pal_type <= 2 && product > 17 && product < 48 && term > 1) ||
       (pal_type == 1 && term > 1))   return;

	/**  now if any not phantom then restore them  **/

   for (cnt = 0, txt_ptr = plot_text; cnt < 32; txt_ptr++, cnt++){
      if (
          !((pal_type <= 3 &&
             (cnt == 14 || cnt == 15 || cnt == 18 || cnt == 19) ) ||
            (pal_type <= 2 &&
             (cnt == 10 || cnt == 11 || cnt == 22 || cnt == 23) ) ||
            (pal_type == 1 &&
             (cnt == 6  || cnt == 7  || cnt == 26 || cnt == 27) )
            ) ){
         if (*txt_ptr == 'O')   *txt_ptr = '-';
         if (*txt_ptr == '0')   *txt_ptr = 'X';
         }
      }
   }


/***********************************************************************
	move base filename to buffer
***********************************************************************/
move_name(dest, source, type)   char *dest, *source, *type;{
   register int cnt;
   static char *dptr;

   dptr = dest;
   for (cnt = 0; cnt < 8 && *source != '\0' && *source != '.'; cnt++){
      if (*source == ':'){          /* picked up device */
         dest = dptr;               /* reset file name pointer */
         *dest++ = *(source - 1);   /* get device name */
         cnt = -1;                  /* count 1 extra for colon */
         }
      *dest++ = *source++;
      }
   *dest++ = '.';
   strncpy(dest, type, 4);
   }


/************************************************************************
	Output header to list file function
************************************************************************/
header(tptr)   char *tptr;{
   static char stars[80] = {"------------------------------------------------\
-----------------------------"};

   fprintf(listfile,"\f\t\t\t\t%s\n%sPart No.: %s%s\n",
		 tptr, title, part_no, stars);
   }
