	TITLE	'Console Command Processor (CCP), ver 2.2	24 oct 81'
		page 58
;
;	assembly language version of the CP/M console command processor
;
;	version 2.2
;
;
;
FALSE		equ	0000h
TRUE		equ	not FALSE
TESTING		equ	FALSE			;true if debugging
SPECIAL		equ	TRUE
;
;
		if TESTING
		org	3400h
BDOSL		equ	$+0800h			;bdos location
		else
		org	0000h
BDOSL		equ	$+0800h			;bdos location
		endif
;
TRAN		equ	0100h
TRANM		equ	$
CCPLOC		equ	$
;
;
;	*********************************************************
;	*	base of CCP contains the following code/data	*
;	*							*
;	*	CCP:	jmp CCPSTART	(start with command)	*
;	*		jmp CCPCLEAR	(start, clear command)	*
;	*	CCP+6	127		(max command length)	*
;	*	CCP+7	COMLEN		(command length = 00)	*
;	*	CCP+8	' ... '		(16 blanks)		*
;	*							*
;	*********************************************************
;	*							*
;	* Normal entry is at CCP, where the command line given	*
;	* at CCP+8 is excuted automatically (normally a null	*
;	* command with COMLEN = 00).  An initializing program	*
;	* can be automatically loaded by storing the command	*
;	* at CCP+8, with the command length at CCP+7.  In this	*
;	* case, the CCP executes the command before prompting	*
;	* the console for input.  Note that the command is exe-	*
;	* cuted on both warm and cold starts.  When the command	*
;	* line is initialized.  A jump to "jmp CCPCLEAR" dis-	*
;	* ables the automatic command execution.		*
;	*							*
;	*********************************************************
;
;
;
		jmp	CCPSTART		;start ccp with possible initial command
		jmp	CCPCLEAR		;clear the command buffer
;
MAXLEN		db	127			;max buffer length
;
COMLEN		db	00			;command length (filled in by bdos)
;
;	(command executed initially if COMLEN non-zero)
;
COMBUF		db	'        '		;8 character fill
		db	'        '		;8 character fill
		db	'COPYRIGHT (C) 1979, DIGITAL RESEARCH  ';38 characters
		ds	128-($-COMBUF)
;
;
;	total buffer length is 128 characters
;
COMADDR		dw	COMBUF			;address of next char to scan
STADDR		ds	2			;starting address of current FILLFCB request
;
;
DISKA		equ	0004h			;disk address for current disk
BDOS		equ	0005h			;primary bdos entry point
BUFF		equ	0080h			;default buffer
FCB		equ	005ch			;default file control block
;
;
RCHARF		equ	1			;read character function
PCHARF		equ	2			;print character function
PBUFF		equ	9			;print buffer function
RBUFF		equ	10			;read buffer function
BREAKF		equ	11			;break key function
LIFTF		equ	12			;lift head function (no operation)
INITF		equ	13			;initialize bdos function
SELF		equ	14			;select disk function
OPENF		equ	15			;open file function
CLOSEF		equ	16			;close file function
SEARF		equ	17			;search for file function
SEARNF		equ	18			;search for next file function
DELF		equ	19			;delete file function
DREADF		equ	20			;disk read function
DWRITF		equ	21			;disk write function
MAKEF		equ	22			;file make function
RENF		equ	23			;rename file function
LOGF		equ	24			;return login vector
CSELF		equ	25			;return currently selected drive number
DMAF		equ	26			;set dma address
USERF		equ	32			;set user number
;
;	special fcb flags
;
ROFILE		equ	9			;read only file
SYSFILE		equ	10			;system file flag
;
;	special characters
;
CR		equ	13			;carriage return
LF		equ	10			;line feed
LA		equ	'_'			;left arrow (underscore)
EOFILE		equ	1ah			;end of file
;
;	utility procedures
;
PRINTCHAR:	mov	e,a
		mvi	c,PCHARF
		jmp	BDOS
;
PRINTBC:	;print character, but save <bc> registers
		push	b
		call	PRINTCHAR
		pop	b
		ret
;
CRLF:		mvi	a,CR
		call	PRINTBC
		mvi	a,LF
		jmp	PRINTBC
;
BLANK:		mvi	a,' '
		jmp	PRINTBC
;
;
PRINT:		;print string starting at <bc> until next 00 entry
		push	b
		call	CRLF
		pop	h			;now print the string
;
PRIN0:		mov	a,m
		ora	a
		rz				;stop on 00
		inx	h
		push	h			;ready for next
		call	PRINTCHAR
		pop	h			;character printed
		jmp	PRIN0			;for another character
;
;
INITIALIZE:	mvi	c,INITF
		jmp	BDOS
;
;
SELECT:		mov	e,a
		mvi	c,SELF
		jmp	BDOS
;
;
BDOS$INR:	call	BDOS
		sta	DCNT
		inr	a
		ret
;
;
OPEN:		;open the file given by <de>
		mvi	c,OPENF
		jmp	BDOS$INR
;
;
OPENC:		;open comfcb
		xra	a
		sta	COMREC			;clear next record to read
		lxi	d,COMFCB
		jmp	OPEN
;
;
CLOSE:		;close the file given by <de>
		mvi	c,CLOSEF
		jmp	BDOS$INR
;
;
SEARCH:		;search for the file given by <de>
		mvi	c,SEARF
		jmp	BDOS$INR
;
;
SEARCHN:	;search for the next occurrence of the file given by <de>
		mvi	c,SEARNF
		jmp	BDOS$INR
;
;
SEARCHCOM:	;search for comfcb file
		lxi	d,COMFCB
		jmp	SEARCH
;
;
DELETE:		;delete the file given by <de>
		mvi	c,DELF
		jmp	BDOS
;
;
BDOS$COND:	call	BDOS
		ora	a
		ret
;
;
DISKREAD:	;read the next record from the file given by <de>
		mvi	c,DREADF
		jmp	BDOS$COND
;
;
DISKREADC:	;read the comfcb file
		lxi	d,COMFCB
		jmp	DISKREAD
;
;
DISKWRITE:	;write the next record to the file given by <de>
		mvi	c,DWRITF
		jmp	BDOS$COND
;
;
MAKE:		;create the file given by <de>
		mvi	c,MAKEF
		jmp	BDOS$INR
;
;
RENAM:		;rename the file given by <de>
		mvi	c,RENF
		jmp	BDOS
;
;
GETUSER:	;return current user code in <a>
		mvi	e,0ffh			;drop through to setuser
;
;
SETUSER:	mvi	c,USERF
		jmp	BDOS			;sets user number
;
;
SAVEUSER:	;save user #/disk # before possible ^C or transient
		call	GETUSER			;code to <a>
		rept 4				;rotate left
		add	a
		endm
		lxi	h,CDISK
		ora	m			;4 bits = user, 4 bits = disk
		sta	DISKA			;stored away in memory for later
		ret
;
;
SETDISKA:	lda	CDISK
		sta	DISKA			;user/disk
		ret
;
;
TRANSLATE:	;translate character in register <a> to upper case
		cpi	'a'
		rc				;return if below lower case a
		cpi	'{'
		rnc				;return if above lower case z
		ani	'_'
		ret				;translated to upper case
;
;
READCOM:	;read the next command into the command buffer
		;check for submit file
		lda	SUBMIT
		ora	a
		jz	NOSUB
;
;	scanning a submit file
;	change drives to open and read the file
;
		lda	CDISK
		ora	a
		mvi	a,00
		cnz	SELECT
;
;	have to open again in case xsub present
;
		lxi	d,SUBFCB
		call	OPEN
		jz	NOSUB			;skip if no submit file
		lda	SUBRC
		dcr	a			;read last record(s) first
		sta	SUBCR			;current record to read
		lxi	d,SUBFCB
		call	DISKREAD		;end of file if last record
		jnz	NOSUB
;
;	disk read is ok, transfer to combuf
;
		lxi	d,COMLEN
		lxi	h,BUFF
		mvi	b,128
		call	MOVE0
;
;	line is transferred, close the with the last
;	deleted record
;
		lxi	h,SUBMOD
		mvi	m,00			;clear fwflag
		inx	h
		dcr	m			;one less record
		lxi	d,SUBFCB
		call	CLOSE
		jz	NOSUB
;
;	close went ok, return to original drive
;
		lda	CDISK
		ora	a
		cnz	SELECT
;
;	print to the 00
;
		lxi	h,COMBUF
		call	PRIN0
		call	BREAK$KEY
		jz	NOREAD
		call	DEL$SUB
		jmp	CCP			;break key depressed
;
;
NOSUB:		;no submit file
		call	DEL$SUB
;
;	translate to upper case, store zero at end
;
		call	SAVEUSER		;user # save in case control C
		mvi	c,RBUFF
		lxi	d,MAXLEN
		call	BDOS
		call	SETDISKA		;no control C, so restore diska
;
NOREAD:		;enter here from submit file
		;set the last character to zero for later scans
		lxi	h,COMLEN
		mov	b,m			;length is in <b>
;
READCOM0:	inx	h
		mov	a,b
		ora	a			;end of scan?
		jz	READCOM1
		mov	a,m			;get character and translate
		call	TRANSLATE
		mov	m,a
		dcr	b
		jmp	READCOM0
;
READCOM1:	;end of scan, <hl> address end of command
		mov	m,a			;store a zero
		lxi	h,COMBUF
		shld	COMADDR			;ready to scan to zero
		ret
;
;
BREAK$KEY:	;check for a character ready at the console
		mvi	c,BREAKF
		call	BDOS
		ora	a
		rz
		mvi	c,RCHARF
		call	BDOS			;character cleared
		ora	a
		ret
;
;
CSELECT:	;get the currently selected drive number to reg <a>
		mvi	c,CSELF
		jmp	BDOS
;
;
SETDMABUFF:	;set default buffer dma address
		lxi	d,BUFF			;(drop through)
;
;
SETDMA:		;set dma address to <de>
		mvi	c,DMAF
		jmp	BDOS
;
;
DEL$SUB:	;delete the submit file, and set submit flag to false
		lxi	h,SUBMIT
		mov	a,m
		ora	a
		rz				;return if no sub file
		mvi	m,00			;submit flag is set to false
		xra	a
		call	SELECT			;on drive A: to erase file
		lxi	d,SUBFCB
		call	DELETE
		lda	CDISK
		jmp	SELECT			;back to original drive
;
;
SERIALIZE:	;check serialization
		lxi	d,SERIAL
		lxi	h,BDOSL
		mvi	b,06			;check six bytes
;
SER0:		ldax	d
		cmp	m
		jnz	BADSERIAL
		inx	d
		inx	h
		dcr	b
		jnz	SER0
		ret				;serial number is ok
;
;
COMERR:		;error in command string starting at position
		;"staddr" and ending with first delimiter
		call	CRLF			;space to next line
		lhld	STADDR			;<hl> address first to print
;
COMERR0:	;print characters until blank or zero
		mov	a,m
		cpi	' '
		jz	COMERR1			;not blank
		ora	a
		jz	COMERR1			;not zero, so print it
		push	h
		call	PRINTCHAR
		pop	h
		inx	h
		jmp	COMERR0			;for another character
;
COMERR1:	;print question mark, and delete sub file
		mvi	a,'?'
		call	PRINTCHAR
		call	CRLF
		call	DEL$SUB
		jmp	CCP			;restart with next command
;
;	fcb scan and fill subroutine (entry is at fillfcb below)
;	fill the comfcb, indexed by <a> (0 or 16)
;	subroutines
;
;
DELIM:		;look for a delimiter
		ldax	d
		ora	a
		rz				;not the last element
		cpi	' '
		jc	COMERR			;non-graphic
		rz				;treat blank as delimiter
		cpi	'='
		rz
		cpi	'_'			;left arrow
		rz
		cpi	'.'
		rz
		cpi	':'
		rz
		cpi	';'
		rz
		cpi	'<'
		rz
		cpi	'>'
		rz
		ret				;delimiter not found
;
;
DEBLANK:	;deblank the input line
		ldax	d
		ora	a
		rz				;treat end of line as blank
		cpi	' '
		rnz
		inx	d
		jmp	DEBLANK
;
;
ADDH:		;add <a> to <hl>
		add	l
		mov	l,a
		rnc
		inr	h
		ret
;
;
FILLFCB0:	;equivalent to fillfcb(0)
		mvi	a,00
;
FILLFCB:	lxi	h,COMFCB
		call	ADDH
		push	h
		push	h			;fcb rescanned a
		xra	a
		sta	SDISK			;clear selected disk (in case A:...)
		lhld	COMADDR
		xchg				;command address in <de>
		call	DEBLANK			;to first non-blank character
		xchg
		shld	STADDR			;in case of errors
		xchg
		pop	h			;<de> has command, <hl> has fcb address
;
;	look for preceding file name A:,B:....
;
		ldax	d
		ora	a
		jz	SETCUR0			;use current disk if empty com line
		sbi	'A'-1
		mov	b,a			;disk name held if <b> if ":" follows
		inx	d
		ldax	d
		cpi	':'
		jz	SETDSK			;set disk name if ":"
;
SETCUR:		;set current disk
		dcx	d			;back to first character of command
;
SETCUR0:	lda	CDISK
		mov	m,a
		jmp	SETNAME
;
;
SETDSK:		;set disk to name in register <b>
		mov	a,b
		sta	SDISK			;mark as disk selected
		mov	m,b
		inx	d			;past the ":"
;
SETNAME:	;set the file name field
		mvi	b,08			;file name length (max)
;
SETNAM0:	call	DELIM
		jz	PADNAME			;not a lelimiter
		inx	h
		cpi	'*'
		jnz	SETNAM1			;must be ?'s
		mvi	m,'?'
		jmp	SETNAM2			;to dec count
;
SETNAM1:	mov	m,a			;store character to fcb
		inx	d
;
SETNAM2:	dcr	b			;count down length
		jnz	SETNAM0
;
;	end of name.  truncate remainder
;
;
TRNAME:		call	DELIM
		jz	SETTY			;set type field if delimiter
		inx	d
		jmp	TRNAME
;
PADNAME:	inx	h
		mvi	m,' '
		dcr	b
		jnz	PADNAME
;
SETTY:		;set the type field
		mvi	b,03
		cpi	'.'
		jnz	PADTY			;skip the type field
		inx	d			;past the "." to the file type field
;
SETTY0:		;set the field from the command buffer
		call	DELIM
		jz	PADTY
		inx	h
		cpi	'*'
		jnz	SETTY1
		mvi	m,'?'			;since "*" specified
		jmp	SETTY2
;
SETTY1:		;not a "*", so copy to type field
		mov	m,a
		inx	d
;
SETTY2:		;decrement count and go again
		dcr	b
		jnz	SETTY0
;
;	end of type field, truncate
;
;
TRTYP:		;truncate type field
		call	DELIM
		jz	EFILL
		inx	d
		jmp	TRTYP
;
;
PADTY:		;pad the type field with blanks
		inx	h
		mvi	m,' '
		dcr	b
		jnz	PADTY
;
;
EFILL:		;end of the filename/filetype fill, save command
		;fill the remaining fields for the fcb
		mvi	b,03
;
EFILL0:		inx	h
		mvi	m,00
		dcr	b
		jnz	EFILL0
		xchg
		shld	COMADDR			;set new starting point
;
;	recover the start address of the fcb and count
;
		pop	h
		lxi	b,0011			;<b>=00, <c>=8+3
;
SCNQ:		inx	h
		mov	a,m
		cpi	'?'
		jnz	SCNQ0
;
;	"?" found, count it in <b>
;
		inr	b
;
SCNQ0:		dcr	c
		jnz	SCNQ
;
;	number of "?"'s in <c>, move to <a> and return with false
;
		mov	a,b
		ora	a
		ret
;
;
INTVEC:		;intrinsic function names (all are four characters)
		db	'DIR '
		db	'ERA '
		db	'TYPE'
		db	'SAVE'
		db	'REN '
		db	'USER'
;
INTLEN		equ	($-INTVEC)/4		;intrinsic function length
;
;
SERIAL		db	0,0,0,0,0,0
;
;
INTRINSIC:	;look for intrinsic functions, (comfcb has been filled)
		lxi	h,INTVEC
		mvi	c,00			;<c> counts intrinsics as scanned
;
INTRIN0:	mov	a,c
		cpi	INTLEN			;done with scan ?
		rnc
;
;	no more to scan
;
		lxi	d,COMFCB+1		;beginning of name
		mvi	b,04			;length of match is in <b>
;
INTRIN1:	ldax	d
		cmp	m			;match?
		jnz	INTRIN2			;skip if no match
		inx	d
		inx	h
		dcr	b
		jnz	INTRIN1			;loop while matching
;
;	complete match on name, check for blank in fcb
;
		ldax	d
		cpi	' '
		jnz	INTRIN3			;otherwise matched
		mov	a,c
		ret				;with intrinsic number in <a>
;
INTRIN2:	;mismatch, move to end of intrinsic
		inx	h
		dcr	b
		jnz	INTRIN2
;
INTRIN3:	;try next intrinsic
		inr	c			;to next intrinsic number
		jmp	INTRIN0			;for another round
;
;
CCPCLEAR:	;clear the command buffer
		xra	a
		sta	COMLEN
;
;	drop through to start ccp
;
;
CCPSTART:	;enter here from boot loader
		lxi	sp,STACK
		push	b			;save initial disk number
;
;	(high order 4 bits=user code, low 4 bits=disk #)
;
		mov	a,c
		rept 4				;user code
		rar
		endm
		ani	0fh			;user code
		mov	e,a
		call	SETUSER			;user code selected
;
;	initialize for this user.  get "$" flag
;
		call	INITIALIZE		;0ff in accum if "$" file present
		sta	SUBMIT			;submit flag set if "$" file present
		pop	b			;recall user code and disk number
		mov	a,c
		ani	0fh			;disk number in accumulator
		sta	CDISK			;clears low memory user code nibble
		call	SELECT			;proper disk is selected, now check sub file
;
;	check for initial command
;
		lda	COMLEN
		ora	a
		jnz	CCP0			;assume typed already
;
CCP:		;enter here on each command, or error condition
		lxi	sp,STACK
		call	CRLF			;print d> prompt, where "d" is disk name
		call	CSELECT			;get current disk number
		adi	'A'
		call	PRINTCHAR
		mvi	a,'>'
		call	PRINTCHAR
		call	READCOM			;command buffer fulled
;
CCP0:		;(enter here from initialization with command full)
		lxi	d,BUFF
		call	SETDMA			;default dma address at buff
		call	CSELECT
		sta	CDISK			;current disk number saved
		call	FILLFCB0		;command fcb filled
		cnz	COMERR			;the name cannot be an ambiguous reference
		lda	SDISK
		ora	a
		jnz	USERFUNC
;
;	check for an intrinsic function
;
		call	INTRINSIC
		lxi	h,JMPTAB		;index is in the accumulator
		mov	e,a
		mvi	d,00
		dad	d
		dad	d			;index in <de>
		mov	a,m
		inx	h
		mov	h,m
		mov	l,a
		pchl
;
;	<pc> changes to the proper intrinsic or user function
;
;
JMPTAB		dw	DIRECT			;directory search
		dw	ERASE			;file erase
		dw	TYPE			;type file
		dw	SAVE			;save memory image
		dw	RENAME			;file rename
		dw	USER			;user number
		dw	USERFUNC		;user-defined function
;
;
BADSERIAL:	lxi	h,di or (hlt shl 8)
		shld	CCPLOC
		lxi	h,CCPLOC
		pchl
;
;	utility subroutines for intrinsic handlers
;
;
READERR:	;print the "read error" message
		lxi	b,RDMSG
		jmp	PRINT
;
RDMSG		db	'Read Error',0
;
;
NOFILE:		;print "no file" message
		lxi	b,NOFMSG
		jmp	PRINT
;
NOFMSG		db	'No File',0
;
;
GETNUMBER:	;read a number from the command line
		call	FILLFCB0		;should be number
		lda	SDISK
		ora	a
		jnz	COMERR			;cannot be prefixed
;
;	convert the byte value in comfcb to binary
;
		lxi	h,COMFCB+1
		lxi	b,0011			;<b>=00, <c>=11
;
;	value accumulated in <b>, <c> counts name length to end
;
;
CONV0:		mov	a,m
		cpi	' '
		jz	CONV1
;
;	more to scan, convert char to binary and save
;
		inx	h
		sui	'0'
		cpi	10
		jnc	COMERR			;value error
		mov	d,a			;save value
		mov	a,b			;multiply by 10
		ani	1110$0000b
		jnz	COMERR
		mov	a,b			;recover value
		rlc
		rlc
		rlc				;*8
		add	b
		jc	COMERR
		add	b
		jc	COMERR			;*8+*2=*10
		add	d
		jc	COMERR			;+digit
		mov	b,a
		dcr	c
		jnz	CONV0			;for another digit
		ret
;
CONV1:		;end of digits, check for all blanks
		mov	a,m
		cpi	' '
		jnz	comerr			;blanks only
		inx	h
		dcr	c
		jnz	CONV1
		mov	a,b			;recover value
		ret
;
;
MOVENAME:	;move 3 characters from <hl> to <de> addresses
		mvi	b,03
;
MOVE0:		mov	a,m
		stax	d
		inx	h
		inx	d
		dcr	b
		jnz	MOVE0
		ret
;
;
ADDHCF:		;buff + <a> + <c> to <hl> followed by fetch
		lxi	h,BUFF
		add	c
		call	ADDH
		mov	a,m
		ret
;
;
SETDISK:	;change disks for this command, if requested
		xra	a
		sta	COMFCB			;clear disk name from fcb
		lda	SDISK
		ora	a
		rz				;no action if not specified
		dcr	a
		lxi	h,CDISK
		cmp	m
		rz				;already selected
		jmp	SELECT
;
;
RESETDISK:	;return to original disk after command
		lda	SDISK
		ora	a
		rz				;no action if not selected
		dcr	a
		lxi	h,CDISK
		cmp	m
		rz				;same disk
		lda	CDISK
		jmp	SELECT
;
;	individual intrinsics follow
;
;
DIRECT:		;directory search
		call	FILLFCB0		;comfcb gets file name
		call	SETDISK			;change disk drives if requested
		lxi	h,COMFCB+1
		mov	a,m			;may be empty request
		cpi	' '
		jnz	DIR1			;skip fill of "???" if not blank
;
;	set comfcb to all "???" for current disk
;
		mvi	b,11			;length of fill (????????.???)
;
DIR0:		mvi	m,'?'
		inx	h
		dcr	b
		jnz	DIR0
;
;	not a blank request, must be in comfcb
;
;
DIR1:		mvi	e,00
		push	d			;<e> counts directory entries
		call	SEARCHCOM		;first one has been found
		cz	NOFILE			;not found message
;
DIR2:		jz	ENDIR
;
;	found, but may be system file
;
		lda	DCNT			;get the location of the element
		rrc
		rrc
		rrc
		ani	0110$0000b
		mov	c,a
;
;	<c> contains base index into buff for dir entry
;
		mvi	a,SYSFILE
		call	ADDHCF			;value to <a>
		ral
		jc	DIR6			;skip if system file
;
;	<c> holds index into buffer
;	another fcb found, new line?
;
		pop	d
		mov	a,e
		inr	e
		push	d
;
;	<e>=0,1,2,3....new line if mod 4 = 0
;
		ani	0000$0011b
		push	psw			;and save the test
		jnz	DIRHDR0			;header on current line
		call	CRLF
		push	b
		call	CSELECT
		pop	b
;
;	current disk in <a>
;
		adi	'A'
		call	PRINTBC
		mvi	a,':'
		call	PRINTBC
		jmp	DIRHDR1			;skip current line hdr
;
DIRHDR0:	call	BLANK			;after last one
		mvi	a,'|'
		call	PRINTBC
;
DIRHDR1:	call	BLANK
;
;	compute position of name in buffer
;
		mvi	b,01			;start with first character of name
;
DIR3:		mov	a,b
		call	ADDHCF			;buff+<a>+<c> fetched
		ani	7fh			;mask flags
;
;	may delete traling blanks
;
		cpi	' '
		jnz	DIR4			;check for blank type
		pop	psw
		push	psw			;may be 3rd item
		cpi	03
		jnz	DIRB			;place blank at end if not
		mvi	a,09
		call	ADDHCF			;first char of type
		ani	7fh
		cpi	' '
		jz	DIR5
;
;	not a blank in the file type field
;
;
DIRB:		mvi	a,' '			;restore trailing filename char
;
DIR4:		call	PRINTBC			;char printed
		inr	b
		mov	a,b
		cpi	12
		jnc	DIR5
;
;	check for break between names
;
		cpi	09
		jnz	DIR3			;for another char
;
;	print a blank between names
;
		call	BLANK
		jmp	DIR3
;
DIR5:		;end of current entry
		pop	psw			;discard the directory counter (mod 4)
;
DIR6:		call	BREAK$KEY		;check for interrupt at keyboard
		jnz	ENDIR			;abort directory search
		call	SEARCHN
		jmp	DIR2			;for another entry
;
ENDIR:		;end of directory scan
		pop	d			;discard directory counter
		jmp	RETCOM
;
;
ERASE:		call	FILLFCB0		;cannot be all "???"'s
		cpi	11
		jnz	ERASEFILE
;
;	erasing all of the disk
;
		lxi	b,ERMSG
		call	PRINT
		call	READCOM
		lxi	h,COMLEN
		dcr	m
		jnz	CCP			;bad input
		inx	h
		mov	a,m
		cpi	'Y'
		jnz	CCP
;
;	ok, erase the entire diskette
;
		inx	h
		shld	COMADDR			;otherwise error at retcom
;
ERASEFILE:	call	SETDISK
		lxi	d,COMFCB
		call	DELETE
		inr	a			;255 returned if not found
		cz	NOFILE			;no file message if so
		jmp	RETCOM
;
ERMSG		db	'All (y/n)?',0
;
;
TYPE:		call	FILLFCB0
		jnz	COMERR			;don't allow ?'s in file name
		call	SETDISK
		call	OPENC			;open the file
		jz	TYPERR			;zero flag indicates not found
;
;	file opened.  read 'till eof
;
		call	CRLF
		lxi	h,BPTR
		mvi	m,255			;read first buffer
;
TYPE0:		;loop on bptr
		lxi	h,BPTR
		mov	a,m
		cpi	128			;end of buffer
		jc	TYPE1
		push	h			;carry if 0,1,... 127
;
;	read another buffer full
;
		call	DISKREADC
		pop	h			;recover address of buffer
		jnz	TYPEOF			;hard end of file
		xra	a
		mov	m,a			;bptr = 00
;
TYPE1:		;read character at bptr and print
		inr	m			;bptr = bptr+1
		lxi	h,BUFF
		call	ADDH			;<hl> addresses char
		mov	a,m
		cpi	EOFILE
		jz	RETCOM
		call	PRINTCHAR
		call	BREAK$KEY
		jnz	RETCOM			;abort if break key
		jmp	TYPE0			;for another character
;
TYPEOF:		;end of file.  check for errors
		dcr	a
		jz	RETCOM
		call	READERR
;
TYPERR:		call	RESETDISK
		jmp	COMERR
;
;
SAVE:		call	GETNUMBER		;value to register <a>
		push	psw			;save it for later
;
;	should be followed by a file name to save
;	the memory image.
;
		call	FILLFCB0
		jnz	COMERR			;cannot be ambiguous
		call	SETDISK			;may be a disk change
		lxi	d,COMFCB
		push	d
		call	DELETE			;existing file
		pop	d
		call	MAKE			;create a new file on the disk
		jz	SAVERR			;no directory space
		xra	a
		sta	COMREC			;clear next record field
		pop	psw			;# of pages to write is in <a>.
						;change to # of sectors
		mov	l,a
		mvi	h,00
		dad	h
		lxi	d,TRAN			;<hl> is sector count, <de> is load address
;
SAVE0:		;check for sector count zero
		mov	a,h
		ora	l
		jz	SAVE1			;may be completed
		dcx	h			;sector count = sector count-1
		push	h			;save it for next time around
		lxi	h,128
		dad	d
		push	h			;next dma address saved
		call	SETDMA			;current dma address set
		lxi	d,COMFCB
		call	DISKWRITE
		pop	d
		pop	h			;dma address, sector count
		jnz	SAVERR			;may be disk full case
		jmp	SAVE0			;for another sector
;
SAVE1:		;end of dump. close the file
		lxi	d,COMFCB
		call	CLOSE
		inr	a			;255 becomes 00 if error
		jnz	RETSAVE			;for another command
;
SAVERR:		;must be full or read only disk
		lxi	b,FULLMSG
		call	PRINT
;
RETSAVE:	;reset dma buffer
		call	SETDMABUFF
		jmp	RETCOM
;
FULLMSG		db	'No Space',00
;
;
RENAME:		;rename a file on a specific disk
		call	FILLFCB0
		jnz	COMERR			;must be unambiguous
		lda	SDISK
		push	psw			;save for later compare
		call	SETDISK			;disk selected
		call	SEARCHCOM		;is new name already there?
		jnz	RENERR3
;
;	file doesn't exist, move to second half of fcb
;
		lxi	h,COMFCB
		lxi	d,COMFCB+16
		mvi	b,16
		call	MOVE0
;
;	check for "=" or left arrow
;
		lhld	COMADDR
		xchg
		call	DEBLANK
		cpi	'='
		jz	REN1			;ok if "="
		cpi	LA
		jnz	RENERR2
;
REN1:		xchg
		inx	h
		shld	COMADDR			;past delimiter
;
;	proper delimiter found
;
		call	FILLFCB0
		jnz	RENERR2
;
;	check for drive conflict
;
		pop	psw
		mov	b,a			;previous drive number
		lxi	h,SDISK
		mov	a,m
		ora	a
		jz	REN2
;
;	drive name was specified.  save one?
;
		cmp	b
		mov	m,b
		jnz	RENERR2
;
REN2:		mov	m,b			;store the name in case drives switched
		xra	a
		sta	COMFCB
		call	SEARCHCOM		;is old file type
		jz	RENERR1
;
;	everything is ok, rename the file
;
		lxi	d,COMFCB
		call	RENAM
		jmp	RETCOM
;
RENERR1:	;no file on disk
		call	NOFILE
		jmp	RETCOM
;
RENERR2:	;ambiguous reference/name conflict
		call	RESETDISK
		jmp	COMERR
;
RENERR3:	;file already exists
		lxi	b,RENMSG
		call	PRINT
		jmp	RETCOM
;
RENMSG		db	'File Exists',00
;
;
USER:		;set user number
		call	GETNUMBER		;leaves the value in the accumulator
		cpi	16
		jnc	COMERR			;must be between 00 and 15
		mov	e,a			;save for setuser call
		lda	COMFCB+1
		cpi	' '
		jz	COMERR
		call	SETUSER			;new user number set
		jmp	ENDCOM
;
;
USERFUNC:	call	SERIALIZE		;check serialization
;
;	load user function and set up for execution
;
		lda	COMFCB+1
		cpi	' '
		jnz	USER0
;
;	no file name, but may be disk switch
;
		lda	SDISK
		ora	a
		jz	ENDCOM			;no disk name if 00
		dcr	a
		sta	CDISK
		call	SETDISKA		;set user/disk
		call	SELECT
		jmp	endcom
;
USER0:		;file name is present
		lxi	d,COMFCB+9
		ldax	d
		cpi	' '
		jnz	COMERR			;type error
;
USER0A:		push	d
		call	SETDISK
		pop	d
		lxi	h,COMTYPE		;.com
		call	MOVENAME		;file type is set to .com
		call	OPENC
;
		if SPECIAL
		lxi	h,SDISK
		ora	m
		jnz	USERER
		inr	m
		lxi	d,COMFCB+9
		jmp	USER0A
;
	else
		jz	USERER
		endif
;
;	file opened properly.  read it into memory
;
		lxi	h,TRAN			;transient program base
;
LOAD0:		push	h			;save dma address
		xchg
		call	SETDMA
		lxi	d,COMFCB
		call	DISKREAD
		jnz	LOAD1
;
;	sector loaded, set new dma address and repeat
;
		pop	h
		lxi	d,128
		dad	d
		lxi	d,TRANM			;has the load overflowed?
		mov	a,l
		sub	e
		mov	a,h
		sbb	d
		jnc	LOADERR
		jmp	LOAD0			;for another sector
;
LOAD1:		pop	h
		dcr	a
		jnz	LOADERR			;end file is 1
		call	RESETDISK		;back to original disk
		call	FILLFCB0
		lxi	h,SDISK
		push	h
		mov	a,m
		sta	COMFCB			;drive number set
		mvi	a,16
		call	FILLFCB			;move entire fcb
		pop	h
		mov	a,m
		sta	COMFCB+16
		xra	a
		sta	COMREC			;record number set to zero
		lxi	d,FCB
		lxi	h,COMFCB
		mvi	b,33
		call	MOVE0
;
;	move command line to buff
;
		lxi	h,COMBUF
;
BMOVE0:		mov	a,m
		ora	a
		jz	BMOVE1
		cpi	' '
		jz	BMOVE1
		inx	h
		jmp	BMOVE0			;for another scan
;
;	first blank position found
;
;
BMOVE1:		mvi	b,00
		lxi	d,BUFF+1		;ready for the move
;
BMOVE2:		mov	a,m
		stax	d
		ora	a
		jz	BMOVE3
;
;	more to move
;
		inr	b
		inx	h
		inx	d
		jmp	BMOVE2
;
BMOVE3:		;<b> has character count
		mov	a,b
		sta	BUFF
		call	CRLF
;
;	now go to the loaded program
;
		call	SETDMABUFF		;default dma
		call	SAVEUSER		;user code saved
;
;	low memory diska contains user code
;
		call	TRAN			;gone to the loaded program
		lxi	sp,STACK		;may come back here
		call	SETDISKA
		call	SELECT
		jmp	CCP
;
USERER:		;arive here on command error
		call	RESETDISK
		jmp	COMERR
;
LOADERR:	;cannot load the program
		lxi	b,LOADMSG
		call	PRINT
		jmp	RETCOM
;
LOADMSG		db	'Bad Load',00
;
COMTYPE		db	'COM'			;for com files
;
;
RETCOM:		;reset disk before end of command check
		call	RESETDISK
;
;
ENDCOM:		;end of intrinsic command
		call	FILLFCB0		;to check for garbage at end of line
		lda	COMFCB+1
		sui	' '
		lxi	h,SDISK
		ora	m
;
;	00 in accumulator if no disk selected, and blank fcb
;
		jnz	COMERR
		jmp	CCP
;
;
;	data areas
;
		ds	16			;8 level stack
STACK		equ	$
;
;
;	"submit" file control block
;
SUBMIT		db	00			;00 if no submit file. 0ff if submitting
SUBFCB		db	00,'$$$     '		;filename is "$$$"
		db	'SUB',00,00		;filetype is "sub"
SUBMOD		db	00			;module number
SUBRC		ds	01			;record count field
		ds	16			;disk map
SUBCR		ds	01			;current record to read
;
;	command file control block
;
COMFCB		ds	32			;fields filled in later
COMREC		ds	1			;current record to read/write
DCNT		ds	1			;disk directory count (used for error count)
CDISK		ds	1			;current disk
SDISK		ds	1			;selected disk for current operation
						;none=00, A=1, B=2 ...
BPTR		ds	1			;buffer pointer

		end	CCPLOC
