;
;	WRITTEN BY KEJ AND RLH
;	LAST MODIFIED 11-Oct-80 by JLZ
;		       1-NOV-80 BY RLH
;		       8-DEC-80 BY RLH
;

;
;	ASSEMBLY CONSTANTS
;

ORIGIN	EQU	100H
TRUE	EQU	0FFH
FALSE	EQU	0

KONAN	EQU	FALSE		;KONAN HARD DISK DRIVER?
NDISKS	EQU	3		;NUMBER OF DISK UNITS ACTIVE
ADAYBOOT EQU	FALSE		;TRUE IF NO CONSOLE AT HUB
NSATS	EQU	1		;NUMBER OF SATELLITES

IOBYTE	EQU	3		;I/O STATUS BYTE ADDRESS
NOCONS	EQU	2		;VALUE OF IO BYTE IF NO CONSLOLE AT HUB
LASTDSK	EQU	4		;ADDRESS OF LAST DISK LOGGED BY CLI
ANOS	EQU	5		;ENTRY-POINT TO OPERATING SYSTEM
TBUF	EQU	80H		;DEFAULT DISK BUFFER
TFCB	EQU	5CH		;DEFAULT FILE CONTROL BLOCK

CR	EQU	0DH		;CARRIAGE RETURN
LF	EQU	0AH		;LINE FEED
ACK	EQU	6		;POSITIVE ACKNOWLEDGE
NAK	EQU	15H		;NEGATIVE ACKNOWLEDGE
SYN	EQU	16H		;SYNCHRONIZATION CHARACTER

CKBR	EQU	40H		;CHECK IF BYTE READY FROM TUART
CPTR	EQU	80H		;CHECK IF OUTPUT PORT READY

ET	EQU	0		;ENTRY TYPE FIELD IN FCB
FN	EQU	ET+1		;FILE NAME FIELD
FT	EQU	FN+8		;FILE TYPE FIELD
XT	EQU	FT+3		;EXTENT FIELD
R1	EQU	XT+1		;RESERVED
R2	EQU	R1+1		;RESERVED
CT	EQU	R2+1		;RECORD COUNT
DM	EQU	CT+1		;DISK MAP
NR	EQU	DM+16		;NEXT RECORD

PRINTB	EQU	9		;PRINT BUFFER REQUEST
OPEN	EQU	15		;OPEN REQUEST
CLOSE	EQU	16		;CLOSE REQUEST
SEARCH	EQU	17		;SEARCH FOR FILE
DELETE	EQU	19		;DELETE FILE REQUEST
OPENX	EQU	33		;EXTENDED OPEN REQUEST
READN	EQU	20		;READ NEXT RECORD REQUEST
SETDMAR	EQU	26		;SET DMA ADDRESS REQUEST
SETUI	EQU	28		;SET USER INDEX REQUEST
SETERRX	EQU	36		;SET ERROR EXIT ADDRESS
SETBUFF	EQU	37		;SET BUFFER AREA ADDRESS
SETUNIT	EQU	14		;SELECT UNIT

	IF KONAN
HDFSTRK	EQU	100		;FIRST TRACK USED ON HARD DISK
	ENDIF

CSP	EQU	0		;CRT STATUS PORT
CDP	EQU	1		;CRT DATA PORT
LSP	EQU	04H		;LIST STATUS PORT 
LDP	EQU	04H		;LIST DATA PORT
LSTROBE	EQU	0C4H		;TUART PORT (TO STROBE DATA READY FOR PRINTER)

	ORG	ORIGIN

;
;	ENTRY-POINT JUMP VECTOR
;

	JMP	COLD		;COLD DEADSTART
	JMP	WARM		;WARM DEADSTART
	JMP	CONSTAT		;RETURN CONSOLE STATUS
	JMP	CONIN		;INPUT CHARACTER FROM CONSOLE
	JMP	CONOUT		;OUTPUT CHARACTER TO CONSOLE
	JMP	BIOS$LIST	;OUTPUT CHARACTER TO LIST DEVICE
	JMP	BIOS$PUNCH	;OUTPUT CHARACTER TO PUNCH DEVICE
	JMP	BIOS$READER	;INPUT  CHARACTER FROM READER DEVICE
	JMP	HOME		;INITIALIZE MASS STORAGE UNIT
	JMP	SELUNIT		;SELECT MASS STORAGE UNIT
	JMP	SELTRK		;SELECT TRACK
	JMP	SELSEC		;SELECT SECTOR
	JMP	SETDMA		;SET DMA ADDRESS
	JMP	READ		;READ FROM MASS STORAGE UNIT
	JMP	WRITE		;WRITE TO MASS STORAGE UNIT
	JMP	SELHEAD		;SELECT HEAD
	JMP	GETUNITS	;RETURN NUMBER OF ACTIVE UNITS
	JMP	SELSAT		;SELECT SATELLITE
	JMP	INTSAT		;INTERROGATE SATELLITE SELECTION
	JMP	SATSTAT		;RETURN SATELLITE STATUS
	JMP	RECEIVE		;RECEIVE MESSAGE FROM SATELLITE
	JMP	SEND		;SEND MESSAGE TO SATELLITE
	JMP	RETCLI		;DUMMY FOR NOW
	JMP	USER0		;4 USER BACKGROUD JOBS FOR ADAY
	JMP	USER1
	JMP	USER2
	JMP	USER3

;
;	STACK SPACE
;

	DS	32
STACK:	DS	0

;
;	ANOS ENTRY-POINT ADDRESS STORAGE
;

ENTRY:	DW	0

;
;	COLDSTART FLAG
;

COLDFLAG:DB	TRUE

;
;	COLD - PROCESS COLD DEADSTART
;
;	THE COLD DEADSTART LOADER CALLS THIS ROUTINE AFTER LOADING
;	THE OPERATING SYSTEM INTO MEMORY.  THIS ROUTINE IS RESPONSIBLE FOR
;	INITIALIZING DEVICES AND SYSTEM PARAMETERS.
;


COLD:	LXI	SP,STACK

;
;	INIT THE CONSOLE (TU-ART)
;
;	***** Deleted.  11-Oct-80  JLZ *****
;

;	MVI	A,11H		;1 FOR RESET 10H FOR OCTUPLE THE CLOCK
;	OUT	CSP+2		;INITIALIZE CONSOLE PORT
;	MVI	A,90H		;1 STOP BIT (80H) 2400(*8) BAUD (10H)
;	OUT	CSP		;BAUD RATE REGISTER
;	XRA	A		;CLEAR INTERRUPTS
;	OUT	CSP+3

;
;	INIT THE SATELLITES TO 76K BAUD
;

	MVI	A,NSATS		;NUMBER OF SATELLITES
	MOV	B,A		;COUNTER
	MVI	C,0		;FIRST SATELLITE

C1:	PUSH	B		;SAVE SATELLITE AND COUNT
	CALL	SELSAT		;SET UP PORT OUTPUT ROUTINES
	MVI	A,11H		;RESET AND OCTUPLE BAUD
	CALL	OCP		;OUTPUT TO COMMAND PORT
	MVI	A,0C0H		;ONE STOP BIT AND 9600*8 BAUD
	CALL	OSP		;OUTPUT TO BAUD RATE PORT
	XRA	A		;CLEAR A FOR INTERRUPTS
	CALL	OIP		;CLEARS THE INTERRUPTS
	POP	B		;GET SATELLITE AND COUNT BACK
	INR	C		;NEXT SATELLITE
	DCR	B		;COUNT
	JNZ	C1		;IF ALL SATELLITES NOT INITIALIZED

;
;	INIT THE PRINTER
;
;	***** Deleted.   11-Oct-80  JLZ  *****

;	MVI	C,01H
;	OUT	LSTROBE
;	MVI	B,0		;DELAY
;C2:	DCR	B
;	JNZ	C2		;FOR AT LEAST 25 NANO SEC
;	MVI	A,03H
;	OUT	LSTROBE
				;INTERRUPTS FOR THIS PORT CLEARED BY SAT INIT

;
;	INIT ENTRY POINTS
;

	MVI	A,0C3H		;STORE JUMP TO WARMSTART
	LXI	H,ORIGIN+3
	STA	0
	SHLD	1
	XRA	A		;INIT I/O STATUS BYTE AND LAST LOGGED DISK
	STA	IOBYTE
	STA	LASTDSK

	IF ADAYBOOT
	MVI	A,NOCONS	;VALUE OF IO BYTE IF NO CONSOLE AT HUB
	STA	IOBYTE		;LOGIN.PRC MUST EXECUTE DESIRED PROGRAM
	ENDIF

	LHLD	ANOS+1		;SAVE ANOS ENTRY-POINT ADDRESS
	SHLD	ENTRY

;
;	FALL THROUGH TO WARMSTART
;

;
;	WARM - PROCESS WARM DEADSTART
;

WARM:	LXI	SP,STACK
	LHLD	ENTRY		;RESTORE ANOS ENTRY-POINT IN CASE IT WAS DAMAGED
	SHLD	ANOS+1
	MVI	A,JMP
	STA	ANOS

	STA	38H		;STORE JUMP TO WARMSTART ON INTERRUPT
	LXI	H,WARM
	SHLD	39H

	MVI	C,SETBUFF	;SET BUFFER AREA ADDRESS
	LXI	D,BUFFER
	CALL	ANOS

	MVI	C,SETUNIT	;SET SYSTEM UNIT
	MVI	E,0
	CALL	ANOS

	MVI	C,SETERRX	;SET ERROR EXIT ADDRESS
	LXI	D,0
	CALL	ANOS

	CALL	LOADCLI		;LOAD THE COMMAND LANGUAGE INTERPRETER
	RET

;
;	LOADCLI - LOAD THE COMMAND LANGUAGE INTERPRETER
;
;	LOADCLI SEARCHES FOR A FILE CALLED 'CLI.IMG' IN THE SYSTEM LIBRARY.
;	IT OPENS THE FILE, DETERMINES THE LENGTH OF THE CLI IMAGE, LOADS
;	IT INTO MEMORY ABUTTING THE OPERATING SYSTEM PROPER, AND RELOCATES
;	ALL THE RELOCATABLE ADDRESSES IN IT.  FINALLY, IT TRANSFERS CONTROL
;	TO THE CLI.
;

LOADCLI:CALL	OPENCLI		;OPEN THE IMAGE FILE
	CALL	GETLEN		;GET THE LENGTH OF THE IMAGE
	CALL	GETMAP		;GET THE BIT MAP
	CALL	RELOC		;READ AND RELOCATE THE IMAGE

;
;	TRANSFER CONTROL TO THE COMMAND LANGUAGE INTERPRETER
;

	LXI	H,COLDFLAG	;PASS COLDSTART FLAG TO CLI
	MOV	B,M
	IF	ADAYBOOT
	MVI	B,0
	ENDIF
	MVI	M,0
	MOV	A,B
	LHLD	CLIADDR
	PCHL

;
;	OPENCLI - OPEN THE CLI IMAGE FILE
;

OPENCLI:XRA	A		;INITIALIZE THE FCB
	STA	CLIFCB+XT
	STA	CLIFCB+NR
	MVI	C,SETDMAR	;SET THE DMA ADDRESS TO TBUF
	LXI	D,TBUF
	CALL	ANOS

	MVI	C,OPENX		;OPEN THE FILE
	LXI	D,CLIFCB
	CALL	ANOS
	INR	A
	JZ	OPENCL1		;IF CLI NOT FOUND
	RET

OPENCL1:MVI	C,PRINTB	;ISSUE ERROR MESSAGE
	LXI	D,NOTFND
	CALL	ANOS

OPENCL2:JMP	OPENCL2		;HANG

;
;	FCB OF CLI
;

CLIFCB:	DB	0
	DB	'CLI     IMG'
	DB	0
	DW	0FFFFH		;USER INDEX OF SYSTEM LIBRARY
	DB	0
	DS	16
	DB	0

CLINDX:	DB	128		;BUFFER INDEX

NOTFND:	DB	'FROM ANOS:  CLI NOT FOUND$'

;
;	GETLEN - GET LENGTH OF CLI IMAGE
;

GETLEN:	MVI	A,128		;INITIALIZE BUFFER INDEX TO FORCE READ OF RECORD
	STA	CLINDX
	CALL	NEXT		;GET LOW BYTE OF LENGTH
	STA	CLILEN
	CALL	NEXT		;GET HIGH BYTE
	STA	CLILEN+1
	RET

;
;	GETMAP - GET BIT MAP
;
;	THE BIT MAP IS THE KEY TO THE RELOCATABILITY OF THE CLI IMAGE.
;	EACH BIT IN THE MAP CORRESPONDS TO A BYTE IN THE CLI IMAGE.  IF
;	A BIT IS SET (1) THEN ITS CORRESPONDING BYTE IS RELOCATABLE.  IF
;	A BIT IS RESET (0) THEN ITS CORRESPONDING BYTE IS NOT RELOCATABLE.
;

GETMAP:	LHLD	CLILEN		;COMPUTE ADDRESS OF CLI
	XCHG
	LHLD	ANOS+1
	LXI	B,-6
	DAD	B
	MOV	A,L
	SUB	E
	MOV	L,A
	MOV	A,H
	SBB	D
	MOV	H,A
	SHLD	CLIADDR

;
;	COMPUTE ADDRESS OF BIT MAP
;

	LHLD	CLILEN		;DIVIDE LENGTH BY 8 TO GET LENGTH OF BIT MAP
	MVI	B,3

GETMAP1:MOV	A,H		;SHIFT LENGTH RIGHT 3 BITS
	ORA	A
	RAR
	MOV	H,A
	MOV	A,L
	RAR
	MOV	L,A
	DCR	B
	JNZ	GETMAP1		;IF SHIFT NOT COMPLETE

	SHLD	BITLEN		;SAVE LENGTH OF BIT MAP
	XCHG
	LHLD	CLIADDR		;COMPUTE ADDRESS OF BIT MAP
	MOV	A,L
	SUB	E
	MOV	L,A
	MOV	A,H
	SBB	D
	MOV	H,A
	SHLD	BITADDR
	XCHG
	LHLD	BITLEN

;
;	READ BIT MAP
;

GETMAP2:PUSH	D
	PUSH	H
	CALL	NEXT		;READ NEXT 8 BITS
	POP	H
	POP	D
	STAX	D		;STORE BITS
	INX	D		;INCREMENT ADDRESS
	DCX	H		;DECREMENT LENGTH
	MOV	A,H
	ORA	L
	JNZ	GETMAP2		;IF MAP NOT COMPLETELY READ
	RET


;
;	RELOC - READ AND RELOCATE CLI IMAGE
;
;	RELOC READS BYTES FROM THE CLI SOURCE IMAGE FILE AND ADDS THE
;	RELOCATION BASE TO THOSE BYTES WHOSE CORRESPONDING BITS IN THE BIT MAP
;	ARE SET TO 1.
;

RELOC:	LHLD	BITADDR		;INITIALIZE BIT MAP ADDRESS
	XCHG
	LHLD	CLIADDR		;INITIALIZE DESTINATION ADDRESS
	SHLD	NEXTADDR

RELOC1:	MVI	B,8		;INITIALIZE COUNT OF BITS IN BYTE
	LDAX	D		;FETCH NEXT MAP ENTRY
	MOV	C,A
	PUSH	D

RELOC2:	PUSH	B
	CALL	NEXT		;READ NEXT BYTE OF IMAGE
	POP	B
	MOV	H,A
	MOV	A,C
	RRC
	MOV	C,A
	MVI	A,0
	JNC	RELOC3		;IF BYTE NOT RELOCATABLE
	LDA	CLIADDR+1

RELOC3:	ADD	H
	LHLD	NEXTADDR	;STORE OBJECT BYTE
	MOV	M,A
	INX	H
	SHLD	NEXTADDR
	LHLD	CLILEN		;DECREMENT COUNT OF BYTES IN CLI
	DCX	H
	SHLD	CLILEN
	MOV	A,H
	ORA	L
	JZ	RELOC4		;IF RELOCATION COMPLETE
	DCR	B
	JNZ	RELOC2		;IF NOT END OF BITS IN BYTE
	POP	D
	INX	D
	JMP	RELOC1

RELOC4:	POP	D
	RET

CLIADDR:DW	0		;ADDRESS OF CLI

BITADDR:DW	0		;ADDRESS OF BIT MAP

CLILEN:	DW	0		;LENGTH OF CLI

BITLEN:	DW	0		;LENGTH OF BIT MAP

NEXTADDR:DW	0		;NEXT ADDRESS TO STORE RELOCATED BYTE

;
;	NEXT - READ NEXT BYTE FROM CLI IMAGE FILE
;
;	EXIT	(A) - BYTE
;

NEXT:	LDA	CLINDX
	ORA	A
	JP	NEXT1		;IF BUFFER NOT EXHAUSTED

	MVI	C,READN		;READ NEXT RECORD
	LXI	D,CLIFCB
	CALL	ANOS
	ORA	A
	JNZ	NEXT2		;IF END OF FILE

NEXT1:	MOV	E,A
	MVI	D,0
	INR	A		;INCREMENT BUFFER INDEX
	STA	CLINDX
	LXI	H,TBUF
	DAD	D
	MOV	A,M
	RET

NEXT2:	MVI	C,PRINTB	;ISSUE ERROR MESSAGE
	LXI	D,DAMAGE
	CALL	ANOS

NEXT3:	JMP	NEXT3		;HANG

DAMAGE:	DB	'FROM ANOS:  CLI DAMAGED$'

;
;	CONSTAT - RETURN CONSOLE STATUS
;
;	EXIT	(A) - 0 IF NO CHARACTER AVAILABLE
;		      0FFH IF CHARACTER AVAILABLE
;

CONSTAT:LDA	IOBYTE
	CALL	DISPATCH

	DW	BIOS$CONSTAT
	DW	BIOS$CONSTAT
	IF NOT ADAYBOOT
	DW	BIOS$CONSTAT
	ENDIF
	IF ADAYBOOT
	DW	BOOT$CONSTAT
	ENDIF
	DW	SAT$CONSTAT

;
;	CONIN - INPUT CHARACTER FROM CONSOLE
;
;	EXIT	(A) - CHARACTER
;

CONIN:	LDA	IOBYTE
	CALL	DISPATCH

	DW	BIOS$CONIN
	DW	BIOS$CONIN
	IF ADAYBOOT
	DW	BOOT$CONSTAT
	ENDIF
	IF NOT ADAYBOOT
	DW	BIOS$CONIN
	ENDIF
	DW	SAT$CONIN

;
;	CONOUT - OUTPUT CHARACTER TO CONSOLE
;
;	ENTRY	(C) - CHARACTER
;

CONOUT:	LDA	IOBYTE
	CALL	DISPATCH

	DW	BIOS$CONOUT
	DW	BIOS$CONOUT
	IF ADAYBOOT
	DW	BOOT$CONOUT
	ENDIF
	IF NOT ADAYBOOT
	DW	BIOS$CONOUT
	ENDIF
	DW	SAT$CONOUT


;
;	DISPATCH - DISPATCH DEVICE DRIVER
;
;	ENTRY	(A) - DEVICE DRIVER ORDINAL RIGHT JUSTIFIED
;		TOP OF STACK - ADDRESS OF TABLE OF DEVICE DRIVER ADDRESSES
;

DISPATCH:ANI	3

DISPAT1:RLC
	MOV	E,A
	MVI	D,0
	POP	H
	DAD	D
	MOV	A,M
	INX	H
	MOV	H,M
	MOV	L,A
	PCHL


;
;	HOME - INITIALIZE MASS STORAGE UNIT
;

HOME:	RET

;
;	SELUNIT - SELECT MASS STORAGE UNIT
;
;	ENTRY	(C) - UNIT NUMBER
;
;	EXIT	(BA) - ADDRESS OF UNIT PARAMETER BLOCK
;

SELUNIT:MOV	A,C
	STA	UNIT
	LXI	D,PRMTABL
	LXI	H,PRMTAB

SELU1:	DCR	C
	JM	SELU2		;IF ADDRESS FOUND
	DAD	D
	JMP	SELU1

SELU2:	MOV	B,H
	MOV	A,L
	RET

;
;	SELTRK - SELECT TRACK
;
;	ENTRY	(BC) - TRACK
;

SELTRK:	MOV	H,B
	MOV	L,C
	SHLD	TRACK
	RET

;
;	SELSEC - SELECT SECTOR
;
;	ENTRY	(BC) - SECTOR
;

SELSEC:	MOV	H,B
	MOV	L,C
	SHLD	SECTOR
	RET

;
;	SELHEAD - SELECT HEAD
;
;	ENTRY	(C) - HEAD
;

SELHEAD:MOV	A,C
	STA	HEAD
	RET

;
;	SETDMA - SET DMA ADDRESS
;
;	ENTRY	(BC) - DMA ADDRESS
;

SETDMA:	MOV	H,B
	MOV	L,C
	SHLD	DMA
	RET

;
;	GETUNITS - RETURN NUMBER OF ACTIVE UNITS
;
;	EXIT	(A) - NUMBER OF UNITS
;

GETUNITS:LDA	UNITS
	RET

;
;	READ - READ FROM MASS STORAGE UNIT
;
;	A 1K BLOCK IS READ FROM THE CURRENT UNIT STARTING AT THE CURRENT
;	HEAD, TRACK, AND SECTOR.  THE BLOCK IS READ INTO A BUFFER SPECIFIED
;	BY THE CURRENT DMA ADDRESS.
;

READ:	LDA	UNIT		;DISPATCH THE APPROPRIATE DEVICE DRIVER
	CALL	DISPAT1

	IF KONAN
	DW	HDREAD		;UNIT 0 IS THE HARD DISK
	ENDIF

	DW	FPREAD		;UNIT 1 IS FLOPPY UNIT 0
	DW	FPREAD		;UNIT 2 IS FLOPPY UNIT 1
	DW	FPREAD		;UNIT 3 IS FLOPPY UNIT 2

;
;	WRITE - WRITE TO MASS STORAGE UNIT
;
;	A 1K BLOCK IS WRITTEN ON THE CURRENT UNIT BEGINNING AT THE CURRENT
;	HEAD, TRACK, AND SECTOR.  THE BLOCK IS WRITTEN FROM THE BUFFER
;	SPECIFIED BY THE CURRENT DMA ADDRESS.
;

WRITE:	LDA	UNIT		;DISPATCH DEVICE DRIVER
	CALL	DISPAT1

	IF KONAN
	DW	HDWRITE		;UNIT 0 IS THE HARD DISK
	ENDIF

	DW	FPWRITE		;UNIT 1 IS FLOPY UNIT 0
	DW	FPWRITE		;UNIT 2 IS FLOPPY UNIT 1
	DW	FPWRITE		;UNIT 3 IS FLOPPY UNIT 2

	IF KONAN
;
;	HDREAD - READ FROM HARD DISK
;

HDREAD:	MVI	A,0
	JMP	HDDRIVER

;
;	HDWRITE - WRITE TO HARD DISK
;

HDWRITE:MVI	A,1
	JMP	HDDRIVER

;
;	HDDRIVER - HARD DISK DRIVER
;
;	HDDRIVER READS/WRITES1 KILOBYTE  FROM/TO THE HARD DISK UNIT.  IT
;	TRANSLATES THE LOGICAL HEAD, TRACK, AND SECTOR INTO PHYSICAL
;	EQUIVALENTS.
;
;	ENTRY	(A) - DISK COMMAND
;
;	EXIT	(A) - STATUS REPLY OF HARD DISK CONTROLLER
;

HDDRIVER:STA	HDCMD		;STORE COMMAND
	MVI	A,4		;SET NUMBER OF SECTORS TO TRANSFER
	STA	HDSECS
	LHLD	DMA		;SET DMA ADDRESS
	MOV	A,H
	MOV	H,L
	MOV	L,A
	SHLD	HDDMA
	LDA	SECTOR		;SET SECTOR
	STA	HDSEC
	LHLD	TRACK		;SET TRACK
	LXI	D,HDFSTRK	;ADD OFFSET TO FIRST LOGICAL TRACK
	DAD	D
	MOV	A,H
	MOV	H,L
	MOV	L,A
	SHLD	HDTRK
	LDA	HEAD		;SET HEAD
	STA	HDHEAD
	XRA	A
	STA	HDUNIT		;SET UNIT
	STA	HDSTAT		;INITIALIZE STATUS
	CALL	HDDRIV1		;EXECUTE OPERATION
	LDA	HDSTAT
	RET

HDDRIV1:CALL	0F840H		;CALL CONTROLLER
	RET

;
;	HARD DISK COMMAND VECTOR
;

HDSTAT:	DB	0		;STATUS
HDCMD:	DB	0		;COMMAND
HDTRK:	DW	0		;TRACK
HDHEAD:	DB	0		;HEAD
HDSEC:	DB	0		;SECTOR
HDDMA:	DW	0		;DMA ADDRESS
HDSECS:	DB	0		;NUMBER OF SECTORS TO TRANSFER
HDUNIT:	DB	0		;UNIT
	DB	0		;OPTIONS
	DB	3		;HIGHEST HEAD
	DB	319 SHR 8,319 AND 0FFH ;HIGHEST TRACK
	DB	63		;HIGHEST SECTOR
	DB	0,0,0,0,0,0,0,0
	DB	0FFH		;END OF TRACK SUBSTITUTION TABLE

	ENDIF

;
;	UNIT STATISTICS
;

UNITS:	DB	NDISKS		;NUMBER OF ACTIVE UNITS
				;SET AT TOP

;
;	PRMTAB - UNIT PARAMETER TABLE
;

PRMTAB:	DS	0

	IF KONAN

;
;	UNIT 0 PARAMETERS
;

	DB	32		;RECORDS PER BLOCK (EACH BLOCK IS 4K BYTES)
	DB	2		;RECORDS PER SECTOR
	DW	64		;SECTORS PER TRACK
	DB	4		;HEADS PER UNIT
	DB	1		;BLOCKS PER DIRECTORY
	DW	64		;DIRECTORIES PER UNIT
	DW	(320-HDFSTRK)*4*4 ;BLOCKS = ACTIVE TRACKS * BLOCKS PER TRACK * HEADS
	DW	AVHD		;ALLOCATION VECTOR ADDRESS
	DW	1		;FIRST TRACK OF LOGICAL BLOCKS

	ENDIF

;
;	UNIT 1 PARAMETERS (UNIT 0 IF NOT KONAN SYSTEM)
;

	DB	8		;RECORDS PER BLOCK
	DB	1		;RECORDS PER SECTOR
	DW	26		;SECTORS PER TRACK
	DB	1		;HEADS PER UNIT
	DB	2		;BLOCKS PER DIRECTORY
	DW	1		;DIRECTORIES PER UNIT
	DW	243		;BLOCKS PER UNIT
	DW	AVFP0		;ALLOCATION VECTOR ADDRESS
	DW	2		;FIRST TRACK OF LOGICAL BLOCKS

;
;	UNIT 2 PARAMETERS
;
	DB	8		;RECORDS PER BLOCK
	DB	1		;RECORDS PER SECTOR
	DW	26		;SECTORS PER TRACK
	DB	1		;HEADS PER UNIT
	DB	2		;BLOCKS PER DIRECTORY
	DW	1		;DIRECTORIES PER UNIT
	DW	243		;BLOCKS PER UNIT
	DW	AVFP1		;ALLOCATION VECTOR ADDRESS
	DW	2		;FIRST TRACK OF LOGICAL BLOCKS

;
;	UNIT 3 PARAMETERS
;

PSTART	EQU	$		;START OF PARAM TABLE
	DB	8		;RECORDS PER BLOCK
	DB	1		;RECORDS PER SECTORS
	DW	26		;SECTORS PER TRACK
	DB	1		;HEADS PER UNITS
	DB	2		;BLOCKS PER DIRECTORY
	DW	1		;DIRECTORIES PER UNIT
	DW	243		;BLOCKS PER UNIT
	DW	AVFP2		;ALLOCATION VECTOR ADDRESS
	DW	2		;FIRST TRACK OF LOGICAL BLOCKS

PRMTABL	EQU	$-PSTART	;PARAMETER TABLE ENTRY LENGTH


;
;	ALLOCATION VECTORS
;

	IF KONAN
AVHD:	DS	(((320-HDFSTRK)*4*4)+7)/8 ;HARD DISK ALLOCATION VECTOR
	ENDIF

AVFP0:	DS	32		;FLOPPY UNIT 0 ALLOCATION VECTOR
AVFP1:	DS	32		;FLOPPY UNIT 1 ALLOCATION VECTOR
AVFP2:	DS	32		;FLOPPY UNIT 2 ALLOCATION VECTOR

;
;	FPREAD - READ FROM MASS STORAGE FLOPPY UNIT 
;
;	A 1K BLOCK IS READ FROM THE CURRENT UNIT STARTING AT THE CURRENT
;	HEAD, TRACK, AND SECTOR.  THE BLOCK IS READ INTO A BUFFER SPECIFIED
;	BY THE CURRENT DMA ADDRESS.
;

FPREAD:	LDA	UNIT
	MOV	C,A
	CALL	BIOS$SELUNIT
	MVI	A,0FFH
	STA	READFLAG
	JMP	FPDRIVER

;
;	FPWRITE - WRITE TO MASS STORAGE FLOPPY UNIT
;
;	A 1K BLOCK IS WRITTEN ON THE CURRENT UNIT BEGINNING AT THE CURRENT
;	HEAD, TRACK, AND SECTOR.  THE BLOCK IS WRITTEN FROM THE BUFFER
;	SPECIFIED BY THE CURRENT DMA ADDRESS.
;

FPWRITE:
	LDA	UNIT
	MOV	C,A
	CALL	BIOS$SELUNIT
	XRA	A
	STA	READFLAG
	JMP	FPDRIVER

;
;	FPDRIVER - FLOPPY DISK DRIVER
;
;	FPDRIVER READS/WRITES A 1K BLOCK FROM/TO A FLOPPY DISK UNIT.
;	IT TRANSLATES THE LOGICAL TRACK AND SECTOR INTO A PHYSICAL
;	TRACK AND SECTOR, AND READS/WRITES 8 SECTORS, BUMPING THE
;	SECTOR NUMBER AND TRACK NUMBER, IF NECESSARY, AFTER EACH
;	OPERATION.
;
;	ENTRY	(READFLAG) - READ/WRITE FLAG
;
;	EXIT	(A) - STATUS REPLY OF FLOPPY CONTROLLER
;

FPDRIVER:MVI	C,8		;INITIALIZE NUMBER OF SECTORS TO TRANSFER

FPDRV1:	LHLD	DMA		;SET DMA ADDRESS
	PUSH	B
	MOV	B,H
	MOV	C,L
	CALL	BIOS$SETDMA

	LDA	TRACK		;SET TRACK
	ADI	2		;ADD OFFSET (LOGICAL TRACK 0 IS PHYSICAL TRACK 2)
	MOV	C,A
	CALL	BIOS$SETTRK

	LHLD	SECTOR		;LOGICAL SECTOR IS INDEX INTO TRANSLATION TABLE
	LXI	D,FPTRANS
	DAD	D
	MOV	C,M
	CALL	BIOS$SETSEC

	LDA	READFLAG
	ORA	A
	JZ	FPDRV2		;IF WRITING
	CALL	BIOS$READ
	JMP	FPDRV3

FPDRV2:	CALL	BIOS$WRITE

FPDRV3:	POP	B
	ORA	A
	RNZ			;IF ERROR WHILE READING OR WRITING
	DCR	C
	RZ			;IF 1K TRANSFERED
	LHLD	DMA		;INCREMENT DMA ADDRESS
	LXI	D,128
	DAD	D
	SHLD	DMA
	LXI	H,SECTOR	;INCREMENT LOGICAL SECTOR
	INR	M
	MVI	A,25
	CMP	M
	JNC	FPDRV1		;IF NOT END OF TRACK
	MVI	M,0
	LXI	H,TRACK		;INCREMENT TRACK
	INR	M
	JMP	FPDRV1

;
;	FPTRANS - FLOPPY DISK SECTOR TRANSLATION TABLE
;

FPTRANS:DB	01H,07H,0DH,13H,19H,05H,0BH,11H
	DB	17H,03H,09H,0FH,15H,02H,08H,0EH
	DB	14H,1AH,06H,0CH,12H,18H,04H,0AH
	DB	10H,16H

;
;	CURRENT UNIT INFORMATION
;

UNIT:	DB	0
HEAD:	DB	0
TRACK:	DW	0
SECTOR:	DW	0
DMA:	DW	0

;
;	READ/WRITE FLAG
;

READFLAG:DB	0

;
;	BUFFER AREA
;

BUFFER:	DS	0

	DB	0
	DS	1029

	DB	0
	DS	1029

	DB	0
	DS	1029

	DB	0
	DS	1029

	DB	0FFH		;END OF BUFFERS

;*********************************************************************
;*
;*	ADAY ROUTINES FOR SATELLITE COMMUNICATIONS
;*
;*********************************************************************


	IF ADAYBOOT

;
;	BOOT$CONSTAT
;
;	EXIT	(A) - 0FFH CHARACTER IS ALWAS READY
;
BOOT$CONSTAT:
	MVI	A,0FFH		;A CHARACTER IS ALWAS READY
	RET

;
;	BOOT$CONOUT
;
;	NO CONSOLE TO GO TO SO JUST RET
;
BOOT$CONOUT:
	RET
;
;	BOOT$CONIN
;
;	EXIT	(A) - NEXT CHARACTER IN BOOTMSG
;		(MSGPTR) - UPDATED IN CIRCULAR FASHION
;
BOOT$CONIN:
	LXI	H,MSGPTR
	MOV	A,M
	INX	H
	SHLD	MSGPTR		;NEXT CHARACTER
	CPI	0FFH		;AT END OF STRING
	RNZ			;IF MORE CHARACTERS TO GIVE
	LXI	H,BOOTMSG	;START AT BEGINNING OF MESSAGE AGAIN
	JMP	BOOT$CONIN	;REPEAT

MSGPTR:	DW	BOOTMSG		;START OF MESSAGE
BOOTMSG: DW	'@LOGIN',CR,LF,0FFH	;FAKE THE USER CONIN

	ENDIF


;
;	SAT$CONSTAT - SATELLITE CONSOLE STATUS
;			USED DURING SUPERVISOR MODE FOR ANOS COMMUNICATION
;			TO SATELLITE ACTING AS CONSOLE DEVICE
;
;	EXIT	(A) -   0FFH IF CHARACTER IS READY FROM SATELLITE
;			0H IF CHARACTER IS NOT READY
;

SAT$CONSTAT:
	CALL	SATSTAT		;IS REQUEST PENDING
	RET

;
;	SAT$CONIN
;			USED DURING SUPERVISOR MODE FOR ANOS COMMUNICATION
;			TO SATELLITE ACTING AS CONSOLE DEVICE
;
;	EXIT	(A) - CHARACTER FROM SATELLITE
;

SAT$CONIN:
	CALL	SAT$CONSTAT
	ORA	A
	JZ	SAT$CONIN		;IF CHARACTER NOT READY
	LXI	B,SATINBUFF		;INPUT BUFFER
	CALL	RECEIVE			;GET BUFFER
	ORA	A
	JNZ	SAT$CONIN		;LOOP IF ERROR IN RECIEVING
	LDA	CHARACTER		;ELSE GET CHARACTER
	RET

;
;	SAT$CONOUT
;			USED DURING SUPERVISOR MODE FOR ANOS COMMUNICATION
;			TO SATELLITE ACTING AS CONSOLE DEVICE
;
;	ENTRY - (C) - HOLDS CHARACTER
;

SAT$CONOUT:
	MVI	A,1		;LENGTH
	STA	LENGTH
	MOV	A,C
	STA	CHARACTER
	LXI	B,SATOUTBUFF
	CALL	SEND		;SEND TO SATELLITE
	RET

;
;	BUFFERS FOR SUPERVISOR MODE HUB SATELLITE COMMUNICATION
;

SATOUTBUFF:
SATINBUFF:
LENGTH:	DS	1		;1 BYTE AT BEGINING FOR LENGTH
CHARACTER:
	DS	1		;WHERE INPUT AND OUTPUT CHARACTER ARE TO GO

;
;	SEND - SEND MESSAGE TO SATELLITE
;
;	SEND SENDS AND VERIFIES RECEIPT OF A MESSAGE.
;
;	ENTRY	(BC) - ADDRESS OF MESSAGE
;
;	EXIT	(A) - 0 IF MESSAGE SENT SUCCESSFULLY
;		      0FFH OTHERWISE
;
;	CALLS	SNDSYN, BYTE$TO$SAT, BYTE$FROM$SAT
;

SEND:	MOV	H,B
	MOV	L,C
	MVI	A,10		;INITIALIZE RETRY COUNT
	STA	RETRY

SEND0:	PUSH	H		;SAVE ADDRESS OF MESSAGE (IN CASE RETRY NECESSARY)
	CALL	SNDSYN		;SYNCHRONIZE
	MOV	B,M		;FETCH MESSAGE LENGTH
	MOV	D,B		;PRESET CHECKSUM
	INX	H
	MOV	C,B		;SEND LENGTH BYTE
	CALL	BYTE$TO$SAT
	JC	SEND2		;IF TRANSMISSION ERROR

SEND1:	MOV	A,M		;FETCH NEXT BYTE
	MOV	C,A
	ADD	D		;UPDATE CHECKSUM
	MOV	D,A
	CALL	BYTE$TO$SAT	;SEND DATA BYTE
	JC	SEND2		;IF TRANSMISSION ERROR
	INX	H
	DCR	B
	JNZ	SEND1		;IF MESSAGE NOT COMPLETELY SENT

;
;	SEND CHECKSUM
;

	MOV	A,D		;NEGATE CUMULATIVE SUM
	CMA
	INR	A
	MOV	C,A
	CALL	BYTE$TO$SAT
	JC	SEND2		;IF TRANSMISSION ERROR

;
;	RECEIVE ACKNOWLEDGEMENT
;

	CALL	BYTE$FROM$SAT	;RECEIVE ACKNOWLEDGE CHARACTER
	JC	SEND2		;IF TRANSMISSION ERROR
	POP	H
	CPI	ACK
	JNZ	SEND3		;IF NOT POSITIVE ACKNOWLEDGE
	XRA	A
	RET

;
;	PROCESS IRRECOVERABLE TRANSMISSION ERROR
;

SEND2:	POP	H
	MVI	A,0FFH
	RET

;
;	PROCESS RECOVERABLE TRANSMISSION ERROR
;

SEND3:	LDA	RETRY
	DCR	A
	STA	RETRY
	JNZ	SEND0		;IF RETRY NOT EXHAUSTED
	MVI	A,0FFH
	RET

RETRY:	DB	0		;TRANSMISSION ERROR RETRY COUNT
				;USED BY SEND AND RECIEVE

;
;	RECEIVE - RECEIVE MESSAGE
;
;	RECEIVE RECEIVES A MESSAGE FROM THE SAT AND INSURES ITS INTEGRITY.
;
;	ENTRY	(BC) - ADDRESS OF MESSAGE BUFFER
;
;	EXIT	(A) - 0 IF MESSAGE RECEIVED SUCCESSFULLY
;		      0FFH OTHERWISE
;
;	CALLS	RCVSYN, BYTE$FROM$SAT, BYTE$TO$SAT
;

RECEIVE:MOV	H,B
	MOV	L,C
	MVI	A,10		;INITIALIZE RETRY COUNT
	STA	RETRY		;RLH 12-8-80

RECEIV1:PUSH	H		;SAVE BUFFER ADDRESS (IN CASE RETRY NECESSARY)
	CALL	RCVSYN		;SYNCHRONIZE
	JC	RECEIV4		;IF SYNCHRONIZATION ERROR
	CALL	BYTE$FROM$SAT	;RECEIVE LENGTH BYTE
	JC	RECEIV4		;IF TRANSMISSION ERROR
	MOV	M,A
	MOV	B,A
	MOV	D,A		;INITIALIZE CHECKSUM
	INX	H

;
;	RECEIVE MESSAGE BODY
;

RECEIV2:CALL	BYTE$FROM$SAT	;RECEIVE NEXT DATA BYTE
	JC	RECEIV4		;IF TRANSMISSION ERROR
	MOV	M,A
	INX	H
	ADD	D		;UPDATE CHECKSUM
	MOV	D,A
	DCR	B
	JNZ	RECEIV2		;IF MESSAGE NOT COMPLETELY RECEIVED

;
;	RECEIVE CHECKSUM
;

	CALL	BYTE$FROM$SAT
	JC	RECEIV4		;IF TRANSMISSION ERROR
	ADD	D
	JNZ	RECEIV3		;IF CHECKSUM DOES NOT JIVE
	MVI	C,ACK		;ACKNOWLEDGE SUCCESSFUL RECEIPT
	CALL	BYTE$TO$SAT
	POP	H
	XRA	A		;CLEAR A FOR RETURN
	RET

;
;	NEGATIVELY ACKNOWLEDGE RECEIPT OF MESSAGE
;

RECEIV3:
	MVI	C,NAK
	CALL	BYTE$TO$SAT
	POP	H
	LDA	RETRY		;DECREMENT RETRY COUNT
	DCR	A
	STA	RETRY		;RLH 12-8-80

	JNZ	RECEIV1		;IF RETRY NOT EXHAUSTED
	MVI	A,0FFH
	RET

;
;	PROCESS IRRECOVERABLE ERROR
;

RECEIV4:POP	H
	MVI	A,0FFH
	RET


;
;	SNDSYN - SEND SYNCHRONIZATION CHARACTER TO  SATELLITE
;

SNDSYN:	MVI	C,SYN
	CALL	BYTE$TO$SAT
	JC	SNDSYN		;IF TRANMISSION ERROR

SNDSYN1:CALL	BYTE$FROM$SAT	;WAIT FOR SATELLITE ACKNOWLEDGEMENT
	JC	SNDSYN1		;IF NO ACKNOWLEDGEMENT BEFORE TIME OUT
	CPI	SYN
	JNZ	SNDSYN		;IF NOT SYNCHRONIZED
	RET

;
;	RCVSYN - RECEIVE SYNCHRONIZATION CHARACTER FROM SATELLITE
;
;	EXIT	(CARRY) - SET IF SYNCHRONIZATION ERROR
;

RCVSYN:	CALL	BYTE$FROM$SAT
	JC	RCVSYN		;IF TIME OUT
	CPI	SYN
	JNZ	RCVSYN1		;IF NOT SYNCHRONIZATION CHARACTER
	MVI	C,SYN		;ACKNOWLEDGE SYNCHRONIZATION
	CALL	BYTE$TO$SAT
	RET

RCVSYN1:
	STC
	RET

;
;	INTSAT - INTERROGATE SATELLITE SELECTION
;
;	EXIT	(A) - SATELLITE IDENTIFICATION NUMBER
;

INTSAT:
	MVI	A,DUMMY		;SET UP IN SELSAT
	RET


;***********************************************************::
;
;	ADAY SATELLITE INTERFACE ROUTINES
;
;************************************************************

SELSAT:

;		SELECT SATELLITE ROUTINE
;		SET UP FOR 2 SATELLITES (#O AND #1)
;
;	ENTRY	(C) - SATELLITE TO SELECT
;
;	EXIT	- PORTS IN ROUTINES ISP, IDP, ODP, OCP, OIP ARE MODIFIED
;		  AS IS AN INSTRUCTION IN INTSAT
;
;		THIS SYSTEM USES A TUART CARD FOR THE COMMUNICATIONS
;		BETWEEN HUB AND SATELLITE
;
;		PRTAB HOLD THE PORTS USED TO MODIFY THE I/O ROUTINES
;
;		** THIS CODE IS SELF MODIFYING PLEASE EXAMINE CLOSELY**
;
;  Mod by JLZ, 11-Oct-80:
;	For use with new hub monitor supporting more satellites.
;	If selected satellite doesn't exist (ie. satellite number
;	is >= NSATS), then SATSTAT will always return FALSE.


	MOV	A,C
	STA	INTSAT+1	;SETS UP INTERROGATE SATELLITE SELECTED ROUTINE

	CPI	NSATS
	JNC	SELS1		;IF UNIMPLEMENTED SATELLITE
	MVI	A,(NOP)
	STA	STSTIN+1	;Set STSTIN to fall through into ISP
	JMP	SELS2
;
SELS1:	MVI	A,(RET)		;Set STSTIN to always return 0
	STA	STSTIN+1

SELS2:	MOV	L,C		;HL GETS INDEX INTO PORT TABLE
	MVI	H,0
	DAD	H		;*4 BECAUSE EACH ENTRY IN PORT TABLE IS 4 LONG

	DAD	H		;RLH FIX OF 11-1-80

	LXI	D,PRTAB		;PORT TABLE
	DAD	D
	MOV	A,M		;GET STATUS PORT
	STA	OSP+1		;OUTPUT TO SATELLITE STATUS PORT (BAUD RATE FOR COLD START)
	STA	ISP+1		;CHANGE PORT IN INPUT STATUS PORT
	INX	H		;GET DATA PORT
	MOV	A,M
	STA	IDP+1		;INPUT DATA PORT
	STA	ODP+1		;OUTPUT DATA PORT

				;THE FOLLOWING USED IN COLD START ROUTINE
	INX	H
	MOV	A,M
	STA	OCP+1		;OUTPUT COMMAND PORT
	INX	H
	MOV	A,M
	STA	OIP+1		;OUTPUT TO INTERRUPT ENABLE PORT

	RET
;
;	TABLE PORTS FIXED BY RLH 11-1-80
;

PRTAB:	DB	0C0H,0C1H,0C2H,0C3H	;PORTS FOR SATELLITE 0
	DB	00H,001H,02H,03H	;FOR SAT 1
	DB	0E0H,0E1H,0E2H,0E3H	;SAT 2

;
;	ISP, IDP, ODP, OSP, OCP, AND OIP HAVE THE CORRECT PORT SET UP IN SELSAT
;	THE ASSEMBLE USES A DUMMY
;

DUMMY	EQU	0H

STSTIN:	XRA	A	;Used by SATSTAT (JLZ)
	RET

ISP:	IN	DUMMY	;INPUT FROM SATELLITE STATUS PORT
	RET

IDP:	IN	DUMMY	;INPUT FROM SATELLITE DATA PORT
	RET

ODP:	OUT	DUMMY	;OUTPUT TO SATELLITE DATA PORT
	RET

OSP:	OUT	DUMMY	;OUTPUT TO SATELLITE STATUS PORT (USED IN COLD START)
	RET

OCP:	OUT	DUMMY	;OUTPUT TO COMMAND PORT (USED IN COLD START)
	RET

OIP:	OUT	DUMMY	;OUTPUT TO INTERRUPT ENABLE PORT (USED IN COLD START)
	RET

;
;	BYTE$FROM$SAT-	BYTE FROM SATELLITE
;
;	EXIT-	(A) -	BYTE FROM SATELLITE IF ONE RECIEVED
;			ELSE CARRY BIT SET IF NO BYTE RECIEVED
;
;
;
;		HARDWARE SPECIFIC CODE EXAMINE CLOSELY
;
;
BYTE$FROM$SAT:
				;INPUT FROM SATELLITE
	MVI	E,0		;RETRY COUNT
BFS1:
	CALL	ISP		;INPUT FROM SATELLITE
	ANI	CKBR		;CHECK BUFFER READY
	JZ	BFS2		;IF NOT READY
	CALL	IDP		;INPUT FROM DATA PORT
	RET

BFS2:
	DCR	E		;COUNT
	JNZ	BFS1		;IF TIME NOT UP
	STC			;SET FAILURE FLAG
	RET
;
;
;	BYT$TO$SAT-	BYTE TO CURRENTLY SELECTED SATELLITE
;
;	ENTRY-		(C)- BYTE TO TRANSMIT TO SATELLITE
;
;	EXIT-		(CARRY)- SET IF TIMED OUT
;
;	USES-		A,E (AND NO OTHERS PLEASE)
;
;
;	HARDWARE SPECIFIC EXAMINE CLOSELY
;
;
BYTE$TO$SAT:
	CALL	ISP
	ANI	CPTR
	JZ	BYTE$TO$SAT
	MOV	A,C
	CALL	ODP
	RET
;
;
;
;	SATSTAT-	SATELLITE STATUS INTEROGATION
;
;	EXIT-		A- 0 IF SATELLITE IS NOT READY
;			   0FFH IF SATELLITE IS READY
;
;
SATSTAT:CALL	STSTIN	;INPUT SATELLITE STATUS PORT
	ANI	CKBR	;CHECK BIT READY
	RZ		;IF NOT READY RETURN ZERO
	MVI	A,0FFH	;A GETS FF IT IS READY TO RECIVE
	RET

;
;	RETCLI - RETURN RESULT TO CLI
;

RETCLI:
	RET

;************************************************
;
;	START OF USER0 CODE (PRINT SPOOLER)
;
;************************************************

;
;	USER0 - USER BACKGROUND ROUTINE (PRINT SPOOLER)
;
;	THIS ROUTINE SEARCHES FOR A FILE WITH TYPE .LPT
;	IF IT FINDS ONE IT DISPOSES IT TO THE PRINTER
;	AT A RATE OF 32 CHARACTERS EACH TIME IT IS CALLED
;

USER0:
	MVI	C,SETUI		;SET USER INDEX
	LXI	D,1
	CALL	5		;SETS USER INDEX TO SPOOLER DIRECTORY

	LDA	ACTFLG		;IS FILE ACTIVE
	ORA	A
	JNZ	US01

	LHLD	SPDELAY		;DELAY USED IF NO FILE WAS FOUND
	DCX	H
	SHLD	SPDELAY
	MOV	A,H
	ORA	L
	RNZ			;RETURN IF DELAY NOT EXPIRED

	CALL	TRYSRC		;TRY TO FIND A .LPT FILE
	ORA	A
	JNZ	US04		;IF NO FILE FOUND

	MVI	A,128		;FORCE A READ IN  GETNEXT
	STA	INDEX

	MVI	A,0FFH		;FLAG SPOOLER AS ACTIVE
	STA	ACTFLG

US01:	CALL	TOPRN		;DUMP 32 CHARACTERS TO PRINTER
	ORA	A
	RZ

	MVI	C,DELETE	;DELETE FILE FUNCTION
	LXI	D,SPLFCB
	CALL	ANOS

US03:
	LXI	H,1		;SET DELAY TO SEARCH FOR NEXT FILE IMMEDIATELY
	SHLD	SPDELAY

	MVI	A,0
	STA	ACTFLG		;INACTIVATE SPOOLER
	RET

US04:
	LXI	H,MXDELAY	;MAXIMUM DELAY
	SHLD	SPDELAY		;DELAY BEFORE NEXT SEARCH
	RET			;SPOOLER STILL INACTIVE

;
;	TOPRN - SHIP 32 CHARACTERS OUT TO THE PRINTER
;
;	EXIT -	A SET TO 0FFH IF FILE IS FINISHED
;		ELSE 0
;
;	CALLS -	LIST DEVICE ROUTINE
;		GETNEXT - GETS BYTE FROM FILE
;

TOPRN:
	MVI	B,32		;CHARACTERS TO GO OUT

TP1:	PUSH	B
	CALL	GETNEXT		;GET NEXT CHARACTER IN A
	POP	B		;GET COUNT BACK
	JC	TP3		;IF DONE

	CPI	LF		;IS IT A LINE FEED
	JNZ	TP2		;IF NOT A LINE FEED
	LXI	H,LNCNT		;LINE COUNT
	DCR	M
	JNZ	TP2		;IF ALL NOT AT END OF PAGE
	MVI	M,MAXCNT	;SET TO TOP OF PAGE

TP2:
	MOV	C,A
	PUSH	B		;SAVE COUNT IN B
	CALL	BIOS$LIST
	POP	B
	DCR	B
	JNZ	TP1		;IF NOT DONE
	XRA	A
	RET


;	NO WAY TO FORCE AN ERROR EXIT SO JUST RETURN

TP3:
	LDA	LNCNT
	CPI	MAXCNT
	JZ	TP5		;IF AT TOP OF FORM ALREADY

TP4:
	PUSH	PSW		;SAVE COUNT TO TOP OF FORM
	CALL	CRLF
	POP	PSW
	DCR	A
	JNZ	TP4		;IF NOT AT TOP OF FORM
	MVI	A,MAXCNT
	STA	LNCNT		;RESET TO TOP OF FORM

TP5:	MVI	A,0FFH
	RET

;
;	CRLF	SEND A CHARAIGE RETURN AND A LINE FEED TO THE PRINTER
;

CRLF:	MVI	C,CR
	CALL	BIOS$LIST	;OUTPUT A LINE FEED
	MVI	C,LF
	CALL	BIOS$LIST	;OUTPUT A RETURN
	RET

;
;	GETNEXT - GET NEXT BYTE FROM .LPT FILE
;
;	ENTRY -	INDEX - INDEX INTO SPLBUF
;	SPBUFF - BUFFER HOLDS CHARACTER TO GO TO PRINTER
;		SPLFCB - FCB OF OPEND FILE TO READ FROM
;
;	EXIT -	CARRY - SET IF DONE (END OF FILE)
;			CLEARED OTHERWISE
;		(A) -	CHARACTER
;

GETNEXT:
	LDA	INDEX
	ORA	A
	JP	GNT1		;IF BUFFER NOT EXHAUSTED


	MVI	C,SETDMAR	;SET DMA ADDRESS
	LXI	D,SPBUFF	;TO SPOOLER BUFFER
	CALL	ANOS

	MVI	C,READN		;READ
	LXI	D,SPLFCB
	CALL	ANOS
	ORA	A		;CHECK FOR END OF FILE
	JNZ	GTN2		;IF END OF FILE OR ERROR

GNT1:
	MOV	E,A
	MVI	D,0
	INR	A
	STA	INDEX		;UPDATE POINTER INTO BUFFER
	LXI	H,SPBUFF
	DAD	D		;HL POINT TO CHARACTER
	MOV	A,M
	CPI	1AH		;IS IT A CONTROL Z CHARACTER
	JZ	GTN2		;IF END OF ASCII FILE
	ORA	A		;CLEAR CARRY
	RET

GTN2:
	STC			;ERROR OR END OF FILE EXIT
	RET

;
;	TRYSRC - TRY SEARCHING FOR A FILE TO OUTPUT
;
;	EXIT -	(A) - 0 IF FILE FOUND
;		OTHERWISE 0FFH
;

TRYSRC:
	LXI	H,SPLTPL	;FCB TEMPLATE
	LXI	D,SPLFCB
	MVI	B,16		;SIZE OF FCB NAME

TR1:	MOV	A,M
	STAX	D
	INX	H
	INX	D
	DCR	B
	JNZ	TR1		;IF FCB NOT COMPLETELY COPIED

	MVI	C,SETDMAR
	LXI	D,SPBUFF	;SET DMA TO SPOOLER BUFFER
	CALL	ANOS

	MVI	C,SEARCH	;SEARCH FOR FILE
	LXI	D,SPLFCB
	CALL	ANOS

	INR	A		;WAS IF FF
	JZ	TR3		;IF NO FILE FOUND

	LXI	H,SPBUFF	;COPY IN FILE
				;ANOS PUTS FILENAME AT START OF DMA BUFFER
	LXI	D,SPLFCB
	MVI	B,32

TR2:	MOV	A,M
	STAX	D
	INX	H
	INX	D
	DCR	B
	JNZ	TR2


	XRA	A		;CLEAR A
	STA	SPLFCB+XT
	STA	SPLFCB+NR
	MVI	C,SETDMAR	;SET DMA ADDRESS TO TBUF
	LXI	D,SPBUFF
	CALL	ANOS

	MVI	A,1		;DISK A SELECT
	STA	SPLFCB+ET	;SET TO THE A DISK

	MVI	C,OPEN		;OPEN FILE
	LXI	D,SPLFCB
	CALL	ANOS

	INR	A
	JZ	TR3		;IF BAD OPEN FOR SOME REASON (GULP)

	XRA	A		;CLEAR A FOR RETURN
	RET

TR3:	MVI	A,0FFH		;FAILURE FLAG
	RET

;
;	DATA STRUCTURES FOR SPOOLING ROUTINES
;
;	ACTFLG-	ACTIVE FLAG TELLS IS FILE IS ACTIVE
;	SPLFCB-	FCB USED BY SPOOLER FOR OPENING AND READING FILES
;	SPLTPL-	SPOOLER TEMPLATE NAME TO PUT IN SPLFCB FOR SEARCH
;	INDEX-	INDEX INTO SPBUFF
;	SPBUFF-	BUFFER FOR DISK FILE
;

ACTFLG:	DB	0	;START OUT INACTIVE

SPLFCB:	DS	33	;32 BYTES FOR FCB

SPLTPL:	DB	1,'????????LPT',0,0,0,0		;FCB TEMPLATE

INDEX:	DB	127	;BUFFER EMPTY TO BEGIN WITH

SPBUFF:	DS	128	;ROOM FOR 1 SECTOR

MAXCNT	EQU	66	;LINES PER PAGE
LNCNT:	DB	MAXCNT	;START AT TOP OF PAGE

MXDELAY	EQU	1000	;MAXIMUM DELAY BEFORE SEARCHING FOR A .LPT FILE

SPDELAY:DW	MXDELAY	;START AT MAXIMUM DELAY

;*******************************************************
;
;	END OF USER0 (PRINT SPOOLER CODE)
;
;*******************************************************

;
;	USER BACKGROUD ROUTINES - ALL DUMMIES FOR NOW
;

USER1:	RET
USER2:	RET
USER3:	RET

;
;	BIOS ENTRY POINT DEFINITIONS
;

BIOS		EQU	($+0FFH) AND 0FF00H
BIOS$CONSTAT	EQU	BIOS+6
BIOS$CONIN	EQU	BIOS+9
BIOS$CONOUT	EQU	BIOS+12
BIOS$LIST	EQU	BIOS+15
BIOS$PUNCH	EQU	BIOS+18
BIOS$READER	EQU	BIOS+21
BIOS$HOME	EQU	BIOS+24
BIOS$SELUNIT	EQU	BIOS+27
BIOS$SETTRK	EQU	BIOS+30
BIOS$SETSEC	EQU	BIOS+33
BIOS$SETDMA	EQU	BIOS+36
BIOS$READ	EQU	BIOS+39
BIOS$WRITE	EQU	BIOS+42
         END
