; hc.xm: Horizontal Concatenation unicum
; /AJK 28.Aug.81, 28.Oct.82

;    _______
;   |      /
;   |     /
;   |    /    Copyright (c) 1981, 1982 by Knowlogy
;   |   //\                            PO Box 283
;   |  //  \                           Wilsonville, Oregon  97070
;   | //    \
;   |//______\

	uses LIB2800
	uses LIB2801

; Mod 28.Oct.82 /AJK: Literal strings are any quoted argument, not arguments
;		      which begin with apostrophe; apostrophe is no longer
;		      required.

	db	13,'HC V2: COPYRIGHT (C) 1981, 1982 BY KNOWLOGY',13,10,26,0

ILSize	equ	1024		; max input line size
OLSize	equ	1024		; max output line size
BufSiz	equ	2048		; buffer size (small because many files
				; can be open at once)

; The argument block consists of several 3-byte entries,
; of the following format:
ARGtyp	equ	0		; (byte) type of argument, as follows:
TYPtab	  equ	0		;   tab position (positive number)
TYPcol	  equ	1		;   start column (negative number)
TYPchn	  equ	2		;   channel number
TYPlit	  equ	3		;   literal string
TYPend	  equ	4		;   end of argument list
ARGval	equ	1		; (word) argument: a number or a channel
ARGLEN	equ	3		; number of bytes per argument
ARGNO	equ	64		; max number of arguments
				; this should be revised upward if SHL2801se
				; is ever modified to allow for more arguments
; Global register usage:
;   IX points to the argument of interest.
; Scalars:
;   LITFLG is on to signify "literal output pending".  This flag is used
;     to force at least one line of output when a literal string is present
;     in the argument list.  If there are no files in the list, or if all
;     files in the list are empty (not even one line can be read from any
;     of them), and there is a literal string argument, then one line
;     containing the literal string will appear.  This allows "hc" to be
;     used as a Unix-like "echo" program.  LITFLG is cleared at program
;     initialization, set when a literal string is encountered during
;     argument scan, cleared when a line is written, and tested when all
;     files encounter EOF to see if a line should be printed anyway.

	entry hc
hc:
	HEAhea [hl=0100h]	; initialize stack and heap
	USKall [hl=(ival)]->[(inlin)=hl] ; allocate input line buffer
	USKall [hl=(oval)]->[(outlin)=hl] ; allocate output line buffer
	USKall [hl=3*ARGNO]->[(args)=hl] ; allocate argument list
	USKini []		; scan command
	USKflg [hl=flgtbl]	; interpret flags
	USKdef [stk="-",stk=0]	; default is to read standard input
	xor	a		; clear the "literal output pending" flag
	ld	(litflg),a
	ld	ix,(args)	; IX -> arguments block
	USKgna []->[hl]+C-a	; get first argument into HL (can't fail)
hc1:				; here with first argument in HL
	ld	b,h		; save argument in HL
	ld	c,l
	ATOI [hl]->[de,hl]+C	; see if it's a numeral
	jr	c,hc4		; branch if it isn't
	ld	a,(hl)		; it may be, check the delimiter
	and	a		; must be null
	jr	nz,hc4		; branch if not null, it's not a numeral
	bit	7,d		; see if it's negative
	jr	nz,hc2		; branch if so, that's start column
	ld	(ix+ARGtyp),TYPtab ; otherwise mark a tab argument
	jr	hc3		; go stow the number
hc2:				; here on negative number
	ld	hl,0		; take number's absolute value
	sbc	hl,de		; (carry still clear from "and a")
	ex	de,hl		; absolute value in DE
	ld	(ix+ARGtyp),TYPcol ; mark a column argument
hc3:				; here to stow numeric argument
	dec	de		; transform so first column is number 0
	ld	(ix+ARGval),e	; stow the number
	ld	(ix+ARGval+1),d
	jr	hc6		; done with this argument, do another
hc4:				; here if argument isn't a valid number
	ld	h,b		; get argument into HL again
	ld	l,c

; Mod 28.Oct.82 /AJK
; The following five lines are affected:
	dec	hl		; HL -> character preceding argument
	ld	a,(hl)		; see if argument is a quoted string
	inc	hl		; restore argument pointer
	cp	'"'
	jr	nz,hc5		; branch if it's not
; End mod 28.Oct.82 /AJK

	ld	(ix+ARGval),l	; stow the pointer
	ld	(ix+ARGval+1),h
	ld	(ix+ARGtyp),TYPlit ; mark a literal string
	ld	a,1		; set the "literal output pending" flag
	ld	(litflg),a
	jr	hc6		; go on to next argument
hc5:				; here if it's not a literal string
	BBIopn [stk=hl,stk=RO+Text+OldOnly,stk=BufSiz]->[a]+C
	jp	c,hc31		; branch if can't open file, fatal error
	ld	(ix+ARGtyp),TYPchn ; mark a channel argument
	ld	(ix+ARGval),a	; stow the channel number
	ld	(ix+ARGval+1),0
hc6:				; here with argument safely stowed
	ld	de,ARGLEN	; advance the argument list pointer
	add	ix,de
	USKgna []->[hl]+C-a	; get the next argument
	jr	nc,hc1		; if we got one, go process it
	ld	(ix+ARGtyp),TYPend ; else mark end of argument list

; Here to start concatenating.
; Throughout concatenation, register usage is as follows:
;   BC = current column in the output line, where first is number 0
;   IX -> argument being processed
; Scalar usage:
;   STCOL is the start column of the input line of the next channel.
;     It corresponds to the number of columns to skip.
;     It is set by a "column" type argument (negative number),
;       and cleared when it is used (so the next channel starts in column 0).
;   EOFFLG is set to 1 at the beginning of one output line, and set to 0
;     when any input line is successfully read.  Thus, after all arguments
;     are processed, it is 1 if no line could be read (EOF on all input
;     channels) and 0 if at least one line was read.
;   LATFLG, the "last argument tab" flag, is set to 0 initially, 1 when a
;     tab argument is seen, 0 when any other argument is seen.  If the last
;     argument in the argument list is a tab, then trailing spaces are not
;     removed, otherwise they are.

; Top of loop for each line of output.
hc7:
	ld	ix,(args)	; IX -> first argument
	ld	bc,0		; BC is column number; we start at column 0.
	ld	(stcol),bc	; that's also the initial start column
	ld	hl,eofflg	; set EOF flag to 1.  If any channel is able
	ld	(hl),1		;   to read a line, this will be zeroed.
	xor	a		; clear LATFLG
	ld	(latflg),a

; Top of loop for each argument.
; Get next argument and dispatch to tab, column, or channel handler.
hc8:
	ld	l,(ix+ARGval)	; get argument value into HL
	ld	h,(ix+ARGval+1)
	ld	a,(ix+ARGtyp)	; A = type of next argument
	dec	a		; see if it's a start column
	jr	z,hc11		; branch if so
	dec	a		; see if it's a channel
	jr	z,hc12		; branch if so
	dec	a		; see if it's a literal string
	jp	z,hc16		; branch if so
	dec	a		; see if it's end of arg list pointer
	jp	z,hc19		; branch if so

; Here with a tab argument.
; If the tab position is less than or equal to the current position,
; just reset the current position; if the tab is greater than the
; current position, then add enough spaces to bring us there.
	ld	a,1		; set LATFLG: this argument is a tab
	ld	(latflg),a
	ld	d,h		; copy tab argument into DE
	ld	e,l
	scf			; see if new column is greater than current
	sbc	hl,bc		; clear carry if so
	jr	nc,hc9		; branch if we've advancing within the line
	ld	b,d		; else we're holding steady or backing up,
	ld	c,e		; make that number the current position
	jp	hc18		; (done with this argument)
hc9:				; here to advance
	inc	hl		; set HL to number of columns to advance
	ld	iy,(outlin)	; -> output line buffer
	add	iy,bc		; IY -> current position in buffer
	ld	b,d		; set BC to new column
	ld	c,e
	call	ChkCol		; check for line overflow
hc10:
	ld	(iy+0),' '	; space over in buffer
	inc	iy
	dec	hl		; decrement advance column count
	ld	a,h		; loop until count becomes zero
	or	l
	jr	nz,hc10
	jr	hc18		; (done with this argument)

; Here with a column argument.
; Just store it for the next file.
hc11:
	xor	a		; clear LATFLG
	ld	(latflg),a
	ld	(stcol),hl	; store column argument
	jr	hc18		; (done with this argument)

; Here with a channel argument; concatenate a line from a file.
; Register usage:
;   BC = current column in output line, where first is number 0 (as usual)
;   DE = current column in input line, where first is number 0
;   HL -> next character in input line
;   IX -> current argument
;   IY -> next position in output line

hc12:
	xor	a		; clear LATFLG
	ld	(latflg),a
	BBIgl [stk=hl,stk=(inlin),stk=(ival)]->[]+C-a ; read a line
	jr	c,hc15		; if EOF, do nothing
	xor	a		; clear EOF flag
	ld	(eofflg),a
	ld	hl,(inlin)	; HL -> line
	ld	iy,(outlin)	; -> start of output buffer
	add	iy,bc		; IY -> current position in output buffer
	ld	de,0		; DE is input column number
hc13:
	ld	a,(hl)		; A = next character from input line
	inc	hl
	and	a		; look for end-of-line
	jr	z,hc15		; branch at end-of-line
	cp	'I'-64		; see if this is a tab
	jr	z,hc14		; branch if it is, do special handling
	call	Chadd		; add the character to the output line
	inc	de		; advance by one input position
	jr	hc13		; loop for next character
hc14:				; here if character is a tab
	ld	a,' '		; add a space to the output line
	call	Chadd
	inc	de		; advance input column number by one
	ld	a,e		; it's a tab stop when the input column number
	and	7		;   is divisible by 8
	jr	nz,hc14		;   (this is why the first column is number 0)
	jr	hc13		; tab advance complete, loop for next character

; Here when done with a channel.
hc15:
	ld	hl,0		; make sure "start column" is zero
	ld	(stcol),hl	;   for next channel
	jr	hc18		; done with this argument

; Here with a literal string argument; concatenate it onto the output line.
; Register usage:
;   BC = current column in output line, where first is number 0 (as usual)
;   HL -> next character in literal string
;   IX -> current argument
;   IY -> next position in output line

hc16:
	xor	a		; clear LATFLG
	ld	(latflg),a
	ld	iy,(outlin)	; -> start of output buffer
	add	iy,bc		; IY -> current position in output buffer
hc17:
	ld	a,(hl)		; A = next character from literal string
	inc	hl
	and	a		; look for end-of-string
	jr	z,hc18		; branch at end-of-string
	call	Chadd		; add the character to the output line
	jr	hc17		; loop for next character

; Here when done with an argument.
; Advance and process the next argument.
hc18:
	ld	de,ARGLEN	; advance IX
	add	ix,de
	jp	hc8

; Here when done with arguments.
; If this is the first line, we ignore file EOF; otherwise
; see if any file did not encounter EOF.
hc19:
	ld	a,(litflg)	; A = "literal output pending" flag
	and	a		; see if we have to do output
	jr	nz,hc20		; branch if so, force output
	ld	a,(eofflg)	; A = EOF flag
	and	a		; see if it's still set
	jp	nz,hc30		; if so, we're all done

; Null-terminate the output line.
hc20:
	ld	hl,(outlin)
	add	hl,bc
	ld	(hl),0

; If the last argument wasn't a tab, then remove trailing spaces.
	ld	a,(latflg)
	and	a
	jr	nz,hc22		; branch if last argument was tab
hc21:				; top of loop to remove a trailing space
	ld	a,b		; see if we've eaten the whole line
	or	c
	jr	z,hc22		; branch if so
	dec	hl		; else back up one character
	ld	a,(hl)		; see if it's a trailing space
	cp	' '
	jr	nz,hc22		; branch if not
	ld	(hl),0		; it is, remove it
	dec	bc		; decrement count of chars in line
	jr	hc21		; loop for all trailing spaces
hc22:

; If -s wasn't specified, do tab compression in the output line.
	ld	a,(sflg)	; did user specify -s?
	and	a
	jr	nz,hc29		; if so, skip tab compression

; Perform tab compression in place.
; Register usage:
;   BC = column number of character being examined.
;   DE = column number of character being placed.
;   IX -> character being examined.
;   IY -> where to put the next character.
; When a space is seen, we just advance BC;
; when a non-space is seen, we add tabs and spaces while advancing DE
;   until DE is equal to BC, then we place the character.
	ld	ix,(outlin)	; IX -> where to get next character
	ld	iy,(outlin)	; IY -> where to put next character
	ld	bc,0		; BC = current "from" column
	ld	de,0		; DE = current "to" column
hc23:
	ld	a,(ix+0)	; examine next character
	cp	' '		; is it a space?
	jr	nz,hc24		; branch if not
	inc	bc		; it's a space, just bump past it
	inc	ix
	jr	hc23		; go do another character

; A non-space character (or end-of-line) has been encountered.
; Add enough tabs and spaces to bring the "to" column up to the "from" column.
hc24:
	ld	h,b		; see if "to" column equals "from" column
	ld	l,c
	and	a
	sbc	hl,de		; they're equal if this sets Z
	jr	z,hc28		; branch if columns are equal
	dec	hl		; see if they differ by just 1
	ld	a,h
	or	l
	jr	z,hc26		; if so, don't fill one space with a tab
	push	de		; save old "to" column
	ld	hl,8		; see if adding a tab would take us too far
	add	hl,de
	ld	a,l
	and	NOT 7
	ld	l,a		; HL = new "to" column after a tab
	ld	d,h		; copy new column number into DE
	ld	e,l
	scf			; see if that takes us too far
	sbc	hl,bc		; it does if new "to" column > "from" column
	jr	nc,hc25		; branch if it does
	ld	(iy+0),'I'-64	; add a tab
	inc	iy
	pop	hl		; discard old "to" column, DE = new "to"
	jr	hc24		; loop to continue advance
hc25:				; here if a tab would take us too far
	pop	de		; restore old "to" column
hc26:				; here to space to new position
	ld	h,b		; compute number of spaces to add
	ld	l,c
	and	a
	sbc	hl,de		; HL = number of spaces to add
hc27:				; add a space
	ld	(iy+0),' '
	inc	iy
	dec	hl		; drop spaces count
	ld	a,h		; loop until there are zero spaces to add
	or	l
	jr	nz,hc27
	ld	d,b		; set "to" column to "from" column
	ld	e,c
hc28:				; here when we've spaced forward sufficiently
	ld	a,(ix+0)	; get non-space character again
	inc	ix
	inc	bc
	ld	(iy+0),a	; add it to output
	inc	iy
	inc	de
	and	a		; was that the null terminator?
	jr	nz,hc23		; if not, loop for next character

; Here after any tab compression.
; Write out the line.
hc29:
	LIOpl [stk=1,stk=(outlin)]
	xor	a		; clear "literal output pending" flag
	ld	(litflg),a
	jp	hc7		; go do another line

; Here when all files are exhausted.
hc30:
	SHLexi [a=0]

; Here on fatal I/O error.
hc31:
	ERRMSG [a=a,b=1,c=1,hl=hl]

; Chadd: routine to provisionally add a character to the output line,
; depending on whether we've come to the input file start column or not.
; Enter with
;   A  = character to put
;   BC = current column in output line
;   IY -> current position in output line
Chadd:
	ld	(iy+0),a	; stow the character
	push	hl		; save HL
	ld	hl,(stcol)	; get the chars left to start column
	ld	a,h		; see if it's zero
	or	l
	jr	z,ch1		; if it is, we don't skip this character
	dec	hl		; else decrement chars left count
	ld	(stcol),hl	; by not incrementing IY, we effectively
	jr	ch2		;   don't add character to output line
ch1:
	inc	iy		; OK to add character, bump output line pointer
	inc	bc		; bump output line position
	call	ChkCol		; check for output buffer overflow
ch2:
	pop	hl		; restore HL
	ret

; ChkCol: routine to check the output line column to make sure buffer
; overflow hasn't occurred.  If it has, an error message is printed and
; the program aborts, otherwise ChkCol returns quietly.
; Enter with BC = current column in output line.
ChkCol:
	push	hl		; save HL
	ld	hl,(oval)	; HL = max size of buffer
	scf			; (set carry for subtract)
	sbc	hl,bc		; clear carry if BC isn't too big
	pop	hl		; restore HL
	ret	nc		; return if OK
	EPUTF [stk="Output line buffer overflow: %u bytes isn't enough^m^j",stk=(oval)]
	SHLexi [a=IoNMM]

; Flag table.
flgtbl:
	db 'i',1
	dw 0,2
	dw 0,0FFFFh
	db 0
	dw 0
ival:	dw ILSize
	db 'o',1
	dw 0,2
	dw 0,0FFFFh
	db 0
	dw 0
oval:	dw OLSize
	db 's',0,0,0,0,0,0,0,0,0,0,0,0
sflg:	dw 0
	db 0

litflg:	ds	1		; literal output pending flag
eofflg:	ds	1		; end-of-file flag
latflg:	ds	1		; last-argument-tab flag
stcol:	ds	2		; start column for next channel
inlin:	ds	2		; -> input line buffer
outlin:	ds	2		; -> output line buffer
args:	ds	2		; -> argument list

	end hc

