(* This is the file archproc.pas, a file of archive processes. *)


{PROCEDURE NOTFOUND}
{ notfound -- print "not found" warning }
procedure notfound;
var
    i : integer;
begin
    for i := 1 to nfiles do
	if (fstat[i] = false) then begin
	    putstr(fname[i], STDERR);
	    message(': not in archive');
	    errcount := errcount + 1;
	end
end {NOTFOUND};

{HELP}
{ help -- print diagnostic for archive }
procedure help;
begin
    error('usage: archive -[cdptux] archname [files...]')
end {HELP};

{FCOPY}
{ fcopy -- copy file fin to fout }
procedure fcopy (fin, fout : filedesc);
var
    c : character;
begin
    while (getcf(c, fin) <> ENDFILE) do
	putcf(c, fout)
end {FCOPY};

{FMOVE}
{ fmove -- move file name1 to name2 }
procedure fmove (var name1, name2 : string);
var
    fd1, fd2 : filedesc;

(* Note: the procedure "close" is a standard procedure in M and MT+, so I
  created the procedure tool_close to do the closing. *)

begin
    fd1 := mustopen(name1, IOREAD);
    fd2 := mustopen(name2, IOWRITE);
    fcopy(fd1, fd2);
    tool_close(fd1);
    tool_close(fd2);
end {FMOVE};

	{SCOPY}
{ scopy -- copy string at src[i] to dest[j] }
procedure scopy (var src : string; i : integer;
	var dest : string; j : integer);

(*Rather than using a modification of the tstring handling procedures presented
 in K&p, we use CONCAT, a standard Pascal/M and MT+ string concatenation
 function.  Note: in order to preserve the generality of the original routine,
 which could copy into the middle of a string, the delete bit must appear. *)

begin {scopy};
  if (length(src) > 0) then
  begin
    if (length(dest) >= j) then (* remove stuff off end so string will be ok *)
      delete(dest, j, (length(dest) - (j - 1)));
    dest := concat(dest, src);
  end;
end {SCOPY};

	{FSIZE}
{ fsize -- size of file in characters }
function fsize (var name : string) : integer;
var
    c : character;
    fd : filedesc;
    n : integer;
begin
    n := 0;
    fd := mustopen(name, IOREAD);
    while (getcf(c, fd) <> ENDFILE) do
	n := n + 1;
    tool_close(fd);
    fsize := n;
end;    {fsize}

	{MAKEHDR}
{ makehdr -- make header line for archive member }
procedure makehdr (var name, head : string );
var
    i : integer;
    afd, tfd : filedesc;
    space_string : string[1];  (* a string to use as a space. *)

(* Note that once again the code here could be shortened considerably by using
  M/MT+ intrinsic string-handling routines.  It has been left as a modified
  version of the original, though it could all be changed to:

  head := concat(archhdr,' ',name,' ');
  i := itoc(fsize(name), head, (length(head) + 1));
  head := concat(head, chr(NEWLINE));
*)

begin (* makehdr *)
	head := '';		(* set length of head to 0 *)
    scopy(archhdr, 1, head, 1);	(* or head := archhdr; *)
    i := length(head) + 1;
    space_string := ' ';	{begins another kludge because of the
    interactions between K&P and the M way of handling strings.}

    head := concat(head, space_string);  {add a space at end of string.
					we use concat because that 
					automagically adjusts length of 
					the string.}
    scopy(name, 1, head, i+1);
    i := length(head) + 1;
    head := concat(head, space_string);
    i := itoc(fsize(name), head, i+1);
		{itoc will return with head = s (within itoc), i unchanged,
		only of course i is now changed by the call; and fsize(name)
		is unchanged by itoc, but is returned.  after the return,
		i is changed to the VALUE OF ITOC, which is not the same as
		all this other stuff.  None of the textbooks give examples
		of this kind of thing, which is a pity.}

    head := concat(head, chr(NEWLINE));	(* concatenates a NEWLINE onto the end
					  of the string head; we know we are
					  always going to be at the end of the
					  string because we start at its end *)
  (*head[i] := chr(NEWLINE);
    head[i+1] := ENDSTR;	Unnecessary because we are using strings *)

end (* makehdr *);

{ADDFILE}
{ addfile -- add file "name" to archive }
procedure addfile (var name : string; fd : filedesc);
var
    head : string;
    nfd : filedesc;

begin (* addfile *)
    nfd := open(name, IOREAD);
    if (nfd = IOERROR) then begin
	putstr(name, STDERR);
	message(': can''t add');
	errcount := errcount + 1
    end;
    if (errcount = 0) then begin
	makehdr(name, head);
	putstr(head, fd);

(* You might check that the file descriptor is the same as the last time.  Note
  that this open statement (to rewind the file to its beginning) does NOT
  appear in the original; without it, you of course get nothing copied (in
  Pascal M). *)

        nfd := open(name, IOREAD);
	fcopy( nfd, fd);
	tool_close(nfd);
    end
end (* addfile *);

(* The following procedures are used by replace and several other processes
  that are declared before they are, so they must be declared "forward". *)

function gethdr (fd : filedesc; var buf, name : string;
	var size : integer) : boolean; forward;

procedure fskip (fd : filedesc; n : integer); forward;

function filearg (var name : string) : boolean; forward;

procedure acopy (fdi, fdo : filedesc; n : integer); forward;

{REPLACE}
{ replace -- replace or delete files }
procedure replace(afd, tfd : filedesc; cmd : character);
var
    inline, uname : string;
    size : integer;
begin
    while (gethdr(afd, inline, uname, size)) do
	if (filearg(uname)) then begin
	    if (cmd = ord('u')) then	{ add new one }
		addfile(uname, tfd);
	    fskip(afd, size)	{ discard old one }
	end
	else begin
	    putstr(inline, tfd);
	    acopy(afd, tfd, size)
	end
end {REPLACE};

{GETHDR}
{ gethdr -- get header info from fd }
function gethdr (* (fd : filedesc; var buf, name : string;
	var size : integer) : boolean *);
var
    temp : string;
    i : integer;
begin

    if (getline(buf, fd, MAXSTR) = false) then
	BEGIN
		gethdr := false
	END
    else begin
	i := getword(buf, 1, temp);
	if (not equal(temp, archhdr)) then
	    error('gethdr: archive not in proper format');
	i := getword(buf, i, name);
	size := ctoi(buf, i);
	gethdr := true
    end
end {GETHDR};

{TABLE}
{ table -- print table of archive contents }
procedure table (var aname : string);
const
  tblmsg1 = 'There are ';
  tblmsg2 = ' files archived in the file ';
var
    head, name : string;
    numarchives,
    i,
    size : integer;
    afd : filedesc;
    outstring : string;

{ tprint -- print table entry for one member }
procedure tprint (var buf : string);
var
    i : integer;
    temp : string;
begin
    i := getword(buf, 1, temp); { header }
    i := getword(buf, i, temp); { name }
    putstr(temp, STDOUT);
    putc(BLANK);
    i := getword(buf, i, temp);	{ size }
    putstr(temp, STDOUT);
    putc(NEWLINE)
end;  {tprint}


begin (* table *)
    afd := mustopen(aname, IOREAD);
    numarchives := 0;
    while (gethdr(afd, head, name, size)) do begin
	if (filearg(name)) then
	begin
	    tprint(head);
	    numarchives := numarchives + 1;
	end; { if }
	fskip(afd, size)
    end; { while }

(* the following prints out how many files are archived *)
    outstring := tblmsg1;
    i := itoc(numarchives, outstring, (length(outstring)+1));
    outstring := concat(outstring, tblmsg2, aname);
    message(outstring);

(* notfound tells which, if any, files were unfound *)
    notfound
end(* table *);

{ fskip -- skip n characters on file fd }
procedure fskip (* (fd : filedesc; n : integer)*) ;
var
  c : character;
    i : integer;
begin
    for i := 1 to n do
	if (getcf(c, fd) = ENDFILE) then
	    error('archive: end of file in fskip')
end; {fskip}

{ filearg -- check if name matches argument list }
function filearg (* (var name : string) : boolean *);
var
    i : integer;
    found : boolean;

(* Converted to use strings rather than tstrings.  The function "equal" could
  be replaced, but for consistency I decided to leave it. *)

begin
    if (nfiles <= 0) then
	filearg := true
    else begin
	found := false;
	i := 1;
	while (not found) and (i <= nfiles) do begin
	    if (equal(name, fname[i])) then begin
		fstat[i] := true;
		found := true
	    end; { if }
	    i := i + 1
	end; {while}
	filearg := found
    end  { if nfiles }
end; {filearg}

{ acopy -- copy n characters from fdi to fdo }
procedure acopy (* (fdi, fdo : filedesc; n : integer) *);
var
    c : character;
    i : integer;
begin
    for i := 1 to n do
	if (getcf(c, fdi) = ENDFILE) then
	    error('archive: end of file in acopy')
	else
	   putcf(c, fdo)
end; {acopy}

{ extract -- extract files from archive }
procedure extract (var aname : string; cmd : character);
var
    ename, inline : string;
    afd, efd : filedesc;
    size : integer;
begin
    afd := mustopen(aname, IOREAD);
    if (cmd = ord('p')) then
	efd := STDOUT
    else
	efd := IOERROR;
    while (gethdr(afd, inline, ename, size)) do
	if (not filearg(ename)) then
	    fskip(afd, size)
	else begin
	    if (efd <> STDOUT) then
		efd := create(ename, IOWRITE);
	    if (efd = IOERROR) then begin
		putstr(ename, STDERR);
		message(': can''t create');
		errcount := errcount + 1;
		fskip(afd, size)
	    end {if efd = IOERROR }
	    else begin
 	        acopy(afd, efd, size);
	        if (efd <> STDOUT) then
		    tool_close(efd)
	    end {else}
	end; {if not filearg}
    notfound
end; {extract}

{ tool_delete -- delete files from archive }
procedure tool_delete (var aname : string);
var
    afd, tfd : filedesc;

(* The name has been changed so that we may use the standard string-handling
  procedure "delete".  You might note that this wouldn't affect any calls
  before this point, but all those after this procedure declaration would be to
  this procedure.  In the interest of sanity I've changed its name. *)

begin
    if (nfiles <= 0) then    { protect the innocents from the slaughter }
	error(
'archive: -d requires explicit file names, no wildcards allowed');
    afd := mustopen(aname, IOREAD);
    tfd := mustcreate(archtemp, IOWRITE);
    replace(afd, tfd, ord('d'));
    notfound;
    tool_close(afd);
    tool_close(tfd);
    if (errcount = 0) then
	fmove(archtemp, aname)
    else
	message('delete: fatal errors - archive not altered');
    remove(archtemp)
end;

