#!/bin/sh
# 5799-WZQ (C) COPYRIGHT IBM CORPORATION 1986,1987,1988
# LICENSED MATERIALS - PROPERTY OF IBM
# REFER TO COPYRIGHT INSTRUCTIONS FORM NUMBER G120-2083
# $Header:restore.tape.sh 12.2$ 
# $ACIS:restore.tape.sh 12.2$ 
# $Source: /ibm/acis/usr/sys/dist/RCS/restore.tape.sh,v $ 
###############################################################################
# restore.{tape,net}	restore filesystems from tape or over network
#
# usage:
#	restore.{tape,net} [option ...] parameter ...
#
# available options:
#	both:		-T hdxx	alternate partition for writable /tmp
#			-fstab	create /etc/fstab based on the available
#				information
#			-config	type
#				create /etc/rc.config for the specified
#				system configuration type (full, reduced,
#				minimal, nfsreduced, or nfsminimal) based
#				on available information
#			-fsck	install /etc/fsck from miniroot
#			-x	extract on top of existing filesystem
#			-q	quiet; don't produce verbose output
#			-m mask	specify a mask of the system
#			-newfs string
#				pass string to newfs(8)'s command line.
#				By default, string is "".
#			-k kernel
#				link specified kernel to /vmunix
#				echo hostname into /etc/hostname 
#	restore.net:	-s sourcehost
#				install from sourcehost rather than master
#			-d desthost
#				configure network as desthost rather than slave
#			-{un,lan}
#				select network interface
#			-ifconfig string
#				pass string to ifconfig(8)'s command line
#
# available parameters:
#	both:		disk	disk is hd[0-2][a-h] or sc[01][a-h]
#			disk=name
#				name is root, user, source, minimal, or site
#	restore.tape:	prompt	prompt the user for a tape
#			prompt=name
#				prompt the user for the "name" tape
#			skip	skip one file on the tape
#	restore.net:	disk1=disk2
#				installs disk1 with the contents of disk2 on
#				sourcehost
#			disk=filesystem
#				filesystem must be the absolute path of a
#				filesystem mountpoint on sourcehost, and is the
#				same as specifying the partition mounted on
#				filesystem as disk2.
#
# examples:
#	to restore both distribution tapes:
#		restore.tape prompt=root/user hd0a=root hd0g=user \
#			prompt=source hd1g=source 
#
#	to restore the sc0a partition from the hd2g partition on myhost:
#		restore.tape -s myhost sc0a=hd2g
#
###############################################################################
CMD=`basename $0`
umask 022

TAPE=1
NET=0
case $CMD in
restore.tape)
	tape=$TAPE
	USAGE="
usage: restore.tape [-T hdxx] [-b[#]] [-q] [-x] [-fstab] [-fsck] [-k kernel]
		[-config type] [-newfs args] [-m mask] ... 
		prompt[=name] disk|disk=name|skip ...
		...

	where disk = hd[0-2][a-h] or sc[01][a-h] and
	      name = root, user, source, minimal, or site and
	      type = full, reduced, minimal, nfsreduced or nfsminimal
"
	;;
restore.net)
	tape=$NET
	USAGE="
usage: restore.net [-T hdxx] [-s sourcehost] [-d desthost] [-un|-lan] [-q] [-x]
		[-fstab] [-fsck] [-k kernel] [-config type] [-newfs args]
		[-ifconfig args] [-m mask] ... 
		disk|disk=name|disk=disk|disk=filesystem ...

	where disk	 = hd[012][a-h] or sc[01][a-h] and
	      name	 = root, user, source, minimal, or site and
	      filesystem = <absolute path of any mountpoint on desthost> and
	      type	 = full, reduced, minimal, nfsreduced, or nfsminimal
"
	;;
*)	echo "$CMD:command name must be restore.{net,tape}" >&2
	exit 1;;
esac

case $# in
0)	echo "$USAGE"
	exit 1;;
esac

# defaults
operator=operator
prompt=first
fstab=0
new_fstab=
fsck=0
ether="un0"
token="lan0"
lan_order="$ether $token"
ifconfig_args=
tmp_part=hd0b
newfs=/etc/newfs
newfs_args=""
kernel=vmunix
hostname=
masks=

case $tape in
$NET)	sourcehost="master"
	desthost="slave";;
$TAPE)	sourcehost="tape"
	desthost="master";;
esac

#		NORMAL VALUE			VALUE IN QUIET MODE
verbose=v	# "v" = verbose /etc/restore;	"" = nonverbose /etc/restore
vecho=echo	# "echo" = real echo;		":" = quiet echo (no-op)
vstdout='1>&1'	# "1>&1" = keep stdout;		"1>/dev/null" = toss stdout
vstderr='2>&2'	# "2>&2" = keep stderr;		"2>/dev/null" = toss stderr

#
# scan and process options
#
while :
do
	case "$tape:$1" in
	# net exclusive options
	$NET:-s)
		sourcehost="$2"
		shift;;
	$NET:-d)
		desthost="$2"
		shift;;
	$NET:-un)
		lan_order="$ether $token";;
	$NET:-lan)
		lan_order="$token $ether";;
	$NET:-ifconfig)
		ifconfig_args="$2"
		shift;;
	
	# tape exclusive options (none now)
	
	# media independent options
	?:-T)	case "$2" in
		hd[012])
			tmp_part="$2"b;;
		hd[012][a-h])
			tmp_part="$2";;
		*)	echo "$CMD:$2:invalid partition for /tmp $USAGE" >&2
			exit 1;;
		esac
		shift;;
	
	?:-fstab)	fstab=1;;

	?:-fsck)	fsck=1;;
	
	?:-m)	if mask test "$2" > /dev/null
		then
			masks="$masks $2"
			shift
		else
			# unknown mask
			exit 1
		fi;;

	?:-x)	newfs=':';;

	?:-newfs)
		newfs_args="$2"
		shift;;
	
	?:-q)	verbose=
		vecho=':'
		vstdout='1> /dev/null'
		vstderr='2> /dev/null';;

	?:-k)	kernel="$2"
		shift;;

	?:-config)
		case "$2" in
		minimal|reduced|full|nfsminimal|nfsreduced)
			config="$2";;
		*)	echo "$CMD:$2:unknown system type $USAGE" >&2
			exit 1;;
		esac
		shift;;

	?:-*)	echo "$CMD:$1:unknown option $USAGE" >&2
		exit 1;;
	*)	break;;
	esac
	shift
done
	
case $# in
0)	echo "$CMD:no partition specified $USAGE" >&2
	exit 1;;
esac
#
# scan parameters and set up /etc/fstab entries
#
for i
do
	case "$tape:$i" in
	# disk
	# mountpoints are missing from new_fstab entries
	?:hd?[a-h] | \
	?:sc?[a-h])
		disk="$i"
		new_fstab="${new_fstab}/dev/$disk /unknown ignore xx 1 0
"
		;;

	# disk=name
	# new_fstab entries are fairly straight-forward
	?:hd?[a-h]=root	| \
	?:sc?[a-h]=root	| \
	?:hd?[a-h]=user	| \
	?:sc?[a-h]=user	| \
	?:hd?[a-h]=source	| \
	?:sc?[a-h]=source	| \
	?:hd?[a-h]=minimal	| \
	?:sc?[a-h]=minimal	| \
	?:hd?[a-h]=site	| \
	?:sc?[a-h]=site)
		disk=`IFS==;set $i;echo $1`
		name=`IFS==;set $i;echo $2`

		case "$name" in
		root | minimal)
			fs=/
			pass=1;;
		site )	fs=/site
			pass=2;;
		*)	fs=`whichdev $name`
			pass=2;;
		esac

		new_fstab="${new_fstab}/dev/$disk $fs ufs rw 1 $pass
"
		;;

	# tape control
	$TAPE:prompt)			;;
	$TAPE:prompt=*)			;;
	$TAPE:skip)			;;

	# disk=disk
	# mountpoints are missing from new_fstab entries
	$NET:hd?[a-h]=hd?[a-h]	| \
	$NET:hd?[a-h]=sc?[a-h]	| \
	$NET:sc?[a-h]=hd?[a-h]	| \
	$NET:sc?[a-h]=sc?[a-h])
		disk=`IFS==;set $i;echo $1`
		new_fstab="${new_fstab}/dev/$disk /unknown ignore xx 1 0
"
		;;

	# this bug could be fixed if we had expr
	$NET:hd?[a-h]=/*=* | \
	$NET:sc?[a-h]=/*=*)
		echo "$CMD:$i:filesystem name includes '=' $USAGE" >&2
		exit 1;;

	# disk=filesystem
	# new_fstab entries all have identical fs_passno values
	$NET:hd?[a-h]=/*	| \
	$NET:sc?[a-h]=/*)
		disk=`IFS==;set $i;echo $1`
		fs=`IFS==;set $i;echo $2`
		new_fstab="${new_fstab}/dev/$disk $fs ufs rw 1 3
"
		;;
	# Nothing needs to be done at this point with the AFS component.
	$TAPE:????,????=afs) :;;
	*) 	echo "$CMD:$i:unknown partition $USAGE" >&2
		exit 1;;
	esac
done

# sanity check restore.tape's parameters
case "$tape" in
$TAPE)	case "$1" in
	prompt*)
		;;
	*)	echo "$CMD:$1:first parameter must be 'prompt' $USAGE" >&2
		exit 1;;
	esac
esac

#
# now mount /dev/$tmp_part as /tmp so we have a writable /tmp
# do it now, since /etc/hosts -> /tmp/hosts
#
trap "cd /;( /etc/umount /dev/$tmp_part) > /dev/null 2>&1 ; exit 1" 0 2

/etc/umount /dev/$tmp_part > /dev/null 2>&1 
if (>>/tmp/x) > /dev/null 2>&1
then
	$vecho
	echo ">>> /tmp writable; skipping mkfs and mount "
	$vecho
else
	$vecho
	echo ">>> creating a /tmp filesystem on the $tmp_part partition"
	$vecho
	if eval /etc/mkfs /dev/$tmp_part 1000 $vstdout
	then 
		/etc/mount /dev/$tmp_part /tmp
		cat /etc/hosts.base > /etc/hosts
	else
		$vecho >&2
		echo " ### error on /dev/$tmp_part, check disk and restart" >&2
		exit 1
	fi
fi

case $tape in
$NET)	$vecho
	echo ">>> establishing network connection from '$desthost' to '$sourcehost'"
	$vecho

	echo $sourcehost $sourcehost >> /etc/hosts
	
	for network in $lan_order abort
	do
		case $network in
		abort)	echo " ### cannot configure network as '$desthost'" >&2
			$vecho " ### either no network interface exists or
 ### '$desthost' is not in /etc/hosts.base" >&2
			exit 1;;
		*)	ifconfig $network inet $desthost $ifconfig_args && break;;
		esac
	done
	
	ifconfig $network inet
	# set up network for mkconfig
	export network

	# network routing daemon lets us use gateways
	# Start routed up and kill it after 10 seconds.
	# That way, it can be restarted on retries.
	# The subshell causes the "killed" message to
	# be suppressed.
	(
	/etc/routed -t > /dev/null 2>&1 &
	routed_pid=${!}
	sleep 10
	exec kill -9 ${routed_pid}
	)

	if eval rsh '$sourcehost' -n -l $operator hostname $vstdout $vstderr 
	then
		: All is well
	else
		echo " ### '$sourcehost' not responding!" >&2
		$vecho " ###	Are we connected to the network?
 ###	Are they in our network tables?
 ###	Are we in their ~operator/.rhosts and /etc/hosts?" >&2
		exit 1
	fi;;
esac

$vecho
echo ">>> Starting '$CMD' with the following parameters '$*'"
$vecho
#
# main loop - scan and process parameters
#
for i
do
	# Handle tape control now, and break other parameters into components.
	# (I would use expr if I had it...)  The variables are:
	# 	hdpart	hard disk partition on this machine
	#	srcpart	disk partition on source machine and name for
	#		determining masking
	case "$i" in

	prompt*)
		# don't rewind before prompting for the first tape
		case $prompt in
		first)	;;
		*)	echo ">>> rewinding tape" < /dev/st0 >&2;;
		esac

		case "$i" in
		prompt=*)
			prompt="'`IFS==; set $i; shift; echo $*`'";;
		esac

		while : wait until tape is ready
		do
			echo -n ">>> Insert $prompt tape and press <ENTER> "
			read answer
			case $answer in
			"")	if echo -n < /dev/st0
				then break # tape is readable and rewound
				else
					echo " ### tape drive not ready!!!" >&2
					$vecho " ### Is the POWER ON?
 ### Is the door closed (handle vertical)?
 ### Is the tape drive plugged into the adapter?
 ### Fix the problem and try again (else type 'quit')" >&2
					continue
				fi;;
			"quit")	exit 1;;
			*)	echo " ### '$answer' invalid response, press <ENTER> or type 'quit'" >&2
				continue;;
			esac
		done

		# change default prompt
		prompt=next
		continue;;

	skip)	echo ">>> skipping tape file" >&2
		if ftc /dev/nrst0 /dev/null
		then : all is well
		else
			echo " ### cannot skip tape file !!! " >&2
			exit 1
		fi
		continue;;

	????)	# disk
		hdpart=$i; srcpart=$i; HDPART=/dev/$hdpart
		;;

	????,????=afs | \
	????=*)	# disk=name or disk=disk or disk=filesystem or disk,disk=afs
		eval `IFS==; set $i; echo "hdpart=$1"
			shift; echo "srcpart=$*"`
		;;

	*)	echo "$CMD:INTERNAL ERROR!  Illegal parameter '$i'" >&2
		exit 1;;
	esac
	case $srcpart in
	afs)
		eval `IFS=,; set $hdpart; echo "HDPART=\"/dev/$1 /dev/$2\"`;;
	*)
		HDPART=/dev/$hdpart;;
	esac

	trap "cd /;( /etc/umount /dev/$tmp_part;
			/etc/umount /dev/$hdpart ) > /dev/null 2>&1;
			exit 1" 0 2
	
	#
	# create filesystem and mount on /mnt
	#
	echo ">>> mounting the '$HDPART' partition(s) of the hard disk"
	/etc/umount $HDPART > /dev/null 2>&1 

	firstpart=1
	for part in $HDPART
	do
	  case $srcpart in
	  afs) :;;
	  *)
	  	if eval $newfs $newfs_args -v $part $vstdout
	  	then : newfs succeeded
	  	else
		  	echo " ### newfs of $part failed, check disk and restart" >&2
		  	exit 1
	  	fi;;
	  esac

	  if eval /etc/fsck -p $part $vstdout
	  then	eval df $part $vstdout
	  else
		  echo " ### fsck of $part failed, check disk and restart" >&2
		  exit 1
	  fi

	  case "$srcpart" in
	  site)	mount=/mnt/site;;
	  afs)  case $firstpart in
		0) mount=/mnt/usr;;
		1) mount=/mnt;;
		esac;;
	  *)	mount=/mnt;;
	  esac

	  /etc/mount $part $mount

	  case $? in
	  0|255) 	;; # mount succeeded
	  *)	echo " ### error mounting $part on $mount" >&2
		exit 1;;
	  esac
	  firstpart=0
	done

	cd /mnt

	# set up /usr for reduced filesystem.
	case "$srcpart:$config" in
	root:reduced)
		ln -s /andrew/usr usr;;
	esac

	# TTYINPUT contains the interactive input to /etc/restore
	TTYINPUT=/tmp/ttyinput
	echo 'add' > $TTYINPUT
	mask $srcpart $masks >> $TTYINPUT 2> /dev/null

	case $tape in
		$NET)
			# don't install an unwanted kernel
			case "$srcpart:$kernel" in
			*:vmunix) echo delete vmunix\* >> $TTYINPUT
				  echo add vmunix >> $TTYINPUT;;
			root:*)	echo delete vmunix\* >> $TTYINPUT;;
			esac

			#don't install an unwanted boot
			case $srcpart in
        			root)
					echo delete boot >> $TTYINPUT;;
				*) :;;
			esac

			# Don't bring over a tmp symbolic link.
			case "$srcpart" in
				root|user) echo delete tmp >> $TTYINPUT;;
			esac;;
		$TAPE) :;;
	esac

	# don't install an unwanted fsck or vfsck
	case "$srcpart:$fsck" in
	root:1)	echo "delete etc/fsck
delete etc/vfsck" >> $TTYINPUT;;
	esac

	# /site is a special case
	case "$srcpart" in
	site)
		echo "add site" > $TTYINPUT
		case "$config" in
		nfsreduced)
			# NFS reduced systems get /site from /usr/site
			# instead of /minroot/site.
			srcpart=/usr;;
		esac;;
	esac

	echo 'extract
yes
quit' >> $TTYINPUT

	case $tape in
	#
	# restore from net - now that filesystem is mounted and network 
	#			connection established.
	#

	$NET)	# figure out which partition to dump
		rdev=`whichdev $srcpart 2>&1`
		case "$rdev" in
		/*)	;;
		*)	echo "$CMD:$srcpart:invalid partition:
$rdev" >&2
			exit 1;;
		esac

		if rsh $sourcehost -n -l $operator /etc/dump 0fs - 20000 $rdev |
			(/etc/restore ifF$verbose - $TTYINPUT
			 EXIT=$?
			 cat > /dev/null
			 exit $EXIT)
		then : all is well
			echo
		else
			echo "
 ### error $? DURING restore" >&2
			$vecho " ### check network connection" >&2

#NOTYET	tell how to restart

			exit 1
		fi;;

	#
	# restore from tape - now that filesystem is mounted and tape is loaded
	#
	$TAPE)	if ftc /dev/nrst0 - |
			(/etc/restore ifF$verbose - $TTYINPUT
			 EXIT=$?
			 cat > /dev/null
			 exit $EXIT)
		then : all is well
			echo
		else
			echo "
 ### error $? DURING restore" >&2

#NOTYET tell how to restart

			exit 1
		fi;;
	esac

	#
	# If root or user pertition, and net install, then
	# make sure a real directory is there.
	#
	case "$tape" in
		$NET)
			case $srcpart in
				root|user)
						echo ">>> Creating tmp directory"
						mkdir tmp 2> /dev/null
						chmod 777 tmp;;
			esac;;
	esac

	#
	# install real fsck
	# install real kernel (if net install)
	# install boot (if net install and (ATR==0)).
	# link $kernel to /vmunix
	#
	case "$srcpart" in
	root)
		case "$fsck" in
		1)	echo ">>> Installing real /etc/fsck"
			cat /etc/fsck > etc/fsck
			chmod 755 etc/fsck;;
		esac

		case "$kernel" in
		vmunix)	:;;
		*)	case $tape in
			$NET)
				echo ">>> Installing $kernel"
				kern_found=`rsh $sourcehost -n -l operator /bin/sh -c "'/bin/test -f /$kernel;/bin/echo \\$?'"`
				case $kern_found in
				0) :;;
				*)
					echo " ### Specified kernel ($kernel) not found!
 ### Your system will not boot!" >&2
					$vecho " ### $kernel should be on the root filesystem on host '$sourcehost'
 ### Restart the installation after fixing '$sourcehost'" >&2
					exit 1;;
				esac
				if rsh $sourcehost -n -l operator \
					/bin/cat /$kernel > $kernel
				then : all is well
				else
					echo " ### Trouble getting $kernel from $sourcehost"
					echo " ### Fix the problem and try again."
					exit 1
				fi;;
			$TAPE) :;;
			esac
			echo ">>> Linking /$kernel to /vmunix"
			chmod 755 $kernel
			ln $kernel vmunix
			rmdir vicep? minroot > /dev/null 2>&1;;
		esac
		case $ATR in
			0)	case $tape in
				$NET)
					echo ">>> Installing boot"
					boot_found=`rsh $sourcehost -n -l operator /bin/sh -c "'/bin/test -f /boot;/bin/echo \\$?'"`
					case $boot_found in
					0) :;;
					*)	echo " ### /boot not found!
 ### Your system will not boot!" >&2
						$vecho " ### boot should be in the root directory on host '$sourcehost'
 ### Restart the installation after fixing '$sourcehost'" >&2
						exit 1;;
					esac
					if rsh $sourcehost -n -l operator \
						/bin/cat /boot > boot
					then : all is well
					else
						echo " ### Trouble getting /boot from $sourcehost"
						echo " ### Fix the problem and try again."
						exit 1
					fi
					chmod 755 boot;;
				$TAPE) :;;
				esac;;
			*) :;;
		esac;;
	minimal)
		# Minimal install assumes net install, so don't bother
		# checking for tape or net install.

		echo ">>> Installing $kernel"
		kern_found=`rsh $sourcehost -n -l operator /bin/sh -c "'/bin/test -f /$kernel;/bin/echo \\$?'"`
		case $kern_found in
		0) :;;
		*)
			echo " ### Specified kernel ($kernel) not found!
 ### Your system will not boot!" >&2
			$vecho " ### $kernel should be on the root filesystem on host '$sourcehost'
 ### Restart the installation after fixing '$sourcehost'" >&2
			exit 1;;
		esac
		if rsh $sourcehost -n -l operator \
			/bin/cat /$kernel > $kernel
		then : all is well
		else
			echo " ### Trouble getting $kernel from $sourcehost"
			echo " ### Fix the problem and try again."
			exit 1
		fi
		chmod 755 $kernel
		case $kernel in
			vmunix) :;;
			*)
				echo ">>> Linking /$kernel to /vmunix"
				ln $kernel vmunix;;
		esac
		case $ATR in
			0)	echo ">>> Installing boot"
				boot_found=`rsh $sourcehost -n -l operator /bin/sh -c "'/bin/test -f /boot;/bin/echo \\$?'"`
				case $boot_found in
				0) :;;
				*)	echo " ### /boot not found!
 ### Your system will not boot!" >&2
					$vecho " ### boot should be in the root directory on host '$sourcehost'
 ### Restart the installation after fixing '$sourcehost'" >&2
					exit 1;;
				esac
				if rsh $sourcehost -n -l operator \
					/bin/cat /boot > boot
				then : all is well
				else
					echo " ### Trouble getting /boot from $sourcehost"
					echo " ### Fix the problem and try again."
					exit 1
				fi
				chmod 755 boot;;
			*) :;;
		esac;;
	esac

	#
	# create /site /etc/ttys /.scm /.profile for reduced root as needed
	# create new /etc/fstab
	# create new /etc/rc.config
	# create new /etc/exports.
	#
	case "$srcpart" in
	root | minimal)
		echo "# This is an sample exports file.  See exports(5) for more detail.
# The syntax is: 
# Export-file-system-name       machine-name-to-export-to
#
# the following are some example lines.
#/usr           client          #export /usr to machine named client
#/usr                           #export /usr to the world" > etc/exports
		case "$config" in
		reduced)
			mkdir site > /dev/null 2>&1

			minroot=`whichdev minimal`
			for file in etc/ttys .scm .profile
			do
				rsh $sourcehost -n -l operator \
					/bin/cat $minroot/$file > $file
			done

			# set up andrew_servers for mkconfig
			andrew_servers="`cat .scm`"
			export andrew_servers
			;;
		nfsreduced)
			mkdir site > /dev/null 2>&1;;
		esac
		case "$fstab" in
		1)	echo ">>> Creating new /etc/fstab"
			test -r etc/fstab && syscall unlink etc/fstab
			echo -n "$new_fstab" > etc/fstab
			case "$srcpart" in
			minimal)
				nfs_server=`rsh ${sourcehost} -n -l operator /bin/hostname`
				afs_servers=`cat .scm`
				echo "$nfs_server:/ /root nfs intr,hard,noauto,ro 0 0" >> etc/fstab
				case "$config" in
				nfsminimal)
					echo "$nfs_server:/usr /usr nfs intr,hard,ro 0 0" >> etc/fstab;;
				minimal)
					echo "$afs_servers /andrew afs rw,cache=/site/venuscache,cache%=80,cache_reserve=2500 0 0" >> etc/fstab;;
				esac;;
			root)
				case "$config" in
				reduced)
					afs_servers=`cat .scm`
					echo "$afs_servers /andrew afs rw,cache=/site/venuscache,cache%=80,cache_reserve=2500 0 0" >> etc/fstab;;
				nfsreduced)
					nfs_server=`rsh ${sourcehost} -n -l operator /bin/hostname`
					echo "$nfs_server:/usr /usr nfs intr,hard,ro 0 0" >> etc/fstab;;
				esac;;
			esac;;
		esac

		case "$tape" in
		$NET)
			#
			# Try to figure out the destination host's real name.
			#
			destname="`rsh $sourcehost -n -l operator /sys/dist/host_name $desthost`";;
		$TAPE)
			destname=$desthost;;
		esac
		case "$config" in
		"")	;;
		*)	hostname="$destname" \
				net_flags="$ifconfig_args" \
				system_type="$config" \
				mkconfig > etc/rc.config;;
		esac
		;;
	esac
	#
	# unmount $HDPART and fsck
	#
	sync
	cd /
	/etc/umount $HDPART > /dev/null 2>&1 
	sync

	echo ">>> checking filesystem consistency"

	if eval /etc/fsck -p $HDPART $vstdout
	then	eval df $HDPART $vstdout
	else
		echo " ### error during fsck" >&2
		$vecho " ### reload last tape or run fsck manually" >&2
		exit 1
	fi

done

case $tape in
$TAPE)	echo ">>> rewinding tape" < /dev/st0 >&2;;
esac


#
# unmount /dev/$tmp_part from /tmp
#
/etc/umount /dev/$tmp_part > /dev/null 2>&1 
#
sync
sync
# turn off the trap since it exits 0
trap 0
exit 0
