/* xbgsun.c:
 *
 * jim frost 9.21.88
 *
 * this reads a sun raster image file and sets the root window background
 * pixmap to it.  this program doesn't deal well with color.  if IMAGE
 * is defined, this will copy the image to an XImage structure and send
 * that across.  this is much faster than plotting every point but may
 * not work for all systems.  works fine on my Sun2, which needs all the
 * help it can get.
 *
 * this program is in the public domain.
 */

#include <stdio.h>
#include <X11/X.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>

/* this describes the header for Sun rasterfiles.  if you have SunOS, a
 * better description is in /usr/include/rasterfile.h.  this is used
 * instead to improve portability and to avoid distribution problems.
 */

struct rheader {
  unsigned char magic[4];   /* magic number */
  unsigned char width[4];   /* width of image in pixels */
  unsigned char height[4];  /* height of image in pixels */
  unsigned char depth[4];   /* depth of each pixel */
  unsigned char length[4];  /* length of the image in bytes */
  unsigned char type[4];    /* format of file */
  unsigned char maptype[4]; /* type of colormap */
  unsigned char maplen[4];  /* length of colormap in bytes */
};

/* following the header is the colormap (unless maplen is zero) then
 * the image.  each row of the image is rounded to 2 bytes.
 */

#define RMAGICNUMBER 0x59a66a95 /* magic number of this file type */

/* these are the possible file formats
 */

#define ROLD       0 /* old format, see /usr/include/rasterfile.h */
#define RSTANDARD  1 /* standard format */
#define RRLENCODED 2 /* run length encoding to compress the image */

/* these are the possible colormap types.  if it's in RGB format,
 * the map is made up of three byte arrays (red, green, then blue)
 * that are each 1/3 of the colormap length.
 */

#define RNOMAP  0 /* no colormap follows the header */
#define RRGBMAP 1 /* rgb colormap */
#define RRAWMAP 2 /* raw colormap; good luck */

#ifndef IMAGE
/* XDrawPoints (at least the Sun version) can handle at most 32767 points at
 * once, so we use a buffer of nearly that size.  this increases throughput
 * substantially but also locks up the server for the duration of the transfer;
 * this is not normally a serious problem.
 */
 
#define PBUFSIZE 32000   /* point buffer size */
int pptr;                /* pointer in buf */
XPoint pbuf[PBUFSIZE];   /* point buf */
#endif

void usage(name)
char *name;
{ fprintf(stderr, "Usage: %s [-display dispname] [-center] [-black] sun_rasterfile\n", name);
  exit(1);
}

main(argc, argv)
int argc;
char *argv[];
{ int a, center, black;
  char *dname, cmd[1024];
  FILE *f;
  int linesize, row, rows, count, col, cols, xorigin, yorigin,
      dwidth, dheight;
#ifdef IMAGE
  char *data, *idata;
  XImage *image;
  int offset, fullbytes, extrabits, linebytes;
#else
  int mask;
  char *data;
#endif
  Display *disp;
  Pixmap pic;
  XGCValues gcv;
  GC gc;

  dname= "";
  center= 0;
  black= 0;
  for (a= 1; (a < argc) && (*argv[a] == '-'); a++) {
    if ((!strcmp(argv[a], "-center")) || (!strcmp(argv[a], "-c")))
      center= 1;
    else if ((!strcmp(argv[a], "-black")) || (!strcmp(argv[a], "-b")))
      black= 1;
    else if ((!strcmp(argv[a], "-display")) || (!strcmp(argv[a], "-d")))
      dname= argv[++a];
    else
      usage(argv[0]);
  }
  if (a != argc - 1)
    usage(argv[0]);

  /* get rasterfile; we get it from uncompress -c if it ends in .Z
   */

  if (!strcmp(argv[a] + strlen(argv[a]) - 2, ".Z")) {
    sprintf(cmd, "uncompress -c %s", argv[a]);
    f= popen(cmd, "r");
  }
  else {
    if ((f= fopen(argv[a], "r")) == NULL) {
      sprintf(cmd, "%s.Z", argv[a]);
      if ((f= fopen(cmd, "r")) != NULL) {
	fclose(f);
	sprintf(cmd, "uncompress -c %s.Z", argv[a]);
	f= popen(cmd, "r");
      }
    }
  }
  if (f == NULL) {
    perror(argv[a]);
    exit(1);
  }
  if (loadimage(f, &cols, &rows, &linesize, &data) < 0)
    exit(1);

  /* X stuff
   */

  if ((disp= XOpenDisplay(dname)) == NULL) {
    fprintf(stderr, "Can't open display\n");
    exit(1);
  }
  dwidth= DisplayWidth(disp, DefaultScreen(disp));
  dheight= DisplayHeight(disp, DefaultScreen(disp));
  if ((pic= XCreatePixmap(disp, RootWindow(disp, DefaultScreen(disp)),
			  (center ? dwidth : cols), (center ? dheight : rows),
			  1)) == NULL) {
    fprintf(stderr, "Can't create pixmap\n");
    exit(1);
  }
  gcv.function= GXset;
  if ((gc= XCreateGC(disp, pic, GCFunction, &gcv)) == NULL) {
    fprintf(stderr, "Could not create GC\n");
    exit(1);
  }
  if (black) {
    XFillRectangle(disp, pic, gc, 0, 0, (center ? dwidth : cols),
		   (center ? dheight : rows));
    gcv.function= GXclear;
    XChangeGC(disp, gc, GCFunction, &gcv);
  }

  /* convert the raster image data into a pixmap.  we buffer points as much
   * as possible to increase throughput.  most of this is from rasttopbm.c.
   */

  xorigin= (center ? (dwidth - cols) / 2 : 0);
  yorigin= (center ? (dheight  - rows) / 2 : 0);

#ifdef IMAGE
  /* this loading technique gets an XImage of the pixmap we're using,
   * loads the raster image into it, then transfers it back.
   */

  /* load raw image of pixmap; mostly this is used because it's easier than
   * creating a new XImage and coloring it as necessary.
   */

  if ((image= XGetImage(disp, pic, 0, 0, (center ? dwidth : cols),
			(center ? dheight : rows), AllPlanes, XYPixmap)) ==
      NULL) {
    fprintf(stderr, "Could not create image\n");
    exit(1);
  }

  /* this attempts to adjust image data for native XImage format.  it is
   * untested and probably doesn't cover every case anyway.
   */

  if (image->bitmap_bit_order == LSBFirst) {   /* reverse bit order in byte */
    int bitnum;  /* bit we're working on */
    int bytenum; /* byte number within line */
    char sbyte;  /* source byte */

    idata= data;
    for (row= 0; (row < rows) && (row < dheight); row++) {
      bitnum= 7;
      bytenum= 0;
      sbyte= *idata;
       *idata= 0;
      for (col= 0; (col < cols) && (col < dwidth); col++) {
	if (bitnum == 0) {
	  bitnum= 7;
	  sbyte= *(idata + (++bytenum));
	  *(idata + bytenum)= 0;
	}
	*(idata + bytenum) |= (sbyte & 0x1) << (bitnum--);
	sbyte= sbyte >> 1;
      }
      idata += linesize;
    }
  }

  if (image->byte_order == LSBFirst) { /* reverse bytes */
    int sbyte;

    idata= data;
    for (row= 0; row < rows; row++) {
      for (col= 0; col < cols / 8; col += 2) {
	sbyte= *(idata + col);
	*(idata + col)= *(idata + col + 1);
	*(idata + col + 1)= sbyte;
      }
      idata += linesize;
    }
  }

  /* place bitmap of image on XImage data area.
   */

  xorigin= xorigin / 8; /* this also forces byte alignment, which is easier */
  idata= image->data + (yorigin * image->bytes_per_line);
  fullbytes= cols / 8;
  extrabits= cols % 8;
  linebytes= fullbytes + (extrabits > 0 ? 1 : 0);

  for (row= 0; row < rows; row++) {
    for (col= 0; col < linebytes; col++) {
      if ((col == linebytes - 1) && (extrabits))
	*(idata + col + xorigin) &= ((char)0xff >> extrabits);
      else
	*(idata + col + xorigin)= 0;
      *(idata + col + xorigin) |= *(data + col);
    }
    idata += image->bytes_per_line;
    data += linesize;
  }

  /* dump whole image back to pixmap
   */

  gcv.function= GXcopy;
  XChangeGC(disp, gc, GCFunction, &gcv);
  XPutImage(disp, pic, gc, image, 0, 0, 0, 0, (center ? dwidth : cols),
	     (center ? dheight : rows));
#else
  /* this sends the image as a series of point plots which works fine but
   * is kind of slow
   */

  pptr= 0;
  for (row= 0; (row < rows) && (row < dheight); row++) {
    count= 0;
    mask= 0x80;
    for (col= 0; (col < cols) && (col < dwidth); col++ ) {
      if (mask == 0) {
	mask = 0x80;
	count++;
      }
      if (((!black) && (*(data + count) & mask)) ||
	  (black && (!(*(data + count) & mask)))) {
	pbuf[pptr].x= xorigin + col;
	pbuf[pptr++].y= yorigin + row;
	if (pptr == PBUFSIZE) {
	  XDrawPoints(disp, pic, gc, pbuf, PBUFSIZE, CoordModeOrigin);
	  pptr= 0;
	}
      }
      mask= mask >> 1;
    }
    data += linesize;
  }
  if (pptr)
    XDrawPoints(disp, pic, gc, pbuf, pptr - 1, CoordModeOrigin);
#endif

  /* set the root window's bacground pixmap to be the image pixmap and
   * die.
   */

  XSetWindowBackgroundPixmap(disp, RootWindow(disp, DefaultScreen(disp)), pic);
  XFreeGC(disp, gc);
  XFreePixmap(disp, pic);
  XClearWindow(disp, RootWindow(disp, DefaultScreen(disp)));
  XCloseDisplay(disp);
}

/* convert Sun integer (68000) format to local integer format.  probably
 * won't work right if sizeof(int) < 4 and may not work if your system
 * has LSBit first.  at any rate, do any int conversions here if you
 * need to.  this could be a macro but for this few calls, who cares.
 */

unsigned int localint(d)
unsigned char *d;
{ int a;
  unsigned int i;

  i= 0;
  for (a= 0; a < 4; a++)
    i= (i << 8) + *(d++);
  return(i);
}

/* load the rasterfile into memory.  this only knows how to deal with
 * monochrome files in standard format.  feel free to expand on this.
 */

int loadimage(file, width, height, bytesperline, data)
FILE *file;
int *width, *height, *bytesperline;
char **data;
{ struct rheader header;
  char *colormap, *image;
  int a, maplen;

  if (fread(&header, sizeof(struct rheader), 1, file) != 1) {
    fprintf(stderr, "Error loading rasterfile header\n");
    return(-1);
  }

  /* check magic number
   */

  if (localint(header.magic) != RMAGICNUMBER) {
    fprintf(stderr, "Bad magic number on rasterfile (%x should be %x)\n",
	    localint(header.magic), RMAGICNUMBER);
    return(-1);
  }

  /* error checks; we can only handle monochrome bitmaps in standard
   * format
   */

  if (localint(header.depth) != 1) {
    fprintf(stderr, "Rasterfile has depth greater than 1 (not monochrome)\n");
    return(-1);
  }
  if (localint(header.type) != RSTANDARD) {
    fprintf(stderr, "Unsupported rasterfile type\n");
    return(-1);
  }

  /* if we have a colormap, skip it
   */

  if ((maplen= localint(header.maplen)) > 0) {
    if ((colormap= (char *)malloc(maplen)) == NULL) {
      perror("malloc");
      return(-1);
    }
    if (fread(colormap, maplen, 1, file) != 1) {
      fprintf(stderr, "Error reading colormap\n");
      return(-1);
    }
    free(colormap);
  }

  /* load image data
   */

  *width= localint(header.width);
  *height= localint(header.height);
  *bytesperline= (*width) / 8;
  if ((*width) % 16 > 8)   /* images are rounded out to 16 bits */
    *bytesperline += 2;
  else if ((*width) % 16)
    *bytesperline += 1;
  if ((image= (char *)malloc((*height) * (*bytesperline))) == NULL) {
    perror("malloc");
    return(-1);
  }
  *data= image;
  for (a= 0; a < *height; a++) {
    if (fread(image, *bytesperline, 1, file) != 1) {
      fprintf(stderr, "Error reading image data\n");
      return(-1);
    }
    image += *bytesperline;
  }
  return(0);
}



