/*-
 * xlock.c - X11 client to lock a display and display the HOPALONG fractals
 *           from page 14 of the September 1986 Scientific American.
 *
 * Copyright (c) 1988 by Patrick J. Naughton (naughton@wind.Sun.COM)
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted,
 * provided that the above copyright notice appear in all copies and that
 * both that copyright notice and this permission notice appear in
 * supporting documentation.
 *
 * This file is provided AS IS with no warranties of any kind.  The author
 * shall have no liability with respect to the infringement of copyrights,
 * trade secrets or any patents by this file or any part thereof.  In no
 * event will the author be liable for any lost revenue or profits or
 * other special, indirect and consequential damages.
 *
 * Comments and additions should be sent to the author:
 *
 *                     Patrick J. Naughton
 *                     Window Systems Group, MS 14-40
 *                     Sun Microsystems, Inc.
 *                     2550 Garcia Ave
 *                     Mountain View, CA  94043
 *                     (415) 336-1080
 *
 * Revision History:
 * 31-Aug-88: Added -time option.
 *            Removed code for fractals to separate file for modularity.
 *            Added signal handler to restore host access.
 *            Installs dynamic colormap with a Hue Ramp.
 *            If grabs fail then exit.
 *            Added VMS Hacks. (password 'iwiwuu').
 * 08-Jun-88: Fixed root password pointer problem and changed PASSLENGTH to 20.
 * 20-May-88: Added -root to allow root to unlock.
 * 12-Apr-88: Added root password override.
 *            Added screen saver override.
 *            Removed XGrabServer/XUngrabServer.
 *            Added access control handling instead.
 * 01-Apr-88: Added XGrabServer/XUngrabServer for more security.
 * 30-Mar-88: Removed startup password requirement.
 *            Removed cursor to avoid phosphor burn.
 * 27-Mar-88: Rotate fractal by 45 degrees clockwise.
 * 24-Mar-88: Added color support. [-color]
 *            wrote the man page.
 * 23-Mar-88: Added HOPALONG routines from Scientific American Sept. 86 p. 14.
 *            added password requirement for invokation
 *            removed option for command line password
 *            added requirement for display to be "unix:0".
 * 22-Mar-88: Recieved Walter Milliken's comp.windows.x posting.
 *
 * Contributors:
 *  milliken@heron.bbn.com
 *  karlton@wsl.dec.com
 *  dana@thumper.bellcore.com
 *  vesper@3d.dec.com
 */

#include <stdio.h>
#include <math.h>
#include <signal.h>
#include <X11/Xos.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include "hopalong.h"

#ifdef VMS

#define DISPLAYNAME "0::0"
#define random rand
#define srandom srand
char *vms_password = "iwiwuu";	/* I wish I were using Unix! */

#else /* VMS */

#include <pwd.h>
char *crypt();
#define DISPLAYNAME ":0"

#endif /* VMS */

char no_bits[] = {0};		/* For blanking the cursor */

char *pname;			/* argv[0] */
Display *dsp = NULL;		/* current display (must be inited) */
int screen;			/* current screen */
Window w, root;			/* window used to cover screen */
GContext gc;			/* graphics context */
Colormap cmap;			/* colormap */
int inittime = 5;		/* time for each fractal on screen */
int color;
int timeout, interval,
    blanking, exposures;	/* screen saver parameters */

#ifdef sparc
#define NPOINTS 1000
#else
#define NPOINTS 100
#endif

void
error(s1, s2)
char *s1, *s2;
{
    fprintf(stderr, s1, pname, s2);
    exit(1);
}

int oldsigmask;

void
block()
{
    oldsigmask = sigblock(sigmask(SIGINT)|sigmask(SIGQUIT));
}

void
unblock()
{
    sigsetmask(oldsigmask);
}

void
allowsig()
{
    unblock();
    block();
}


XHostAddress *XHosts;
int HostAccessCount;
Bool HostAccessState;

void
XGrabHosts(dsp)
Display *dsp;
{
    XHosts = XListHosts(dsp, &HostAccessCount, &HostAccessState);
    XRemoveHosts(dsp, XHosts, HostAccessCount);
    XEnableAccessControl(dsp);
}

void
XUngrabHosts(dsp)
Display *dsp;
{
    XAddHosts(dsp, XHosts, HostAccessCount);
    XFree(XHosts);
    if (HostAccessState == False)
	XDisableAccessControl(dsp);
}


void
finish()
{
    XSync(dsp);
    XUngrabHosts(dsp);
    XUngrabPointer(dsp, CurrentTime);
    XUngrabKeyboard(dsp, CurrentTime);
    XSetScreenSaver(dsp, timeout, interval, blanking, exposures);
    XDestroyWindow(dsp, w);
    if (color)
	XUninstallColormap(dsp, cmap);
    XFlush(dsp);
    XCloseDisplay(dsp);
}


void
sigcatch()
{
    finish();
    error("%s: caught terminate signal.\nAccess control list restored.\n",
	NULL);
}


void
ReadXString(s, slen, callback, reinit, init)
char *s;
int slen;
void (*callback)();
int (*reinit)();
void (*init)();
{
    XEvent evt;
    char keystr[20];
    char c;
    int i, bp, len;

    init(dsp, w, gc, color, inittime, NPOINTS);
    bp = 0;
    while (True) {
	while (!XPending(dsp)) {
	    allowsig();
	    callback();
	    if (reinit())
		init(dsp, w, gc, color, inittime, NPOINTS);
	}
	XNextEvent(dsp, &evt);
	if (evt.type != KeyPress)
	    continue;
	len = XLookupString((XKeyEvent *) &evt, keystr, 20, NULL, NULL);
	for (i = 0; i < len; i++) {
	    c = keystr[i];
	    switch (c) {
		case 8:			/* ^H */
		case 127:		/* DEL */
		    if (bp > 0) bp--;
		    break;
		case 10:		/* ^J */
		case 13:		/* ^M */
		    s[bp] = '\0';
		    return;
		case 21:		/* ^U */
		    bp = 0;
		    break;
		default:
		    s[bp] = c;
		    if (bp < slen-1) bp++;
	    }
	}
    }
}


void
usage()
{
    error("\tusage: %s [-color] [-time n] [-root]\n", NULL);
}


main(argc, argv)
int argc;
char *argv[];
{
#define PASSLENGTH 20
    char buffer[PASSLENGTH];
    char userpass[PASSLENGTH];
    char rootpass[PASSLENGTH];
    XSetWindowAttributes xswa;
    XGCValues xgcv;
    XColor black, white;
    Cursor mycursor;
    Pixmap lockc, lockm;   
    int skipRoot = True;
    struct passwd *pw;
    int i, status;
    char *s;
    int n;

    block();
    signal(SIGINT, sigcatch);
    signal(SIGQUIT, sigcatch);

    pname = argv[0];
    color = False;

    for (i = 1; i < argc; i++) {
	s = argv[i];
	n = strlen(s);
	if (!strncmp("-color", s, n)) {
	    color = True;
	}
	else if (!strncmp("-root", s, n)) {
	    skipRoot = False;
	}
	else if (!strncmp("-time", s, n)) {
	    if (++i >= argc) usage();
	    inittime = atoi(argv[i]);
	}
#ifdef VMS
	else if (!strncmp("-pw", s, n)) {
	    if (++i >= argc) break;
	    vms_password = argv[i];
	}
#endif VMS
	else usage();
    }

    if (!(dsp = XOpenDisplay(DISPLAYNAME)))
	error("%s: unable to open display.\n", NULL);

    screen = DefaultScreen(dsp);
    root = RootWindow(dsp, screen);

    if (color) {
	XCreateHSBColormap(dsp, screen, &cmap, 0.0, 1.0, 1.0, 1.0, 1.0, 1.0);
	XInstallColormap(dsp, cmap);
    }
    else cmap = DefaultColormap(dsp, screen);

    black.pixel = BlackPixel(dsp, screen);
    XQueryColor(dsp, cmap, &black);

    white.pixel = WhitePixel(dsp, screen);
    XQueryColor(dsp, cmap, &white);

    lockc = XCreateBitmapFromData(dsp, root, no_bits, 1, 1);
    lockm = XCreateBitmapFromData(dsp, root, no_bits, 1, 1);
    mycursor = XCreatePixmapCursor(dsp, lockc, lockm, &black, &black, 0, 0);
    XFreePixmap(dsp, lockc);
    XFreePixmap(dsp, lockm);

    xswa.cursor = mycursor;
    xswa.override_redirect = True;
    xswa.background_pixel = black.pixel;
    xswa.event_mask = KeyPressMask;

    w = XCreateWindow(
	dsp,
	DefaultRootWindow(dsp),
	0, 0,
	DisplayWidth(dsp, screen),
	DisplayHeight(dsp, screen),
	0,			/* borderwidth */
	CopyFromParent,		/* depth */
	InputOutput,		/* class */
	CopyFromParent,		/* visual */
	CWCursor |
	CWOverrideRedirect |
	CWBackPixel |
	CWEventMask,		/* window attribute mask */
	&xswa			/* the attributes */
    );

    XMapWindow(dsp, w);
    XRaiseWindow(dsp, w);

    xgcv.foreground = white.pixel;
    xgcv.background = black.pixel;
    gc = (GContext) XCreateGC(dsp, w, GCForeground | GCBackground, &xgcv);

    XGetScreenSaver(dsp, &timeout, &interval, &blanking, &exposures);
    XSetScreenSaver(dsp, 0, 0, 0, 0); /* disable screen saver */

    XGrabHosts(dsp);

    status = XGrabKeyboard(dsp, w, True,
	GrabModeAsync, GrabModeAsync, CurrentTime);
    if (status != GrabSuccess)
	error("%s: couldn't grab keyboard! (%d)\n", status);

    status = XGrabPointer(dsp, w, False, -1,
	GrabModeAsync, GrabModeAsync, None, mycursor, CurrentTime);
    if (status != GrabSuccess)
	error("%s: couldn't grab pointer! (%d)\n", status);

    allowsig();

    pw = getpwuid(0);
    strcpy(rootpass, pw->pw_passwd);

    pw = getpwuid(getuid());
    strcpy(userpass, pw->pw_passwd);

    srandom(getpid());
    do {
	ReadXString(buffer, PASSLENGTH, hop, hopdone, randomInithop);
#ifdef VMS
    } while (strcmp(buffer, vms_password));
#else
    } while (            (strcmp(crypt(buffer, userpass), userpass))
	  && (skipRoot || strcmp(crypt(buffer, rootpass), rootpass)));
#endif VMS

    finish();

    unblock();

    exit(0);
}
