/****************************************************************************

   This file contains the entire X interface of xperfmon.  Routines in this
   file have virtually no knowledge of anything happening in xperfmon.
   The routines here could easily be used by any non-XToolKit X11 program
   that wishes to make a set of parallel line graphs.

   These routines take an array of integers and graph them by drawing
   parallel line graphs inside of a single window.  Each graph is called
   a "graph."  The entire set of graphs is called a "graph set."

   This program keeps track of afew kinds of values.  There are
   1.  Raw values: These are the values that are passed to this
       section of the program from main_event_loop() in xperfmon.c.  
       They are simply a list of numbers as far as window.c is concerned.
       Before they can be used by X, they need to be scaled.  See the
       comment on the SCALE macro below for more information.
   2.  Pixel values: These values represent actual numbers of pixels.
   3.  Saved values: These correspond to values on the graph.  If
       the stepsize of the graph were always one, there would be no 
       difference between these and pixel values, but since the step 
       size can be set by the user of the program from the commandline,
       there must be a distinction.  Since it is important to know 
       which values are Pixel values and which ones are Saved values, 
       the comments on the data structure and variables will tell 
       which type of values each variable or field is.  A lot of subtle
       bugs (especially in refresh) can occur if the values are mixed up.

*****************************************************************************/
 
/****************** #defines *******************/
#define MAX(a,b) (a > b ? a : b)
#define MIN(a,b) (a < b ? a : b)
#define WINDOW_NAME "X Performance Monitor"
/* The following four definitions were created by the bitmap program and
   are parameters of the border and icon bitmaps respectively. */
#define gray_width 2
#define gray_height 2
#define icon_width 40
#define icon_height 58
/* This is a padding along the inner border of the window inside of which
   nothing will be drawn. */
#define INNER_BORDER ((int)(.3 * (float)font_height))
/* This is a similar border but it appears around each graph. */
#define GRAPH_BORDER ((int)(.2 * (float)font_height))
/* This appears around each label. */
#define FONT_PAD ((int)(.1 * (float)font_height))
/* height in pixels of the name of the host and the padding around it */
#define HOST_HEIGHT (FONT_PAD*2 + font_height)
/* height in pixels of a single graph when the window is at its smallest */
#define MIN_GRAPH_HEIGHT ((graph_set.lines_per_label+1)*(font_height+2*FONT_PAD))
/* default height of the graph */
#define GRAPH_HEIGHT ((graph_set.lines_per_label*2)*font_height)
/* height of the drawable area inside of a graph */
#define DRAW_GRAPH_HEIGHT(i) (graph_list[i]->min_pix - graph_list[i]->max_pix)

/* Convenient macros to do text stuff in the window */
#define XTW(s) XTextWidth(fontstruct,s,strlen(s))
#define XDS(s) XDrawString(display,window,gcontext,x,y,s,strlen(s))

/* Amount of space on the top and bottom of the window that will not be
   used by the graphs */
#define TOPSTUFF (INNER_BORDER + (int)x_params.border_width + HOST_HEIGHT)
#define BOTTOMSTUFF (INNER_BORDER + (int)x_params.border_width)

/* The maximum number of values that can be saved - this is the number of 
   horizontal pixels in the screen */
#define NUM_VALS (DisplayWidth(display,screen))

/* An expression that will scale a pixel value for graph number i.  This
   macro, unlike other numerical values in not surrounded by parentheses.
   The idea here is to take (num*SCALE(i)) which will result in num*int/int.
   This is MUCH faster than making the scale a floating point value.  If
   SCALE is surrounded by parentheses, it will be rounded to an integer
   and the picture will not look right. */
#define SCALE(i) DRAW_GRAPH_HEIGHT(i)/(graph_list[i]->max_value-graph_list[i]->min_value) /* no parentheses on purpose */

/****************** #includes ******************/
#include "all.h"
#include "window.h"

#include <stdio.h>

#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/cursorfont.h>

/* This structure forms the WM_HINTS property of the window,
   letting the window manager know how to handle this window. */

XWMHints xwmh = {
  (InputHint | StateHint),       /* Flags */
  FALSE,                         /* input */
  NormalState,                   /* Initial state */
  NULL,                          /* Icon Pixmap */
  NULL,                          /* Icon Window */
  0,0,                           /* Icon location */
  0,                             /* Icon mask */
  0,                             /* Window group */
};

/* External declarations */
char *index();
/* see the file geometry.c */
void init_hints();
void parse_geometry();

/* X variables */
/* These variables are used by the X window system */
unsigned long        background;    /* background color */
Display *            display;       /* the display - see X docs */
int		     screen;
XEvent               event;         /* the current event */
unsigned long        font_height;
unsigned long        foreground;    /* foreground color */
XFontStruct *        fontstruct;    /* the font being used */
GC                   gcontext;      /* graphics context */
XGCValues            gcontext_vals;

/* bits for border - created by "bitmap" */
static char gray_bits[] = 
{
  0x01, 0x02
};
/* border pixmap */
Pixmap               gray_pixmap;

/* bits for window icon - created by "bitmap" */
static char icon_bits[] = {
   0x55, 0x55, 0x55, 0x55, 0x55, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0x55, 0x55,
   0x55, 0x55, 0x55, 0xfa, 0xff, 0xff, 0xff, 0xbf, 0xfd, 0xff, 0xff, 0xff,
   0x5f, 0xda, 0xf3, 0xff, 0xff, 0xbf, 0x4d, 0xe2, 0xff, 0xff, 0x5f, 0x7a,
   0xff, 0xff, 0xff, 0xbf, 0xfd, 0xff, 0xff, 0xff, 0x5f, 0xfa, 0x1f, 0x00,
   0x00, 0xa0, 0xfd, 0xf3, 0xff, 0xff, 0x5f, 0xca, 0x7f, 0xfc, 0xf8, 0xa3,
   0x8d, 0x3f, 0x09, 0xf2, 0x58, 0xfa, 0x9f, 0xe3, 0x07, 0xbe, 0xfd, 0xf3,
   0xff, 0xff, 0x5f, 0xfa, 0xff, 0xff, 0xff, 0xbf, 0xfd, 0x1f, 0x00, 0x00,
   0x40, 0xfa, 0xf3, 0xff, 0xff, 0xbf, 0xfd, 0x1f, 0xff, 0xff, 0x5f, 0xca,
   0x7f, 0x3f, 0x1e, 0xa0, 0x8d, 0x7f, 0xc2, 0xe0, 0x5f, 0xfa, 0xf3, 0xf8,
   0xff, 0xbf, 0xfd, 0xff, 0xff, 0xff, 0x5f, 0xfa, 0xff, 0xff, 0xff, 0xbf,
   0xfd, 0x1f, 0x00, 0x00, 0x40, 0xfa, 0xf3, 0xff, 0xff, 0xbf, 0xfd, 0x9f,
   0xff, 0xff, 0x47, 0x5a, 0x7f, 0x0c, 0x0c, 0xb0, 0x8d, 0xff, 0xf1, 0xe3,
   0x5f, 0xfa, 0xf3, 0xff, 0xff, 0xbf, 0xfd, 0xff, 0xff, 0xff, 0x5f, 0xfa,
   0x1f, 0x00, 0x00, 0xa0, 0xfd, 0xf3, 0xff, 0xff, 0x5f, 0x7a, 0xfe, 0xcf,
   0xf9, 0xbe, 0x8d, 0x7e, 0xb0, 0x46, 0x5d, 0xfa, 0x1f, 0x7f, 0xbf, 0xa3,
   0xfd, 0xf3, 0xff, 0xff, 0x5f, 0xfa, 0xff, 0xff, 0xff, 0xbf, 0xfd, 0x1f,
   0x00, 0x00, 0x40, 0xfa, 0xf3, 0xff, 0xff, 0xbf, 0xcd, 0x9f, 0xff, 0xff,
   0x5f, 0x9a, 0x7f, 0x7e, 0x3c, 0xb0, 0xfd, 0xff, 0x01, 0x83, 0x47, 0xfa,
   0xf3, 0xff, 0xff, 0xbf, 0xfd, 0xff, 0xff, 0xff, 0x5f, 0xfa, 0x1f, 0x00,
   0x00, 0xa0, 0xfd, 0xf3, 0xff, 0xff, 0x5f, 0xda, 0xff, 0xff, 0xff, 0xa7,
   0x9d, 0xff, 0x0f, 0x0e, 0x5b, 0xfa, 0xff, 0xe0, 0x60, 0xbc, 0xfd, 0x13,
   0xfe, 0xff, 0x5f, 0xfa, 0xff, 0xff, 0xff, 0xbf, 0xfd, 0x1f, 0x00, 0x00,
   0x40, 0xfa, 0xff, 0xff, 0xff, 0xbf, 0xfd, 0xff, 0xff, 0xff, 0x5f, 0xaa,
   0xaa, 0xaa, 0xaa, 0xaa, 0x55, 0x55, 0x55, 0x55, 0x55, 0xaa, 0xaa, 0xaa,
   0xaa, 0xaa};
/* pixmap for icon */
Pixmap               icon_pixmap;
XSizeHints           size_hints;
Window               window;       /* the window */
XWindowAttributes    window_attr;  /* a global struct for window attributes */
/* This structure contains the information that can be specified on the
   commandline that is relevant to the x section of the program.  This
   structure is used by various routines and initialized by set_x_defautls()
   and x_process_cmdline(). */
struct 
{
  char *geometry;
  char *display_name;
  short reversed;
  unsigned long border_width;
  char *font_name;
/*  char *border_color; */
/*  char *fore_color; */
/*  char *back_color; */
/*  char *highlight_color; */
  int step_size;
} x_params;

/* graph variables */
/* These variables are necessary for the interface between a list of numbers
   and a set of graphs */

/* graph_set contains some constant information about the set of graphs */
struct
{
  int num_graphs;
  int lines_per_label;
} graph_set;


/* There is an instance of this structure for each graph.  It contains 
   information needed for drawing the graph.  The first three fields (label,
   min_value, and max_value) are initialized at the beginning of the program.
   The remaining fields change every time the window is resized. */
struct graph
{
  char **label;
  int min_value;  /* min and max are raw values used to calculate scale. */
  int max_value;  /* See the comment at the top for a description of */
                  /* what is meant by "raw values." */
  int min_pix;  /* y pixel values for the bottom and top of the */
  int max_pix;  /* area in which the graph is actually drawn. */
  int zero;     /* y pixel value for zero on the graph */
};

struct
{
  int g_width;    /* width of drawable graph in pixels */
  int left_pix;   /* x pixel values for left of graph drawing area */
  int cur_value;  /* saved value -index to the current item in saved_values */
  int **saved_values; /* This is an array int[value_num][graph_num] of
			 raw values.  It is used for refreshing the graph.
			 There is one saved value for each step on the 
			 graph.  Each saved value corresponds to 
			 x_params.step_size pixel values. */
} graph_data;

/* I believe in dynamic memory allocation. */
struct graph **graph_list;  /* This will become an array of pointers to
			       one graph structure per graph */

/****************** functions ******************/

void calc_graph_size(g_width,g_height)
int *g_width;
int *g_height;
/* This calculates the size of a single graph based on the size of the 
   window.  It takes as arguments pointers to the variables that are to
   contain the calculted width and height. */
{
  (void)XGetWindowAttributes(display,window,&window_attr);
  *g_width = window_attr.width - label_width();
  *g_height = (window_attr.height - TOPSTUFF - BOTTOMSTUFF)/graph_set.num_graphs;
}


void draw_background()
/* This routine draws the background for a window - that is, the hostname,
   the dividers, and the labels including maximum and minimum values. */
{
  int g_height;
  int x,x1,x2;
  int y,y1,y2;
  int i;
  int j;
  char hostname[40];
  char numstring[20];
  
  calc_graph_size(&(graph_data.g_width),&g_height);
  graph_data.left_pix = label_width();

  /* draw hostname */
  x = INNER_BORDER;
  y = font_height + FONT_PAD + INNER_BORDER;
  (void)gethostname(hostname,sizeof(hostname));
  XDS(hostname);

  /* draw dividers */
  for (i = 0; i <= graph_set.num_graphs; i++)
  {
    /* Draw a line from the left of the drawing area to the right of the 
       window at the level of the top of each graph.  Also draw a line
       at the bottom of the last graph hence the <= graph_set.num_graphs
       rather than the < as in all other places */
    x1 = graph_data.left_pix;
    y1 = g_height * i + TOPSTUFF;
    x2 = x1 + graph_data.g_width;
    y2 = y1;
    XDrawLine(display,window,gcontext,x1,y1,x2,y2);
  }

  /* draw labels */
  for (i = 0; i < graph_set.num_graphs; i++)
    {
      /* set min_pix, max_pix, and zero values based on the graph number
	 remembering to add in the TOPSTUFF and GRAPH_BORDER - see comments
	 in #defines section */
      graph_list[i]->min_pix = g_height * (i+1) + TOPSTUFF - GRAPH_BORDER;
      graph_list[i]->max_pix = g_height * i + TOPSTUFF + GRAPH_BORDER;
      graph_list[i]->zero = graph_list[i]->min_pix + (graph_list[i]->min_value*SCALE(i));
      /* Note that since vertical pixel values increase as you move down
	 the screen, max_pix < min_pix. */
      /* Notes "%d " rather than "%d ".  This to leave a space after the
	 number before the beginning of the graph. */
      (void)sprintf(numstring,"%d ",graph_list[i]->min_value);
      x = x1-XTW(numstring);
      y = graph_list[i]->min_pix;
      XDS(numstring);
      (void)sprintf(numstring,"%d ",graph_list[i]->max_value);
      x = x1-XTW(numstring);
      y = graph_list[i]->max_pix + font_height;
      XDS(numstring);
      x = INNER_BORDER;
      /* Set the y value for the first string so that the members of the 
	 label array will be centered in the graph area. */
      y = graph_list[i]->max_pix + DRAW_GRAPH_HEIGHT(i)/2 - (graph_set.lines_per_label-2) * (font_height / 2 + FONT_PAD);
      for (j = 0; j < graph_set.lines_per_label; j++)
	{
	  XDS(graph_list[i]->label[j]);
	  y = y + font_height + 2*FONT_PAD;
	}
    }
}

void set_x_defaults()
/* This routine initializes the x parameters. Initial values are set here
   so that numerous constants do not have to be #defined. */
{
  x_params.geometry = NULL; /* "=%dx%d-0+0"; */
  x_params.display_name = NULL;
  x_params.reversed = TRUE;
  x_params.border_width = 2;
  x_params.font_name = "6x10";
/* These fields should be uncommented if color is added to this program.  
   There are several other references in this program to these fields.  Also
   note that the items in the usage() routine in xperfmon.c need to be 
   uncommented. */
/*  x_params.border_color = NULL; */
/*  x_params.fore_color = NULL; */
/*  x_params.back_color = NULL; */
/*  x_params.highlight_color = NULL; */
  x_params.step_size = 1;
}

short x_process_cmdline(argc,argv,args_found)
int argc;
char **argv;
short args_found[];
/* This routine allows this section of xperfmon to take the command line
   information that it needs leaving the argv array untouched for other
   parts of the program.  This routine returns FALSE if there is an error
   in a parameter and TRUE otherwise.  The array args_found[] contains
   either 1's or 0's.  A 0 means that the argument was not used by X and
   a 1 means that it was; i.e., if a member of argv[] is used, args_found[]
   (the same index) should be set to 1 as is done.  This way, xperfmon.c
   will know that that element has been considered and is not an error. */
{
  int i;

  for (i = 1; i < argc; i++)
    {
      if ((argv[i][0] == '='))               /* define window geometry */
	{
	  args_found[i] = TRUE;
	  x_params.geometry = argv[i];
	}
      else if (index(argv[i],':') != NULL)   /* host:display */
	{
	  args_found[i] = TRUE;
	  x_params.display_name = argv[i];
	}
      else if (ARGIS(i,"-rv","-reverse"))    /* black on white */
	{
	  args_found[i] = TRUE;
	  x_params.reversed = TRUE;
	}
      else if (ARGIS(i,"-fw","-forward"))    /* white on black */
	{
	  args_found[i] = TRUE;
	  x_params.reversed = FALSE;
	}
      else if (ARGIS(i,"-bw","-border"))     /* border width */
	{
	  args_found[i] = TRUE;
	  if (++i >= argc)
	    return(FALSE);
	  else
	    {
	      args_found[i] = TRUE;
	      x_params.border_width = atoi(argv[i]);
	    }
	}
      else if (ARGIS(i,"-fn","-font"))       /* host font name */
	{
	  args_found[i] = TRUE;
	  if (++i >= argc)
	    return(FALSE);
	  else
	    {
	      args_found[i] = TRUE;
	      x_params.font_name = argv[i];
	    }
	}
      /* If this gets added, don't forget to recomment this     ||||  */
/*      else if (ARGIS(i,"-bd","-color"))      /* border color  VVVV 
	{
	  args_found[i] = TRUE;
	  if (++i >= argc)
	    return(FALSE);
	  else
	    {
	      args_found[i] = TRUE;
	      x_params.border_color = argv[i];
	    }
	}
      else if (ARGIS(i,"-fg","-foreground")) /* foreground color
	{
	  args_found[i] = TRUE;
	  if (++i >= argc)
	    return(FALSE);
	  else
	    {
	      args_found[i] = TRUE;
	      x_params.fore_color = argv[i];
	    }
	}
      else if (ARGIS(i,"-bg","-background")) /* background color
	{
	  args_found[i] = TRUE;
	  if (++i >= argc)
	    return(FALSE);
	  else
	    {
	      args_found[i] = TRUE;
	      x_params.back_color = argv[i];
	    }
	}
      else if (ARGIS(i,"-hl","-highlight"))  /* highlight color
	{
	  args_found[i] = TRUE;
	  if (++i >= argc)
	    return(FALSE);
	  else
	    {
	      args_found[i] = TRUE;
	      x_params.highlight_color = argv[i];
	    }
	} */
      else if (ARGIS(i,"-st","-stepsize"))  /* step size */
	{
	  args_found[i] = TRUE;
	  if (++i >= argc)
	    return(FALSE);
	  else
	    {
	      args_found[i] = TRUE;
	      x_params.step_size = atoi(argv[i]);
	    }
	}
    }
  return(TRUE); /* If the routine has gotten to this point, it has finished
		   successfully. */
}

int label_width()
/* This routine calcuates the the width in pixels of the label section of 
   the graph set.  It does this by takintg the width of the longest string
   in the label array and adding to it the width of a five-digit number and
   a space (as represented by "99999 " below. */
{
  int longest = 0;
  int graph;
  int line;

  for (graph = 0; graph < graph_set.num_graphs; graph++)
    for (line = 0; line < graph_set.lines_per_label; line++)
      if (XTW(graph_list[graph]->label[line]) > longest)
	longest = XTW(graph_list[graph]->label[line]);
  /* If the min and max value labels ever start colliding with the
     names of the graphs, "99999 " will have to be increased in length. */
  return(longest + XTW("99999 ") + INNER_BORDER + FONT_PAD);
}

void calc_min_size(width,height)
int *width;
int *height;
/* This routine calculates the minimum width and height of the window. */
{
  *width = 2*label_width();
  *height = HOST_HEIGHT + 2 * INNER_BORDER + graph_set.num_graphs*MIN_GRAPH_HEIGHT;
}

void calc_sugg_geom(x,y,width,height)
int *x;
int *y;
int *width;
int *height;
/* This routine calcuates the suggested size and position of the window. */
{
  *width = 3*label_width();
  *height = HOST_HEIGHT + 2 * INNER_BORDER + graph_set.num_graphs*GRAPH_HEIGHT;
  *x = DisplayWidth(display,screen) - *width;
  *y = 0;
}

int win_setup(argc,argv)
int argc;
char **argv;
/* This routine does all the window setup; i.e., it opens the display, creates
   the window and all bitmaps, etc.  It should be called by main() during the
   setup stages of the program and is not needed after that.  This routine 
   returns the file descriptor that will be used to trap X events via
   select() from main_event_loop() in xperfmon.c. */
{
  Cursor cursor;
  int i;
  XSetWindowAttributes attr;

  /* Open the display establishing communication with the X server. */

  if ((display = XOpenDisplay(x_params.display_name)) == NULL) 
    {
      fprintf(stderr,"%s: can't open dislay '%s'\n",argv[0],x_params.display_name);
      exit(1);
    }

  screen = DefaultScreen (display);

  /* allocate memory for saved_values.  This has to be done after the display
     is opened because before that there is no way of finding out the width
     of the screen. */
  if ((graph_data.saved_values = (int **)calloc(NUM_VALS,sizeof(int *))) == NULL)
    {
      perror("Failure getting memory for saved_values");
      exit(1);
    }
  else
    {
      for (i = 0; i < NUM_VALS; i++)
	if ((graph_data.saved_values[i] = (int *)calloc(graph_set.num_graphs,sizeof(int))) == NULL)
	  {
	    perror("Failure getting memory for saved_values[i]");
	    exit(1);
	  }
    }

  /* Load the font. */
  if ((fontstruct = XLoadQueryFont(display,x_params.font_name)) == NULL)
    {
      fprintf(stderr,"%s: display %s doesn't know font '%s'\n",argv[0],DisplayString(display),x_params.font_name);
      exit(1);
    }
  font_height = fontstruct->max_bounds.ascent+fontstruct->max_bounds.descent;

  /* Select colors for the window background foreground. */

  if (x_params.reversed)
    {
      background = BlackPixel(display,screen);
      foreground = WhitePixel(display,screen);
    }
  else
    {
      background = WhitePixel(display,screen);
      foreground = BlackPixel(display,screen);
    }

  /* Set size hints. */
  calc_min_size(&(size_hints.min_width),&(size_hints.min_height));
  size_hints.flags = PMinSize;

  calc_sugg_geom(&(size_hints.x),&(size_hints.y),&(size_hints.width),&(size_hints.height));
  size_hints.flags |= (PPosition | PSize);
  
  if (x_params.geometry != NULL)
    {
      init_hints(&size_hints);
      parse_geometry(display,screen,x_params.geometry,&size_hints);
      if (size_hints.width < size_hints.min_width)
	size_hints.width = size_hints.min_width;
      if (size_hints.height < size_hints.min_height)
	size_hints.height = size_hints.min_height;
      size_hints.flags |= PMinSize;
    }

  /* Create the window */

  window = XCreateSimpleWindow(display,DefaultRootWindow(display),size_hints.x,size_hints.y,size_hints.width,size_hints.height,x_params.border_width,0L,background);

  /* Create a cursor and associate it with this window. */

  cursor = XCreateFontCursor(display,XC_sb_up_arrow);
  XDefineCursor(display,window,cursor);

  /* Set the border pixmap */

  if ((gray_pixmap = XCreatePixmapFromBitmapData (display, window, gray_bits,
						  gray_width, gray_height,
						  foreground, background,
						  DefaultDepth (display,
								screen))) ==
       (Pixmap) 0) {
      perror("Failure creating border pixmap");
      exit(1);
    }
  else
    {
      XSetWindowBorderPixmap(display,window,gray_pixmap);
      XFreePixmap(display,gray_pixmap);
    }

  /* Set the icon pixmap. */
  icon_pixmap = XCreateBitmapFromData(display,DefaultRootWindow(display),icon_bits,icon_width,icon_height);

  /* Set standard properties for the window manager. */
  XSetWMHints (display, window, &xwmh);
  XSetStandardProperties(display,window,WINDOW_NAME,NULL,icon_pixmap,argv,argc,&size_hints);
/* !!!! Note that the pixmap for the icon cannot be freed - the window manager 
   needs it. !!!! */

  /* Ensure that the window's colormap field points to the default 
   * colormap so that the window manager knows the correct colormap to 
   * use for the window. */

  attr.colormap = DefaultColormap(display,screen);
  XChangeWindowAttributes(display,window,CWColormap,&attr);

  /* Create the graphics context for drawing text. */

  gcontext_vals.font = fontstruct->fid;
  gcontext_vals.foreground = foreground;
  gcontext_vals.background = background;
  gcontext = XCreateGC(display,window,(GCFont | GCForeground | GCBackground),&gcontext_vals);

  /* Set the event mask so that we get told about the appropriate events. */

  XSelectInput(display,window,ExposureMask | KeyPressMask);

  /* 
   * Map the window to make it visible.  See Section3.5.
   */
  XMapWindow(display,window);

  /* Return the file descriptor for X events. */
  return(ConnectionNumber(display));
}

void shift_graph()
{
/* This routine takes care of shifting the graph to the left when the
   information gets drawn off the right end.  It is also called from
   redisplay_window if the current_value is larger than will fit on the
   graph.  It should be called whenever this occurs.  This routine 
   moves the current_value to the center of the graph and takes care
   of moving all saved information so that the graph will be updated 
   properly.  In addition, this routine sees that the max_value field
   for each graph is as high as it needs to be. */
  register int i,j,t;
  int num_vals_shift;
  int max;
  int min;

  num_vals_shift = (graph_data.g_width/2)/x_params.step_size;

  for (i = 0; i < num_vals_shift; i++)
    graph_data.saved_values[i] = graph_data.saved_values[graph_data.cur_value-num_vals_shift+i];
  for (j = 0; j < graph_set.num_graphs; j++)
    {
      max = min = 0;
      for (i = 0; i < num_vals_shift; i++)
	{
	  t = graph_data.saved_values[i][j];
	  max = (max>t) ? max : t;
	  min = (min<t) ? min : t;
	}
      max = (max == min) ? graph_list[j]->max_value : max;
      if (max != graph_list[j]->max_value)
	graph_list[j]->max_value = max;
      min = (graph_list[j]->max_value == min) ? min-1 : min;
      if (min != graph_list[j]->min_value)
	graph_list[j]->min_value = min;
    }
  graph_data.cur_value = num_vals_shift;
  
  XGetWindowAttributes(display,window,&window_attr);
  XClearArea(display,window,0,0,graph_data.g_width,window_attr.height,True);
}

void val_to_pixels(graph_num,which_val,x,y)
int graph_num;
int which_val;
int *x;
int *y;
/* This routine takes care of all the scaling of values.  This routine takes
   the graph number and the raw value and sets the values of x and y to the
   actual point on the window corresponding to those values. */
{
  *x = graph_data.left_pix + x_params.step_size * which_val;
  *y = MAX(graph_list[graph_num]->zero - graph_data.saved_values[which_val][graph_num]*SCALE(graph_num),graph_list[graph_num]->max_pix);
  *y = MIN(*y,graph_list[graph_num]->min_pix);
  /* If the choice of MAX and MIN look odd, just remember that pixel values
     go up as actual values go down; i.e., min_pix is greater than max_pix. */
}

void graph(vals)
int vals[];
/* This routine adds the values that are given to it to the graph. */
{
  int i;
  int x1,y1,x2,y2;

  if (graph_data.g_width == 0)
    {
      /* This means that the window has not been drawn yet */
      return;
    }

  for (i = 0; i < graph_set.num_graphs; i++)
    {
      graph_data.saved_values[graph_data.cur_value][i] = vals[i];
      val_to_pixels(i,graph_data.cur_value,&x2,&y2);
      if (graph_data.cur_value == 0)
	{
	  /* If this is the leftmost point, then just plot it. */
	  XDrawPoint(display,window,gcontext,x2,y2);
	}
      else
	{

	  /* If this is not the left most point, then draw a line connecting
	     this point with the one to its left. */
	  val_to_pixels(i,(graph_data.cur_value-1)%NUM_VALS,&x1,&y1);
	  XDrawLine(display,window,gcontext,x1,y1,x2,y2);
	}
    }
  /* Check to see whether the graph needs to be shifted. */
  if ((graph_data.cur_value+1)*x_params.step_size > graph_data.g_width)
    shift_graph();
  /* update the current value.  This needs to be done even when the graph
     is shifted, as all shift_graph does is move the pointers. */
  graph_data.cur_value++;
}

void refresh_graph()
/* This routine redraws the graph from the values stored in 
   graph_data.saved_values[][]. */
{
  int save_cur_value;

  save_cur_value = graph_data.cur_value;

  for (graph_data.cur_value = 0; graph_data.cur_value <= save_cur_value;)
    graph(graph_data.saved_values[graph_data.cur_value]);
}

void graph_setup(num_graphs,lines_per_label)
int num_graphs;
int lines_per_label;
/* This routine initializes the graph data structures.  It is called from
   main() during the setup period of the program and is not needed after
   that.  This routine must be called before any other calculations are
   done because other routines assume that the value in these data structures
   will be accurate. */
{
  int i;

  graph_set.num_graphs= num_graphs;
  graph_set.lines_per_label = lines_per_label;

  if ((graph_list = (struct graph **)calloc(num_graphs,sizeof(struct graph *))) == NULL)
    {
      perror("Failure getting memory for graph_list");
      exit(1);
    }

  for (i = 0; i < num_graphs; i++)
    {
      if ((graph_list[i] = (struct graph *)malloc(sizeof(struct graph))) == NULL)
	{
	  perror("Failure getting memory for graph_list[i]");
	  exit(1);
	}
    }

  /* Note that saved_values is not initialized until win_setup because 
     the display has to be opened before the number of pixels in the screen
     can be determined. */
}

void redisplay_window()
/* This routine gets called when an exposure event occurs.  It takes care
   of calling routines to refresh the various parts of the window. */

{
/*  XClearArea(display,window,event.xexpose.x,event.xexpose.y,event.xexpose.width,event.xexpose.height,False); */

  /* Since X11 does not currently provide a convenient or efficient way
     of updating only the section of a window that needs to be
     updated (other than the toolkit), this program clears the whole
     window once for each flury of expose events. */
  
  XClearWindow(display,window);
  draw_background();
  if ((graph_data.cur_value+1)*x_params.step_size > graph_data.g_width)
    shift_graph();
  refresh_graph();

  /* Remove any other pending Expose events from the queue to 
   * avoid multiple repaints. */

  while (XCheckTypedEvent(display,Expose,&event));
}

void reset_graph()
/* This routine resets the graph moving the current value pointer to zero
   so that all saved values will be ignored. */
{
  /* Clear the window, signaling an exposure event. */
  XGetWindowAttributes(display,window,&window_attr);
  XClearArea(display,window,0,0,window_attr.width,window_attr.height,True);
  graph_data.cur_value = 0;
}

short check_win_events(ch)
char *ch;  /* This is intended to be a pointer to a single character */
/* This routine returns a constant starting with WE_ defined in window.h.  It
   takes care of handling all window events and is called from 
   main_event_loop() in xperfmon.c when select is awakened from an x event. */
{
  short retcode;
  char buf[5];
  XComposeStatus status;
  int nbytes;

  /* Get the next event if there is one */

  *ch = 0; /* if this is NULL, it is assumed that no key was pressed. */
  if (XPending(display))
    {
      retcode = WE_MORE_EVENTS;

      XNextEvent(display,&event);
      
      /* Only refresh window once; see comments in redisplay_window(). */
      if ((event.type == Expose) && (event.xexpose.count == 0))
	redisplay_window();
      else if (event.type == KeyPress)
	{
	  nbytes = XLookupString(&event,buf,sizeof(buf),0,status);
	  /* Ignore keys that do not correspond to characters */
	  if (nbytes > 0) 
	      *ch = buf[0];
	}
    }
  else
    retcode = WE_NO_MORE_EVENTS;
  return(retcode);
}

void init_graph_list(label,min,max)
char **label;
int min;
int max;
/* This routine is designed to be called exactly graph_set.num_graphs times.
   Each time it is called, it initializes another element of graph_list by
   keeping track of which element it is on via a statically declared index. */
{
  static int index = 0;
 
  if (index < graph_set.num_graphs)
    {
      graph_list[index]->label = label;
      graph_list[index]->min_value = min;
      graph_list[index]->max_value = max;
      index++;
    }
  else
    ; /* Do nothing - the list is already initialized. */
}
