/* $Header: x2ps.c,v 1.3 88/08/31 23:46:44 moraes Exp $ */
#include <stdio.h>
#include <pwd.h>
#include <ctype.h>
#include <sys/param.h>
#include "xpic.h"
#include "tune.h"

/*
 *  This simply reads in xpic format, and emits it (slightly massaged)
 *  with a tagged on PostScript prolog that does most of the work. The
 *  only real work done by this program is for splines which are
 *  converted to bezier curves by some gory math, and arrows which are
 *  computed here by essentially the same algorithm used by xpic
 */
/* To do:
	2. Allow some sort of positioning arguments.
 */

#define MAXFONTS 127
#define FONTDESCFILE "x2ps"

#ifndef MAXHOSTNAMELEN
#define MAXHOSTNAMELEN 64
#endif

#define PROLOGUE	"x2ps.pro"
#define TRAILER		"x2ps.tra"

/* PostScript (well, typesetting) constant */
#define PPI		72.0	/* 72 points per inch */

/* 15 gives a decent margin on LaserWriters */
#define xtransform(x)	((x) - orgX)
#define ytransform(y)	((orgY - (y)))

extern char *malloc();

static int orgX, orgY, picWidth, picHeight;

static int x, y;

static char *halign_words[] = {
	"ce", "lj", "rj"
};

static int arrow_type = -1;
static int curStyle = -1;
static char *curFont = NULL;
static int curSize = -1;
static int curWidth = -1;
static int halign, valign;
static int showPage = TRUE;
static double hoffset = 0;
static double voffset = 0;
#ifdef DEBUG
static char *prologueFile = PROLOGUE;
static char *trailerFile = TRAILER;
#else DEBUG
static char *prologueFile = NULL;
static char *trailerFile = NULL;
#endif DEBUG

static char *progname;
extern int optind;
extern char *optarg;

static void ChangeThickness(t)
{
	if (curWidth != t) {
		curWidth = t;
		if (curWidth == 0)
			fprintf(outFile, "0.5 W\n");
		else
			fprintf(outFile, "%d W\n", curWidth);
	}
}

	
static void ChangeStyle(style)
{
	if ((style < 0) || (style >= NSTYLES)) {
		fprintf(stderr, "Unknown style - %d\n", style);
		style = 0;
	}
	if (style != curStyle) {
		fprintf(outFile, "%d D\n", style);
		curStyle = style;
	}
}

static void ChangeFont(font)
char *font;
{
	extern char *HashSearch();
	char *psfont = HashSearch(font);
	if (psfont == NULL) {
		fprintf(stderr, "Unavailable font %s\n", font);
		return;
	}
	if (psfont != curFont) {
		fprintf(outFile, "/%s F\n", psfont);
		curFont = psfont;
	}
}

static void ScaleFont(size)
{
	if (size <= 0) {
		fprintf(stderr, "Incorrect font size %d\n", size);
		size = 10;
	}
	if (size != curSize) {
		fprintf(outFile, "%d S\n", size);
		curSize = size;
	}
}


static void MakeBox(x1, y1, x2, y2, attr)
{
	ChangeStyle(getlinestyle(attr));
	x = xtransform(x1);
	y = ytransform(y1);
	fprintf(outFile, "%d %d ", x, y);
	x = xtransform(x2);
	y = ytransform(y2);
	fprintf(outFile, "%d %d b\n", x, y);
}


static void MakeEllipse(xc, yc, xr, yr, attr)
{
	ChangeStyle(getlinestyle(attr));
	x = xtransform(xc);
	y = ytransform(yc);
	fprintf(outFile, "%d %d %d %d e\n", xr, yr, x, y);
}


/*
 *  print out a string, escaping '(' and ')' with a \ - we don't escape \, to
 *  give people a mechanism for sending strange numbers to the lw
 */
static void PrintText(s)
char *s;
{
	register char *cp = s;

	for(; *cp != 0; cp++) {
		if (isascii(*cp) && isprint(*cp)) {
 			if (*cp != '(' && *cp != ')')
		 		(void) fputc(*cp, outFile);
			else
				(void) fprintf(outFile, "\\%c", *cp);
		} else {
			/* Ok - but we're only printing the lower byte! */
			(void) fprintf(outFile, "\\%o", *cp & 0xff);
		} 
	}
}
			
static void MakeText(s, len, font, size, attr, xc, yc)
char *s;
{
	ChangeFont(font);
	ScaleFont(size);
	halign = gettext_halign(attr);
	valign = gettext_valign(attr);
	if ((valign < 0) || (valign >= 3)) {
		fprintf(stderr, "Incorrect vert alignment %d\n", valign);
		valign = 0;
	}
	if ((halign < 0) || (halign >= 3)) {
		fprintf(stderr, "Incorrect horiz alignment %d\n", halign);
		halign = 0;
	}
	x = xtransform(xc);
	y = ytransform(yc);
	fprintf(outFile, "%d %d m\n(", x, y);
	PrintText(s);
	fprintf(outFile, ") %s %d t\n", halign_words[halign], valign);
}
	

#define PI 			3.14159265358979323846264338327950288419716939937511
#define ARROWSIZE	10.
#define ARROWANGLE	0.46

#define round(x)	((int) ((x) + 0.5))

/*
 *  Draws an arrowhead on the end of the line going from x1, y1 to x2, y2.
 *  It calculates the direction, and draws two lines ARROWSIZE long at
 *  angles of ARROWANGLE to the line.
 */
static void DrawArrow (x1, y1, x2, y2) 
int x1, y1, x2, y2; 
{
	double dir, dx, dy, atan2(), cos(), sin();

	dx = (double) (x2 - x1);
	dy = (double) (y2 - y1);
	if (dx == 0. && dy == 0.) 
		return;
	dir = atan2(dy, dx) + PI;

	/* Arrows should be solid lines, say the critics */
	fprintf(outFile, "0 D\n");
	x = x2 + round(ARROWSIZE * cos (dir - ARROWANGLE));
	y = y2 + round(ARROWSIZE * sin (dir - ARROWANGLE));
	fprintf(outFile, "%d %d m\n", x, y);
	fprintf(outFile, "%d %d l\n", x2, y2);
	x = x2 + round(ARROWSIZE * cos (dir + ARROWANGLE));
	y = y2 + round(ARROWSIZE * sin (dir + ARROWANGLE));
	fprintf(outFile, "%d %d l\n", x, y);
	fprintf(outFile, "%d D\n", curStyle);
	fprintf(outFile, "%d %d m\n", x2, y2);
}

static int oldX, oldY;
static int lastX, lastY;
static BOOL firstpoint = FALSE;

static void StartLine(xc, yc, n, attr)
{
	int x, y;
	
	arrow_type = getlinearrow(attr);
	if ((arrow_type < 0) || (arrow_type >= 4)) {
		fprintf(stderr, "Unknown arrow type %d\n", arrow_type);
		arrow_type = 0;
	}
	ChangeStyle(getlinestyle(attr));
	x = xtransform(xc);
	y = ytransform(yc);
	fprintf(outFile, "%d %d m\n", x, y);
	firstpoint=TRUE;
	oldX = lastX = x;
	oldY = lastY = y;
}


static void NextLinePointAt(xc, yc)
{
	int x, y;
	
	x = xtransform(xc);
	y = ytransform(yc);
	if (firstpoint) {
		if (arrow_type & 0x1) DrawArrow(x, y, oldX, oldY);
		firstpoint = FALSE;
	}
	fprintf(outFile, "%d %d l\n", x, y);
	oldX = lastX;
	oldY = lastY;
	lastX = x;
	lastY = y;
}


static void EndLine()
{
	if (arrow_type & 0x2) DrawArrow(oldX, oldY, lastX, lastY);
	fprintf(outFile, "x\n");
}


/*
 *  This takes a set of points for a quadratic B-Spline (passing thru'
 *  all the points and lying within their convex hull, and produces a
 *  PostScript curve using Bezier segments. Mark Moraes, 1987.
 */
StartSpline(xc, yc, n, attr)
{
	arrow_type = getlinearrow(attr);
	if ((arrow_type < 0) || (arrow_type >= 4)) {
		fprintf(stderr, "Unknown arrow type %d\n", arrow_type);
		arrow_type = 0;
	}
	ChangeStyle(getlinestyle(attr));
	x = xtransform(xc);
	y = ytransform(yc);
	fprintf(outFile, "%d %d m\n", x, y);
	firstpoint=TRUE;
	oldX = lastX = x;
	oldY = lastY = y;
}


static void NextSplinePointAt(xc, yc, lastpoint)
{
	int x, y;
	int dX, dY, dX2, dY2;
	
	x = xtransform(xc);
	y = ytransform(yc);
	if (firstpoint) {
		if (arrow_type & 0x1) DrawArrow(x, y, oldX, oldY);
		firstpoint = FALSE;
	}
	dX = x - oldX;
	dY = y - oldY;
	if (lastpoint) {
		dX += x - lastX;
		dY += y - lastY;
	}
	dX2 = lastX - oldX;
	dY2 = lastY - oldY;
	fprintf(outFile, "%d %d ", dX2 / 3, dY2 / 3);
	fprintf(outFile, "%d %d ", (2 * dX2 + dX) / 6, (2 * dY2 + dY) / 6);
	fprintf(outFile, "%d %d s\n", dX / 2, dY / 2);
	oldX = lastX;
	oldY = lastY;
	lastX = x;
	lastY = y;
}


void EndSpline()
{
	if (arrow_type & 0x2) DrawArrow(oldX, oldY, lastX, lastY);
	fprintf(outFile, "x\n");
}

	 
/*
 *  Puts a PostScript prologue from the file 'prologue' to stdout
 *  preceded by a header comment according to Adobe's Structuring
 *  Conventions. 'filename' is the name of the xpic file, and the four
 *  values denote the bounding box of the figure
 */
WritePrologue(prologue, filename, llx, lly, urx, ury)
char *prologue;
char *filename;
double llx, lly, urx, ury;
{
	FILE *f;
	char hostname[MAXHOSTNAMELEN];
	struct passwd *pw;
	long clock = time(0);
	char s[BUFSIZ];

	if ((f = fopen(prologue, "r")) == NULL) {
		fprintf(stderr, "Cannot open prologue file %s\n", prologue);
		exit(-1);
	}

	/* The comments section - according to Appendix C in the Red Book */
	fprintf(outFile, "%%!\n");
	gethostname(hostname, sizeof(hostname));
	if ((pw = getpwuid(getuid())) == NULL) 
		fprintf(stderr, "Cannot get passwd info!");
	else
		fprintf(outFile, "%%%%Creator: %s@%s (%s)\n", pw->pw_name,
 		 hostname, pw->pw_gecos);
	if (filename != NULL)
		fprintf(outFile, "%%%%Title: %s (xpic)\n", filename);
	fprintf(outFile, "%%%%CreationDate: %s", ctime(&clock));
	/* Only one page pictures allowed */
	fprintf(outFile, "%%%%Pages: 1\n");
#if 0
	/* !! Ouch. */
	fprintf(outFile, "%%%%DocumentFonts: (atend)\n");
#endif
	fprintf(outFile, "%%%%BoundingBox: %g %g %g %g\n", llx, lly, urx, ury);
	fprintf(outFile, "%%\t(in inches) at %g %g, width %g, height %g\n",
	 llx / PPI, lly / PPI, (urx - llx) / PPI, (ury - lly) / PPI);
	fprintf(outFile, "%%%%EndComments\n");

	/* Now copy the prologue - stdio should be fast enough */
	while(fgets(s, sizeof(s), f) != NULL)
		fputs(s, outFile);

	fclose(f);

	fprintf(outFile, "%%%%EndProlog\n");
}

/*
 *  Put a PostScript trailer on stdout from the file 'trailer' as per
 *  the Structuring Conventions. 
 */
WriteTrailer(trailer, showpage)
char *trailer;
{
	FILE *f;
	char s[BUFSIZ];

	if ((f = fopen(trailer, "r")) == NULL) {
		fprintf(stderr, "Cannot open trailer file %s\n", trailer);
		exit(-1);
	}

	fprintf(outFile, "%%%%Trailer\n");
	if (showpage)
		fprintf(outFile, "showpage\n");

	/* Now copy the trailer - stdio should be fast enough */
	while(fgets(s, sizeof(s), f) != NULL)
		fputs(s, outFile);

	fclose(f);
}



/*
 *  Read in a file output by xpic, and emit the appropriate PostScript text
 */
static void convert(land, scale, infilename)
double scale;
char *infilename;
{
	int type;
	int xc, yc, xr, yr, len, attr;
	int size;
	int x1, y1, x2, y2;
	char *s;
	char font[MAXSTR];
	int c, i, n;
	int err, nf, gs;
	int x, y;
	int xx, yy;
	int num, thickness;
	int xoff, yoff;
	double xscale = scale;	/* Allow for grid spacing as well */

#define INPERR 2
#define INPEOF 3

	err = 0;
#ifdef MAGIC
	/* Check for the magic header that the new xpic puts out */
	if ((c = fgetc(inFile)) == EOF) {
		fprintf(stderr, "Incorrect input format");
		return;
	}
	ungetc(c, inFile);
	if (c == '#') {
		/* Magic header - ignore */
		fscanf(inFile, "%*[^\n]");
	}
#endif MAGIC
	/* Read in (and ignore) the gel bounding box */
	(void) fscanf(inFile, " %d %d %d %d %d", &x1, &y1, &x2, &y2, &gs);
	/* Set up constants for scaling, translation etc. */
	orgX = x1;
	/* Take advantage of the already mirrored coordinates if rotating */
	orgY = (land) ? y1 : y2;
	picWidth = x2 - x1;
	picHeight = y2 - y1;
	/* Convert x and y offsets to xpic units */
	xoff = hoffset * 10.0 * gs;
	yoff = voffset * 10.0 * gs;
	orgX -= xoff;
	orgY += yoff;
	/*
	 *  We need to tell psfig the correct bounding box in points (72
	 *  points = 1 inch) and xpic has gridspacing * 10 pixels per
	 *  inch, hence the xscale bit. We also have a possibly
	 *  user-specified scale to deal with. The actual scaling is
	 *  done in PostScript - we need to calculate this here only for
	 *  the bounding box.
	 */
	xscale *= PPI / (double) (gs * 10.0);
	if (land) {
		double w = xscale * (x2 - x1);
		double h = xscale * (y2 - y1);
		WritePrologue(prologueFile, infilename,
 		 (double) (hoffset * PPI), (double) (voffset * PPI),
		 (double) (xscale * picHeight), (double) (xscale * picWidth));
		fprintf(outFile, "land\n");
	} else {
		WritePrologue(prologueFile, infilename,
 		 (double) (hoffset * PPI), (double) (voffset * PPI),
		 (double) (xscale * picWidth), (double) (xscale * picHeight));
	}
	fprintf(outFile, "%lg %lg scale\n", scale, scale);
	fprintf(outFile, "%d ss\n", gs*10);
	/* Read in the actual picture */
	do {
		if ((nf = fscanf(inFile, " %d", &type)) != 1) {
			err = INPEOF;
			break;
		}
		nf = fscanf(inFile, " %d %d %d %d %d %x %d", &num, &x1, &y1, 
		 &x2, &y2, &attr, &thickness);
		if (nf != 7) {
			err = INPERR;
			break;
		}
		ChangeThickness(thickness);
		switch (type) {
		case BOX:
			MakeBox(x1, y1,x2, y2, attr);
			break;
		case ELLIPSE:
			nf = fscanf(inFile, " %d %d %d %d", &xc, &yc, &xr, &yr) ;
			if (nf != 4) {
				err = INPERR;
				break;
			}
			MakeEllipse(xc, yc, xr, yr, attr);
			break;
		case CIRCLE:
			nf = fscanf(inFile, " %d %d %d", &xc, &yc, &xr);
			if (nf != 3) {
				err = INPERR;
				break;
			}
			MakeEllipse(xc, yc, xr, xr, attr);
			break;
		case TEXT:
			nf = fscanf(inFile, " %d %d %d %s %d", &xc, &yc, &len, font, 
			 &size);
			if (nf != 5) {
				err = INPERR;
				break;
			}
			/*
			 *  For backward compatibility with the bad old days. The
			 *  old convention of storing font information was really
			 *  ugly - a font number from 0-3, (corresponding to Roman,
			 *  Bolld, Italic, Special) and a size from 0-9
			 *  (corresponding to point sizes 6 - 24)
			 */
			if (font[1] == '\0') {
				int oldfontconvention = TRUE;
				
				switch (font[0]) {
				case '0':
					strcpy(font, "Roman");
					break;
				case '1':
					strcpy(font, "Bold");
					break;
				case '2':
					strcpy(font, "Italic");
					break;
				case '3':
					strcpy(font, "Special");
					break;
				default:
					/* Must a new font with a one letter name. Eeep! */
					oldfontconvention = FALSE;
				}
				if (oldfontconvention)
					/* Convert to pointsize */
					size = size * 2 + 6;
			}
			/* Go to the next line */
			while ((c = fgetc(inFile)) != '\n' && c != EOF)
				;
			if (c == EOF) {
				err = INPERR;
				break;
			}
			if (( s = malloc(len + 2)) == NULL) {
				fprintf("No more memory for text string");
				break;
			}
			if (fgets(s, len + 1, inFile) == NULL) {
				free(s);
				err = INPERR;
				break;
			}
			s[len] = '\0';
			MakeText(s, len, font, size, attr, xc, yc);
			free(s);
			break;
		case LINE:
			if (fscanf(inFile, " %d %d %d", &n, &xc, &yc) != 3) {
				err = INPERR;
				break;
			}
			StartLine(xc, yc, n, attr);
			for (i = 1; i < n; i++) {
				if (fscanf(inFile, " %d %d", &xc, &yc) != 2) {
					err = INPERR;
					break;
				}
				NextLinePointAt(xc, yc);
			}
			if (err != INPERR)
				EndLine();
			break;
		case SPLINE:
			if (fscanf(inFile, " %d %d %d", &n, &xc, &yc) != 3) {
				err = INPERR;
				break;
			}
			StartSpline(xc, yc, n, attr);
			for (i = 1; i < n; i++) {
				if (fscanf(inFile, " %d %d", &xc, &yc) != 2) {
					err = INPERR;
					break;
				}
				NextSplinePointAt(xc, yc, (i == n - 1));
			}
			if (err != INPERR)
				EndSpline();
			break;
		}
	} while (err == 0);
	WriteTrailer(trailerFile, showPage);
	if (err == INPERR)
		fprintf(stderr, "Incorrect input format");

	return;

#undef INPEOF
#undef INPERR
}

usage(s)
char *s;
{
	fprintf(stderr, "\
Usage: %s [-r] [-s scale] [-p prologuefile] [-t trailerfile] \n\
	[-h hoffset] [-v voffset] [-x] [-f maxfonts] [filename]\n", s);
	exit(-1);
}


/* Font tables are of the form
	xpic-font-name	PostScript-font-name
 */
readfonttable(file)
char *file;
{
	FILE *fp;
	char xfontname[MAXSTR], psfontname[MAXSTR];
	char *s1, *s2;
	extern char *strsave();
	
	if ((fp = fopen(file, "r")) == NULL)
		return;
	while(fscanf(fp, " %s %s", xfontname, psfontname) == 2) {
		if ((s1 = strsave(xfontname)) && (s2 = strsave(psfontname)))
			HashInsert(s1, s2);
		else {
			fprintf(stderr, "Out of memory\n");
			exit(-1);
		}
	}
	fclose(fp);
}


main(argc, argv)
int argc;
char **argv;
{
	int land = FALSE;
	double scale = 1.0;
	double atof();
	int c;
	int nfonts = MAXFONTS;
	char fontfile[MAXSTR];
	char *hdir;
	char *getenv();
	
	progname = argv[0];
	inFile = stdin;
	outFile = stdout;
	while((c = getopt(argc, argv, "rs:t:p:xf:h:v:")) != EOF) {
		switch (c) {
		case 'r':
			land = TRUE;
			break;
		case 's':
			scale = atof(optarg);
			break;
		case 'p':
			prologueFile = optarg;
			break;
		case 't':
			trailerFile = optarg;
			break;
		case 'x':
			/* Pronounce this 'Ech' to rhyme with TeX! */
			showPage = !showPage;
			break;
		case 'f':
			nfonts = atoi(optarg);
			break;
		case 'h':
			hoffset = atof(optarg);
			break;
		case 'v':
			voffset = atof(optarg);
			break;
		case '?':
			usage(progname);
			break;
		}
	}

	HashInit(nfonts);
	sprintf(fontfile, "%s/fontdesc/%s", LIBDIR, FONTDESCFILE);
	readfonttable(fontfile);
	if (hdir = getenv("HOME")) {
		sprintf(fontfile, "%s/.%s", hdir, FONTDESCFILE);
		readfonttable(fontfile);
	}
	if (!prologueFile) {
		prologueFile = malloc(sizeof(PROLOGUE) + sizeof(LIBDIR));
		sprintf(prologueFile, "%s/%s", LIBDIR, PROLOGUE);
	}

	if (!trailerFile) {
		trailerFile = malloc(sizeof(TRAILER) + sizeof(LIBDIR));
		sprintf(trailerFile, "%s/%s", LIBDIR, TRAILER);
	}

	/* Must have some files */
	if (optind >= argc) {
		convert(land, scale, "stdin");
	} else {
		while (optind < argc) {
			if ((inFile = fopen(argv[optind], "r")) == NULL) {
				fprintf(stderr, "Can't open %s for reading\n", 
				 argv[optind]);
				break;
			}
			outFile = stdout;
			convert(land, scale, argv[optind]);
			optind++;
		}
	}
}
