/*
 *	SunTroff -  A program to display the output of Device Independent
 *		Troff as a window on a Sun Workstation.
 *
 *	Suntroff - Code to interface the rather generic user interface
 *		provided by windows.c and the low level troff parsing 
 *		and drawing routines.
 *
 *	Authors - The original version of this program was written by
 *			Richard L. Hyde (Purdue)
 *			David Slattengren (Berkeley)
 *		It was nearly all rewritten, cleaned up and a more elegant
 *		user interface installed by
 *			Malcolm Slaney (Schlumberger Palo Alto Research)
 *
 *	Legalese -  This command was developed as an independent project
 *		to satisfy a need of the author.  This program may contain
 *		bugs and the user is cautioned to independently verify that
 *		the program is suitable for the user's intended purpose.
 *		The program is made available on an ``as is'' basis with
 *		all faults and without any implied or expressed warranties
 *		or support from either the author, Malcolm Slaney, or
 *		Schlumberger Palo Alto Research Laboratory.
 *
 *		I am putting this program in the public domain.  You are
 *		free to use it as you wish.  In return I ask two things.
 *		First, that you do not remove the names of the authors
 *		from this work.  Secondly, if you make changes or
 *		improvements to this program that you pass these back to
 *		the author so that everybody can benefit from the
 *		improvements.
 *
 *					Malcolm Slaney  (December 1986)
 *					Schlumberger Palo Alto Research
 *					3340 Hillview Avenue
 *					Palo Alto, CA 94304
 *					(415) 496-4669
 *					spar!malcolm@decwrl.dec.com
 *					malcolm@ecn.purdue.edu
 *					malcolm@spar.slb.com (Someday)
 */

#include	"suntroff.h"
#ifdef SUNTOOLS
#include	<suntool/sunview.h>
#endif SUNTOOLS
#define	MAXPAGES	500		        /* Pages to remember */

static FILE *CurrentFilePointer;		/* Current input file. */
static FILE *RealBufferPointer;	    		/* Buffer file pointer */
static FILE *RealFilePointer;			/* Real File Pointer */

static long PagePlace[MAXPAGES];		/* Remembered ftell
						   positions */


					/* TYPESETTER ENVIRONMENT VARIABLES */
int	size = 1;			/* Current Font Size (internal
					   pstable index) */
int	font = 1;			/* Current Font (internal font
					   number */
int	linmod;				/* Line Style....unused. */

int	linethickness;			/* unused */

int	HorizontalPosition;		/* Horizontal Position (Device
					   Coordinates) */
int	VerticalPosition;		/* Vertical Position on the page
					   (Device Coordinates) */

int	NextPage;			/* Next page to be parsed in the input
					   file. */
char	DeviceName[11];			/* Output produced for this device */

int	DeviceResolution;		/* Output produced at this resolution*/

struct FontBitStruct *CurrentFont = 0;	/* Pointer to the current font
					   information. */
int	DisplayOutput = 1;		/* Don't display output (just parse)
					   when this flag is zero. */
int	LineNumber = 0;			/* Input file line number */

#ifdef SUNTOOLS
char	*DefaultTitle = "SUNTROFF (3.0) ";
#else
char	*DefaultTitle = "XTROFF (3.0) ";
#endif

int	CurrentPage = 0;		/* Current Page in Input File */

int	LastPage = 0;			/* Last Page of input file */

ShowPage(PageNumber){
	int	i;

	if (!CurrentFilePointer){
		warning("No file open for input.");
		return(0);
	}

	if (PageNumber < 1)
		PageNumber = 1;

	if (LastPage && PageNumber > LastPage){
		warning("There are only %d pages in\nthis document.",
		      LastPage);
		return(CurrentPage);
	}

	if (PageNumber < MAXPAGES){
		if (PagePlace[PageNumber]){
			FileSeek(PagePlace[PageNumber]);
			CurrentPage = PageNumber;
		} else {
			for (i=PageNumber;i>0 && !PagePlace[i];i--)
				;
			FileSeek(PagePlace[i]);
			
			SetTitleBar("Skipping",i);
			DisplayOutput = 0;
			while (!feof(CurrentFilePointer) &&
			       (i = ParseInput()) !=
				PageNumber){
					CurrentPage = i;
					SetTitleBar("Skipping",i);
			       }
			CurrentPage = i;
		}
		DisplayOutput = 1;
		ClearPage();
		SetTitleBar("Rasterizing",PageNumber);
		ParseInput();
		RefreshPage();
	}

	if (LastPage && PageNumber > LastPage){
		warning("There are only %d pages in\nthis document.",
		      LastPage);
		SetTitleBar("Displaying",CurrentPage);
		return(CurrentPage);
	}

	SetTitleBar("Displaying",PageNumber);
	return(PageNumber);
}


ClearPagePositions(){
	int	i;

	for (i=0;i<MAXPAGES;i++){
		PagePlace[i] = 0;
	}
}

#ifdef SUNTOOLS
static short IconImage[] = {
#include	"ditroff.icon"
}; 
DEFINE_ICON_FROM_IMAGE(DitroffIcon,IconImage);

InitializeApplication(Frame,Canvas)
Window Frame, Canvas;
{
	window_set(Frame,
		   FRAME_ICON,&DitroffIcon,
		   0);
	SetTitleBar("Initializing",0);
}
#endif SUNTOOLS
	

InitializeFile(RealFile, TempFile)
FILE	*RealFile, *TempFile;
{
	CurrentFilePointer = RealFilePointer = RealFile;
	RealBufferPointer = TempFile;
	FileSeek(0L);
	ClearPagePositions();
	CurrentPage = LastPage = 0;
}

HorizontalMove(delta)
int	delta;
{
	HorizontalPosition += delta;
}

HorizontalGoto(NewPosition)
int	NewPosition;
{
	HorizontalPosition = NewPosition;
}

VerticalMove(delta)
int	delta;
{
	VerticalPosition += delta;
}

VerticalGoto(NewPosition)
int	NewPosition;
{
	VerticalPosition = NewPosition;
}

PutCharacter(c)
int	c;
{
	int	OldFont, i, cwidth;
	char	**AsciiTable, *SpecialCharacterName, **CodeTable;
	short	*SpecialCharacterNumber;
	struct Font **FontInformation;
	struct dev *Device;

	AsciiTable = OutputTypesetter->AsciiTable;
	SpecialCharacterNumber = OutputTypesetter->SpecialCharacterNumber;
	SpecialCharacterName = OutputTypesetter->SpecialCharacterName;
	CodeTable = OutputTypesetter->CodeTable;
	FontInformation = OutputTypesetter->FontInformation;
	Device = &OutputTypesetter->Device;


	if (!DisplayOutput){
		return;
	}

	if (c == ' ')				/* Spaces are easy */
		return;
	  
	c -= 32;
	if (c < 0 || c > 128-32+Device->nchtab){
		warning(
		      "Bad character (%d) passed to PutCharacter at line %d.\n"
		      ,c+32,LineNumber);
	}
	
	OldFont = font;

	if (AsciiTable[font][c] == 0){			/* If its not in the
							   table then look for
							   it in the other 
							   fonts. */
		for (i=0;i<=Device->nfonts;i++){
			if (!FontInformation)
				continue;
#ifdef	MAYBENOT			
			if (!FontInformation[i]->specfont)
				continue;
#endif	MAYBENOT			
			if (!AsciiTable[i])
				continue;
			if (AsciiTable[i][c])
				break;
		}
		if (i <= Device->nfonts){
			font = i;
		        VirtualLoadFont(FontInformation[i]->namefont, size);
		} else {
			warning(
			      "Can't find (%s)%d in %s character table.\n",
			      &SpecialCharacterName[SpecialCharacterNumber[
							     c+32-128]],
			      c+32,
			      OutputTypesetter->Name);
			return;
		}
	}

	LoadFontBits();

#ifndef NOADJUST
	/*
	 *  A hack to help centre the X11 font in the space of the laser
	 *  printer font so it looks much nicer. Taken from David
	 *  Blythe's xproof previewer for X10, at the University of
	 *  Toronto. It might work in Suntools as well - I haven't
	 *  tried. - moraes
	 */
	cwidth = UserTypesetter->WidthTable[font]
		[UserTypesetter->AsciiTable[font][c]&0xff]&0xff;
	cwidth = (cwidth * UserTypesetter->PointSizeTable[size - 1]
	 + UserTypesetter->Device.unitwidth/2)
	 / UserTypesetter->Device.unitwidth;
#endif
	
	DrawCharacter(HorizontalPosition,VerticalPosition,
		      CodeTable[font][AsciiTable[font][c]&0xff], cwidth);
	SetFont(OldFont);
}

PutSpecialCharacter(CharacterName)
char	*CharacterName;
{
	int 	i;
	struct dev *Device;
	short	*SpecialCharacterNumber;
	char	*SpecialCharacterName;

	Device = &OutputTypesetter->Device;
	SpecialCharacterNumber = OutputTypesetter->SpecialCharacterNumber;
	SpecialCharacterName = OutputTypesetter->SpecialCharacterName;

	if (!DisplayOutput){
		return;
	}

	for (i=0;i<Device->nchtab;i++){
		if (strcmp(&SpecialCharacterName[SpecialCharacterNumber[i]],
			   CharacterName) == 0)
			break;
	}
	if (i < Device->nchtab){
		PutCharacter(i+128);
	} else {
		warning(
		  "Couldn't find special character %s in %s character list.\n",
		  CharacterName, OutputTypesetter->Name);



	}
}

	
PrintDocument(ActualFileName,Printer)
char	*ActualFileName, *Printer;
{
	char	*Command[BUFSIZ];	
	int	i, SavedPageNumber;

	SavedPageNumber = CurrentPage;		/* Save this, just in case */
	SaveTitleBar();
	if (!LastPage && RealBufferPointer != RealFilePointer){
		for (i=1; i < MAXPAGES; i++){
			if (PagePlace[i])
				CurrentPage = i;
		}
		FileSeek(PagePlace[CurrentPage]);
		DisplayOutput = 0;		/* Now flush the rest of input
						   */
		while (!LastPage || !feof(RealFilePointer)){
			SetTitleBar("Flushing", CurrentPage);
			CurrentPage = ParseInput();
		}
	}
	
	SetTitleBar("Printing Document", -1);
	fflush(RealBufferPointer);

	sprintf(Command,"%s%s %s",LPRCOMMAND,Printer,
		ActualFileName);
	system(Command);
	RestoreTitleBar();
	CurrentPage = SavedPageNumber;
}


PrintPage(PageNumber,Printer)
int	PageNumber;
char	*Printer;
{
	char	FileName[BUFSIZ], Command[BUFSIZ];
	FILE	*fp;

	strcpy(FileName,"/tmp/suntroff.XXXXXX");
	mktemp(FileName);
	
	fp = fopen(FileName,"w");
	if (!fp){
		warning("Can't open %s for writing page image.\n",
		      FileName);
		return;
	}

	OutputPage(0,PagePlace[1],CurrentFilePointer,fp);
	OutputPage(PagePlace[PageNumber],PagePlace[PageNumber+1],
		   CurrentFilePointer,fp);
	fprintf(fp, "\n");
	fprintf(fp, "x trailer\n");
	fprintf(fp, "x stop\n");
	fclose(fp);
	
	sprintf(Command,"%s%s -n %s",LPRCOMMAND,Printer,FileName);
	system(Command);
        unlink(FileName);

}

OutputPage(Start,End,Input,Output)
long	Start, End;
FILE	*Input, *Output;
{
	char	Buffer[BUFSIZ],  c;

	if (End != 0 && Start > End){
		fatal("PrintPage: starting offset (%d) is less than\nending offset (%d)\n",Start,End);
		return;
	}

	FileSeek(Start);

	do {
		c = GetChar();
		if (c != EOF){
			putc(c, Output);
		}
		Start ++;
	} while (c != EOF && (End == 0 || Start < End));
	
}
	

SearchFile(String,PageNumber,Direction)
int	PageNumber, Direction;
char	*String;
{
	PageNumber += Direction;		/* Skip Current Page */

	if (PageNumber <= 0 || (LastPage && PageNumber > PageNumber) ||
	    !String || String[0] == NULL){
		return(0);
	}

	if (PagePlace[PageNumber] == 0){
		warning("Can't find the current page while searching.");
		return(0);
	}

	FileSeek(PagePlace[PageNumber]);
	for (;PageNumber>0 ;PageNumber += Direction){
		if (LastPage && PageNumber > LastPage){
			return(0);
		}
		if (feof(CurrentFilePointer)){
			return(0);
		}
		if (Direction < 0){
			FileSeek(PagePlace[PageNumber]);
		}

		SetTitleBar("Searching",PageNumber);
		if (SearchPage(String)) {
			return(PageNumber);
		}
	}
	return(0);
}

SearchPage(String)
char	*String;
{
	char	*StringP = String;
	int	c, NextPage;

	while ((c = GetChar()) != EOF){
		switch(c){
		case ' ':
		case 0:
		case '{':
		case '}':
		case '\n':
			break;
		case '0': case '1': case '2': case '3': case '4':
		case '5': case '6': case '7': case '8': case '9':
			GetChar();
		case 'c':
			c = GetChar();
			if (c == *StringP){
				StringP++;
				if (*StringP == 0){
					return(1);
				}
			} else {
				StringP = String;
			}
			break;
		case 'C':
			GetChar();
			GetChar();
			StringP = String;
			break;
		case 'D':
		case 'x':
		case '#':
			do {
				c = GetChar();
			} while (c != '\n' && c != EOF);
			StringP = String;
			break;
		case 'w':
			if (*StringP == ' '){
				*StringP++;
				if (*StringP == 0){
					return(1);
				}
			} else {
				StringP = String;
			}
			break;
		case 'n':
			if (*StringP == ' '){
				*StringP++;
				if (*StringP == 0){
					return(1);
				} 
			} else {
				StringP = String;
			}
			GetNumber();
			GetNumber();
			break;
		case 's':
		case 'f':
		case 'H':
		case 'V':
		case 'h':
		case 'v':
			GetNumber();
			break;
		case 'p':
			NextPage = GetNumber();
			RememberPagePosition(CurrentFilePointer,NextPage);
			return(0);
		default:
			warning("Unknown input character %c(%d)\n",
			      c,c);
			break;
		}
	}
	return(0);
}

static	UnreadCharacter = 0;

RememberPagePosition(PageNumber)
int	PageNumber;
{
	extern long ftell();

	if (PageNumber > 0 && PageNumber < MAXPAGES){
		if (!PagePlace[PageNumber]){
			if (UnreadCharacter){
				PagePlace[PageNumber] =
						ftell(RealBufferPointer)-1;
				UnreadCharacter = 0;
#ifdef	SEEK					
		printf("Remembering page %d at %x with unread character.\n", PageNumber, PagePlace[PageNumber]);
#endif	SEEK			
			} else {
				PagePlace[PageNumber] =
						ftell(RealBufferPointer);
#ifdef	SEEK					
				printf("Remembering page %d at %x.\n",
				    PageNumber, PagePlace[PageNumber]);
#endif	SEEK				  
			}
		}
	}
}

FileSeek(Position)
long	Position;
{
	UnreadCharacter = 0;
	CurrentFilePointer = RealBufferPointer;
	fseek(CurrentFilePointer,Position,0);
#ifdef	SEEK		
	printf("Seeking to %x of real buffer.\n", Position);
#endif	SEEK
}

GetChar(){
	int	i;
	
	if (UnreadCharacter){
		i = UnreadCharacter;
		UnreadCharacter = 0;
		return(i);
	}

	i = getc(CurrentFilePointer);
	if (CurrentFilePointer != RealBufferPointer){
		putc(i, RealBufferPointer);
	}			
	
	if (i == EOF){
		if (RealFilePointer != RealBufferPointer){
			if (CurrentFilePointer == RealBufferPointer){
				CurrentFilePointer = RealFilePointer;
				i = GetChar();
			}
		}
	}
	
	return(i);
}

UnGetChar(c)
int	c;
{
	if (UnreadCharacter){
		fatal("Can't UnGetChar more than one character.\n");
	}

	UnreadCharacter = c;
}

char *
GetLine(Buffer, Length)
char	*Buffer;
int	Length;
{
	int 	i = 0, c;
	char	*p = Buffer;
	
	Length--;			    /* Save room for final NULL */
	
	while (i < Length && (c = GetChar()) != EOF && c != '\n'){
		if (p)
			*p++ = c;
	}
	if (c == '\n' && p){		    /* Retain the newline like fgets */
		*p++ = c;
	}
	if (c == '\n')
		UnGetChar(c);
	

	if (p)	
		*p = NULL;
	return(Buffer);
} 

char *
GetWord(Buffer, Length)
char	*Buffer;
int	Length;
{
	int 	i = 0, c;
	char	*p = Buffer;
	
	Length--;			    /* Save room for final NULL */
	
	while ((c = GetChar()) != EOF && isspace(c));
	if (c != EOF){
		UnGetChar(c);
	}

	while (i < Length && (c = GetChar()) != EOF && !isspace(c)){
		if (p)
			*p++ = c;
	}
	if (c != EOF)
		UnGetChar(c);
	
	if (p)
		*p = NULL;
	return(Buffer);
} 

GetNumber(){
	int	i = 0,  c;

	while ((c = GetChar()) != EOF && isspace(c));

	if (c != EOF){
		UnGetChar(c);
	}

	while ((c = GetChar()) != EOF && isdigit(c)){
		i = i*10 + c - '0';
	}

	if (c != EOF)
		UnGetChar(c);
	return (i);
}
	
	    
