/* 
 * pexpong.c - The PEX xpong ae routines.
 * 
 * Copyright 1988
 * Center for Information Technology Integration (CITI)
 * Information Technology Division
 * University of Michigan
 * Ann Arbor, Michigan
 *
 *                         All Rights Reserved
 * 
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose and without fee is hereby
 * granted, provided that the above copyright notice appear in all
 * copies and that both that copyright notice and this permission
 * notice appear in supporting documentation, and that the names of
 * CITI or THE UNIVERSITY OF MICHIGAN not be used in advertising or
 * publicity pertaining to distribution of the software without
 * specific, written prior permission.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS." CITI AND THE UNIVERSITY OF
 * MICHIGAN DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN
 * NO EVENT SHALL CITI OR THE UNIVERSITY OF MICHIGAN BE LIABLE FOR ANY
 * SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN
 * AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING
 * OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 */


#include "pexpoing.h"
#include <sys/types.h>
#include <sys/time.h>
#ifdef hpux
/* Select stuff cause HP doesn't define it in types.h */
#define FD_ZERO(p)	((p)->fds_bits[0] = 0)
#define FD_SET(n,p)	((p)->fds_bits[0] |= (1 << (n)))
#define FD_CLR(n,p)	((p)->fds_bits[0] &= ~(1 << (n)))
#define FD_ISSET(n,p)	((p)->fds_bits[0] & (1 << (n)))
#endif /* hpux */
#ifdef vax
/* Select stuff cause VAX doesn't define it in types.h */
#define FD_ZERO(p)	((p)->fds_bits[0] = 0)
#define FD_SET(n,p)	((p)->fds_bits[0] |= (1 << (n)))
#define FD_CLR(n,p)	((p)->fds_bits[0] &= ~(1 << (n)))
#define FD_ISSET(n,p)	((p)->fds_bits[0] & (1 << (n)))
#endif /* vax */


/*  Debugging switches.  */
#define XPONG_DEBUGGING FALSE 		/* debugging messages printed out. */
#define XPONG_DEBUG_CALLS FALSE		/* for function calls. */
#define XPONG_DEBUG_BALL_POSITION FALSE   /* ball position messages. */
#define XPONG_DEBUG_INTERSECTS FALSE 	/* for intersections of ball path. */

/* Flag to see if we are doing various testing type things (with the 
   environment variables). */

Bool debugShowVertices = FALSE;		/* For putting labels on the corners */
Bool debugChangeViews = FALSE; 		/* For changing the view parameters. */


/* Colors to use. */

#define XPONG_SUN_VERSION 1
/* *** #define XPONG_HP_VERSION 1 */
/* *** #define XPONG_LYNX_VERSION 1 */
/* *** #define XPONG_MI_VERSION 1*/ 

#ifdef XPONG_SUN_VERSION
#define XPONG_WALL_COLOR pexYellow

#define XPONG_BALL_COLOR pexWhite

#define XPONG_PLAYER1_COLOR pexMagenta
#define XPONG_PLAYER2_COLOR pexCyan
#define XPONG_PLAYER3_COLOR pexGreen
#define XPONG_PLAYER4_COLOR pexYellow
#define XPONG_PLAYER5_COLOR pexRed
#define XPONG_PLAYER6_COLOR pexBlue

#define XPONG_PADDLE_COLOR pexGreen

#define XPONG_PADDLE_BOUNCE_COLOR pexWhite
#define XPONG_BOUNCE_COLOR pexMagenta

#else

#define XPONG_WALL_COLOR pexCyan

#define XPONG_BALL_COLOR pexCyan

#define XPONG_PLAYER1_COLOR pexCyan
#define XPONG_PLAYER2_COLOR pexCyan
#define XPONG_PLAYER3_COLOR pexCyan
#define XPONG_PLAYER4_COLOR pexCyan
#define XPONG_PLAYER5_COLOR pexCyan
#define XPONG_PLAYER6_COLOR pexCyan

#define XPONG_PADDLE_COLOR pexCyan

#define XPONG_PADDLE_BOUNCE_COLOR pexCyan
#define XPONG_BOUNCE_COLOR pexCyan

#endif

/* For looping through the colors... */
int playerColors[] = {
    XPONG_PLAYER1_COLOR,
    XPONG_PLAYER2_COLOR,
    XPONG_PLAYER3_COLOR,
    XPONG_PLAYER4_COLOR,
    XPONG_PLAYER5_COLOR,
    XPONG_PLAYER6_COLOR,
   };
 
/* Global Phigs view variables */
Ppoint3		vrp;		/* view reference point	*/
Pvector3	vpn, nvpn;	/* view plane normal	*/
Pvector3	vup;		/* view up vector	*/
Pint		error;		/* error indicator	*/
Pmatrix3	roty;		/* y rotation matrix	*/
Pmatrix3	rotz;		/* z rotation matrix	*/
Pviewmapping3 map;		/* view mapping structure */
Pviewrep3 viewrep;		/* Phigs view representation */

 
/* One global for each player (possibly one for each side of the cube.*/ 
pexC *Pexi[6];    
/* Bitmask for select.  One bit set per active X display. */
fd_set selmask;
int numfds;

/* These are for the position of the paddles for the players. */
pexCoord3D paddlePosition[6];
pexCoord3D prevPaddlePosition[6];

/* For the score stuff for the players. */
int playerScore[6];
Bool anyScoreChanged[6] = { TRUE, TRUE, TRUE, TRUE, TRUE, TRUE };
char scoreString[255];

int spinning[6] = { 0, 0, 0, 0, 0, 0 };
Bool farView[6] = { FALSE, FALSE, FALSE, FALSE, FALSE, FALSE };
Bool solidModel[6] = { FALSE, FALSE, FALSE, FALSE, FALSE, FALSE };

int windowMin[6];	/* For the scaling for input for each of the windows */

Bool readyToDraw[6]; 	/* To say if event we sent out has been recieved. */ 

char *playerDisplay[6];	/* The displays the players are using. */
Window scoreWindow[6]; 	/* X's windows for the score stuff. */

numberPlayers = 0; 	/* The number of players (i.e. displays) therer are. */


/* Structure containing the information about the state of the ball. */
BallInfo ballInfo, *ball;

/* Definitions for the icosohedron for the ball ( it is defined within 
   a radius of 1.0 of the origin). */ 
#include "polyinfo.h"
Polyinfo ballPoly = {
#include "objico.h"
};
Polyinfo ballWire = {
#include "wireico.h"
};


/* External routines. */ 

int rand();
double sqrt();
double rint();    



/* Our routines. */

void XpongInit();		/* Main initializing routine. */
void XpongPexInit();		/* Initialize the PEX (and X) stuff. */
void XpongUpdate();		/* Main processing loop. */
void XpongXEvent();		/* Deal with the X events. */
void GetPaddlePosition();	/* Query the X pointer for the paddles. */
void ConvertPaddlePosition();	/* Convert screen coords to ones for paddle */
void GameInitialize();		/* Initialize the game stuff. */
void InitializeIntersectItems();	/* Set up list of walls and paddes. */
void BallInitialize(); 		/* Initialize ball stuff. */
pexVector3D GetInitialVelocity();	/* Gets initial velocity. */
float RealRand();		/* Get random between 0.0 and 1.0 */
void DrawWalls();		/* Draws the walls. */
void DrawPlane();		/* Draws a plane from the intersect list. */
void HighlightPlane();		/* Highlights a plane from intersect list. */
void SavePaddlePositions();	/* Updates position for all paddles. */
void SavePaddlePosition();	/* Updates intersect list for each paddle. */
void DrawPaddles();		/* Draws all the paddles on the screen. */
void DrawPaddle();		/* Draws the given paddle on the screen. */
void DrawBallPosition(); 	/* Draws the ball on the screen. */
void UpdateBallPosition();	/* Gets the next position for ball. */
Bool GetIntersectionPoint();	/* Gets intersection pt for ball and plane. */
float Distance();		/* Get distance between two pts. */
int ConvertWCToSC();		/* Converts world coords. to screen coords. */
float ConvertSCToWC();		/* Converts screen coords. to world coords. */
void PrintWCPoint();		/* Print out a 3D point. */
void PrintViewValues();		/* Prints out the PEX global view variables. */
void GetViewValues();		/* Get new values for global view variables. */
void DrawPlayersRenderedPolyline();	/* Draws polylines for all players. */
void DrawPlayersFillArea(); 	 /* Draws polylines for all players. */
Cursor MakeCursor();		/*  Make or get a cursor for our window. */
int SetPlayersCurrentLineColor();	/* Sets line color for given player. */
int SetPlayersCurrentSurfaceColor();	/* Sets surface color for player. */
Window MakeAnotherWindowPlease();	/* Makes windows (in X) for  score. */
int stringLen();		/* Returns the length of a string. */
 
/* For random number generator for initial velocity. */
int seed = 131;     


/* This is the list of the things that the ball can bounce off of (like the 
   walls and paddles etc. */
#define NUMTHINGS 12 		/* The six walls and up to six paddles. */
IntersectItem ii[NUMTHINGS];


/* Global parameter stuff.  *** not implemented yet */

float velocityScale = VELOCITY_SCALE;


/*****************************************************************
 * TAG( main )
 * 
 * 	The main driving loop for the xpong routines.
 *
 * Inputs:
 *	[None]
 * Outputs:
 *	[None]
 * Assumptions:
 *	[None]
 * Algorithm:
 *	[None]
 */

main(argc, argv)
int     argc;
char    *argv[];
{ 
   
    /* Set things up. */
    XpongInit(argc, argv);

    /* And then have a good time... */
    XpongUpdate();   
}



/*****************************************************************
 * TAG( XpongInit )
 * 
 * 	Initialize Xpong
 * 	
 * Inputs:
 *	argc, argv -- from main programm.
 * Outputs:
 *	[None]
 * Assumptions:
 *	[None]
 * Algorithm:
 *	[None]
 */
void
XpongInit(argc, argv)
    int argc;
    char *argv[];
{
    /* Set up thee PEX stuff, the game stuff and then the ball... */
    XpongPexInit(argc, argv);
    GameInitialize();
    ball = &ballInfo;
    BallInitialize(ball);

}

/*****************************************************************
 * TAG( XpongPexInit )
 * 
 * 	Set the PEX stuff up for Xpong.
 * 	
 * Inputs:
 *	argc, argv -- from main program.
 * Outputs:
 *	[None]
 * Assumptions:
 *	[None]
 * Algorithm:
 *	[None]
 */
void
XpongPexInit(argc, argv)
    int argc;
    char *argv[];
{
    static char *envDisplay[] = {
	"DISPLAY1",
	"DISPLAY2",
	"DISPLAY3",
	"DISPLAY4",
	"DISPLAY5",
	"DISPLAY6",
    };

    Display *dpy;

    register int i = 0;
    register Bool noMorePlayers = FALSE;

    Cursor theCursor;

    /* See if we want to do some checking of things. */
    if ( getenv("XPONGVERTICES") )
	debugShowVertices = TRUE;
     
    /* See if we want to do some checking of things. */
    if ( getenv("XPONGVIEW") )
	debugChangeViews = TRUE;


    /* Get the displays (and number of players). */
    numberPlayers = 0;
    i = 0;
    FD_ZERO(&selmask);
    numfds = 0;
    do
    {
	playerDisplay[i] = (char *)getenv(envDisplay[i]);
	if ( playerDisplay[i] != NULL )
	{
	    if ( !(dpy = XOpenDisplay(playerDisplay[i])) )
	    {
		perror("Cannot open display\n");
		exit(-1);
	    }
	    FD_SET(ConnectionNumber(dpy), &selmask);
	    if (ConnectionNumber(dpy) > numfds)
		numfds = ConnectionNumber(dpy);

	    /* Set the default window size to 350. */
	    Pexi[i] = popenphigsextra(dpy, 0, 100, 200, 350, 350, argc, argv); 
	    popenws(Pexi[i], 0, 0, 0);

	    /* Make the window to keep score etc. in. */
	    scoreWindow[i] = MakeAnotherWindowPlease(i, 100, 600, 300, 100, 
					 	     " pexPoing score ");

	    numberPlayers += 1;
	    i += 1;
	}
	else
	{
	    noMorePlayers = TRUE;
	}
    } while ( ! noMorePlayers );

    if ( numberPlayers == 0 )
    {
	printf("Xpong: No players, this does not compute.\n");
	printf("Try typing 'setenv DISPLAY1 <machine>' and rerunning.\n");
	exit(-1);
    }

    /* Change some X stuff for our windows. */
    for ( i = 0; i < numberPlayers; i++ )
    {
	/* Allow more events than default (ExposureMask | StructureNotifyMask).
	   We are going to use the Button5MotionMask event to deal with
	   our asynchronous output stuff. */
	XSelectInput(Pexi[i]->phigsDisplay, Pexi[i]->phigsWindow, 
		     ExposureMask | StructureNotifyMask | KeyPressMask | 
		     ButtonPressMask  );
	
	XChangeProperty(Pexi[i]->phigsDisplay, Pexi[i]->phigsWindow, XA_WM_NAME,
			XA_STRING, 8, PropModeReplace, " pexPoing ", 10);
	XChangeProperty(Pexi[i]->phigsDisplay, Pexi[i]->phigsWindow,
			XA_WM_ICON_NAME, XA_STRING, 8, PropModeReplace, 
			" PEXPOING ", 10);

	readyToDraw[i] = TRUE;   /* For the first drawing. */

	/* Change the cursor to what we want. */
	theCursor = MakeCursor(Pexi[i]->phigsDisplay, Pexi[i]->phigsWindow);
	XDefineCursor(Pexi[i]->phigsDisplay, Pexi[i]->phigsWindow, theCursor); 

    }

    /* Set up our stuff for mapping.  */
    for ( i = 0; i < numberPlayers; i++ )
	windowMin[i] = Minimum(Pexi[i]->winh,Pexi[i]->winw);

    /* Put stuff in the score windows of each player. */
    for ( i = 0; i < numberPlayers; i++ )
    {
	static char titleString[] = "PexPoing scores: ";

	Display *dpy = Pexi[i]->phigsDisplay;
	
	XDrawImageString(dpy, scoreWindow[i], 
			 DefaultGC(dpy, DefaultScreen(dpy)), 
			 SCORE_TITLE_X, SCORE_TITLE_Y, 
			 titleString, stringLen(titleString));
    } /* for */
    
    for ( i = 0; i < numberPlayers; i++ )
    {
	/* Set up th>e PHIGS view */
	vrp.x = 0.0; vrp.y = 0.0; vrp.z = 0.0;
	vpn.x = 0.0; vpn.y = 0.0; vpn.z = 0.0;
	vup.x = 0.0; vup.y = 1.0; vup.z = 0.0;
	map.window.xmin = -1.1;	map.window.ymin = -1.1;
	map.window.xmax =  1.1;	map.window.ymax =  1.1;
	map.view_plane = 1.1; 
	
	if ( debugShowVertices )
	{
	    map.window.xmin = -1.6;	map.window.ymin = -1.6;
	    map.window.xmax =  1.6;	map.window.ymax =  1.6;
	    map.view_plane = 1.6; 
	}

	map.back_plane = -3.0;
	map.front_plane = 3.0;
	
	map.viewport.xmin= 0.0; map.viewport.ymin= 0.0; map.viewport.zmin= -1.0;
	map.viewport.xmax= 1.0; map.viewport.ymax= 1.0; map.viewport.zmax= 0.0;
	map.prp.x = 0.0; map.prp.y = 0.0; map.prp.z = 3.2; 
	map.proj = PPERSPECTIVE; 
	
	viewrep.clip_xy = PCLIP;
	viewrep.clip_back = PCLIP;
	viewrep.clip_front = PCLIP;
	viewrep.clip_limit.xmin = map.viewport.xmin;
	viewrep.clip_limit.xmax = map.viewport.xmax;
	viewrep.clip_limit.ymin = map.viewport.ymin;
	viewrep.clip_limit.ymax = map.viewport.ymax;
	viewrep.clip_limit.zmin = map.viewport.zmin;
	viewrep.clip_limit.zmax = map.viewport.zmax;
	
	switch ( i ) 
	{
	case 0:
	    /* Player 1 has +Z paddle. */
	    vpn.z = 1.0;
	    break;
	case 1:
	    /* Player 1 has -Z paddle. */
	    vpn.z = -1.0;
	    break;
	case 2:
	    /* Player 3 has +X paddle. */
	    vpn.x = 1.0;
	    break;
	case 3:
	    /* Player 4 has -X paddle. */
	    vpn.x = -1.0;
	    break;
	case 4:
	    /* Player 5 has +Y paddle. */
	    vpn.y = 1.0;
	    break;
	case 5:
	    /* Player 6 has -Y paddle. */
	    vpn.y = -1.0;
	    break;
	}

	if (debugChangeViews )
	{ 
	    char answer[255];

	    printf("\nViewing settings: -- Player number %d\n", i + 1);
	    PrintViewValues();

	    printf("\nGive new values? ");
	    scanf("%s", answer);
	    while ( answer[0] == 'y' )
	    {
		GetViewValues();
		printf("\n New values:\n");
		PrintViewValues();
		printf("\nRevise values? ");
		scanf("%s", answer);
	    }

	}

	pevalvieworientationmatrix3(&vrp, &vpn, &vup, 
				    &error, viewrep.orientation_matrix);
	pevalviewmappingmatrix3(&map, &error, viewrep.mapping_matrix);
	psetviewrep3(Pexi[i], 1, &viewrep );

	
	/* Set up a different view for spinning. */

	vpn.x = 1.0; vpn.y = 2.0; vpn.z = 3.0;

	map.window.xmin = -2.0;	map.window.ymin = -2.0;
	map.window.xmax =  2.0;	map.window.ymax =  2.0;
	map.view_plane = 2.0; 
	map.back_plane = -10.0;
	map.front_plane = 10.0;
	map.prp.x = 0.0; map.prp.y = 0.0; map.prp.z = 20.0;

	pevalvieworientationmatrix3(&vrp, &vpn, &vup, 
				    &error, viewrep.orientation_matrix);
	pevalviewmappingmatrix3(&map, &error, viewrep.mapping_matrix);
	psetviewrep3(Pexi[i], 2, &viewrep );

    }
    
    printf("\n XPong -- starting with %d players.\n\n", numberPlayers);
    for ( i = 0; i < numberPlayers; i++ )
	printf("   Player %d will be using display %s\n", (i + 1), playerDisplay[i]);
    printf("\n");
    printf("Hit return when ready to start");
    while ( (i = getchar()) != '\n' )
	;
    
    /* Set up our rotation matrix for spinning. */
    protatey(1.0 * 3.1415 / 180.0, &error, roty );
    protatez(.707 * 3.1415 / 180.0, &error, rotz );
}

/*****************************************************************
 * TAG( XpongUpdate )
 * 
  * 
 * 	This routine takes care of one iteration of xpong.
 * 	
 * Inputs:
 *	[None]
 * Outputs:
 *	[None]
 * Assumptions:
 * 	The pex stuff is all set up..
 * Algorithm:
 *	[None]
 */
void
XpongUpdate()
{  
    register int i;
    struct timeval timeout;
    XButtonPressedEvent xButtonPressedEvent;
    
    
    while ( 1 )
    {
	/* Check and see if there are any events waiting. */
	XpongXEvent();

	/* Get the new paddle position. */
	SavePaddlePositions();
	
	/* Update the position of the ball. */
	UpdateBallPosition(ball); 
	
	/* This loop is to go through different displays for each player. */
	for ( i = 0; i < numberPlayers; i++ )
	{
	    
	    /* See if the event we sent out has been recieved back from
	       the servers.  If it has then we will send a new set of 
	       output commands, and query the cursor location.  If not, 
	       then we have to wait till next time around... */
	    if ( readyToDraw[i] ) 
	    {
		/* The drawing stuff... */

		/* Deal with the score if needed. */	
		/* If any score has changed since our last update */
		if ( anyScoreChanged[i] )
		{
		    static char titleString[] = "PexPoing scores: ";
		    Display *dpy = Pexi[i]->phigsDisplay;
		    register int j;

		    XDrawImageString(dpy, scoreWindow[i], 
				     DefaultGC(dpy, DefaultScreen(dpy)), 
				     SCORE_TITLE_X, SCORE_TITLE_Y, 
				     titleString, stringLen(titleString));
			    
		    /* Get the score for each player and redraw them. */
		    for ( j = 0; j < numberPlayers; j++ )
		    {	
			sprintf(scoreString, "Player %d:%4d   (%s)",
				j + 1, playerScore[j], playerDisplay[j]);
			
			XDrawImageString(dpy, scoreWindow[i],
					 DefaultGC(dpy, DefaultScreen(dpy)), 
					 SCORE_X_START, 
					 SCORE_Y_START + (j * SCORE_Y_OFFSET), 
					 scoreString, stringLen(scoreString));
		    } /* for */

		    anyScoreChanged[i] = FALSE;
		}

                /* Begin the PEX rendering */
		PexBeginRendering(Pexi[i]);

		if ( spinning[i] != 0 )
		{
		    /* If we are spinning, set up a new PHIGS view */
		    ptranvc3(&vpn, roty, &error, &nvpn);
		    
		    vpn.x = nvpn.x; vpn.y = nvpn.y; vpn.z = nvpn.z;
		    
		    if ( spinning[i] >= 2 )
		    {
			ptranvc3(&vpn, rotz, &error, &nvpn);
			
			vpn.x = nvpn.x; vpn.y = nvpn.y; vpn.z = nvpn.z;
		    }
		    
		    pevalvieworientationmatrix3(&vrp, &vpn, &vup, &error, 
						viewrep.orientation_matrix);
		    psetviewrep3( Pexi[i], 2, &viewrep ); 
		}

		if ( farView[i] == FALSE ) 
		    psetviewind(Pexi[i], 1);
		else
		    psetviewind(Pexi[i], 2);

		PexRenderOutputCommands(Pexi[i]);
		if (solidModel[i]){	
	   	    PexInteriorStyle(Pexi[i], InteriorStyleSolid);
		    PexCullMode(Pexi[i], BackFaces);
		    PexHlhsrMode(Pexi[i], HlhsrZBuffer);
		    }
			
		SetPlayersCurrentLineColor(i, XPONG_WALL_COLOR);
		DrawWalls(i);
		
		DrawPaddles(i); 
		if (solidModel[i])	
			SetPlayersCurrentSurfaceColor(i, XPONG_BALL_COLOR);
		else
			SetPlayersCurrentLineColor(i, XPONG_BALL_COLOR);
		DrawBallPosition(i, ball);
		
		PexEndRendComm(Pexi[i]);
		
		PexEndRendering(Pexi[i]);


		/* Send out an event to see whether things are caught up. */
		XStartQueryPointer(Pexi[i]->phigsDisplay, Pexi[i]->phigsWindow);

		/* Need to wait till we get this back. */
		readyToDraw[i] = FALSE;
	
		/* Flush X. */
		XFlush(Pexi[i]->phigsDisplay);
	
	    } /* if readyToDraw */

	} /* for */
	
	/* Wait a bit. */
	timeout.tv_sec = 0;
	timeout.tv_usec = 20000;
	(void)select(numfds, &selmask, 0, 0, &timeout);
	
    }
}

/*****************************************************************
 * TAG( XpongXEvent )
 * 
 * 	Check up on any events given back by X.
 * Inputs:
 *	[None]
 * Outputs:
 *	[None]
 * Assumptions:
 * 	The X stuff is set up okay.
 * Algorithm:
 *	[None]
 */
void
XpongXEvent()
{
    register int i = 0;
    Window root, child;
    int root_x, root_y, win_x, win_y;
    unsigned int mask; 
    

    if ( XPONG_DEBUG_CALLS )
    {
	printf("\nCheckXEvent -- \n");
    }

    for ( i = 0; i < numberPlayers; i++ )
    {
	
	/* Get the next X event (if there is one). */
	if (XEndQueryPointer(Pexi[i]->phigsDisplay, Pexi[i]->phigsWindow,
			      &root, &child, &root_x, &root_y,
			      &win_x, &win_y, &mask))
	{
	    XEvent xEvent;
	    
	    /* Move the paddle */
	    ConvertPaddlePosition(i, win_x, win_y);
	    readyToDraw[i] = TRUE;

	    /* Now it's safe to check for other events */
	    while (XPending(Pexi[i]->phigsDisplay))
	    {
		XNextEvent(Pexi[i]->phigsDisplay, &xEvent); 
		if (xEvent.type == KeyPress)
		{
		    XKeyPressedEvent *xKeyPressedEventPtr;
		    xKeyPressedEventPtr = (XKeyPressedEvent *) &xEvent;
		}
		else if (xEvent.type == ButtonPress)
		{
		    XButtonPressedEvent *xButtonPressedEventPtr;
		    xButtonPressedEventPtr = (XButtonPressedEvent *) &xEvent;
		
		    if ( xButtonPressedEventPtr->button == 1 )
		    {
			/* This means we want to start the spinning. */
			farView[i] = TRUE;
			spinning[i]++;
		    }
		    else if ( xButtonPressedEventPtr->button == 2 )
		    {
			/* This means we want to stop the spinning and get back
			   up close. */
			farView[i] = FALSE;
			spinning[i] = 0;
		    }
		    else if ( xButtonPressedEventPtr->button == 3 )
		    {
			/* This means we toggle the view. */
			if (solidModel[i])
			    solidModel[i]=0; 
			else solidModel[i]=1;
		    
		    }

		}
		else if (xEvent.type == Expose)
		{
		    XExposeEvent *xExposeEventPtr;
		    xExposeEventPtr = (XExposeEvent *) &xEvent;
		    while (xExposeEventPtr->count)
		    {
			XNextEvent(Pexi[i]->phigsDisplay, &xEvent);	    
			xExposeEventPtr = (XExposeEvent *) &xEvent;
		    }
		}
		else if (xEvent.type == ConfigureNotify)
		{
		    XConfigureEvent *xConfigureEventPtr = 
			(XConfigureEvent *)&xEvent;
		    Pexi[i]->winx = xConfigureEventPtr->x;
		    Pexi[i]->winy = xConfigureEventPtr->y;
		    Pexi[i]->winw = xConfigureEventPtr->width;
		    Pexi[i]->winh = xConfigureEventPtr->height;
		    windowMin[i] = Minimum(Pexi[i]->winh,Pexi[i]->winw);
		}
	    }
	}
 
    } /* for */
}


/*****************************************************************
 * TAG( ConvertPaddlePosition )
 * 
 * 	Get the location of the cursor as the paddle position for each player.
 * Inputs:
 *	[None]
 * Outputs:
 *	[None]
 * Assumptions:
 *	[None]
 * Algorithm:
 *	[None]
 */
void
ConvertPaddlePosition(i, winX, winY)
    int i;	/* The player number - 1. */
    int winX, winY; 	/* Coordinate values given in screen values. */
{

    if ( i < numberPlayers )	/* Goes from 0 to player# - 1 */
    {
	prevPaddlePosition[i] = paddlePosition[i];
	
	/* Get our paddle position in World Coordinates.  Note that 
	   the origin in X is the upper left, so the coords have 
	   to be flipped. */
	switch ( i )
	{
	case 0:
	    /* Player 1. (+ Z plane) */
	    paddlePosition[i].x = ConvertSCToWC(winX, i);
	    paddlePosition[i].y = -1.0 * (ConvertSCToWC(winY, i));
	    paddlePosition[i].z = 0.0;
	    break;
	case 1:
	    /* Player 2. (- Z plane) */
	    /* The negative Z paddle has view that has been flipped 
	       around so the x coords run in the opposite direction. */
	    paddlePosition[i].x = -1.0 * ConvertSCToWC(winX, i);
	    paddlePosition[i].y = -1.0 * (ConvertSCToWC(winY, i));
	    paddlePosition[i].z = 0.0;
	    break;
	case 2:
	    /* Player 3. (+ X plane) */
	    paddlePosition[i].x = 0.0;
	    paddlePosition[i].y = -1.0 * (ConvertSCToWC(winY, i));
	    paddlePosition[i].z = -1.0 * (ConvertSCToWC(winX, i));
	    break;
	case 3:
	    /* Player 4. (- X plane) */
	    paddlePosition[i].x = 0.0;
	    paddlePosition[i].y = -1.0 * (ConvertSCToWC(winY, i));
	    paddlePosition[i].z = ConvertSCToWC(winX, i);
	    break;
	case 4:
	    /* Player 5. (+ Y plane) */
	    paddlePosition[i].x = ConvertSCToWC(winX, i);
	    paddlePosition[i].y = 0.0;
	    paddlePosition[i].z = ConvertSCToWC(winY, i);
	    break;
	case 5:
	    /* Player 6. (- Y plane) */
	    paddlePosition[i].x = ConvertSCToWC(winX, i);
	    paddlePosition[i].y = 0.0;
	    paddlePosition[i].z = -1.0 * (ConvertSCToWC(winY, i));
	    break;
	} /* switch */
	
    } /* if */
} 

/*****************************************************************
 * TAG( GameInitialize )
 * 
 * 
 * Inputs:
 * 	[None]
 * Outputs:
 *      [None] 
 * Assumptions:
 * 	[None]
 * Algorithm:
 * 	Sets up the game stuff.
 */
void
GameInitialize()
{
    if ( XPONG_DEBUG_CALLS )
    {
	printf("\nGameInitialize -- \n");
    }

    /* Set up the list of things that can intersect with the ball. */
    InitializeIntersectItems();

    /* Set up the random number stuff. */
    srand(seed);

}

/*****************************************************************
 * TAG( InitializeIntersectItems )
 * 
 * 
 * Inputs:
 * 	[None]
 * Outputs:
 *      [None] 
 * Assumptions:
 * 	Uses the global array of intersect items ii.
 * Algorithm:
 * 	Sets up the list.
 */
void
InitializeIntersectItems()
{
    register int i;
    pexCoord3D zeroPt;

    if ( XPONG_DEBUG_CALLS )
    {
	printf("\nInitializeIntersectItems -- \n");
    }

    zeroPt.x = 0.0;
    zeroPt.y = 0.0;
    zeroPt.z = 0.0;

    /* Set up the list of things that can intersect with the ball. */

    /* Set up the common entries. */
    for (i = 0; i < NUMTHINGS; i++) 
    {
	ii[i].isPaddle  = ((i < 6) ? FALSE : TRUE);  

	ii[i].showBounce = 0;

	ii[i].interPt= zeroPt;

	ii[i].maxX = ii[i].maxY = ii[i].maxZ = WALL_DISTANCE;
	ii[i].minX = ii[i].minY = ii[i].minZ = (-1.0 * WALL_DISTANCE);
    }
    /* The z=+1 plane. */
    ii[0].planeKind = ZPLANE;
    ii[0].maxZ = ii[0].minZ = WALL_DISTANCE;

    /* The z=-1 plane. */
    ii[1].planeKind = ZPLANE;
    ii[1].maxZ = ii[1].minZ = -1.0 * WALL_DISTANCE;

    /* The x=+1 plane. */
    ii[2].planeKind = XPLANE;
    ii[2].maxX = ii[2].minX = WALL_DISTANCE;

    /* The x=-1 plane. */
    ii[3].planeKind = XPLANE;
    ii[3].maxX = ii[3].minX = -1.0 * WALL_DISTANCE;

    /* The y=+1 plane. */
    ii[4].planeKind = YPLANE;
    ii[4].maxY = ii[4].minY = WALL_DISTANCE;

    /* The y=-1 plane. */
    ii[5].planeKind = YPLANE;
    ii[5].maxY = ii[5].minY = -1.0 * WALL_DISTANCE;


    /* The paddles. */
    
    /* The + Z Paddle. */
    ii[6].planeKind = ZPLANE;
    ii[6].minZ = ii[6].maxZ = 
	WALL_DISTANCE - PADDLE_INSET_DISTANCE; 
    SavePaddlePosition(6, zeroPt);  
    
    /* The - Z Paddle. */
    ii[7].planeKind = ZPLANE;
    ii[7].minZ = ii[7].maxZ = 
	-1.0 * (WALL_DISTANCE - PADDLE_INSET_DISTANCE);
    SavePaddlePosition(7, zeroPt);  

    /* The + X Paddle. */
    ii[8].planeKind = XPLANE;
    ii[8].minX = ii[8].maxX = 
	WALL_DISTANCE - PADDLE_INSET_DISTANCE; 
    SavePaddlePosition(8, zeroPt);  
    
    /* The - X Paddle. */
    ii[9].planeKind = XPLANE;
    ii[9].minX = ii[9].maxX = 
	-1.0 * (WALL_DISTANCE - PADDLE_INSET_DISTANCE);
    SavePaddlePosition(9, zeroPt);  

    /* The + Y Paddle. */
    ii[10].planeKind = YPLANE;
    ii[10].minY = ii[10].maxY = 
	WALL_DISTANCE - PADDLE_INSET_DISTANCE; 
    SavePaddlePosition(10, zeroPt);  
    
    /* The - Y Paddle. */
    ii[11].planeKind = YPLANE;
    ii[11].minY = ii[11].maxY = 
	-1.0 * (WALL_DISTANCE - PADDLE_INSET_DISTANCE);
    SavePaddlePosition(11, zeroPt);  

}

/*****************************************************************
 * TAG( BallInitialize )
 * 
 * 
 * Inputs:
 * 	ball -- pointer to BallInfo structure.
 * Outputs:
 *      BallInfo struture filled in.
 * Assumptions:
 * 	[None]
 * Algorithm:
 * 	[None]
 */
void
BallInitialize(ball)
    BallInfo *ball;
{
    pexVector3D GetInitialVelocity();

    register int i;

    if ( XPONG_DEBUG_CALLS )
    {
	printf("\nBallInitialize -- \n");
    }

    /* Get initial position and velocity. */
    ball->position.x = 0.0;
    ball->position.y = 0.0;
    ball->position.z = 0.0;

    ball->velocity = GetInitialVelocity();

    /* Now shrink the ball coordinates down to what we want. (The ball 
       coordinates are defined so they fill the -1.0 to +1.0 cube, an then 
       are scaled down to what we want. */

    for ( i = 0; i < ballPoly.numverts; i++ ) 
    {
	ballPoly.v[i].x *= BALL_RADIUS;
	ballPoly.v[i].y *= BALL_RADIUS;
	ballPoly.v[i].z *= BALL_RADIUS;
    }
    for ( i = 0; i < ballWire.numverts; i++ ) 
    {
	ballWire.v[i].x *= BALL_RADIUS;
	ballWire.v[i].y *= BALL_RADIUS;
	ballWire.v[i].z *= BALL_RADIUS;
    }


}

/*****************************************************************
 * TAG( GetInitialVelocity )
 * 
 * 	Gets the initial velocity for the ball.
 * 	
 * Inputs:
 * 	[None]
 * Outputs:
 * 	returns the initial velocity (as pexVector3D).
 * Assumptions:
 *	Assumes that the point will initially be positioned such that
 * 	the velocity in any direction will be legal (i.e. the ball is
 *	not on one of the walls, etc.).
 * Algorithm:
 * 	Sets each component of the vector to a value between 
 * 	-1.0 * VELOCITY_SCALE and 1.0 * VELOCITY_SCALE.
 */
pexVector3D
GetInitialVelocity()
{
    pexVector3D velocity;

    if ( XPONG_DEBUGGING ) 
    {
	printf("\nGetInitialVelocity -- \n");
    }

    velocity.x = (2.0 * RealRand() - 1.0) * velocityScale;
    velocity.y = (2.0 * RealRand() - 1.0) * velocityScale;
    velocity.z = (2.0 * RealRand() - 1.0) * velocityScale;

    if ( XPONG_DEBUGGING )
    {
	printf("   initial velocity: \n");
	PrintWCPoint(velocity);
    }

    return velocity;
}

/*****************************************************************
 * TAG( RealRand )
 * 
 *   Return a "random" number between 0.0 and 1.0
 *   
 * Inputs:
 * 	[None]
 * Outputs:
 * 	returns float between 0.0 and 1.0
 * Assumptions:
 *      [None]	
 * Algorithm:
 * 	rand returns random number between 0 and 32767, so map that to 
 *      a value between 0.0 and 1.0.
 */
float
RealRand()
{
    int n;
    float x;

    if ( XPONG_DEBUG_CALLS )
    {
	printf("\nRealRand -- \n");
    }

    /* Get "random" integer from 0 - 32767 */
    n = rand();
    n = n % 32767;        /* Get rid of unwanted bits. */

    /* Get number between 0.0 and 1.0 */
    x = n / 32767.0;

    if ( XPONG_DEBUGGING )
    {
	printf("   random float is %f\n", x);
    }

    /* Return the number. */
    return x;
}

/*****************************************************************
 * TAG( DrawWalls )
 * 
 * 	Draw the walls for the ball to bounce around in.
 * 	
 * Inputs:
 *	[None]
 * Outputs:
 * 	(The walls are drawn).
 * Assumptions:
 * 	Assumes that the pex stuff is all set up for the PEX calls,
 *      dpy and gc (both global) are set up as the display and graphic 
 *      context.  The walls are defined as being WALL_DISTANCE (which 
 *      is BOUNCE_DISTANCE +BALL_RADIUS) away from the origin 
 *      (0.0, 0.0, 0.0).  
 * Algorithm:
 * 	Calls DrawPlane for each of the planes.
 */
void
DrawWalls(playerNumber)
    int playerNumber;
{
    static char *textText[] = {
        "(-1,-1,1)",
        "(1,-1,1)",
        "(1,1,1)",
        "(-1,1,1)", 
        "(-1,-1,-1)",
        "(1,-1,-1)",
        "(1,1,-1)", 
        "(-1,1,-1)",
    };

    static pexCoord3D textPt[] = {
        { -1.0, -1.0, 1.0 },
        { 1.0, -1.0, 1.0 },
        { 1.0, 1.0, 1.0 },
        { -1.0, 1.0, 1.0 },
        { -1.0, -1.0, -1.0 },
        { 1.0, -1.0, -1.0 },
        { 1.0, 1.0, -1.0 },
        { -1.0, 1.0, -1.0 },
    };

    static pexCoord3D textOff = { 0.0, 0.0, 0.0 };

    register int i, j;
 
    if ( XPONG_DEBUG_CALLS )
    {
	printf("\nDrawWalls -- \n");
    }

    /* Draw each plane, one at a time. */
    for ( i = 0; i < 6; i++ )	
    {
	DrawPlane(playerNumber, i, FALSE); 
	HighlightPlane(playerNumber, &(ii[i]), playerColors[i]);
    }

    if ( debugShowVertices )
    {
   	for ( i = 0; i < numberPlayers; i++ )
	    for ( j = 0; j < 8; j++ )
		PexAnnoText(Pexi[i], textText[j], &textPt[j], &textOff);

    }

}

/*****************************************************************
 * TAG( DrawPlane )
 * 
 * 	This is called to draw a plane from its description in the 
 *      IntersectItem.
 * 	
 * Inputs:
 *	ii -- IntersectItem  (describes the plane)
 * Outputs:
 * 	(updates the window contents)
 * Assumptions:
 * 	Assumes the PEX stuff is set up and in the rendering part.
 * Algorithm:
 *	[None]
 */
void
DrawPlane(playerNumber, planeNumber, wantOutlines)
    int playerNumber;
    int planeNumber;
    Bool wantOutlines;
{
    register int i;
    int numPts, saveLineColorIndex;
    pexCoord3D pt[10];

    if ( XPONG_DEBUG_CALLS )
    {
	printf("\nDrawPlane -- \n");
    }

    if (wantOutlines )
    {
	
	/* Draw the outlines of the given plane. */
	
	switch ( ii[planeNumber].planeKind )
	{
	case XPLANE:
	    pt[0].x = ii[planeNumber].minX;
	    pt[0].y = ii[planeNumber].maxY;
	    pt[0].z = ii[planeNumber].maxZ;
	    
	    pt[1].x = ii[planeNumber].minX;
	    pt[1].y = ii[planeNumber].minY;
	    pt[1].z = ii[planeNumber].maxZ;
	    
	    pt[2].x = ii[planeNumber].minX;
	    pt[2].y = ii[planeNumber].minY;
	    pt[2].z = ii[planeNumber].minZ;
	    
	    pt[3].x = ii[planeNumber].minX;
	    pt[3].y = ii[planeNumber].maxY;
	    pt[3].z = ii[planeNumber].minZ;
	    
	    pt[4].x = ii[planeNumber].minX;
	    pt[4].y = ii[planeNumber].maxY;
	    pt[4].z = ii[planeNumber].maxZ;
	    
	    numPts = 5;
	    
	    break;
	    
	case YPLANE:
	    pt[0].x = ii[planeNumber].maxX;
	    pt[0].y = ii[planeNumber].minY;
	    pt[0].z = ii[planeNumber].maxZ;
	    
	    pt[1].x = ii[planeNumber].minX;
	    pt[1].y = ii[planeNumber].minY;
	    pt[1].z = ii[planeNumber].maxZ;
	    
	    pt[2].x = ii[planeNumber].minX;
	    pt[2].y = ii[planeNumber].minY;
	    pt[2].z = ii[planeNumber].minZ;
	    
	    pt[3].x = ii[planeNumber].maxX;
	    pt[3].y = ii[planeNumber].minY;
	    pt[3].z = ii[planeNumber].minZ;
	    
	    pt[4].x = ii[planeNumber].maxX;
	    pt[4].y = ii[planeNumber].minY;
	    pt[4].z = ii[planeNumber].maxZ;
	    
	    numPts = 5;
	    
	    break;
	    
	    
	case ZPLANE:
	    pt[0].x = ii[planeNumber].maxX;
	    pt[0].y = ii[planeNumber].maxY;
	    pt[0].z = ii[planeNumber].minZ;
	    
	    pt[1].x = ii[planeNumber].minX;
	    pt[1].y = ii[planeNumber].maxY;
	    pt[1].z = ii[planeNumber].minZ;
	    
	    pt[2].x = ii[planeNumber].minX;
	    pt[2].y = ii[planeNumber].minY;
	    pt[2].z = ii[planeNumber].minZ;
	    
	    pt[3].x = ii[planeNumber].maxX;
	    pt[3].y = ii[planeNumber].minY;
	    pt[3].z = ii[planeNumber].minZ; 
	    
	    pt[4].x = ii[planeNumber].maxX;
	    pt[4].y = ii[planeNumber].maxY;
	    pt[4].z = ii[planeNumber].minZ;
	    
	    numPts = 5;
	    
	    break;
	    
	}
	
	DrawPlayersRenderedPolyline(playerNumber, pt, numPts);

    } /* if (wantOutlines ) */

    /* Now see if we need to mark the wall as "bounced off of".  The
       number in showBounce is the number of iterations we want to 
       have the mark show up for. */
    if ( ii[planeNumber].showBounce > 0 )
    {
	/* Mark the wall as bounced off of. */
	
	float sideSize;
	
	/* Make the X shrink down over time. */
	sideSize = X_SIDE_SIZE * 
	    ((float) ii[planeNumber].showBounce /(float) TIMES_TO_SHOW_BOUNCE);
	

	/* The paddles will just be highlighted if they have been
	   bounced off of, the planes will get an X. */
	if ( ii[planeNumber].isPaddle ) 
	{
	    int m;
	    m = IntersectItemPaddleIndex(planeNumber);

	    /* Move the intersection pt. to where it would be if it was
	       moved with the paddle. */
	    ii[planeNumber].interPt.x += paddlePosition[m].x -
		prevPaddlePosition[m].x;
	    ii[planeNumber].interPt.y += paddlePosition[m].y -
		prevPaddlePosition[m].y;
	    ii[planeNumber].interPt.z += paddlePosition[m].z -
		prevPaddlePosition[m].z;

	    /* Make the paddle highlight last only half as long as the X. */
	    if ( ((2 * ii[planeNumber].showBounce) / TIMES_TO_SHOW_BOUNCE) >= 1 )
	    {
		int i;
		/* Loop through all the displays we have to draw on. */
		for ( i = 0; i < numberPlayers; i++ )
		{
		    if ( readyToDraw[i] )
			HighlightPlane(i, &ii[planeNumber], 
				       XPONG_PADDLE_BOUNCE_COLOR);
		}
	    }

	    switch ( ii[planeNumber].planeKind )
	    {
	    case XPLANE:
		
		pt[0].x = ii[planeNumber].minX;
		pt[0].y = ii[planeNumber].interPt.y + sideSize;
		pt[0].z = ii[planeNumber].interPt.z;
		
		pt[1].x = ii[planeNumber].minX;
		pt[1].y = ii[planeNumber].interPt.y;
		pt[1].z = ii[planeNumber].interPt.z - sideSize;
		
		pt[2].x = ii[planeNumber].minX;
		pt[2].y = ii[planeNumber].interPt.y - sideSize;
		pt[2].z = ii[planeNumber].interPt.z;
		
		pt[3].x = ii[planeNumber].minX;
		pt[3].y = ii[planeNumber].interPt.y;
		pt[3].z = ii[planeNumber].interPt.z + sideSize;
		
	    case YPLANE:
		pt[0].x = ii[planeNumber].interPt.x + sideSize;
		pt[0].y = ii[planeNumber].minY;
		pt[0].z = ii[planeNumber].interPt.z;
		
		pt[1].x = ii[planeNumber].interPt.x;
		pt[1].y = ii[planeNumber].minY;
		pt[1].z = ii[planeNumber].interPt.z - sideSize; 
		
		pt[2].x = ii[planeNumber].interPt.x - sideSize; 
		pt[2].y = ii[planeNumber].minY;
		pt[2].z = ii[planeNumber].interPt.z;
		
		pt[3].x = ii[planeNumber].interPt.x;
		pt[3].y = ii[planeNumber].minY;
		pt[3].z = ii[planeNumber].interPt.z + sideSize; 
		
		break;
		
	    case ZPLANE:
		pt[0].x = ii[planeNumber].interPt.x + sideSize; 
		pt[0].y = ii[planeNumber].interPt.y;
		pt[0].z = ii[planeNumber].minZ;
		
		pt[1].x = ii[planeNumber].interPt.x;
		pt[1].y = ii[planeNumber].interPt.y - sideSize;
		pt[1].z = ii[planeNumber].minZ;
		
		pt[2].x = ii[planeNumber].interPt.x - sideSize;
		pt[2].y = ii[planeNumber].interPt.y;
		pt[2].z = ii[planeNumber].minZ;
		
		pt[3].x = ii[planeNumber].interPt.x;
		pt[3].y = ii[planeNumber].interPt.y + sideSize; 
		pt[3].z = ii[planeNumber].minZ;
		
		break;
	 
	    }

	    pt[4] = pt[0];	/* Hook it back together. */
	
	    /* Draw the mark in the player's color. */
	    saveLineColorIndex = SetPlayersCurrentLineColor(playerNumber, 
			 playerColors[IntersectItemPaddleIndex(planeNumber)]);

	    DrawPlayersRenderedPolyline(playerNumber, pt, 5);
    
	}
	else
	{
	    /* It is a plane, so draw an X. */

	    switch ( ii[planeNumber].planeKind )
	    {
	    case XPLANE:
		
		pt[0].x = ii[planeNumber].minX;
		pt[0].y = ii[planeNumber].interPt.y + sideSize;
		pt[0].z = ii[planeNumber].interPt.z + sideSize;
		
		pt[1].x = ii[planeNumber].minX;
		pt[1].y = ii[planeNumber].interPt.y - sideSize;
		pt[1].z = ii[planeNumber].interPt.z - sideSize;
		
		pt[2].x = ii[planeNumber].minX;
		pt[2].y = ii[planeNumber].interPt.y + sideSize;
		pt[2].z = ii[planeNumber].interPt.z - sideSize;
		
		pt[3].x = ii[planeNumber].minX;
		pt[3].y = ii[planeNumber].interPt.y - sideSize;
		pt[3].z = ii[planeNumber].interPt.z + sideSize;
		
		break;
		
	    case YPLANE:
		pt[0].x = ii[planeNumber].interPt.x + sideSize;
		pt[0].y = ii[planeNumber].minY;
		pt[0].z = ii[planeNumber].interPt.z + sideSize;
		
		pt[1].x = ii[planeNumber].interPt.x - sideSize;
		pt[1].y = ii[planeNumber].minY;
		pt[1].z = ii[planeNumber].interPt.z - sideSize;
		
		pt[2].x = ii[planeNumber].interPt.x + sideSize;
		pt[2].y = ii[planeNumber].minY;
		pt[2].z = ii[planeNumber].interPt.z - sideSize;
		
		pt[3].x = ii[planeNumber].interPt.x - sideSize;
		pt[3].y = ii[planeNumber].minY;
		pt[3].z = ii[planeNumber].interPt.z + sideSize;
		
		break;
		
	    case ZPLANE:
		pt[0].x = ii[planeNumber].interPt.x + sideSize;
		pt[0].y = ii[planeNumber].interPt.y + sideSize;
		pt[0].z = ii[planeNumber].minZ;
		
		pt[1].x = ii[planeNumber].interPt.x - sideSize;
		pt[1].y = ii[planeNumber].interPt.y - sideSize;
		pt[1].z = ii[planeNumber].minZ;
		
		pt[2].x = ii[planeNumber].interPt.x + sideSize ;
		pt[2].y = ii[planeNumber].interPt.y - sideSize;
		pt[2].z = ii[planeNumber].minZ;
		
		pt[3].x = ii[planeNumber].interPt.x - sideSize;
		pt[3].y = ii[planeNumber].interPt.y + sideSize;
		pt[3].z = ii[planeNumber].minZ;
		
		break;
		
	    }
	    /* Draw the mark in the player's color. */
	    saveLineColorIndex = 
		SetPlayersCurrentLineColor(playerNumber,
					   playerColors[planeNumber]);

	    DrawPlayersRenderedPolyline(playerNumber, pt, 2);
	    DrawPlayersRenderedPolyline(playerNumber, pt+2, 2);
	    
	}

	/* Don't care about the previous color this time */
	SetPlayersCurrentLineColor(playerNumber, saveLineColorIndex);
	
	
	/* Get rid of the flag (do this only if it is player 1 (who is
	   always going to be playing).  */
	/* ### Later we will check to see if the player is active and 
	   if not, go up to the next player. */
	if ( playerNumber == 0 )
	{
	    if ( ii[planeNumber].isPaddle )
		ii[planeNumber].showBounce -= 2;
	    else
		ii[planeNumber].showBounce -= 1;
	}

	/* We don''t want it to go less than zero. */
	ii[planeNumber].showBounce = Maximum(ii[planeNumber].showBounce, 0);
	
	if (ii[planeNumber].showBounce == 0 )
	{	
	    ii[planeNumber].interPt.x = 0.0;
	    ii[planeNumber].interPt.y = 0.0;
	    ii[planeNumber].interPt.z = 0.0;
	}
    }	
}

/*****************************************************************
 * TAG( HighlightPlane )
 * 
 * 	This is called to draw a plane from its description in the 
 *      IntersectItem.
 * 	
 * Inputs:
 *	ii -- IntersectItem  (describes the plane)
 * Outputs:
 * 	(updates the window contents)
 * Assumptions:
 * 	Assumes the PEX stuff is set up and in the rendering part.
 * Algorithm:
 *	[None]
 */
void
HighlightPlane(playerNumber, ii, highlightColor)
    int playerNumber;
    IntersectItem *ii;
    int highlightColor;
{
    register int i;
    int numPts, saveLineColorIndex;
    pexCoord3D pt[10];

    if ( XPONG_DEBUG_CALLS )
    {
	printf("\nHighlightPlane -- \n");
    }

    /* Draw lines just inside of the given plane. */
    
    switch ( ii->planeKind )
    {
    case XPLANE:
	pt[0].x = ii->minX;
	pt[0].y = ii->maxY - HIGHLIGHT_DISTANCE;
	pt[0].z = ii->maxZ - HIGHLIGHT_DISTANCE;

	pt[1].x = ii->minX;
	pt[1].y = ii->minY + HIGHLIGHT_DISTANCE;
	pt[1].z = ii->maxZ - HIGHLIGHT_DISTANCE;

	pt[2].x = ii->minX;
	pt[2].y = ii->minY + HIGHLIGHT_DISTANCE;
	pt[2].z = ii->minZ + HIGHLIGHT_DISTANCE;
 
	pt[3].x = ii->minX;
	pt[3].y = ii->maxY - HIGHLIGHT_DISTANCE;
	pt[3].z = ii->minZ + HIGHLIGHT_DISTANCE;

	pt[4].x = ii->minX;
	pt[4].y = ii->maxY - HIGHLIGHT_DISTANCE;
	pt[4].z = ii->maxZ - HIGHLIGHT_DISTANCE;
	
	numPts = 5;
	
	break;

    case YPLANE:
	pt[0].x = ii->maxX - HIGHLIGHT_DISTANCE;
	pt[0].y = ii->minY;
	pt[0].z = ii->maxZ - HIGHLIGHT_DISTANCE;

	pt[1].x = ii->minX + HIGHLIGHT_DISTANCE;
	pt[1].y = ii->minY;
	pt[1].z = ii->maxZ - HIGHLIGHT_DISTANCE;

	pt[2].x = ii->minX + HIGHLIGHT_DISTANCE;
	pt[2].y = ii->minY;
	pt[2].z = ii->minZ + HIGHLIGHT_DISTANCE;
 
	pt[3].x = ii->maxX - HIGHLIGHT_DISTANCE;
	pt[3].y = ii->minY;
	pt[3].z = ii->minZ + HIGHLIGHT_DISTANCE;

	pt[4].x = ii->maxX - HIGHLIGHT_DISTANCE;
	pt[4].y = ii->minY;
	pt[4].z = ii->maxZ - HIGHLIGHT_DISTANCE;
	
	numPts = 5;
	
	break;


    case ZPLANE:
	pt[0].x = ii->maxX - HIGHLIGHT_DISTANCE;
	pt[0].y = ii->maxY - HIGHLIGHT_DISTANCE;
	pt[0].z = ii->minZ;

	pt[1].x = ii->minX + HIGHLIGHT_DISTANCE;
	pt[1].y = ii->maxY - HIGHLIGHT_DISTANCE;
	pt[1].z = ii->minZ;

	pt[2].x = ii->minX + HIGHLIGHT_DISTANCE;
	pt[2].y = ii->minY + HIGHLIGHT_DISTANCE;
	pt[2].z = ii->minZ;
 
	pt[3].x = ii->maxX - HIGHLIGHT_DISTANCE;
	pt[3].y = ii->minY + HIGHLIGHT_DISTANCE;
	pt[3].z = ii->minZ; 

	pt[4].x = ii->maxX - HIGHLIGHT_DISTANCE;
	pt[4].y = ii->maxY - HIGHLIGHT_DISTANCE;
	pt[4].z = ii->minZ;
	
	numPts = 5;

	break;

    }
    
    saveLineColorIndex = SetPlayersCurrentLineColor(playerNumber,
						    highlightColor);
    
    DrawPlayersRenderedPolyline(playerNumber, pt, numPts);
    
    /* Don't care about the previous color this time */
    SetPlayersCurrentLineColor(playerNumber, saveLineColorIndex);
    

	
}

/*****************************************************************
 * TAG( SavePaddlePositions )
 * 
 * 	Get the location of the cursor as the paddle position for each player.
 * Inputs:
 *	[None]
 * Outputs:
 *	[None]
 * Assumptions:
 *	[None]
 * Algorithm:
 *	[None]
 */
void
SavePaddlePositions()
{
    register int i;

    for ( i = 0; i < numberPlayers; i++ ) 
    {
	SavePaddlePosition(PaddleIntersectItemIndex(i), 
			   paddlePosition[i]); 
    }

}


/*****************************************************************
 * TAG( SavePaddlePosition )
 * 
 * 	Put the plane for the paddle into the intersect item listt in the new
 * 	position.
 * 	
 * Inputs:
 * 	paddle -- which paddle this is.
 * 	position -- the new position.
 * Outputs:
 * 	(puts the values into the intersect list)
 * Assumptions:
 * 	The intersect list is set up.
 * Algorithm:
 *	[None]
 */
void
SavePaddlePosition(paddle, position)
    int paddle;
    pexCoord3D position;
{
    float dist = PADDLE_WIDTH / 2.0;

    switch ( ii[paddle].planeKind )
    {
    case XPLANE:
	/* Make sure we stay in the box. */
	if ( (position.y + dist) > WALL_DISTANCE )
	    position.y = WALL_DISTANCE - dist;
	else if ( (position.y - dist) < -1.0 * WALL_DISTANCE )
	    position.y = -1.0 * WALL_DISTANCE + dist;

	if ( (position.z + dist) > WALL_DISTANCE )
	    position.z = WALL_DISTANCE - dist;
	else if ( (position.z - dist) < -1.0 * WALL_DISTANCE )
	    position.z = -1.0 * WALL_DISTANCE + dist;

	ii[paddle].maxY = position.y + dist;
	ii[paddle].minY = position.y - dist;
	ii[paddle].maxZ = position.z + dist;
	ii[paddle].minZ = position.z - dist;
	break;

    case YPLANE:
	/* Make sure we stay in the box. */
	if ( (position.x + dist) > WALL_DISTANCE )
	    position.x = WALL_DISTANCE - dist;
	else if ( (position.x - dist) < -1.0 * WALL_DISTANCE )
	    position.x = -1.0 * WALL_DISTANCE + dist;

	if ( (position.z + dist) > WALL_DISTANCE )
	    position.z = WALL_DISTANCE - dist;
	else if ( (position.z - dist) < -1.0 * WALL_DISTANCE )
	    position.z = -1.0 * WALL_DISTANCE + dist;

	ii[paddle].maxX = position.x + dist;
	ii[paddle].minX = position.x - dist;
	ii[paddle].maxZ = position.z + dist;
	ii[paddle].minZ = position.z - dist;
	break;

    case ZPLANE:
	/* Make sure we stay in the box. */
	if ( (position.x + dist) > WALL_DISTANCE )
	    position.x = WALL_DISTANCE - dist;
	else if ( (position.x - dist) < -1.0 * WALL_DISTANCE )
	    position.x = -1.0 * WALL_DISTANCE + dist;
	
	if ( (position.y + dist) > WALL_DISTANCE )
	    position.y = WALL_DISTANCE - dist;
	else if ( (position.y - dist) < -1.0 * WALL_DISTANCE )
	    position.y = -1.0 * WALL_DISTANCE + dist;

	ii[paddle].maxX = position.x + dist;
	ii[paddle].minX = position.x - dist;
	ii[paddle].maxY = position.y + dist;
	ii[paddle].minY = position.y - dist;
	break;

    } /* switch */

}

/*****************************************************************
 * TAG( DrawPaddles )
 * 
 *    Draws the paddles for all the players on the screen.
 *    
 * Inputs:
 *      playerNumber -- the player upon whose display to draw.
 * Outputs:
 * 	(draws things on the window.)
 * Assumptions:
 * 	Assumes the window is all set up to draw on, and paddle stuff
 *      is set up right.
 * Algorithm:
 *      [None]
 */
void
DrawPaddles(playerNumber)
    int playerNumber;
{  
    register int i;

    if ( XPONG_DEBUG_CALLS )
    {
	printf("\nDrawPaddles -- \n");
    }

    for ( i = 0; i < numberPlayers; i++ )
    {
	SetPlayersCurrentLineColor(playerNumber, playerColors[i]); 
	DrawPaddle(playerNumber, i);
    }

}

/*****************************************************************
 * TAG( DrawPaddle )
 * 
 *    Draw the paddle on the screen.
 *    
 * Inputs:
 *      playerNumber -- the player upon whose display to draw.
 *      paddle -- the player whose paddle to draw.
 * Outputs:
 * 	(draws things on the window.)
 * Assumptions:
 * 	Assumes the window is all set up to draw on, and paddle stuff
 *      is set up right.
 * Algorithm:
 *      [None]
 */
void
DrawPaddle(playerNumber, paddle)
    int playerNumber;
    int paddle;
{  
    if ( XPONG_DEBUG_CALLS )
    {
	printf("\nDrawPaddle -- paddle\n");
    }

    /* Draw the paddle plane.  The intersection list's first 6 places are 
       for the walls of the cube (0 - 5), and so 6 on up are for the paddles 
       (PaddleIntersectItemIndex gives these numbers). */
    DrawPlane(playerNumber, PaddleIntersectItemIndex(paddle), TRUE);

}

/*****************************************************************
 * TAG( DrawBallPosition )
 * 
 *    Draw the ball on the screen.
 *    
 * Inputs:
 * 	ball -- pointer to BallInfo.
 * Outputs:
 * 	(draws things on the window.)
 * Assumptions:
 * 	Assumes the window is all set up to draw on.
 * Algorithm:
 * 	Removes the ball from the last position, and draws it in
 * 	the new one.
 */
void
DrawBallPosition(playerNumber, ball)
    int playerNumber;
    BallInfo *ball;
{
    int nfaces = ballPoly.numfaces;	/* quicker this way? */
    pexCoord3D *verts = ballPoly.v;

    register int i, j;
    int nverts;
    register int *pf;

    pexCoord3D ppts[MAXVERTS+1];


    if ( XPONG_DEBUG_CALLS )
    {
	printf("\nDrawBallPosition -- ball\n");
    }


    if ( solidModel[i] )
    {
	/* Get initial vertex count. */
	pf = ballPoly.f;	

	/* Loop through the faces. */
	for ( i = 0; i < nfaces; i++ )
	{
	    nverts = *pf;	/* number of vertices in next polygon. */
	    pf++;		/* pf -> first vertex number. */

	    /* Get the vertices for this polygon. */
	    for ( j = 0; j < nverts; j++ )
	    {
		ppts[j].x = ball->position.x + verts[pf[j]].x;
		ppts[j].y = ball->position.y + verts[pf[j]].y;
		ppts[j].z = ball->position.z + verts[pf[j]].z;
	    }

	    DrawPlayersFillArea(playerNumber, ppts, nverts);

	    /* Get ready for next polygon. */
	    pf += nverts;
	}

    }
    else
    {
	/* Get initial vertex count. */
	pf = ballWire.f;	

	/* Loop through the faces. */
	for ( i = 0; i < nfaces; i++ )
	{
	    nverts = *pf;	/* number of vertices in next polygon. */
	    pf++;		/* pf -> first vertex number. */

	    /* Get the vertices for this polygon. */
	    for ( j = 0; j < nverts; j++ )
	    {
		ppts[j].x = ball->position.x + verts[pf[j]].x;
		ppts[j].y = ball->position.y + verts[pf[j]].y;
		ppts[j].z = ball->position.z + verts[pf[j]].z;
	    }

	    DrawPlayersRenderedPolyline(playerNumber, ppts, nverts);

	    /* Get ready for next polygon. */
	    pf += nverts;
	}

    }
}

/*****************************************************************
 * TAG( UpdateBallPosition )
 * 
 * 	Gets the next position for the ball, checking for intersections 
 * 	different things.
 * 
 * Inputs:
 * 	ball -- pointer to BallInfo.
 * Outputs:
 * 	updates stuff in the ball struct.
 * Assumptions:
 *      the ball has been initialized and the intersect list exists.
 * Algorithm:
 * 	vector arithmetic.
 */
void
UpdateBallPosition(ball)
    BallInfo *ball;
{
    pexCoord3D curPos, nextPos, iPos, iPosNew;
    pexVector3D curVel;
    register int thingNum, i;
    float dist, minDist;

    if ( XPONG_DEBUG_CALLS ) 
    {
	printf("\nUpdateBallPosition -- ball \n");
    }


    /* Calculate new ball position. */
    curPos.x = ball->position.x;
    curPos.y = ball->position.y;
    curPos.z = ball->position.z;

    curVel.x = ball->velocity.x;
    curVel.y = ball->velocity.y;
    curVel.z = ball->velocity.z;

    do
    {
	thingNum = -1;		/* Nothing found yet. */

	/* Check the ball path against everything it could intersect with. */

	/* Get the distance to the endpoint. */
	nextPos.x = curPos.x + curVel.x;
	nextPos.y = curPos.y + curVel.y;
	nextPos.z = curPos.z + curVel.z;
	minDist = Distance(curPos, nextPos);

	/* Now check the possible intersections (PaddleIntersectItemIndex gives
	   the number of intersect item for the paddle including the walls. */
	for ( i = 0; i < PaddleIntersectItemIndex(numberPlayers); i++ )
	{
	    if ( GetIntersectionPoint(curPos, curVel, &iPosNew, ii[i]) )
	    {
		/* See what the distance to the intersect is. */
		dist = Distance(curPos, iPosNew);
		if ( dist < minDist )
		{
		    iPos = iPosNew;
		    thingNum = i;
		    minDist = dist;
		}
	    }
	} /* for */

	/* If we found something that intersects, then we will
	   move the ball to the intersection point, adjust the 
	   velocity and check the remaining path with the list of 
	   things to intersect again, until we get to the end of 
	   the path. */
	if ( thingNum != -1 )
	{
	    /* Give the player his score. */
	    if ( thingNum < numberPlayers )
	    {
		for ( i = 0; i < numberPlayers; i++ )
		    anyScoreChanged[i] = TRUE;
		playerScore[thingNum]++;
	    }

	    /* Mark this plane as having the ball bounce off it. */
	    ii[thingNum].showBounce = TIMES_TO_SHOW_BOUNCE;
	    ii[thingNum].interPt = iPos;

	    /* Get the remaining velocity. */
	    curVel.x -= iPos.x - curPos.x;
	    curVel.y -= iPos.y - curPos.y;
	    curVel.z -= iPos.z - curPos.z;

	    /* Move the current point to the intersection point. */
	    curPos = iPos;

	    /* Adjust the direction of the velocity for the bounce. */
	    switch ( ii[thingNum].planeKind )
	    {
	    case XPLANE:
		curVel.x *= -1.0;
		ball->velocity.x *= -1.0;
		break;
	    case YPLANE:
		curVel.y *= -1.0;
		ball->velocity.y *= -1.0;
		break;
	    case ZPLANE:
		curVel.z *= -1.0;
		ball->velocity.z *= -1.0;
		break;
	    } /* switch */

	} /* if (thingNum != -1 ) */

    } while ( thingNum != -1 );

    /* Update ball positions.  The velocity has already been taken care of. */
    ball->previousPosition = ball->position;

    ball->position.x = curPos.x + curVel.x;
    ball->position.y = curPos.y + curVel.y;
    ball->position.z = curPos.z + curVel.z;

}

/*****************************************************************
 * TAG( GetIntersectionPoint )
 * 
 * 	Fill in the next intersection point for the ball.
 * 
 * Inputs:
 * 	pos -- the position.
 *	vel -- the velocity to move in.
 *	ii -- an IntersectItem to check against.
 * Outputs:
 *	iPos -- the resulting intersection point.
 * 	returns TRUE if there is an intersection, FALSE otherwise.
 * Assumptions:
 *      the ball has been initialized, ii is set up properly.
 * Algorithm:
 * 	vector arithmetic.
 */
Bool
GetIntersectionPoint(pos, vel, iPos, ii)
    pexCoord3D pos, *iPos;
    pexVector3D vel;
    IntersectItem ii;
{
    pexCoord3D iPosTemp, nextPos;

    if ( XPONG_DEBUG_CALLS ) 
    {
	printf("\nGetIntersectionPoint -- ball \n");
    }

    (*iPos).x = 0.0;
    (*iPos).y = 0.0;
    (*iPos).z = 0.0;

    /* Make sure that we have some sort of velocity. */
    if ( (vel.x == 0.0) && (vel.y == 0.0) && (vel.z == 0.0) )

    {
	fprintf(stderr, "\nNo velocity for the ball\n");
	exit(1);
    }

    /* See if there is an intersection.  First the endpoints of the line 
       are checked against the min and max positions of the plane.  If they
       are both on one side or the other, then it isn't going to
       intersect. */

    /* Get what would be the next point. */
    nextPos.x = pos.x + vel.x;
    nextPos.y = pos.y + vel.y;
    nextPos.z = pos.z + vel.z;

    if ( ((pos.x <= ii.minX) && (nextPos.x < ii.minX)) || 
	((pos.x >= ii.maxX) && (nextPos.x > ii.maxX)) )
    {
	return FALSE;
    }
    
    if ( ((pos.y <= ii.minY) && (nextPos.y < ii.minY)) || 
	((pos.y >= ii.maxY) && (nextPos.y > ii.maxY)) )
    {
	return FALSE;
    }

    if ( ((pos.z <= ii.minZ) && (nextPos.z < ii.minZ)) || 
	((pos.z >= ii.maxZ) && (nextPos.z > ii.maxZ)) )
    {
	return FALSE;
    }

    switch (ii.planeKind )
    {
    case XPLANE:
	/* See where the point hits the plane x = ii.minX (same as ii.maxX). */
	iPosTemp.x = ii.minX;
	iPosTemp.y = pos.y + (vel.y / vel.x) * (ii.minX - pos.x);
	iPosTemp.z = pos.z + (vel.z / vel.x) * (ii.minX - pos.x);

	/* Now see if it is in the plane. */
	if ( (iPosTemp.y < ii.minY) || (iPosTemp.y > ii.maxY) ) 
	{
	    return FALSE;
	}
	if ( (iPosTemp.z < ii.minZ) || (iPosTemp.z > ii.maxZ) ) 
	{
	    return FALSE;
	}
	break;

    case YPLANE:
	/* See where the point hits the plane y = ii.minY (same as ii.maxY). */
	iPosTemp.x = pos.x + (vel.x / vel.y) * (ii.minY - pos.y);
	iPosTemp.y = ii.minY;
	iPosTemp.z = pos.z + (vel.z / vel.y) * (ii.minY - pos.y);

	/* Now see if it is in the plane. */
	if ( (iPosTemp.x < ii.minX) || (iPosTemp.x > ii.maxX) ) 
	{
	    return FALSE;
	}
	if ( (iPosTemp.z < ii.minZ) || (iPosTemp.z > ii.maxZ) ) 
	{
	    return FALSE;
	}
	break;

    case ZPLANE:
	/* See where the point hits the plane z = ii.minZ (same as ii.maxZ). */
	iPosTemp.x = pos.x + (vel.x / vel.z) * (ii.minZ - pos.z);
	iPosTemp.y = pos.y + (vel.y / vel.z) * (ii.minZ - pos.z);
	iPosTemp.z = ii.minZ;

	/* Now see if it is in the plane. */
	if ( (iPosTemp.x < ii.minX) || (iPosTemp.x > ii.maxX) ) 
	{
	    return FALSE;
	}
	if ( (iPosTemp.y < ii.minY) || (iPosTemp.y > ii.maxY) ) 
	{
	    return FALSE;
	}
	break;
    } /* switch */

    /* If we get here, we have a valid intersection point. */
    *iPos = iPosTemp;
    return TRUE;

}

/*****************************************************************
 * TAG( Distance )
 * 
 * 	Get the distance of the line going from position1 to position2.
 * 	
 * Inputs:
 * 	position1, position2 -- endpoints of the line.
 * Outputs:
 * 	returns the distance (always >= 0.0).
 * Assumptions:
 * 	[None]
 * Algorithm:
 * 	vector arithmetic.
 */
float
Distance(position1, position2)
    pexCoord3D position1, position2;
{
    float dx, dy, dz, distance;

    if ( XPONG_DEBUGGING ) 
    {
	printf("\nDistance -- position1, position2\n");
    }

    dx = position2.x - position1.x;
    dy = position2.y - position1.y;
    dz = position2.z - position1.z;

    distance = (float) sqrt(sqrt((double) ((dx * dx) + (dy * dy))) + 
			    (double) dz * dz);

    if ( XPONG_DEBUGGING )
    {
	printf("     distance: %f\n", distance);
    }

    return distance;
}


/*****************************************************************
 * TAG( ConvertSCToWC )
 *   
 * Inputs: 
 *      wCValue -- the point in World Coordinates.  
 * Outputs: 
 *	returns the point in World Coordinates.  
 * Assumptions: 
 *      Assumes that windowMin is set (which gives the smaller of
 *      the two dimensions of the window) and that the transformation from 
 *      World Coordinates to Screen Coordinates are going to be scaled 
 *      the same for each of the components of the points.  
 * Algorithm:
 *     Converts the points (which are in Screen Coordinates) to
 *     the World Coordinates.
 */
float
ConvertSCToWC(sCValue, playerNum)
    int sCValue;
    int playerNum;
{
    float wCValue;

    if ( XPONG_DEBUG_CALLS )
    {
	printf("\nConvertSCToWC -- \n");
	printf("      wCValue: %f\n", wCValue);
    }

    /* Map [(0+ED) to (windowMin-ED)] to [-1.0 to 1.0] */
    wCValue = (sCValue - EDGE_DISTANCE) /  (windowMin[playerNum] / 2.0 - 
					    EDGE_DISTANCE) - 1.0; 

    if ( XPONG_DEBUGGING )
    {
	printf("   converted value (wc): %f\n", wCValue);
    }

    return wCValue;
}

/*****************************************************************
 * TAG( PrintWCPoint )
 * 
 * 	Print out a point in World Coordinates
 * 	
 * Inputs:
 * 	point -- a point in World Coordinates.
 * Outputs:
 * 	(prints out the point on stdout).
 * Assumptions:
 *	[None]
 * Algorithm:
 *	If the Z value is zero, prints out only 2D point.
 */
void
PrintWCPoint(point)
    pexCoord3D point;
{
    printf("   [%f,%f,%f] \n", point.x, point.y, point.z);
}

/*****************************************************************
 * TAG( PrintViewValues )
 * 
 * 	Print out the values for the PEX viewing parameters.
 * 	
 * Inputs:
 * 	[None]
 * Outputs:
 *	[None]
 * Assumptions:
 *	Uses the global variables.
 * Algorithm:
 * 	[None]
 */
void
PrintViewValues()
{  
    printf("     VPN: "); PrintWCPoint(vpn);
    printf("     VRP: "); PrintWCPoint(vrp);
    printf("     VUP: "); PrintWCPoint(vup);

    printf("     map - window X (max,min): %f %f\n", map.window.xmax, map.window.xmin);
    printf("                  Y (max,min): %f %f\n", map.window.ymax, map.window.ymin);

    printf("     map - viewport X (max,min): %f %f\n", map.viewport.xmax, map.viewport.xmin);
    printf("                    Y (max,min): %f %f\n", map.viewport.ymax, map.viewport.ymin);
    printf("                    Z (max,min): %f %f\n", map.viewport.zmax, map.viewport.zmin);

    printf("           prp:"); PrintWCPoint(map.prp); 
    printf("           view_plane: %f\n  front_plane: %f\n back_plane %f\n", map.view_plane, map.front_plane, map.back_plane);

}

/*****************************************************************
 * TAG( GetViewValues )
 * 
 * 	Print out the values for the PEX viewing parameters.
 * 	
 * Inputs:
 * 	[None]
 * Outputs:
 *	Changes the values of the local variables.
 * Assumptions:
 *	Uses the global variables.
 * Algorithm:
 * 	[None]
 */
void
GetViewValues()
{
    float temp1, temp2, temp3, temp4;
    register int itemp;
    char ans[255];

    printf("Want VPN? ");
    scanf("%s", ans);
    if ( ans[0] == 'y' )
    {
	printf("VPN (x,y,z): ");
	if ( scanf("%f %f %f", &temp1, &temp2, &temp3) == 3 )
	{
	    vpn.x = temp1;
	    vpn.y = temp2; 
	    vpn.z = temp3;
	}
    }

    printf("Want VRP? ");
    scanf("%s", ans);
    if ( ans[0] == 'y' )
    {
	printf("VRP (x,y,z): ");
	if ( scanf("%f %f %f", &temp1, &temp2, &temp3) == 3 )
	{
	    vrp.x = temp1;
	    vrp.y = temp2; 
	    vrp.z = temp3;
	}
    }

    printf("Want VUP? ");
    scanf("%s", ans);
    if ( ans[0] == 'y' )
    {
	printf("VUP (x,y,z): ");
	if ( scanf("%f %f %f", &temp1, &temp2, &temp3) == 3 )
	{
	    vup.x = temp1;
	    vup.y = temp2; 
	    vup.z = temp3;
	}
    }

    printf("Want Map-window? ");
    scanf("%s", ans);
    if ( ans[0] == 'y' )
    {
	printf("Map-window X (max,min): ");
	if ( scanf("%f %f", &temp1, &temp2) == 2 )
	{
	    map.window.xmax = temp1;
	    map.window.xmin = temp2; 
	}
	printf("Map-window Y (max,min): ");
	if ( scanf("%f %f", &temp1, &temp2) == 2 )
	{
	    map.window.ymax = temp1;
	    map.window.ymin = temp2; 
	}
    }
    
    printf("Want Map-viewport? ");
    scanf("%s", ans);
    if ( ans[0] == 'y' )
    {
	printf("Map-viewport X (max,min): ");
	if ( scanf("%f %f", &temp1, &temp2) == 2 )
	{
	    map.viewport.xmax = temp1;
	    map.viewport.xmin = temp2; 
	}
	printf("Map-viewport Y (max,min): ");
	if ( scanf("%f %f", &temp1, &temp2) == 2 )
	{
	    map.viewport.ymax = temp1;
	    map.viewport.ymin = temp2; 
	}
	printf("Map-viewport Z (max,min): ");
	if ( scanf("%f %f", &temp1, &temp2) == 2 )
	{
	    map.viewport.zmax = temp1;
	    map.viewport.zmin = temp2; 
	}
    }
    
    printf("Want Map-prp? ");
    scanf("%s", ans);
    if ( ans[0] == 'y' )
    {
	printf("Map-prp (x,y,z): ");
	if ( scanf("%f %f %f", &temp1, &temp2, &temp3) == 3 )
	{
	    map.prp.x = temp1;
	    map.prp.y = temp2; 
	    map.prp.z = temp3;
	}
    }
    
    printf("Want planes? ");
    scanf("%s", ans);
    if ( ans[0] == 'y' )
    {
	printf("view_plane (x): ");
	if ( scanf("%f", &temp1) == 1 )
	{
	    map.view_plane = temp1;
	}
	printf("front_plane (x): ");
	if ( scanf("%f", &temp1) == 1 )
	{
	    map.front_plane = temp1;
	}
	printf("back_plane (x): ");
	if ( scanf("%f", &temp1) == 1 )
	{
	    map.back_plane = temp1;
	}
    }
    
}

/*****************************************************************
 * TAG( DrawPlayersRenderedPolyline )
 * 
 * 	Draws PexRenderPolylines for each of the players.
 * Inputs:
 *	[None]
 * Outputs:
 *	[None]
 * Assumptions:
 *	[None]
 * Algorithm:
 *	[None]
 */
void
DrawPlayersRenderedPolyline(playerNumber, polylinePoints, numPts)
    int playerNumber;	
    pexCoord3D polylinePoints[];
    int numPts;
{
    PexPolyline(Pexi[playerNumber], numPts, polylinePoints); 
}

/*****************************************************************
 * TAG( DrawPlayersFillArea )
 * 
 * 	Draws PexRenderPolylines for each of the players.
 * Inputs:
 *	[None]
 * Outputs:
 *	[None]
 * Assumptions:
 *	[None]
 * Algorithm:
 *	[None]
 */
void
DrawPlayersFillArea(playerNumber, polylinePoints, numPts)
    int playerNumber;	
    pexCoord3D polylinePoints[];
    int numPts;
{
if (solidModel[playerNumber]){
    PexFillAreaWithData(Pexi[playerNumber], Convex, 0, SurfaceEdgeSolid, 
			numPts, polylinePoints, NULL, NULL, NULL, NULL); 
			}
			else{
			polylinePoints[numPts]=polylinePoints[0];
    PexPolyline(Pexi[playerNumber], numPts+1, polylinePoints); 
			}
}


/*
 *****************************************************************
 *  Stuff for making cursors. 
 ***************************************************************** 
 */ 

extern Cursor XCreatePixmapCursor();

#define foo_width 1
#define foo_height 1
#define foo_x_hot 0
#define foo_y_hot 0
static char foo_bits[] = {
   0x01
   };

#define Foo_width 3
#define Foo_height 3
#define Foo_x_hot 1
#define Foo_y_hot 1
static char Foo_bits[] = {
   0x09
   };

/*****************************************************************
 * TAG( MakeCursor )
 * 
 * 	Make a little cursor for the windows.
 * 	
 * Inputs:
 * 	d -- the display.
 * 	w -- the window.
 * Outputs:
 * 	returns the new cursor.
 * Assumptions:
 *	[None]
 * Algorithm:
 *	[None]
 */
Cursor 
MakeCursor(d, w)
    Display *d;
    Window w;
{
    int numberCursors = 7;

    static int spiffyCursors[] = {
	XC_tcross,
	XC_iron_cross,
	XC_pirate,
	XC_gumby,
	XC_gobbler,
	XC_heart,
	XC_man,
    };
    
    Pixmap p;
    XColor b;
    char *cursorBusiness = NULL;
    
    /* Set the color we want this to be. */
    b.red = b.green = b.blue = 150;

    /* If this env variable is set then we can have different cursors. */
    cursorBusiness = (char *)getenv("XPONGCURSOR");
    if ( cursorBusiness == NULL )
    {  
	/* We want the teeny weeny cursor. */
	p = XCreateBitmapFromData( d, w, foo_bits, foo_width, foo_height );
	return XCreatePixmapCursor( d, p, p, &b, &b, foo_x_hot, foo_y_hot );
    }
    else if ( cursorBusiness[0] == 'A' )
    {
	return XCreateFontCursor(d, spiffyCursors[0]); 
    }
   else if ( cursorBusiness[0] == 'B' )
    {  
	p = XCreateBitmapFromData( d, w, Foo_bits, Foo_width, Foo_height );
	return XCreatePixmapCursor( d, p, p, &b, &b, Foo_x_hot, Foo_y_hot );
    }
    else if ( cursorBusiness[0] <= (numberCursors + '0') )
    {  
	int i = cursorBusiness[0];
	return XCreateFontCursor(d, spiffyCursors[i - '0']); 
    }
    else
    {
	/* We want the parent cursor. */
	return NULL;
    }
}


/*****************************************************************
 * TAG( SetPlayersCurrentLineColor )
  * 
 * 	Sets the given players current line color, saves the value
 * 	of the current color in the pexi globals for them, and 
 *	returns the old value of the color.
 * 	
 * Inputs:
 * 	playerNumber -- which player this is.
 * 	lineColor -- the new color to set.
 * Outputs:
 * 	sets the current line color in the Pexi globals for that player.
 *	returns the old color.
 * Assumptions:
 *	[None]
 * Algorithm:
 *	[None]
 */
int
SetPlayersCurrentLineColor(playerNumber, lineColor)
    int playerNumber;
    int lineColor;
{
    int saveLineColor;

    saveLineColor = Pexi[playerNumber]->lineColorIndex;

    Pexi[playerNumber]->lineColorIndex = lineColor;
    PexLineColorIndex(Pexi[playerNumber], Pexi[playerNumber]->lineColorIndex);
 
    return saveLineColor;
}


/*****************************************************************
 * TAG( SetPlayersCurrentSurfaceColor )
 * 
 * 	Sets the given players current surface color, saves the value
 * 	of the current color in the pexi globals for them, and 
 *	returns the old value of the color.
 * 	
 * Inputs:
 * 	playerNumber -- which player this is.
 * 	surfaceColor -- the new color to set.
 * Outputs:
 * 	sets the current surface color in the Pexi globals for that player.
 *	returns the old color.
 * Assumptions:
 *	[None]
 * Algorithm:
 *	[None]
 */
int
SetPlayersCurrentSurfaceColor(playerNumber, surfaceColor)
    int playerNumber;
    int surfaceColor;
{
    register int saveSurfaceColor;

    saveSurfaceColor = Pexi[playerNumber]->surfaceColorIndex;

    Pexi[playerNumber]->surfaceColorIndex = surfaceColor;
    PexSurfaceColorIndex(Pexi[playerNumber], 
			 Pexi[playerNumber]->surfaceColorIndex);

    return saveSurfaceColor;
}

/*****************************************************************
 * TAG( MakeAnotherWindowPlease )
 * 
 * 	Makes an X window for the given player on their display.
 * 	
 * Inputs:
 *      playerNumber -- the number of the player to make the window for.
 * 	windowTitle -- the name to give to the window (if there).
 *	x, y, w, h -- window positioning. 
 * Outputs:
 *	window is returned (from X).
 * Assumptions:
 *	[None]
 * Algorithm:
 *	[None]
 */
Window 
MakeAnotherWindowPlease(playerNumber, x, y, w, h, windowTitle)
    int playerNumber;
    int x, y, w, h; 	
    char *windowTitle;
{
    register Window winder;
    register Display *dpy;
    register int scrn;

    /* Use the display we already have for this guy... */
    dpy = Pexi[playerNumber]->phigsDisplay;

    scrn = DefaultScreen( dpy );

    winder = XCreateSimpleWindow( dpy, DefaultRootWindow( dpy ), x, y, w, h,
			    1, BlackPixel( dpy, scrn ),
			    WhitePixel( dpy, scrn ) );

    XSelectInput( dpy, winder, ExposureMask );
    XMapWindow( dpy, winder );
    XFlush(dpy);

    XChangeProperty(dpy, winder, XA_WM_NAME, XA_STRING, 8, 
		    PropModeReplace, " pexPoing score: ", 17);
    XChangeProperty(dpy, winder, XA_WM_ICON_NAME, XA_STRING, 8, 
		    PropModeReplace, " PEXPOING SCORE: ", 17);

    return (winder);
}




/*****************************************************************
 * TAG( stringLen )
 * 
 * 
 * Inputs:
 *	s -- a string (null terminated).
 * Outputs:
 *	returns the length of the string.
 * Assumptions:
 *	[None]
 * Algorithm:
 *	[None]
 */

stringLen(s)
    char *s;
{
    register int i = 0;
    while ( s[i] != '\0' ) 
	i++;
    return i;
}
