/* 
 * 3Dpolyl.c - Machine Independant 3D Polyline drawing function ("dd" layer)
 * 
 * Copyright 1988
 * Center for Information Technology Integration (CITI)
 * Information Technology Division
 * University of Michigan
 * Ann Arbor, Michigan
 *
 *                         All Rights Reserved
 * 
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose and without fee is hereby
 * granted, provided that the above copyright notice appear in all
 * copies and that both that copyright notice and this permission
 * notice appear in supporting documentation, and that the names of
 * CITI or THE UNIVERSITY OF MICHIGAN not be used in advertising or
 * publicity pertaining to distribution of the software without
 * specific, written prior permission.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS." CITI AND THE UNIVERSITY OF
 * MICHIGAN DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN
 * NO EVENT SHALL CITI OR THE UNIVERSITY OF MICHIGAN BE LIABLE FOR ANY
 * SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN
 * AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING
 * OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 */

#include <stdio.h>
#include "X.h"
#include "pixmap.h"
#include "gc.h"
#include "miscstruct.h"
#include "gcstruct.h"
#include "extnsionst.h"
#include "dix.h"
#include "dixstruct.h"
#include "resource.h"
#include "colormap.h"

#include "PEX.h"
#include "PEXproto.h"
#include "pubstr.h"
#include "polyline.h"
#include "renderer.h"
#include "colortable.h"
#include "pexmath.h"
#include "mipex.h"

/*****************************************************************
 * TAG( miPexPolyline3D )
 * 
 * Draws polylines through an X graphics context to maintain compatibility
 * with X machines which have no dd layer.
 * 
 * Inputs:
 * 	pRend - ptr to a renderer
 * 	pPoly - ptr to a polyline struct
 * Outputs:
 * 	an integer error code or zero.
 * Assumptions:
 *	[None]
 * Algorithm:
 *	[None]
 */

#define ARRAY_SIZE 400

int
miPexPolyline3D(pRend, pPoly)
    pexRendererPtr pRend;
    pexPolylinePtr pPoly;
{
    DrawablePtr pDraw;
    GCPtr pGC;
    Pixel index;
    
    pDraw = pRend->pDraw;
    pGC =  pRend->pGC;

    miPexSetGCLineValue(pGC, pDraw,
			pRend->pPC->lineColor.color.format.indexed.index,
			index);

    miPexDrawPolypts( PexPolylineVertex( pPoly, 0 ), FALSE, pPoly->numVertices,
		      pPoly->vertexStride, pRend );

    return (Success);
}


/*****************************************************************
 * TAG( miPexDrawPolypts )
 * 
 * Transform, clip, and draw a polyline from an array of points.
 * Inputs:
 * 	wPts:		Input points.
 * 	inNPC:		If TRUE, points are in NPC already (and are of
 * 			type pexCoord4D).
 * 	nPts:		Number of points.
 * 	stride:		Offset between points.
 * 	pRend:		renderer to use.
 * Outputs:
 * 	Draws it.
 * Assumptions:
 *	[None]
 * Algorithm:
 *	[None]
 */
miPexDrawPolypts( wPts, inNPC, nPts, stride, pRend )
register pexCoord3D *wPts;
Bool inNPC;
int nPts;
int stride;
pexRendererPtr pRend;
{
    CARD32 i = 0;			/* Loop counter */
    CARD32 BufferCounter = 0;		/* always starts at zero */
    CARD32 Continuing = 0;		/* Haven't run yet */
    CARD32 BigCounter = 0;


    static pexCoord4D NpcPoints[ARRAY_SIZE];	/* Transformed points */
						/* stored here */
    register pexCoord3D * inPt;
    register pexCoord4D * npcPt;

					/* Booleans used for clipping */
    int change1, change2;		/* TRUE if endpoint moved */
    int curr_out, prev_out;			/* outcode for point */
    int lvis;				/* TRUE if line visible */
    DDXPointRec ddxpt[ARRAY_SIZE];	/* GC coordinates */
    DrawablePtr pDraw;
    GCPtr pGC;

    pDraw = pRend->pDraw;
    pGC = pRend->pGC;

    if ( inNPC )
	npcPt = (pexCoord4D *)wPts;
    else
	inPt = wPts;

    while ( (BigCounter < nPts) )	/* only check counter at top */
    {
	if ( Continuing )
	{
	    printf("Continuing in miPolyline\n");
	    NpcPoints[0] = NpcPoints[i-1];
	    ddxpt[0] = ddxpt[i-1];
	}

	for(i = Continuing, lvis = 0;
	    ((i < ARRAY_SIZE) && (BigCounter < nPts));
	    i++, BigCounter++,
	    (inNPC ?
	     (char *)(npcPt = (pexCoord4D *)((char *)npcPt + stride)) :
	     (char *)(inPt = (pexCoord3D *)((char *)inPt + stride))),
	    prev_out = curr_out)
	{
	    /* Transform to NPC space */
	    if ( inNPC )
		NpcPoints[i] = *npcPt;
	    else
		PexTransformPoint( pRend, inPt, &NpcPoints[i] );

	    curr_out = PexClipPoint(NpcPoints[i]);
	    if (!i)		/* Clip against constant */
		continue;

	    if ( curr_out & prev_out )	/* only happens when i = 1 */
		break;

	    if ( curr_out | prev_out )	/* At least one point is out, clip */
	    {
		lvis = PexClipLine(&NpcPoints[i-1],
				   &NpcPoints[i],
				   &change1, &change2 );

		if ( !lvis )
		    break;
	    }
	    lvis = 1;

	    /* Transform previous point now, didn't do it earlier */
	    if( NpcPoints[i-1].w != 1.0 )
	    {
		NpcPoints[i-1].x /= NpcPoints[i-1].w;
		NpcPoints[i-1].y /= NpcPoints[i-1].w;
	    };

	    /* Transform to physical device coordinates */
	    PexTransformNpcToDev(pRend,
			NpcPoints[i-1].x, NpcPoints[i-1].y,
			&ddxpt[i-1].x, &ddxpt[i-1].y );

	    if( curr_out ||
		(i == ARRAY_SIZE - 1) || (BigCounter == nPts - 1) )
	    {
		/* Do the perspective divide on clipped current point */
		if( NpcPoints[i].w != 1.0 )
		{
		    NpcPoints[i].x /= NpcPoints[i].w;
		    NpcPoints[i].y /= NpcPoints[i].w;
		}
		
		/* Transform to physical device coordinates */
		PexTransformNpcToDev(pRend,
			    NpcPoints[i].x, NpcPoints[i].y,
			    &ddxpt[i].x, &ddxpt[i].y );
		i++;
		break;
	    }
	}
	/*
	 * Here the above loop has dropped out.  If the line has
	 * crossed out of the clipping boundry we must draw it now
	 * before starting a new line.  In the case that the loop is
	 * waiting for the polyline to reenter the clipping volume,
	 * lvis will be 0.  We must not draw if there is only one
	 * segment and it is not visible.
	 */
	if (i == ARRAY_SIZE)
	    Continuing = 1;
	else
	    Continuing = 0;
	
	if ( lvis )
	    /* Draw the polyline on the display */
	    (*GetGCValue(pGC, Polylines)) (pDraw, pGC,
					   CoordModeOrigin,
					   i, ddxpt);
    }
}
