/* 
 * quadmesh.c - handle quad mesh OC requests
 * 
 * Copyright 1988
 * Center for Information Technology Integration (CITI)
 * Information Technology Division
 * University of Michigan
 * Ann Arbor, Michigan
 *
 *                         All Rights Reserved
 * 
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose and without fee is hereby
 * granted, provided that the above copyright notice appear in all
 * copies and that both that copyright notice and this permission
 * notice appear in supporting documentation, and that the names of
 * CITI or THE UNIVERSITY OF MICHIGAN not be used in advertising or
 * publicity pertaining to distribution of the software without
 * specific, written prior permission.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS." CITI AND THE UNIVERSITY OF
 * MICHIGAN DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN
 * NO EVENT SHALL CITI OR THE UNIVERSITY OF MICHIGAN BE LIABLE FOR ANY
 * SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN
 * AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING
 * OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 */

#include <stdio.h>
#include "X.h"
#include "pixmap.h"
#include "gc.h"
#include "miscstruct.h"
#include "gcstruct.h"
#include "extnsionst.h"
#include "dix.h"
#include "dixstruct.h"
#include "resource.h"
#include "colormap.h"

#include "PEX.h"
#include "PEXproto.h"
#include "pubstr.h"
#include "quadmesh.h"
#include "renderer.h"
#include "colortable.h"
#include "pexmath.h"
#include "mipex.h"

/*****************************************************************
 * TAG( miPexQuadMesh3DWire )
 * 
 * Draws an array of polygons through an X graphics context to maintain
 * compatibility with X machines which have no dd layer.
 * 
 * Inputs:
 * 	pRend - ptr to a renderer
 * 	pPoly - ptr to a quadmesh struct
 * Outputs:
 * 	an integer error code or zero.
 * Assumptions:
 *	[None]
 * Algorithm:
 *	[None]
 */

#define MaxVertCount 10

extern CARD32 PexClipPolygon ();

int
miPexQuadMesh3DWire(pRend, pStuff)
    pexRendererPtr pRend;
    pexQuadMeshPtr pStuff;
{
    register CARD32 m, n, i;	/* Loop counter */
    CARD32 vertsVis;            /* # of visible verts after clipping */
    pexCoord3D *tpt1;           /* unTransformed points stored here */
    pexCoord4D pt1[MaxVertCount],     	/* Transformed points stored here */
       cpt[MaxVertCount];       /* clipped points stored here */
    DDXPointRec ddxpt[MaxVertCount];	/* GC coordinate endpoints */
    DrawablePtr pDraw;
    GCPtr pGC;
    Pixel index;
    pexQuadMeshPtr npcMesh, PexTransformQuadmesh ();
    
    pDraw = pRend -> pDraw;
    pGC =  pRend -> pGC;

    miPexSetGCLineValue(pGC, pDraw,
			pRend->pPC->surfaceColor.color.format.indexed.index,
			index);

    npcMesh = PexTransformQuadmesh(pRend, pStuff);

    if (!npcMesh)
	return BadAlloc;
    
    /* Draw the rows */
    for (m = 0; m < npcMesh->mPoints; m++)
	miPexDrawPolypts( PexQuadMeshVertex(npcMesh,m,0), TRUE,
			  npcMesh->nPoints, npcMesh->vertexNStride, pRend );

    for (n = 0; n < npcMesh->nPoints; n++)
	miPexDrawPolypts( PexQuadMeshVertex(npcMesh,0,n), TRUE,
			  npcMesh->mPoints, npcMesh->vertexMStride, pRend );


    FreePexQuadMesh( npcMesh );

    return (Success);
}





int
miPexQuadMesh3DFlat(pRend, pStuff)
    pexRendererPtr pRend;
    pexQuadMeshPtr pStuff;
{
    register CARD32 m, n, i;	/* Loop counter */
    CARD32 vertsVis;            /* # of visible verts after clipping */
    pexCoord3D *tpt1;           /* unTransformed points stored here */
    pexCoord4D pt1[MaxVertCount],     	/* Transformed points stored here */
    cpt[MaxVertCount];		/* clipped points stored here */
    DDXPointRec ddxpt[MaxVertCount];	/* GC coordinate endpoints */
    register pexCoord4D *npt;
    register CARD32 code, incode, outcode;
    DrawablePtr pDraw;
    GCPtr pGC;
    pexVector3D normal;
    int shade;
    Pixel index;
    pexQuadMeshPtr npcMesh, PexTransformQuadmesh ();
    
    pDraw = pRend -> pDraw;
    pGC =  pRend -> pGC;

    npcMesh = PexTransformQuadmesh(pRend, pStuff);

    if (!npcMesh)
	return BadAlloc;
    
    for (m=0; m<(pStuff->mPoints)-1; m++)
	for (n=0; n<(pStuff->nPoints)-1; n++)
	{
	    pt1[0] = *(pexCoord4D *)PexQuadMeshVertex (npcMesh, m, n);
	    pt1[1] = *(pexCoord4D *)PexQuadMeshVertex (npcMesh, m+1, n);
	    pt1[2] = *(pexCoord4D *)PexQuadMeshVertex (npcMesh, m+1, n+1);
	    pt1[3] = *(pexCoord4D *)PexQuadMeshVertex (npcMesh, m, n+1);

	    outcode = ~0; incode = 0;
	    for (i = 0; i < 4; i++)
	    {
		code = PexClipPoint(pt1[i]);
		outcode &= code;
		incode |= code;
	    };
	    
	    if (outcode)
		continue;

	    /* Check to see if we're culling, and if we are, check to see if
             * this facet should be culled. If so, get out fast.
	     */
	    if ((pRend -> pPC -> cullMode == BackFaces) &&
		PexCullFacet (pt1, 4))
		continue;

	    if (incode)
	    {
		vertsVis = PexClipPolygon (4, pt1, MaxVertCount, cpt);
		npt = cpt;
	    }
	    else
	    {
		vertsVis = 4;
		npt = pt1;
	    }

	    if (vertsVis)
	    {
		for(i=0; i<vertsVis; i++)
		{
		    /* Do the perspective divide */
		    if( npt[i].w != 1.0 )
		    {
			npt[i].x /= npt[i].w;
			npt[i].y /= npt[i].w;
		    }
	    
		    /* Transform to physical device coordinates */
		    PexTransformNpcToDev( pRend, npt[i].x, npt[i].y, &ddxpt[i].x, &ddxpt[i].y );
		}
    
		/* Get a normal (one of these will be set) */
		if ( npcMesh->facetAttributes & GANormal )
		    normal = * PexQuadMeshFacetNorm(npcMesh, m, n);
		else if ( npcMesh->vertexAttributes & GANormal )
		    normal = * PexQuadMeshVertexNorm(npcMesh, m, n);
		else
		    fprintf (stderr, "ERROR in quad mesh 3D flat shading -- no normals at facet %d, %d.\n", m, n);
		
		shade = PexCalcShade(pRend, &normal);
		miPexSetGCValues(pGC, pDraw,
				 pRend->pPC->surfaceColor.color.format.indexed.index,
				 shade, index);

		/* Draw the area on the display */
		(*GetGCValue(pGC, FillPolygon)) (pDraw, pGC, Complex,
						 CoordModeOrigin, vertsVis,
						 ddxpt);
	    }
	}

    FreePexQuadMesh(npcMesh);

    return (Success);
}
	    


int miPexQuadMesh3DSmooth(pRend, pStuff)
    pexRendererPtr pRend;
    pexQuadMeshPtr pStuff;
{
    register CARD32 m, n, i;	/* Loop counter */
    CARD32 vertsVis;            /* # of visible verts after clipping */
    pexCoord3D *tpt1;           /* unTransformed points stored here */
    pexCoord4D pt1[MaxVertCount],     	/* Transformed points stored here */
    cpt[MaxVertCount];		/* clipped points stored here */
    DDXPointRec ddxpt[MaxVertCount];	/* GC coordinate endpoints */
    DrawablePtr pDraw;
    GCPtr pGC;
    pexVector3D normal;
    int shade;
    Pixel index;
    pexQuadMeshPtr npcMesh, PexTransformQuadmesh ();
    int cornerShades[4];
    
    if ( !(pStuff->vertexAttributes & GANormal) )
	return miPexQuadMesh3DFlat(pRend, pStuff);	/* bug out if flat */

    pDraw = pRend -> pDraw;
    pGC =  pRend -> pGC;

    npcMesh = PexTransformQuadmesh(pRend, pStuff);

    if (!npcMesh)
	return BadAlloc;

    for (m=0; m<(pStuff->mPoints)-1; m++)
	for (n=0; n<(pStuff->nPoints)-1; n++)
	{
	    pt1[0] = *(pexCoord4D *)PexQuadMeshVertex (npcMesh, m, n);
	    pt1[1] = *(pexCoord4D *)PexQuadMeshVertex (npcMesh, m+1, n);
	    pt1[2] = *(pexCoord4D *)PexQuadMeshVertex (npcMesh, m+1, n+1);
	    pt1[3] = *(pexCoord4D *)PexQuadMeshVertex (npcMesh, m, n+1);

	    cornerShades[0] = 
		PexCalcShade( pRend,
				PexQuadMeshVertexNorm(npcMesh, m, n ) );
	    cornerShades[1] = 
		PexCalcShade( pRend,
				PexQuadMeshVertexNorm(npcMesh, m+1, n ) );
	    cornerShades[2] = 
		PexCalcShade( pRend,
				PexQuadMeshVertexNorm(npcMesh, m+1, n+1 ) );
	    cornerShades[3] = 
		PexCalcShade( pRend,
				PexQuadMeshVertexNorm(npcMesh, m, n+1 ) );

	    if ( !miPexPolyptSmooth(
		pt1, cornerShades, 4, pRend,
		pRend->pPC->surfaceColor.color.format.indexed.index) )
	    {
		/* culling is done with miPexPolyptSmooth */

		vertsVis = PexClipPolygon (4, pt1, MaxVertCount, cpt);

		if (vertsVis)
		{
		    for(i=0; i<vertsVis; i++)
		    {
			/* Do the perspective divide */
			if( cpt[i].w != 1.0 )
			{
			    cpt[i].x /= cpt[i].w;
			    cpt[i].y /= cpt[i].w;
			}
	    
			/* Transform to physical device coordinates */
			PexTransformNpcToDev( pRend, cpt[i].x, cpt[i].y,
				     &ddxpt[i].x, &ddxpt[i].y );
		    }
    
		    miPexSetGCValues(pGC, pDraw,
				     pRend->pPC->surfaceColor.color.format.indexed.index,
				     cornerShades[0], index);
		    /* Draw the area on the display */
		    (*GetGCValue(pGC, FillPolygon)) (pDraw, pGC, Complex,
						     CoordModeOrigin, vertsVis,
						     ddxpt);
		}
	    }
	}

    FreePexQuadMesh(npcMesh);

    return (Success);
}
