/* 
 * qmhidesqr.c - Routines dealing with classification of facets into squares.
 * 
 * Copyright 1988
 * Center for Information Technology Integration (CITI)
 * Information Technology Division
 * University of Michigan
 * Ann Arbor, Michigan
 *
 *                         All Rights Reserved
 * 
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose and without fee is hereby
 * granted, provided that the above copyright notice appear in all
 * copies and that both that copyright notice and this permission
 * notice appear in supporting documentation, and that the names of
 * CITI or THE UNIVERSITY OF MICHIGAN not be used in advertising or
 * publicity pertaining to distribution of the software without
 * specific, written prior permission.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS." CITI AND THE UNIVERSITY OF
 * MICHIGAN DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN
 * NO EVENT SHALL CITI OR THE UNIVERSITY OF MICHIGAN BE LIABLE FOR ANY
 * SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN
 * AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING
 * OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 */

#include <stdio.h>
#include "PEX.h"
#include "PEXprotostr.h"
#include "pubstr.h"
#include "quadmesh.h"
#include "qmhide.h"

static qm_square squares[NSQUARE][NSQUARE];

/*****************************************************************
 * TAG( facet_sqrs )
 * 
 * Return the set of squares covered by a facet.
 * Inputs:
 * 	facet:		Facet to test.
 * Outputs:
 * 	minx_ret,etc.:	Minimum and maximum indices for the squares
 * 			possibly intersected by the facet.
 * 	Returns:	TRUE if facet is in the window, FALSE otherwise.
 * Assumptions:
 *	[None]
 * Algorithm:
 * 	Figure out the set of squares covered by the bounding box.
 */
Bool
facet_sqrs( facet, minx_ret, maxx_ret, miny_ret, maxy_ret )
qm_facet *facet;
int *minx_ret;
int *maxx_ret;
int *miny_ret;
int *maxy_ret;
{
    float minval, maxval;

    minval = facet->bbox[0][0];
    if ( minval < -1 ) minval = -1;
    maxval = facet->bbox[0][1];
    if ( maxval > 1 ) maxval = 1;
    if ( maxval <= minval )
	return FALSE;
    /* Note: won't have minval == 1 or maxval == -1, so no rounding problems */
    *minx_ret = ((minval + 1) / 2) * NSQUARE;
    *maxx_ret = NSQUARE - 1 - (int)(((1 - maxval) / 2) * NSQUARE);

    minval = facet->bbox[1][0];
    if ( minval < -1 ) minval = -1;
    maxval = facet->bbox[1][1];
    if ( maxval > 1 ) maxval = 1;
    if ( maxval <= minval )
	return FALSE;
    /* Note: won't have minval == 1 or maxval == -1, so no rounding problems */
    *miny_ret = ((minval + 1) / 2) * NSQUARE;
    *maxy_ret = NSQUARE - 1 - (int)(((1 - maxval) / 2) * NSQUARE);

    return TRUE;
}


/*****************************************************************
 * TAG( ins_sqr_facet )
 * 
 * Insert a facet into the squares it covers.
 * Inputs:
 * 	facet:		Facet to be inserted.
 * Outputs:
 * 	Adds facet to all squares covered by it.
 * Assumptions:
 *	[None]
 * Algorithm:
 * 	Call facet_sqrs to get range, add facet pointer to each one.
 */
Bool
ins_sqr_facet( facet )
qm_facet *facet;
{
    int minx, maxx, miny, maxy, i, j, n, n2;
    register qm_square *qs;

    if ( !facet_sqrs( facet, &minx, &maxx, &miny, &maxy ) )
	return FALSE;

    for ( i = minx; i <= maxx; i++ )
    {
	if ( i < 0 || i >= NSQUARE )
	    continue;
	for ( j = miny; j <= maxy; j++ )
	{
	    if ( j < 0 || j >= NSQUARE )
		continue;
	    qs = &squares[j][i];
	    n = qs->nfacet;
	    qs->nfacet++;
	    if ( n < NDIRECT )
		qs->direct[n] = facet;
	    else if ( (n -= NDIRECT) < NINDIRECT )
	    {
		if ( n == 0 )	/* allocate indirect block */
		    qs->indirect =
			(qm_facet **)xalloc( sizeof(qm_facet *) * NINDIRECT );
		qs->indirect[n] = facet;
	    }
	    else
	    {
		n -= NINDIRECT;
		if ( n == 0 )
		    qs->dbl_i = (qm_facet ***)
			xalloc( sizeof(qm_facet **) * NINDIRECT );
		/* Find which double indirect block to use */
		for ( n2 = 0; n >= NINDIRECT; n2++ )
		    n -= NINDIRECT;

		if ( n == 0 )
		    qs->dbl_i[n2] = (qm_facet **)
			xalloc( sizeof(qm_facet *) * NINDIRECT );

		qs->dbl_i[n2][n] = facet;
	    }
	}
    }

    return TRUE;
}

/*****************************************************************
 * TAG( get_sqr_facet )
 * 
 * Return the nth facet from a square.
 * Inputs:
 * 	x, y:		Specify square number.
 * 	n:		Facet within square.
 * Outputs:
 * 	Returns facet pointer, or NULL if n is out of range.
 * Assumptions:
 *	[None]
 * Algorithm:
 *	[None]
 */
qm_facet *
get_sqr_facet( x, y, n )
int n;
{
    register qm_square * qs;
    register int n2;

    if ( x < 0 || x >= NSQUARE || y < 0 || y >= NSQUARE
	 || n < 0 || n >= (qs = &squares[y][x])->nfacet )
	return NULL;

    if ( n < NDIRECT )
	return qs->direct[n];
    else if ( (n -= NDIRECT) < NINDIRECT )
	return qs->indirect[n];
    else
    {
	n -= NINDIRECT;

	/* Find which double indirect block to use */
	for ( n2 = 0; n >= NINDIRECT; n2++ )
	    n -= NINDIRECT;

	return qs->dbl_i[n2][n];
    }
    
}

/*****************************************************************
 * TAG( qm_free_sqr )
 * 
 * Free all indirect blocks and reset counts in the square.
 * Inputs:
 *	[None]
 * Outputs:
 *	[None]
 * Assumptions:
 *	[None]
 * Algorithm:
 *	[None]
 */
void
qm_free_sqr()
{
    register qm_square * qs;
    register int n, n2;
    int i;

    for ( i = 0, qs = (qm_square *)squares; i < NSQUARE*NSQUARE; i++, qs++ )
    {
	if ( qs->nfacet <= NDIRECT )
	    ;			/* nothing to do */
	else
	{
	    n = qs->nfacet - NDIRECT;
	    xfree( qs->indirect );

	    if ( n > NINDIRECT )
	    {
		for ( n2 = 0, n -= NINDIRECT; n > 0;
		      n2++, n -= NINDIRECT )
		    xfree( qs->dbl_i[n2] );
		xfree( qs->dbl_i );
	    }
	}
	qs->nfacet = 0;
    }
}

