/* 
 * DPClip.c - volume clipping routines
 * 
 * Copyright 1988
 * Center for Information Technology Integration (CITI)
 * Information Technology Division
 * University of Michigan
 * Ann Arbor, Michigan
 *
 *                         All Rights Reserved
 * 
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose and without fee is hereby
 * granted, provided that the above copyright notice appear in all
 * copies and that both that copyright notice and this permission
 * notice appear in supporting documentation, and that the names of
 * CITI or THE UNIVERSITY OF MICHIGAN not be used in advertising or
 * publicity pertaining to distribution of the software without
 * specific, written prior permission.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS." CITI AND THE UNIVERSITY OF
 * MICHIGAN DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN
 * NO EVENT SHALL CITI OR THE UNIVERSITY OF MICHIGAN BE LIABLE FOR ANY
 * SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN
 * AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING
 * OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 */

#include "X.h"
#include "misc.h"

/*****************************************************************
 * TAG( PexClipLine )
 * 
 * Clips lines in homogeneous coordinates.
 * 
 * Inputs:
 * 	pt1     Unclipped point 1.
 * 	pt2     Unclipped point 2.
 * Outputs:
 * 	Points clipped.
 * 	change1 true if pt1 was modified.
 * 	change2 true if pt2 was modified.
 * 	Returns true if anything visible, else False.
 * Assumptions:
 * 	clip region is -w<x<w, -w<y<w, 0<z<w.
 * Algorithm:
 *      We use a 3-D Cohen-Sutherland algorithm,  this closely
 *      follows Foley and VanDam, pg 297.  However, we clip
 * 	on w instead of the view frustrum.
 */

int
PexClipLine(pt1, pt2, change1, change2)
    register float pt1[4], pt2[4];
    int *change1, *change2;
{
    register unsigned outcode1;	/* Outcode for point 1 */
    register unsigned outcode2;	/* Outcode for point 2 */

    int swapped;		/* Flag indicating coordinates swapped */

    float tmpf;			/* Temporary used for swapping points */
    float tmpi;			/* Temporary used for codes */
    float x1,y1,z1,w1;		/* Point 1 */
    float x2,y2,z2,w2;		/* Point 2 */

    *change1 = FALSE;
    *change2 = FALSE;

#define x 0
#define y 1
#define z 2
#define w 3

    /* Calculate outcodes of line endpoints do this shit fast! */
    outcode1 =
	((pt1[y] >  pt1[w])       | /* Bit 1 - point is above view volume */
	 (pt1[y] < -pt1[w])  << 1 | /* Bit 2 - point is below view volume */
	 (pt1[x] >  pt1[w])  << 2 | /* Bit 3 - point is right of view volume */
	 (pt1[x] < -pt1[w])  << 3 | /* Bit 4 - point is left of view volume */
	 (pt1[z] >  pt1[w])  << 4 | /* Bit 5 - point is behind view volume */
	 (pt1[z] < 0.0)	     << 5); /* Bit 6 - point is front of view volume */
    
    outcode2 = 
	((pt2[y] >  pt2[w])       | /* Bit 1 - point is above view volume */
	 (pt2[y] < -pt2[w])  << 1 | /* Bit 2 - point is below view volume */
	 (pt2[x] >  pt2[w])  << 2 | /* Bit 3 - point is right of view volume */
	 (pt2[x] < -pt2[w])  << 3 | /* Bit 4 - point is left of view volume */
	 (pt2[z] >  pt2[w])  << 4 | /* Bit 5 - point is behind view volume */
	 (pt2[z] < 0.0)      << 5); /* Bit 6 - point is front of view volume */
    
    /* Initially nothing swapped, nothing changed */
    swapped = FALSE;

    /* Trivial cases... */
    if( !outcode1 && !outcode2 )  /* Both outcodes 0x00, accept */
	return(TRUE);
    
    if( outcode1 & outcode2 )     /* Trivial reject */
	return(FALSE);

    /* Now that our case is not trivial */
    /* Copy points into local variables */

    x1 = pt1[0];  y1 = pt1[1];  z1 = pt1[2]; w1 = pt1[3];
    x2 = pt2[0];  y2 = pt2[1];  z2 = pt2[2]; w2 = pt2[3];

    for(;;) {
        
        /* Subdivide line since at most one endpoint is inside.
         * First, guarantee that p1 is outside the window.
         */

        if( !outcode1 ) {    /* Swap p1, p2,  so as not to replicate code???*/
            swapped = !swapped;
            tmpf = x1; x1 = x2; x2 = tmpf;
            tmpf = y1; y1 = y2; y2 = tmpf;
            tmpf = z1; z1 = z2; z2 = tmpf;
            tmpf = w1; w1 = w2; w2 = tmpf;
            tmpi = outcode1; outcode1 = outcode2; outcode2 = tmpi;
            *change2 = *change1;
        }
	
        /* Now perform a subdivision, move p1 to the intersection point;
         * use the formulas from Foley and VanDam, pg 297.
         */
       	*change1 = TRUE;
        if( outcode1 & 0x01 ) {  /* Divide line at y=w plane */
            tmpf = (w1-y1) / ((w1-y1) - (w2-y2));
            x1   = (x2-x1) * tmpf + x1;
            y1   = (y2-y1) * tmpf + y1;
            z1   = (z2-z1) * tmpf + z1;
	    w1   = y1;
        }

        else if( outcode1 & 0x02 ) {  /* Divide line at y= -w plane */
            tmpf = (w1+y1) / ((w1+y1) - (w2+y2));
            x1   = (x2-x1) * tmpf + x1;
            y1   = (y2-y1) * tmpf + y1;
            z1   = (z2-z1) * tmpf + z1;
            w1   = -y1;
        }
	
        else if( outcode1 & 0x04 ) {  /* Divide line at x=w plane */
            tmpf = (w1-x1) / ((w1-x1) - (w2-x2));
            x1   = (x2-x1) * tmpf + x1;
            y1   = (y2-y1) * tmpf + y1;
            z1   = (z2-z1) * tmpf + z1;
            w1   = x1;
        }

        else if( outcode1 & 0x08 ) {  /* Divide line at x= -w plane */
            tmpf = (w1+x1) / ((w1+x1) - (w2+x2));
            x1   = (x2-x1) * tmpf + x1;
            y1   = (y2-y1) * tmpf + y1;
            z1   = (z2-z1) * tmpf + z1;
            w1   = -x1;
        }

        else if( outcode1 & 0x10 ) {  /* Divide line at z=w plane */
            tmpf = (w1-z1) / ((w1-z1) - (w2-z2));
            x1   = (x2-x1) * tmpf + x1;
            y1   = (y2-y1) * tmpf + y1;
            z1   = (z2-z1) * tmpf + z1;
            w1   = z1;
        }

        else if( outcode1 & 0x20 ) {  /* Divide line at z=0 plane */
            tmpf = (z1) / (z1-z2);
            x1   = (x2-x1) * tmpf + x1;
            y1   = (y2-y1) * tmpf + y1;
            z1   = 0.0;
            w1   = (w2-w1) * tmpf + w1;
        }
	
	/* Calculate outcodes of line endpoints */
	outcode1 =
	    ((y1 >  w1)       |  /* Bit 1 - point is above view volume */
	     (y1 < -w1)  << 1 |  /* Bit 2 - point is below view volume */
	     (x1 >  w1)  << 2 |  /* Bit 3 - point is right of view volume */
	     (x1 < -w1)  << 3 |  /* Bit 4 - point is left of view volume */
	     (z1 >  w1)  << 4 |  /* Bit 5 - point is behind view volume */
	     (z1 < 0.0)  << 5);  /* Bit 6 - point is front of view volume */
	     
	outcode2 = 
	    ((y2 >  w2)       |  /* Bit 1 - point is above view volume */
	     (y2 < -w2)  << 1 |  /* Bit 2 - point is below view volume */
	     (x2 >  w2)  << 2 |  /* Bit 3 - point is right of view volume */
	     (x2 < -w2)  << 3 |  /* Bit 4 - point is left of view volume */
	     (z2 >  w2)  << 4 |  /* Bit 5 - point is behind view volume */
	     (z2 < 0.0)  << 5);  /* Bit 6 - point is front of view volume */
	     
	/* Trivial cases... */
	if( !outcode1 && !outcode2 ) { /* Both outcodes 0x00, accept */
	    if( swapped ) {
		pt1[0] = x2;  pt1[1] = y2;  pt1[2] = z2;  pt1[3] = w2;
		pt2[0] = x1;  pt2[1] = y1;  pt2[2] = z1;  pt2[3] = w1;
		tmpi = *change1; *change1 = *change2; *change2 = tmpi;
	    }
	    else {
		pt1[0] = x1;  pt1[1] = y1;  pt1[2] = z1;  pt1[3] = w1;
		pt2[0] = x2;  pt2[1] = y2;  pt2[2] = z2;  pt2[3] = w2;
	    }
	    return(TRUE);
	}
        
	if( outcode1 & outcode2 ) {     /* Trivial reject */
	    return(FALSE);
	}
        
    }
}
