#include "gfto.h"
/*
 * Copyright 1988 Barry Shein
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted,
 * provided that the above copyright notice appear in all copies and that
 * both that copyright notice and this permission notice appear in
 * supporting documentation, and that the name of Barry Shein not be used
 * in advertising or publicity pertaining to distribution of the software
 * without specific, written prior permission.  Barry Shein makes no
 * representations about the suitability of this software for any
 * purpose.  It is provided "as is" without express or implied warranty.
 * 
 * BARRY SHEIN DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO
 * EVENT SHALL BARRY SHEIN BE LIABLE FOR ANY SPECIAL, INDIRECT OR
 * CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF
 * USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR
 * OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
 * PERFORMANCE OF THIS SOFTWARE.
 *
 * Author:  Barry Shein, Encore Computer Corporation, BZS@ENCORE.COM
 */



#ifdef SYSV
/*
 * taken from server/os/sysV/util.c
 *
 * Written by Jack Palevich.  
 *            HP Labs
 *            April 1987
 */

bcopy (b1, b2, length)
register unsigned char *b1, *b2;
register length;
{
    if (b1 < b2) {
        b2 += length;
        b1 += length;
        while (length--) {
            *--b2 = *--b1;
        }
    }
    else {
        while (length--) {
            *b2++ = *b1++;
        }
    }
}
#endif /* SYSV */

/*
 * Main driving routine for reading in GF file, hopefully
 * this would make it easy to chop this out and make it a subroutine
 * in someone else's program.
 */
GFread(gfp) FILE *gfp;
  {
  if(getpreamble(gfp))
    fatal(1,"No preamble, not a GF file?");

  if(debug)
    eprintf("Preamble read ok\n");

  if(getpostpost(gfp) < 0)
    fatal(1,"Error reading postpostamble\n");

  if(getpost(gfp) < 0)
    fatal(1,"Error reading postamble\n");

  getchars(gfp);
}
/*
 * The preamble really has very little information
 * but does tell us whether or not this is really
 * likely to be a GF file (the GF_ID_BYTE)
 */
getpreamble(gfp) FILE *gfp;
{
  register int c, k;

  if((c = getc(gfp)) != PRE) goto bad;

  if((c = getc(gfp)) == EOF) goto bad;
  if(c != GF_ID_BYTE) goto bad;

  if((c = getc(gfp)) == EOF) goto bad;
  k = c;

  /* skip commentary */
  while(k--)
    if(getc(gfp) == EOF) goto bad;

  return(0);
 bad:
  return(-1);
}
/*
 *
 * The post-amble of a GF file must be read first. Of course, it's not at
 * a fixed or even predictable location but requires that you read the
 * file itself backwards searching for magic cookies.  Unbelieveable. The
 * logic from the GF description goes essentially like this:
 *
 *
 * Read backwards skipping an arbitrary # of '\337' bytes (>= 4 of them)
 * Next backwards byte is a GF_ID_BYTE
 * Next backwards word (4 bytes) is a byte seek pointer into the
 * file to find the postamble.
 *
 * Ah well, it could be worse...obviously he's kowtowing to systems
 * that don't allow rewriting of files (or perhaps to the fact that
 * the output device might be a tape or other sequential device.)
 *
 * The side effect of this routine is it sets the global 'postseek'
 * to the file position of the postamble (or else die.)
 */
getpostpost(gfp) FILE *gfp;
{
  unsigned char buf[BUFSIZ];
  unsigned register char *cp;
  register long i;
  struct stat st;
  char *errmsg;
  long *lp;
    

  if(fstat(fileno(gfp),&st) < 0)
    goto bad;

  errmsg = "could not stat file?";
  i = (st.st_size >= BUFSIZ ? BUFSIZ : st.st_size);
  if(fseek(gfp,-i,2) < 0)
    goto bad;

  errmsg = "short read?";
  if(fread(buf,sizeof(buf[0]),i,gfp) != i)
    goto bad;

  cp = &buf[i-1];	/* last char */

  errmsg = "unlikely, last block all pad bytes? probably a program bug";
  while(*cp == GF_PAD)
    if(cp == buf)
      goto bad;		/* in theory we should just go to the previous block */
    else
      --cp;

  errmsg = "expected GF-ID-BYTE (131 decimal), not a GF file?";
  if(*cp != GF_ID_BYTE)
    goto bad;

  errmsg =
    "not enough bytes left for postamble pointer? probably a program bug";
  if((cp - buf) < 4)
    goto bad;

  cp -= 4;
  lp = (long *) cp;
  postseek = ntohl(*lp);

  if(debug)
    eprintf("postpostamble: file ptr to postamble = %u\n",postseek);

  errmsg = "seek pointer to postamble impossible, giving up";
  if((postseek == 0) || (postseek >= st.st_size))
    goto bad;
  return(0);
 bad:
  eprintf("%s\n",errmsg);
  return(-1);
}
/*
 * Read in the postamble, basically just a straightforward record
 */
getpost(gfp) FILE *gfp;
{
  char *errmsg;
  register int c;

  errmsg = "seek to postamble failed";
  if(fseek(gfp,postseek,0) < 0)
    goto bad;

  errmsg = "Unexpected EOF while getting postamble?";
  if((c = getc(gfp)) == EOF)
    goto bad;

  errmsg = "Postamble does not start with post opcode, bad file?";
  if(c != POST)
    goto bad;

  errmsg = "Error or EOF while reading postamble final EOC ptr";
  if(get4(&pa.p,gfp))
    goto bad;

  errmsg = "Error or EOF while reading postamble design size";
  if(get4(&pa.ds,gfp))
    goto bad;

  errmsg = "Error or EOF while reading postamble checksum";
  if(get4(&pa.cs,gfp))
    goto bad;

  errmsg = "Error or EOF while reading postamble hppp";
  if(get4(&pa.hppp,gfp))
    goto bad;

  errmsg = "Error or EOF while reading postamble vppp";
  if(get4(&pa.vppp,gfp))
    goto bad;

  errmsg = "Error or EOF while reading postamble min_m";
  if(get4(&pa.min_m,gfp))
    goto bad;

  errmsg = "Error or EOF while reading postamble max_m";
  if(get4(&pa.max_m,gfp))
    goto bad;

  errmsg = "Error or EOF while reading postamble min_n";
  if(get4(&pa.min_n,gfp))
    goto bad;

  errmsg = "Error or EOF while reading postamble max_n";
  if(get4(&pa.max_n,gfp))
    goto bad;

  if(debug) {
    double d;

    eprintf("Postamble:\n");
    eprintf("\tp:\t%u\n",pa.p);

    eprintf("\tds:\t%u\n",pa.ds);
    d = pa.ds;
    d /= TWOTO16;
    d /= 16.0;
    eprintf("\t\t=> designed at %d points\n",(int)d);

    eprintf("\tcs:\t%u\n",pa.cs);
    d = pa.hppp;
    d /= TWOTO16;
    eprintf("\thppp:\t%u pix/pt * 2^16 (%g)\n",pa.hppp,d);
    d *= PTPI;
    eprintf("\t\t=> %d pix/in [magstep]\n",(int)(d+0.5));

    d = pa.vppp;
    d /= TWOTO16;
    eprintf("\tvppp:\t%u pix/pt * 2^16 (%g)\n",pa.vppp,d);
    d *= PTPI;
    eprintf("\t\t=> %d pix/in [magstep]\n",(int)(d+0.5));

    eprintf("\tmin_m:\t%d\tmax_m:\t%d\n",pa.min_m,pa.max_m);
    eprintf("\tmin_n:\t%d\tmax_n:\t%d\n",pa.min_n,pa.max_n);
  }
  return(0);
 bad:
  eprintf("%s\n",errmsg);
  return(-1);
}
/*
 * Expand the perchar pointer table if necessary. Always
 * done once as it starts out unallocated (charmax == 0.)
 * I doubt the re-expand code has been exercised yet.
 */
void
percharexpand()
{
  struct perchar **tp;

  if(charmax == 0) {
    if((pc = (struct perchar **)
	calloc(INITCHARMAX,sizeof(struct perchar *))) == NULL)
      goto bad;
    charmax = INITCHARMAX;
  }
  else {
    /* DEFINITELY UNTESTED */
    if((tp = (struct perchar **)
	calloc(2*charmax,sizeof(struct perchar *))) == NULL)
      goto bad;

    bcopy(pc,tp,nchars*sizeof(struct perchar *));
    charmax *= 2;
    cfree(pc);
    pc = tp;
  }
  return;

 bad:
  fatal(1,"Out of Memory while expanding per char table");
}
/*
 * Allocate a new perchar structure
 */
struct perchar *
newperchar()
{
  struct perchar *p;

  if((p = (struct perchar *)calloc(1,sizeof(*p))) == NULL)
    fatal(1,"Out of memory allocating per char structure");

  return(p);
}
/*
 * Read in the Character Locator information which follows the
 * postamble.
 */
getchars(gfp) FILE *gfp;
{
  register int c,i;
  register struct perchar **p;
  char *iarray;
  int done = FALSE;

  for(p=pc; !done ;) {
    switch(i = getc(gfp)) {

    case EOF:
      goto bad;

    case CHAR_LOC:
    case CHAR_LOC0:
      /* Ensure we have space */
      if(nchars == charmax) {
	percharexpand();	/* will exit if no more space possible */
	p = &pc[nchars];
      }
      *p = newperchar();	/* get a new structure (exits on failure) */
      if((c = getc(gfp)) == EOF)
	goto bad;
      (*p)->residue = c;
      if(i == CHAR_LOC) {
	if(get4(&(*p)->dx,gfp))
	  goto bad;
	if(get4(&(*p)->dy,gfp))
	  goto bad;
      }
      else {
	if((c = getc(gfp)) == EOF)
	  goto bad;
	(*p)->dx = c * 65536;
	(*p)->dy = 0;
      }
      if(get4(&(*p)->w,gfp))
	goto bad;
      if(get4(&(*p)->p,gfp))
	goto bad;
      p++;
      nchars++;
      break;

    default:
      done = TRUE;
      break;
    }
  }
  /* Should have stopped when we hit the POST_POST code */
  if(debug)
    eprintf("Read %d char locators, stopped by: %d (?POST_POST:249?)\n",
	   nchars,i);
  /*
   * Allocate the bit image array
   * We allocate on big one and break it up.
   */
  width = pa.max_m-pa.min_m+1;
  height = pa.max_n-pa.min_n+1;
  widthb = (width+(BPERB-1))/BPERB;

  if((iarray = (char *)calloc(1,widthb*height*nchars)) == NULL)
    fatal(1,"Out of memory allocating character image array");

  for(i=0,p=pc; i<nchars; i++,p++,iarray += widthb*height) {
    (*p)->image = iarray;
    getcharimage(*p,gfp);
  }
  return;
bad:
  fatal(1,"Error or EOF while reading char locators\n");
}
/*
 * Read in the actual character images.
 * Note that this DOES NOT implement the possibility
 * of back pointers for characters sharing the same
 * char locator info, I've never seen a GF file that
 * actually uses it. There's a message put to the diagnostic
 * output if we find one.
 */
getcharimage(p,gfp) register struct perchar *p; FILE *gfp;
{
  int cmd;
  long k;
  char *errmsg;
  /*
   * It is possible that a pointer is -1
   * indicating it exists in the TFM file but
   * not in the GF file. Just ignore. We could
   * have considered this earlier I suppose and
   * not included it in the array and allocations
   * at all but I wasn't sure if the info in the
   * the char locator might still be useful to someone.
   */
  if(p->p == -1) {
    p->exists = FALSE;
    return;
  }
  p->exists = TRUE;
  errmsg = "Error or EOF while looking for BOC";
  if(fseek(gfp,p->p,0))
    goto bad;

  cmd = EOC;
  /*
   * There can be various XXX and YYY cmds before
   * we actually get to the desired BOC (beginning of char)
   * They are all currently treated as NO-OPs.
   */
  while(!((cmd == BOC) || (cmd == BOC1))) {
    switch(cmd = getc(gfp)) {

    case EOF:
      goto bad;

    case XXX1:
      if((k = getc(gfp)) == EOF)
	goto bad;
      while(k--)
	if(getc(gfp) == EOF)
	  goto bad;
      break;

    case XXX2:
      if(get2(&k,gfp) < 0)
	goto bad;
      while(k--)
	if(getc(gfp) == EOF)
	  goto bad;
      break;

    case XXX3:
      if(get3(&k,gfp) < 0)
	goto bad;
      while(k--)
	if(getc(gfp) == EOF)
	  goto bad;
      break;

    case XXX4:
      if(get4(&k,gfp) < 0)
	goto bad;
      while(k--)
	if(getc(gfp) == EOF)
	  goto bad;

    case YYY:
      if(get4(&k,gfp) < 0)
	goto bad;

    case BOC:
      if(get4(&p->boc.c,gfp) < 0)
	goto bad;
      if(get4(&p->boc.p,gfp) < 0)
	goto bad;
      if(get4(&p->boc.min_m,gfp) < 0)
	goto bad;
      if(get4(&p->boc.max_m,gfp) < 0)
	goto bad;
      if(get4(&p->boc.min_n,gfp) < 0)
	goto bad;
      if(get4(&p->boc.max_n,gfp) < 0)
	goto bad;
      if(debug && (p->boc.p != -1))
	eprintf("back pointer\n");
      break;

    case BOC1:
      p->boc.p = -1;
      if((p->boc.c = getc(gfp)) == EOF)
	goto bad;
      if((p->boc.min_m = getc(gfp)) == EOF)
	goto bad;
      if((p->boc.max_m = getc(gfp)) == EOF)
	goto bad;
      if((p->boc.min_n = getc(gfp)) == EOF)
	goto bad;
      if((p->boc.max_n = getc(gfp)) == EOF)
	goto bad;
      p->boc.min_m = p->boc.max_m - p->boc.min_m;
      p->boc.min_n = p->boc.max_n - p->boc.min_n;
      break;

    default:
      eprintf("Warning: unexpected code found while looking for BOC(1), %d\n",
	      cmd);
      break;
    }
  }
  paintimage(p,gfp);
  return;
 bad:
  if(debug)
    eprintf("cmd = %d\n",cmd);
  fatal(1,errmsg);
}
/*
 * Set or get a particular pixel in the bit image array.
 */
setpix(p,r,c,b) char *p; int r,c,b;
{
  p += r*widthb+c/BPERB;
  if(b)
    *p |= HIBIT >> (c % BPERB);
  else
    *p &= ~(HIBIT >> (c % BPERB));
}
getpix(p,r,c) char *p; int r,c;
{
  p += r*widthb+c/BPERB;
  return((*p & (HIBIT >> (c % BPERB))) != 0);
}
/*
 * This actually interprets the GF image painting commands
 */
paintimage(p,gfp) register struct perchar *p; FILE *gfp;
{
  int cmd;
  long d;
  int paint_switch;
  register char *imagep;
  register long m,n;
  long maxn,minm;

  imagep = p->image;
  maxn = p->boc.max_n;
  minm = p->boc.min_m;

#define SETPIX(M,N) setpix(imagep,maxn-N,M-minm,paint_switch)

  m = p->boc.min_m;
  n = p->boc.max_n;
  paint_switch = WHITE;
  cmd = BOC;
  /*
   * This is the most disgusting thing I have ever written, I think.
   * Not sure if there's really any much better way to do it tho without
   * encoding lots of assumptions about things. Feel free to try...
   */
  while(cmd != EOC) {
    if((cmd = getc(gfp)) == EOF)
      goto bad;

    if(cmd == PAINT_0)
      paint_switch ^= 1;
    else if((cmd >= PAINT_1) && (cmd <= PAINT_63)) {
      while(cmd--) {
	SETPIX(m,n);
	m++;
      }
      paint_switch ^= 1;
    }
    else if(cmd == PAINT1) {
      if((d = getc(gfp)) == EOF)
	goto bad;
      while(d--) {
	SETPIX(m,n);
	m++;
      }
      paint_switch ^= 1;
    }
    else if(cmd == PAINT2) {
      if(get2(&d,gfp) < 0)
	goto bad;
      while(d--) {
	SETPIX(m,n);
	m++;
      }
      paint_switch ^= 1;
    }
    else if(cmd == PAINT3) {
      if(get3(&d,gfp) < 0)
	goto bad;
      while(d--) {
	SETPIX(m,n);
	m++;
      }
      paint_switch ^= 1;
    }
    else if(cmd == SKIP0) {
      n--;
      m = p->boc.min_m;
      paint_switch = WHITE;
    }
    else if(cmd == SKIP1) {
      if((d = getc(gfp)) == EOF)
	goto bad;
      n -= (d+1);
      m = p->boc.min_m;
      paint_switch = WHITE;
    }
    else if(cmd == SKIP2) {
      if(get2(&d,gfp) < 0)
	goto bad;
      n -= (d+1);
      m = p->boc.min_m;
      paint_switch = WHITE;
    }
    else if(cmd == SKIP3) {
      if(get3(&d,gfp) < 0)
	goto bad;
      n -= (d+1);
      m = p->boc.min_m;
      paint_switch = WHITE;
    }
    else if(cmd == NEW_ROW_0) {
      n--;
      m = p->boc.min_m;
      paint_switch = BLACK;
    }
    else if((cmd >= NEW_ROW_1) && (cmd <= NEW_ROW_164)) {
      n--;
      m = p->boc.min_m+(cmd-NEW_ROW_1+1);
      paint_switch = BLACK;
    }
    else if(cmd != EOC) {
      eprintf("Bad opcode == %d\n",cmd);
      fatal(1,"While painting image");
    }
  }
  return;
 bad:
  fatal(1,"EOF or Error while reading paint commands\n");
}
