/* main.c */
/*
 * 	(c) Copyright 1987 Gould Inc.
 * 	    All Rights Reserved.
 */

/*
 *	Gould, Inc. Base Register Assembler
 */

#ifndef lint
/*NOBASE*/
static char *rcsid = "@(#) (Gould) $Header: main.c,v 5.5 89/05/12 12:35:56 pcc Rel-3_0 $";
#endif

#include <stdio.h>
#include <signal.h>
#include "struct.h"

Tuchar *infile;
Tuchar tmpname[] = "/tmp/asinXXXXXX";
extern Tuchar tmpout[];		/* full declaration is in aout.c */
Tuchar *outfile = (Tuchar *) "a.out";

Tseginf *cur_seg;

int nerrors = 0;
int lineno = 0;
int pass;
int far_flag;
extern int stab_num;
extern int deflab();
extern int checklab();
int dbflag = 0;
int lineflag = 0;
int L_labflag = 0;
int R_readflag = 0;






/*
NAME: main
PURPOSE: main procedure for the Gould, Inc. UTX base register assembler
PRECONDITIONS:
if no input source file is named, standard input is assumed. a temporary
file is created in this case for all input from stdin up to an end of
file. at EOF, the file is processed.
POSTCONDITIONS:
successful assembly causes 0 (zero) to be returned, errors will result
in a non-zero return.
ALGORITHM:
trap signals, check command line arguments, handle the input from stdin
case, open the input source file, initialize for pass 1 of the source
file, call pass_handler for pass 1, do between pass cleanup and pass 2
initialization, call pass_handler for pass 2, fix current segment size
and call post_pass2 to generate the object file, use dexit to check for
errors and return an appropriate return code.
signals are trapped to the routine siexit to avoid the unpleasantries
of not being able to pass an argument (via the signal) to dexit.
NOTE: if debugging (-d) and errors exist, pass 2 may dump as some pass 1
routines do not clean up after themselves on errors.
HISTORY:
part of initial code DJK RJM
*/






main(argc,argv,env)
register char **argv;
{
    register int c;
    register char **ap;
    int siexit();

    /* trap signals */
    if (signal (SIGHUP, SIG_IGN) != SIG_IGN)
	signal (SIGHUP, siexit);
    if (signal (SIGINT, SIG_IGN) != SIG_IGN)
	signal (SIGINT, siexit);
    if (signal (SIGTERM, SIG_IGN) != SIG_IGN)
	signal (SIGTERM, siexit);

    /* handle command line arguments */
    for( ap = argv+1;*ap;ap++ ){
	char *cp = *ap;
	if( *cp != '-' ){
	    if( infile==0 ){
		infile = (Tuchar *) cp;
	    }
	} else {
	    switch( cp[1] ){
		
	    case 'o':
		outfile = (Tuchar *) *++ap;
		if( outfile==0 ){
		    ap--;
		}
		break;
	    
	    case 'Q':		/* special debug flag */
		++lineflag;
		break;
	    
	    case 'd':
		++dbflag;
		break;
	    
	    case 'L':
		++L_labflag;
		break;

	    case 'R':
		++R_readflag;
		break;

	    default:
		error ("bad command line option specified %s", cp);
		break;
	    }
	}
    }

    /* temporary work file from stdin if no input file on command line */
    if (infile == 0)
    {
	FILE *hold;
	mktemp(tmpname);
	hold = fopen(tmpname,"w");
	while( (c=getchar()) >=0 )
	{
	    putc(c,hold);
	}
	fclose(hold);
	infile = tmpname;
    }

    if (freopen(infile,"r",stdin)==NULL)
    {
	perror(infile);
	error ("Fatal assembler error (input file %s)", infile);
	exit(2);
    }

    /* initialization for pass 1 */
    initsymtab ();
    yyinit();
    far_flag = 0;
    stab_num = 0;
    cur_seg = &seginf[(int) S_text];

    pass = 1;
#ifdef POOL
#ifdef LDPOOL
    pre_pass1();
#endif /* LDPOOL */
#endif /* POOL */
    e_init();
    pass_handler(deflab);


    /* if errors in pass1 and not debugging, skip pass 2 */
    /* note: if debugging and errors exist, pass 2 may dump */
    /* some routines do not clean up after themselves on errors */
    if (nerrors && !dbflag)
    {
	dexit (1);
    }

    /* assure current segment closed up properly */
    cur_seg -> si_dot = dot -> svalue;
#ifndef OLDCALL
#ifndef GOULD_NP1
    e_fixup();
#endif /* NOT GOULD_NP */
#endif /* NOT OLDCALL */

#ifdef DEBUG
    if (dbflag)
    {
	dump_segs ();
	dump_syms ();
    }
#endif

    /* set up header here using seginf array */
    /* then zero size of all segs for pass2 usage */
    pass2_init();
    far_flag = 0;
    stab_num = 0;
    cur_seg = &seginf[(int) S_text];

    /* set up dot for pass 2 */
    dot -> svalue = 0;
    dot -> sspace = S_text;

    pass = 2;
    pass_handler(checklab);

/* assure segments updated, may not be needed here */
    cur_seg -> si_dot = dot -> svalue;

    /* note - all file handling (including linking) is done in post_pass2 */
    post_pass2 ();	/* put the a.out together */

#ifdef DEBUG
    if (dbflag)
    {
	dump_segs ();
	dump_syms ();
    }
#endif

    dexit(nerrors!=0);
}






/*
NAME: dexit and siexit
PURPOSE: exit gracefully (?)
PRECONDITIONS:
parameter v is the return code for dexit.
siexit is called ONLY through the signal handler. it exists only to
allow a proper parameter to be passed to dexit.
POSTCONDITIONS:
if a temporary file was created for source from stdin, it is unlinked
before exiting. creation is done in main.
if the code generation temporary file has been created , it is unlinked
before exiting on a signal. creation is done in pass2_init of aout.c.
HISTORY:
part of initial code DJK RJM
*/






dexit(v){
	if (infile==tmpname) unlink(infile);
	if (dbflag)
	{
	    printf ("dexit: assembler returns %d\n", v);
	}
	exit(v);
}

siexit ()
{
    if (pass == 2) unlink (tmpout);
    dexit (1);
}






/*
NAME: error
PURPOSE: output error messages
PRECONDITIONS:
parameter f is a character string identical in syntax to a printf format
string. parameter arg is any additional information to be printed, again
as in a printf call.
POSTCONDITIONS:
an error message is set up and printed to standard error.
the variable nerrors (number of errors) is incremented.
HISTORY:
part of initial code DJK
*/






/*VARARGS1*/
error(f, arg)
char *f;
{
    extern Tuchar *curfile;

    fflush (stdout);
    fprintf(stderr,"\"%s\", line %d: "
	,curfile ? curfile : (Tuchar *) "",lineno);
    frprintf (stderr, f, &arg);
    fprintf(stderr,"\n");
    fflush (stderr);
    nerrors++;
}






/*
NAME: warning
PURPOSE: output warning messages
PRECONDITIONS:
parameter f is a character string identical in syntax to a printf format
string. parameter arg is any additional information to be printed, again
as in a printf call.
POSTCONDITIONS:
a warning message is set up and printed to standard error.
HISTORY:
essentially a copy of routine error RJM
*/






/*VARARGS1*/
warning(f, arg)
char *f;
{
    extern Tuchar *curfile;

    fflush (stdout);
    fprintf(stderr,"\"%s\", line %d: warning: "
	,curfile ? curfile : (Tuchar *) "",lineno);
    frprintf (stderr, f, &arg);
    fprintf(stderr,"\n");
    fflush (stderr);
}






/*
NAME: frprintf
PURPOSE: special print routine
PRECONDITIONS:
Equivalent to fprintf, but allows caller to expand parameters that were
passed to it.
parameter iop is a pointer to the stdio file descriptor, fmt is a pointer
to a character string which is the printf-like format string, and arg is
a pointer to the stack frame with args.
arg is the address of the first parameter TO CALLER after the format.
POSTCONDITIONS:
returns:
int	EOF	If hit end of file
	0	If no EOF
ALGORITHM:
use _doprnt
HISTORY:
16 Nov 82	David A Willcox of Compion	Adapted from frpintf
*/






/*VARARGS2*/
frprintf(iop, fmt, args)
FILE *iop;
char *fmt;
int  *args;
{
	_doprnt(fmt, args, iop);
	return(ferror(iop)? EOF: 0);
}






/*
NAME: dump_segs
PURPOSE: debugging routine
ALGORITHM:
dumps the seginf structure
HISTORY:
part of initial code RJM
*/






dump_segs ()
{
    register int i;

    printf ("main segment dump:\n");
    for (i = (int) S_text; i < (int) last_SEGS; i++)
    {
	printf ("segment %d ntype %d size %d dot %d nrel %d\n",
	    i, seginf[i].si_ntype, seginf[i].si_size, seginf[i].si_dot,
	    seginf[i].si_nrel);
    }
}

/*
 * 	(c) Copyright 1987 Gould Inc.
 * 	    All Rights Reserved.
 */
