/* activity.c */
/*
 * HCR Confidential
 *
 * These computer programs are the confidential, proprietary property
 * of HCR (Human Computing Resources Corporation, 10 St. Mary Street,
 * Toronto, Ontario, Canada), and may not be disclosed except with the
 * prior written agreement of HCR.
 *
 * Copyright (c) 1984, 1985, 1986 Human Computing Resources Corporation
 * All Rights Reserved
 */

/*
 *	Activity Counts: Compute dependency information
 */

#ifndef lint
static char *rcsid = "@(#) (Gould) $Header: activity.c,v 5.5 89/05/12 12:48:57 pcc Rel-3_0 $";
/* static char ID[] = "@(#)activity.c	15.3	of 86/12/01"; */
#endif

/*
 *	Import
 */

# include <assert.h>
# include <activity.h>
# include <bool.h>
# include <dag.h>
# include <erroro.h>
# include <ops.h>
# include <storage.h>

/*
 *	Export
 */

DAG_Node ActRoot;

/*
 *	Private
 */


/*
 *	Stack Submodule
 *
 *	This provides a stack of lists of dag nodes.  These are 
 *	used to collect the list of nodes in a conditional zone
 *	that have side effects.  When the zone is finally evaluated,
 *	this list is traversed to find all the nodes that have side
 *	effects, in the correct order.
 *
 *	A zone is pushed when the first node in it is discovered,
 *	and popped when its controlling conditional operator is
 *	evaluated.  During the time a zone is on TOS, nodes can 
 *	be added to the list.  Each list starts with an entry that
 *	points to no node at all.  This marker allows the two lists
 *	that are on either side of a COLON to be distinguished.
 */

static DagNodeList DListAllocate();	/* forward for ExtendTos */

typedef struct CSE {
	struct CSE *prev;		/* previous tos */
	DagNodeList nodes;		/* in order */
	DagNodeList last;		/* helps with ordering */
	DagCount cond_zone;		/* which zone is this one ? */
} CSElement, *CondStack;

static CondStack FreeStack = NULL;	/* Free List for stack */
static CondStack Stack = NULL;		/* The actual stack */

#define TosZone	(Stack == NULL ? NotConditional : Stack->cond_zone)

static CondStack
StackAllocate()
{
	CondStack tos;

	if (FreeStack == NULL) {
		tos = GetStorage(s_NoStats, CSElement);
		CheckStorage(tos, "storage for conditional stack", 0);
	} else {
		tos = FreeStack;
		FreeStack = FreeStack->prev;
	}
	tos->prev = NULL;
	tos->nodes = NULL;
	tos->last = NULL;
	return tos;
}

static void
StackFree(tos)
	CondStack tos;
{
	tos->prev = FreeStack;
	FreeStack = tos;
}

static void
ExtendTos(n, zone)
	DAG_Node n;
	DagCount zone;
{
	DagNodeList new_e;

	/*
	 *	n is a node in "zone" that has side effects
	 */

	assert(Stack != NULL);
	assert(Stack->cond_zone == zone);

	new_e = DListAllocate();
	new_e->n = n;

	if (Stack->nodes == NULL) {
		Stack->nodes = new_e;
	} else
		Stack->last->next = new_e;
	Stack->last = new_e;
}

static void
Push(zone)
	DagCount zone;
{
	CondStack new_tos;

	new_tos = StackAllocate();

	new_tos->prev = Stack;
	Stack = new_tos;
	new_tos->cond_zone = zone;
	ExtendTos((DAG_Node) NULL, zone);	/* marker */
}

static DagNodeList
Pop(zone)
	DagCount zone;
{
	DagNodeList l;
	CondStack tos;

	assert(Stack != NULL);		/* Stack is empty */

	tos = Stack;
	Stack = Stack->prev;

	assert(tos->cond_zone == zone);
	l = tos->nodes;
	StackFree(tos);
	return l;
}


/*
 *	End Of SubModule
 */

/*
 *	The following procedures manage lists of Dag Nodes that
 *	have side effects in a conditional. These are the nodes that
 *	DecrementActivity will handle specially when their controlling
 *	conditional node's activity count becomes zero.  The nodes
 *	on that list must be processed in the order they appear in
 *	the Dag, so they are added to the tos in order.
 */

static DagNodeList FreeDList = NULL;	/* Free List for Dag Nodes */

static DagNodeList
DListAllocate()
{
	DagNodeList t;

	if (FreeDList == NULL) {
		t = GetStorage(s_NoStats, DagListElement);
		CheckStorage(t, "storage for cond'l list", 0);
	} else {
		t = FreeDList;
		FreeDList = FreeDList->next;
	}
	t->next = NULL;
	t->n = NULL;
	return t;
}

static void
DListFree(l)
	DagNodeList l;
{
	DagNodeList e;

	while (l != NULL) {
		e = l->next;
		l->next = FreeDList;
		FreeDList = l;
		l = e;
	}
}

static DAG_Node last_node;	/* node examined on last call to Update..*/

void
InitActivity(d)			/* Initialize d for a.c. algorithm */
	DAG_Node d;
{
	DAG_Node n;

	for (n = d; n != NULL; n = n->next) {
		n->activity = n->in_degree + n->delay_count + 1;
		n->evaluated = False;
		n->chain.first_cond = NULL;
	}
	last_node = NULL;
}

void
DoneActivity(d, premature)			/* Done with this Dag */
	DAG_Node d;
	Boolean premature;			/* If premature termination */
{
	DAG_Node n;

	if (!premature) {
		assert(Stack == NULL);	/* orphan conditional zone */
	} else {

		/* Clean up the stack */

		while (Stack != NULL)
			DListFree(Pop(Stack->cond_zone));
	}
	for (n = d; n != NULL; n = n->next) {
		assert(premature || n->evaluated);
		/* assert(n->activity == 0) */ /* must check idioms */
		if (n->chain.first_cond != NULL)
			DListFree(n->chain.first_cond);
	}
}

static void
DecrChildren(n, FinishNode)
	DAG_Node n;
	void (*FinishNode)();
{
	switch(optype(n->op)) {
	case UTYPE:
		DecrementActivity(n->u.in.left, FinishNode);
		break;
	case BITYPE:
		if( asgop(n->op)) {
			DecrementActivity(n->u.in.right, FinishNode);
			DecrementActivity(n->u.in.left, FinishNode);
		} else {
			DecrementActivity(n->u.in.left, FinishNode);
			DecrementActivity(n->u.in.right, FinishNode);
		}
		break;
	}
}

void
UpdateActivity(n, FinishNode)
	DAG_Node n;
	void (*FinishNode)();
{

	/* Update activity counts.  This is trickier than it looks.
	 * DecrementActivity will not continue past a node with a carrier
	 * since the children of nodes with carriers will never
	 * be referenced after the initial evaluation.  Therefore, if this
	 * node has a carrier, we must disconnect its children
	 * here and now.  If this node is no longer active (immediately upon
	 * evaluation) we need to reclaim its resources by calling
	 * DecrementActivity.
	 * Note:  That call won't do the children if there is a carrier
	 * so we must do it ourselves.
	 * Note:  Although we say "nodes with carriers" in the above text,
	 * the test is slightly more complicated than that because this
	 * algorithm is used in three separate instances:
	 *
	 *	1) when delayed stores are being removed.  Since carriers
	 *	   have not yet been chosen, attached identifiers must
	 *	   be checked instead of carriers.  However, attached
	 *	   identifiers can be removed in that process when things
	 *	   are being delayed.  So we also check delay_count.
	 *	2) During choosing of carriers.  The algorithm is running
	 *	   on a DAG for which carriers have not yet been chosen.
	 *	   However, at this point we know that carriers will
	 *	   be chosen if there is an attached id, or if delay_count
	 *	   is nonzero, so we do the same tests as in 1.
	 *	3) After carriers are chosen.  Now things work as described
	 *	   above.
	 *
	 */

	assert(last_node == NULL || last_node->order < n->order);

	if( n->activity == 1 ) {
		DecrementActivity(n, FinishNode);
	} else {
		assert(n->activity != 0);
		--n->activity;		/* We have done this node */
	}

	/* assert(n->activity >= 0); shouldn't be needed now */

	/*
	 *	If the node would have side effects, disconnect
	 *	its children.
	 */

	if (WillHaveCarrier(n)) {
		DecrChildren(n, FinishNode);
	}
}

void
DecrementActivity(n, FinishNode)
			/* Reduce activity count of node and descendents */
	DAG_Node n;
	void (*FinishNode)();
{

	assert(n->activity != 0);
	--(n->activity);

	if (n->activity == 0) {	/*  Finished this node */
	
		if (FinishNode != NULL)
			(*FinishNode)(n);

		if( ! WillHaveCarrier(n) ) 
			DecrChildren(n, FinishNode);
	}
}

/*
 * The inverse of DecrementActivity
 */

void
IncrementActivity(d)
	DAG_Node d;
{
	int ty;

	d->activity++;
	ty = optype(d->op);

	/*
	 *	If this is a conditional node, then we must increment
	 *	the activity of the side effect nodes.
	 */

	if (d->chain.first_cond != NULL) {

		DagNodeList ln;

		assert(d->op == ANDAND || d->op == OROR || d->op == COLON);
		ln = d->chain.first_cond;
		while (ln != NULL) {
			if( WillHaveCarrier(ln->n) )
			{
				if (optype(ln->n->op) != LTYPE)
					IncrementActivity(ln->n->u.in.left);
				if (optype(ln->n->op) == BITYPE)
					IncrementActivity(ln->n->u.in.right);
			}
			IncrementActivity(ln->n);
			ln = ln->next;
		}
	}

	if( ty != LTYPE && !WillHaveCarrier(d))
	{
		IncrementActivity(d->u.in.left);
		if( ty == BITYPE )
			IncrementActivity(d->u.in.right);
	}
}

static Boolean
Evaluate(n, EvalNode)
	DAG_Node n;
	Boolean (*EvalNode)();
{
	Boolean result;

	if (n->evaluated)
/**/		return False;		/* already evaluated */

	/*
	 *	First, evaluate LHS.  This ensures that if the operator
	 *	is conditional, the LHS will happen before the side
	 *	effects on the RHS.  Then, if side effects are waiting,
	 *	they are evaluated (before the RHS).
	 *
	 *	As a special case, must evaluate the RHS of an assignment
	 *	before the LHS.
	 */

	if (optype(n->op) != LTYPE) {
	    if(asgop(n->op)) {
		if (Evaluate(n->u.in.right, EvalNode))
/**/			return True;	/* Terminate this pass */
	    } else {
		if (Evaluate(n->u.in.left, EvalNode))
/**/			return True;	/* Terminate this pass */
	    }
	}
		
	if (n->chain.first_cond != NULL) {

		DagNodeList ln;		/* list of nodes to do */

		assert(n->op == ANDAND || n->op == OROR || n->op == COLON);
		ln = n->chain.first_cond;
		while (ln != NULL) {
			if (ln->n != NULL) {	/* if not a marker */
				if (Evaluate(ln->n, EvalNode))
/**/					return True;	/* terminate early */
			}
			ln = ln->next;
		}
	}

	/*
 	 *	Now, evaluate RHS, if any. (LHS for assignments)
	 */

	if (optype(n->op) == BITYPE) {
	    if(asgop(n->op)) {
		if (Evaluate(n->u.in.left, EvalNode))
/**/			return True;		/* terminate early */
	    } else {
		if (Evaluate(n->u.in.right, EvalNode))
/**/			return True;		/* terminate early */
	    }
	}

	/*
	 *	Now, evaluate this node by invoking the routine from
	 *	the module.  It is responsible for calling UpdateActivity.
	 */

	result = (*EvalNode)(n);
	n->evaluated = True;
	return result;
}

/*
 *	Version 2:
 *
 *	Nodes are now evaluated "top-down".  We sweep through the
 *	DAG, looking for nodes to evaluate.  Currently we evaluate
 *	if:
 *		- indegree == 0 (=> a root node)
 *		- WillHaveCarrier (=> won't be embedded)
 *		  This will be sharpened in the future.
 *
 *	Evaluating a node is:
 *		- Evaluate side effect nodes (see below)
 *		- Evaluate children
 *		  if activity > indegree (i.e., not already done)
 *		- Call (*EvalNode)()
 *		  - evaluate node
 *		  - call UpdateActivity
 *		  - do side effects.
 *
 *	When we enter a conditonal zone, we push it onto the stack.
 *	Side effects within the conditional zone are accumulated
 *	on the top of stack, and attached to the conditional operator
 *	node.  When that node is evaluated, all the attached side
 *	effects are evaluated before the children are.
 *
 *	Note that this routine does not call UpdateActivity - that is
 *	still called at the appropriate point in (*EvalNode)().
 *
 *	This routine returns the result of the last call on EvalNode.
 *	If it is true, it indicates the module requested the pass
 *	through the DAG be terminated.  Usually, this is followed
 *	by another pass.
 */

/*
 *	This is the same macro definition as dagtree uses to decide
 *	if a node should be embedded.
 */

#define DoNotEmbed(n)		((n)->attached != NULL || \
					((n)->carrier != NoId && \
					 optype((n)->op) != LTYPE && \
					 !(n)->prev_valid_carrier))

Boolean
ActivityDag(d, EvalNode)
	DAG_Node d;				/* the Dag to process */
	Boolean (*EvalNode)(/*n*/);	/* Routine to evaluate node */
{
	DAG_Node n;
	DagNodeList first_cond, right_cond;
	Boolean stop = False;

	InitActivity(d);
	for (n = d; (n != NULL) && !stop; n = n->next) {

		ActRoot = n;

		/*
		 *	For conditional operators, remove side effects from
		 *	the stack.
		 */
	
		if (n->op == ANDAND || n->op == OROR) {
			first_cond = Pop(n->u.in.right->in_cond);
		} else
		if (n->op == COLON) {
			/* Correct order is left, then right */
			right_cond = Pop(n->u.in.right->in_cond);
			first_cond = Pop(n->u.in.left->in_cond);
			if (right_cond != NULL) {
				if (first_cond == NULL)
					first_cond = right_cond;
				else {			/* concatenate the lists */
					DagNodeList lc;
	
					lc = first_cond;
					while (lc->next != NULL) lc = lc->next;
					lc->next = right_cond;
				}
			}
		} else
			first_cond = NULL;
	
		n->chain.first_cond = first_cond;

		/*
		 *	Do we evaluate this node?
		 *	If it is in a conditional, and it has a side effect,
		 *	attach it to top of stack.  Otherwise, evaluate
		 *	it if appropriate.
		 */

		if (n->in_cond != NotConditional) {
			if (n->in_cond > TosZone)	/* New conditional */
				Push(n->in_cond);

			if (n->in_degree == 0 || DoNotEmbed(n)) {
				/*
				 * Currently, we don't allow delays in
				 * conditionals.
				 */
				assert(n->delay_count == 0);
				ExtendTos(n, n->in_cond);
			}
			
		} else {
			if (n->in_degree == 0 ||
			    n->delay_count > 0 ||  /* For now, don't embed */
			    DoNotEmbed(n)
			   )
			{
				stop = Evaluate(n, EvalNode);
			}
		}
	}
	DoneActivity(d, stop);
	return stop;
}
