/* flow.c */
/*
 * HCR Confidential
 *
 * These computer programs are the confidential, proprietary property
 * of HCR (Human Computing Resources Corporation, 10 St. Mary Street,
 * Toronto, Ontario, Canada), and may not be disclosed except with the
 * prior written agreement of HCR.
 *
 * Copyright (c) 1984, 1985, 1986 Human Computing Resources Corporation
 * All Rights Reserved
 */
/*
 *	This module handles the flow graph representation of the function.
 */

#ifndef lint
static char *rcsid = "@(#) (Gould) $Header: flow.c,v 5.5 89/05/12 12:50:37 pcc Rel-3_0 $";
/* static char ID[] = "@(#)flow.c	14.3	of 86/08/13"; */
#endif

/*
 *	Imported Objects
 */

# include <blocks.h>
# include <dag.h>
# include <dagsymbol.h>
# include <flow.h>
# include <longset.h>
# include <assert.h>
# include <bool.h>
# include <storage.h>
# include <labels.h>
# include <erroro.h>

int gdebug = 0;		/* Flow Graph */

/*
 *	Exported Objects
 */

FlowNode *FlowGraph;		/* a dynamically allocated array */

FlowIndex *DFN;			/* Depth-first Numbering */

int	NumFlowNodes;		/* Number of nodes in the flow graph */
int	NumReachableNodes;	/* Number of reachable nodes */

/*	Globals
 */

#define SizeInc		10	/* minimum size increment (in nodes) */

static int DFN_number;		/* Current Depth First Number */
static int CurSize = 0;		/* Current size (in nodes) of arrays */
static int CurNodes = 0;	/* Current number of valid nodes in use 
				   (for storage management) */

static PredNode PredFreeList = NULL;	/* Free list of PredNodes */

/*	Forward Declarations
 */

static void	DeallocateFlow();
static void	EliminateCBranch();
static void	ReBuildGraph();

/*
 *	Storage Management
 */

static void
FreePred(p)
	PredNode p;
{
	p->next = PredFreeList;
	PredFreeList = p;
}

void
FreeFlow()			/* This recovers the program */
{
	DeallocateFlow(True);
}

static void
DeallocateFlow(RecoverProgram)
	Boolean RecoverProgram;
{
	int n;
	PredNode p, next;

	for (n = 0; n < CurNodes; ++n) {
		DestroySet(Dom(n));
		Dom(n) = NULL;
		p = FlowGraph[n].preds;
		while (p != NULL) {
			next = p->next;
			FreePred(p);
			p = next;
		}
		if(IsMultExit(n)) {
			free((char *) FlowGraph[n].u.exit_list);
			DecreaseSpace(s_FlowExits,
				     FlowGraph[n].nexits * sizeof(FlowIndex));
		}

		if( RecoverProgram )
			FreeBlock(FlowGraph[n].block);
	}
	CurNodes = 0;
}

static void
FreeGraph()
{
	free(FlowGraph); FlowGraph = NULL;
	DecreaseSpace(s_FlowGraph, CurSize * sizeof(FlowNode));
	free((char *) DFN); DFN = NULL;
	DecreaseSpace(s_DFN, CurSize * sizeof(FlowIndex));
}

static void
AllocFlow(nodes)
	int nodes;
{
	int n;

	assert(CurNodes == 0);

	if (CurSize < nodes) {	/* Need more room, re-allocate */
		if (CurSize > 0)
			FreeGraph();
		FlowGraph = GetArray(s_FlowGraph, nodes, FlowNode);
		CheckStorage(FlowGraph, "storage for %d flow graph nodes", nodes);
		DFN = GetArray(s_DFN, nodes, FlowIndex);
		CheckStorage(DFN, "storage for %d DFN entries", nodes);
		CurSize = nodes;
	}
	CurNodes = nodes;
	for (n=0; n < nodes; ++n) {
		Dom(n) = CreateSet(NumFlowNodes);
		FlowGraph[n].preds = NULL;
	}
}

static void
ExpandFlow()			/* Make sure storage available for another
				   flow graph node
				 */
{
	unsigned int new_size;

	if (CurNodes == CurSize) {	/* all nodes are in use */
		new_size = (CurSize + SizeInc) * sizeof(FlowNode);

		/*	Make the flow graph bigger */

		FlowGraph = (FlowNode *) realloc((char *) FlowGraph,new_size);
		CheckStorage(FlowGraph, "storage for FlowGraph, size: %d", new_size);
		CurSize += SizeInc;
		IncreaseSpace(s_FlowGraph, SizeInc * sizeof(FlowNode));

		/*	Make the DFN array bigger */

		new_size = CurSize * sizeof(FlowIndex);
		DFN = (FlowIndex *) realloc((char *) DFN, new_size);
		CheckStorage(DFN, "storage for DFN, size: %d", new_size);
		IncreaseSpace(s_DFN, SizeInc * sizeof(FlowIndex));

	}
}

static PredNode
PredAllocate()
{
	PredNode p, c;
	int i;

	if (PredFreeList == NULL) {
		c = GetArray(s_Pred, PredChunk, PrdNd);
		CheckStorage(c, "storage for predecessor links", 0);

		/* Turn array into linked list (backwards!) */

		p = NULL;
		for (i=0; i < PredChunk; ++i) {
			c->next = p;
			p = c++;
		}

		/*
		 * Now p points to the head of the free list (the
		 * last entry in the chunk.
		 */
		
	} else
		p = PredFreeList;

	PredFreeList = p->next;
	return p;
}

/*
 *	Determine which nodes of the flow graph are reachable from "f"
 *	and (recursively) mark them reachable if they have not already
 *	been reached
 */

static void
MarkReachable(f)
	FlowIndex f;
{
	int n;
	FlowIndex fn;

	FlowGraph[f].block->reachable = True;
	NumReachableNodes++;

	for( n = 0; n < NumExits(f); n++ )
	{
		fn = Successor(f,n);
		if( !FlowGraph[fn].block->reachable )
			MarkReachable(fn);
	}
}

/*
 *	Compute the depth-first ordering
 */

static void
DFN_Search(n)
	register FlowIndex n;
{
	register int s;

	FlowGraph[n].visited = True;

	for (s=NumExits(n) - 1; s >= 0; --s) {
		if (Successor(n, s) != NullFlow) {
			if (!FlowGraph[Successor(n, s)].visited)
				DFN_Search(Successor(n, s));
		}
	}
	FlowGraph[n].DFN = DFN_number;
	DFN[DFN_number] = n;
	--DFN_number;
}

static void
DFN_Compute()
{	
	/*	Computes the Depth-First ordering for the Flow Graph.
	 *	DFN[i] is the node index of the node with depth-first
	 *	number i.
	 */

	FlowIndex f;

	for (f=0; f < NumFlowNodes; ++f)
	{
		FlowGraph[f].visited = False;
		FlowGraph[f].DFN = -1;
	}

	DFN_number = NumReachableNodes-1;

	DFN_Search(0);
}

/*
 *	Dominator Computation
 */

static void
DOM_Compute()		/* Fig. 13.5 Aho and Ullman */
{
	/* 	Warning: this code currently assumes that the initial
	 *	node is node zero, and that the DFN[0] == 0
	 */

	register FlowIndex n;
	register int i;
	register LongSet u, new_d;
	Boolean changed;
	PredNode p;

	u = CreateSet(NumFlowNodes);
	new_d = CreateSet(NumFlowNodes);

	for (n=0; n < NumFlowNodes; ++n) {
		if( FlowGraph[n].block->reachable )
			Insert( (int) n, u);
	}

	NullSet(FlowGraph[InitialNode].dom);
	Insert( (int) InitialNode, Dom(InitialNode));  /* Hmmm ! */
	
	for (n=1; n < NumFlowNodes; ++n)	{/* succ(InitialNode) */
		if( FlowGraph[n].block->reachable )
			CopySet(Dom(n), u);
		else
			NullSet(Dom(n));
	}

	changed = True;
	while (changed) {
		changed = False;
		assert(DFN[0] == 0 && InitialNode == 0);
		for (i=1; i < NumReachableNodes; ++i) {	/* n in N - {n0} */
			n = DFN[i];
			CopySet(new_d, u);
			for (p=FlowGraph[n].preds; p != NULL; p=p->next) {
				if( FlowGraph[p->p].block->reachable )
					Intersection(new_d, new_d, Dom(p->p));
			}
			Insert( (int) n, new_d);
			if (Not(SetEq(new_d, Dom(n)))) {
				changed = True;
				CopySet(Dom(n), new_d);
			}
		}
	}
	DestroySet(u);
	DestroySet(new_d);
}

Boolean
BackEdge(a, b)				/* is a --> b a back edge ? */
	FlowIndex a, b;	
{
	return IsElement( (int) b, Dom(a));
}

/*
 *	Flow Graph Construction
 */

void
AddPred(n, p)
	FlowIndex n, p;
{
	PredNode lp;
	
	lp = PredAllocate();
	lp->p = p;
	lp->next = FlowGraph[n].preds;
	FlowGraph[n].preds = lp;
}

void
DelPred(n,pred)
	FlowIndex n;
	FlowIndex pred;
{
	PredNode p, prev;

	p = FlowGraph[n].preds;
	assert(p != NULL);

	if( p->p == pred )
	{
		FlowGraph[n].preds = p->next;
	}
	else
	{
		do {
			prev = p;
			p = p->next;
		} while( p!= NULL && p->p != pred );

		assert(p != NULL);
		prev->next = p->next;
	}
	FreePred(p);
}

void	
BuildGraph()
{
	register BasicBlock b;
	BasicBlock sb;
	register Instruction l;
	register FlowIndex n;
	register int i;
	register FlowIndex sn;

	NumFlowNodes = NumBlocks;

	AllocFlow(NumFlowNodes);


	n = 0;
	for( b = FirstBlock; b != NULL; b = b->next )
	{
		b->FGindex = n;
		b->reachable = False;
		FlowGraph[n].block = b;
		FlowGraph[n].preds = NULL;
		FlowGraph[n].nesting = 0;
		FlowGraph[n].loopweight = 0;
		n++;
	}

	/*
	 *	Set up the initial block
	 */

	if (NumEntryPoints > 1) {		/* f77 only */
		FlowGraph[InitialNode].mult_exit = True;
		FlowGraph[InitialNode].nexits = 1;	/* updated below */
		FlowGraph[InitialNode].u.exit_list =
				GetArray(s_FlowExits, NumEntryPoints, FlowIndex);
		CheckStorage(FlowGraph[InitialNode].u.exit_list, "storage for entry point links (%d)", NumEntryPoints);
		FlowGraph[InitialNode].u.exit_list[0] = FirstBlock->next->FGindex;
	} else {			/* Normal case */
		FlowGraph[InitialNode].mult_exit = False;
		FlowGraph[InitialNode].nexits = 1;
		FlowGraph[InitialNode].u.exits[0] = FirstBlock->next->FGindex;
		FlowGraph[InitialNode].u.exits[1] = NullFlow;

	}

	/*
	 *	This loop starts with the second block
	 */

	for (b=FirstBlock->next; b != NULL; b = b->next) {
		n = b->FGindex;
		l = b->code.last;

		if( l == NULL )
		{
			/* empty block created by folding a CBRANCH when
			 * it is the only instruction in the block
			 */

			FlowGraph[n].mult_exit = False;
			FlowGraph[n].u.exits[0] = n + 1;
			FlowGraph[n].u.exits[1] = NullFlow;
			FlowGraph[n].nexits = 1;
		}
		else
		if( IsSwitch(l) )
		{
			/* The multi-way branch of a switch statement */

			FlowGraph[n].mult_exit = True;
			FlowGraph[n].nexits = l->u.sw.ncases + 1;
			FlowGraph[n].u.exit_list = GetArray(s_FlowExits,
					      FlowGraph[n].nexits, FlowIndex);
			CheckStorage(FlowGraph[n].u.exit_list, "storage for switch flow (%d)", FlowGraph[n].nexits);
				/* Find block numbers for all labels.  The
				 * default case goes at the end of the list.
				 */

			for( i = 0; i < l->u.sw.ncases; i++)
			{
				sb = LookUpLabel( (LabelType) l->u.sw.swtable[i].slab);
				assert(sb != NULL);
				FlowGraph[n].u.exit_list[i] = sb->FGindex;
			}
			sb = LookUpLabel(l->u.sw.defaultlabel);
			assert(sb != NULL);
			FlowGraph[n].u.exit_list[l->u.sw.ncases] = sb->FGindex;
		}
#ifdef FORT
		else
		if( IsArithIf(l) )
		{
			FlowGraph[n].mult_exit = True;
			FlowGraph[n].nexits = ArithIfExits;
			FlowGraph[n].u.exit_list = GetArray(s_FlowExits,
					      FlowGraph[n].nexits, FlowIndex);
			CheckStorage(FlowGraph[n].u.exit_list, "storage for arith. if exits", 0);
			for( i = 0; i < ArithIfExits; ++i )
			{
				sb = LookUpLabel(l->u.aif.labs[i]);
				assert(sb != NULL);
				FlowGraph[n].u.exit_list[i] = sb->FGindex;
			}
		}
#endif
		else
		{
			FlowGraph[n].mult_exit = False;
			FlowGraph[n].u.exits[0] = n + 1;
			FlowGraph[n].u.exits[1] = NullFlow;
			FlowGraph[n].nexits = 1;
#ifdef FORT
			if (IsEntry(l)) { /* Initial Node exits to here */
				FlowGraph[InitialNode].u.
					exit_list[FlowGraph[InitialNode].nexits] = n;
				++(FlowGraph[InitialNode].nexits);
			} else
#endif /* FORT */
			if (InstType(l) == FcnEnd)
			{
				FlowGraph[n].nexits = 0;
				FlowGraph[n].u.exits[0] = NullFlow;
			}
			else
			if (IsReturn(l))
			{
				FlowGraph[n].u.exits[0] = FinalBlock->FGindex;
			}
			else
			if (IsBranch(l)) {
				FlowGraph[n].nexits = IsCBranch(l) ? 2 : 1;
				sb = LookUpLabel(BranchTarget(l));
				assert(sb != NULL);
				FlowGraph[n].u.exits[IsCBranch(l) ? 1 : 0] = sb->FGindex;
			}
		}
	
		/* Set up predecessor Links */

		for (i=0; i < FlowGraph[n].nexits; ++i) {
			sn = Successor(n, i);
			if (sn != NullFlow)
				AddPred(sn, n);
		}
	}


	/*
	 *	Set up predecessor links for the initial node.
	 *	It had to be delayed until now because the exit list
	 *	is built piecemeal for multiple entry points.
	 */

	for (i = 0; i < FlowGraph[InitialNode].nexits; ++i) {
		sn = Successor(InitialNode, i);
		if (sn != NullFlow)
			AddPred(sn, InitialNode);
	}

	/* Now figure out which nodes of the flow graph can be reached
	 * from the initial node.  The present scheme assumes that the
	 * initial node is 0, and that the function can be entered only
	 * via this node.
	 */

	NumReachableNodes = 0;
	MarkReachable(InitialNode);

	DFN_Compute();
	DOM_Compute();
	if (gdebug) DumpGraph();
}

/*
 *	Extend the flow graph.  This allows new nodes to be
 *	inserted into the flow graph.  Note that only the basic
 *	information is constructed - exits information, the DFN,
 *	dominators and nesting must be set up externally, although
 *	initial values are entered.  The dom set is created.
 *
 *	It returns the FlowIndex of the new node.
 */

FlowIndex
ExtendGraph(b)			/* Extend graph with block b */
	BasicBlock b;
{
	FlowIndex new;

	ExpandFlow();		/* ensure adequate storage */
	new = CurNodes++;
	++NumFlowNodes;

	b->FGindex = new;
	FlowGraph[new].block = b;
	FlowGraph[new].nexits = 0;
	FlowGraph[new].u.exits[0] = NullFlow;
	FlowGraph[new].u.exits[1] = NullFlow;
	FlowGraph[new].mult_exit = False;
	FlowGraph[new].preds = NULL;
	FlowGraph[new].nesting = -1;		/* illegal value */
	FlowGraph[new].loopweight = 0;
	FlowGraph[new].DFN = -1;		/* ditto */
	Dom(new) = CreateSet(NumFlowNodes);

	return new;
}

/*
 * Re build the flow graph.  Some day this should take advantage of the
 * fact that we already know a depth first ordering
 */
static void
ReBuildGraph()
{
	BasicBlock b;

	DeallocateFlow(False);
	BuildGraph();
	for( b=FirstBlock; b != NULL; b = b->next )
		InitFGindex(b);
}

/*
 *	Reducibility Determiniation
 */

Boolean
IsReducible()
{
	FlowIndex *stack, *stackp;
	int *indegree;
	FlowIndex n, nprime;
	int i, nactive;
	PredNode lp;

	stack = GetArray(s_NoStats, NumFlowNodes, FlowIndex);
	CheckStorage(stack, "storage for irr. test stack (%d)", NumFlowNodes);
	stackp = stack;
	indegree = GetArray(s_NoStats, NumFlowNodes, int);
	CheckStorage(indegree, "storage for irr. test indegree (%d)", NumFlowNodes);

	/* Flow graph is reducible iff the graph less its back edges is
	 * acyclic.  To determine acyclic-ness, do a topological sort of
	 * the flow graph
	 *
	 * Begin by finding the indegree of each node, and keeping track
	 * of the number of nodes with indegree 0
	 */
	for( n = 0; n < NumFlowNodes; n++ )
 	if(FlowGraph[n].block->reachable)
	{
		indegree[n] = 0;
		for( lp = FlowGraph[n].preds; lp != NULL; lp = lp->next)
		{
			if( !BackEdge(lp->p,n) && 
					FlowGraph[lp->p].block->reachable )
			{
				indegree[n]++;
			}
		}

		if( indegree[n] == 0 )
			*stackp++ = n;
	}

	/* Now walk around removing nodes of indegree 0 from the stack and
	 * decrementing the indegrees of their successors.
	 */
	nactive = NumReachableNodes;
	while( stackp != stack )
	{
		n = *--stackp;
		nactive--;
		for( i = 0; i < NumExits(n); i++ )
		{
			nprime = Successor(n,i);
			if( nprime != NullFlow  &&  !BackEdge(n,nprime))
			{
				assert(indegree[nprime] > 0);
				if( --indegree[nprime] == 0 )
					*stackp++ = nprime;
			}
		}
	}
	free((char *)stack); free((char *)indegree);
	return( nactive == 0 );
}

/*
 * Try to find CBRANCHes that have become unconditional and change them
 * to branches.
 */
void
AdjustFlowGraph()
{
	BasicBlock b;
	DAG_Node d, l;
	Boolean changed = False;

	for( b=FirstBlock; b != NULL; b = b->next )
	{
		if( b->code.last != NULL && IsCBranch(b->code.last) )
		{
			/* There should be a faster way to do this */
			for( d = b->Dag; d != NULL; d = d->next )
			if( d->op == CBRANCH )
			{
				l = d->u.in.left;
				if( l->op == ICON )
				{
					EliminateCBranch(d,b);
					changed = True;
			/**/		break;
				}
			}
		}
	}

	if( changed )
	{
		if( gdebug > 1 )
			printf("Re-building flow graph\n");
		ReBuildGraph();
	}
}

/*
 * DAG node "d" is a CBRANCH with a condition that has become an ICON.
 * Remove it and replace it with an unconditional branch if necessary.
 */
static void
EliminateCBranch(d,b)
	DAG_Node d;		/* The CBRANCH */
	BasicBlock b;		/* Block in which this node found */
{
	DAG_Node left, right;
	Instruction c;
	TreeNode tleft;

	if( gdebug > 1 )
		printf("Fixing CBRANCH %d (%x, block %d)\n",
			d->order, d, b->blocknum);

	left = d->u.in.left;
	right = d->u.in.right;

	/* Delete the last instruction in the block (the CBRANCH).
	 * Then add a branch if necessary
	 */

	DelInstruction(b->code.last,&b->code);
	tleft = IdTree(left->leaf_id);
	if( ! tleft->tn.lval )
	{
		/* CBRANCH, 0 => take branch (sic)
		 * Add an uncondtional branch to the block
		 */

		c = AddInstruction(UBranch, &b->code);
		c->u.lb.lab_num = right->u.tn.lval;
		if( gdebug > 1 )
		{
			printf("Add branch:\n");
			PrintInstruction(c);
		}
	}
	/* Finally, make this node vanish */

	left->in_degree--;
	right->in_degree--;
	d->op = LEAFNOP;
}

/*
 *	Provide flow graph indices for the DAG for a single basic block
 *	By convention, if the block is not reachable, the FGIndex is
 *	set to NullFlow
 */

void
InitFGindex(b)
	BasicBlock b;
{
	DAG_Node d;
	FlowIndex f;

	f = b->reachable ? b->FGindex : NullFlow;

	for( d = b->Dag; d != NULL; d = d->next )
		d->FGindex = f;
}

/*
 *	Debugging
 */

static void
DumpPreds(n)
	FlowIndex n;
{
	PredNode lp;

	for (lp = FlowGraph[n].preds; lp != NULL; lp = lp->next)
		printf(" %d ", lp->p);
}

void
DumpGraph()
{
	FlowIndex n;
	int i;

	for (n = 0; n < NumFlowNodes; ++n) {
		printf("Node: %d block %o DFN %d nesting %d exits(%d):",
			n, FlowGraph[n].block, FlowGraph[n].DFN, 
			FlowGraph[n].nesting, FlowGraph[n].nexits);
		if( IsMultExit(n) )
			for( i = 0; i < FlowGraph[n].nexits; i++ )
				printf(" %d", FlowGraph[n].u.exit_list[i]);
		else
			printf(" %d %d",
				FlowGraph[n].u.exits[0],
				FlowGraph[n].u.exits[1]);
		printf("\n\tPredecessors: ");
		DumpPreds(n);
		putchar('\n');
		printf("\tDominators:");
		DumpSet(FlowGraph[n].dom);
		putchar('\n');
	}

	for (n = 0; n < NumReachableNodes; ++n)
		printf("DFN[%d] = %d\n", n, DFN[n]);
}
