/* livedead.c */
/*
 * HCR Confidential
 *
 * These computer programs are the confidential, proprietary property
 * of HCR (Human Computing Resources Corporation, 10 St. Mary Street,
 * Toronto, Ontario, Canada), and may not be disclosed except with the
 * prior written agreement of HCR.
 *
 * Copyright (c) 1984, 1985, 1986 Human Computing Resources Corporation
 * All Rights Reserved
 */
/*
 *	Live-Dead Information Computation
 */

/*
 *	See: Aho and Ullman, Principles of Compiler Design, Section 12.5
 */

#ifndef lint
static char *rcsid = "@(#) (Gould) $Header: livedead.c,v 5.5 89/05/12 12:51:24 pcc Rel-3_0 $";
/* static char ID[] = "@(#)livedead.c	15.4	of 86/10/24"; */
#endif

/*	Import
 */

# include <activity.h>
# include <assert.h>
# include <bool.h>
# include <livedead.h>
# include <blocks.h>
# include <dag.h>
# include <flow.h>
# include <identifier.h>
# include <dagsymbol.h>
# include <longset.h>
# include <option.h>
# include <erroro.h>

/*	Import
 */

/*	Private
 */

	/* Cases for which we keep track of LD information in symbol table */
#define InterestingCases	case REG: \
				case NAME: \
				case LNAME: \
				case PNAME: \
				case STATNAME

#define FirstCall	True		/* Is this first call to LDInit() for
					 * this function? */

static void LDInit();
static void Done();
static Boolean LDNode();
static void GenDly();
static void SetUse();
static void SetDef();
static void CallUse();
static void IndirUse();
static void IndirDef();

static BasicBlock GBlock;	/* Tunnel from LDWalk to SetDef(), SetUse() and
				 * IndirUse()
				 */

static LongSet LiveOnCall;	/* For keeping track of things live on call
				 * (and also on exit from function) */
static LongSet LDTmpSet;

/*
 * Solve data flow equations for LD information
 */

void
LiveDead()
{
	int i;
	BasicBlock b;

	if( LiveOnCall != NULL )
	{
		DestroySet(LiveOnCall);
		DestroySet(LDTmpSet);
	}
	LiveOnCall = CreateSet((int)MaxIdentifier+1);
	LDTmpSet   = CreateSet((int)MaxIdentifier+1);

	for( i = 0; i < NumReachableNodes; i++ )
	{
		b = FlowGraph[DFN[i]].block;
		if( b->ld.Use == NULL )
		{
			LDNewBlock(b);
		}
		else
		{
			NullSet(b->ld.In);
			NullSet(b->ld.Out);
		}
		LDWalk(b);
	}
	LDInit(FirstCall);
	LDDataFlow();
	LDInit(!FirstCall);
	LDDataFlow();
}

/*
 * Initialization routine called from loop invariant analysis
 */

void
LDNewBlock(b)
	BasicBlock b;
{
	b->ld.Use = CreateSet((int)MaxIdentifier+1);
	b->ld.Def = CreateSet((int)MaxIdentifier+1);
	b->ld.PDef= CreateSet((int)MaxIdentifier+1);
	b->ld.In  = CreateSet((int)MaxIdentifier+1);
	b->ld.Out = CreateSet((int)MaxIdentifier+1);
	NullSet(b->ld.In);
	NullSet(b->ld.Out);
}

/*
 * Solve the data flow equations.  Also called after global common
 * subexpressions.
 */

void
LDDataFlow()
{
	FlowIndex n;
	int i, j;
	BasicBlock b, bb;
	LongSet NewOut;
	Boolean changed;
	Boolean infinite_loop = !FinalBlock->reachable;

	NewOut = CreateSet((int)MaxIdentifier+1);
	do
	{
		changed = False;
		for( i = NumReachableNodes - 1; i >= 0; i-- )
		{
			n = DFN[i];
			b = FlowGraph[n].block;

			if( b == FinalBlock )
		/**/		continue;	/* skip final block */

			NullSet(NewOut);

			for( j = 0; j < NumExits(n); j++ )
			{
				bb = FlowGraph[Successor(n,j)].block;
				Union(NewOut, NewOut, bb->ld.In);
			}
			if( Not(SetEq(NewOut,b->ld.Out)))
			{
				changed = True;
				CopySet(b->ld.Out, NewOut);
			}
			Difference(NewOut, NewOut, b->ld.Def);
			Union( b->ld.In, NewOut, b->ld.Use);
		}
	} while( changed );

	DestroySet(NewOut);

	if( ddebug > 1 )
		PLDInfo();
}

/*
 * Set up initial conditions on Out for last block
 */

static void
LDInit(first_time)
	Boolean first_time;
{
	BasicBlock final, initial;
	Identifier id;
	LongSet suses;

	final = FinalBlock;
	initial = FirstBlock;

	if( !first_time )
		assert(initial->ld.In != NULL);

	/* Now determine what things might be referenced by a call:
	 * - all NAMEs
	 * - all STATNAMEs that are live on entry to this function.
	 *   If we don't know this yet (first time through), say all
	 *   STATNAMEs that appear as leaves.
	 * - anything whose address has been taken.
	 *
	 * Of these three classes of identifier, the first two must also
	 * be marked live on a return.
	 */

	NullSet(LiveOnCall);

	for( id = FirstId; id <= MaxIdentifier; id++ )
	{
		switch(IdOp(id))
		{
		case NAME:
			Insert((int)id, LiveOnCall);
			break;

		case STATNAME:
			if( first_time )
			{
				suses = GetUses(id);
				if( suses == NULL || Cardinality(suses) != 0 )
					Insert((int)id, LiveOnCall);
			}
			else
			{
				if( IsElement((int)id, initial->ld.In) )
					Insert((int)id, LiveOnCall);
			}
			break;
		}
	}

	if( final->reachable )
	{
		/* Add information to date to final block */

		CopySet(final->ld.Out, LiveOnCall);
		Difference(final->ld.In, final->ld.Out, final->ld.Def);
		Union( final->ld.In, final->ld.In, final->ld.Use);
	}

	/* Finally, mark as live any identifier whose address has already
	 * been taken.  These are not necessarily live on exit.
	 */

	for( id = FirstId; id <= MaxIdentifier; id++ )
		if( WasAddressed(id) )
			Insert((int)id, LiveOnCall);
}

/*
 * Walk over DAG to collect LD information
 */

void
LDWalk(b)
	register BasicBlock b;
{
	GBlock = b;

	NullSet(b->ld.Use);
	NullSet(b->ld.Def);
	NullSet(b->ld.PDef);

	(void) ActivityDag(b->Dag, LDNode);

}

static Boolean
LDNode(d)
	DAG_Node d;
{
	AttachedID a;

	/* First, figure out action implied by the DAG node itself */

	switch(d->op)
	{
	InterestingCases:
		SetUse(d->leaf_id);
		break;

	case CALL:
	case UNARY CALL:
	case FORTCALL:
	case UNARY FORTCALL:
	case STCALL:
	case UNARY STCALL:
		CallUse();
		break;

	case UNARY MUL:
		if( d->is_fetch )
		{
			if( d->indirect == NoId )
				IndirUse();
			else
				SetUse(d->indirect);
		}
		else
		{
			if( d->indirect == NoId )
				IndirDef(d->in_cond);
			else
				SetDef(d->indirect, d->in_cond);
		}
		break;
	}

	/* For each attached identifier, call SetDef to indicate that
	 * identifier defined here.
	 */

	for( a = d->attached ; a != NULL; a = a->next)
	switch(IdOp(a->id))
	{
	InterestingCases:
		SetDef(a->id, d->in_cond);
		break;
	}
	if (d->delayed != NULL && d->special_delay) {
		/* Do special delayed store.  This mimics the
		 * algorithm used in dagtree.c */
		GenDly(d);
	}

	UpdateActivity(d, Done);
	return False;
}

/*
 *	If "id" has not already been marked as defined, mark it as
 *	used.
 */

static void
SetUse(id)
	Identifier id;
{
	assert(id != NoId);
	if( !IsElement((int)id, GBlock->ld.Def) )
		Insert((int)id, GBlock->ld.Use);
}

/*
 *	If "id" has not already been marked as used, mark it as possibly
 *	defined, and as defined if the DAG node is not conditional.
 */

static void
SetDef(id, cond)
	Identifier id;
	DagCount cond;		/* condition on defining DAG node */
{
	assert(id != NoId);
	if( cond == NotConditional &&  !IsElement((int)id, GBlock->ld.Use) )
		Insert((int)id, GBlock->ld.Def);
	Insert((int)id, GBlock->ld.PDef);
}

/*
 *	A pointer fetch where we do not know what the pointer might point
 *	to.  Mark as used anything that might be referenced.
 */

static void
IndirUse()
{
	Identifier id;

	for( id = FirstId; id <= MaxIdentifier; id++ )
		if( IdOp(id) == NAME || WasAddressed(id) )
			SetUse(id);
}

/*
 *	Note the things that might be used by a call: anything that we know
 *	is live on a call, less those things already defined in this block.
 */

static void
CallUse()
{
	Difference(LDTmpSet, LiveOnCall, GBlock->ld.Def);
	Union(GBlock->ld.Use, GBlock->ld.Use, LDTmpSet);
}

/*
 *	A pointer store where we do not know what the pointer might
 *	point to.  Mark as possibly defined anything that might be hit.
 */

/* ARGSUSED */
static void
IndirDef(cond)
	DagCount cond;
{
	Identifier id;

	for( id = FirstId; id <= MaxIdentifier; id++ )
		if( IdOp(id) == NAME || WasAddressed(id) )
			Insert((int)id, GBlock->ld.PDef);
}

/*
 * The activity count on node "d" has just gone to 0.  Do delayed stores
 * that are waiting on "d".
 */

static void
Done(d)
	DAG_Node d;
{
	assert(d->activity == 0);
	if( d->delayed && !(d->special_delay) )
		GenDly(d);
}

/*
 * "d" has a delayed store waiting on it.  Do it and (recursively) any
 * delayed stores waiting on this delayed store
 */

static void
GenDly(d)
	DAG_Node d;
{
	assert(optype(d->op) == LTYPE);
	switch(IdOp(d->leaf_id))
	{
	InterestingCases:
		assert(d->in_cond == d->delayed->in_cond);
		SetDef(d->leaf_id, d->in_cond);
		break;
	}
	assert(d->delayed->activity > 0);
	--(d->delayed->activity);
	if( d->delayed->activity == 0 )
		Done(d->delayed);
}

static void
PLDSet(s)
	LongSet s;
{
	int i;
	int count = 0;

	i = FirstElement(s);
	if( i != NoElement )
	{
		printf("\t\t");
		do
		{
			if( count > 20 )
			{
				count = 0;
				printf("\n\t\t");
			}
			printf("%d,", i);
			count++;
			i = NextElement(i,s);
		} while (i != NoElement);

		if( count != 0 )
			printf("\n");
	}
}

static void
PLDBlock(b)
	BasicBlock b;
{
	printf("LD Block: %o (#%d)\n", b, b->blocknum);
	printf("\tUse:\n"); PLDSet(b->ld.Use);
	printf("\tDef:\n"); PLDSet(b->ld.Def);
	printf("\tPDef:\n"); PLDSet(b->ld.PDef);
	if( ddebug > 2 )
	{
		printf("\tIn:\n"); PLDSet(b->ld.In);
		printf("\tOut:\n"); PLDSet(b->ld.Out);
	}
}

void
PLDInfo()
{
	int n;
	for( n = 0; n < NumBlocks; n++ )
	{
		PLDBlock(FlowGraph[n].block);
	}
}
