/* longset.c */
/*
 * HCR Confidential
 *
 * These computer programs are the confidential, proprietary property
 * of HCR (Human Computing Resources Corporation, 10 St. Mary Street,
 * Toronto, Ontario, Canada), and may not be disclosed except with the
 * prior written agreement of HCR.
 *
 * Copyright (c) 1984, 1985, 1986 Human Computing Resources Corporation
 * All Rights Reserved
 */

/*
 *	Long Set Implementation
 */

#ifndef lint
static char *rcsid = "@(#) (Gould) $Header: longset.c,v 5.5 89/05/12 12:51:32 pcc Rel-3_0 $";
/* static char ID[] = "@(#)longset.c	15.2	of 86/11/24"; */
#endif

/*
 *	Imported Objects
 */

#include <assert.h>
#include <memory.h>
#include <cmanifest.h>
#include <config.h>
#include <storage.h>
#include <longset.h>
#include <erroro.h>

/*	Private Definitions
 */

#define SetExpand	4	/* words per set expansion.  Move
				 * to storage.h at some point.
				 * Don't forget to move the def of
				 * RoundUp() too.
				 */

#define RoundUp(e)	( ((e) + (SetExpand - 1)) & ~(SetExpand - 1) )

#ifdef SetPower

/*
 *	A faster implementation.  Note that the expression e is always
 *	nonnegative.
 */

#define word(e)		((e) >> SetPower)
#define bit(e)		((e) & SetMask)
#define WordsFor(n)	(((n)+SETBITS-1) >> SetPower)
#define Bytes(w)	((w)*(sizeof (unsigned long)))

#else

/*
 *	A general purpose version, but may be slower.  On a good compiler,
 *	it wouldn't be.
 */

#define word(e)		((e)/SETBITS)
#define bit(e)		((e)%SETBITS)
#define WordsFor(n)	(((n)+SETBITS-1)/SETBITS)
#define Bytes(w)	((w)*(sizeof (unsigned long)))

#endif

#ifdef UseVariableShifts
#define ShiftBit(e)	(1 << (e))
#else

#define ShiftBit(e)	(ShiftTable[(e)])

static unsigned long ShiftTable[SETBITS];	/* Filled in below */

#endif


static LongSet LSFree;		/* LSet free list */

void
InitSets()
{

#ifndef UseVariableShifts

	/*
 	 *	Initialize the bit shift table.  Entry i is an unsigned
	 *	long with bit i set and all other bits zero.
	 */

	{
		register int bit_num;

		for (bit_num = 0; bit_num < SETBITS; ++bit_num)
			ShiftTable[bit_num] = 1 << bit_num;
	}

#endif

}

LongSet
CreateSet(n)		/* Create a Set with room for n elements */
	register int n;
{
	register LongSet l;
	SetDescriptor *c;		/* a collection of LSets */
	register int i;

	if (LSFree == NULL) {
		c = GetArray(s_Sets, SetChunk, SetDescriptor);
		CheckStorage(c, "storage for set descriptors", 0);

		/* Turn array into linked list (backwards!) */

		l = NULL;
		for (i=0; i < SetChunk; ++i) {
			c->u.next = l;
			l = c++;
		}

		/*
		 * Now l points to the head of the free list (the
		 * last entry in the chunk.
		 */
		
	} else
		l = LSFree;

	LSFree = l->u.next;

	/*
	 *	Set up the set
	 */

	l->words = WordsFor(n);
	l->u.v = (unsigned long *) calloc((unsigned) l->words, sizeof(unsigned long));
	CheckStorage(l->u.v, "storage for set bit vectors (%d words)", l->words);
	IncreaseSpace(s_SetV, (l->words)*(sizeof(unsigned long)));

	NullSet(l);
	return l;
}

static void
IncreaseSet(S, w)		/* Expand S to w words */
	register LongSet S;
	register int w;
{
	register int more;

	more = w - S->words;

	S->u.v = (unsigned long *) realloc( (char *) S->u.v,
					 (unsigned) Bytes(S->words+more));
	CheckStorage(S->u.v, "storage for set bit vectors", 0);

	/*
	 *	Bit vector expanded.  Ensure it is zero
	 */

#ifdef FASTSET
	(void) memset((char *) &S->u.v[S->words], 0, (int) Bytes(more));
#else
	for( i = 0; i < more; i++ )
		S->u.v[S->words+i] = 0L;
#endif
	S->words += more;
}

void
ExpandSet(S, e)			/* Expand S to have e elements */
	LongSet S;
	int e;
{
	register int newwords;

	newwords = RoundUp(WordsFor(e));
	if( newwords != S->words )
		IncreaseSet(S, newwords);
}

void
DestroySet(S)
	LongSet S;
{
	free((char *) S->u.v);
	DecreaseSpace(s_SetV, (S->words * sizeof(unsigned long)));
	S->u.next = LSFree;
	LSFree = S;
}

Boolean
IsElement(e, S)			/* Is e in S? */
	register int e;
	register LongSet S;
{
	assert(e >= 0);
	if (word(e) < S->words)		/* It's in the array */
		return ((S->u.v[word(e)] & ShiftBit(bit(e))) != 0);
	else
		return False;
}

void
Insert(e, S)			/* Insert e into S */
	register int e;
	register LongSet S;
{
	assert(e >= 0);
	if( word(e) >= S->words)	/* e is outside the array */
		ExpandSet(S,e+1);	/* e+1 elements to set element e */

	S->u.v[word(e)] |= ShiftBit(bit(e));
}

void
DelMember(e, S)			/* Delete e from S */
	register int e;
	register LongSet S;
{
	/*	This operation won't expand the set */

	assert(e >= 0 && word(e) < S->words);

	S->u.v[word(e)] &= ~ShiftBit(bit(e));
}

void
NullSet(S)			/* S = null */
	register LongSet S;
{

#ifndef FASTSET
	register int i;
	register int words;

	words = S->words;
	for (i=0; i < words; ++i)
		S->u.v[i] = 0L;
#else
	(void) memset((char *) S->u.v, 0, (int) Bytes(S->words));
#endif
}

void
Intersection(r, a, b)		/* r = a intersect b */
	LongSet r, a, b;
{

#ifndef FASTSET

	register int i;
	register int maxwords;

	maxwords = (a->words > b->words) ? a->words : b->words;
	if( maxwords > r->words )
		ExpandSet(r, maxwords*SETBITS);

	/* Ensure that the shorter set is b */

	if( a->words < b->words )
	{
		LongSet t;
		t = a; a = b; b = t;
	}

	/* Take intersection of common part */

	for (i=0; i < b->words; ++i)
		r->u.v[i] = a->u.v[i] & b->u.v[i];

	/* Then zero out the rest */

	for (; i < r->words; ++i)

#else
	register unsigned long *rp, *ap, *bp;
	register int words;
	register int maxwords;

	maxwords = (a->words > b->words) ? a->words : b->words;
	if( maxwords > r->words )
		ExpandSet(r, maxwords*SETBITS);

	/* Ensure shorter set is b */

	if( a->words < b->words )
	{
		LongSet t;
		t = a; a = b; b = t;
	}

	rp = r->u.v;
	ap = a->u.v;
	bp = b->u.v;

	/* Take intersection of common part */

	for (words = b->words; words > 0; --words) {
		*rp++ = *ap++ & *bp++;
	}

	/* Then zero out the rest */

	words = r->words - b->words;
	if( words > 0 )
		(void) memset((char *)rp, 0, (int) Bytes(words));
#endif
}

void
Union(r, a, b)			/* r = a union b */
	LongSet r, a, b;
{

#ifndef FASTSET

	register int maxwords;
	register int i;

	maxwords = (a->words > b->words) ? a->words : b->words;
	if( maxwords > r->words )
		ExpandSet(r, maxwords*SETBITS);

	/* Ensure that the shorter set is b */

	if( a->words < b->words )
	{
		LongSet t;
		t = a; a = b; b = t;
	}

	/* First, take union of parts common to a and b */

	for (i=0; i < b->words; ++i)
		r->u.v[i] = a->u.v[i] | b->u.v[i];

	/*
	 * Now if a and b had unequal sizes, copy the extra part
	 * If the result and longer set are aliased, the copy
	 * isn't necessary.
	 */
	
	if( a != r )
		for( ; i < a->words; ++i )
			r->u.v[i] = a->u.v[i];

	/* Finally, zero out all the stuff on the end */

	for (; i < r->words; ++i)
		r->u.v[i] = 0L;

#else
	register unsigned long *rp, *ap, *bp;
	register int words;
	register int maxwords;

	maxwords = (a->words > b->words) ? a->words : b->words;
	if( maxwords > r->words )
		ExpandSet(r, maxwords*SETBITS);

	/* Ensure that shorter set is b */

	if( a->words < b->words )
	{
		LongSet t;
		t = a; a = b; b = t;
	}

	/* First, take union of parts common to a and b */

	rp = r->u.v;
	ap = a->u.v;
	bp = b->u.v;

	for (words = b->words; words > 0; --words) {
		*rp++ = *ap++ | *bp++;
	}

	/*
	 * Now if a and b had unequal sizes, copy the extra part.
	 * If the result and a are aliased, copy isn't required.
	 * It is not clear if we should use memcpy here or just do it.
	 */

	words = a->words - b->words;
	if( words > 0 )
	{
		if( r != a )
			(void) memcpy((char *)rp, (char *)ap, (int) Bytes(words));
		rp += words;
	}

	/* Finally, zero out all the stuff on the end */

	words = r->words - a->words;
	if( words > 0 )
		(void) memset((char *)rp, 0, (int) Bytes(words));
#endif
}

void
Difference(r, a, b)		/* r = a - b */
	LongSet r, a, b;
{

#ifndef FASTSET
	register int words;
	register int i;
	int maxwords, minwords;

	minwords = (a->words < b->words) ? a->words : b->words;
	maxwords = (a->words > b->words) ? a->words : b->words;
	if( maxwords > r->words )
		ExpandSet(r, maxwords*SETBITS);

	/* Perform difference op up to the end of the shorter of a and b */

	for (i=0; i < minwords; ++i)
		r->u.v[i] = a->u.v[i] & ~(b->u.v[i]);

	/* Now, if a and r are the same we are done.  Otherwise copy the rest
	 * of a into r and fill anything left over with zeros
	 */

	if( a != r )
	{
		words = a->words;
		for (; i < words; ++i)
			r->u.v[i] = a->u.v[i];
		words = r->words;
		for (; i < words; ++i )
			r->u.v[i] = 0L;
	}
#else
	register unsigned long *rp, *ap, *bp;
	register int words;
	int maxwords, minwords;

	minwords = (a->words < b->words) ? a->words : b->words;
	maxwords = (a->words > b->words) ? a->words : b->words;
	if( maxwords > r->words )
		ExpandSet(r, maxwords*SETBITS);

	rp = r->u.v;
	ap = a->u.v;
	bp = b->u.v;

	/* Perform difference op up to the end of the shorter of a and b */

	for (words = minwords; words > 0; --words) {
		*rp++ = *ap++ & ~(*bp++);
	}

	/* Now, if a and r are the same we are done.  Otherwise copy the rest
	 * of a into r and fill anything left over with zeros
	 */

	if( a != r )
	{
		words = a->words - minwords;
		if( words > 0 )
		{
			(void) memcpy((char *)rp, (char *)ap, (int) Bytes(words));
			rp += words;
		}

		words = r->words - a->words;
		if( words > 0 )
			(void) memset((char *)rp, 0, (int) Bytes(words));
	}
#endif
}

Boolean
SetEq(a, b)		/* is a == b ? */
	register LongSet a, b;
{

#ifndef FASTSET
	register int i, l;

	/* Get the sets the same size */

	if (a->words < b->words)
		IncreaseSet(a, b->words);
	else
	if (a->words > b->words)
		IncreaseSet(b, a->words);

	i = 0; l = a->words;

	while (i < l && a->u.v[i] == b->u.v[i]) ++i;

	return (i == l);
#else

	/* Get the sets the same size */

	if (a->words < b->words)
		IncreaseSet(a, b->words);
	else
	if (a->words > b->words)
		IncreaseSet(b, a->words);

	return (memcmp((char *)a->u.v, (char *)b->u.v, (int) Bytes(a->words)) == 0);
#endif
}

Boolean
Subset(a, b)		/* Is a a subset of b? */
	LongSet a, b;
{

#ifndef	FASTSET

	register int i, l;

	if (a->words < b->words)
		IncreaseSet(a, b->words);
	else
	if (a->words > b->words)
		IncreaseSet(b, a->words);

	i = 0; l = a->words;

	while (i < l && (a->u.v[i] & ~(b->u.v[i])) == 0) ++i;

#else

	register unsigned long *ap, *bp;
	register int i, l;

	if (a->words < b->words)
		IncreaseSet(a, b->words);
	else
	if (a->words > b->words)
		IncreaseSet(b, a->words);

	i = 0; l = a->words;

	ap = a->u.v;
	bp = b->u.v;
	while (i < l && (*ap++ & ~(*bp++)) == 0) ++i;

#endif

	return (i == l);
}

Boolean
Disjoint(a, b)		/* Are a and b disjoint? */
	LongSet a, b;
{

#ifndef	FASTSET

	register int i, l;

	if (a->words < b->words)
		IncreaseSet(a, b->words);
	else
	if (a->words > b->words)
		IncreaseSet(b, a->words);

	i = 0; l = a->words;

 	while (i < l && (a->u.v[i] & b->u.v[i]) == 0) ++i;

#else

	register unsigned long *ap, *bp;
	register int i, l;

	if (a->words < b->words)
		IncreaseSet(a, b->words);
	else
	if (a->words > b->words)
		IncreaseSet(b, a->words);

	i = 0; l = a->words;

	ap = a->u.v;
	bp = b->u.v;
	while (i < l && (*ap++ & *bp++) == 0) ++i;

#endif

	return (i == l);
}

void
CopySet(a, b)		/* a = b;  (a must exist!) */
	register LongSet a, b;
{

#ifndef FASTSET
	register int i, l;

	if (a->words < b->words)
		ExpandSet(a, b->words*SETBITS);

	l = b->words;	
	for (i=0; i < l; ++i)
		a->u.v[i] = b->u.v[i];

	/* Clear any extra elements */

	l = a->words;
	for (; i > l; ++i)
		a->u.v[i] = 0L;
#else

	register int words;

	if (a->words < b->words)
		ExpandSet(a, b->words*SETBITS);

	(void) memcpy((char *)a->u.v, (char *)b->u.v, (int) Bytes(b->words));
	words = a->words - b->words;
	if (words > 0)
		(void) memset((char *)&a->u.v[b->words], 0, (int) Bytes(words));
	
#endif

}

/*
 * Number of bits in each possible pattern of 8 bits
 */

static unsigned char TranslateTable[256] = {
 0, 1, 1, 2, 1, 2, 2, 3, 1, 2, 2, 3, 2, 3, 3, 4,	/* 0x00 - 0x0f */
 1, 2, 2, 3, 2, 3, 3, 4, 2, 3, 3, 4, 3, 4, 4, 5,	/* 0x10 - 0x1f */
 1, 2, 2, 3, 2, 3, 3, 4, 2, 3, 3, 4, 3, 4, 4, 5,	/* 0x20 - 0x2f */
 2, 3, 3, 4, 3, 4, 4, 5, 3, 4, 4, 5, 4, 5, 5, 6,	/* 0x30 - 0x3f */
 1, 2, 2, 3, 2, 3, 3, 4, 2, 3, 3, 4, 3, 4, 4, 5,	/* 0x40 - 0x4f */
 2, 3, 3, 4, 3, 4, 4, 5, 3, 4, 4, 5, 4, 5, 5, 6,	/* 0x50 - 0x5f */
 2, 3, 3, 4, 3, 4, 4, 5, 3, 4, 4, 5, 4, 5, 5, 6,	/* 0x60 - 0x6f */
 3, 4, 4, 5, 4, 5, 5, 6, 4, 5, 5, 6, 5, 6, 6, 7,	/* 0x70 - 0x7f */
 1, 2, 2, 3, 2, 3, 3, 4, 2, 3, 3, 4, 3, 4, 4, 5,	/* 0x80 - 0x8f */
 2, 3, 3, 4, 3, 4, 4, 5, 3, 4, 4, 5, 4, 5, 5, 6,	/* 0x90 - 0x9f */
 2, 3, 3, 4, 3, 4, 4, 5, 3, 4, 4, 5, 4, 5, 5, 6,	/* 0xa0 - 0xaf */
 3, 4, 4, 5, 4, 5, 5, 6, 4, 5, 5, 6, 5, 6, 6, 7,	/* 0xb0 - 0xbf */
 2, 3, 3, 4, 3, 4, 4, 5, 3, 4, 4, 5, 4, 5, 5, 6,	/* 0xc0 - 0xcf */
 3, 4, 4, 5, 4, 5, 5, 6, 4, 5, 5, 6, 5, 6, 6, 7,	/* 0xd0 - 0xdf */
 3, 4, 4, 5, 4, 5, 5, 6, 4, 5, 5, 6, 5, 6, 6, 7,	/* 0xe0 - 0xef */
 4, 5, 5, 6, 5, 6, 6, 7, 5, 6, 6, 7, 6, 7, 7, 8		/* 0xf0 - 0xff */
};

/*
 * Find the cardinality of a set.  This code is machine dependent in that it
 * assumes
 * 1) all the bits in the set can be accessed by converting s->u.v to
 *    an unsigned character pointer
 * 2) these unsigned characters can be used as indices in the translate table
 * 3) there are 8 bits per byte.
 * If these assumptions are wrong, do not define MDEP_CARDINALITY, and the
 * portable code below will be used.
 */

#ifdef MDEP_CARDINALITY
int
Cardinality(s)
	LongSet s;
{
	unsigned char *p;
	int i, max, count;

	max = s->words * (SETBITS/8);
	p = (unsigned char *) s->u.v;
	count = 0;

	for( i = 0; i < max; i++ )
		count += TranslateTable[*p++];
	return(count);
}

#else
int
Cardinality(s)
	LongSet s;
{
	int count, i, j;
	unsigned long Word;

	count = 0;

	for( i = 0; i < s->words; i++ )
	if( (Word = s->u.v[i]) != 0 )
	{
		for(j = 0; j < (SETBITS+7)/8; j++)
		{
			count += TranslateTable[Word&0xff];
			Word >>= 8;
		}
	}
	return(count);
}
#endif

/*
 * Find the first element in a set.
 */

int
FirstElement(s)
	LongSet s;
{
	register int Word, Bit;
	register unsigned long *p;
	register unsigned long Elements;

	p = s->u.v;
	for( Word = 0; Word < s->words; Word++ )
	{
		Elements = *p++;
		if( Elements != 0 )
		{
			for( Bit = 0; Bit < SETBITS; Bit++)
			{
				if( (Elements & 1) != 0 )
					return Word*SETBITS + Bit;
				Elements >>= 1;
			}
		}
	}
	return NoElement;
}

/*
 * Find the next element of the set S following element e
 */
int
NextElement(e, S)
	int e;
	LongSet S;
{
	int Word, Bit;
	unsigned long *p;
	unsigned long Elements;

	Word = word(e);
	assert(Word >= 0 && Word < S->words);
	Bit = bit(e) + 1;
	if( Bit >= SETBITS )
	{
		Word++;
		if( Word >= S->words )
			return NoElement;
		Bit = 0;
	}
	p = &S->u.v[Word];

	for(Elements = (*p++) >> Bit; ;Elements = *p++)
	{
		if( Elements != 0 )
		{
			for(; Bit < SETBITS; Bit++ )
			{
				if( (Elements & 1) != 0)
					return Word*SETBITS + Bit;
				Elements >>= 1;
			}
		}
		Word++;
		if( Word >= S->words )
			return NoElement;
		Bit = 0;
	}
	/*NOTREACHED*/
}

/*
 * Print a set in readable form.  Used for debugging
 */

void
DumpSet(s)
	LongSet s;
{
	unsigned long *p;
	unsigned long Elements;
	int Word, Bit;

	printf("{");
	p = s->u.v;
	for( Word = 0; Word < s->words; Word++)
	{
		Elements = *p++;
		Bit = 0;
		while( Elements != 0 )
		{
			if( (Elements & 1) != 0)
				printf("%d ", Word*SETBITS + Bit);
			Elements >>= 1;
			Bit++;
		}
	}
	printf("}");
}
