#ifndef E3MEMOBJ_H
#define E3MEMOBJ_H

#include "memmap.h"
#include "macroop.h"
#include "lispm.h"		/* For global (ugh) lispm */

/*
 * Base functionality for looking at memory object
 */

class memObj
{
public:
  memObj(e3WordAddr);
  virtual ~memObj();

  virtual e3u32   sizeOfInWords(void) const; /* If not over-ridden, returns 1. */
  virtual void    dump(ostream &) const; /* If not over-ridden, dumps single e3Word at address. */

  static  memObj  *makeRightObjType(e3WordAddr);

protected:

  e3Word           readMe(void) const;
  e3Word           readWordAt(e3WordAddr) const;

  e3WordAddr       myWordAddress;
};

/*
 * Specific types of memory object
 */

class memTrap: public memObj
{
public:
  memTrap(e3WordAddr);
};

class memList: public memObj
{
public:
  memList(e3WordAddr);
  e3u32 sizeOfInWords(void) const;
  void   dump(ostream &os) const;
};

class memStackList: public memObj
{
public:
  memStackList(e3WordAddr);
  e3u32  sizeOfInWords(void) const;
  void    dump(ostream &os) const;
};

class memSymbol: public memObj
{
public:
  memSymbol(e3WordAddr);
};

class memArray: public memObj
{
public:
  memArray(e3WordAddr);
  void       dump(ostream &os) const;
  e3WordAddr getArrayHeaderAddress(void) const;
};

class memFix: public memObj
{
public:
  memFix(e3WordAddr);

  e3s32 value(void) const;
};

class memCharacter: public memObj
{
public:
  memCharacter(e3WordAddr);
};

class memSingleFloat: public memObj
{
public:
  memSingleFloat(e3WordAddr);
  e3u32 sizeOfInWords(void) const;
  void   dump(ostream &os) const;
};

class memShortFloat: public memObj
{
public:
  memShortFloat(e3WordAddr);
};

class memInstance: public memObj
{
public:
  memInstance(e3WordAddr);
};

class memExtendedNumber: public memObj
{
public:
  memExtendedNumber(e3WordAddr);
};

class memLocative: public memObj
{
public:
  memLocative(e3WordAddr);
};

class memFunction: public memObj
{
public:
  memFunction(e3WordAddr);
};

class memClosure: public memObj
{
public:
  memClosure(e3WordAddr);
};

class memLexicalClosure: public memObj
{
public:
  memLexicalClosure(e3WordAddr);
  e3u32 sizeOfInWords(void) const;
  void   dump(ostream &os) const;
};

class memUEntry: public memObj
{
public:
  memUEntry(e3WordAddr);
};

/* Is a kind-of array pointer. */
/* SSDN2 pg 14-2 ff. */
class memStackGroup: public memArray
{
public:
  memStackGroup(e3WordAddr);
  void   dump(ostream &os) const;
};

class memGCForward: public memObj
{
public:
  memGCForward(e3WordAddr);
};

class memExternalValueCellPointer: public memObj
{
public:
  memExternalValueCellPointer(e3WordAddr);
};

class memOneQForward: public memObj
{
public:
  memOneQForward(e3WordAddr);
};

class memHeaderForward: public memObj
{
public:
  memHeaderForward(e3WordAddr);
};

class memBodyForward: public memObj
{
public:
  memBodyForward(e3WordAddr);

};

class memSymbolHeader: public memObj
{
public:
  memSymbolHeader(e3WordAddr);
  e3u32  sizeOfInWords(void) const;
  void    dump(ostream &os) const;

  e3Word printNameCell(void) const;
  e3Word valueCell(void) const;
  e3Word functionCell(void) const;
  e3Word propertyCell(void) const;
  e3Word packageCell(void) const;
};

/* SSDN2 pg 7-14 */
typedef enum
{
  HEADER_TYPE_ERROR,
  HEADER_UNUSED_1,
  HEADER_TYPE_ARRAY_LEADER,
  HEADER_TYPE_UNUSED_3,
  HEADER_TYPE_SINGLE_FLOAT,
  HEADER_TYPE_COMPLEX,
  HEADER_TYPE_BIGNUM,
  HEADER_TYPE_RATIONAL,
  HEADER_TYPE_DOUBLE_FLOAT
} memHeaderType;

/* SSDN2 pg 8-4 */
class memHeader: public memObj
{
public:
  memHeader(e3WordAddr);
  void          dump(ostream &os) const;

  memHeaderType getHeaderType(void) const;
  e3u32         getHeaderRestField(void) const;
};

typedef enum
{
  ART_ERROR,
  ART_1B,
  ART_2B,
  ART_4B,
  ART_8B,
  ART_16B,
  ART_32B,
  ART_Q,
  ART_Q_LIST,
  ART_STRING,
  ART_STACK_GROUP_HEAD,
  ART_SPECIAL_PDL,
  ART_HALF_FIX,
  ART_REG_PDL,
  ART_DOUBLE_FLOAT,
  ART_SINGLE_FLOAT,
  ART_FAT_STRING,
  ART_COMPLEX_DOUBLE_FLOAT,
  ART_COMPLEX,
  ART_COMPLEX_SINGLE_FLOAT,
  ART_FIX,
  ART_EXTENDED_FIX
} e3ArrayTypes;

class memArrayHeader: public memObj
{
public:
  memArrayHeader(e3WordAddr);
  void          dump(ostream &os) const;

  e3ArrayTypes  arrayType(void) const;
  e3Boolean     physicalBit(void) const;
  e3Boolean     hasLeader(void) const;
  e3Boolean     isDisplaced(void) const;
  e3Boolean     simpleBit(void) const;
  e3u8          numberOfDims(void) const;
  e3Boolean     namedStructureFlag(void) const;
  e3Boolean     longLengthFlag(void) const;
  e3u16         indexLength(void) const;
  
#ifdef NOTDEF
  e3u32         sizeOfInWords(void) const;
  e3u32         maxIndexLength(void) const;
#endif

  /* Deal with leader. */
  void         dumpLeader(ostream &os) const;
  e3s32        leaderLength(void) const;
  e3Word       leaderWordFromEnd(e3s32 index) const;

 protected:

  e3WordAddr   leaderAddress(void) const;
};

/*
 * These are the stack group "states"
 * of Table 14-1 on page 14-6.
 */

typedef enum
{
    SG_STATE_ERROR = 0,		/* Should never get this. */
    SG_STATE_ACTIVE,		/* Actually executing on the machine */
    SG_STATE_RESUMABLE,
    SG_STATE_AWAITING_RETURN,
    SG_STATE_INVOKE_CALL_ON_RETURN,
    SG_STATE_INTERRUPTED_DIRTY,
    SG_STATE_AWAITING_ERROR_RECOVERY,
    SG_STATE_AWAITING_CALL,
    SG_STATE_AWAITING_INITIAL_CALL,
    SG_STATE_EXHAUSTED
} memStackGroupState;

/*
 * This is the stack group array header subtype of
 * the vanilla memArrayHeader.  It has no analog in the
 * SSDN2 document -- it just makes some things easier.
 */
class memStackGroupArrayHeader: public memArrayHeader
{
 public:
  memStackGroupArrayHeader(e3WordAddr);
  void   dump(ostream &os) const;

  /* Static Section. */
  e3Word getName(void) const                 { return(readWordAt(myWordAddress-2)); };
  e3Word getRegularPDL(void) const           { return(readWordAt(myWordAddress-3)); };
  e3Word getRegularPDLLimit(void) const      { return(readWordAt(myWordAddress-4)); };
  e3Word getSpecialPDL(void) const           { return(readWordAt(myWordAddress-5)); };
  e3Word getSpecialPDLLimit(void) const      { return(readWordAt(myWordAddress-6)); };
  e3Word getInitialFunctionIndex(void) const { return(readWordAt(myWordAddress-7)); };

  /* Debugging Section. */
  e3Word getTrapTag(void) const              { return(readWordAt(myWordAddress-8)); };
  e3Word getRecoveryHistory(void) const      { return(readWordAt(myWordAddress-9)); };
  e3Word getFootholdData(void) const         { return(readWordAt(myWordAddress+-10)); };

  /* High Level Section. */

  e3Word getState(void) const                { return(readWordAt(myWordAddress-11)); };
  void getStateDetail(ostream& os, char *fill) const;
  memStackGroupState getCurrentState(void) const;
  e3Boolean          getFootholdExecuting(void) const;
  e3Boolean          getProcessingError(void) const;
  e3Boolean          getProcessingInterruptRequest(void) const;
  e3Boolean          getSafe(void) const;
  e3u8               getInstDisp(void) const;
  e3Boolean          getSwapSvOfSgThatCallsMe(void) const;
  e3Boolean          getSwapSvOnCallOut(void) const;
  e3Boolean          getInSwappedInState(void) const;
  e3Boolean          getRestoreMicrostack(void) const;

  e3Word getPreviousStackGroup(void) const   { return(readWordAt(myWordAddress-12)); };
  e3Word getCallingArgsPointer(void) const   { return(readWordAt(myWordAddress-13)); };
  e3Word getCallingArgsNumber(void) const    { return(readWordAt(myWordAddress-14)); };
  e3Word getTrapAPLevel(void) const          { return(readWordAt(myWordAddress-15)); };

  /* Dynamic Section. */
  e3Word getRegularPDLPointer(void) const    { return(readWordAt(myWordAddress-16)); };
  e3Word getSpecialPDLPointer(void) const    { return(readWordAt(myWordAddress-17)); };
  e3Word getTrapMicroPC(void) const          { return(readWordAt(myWordAddress-18)); };
  e3Word getSavedMFlags(void) const          { return(readWordAt(myWordAddress-19)); };
  void   getSavedMFlagsDetail(ostream& os, char *fill) const;
  e3Word getPDLPhase(void) const             { return(readWordAt(myWordAddress-20)); };
  e3Word getSavedVMA(void) const             { return(readWordAt(myWordAddress-21)); };
  e3Word getVMAM1M2Tags(void) const          { return(readWordAt(myWordAddress-22)); };
  e3Word getM3M4Tags(void) const             { return(readWordAt(myWordAddress-23)); };

  /* Accumulators. */

  e3Word getSGAC1(void) const                { return(readWordAt(myWordAddress-24)); };
  e3Word getSGAC2(void) const                { return(readWordAt(myWordAddress-25)); };
  e3Word getSGAC3(void) const                { return(readWordAt(myWordAddress-26)); };
  e3Word getSGAC4(void) const                { return(readWordAt(myWordAddress-27)); };
  e3Word getSGACA(void) const                { return(readWordAt(myWordAddress-28)); };
  e3Word getSGACB(void) const                { return(readWordAt(myWordAddress-29)); };
  e3Word getSGACC(void) const                { return(readWordAt(myWordAddress-30)); };
  e3Word getSGACD(void) const                { return(readWordAt(myWordAddress-31)); };
  e3Word getSGACE(void) const                { return(readWordAt(myWordAddress-32)); };
  e3Word getSGACF(void) const                { return(readWordAt(myWordAddress-33)); };
  e3Word getSGACG(void) const                { return(readWordAt(myWordAddress-34)); };
  e3Word getSGACH(void) const                { return(readWordAt(myWordAddress-35)); };
  e3Word getSGACI(void) const                { return(readWordAt(myWordAddress-36)); };
  e3Word getSGACJ(void) const                { return(readWordAt(myWordAddress-37)); };
  e3Word getSGACK(void) const                { return(readWordAt(myWordAddress-38)); };
  e3Word getSGACL(void) const                { return(readWordAt(myWordAddress-39)); };
  e3Word getSGACQ(void) const                { return(readWordAt(myWordAddress-40)); };
  e3Word getSGACR(void) const                { return(readWordAt(myWordAddress-41)); };
  e3Word getSGACS(void) const                { return(readWordAt(myWordAddress-42)); };
  e3Word getSGACZR(void) const               { return(readWordAt(myWordAddress-43)); };
  e3Word getSGACT(void) const                { return(readWordAt(myWordAddress-44)); };

  e3Word getCatchPointer(void) const         { return(readWordAt(myWordAddress-45)); };
};

class memInstanceHeader: public memObj
{
public:
  memInstanceHeader(e3WordAddr);
  e3u32 sizeOfInWords(void) const;
  void   dump(ostream &os) const;
};

typedef enum
{
  CALL_SIMPLE = 0,
  CALL_OPTIONALS,
  CALL_LOCALS,
  CALL_REST,
  CALL_OPTIONALS_AND_LOCALS,
  CALL_OPTIONALS_AND_REST,
  CALL_UNUSED,
  CALL_LONG
} vlmCallType;

class memFEFHeader: public memObj
{
public:
  memFEFHeader(const e3WordAddr);
  e3u32 sizeOfInWords(void) const;
  void   dump(ostream &os) const;

  /* 1st word and accessors. */
  e3Word     getHeaderQ(void) const;
  e3Boolean  isSpecialForm(void) const; // 1 if &QUOTE or &FUNCTIONAL
  e3Boolean  isSubst(void) const; // 1 if DEFSUBST
  e3Boolean  selfMappingTable(void) const;
  vlmCallType callType(void) const;
  e3u8       numberOptionalArgs(void) const;
  e3u8       numberArgs(void) const;
  e3u8       numberLocals(void) const;
  e3u16      locationCounterOffset(void) const;
  
  /* 2nd word and accessors?. */
  e3Word     getStorageLengthQ(void) const;
  e3Boolean  isGenericFunction(void) const; /* lroy has this, but SSDN2 doesn't. */

  /* 3rd word. */
  e3Word     getPointerToDebuggingInfo(void) const;

  /* Optional 4th word and accessors. */
  e3Word     getOptionalLongArgsWordQ(void) const;
  e3Boolean  longArgsOptionals(void) const;
  e3Boolean  longArgsLocals(void) const;
  e3Boolean  longArgsRestArg(void) const;
  e3u8       longArgsMinArgs(void) const;
  e3u8       longArgsMaxArgs(void) const;
  e3u8       longArgsNumberOfLocals(void) const;

  /* Optional 4th/5th ($$$$$?) word. */
  e3Word     getFlavorNameQ(void) const;

  e3MacroInstruction readMacroInstruction(e3u32 entry_offset, e3u8 relative_pc) const; /* Inline later. */
  e3u32              numberOfMacroInstructions(void) const;

 private:

};

class memSelfRefPointer: public memObj
{
public:
  memSelfRefPointer(e3WordAddr);
};

class memGCYoungPointer: public memObj
{
public:
  memGCYoungPointer(e3WordAddr);
  e3u32 sizeOfInWords(void) const;
  void   dump(ostream &os) const;
};

class memFree: public memObj
{
public:
  memFree(e3WordAddr);
};

class memNull: public memObj
{
public:
  memNull(e3WordAddr);
};

class memOnesTrap: public memObj
{
public:
  memOnesTrap(e3WordAddr);
};

#endif /* E3MEMOBJ_H */
