/*
 * disasm.c
 *
 * disassembler for macrocode
 */

#include "utils.h"

#include <common/types.h>
#include <stdio.h>

#define OPT_UNKNOWN 0
#define OPT_MAIN    1
#define OPT_BRANCH  2
#define OPT_IMMED   3
#define OPT_MISC    4
#define OPT_AUX     5
#define OPT_AREFI   6
#define OPT_MODULE  7

struct instr {
    int type; /* OPT_xxx constants */
    char *name;
} instructions[0x80] = {
    /* 00 - 07 */
    {OPT_AUX},
    {OPT_MISC},
    {OPT_MODULE},
    {OPT_IMMED, "EQ-IMMED"},
    {OPT_IMMED, "=-IMMED"},
    {OPT_IMMED, ">-IMMED"},
    {OPT_IMMED, "<-IMMED"},
    {OPT_AREFI},

    /* 08 - 0f */
    {OPT_MAIN, "TEST"},
    {OPT_MAIN, "TEST-CAR"},
    {OPT_MAIN, "TEST-CDR"},
    {OPT_MAIN, "TEST-CADR"},
    {OPT_MAIN, "TEST-CDDR"},
    {OPT_MAIN, "TEST-CAAR"},
    {OPT_MAIN, "TEST-MEMO"},
    {OPT_MAIN, "RETURN"},
    
    /* 10 - 17 */
    {OPT_UNKNOWN},
    {OPT_UNKNOWN},
    {OPT_UNKNOWN},
    {OPT_MAIN, "EQ"},
    {OPT_MAIN, "EQL"},
    {OPT_MAIN, "EQUAL"},
    {OPT_MAIN, "EQUALP"},
    {OPT_UNKNOWN},

    /* 18 - 1f */
    {OPT_MAIN, "NUMBERP"},
    {OPT_MAIN, "ARRAYP"},
    {OPT_MAIN, "LISTP"},
    {OPT_MAIN, "STRINGP"},
    {OPT_MAIN, "FIXNUMP"},
    {OPT_MAIN, "INTEGERP"},
    {OPT_MAIN, "PLUSP"},
    {OPT_MAIN, "MINUSP"},
    
    /* 20 - 27 */
    {OPT_UNKNOWN},
    {OPT_MISC},
    {OPT_MODULE},
    {OPT_UNKNOWN},
    {OPT_UNKNOWN},
    {OPT_IMMED, "PUSH-NUMBER"}, /* ??? SSDN2 disagrees? */
    {OPT_IMMED, "PUSH-NEG-NUMBER"},
    {OPT_AREFI}, /* ??? SSDN2 says this is PUSH-NUMBER? */

    /* 28 - 2f */
    {OPT_MAIN, "PUSH"},
    {OPT_MAIN, "PUSH-CAR"},
    {OPT_MAIN, "PUSH-CDR"},
    {OPT_MAIN, "PUSH-CADR"},
    {OPT_MAIN, "PUSH-CDDR"},
    {OPT_MAIN, "PUSH-CADDR"},
    {OPT_MAIN, "PUSH-CONS"},
    {OPT_MAIN, "PUSH-GET"},
    
    /* 30 - 37 */
    {OPT_MAIN, "+"},
    {OPT_MAIN, "-"},
    {OPT_MAIN, "*"},
    {OPT_MAIN, "LOGAND"},
    {OPT_MAIN, "LOGXOR"},
    {OPT_MAIN, "1+"},
    {OPT_MAIN, "1-"},
    {OPT_MAIN, "PUSH-AR-1"},

    /* 38 - 3f */
    {OPT_IMMED, "PUSH-LONG-FEF"},
    {OPT_MAIN, "SELECT"},
    {OPT_MAIN, "DISPATCH"}, /* ??? The descrition implies OPT_IMMED */
    {OPT_UNKNOWN},
    {OPT_UNKNOWN},
    {OPT_UNKNOWN},
    {OPT_MAIN, "LEXICAL-UNSHARE"}, /* ??? Doesn't specify OPT_xxx? */
    {OPT_MAIN, "LOCATE-LEXICAL-ENVIRONMENT"},
    
    /* 40 - 47 */
    {OPT_MAIN, "CALL-0-DEST-INDS"},
    {OPT_MAIN, "CALL-0-DEST-PUSH"},
    {OPT_MAIN, "CALL-0-DEST-RETURN"},
    {OPT_MAIN, "CALL-0-DEST-TAIL-REC"},
    {OPT_MAIN, "CALL-1-DEST-INDS"},
    {OPT_MAIN, "CALL-1-DEST-PUSH"},
    {OPT_MAIN, "CALL-1-DEST-RETURN"},
    {OPT_MAIN, "CALL-1-DEST-TAIL-REC"},

    /* 48 - 4f */
    {OPT_MAIN, "CALL-2-DEST-INDS"},
    {OPT_MAIN, "CALL-2-DEST-PUSH"},
    {OPT_MAIN, "CALL-2-DEST-RETURN"},
    {OPT_MAIN, "CALL-2-DEST-TAIL-REC"},
    {OPT_MAIN, "CALL-3-DEST-INDS"},
    {OPT_MAIN, "CALL-3-DEST-PUSH"},
    {OPT_MAIN, "CALL-3-DEST-RETURN"},
    {OPT_MAIN, "CALL-3-DEST-TAIL-REC"},

    /* 50 - 57 */
    {OPT_MAIN, "CALL-4-DEST-INDS"},
    {OPT_MAIN, "CALL-4-DEST-PUSH"},
    {OPT_MAIN, "CALL-4-DEST-RETURN"},
    {OPT_MAIN, "CALL-4-DEST-TAIL-REC"},
    {OPT_MAIN, "CALL-5-DEST-INDS"},
    {OPT_MAIN, "CALL-5-DEST-PUSH"},
    {OPT_MAIN, "CALL-5-DEST-RETURN"},
    {OPT_MAIN, "CALL-5-DEST-TAIL-REC"},

    /* 58 - 5f */
    {OPT_MAIN, "CALL-6-DEST-INDS"},
    {OPT_MAIN, "CALL-6-DEST-PUSH"},
    {OPT_MAIN, "CALL-6-DEST-RETURN"},
    {OPT_MAIN, "CALL-6-DEST-TAIL-REC"},
    {OPT_MAIN, "CALL-N-DEST-INDS"},
    {OPT_MAIN, "CALL-N-DEST-PUSH"},
    {OPT_MAIN, "CALL-N-DEST-RETURN"},
    {OPT_MAIN, "CALL-N-DEST-TAIL-REC"},
    
    /* 60 - 67 */
    {OPT_MAIN, "POP"},
    {OPT_MAIN, "MOVEM"},
    {OPT_MAIN, "SETE-CDR"},
    {OPT_MAIN, "SETE-CDDR"},
    {OPT_MAIN, "SETE-1+"},
    {OPT_MAIN, "SETE-1-"},
    {OPT_UNKNOWN},
    {OPT_MAIN, "PUSH-CDR-STORE-CAR-IF-CONS"},

    /* 68 - 6f */
    {OPT_MAIN, "PUSH-LOC"},
    {OPT_MAIN, "BIND-NIL"},
    {OPT_MAIN, "BIND-T"},
    {OPT_MAIN, "BIND-POP"},
    {OPT_MAIN, "BIND-CURRENT"},
    {OPT_MAIN, "SET-NIL"},
    {OPT_MAIN, "SET-T"},
    {OPT_MAIN, "SET-ZERO"},
    
    /* 70 - 77 */
    {OPT_BRANCH, "BR-NIL-ELSE-POP"},
    {OPT_BRANCH, "BR-NOT-NIL-ELSE-POP"},
    {OPT_BRANCH, "BR-NIL"},
    {OPT_BRANCH, "BR-NOT-NIL"},
    {OPT_BRANCH, "BR-ATOM"},
    {OPT_BRANCH, "BR-NOT-ATOM"},
    {OPT_BRANCH, "BR-ZEROP"},
    {OPT_BRANCH, "BR-NOT-ZEROP"},

    /* 78 - 7f */
    {OPT_BRANCH, "BR-SYMBOLP"},
    {OPT_BRANCH, "BR-NOT-SYMBOLP"},
    {OPT_BRANCH, "unused"},
    {OPT_BRANCH, "unused"},
    {OPT_BRANCH, "BR-NIL-LIKELY"},
    {OPT_BRANCH, "BR-NOT-NIL-LIKELY"},
    {OPT_BRANCH, "BR"},
    {OPT_BRANCH, "illegal"},
};

void handle_mainop(int address, u16 opcode, struct instr *instr)
{
    int base;
    int offset;

    base = (opcode >> 6) & 7;
    offset = opcode & 0x3f;

    printf("%04x: %04hx %s", address, opcode, instr->name);

    switch (base) {
    case 0:
	printf(" FEF[%d]\n", offset);
	break;
	
    case 1:
	printf(" FEF[%d]\n", offset + 0x40);
	break;
	
    case 2:
	printf(" FEF[%d]\n", offset + 0x80);
	break;
	
    case 5:
	printf(" local[%d]\n", offset);
	break;
	
    case 6:
	printf(" arg[%d]\n", offset);
	break;
	
    case 7:
	printf(" PDL-POP\n");
	break;
	
    default:
	printf(" %d[#o%o]\n", (opcode >> 6) & 3, opcode & 0x3f);
	break;
    }
}

void handle_branchop(int address, u16 opcode, struct instr *instr)
{
    int target;

    target = opcode & 0xff;
    if (opcode & 0x100) target |= 0xffffff00;
    target += address + 1;
    
    printf("%04x: %04hx %s %04x\n", address, opcode, instr->name, target);
}

void handle_immedop(int address, u16 opcode, struct instr *instr)
{
    u16 value;

    value = opcode & 0x1ff;
    
    printf("%04x: %04hx %s %x\n", address, opcode, instr->name, value);
}

void handle_unknownop(int address, u16 opcode, struct instr *instr)
{
    printf("%04x: %04hx %s\n", address, opcode,
	   instr->name? instr->name: "unknown");
}

void handle_miscop(int address, u16 opcode, struct instr *instr)
{
    printf("%04x: %04hx miscop #o%o\n", address, opcode, opcode & 0x1ff);
}

void handle_auxop(int address, u16 opcode, struct instr *instr)
{
    printf("%04x: %04hx auxop #o%o\n", address, opcode, opcode & 0x1ff);
}

void handle_arefiop(int address, u16 opcode, struct instr *instr)
{
    printf("%04x: %04hx arefiop #o%o\n", address, opcode, opcode & 0x1ff);
}

void handle_moduleop(int address, u16 opcode, struct instr *instr)
{
    printf("%04x: %04hx moduleop #o%o:%o\n", address, opcode,
	   (opcode & 0x1f8) >> 3, opcode & 7);
}

void (*instr_handlers[8])(int, u16, struct instr *) = {
    handle_unknownop, /* OPT_UNKNOWN */
    handle_mainop,    /* OPT_MAIN    */
    handle_branchop,  /* OPT_BRANCH  */
    handle_immedop,   /* OPT_IMMED   */
    handle_miscop,    /* OPT_MISC    */
    handle_auxop,     /* OPT_AUX     */
    handle_arefiop,   /* OPT_AREFI   */
    handle_moduleop,  /* OPT_MODULE  */
};

void disassemble_instr(int address, u16 opcode)
{
    struct instr *cur_instr;

    cur_instr = &instructions[opcode >> 9];

    instr_handlers[cur_instr->type](address, opcode, cur_instr);
}

/* EOF */
