/*
 * interp.c
 *
 * macrocode interpreter
 */

#include "exploiter.h"
#include "context.h"
#include "memory.h"
#include "funcall.h"
#include "utils.h"

#include <common/fileslurp.h>

struct context context;

u16 get_opcode(void)
{
    u32 tmp;

    tmp = memread(context.function + (context.location_counter >> 1));

    if (context.location_counter & 1) {
	return tmp >> 16;
    } else {
	return tmp;
    }
}

int step_miscop(u16 opcode)
{
    u16 opnum;
    lisp_q destination;

    opnum = opcode & 0x1ff;

    if (opnum == 077) { /* NOT-INDICATORS */
	destination = (NOT_CDRCODE(context.indicators) == C_NIL)? C_T: C_NIL;
    } else if (opnum == 0240) { /* BOUNDP */
	lisp_q foo;

	foo = memread_inviz(pop() + SYM_VALUE);

	/* If foo is DTP_NULL, return C_T, else C_NIL */
	destination = (DTP(foo) == DTP_NULL)? C_T: C_NIL;
    } else if (opnum == 0613) { /* MEMQ */
	lisp_q foo;
	lisp_q list;
	
	list = pop();
	foo = NOT_CDRCODE(pop());
	
	while((NOT_CDRCODE(list) != (C_NIL)) &&
	      (NOT_CDRCODE(car(list)) != foo)) {
	    list = cdr(list);
	}

	destination = list;
    } else {
	printf("Unknown miscop.\n");
	return 0;
    }

    if (opcode & 0x4000) {
	push_cdrnext(destination);
    } else {
	context.indicators = destination;
    }

    return 1;
}

int step_auxop(u16 opcode)
{
    u16 opnum;

    opnum = opcode & 0x1ff;

    if (opnum == 0136) { /* RETURN-PRED */
	return_1((NOT_CDRCODE(context.indicators) == C_NIL)? C_T: C_NIL);
    } else {
	printf("Unknown auxop.\n");
	return 0;
    }

    return 1;
}

int step(void)
{
    u16 opcode;

    opcode = get_opcode();

    disassemble_instr(context.location_counter, opcode);

    context.location_counter++;

    if ((opcode & 0xfe00) == 0x0000) { /* auxop */
	return step_auxop(opcode);
    } else if ((opcode & 0xbe00) == 0x0200) { /* miscop */
	return step_miscop(opcode);
    } else if (opcode == 0xdbff) { /* SET-NIL PDL */
	push_cdrnext(C_NIL);
    } else if (((opcode & 0xc1c0) == 0x8000) &&
	       ((opcode & 0xf9c0) != 0xb800)) { /* CALL-[0-6]-DEST-* FEF|xx */
	lisp_q function;

	function = memread_inviz(context.function + (opcode & 0x3f));
	funcall(function, DTP_FIX | 0x004000 |
		((opcode >> 11) & 7) | (opcode & 0x600));
    } else if (opcode == 0x1fff) { /* RETURN PDL */
	return_1(pop());
    } else if (opcode == 0x3dff) { /* PLUSP PDL */
	int x;

	x = pop();
	/* FIXME: Raise ARGTYP if not of type Number */
	/* FIXME: Don't assume it's a Fixnum */
	x &= Q_BITS_ADDRESS;

	context.indicators = ((x == 0) | (x & 0x1000000))? C_NIL: C_T;
    } else if ((opcode & 0xfe00) == 0xe000) { /* BR-NIL-ELSE-POP */
	int x;

	x = context.indicators;

	if (NOT_CDRCODE(x) == C_NIL) {
	    if (opcode & 0x100) {
		context.location_counter += (0xffffff00 + (opcode & 0xff));
	    } else {
		context.location_counter += (opcode & 0xff);
	    }
	} else {
	    pop();
	}
    } else if ((opcode & 0xfe00) == 0xe400) { /* BR-NIL */
	int x;

	x = context.indicators;

	if (NOT_CDRCODE(x) == C_NIL) {
	    if (opcode & 0x100) {
		context.location_counter += (0xffffff00 + (opcode & 0xff));
	    } else {
		context.location_counter += (opcode & 0xff);
	    }
	}
    } else if ((opcode & 0xfe00) == 0xe600) { /* BR-NOT-NIL */
	int x;

	x = context.indicators;

	if (NOT_CDRCODE(x) != C_NIL) {
	    if (opcode & 0x100) {
		context.location_counter += (0xffffff00 + (opcode & 0xff));
	    } else {
		context.location_counter += (opcode & 0xff);
	    }
	}
    } else if ((opcode & 0xffc0) == 0x5000) { /* PUSH FEF|xx */
	push_cdrnext(memread_inviz(context.function + (opcode & 0x3f)));
    } else if ((opcode & 0xffc0) == 0x5140) { /* PUSH local|xx */
	push_cdrnext(memread_inviz(context.local_pointer + (opcode & 0x3f)));
    } else if ((opcode & 0xffc0) == 0x5180) { /* PUSH arg|xx */
	push_cdrnext(memread_inviz(context.arg_pointer + (opcode & 0x3f)));
    } else if ((opcode & 0xffc0) == 0xc000) { /* POP FEF|xx */
	memwrite_inviz(context.function + (opcode & 0x3f), pop());
    } else if ((opcode & 0xffc0) == 0xc140) { /* POP local|xx */
	memwrite(context.local_pointer + (opcode & 0x3f), pop());
    } else if ((opcode & 0xffc0) == 0xc180) { /* POP arg|xx */
	memwrite(context.arg_pointer + (opcode & 0x3f), pop());
    } else if ((opcode & 0xffc0) == 0xdc00) { /* SET-T FEF|xx */
	memwrite_inviz(context.function + (opcode & 0x3f), C_T);
    } else if ((opcode & 0xffc0) == 0xda00) { /* SET-NIL FEF|xx */
	memwrite_inviz(context.function + (opcode & 0x3f), C_NIL);
    } else if ((opcode & 0xffc0) == 0x1180) { /* TEST arg|xx */
	context.indicators = memread_inviz(context.arg_pointer + (opcode & 0x3f));
    } else if ((opcode & 0xffc0) == 0x2600) { /* EQ FEF|xx */
	lisp_q foo;
	lisp_q bar;

	foo = NOT_CDRCODE(pop());
	bar = NOT_CDRCODE(memread_inviz(context.function + (opcode & 0x3f)));
	context.indicators = (foo == bar)? C_T: C_NIL;
    } else if ((opcode & 0xffc0) == 0x4a00) { /* PUSH-NUMBER */
	lisp_q foo;

	foo = opcode & 0xff;
	if (opcode & 0x100) foo |= 0x01ffff00;

	foo |= DTP_FIX;
	
	push_cdrnext(foo);
    } else if ((opcode & 0xffc0) == 0x6000) { /* + FEF|xx */
	lisp_q src1;
	lisp_q src2;
	lisp_q result;

	src1 = pop();
	src2 = memread_inviz(context.function + (opcode & 0x3f));

	if ((DTP(src1) != DTP_FIX) || (DTP(src2) != DTP_FIX)) {
	    printf("One or both args to + is not DTP_FIX, failing.\n");
	    dump_q(src1, 1);
	    dump_q(src2, 2);
	    return 0;
	}

	/* get fixnum values and sign extend */
	src1 = ADDRESS(src1);
	src2 = ADDRESS(src2);
	if (src1 & 0x01000000) src1 |= 0xfe000000;
	if (src2 & 0x01000000) src2 |= 0xfe000000;

	result = src1 + src2;
	
	if ((result > 0x00ffffff) && (result < 0xff000000)) {
	    printf("Result from + overflows FIXNUM.\n");
	    return 0;
	}

	result = DTP_FIX | ADDRESS(result);
	push_cdrnext(result);
    } else {
	printf("Unknown opcode.\n");
	return 0;
    }

    return 1;
}

void load_sg(lisp_q sg)
{
    context.current_sg = sg;
    context.pdl = memread(sg - 3);
    context.pdl_pointer = context.pdl + 3;
}

int process_data(u8 *data, int length)
{
    lisp_q spia; /* Scratch Pad Init Area */

    memory_init((lisp_q *)data);

    spia = get_area_address(2);

    printf("scratch pad init area:\n");
    dump_q(memread(spia), spia);
    dump_q(memread(spia+1), spia+1);
    dump_q(memread(spia+2), spia+2);
    dump_q(memread(spia+3), spia+3);
    dump_q(memread(spia+4), spia+4);

#if 0
    printf("initial stack group:\n");
/*     dump_raw_array(spia[3]); */
    dump_q(memread(spia[3]-3), spia[3]-3);
    dump_q(memread(memread(spia[3]-3)), memread(spia[3]-3));
    dump_q(memread(memread(spia[3]-3)+1), memread(spia[3]-3)+1);
    dump_q(memread(memread(spia[3]-3)+2), memread(spia[3]-3)+2);
    dump_q(memread(memread(spia[3]-3)+3), memread(spia[3]-3)+3);
    dump_q(memread(memread(spia[3]-3)+4), memread(spia[3]-3)+4);
#endif

    /*
     * Load the static part of the initial stack group, set the stack
     * pointers to 0, and fake a function call to the initial function.
     * The call-info word for the function call ends up as the first
     * thing on the stack.
     *
     * The call-info word will be 0x4a001000. (no, we need to set D-MICRO)
     * Just as a quick hack, we will push 4 NILs after the call-info
     * word and fake up whatever is nessecary for the initial function.
     *
     * The call process seems to be a matter of:
     *
     *     find the call target
     *     fixup for outgoing function
     *     fixup arglist
     *     save outgoing state
     *     load incoming state
     *     fixup local variables
     */

    load_sg(memread(spia + SPIA_INITIAL_SG));

    funcall(memread(memread(spia + SPIA_INITIAL_FUNCTION)), DTP_FIX | 0x001000); /* FIXME: wrong dest */

    while(step());
    
#if 0
    printf("initial function:\n");
    dump_q(memread(spia[0]), spia[0]);
    dump_raw_fef(memread(spia[0]));
#endif

    return 0;
}

/* EOF */
