/*
 * memory.c
 *
 * memory load and access functions
 */

#include "memory.h"

#include "exploiter.h"
#include "context.h"
#include "funcall.h"
#include "utils.h"

#include <stdio.h>

lisp_q *image;
lisp_q *sca;
lisp_q *area_origin_area;
lisp_q *dpmt;

int num_wired_words;

#define CLUSTER_NUMBER(q) (((q) >> 13) & 0xfff)
#define PAGE_NUMBER(q) (((q) >> 9) & 0xf)

#if 0 /* VMM emulation not complete yet */

/* VMM status has 128K 2-bit entries. That's 256 bits. That's 32k bytes. */

u8 vmm_status[0x8000];

#define VMS_VALID 2
#define VMS_BANK  1

#define VMSB_LEFT  0
#define VMSB_RIGHT 1

int get_vmm_status(int index)
{
    u8 vmm_status_byte;

    vmm_status_byte = vmm_status[index >> 2];
    return (vmm_status_byte >> ((index & 3) << 1)) & 3;
}

void set_vmm_status(int index, int value)
{
    u8 vmm_status_byte;

    vmm_status_byte = vmm_status[index >> 2];

    vmm_status_byte &= ~(3 << ((index & 3) << 1));
    vmm_status_byte |= value << ((index & 3) << 1);

    vmm_status[index >> 2] = vmm_status_byte;
}

/* the VMM is two 16k entry 32bit maps */
u32 vmm_left_bank[0x4000];
u32 vmm_right_bank[0x4000];
u32 const *vmm_banks[2] = { vmm_left_bank, vmm_right_bank };

#define VMME_ACCESS 0x80000000

u32 mapped_read(lisp_q address)
{
    int vmm_status;
    u32 vmm_entry;
    u32 physaddr;
    
    address = ADDRESS(address);

    vmm_status = get_vmm_status((address >> 8) & 0x1ffff);

    if (!(vmm_status & VMS_VALID)) {
	/* FIXME: map miss page exception */
    }

    vmm_entry = vmm_banks[vmm_status & VMS_BANK][(address >> 8) & 0x3fff];

    if (!(vmm_entry & VMME_ACCESS)) {
	/* FIXME: page exception */
    }

    physaddr = vmm_entry << 10;
    physaddr |= (address & 0xff) << 2;

    return phys_read_u32(physaddr);
}
#endif /* VMM emulation */

u32 physical_memory_map[8] = {
    /*
     * We pretend that the first memory board found was in slot 4 and that
     * it has been set up with 1 Quantum (2 megs) of memory. All other boards
     * in the map are empty.
     */
    0xf4000001, -1, -1, -1, -1, -1, -1, -1
};

lisp_q read_amem(int address)
{
    if (address == 0x52) { /* %COUNTER-BLOCK-A-MEM-ADDRESS */
	printf("AMEM: read %%COUNTER-BLOCK-A-MEM-ADDRESS.\n");
	return DTP_FIX | 0x100; /* FIXME: where is it really? */
    } else if (address == 0x56) { /* inhibit scheduling flag */
	printf("AMEM: read inhibit scheduling flag.\n");
	return 0;
    } else if (address == 0x65) { /* alphabetic case affects string comparison */
	printf("AMEM: read alphabetic case affects string comparison.\n");
	return 0;
    } else if (address == 0x8a) { /* microcode type */
	printf("AMEM: read microcode type.\n");
	return 0;
    } else if (address == 0x8b) { /* processor type */
	printf("AMEM: read processor type.\n");
	return 0;
    } else if ((address >= 0x198) && (address < 0x200)) {
	printf("AMEM: read PMM[%d].\n", address & 7);
	return physical_memory_map[address & 7];
    } else {
	printf("AMEM: 0x%03x.\n", address);
    	return 0;
    }
}

void write_amem(int address, lisp_q data)
{
    printf("AMEM: write 0x%03x = 0x%08lx.\n", address, data);
}

lisp_q memread_unboxed(lisp_q address)
{
    lisp_q *dpmt_entry;
    
    address &= Q_BITS_ADDRESS;

    if (address < num_wired_words) {
	return image[address];
    } else if (address >= 0x01fffc00) {
	return read_amem(address & 0x3ff);
    } else if (address >= 0x01fdfc00) {
	printf("IO: 0x%05lx.\n", (address + 0x400) & 0x1ffff);
	return 0;
    }

    dpmt_entry = &dpmt[CLUSTER_NUMBER(address) << 1];
/*     printf("DPMT: %08lx %08lx.\n", dpmt_entry[0], dpmt_entry[1]); */

    return image[0x2000 * (dpmt_entry[1] & 0xffff) + (address & 0x1fff)];
}

lisp_q memread(lisp_q address)
{
    /* FIXME: Read barrier goes here */
    return memread_unboxed(address);
}

void memwrite_unboxed(lisp_q address, lisp_q data)
{
    lisp_q *dpmt_entry;
    
    address &= Q_BITS_ADDRESS;

    if (address < num_wired_words) {
	image[address] = data;
	return;
    } else if (address >= 0x01fffc00) {
	write_amem(address & 0x3ff, data);
	return;
    } else if (address >= 0x01fdfc00) {
	printf("IO: 0x%05lx = 0x%08lx.\n", (address + 0x400) & 0x1ffff, data);
	return;
    }

    dpmt_entry = &dpmt[CLUSTER_NUMBER(address) << 1];

    image[0x2000 * (dpmt_entry[1] & 0xffff) + (address & 0x1fff)] = data;
}

void memwrite(lisp_q address, lisp_q data)
{
    /* FIXME: Write barrier goes here */
    memwrite_unboxed(address, data);
}

lisp_q inviz(lisp_q address, lisp_q *final_address)
{
    /*
     * Oh, for multiple-value-bind...
     */
    lisp_q foo;

    foo = memread(address);
    
    if (DTP(foo) == DTP_EVCP) {
	printf("INVIZ: EVCP 0x%08lx.\n", foo);
    } else if (DTP(foo) == DTP_ONE_Q_FORWARD) {
	printf("INVIZ: 1QF 0x%08lx.\n", foo);
    } else {
	if (final_address) *final_address = address;
	return foo;
    }

    return inviz(foo, final_address);
}

lisp_q memread_inviz(lisp_q address)
{
    return inviz(address, NULL);
}

void memwrite_inviz(lisp_q address, lisp_q data)
{
    lisp_q tmp;
    lisp_q tmpaddr;

    tmp = inviz(address, &tmpaddr);
    memwrite(tmpaddr, CDRCODE(tmp) | NOT_CDRCODE(data));
}

lisp_q return_barrier(lisp_q data)
{
    /* FIXME: Implement */
    return data;
}

void push(lisp_q foo)
{
    memwrite(context.pdl_pointer++, foo);
}

void push_cdrnext(lisp_q foo)
{
    push(foo | CDR_NEXT);
}

lisp_q pop(void)
{
    return memread(--context.pdl_pointer);
}

lisp_q car(lisp_q cons)
{
    lisp_q foo;
    lisp_q fooaddr;

    foo = inviz(cons, &fooaddr);
    
    if (NOT_CDRCODE(fooaddr) == C_NIL) return C_NIL;
    
    return foo;
}

lisp_q cdr(lisp_q cons)
{
    lisp_q foo;

    /*
     * Okay, we're passed a DTP_LIST, a DTP_STACK_LIST, or a DTP_SYMBOL.
     * If we're a symbol, it had best be NIL, for which we are NIL. Next,
     * we have to find the CDR of the list, which means checking a CDR
     * code. Is this the CDR code on the forwarding pointer? It is, isn't
     * it? We then have to clamp out any forwarding pointers on it.
     */

    if (NOT_CDRCODE(cons) == C_NIL) return C_NIL;

    foo = memread(cons);
    
    switch (CDRCODE(foo)) {
    case CDR_NIL:
	return C_NIL;

    case CDR_NEXT:
	return cons + 1;

    case CDR_NORMAL:
	return memread_inviz(cons + 1);

    case CDR_ERROR:
	printf("CDR of CDR-ERROR taken.\n");
	exit(-1);

    default:
	exit(-1); /* GCC can't see that this can never be reached. :-( */
    }
}

void spdl_push(lisp_q data)
{
    memwrite(context.spdl + context.spdl_pointer++, data);
}

lisp_q spdl_pop(void)
{
    return memread(context.spdl + --context.spdl_pointer);
}

void bind(lisp_q where, lisp_q data)
{
    if (DTP(where) != DTP_LOCATIVE) {
	printf("bind(): where not DTP_LOCATIVE.\n");
	exit(-1);
    }

    where = NOT_CDRCODE(where);
    if (!(context.call_info & CI_BINDINGBLOCK)) {
	where |= SPECPDL_BLOCK_START_FLAG;
	context.call_info |= CI_BINDINGBLOCK;
    }
    
    spdl_push(memread(where));
    spdl_push(where);

    memwrite(where, data);
}

void unbind_1(void)
{
    lisp_q where;
    lisp_q data;

    /* FIXME: What about interactions with catch and unwind-protect? */
    
    where = spdl_pop();
    data = spdl_pop();

    if (where & SPECPDL_BLOCK_START_FLAG) {
	context.call_info &= ~CI_BINDINGBLOCK;
    }

    memwrite(where, data);
}

void unbind_block(void)
{
    while (context.call_info & CI_BINDINGBLOCK) {
	unbind_1();
    }
}

void unbind_n(int n)
{
    while(n--) {
	unbind_1();
    }
}

lisp_q get_area_address(int area)
{
    return area_origin_area[area];
}

void memory_init(lisp_q *data)
{
    image = data;
    sca = image + 01000;
    area_origin_area = image + ADDRESS(sca[0]);
    dpmt = image + ADDRESS(area_origin_area[9]);
    num_wired_words = ADDRESS(sca[13]);

    sca[12] = DTP_FIX | 0x80000; /* pretend we have 2 megs of memory */
    sca[28] = DTP_FIX | 0xf1; /* pretend processor is in NuBus slot 1 */
    sca[32] = DTP_FIX | 0x1fffd98; /* pretend the PMM is at AMEM:0x298 */
}

/* EOF */
