#include <iomanip.h>
#include <iostream.h>
#include <unistd.h>
#include <stdlib.h>		// for abort
#include <ctype.h>

#include "word.h"

void e3Word::setDataType(e3DataType t)
{
  myBits &= ~E3_DTP_ONES_TRAP;
  myBits |= t;
}

e3u32 e3Word::getBits(void) const
{
#ifdef E3_LITTLE_ENDIAN_PLATFORM

  return(myBits);

#else

  e3u32 big_endian_bits
    = ((myBits & 0x000000ff) << 24)
    | ((myBits & 0x0000ff00) << 8)
    | ((myBits & 0x00ff0000) >> 8)
    | ((myBits & 0xff000000) >> 24);

  return(big_endian_bits);

#endif
}

void e3Word::setBits(e3u32 new_bits)
{
#ifdef E3_LITTLE_ENDIAN_PLATFORM

  myBits = new_bits;

#else

  e3u32 little_endian_bits
    = ((new_bits & 0x000000ff) << 24)
    | ((new_bits & 0x0000ff00) << 8)
    | ((new_bits & 0x00ff0000) >> 8)
    | ((new_bits & 0xff000000) >> 24);

  myBits = little_endian_bits;

#endif
}

e3u32 e3Word::selectBitsMask(e3u32 mask) const
{
  e3u32 my_bits = getBits();
  e3u32 result = my_bits & mask;
  return(result);
}
  

e3u32 e3Word::selectBitsPPSS(e3u32 position, e3u32 size) const
{
#ifdef PARANOID
  if (size >= 32)
    {
      cerr << "e3Word::selectBitsPPSS -- size too big for word size " << size << endl;
      abort();
      return(0);
    }
  else if (position >= 32)
    {
      cerr << "e3Word::selectBitsPPSS -- size too big for word size " << size << endl;
      abort();
      return(0);
    }
  else
#endif
    {
      /* Construct mask of correct width. */

      e3u32 mask = 0;

      for (e3u32 i=0; i<size; i++)
	{
	  mask = (mask << 1) | 1;
	}

      /* Shift mask to position. */

      mask = mask << position;
#ifdef DEBUG
      printf("e3Word::selectBitsPPSS(%ld pos, %ld size) -- mask=0x%08X\n", position, size, mask); 
#endif

      /* Extract the bits. */

      e3u32 bits = selectBitsMask(mask);

      /* Shift back down to low bits. */

      bits = bits >> position;
      return(bits);
    }
}



e3CdrCode e3Word::getCdrCode(void) const
{
  e3u8 cdr_code = selectBitsPPSS(30, 2);

  return((e3CdrCode)cdr_code);
}

e3DataType e3Word::getDataType(void) const
{
  /* lroy-qcom.lisp says byte specifier #o3105. */
  e3u8 data_type = selectBitsPPSS(031, 5);

  return((e3DataType)data_type);
};

e3WordAddr e3Word::getPointer(void) const
{
  /* lroy-qcom.lisp says byte specifier #o0031. */

  e3WordAddr a = (getBits() & 0x01ffffff); // 25 bits, starting at bit 0.

  return(a);
}

void e3Word::clear(void)
{
  setBits(0);
}

/*
 * Type coercion.
 */

/*
 * 
 */

e3Boolean e3Word::isInvisibleForwardingPointer(void) const
{
  switch(getDataType())
    {
    case E3_DTP_TRAP:
      {
	abort();
	return(e3False);
      }
    case E3_DTP_LIST:
    case E3_DTP_STACK_LIST:
    case E3_DTP_SYMBOL:
    case E3_DTP_ARRAY:
    case E3_DTP_FIX:
    case E3_DTP_CHARACTER:
    case E3_DTP_SINGLE_FLOAT:
    case E3_DTP_SHORT_FLOAT:
    case E3_DTP_INSTANCE:
    case E3_DTP_EXTENDED_NUMBER:
    case E3_DTP_LOCATIVE:
    case E3_DTP_FUNCTION:
    case E3_DTP_CLOSURE:
    case E3_DTP_LEXICAL_CLOSURE:
    case E3_DTP_U_ENTRY:
    case E3_DTP_STACK_GROUP:
      return(e3False);

    case E3_DTP_GC_FORWARD:
    case E3_DTP_EXTERNAL_VALUE_CELL_POINTER:
    case E3_DTP_ONE_Q_FORWARD:
    case E3_DTP_HEADER_FORWARD:
    case E3_DTP_BODY_FORWARD:
    case E3_DTP_SYMBOL_HEADER:
      return(e3True);

    case E3_DTP_HEADER:
    case E3_DTP_ARRAY_HEADER:
    case E3_DTP_INSTANCE_HEADER:
    case E3_DTP_FEF_HEADER:
    case E3_DTP_SELF_REF_POINTER: // ??????
    case E3_DTP_GC_YOUNG_POINTER:
    case E3_DTP_FREE:
    case E3_DTP_NULL:
      return(e3False);


    case E3_DTP_ONES_TRAP:
    default:
      {
	abort();
	return(e3False);
      }
    }
}


/*
 * 
 */

void e3Word::dump(ostream &os)
{
  os.fill('0');
  os.width(8);
  os << hex << getBits() << " = ";

  /*
   * Print out CDR-CODE in human readable form.
   */

  e3CdrCode cdr_code = getCdrCode();
  os << "CDR_CODE=";
  switch(cdr_code)
    {
    case E3_CDR_CODE_NORMAL:
      os << "NORMAL";
      break;
    case E3_CDR_CODE_ERROR:
      os << "ERROR ";
      break;
    case E3_CDR_CODE_NIL:
      os << "NIL   ";
      break;
    case E3_CDR_CODE_NEXT:
      os << "NEXT  ";
      break;
    default:
      os << "??????";
      break;
    }
  os  << ";";

  /*
   * Print out DTP in human readable form.
   */

  e3DataType tagonly = getDataType();
  os << "DTP=";
  switch(tagonly)
    {
    case E3_DTP_TRAP:
      os << "TRAP            ";
      break;
    case E3_DTP_LIST:
      os << "LIST            ";
      break;
    case E3_DTP_STACK_LIST:
      os << "STACK_LIST      ";
      break;
    case E3_DTP_SYMBOL:
      os << "SYMBOL          ";
      break;
    case E3_DTP_ARRAY:
      os << "ARRAY           ";
      break;
    case E3_DTP_FIX:
      os << "FIX             ";
      break;
    case E3_DTP_CHARACTER:
      os << "CHARACTER       ";
      break;
    case E3_DTP_SINGLE_FLOAT:
      os << "SINGLE_FLOAT    ";
      break;
    case E3_DTP_SHORT_FLOAT:
      os << "SHORT_FLOAT     ";
      break;
    case E3_DTP_INSTANCE:
      os << "INSTANCE        ";
      break;
    case E3_DTP_EXTENDED_NUMBER:
      os << "EXTENDED_NUMBER ";
      break;
    case E3_DTP_LOCATIVE:
      os << "LOCATIVE        ";
      break;
    case E3_DTP_FUNCTION:
      os << "FUNCTION        ";
      break;
    case E3_DTP_CLOSURE:
      os << "CLOSURE         ";
      break;
    case E3_DTP_LEXICAL_CLOSURE:
      os << "LEXICAL_CLOSURE ";
      break;
    case E3_DTP_U_ENTRY:
      os << "U_ENTRY         ";
      break;
    case E3_DTP_STACK_GROUP:
      os << "STACK_GROUP     ";
      break;
    case E3_DTP_GC_FORWARD:
      os << "GC_FORWARD      ";
      break;
    case E3_DTP_EXTERNAL_VALUE_CELL_POINTER:
      os << "EVCP            ";
      break;
    case E3_DTP_ONE_Q_FORWARD:
      os << "ONE_Q_FORWARD   ";
      break;
    case E3_DTP_HEADER_FORWARD:
      os << "HEADER_FORWARD  ";
      break;
    case E3_DTP_BODY_FORWARD:
      os << "BODY_FORWARD    ";
      break;
    case E3_DTP_SYMBOL_HEADER:
      os << "SYMBOL_HEADER   ";
      break;
    case E3_DTP_HEADER:
      os << "HEADER          ";
      break;
    case E3_DTP_ARRAY_HEADER:
      os << "ARRAY_HEADER    ";
      return;
    case E3_DTP_INSTANCE_HEADER:
      os << "INSTANCE_HEADER ";
      break;
    case E3_DTP_FEF_HEADER:
      os << "FEF_HEADER      ";
      break;
    case E3_DTP_SELF_REF_POINTER:
      os << "SELF_REF_POINTER";
      break;
    case E3_DTP_GC_YOUNG_POINTER:
      os << "GC_YOUNG_POINTER";
      break;
    case E3_DTP_FREE:
      os << "FREE            ";
      break;
    case E3_DTP_NULL:
      os << "NULL            ";
      return;

    case E3_DTP_ONES_TRAP:
      os << "ONES_TRAP       ";
      break;
    default:
      os << "????????????????";
    }

  /*
   * Print out POINTER.
   */

  os << ";POINTER=" << getPointer();

  /* Try and dump chars. */
  {
    /*
     * I used this to figure out the right byte order.
     * If byte order OK, get readable strings at end of MCR band.
     */

    char b3 = (getBits() >> 24) & 0xff;
    char b2 = (getBits() >> 16) & 0xff;
    char b1 = (getBits() >> 8) & 0xff;
    char b0 = getBits() & 0xff;

    if ((isprint((int)b3) || (b3 == 0))
	&& (isprint((int)b2) || (b2 == 0))
	&& (isprint((int)b1) || (b1 == 0))
	&& (isprint((int)b0) || (b0 == 0)))
      {
	os << " \""  << b3 << b2 << b1 << b0 << "\"";
      };
  }
}


#undef BYTE_AT_A_TIME

ifstream& operator >> (register ifstream& is, register e3Word &w)
{
#ifdef BYTE_AT_A_TIME
  e3u8 b3;
  e3u8 b2;
  e3u8 b1;
  e3u8 b0;

  is.get(b0);
  is.get(b1);
  is.get(b2);
  is.get(b3);

#ifdef NOTDEF
  printf("%02x%02x%02x%02x\n", b3, b2, b1, b0);
#endif

  //  w.myBits = (b3 << 24) | (b2 << 16) | (b1 << 8) | b0;
  setBits((b3 << 24) | (b2 << 16) | (b1 << 8) | b0);

#else

  /* $$$$$$$$$$$ THIS MIGHT NOT BE PORTABLE!!!!!!!!!! */

  is.read((char *)&(w.myBits), sizeof(w.myBits));

  //  printf("0x%08X\n", w.myBits);

#endif
  
  return (is);
}
	      

ostream& operator << (ostream &os, e3Word w)
{
  w.dump(os);
  return(os);
}


e3Word e3Word::allOnesTrap(void)
{
  e3Word retval;

  retval.clear();
  retval.setDataType(E3_DTP_ONES_TRAP);
  return(retval);
}


/*
 * Should probably throw or something.
 */

void e3Word::abortIfNot(e3DataType t, char *diag) const
{
  e3DataType tag = getDataType();

  if (tag != t)
    {
      abort();			// $$$$$$$ Should throw into Lisp?
      cerr << diag << endl;
    }
}


//
// Local Variables:
// compile-command: "make cycle"
// fill-column: 78
// End:
