/*
 * miscop.c
 *
 * miscop instruction handlers
 */

#include "exploiter.h"
#include "context.h"
#include "memory.h"
#include "interp.h"
#include "utils.h"
#include "nubus.h"
#include <stdio.h>

#include <common/hexdump.h>

#define ASSERT_DTP(q, dtp) if (DTP(q) != (dtp)) { printf("DTP assertion failure.\n"); return 0; }

static lisp_q destination; /* FIXME: This is more than a little ugly */

typedef int (*miscop_handler)(u16 opcode, u16 opnum);

#define MISCOP(foo) int miscop_##foo(u16 opcode, u16 opnum)

MISCOP(002) { /* CAR */
    lisp_q list;

    list = pop();

    /* FIXME: Typecheck */

    destination = car(list);

    return 1;
}

MISCOP(003) { /* CDR */
    lisp_q list;

    list = pop();

    /* FIXME: Typecheck */

    destination = cdr(list);

    return 1;
}

MISCOP(005) { /* CADR */
    lisp_q list;

    list = pop();

    /* FIXME: Typecheck */

    destination = car(cdr(list));
    
    return 1;
}

MISCOP(013) { /* CADDR */
    lisp_q list;

    list = pop();

    /* FIXME: Typecheck */

    destination = car(cdr(cdr(list)));

    return 1;
}

MISCOP(041) { /* SHRINK-PDL-SAVE-TOP */
    lisp_q value_to_move;
    lisp_q n_slots;

    n_slots = pop();
    value_to_move = pop();

    context.pdl_pointer -= ADDRESS(n_slots);
    push_cdrnext(value_to_move);

    return 1;
}

MISCOP(050) { /* %P-DPB */
    lisp_q value;
    lisp_q ppss;
    lisp_q pointer;

    u32 dest;

    pointer = pop();
    ppss = pop();
    value = pop();

    dump_q(value, 0);
    dump_q(ppss, 1);
    dump_q(pointer, 2);

    printf("%%P-DPB: Not implemented.\n");

    dest = memread_unboxed(pointer);
    dump_q(dest, 3);
    dest &= ~(ppss_mask(ppss) << ppss_position(ppss));
    dest |= (value & ppss_mask(ppss)) << ppss_position(ppss);
    dump_q(dest, 4);
    
    memwrite_unboxed(pointer, dest);
    
    return 1;
}

MISCOP(054) { /* %PHYS-LOGLDB */
    lisp_q ppss;
    lisp_q slot;
    lisp_q offset;
    u32 data;
    nubus_result result;

    offset = pop();
    slot = pop();
    ppss = pop();

    dump_q(ppss, 0);
    dump_q(slot, 1);
    dump_q(offset, 2);

    result = nubus_read32(ADDRESS(slot), ADDRESS(offset), &data);

    destination = DTP_FIX | ((data >> ppss_position(ppss)) & ppss_mask(ppss));
    
    return result == NR_SUCCESS;
}

MISCOP(055) { /* %PHYS-LOGDPB */
    lisp_q val;
    lisp_q ppss;
    lisp_q slot;
    lisp_q offset;

    u32 real_data;
    nubus_result result;

    offset = pop();
    slot = pop();
    ppss = pop();
    val = pop();

    dump_q(val, 0);
    dump_q(ppss, 1);
    dump_q(slot, 2);
    dump_q(offset, 3);

    slot = ADDRESS(slot);
    offset = ADDRESS(offset);

    result = nubus_read32(slot, offset, &real_data);
    if (result != NR_SUCCESS) {
	printf("%%PHYS-LOGDPB: NuBus failure during read.\n");
	return 0;
    }

    real_data &= ~(ppss_mask(ppss) << ppss_position(ppss));
    real_data |= (val & ppss_mask(ppss)) << ppss_position(ppss);

    result = nubus_write32(slot, offset, real_data);
    if (result != NR_SUCCESS) {
	printf("%%PHYS-LOGDPB: NuBus failure during write.\n");
	return 0;
    }
    
    return 1;
}

MISCOP(057) { /* %GC-SCAV-RESET */
    lisp_q region = pop();

    dump_q(region, 0);

    destination = C_NIL;

    return 1;
}

MISCOP(062) { /* %P-STORE-CONTENTS */
    lisp_q pointer;
    lisp_q value;
    lisp_q oldvalue;

    value = pop();
    pointer = pop();

    dump_q(pointer, 0);
    dump_q(value, 1);

    oldvalue = memread(pointer);
    memwrite(pointer, CDRCODE(oldvalue) | NOT_CDRCODE(value));

    return 1;
}

MISCOP(063) { /* %P-STORE-TAG-AND-POINTER */
    lisp_q pointer;
    lisp_q misc_fields;
    lisp_q pointer_field;

    pointer_field = pop();
    misc_fields = pop();
    pointer = pop();

    dump_q(pointer, 0);
    dump_q(misc_fields, 1);
    dump_q(pointer_field, 2);

    destination = ADDRESS(pointer_field) | ((misc_fields & 0x7f) << 25);
    dump_q(destination, 3);

    memwrite(pointer, destination);

    return 1;
}

MISCOP(067) { /* %P-STORE-CONTENTS-OFFSET */
    lisp_q value;
    lisp_q base_pointer;
    lisp_q offset;

    offset = pop();
    base_pointer = pop();
    value = pop();

    dump_q(value, 0);
    dump_q(base_pointer, 1);
    dump_q(offset, 2);

    memwrite(offset + base_pointer, value);
    
    return 1;
}

MISCOP(070) { /* INTERNAL-GET-2 */
    lisp_q sym;
    lisp_q prop;
    lisp_q plist;

    prop = pop();
    sym = pop();

    dump_q(sym, 0);
    dump_q(prop, 1);

    if (DTP(sym) != DTP_SYMBOL) {
	printf("INTERNAL-GET-2: not SYMBOL.\n");
	return 0;
    }

    plist = memread(sym + SYM_PROPERTY);
    dump_q(plist, 44);
/*     print_list(plist); */
    
    destination = C_NIL;
    if (DTP(plist) != DTP_NULL) {
	while(NOT_CDRCODE(plist) != (C_NIL)) {
	    if (NOT_CDRCODE(car(plist)) == prop) {
		destination = car(cdr(plist));
	    }
	    plist = cdr(cdr(plist));
	}
    }

    return 1;
}

MISCOP(076) { /* PREDICATE */
    destination = (NOT_CDRCODE(context.indicators) == C_NIL)? C_NIL: C_T;

    return 1;
}

MISCOP(077) { /* NOT-INDICATORS */
    destination = (NOT_CDRCODE(context.indicators) == C_NIL)? C_T: C_NIL;

    return 1;
}

MISCOP(200) { /* BIND */
    lisp_q loc;
    lisp_q val;

    val = pop();
    loc = pop();

    dump_q(loc, 0);
    dump_q(val, 1);

    bind(loc, val);
    
    return 1;
}

void nupi_io_command(lisp_q rqb, lisp_q real_device_desc)
{
    u32 command_word;
    lisp_q rqb_base;

    rqb_base = rqb + 0x1f1;
/*     rqb_base += !!ARY_LL(memread(rqb)); */
    if (ARY_LL(memread(rqb))) {
	printf("LL.\n");
	dump_q(memread(rqb + 1), 0);
/* 	exit(-1); */
    }

    {
	lisp_q buf;
/* 	lisp_q disp; */

	buf = memread(rqb - 4);
/* 	disp = memread(buf); */
	dump_q(buf, 0);
	dump_raw_array(buf);
    }
    
    command_word = memread(rqb + 4);
    dump_q(command_word, 0);
    if ((command_word >> 24) == 0x82) {
	int i;
	
	printf("%%nupi-command-rq-nupi-status.\n");

/* 	buffer = memread(rqb + 7); */
/* 	dump_q(rqb, 0); */

	/* Cheating like mad: Pretend all the formatters are offline */
	for (i = 0; i < 7; i++) {
/* 	    printf("%08lx.\n", rqb + 0x1f3 + i); */
	    memwrite(rqb + 0x1f3 + i, 0x10000000);
	}
	/* Cheating like mad: Except the first one */
	memwrite(rqb + 0x1f3, 0);

	/* And pretend that the first drive is online, and a disk */
	memwrite(rqb + 0x1f3 + 7, 0x40000000);

	/* But not the second drive */
	memwrite(rqb + 0x1f3 + 7 + 1, 0x10000000);
/* 	exit(-1); */
    } else if ((command_word >> 24) == 0x12) {
	u32 i;
	u32 transfer_length;
	u32 device_address;
	u32 *sectorbase;

	transfer_length = memread(rqb+7);
	device_address = memread(rqb+8);
	
	printf("%%nupi-command-read.\n");
	printf("transfer length: %08lx.\n", transfer_length);
	printf("device address: %08lx.\n", device_address);

	if (((device_address + 1) * 1024) > diskimage.size) {
	    printf("disk block %ld beyond end of image %ld.\n",
		   device_address, diskimage.size >> 10);
	    exit(-1);
	}

	sectorbase = (u32 *)(diskimage.data + (device_address * 1024));

	transfer_length >>= 2;
	for (i = 0; i < transfer_length; i++) {
	    u32 data;

	    data = sectorbase[i];

	    memwrite(rqb_base + i, data);
	}

	hexdump((u8 *)sectorbase, 0, 256);
/* 	exit(-1); */
    } else {
	printf("Unknown request.\n");
	exit(-1);
    }

    dump_q(memread(rqb + 5), 42);

    memwrite(rqb + 5, 0x40000000); /* RQB status */
}

MISCOP(210) { /* %IO */
    lisp_q rqb;
    lisp_q device_desc;
    lisp_q real_device_desc;
    u32 information_word;
    int interrupt_level;

    device_desc = pop();
    rqb = pop();

    dump_q(rqb, 0);
    dump_q(device_desc, 1);

    dump_raw_array(device_desc);

    /* FIXME: Cheap hacks start here. */
    real_device_desc = memread(device_desc + 1);

    information_word = memread_unboxed(real_device_desc + 1);
    dump_q(information_word, 22);

    interrupt_level = information_word & 7;

    if (interrupt_level == 1) {
	printf("NuPI device descriptor.\n");

	nupi_io_command(rqb, real_device_desc);
    } else if (interrupt_level == 3) {
	printf("*keyboard*.\n");
	dump_q(memread(0x2c04), -1);
	if (NOT_CDRCODE(rqb) == (DTP_FIX | 2)) {
	    printf("%%KBD-Get-Char.\n");
	} else {
	    printf("Unknown request.\n");
	}
	/* FIXME: Return values? We know that keyboard %IO returns them */
    } else {
	printf("Unrecognized device interrupt level %d.\n", interrupt_level);
	return 0;
    }
    
    destination = DTP_FIX;

    return 1;
}

MISCOP(211) { /* %ADD-INTERRUPT */
    lisp_q device_desc;
    lisp_q level;

    level = pop();
    device_desc = pop();

    dump_q(device_desc, 0);
    dump_q(level, 1);

    printf("%%ADD-INTERRUPT: Unimplemented, ignoring.\n");
    
    return 1;
}

MISCOP(217) { /* %NUBUS-WRITE */
    lisp_q slot;
    lisp_q offset;
    lisp_q data;
    u32 real_data;
    nubus_result result;

    data = pop();
    offset = pop();
    slot = pop();

    real_data = positive_integer_value(data);

    if ((DTP(slot) != DTP_FIX) || (DTP(offset) != DTP_FIX)) {
	printf("%%NUBUS-WRITE: slot or offset not DTP_FIX.\n");
	dump_q(slot, 0);
	dump_q(offset, 1);
	return 0;
    }

    result = nubus_write32(ADDRESS(slot), ADDRESS(offset), real_data);
    
    return result == NR_SUCCESS;
}

MISCOP(220) { /* %NuBus-Write-8B */
    lisp_q slot;
    lisp_q offset;
    lisp_q data;
    nubus_result result;

    data = pop();
    offset = pop();
    slot = pop();

    if ((DTP(slot) != DTP_FIX) || (DTP(offset) != DTP_FIX)
	|| (DTP(data) != DTP_FIX)) {
	printf("arg to %%NuBus-Write-8B not FIXNUM.\n");
	return 0;
    }

    /* FIXME: Do we need to check the ranges on the slot / offset? */

    result = nubus_write8(ADDRESS(slot), ADDRESS(offset), data & 0xff);

    return result == NR_SUCCESS;
}

MISCOP(227) { /* %TEST&SET-68K */
    lisp_q slot;
    lisp_q offset;
/*     u8 value; */

    offset = pop();
    slot = pop();

    dump_q(slot, 0);
    dump_q(offset, 1);

    printf("%%TEST&SET-68K: %lx %lx.\n", ADDRESS(slot),
		ADDRESS(offset));
    /*value = nubus_read8(0xf0 + ADDRESS(slot), ADDRESS(offset));*/

    destination = C_T;
    return 1;
}

MISCOP(232) { /* INTERNAL-CHAR-EQUAL */
    lisp_q ch1;
    lisp_q ch2;

    ch2 = pop();
    ch1 = pop();

    dump_q(ch1, 0);
    dump_q(ch2, 1);

    /* FIXME: Typecheck, ch1 and ch2 must be either FIXNUMs or CHARACTERS. */
    /* FIXME: Should check A-ALPHABETIC-CASE-AFFECTS-STRING-COMPARISON. */
    /* (first time I've cared about -that- A-Memory variable) */
    
    destination = (ADDRESS(ch1) == ADDRESS(ch2))? C_T: C_NIL;
    
    return 1;
}

MISCOP(233) { /* %STRING-EQUAL */
    lisp_q string1;
    lisp_q index1;
    lisp_q string2;
    lisp_q index2;
    lisp_q count;

    lisp_q aryhdr1;
    lisp_q aryhdr2;

    count = pop();
    index2 = pop();
    string2 = pop();
    index1 = pop();
    string1 = pop();
    
    dump_string(string1);
    dump_q(index1, 0);
    dump_string(string2);
    dump_q(index2, 1);
    dump_q(count, 2);

    /* FIXME: implement properly */

    if ((NOT_CDRCODE(index1) != DTP_FIX)
	|| (NOT_CDRCODE(index2) != DTP_FIX)
	|| (NOT_CDRCODE(count) != C_NIL)) {
	printf("%%STRING-EQUAL: hack failure.\n");
	return 0;
    }

    aryhdr1 = memread(string1);
    aryhdr2 = memread(string2);

    dump_q(aryhdr1, 3);
    dump_q(aryhdr2, 3);

    if (ARY_INDEX(aryhdr1) != ARY_INDEX(aryhdr2)) {
	printf("disparate lengths.\n");
	destination = C_NIL;
	return 1;
    }

    if ((NOT_CDRCODE(aryhdr1) == 0x30489003)
	&& (NOT_CDRCODE(aryhdr2) == 0x30489003)) {
	destination = ((memread(string1 + 1) & 0x00ffffff) ==
		       (memread(string2 + 1) & 0x00ffffff))? C_T: C_NIL;
	return 1;
    }

#if 1
    if ((NOT_CDRCODE(aryhdr1) == 0x30489004)
	&& (NOT_CDRCODE(aryhdr2) == 0x30489004)) {
	destination = ((memread(string1 + 1) & 0xffffffff) ==
		       (memread(string2 + 1) & 0xffffffff))? C_T: C_NIL;
	return 1;
    }
#endif
    
    printf("%%STRING-EQUAL: hack failure 2.\n");
    return 0;
}

MISCOP(237) { /* FBOUNDP */
    lisp_q foo;

    foo = memread_inviz(pop() + SYM_FUNCTION);

    /* FIXME: Verify behavior, may be backwards */
    
    /* If foo is DTP_NULL, return C_T, else C_NIL */
    destination = (DTP(foo) == DTP_NULL)? C_T: C_NIL;

    return 1;
}

MISCOP(240) { /* BOUNDP */
    lisp_q sym;
    lisp_q foo;

    sym = pop();
    
    if (DTP(sym) != DTP_SYMBOL) {
	printf("MISCOP(240) /* BOUNDP */: sym not DTP_SYMBOL.\n");
	dump_q(sym, 0);
	return 0;
    }

    printf("BOUNDP for ");
    dump_string(memread_inviz(sym + SYM_PRINTNAME));
    
    foo = memread_inviz(sym + SYM_VALUE);

    dump_q(foo, 0);

    /* If foo is DTP_NULL, return C_NIL, else C_T */
    /* NOTE: SSDN2 has this backwards */
    destination = (DTP(foo) == DTP_NULL)? C_NIL: C_T;

    return 1;
}

MISCOP(252) { /* TYPEP-STRUCTURE-OR-FLAVOR */
    lisp_q object;
    lisp_q type;
    
    lisp_q header;
    lisp_q structure_name;

    type = pop();
    object = pop();

    dump_q(object, 0);
    dump_q(type, 1);

    if (DTP(object) != DTP_ARRAY) {
	printf("TYPEP-STRUCTURE-OR-FLAVOR: Not array.\n");
	return 0;
    }

/*     dump_raw_array(object); */
/*     return 0; */

    header = memread(object);
    if (!ARY_NS(header)) {
	printf("TYPEP-STRUCTURE-OR-FLAVOR: Array not named structure.\n");
	return 0;
    }

    if (!ARY_LEADER(header)) {
	printf("TYPEP-STRUCTURE-OR-FLAVOR: Array has no leader.\n");
	return 0;
    } else {
	structure_name = memread(object - 3);
    }

    destination = (NOT_CDRCODE(structure_name) == NOT_CDRCODE(type))? C_T: C_NIL;

    return 1;
}

MISCOP(254) { /* CHARACTERP */
    lisp_q object;

    object = pop();

    destination = (DTP(object) == DTP_CHARACTER)? C_T: C_NIL;
    
    return 1;
}

MISCOP(261) { /* EQ-T */
    lisp_q object;

    object = pop();

    destination = (NOT_CDRCODE(object) == C_T)? C_T: C_NIL;
    
    return 1;
}

MISCOP(300) { /* RPLACA */
    lisp_q cons;
    lisp_q newcar;
    lisp_q oldcar;

    newcar = pop();
    cons = pop();

    if ((DTP(cons) != DTP_LIST) && (DTP(cons) != DTP_STACK_LIST)) {
	printf("RPLACA: cons not DTP_LIST or DTP_STACK_LIST.\n");
	dump_q(cons, 0);
	dump_q(newcar, 1);
	return 0;
    }

    oldcar = memread(cons);
    memwrite(cons, NOT_CDRCODE(newcar) | CDRCODE(oldcar));

    destination = cons;
    
    return 1;
}

MISCOP(302) { /* SETCAR */
    lisp_q cons;
    lisp_q newcar;
    lisp_q oldcar;

    newcar = pop();
    cons = pop();

    if (DTP(cons) == DTP_LOCATIVE) {
	printf("SETCAR: Locative case.\n");
    } else if ((DTP(cons) != DTP_LIST) && (DTP(cons) != DTP_STACK_LIST)) {
	printf("SETCAR: cons not DTP_LIST or DTP_STACK_LIST.\n");
	dump_q(cons, 0);
	dump_q(newcar, 1);
	return 0;
    }

    oldcar = memread(cons);
    memwrite(cons, NOT_CDRCODE(newcar) | CDRCODE(oldcar));

    destination = newcar;
    
    return 1;
}

MISCOP(330) { /* SET-ARRAY-LEADER */
    lisp_q array;
    lisp_q index;
    lisp_q value;
    lisp_q header;
    lisp_q length;

    value = pop();
    index = pop();
    array = pop();
    
    dump_q(array, 0);
    dump_q(index, 1);
    dump_q(value, 2);

    if (DTP(array) != DTP_ARRAY) {
	printf("SET-ARRAY-LEADER: ARGTYP (array not DTP_ARRAY).\n");
	return 0;
    }

    header = memread(array);

    if (DTP(header) != DTP_ARRAY_HEADER) {
	printf("SET-ARRAY-LEADER: ??? (array header not DTP_ARRAY_HEADER).\n");
	return 0;
    }

    if (!ARY_LEADER(header)) {
	printf("SET-ARRAY-LEADER: ARRAY-HAS-NO-LEADER.\n");
	return 0;
    }

    if (DTP(index) != DTP_FIX) {
	printf("SET-ARRAY-LEADER: ARGTYP (index not DTP_FIX).\n");
	return 0;
    }

    length = memread(array - 1);

    if (DTP(length) != DTP_FIX) {
	printf("SET-ARRAY-LEADER: ??? (header length not DTP_FIX).\n");
	return 0;
    }

    if (fixnum_value(index) >= fixnum_value(length)) {
	printf("SET-ARRAY-LEADER: SUBSCRIPT-OOB.\n");
	return 0;
    }

    memwrite(array - fixnum_value(index) - 2, value);
    
    return 1;
}

MISCOP(331) { /* SET-AR-1 */
    lisp_q array;
    lisp_q value;

    int status;
    lisp_q locative;
    int offset;
    lisp_q header;

    value = pop();
    
    array = memread(context.pdl_pointer - 2);

    dump_q(array, 0);
    
    status = resolve_aref(array, 1, 0, context.pdl_pointer - 1, &header, &locative, &offset);
    if (!status) return 0;

    status = write_aref(header, locative, offset, value);

    context.pdl_pointer -= 1;

    return status;
}

MISCOP(332) { /* SET-AR-2 */
    lisp_q array;
    lisp_q value;

    int status;
    lisp_q locative;
    int offset;
    lisp_q header;

    value = pop();
    
    array = memread(context.pdl_pointer - 3);

    dump_q(array, 0);
    
    status = resolve_aref(array, 2, 0, context.pdl_pointer - 2, &header, &locative, &offset);
    if (!status) return 0;

    status = write_aref(header, locative, offset, value);

    context.pdl_pointer -= 3;

    return status;
}

MISCOP(335) { /* SET-AR-1-FORCE */
    lisp_q array;
    lisp_q value;

    int status;
    lisp_q locative;
    int offset;
    lisp_q header;

    value = pop();
    
    array = memread(context.pdl_pointer - 2);

    dump_q(array, 0);
    
    status = resolve_aref(array, 1, 1, context.pdl_pointer - 1, &header, &locative, &offset);
    if (!status) return 0;

    printf("Locative: %08lx.\n", locative);

    status = write_aref(header, locative, offset, value);

    context.pdl_pointer -= 1;

    return status;
}

MISCOP(344) { /* COPY-ARRAY-PORTION */
    lisp_q from_array;
    lisp_q from_start;
    lisp_q from_end;
    lisp_q to_array;
    lisp_q to_start;
    lisp_q to_end;

    int num_elements;
    int i;

    lisp_q value;
    int status;
    lisp_q locative;
    int offset;
    lisp_q header;

    to_end = pop();
    to_start = pop();
    to_array = pop();
    from_end = pop();
    from_start = pop();
    from_array = pop();

    dump_q(from_array, 0);
    dump_q(from_start, 1);
    dump_q(from_end, 2);
    dump_q(to_array, 3);
    dump_q(to_start, 4);
    dump_q(to_end, 5);

    ASSERT_DTP(from_array, DTP_ARRAY);
    ASSERT_DTP(from_start, DTP_FIX);
    ASSERT_DTP(from_end, DTP_FIX);
    ASSERT_DTP(to_array, DTP_ARRAY);
    ASSERT_DTP(to_start, DTP_FIX);
    ASSERT_DTP(to_end, DTP_FIX);

    num_elements = from_end - from_start;
    if (num_elements != (to_end - to_start)) {
	printf("COPY-ARRAY-PORTION: portion size mismatch.\n");
	return 0;
    }

    for (i = 0; i < num_elements; i++) {
	push_cdrnext(from_start + i);
	status = resolve_aref(from_array, 1, 0, context.pdl_pointer - 1, &header, &locative, &offset);
	if (!status) return 0;
	status = read_aref(header, locative, offset, &value, 0);
	if (!status) return 0;
	pop();
	
	push_cdrnext(to_start + i);
	status = resolve_aref(to_array, 1, 0, context.pdl_pointer - 1, &header, &locative, &offset);
	if (!status) return 0;
	status = write_aref(header, locative, offset, value);
	if (!status) return 0;
	pop();
    }

    /* FIXME: Destination? */
    
    return 1;
}

MISCOP(346) { /* %BLT */
    lisp_q from;
    lisp_q to;
    lisp_q count;
    lisp_q increment;

    increment = pop();
    count = pop();
    to = pop();
    from = pop();

    dump_q(from, 0);
    dump_q(to, 1);
    dump_q(count, 3);
    dump_q(increment, 4);

    count = ADDRESS(count);
    increment = ADDRESS(increment);

    printf("%%BLT from %08lx to %08lx for %ld by %ld.\n",
	   from, to, count, increment);

    while (count--) {
	memwrite_unboxed(to, memread_unboxed(from));
	from += increment;
	to += increment;
    }

    return 1;
}

MISCOP(347) { /* %BLT-TYPED */
    lisp_q from;
    lisp_q to;
    lisp_q count;
    lisp_q increment;

    increment = pop();
    count = pop();
    to = pop();
    from = pop();

    dump_q(from, 0);
    dump_q(to, 1);
    dump_q(count, 3);
    dump_q(increment, 4);

    count = ADDRESS(count);
    increment = ADDRESS(increment);

    printf("%%BLT-TYPED from %08lx to %08lx for %ld by %ld.\n",
	   from, to, count, increment);

    while (count--) {
	memwrite(to, memread(from));
	from += increment;
	to += increment;
    }

    return 1;
}

MISCOP(360) { /* %CHANGE-PAGE-STATUS */
    lisp_q virt_addr;
    lisp_q swap_status;
    lisp_q access_and_meta;

    access_and_meta = pop();
    swap_status = pop();
    virt_addr = pop();

    dump_q(virt_addr, 0);
    dump_q(swap_status, 1);
    dump_q(access_and_meta, 2);

    /* FIXME: Implement this once we run the VMM properly */

/*     push_cdrnext(C_T); */
    destination = C_T;
    
    return 1;
}

MISCOP(401) { /* %MAKE-EXPLICIT-STACK-LIST */
    lisp_q num_elements;

    num_elements = pop();
    dump_q(num_elements, 0);
    memwrite(context.pdl_pointer - 1, CDR_NIL | NOT_CDRCODE(memread(context.pdl_pointer - 1)));

    destination = ADDRESS(context.pdl_pointer - fixnum_value(num_elements)) | DTP_STACK_LIST;
    print_list(destination);
    printf("\n");
    
    return 1;
}

MISCOP(402) { /* %MAKE-EXPLICIT-STACK-LIST* */
    lisp_q num_elements;

    num_elements = pop();
    dump_q(num_elements, 0);
    
    memwrite(context.pdl_pointer - 1, CDR_ERROR | NOT_CDRCODE(memread(context.pdl_pointer - 1)));
    memwrite(context.pdl_pointer - 2, CDR_NORMAL | NOT_CDRCODE(memread(context.pdl_pointer - 2)));

    destination = ADDRESS(context.pdl_pointer - fixnum_value(num_elements)) | DTP_STACK_LIST;
    print_list(destination);
    printf("\n");
    
    return 1;
}

MISCOP(405) { /* SPECIAL-PDL-INDEX */
    destination = context.spdl_pointer;
    
    return 1;
}

MISCOP(412) { /* CONS */
    lisp_q car;
    lisp_q cdr;
    lisp_q cons;

    cdr = pop();
    car = pop();
    
    cons = allocate_list_qs(C_NIL, 2);

    memwrite(cons, CDR_NORMAL | NOT_CDRCODE(car));
    memwrite(cons+1, CDR_ERROR | NOT_CDRCODE(cdr));

    destination = DTP_LIST | ADDRESS(cons);
    
    return 1;
}

MISCOP(415) { /* %ALLOCATE-AND-INITIALIZE */
    lisp_q data_type;
    lisp_q header_type;
    lisp_q header;
    lisp_q second_word;
    lisp_q area;
    lisp_q size;

    lisp_q result;
    int i;

    size = pop();
    area = pop();
    second_word = pop();
    header = pop();
    header_type = pop();
    data_type = pop();

    dump_q(data_type, 0);
    dump_q(header_type, 1);
    dump_q(header, 2);
    dump_q(second_word, 3);
    dump_q(area, 4);
    dump_q(size, 5);

    ASSERT_DTP(size, DTP_FIX);

    size &= Q_BITS_ADDRESS;

    result = allocate_structure_qs(area, fixnum_value(size));

    memwrite(result, CDR_NEXT | ((header_type << 25) & Q_BITS_DTP) | ADDRESS(header));
    memwrite(result+1, CDR_NEXT | NOT_CDRCODE(second_word));

    for (i = 2; i < size; i++) {
	memwrite(result+i, ((i == size)? CDR_NIL: CDR_NEXT) | C_NIL);
    }

    destination = ADDRESS(result) | ((data_type << 25) & Q_BITS_DTP);

    dump_q(destination, 42);
    dump_q(memread(destination), 42);
    dump_string(memread(destination));
    dump_raw_array(memread(destination));
    
    return 1;
}

MISCOP(416) { /* %ALLOCATE-AND-INITIALZE-ARRAY */
    lisp_q header;
    lisp_q index_length;
    lisp_q leader_length;
    lisp_q area;
    lisp_q nqs;
    int i;

    int qs_needed;
    lisp_q array;

    nqs = pop();
    area = pop();
    leader_length = pop();
    index_length = pop();
    header = pop();

    printf("%%ALLOCATE-AND-INITIALIZE-ARRAY.\n");
    
    dump_q(header, 0);
    dump_q(index_length, 1);
    dump_q(leader_length, 2);
    dump_q(area, 3);
    dump_q(nqs, 4);

    qs_needed = ADDRESS(nqs) + 1;
    if (ADDRESS(leader_length)) {
	qs_needed += ADDRESS(leader_length) + 1;
    }

    array = allocate_structure_qs(area, qs_needed);

    if (ADDRESS(leader_length)) {
	/* FIXME: initialize leader here */
	printf("%%ALLOCATE-AND-INITIALIZE-ARRAY: take me to your leader.\n");
	return 0;

	array += ADDRESS(leader_length) + 1;
    }

    memwrite(array, NOT_DTP(header) | DTP_ARRAY_HEADER);

    for (i = 1; i < ADDRESS(nqs); i++) {
	memwrite(array + i, C_NIL);
    }

    destination = ADDRESS(array) + DTP_ARRAY;
    
    return 1;
}

MISCOP(430) { /* %PHYSICAL-ADDRESS */
    lisp_q x;

    x = pop();

    dump_q(x, 0);

    printf("%%PHYSICAL-ADDRESS: No VMM and no BIGNUMS, returning bogus value.\n");

    /* FIXME: Cheap hack city. */

    destination = DTP_FIX;

    return 1;
}

MISCOP(434) { /* %NuBus-Read */
    lisp_q slot;
    lisp_q offset;
    u32 data;
    nubus_result result;

    offset = pop();
    slot = pop();
    
    if ((DTP(slot) != DTP_FIX) || (DTP(offset) != DTP_FIX)) {
	printf("arg to %%NuBus-Read not FIXNUM.\n");
	return 0;
    }

    result = nubus_read32(ADDRESS(slot), ADDRESS(offset), &data);
    printf("%%NuBus-Read: incompletely implemented, possible data loss.\n");
    dump_q(data, 0);
    destination = DTP_FIX | ADDRESS(data);

    return result == NR_SUCCESS;
}

MISCOP(435) { /* %NuBus-Read-8B */
    lisp_q slot;
    lisp_q offset;
    u8 data;
    nubus_result result;

    offset = pop();
    slot = pop();

    if ((DTP(slot) != DTP_FIX) || (DTP(offset) != DTP_FIX)) {
	printf("arg to %%NuBus-Read-8B not FIXNUM.\n");
	return 0;
    }

    /* FIXME: Do we need to check the ranges on the slot / offset? */

    result = nubus_read8(ADDRESS(slot), ADDRESS(offset), &data);
    destination = DTP_FIX | data;

    return result == NR_SUCCESS;
}

MISCOP(436) { /* %NuBus-Read-16B */
    lisp_q slot;
    lisp_q offset;
    u16 data;
    nubus_result result;

    offset = pop();
    slot = pop();

    if ((DTP(slot) != DTP_FIX) || (DTP(offset) != DTP_FIX)) {
	printf("arg to %%NuBus-Read-16B not FIXNUM.\n");
	return 0;
    }

    /* FIXME: Do we need to check the ranges on the slot / offset? */

    result = nubus_read16(ADDRESS(slot), ADDRESS(offset), &data);
    destination = DTP_FIX | data;

    return result == NR_SUCCESS;
}

MISCOP(440) { /* %NuBus-Read-8B-Careful */
    lisp_q slot;
    lisp_q offset;
    u8 data;
    nubus_result result;

    offset = pop();
    slot = pop();

    if ((DTP(slot) != DTP_FIX) || (DTP(offset) != DTP_FIX)) {
	printf("arg to %%NuBus-Read-8B not FIXNUM.\n");
	return 0;
    }

    /* FIXME: Do we need to check the ranges on the slot / offset? */

    result = nubus_read8(ADDRESS(slot), ADDRESS(offset), &data);
    destination = DTP_FIX | data;

    /* FIXME: Should be NIL for bus error, T for timeout */
    if (result == NR_BUSERR) {
	destination = C_T;
    }

    return 1;
}

MISCOP(445) { /* %POINTER */
    lisp_q x;

    x = pop();

    destination = DTP_FIX | ADDRESS(x);

    return 1;
}

MISCOP(446) { /* %MAKE-POINTER */
    lisp_q newdtp;
    lisp_q pointer;
	
    pointer = pop();
    newdtp = pop();

    destination = ADDRESS(newdtp) << 25;
    destination |= ADDRESS(pointer);
    
    return 1;
}

MISCOP(447) { /* %MAKE-POINTER-OFFSET */
    lisp_q newdtp;
    lisp_q pointer;
    lisp_q offset;

    offset = pop();
    pointer = pop();
    newdtp = pop();

#if 0 /* If they want negative offsets... */
    if (offset & 0x1000000) {
	printf("%%MAKE-POINTER-OFFSET: negative offset.\n");
	return 0;
    }
#endif

    destination = ADDRESS(newdtp) << 25;
    destination |= ADDRESS(pointer) + ADDRESS(offset);

    return 1;
}

MISCOP(450) { /* %DATA-TYPE */
    lisp_q pointer;

    pointer = pop_loc();

    destination = DTP_FIX | (DTP(memread_inviz(pointer)) >> 25);

    return 1;
}

MISCOP(451) { /* %P-CDR-CODE */
    lisp_q pointer;

    pointer = pop();

    destination = DTP_FIX | (CDRCODE(memread(pointer)) >> 30);

    return 1;
}

MISCOP(452) { /* %P-DATA-TYPE */
    lisp_q pointer;

    pointer = pop();

    destination = DTP_FIX | (DTP(memread(pointer)) >> 25);

    return 1;
}

MISCOP(454) { /* %P-LDB */
    lisp_q ppss;
    lisp_q pointer;
    u32 value;

    pointer = pop();
    ppss = pop();

    printf("%%P-LDB: %04lo.\n", ADDRESS(ppss));
    dump_q(ppss, 0);
    dump_q(pointer, 0);
    
    value = memread_unboxed(pointer);
    value >>= ppss_position(ppss);

    value &= ppss_mask(ppss);

    printf("%%P-LDB: result 0x%08lx.\n", value);
	
    destination = DTP_FIX | value;

    return 1;
}

MISCOP(457) { /* %P-LDB-OFFSET */
    lisp_q ppss;
    lisp_q pointer;
    lisp_q offset;
    u32 value;

    offset = pop();
    pointer = pop();
    ppss = pop();

    printf("%%P-LDB-OFFSET: %04lo.\n", ADDRESS(ppss));
    dump_q(ppss, 0);
    dump_q(pointer, 0);
    dump_q(offset, 0);

    inviz(pointer, &pointer);

    pointer += offset;
    
    value = memread_unboxed(pointer);
    value >>= ppss_position(ppss);

    value &= ppss_mask(ppss);

    printf("%%P-LDB-OFFSET: result 0x%08lx.\n", value);
	
    destination = DTP_FIX | value;

    return 1;
}

MISCOP(461) { /* %P-CONTENTS-AS-LOCATIVE */
    lisp_q pointer;

    pointer = pop();

    destination = DTP_LOCATIVE | ADDRESS(memread(pointer));

    return 1;
}

MISCOP(463) { /* %POINTER-DIFFERENCE */
    lisp_q ptr1;
    lisp_q ptr2;

    ptr2 = pop();
    ptr1 = pop();

    destination = DTP_FIX | ADDRESS(ptr1 - ptr2);
    
    return 1;
}

MISCOP(474) { /* %PAGE-STATUS */
    lisp_q ptr;

    /*
     * From SSDN2, section 6:
     *
     * %page-status (virtual-address)
     *     If the page containing virtual-address is swapped out, or if
     *     it is part of  one  of  the  low-numbered  permanently-wired
     *     system  areas,  this returns NIL.  Otherwise, it returns the
     *     entire first word of the Page Hash Table (PHT) entry for the
     *     page.
     */

    ptr = pop();

    printf("%%PAGE-STATUS.\n");
    dump_q(ptr, 0);
    destination = get_pht_entry(ptr);
    dump_q(destination, 1);
    /* FIXME: Cheap hack */
    destination = 0x4c45 | DTP_FIX;
    printf("%%PAGE-STATUS: cheap hack, returning #x4c45 as DTP_FIX.\n");
    
    return 1;
}

MISCOP(500) { /* %AREA-NUMBER */
    lisp_q x;

    x = pop();

    dump_q(x, 0);

    /*
     * FIXME: find_region_number and find_region_area bomb if they
     * can't find a matching region or area, %AREA-NUMBER should
     * return C_NIL.
     */
    
    destination = DTP_FIX | find_region_area(find_region_number(x));

    dump_q(destination, 42);
    
    return 1;
}

MISCOP(501) { /* %REGION-NUMBER */
    /*
     * FIXME: find_region_number bombs if not in region,
     * %REGION-NUMBER should return C_NIL.
     */

    destination = DTP_FIX | find_region_number(pop());
    
    return 1;
}

MISCOP(504) { /* %STRUCTURE-TOTAL-SIZE */
    destination = structure_total_size(pop());

    return 1;
}

MISCOP(506) { /* %FIND-STRUCTURE-LEADER */
    destination = find_structure_leader(pop());
    
    return 1;
}

MISCOP(514) { /* GLOBAL:MINUS */
    lisp_q foo;

    foo = pop();

    if (DTP(foo) != DTP_FIX) {
	printf("MINUS of non-FIXNUM value.\n");
	return 0;
    }

    /* FIXME: BIGNUMs */

    destination = DTP_FIX | ADDRESS(0 - ADDRESS(foo));
    return 1;
}

MISCOP(515) { /* HAULONG */
    lisp_q val;

    val = pop();

    dump_q(val, 0);

    if (DTP(val) != DTP_FIX) {
	printf("HAULONG: Val not DTP_FIX.\n");
	return 0;
    }

    if (val & 0x01000000) {
	printf("HAULONG: Val negative.\n");
	return 0;
    }

    val = ADDRESS(val);

    for (destination = DTP_FIX; val; destination++) val >>= 1;

    return 1;
}

MISCOP(520) { /* LDB */
    lisp_q val;
    lisp_q ppss;

    val = pop();
    ppss = pop();

    if (DTP(ppss) != DTP_FIX) {
	printf("LDB: ppss not FIXNUM.\n");
	return 0;
    }

    return ldb_generic(val, ppss_mask(ppss), ppss_position(ppss),
		       ppss_numbits(ppss), &destination);
}

MISCOP(524) { /* DPB */
    lisp_q val;
    lisp_q ppss;
    lisp_q num;

    num = pop();
    ppss = pop();
    val = pop();

    printf("DPB: %04lo.\n", ADDRESS(ppss));
    dump_q(num, 0);
    dump_q(val, 0);

    /* FIXME: typecheck? BIGNUMs? */

    if (DTP(num) != DTP_FIX) {
	printf("DPB: Number is not FIXNUM.\n");
	return 0;
    }

    if ((ppss_position(ppss) + ppss_numbits(ppss)) > 24) {
	/* FIXME: Improve BIGNUM result test to handle the general case */
	if ((Q_BITS_CDRCODE | Q_BITS_DTP) >> ppss_position(ppss) & ppss_mask(ppss) & val) {
	    int bitlength;
	    int wordlength;
	    lisp_q bignum;

	    /* FIXME: Improve to handle the general case. */
	    /* FIXME: Take val into account when calculating wordlength. */
	    
	    printf("DPB: Result would be BIGNUM.\n");

	    bitlength = ppss_position(ppss) + ppss_numbits(ppss);
	    wordlength = (bitlength + 30) / 31;

/* 	    printf("bitlength %d, wordlength %d.\n", bitlength, wordlength); */

	    bignum = allocate_number_qs(wordlength+1);
	    bignum |= DTP_EXTENDED_NUMBER;

	    memwrite(bignum, DTP_HEADER | HT_BIGNUM | wordlength);

	    {
		int i;
		u32 mask;
		long position;
		u32 numbits;

		mask = ppss_mask(ppss);
		position = ppss_position(ppss);
		numbits = ppss_numbits(ppss);

		for (i = 0; i < wordlength; i++) {
		    u32 tmp;

		    if (i == 0) {
			tmp = ADDRESS(num);
		    } else {
			tmp = 0;
		    }
		    
		    if (mask && (position <= 30)) {
			tmp &= ~(mask << position);
			tmp |= ((val & mask) << position) & 0x7fffffff;
		    }

		    memwrite(bignum + i + 1, tmp);
		    printf("BIGNUM word %d: 0x%08lx.\n", i, tmp);

		    position -= 31;

		    if (position < 0) {
/* 			printf("pmv: %ld %lx %lx.\n", position, mask, val); */
			mask >>= -position;
			val >>= -position;
			position = 0;
		    }
		}
	    }

	    destination = bignum;
	    
	    return 1;
	}
    }
    
    destination = num;
    destination &= ~(ppss_mask(ppss) << ppss_position(ppss));
    destination |= (val & ppss_mask(ppss)) << ppss_position(ppss);

    destination = DTP_FIX | ADDRESS(destination);

    return 1;
}

MISCOP(525) { /* %LOGDPB */
    lisp_q val;
    lisp_q ppss;
    lisp_q num;

    num = pop();
    ppss = pop();
    val = pop();

    printf("%%LOGDPB: %04lo.\n", ADDRESS(ppss));
    dump_q(num, 0);
    dump_q(val, 0);

    destination = num;
    destination &= ~(ppss_mask(ppss) << ppss_position(ppss));
    destination |= (val & ppss_mask(ppss)) << ppss_position(ppss);

    destination = DTP_FIX | ADDRESS(destination);

    return 1;
}

MISCOP(530) { /* LSH */
    lisp_q n;
    lisp_q nbits;

    nbits = pop();
    n = pop();

    /* FIXME: Signal ARGTYP if n or nbits is not DTP_FIX */

    dump_q(n, 0);
    dump_q(nbits, 1);

    if (nbits & 0x01000000) { /* shift right */
	printf("LSH: right shift not supported.\n");
	return 0;
    } else { /* shift left */
	/* FIXME: Do we create a BIGNUM here if we overflow? */
	/* FIXME: Do we support shift counts larger than 31? */
	destination = DTP_FIX | ADDRESS(n << ADDRESS(nbits));
    }
    
    return 1;
}

MISCOP(531) { /* ASH */
    lisp_q n;
    lisp_q nbits;
    int shift_count;

    nbits = pop();
    n = pop();

    /* FIXME: Signal ARGTYP if n or nbits is not DTP_FIX */

    dump_q(n, 0);
    dump_q(nbits, 1);

    if (DTP(n) != DTP_FIX) {
	printf("ASH: n not FIXNUM.\n");
	return 0;
    }

    if (DTP(nbits) != DTP_FIX) {
	printf("ASH: nbits not FIXNUM.\n");
	return 0;
    }

    shift_count = fixnum_value(nbits);

    if (shift_count < 0) { /* shift right */
	/* FIXME: Do we create a BIGNUM here if we overflow? */
	/* FIXME: Do we support shift counts larger than 31? */
	destination = DTP_FIX | (ADDRESS(n) >> -shift_count);
    } else { /* shift left */
	/* FIXME: Do we create a BIGNUM here if we overflow? */
	/* FIXME: Do we support shift counts larger than 31? */
	destination = DTP_FIX | ADDRESS(n << ADDRESS(nbits));
    }

    dump_q(destination, 2);
    
    return 1;
}

MISCOP(534) { /* MAX */
    lisp_q num1;
    lisp_q num2;

    num1 = pop();
    num2 = pop();

    /* FIXME: Not even checking datatypes (!) */

    destination = DTP_FIX | ((ADDRESS(num1) > ADDRESS(num2)) ?
			     ADDRESS(num1) : ADDRESS(num2));
    return 1;
}

MISCOP(535) { /* MIN */
    lisp_q num1;
    lisp_q num2;

    num1 = pop();
    num2 = pop();

    /* FIXME: Not even checking datatypes (!) */

    destination = DTP_FIX | ((ADDRESS(num1) < ADDRESS(num2)) ?
			     ADDRESS(num1) : ADDRESS(num2));
    return 1;
}

MISCOP(536) { /* EXPT */
    lisp_q base;
    lisp_q exponent;

    exponent = pop();
    base = pop();
	
    if ((NOT_CDRCODE(exponent) != (DTP_FIX | 0x0b)) ||
	(NOT_CDRCODE(base) != (DTP_FIX | 0x02))) {
	printf("EXPT: Not 2 ^ 11.\n");
	
	dump_q(base, 0);
	dump_q(exponent, 1);

	return 0;
    }
	
    destination = DTP_FIX | 0x800;

    return 1;
}

MISCOP(542) { /* LOGIOR */
    lisp_q x;
    lisp_q y;

    y = pop();
    x = pop();

    dump_q(x, 0);
    dump_q(y, 1);

    ASSERT_DTP(x, DTP_FIX);
    ASSERT_DTP(y, DTP_FIX);

    destination = x | y;
    
    return 1;
}

MISCOP(550) { /* FLOOR-1 */
    lisp_q dividend;
    lisp_q divisor;

    divisor = pop();
    dividend = pop();

    dump_q(dividend, 0);
    dump_q(divisor, 1);

    /* FIXME: Not even checking datatypes (!) */
	
    destination = DTP_FIX | (ADDRESS(dividend) / ADDRESS(divisor));

    return 1;
}

MISCOP(551) { /* CEILING-1 */
    lisp_q dividend;
    lisp_q divisor;

    divisor = pop();
    dividend = pop();

    dump_q(dividend, 0);
    dump_q(divisor, 1);

    /* FIXME: Not even checking datatypes (!) */

#if 0
    destination = DTP_FIX | ((ADDRESS(dividend) / ADDRESS(divisor)) + 1);
#else
    dividend &= Q_BITS_ADDRESS;
    divisor &= Q_BITS_ADDRESS;

    destination = DTP_FIX | ((dividend + divisor - 1) / divisor);
#endif

    return 1;
}

MISCOP(552) { /* TRUNCATE-1 */
    lisp_q dividend;
    lisp_q divisor;

    divisor = pop();
    dividend = pop();

    dump_q(dividend, 0);
    dump_q(divisor, 1);

    /* FIXME: Not even checking datatypes (!) */
    
    /* FIXME: How is this different from FLOOR-1? */
    
    destination = DTP_FIX | (ADDRESS(dividend) / ADDRESS(divisor));

    return 1;
}

MISCOP(554) { /* FLOOR-2 */
    lisp_q dividend;
    lisp_q divisor;

    divisor = pop();
    dividend = pop();

    dump_q(dividend, 0);
    dump_q(divisor, 1);

    /* FIXME: Not even checking datatypes (!) */

    push_cdrnext(DTP_FIX | (ADDRESS(dividend) / ADDRESS(divisor)));
    destination = DTP_FIX | (ADDRESS(dividend) % ADDRESS(divisor));

    return 1;
}

MISCOP(555) { /* CEILING-2 */
    lisp_q dividend;
    lisp_q divisor;

    divisor = pop();
    dividend = pop();

    dump_q(dividend, 0);
    dump_q(divisor, 1);

    /* FIXME: Not even checking datatypes (!) */

    push_cdrnext(DTP_FIX | (ADDRESS(dividend) / ADDRESS(divisor)));
    destination = DTP_FIX | (ADDRESS(dividend) % ADDRESS(divisor));

    return 1;
}

MISCOP(556) { /* TRUNCATE-2 */
    lisp_q dividend;
    lisp_q divisor;

    divisor = pop();
    dividend = pop();

    dump_q(dividend, 0);
    dump_q(divisor, 1);

    /* FIXME: Not even checking datatypes (!) */
    
    /* FIXME: How is this different from FLOOR-2? */

    push_cdrnext(DTP_FIX | (ADDRESS(dividend) % ADDRESS(divisor)));
    destination = DTP_FIX | (ADDRESS(dividend) % ADDRESS(divisor));

    return 1;
}

MISCOP(560) { /* ZEROP */
    lisp_q foo;
    
    foo = pop();
    
    if (DTP(foo) != DTP_FIX) {
	printf("ZEROP: Not FIXNUM.\n");
	return 0;
    }
    
    destination = (NOT_CDRCODE(foo) == DTP_FIX)? C_T: C_NIL;

    return 1;
}

MISCOP(562) { /* EQUAL */
    lisp_q foo;
    lisp_q bar;

    bar = pop();
    foo = pop();

    destination = equal(foo, bar);
    
    return 1;
}

MISCOP(565) { /* NUMBERP */
    lisp_q foo;

    foo = pop();

    dump_q(foo, 0);

    /* FIXME: Only detects FIXNUM types. */
    destination = (DTP(foo) == DTP_FIX)? C_T: C_NIL;

    return 1;
}

MISCOP(566) { /* PLUSP */
    lisp_q foo;

    foo = pop();

    dump_q(foo, 0);

    if (DTP(foo) != DTP_FIX) {
	printf("PLUSP: Not FIXNUM.\n");
	return 0;
    }

    destination = ((foo & 0x01000000) || !ADDRESS(foo)) ? C_NIL : C_T;

    dump_q(destination, 0);

    return 1;
}

MISCOP(570) { /* < */
    lisp_q foo;
    lisp_q bar;

    /* FIXME: Check datatypes. Make work on non-FIXNUM types */

    bar = NOT_CDRCODE(pop());
    foo = NOT_CDRCODE(pop());
    dump_q(foo, 0);
    dump_q(bar, 1);
    destination = (foo < bar)? C_T: C_NIL;

    return 1;
}

MISCOP(571) { /* > */
    lisp_q foo;
    lisp_q bar;

    /* FIXME: Check datatypes. Make work on non-FIXNUM types */

    bar = NOT_CDRCODE(pop());
    foo = NOT_CDRCODE(pop());
    dump_q(foo, 0);
    dump_q(bar, 1);
    destination = (foo > bar)? C_T: C_NIL;

    return 1;
}

MISCOP(572) { /* = */
    lisp_q foo;
    lisp_q bar;

    /* FIXME: Check datatypes. Make work on non-FIXNUM types */

    bar = NOT_CDRCODE(pop());
    foo = NOT_CDRCODE(pop());
    dump_q(foo, 0);
    dump_q(bar, 1);
    destination = (foo == bar)? C_T: C_NIL;

    return 1;
}

MISCOP(577) { /* SYMBOLP */
    lisp_q foo;

    foo = pop();

    destination = (DTP(foo) == DTP_SYMBOL)? C_T : C_NIL;

    return 1;
}

MISCOP(600) { /* ARRAYP */
    lisp_q foo;

    foo = pop();

    destination = (DTP(foo) == DTP_ARRAY)? C_T: C_NIL;
    
    return 1;
}

MISCOP(610) { /* ASSQ */
    lisp_q foo;
    lisp_q alist;

    alist = pop();
    foo = NOT_CDRCODE(pop());

    dump_q(foo, 0);
    dump_q(alist, 1);

    print_list(alist);
    printf("\n");

    destination = C_NIL;
    while(NOT_CDRCODE(alist) != (C_NIL)) {
	if (NOT_CDRCODE(car(car(alist))) == foo) {
	    destination = car(alist);
	    break;
	}
	alist = cdr(alist);
    }

    dump_q(destination, 2);

    return 1;
}

MISCOP(611) { /* LAST */
    lisp_q list;
    lisp_q next;
    
    list = pop();

    if ((DTP(list) != DTP_LIST)
	&& (DTP(list) != DTP_STACK_LIST)
	&& (NOT_CDRCODE(list) != C_NIL)) {
	printf("LAST: list not list.\n");
	return 0;
    }

    while(1) {
	next = NOT_CDRCODE(cdr(list));
	if (next == C_NIL) {
	    destination = list;
	    return 1;
	}
	list = next;
	
	if ((DTP(list) != DTP_LIST)
	    && (DTP(list) != DTP_STACK_LIST)) {
	    printf("LAST: non-cons in list.\n");
	    return 0;
	}
    }
}

MISCOP(612) { /* LENGTH */
    lisp_q list_or_array;
    
    printf("LENGTH.\n");

    list_or_array = pop();
    dump_q(list_or_array, 0);

    if (NOT_CDRCODE(list_or_array) == C_NIL) {
	destination = DTP_FIX | 0;
	return 1;
    } else if ((DTP(list_or_array) == DTP_LIST) ||
	       (DTP(list_or_array) == DTP_STACK_LIST)){
	int length;
	
	length = 0;
	
	while ((DTP(list_or_array) == DTP_LIST) ||
	       (DTP(list_or_array) == DTP_STACK_LIST)){
	    length++;
	    list_or_array = cdr(list_or_array);
	}
	
	destination = DTP_FIX | length;
	return 1;
    } else if (DTP(list_or_array) == DTP_ARRAY) {
	MISCOP(662);
	push(list_or_array);
	return miscop_662(opcode, opnum);
    }

    printf("Unsupported parameter type in LENGTH.\n");
    
    return 0;
}

MISCOP(613) { /* MEMQ */
    lisp_q foo;
    lisp_q list;
	
    list = pop();
    foo = NOT_CDRCODE(pop());
	
    while((NOT_CDRCODE(list) != (C_NIL)) &&
	  (NOT_CDRCODE(car(list)) != foo)) {
	list = cdr(list);
    }

    destination = list;

    return 1;
}

MISCOP(614) { /* NTH */
    lisp_q n;
    lisp_q list;

    list = pop();
    n = pop();

    dump_q(n, 0);
    dump_q(list, 1);

    /* FIXME: Only works on FIXNUMs? */
    n = ADDRESS(n);

    while (n--) {
	list = cdr(list);
    }

    destination = car(list);
    
    return 1;
}

MISCOP(616) { /* FIND-POSITION-IN-LIST */
    lisp_q foo;
    lisp_q list;
    int index;

    index = 0;
    list = pop();
    foo = NOT_CDRCODE(pop());
	
    while((NOT_CDRCODE(list) != (C_NIL))) {
	if(NOT_CDRCODE(car(list)) == foo) {
	    destination = DTP_FIX | index;
	    return 1;
	}
	index++;
	list = cdr(list);
    }

    destination = C_NIL;

    return 1;
}

MISCOP(626) { /* COMMON-LISP-ELT */
    lisp_q sequence;
    lisp_q index;

    index = pop();
    sequence = pop();

    dump_q(sequence, 0);
    dump_q(index, 1);

    if ((DTP(index) != DTP_FIX) || (index & 0x01000000)) {
	printf("COMMON-LISP-ELT: index not positive FIXNUM.\n");
	return 0;
    } else if ((DTP(sequence) == DTP_LIST) || (DTP(sequence) == DTP_STACK_LIST)) {
	printf("COMMON-LISP-ELT: List case unimplemented.\n");
	return 0;
    } else if (DTP(sequence) == DTP_ARRAY) {
	int status;
	lisp_q locative;
	int offset;
	lisp_q header;

	push(index);
	
	status = resolve_aref(sequence, 1, 0, context.pdl_pointer - 1, &header, &locative, &offset);
	if (!status) return 0;

	status = read_aref(header, locative, offset, &destination, 1);
	
	context.pdl_pointer--;

	return status;
    }

    printf("COMMON-LISP-ELT: Sequence not List or Array.\n");
    return 0;
}

MISCOP(627) { /* SYMBOL-FUNCTION */
    lisp_q symbol;
    
    symbol = pop();
    if (DTP(symbol) != DTP_SYMBOL) {
	/* FIXME: signal ARGTYP */
    }
    destination = memread((ADDRESS(symbol) + SYM_FUNCTION));

    return 1;
}

MISCOP(631) { /* SYMBOL-NAME */
    lisp_q symbol;

    symbol = pop();

    ASSERT_DTP(symbol, DTP_SYMBOL);

    destination = DTP_ARRAY | ADDRESS(memread(symbol));
    
    return 1;
}

MISCOP(632) { /* VALUE-CELL-LOCATION */
    lisp_q symbol;

    symbol = pop();
    if (DTP(symbol) != DTP_SYMBOL) {
	/* FIXME: signal ARGTYP */
    }
    destination = DTP_LOCATIVE | (ADDRESS(symbol) + SYM_VALUE);

    return 1;
}

MISCOP(633) { /* FUNCTION-CELL-LOCATION */
    lisp_q symbol;

    symbol = pop();
    if (DTP(symbol) != DTP_SYMBOL) {
	/* FIXME: signal ARGTYP */
    }
    destination = DTP_LOCATIVE | (ADDRESS(symbol) + SYM_FUNCTION);

    return 1;
}

MISCOP(634) { /* PROPERTY-CELL-LOCATION */
    lisp_q symbol;

    symbol = pop();
    if (DTP(symbol) != DTP_SYMBOL) {
	/* FIXME: signal ARGTYP */
    }
    destination = DTP_LOCATIVE | (ADDRESS(symbol) + SYM_PROPERTY);

    return 1;
}

MISCOP(635) { /* SYMBOL-PACKAGE */
    lisp_q symbol;
    
    symbol = pop();
    if (DTP(symbol) != DTP_SYMBOL) {
	/* FIXME: signal ARGTYP */
    }
    destination = memread((ADDRESS(symbol) + SYM_PACKAGE));

    return 1;
}

MISCOP(636) { /* SYMBOL-VALUE */
    lisp_q symbol;
    
    symbol = pop();
    if (DTP(symbol) != DTP_SYMBOL) {
	/* FIXME: signal ARGTYP */
    }
    destination = memread((ADDRESS(symbol) + SYM_VALUE));

    return 1;
}

MISCOP(637) { /* %EXTERNAL-VALUE-CELL */
    lisp_q symbol;

    symbol = pop();

    dump_q(symbol, 0);
    
    destination = DTP_LOCATIVE | ADDRESS(symbol + SYM_VALUE);

    dump_q(destination, 1);
    
    return 1;
}

MISCOP(640) { /* ARRAY-LEADER */
    lisp_q array;
    lisp_q index;
    lisp_q header;
    lisp_q length;

    index = pop();
    array = pop();

    dump_q(array, 0);
    dump_q(index, 1);

    if (DTP(array) != DTP_ARRAY) {
	printf("ARRAY-LEADER: ARGTYP (array not DTP_ARRAY).\n");
	return 0;
    }

    header = memread(array);

    if (DTP(header) != DTP_ARRAY_HEADER) {
	printf("ARRAY-LEADER: ??? (array header not DTP_ARRAY_HEADER).\n");
	return 0;
    }

    if (!ARY_LEADER(header)) {
	printf("ARRAY-LEADER: ARRAY-HAS-NO-LEADER.\n");
	return 0;
    }

    if (DTP(index) != DTP_FIX) {
	printf("ARRAY-LEADER: ARGTYP (index not DTP_FIX).\n");
	return 0;
    }

    length = memread(array - 1);

    if (DTP(length) != DTP_FIX) {
	printf("ARRAY-LEADER: ??? (header length not DTP_FIX).\n");
	return 0;
    }

    if (fixnum_value(index) >= fixnum_value(length)) {
	printf("ARRAY-LEADER: SUBSCRIPT-OOB.\n");
	return 0;
    }

    destination = memread(array - fixnum_value(index) - 2);
    dump_q(destination, 0);
    
    return 1;
}

MISCOP(645) { /* AP-1 */
    lisp_q ary;
    lisp_q index;
    lisp_q ary_header;

    index = pop();
    ary = pop();

    dump_q(ary, 0);
    dump_q(index, 1);

    ary_header = memread(ary);
    dump_q(ary_header, 2);
    dump_array_header(ary_header);

    if ((NOT_CDRCODE(ary_header) != 0x30389080) &&
	(NOT_CDRCODE(ary_header) != 0x30391002)) return 0;

    /* FIXME: Cheating like mad again... */

    if (ARY_DISPLACED(ary_header)) {
	destination = DTP_LOCATIVE | (ADDRESS(memread(ary + 1)) + ADDRESS(index));
    } else {
	destination = DTP_LOCATIVE | (ADDRESS(ary) + ADDRESS(index));
    }

    return 1;
}

MISCOP(660) { /* ARRAY-LENGTH */
    lisp_q array;
    lisp_q array_header;

    array = pop();

    /*
     * Just like ARRAY-ACTIVE-LENGTH, below, except without the fill
     * pointer bit.
     */

    array_header = memread(array);

    if (!ARY_LL(array_header)) {
	destination = DTP_FIX | ARY_INDEX(array_header);
	return 1;
    }

    /* FIXME: displaced arrays? long-length arrays? */
    
    return 0;
}

MISCOP(662) { /* ARRAY-ACTIVE-LENGTH */
    lisp_q array;
    lisp_q array_header;
    lisp_q fill_pointer;

    array = pop();

    /*
     * If array has a fill pointer, then we return the fill pointer.
     * Otherwise, we return the number of elements in array.
     *
     * An array has a fill pointer if it has a leader, and the first
     * leader entry is a FIXNUM (which is then the fill pointer).
     */

    array_header = memread(array);

    if (ARY_LEADER(array_header)) {
	fill_pointer = memread(array - 2);
	if (DTP(fill_pointer) == DTP_FIX) {
	    destination = fill_pointer;
	    return 1;
	}
    }

    if (ARY_DISPLACED(array_header)) {
	destination = memread(array + ARY_DIMS(array_header) + 1);
	return 1;
    }

    if (!ARY_LL(array_header)) {
	destination = DTP_FIX | ARY_INDEX(array_header);
	return 1;
    }

    /* FIXME: displaced arrays? long-length arrays? */
    destination = memread(array + 1);
    
    return 1;
}

MISCOP(664) { /* ARRAY-RANK */
    lisp_q array;
    lisp_q array_header;

    array = pop();

    /*
     * Just returns the number of dimentions for the array
     */

    array_header = memread(array);

    destination = DTP_FIX | (ARY_DIMS(array_header));
    
    return 1;
}

MISCOP(665) { /* ARRAY-DIMENSION */
    lisp_q array;
    lisp_q dimension;

    lisp_q header;

    dimension = pop();
    array = pop();

    dump_q(array, 0);
    dump_q(dimension, 1);

    header = memread(array);
    dump_array_header(header);

    destination = memread(array + !!ARY_LL(header) + dimension + 1);
    dump_q(destination, 44);
    
    return 1;
}

MISCOP(670) { /* COMMON-LISP-AR-1 */
    lisp_q array;

    int status;
    lisp_q locative;
    int offset;
    lisp_q header;

    array = memread(context.pdl_pointer - 2);

    dump_q(array, 0);
    
    status = resolve_aref(array, 1, 0, context.pdl_pointer - 1, &header, &locative, &offset);
    if (!status) return 0;

    printf("Locative: %08lx.\n", locative);

    status = read_aref(header, locative, offset, &destination, 1);

    context.pdl_pointer -= 2;

    return status;
}

MISCOP(671) { /* COMMON-LISP-AR-2 */
    lisp_q array;

    int status;
    lisp_q locative;
    int offset;
    lisp_q header;

    array = memread(context.pdl_pointer - 3);

    dump_q(array, 0);
    
    status = resolve_aref(array, 2, 0, context.pdl_pointer - 2, &header, &locative, &offset);
    if (!status) return 0;

    status = read_aref(header, locative, offset, &destination, 1);

    context.pdl_pointer -= 3;

    return status;
}

MISCOP(700) { /* %SXHASH-STRING */
    lisp_q string;
    lisp_q character_mask;

    u32 length;
    u32 i;
    u32 acc;

    int status;
    lisp_q locative;
    int offset;
    lisp_q header;

    character_mask = pop();
    string = pop();

    dump_q(string, 0);
    dump_q(character_mask, 1);

    dump_string(string);
    dump_array_header(memread(string));

    push(string);
    miscop_662(opcode, opnum);

    length = fixnum_value(destination);
    character_mask = fixnum_value(character_mask);

    acc = 0;
    for (i = 0; i < length; i++) {
	push(DTP_FIX | i);
	status = resolve_aref(string, 1, 0, context.pdl_pointer - 1, &header, &locative, &offset);
	if (!status) return 0;
	status = read_aref(header, locative, offset, &destination, 1);
	if (!status) return 0;
	context.pdl_pointer--;

	acc ^= destination & character_mask;
	acc <<= 7;
	acc |= (acc >> 24) & 0x7f;
    }

    if (acc & 0x800000) acc ^= 0x800001;

    destination = DTP_FIX | (acc & 0x00ffffff);

    dump_q(destination, 0);
    
    return 1;
}

MISCOP(701) { /* %STRING-SEARCH-CHAR */
    lisp_q character;
    lisp_q string;
    lisp_q start;
    lisp_q end;

    int i;

    end = pop();
    start = pop();
    string = pop();
    character = pop();

    dump_q(character, 0);
    dump_q(string, 1);
    dump_q(start, 2);
    dump_q(end, 3);

    ASSERT_DTP(start, DTP_FIX);
    ASSERT_DTP(end, DTP_FIX);

    start = fixnum_value(start);
    end = fixnum_value(end);

    /*
     * Search string from start to end looking for character.
     * If character is found, return its index.
     * If character is not found, return NIL.
     */

    for (i = start; i < end; i++) {
	/* FIXME: Aref character here. */
	printf("string-search-char: non-0-length string.\n");
	return 0;
    }

    destination = C_NIL;
    
    return 1;
}

MISCOP(703) { /* INT-CHAR */
    int value;

    value = pop();

    /* FIXME: Typechecking? */

    destination = DTP_CHARACTER | ADDRESS(value);
    
    return 1;
}

MISCOP(704) { /* CHAR-INT */
    int value;

    value = pop();

    /* FIXME: Typechecking? */

    destination = DTP_FIX | (value & 0xff);
    
    return 1;
}

miscop_handler miscop_dispatch[0x200] = {
    /* 000 - 017 */
    NULL,       NULL,       miscop_002, miscop_003,
    NULL,       miscop_005, NULL,       NULL,
    NULL,       NULL,       NULL,       miscop_013,
    NULL,       NULL,       NULL,       NULL,

    /* 020 - 037 */
    NULL,       NULL,       NULL,       NULL,
    NULL,       NULL,       NULL,       NULL,
    NULL,       NULL,       NULL,       NULL,
    NULL,       NULL,       NULL,       NULL,

    /* 040 - 057 */
    NULL,       miscop_041, NULL,       NULL,
    NULL,       NULL,       NULL,       NULL,
    miscop_050, NULL,       NULL,       NULL,
    miscop_054, miscop_055, NULL,       miscop_057,

    /* 060 - 077 */
    NULL,       NULL,       miscop_062, miscop_063,
    NULL,       NULL,       NULL,       miscop_067,
    miscop_070, NULL,       NULL,       NULL,
    NULL,       NULL,       miscop_076, miscop_077,


    /* 100 - 117 */
    NULL,       NULL,       NULL,       NULL,
    NULL,       NULL,       NULL,       NULL,
    NULL,       NULL,       NULL,       NULL,
    NULL,       NULL,       NULL,       NULL,

    /* 120 - 137 */
    NULL,       NULL,       NULL,       NULL,
    NULL,       NULL,       NULL,       NULL,
    NULL,       NULL,       NULL,       NULL,
    NULL,       NULL,       NULL,       NULL,

    /* 140 - 157 */
    NULL,       NULL,       NULL,       NULL,
    NULL,       NULL,       NULL,       NULL,
    NULL,       NULL,       NULL,       NULL,
    NULL,       NULL,       NULL,       NULL,

    /* 160 - 177 */
    NULL,       NULL,       NULL,       NULL,
    NULL,       NULL,       NULL,       NULL,
    NULL,       NULL,       NULL,       NULL,
    NULL,       NULL,       NULL,       NULL,


    /* 200 - 217 */
    miscop_200, NULL,       NULL,       NULL,
    NULL,       NULL,       NULL,       NULL,
    miscop_210, miscop_211, NULL,       NULL,
    NULL,       NULL,       NULL,       miscop_217,

    /* 220 - 237 */
    miscop_220, NULL,       NULL,       NULL,
    NULL,       NULL,       NULL,       miscop_227,
    NULL,       NULL,       miscop_232, miscop_233,
    NULL,       NULL,       NULL,       miscop_237,

    /* 240 - 257 */
    miscop_240, NULL,       NULL,       NULL,
    NULL,       NULL,       NULL,       NULL,
    NULL,       NULL,       miscop_252, NULL,
    miscop_254, NULL,       NULL,       NULL,

    /* 260 - 277 */
    NULL,       miscop_261, NULL,       NULL,
    NULL,       NULL,       NULL,       NULL,
    NULL,       NULL,       NULL,       NULL,
    NULL,       NULL,       NULL,       NULL,


    /* 300 - 317 */
    miscop_300, NULL,       miscop_302, NULL,
    NULL,       NULL,       NULL,       NULL,
    NULL,       NULL,       NULL,       NULL,
    NULL,       NULL,       NULL,       NULL,

    /* 320 - 337 */
    NULL,       NULL,       NULL,       NULL,
    NULL,       NULL,       NULL,       NULL,
    miscop_330, miscop_331, miscop_332, NULL,
    NULL,       miscop_335, NULL,       NULL,

    /* 340 - 357 */
    NULL,       NULL,       NULL,       NULL,
    miscop_344, NULL,       miscop_346, miscop_347,
    NULL,       NULL,       NULL,       NULL,
    NULL,       NULL,       NULL,       NULL,

    /* 360 - 377 */
    miscop_360, NULL,       NULL,       NULL,
    NULL,       NULL,       NULL,       NULL,
    NULL,       NULL,       NULL,       NULL,
    NULL,       NULL,       NULL,       NULL,


    /* 400 - 417 */
    NULL,       miscop_401, miscop_402, NULL,
    NULL,       miscop_405, NULL,       NULL,
    NULL,       NULL,       miscop_412, NULL,
    NULL,       miscop_415, miscop_416, NULL,

    /* 420 - 437 */
    NULL,       NULL,       NULL,       NULL,
    NULL,       NULL,       NULL,       NULL,
    miscop_430, NULL,       NULL,       NULL,
    miscop_434, miscop_435, miscop_436, NULL,

    /* 440 - 457 */
    miscop_440, NULL,       NULL,       NULL,
    NULL,       miscop_445, miscop_446, miscop_447,
    miscop_450, miscop_451, miscop_452, NULL,
    miscop_454, NULL,       NULL,       miscop_457,

    /* 460 - 477 */
    NULL,       miscop_461, NULL,       miscop_463,
    NULL,       NULL,       NULL,       NULL,
    NULL,       NULL,       NULL,       NULL,
    miscop_474, NULL,       NULL,       NULL,


    /* 500 - 517 */
    miscop_500, miscop_501, NULL,       NULL,
    miscop_504, NULL,       miscop_506, NULL,
    NULL,       NULL,       NULL,       NULL,
    miscop_514, miscop_515, NULL,       NULL,

    /* 520 - 537 */
    miscop_520, NULL,       NULL,       NULL,
    miscop_524, miscop_525, NULL,       NULL,
    miscop_530, miscop_531, NULL,       NULL,
    miscop_534, miscop_535, miscop_536, NULL,

    /* 540 - 557 */
    NULL,       NULL,       miscop_542, NULL,
    NULL,       NULL,       NULL,       NULL,
    miscop_550, miscop_551, miscop_552, NULL,
    miscop_554, miscop_555, miscop_556, NULL,

    /* 560 - 577 */
    miscop_560, NULL,       miscop_562, NULL,
    NULL,       miscop_565, miscop_566, NULL,
    miscop_570, miscop_571, miscop_572, NULL,
    NULL,       NULL,       NULL,       miscop_577,


    /* 600 - 617 */
    miscop_600, NULL,       NULL,       NULL,
    NULL,       NULL,       NULL,       NULL,
    miscop_610, miscop_611, miscop_612, miscop_613,
    miscop_614, NULL,       miscop_616, NULL,

    /* 620 - 637 */
    NULL,       NULL,       NULL,       NULL,
    NULL,       NULL,       miscop_626, miscop_627,
    NULL,       miscop_631, miscop_632, miscop_633,
    miscop_634, miscop_635, miscop_636, miscop_637,

    /* 640 - 657 */
    miscop_640, NULL,       NULL,       NULL,
    NULL,       miscop_645, NULL,       NULL,
    NULL,       NULL,       NULL,       NULL,
    NULL,       NULL,       NULL,       NULL,

    /* 660 - 677 */
    miscop_660, NULL,       miscop_662, NULL,
    miscop_664, miscop_665, NULL,       NULL,
    miscop_670, miscop_671, NULL,       NULL,
    NULL,       NULL,       NULL,       NULL,


    /* 700 - 717 */
    miscop_700, miscop_701, NULL,       miscop_703,
    miscop_704, NULL,       NULL,       NULL,
    NULL,       NULL,       NULL,       NULL,
    NULL,       NULL,       NULL,       NULL,

    /* 720 - 737 */
    NULL,       NULL,       NULL,       NULL,
    NULL,       NULL,       NULL,       NULL,
    NULL,       NULL,       NULL,       NULL,
    NULL,       NULL,       NULL,       NULL,

    /* 740 - 757 */
    NULL,       NULL,       NULL,       NULL,
    NULL,       NULL,       NULL,       NULL,
    NULL,       NULL,       NULL,       NULL,
    NULL,       NULL,       NULL,       NULL,

    /* 760 - 777 */
    NULL,       NULL,       NULL,       NULL,
    NULL,       NULL,       NULL,       NULL,
    NULL,       NULL,       NULL,       NULL,
    NULL,       NULL,       NULL,       NULL,
};

int step_miscop(u16 opcode)
{
    u16 opnum;
    int retval;

    opnum = opcode & 0x1ff;

    retval = 1;

    if (miscop_dispatch[opnum]) {
	retval = miscop_dispatch[opnum](opcode, opnum);
    } else {
	printf("Unknown miscop.\n");
	return 0;
    }

    if (!retval) return 0;

    if (opcode & 0x4000) {
	push_cdrnext(destination);
    }

    context.indicators = destination;

    return 1;
}

/* EOF */
