;;; -*- Mode:Common-Lisp; Package:SYSTEM-INTERNALS; Base:8; Fonts:(CPTFONT HL10B HL12BI HL12B CPTFONTB)1; *Patch-File:T -*-

;1;;                     RESTRICTED RIGHTS LEGEND          *
;1;; Use, duplication, or disclosure by the Government is subject to*
;1;; restrictions as set forth in subdivision (b)(3)(ii) of the Rights in*
;1;; Technical Data and Computer Software clause at 52.227-7013.*
;1;;                   TEXAS INSTRUMENTS INCORPORATED.*
;1;;                            P.O. BOX 149149*
;1;;                         AUSTIN, TEXAS 78714-9149*
;1;;                             MS 2151*
;1;; Copyright (C) 1986,1987 Texas Instruments Incorporated. All rights reserved.*
;1;; Copyright (C) 1984-1989 Texas Instruments Incorporated. All rights reserved.*

;;; 04-24-89 DAB Added all function, variables, flavors and instances that are documented to export list.

;1;; Ultra-simple stepper for lisp-machine.*
;1;; Wins with multiple values*
;1;; Does not attempt to win with editor top level*
;1;; Compile with QC-FILE.*

;1 NOTES:*
;1 The way it decides whether it needs to reprint the form when showing*
;1 you the values is pretty kludgey right now.  Can it check the cursorpos*
;1 or ask itself whether it typed anything or something?*
;
;1 Would like to be able to evaluate and/or substitute in atoms and forms*
;1 without having to break first.*
;
;1 Would like to be able to type A and have it stop after evaluating the*
;1 args, before calling the function.*
;
;1 Raid registers*
;
;1 Hook up to DDT?*
;1 *
;1 If an error happens, user should be able to throw back into the stepper.*

(DEFVAR 4*step-level** nil "2Depth within STEP-EVAL, minus one, within call to STEP*")
(DEFVAR 4*step-array** nil "2Holds forms to evaluate, indexed by *STEP-LEVEL* value.*")
(DEFVAR 4*step-apply-p-array** nil "2Holds the APPLY-P flag for each level, indexed by *STEP-LEVEL* value.*")
(DEFVAR 4*step-max** nil "2Do not tell user about evaluations with *STEP-LEVEL* deeper than this.*")
(DEFVAR 4*step-form** nil "2Form to be or just evaluated, in STEP command loop*")
(DEFVAR 4*step-value** nil "2First value just computed.  May be changed in a breakpoint.*")
(DEFVAR 4*step-values** nil "2List of values just computed.  May be changed in a breakpoint.*")

(PROCLAIM '(SPECIAL step-form step-value step-values))
(FORWARD-VALUE-CELL 'step-form '*step-form*)
(FORWARD-VALUE-CELL 'step-value '*step-value*)
(FORWARD-VALUE-CELL 'step-values '*step-values*)

(defvar	 4*step-auto** nil)     ;1; If non NIL, simulate cntrl-n at step-cmdr.*
			       ;1; normal printout produced unless NO-PRINT.*
			       ;1; User's program can turn on auto mode by*
			       ;1; (si:step-auto-on &optional (mode 'no-print))*
			       ;1; and (si:step-auto-off) to reable stepping.*

(DEFVAR 4*ucl-stepper** nil "2Holds the currently executing instance of stepper.*")

(DEFVAR 4*step-function-not-recursive** t)   ;1; Is this the first level call to this function?*

;1;; NEW SUGGESTIONS!!!*
(sys:declare-suggestions-for 'si::step-eval 
			     :around (sys::make-default-around-form
				       si::step-eval single-stepper-top-menu
				       single-stepper-bottom-menu
				       single-stepper-commands-menu))

(DEFFLAVOR 4ucl-step*
	   ((apply-p)
	    (*step-max*)
	    (*step-level*)
	    ;1;;(*step-auto*)*
	    (form)
	    (VALUES)
	    (*step-array*)
	    (*step-apply-p-array*)
	    (*step-form*)
	    (*step-value*)
	    (*step-values*)
            name                                   ;1; NEW*
	    value-to-return)
	   (ucl:basic-command-loop)
  :gettable-instance-variables                     ;1; NEW*
  :settable-instance-variables
  (:special-instance-variables
    apply-p *step-max* *step-level* form values *step-array* *step-apply-p-array* 
    *step-form* *step-value* *step-values*)
  (:default-init-plist
    :active-command-tables '(step-cmd-table)
    :all-command-tables '(step-cmd-table)
    :menu-panes  nil
    :typein-handler :handle-typein-input
    :name        "3Step*"                            ;1; NEW*
    :basic-help '(documentation-cmd)))

(DEFRESOURCE 4ucl-step* ()
  :constructor (MAKE-INSTANCE 'ucl-step)
  :initial-copies 0)

;1; Redefine this function to call the ucl command loop.*
;1; print whatever is necessary, read a command, set special variables*
;1; and return how to proceed:  *eval (just *eval), evalhook (recurse), more options later.*
;1; If calling for *eval, *step-values* is nil, otherwise calling for return.*
(DEFUN 4step-cmdr* (form values print-form-p &optional apply-p)
  (DECLARE (SPECIAL apply-p *step-max* *step-level*  form values))
  (BLOCK ()
    (IF *step-auto*
       (IF (EQUAL *step-auto* :no-print) ;1; Change no-print to keyword to avoid package conflict*
	  (PROGN
	    (SETQ *step-max* (1+ *step-level*))
	    (RETURN 'EVALHOOK)) ;1; Add else clause to handle *step-auto* being non-nil but not*
	  (SEND tv:selected-window :force-kbd-input #\c-n))) ;1; eq to :no-print*
    (AND print-form-p (step-print-form form *step-level* apply-p))
    (print-values)
    ;1; The following code has been reworked to make the stepper run in its own temporary stack group, *
    ;1; so that its UCL instance variables are not bound when user evaluates a form.  This is particularly*
    ;1; important if the user is stepping through a UCL application, where he expects to find*
    ;1; *his* instance vars bound.  The three tricks below are a mechanism of communication between*
    ;1; two stack groups--the current one in which this function is called, and the temporary stack group*
    ;1; which is used (in UCL:IN-OWN-STACK-GROUP) to execute the stepper command loop.  In the current*
    ;1; sg, we have bound a group of special vars.  Trick#1 gives the stepper sg access to these *
    ;1; variables' current values.  Trick#2 makes sure the current sg gets any modifications the stepper *
    ;1; sg (stepper commands) make to the vars.*
    (USING-RESOURCE (*ucl-stepper* ucl-step)
		    ;1; Trick #1*
       (DOLIST (ivar
	 '(apply-p *step-max* *step-level*  form values *step-array* *step-apply-p-array*
	   *step-form* *step-value* *step-values*))
	 (SET-IN-INSTANCE *ucl-stepper* ivar (SYMBOL-VALUE ivar)))
       ;1; Trick#3: UCL:IN-OWN-STACK-GROUP can not return any values, so we*
       ;1; return a value through an instance variable.*
       (ucl::in-own-stack-group (*ucl-stepper*) ()
	  (SEND *ucl-stepper* :set-value-to-return (SEND *ucl-stepper* :command-loop)))
       ;1; Trick#2: some of these vars may have been modified during the command loop.*
       ;1; set them to the values of the corresponding instance variables.*
       (DOLIST (special-var
	 '(apply-p *step-max* *step-level* form values *step-array* *step-apply-p-array*
	   *step-form* *step-value* *step-values*))
	 (SET special-var (SYMEVAL-IN-INSTANCE *ucl-stepper* special-var)))
       (LET ((ret-val (SEND *ucl-stepper* :value-to-return))) ;1; If return value is 'exiting due to quit-cmd, then throw to*
	 (IF (EQ ret-val 'exiting)                            ;1; top level of step, otherwise continue*
              (THROW 'exit-stepper ())
	    (RETURN ret-val))))))       

(DEFMETHOD 4(ucl-step :around :fetch-and-execute*) (cont mt ignore)
  "2Intercept a ucl method to break out of the command loop for some of the commands.
   This is necessary due to the nature of the implementation of the stepper.
   It sets a var in eval as a hook to regain control of the evaluation.*"
  (CATCH 'ucl::command-abort
    (LET ((tv:kbd-intercepted-characters ;1; Let this code handle the break key.*
	   (REMOVE (ASSOC #\Break tv:kbd-intercepted-characters :test #'EQ)
		   (THE list tv:kbd-intercepted-characters) :test #'EQUAL)))
      (FUNCALL-WITH-MAPPING-TABLE cont mt :fetch-and-execute)
      (WHEN (AND (NOT ucl::preempting?) ucl::command-entry (TYPEP ucl::command-entry 'ucl::command)
		 ;1; Several stepper commands are given this property in their DEFCOMMANDs.*
	  (SEND ucl::command-entry :get 'exit-loop?))
	(THROW 'ucl::exit-command-loop
	       (CAR
		/)))))) 

(DEFMETHOD 4(ucl-step :around :handle-typein-input*) (cont mt ignore &optional (untyi-first-char? t))
  "2Do the lisp prompt and afterwards output the stepper forms for Lisp typein.*"
  (DECLARE (SPECIAL *step-array* *step-level* *step-apply-p-array*))
  (FORMAT t "3    Eval: *")
  (UNWIND-PROTECT (FUNCALL-WITH-MAPPING-TABLE
		   cont
		   mt
		   :handle-typein-input
		   untyi-first-char?)
    (DO ((i (MAX 0 (- *step-level* 10)) (1+ i)))
	((> i *step-level*)
	 nil)
      (step-print-form (AREF *step-array* i) i (AREF *step-apply-p-array* i)))
    (print-values))) 

(DEFMETHOD 4(ucl-step :handle-prompt*) ()
  "2Redefine UCL method to do nothing so that a prompt is not printed.*" nil)

(DEFMETHOD 4(ucl-step :designate-io-streams*) ()
  "2Redefine a ucl method to set up the correct bindings.*"
  (SETQ *standard-input* *terminal-io*
        *standard-output* *terminal-io*))

(DEFCOMMAND 4next-thing-cmd* ()
  '(:description "3Proceed to next thing evaled.*"
    :names ("3Next thing*")
    :keys (#\Control-n)
    :property-list (exit-loop? t))
  (DECLARE (SPECIAL *step-max* *step-level*))
  (SETQ *step-max* (1+ *step-level*))
  'EVALHOOK)

(DEFCOMMAND 4next-this-level-cmd* () 
  '(:description "3Proceed to next thing evaled at same level.*"
    :names ("3Next this level*") 
    :keys (#\Space)
    :property-list (exit-loop? t :cant-preempt t)) ;1;Keep space from preempting type-in.*
  (DECLARE (SPECIAL *step-max* *step-level*))
  (SETQ *step-max* *step-level*)
  '*eval)    ;1eval1 in rel 2*

(DEFCOMMAND 4eval-args-cmd* ()
   '(:description "3Eval the args without stepping; stop before applying the function.*"
     :names ("3Eval args*")
     :keys (#\c-a)
     :property-list (exit-loop? t))
   (DECLARE (SPECIAL apply-p *step-max* *step-level*))
   (IF (NOT apply-p)
       (PROGN (SETQ *step-max* (1+ *step-level*))
              'APPLYHOOK)
       (PROGN (BEEP)
              (THROW 'ucl::command-abort t)))) 

(DEFCOMMAND 4proceed-up-cmd* ()
  '(:description "3Proceed to first thing up one level.*"
    :names ("3Proceed up*")
    :keys (#\Control-u)
    :property-list (exit-loop? t))
  (DECLARE (SPECIAL *step-max* *step-level*))
  (SETQ *step-max* (MAX 0 (1- *step-level*)))
  '*eval)   ;1; eval1 in rel 2*

(DEFCOMMAND 4execute-no-stepping-cmd* ()
  '(:description "3Continue executing without further stepping.*"
    :names ("3Exit Stepper*")
    :keys (#\End #\Control-x)
    :property-list (exit-loop? t))
  (DECLARE (SPECIAL *step-max*))
  (SETQ *step-max* -1)
  '*eval)   ;1; eval1 in rel 2*

(DEFCOMMAND 4quit-cmd* ()
   '(:description "3Throw out of Stepper without further execution of program.*" :names
     ("3Abort Program*") :keys (#\s-end))
   (THROW  'ucl::exit-command-loop
	  'exiting))                           ;1Throw 'exiting so we can check at next level up*

(DEFCOMMAND 4escape-to-editor-cmd* ()
   '(:description "3Escape to editor.*" :names ("3Escape to editor*") :keys (#\c-e))
   (DECLARE (SPECIAL *step-level* *step-apply-p-array* *step-array* *ucl-stepper*)) (ED)
   (SEND *standard-output* :clear-screen)
   (DO ((i (MAX 0 (- *step-level* 10)) (1+ i)))
       ((> i *step-level*)
	nil)
     (step-print-form (AREF *step-array* i) i (AREF *step-apply-p-array* i)))
   (print-values)) 

(DEFCOMMAND 4retype-cmd* ()
  '(:description "3Retype current form in full.*"
    :names ("3Retype*")
    :keys (#\C-t))
  (DECLARE (SPECIAL values form *ucl-stepper*))  
  (IF (NULL values) (PRINT form)
    (DOLIST (value values) (PRINT value))))

(DEFCOMMAND 4grind-cmd* ()
  '(:description "3Grind current form.*"
    :names ("3Grind*")
    :keys (#\C-g))
  (DECLARE (SPECIAL values form *ucl-stepper*))
  (IF (NULL values) (GRIND-TOP-LEVEL form)
    (DOLIST (value values) (GRIND-TOP-LEVEL value))))

(DEFCOMMAND 4breakpoint-cmd* ()
   '(:description "3Enter breakpoint with STEP-FORM, STEP-VALUE and STEP-VALUES bound.*"
     :names ("3Breakpoint*")
     :keys (#\Break #\c-b))
   (DECLARE (SPECIAL *step-form* *step-array* *step-level* apply-p *step-apply-p-array* *ucl-stepper*))
   (UNWIND-PROTECT (PROGN
		    (BREAK "3step - with *STEP-FORM*, *STEP-VALUE* and *STEP-VALUES* bound.*" t)
		    (SETF (AREF *step-array* *step-level*) *step-form*)
		    (SETF (AREF *step-apply-p-array* *step-level*) apply-p))
     (DO ((i (MAX 0 (- *step-level* 10)) (1+ i)))
	 ((> i *step-level*)
	  nil)
       (step-print-form (AREF *step-array* i) i (AREF *step-apply-p-array* i)))
     (print-values))) 
  
(DEFCOMMAND 4clear-and-list-cmd* ()
   '(:description "3Clear screen then list last ten forms.*" :names ("3Clear and list*") :keys
     (#\Page #\c-l))
   (DECLARE (SPECIAL *step-level* *step-apply-p-array* *step-array* *ucl-stepper*))
   (SEND *standard-output* :clear-screen)
   (DO ((i (MAX 0 (- *step-level* 10)) (1+ i)))
       ((> i *step-level*)
	nil)
     (step-print-form (AREF *step-array* i) i (AREF *step-apply-p-array* i)))
   (print-values)) 

(DEFCOMMAND 4list-ten-forms-cmd* ()
   '(:description "3List last ten forms.*" :names ("3List ten*") :keys (#\m-l))
   (DECLARE (SPECIAL *step-level* *step-apply-p-array* *step-array* *ucl-stepper*))
   (DO ((i (MAX 0 (- *step-level* 10)) (1+ i)))
       ((> i *step-level*)
	nil)
     (step-print-form (AREF *step-array* i) i (AREF *step-apply-p-array* i)))
   (print-values)) 

(DEFCOMMAND 4clear-list-all-cmd* ()
   '(:description "3Clear and list all forms.*" :names ("3Clear list all*") :keys (#\c-m-l))
   (DECLARE (SPECIAL *step-level* *step-apply-p-array* *step-array* *ucl-stepper*))
   (SEND *standard-output* :clear-screen)
   (DO ((i 0 (1+ i)))
       ((> i *step-level*)
	nil)
     (step-print-form (AREF *step-array* i) i (AREF *step-apply-p-array* i)))
   (print-values)) 

(DEFCOMMAND 4documentation-cmd* ()
   '(:description "3Describe what is about to be done next.*" :names ("3Documentation*") :keys
     (#\c-help #\m-help))
   (DECLARE (SPECIAL apply-p *step-values* *ucl-stepper*)) (TERPRI)
   (PRINC
    (COND
      ((NULL *step-values*)
       (IF apply-p "3You are about to apply the above function to the above arguments.*"
	  "3You are about to evaluate the above form.*"))
      (t
       (IF apply-p
	  "3You have applied a function to arguments and are about to return the above values.*"
	  "3You have evaluated a form and are about to return the above values.*"))))
   (TERPRI)
   (PRINC
    "3Magic flags preceding output:

       Ordinary LISP form
       About to apply a function
       Macro
       Values
       Separates multiple values

    For the UCL command display press HYPER-CONTROL-HELP.*")
   (TERPRI)
   (DO ((i (MAX 0 (- *step-level* 10)) (1+ i)))
       ((> i *step-level*)
	nil)
     (step-print-form (AREF *step-array* i) i (AREF *step-apply-p-array* i)))
   (print-values)) 

(DEFUN 4print-values* ()
  (DECLARE (SPECIAL values))
  (DO ((l values (CDR l))
       (ch #\ #\))
      ((NULL l))
    (terpri-if-insufficient-space 120)
    (WRITE-CHAR #\Space)
    (WRITE-CHAR (INT-CHAR ch))
    (WRITE-CHAR #\Space)
    (print-truncated (CAR l) 142))) 	   ;1; Several windows lose if this is 100.*


;1Make the UCL command table.*

(BUILD-COMMAND-TABLE 'step-cmd-table 'ucl-step
  '(next-thing-cmd
    next-this-level-cmd
    eval-args-cmd
    proceed-up-cmd
    execute-no-stepping-cmd
    quit-cmd
    escape-to-editor-cmd
    retype-cmd
    grind-cmd
    breakpoint-cmd
    clear-and-list-cmd
    list-ten-forms-cmd
    clear-list-all-cmd
    documentation-cmd)
  :init-options
  '(:name "3Step Command Table*"
    :documentation "3The Step commands.*"))


(COMPILE-FLAVOR-METHODS ucl-step)


;1; Do these step-* vars need to be declared special?     ;!*
;1; 9/13/88 clm - fixed two misspellings in doc string and added compiler-let*
;1; form to prevent redefinition warnings.*
(COMPILER-LET ((local-declarations '((:expr-sxhash 6217301.))))	
(DEFMACRO 4step* (form &optional *step-auto*)
  "2Allows the user to single step an evaluated function or functions.*"
  `(LET ((*step-level* -1)
	 (*step-max* 0)
	 (*step-array* (MAKE-ARRAY 200))
	 (*step-apply-p-array* (MAKE-ARRAY 200)))
	 ;1; (*step-auto* ,*step-auto*))*
     (WITH-STACK-LIST (environment si::*interpreter-environment*           ;1! allow access to locally bound variables*
                                   si::*interpreter-function-environment*  ;1! allow access to locally bound functions*
                                   nil) 
      (CATCH 'exit-stepper     ;1; Upper level catch from step-cmdr*
       (step-eval (QUOTE ,form) environment)))))
)

;1This is for TRACE, mainly.  The idea is to do an apply,*
;1stepping under it but not showing the user the apply itself.*
(DEFUN 4step-apply* (fcn args &aux (*evalhook* #'step-hook))
  (CATCH-CONTINUATION-IF *step-function-not-recursive* 'exit-stepper nil nil  ;1; Only set label to "EXIT-STEPPER" the first call.*
    (UNWIND-PROTECT 
        (PROGN (SETQ *step-function-not-recursive* nil)
               (IF (ATOM fcn) (FORMAT t "3~% Warning: Cannot step inside function, ~S~%*" fcn))
               (APPLY fcn args))
      (SETQ *step-function-not-recursive* t))))
         
;1; Main entry point.*
(DEFUN 4step-hook* (form &optional env &aux (*step-level* -1) (*step-max* 0)
		  ;1; *step-auto* *
		  (*step-array* (MAKE-ARRAY 200))
		  (*step-apply-p-array* (MAKE-ARRAY 200)))
  (step-eval form env))
  
;1(defun step-hook (form &optional env &aux (*step-auto* t) (*step-level* -1) (*step-max* 0)*
;		1  (*step-array* (make-array 200))*
;		1  (*step-apply-p-array* (make-array 200)))*
;1  (step-eval form env))*
  
;1Check for macros, they are treated specially.*
(DEFUN 4step-macro-p* (form)
  (WHEN (CONSP form)
    (LET ((defn (si:fdefinition-safe (CAR form))))
      (EQ (CAR-SAFE defn) 'MACRO))))

;1(DEFUN STEP-MACRO-P (FORM)*
;1  (AND (CONSP FORM)*
;1       (SYMBOLP (CAR FORM))*
;1       (FBOUNDP (CAR FORM))*
;1     (CONSP (SETQ FORM (SYMBOL-FUNCTION (CAR FORM)))) (EQ (CAR FORM) 'MACRO))) *

(DEFUN 4step-auto-on* (&optional (mode t))   ;1change no-print to keyword*
  "2The STEP-AUTO-ON function allows the stepper to automatically step through
   the routine. This option remains in effect until specificaly turned off.*"
  (SETQ *step-auto* mode))

(DEFUN 4step-auto-off* ()
  "2Disables automatic stepping.*"
  (SETQ *step-auto* nil))

;1; Print a form, suitably indented, marked, and truncated to one line.*
(DEFUN 4step-print-form* (form level apply-p)
  (TERPRI)
  (DO ((n (* 2 level) (1- n)))
      ((= n 0)
       nil)
    (WRITE-CHAR #\Space))
  (WRITE-CHAR (INT-CHAR (COND
			  (apply-p #\)
			  ((step-macro-p form) #\)
			  (t #\))))
  (WRITE-CHAR #\Space)
  (IF apply-p
     (PROGN
       (print-truncated (FUNCTION-NAME (CAR form)) 113)
       (PRINC "3: *")
       (print-elements-truncated (CDR form) 132 113))
     (print-truncated form 113)))  



;1; This is evalhooked in in place of EVAL.  Works by calling step-cmdr*
;1; to let the user see what's going on and say what to do, then continues*
;1; evaluation using either EVAL or EVALHOOK based on what the user typed.*
;1; Has special hair for macros and for atoms.*

(DEFUN 4step-eval* (*step-form* &optional environment)
  (sys:with-suggestions-menus-for si::step-eval     ;1; NEW SUGGESTIONS!!!*
   (PROG ((*step-level* (1+ *step-level*))
          *step-value*
          *step-values*
	  (ucl::*env* sys::environment)
          tem
          val)
         (WHEN (>= *step-level* (ARRAY-TOTAL-SIZE *step-array*))
           (ADJUST-ARRAY *step-array* (+ 100 *step-level*))
           (ADJUST-ARRAY *step-apply-p-array* (+ 100 *step-level*)))
      mc
         (SETF (AREF *step-array* *step-level*) *step-form*)
         (SETF (AREF *step-apply-p-array* *step-level*) nil)
         (COND
           ((ATOM *step-form*) (SETQ *step-values* (LIST (*eval *step-form*)))
                               (SETQ tem 'ATOM)
                               (GO rl))
           ((<= *step-level* *step-max*) (SETQ tem (step-cmdr *step-form* nil t)))
           (t (SETQ tem '*eval)))
         (COND
           ((step-macro-p *step-form*) (SETQ *step-form* (MACROEXPAND-1 *step-form*))
                                       (GO mc))
           ((EQ tem '*eval)
            (SETQ *step-values* (MULTIPLE-VALUE-LIST (EVALHOOK *step-form* nil nil environment))))
           ((EQ tem 'EVALHOOK)
            (SETQ *step-values* (MULTIPLE-VALUE-LIST (EVALHOOK *step-form* #'step-eval nil environment))))
           ((EQ tem 'APPLYHOOK)
            (SETQ *step-values*
                  (MULTIPLE-VALUE-LIST (EVALHOOK *step-form* nil #'step-applyhook environment))))
           ((FERROR () "3Unknown function ~S*" tem)))
      rl
         (SETQ *step-value* (SETQ val (CAR *step-values*)))
         (COND ((<= *step-level* *step-max*)
                (SETQ tem (step-cmdr *step-form* *step-values* (NEQ tem '*eval))))    ;1; 2 eval1*
               (t (SETQ tem '*eval))) 
         (AND (NEQ *step-value* val) (RETURN *step-value*))
         (COND ((NULL (CDR *step-values*))
                (RETURN (CAR *step-values*)))
               (t (RETURN (VALUES-LIST *step-values*)))))))
;1    RT*
;1    (COND*
;1      ((NULL (CDR *STEP-VALUES*))*
;1       (RETURN (CAR *STEP-VALUES*)))*
;1      (T (RETURN-NEXT-VALUE (CAR *STEP-VALUES*))*
;1         (SETQ *STEP-VALUES* (CDR *STEP-VALUES*))*
;1         (GO RT))))) *

(DEFUN 4step-applyhook* (function args &optional environment &aux
			 *interpreter-environment* *interpreter-function-environment*
			 (*step-form* (CONS function args)))
  environment ;1; so compiler doesn't complain *
  (PROG ((*step-level* (1+ *step-level*))
	 *step-value*
	 *step-values*
	 tem
	 val)
    (WHEN (>= *step-level* (ARRAY-TOTAL-SIZE *step-array*))
      (ADJUST-ARRAY *step-array* (+ 100 *step-level*))
      (ADJUST-ARRAY *step-apply-p-array* (+ 100 *step-level*)))
    mc
    (SETF (AREF *step-array* *step-level*) *step-form*)
    (SETF (AREF *step-apply-p-array* *step-level*) t)
    (COND
      ((<= *step-level* *step-max*) (SETQ tem (step-cmdr *step-form* nil t t)))
      (t (SETQ tem '*eval)))
    (COND
      ((EQ tem '*eval)
       (SETQ *step-values* (MULTIPLE-VALUE-LIST (APPLY (CAR *step-form*) (CDR *step-form*)))))
      ((EQ tem 'EVALHOOK)
       (SETQ *step-values*
	     (MULTIPLE-VALUE-LIST
	      (LET ((*evalhook* #'step-eval))
		(APPLY (CAR *step-form*) (CDR *step-form*))))))
      ((FERROR () "3Unknown function ~S*" tem)))
    rl
    (SETQ *step-value* (SETQ val (CAR *step-values*)))
    (COND ((<= *step-level* *step-max*)
           (SETQ tem (step-cmdr *step-form* *step-values* (NEQ tem '*eval) t)))
          (t (SETQ tem '*eval)))
    (AND (NEQ *step-value* val) (RETURN *step-value*))
    (COND ((NULL (CDR *step-values*))
           (RETURN (CAR *step-values*)))
          (t (RETURN (VALUES-LIST *step-values*))))))
;1    RT*
;1    (COND ((NULL (CDR *STEP-VALUES*))*
;1           (RETURN (CAR *STEP-VALUES*)))*
;1          (T (RETURN-NEXT-VALUE (CAR *STEP-VALUES*))*
;1             (SETQ *STEP-VALUES* (CDR *STEP-VALUES*))*
;1             (GO RT))))) *

;1; PRINT abbreviated spacewise rather than listwise*

(DEFVAR 4print-truncated)* ;1YECH*

(DEFUN 4terpri-if-insufficient-space* (percent-width)
  (LET ((x (TRUNCATE (* percent-width (FUNCALL *standard-output* :inside-size)) 144)))
    (AND (>= (FUNCALL *standard-output* :read-cursorpos :pixel) x) (TERPRI)))) 

(DEFUN 4print-truncated* (sexp percent-width)
  (LET ((print-truncated (TRUNCATE (* percent-width (FUNCALL *standard-output* :inside-size)) 144)))
    (CATCH 'print-truncated
      (PRIN1 sexp (CLOSURE '(print-truncated *standard-output*) #'print-truncated-stream))))) 

(DEFUN 4print-elements-truncated* (LIST truncation-percent-width terpri-percent-width)
  (DOLIST (element list)
    (terpri-if-insufficient-space terpri-percent-width)
    (print-truncated element truncation-percent-width)
    (WRITE-CHAR #\Space))) 

(DEFUN 4print-truncated-stream* (op &optional arg1 &rest rest)
  (CASE op
    (:tyo
     (COND
       ((>= (FUNCALL *standard-output* :read-cursorpos :pixel) print-truncated)
	(THROW 'print-truncated
	       ()))
       (t (FUNCALL *standard-output* :tyo arg1))))
    (:which-operations '(:tyo))
    (otherwise (STREAM-DEFAULT-HANDLER 'print-truncated-stream op arg1 rest)))) 



(EXPORT '(sys:*step-auto*           ;1variable*
	   sys:step-auto-on         ;1function*
	   sys:step-auto-off        ;1function*
	   )
	'sys)