;LISP machine character level I/O stuff -*- Mode:Common-Lisp; Package:SI; Cold-load:T; Base:10; Fonts:(CPTFONT HL12B) -*-

;1;;                           RESTRICTED RIGHTS LEGEND*

;1;;Use, duplication, or disclosure by the Government is subject to*
;1;;restrictions as set forth in subdivision (c)(1)(ii) of the Rights in*
;1;;Technical Data and Computer Software clause at 52.227-7013.*
;1;;*
;1;;                     TEXAS INSTRUMENTS INCORPORATED.*
;1;;                              P.O. BOX 2909*
;1;;                           AUSTIN, TEXAS 78769*
;1;;                                 MS 2151*
;1;;*
;1;; Copyright (C) 1985-1989 Texas Instruments Incorporated.  All rights reserved.*
;1 ** (c) Copyright 1980 Massachusetts Institute of Technology ***


;1;;*
;1;; Change history:*
;1;;*
;1;;  Date      Author*	1Description*
;1;; -------------------------------------------------------------------------------------*
;1;;11/13/86    TWE*	1Fixed up the stream-default-handler, eval-read-prompt-and-read and*
;1;;*			1expression-or-end-prompt-and-read to correctly compare character*
;1;;*			1objects to characters read via :tyi.*
;1;; 7/29/87     AB      o Fix NULL-STREAM to return 0 on :READ-CURSORPOS or :READ-POINTER;*
;1;;                     and return T on :SET-CURSORPOS. [SPR 5891]*
;1;;                     o Fix :LINE-IN operation of STREAM-DEFAULT-HANDLER always to cons *
;1;;                     in BACKGROUND-CONS-AREA and not to write into deallocated readstrings *
;1;;                     (in fact, don't use the readstring pool at all).  [SPR 5985]*
;1;; 7/29/87     AB      o Fix STREAMP for compiled functions which don't handle messages.  [SPR 2193]*
;1;; 8/12/87     PHD     Fixed READ-FROM-STRING-STREAM :tyi so it returns a fixnum, SPR #6198*
;1;; 1/04/88     RJF     Fixed Make-symonym-strean to check its argument better, SPR #7069*

(setq stream-input-operations '(:tyi :listen :untyi :line-in :rubout-handler)) 

(setq stream-output-operations
      '(:tyo :force-output :finish :string-out :line-out :fresh-line :untyo-mark :untyo)) 

;1 Naming conventions:*
;1   Symbols whose names end in "-INPUT", "-OUTPUT", or "-IO" should*
;1      normally be BOUND to streams; which of the three you use depends on*
;1      what directions the stream normally supports.*
;1   Symbols whose names end in "-STREAM" are DEFINED as streams.*


;1 Synonyms.*
;1 MAKE-SYN-STREAM takes a symbol, and returns a stream which will forward all operations*
;1   to the binding of the symbol.  After (SETQ BAR (MAKE-SYN-STREAM 'FOO)), one says*
;1   that BAR is SYNned to FOO.*


;1 The initial environment.*
;1   The initial binding of streams (set up by LISP-REINITIALIZE) is*
;1      as follows:*
;1   TERMINAL-IO     - This is how to get directly to the user's terminal.  It is set*
;1                     up to go to the TV initially.  Other places it might go are to*
;1                     the SUPDUP server, etc.  It is initially bound to a TV-MAKE-STREAM*
;1                     of CONSOLE-IO-PC-PPR.*
;1   *STANDARD-INPUT*  - This is initially bound to SYN to TERMINAL-IO.*
;1   *STANDARD-OUTPUT* - This is initially bound to SYN to TERMINAL-IO. *STANDARD-INPUT**
;1                     and *STANDARD-OUTPUT* are the default streams for READ, PRINT and*
;1                     other things.  *STANDARD-OUTPUT* gets hacked when the session is*
;1                     being scripted, for example.*
;1   *ERROR-OUTPUT*    - This is where error messages should eventually get sent. Initially*
;1                     SYNned to TERMINAL-IO.*
;1   *QUERY-IO*        - This is for unexpected user queries*
;1                     of the "Do you really want to ..." variety.  Initially SYNned to*
;1                     TERMINAL-IO.  It supersedes "QUERY-INPUT".*
;1   *TRACE-OUTPUT*    - Output produced by TRACE goes here.  Initially SYNned to *ERROR-OUTPUT*.*


(defvar rubout-handler :unbound "T while executing inside of a rubout handler.") 

;1;AB 7/29/87.  Fix for compiled functions which don't handle messages. [SPR 2193]*
(defun streamp (object)
  "Returns non-() if OBJECT is a stream.
This predicate considers the following to be streams:
 Any instance incorporating SI:STREAM or TV:SHEET
 Any function handling either :TYI or :TYO.
 Any symbol with a non-NIL SI:IO-STREAM-P property."
  (or
    (and (instancep object)
	 (or (typep-structure-or-flavor object 'stream) (typep object 'tv:sheet)))	;1 Explicit FUNCALLed things*
						;1that accept messages*
    (and (or (closurep object) (functionp object)) (arglist object t)
	 (let ((wo (IGNORE-ERRORS (send object :which-operations))))
	   (or (member :tyo wo :test #'eq) (member :tyi wo :test #'eq))))
    (and (symbolp object) (get object 'io-stream-p)))) 


(defun input-stream-p (stream)
  "T if STREAM, assumed to be a stream, supports input."
  (member (send stream :direction) '(:input :bidirectional) :test #'eq)) 


(defun output-stream-p (stream)
  "T if STREAM, assumed to be a stream, supports output."
  (member (send stream :direction) '(:output :bidirectional) :test #'eq)) 


(defun io-stream-p (x)
  "T if X is a plausible i/o stream.
It must be an insance, entity, closure, compiled function
or a symbol which has a non-NIL SI:IO-STREAM-P property."
  (typecase x
    ((or instance closure compiled-function) t)
    (symbol (get x 'io-stream-p))
    (t nil))) 

(defun stream-element-type (stream)
  "Return a Common Lisp type describing the objects input or output by STREAM.
This will be either CHARACTER or STRING-CHAR or a subtype of INTEGER."
  (or (send stream :send-if-handles :element-type)
      (if (send stream :characters)
	  'character
	  (let ((value (send stream :send-if-handles :byte-size)))
	    (if value
		`(unsigned-byte ,value)
		'fixnum))))) 

;1;; Given the 2 arguments to READ (or TYI or READCH or TYIPEEK or READLINE)*
;1;; in the form of a REST argument this returns the input stream and the eof option.*
;1;; Note that the first arg would rather be the stream than the eof option.*
;1;; This is set up for Maclisp compatibility.*
;1;; HOWEVER, if the second argument is NIL or unsupplied, the first is*
;1;; assumed to be a stream if that is plausible,*
;1;; which is not compatible with Maclisp but more winning.*
;1;; If the user didn't supply an eof-option, the second value returned will*
;1;; be the symbol NO-EOF-OPTION.*

(defun decode-read-args (arg-list)
  (case (length arg-list)
    (0 (values *standard-input* 'no-eof-option))
    (1
     (let ((arg1 (first arg-list)))
       (if (or (eq arg1 ()) (eq arg1 t) (io-stream-p arg1))
	   ;1; The arg is a plausible stream.*
	   (values (cond
		     ((eq arg1 ()) *standard-input*)
		     ((eq arg1 t) *terminal-io*)
		     (t arg1))
		   'no-eof-option)
	   ;1; It is not a stream and must be an EOF option.*
	   (values *standard-input* arg1))))
    (2
     (let ((arg1 (first arg-list))
	   (arg2 (second arg-list)))
       (cond
	 ((or (eq arg1 ()) (eq arg1 t) (io-stream-p arg1))
	  (values (cond
		    ((eq arg1 ()) *standard-input*)
		    ((eq arg1 t) *terminal-io*)
		    (t arg1))
		  arg2))
	 ((or (eq arg2 ()) (eq arg2 t) (io-stream-p arg2))
	  (values (cond
		    ((eq arg2 ()) *standard-input*)
		    ((eq arg2 t) *terminal-io*)
		    (t arg2))
		  arg1))
	 (t (values arg1 arg2)))))
    (otherwise
     (ferror () "Too many arguments were given to one of the READ-like functions: ~S" arg-list)))) 



(defun global:terpri (&optional stream)
  "Go to a new line on STREAM."
  (send (decode-print-arg stream) :tyo (CHAR-INT #\NEWLINE))
  t) 

(DEFUN TERPRI (&OPTIONAL STREAM)
  "Go to a new line on STREAM."
  (FUNCALL (DECODE-PRINT-ARG STREAM) :TYO (CHAR-INT #\RETURN))
  NIL)

(defun fresh-line (&optional stream)
  "Go to a new line on STREAM if not already at the beginning of one.
Returns T if a Return was output, NIL if nothing output."
  (send (decode-print-arg stream) :fresh-line)) 


(defun zlc:tyo (char &optional stream)
  "Output CHAR to STREAM."
  (send (decode-print-arg stream) :tyo (char-int char))
  char) 


(defun write-char (char &optional stream )
  "Output CHAR to STREAM.  Returns CHAR."
  (send (decode-print-arg stream) :tyo (char-int char))
  char)   


(defun write-byte (byte &optional stream)
  "Output BYTE to STREAM.  Returns BYTE."
  (send (decode-print-arg stream) :tyo byte)
  byte)   


(defun write-string (string &optional stream  &key &optional (start 0) end)
  "Output all or part of STRING to STREAM.
START and END are indices specifying the part.
START defaults to 0 and END to NIL (which means the end of STRING.)"
  (send (decode-print-arg stream) :string-out string start end)
  string) 


(defun write-line (string &optional stream  &key &optional (start 0) end)
  "Output all or part of STRING to STREAM, followed by a Return.
START and END are indices specifying the part.
START defaults to 0 and END to NIL (which means the end of STRING.)"
  (setq stream (decode-print-arg stream))
  (send stream :string-out string start end)
  (send stream :tyo #\NEWLINE)
  string) 


(defun force-output (&optional stream)
  "Force output buffers on STREAM to begin being transmitted immediately.
Useful on asynchronous streams such as the chaosnet, which normally
wait until a buffer is full before even starting to transmit."
  (send (decode-print-arg stream) :force-output)
  ()) 


(defun finish-output (&optional stream)
  "Wait until output buffers on STREAM are transmitted and processed completely.
For a file stream, this will not return until the data is recorded permanently
in the file system."
  (send (decode-print-arg stream) :finish)
  ()) 


(defun clear-output (&optional stream)
  "Discard buffer output buffers on STREAM, if it is an interactive stream.
The discarded output will never appear where it was going.
For noninteractive streams, this usually does nothing."
  (send (decode-print-arg stream) :clear-output)
  ()) 

;1;; Common Lisp low level input functions*


(defun read-char (&optional stream  (eof-errorp t) eof-value recursive-p)
  "Read one character from STREAM, and return it as a character object.
If EOF-ERRORP is T (the default), EOF is an error.
Otherwise, at EOF we return EOF-VALUE.
RECURSIVE-P is not used; it is a confusion in Common Lisp."
  recursive-p
  (let ((value
	  (send (if (eq stream t)
		    *terminal-io*
		    (if (eq stream ())
			*standard-input*
			stream))
		:tyi eof-errorp)))
    (if (null value)
	eof-value
	(int-char value)))) 


(defun unread-char (char &optional stream )
  "Put CHAR back in STREAM to be read out again as the next input character.
CHAR must be the same character last read from STREAM,
or this may not work or might even signal an error."
  (send (if (eq stream t)
	    *terminal-io*
	    (if (eq stream ())
		*standard-input*
		stream))
	:untyi (char-int char))) 


(defun read-byte (&optional stream  (eof-errorp t) eof-value)
  "Read one byte from STREAM, and return it.
If EOF-ERRORP is T (the default), EOF is an error.
Otherwise, at EOF we return EOF-VALUE."
  (or
    (send (if (eq stream t)
	      *terminal-io*
	      (if (eq stream ())
		  *standard-input*
		  stream))
	  :tyi eof-errorp)
    eof-value)) 


(defun peek-char (&optional peek-type stream  (eof-errorp t) eof-value recursive-p)
  "Peek ahead at input from STREAM without discarding it.
The character peeked at is returned as a character object.
If PEEK-TYPE is NIL, peek at the next input character on STREAM,
 but leave it in the input stream so the next input will reread it.
If PEEK-TYPE is T, discard all whitespace chars and peek at first non-whitespace.
 The current readtable says what is whitespace.
Otherwise, discard all chars before the first one that is equal to PEEK-TYPE,
 which should be a number or a character.
EOF-ERRORP and EOF-VALUE are as for READ-CHAR."
  recursive-p
  (setq stream (if (eq stream t)
		   *terminal-io*
		   (if (eq stream ())
		       *standard-input*
		       stream)))
  (cond
    ((null peek-type)
     (let ((value (send stream :tyi)))
       (if (null value)
	   (if eof-errorp
	       (ferror 'end-of-file-1 "End of file encountered on stream ~S." stream)
	       eof-value)
	   (progn
	     (send stream :untyi value)
	     (int-char value)))))
    (t
     (do ((character-attribute-table (character-attribute-table *readtable*)))
	 (nil)
       (let ((value (send stream :tyi)))
	 (if (null value)
	     (if eof-errorp
		 (ferror 'end-of-file-1 "End of file encountered on stream ~S." stream)
		 (return eof-value))
	     (when (cond
		     ((eq peek-type t)
		      (or (not (= (char-code value) value))
			  (not (= (elt character-attribute-table value) whitespace))))
		     (t (= peek-type value)))
	       (send stream :untyi value)
	       (return (int-char value))))))))) 


(defun listen (&optional stream )
  "T if input is available on STREAM.
On a noninteractive stream, this is T if not at EOF."
  (send (if (eq stream t)
	    *terminal-io*
	    (if (eq stream ())
		*standard-input*
		stream))
	:listen)) 


(defun clear-input (&optional stream)
  "Discard any buffered input on STREAM, if it is an interactive stream."
  (send (if (eq stream t)
	    *terminal-io*
	    (if (eq stream ())
		*standard-input*
		stream))
	:clear-input)
  ()) 


(defun read-char-no-hang (&optional stream  (eof-errorp t) eof-value recursive-p)
  "Read one character from STREAM, and return it as a character object, but don't wait.
On an interactive stream, if no input is currently buffered, NIL is returned.
If EOF-ERRORP is T (the default), EOF is an error.
Otherwise, at EOF we return EOF-VALUE.
RECURSIVE-P is not used; it is a confusion in Common Lisp."
  (declare (ignore recursive-p))
  (condition-case-if (not eof-errorp) ()
      (let ((value
	      (send (if (eq stream t)
			*terminal-io*
			(if (eq stream ())
			    *standard-input*
			    stream))
		    :tyi-no-hang t)))
	(if (null value)
	    ()
	    (int-char value)))
    (end-of-file eof-value))) 


(defun read-any (&optional stream (eof-errorp t) eof-value recursive-p)
  "Read one character or blip from STREAM.
If EOF-ERRORP is T (the default), EOF is an error.
Otherwise, at EOF we return EOF-VALUE.
RECURSIVE-P is not used; it is a confusion in Common Lisp."
  (declare (ignore recursive-p))
  (let ((value (send (if (eq stream t)
			 *terminal-io*
			 (if (eq stream nil)
			     *standard-input*
			     stream))
		     :any-tyi eof-errorp)))
    (if (null value)
	eof-value
	(if (fixnump value)
	    (int-char value)
	    value))))

(defun read-list (&optional stream (eof-errorp t) eof-value recursive-p)
  "Read one blip from STREAM.
If EOF-ERRORP is T (the default), EOF is an error.
Otherwise, at EOF we return EOF-VALUE.
RECURSIVE-P is not used; it is a confusion in Common Lisp."
  (declare (ignore recursive-p))
  (loop for value = (send (if (eq stream t)
			      *terminal-io*
			      (if (eq stream nil)
				  *standard-input*
				  stream))
			  :any-tyi eof-errorp)
        when (null value)
        do (return eof-value)
        when (listp value)
        do (return value)))

(defun read-mouse-or-kbd (&optional stream (eof-errorp t) eof-value recursive-p)
  "Read one character or :mouse-button blip from STREAM.
If EOF-ERRORP is T (the default), EOF is an error.
Otherwise, at EOF we return EOF-VALUE.
RECURSIVE-P is not used; it is a confusion in Common Lisp."
  (declare (ignore recursive-p))
  (loop for value = (send (if (eq stream t)
			      *terminal-io*
			      (if (eq stream nil)
				  *standard-input*
				  stream))
			  :any-tyi eof-errorp)
        when (characterp value)
        do (return value)
        when (fixnump value)
        do (return (int-char value))
        when (null value)
        do (return eof-value)
        when (and (listp value) (eq (car value) :mouse-button))
        do (return (values (int-char (second value)) value))))

(defun read-mouse-or-kbd-no-hang (&optional stream (eof-errorp t) eof-value recursive-p)
  "Read one character or :mouse-button blip from STREAM.
If EOF-ERRORP is T (the default), EOF is an error.
Otherwise, at EOF we return EOF-VALUE.
RECURSIVE-P is not used; it is a confusion in Common Lisp."
  (declare (ignore recursive-p))
  (condition-case-if (not eof-errorp) ()
      (loop for value = (send (if (eq stream t)
                                  *terminal-io*
                                  (if (eq stream nil)
                                      *standard-input*
                                      stream))
                              :any-tyi-no-hang t)
            when (null value)
            do (return nil)
            when (characterp value)
            do (return value)
            when (fixnump value)
            do (return (int-char value))
            when (and (listp value) (eq (car value) :mouse-button))
            do (return (values (int-char (second value)) value)))
    (end-of-file eof-value)))

(defun read-any-no-hang (&optional stream (eof-errorp t) eof-value recursive-p)
  "Read one character or blip from STREAM but don't wait.
On an interactive stream, if no input is currently buffered, NIL is returned.
If EOF-ERRORP is T (the default), EOF is an error.
Otherwise, at EOF we return EOF-VALUE.
RECURSIVE-P is not used; it is a confusion in Common Lisp."
  (declare (ignore recursive-p))
  (condition-case-if (not eof-errorp) ()
      (let ((value (send (if (eq stream t)
			     *terminal-io*
			     (if (eq stream nil)
				 *standard-input*
				 stream))
			 :any-tyi-no-hang t)))
	(if (null value)
	    nil
	    (if (fixnump value)
		(int-char value)
		value)))
    (end-of-file eof-value)))

(defun unread-any (char &optional stream)
  "Put CHAR back in STREAM to be read out again as the next character or blip.
CHAR must be the same character or blip last read from STREAM,
or this may not work or might even signal an error."
  (send (if (eq stream t)
	    *terminal-io*
	    (if (eq stream nil)
		*standard-input*
		stream))
	:untyi (if (characterp char)
		   (char-int char)
		   char))
  char)

;1;; Old-fashioned low level input functions.*

;1This function is compatible with the regular Maclisp TYI.  If you want speed,*
;1FUNCALL the stream directly.  We have to echo, but cannot use the rubout handler*
;1because the user wants to see rubout, form, etc. characters.  Inside the rubout*
;1handler, we do not echo since echoing will have occurred already.*

(defun tyi (&rest read-args &aux ch)
  "Read one character from a stream.  Args are a stream and an eof-option.
The order is irrelevant; an arg which is not a reasonable stream
is taken to be the eof-option, which is returned if end of file is reached.
If there is no eof-option, end of file is an error.
If the stream supports rubout handling but we are not inside the input editor,
then the character read is echoed."
  (declare (arglist stream eof-option))
  (multiple-value-bind (stream eof-option)
      (decode-read-args read-args)
    (cond
      ((null (setq ch (send stream :tyi)))	;1Get a character, check for EOF*
       (if (eq eof-option 'no-eof-option)
	   (ferror 'end-of-file-1 "End of file encountered on stream ~S." stream)
	   eof-option))
      ((or rubout-handler			;1 If inside rubout handler, or*
	   (not (member :rubout-handler (send stream :which-operations) :test #'eq)))
       ch)					;1  ordinary device, just return char*
      (t
       ;1; Echo anything but blips and rubout, even control and meta charcters.*
       (if (and (fixnump ch) (/= ch (char-int #\RUBOUT)))
	   (format stream "~C" ch))
       ch)))) 


;1This function is compatible, more or less, with the regular Maclisp TYIPEEK.*
;1It does not echo, since the echoing will occur when READ or TYI is called.*
;1It does echo characters which it discards.*

(defun tyipeek (&optional peek-type &rest read-args
		&aux (character-attribute-table (character-attribute-table *readtable*)))
  (declare (arglist peek-type stream eof-option))
  (multiple-value-bind (stream eof-option)
      (decode-read-args read-args)
    (and (numberp peek-type) (>= peek-type 512)
	 (ferror () "The ~S flavor of TYIPEEK is not implemented." peek-type))
    (do ((ch))					;1Pass over characters until termination condition reached*
	(nil)
      (or (setq ch (send stream :tyi))
	  (if (eq eof-option 'no-eof-option)
	      (ferror 'end-of-file-1 "End of file encountered on stream ~S." stream)
	      (return eof-option)))
      (send stream :untyi ch)			;1Put it back*
      (and
	(cond
	  ((null peek-type))			;1Break on every*
	  ((eq ch peek-type))			;1Break on specified character*
	  ((eq peek-type t)			;1Break on start-of-object*
	   (or (not (= (char-code ch) ch))
	       (not (= (elt character-attribute-table ch) whitespace)))))
	(return ch))				;1Break here*
      (tyi stream))))				;1Echo and eat this character*


(defun stream-copy-until-eof (from-stream to-stream &optional (leader-size nil))
  "Copy data from FROM-STREAM to TO-STREAM, until EOF on FROM-STREAM.
The default is to use the most efficient mode, but the third argument
may be used to force use of :LINE-IN/:LINE-OUT mode, especially useful
when the to-stream is an editor interval stream.  If you use this to
copy binary files, note that you had better open the streams with
appropriate host-dependent byte sizes, and that if the from-stream
supports :LINE-IN but not :READ-INPUT-BUFFER you will probably lose."
  (let ((fwo (send from-stream :which-operations))
	(two (send to-stream :which-operations)))
    (cond
      ((and (not leader-size) (member :read-input-buffer fwo :test #'eq)
	    (member :string-out two :test #'eq))
       ;1; If it can go, this mode is the most efficient by far.*
       (do ((buf)
	    (offset)
	    (limit))
	   (nil)
	 (multiple-value-setq (buf offset limit)
			      (send from-stream :read-input-buffer))
	 (cond
	   ((null buf) (return ())))
	 (send to-stream :string-out buf offset limit)
	 (send from-stream :advance-input-buffer)))
      ((and (member :line-in fwo :test #'eq) (member :line-out two :test #'eq))
       ;1; Not as good, but better than :TYI/:TYO*
       (do ((line)
	    (eof))
	   (nil)
	 (multiple-value-setq (line eof)
			      (send from-stream :line-in leader-size))
	 (cond
	   ((not eof) (send to-stream :line-out line))
	   (t (send to-stream :string-out line) (return ())))))
      ;1; This always wins, but is incredibly slow.*
      (t
       (do ((char))
	   ((null (setq char (send from-stream :tyi))))
	 (send to-stream :tyo char)))))) 



(deff zlc:make-syn-stream 'make-synonym-stream) 

(defun make-synonym-stream (stream-symbol)
  "Return an I/O stream which passes all operations to the value of STREAM-SYMBOL.
This is most often used with STREAM-SYMBOL equal to '*TERMINAL-IO*.
STREAM-SYMBOL can be a locative instead of a symbol."
  (if (symbolp stream-symbol)
      ;1; Changed 10/16/83 to make an uninterned symbol*
      ;1; but record it on STREAM-SYMBOL's plist so only one symbol needs to be made.*
      (or (get stream-symbol 'syn-stream)
	  (let ((sym (make-symbol (string-append stream-symbol "-SYN-STREAM"))))
	    (%p-store-tag-and-pointer (locf (symbol-function sym)) dtp-external-value-cell-pointer
				      (locf (symbol-value stream-symbol)))
	    (setf (get sym 'io-stream-p) t)
	    (setf (get stream-symbol 'syn-stream) sym)
	    sym))
      (if (locativep stream-symbol)
	  (let ((sym (make-symbol "SYN-STREAM")))
	    (%p-store-tag-and-pointer (locf (symbol-function sym)) dtp-external-value-cell-pointer
				  stream-symbol)
	    (setf (get sym 'io-stream-p) t)
	    sym)
	  (ferror () "The argument to Make-Synonym-Stream must be a symbol or an locative."))))


(defun follow-syn-stream (stream)
  "If STREAM is a synonym stream symbol, return the stream it is currently a synonym for.
Otherwise return STREAM."
  (cond
    ((not (symbolp stream)) stream)
    ((neq (locf (symbol-function stream)) (follow-cell-forwarding (locf (symbol-function stream)) t))
     (symbol-function stream))
    (t stream))) 


(defun make-broadcast-stream (&rest streams)
  "Return an I/O stream which passes all operations to all of the STREAMS.
Thus, output directed to the broadcast stream will go to multiple places."
  (if (null streams)
      'null-stream
      (let-closed
	((broadcast-stream-streams (copy-list streams))
	 (which-operations
	   (loop with wo = (send (car streams) :which-operations) with copyp = t for stream in
		 (cdr streams) do
		 (loop with wo2 = (send stream :which-operations) for op in wo unless
		       (member op wo2 :test #'eq) do (if copyp
							 (setq wo (copy-list wo)))
		       (setq copyp ()) (setq wo (delete op (the list wo) :test #'eq)))
		 finally (return wo))))
	#'(lambda (&rest args)
	    (cond
	      ((eq (car args) :which-operations) which-operations)
	      ((eq (car args) :operation-handled-p)
	       (member (cadr args) which-operations :test #'eq))
	      ((eq (car args) :send-if-handles)
	       (do ((l broadcast-stream-streams (cdr l)))
		   ((null (cdr l))		;1Last one gets to return multiple values*
		    (apply (car l) :send-if-handles args))
		 (apply (car l) :send-if-handles args)))
	      (t
	       (do ((l broadcast-stream-streams (cdr l)))
		   ((null (cdr l))		;1Last one gets to return multiple values*
		    (apply (car l) args))
		 (apply (car l) args)))))))) 


(defun make-concatenated-stream (&rest streams)
  "Return a stream which will read from each of the STREAMS, one by one.
Reading from the concatenated stream will first read data from the first STREAM.
When that reaches eof, it will then read data from the second STREAM,
and so on until all the STREAMS are exhausted.  Then the concatenated stream gets eof."
  (let-closed
    ((concatenated-stream-streams (copy-list streams))
     (which-operations
       (and streams 
	 (loop with wo = (send (car streams) :which-operations) with copyp = t for stream in
	       (cdr streams) do
	       (loop with wo2 = (send stream :which-operations) for op in wo unless
		     (member op wo2 :test #'eq) do (if copyp
						       (setq wo (copy-list wo)))
		     (setq copyp ()) (setq wo (delete op (the list wo) :test #'eq)))
	       finally (return wo))))
     (concatenated-stream-function nil))
    (setq concatenated-stream-function
	  #'(lambda (op &rest args)
	      (prog ()
		 loop
		    (return
		      (case op
			((:tyi :tyi-no-hang :any-tyi-no-hang :any-tyi)
			 (if (null concatenated-stream-streams)
			     (and (car args)
				  (ferror 'end-of-file-1 "End of file on ~S." concatenated-stream-function))
			     (let ((value (send (car concatenated-stream-streams) :tyi)))
			       (if value
				   value
				   (progn
				     (pop concatenated-stream-streams)
				     (go loop))))))
			(:which-operations which-operations)
			(:send-if-handles
			 (and (member (car args) which-operations :test #'eq)
			      (apply concatenated-stream-function args)))
			(:operation-handled-p
			 (not (null (member (car args) which-operations :test #'eq))))
			(:untyi (send (car concatenated-stream-streams) :untyi (car args)))
			(:direction :input)
			(t
			 (stream-default-handler concatenated-stream-function op (car args) (cdr args)))))))))) 


(defparameter two-way-input-operations
	      '(:tyi :tyi-no-hang :any-tyi-no-hang :any-tyi :untyi :tyipeek :listen :line-in :string-in
		     :get-input-buffer :advance-input-buffer :read-input-buffer :read-until-eof :clear-input)) 


(defun make-two-way-stream (input-stream output-stream)
  "Return a stream which does its input via INPUT-STREAM and its output via OUTPUT-STREAM.
This works by knowing about all the standard, ordinary input operations.
Use of unusual input operations, or operations that affect both input and output
(such as random access) will not work."
  (let-closed
    ((two-way-input-stream input-stream) (two-way-output-stream output-stream)
     (which-operations
       (union
	 (union
	   (intersection two-way-input-operations (send input-stream :which-operations) :test #'eq)
	   (remove-if #'(lambda (elt)
			  (member elt two-way-input-operations :test #'eq))
		      (send output-stream :which-operations))
	   :test #'eq)
	 '(:send-if-handles :operation-handled-p) :test #'eq))
     (two-way-stream-function nil))
    (setq two-way-stream-function
	  #'(lambda (op &rest args)
	      (cond
		((member op two-way-input-operations :test #'eq)
		 (apply two-way-input-stream op args))
		((eq op :which-operations) which-operations)
		((eq op :send-if-handles)
		 (and (member (car args) which-operations :test #'eq)
		      (apply two-way-stream-function args)))
		((eq op :operation-handled-p)
		 (not (null (member (car args) which-operations :test #'eq))))
		(t (apply two-way-output-stream op args))))))) 


(defun make-echo-stream (input-stream output-stream)
       "Return a stream which does output via OUTPUT-STREAM, and input via INPUT-STREAM with echo.
All characters that this stream reads from INPUT-STREAM are also
echoed to OUTPUT-STREAM.
This works by knowing about all the standard, ordinary input operations.
Use of unusual input operations, or operations that affect both input and output
(such as random access) will not work."
  (let-closed
    ((two-way-input-stream input-stream) (two-way-output-stream output-stream)
     (echo-stream-unread-char nil)
     (which-operations
       (union
	 (union '(:tyi :untyi)
		(remove-if #'(lambda (elt)
			       (member elt two-way-input-operations :test #'eq))
			   (send output-stream :which-operations))
		:test #'eq)
	 '(:send-if-handles :operation-handled-p) :test #'eq))
     (two-way-stream-function nil))
    (setq two-way-stream-function
	  #'(lambda (op &rest args)
	      (cond
		((eq op :tyi)
		 (or (prog1
		       echo-stream-unread-char
		       (setq echo-stream-unread-char ()))
		     (let ((value (send two-way-input-stream :tyi (car args))))
		       (if value
			   (send two-way-output-stream :tyo value))
		       value)))
		((eq op :untyi) (setq echo-stream-unread-char (car args)))
		((member op two-way-input-operations :test #'eq)
		 (stream-default-handler two-way-stream-function op (car args) (cdr args)))
		((eq op :which-operations) which-operations)
		((eq op :send-if-handles)
		 (and (member (car args) which-operations :test #'eq)
		      (apply two-way-stream-function args)))
		((eq op :operation-handled-p)
		 (not (null (member (car args) which-operations :test #'eq))))
		(t (apply two-way-output-stream op args))))))) 

;1;AB 7/29/87.  Fix :LINE-IN always to cons in BACKGROUND-CONS-AREA and not to write into*
;1;             deallocated readstrings (in fact, don't use the readstring pool at all).  [SPR 5985]*
(defun stream-default-handler (fctn op arg1 args &aux tem)
  "Subroutine which provides default definition of certain stream operations.
If a stream does not recognize an operation, it may call this function
to have the operation handled.  The stream should return whatever
this function returns to it.  OP should be the operation, FCTN should
be the stream which received the operation, and ARG1 and ARGS should be
the arguments that came with the operation."
  (case op
    ((:tyipeek :listen)
     (cond ((setq tem (send fctn :tyi ()))
            (send fctn :untyi tem)
            tem)))
    ((:any-tyi :tyi-no-hang)
     (send fctn :tyi arg1))
    (:any-tyi-no-hang
     (send fctn :any-tyi arg1))
    ((:clear-output :clear-input :force-output :finish :close :eof)
     nil)
    (:fresh-line
     (send fctn :tyo #\NEWLINE)
     t)
    ((:string-out :line-out)
     (setq tem (string arg1))
     (do ((len (cond ((second args))
                     (t (length tem))))
          (i (cond ((first args)) (t 0))
             (1+ i)))
         ((>= i len) nil)
       (send fctn :tyo (aref tem i)))
     (and (eq op :line-out)
          (send fctn :tyo #\NEWLINE)))
    (:line-in
     (let* ((sys:default-cons-area sys:background-cons-area)
	    (buf (if (null arg1)
		     (make-array 64. :element-type 'string-char
				 :fill-pointer 0)
		     (make-array 64. :element-type 'string-char
				 :leader-length (if (numberp arg1)
						    arg1
						    1)))))
       (setf (fill-pointer buf) 0)
       (values buf
	       (do ((tem (send fctn :tyi nil) (send fctn :tyi nil)))
		   ((or (null tem) (= (int-char tem) #\NEWLINE) (= (int-char tem) #\END))
		    (when arg1  (adjust-array buf (array-active-length buf)))
		    (null tem))
		 (vector-push-extend tem buf)))))
    (:string-in
     ;1; ARG1 = EOF, (CAR ARGS) = STRING*
     (loop with start = (or (cadr args) 0)
           and end = (or (caddr args) (array-total-size (car args)))
           while (< start end)
           as ch = (send fctn :tyi)
           while ch
           do (setf (aref (car args) (prog1 start (incf start))) ch)
	   finally (and (array-has-leader-p (car args)) (setf (fill-pointer  (car args)) start))
	   (and (null ch) arg1 (ferror 'end-of-file-1 "End of file on ~S." fctn))
	   (return (values start (null ch)))))
    (:string-line-in
     ;1; ARG1 = EOF, (CAR ARGS) = STRING*
     (loop WITH start = (or (cadr args) 0)
           AND END = (or (caddr args) (array-total-size (car args)))
           WHILE (< start end)
           AS ch = (send fctn :tyi)
           WHILE (and ch (not (eql ch (char-int #\NEWLINE))))
           DO (setf (aref (car args) (prog1
                                       start
                                       (incf start))) ch)
           FINALLY (and (array-has-leader-p (car args))
                        (setf (fill-pointer  (car args)) start))
	           (and (null ch) arg1 (ferror 'end-of-file-1 "End of file on ~S." fctn))
                   (return (values start (null ch) (not (eql ch (char-int #\NEWLINE))))))) ;1changed from neq PMH*
    (:operation-handled-p (member arg1 (send fctn :which-operations) :test #'eq))
    (:characters t)
    (:element-type
     (if (send fctn :characters)
	 'character
	 (let ((value (send fctn :send-if-handles :byte-size)))
	   (if value
	       `(unsigned-byte ,value)
	       'fixnum))))
    (:direction
     (let ((ops (send fctn :which-operations)))
       (if (member :tyi ops :test #'eq)
	   (if (member :tyo ops :test #'eq)
	       :bidirectional
	       :input)
	   (if (member :tyo ops :test #'eq)
	       :output
	       ()))))
    (:send-if-handles
     (if (member arg1 (send fctn :which-operations) :test #'eq)
	 (apply fctn arg1 args)))
    (otherwise
     (ferror :unclaimed-message "The stream operation ~S is not supported by ~S" op fctn)))) 

(defmacro selectq-with-which-operations (thing &body clauses)
  "Like SELECTQ, but automatically recognizes :WHICH-OPERATIONS.
:WHICH-OPERATIONS is handled by returning a list of all the
keywords which are tested for in the clauses."
  (let (otherwise)
    (cond
      ((string-equal (caar (last clauses)) 'otherwise)
       (setq otherwise (last clauses)
	     clauses (butlast clauses))))
    `(case ,thing ,@clauses
	   (:which-operations
	    ',(loop for clause in clauses appending
		    (if (consp (car clause))
			(car clause)
			(list (car clause)))))
	   . ,otherwise))) 


(defprop null-stream t io-stream-p) 

(defconstant *null-stream* 'null-stream
	     "The value of this variable is a stream which discards output.")

;1;AB 7/29/87.  Fix NULL-STREAM to return 0 on :READ-CURSORPOS or :READ-POINTER; T on :SET-CURSORPOS. [SPR 5891]*
(defun null-stream (operation &rest args &aux tem)
  "An i/o stream which ignores output and gives instant end-of-file on input."
  (selectq-with-which-operations operation
    ;1; These operations signal EOF.*
    ((:tyi :read-char :tyi-no-hang :tyipeek :get-input-buffer :read-input-buffer :any-tyi :any-tyi-no-hang)
     (when (first args) 
       (ferror 'read-end-of-file "End of file on SYS:NULL-STREAM.")))
    ;1; Signals EOF differently.*
    (:string-in (if (first args)
 		    (ferror 'read-end-of-file "End of file on SYS:NULL-STREAM.")
 		    (values (third args) t)))
    ;1; Signals EOF still differently.*
    (:line-in
      (setq tem (make-array 0 :element-type 'string-char :leader-length (and (numberp (first args)) (first args))))
      (when (and (numberp (first args)) (plusp (first args)))
  	(setf (fill-pointer tem ) 0))
      (values tem t))
    ;1; These operations should all return their argument.*
    ((:tyo :string-out :write-char :line-out :untyi) (first args))
    ;1; These operations should return NIL*
    ((:increment-cursorpos :finish :force-output :clear-output :clear-input :listen
			   :input-chars-available-p :read-until-eof :reset :reset-hardware :set-pointer) nil)
    ;1; Supports nothing in both directions.*
    (:direction :bidirectional)
    ;1; These operations should always return T.*
    ((:beep :characters :set-cursorpos) t)
    ;1; Positional operations return 0.*
    (:read-cursorpos (values 0 0))
    (:read-pointer 0)
    ;1; Handle obscure operations.*
    (otherwise (stream-default-handler 'null-stream operation (first args) (rest args)))))



(defvar *iolst :unbound "String or list of data to read, in READLIST or READ-FROM-STRING.") 


(defvar *ioch :unbound "Character position in *IOLST, in READ-FROM-STRING.") 


(defvar *ioend :unbound "Character position to stop at, in READ-FROM-STRING.") 


(defun readlist (charlist &aux (*ioch nil) (*iolst charlist))
  "Read an expression from the list of characters CHARLIST."
  (read 'readlist-stream t)) 


(defprop readlist-stream t io-stream-p) 


(defun readlist-stream (operation &optional arg1 &rest rest)
  (cond
    ((or (eq operation :any-tyi)
	 (member operation '(:tyi :tyi-no-hang :any-tyi-no-hang) :test #'eq))
     (cond
       ((eq *ioch t) (ferror () "EOF in middle of READLIST"))
       ((not (null *ioch)) (prog2
			     ()
			     *ioch
			     (setq *ioch ())))
       ((null *iolst) (setq *ioch t) 32)
       (t (prog1
	    (character (car *iolst))
	    (setq *iolst (cdr *iolst))))))
    ((eq operation :untyi) (setq *ioch arg1))
    ((eq operation :which-operations) '(:tyi :untyi))
    (t (stream-default-handler 'readlist-stream operation arg1 rest)))) 


(defun read-from-string (string &optional (eof-errorp t) eof-value &key &optional (start 0) end preserve-whitespace)
  "Read an expression out of the characters in STRING.
START and END are indices which specify a substring of STRING to be used.
 START defaults to 0 and END to NIL (which means the end of STRING).
Reaching the end of STRING or the specified substring constitutes EOF.
EOF-ERRORP and EOF-VALUE are passed to the READ function,
 and PRESERVE-WHITESPACE non-NIL causes READ-PRESERVING-WHITESPACE to be used.
Only one object is read.  The first value is that object (or perhaps
 EOF-VALUE) and the second value is the index in STRING at which reading stopped."
  (declare (VALUES contents end-char-position))
  (let ((*iolst string)
	(*ioch start)
	(*ioend (or end (length string))))
    (values (internal-read 'read-from-string-stream eof-errorp eof-value () preserve-whitespace)
	    *ioch))) 


(defun global:read-from-string 
       (string &optional (eof-option 'no-eof-option) (start 0) end &aux (*iolst string) (*ioch start)
	(*ioend (or end (length string))))
  "Read an expression out of the characters in STRING.
If EOF-OPTION is non-NIL, it is returned on end of file;
otherwise, end of file is an error.  START (default 0)
is the position in STRING to start reading at; END is where to stop.

The second value is the index in the string at which reading stopped.
It stops after the first object, even if not all the input is used."
  (declare (VALUES contents end-char-position))
  (values (read 'read-from-string-stream () eof-option) *ioch)) 


(defprop read-from-string-stream t io-stream-p) 


(defun make-string-input-stream (string &optional (start 0) end)
  "Return a stream from which one can read the characters of STRING, or some substring of it.
START and END are indices specifying a substring of STRING;
they default to 0 and NIL (NIL for END means the end of STRING)."
  (setq string (string string))
  (let-closed ((*ioch start) (*ioend (or end (length string))) (*iolst string))
     'read-from-string-stream)) 
    

;1;PHD 8/12/87 Fixed READ-FROM-STRING-STREAM :tyi so it returns a fixnum, SPR#6198*
(defun read-from-string-stream (operation &optional (arg1 nil) &rest rest)
  (cond
    ((or (eq operation :tyi) (eq operation :any-tyi))
     (cond
       ((= *ioch *ioend)
	(if arg1
	    (ferror 'sys:read-end-of-file "end of file encountered on stream ~S" *iolst)
	    nil))
       (t (prog2
	    ()
	    (char-int (aref *iolst *ioch))
	    (setq *ioch (1+ *ioch))))))
    ((eq operation :untyi) (when arg1
			     (setq *ioch (1- *ioch))))
    ((eq operation :get-string-index) *ioch)
    ((eq operation :which-operations) '(:tyi :untyi :get-string-index))
    (t (stream-default-handler 'read-from-string-stream operation arg1 rest))))

(defun flatsize (x &aux (*ioch 0))
  "Return the number of characters it takes to print X with quoting."
  (prin1 x #'flatsize-stream)
  *ioch) 


(defun flatc (x &aux (*ioch 0))
  "Return the number of characters it takes to print X with no quoting."
  (princ x #'flatsize-stream)
  *ioch) 


(defprop flatsize-stream t io-stream-p) 


(defun flatsize-stream (operation &optional arg1 &rest rest)
  (cond
    ((eq operation :tyo) (setq *ioch (1+ *ioch)))
    ((eq operation :which-operations) '(:tyo))
    (t (stream-default-handler 'flatsize-stream operation arg1 rest)))) 


(defun zlc:readline (&rest read-args)
  "Read a line from STREAM and return it as a string.
The string does not include a Return character, and is empty for a blank line.
If EOF-OPTION is non-NIL, it is returned on end of file at beginning of line;
 otherwise, end of file with no text first is an error.
End of file after reading some text is never an error.

If the stream supports the :RUBOUT-HANDLER operation, we use it.
OPTIONS is a list of rubout handler options, passed to WITH-INPUT-EDITING if it is used.

The second value is EOF-OPTION if we exit due to end of file.

The third value is the delimiter which ended the input, or NIL if
it ended due to EOF."
  (declare (arglist &optional stream eof-option options)
	   (values string-or-eof-option eof-flag delimiters))
  (let ((options nil))
    ;1; This kludge is to let us take a third, optional argument.*
    (cond
      ((> (length read-args) 2) (setq options (third read-args))
				(setq read-args (list (first read-args) (second read-args)))))
    (multiple-value-bind (stream eof-option)
	(decode-read-args read-args)
      (multiple-value-bind (string eof terminator)
	  (read-delimited-string '(#\NEWLINE #\END) stream (eq eof-option 'no-eof-option) options)
	(values (if (and eof (zerop (length string)))
		    eof-option
		    string)
		(if eof
		    eof-option)
		terminator))))) 


;1;PAD 4/7/87 If input-stream is nil, default to *standard-input*.*
;1;           If input-stream is t, default to *terminal-io*. [SPR 4499] (CR:PHD)*
(defun read-line (&optional (stream *standard-input*) (eof-error-p t) eof-value recursive-p rh-options)
  "Read a line from STREAM and return it as a string.
The string does not include the final Newline character, and is empty if nothing was read.
The second value is T if the line was terminated by EOF.
EOF-ERROR-P says whether an error should be signalled if eof occurs at the start of
the line. If it is NIL and eof occurs at the start of the line, we return EOF-VALUE and T
RECURSIVE-P is ignored.
If the stream supports the :RUBOUT-HANDLER operation, we use it.
OPTIONS is a list of rubout handler options, passed to WITH-INPUT-EDITING if it is used."
  (declare (values line eof-flag)
	   (ignore recursive-p))
  (case stream
	(nil (setf stream *standard-input*))
	((t) (setf stream  *terminal-io*)))
  (with-stack-list* (options nil  rh-options)
    (with-input-editing (stream options)
      (multiple-value-bind (string eof-flag)
	  (send stream :line-in t)
	(if  eof-flag
	     (if eof-error-p
		 (ferror 'sys:end-of-file "end of file reached during read-line on stream ~S" stream)
		 (if (zerop (length string))
		     (values eof-value t)
		     (values string eof-flag)))
	     (values string nil))))))


(defun readline-trim (&rest read-args)
  "Read a line from STREAM and return it as a string, sans leading and trailing whitespace.
The string does not include a Return character, and is empty for a blank line.
If EOF-OPTION is non-NIL, it is returned on end of file at beginning of line;
 otherwise, end of file with no text first is an error.
End of file after reading some text is never an error.

If the stream supports the :RUBOUT-HANDLER operation, we use it.
OPTIONS is a list of rubout handler options, passed to :RUBOUT-HANDLER if it is used.

The second value is T if we exit due to end of file."
  (declare (arglist &optional stream eof-option options) (values string eof))
  (multiple-value-bind (string eof)
    (apply 'zlc:readline read-args)
    (values (if eof
	      string
	      (string-trim '(#\SPACE #\TAB) string))
	    eof))) 


(defun readline-or-nil (&rest read-args)
  "Read a line from STREAM and return it as a string, or return NIL if line is empty.
The string does not include a Return character.
If EOF-OPTION is non-NIL, it is returned on end of file at beginning of line;
 otherwise, end of file with no text first is an error.
End of file after reading some text is never an error.

If the stream supports the :RUBOUT-HANDLER operation, we use it.
OPTIONS is a list of rubout handler options, passed to :RUBOUT-HANDLER if it is used.

The second value is T if we exit due to end of file."
  (declare (arglist &optional stream eof-option options) (values string-or-nil eof))
  (multiple-value-bind (string eof)
    (apply 'zlc:readline read-args)
    (values
     (if eof
       string
       (progn
	 (setq string (string-trim '(#\SPACE #\TAB) string))
	 (if (equal string "")
	   ()
	   string)))
     eof))) 


(defvar read-string-pool  nil
;1;;       '((nil . #.(make-array 128 :element-type 'string-char :fill-pointer 0) )*
;1;;*			1          (nil . #.(make-array 128 :element-type 'string-char :fill-pointer 0) )*
;1;;                                  (nil . #.(make-array 128 :element-type 'string-char :fill-pointer 0) )*
;1;;*			1          (nil . #.(make-array 128 :element-type 'string-char :fill-pointer 0) ))*
  "Pool of strings for READ-DELIMITED-STRING  List of entries.
Car of entry is T if string is in use, else NIL.
Cdr of entry is string.")

(defun get-readstring (&aux temp)
  "Gets an available string from pool."
  (without-interrupts
    (if (setq temp (assoc nil read-string-pool :test #'eq))
	(progn
	  (rplaca temp t)
	  (setf (fill-pointer (cdr temp )) 0)
	  (cdr temp))
	;1;*** phd 9/2/85 force the area because this array is going to be kept around, so beware of temp areas.*
	(let ((default-cons-area  si:background-cons-area))
	  (cdar (push `(t . ,(make-array 128 :element-type 'string-char :fill-pointer 0 :area background-cons-area) )
		      read-string-pool))))))

(defun return-readstring (array &aux temp)
  "Returns a string to pool."
  (without-interrupts
    (when (setq temp (rassoc array read-string-pool :test #'eq))
      (rplaca temp nil))))

(defmacro with-readstring (string &body body)	;1MBC 8-13-86*
  `(let ((,string (si:get-readstring)))
     (unwind-protect
	 (Block nil
	   ,@Body)
       (return-readstring ,string))))

;;CLM for PHD 4/7/88 Fixed third value of Read-Delimited-String. 
;;It has to be a character object. [sprs 7640 & 7502]
(defun read-delimited-string (&optional (delimiters #\END) (stream *standard-input*) eof rh-options (buffer-size 100))
  "Reads input from STREAM until DELIMITERS is found; returns a string.
Uses the rubout handler if STREAM supports that.
DELIMITERS is either a character or a list of characters.
 (Characters may be fixnums or character objects).
Values are:
 The string of characters read, not including the delimiter
 T if input ended due to end of file
 The delimiter character read (as a fixnum), or NIL if ended at EOF.
EOF if non-NIL means get error on end of file before any input is got.
RH-OPTIONS are passed to WITH-INPUT-EDITING.
BUFFER-SIZE is the size to make the buffer string, initially."
  (declare (values string eof-flag delimiters)
	   (ignore buffer-size))
  (with-stack-list (activation :activation (if (consp delimiters)
					       'zlc:memq
					       'eq)
			       delimiters)
    (with-stack-list* (options activation rh-options)
      (with-input-editing (stream options)
	(do ((buffer (get-readstring)))
	    (nil)
	  (let ((ch
		  (send stream (if rubout-handler
				   :any-tyi
				   :tyi)
			(and (zerop (length buffer)) eof))))
	    (cond
	      ((null ch)
	       (return (values 
			 (prog1 (string-append buffer)
				(return-readstring buffer))
			 t)))
	      ((consp ch)
	       (when (eq (car ch) :activation)
		 (send stream :tyo (cadr ch))
		 (return
		   (values
		     (prog1
		       (string-append buffer)
		       (return-readstring buffer))
		     () (int-char (cadr ch))))))
	      ((if (consp delimiters)
		   (or (member (int-char ch) delimiters :test #'eq) 
		       (member ch delimiters :test #'eq) )
		   (eq (int-char ch) (int-char delimiters)))
	       (return (values 
			 (prog1 (string-append buffer)
				(return-readstring buffer))
			  () (int-char ch))))
	      (t (vector-push-extend ch buffer))))))))) 


(defvar prompt-and-read-format-string :unbound
   "Within PROMPT-AND-READ, holds the FORMAT-STRING argument.") 


(defvar prompt-and-read-format-args :unbound
   "Within PROMPT-AND-READ, holds the FORMAT-ARGS arguments.") 


(defun prompt-and-read (option format-string &rest format-args)
  "Read an object from *QUERY-IO* according to OPTION, prompting using FORMAT-STRING and -ARGS.
OPTION says how to read the object and what its syntax is.  It can be:
 :READ -- use READ to read the object.
 :EVAL-READ -- read an s-expression and evaluate it.  Return the value.
 :EVAL-READ-OR-END -- Like :EVAL-READ, but user can also type just End,
   in which case we return NIL as first value and :END as second.
 (:EVAL-READ :DEFAULT <DEFAULT>) -- Like :EVAL-READ, but user can
   also type just Space to use the default.  Second value is :DEFAULT then.
 (:EVAL-READ-OR-END :DEFAULT <DEFAULT>) -- Analogous.
 :NUMBER -- read a number, terminated by Return or End.
 (:NUMBER :INPUT-RADIX <RADIX> :OR-NIL <BOOLEAN>) -- read using <RADIX> for *READ-BASE*,
   and if <BOOLEAN> is non-NIL it allows you to type just Return and returns NIL.
 :CHARACTER -- read one character and return it as a fixnum.
 :DATE -- read a date and return in universal time format.
 (:DATE :PAST-P <PAST-P> :NEVER-P <NEVER-P>) -- read a date.
   The value is in universal time format.
   If <NEVER-P> is non-NIL, \"never\" is accepted, meaning return NIL.
   If <PAST-P> is non-NIL, the date is required to be before the present.
 :STRING -- read a string, terminated by Return.
 :STRING-TRIM -- read a string, terminated by Return.
   Discard leading and trailing whitespace.
 :STRING-OR-NIL -- read a string, terminated by Return.
   Discard leading and trailing whitespace.  If string is empty, return NIL.
 :PATHNAME -- read a pathname and default it.
 (:PATHNAME :DEFAULTS <DEFAULTS-LIST> :VERSION <VERSION-DEFAULT>) --
   read a pathname and default it using the defaults list specified.
   <VERSION-DEFAULT> is passed as the fourth arg to FS:MERGE-PATHNAME-DEFAULTS.
 :PATHNAME-OR-NIL -- like :PATHNAME but if user types just End then NIL is returned.
 (:DELIMITED-STRING :DELIMITER <DELIM> :BUFFER-SIZE <SIZE>) --
   read a string terminated by <DELIM>, which should be a character or a list of them.
   <SIZE> specifies the size of string to allocate initially.
 :DELIMITED-STRING-OR-NIL -- like :DELIMITED-STRING but if user types
   an empty string then NIL is returned.
 (:FQUERY . FQUERY-OPTIONS) -- calls FQUERY with the options."
  (prompt-and-read-internal option nil format-string format-args))

(defun prompt-and-read-internal (option rubout-handler-options format-string &optional format-args)
  ;1; 11/20/87 CLM - Fix to use ZLC:MEMQ instead of MEMQ.  The function was looking for MEMQ in the*
  ;1;                SYS package, which used to work.  At some time MEMQ must have been exported to*
  ;1;                SYS, but this is no longer the case.*
  (or (lexpr-send *query-io* :send-if-handles :prompt-and-read option format-string format-args)
      (let* ((option-type (if (consp option)
			      (car option)
			      option))
	     (function (get option-type 'prompt-and-read-function))
	     (prompt-and-read-format-string format-string)
	     (prompt-and-read-format-args format-args))
	(cond
	  ((get option-type 'prompt-and-read-no-rubout-function)
	   (send (get option-type 'prompt-and-read-no-rubout-function) option *query-io*))
	  ((null function)
	   (ferror () "~S is not a known PROMPT-AND-READ option keyword." option-type))
	  ((send *query-io* :operation-handled-p :rubout-handler)
	   (let ((rh-options (get option-type 'prompt-and-read-rubout-options
				  '((:prompt prompt-and-read-prompt-function)
				    (:activation zlc:memq (#\END #\NEWLINE))))))
	     (if rubout-handler-options
		 (setf rh-options (cons rubout-handler-options rh-options)))
	     (send *query-io* :rubout-handler rh-options function option *query-io*)))
	  (t (funcall function option *query-io*))))))

(defun prompt-and-read-prompt-function (stream ignore)
  (apply 'global:format stream prompt-and-read-format-string prompt-and-read-format-args)) 


(defparameter eval-read-prinlevel 2) 

(defparameter eval-read-prinlength 4) 


(defprop :eval-read eval-read-prompt-and-read prompt-and-read-no-rubout-function) 

(defprop :eval-sexp eval-read-prompt-and-read prompt-and-read-no-rubout-function) 

(defprop :eval-form eval-read-prompt-and-read prompt-and-read-no-rubout-function) 

(defun eval-read-prompt-and-read (option stream)
  (do (value
       form
       flag)
      (nil)
    (error-restart (error "Try again to type this input.")
      (multiple-value-setq (form flag)
			   (with-input-editing
			     (stream '((:prompt prompt-and-read-prompt-function) (:activation = #\END)))
			     (let ((ch (send stream :tyi)))
			       (cond
				 ((and (consp option) (get-location-or-nil option :default) (= (int-char ch) #\SPACE))
				  (values (get option :default) :default))
				 (t (send stream :untyi ch) (values (read stream)))))))
      (if flag
	  (return (values form flag))
	  (setq value (eval-abort-trivial-errors form))))
    ;1; If FORM was not trivial, ask for confirmation of the value it returned.*
    (when (or (trivial-form-p form)
	      (let ((*print-level* eval-read-prinlevel)
		    (*print-length* eval-read-prinlength))
		(fquery '(:list-choices nil) "The object is ~S, ok? " value)))
      (return value))
    (terpri stream))) 


(defun trivial-form-p (form)
  "T if what FORM evaluates to is inherent in its appearance."
  (cond
    ((symbolp form) (or (eq form 't) (null form)))
    ((keywordp form))
    ((eq (car-safe form) 'quote) t)
    ((numberp form) t)
    ((stringp form) t))) 


(defprop :eval-read-or-end eval-read-or-end-prompt-and-read prompt-and-read-no-rubout-function) 

(defprop :eval-sexp-or-end eval-read-or-end-prompt-and-read prompt-and-read-no-rubout-function) 

(defprop :eval-form-or-end eval-read-or-end-prompt-and-read prompt-and-read-no-rubout-function) 

;1;;Clm-phd Fix: (int-char ch) was called when ch was a list.*
(defun eval-read-or-end-prompt-and-read (option stream)
  (do (value
       form
       flag)
      (nil)
    (error-restart (error "Try again to type this input.")
       (multiple-value-setq (form flag)
	 (with-input-editing
	  (stream '((:prompt prompt-and-read-prompt-function) (:activation = #\END)))
	  (let ((ch (send stream :any-tyi)))
	    (cond
	      ((and (consp option) (get-location-or-nil option :default)
		    (eql (and (fixnump ch)
			      (int-char ch)) #\SPACE))
	       (values (get option :default) :default))
	       ((and (consp ch) (eq (car ch) :activation)) (send stream :tyo (cadr ch))
	       (values () :end))
	      ((eql (and (fixnump ch) (int-char ch)) #\END)
	       (values () :end))
	      (t (unless (consp ch)
		   (send stream :untyi ch))
	       (values (read stream)))))))
       (if flag
	 (return (values form flag))
	 (setq value (eval-abort-trivial-errors form))))
    ;1; If FORM was not trivial, ask for confirmation of the value it returned.*
    (when (or (trivial-form-p form)
	(let ((*print-level* eval-read-prinlevel)
	      (*print-length* eval-read-prinlength))
	  (fquery '(:list-choices nil) "The object is ~S, ok? " value)))
      (return value))
    (terpri stream)))

(defprop :read read-prompt-and-read prompt-and-read-function) 

(defprop :expression read-prompt-and-read prompt-and-read-function) 

(defun read-prompt-and-read (ignore stream)
  (values (read stream t))) 


(defprop :read ((:prompt prompt-and-read-prompt-function) (:activation = #\END))
   prompt-and-read-rubout-options) 


(defprop :expression ((:prompt prompt-and-read-prompt-function) (:activation = #\END))
   prompt-and-read-rubout-options) 


(defprop :expression-or-end expression-or-end-prompt-and-read prompt-and-read-function) 

(defprop :expression-or-end ((:prompt prompt-and-read-prompt-function) (:activation = #\END))
   prompt-and-read-rubout-options) 

(defun expression-or-end-prompt-and-read (ignore stream)
  (let ((ch (send stream :any-tyi)))
    (if (or (and (consp ch) (eq (car ch) :activation)) (and (not rubout-handler) (eql (int-char ch) #\END)))
      (progn
	(if (consp ch)
	  (send stream :tyo (cadr ch)))
	(values () :end))
      (progn
	(when (atom ch)
	  (send stream :untyi ch))
	(values (read stream t)))))) 


(defun (:property :character prompt-and-read-no-rubout-function) (option stream)
  (block char
    (prompt-and-read-prompt-function stream ())
    (let ((char (read-char stream () ()) )
	  (*standard-output* stream))
      (when (and (consp option) (get option :or-nil))
	(cond
	  ((member char '(#\QUOTE #\c-Q) :test #'eq) (setq char (read-char stream () ()) ))
	  ((= char #\CLEAR-INPUT) (princ "none") (return-from char ()))))
      (format t "~:C" char)
      char)))


(defun (:property :character-list prompt-and-read-function) (ignore stream)
  (concatenate 'list (read-line stream))) 


(defun (:property :number prompt-and-read-function) (option stream)
  (let ((*read-base* (or (and (consp option) (get option :input-radix)) *read-base*))
	(string (readline-trim stream)))
    (if (and (consp option) (get option :or-nil) (equal string ""))
      ()
      (condition-case ()
	 (let* ((number (read-from-string string t)))
	   (if (numberp number)
	     number
	     (ferror 'read-error-1 "That is not a number.")))
	 (end-of-file (ferror 'read-error-1 "That is not a number.")))))) 


(defun (:property :integer prompt-and-read-function) (option stream)
  (let ((*read-base* (or (and (consp option) (get option :input-radix)) *read-base*))
	(string (readline-trim stream)))
    (if (and (consp option) (get option :or-nil) (equal string ""))
      ()
      (condition-case ()
	 (let* ((number (read-from-string string t)))
	   (if (and (numberp number) (integerp number))
	     number
	     (ferror 'read-error-1 "That is not an integer.")))
	 (end-of-file (ferror 'read-error-1 "That is not an integer.")))))) 


(defun (:property :small-fraction prompt-and-read-function) (option stream)
  (let ((string (readline-trim stream)))
    (if (and (consp option) (get option :or-nil) (equal string ""))
      ()
      (condition-case ()
	 (let* ((number (read-from-string string t)))
	   (if (and (numberp number) (realp number) (<= 0.0 number 1.0))
	     (float number)
	     (ferror 'read-error-1 "That is not a fraction between 0 and 1.")))
	 (end-of-file (ferror 'read-error-1 "That is not a fraction between 0 and 1.")))))) 


(defun (:property :date prompt-and-read-function) (option stream)
  (let ((string (readline-trim stream)))
    (if (equalp string "never")
      (if (and (consp option) (get option :never-p))
	()
	(ferror 'read-error-1 "Never is not allowed here."))
      (let* ((past-p (and (consp option) (get option :past-p)))
	     (date
	      (condition-case (error) (time:parse-universal-time string 0 () (not past-p))
		 (time:parse-error (ferror 'read-error-1 "~A" (send error :report-string))))))
	(and past-p (> date (get-universal-time))
	   (ferror 'read-error-1 "~A is not in the past." (time:print-universal-time date ())))
	date)))) 


(defun (:property :string-or-nil prompt-and-read-function) (ignore stream)
  (readline-or-nil stream)) 


(defun (:property :string prompt-and-read-function) (ignore stream)
  (zlc:readline stream)) 


(defun (:property :string-trim prompt-and-read-function) (ignore stream)
  (readline-trim stream)) 


(defun (:property :string-list prompt-and-read-function) (ignore stream)
  (let ((str1 (zlc:readline stream))
	j
	accum)
    (do ((i 0))
	(nil)
      (setq j (position #\, (the string (string str1)) :start i :test #'char-equal))
      (let ((str2 (string-trim " " (subseq str1 i j))))
	(unless (equal str2 "")
	  (push str2 accum)))
      (unless j
	(return (nreverse accum)))
      (setq i (1+ j))))) 


(defun (:property :pathname prompt-and-read-function) (option stream)
  (let ((defaults (if (consp option)
		    (get option :defaults)
		    *default-pathname-defaults*))
	(string (zlc:readline stream)))
    (fs:merge-pathname-defaults string defaults fs:*name-specified-default-type*
				(or (and (consp option) (get option :default-version)) :newest)))) 


(defun (:property :pathname-or-end prompt-and-read-function) (option stream)
  (let ((defaults (if (consp option)
		      (get option :defaults)
		      *default-pathname-defaults*)))
    (multiple-value-bind (string () terminator)
	(zlc:readline stream)
      (if (and (equal string "") (eql terminator #\END))
	  #\END
	  (fs:merge-pathname-defaults string defaults fs:*name-specified-default-type*
				      (or (and (consp option) (get option :default-version))
					  :newest)))))) 


(defun (:property :pathname-or-nil prompt-and-read-function) (option stream)
  (let ((defaults (if (consp option)
		      (get option :defaults)
		      *default-pathname-defaults*)))
    (multiple-value-bind (string () terminator)
	(zlc:readline stream)
      (unless (and (equal string "") (eql terminator #\END))
	(fs:merge-pathname-defaults string defaults fs:*name-specified-default-type*
				    (or (and (consp option) (get option :default-version))
					:newest)))))) 


(defun (:property :fquery prompt-and-read-no-rubout-function) (option *query-io*)
  (apply 'fquery (if (consp option)
		     (cdr option))
	 prompt-and-read-format-string prompt-and-read-format-args)) 

;1Obsolete.*

(defun (:property fquery prompt-and-read-no-rubout-function) (option *query-io*)
  (apply 'fquery (if (consp option)
		     (cdr option))
	 prompt-and-read-format-string prompt-and-read-format-args)) 


(defun (:property :delimited-string prompt-and-read-no-rubout-function) (option stream)
  (read-delimited-string (or (and (consp option) (get option :delimiter)) #\END) stream ()
			 '((:prompt prompt-and-read-prompt-function))
			 (or (and (consp option) (get option :buffer-size)) 64))) 


(defun (:property :delimited-string-or-nil prompt-and-read-no-rubout-function) (option stream)
  (let ((string
	  (read-delimited-string (or (and (consp option) (get option :delimiter)) #\END) stream
				 () '((:prompt prompt-and-read-prompt-function))
				 (or (and (consp option) (get option :buffer-size)) 64))))
    (if (equal string "")
	()
	string)))

(defun (:property :choose prompt-and-read-no-rubout-function) (option *query-io*)
  ;1; 11/20/87 CLM - Modified to use ZLC:MEMQ instead of MEMQ.  MEMQ no longer accessible in*
  ;1;                SYS package.*
  (let ((choices (get option :choices)))
    (with-input-editing
      (*query-io*
	`((:prompt
	    ,(function
	       (lambda (&rest args)
		 (apply 'prompt-and-read-prompt-function args)
		 (fresh-line *query-io*)
		 (do ((choices choices (cdr choices))
		      (i 0 (1+ i)))
		     ((null choices))
		   (format *query-io* "~& Type ~D for ~S" i (car choices)))
		 (terpri *query-io*))))
	  (:activation zlc:memq (#\END #\NEWLINE))))
      (nth (read *query-io*) choices)))) 


(defun (:property :assoc prompt-and-read-no-rubout-function) (option *query-io*)
  ;1; 11/20/87 CLM - Modified to use ZLC:MEMQ instead of MEMQ.  MEMQ no longer accessible in*
  ;1;                SYS package.*
  (let ((choices (get option :choices)))
    (with-input-editing
      (*query-io*
	`((:prompt
	    ,(function
	       (lambda (&rest args)
		 (apply 'prompt-and-read-prompt-function args)
		 (fresh-line *query-io*)
		 (do ((choices choices (cdr choices))
		      (i 0 (1+ i)))
		     ((null choices))
		   (format *query-io* "~& Type ~D for ~S" i (caar choices)))
		 (terpri *query-io*))))
	  (:activation zlc:memq (#\END #\NEWLINE))))
      (cdr (nth (read *query-io*) choices))))) 


(defun (:property :boolean prompt-and-read-no-rubout-function) (ignore *query-io*)
  (apply 'y-or-n-p prompt-and-read-format-string prompt-and-read-format-args)) 













