/*
 * dumpq.c
 *
 * dump the contents of a Q semi-readably.
 */

#include "utils.h"

#include <stdio.h>
#include <ctype.h>
#include "types.h"

#include "memory.h"
#include "interp.h"

char *cdrcodes[4] = {
    "CDR-NORMAL",
    "CDR-ERROR ",
    "CDR-NIL   ",
    "CDR-NEXT  "
};

char *dtpcodes[32] = {
    "DTP-Trap",
    "DTP-List",
    "DTP-Stack-List",
    "DTP-Symbol",
    "DTP-Array",
    "DTP-Fix",
    "DTP-Character",
    "DTP-Single-Float",
    "DTP-Short-Float",
    "DTP-Instance",
    "DTP-Extended-Number",
    "DTP-Locative",
    "DTP-Function",
    "DTP-Closure",
    "DTP-Lexical-Closure",
    "DTP-U-Entry",
    "DTP-Stack-Group",
    "DTP-GC-Forward",
    "DTP-EVCP",
    "DTP-One-Q-Forward",
    "DTP-Header-Forward",
    "DTP-Body-Forward",
    "DTP-Symbol-Header",
    "DTP-Header", /* _Not_ in SSDN2 */
    "DTP-Array-Header",
    "DTP-Instance-Header",
    "DTP-Fef-Header",
    "DTP-Self-Ref-Pointer",
    "DTP-GC-Young-Pointer",
    "DTP-Free",
    "DTP-Null",
    "DTP-Ones-Trap"
};

void dump_dtp(u32 data)
{
    printf("%s", dtpcodes[(data >> 25) & 31]);

    if (DTP(data) == DTP_SYMBOL) {
	printf("\t");
	dump_string(memread(data));
    } else if (DTP(data) == DTP_FIX) {
	printf("\t\t%d\n", fixnum_value(data));
    } else if (DTP(data) == DTP_EVCP) {
	lisp_q foo;
	lisp_q bar;
	
	foo = find_structure_header(data);

	bar = memread(foo);
	if (DTP(bar) == DTP_SYMBOL_HEADER) {
	    if ((ADDRESS(foo) + SYM_VALUE) == ADDRESS(data)) {
		printf("\t");
		dump_string(bar);
	    } else if ((ADDRESS(foo) + SYM_FUNCTION) == ADDRESS(data)) {
		printf("\t#'");
		dump_string(bar);
	    } else {
		printf("\n");
	    }
	} else {
	    printf("\n");
	}
    } else if (DTP(data) == DTP_FUNCTION) {
	printf("\t");
	dump_function_name(data);
    } else if (DTP(data) == DTP_ARRAY) {
	lisp_q array_header;

	printf("\t");

	array_header = memread(data);
	if (DTP(array_header) != DTP_ARRAY_HEADER) {
	    printf("<corrupt header>\n");
	    return;
	}

	if (ARY_NS(array_header)) {
	    lisp_q ary_type_name;

	    if (ARY_LEADER(array_header)) {
		ary_type_name = memread(data - 3);
	    } else {
		ary_type_name = memread(data + !!ARY_LL(array_header) +
					ARY_DIMS(array_header));
	    }

	    printf("<a ");
	    internal_dump_string(memread(ary_type_name));
	    printf(">\n");
	} else if (ARY_TYPE(array_header) == ART_STRING) {
	    printf("\"");
	    internal_dump_string(data);
	    printf("\"\n");
	} else {
	    printf("<%s>\n", ary_type_names[ARY_TYPE(array_header) >> 19]);
	}
    } else {
	printf("\n");
    }
}

void dump_q(u32 data, u32 address)
{
    printf("%08lx: %08lx '%c%c%c%c'  %s   ",
	   address, data,
	   (int)(isprint( data        & 0xff)?  data        & 0xff: '.'),
	   (int)(isprint((data >>  8) & 0xff)? (data >>  8) & 0xff: '.'),
	   (int)(isprint((data >> 16) & 0xff)? (data >> 16) & 0xff: '.'),
	   (int)(isprint( data >> 24        )?  data >> 24        : '.'),
	   cdrcodes[data >> 30]);

    dump_dtp(data);
}

/* EOF */
