/*
 * mainop.c
 *
 * mainop instruction handlers
 */

#include "exploiter.h"
#include "context.h"
#include "memory.h"
#include "interp.h"
#include "utils.h"
#include "funcall.h"
#include "nubus.h"
#include <stdio.h>
#include <stdlib.h>

typedef int (*mainop_handler)(u16 opcode);

#define MAINOP(foo) int mainop_##foo(u16 opcode)


MAINOP(003) { /* EQ-IMMED */
    lisp_q foo;

    foo = (DTP_FIX | (opcode & 0xff) | ((opcode & 0x100)? 0x01ffff00: 0));
    context.indicators = (NOT_CDRCODE(pop()) == foo)? C_T: C_NIL;

    return 1;
}

MAINOP(004) { /* =-IMMED */
    lisp_q foo;
    lisp_q bar;
	
    foo = opcode & 0xff;
    if (opcode & 0x100) foo |= 0x01ffff00;
    foo |= DTP_FIX;

    bar = NOT_CDRCODE(pop());

    context.indicators = (foo == bar)? C_T: C_NIL;
    
    return 1;
}

MAINOP(005) { /* >-IMMED */
    lisp_q foo;
    lisp_q bar;
	
    foo = NOT_CDRCODE(pop());
    
    bar = opcode & 0xff;
    if (opcode & 0x100) bar |= 0x01ffff00;
    bar |= DTP_FIX;

    context.indicators = (foo > bar)? C_T: C_NIL;

    return 1;
}

MAINOP(006) { /* <-IMMED */
    lisp_q foo;
    lisp_q bar;
	
    foo = NOT_CDRCODE(pop());

    bar = opcode & 0xff;
    if (opcode & 0x100) bar |= 0x01ffff00;
    bar |= DTP_FIX;

    context.indicators = (foo < bar)? C_T: C_NIL;

    return 1;
}

MAINOP(010) { /* TEST */
    context.indicators = load_target(opcode);

    dump_q(context.indicators, 0);

    return 1;
}

MAINOP(011) { /* TEST-CAR */
    context.indicators = car(load_target(opcode));

    return 1;
}

MAINOP(016) { /* TEST-MEMQ */
    lisp_q foo;
    lisp_q list;
    
    list = load_target(opcode);
    foo = NOT_CDRCODE(pop());
    
    while((NOT_CDRCODE(list) != (C_NIL)) &&
	  (NOT_CDRCODE(car(list)) != foo)) {
	list = cdr(list);
    }
    
    context.indicators = list;

    return 1;
}

MAINOP(017) { /* RETURN */
    return_1(load_target(opcode));

    return 1;
}

MAINOP(020) { /* = */
    lisp_q foo;
    lisp_q bar;

    /* FIXME: Check datatypes. Make work on non-FIXNUM types */

    foo = NOT_CDRCODE(load_target(opcode));
    bar = NOT_CDRCODE(pop());
    dump_q(foo, 0);
    dump_q(bar, 1);

    if ((DTP(foo) != DTP_FIX) && (DTP(foo) != DTP_CHARACTER)) {
	printf("Argument not DTP_FIX.\n");
	return 0;
    }
    
    if ((DTP(bar) != DTP_FIX) && (DTP(bar) != DTP_CHARACTER)) {
	printf("Argument not DTP_FIX.\n");
	return 0;
    }
    
    context.indicators = (ADDRESS(foo) == ADDRESS(bar))? C_T: C_NIL;
    
    return 1;
}

MAINOP(021) { /* > */
    lisp_q foo;
    lisp_q bar;

    /* FIXME: Check datatypes. Make work on non-FIXNUM types */

    bar = load_target(opcode);
    foo = pop();
    dump_q(foo, 0);
    dump_q(bar, 1);

    if ((DTP(foo) != DTP_FIX) && (DTP(foo) != DTP_CHARACTER)) {
	printf("Argument not DTP_FIX or DTP_CHARACTER.\n");
	return 0;
    }
    
    if ((DTP(bar) != DTP_FIX) && (DTP(bar) != DTP_CHARACTER)) {
	printf("Argument not DTP_FIX or DTP_CHARACTER.\n");
	return 0;
    }

    foo = fixnum_value(foo);
    bar = fixnum_value(bar);

    context.indicators = (foo > bar)? C_T: C_NIL;

    return 1;
}

MAINOP(022) { /* < */
    lisp_q foo;
    lisp_q bar;

    /* FIXME: Check datatypes. Make work on non-FIXNUM types */

    bar = load_target(opcode);
    foo = pop();
    dump_q(foo, 0);
    dump_q(bar, 1);

    if ((DTP(foo) != DTP_FIX) && (DTP(foo) != DTP_CHARACTER)) {
	printf("Argument not DTP_FIX or DTP_CHARACTER.\n");
	return 0;
    }
    
    if ((DTP(bar) != DTP_FIX) && (DTP(bar) != DTP_CHARACTER)) {
	printf("Argument not DTP_FIX or DTP_CHARACTER.\n");
	return 0;
    }

    foo = fixnum_value(foo);
    bar = fixnum_value(bar);

    context.indicators = (foo < bar)? C_T: C_NIL;

    return 1;
}

MAINOP(023) { /* EQ */
    lisp_q foo;
    lisp_q bar;

    foo = NOT_CDRCODE(pop());
    bar = NOT_CDRCODE(load_target(opcode));
    context.indicators = (foo == bar)? C_T: C_NIL;

    return 1;
}

MAINOP(024) { /* EQL */
    lisp_q foo;
    lisp_q bar;

    foo = NOT_CDRCODE(pop());
    bar = NOT_CDRCODE(load_target(opcode));
    dump_q(foo, 0);
    dump_q(bar, 1);
    context.indicators = (foo == bar)? C_T: C_NIL;

    return 1;
}

MAINOP(025) { /* EQUAL */
    lisp_q foo;
    lisp_q bar;
    lisp_q result;
    
    foo = load_target(opcode);
    bar = pop();
/*     dump_q(foo, 0); */
/*     dump_q(bar, 1); */

/*     print_list(foo); */
/*     printf("\n"); */
/*     print_list(bar); */
/*     printf("\n"); */

/*     dump_q(equal(foo, bar), 0); */

    result = equal(foo, bar);
    
    push_cdrnext(result);
    context.indicators = result;
    
    return 1;
}

MAINOP(030) { /* NUMBERP */
    lisp_q foo;

    foo = NOT_CDRCODE(load_target(opcode));

    dump_q(foo, 0);
    
    if (DTP(foo) == DTP_FIX) {
	context.indicators = C_T;
    } else if (DTP(foo) == DTP_EXTENDED_NUMBER) {
	context.indicators = C_T;
    } else if (DTP(foo) == DTP_SINGLE_FLOAT) {
	context.indicators = C_T;
    } else if (DTP(foo) == DTP_SHORT_FLOAT) {
	context.indicators = C_T;
    } else {
	context.indicators = C_NIL;
    }
    
    return 1;
}

MAINOP(031) { /* ARRAYP */
    lisp_q foo;

    foo = NOT_CDRCODE(load_target(opcode));
    context.indicators = (DTP(foo) == DTP_ARRAY) ? C_T: C_NIL;

    return 1;
}

MAINOP(032) { /* LISTP */
    lisp_q foo;

    foo = NOT_CDRCODE(load_target(opcode));
    context.indicators = ((DTP(foo) == DTP_LIST)
			  || (DTP(foo) == DTP_STACK_LIST)) ? C_T: C_NIL;

    return 1;
}

MAINOP(033) { /* STRINGP */
    lisp_q foo;
    lisp_q bar;

    context.indicators = C_NIL;
    foo = NOT_CDRCODE(load_target(opcode));
    if (DTP(foo) == DTP_ARRAY) {
	bar = memread(foo);
	if (DTP(bar) == DTP_ARRAY_HEADER) {
	    dump_array_header(bar);
	    if ((ARY_TYPE(bar) == ART_STRING) ||
		(ARY_TYPE(bar) == ART_FAT_STRING))
		context.indicators = C_T;
	} else {
	    printf("STRINGP: Array header not DTP-ARRAY-HEADER.\n");
	    dump_q(bar, 0);
	    return 0;
	}
    }

    return 1;
}

MAINOP(034) { /* FIXNUMP */
    lisp_q foo;

    foo = NOT_CDRCODE(load_target(opcode));
    context.indicators = (DTP(foo) == DTP_FIX) ? C_T: C_NIL;

    return 1;
}

MAINOP(035) { /* INTEGERP */
    lisp_q foo;

    foo = NOT_CDRCODE(load_target(opcode));
    context.indicators = (DTP(foo) == DTP_FIX) ? C_T: C_NIL;

    return 1;
}

MAINOP(036) { /* PLUSP */
    lisp_q foo;

    /* FIXME: Check datatypes. Make work on non-FIXNUM types */
    
    foo = load_target(opcode);
    dump_q(foo, 0);
    
    if (DTP(foo) != DTP_FIX) {
	printf("Argument not DTP_FIX.\n");
	return 0;
    }
    
    context.indicators = ((foo & 0x01000000) || !ADDRESS(foo)) ? C_NIL : C_T;
    dump_q(context.indicators, 0);

    return 1;
}

MAINOP(037) { /* MINUSP */
    lisp_q foo;

    /* FIXME: Check datatypes. Make work on non-FIXNUM types */
    
    foo = load_target(opcode);

    dump_q(foo, 0);
    
    if (DTP(foo) != DTP_FIX) {
	printf("Argument not DTP_FIX.\n");
	return 0;
    }
    
    context.indicators = (foo & 0x01000000) ? C_T: C_NIL;

    return 1;
}

MAINOP(043) { /* ADD-IMMED */
    lisp_q foo;
    int value;

    foo = pop();
    if (DTP(foo) != DTP_FIX) {
	printf("ADD-IMMED: Not FIXNUM.\n");
	dump_q(foo, 0);
	return 0;
    }

    value = (opcode & 0x1ff);
    if (value & 0x100) value |= ~0x1ff;
    
    push_cdrnext(DTP_FIX | ADDRESS(ADDRESS(foo) + value));

    return 1;
}

MAINOP(044) { /* LDB-IMMED */
    lisp_q foo;
    lisp_q result;
    int retval;

    foo = pop();

    retval = ldb_generic(foo, (1 << (opcode & 15)) -1,
			 (opcode >> 4) & 0x1f, opcode & 15, &result);
    
    push_cdrnext(result);

    return retval;
}

MAINOP(045) { /* PUSH-NUMBER */
    push_cdrnext(DTP_FIX | (opcode & 0x1ff));

    return 1;
}

MAINOP(046) { /* PUSH-NEG-NUMBER */
    push_cdrnext(DTP_FIX | (Q_BITS_ADDRESS & (0 - (opcode & 0x1ff))));

    return 1;
}

MAINOP(050) { /* PUSH */
    context.indicators = load_target(opcode);

    push_cdrnext(context.indicators);

    return 1;
}

MAINOP(051) { /* PUSH-CAR */
    push_cdrnext(car(load_target(opcode)));

    return 1;
}

MAINOP(052) { /* PUSH-CDR */
    push_cdrnext(cdr(load_target(opcode)));

    return 1;
}

MAINOP(053) { /* PUSH-CADR */
    push_cdrnext(car(cdr(load_target(opcode))));

    return 1;
}

MAINOP(054) { /* PUSH-CDDR */
    push_cdrnext(cdr(cdr(load_target(opcode))));

    return 1;
}

MAINOP(056) { /* PUSH-CONS */
    lisp_q car;
    lisp_q cdr;
    lisp_q cons;

    cdr = load_target(opcode);
    car = pop();
    
    cons = allocate_list_qs(C_NIL, 2);

    memwrite(cons, CDR_NORMAL | NOT_CDRCODE(car));
    memwrite(cons+1, CDR_ERROR | NOT_CDRCODE(cdr));

    push_cdrnext(DTP_LIST | ADDRESS(cons));
    
    return 1;
}

MAINOP(057) { /* PUSH-GET */
    lisp_q sym;
    lisp_q prop;
    lisp_q plist;
    lisp_q result;

    prop = NOT_CDRCODE(load_target(opcode));
    sym = pop();

    dump_q(sym, 0);
    dump_q(prop, 1);

    plist = memread(sym + SYM_PROPERTY);

    dump_q(plist, 2);
    
    result = C_NIL;
    if (DTP(plist) != DTP_NULL) {
	while(NOT_CDRCODE(plist) != (C_NIL)) {
	    if (NOT_CDRCODE(car(plist)) == prop) {
		result = car(cdr(plist));
	    }
	    plist = cdr(cdr(plist));
	}
    }

    dump_q(result, 2);

    push_cdrnext(result);

    return 1;
}

MAINOP(060) { /* + */
    lisp_q src1;
    lisp_q src2;

    src1 = pop();
    src2 = load_target(opcode);

    return math_plus(src1, src2);
}

MAINOP(061) { /* - */
    lisp_q src1;
    lisp_q src2;
    lisp_q result;
    int retval;

    src2 = load_target(opcode);
    src1 = pop();

    retval = math_minus(src1, src2);

    result = pop();
    push_cdrnext(result);

    dump_q(src1, 0);
    dump_q(src2, 1);
    dump_q(result, 2);

    return retval;
}

MAINOP(062) { /* * */
    lisp_q src1;
    lisp_q src2;

    src1 = pop();
    src2 = load_target(opcode);

    return math_times(src1, src2);
}

MAINOP(063) { /* LOGAND */
    lisp_q src1;
    lisp_q src2;
    lisp_q result;

    /* The description of this op in SSDN2 is rather unilluminating */
	
    src1 = pop();
    src2 = load_target(opcode);
    
    if ((DTP(src1) != DTP_FIX) || (DTP(src2) != DTP_FIX)) {
	printf("One or both args to LOGAND is not DTP_FIX, failing.\n");
	dump_q(src1, 1);
	dump_q(src2, 2);
	return 0;
    }

    result = DTP_FIX | ADDRESS(src1 & src2);
    push_cdrnext(result);

    return 1;
}

MAINOP(064) { /* LOGXOR */
    lisp_q src1;
    lisp_q src2;
    lisp_q result;

    /* The description of this op in SSDN2 is rather unilluminating */
	
    src1 = pop();
    src2 = load_target(opcode);
    
    if ((DTP(src1) != DTP_FIX) || (DTP(src2) != DTP_FIX)) {
	printf("One or both args to LOGXOR is not DTP_FIX, failing.\n");
	dump_q(src1, 1);
	dump_q(src2, 2);
	return 0;
    }

    result = DTP_FIX | ADDRESS(src1 ^ src2);
    push_cdrnext(result);

    return 1;
}

MAINOP(065) { /* 1+ */
    lisp_q foo;
	
    /* FIXME: What about numbers other than FIXNUMs? */

    foo = load_target(opcode);
    foo = ADDRESS(foo);
    /* FIXME: Check for FIXNUM overflow */
    push_cdrnext(DTP_FIX | (foo + 1));

    return 1;
}

MAINOP(066) { /* 1- */
    lisp_q foo;

    foo = load_target(opcode);
    foo--;
    push_cdrnext(DTP_FIX | ADDRESS(foo));

    return 1;
}

MAINOP(067) { /* PUSH-AR-1 */
    lisp_q array;
    lisp_q result;

    int status;
    lisp_q locative;
    int offset;
    lisp_q header;

    array = load_target(opcode);

    dump_q(array, 0);
    
    status = resolve_aref(array, 1, 0, context.pdl_pointer - 1, &header, &locative, &offset);
    if (!status) return 0;

    status = read_aref(header, locative, offset, &result, 0);

    memwrite(context.pdl_pointer - 1, CDR_NEXT | NOT_CDRCODE(result));

    return status;
}

MAINOP(071) { /* SELECT */
    lisp_q x;
    lisp_q table;
    int table_length;
    int i;
    
    x = NOT_CDRCODE(pop());
    table = context.function + (opcode & 0x1ff);

    table_length = fixnum_value(memread(table));

    for (i = 0; i < table_length; i++) {
	if (NOT_CDRCODE(memread(table + i + 1)) == x) break;
    }

    if (i == table_length) {
	i = 0;
    } else {
	i++;
    }

    x = memread(table + table_length + i + 2);

    context.location_counter = fixnum_value(x);
    
    return 1;
}

MAINOP(072) { /* DISPATCH */
    lisp_q index;
    lisp_q max_index;
    lisp_q offset;

    index = pop();
    printf("DISPATCH.\n");
    dump_q(index, 0);
    max_index = memread(context.function + (opcode & 0x1ff));
    /* FIXME: Check index against max_index */
    /* What's the number at FEF[whatever] + 1? */
    offset = memread(context.function + (opcode & 0x1ff) + 2 + ADDRESS(index));
    dump_q(offset, 1);
    context.location_counter = ADDRESS(offset);

    return 1;
}

MAINOP(140) { /* POP */
    store_target(opcode, pop());

    return 1;
}

MAINOP(141) { /* MOVEM */
    lisp_q foo;

    foo = pop();
    push_cdrnext(foo);
    store_target(opcode, foo);

    return 1;
}

MAINOP(142) { /* SETE-CDR */
    lisp_q foo;

    foo = load_target(opcode);
    foo = cdr(foo);
    store_target(opcode, foo);

    return 1;
}

MAINOP(143) { /* SETE-CDDR */
    lisp_q foo;

    foo = load_target(opcode);
    foo = cdr(cdr(foo));
    store_target(opcode, foo);

    return 1;
}

MAINOP(144) { /* SETE-1+ */
    lisp_q foo;

    foo = load_target(opcode);
    foo++;
    store_target(opcode, foo);

    return 1;
}

MAINOP(145) { /* SETE-1- */
    lisp_q foo;

    foo = load_target(opcode);
    dump_q(foo, 0);
    foo = NOT_ADDRESS(foo) | ADDRESS(foo-1);
    store_target(opcode, foo);

    return 1;
}

MAINOP(147) { /* PUSH-CDR-Store-CAR-IF-CONS */
    lisp_q foo;

    foo = pop();

    /* FIXME: Typecheck. */
    
    /* FIXME: If this is called for target PDL, which gets pushed first? */
    /* FIXME: Is this really the correct logic for when it isn't a cons? */
    if ((DTP(foo) == DTP_LIST) ||
	(DTP(foo) == DTP_STACK_LIST)) {
	push_cdrnext(cdr(foo));
	store_target(opcode, car(foo));
	context.indicators = C_T;
    } else {
	context.indicators = C_NIL;
    }

    return 1;
}

MAINOP(150) { /* PUSH-LOC */
    push_cdrnext(locative_target(opcode));

    return 1;
}


MAINOP(151) { /* BIND-NIL */
    bind(locative_target(opcode), C_NIL);

    return 1;
}

MAINOP(152) { /* BIND-T */
    bind(locative_target(opcode), C_T);

    return 1;
}

MAINOP(153) { /* BIND-POP */
    bind(locative_target(opcode), pop());

    return 1;
}

MAINOP(155) { /* SET-NIL */
    store_target(opcode, C_NIL);

    return 1;
}

MAINOP(156) { /* SET-T */
    store_target(opcode, C_T);

    return 1;
}

MAINOP(157) { /* SET-ZERO */
    store_target(opcode, DTP_FIX);

    return 1;
}

MAINOP(160) { /* BR-NIL-ELSE-POP */
    if (NOT_CDRCODE(context.indicators) == C_NIL) {
	do_branch(opcode);
    } else {
	pop();
    }

    return 1;
}

MAINOP(161) { /* BR-NOT-NIL-ELSE-POP */
    if (NOT_CDRCODE(context.indicators) != C_NIL) {
	do_branch(opcode);
    } else {
	pop();
    }

    return 1;
}

MAINOP(162) { /* BR-NIL */
    if (NOT_CDRCODE(context.indicators) == C_NIL) {
	do_branch(opcode);
    }

    return 1;
}

MAINOP(163) { /* BR-NOT-NIL */
    if (NOT_CDRCODE(context.indicators) != C_NIL) {
	do_branch(opcode);
    }

    return 1;
}

MAINOP(164) { /* BR-ATOM */
    int x;

    x = context.indicators;

    if ((DTP(x) != DTP_LIST) && (DTP(x) != DTP_STACK_LIST)) {
	do_branch(opcode);
    }
    
    return 1;
}

MAINOP(165) { /* BR-NOT-ATOM */
    int x;

    x = context.indicators;

    if ((DTP(x) == DTP_LIST) || (DTP(x) == DTP_STACK_LIST)) {
	do_branch(opcode);
    }
    
    return 1;
}

MAINOP(166) { /* BR-ZEROP */
    int x;

    x = context.indicators;

    /* FIXME: What about numbers other than FIXNUMs? */
    if (NOT_CDRCODE(x) == DTP_FIX) {
	do_branch(opcode);
    }

    return 1;
}

MAINOP(167) { /* BR-NOT-ZEROP */
    int x;

    x = context.indicators;

    /* FIXME: What about numbers other than FIXNUMs? */
    if ((DTP(x) == DTP_FIX) && ADDRESS(x)) {
	do_branch(opcode);
    }

    return 1;
}

MAINOP(170) { /* BR-SYMBOLP */
    lisp_q x;

    x = context.indicators;

    if (DTP(x) == DTP_SYMBOL) {
	do_branch(opcode);
    }

    return 1;
}

MAINOP(171) { /* BR-NOT-SYMBOLP */
    lisp_q x;

    x = context.indicators;

    if (DTP(x) != DTP_SYMBOL) {
	do_branch(opcode);
    }

    return 1;
}

MAINOP(176) { /* BR */
    do_branch(opcode);

    return 1;
}

int arefiop(u16 opcode) { /* *-AREFI */
    lisp_q src;
    lisp_q ary_header;
    lisp_q data;
    lisp_q address;
    int size;
    nubus_result result;
    int physical;
    lisp_q retval;
    int status;
    lisp_q locative;
    int offset;

    if (opcode & 0x0100) data = pop();
    
    src = pop();
    physical = 0;

    ary_header = memread(src);
    if (DTP(ary_header) != DTP_ARRAY_HEADER) {
	printf("AREFI: bad array header.\n");
	dump_q(src, 0);
	dump_q(ary_header, 0);
	return 0;
    }

    dump_array_header(ary_header);

    switch((opcode >> 6) & 7) {
    case 1:			/* ARRAY-LEADER */
	if (ARY_LEADER(ary_header)) {

	    size = memread(ADDRESS(src) - 1);
	    printf("PUSH-AREFI (ARRAY-LEADER) size %d.\n", ADDRESS(size));
	    address = ADDRESS(src) - ((opcode & 0x3f) + 2);
	} else {
	    printf("PUSH-AREFI: type mismatch.\n");
	    return 0;
	}
	break;

    case 3:			/* AREF (Common Lisp) */
	if (ARY_DISPLACED(ary_header)) {
	    if (ARY_PHYSICAL(ary_header)) {
		address = memread_unboxed(src + 1);
	    } else {
		address = memread(src + 1);
	    }
	    printf("array is displaced to %lx.\n", address);
	    dump_q(address, 0);
	} else if (ARY_LL(ary_header)) {
	    address = ADDRESS(src) + 2;
	} else {
	    address = ADDRESS(src) + 1;
	}
	printf("type %ld offset %d.\n", ARY_TYPE(ary_header) >> 19,
	       (opcode & 0x3f));

	/* FIXME: Doesn't play nice with non-Q-sized elements */
	
	if (ARY_PHYSICAL(ary_header)) {
	    printf("array is physical.\n");
	    address += (opcode & 0x3f) << 2;
	    physical = 1;
	} else {
	    address += (opcode & 0x3f);
	}
	break;

    case 4: /* SETF AREF */
	printf("SETF AREF.\n");

	push_cdrnext(DTP_FIX | (opcode & 0x3f));

	status = resolve_aref(src, 1, 0, context.pdl_pointer - 1, &ary_header, &locative, &offset);
	if (!status) return 0;

	status = write_aref(ary_header, locative, offset, data);
	if (!status) return 0;

	context.pdl_pointer -= 1;
	
	return 1;
	break;

    case 5: /* SETF ARRAY-LEADER */
	if (ARY_LEADER(ary_header)) {
	    size = memread(ADDRESS(src) - 1);
	    printf("PUSH-AREFI (SETF ARRAY-LEADER) size %d.\n", ADDRESS(size));
	    address = ADDRESS(src) - ((opcode & 0x3f) + 2);
	    memwrite(address, data);
	    return 1;
	} else {
	    printf("PUSH-AREFI: type mismatch.\n");
	    return 0;
	}
	break;
	
    default:
	printf("AREFI: Unsupported operation type %d.\n", (opcode >> 6) & 7);
	return 0;
    }

    if (physical) {
	result = nubus_read32(address >> 24, address & 0xffffff, &data);
	if (result != NR_SUCCESS) exit(-1); /* FIXME: Should signal error */
	/* FIXME: Should handle bignum conversions as neccessary. */
	retval = DTP_FIX | ADDRESS(data);
    } else {
	retval = memread(address);
	dump_q(retval, 0);
    }

    if (opcode & 0x4000) {
	push_cdrnext(retval);
    }

    context.indicators = retval;
    
    return 1;
}

int callop(u16 opcode) { /* CALL-* */
    lisp_q function;
    int num_args;

    num_args = (opcode >> 11) & 7;

    function = load_target(opcode);

    if (num_args == 7) {
	/* FIXME: check data type? */
	num_args = ADDRESS(pop());

	printf("callop(): CALL-N case.\n");
    }
    
    funcall(function, DTP_FIX | 0x004000 |
	    num_args | (opcode & 0x600));

    return 1;
}

mainop_handler mainop_dispatch[0x80] = {
    /* 000 - 017 */
    step_auxop, step_miscop,NULL,       mainop_003,
    mainop_004, mainop_005, mainop_006, arefiop,
    mainop_010, mainop_011, NULL,       NULL,
    NULL,       NULL,       mainop_016, mainop_017,

    /* 020 - 037 */
    mainop_020, mainop_021, mainop_022, mainop_023,
    mainop_024, mainop_025, NULL,       NULL,
    mainop_030, mainop_031, mainop_032, mainop_033,
    mainop_034, mainop_035, mainop_036, mainop_037,

    /* 040 - 057 */
    NULL,       step_miscop,NULL,       mainop_043,
    mainop_044, mainop_045, mainop_046, arefiop,
    mainop_050, mainop_051, mainop_052, mainop_053,
    mainop_054, NULL,       mainop_056, mainop_057,

    /* 060 - 077 */
    mainop_060, mainop_061, mainop_062, mainop_063,
    mainop_064, mainop_065, mainop_066, mainop_067,
    NULL,       mainop_071, mainop_072, NULL,
    NULL,       NULL,       NULL,       NULL,


    /* 100 - 117 */
    callop,     callop,     callop,     callop,
    callop,     callop,     callop,     callop,
    callop,     callop,     callop,     callop,
    callop,     callop,     callop,     callop,

    /* 120 - 137 */
    callop,     callop,     callop,     callop,
    callop,     callop,     callop,     callop,
    callop,     callop,     callop,     callop,
    callop,     callop,     callop,     callop,

    /* 140 - 157 */
    mainop_140, mainop_141, mainop_142, mainop_143,
    mainop_144, mainop_145, NULL,       mainop_147,
    mainop_150, mainop_151, mainop_152, mainop_153,
    NULL,       mainop_155, mainop_156, mainop_157,

    /* 160 - 177 */
    mainop_160, mainop_161, mainop_162, mainop_163,
    mainop_164, mainop_165, mainop_166, mainop_167,
    mainop_170, mainop_171, NULL,       NULL,
    NULL,       NULL,       mainop_176, NULL,
};

int step(void)
{
    u16 opcode;
    u16 opnum;

    opcode = get_opcode();
    opnum = opcode >> 9;

    disassemble_instr(context.function, context.location_counter, opcode);

    context.location_counter++;

    if (mainop_dispatch[opnum]) {
	return mainop_dispatch[opnum](opcode);
    } else {
	printf("Unknown opcode (#o%o).\n", opnum);
	return 0;
    }
}

/* EOF */
