/*
 * nubus.c
 *
 * NuBus simulation
 */

#include "nubus.h"

#include <stdio.h>
#include <stdlib.h>

/* FIXME: All of the memory board stuff needs to be moved to a separate file */
#define MEMORY_BOARD_NAME "xxx" /* FIXME: Real name? */

struct nubus_board *nubus_boards[0x10];

nubus_result nubus_read8(u8 slot, u32 offset, u8 *data)
{
    struct nubus_board *board;
    
    if ((slot & 0xf0) != 0xf0) {
	printf("NuBus: Not in slot space.\n");
    } else {
	board = nubus_boards[slot & 15];
    
	if (board) {
	    return board->read8(board, offset, data);
	}

	printf("NuBus: No board in slot.\n");
    }

    printf("nubus_read8: %02x %06lx.\n", slot, offset);

    return NR_BUSERR;
}

nubus_result nubus_read16(u8 slot, u32 offset, u16 *data)
{
    struct nubus_board *board;
    
    if (offset & 1) {
	printf("NuBus: Unaligned halfword access.\n");
    } else if ((slot & 0xf0) != 0xf0) {
	printf("NuBus: Not in slot space.\n");
    } else {
	board = nubus_boards[slot & 15];
	
	if (board) {
	    return board->read16(board, offset, data);
	}
	
	printf("NuBus: No board in slot.\n");
    }
    
    printf("nubus_read16: %02x %06lx.\n", slot, offset);
    
    return NR_BUSERR;
}

nubus_result nubus_read32(u8 slot, u32 offset, u32 *data)
{
    struct nubus_board *board;
    
    if (offset & 3) {
	printf("NuBus: Unaligned word access.\n");
    } else if ((slot & 0xf0) != 0xf0) {
	printf("NuBus: Not in slot space.\n");
    } else {
	board = nubus_boards[slot & 15];
	
	if (board) {
	    return board->read32(board, offset, data);
	}
	
	printf("NuBus: No board in slot.\n");
    }
    
    printf("nubus_read32: %02x %06lx.\n", slot, offset);

    return NR_BUSERR;
}

nubus_result nubus_write8(u8 slot, u32 offset, u8 data)
{
    struct nubus_board *board;
    
    if ((slot & 0xf0) != 0xf0) {
	printf("NuBus: Not in slot space.\n");
    } else {
	board = nubus_boards[slot & 15];
    
	if (board) {
	    return board->write8(board, offset, data);
	}

	printf("NuBus: No board in slot.\n");
    }

    printf("nubus_write8: %02x %06lx = %02x.\n", slot, offset, data);

    return NR_BUSERR;
}

nubus_result nubus_write16(u8 slot, u32 offset, u16 data)
{
    printf("nubus_write16: %02x %06lx = %04hx.\n", slot, offset, data);
    exit(-1);
}

nubus_result nubus_write32(u8 slot, u32 offset, u32 data)
{
    struct nubus_board *board;
    
    if (offset & 3) {
	printf("NuBus: Unaligned word access.\n");
    } else if ((slot & 0xf0) != 0xf0) {
	printf("NuBus: Not in slot space.\n");
    } else {
	board = nubus_boards[slot & 15];
    
	if (board) {
	    return board->write32(board, offset, data);
	}

	printf("NuBus: No board in slot.\n");
    }
    
    printf("nubus_write32: %02x %06lx = %08lx.\n", slot, offset, data);

    return NR_BUSERR;
}

void *nubus_get_memory_base(u8 slot, u32 offset)
{
    struct nubus_board *board;

    board = nubus_boards[slot & 15];

    return board->get_memory_base(board, offset);
}

struct memboard {
    struct nubus_board board;
    u32 ram_size;
    u8 ram_size_byte;
    u8 memory[0];
};

nubus_result memboard_read8(struct nubus_board *board, u32 address, u8 *data)
{
    struct memboard *memboard;

    memboard = (struct memboard *)board;

    /* FIXME: Add #defined constants for all these magic numbers */
    if (address == 0xfffe00) {
	*data = 0x00;
    } else if (address == 0xffff04) {
	*data = 0xc3;
    } else if (address == 0xffff38) {
	*data = 0x00;
    } else if (address == 0xffff3c) {
	*data = 0xfe;
    } else if (address == 0xffff40) {
	*data = 0xff;
    } else if (address == 0xffff84) {
	*data = MEMORY_BOARD_NAME[0];
    } else if (address == 0xffff88) {
	*data = MEMORY_BOARD_NAME[1];
    } else if (address == 0xffff8c) {
	*data = MEMORY_BOARD_NAME[2];
    } else if (address == 0xffff90) {
	*data = memboard->ram_size_byte;
    } else {
	printf("memboard_read8: 0x%06lx Not CROMO-Board-Type-Memory-Size-Offset.\n", address);
	exit(-1);
    }

    return NR_SUCCESS;
}

nubus_result memboard_read16(struct nubus_board *board, u32 address, u16 *data)
{
    struct memboard *memboard;

    memboard = (struct memboard *)board;

    if (address < memboard->ram_size) {
	*data = *((u16 *)(memboard->memory + address));
	return NR_SUCCESS;
    }

    printf("memboard: Read to unmapped space (0x%06lx).\n", address);
    return NR_BUSERR;
}

nubus_result memboard_read32(struct nubus_board *board, u32 address, u32 *data)
{
    struct memboard *memboard;

    memboard = (struct memboard *)board;

    if (address < memboard->ram_size) {
	*data = *((u32 *)(memboard->memory + address));
	return NR_SUCCESS;
    }

    printf("memboard: Read to unmapped space (0x%06lx).\n", address);
    return NR_BUSERR;
}

nubus_result memboard_write8(struct nubus_board *board, u32 address, u8 data)
{
    struct memboard *memboard;

    memboard = (struct memboard *)board;

    if (address < memboard->ram_size) {
	*((u8 *)(memboard->memory + address)) = data;
	return NR_SUCCESS;
    }

    printf("memboard: Write to unmapped space (0x%06lx = 0x%02x).\n", address, data);
    return NR_BUSERR;
}

nubus_result memboard_write32(struct nubus_board *board, u32 address, u32 data)
{
    struct memboard *memboard;

    memboard = (struct memboard *)board;

    if (address < memboard->ram_size) {
	*((u32 *)(memboard->memory + address)) = data;
	return NR_SUCCESS;
    }

    printf("memboard: Write to unmapped space (0x%06lx = 0x%08lx).\n", address, data);
    return NR_BUSERR;
}

void *memboard_get_memory_base(struct nubus_board *board, u32 address)
{
    struct memboard *memboard;

    memboard = (struct memboard *)board;

    if (address < memboard->ram_size) {
	return memboard->memory + address;
    }

    printf("memboard: get base of unmapped space (0x%06lx).\n", address);
    exit(-1);
}

void nubus_create_memory_board(u8 slot, u8 size_byte)
{
    struct memboard *board;
    u32 ram_size;

    ram_size = 1024 * (1 << (size_byte & 15)) * (size_byte >> 4);

    printf("creating memory board in slot 0x%02x, size 0x%08lx.\n", slot, ram_size);

    board = malloc(sizeof(*board) + ram_size);
    board->board.read8 = memboard_read8;
    board->board.read16 = memboard_read16;
    board->board.read32 = memboard_read32;
    board->board.write8 = memboard_write8;
    board->board.write32 = memboard_write32;
    board->board.get_memory_base = memboard_get_memory_base;
    
    board->ram_size = ram_size;
    board->ram_size_byte = size_byte;

    nubus_add_board(slot, &board->board);
}

void nubus_add_board(u8 slot, struct nubus_board *board)
{
    nubus_boards[slot & 15] = board;
}

/* EOF */
