/*
 * miscop.c
 *
 * miscop instruction handlers
 */

#include "exploiter.h"
#include "context.h"
#include "memory.h"
#include "interp.h"
#include "utils.h"
#include <stdio.h>

static lisp_q destination; /* FIXME: This is more than a little ugly */

typedef int (*miscop_handler)(u16 opcode, u16 opnum);

#define MISCOP(foo) int miscop_##foo(u16 opcode, u16 opnum)

MISCOP(054) { /* %PHYS-LOGLDB */
    lisp_q ppss;
    lisp_q slot;
    lisp_q offset;

    offset = pop();
    slot = pop();
    ppss = pop();

    dump_q(ppss, 0);
    dump_q(slot, 1);
    dump_q(offset, 2);

    printf("%%PHYS-LOGLDB: Not implemented.\n");
    
    return 0;
}

MISCOP(063) { /* %P-STORE-TAG-AND-POINTER */
    lisp_q pointer;
    lisp_q misc_fields;
    lisp_q pointer_field;

    pointer_field = pop();
    misc_fields = pop();
    pointer = pop();

    dump_q(pointer, 0);
    dump_q(misc_fields, 1);
    dump_q(pointer_field, 2);

    destination = ADDRESS(pointer_field) | ((misc_fields & 0x7f) << 25);
    dump_q(destination, 3);

    memwrite(pointer, destination);

    return 1;
}

MISCOP(067) { /* %P-STORE-CONTENTS-OFFSET */
    lisp_q value;
    lisp_q base_pointer;
    lisp_q offset;

    offset = pop();
    base_pointer = pop();
    value = pop();

    dump_q(value, 0);
    dump_q(base_pointer, 1);
    dump_q(offset, 2);

    memwrite(offset + base_pointer, value);
    
    return 1;
}

MISCOP(077) { /* NOT-INDICATORS */
    destination = (NOT_CDRCODE(context.indicators) == C_NIL)? C_T: C_NIL;

    return 1;
}

MISCOP(237) { /* FBOUNDP */
    lisp_q foo;

    foo = memread_inviz(pop() + SYM_FUNCTION);

    /* If foo is DTP_NULL, return C_T, else C_NIL */
    destination = (DTP(foo) == DTP_NULL)? C_T: C_NIL;

    return 1;
}

MISCOP(240) { /* BOUNDP */
    lisp_q foo;

    foo = memread_inviz(pop() + SYM_VALUE);

    /* If foo is DTP_NULL, return C_T, else C_NIL */
    destination = (DTP(foo) == DTP_NULL)? C_T: C_NIL;

    return 1;
}

MISCOP(335) { /* SET-AR-1-FORCE */
    lisp_q array;
    lisp_q subscript;
    lisp_q value;

    lisp_q ary_header;
    lisp_q tmp;

    value = pop();
    subscript = pop();
    array = pop();

    dump_q(array, 0);
    dump_q(subscript, 1);
    dump_q(value, 2);

    ary_header = memread(array);
    
    dump_q(ary_header, 3);

    /* FIXME: This supporteth not anything but 4-character string arrays */
    if (NOT_CDRCODE(ary_header) != 0x30489003) return 0;
    if (DTP(subscript) != DTP_FIX) return 0;
    if (DTP(value) != DTP_FIX) return 0;

    /* FIXME: Is this the right way around? */
    tmp = memread(array + 1 + (ADDRESS(subscript) >> 2));
    tmp &= ~(0xff << ((subscript & 3) << 3));
    tmp |= (value & 0xff) << ((subscript & 3) << 3);
    memwrite(array + 1 + (ADDRESS(subscript) >> 2), tmp);
    
    return 1;
}

MISCOP(416) { /* %ALLOCATE-AND-INITIALZE-ARRAY */
    lisp_q header;
    lisp_q index_length;
    lisp_q leader_length;
    lisp_q area;
    lisp_q nqs;
    int i;

    int qs_needed;
    lisp_q array;

    nqs = pop();
    area = pop();
    leader_length = pop();
    index_length = pop();
    header = pop();

    printf("%%ALLOCATE-AND-INITIALIZE-ARRAY unimplemented.\n");
    
    dump_q(header, 0);
    dump_q(index_length, 1);
    dump_q(leader_length, 2);
    dump_q(area, 3);
    dump_q(nqs, 4);

    qs_needed = ADDRESS(nqs) + 1;
    if (ADDRESS(leader_length)) {
	qs_needed += ADDRESS(leader_length) + 1;
    }

    array = allocate_structure_qs(area, qs_needed);

    if (ADDRESS(leader_length)) {
	/* FIXME: initialize leader here */
	printf("%%ALLOCATE-AND-INITIALIZE-ARRAY: take me to your leader.\n");
	return 0;

	array += ADDRESS(leader_length) + 1;
    }

    memwrite(array, NOT_DTP(header) | DTP_ARRAY_HEADER);

    for (i = 1; i < ADDRESS(nqs); i++) {
	memwrite(array + i, C_NIL);
    }

    destination = ADDRESS(array) + DTP_ARRAY;
    
    return 1;
}

MISCOP(435) { /* %NuBus-Read-8B */
    lisp_q slot;
    lisp_q offset;

    offset = pop();
    slot = pop();

    if ((DTP(slot) != DTP_FIX) || (DTP(offset) != DTP_FIX)) {
	printf("arg to %%NuBus-Read-8B not FIXNUM.\n");
	return 0;
    }

    if (ADDRESS(slot) != 0xf4) {
	printf("%%NuBus-Read-8B: Not slot 4.\n");
	return 0;
    }

    if (ADDRESS(offset) != 0xffff90) {
	printf("%%NuBus-Read-8B: Not CROMO-Board-Type-Memory-Size-Offset.\n");
	return 0;
    }

    /*
     * At this point we're looking at the memory size byte in the config
     * ROM for slot 4. This is the slot we set the PMM up to say we have.
     * To make things easier wrt VMM setup, we only want 2 megs on this
     * board. This will hopefully make it so the system doesn't have to
     * set up the PMM, PPD, or PHT in %boot-virtual-memory.
     *
     * "Mem board size: # KB = Mantissa * (2 ** Exp)" (vm-boot)
     *
     * 2 MB is 0x200000 bytes.
     * 1 KB is 0x400 bytes.
     * 0x200000 / 0x400 = 0x800 KB.
     *
     * Exponent is low nybble, Mantissa is high nybble. (lroy-qdev)
     *
     * (1 << 11) is 0x800.
     *
     * So we return 0x1b as a FIXNUM.
     */
    
    destination = DTP_FIX | 0x1b;

    return 1;
}

MISCOP(446) { /* %MAKE-POINTER */
    lisp_q newdtp;
    lisp_q pointer;
	
    pointer = pop();
    newdtp = pop();

    destination = ADDRESS(newdtp) << 25;
    destination |= ADDRESS(pointer);
    
    return 1;
}

MISCOP(447) { /* %MAKE-POINTER-OFFSET */
    lisp_q newdtp;
    lisp_q pointer;
    lisp_q offset;

    offset = pop();
    pointer = pop();
    newdtp = pop();

#if 0 /* If they want negative offsets... */
    if (offset & 0x1000000) {
	printf("%%MAKE-POINTER-OFFSET: negative offset.\n");
	return 0;
    }
#endif

    destination = ADDRESS(newdtp) << 25;
    destination |= ADDRESS(pointer) + ADDRESS(offset);

    return 1;
}

MISCOP(450) { /* %DATA-TYPE */
    lisp_q pointer;

    pointer = pop_loc();

    destination = DTP_FIX | (DTP(memread_inviz(pointer)) >> 25);

    return 1;
}

MISCOP(451) { /* %P-CDR-CODE */
    lisp_q pointer;

    pointer = pop();

    destination = DTP_FIX | (CDRCODE(memread(pointer)) >> 30);

    return 1;
}

MISCOP(452) { /* %P-DATA-TYPE */
    lisp_q pointer;

    pointer = pop();

    destination = DTP_FIX | (DTP(memread(pointer)) >> 25);

    return 1;
}

MISCOP(454) { /* %P-LDB */
    lisp_q ppss;
    lisp_q pointer;
    u32 value;

    pointer = pop();
    ppss = pop();

    printf("%%P-LDB: %04lo.\n", ADDRESS(ppss));
    dump_q(ppss, 0);
    dump_q(pointer, 0);
    
    value = memread_unboxed(pointer);
    value >>= ppss_position(ppss);

    value &= ppss_mask(ppss);

    printf("%%P-LDB: result 0x%08lx.\n", value);
	
    destination = DTP_FIX | value;

    return 1;
}

MISCOP(461) { /* %P-CONTENTS-AS-LOCATIVE */
    lisp_q pointer;

    pointer = pop();

    destination = DTP_LOCATIVE | ADDRESS(memread(pointer));

    return 1;
}

MISCOP(463) { /* %POINTER-DIFFERENCE */
    lisp_q ptr1;
    lisp_q ptr2;

    ptr2 = pop();
    ptr1 = pop();

    destination = DTP_FIX | ADDRESS(ptr1 - ptr2);
    
    return 1;
}

MISCOP(474) { /* %PAGE-STATUS */
    lisp_q ptr;

    /*
     * From SSDN2, section 6:
     *
     * %page-status (virtual-address)
     *     If the page containing virtual-address is swapped out, or if
     *     it is part of  one  of  the  low-numbered  permanently-wired
     *     system  areas,  this returns NIL.  Otherwise, it returns the
     *     entire first word of the Page Hash Table (PHT) entry for the
     *     page.
     */

    ptr = pop();

    dump_q(ptr, 0);

    printf("%%PAGE-STATUS: Unimplemented (no PHT, and not faking it yet).\n");
    
    return 0;
}

MISCOP(520) { /* LDB */
    lisp_q val;
    lisp_q ppss;

    val = pop();
    ppss = pop();

    if (DTP(val) != DTP_FIX) {
	printf("LDB of non-FIXNUM value.\n");
	return 0;
    }

    /* FIXME: BIGNUMs? */

    destination = ((val >> ppss_position(ppss)) & ppss_mask(ppss));
    destination |= DTP_FIX;

    return 1;
}

MISCOP(524) { /* DPB */
    lisp_q val;
    lisp_q ppss;
    lisp_q num;

    num = pop();
    ppss = pop();
    val = pop();

    printf("DPB: %04lo.\n", ADDRESS(ppss));
    dump_q(num, 0);
    dump_q(val, 0);

    /* FIXME: typecheck? BIGNUMs? */
	
    destination = num;
    destination &= ~(ppss_mask(ppss) << ppss_position(ppss));
    destination |= (val & ppss_mask(ppss)) << ppss_position(ppss);

    destination = DTP_FIX | ADDRESS(destination);

    return 1;
}

MISCOP(525) { /* %LOGDPB */
    lisp_q val;
    lisp_q ppss;
    lisp_q num;

    num = pop();
    ppss = pop();
    val = pop();

    printf("%%LOGDPB: %04lo.\n", ADDRESS(ppss));
    dump_q(num, 0);
    dump_q(val, 0);

    destination = num;
    destination &= ~(ppss_mask(ppss) << ppss_position(ppss));
    destination |= (val & ppss_mask(ppss)) << ppss_position(ppss);

    destination = DTP_FIX | ADDRESS(destination);

    return 1;
}

MISCOP(530) { /* LSH */
    lisp_q n;
    lisp_q nbits;

    nbits = pop();
    n = pop();

    /* FIXME: Signal ARGTYP if n or nbits is not DTP_FIX */

    dump_q(n, 0);
    dump_q(nbits, 1);

    if (nbits & 0x01000000) { /* shift right */
	printf("LSH: right shift not supported.\n");
	return 0;
    } else { /* shift left */
	/* FIXME: Do we create a BIGNUM here if we overflow? */
	/* FIXME: Do we support shift counts larger than 31? */
	destination = DTP_FIX | ADDRESS(n << ADDRESS(nbits));
    }
    
    return 1;
}

MISCOP(534) { /* MAX */
    lisp_q num1;
    lisp_q num2;

    num1 = pop();
    num2 = pop();

    /* FIXME: Not even checking datatypes (!) */

    destination = DTP_FIX | ((ADDRESS(num1) > ADDRESS(num2)) ?
			     ADDRESS(num1) : ADDRESS(num2));
    return 1;
}

MISCOP(535) { /* MIN */
    lisp_q num1;
    lisp_q num2;

    num1 = pop();
    num2 = pop();

    /* FIXME: Not even checking datatypes (!) */

    destination = DTP_FIX | ((ADDRESS(num1) < ADDRESS(num2)) ?
			     ADDRESS(num1) : ADDRESS(num2));
    return 1;
}

MISCOP(536) { /* EXPT */
    lisp_q base;
    lisp_q exponent;

    exponent = pop();
    base = pop();
	
    if ((NOT_CDRCODE(exponent) != (DTP_FIX | 0x0b)) ||
	(NOT_CDRCODE(base) != (DTP_FIX | 0x02))) {
	printf("EXPT: Not 2 ^ 11.\n");
	
	dump_q(base, 0);
	dump_q(exponent, 1);

	return 0;
    }
	
    destination = DTP_FIX | 0x800;

    return 1;
}

MISCOP(550) { /* FLOOR-1 */
    lisp_q dividend;
    lisp_q divisor;

    divisor = pop();
    dividend = pop();

    dump_q(dividend, 0);
    dump_q(divisor, 1);

    /* FIXME: Not even checking datatypes (!) */
	
    destination = DTP_FIX | (ADDRESS(dividend) / ADDRESS(divisor));

    return 1;
}

MISCOP(551) { /* CEILING-1 */
    lisp_q dividend;
    lisp_q divisor;

    divisor = pop();
    dividend = pop();

    dump_q(dividend, 0);
    dump_q(divisor, 1);

    /* FIXME: Not even checking datatypes (!) */

    destination = DTP_FIX | ((ADDRESS(dividend) / ADDRESS(divisor)) + 1);

    return 1;
}

MISCOP(552) { /* TRUNCATE-1 */
    lisp_q dividend;
    lisp_q divisor;

    divisor = pop();
    dividend = pop();

    dump_q(dividend, 0);
    dump_q(divisor, 1);

    /* FIXME: Not even checking datatypes (!) */
    
    /* FIXME: How is this different from FLOOR-1? */
    
    destination = DTP_FIX | (ADDRESS(dividend) / ADDRESS(divisor));

    return 1;
}

MISCOP(554) { /* FLOOR-2 */
    lisp_q dividend;
    lisp_q divisor;

    divisor = pop();
    dividend = pop();

    dump_q(dividend, 0);
    dump_q(divisor, 1);

    /* FIXME: Not even checking datatypes (!) */

    push_cdrnext(DTP_FIX | (ADDRESS(dividend) / ADDRESS(divisor)));
    destination = DTP_FIX | (ADDRESS(dividend) % ADDRESS(divisor));

    return 1;
}

MISCOP(555) { /* CEILING-2 */
    lisp_q dividend;
    lisp_q divisor;

    divisor = pop();
    dividend = pop();

    dump_q(dividend, 0);
    dump_q(divisor, 1);

    /* FIXME: Not even checking datatypes (!) */

    push_cdrnext(DTP_FIX | (ADDRESS(dividend) / ADDRESS(divisor)));
    destination = DTP_FIX | (ADDRESS(dividend) % ADDRESS(divisor));

    return 1;
}

MISCOP(560) { /* ZEROP */
    lisp_q foo;
    
    foo = pop();
    
    if (DTP(foo) != DTP_FIX) {
	printf("ZEROP: Not FIXNUM.\n");
	return 0;
    }
    
    destination = (NOT_CDRCODE(foo) == DTP_FIX)? C_T: C_NIL;

    return 1;
}

MISCOP(566) { /* PLUSP */
    lisp_q foo;

    foo = pop();

    if (DTP(foo) != DTP_FIX) {
	printf("PLUSP: Not FIXNUM.\n");
	return 0;
    }

    foo = ADDRESS(foo);
    if (foo & 0x01000000) foo |= 0xfe000000;
    destination = (foo > 0) ? C_T : C_NIL;

    return 1;
}

MISCOP(571) { /* > */
    lisp_q foo;
    lisp_q bar;

    /* FIXME: Check datatypes. Make work on non-FIXNUM types */

    bar = NOT_CDRCODE(pop());
    foo = NOT_CDRCODE(pop());
    dump_q(foo, 0);
    dump_q(bar, 1);
    destination = (foo > bar)? C_T: C_NIL;

    return 1;
}

MISCOP(577) { /* SYMBOLP */
    lisp_q foo;

    foo = pop();

    destination = (DTP(foo) == DTP_SYMBOL) ? C_T : C_NIL;

    return 1;
}

MISCOP(610) { /* ASSQ */
    lisp_q foo;
    lisp_q alist;

    alist = pop();
    foo = NOT_CDRCODE(pop());

    destination = C_NIL;
    while(NOT_CDRCODE(alist) != (C_NIL)) {
	if (NOT_CDRCODE(car(car(alist))) == foo) {
	    destination = cdr(car(alist));
	}
	alist = cdr(alist);
    }

    return 1;
}

MISCOP(614) { /* NTH */
    lisp_q n;
    lisp_q list;

    list = pop();
    n = pop();

    dump_q(n, 0);
    dump_q(list, 1);

    /* FIXME: Only works on FIXNUMs? */
    n = ADDRESS(n);

    while (n--) {
	list = cdr(list);
    }

    destination = car(list);
    
    return 1;
}

MISCOP(613) { /* MEMQ */
    lisp_q foo;
    lisp_q list;
	
    list = pop();
    foo = NOT_CDRCODE(pop());
	
    while((NOT_CDRCODE(list) != (C_NIL)) &&
	  (NOT_CDRCODE(car(list)) != foo)) {
	list = cdr(list);
    }

    destination = list;

    return 1;
}

MISCOP(616) { /* FIND-POSITION-IN-LIST */
    lisp_q foo;
    lisp_q list;
    int index;

    index = 0;
    list = pop();
    foo = NOT_CDRCODE(pop());
	
    while((NOT_CDRCODE(list) != (C_NIL))) {
	if(NOT_CDRCODE(car(list)) == foo) {
	    destination = DTP_FIX | index;
	    return 1;
	}
	index++;
	list = cdr(list);
    }

    destination = C_NIL;

    return 1;
}

MISCOP(632) { /* VALUE-CELL-LOCATION */
    lisp_q symbol;

    symbol = pop();
    if (DTP(symbol) != DTP_SYMBOL) {
	/* FIXME: signal ARGTYP */
    }
    destination = DTP_LOCATIVE | (ADDRESS(symbol) + SYM_VALUE);

    return 1;
}

MISCOP(634) { /* PROPERTY-CELL-LOCATION */
    lisp_q symbol;

    symbol = pop();
    if (DTP(symbol) != DTP_SYMBOL) {
	/* FIXME: signal ARGTYP */
    }
    destination = DTP_LOCATIVE | (ADDRESS(symbol) + SYM_PROPERTY);

    return 1;
}

MISCOP(635) { /* SYMBOL-PACKAGE */
    lisp_q symbol;
    
    symbol = pop();
    if (DTP(symbol) != DTP_SYMBOL) {
	/* FIXME: signal ARGTYP */
    }
    destination = memread((ADDRESS(symbol) + SYM_PACKAGE));

    return 1;
}

MISCOP(637) { /* %EXTERNAL-VALUE-CELL */
    lisp_q symbol;

    symbol = pop();
    destination = DTP_LOCATIVE | ADDRESS(memread(ADDRESS(symbol) + SYM_VALUE));

    return 1;
}

MISCOP(660) { /* ARRAY-LENGTH */
    lisp_q array;
    lisp_q array_header;

    array = pop();

    /*
     * Just like ARRAY-ACTIVE-LENGTH, below, except without the fill
     * pointer bit.
     */

    array_header = memread(array);

    if (!ARY_LL(array_header)) {
	/* FIXME: Is +1 right here? */
	destination = DTP_FIX | (ARY_INDEX(array_header) + 1);
	return 1;
    }

    /* FIXME: displaced arrays? long-length arrays? */
    
    return 0;
}

MISCOP(662) { /* ARRAY-ACTIVE-LENGTH */
    lisp_q array;
    lisp_q array_header;
    lisp_q fill_pointer;

    array = pop();

    /*
     * If array has a fill pointer, then we return the fill pointer.
     * Otherwise, we return the number of elements in array.
     *
     * An array has a fill pointer if it has a leader, and the first
     * leader entry is a FIXNUM (which is then the fill pointer).
     */

    array_header = memread(array);

    if (ARY_LEADER(array_header)) {
	fill_pointer = memread(array - 2);
	if (DTP(fill_pointer) == DTP_FIX) {
	    destination = fill_pointer;
	    return 1;
	}
    }

    if (!ARY_LL(array_header)) {
	/* FIXME: Is +1 right here? */
	destination = DTP_FIX | (ARY_INDEX(array_header) + 1);
	return 1;
    }

    /* FIXME: displaced arrays? long-length arrays? */
    
    return 0;
}

miscop_handler miscop_dispatch[0x200] = {
    /* 000 - 017 */
    NULL,       NULL,       NULL,       NULL,
    NULL,       NULL,       NULL,       NULL,
    NULL,       NULL,       NULL,       NULL,
    NULL,       NULL,       NULL,       NULL,

    /* 020 - 037 */
    NULL,       NULL,       NULL,       NULL,
    NULL,       NULL,       NULL,       NULL,
    NULL,       NULL,       NULL,       NULL,
    NULL,       NULL,       NULL,       NULL,

    /* 040 - 057 */
    NULL,       NULL,       NULL,       NULL,
    NULL,       NULL,       NULL,       NULL,
    NULL,       NULL,       NULL,       NULL,
    miscop_054, NULL,       NULL,       NULL,

    /* 060 - 077 */
    NULL,       NULL,       NULL,       miscop_063,
    NULL,       NULL,       NULL,       miscop_067,
    NULL,       NULL,       NULL,       NULL,
    NULL,       NULL,       NULL,       miscop_077,


    /* 100 - 117 */
    NULL,       NULL,       NULL,       NULL,
    NULL,       NULL,       NULL,       NULL,
    NULL,       NULL,       NULL,       NULL,
    NULL,       NULL,       NULL,       NULL,

    /* 120 - 137 */
    NULL,       NULL,       NULL,       NULL,
    NULL,       NULL,       NULL,       NULL,
    NULL,       NULL,       NULL,       NULL,
    NULL,       NULL,       NULL,       NULL,

    /* 140 - 157 */
    NULL,       NULL,       NULL,       NULL,
    NULL,       NULL,       NULL,       NULL,
    NULL,       NULL,       NULL,       NULL,
    NULL,       NULL,       NULL,       NULL,

    /* 160 - 177 */
    NULL,       NULL,       NULL,       NULL,
    NULL,       NULL,       NULL,       NULL,
    NULL,       NULL,       NULL,       NULL,
    NULL,       NULL,       NULL,       NULL,


    /* 200 - 217 */
    NULL,       NULL,       NULL,       NULL,
    NULL,       NULL,       NULL,       NULL,
    NULL,       NULL,       NULL,       NULL,
    NULL,       NULL,       NULL,       NULL,

    /* 220 - 237 */
    NULL,       NULL,       NULL,       NULL,
    NULL,       NULL,       NULL,       NULL,
    NULL,       NULL,       NULL,       NULL,
    NULL,       NULL,       NULL,       miscop_237,

    /* 240 - 257 */
    miscop_240, NULL,       NULL,       NULL,
    NULL,       NULL,       NULL,       NULL,
    NULL,       NULL,       NULL,       NULL,
    NULL,       NULL,       NULL,       NULL,

    /* 260 - 277 */
    NULL,       NULL,       NULL,       NULL,
    NULL,       NULL,       NULL,       NULL,
    NULL,       NULL,       NULL,       NULL,
    NULL,       NULL,       NULL,       NULL,


    /* 300 - 317 */
    NULL,       NULL,       NULL,       NULL,
    NULL,       NULL,       NULL,       NULL,
    NULL,       NULL,       NULL,       NULL,
    NULL,       NULL,       NULL,       NULL,

    /* 320 - 337 */
    NULL,       NULL,       NULL,       NULL,
    NULL,       NULL,       NULL,       NULL,
    NULL,       NULL,       NULL,       NULL,
    NULL,       miscop_335, NULL,       NULL,

    /* 340 - 357 */
    NULL,       NULL,       NULL,       NULL,
    NULL,       NULL,       NULL,       NULL,
    NULL,       NULL,       NULL,       NULL,
    NULL,       NULL,       NULL,       NULL,

    /* 360 - 377 */
    NULL,       NULL,       NULL,       NULL,
    NULL,       NULL,       NULL,       NULL,
    NULL,       NULL,       NULL,       NULL,
    NULL,       NULL,       NULL,       NULL,


    /* 400 - 417 */
    NULL,       NULL,       NULL,       NULL,
    NULL,       NULL,       NULL,       NULL,
    NULL,       NULL,       NULL,       NULL,
    NULL,       NULL,       miscop_416, NULL,

    /* 420 - 437 */
    NULL,       NULL,       NULL,       NULL,
    NULL,       NULL,       NULL,       NULL,
    NULL,       NULL,       NULL,       NULL,
    NULL,       miscop_435, NULL,       NULL,

    /* 440 - 457 */
    NULL,       NULL,       NULL,       NULL,
    NULL,       NULL,       miscop_446, miscop_447,
    miscop_450, miscop_451, miscop_452, NULL,
    miscop_454, NULL,       NULL,       NULL,

    /* 460 - 477 */
    NULL,       miscop_461, NULL,       miscop_463,
    NULL,       NULL,       NULL,       NULL,
    NULL,       NULL,       NULL,       NULL,
    miscop_474, NULL,       NULL,       NULL,


    /* 500 - 517 */
    NULL,       NULL,       NULL,       NULL,
    NULL,       NULL,       NULL,       NULL,
    NULL,       NULL,       NULL,       NULL,
    NULL,       NULL,       NULL,       NULL,

    /* 520 - 537 */
    miscop_520, NULL,       NULL,       NULL,
    miscop_524, miscop_525, NULL,       NULL,
    miscop_530, NULL,       NULL,       NULL,
    miscop_534, miscop_535, miscop_536, NULL,

    /* 540 - 557 */
    NULL,       NULL,       NULL,       NULL,
    NULL,       NULL,       NULL,       NULL,
    miscop_550, miscop_551, miscop_552, NULL,
    miscop_554, miscop_555, NULL,       NULL,

    /* 560 - 577 */
    miscop_560, NULL,       NULL,       NULL,
    NULL,       NULL,       miscop_566, NULL,
    NULL,       miscop_571, NULL,       NULL,
    NULL,       NULL,       NULL,       miscop_577,


    /* 600 - 617 */
    NULL,       NULL,       NULL,       NULL,
    NULL,       NULL,       NULL,       NULL,
    miscop_610, NULL,       NULL,       miscop_613,
    miscop_614, NULL,       miscop_616, NULL,

    /* 620 - 637 */
    NULL,       NULL,       NULL,       NULL,
    NULL,       NULL,       NULL,       NULL,
    NULL,       NULL,       miscop_632, NULL,
    miscop_634, miscop_635, NULL,       miscop_637,

    /* 640 - 657 */
    NULL,       NULL,       NULL,       NULL,
    NULL,       NULL,       NULL,       NULL,
    NULL,       NULL,       NULL,       NULL,
    NULL,       NULL,       NULL,       NULL,

    /* 660 - 677 */
    miscop_660, NULL,       miscop_662, NULL,
    NULL,       NULL,       NULL,       NULL,
    NULL,       NULL,       NULL,       NULL,
    NULL,       NULL,       NULL,       NULL,


    /* 700 - 717 */
    NULL,       NULL,       NULL,       NULL,
    NULL,       NULL,       NULL,       NULL,
    NULL,       NULL,       NULL,       NULL,
    NULL,       NULL,       NULL,       NULL,

    /* 720 - 737 */
    NULL,       NULL,       NULL,       NULL,
    NULL,       NULL,       NULL,       NULL,
    NULL,       NULL,       NULL,       NULL,
    NULL,       NULL,       NULL,       NULL,

    /* 740 - 757 */
    NULL,       NULL,       NULL,       NULL,
    NULL,       NULL,       NULL,       NULL,
    NULL,       NULL,       NULL,       NULL,
    NULL,       NULL,       NULL,       NULL,

    /* 760 - 777 */
    NULL,       NULL,       NULL,       NULL,
    NULL,       NULL,       NULL,       NULL,
    NULL,       NULL,       NULL,       NULL,
    NULL,       NULL,       NULL,       NULL,
};

int step_miscop(u16 opcode)
{
    u16 opnum;
    int retval;

    opnum = opcode & 0x1ff;

    retval = 1;

    if (miscop_dispatch[opnum]) {
	retval = miscop_dispatch[opnum](opcode, opnum);
#if 0
    } else if (opnum == 0405) { /* SPECIAL-PDL-INDEX */
	/* FIXME: Should be context.specpdl_pointer */
	destination = DTP_LOCATIVE | ADDRESS(context.pdl_pointer);
#endif

    } else {
	printf("Unknown miscop.\n");
	return 0;
    }

    if (!retval) return 0;

    if (opcode & 0x4000) {
	push_cdrnext(destination);
    } else {
	context.indicators = destination;
    }

    return 1;
}

/* EOF */
