;;               -*- Mode: LISP; Package: User; Base: 10 -*-
;;
;;    PRODUCE THE CALLING TREE AND CROSS REFERENCE FOR LISP FUNCTION(S)
;;
;; EDITTED 7/27/83 RAF ... Ignore "(COMMENT ... )" forms; add DOLIST, SELECTQ,
;;     & DEFCOM forms.  Add function to print only functions called directly.
;; EDITTED 6/23/83 RAF ... Add XREF & CALLED-FROM functions; fix DO and
;;     dotted list problem.
;; EDITTED 6/16/83 RAF ... Add capability to descend down the calling tree.
;; EDITTED 6/16/83 RAF ... Change from 3 lists to an a-list, read from file.
;; WRITTEN 2/14/83 TWE


#M(load "ps:<Maclisp>GCdemn")      ;don't crap out over GC

(DEFVAR *CALLING-DATA* nil)        ;a-list of calling data
(DEFVAR *DEFINED-ONLY* nil)        ;if => T, prints only defined functions
(DEFVAR *FILES* nil)               ;List of all files read this session
(DEFVAR EXCLUDE-LIST nil)          ; list of; functions to ignore.
(Defconst *CROSS-REF-FUNCS* nil      "List of User callable functions")
                                                
(Declare (Special CALLS-LIST))     ;This symbol is used dynamically.

(Defmacro DEFUSER-FUNC (symbol arglist &Body body)
  "Define SYMBOL to be a function & push it on the list of user functions."
  `(progn
     (Unless (memq ',symbol  *CROSS-REF-FUNCS*)
       (push ',symbol *CROSS-REF-FUNCS*))
     (Defun ,Symbol ,arglist ,@body)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;                            USER INTERFACE
;;
;; Prints the effective "ARGLIST" for each user function defined.

(DEFUSER-FUNC HELP ()                                  ;4/3/84 RAF
  "Prints the list of user callable functions"
  (Dolist (func *CROSS-REF-FUNCS*)
    (Format t "~&~A~25T: ~A~%" func (documentation func)))
  (Print "To print only defined functions, (SETQ *DEFINED-ONLY* T)")
  (terpri))



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;                       DATA STRUCTURE FUNCTIONS


;; Returns an item from a FUNCTION-LIST, creating a new one if not present.

(DEFUN FIND-FUNCTION (name)                     ;6/13/83 RAF
  "Return the function NAME in *CALLING-DATA*,
   creating an entry if not already there."
  (OR (ASSQ name *CALLING-DATA*)         ;return existing function.
      (Car (PUSH (MAKE-FUNCTION name) *CALLING-DATA*))  ;make a new entry
      ))


;; FUNCTION DATA STRUCTURE
;;
;; returns the list of callers of ITEM & Called functions from ITEM

(DEFUN MAKE-FUNCTION (name) (LIST name nil nil nil))

;; Function-Callers & Function-Callees are lists of FUNCTIONS, i.e. the data
;;structure is recursive.

(DEFUN FUNCTION-XREF-NAME    (func) (Car    func))   ;Access function's name
(DEFUN FUNCTION-CALLERS (func) (Cadr   func))   ;Access function's callers
(DEFUN FUNCTION-CALLEES (func) (Caddr  func))   ;Access its called functions
(DEFUN FUNCTION-DEFINED (func) (Cadddr func))   ;Is this function defined?

(DEFUN SET-FUNCTION-CALLERS (func caller) (RPLACA (cdr func) caller))
(DEFUN SET-FUNCTION-CALLEES (func called) (RPLACA (cddr func) called))
(DEFUN SET-FUNCTION-DEFINED (func state)  (RPLACA (cdddr func) state))


;; Add a new caller to a function; side effect only

(DEFUN FUNCTION-CALLER-ADD (func caller)        ;6/14/83 RAF
  (If (not (Memq caller (FUNCTION-CALLERS func)))
      (SET-FUNCTION-CALLERS func (CONS caller (FUNCTION-CALLERS func)))))

;; add a new function, NAME, called from FUNC.

(DEFUN FUNCTION-CALLEE-ADD (func callee)        ;6/14/83 RAF
  (If (not (Memq callee (FUNCTION-CALLEES func)))
      (SET-FUNCTION-CALLEES func (CONS callee (FUNCTION-CALLEES func)))))


(DEFUN DEFINEDP (function)                      ;6/23/83 RAF
  "Returns non-nil if FUNCTION is 'defined' to be referenced."
  (or (not *DEFINED-ONLY*) (FUNCTION-DEFINED function)))



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;                            USER FUNCTIONS
;;
;; Arguements - IN:   String name of program source file to read.
;;              OUT:  String name for output file; if T, use screen.
;;              FUNC: Symbol name of some function.

(DEFUSER-FUNC CLEAR-CROSS-REF ()                ;4/3/84 RAF
  "Clear all cross reference data."
  (Setq *CALLING-DATA* nil)
  (Setq *FILES* nil))


(DEFUSER-FUNC CALLING-TREE (in out)             ;6/16/83 RAF
  "Builds the calling data from file IN, & prints the call tree to OUT."
  (BUILD-CALLING-DATA in)
  (PRINT-CALLING-TREE out))


(DEFUSER-FUNC BUILD-CALLING-DATA (in)           ;7/28/83 RAF
  "Builds the calling data for file IN."      
  (Setq *FILES* (nconc *FILES* (list in)))
  (Setq in (OPEN in))
  (DO ((X nil (READ in '*EOF*))
       (calls-list nil))
      ((eq X '*EOF*)
       (Close in)
       (Setq *CALLING-DATA* (Sortcar *CALLING-DATA* #'Listlessp))
       )
    (PARSE-FORM X))
  in)

(DEFUSER-FUNC PRINT-CALLING-TREE (out)          ;6/22/83 RAF
  "Prints the calling data to file OUT."
  (Let ((out-stream (OPEN-OUT out)))
    (PRINT-HEADER "Calling Tree" out-stream)
    (Do ((f-list *CALLING-DATA* (cdr f-list)))
        ((null f-list) (Close-out out-stream))
      (If (FUNCTION-DEFINED (car f-list))
          (Progn (PRINT-ONE-FUNCTION (car f-list) nil out-stream)
                 (terpri out-stream)
                 (terpri out-stream))))))


(DEFUSER-FUNC PRINT-CALLED-FUNCTIONS (func out) ;6/16/83 RAF
  "Prints the calling data for function FUNC to file OUT."
  (setq out (OPEN-OUT out))
  (PRINT-ONE-FUNCTION (assq func *CALLING-DATA*) nil out)
  (CLOSE-OUT out))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;                    Cross-reference user functions
;;

(DEFUSER-FUNC PRINT-CALLED-FROM (func out)      ;6/17/83 RAF
  "Prints all functions which call FUNC, directly or indirectly."
  (Setq out (OPEN-OUT out))
  (PRINT-ONE-CALLED-FROM (assq func *CALLING-DATA*) nil out)
  (CLOSE-OUT out))


(DEFUSER-FUNC PRINT-XREF-CALLERS (out)          ;6/20/83 RAF
  "Print a complete Cross reference to file OUT."
  (Setq out (OPEN-OUT out))
  (DO ((X *CALLING-DATA* (cdr X)))
      ((null X))
    (If (FUNCTION-DEFINED (car X)) (PRINT-ONE-XREF (car x) out)))
  (CLOSE-OUT out))


(DEFUSER-FUNC PRINT-XREF-CALLED (out)           ;7/27/83 RAF
  "Prints functions called directly from each function."
  (Setq out (OPEN-OUT out))
  (Print-header "Called Functions" out)
  (DOLIST (function *CALLING-DATA*)
    (Cond ((FUNCTION-DEFINED function)
           (Print (FUNCTION-XREF-NAME function) out)
           (DOLIST (X (FUNCTION-CALLEES function))
             (Progn (terpri out)
                    (INDENT 2 out)
                    (Princ (FUNCTION-XREF-NAME X) out)))
           (terpri out)))))
  

;; Print the names of all functions referenced.

(Declare (special tuo))
(DEFUSER-FUNC PRINT-NAMES (out)                 ;6/22/83 RAF
  "Print all the functions defined in the calling tree."          
  (Let ((tuo (OPEN-OUT out)))
    (Mapc #'(lambda (x) (If (DEFINEDP x) (Print (car x) tuo)))
          *CALLING-DATA*)
    (CLOSE-OUT tuo)))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;                       SPECIALIZED I/O FUNCTIONS

(DEFUN PRINT-HEADER (title out)                 ;7/27/83 raf
  (terpri out)
  (Princ "   ** Print " out)
  (Princ title out)
  (Princ " - " out)
  (Princ *FILES* out)
  (terpri out)
  (terpri out))
;;
;; Basic function called to print a function and its callers (recursively).

(DEFUN PRINT-ONE-FUNCTION (function callers out)        ;6/21/83 RAF
  "Prints data for FUNCTION to the stream OUT."
  (Cond ((DEFINEDP function)
         (Print (length callers) out)
         (Princ "  " out)
         (INDENT (Times 2 (Length callers)) out)
         (Princ (FUNCTION-XREF-NAME function) out)
         (IF (not (memq function callers))      ;go down if no recursion.
             (DO ((X (FUNCTION-CALLEES function) (cdr X))
                  (newcallers (cons function callers)))
                 ((null X))
               (PRINT-ONE-FUNCTION (car X) newcallers out))))))


(DEFUN INDENT (n out)                           ;6/16/83 RAF
  (DO ((i 0 (plus i 1)))
      ((= i n))
    (TYO #/  out)))

(DEFUN PRINT-ONE-XREF (func out)                ;6/20/83 RAF
  (terpri out)
  (Princ (FUNCTION-XREF-NAME func) out)
  (DO ((X (FUNCTION-CALLERS func) (cdr X)))
      ((null X))
    (terpri out)
    (INDENT 2 out)
    (Princ (FUNCTION-XREF-NAME (car X)) out))
  (terpri out))
  

(DEFUN PRINT-ONE-CALLED-FROM (function callees out)        ;6/16/83 RAF
  "Prints data for FUNCTION to the stream OUT."
  (terpri out)
  (INDENT (Times 2 (Length callees)) out)
  (Princ (FUNCTION-XREF-NAME function) out)
  (IF (not (memq function callees))             ;go down if no recursion.
      (DO ((X (FUNCTION-CALLERS function) (cdr X)))
          ((null X))
        (PRINT-ONE-CALLED-FROM (car X) (cons function callees) out))))


;; Opens FILE-NAME for output.  Terminal output is T

(DEFUN OPEN-OUT (file-name)                     ;6/22/83 RAF
  "Opens FILE-NAME for output, goes thru crap to open new version."
  (Let ((out (If (eq file-name T) T 
                 #Q(OPEN file-name ':out)                  ;for LISP Machine
                 #M(OPEN (mergef file-name
                                 (list (list 'ps (status udir))  ;for MACLISP
                                       'file 'lsp '/-1))
                         '(out)))))
    (Terpri out)
    out))

(DEFUN CLOSE-OUT (file)                         ;6/22/83 RAF
  (terpri file)
  (If (not (eq file T)) (close file)))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;                           PARSE A GENERAL FORM
;;
;; Parse one statement of a function.  This function handles
;; nested function calls too.
;;
;; Input parameter:
;;   form - one complete statement of a function

(DEFUN PARSE-FORM (form)                        ;mod. 7/27/83 RAF 
  (If (listp form)
      (Let ((func (car form))
            (args (cdr form)))
        (selectq func
          ((Defun Defmacro Defsubst Defmethod) (PARSE-DEFUN form))
          ((Defcom)         (PARSE-DEFCOM form))
          ((Cond)           (PARSE-COND form))
          ((Do Do*)         (PARSE-DO form))
          ((Let Let* Prog Progw) (PARSE-LET form))
          ((Function)       (INSERT-IN-CALLS-LIST (car args)))
          ((Dotimes Dolist) (PARSE-DOLIST form))
          ((Select Selectq) (PARSE-SELECTQ form))
          ((Mapc Mapcar Mapcan Map Maplist Mapcon Mapatoms Maphash)
           (PARSE-MAP form))
          ((Quote Defstruct Defflavor Defvar Defconst Declare Comment)
           nil)                                 ;ignore
          (otherwise                            ; Normal functions.
           (INSERT-IN-CALLS-LIST func)
           (MAPDOT (function PARSE-FORM) args))
          ))))




;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;                            PARSING FUNCTIONS
;;
;; Parse an entire defun style function.
;;
;; Input: DEFUN-LIST - list which contains the defun to be parsed

(DEFUN PARSE-DEFUN (defun)
  (LET ((func-name (cadr defun))                ; name of the function
        (calls-list nil))                       ; functions called
    (MAPDOT (function PARSE-ARG-LIST) (caddr defun))
    (MAPDOT (function PARSE-FORM) (cdddr defun))
    (PROCESS-CALLS-AND-CALLERS func-name (reverse calls-list))))


(DEFUN PARSE-DEFCOM (defun)                     ; 7/27/83 RAF
  "Parses defining forms which have NO arg list (NOT a Nil one)."
  (LET ((func-name (cadr defun))                ; name of the function
        (calls-list nil))                       ; functions called
    (MAPDOT (function PARSE-FORM) (cddr defun))
    (PROCESS-CALLS-AND-CALLERS func-name (reverse calls-list))))


(DEFUN PARSE-LET (form)                         ;6/15/83 RAF
  (MAPDOT (function PARSE-ARG-LIST) (cadr form))  ;var list
  (MAPDOT (function PARSE-FORM) (cddr form)))     ;LET forms


(DEFUN PARSE-DO (form)                          ;6/20/83 RAF
  (If (symbolp (cadr form))                    
      (MAPDOT (function PARSE-FORM) (Cddr form))  ;Old DO
    (MAPDOT (function PARSE-ARG-LIST) (cadr form))  ;var list - New DO
    (MAPDOT (function PARSE-FORM) (caddr form))     ;exit clause
    (MAPDOT (function PARSE-FORM) (cdddr form))))   ;interation forms


(DEFUN PARSE-SELECTQ (form)                     ;7/27/83 RAF
  (PARSE-FORM (cadr form))
  (Dolist (case (cddr form))
    (MAPDOT (Function PARSE-FORM) (cdr case))))


(DEFUN PARSE-DOLIST (form)                      ;7/27/83 RAF
  (PARSE-ARG-LIST (cadr form))
  (MAPDOT (function PARSE-FORM) (cddr form)))


(DEFUN PARSE-ARG-LIST (var)                     ;6/15/83 RAF
  (IF (not (atom var))
      (MAPDOT (function PARSE-FORM) (cdr var))))


(DEFUN PARSE-COND (form)                        ;6/15/83 RAF
  (DO ((list (cdr form) (cdr list)))
      ((null list))
    (MAPDOT (function PARSE-FORM) (car list))))




;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;                     PARSE FUNCTION CALLING FORMS
;;
;;                            FUNCALL & SEND

;; This function not yet working, 7/27/83 raf
;(DEFUN PARSE-FUNCALL (form)                     ;7/27/83 RAF
;  (Selectq (car form)
;    (FUNCALL (Comment Place 1st arg as a called function, parse remaining))
;    (FUNCALL-SELF (Comment just parse remaining args))
;    (SEND (Comment Place (arg1 arg2) as a called function & parse rest ...
;                   OR place (SEND arg2) as a called function.
;                   NOTE - for (DEFMETHOD (foo bar) ...) also ought to place
;                          (SEND :bar) as a locally defined function.))
;    (SEND-SELF (Comment same as funcall-self))))
;

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;                                MAPPING
;; (Map (function F) arg)            => (F              arg)
;; (Map (function (lambda ...)) arg) => ((Lambda ...)   arg)
;; (Map sym arg)                     => ((Map sym)      arg)

(DEFUN PARSE-MAP (form)                         ;6/15/83 RAF
  (Let ((map (car form))
        (func (cadr form))
        (args (cddr form)))
    (Setq func (If (eq 'FUNCTION (car func))
                   (cadr func)               ;(map (function ...) ...)
                   (list map func)))         ;(map ?? ...)
    (PARSE-FORM (cons func args))))




;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Insert a function reference into calls-list
;;
;; Global: CALLS-LIST - reference list for the current function

(DEFUN INSERT-IN-CALLS-LIST (this-func-call)
  (IF (and (not (member this-func-call exclude-list)) ;normal LISP function?
           (not (member this-func-call CALLS-LIST)))  ;already on list?
      (push this-func-call calls-list))         ;add to list.
  )


;; Update the reference lists using the data generated by PARSE-DEFUN.
;;
;; Input: FUNC-NAME -  name of the defun-ed function.
;;        CALLS-LIST - list of functions called by FUNC-NAME.
;;
;; Globals: *CALLING-DATA*

(DEFUN PROCESS-CALLS-AND-CALLERS (func-name calls-list) ;6/14/83 RAF
  "Remember Function and those it calls."
  (Let ((function  (FIND-FUNCTION func-name)))
    (SET-FUNCTION-CALLEES function              ;set guys I call.
                          (MAPCAR (function FIND-FUNCTION) calls-list))
    (SET-FUNCTION-DEFINED function T)           ;This guy is defined.
    (ADD-CALLED-FROM-LIST func-name calls-list) ;set guys who call me.
  ))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Add FUNC-NAME to each of the CALLS-LIST in (car calls-list).
;;
;; This function takes the CALLS-LIST list generated by PARSE-DEFUN and
;; uses that to update the *CALLING-DATA* list.  Every name in the CALLS-LIST
;; is called from the function, FUNC-NAME.
;;
;; The pseudo-code looks like:
;; FOR each item X in CALLS-LIST DO
;;   Find X's reference in *CALLING-DATA*, creating an empty one if necessary.
;;   Add func-name to X's called-from list
;;
;; Input: FUNC-NAME  - the function name.
;;        CALLS-LIST - list containing function names called by 'func-name'.
;;
;; Globals:
;;   *CALLING-DATA* - A-List of (function names, callers, & called functions)

(DEFUN ADD-CALLED-FROM-LIST (func-name calls-list)      ;6/14/83 RAF
  (cond ((not (null calls-list))
         (FUNCTION-CALLER-ADD (FIND-FUNCTION (car calls-list))
                              (FIND-FUNCTION func-name))
	 (ADD-CALLED-FROM-LIST func-name (cdr calls-list)))
	))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;                           UTILITY FUNCTIONS


;; Define MAPDOT to work on "dotted lists" of the form (A B C . D).

(DEFUN MAPDOT (function list)                     ;6/22/83 RAF
  (DO ((sublist list (if (listp sublist) (cdr sublist))))
      ((null sublist) list)
    (LET ((item (if (listp sublist) (car sublist) sublist)))
      (apply function (list item)))))


;; Extends ALPHALESSP to work on trees by comparing the left most leaves.

(DEFUN LISTLESSP (A B)                          ;6/20/83 RAF
  (Cond ((not (atom A)) (Listlessp (car A) B))
        ((not (atom B)) (Listlessp A (car B)))
        (T (Alphalessp A B))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Initialization
;;

;; list of built-in function names

(Setq EXCLUDE-LIST '(append atom car cdr cadr cddr caddr cdddr cons list setq))
(Nconc EXCLUDE-LIST '(quote set boundp select selectq memq eq new equal))
(Nconc EXCLUDE-LIST '(nil not null and or if plusp minusp zerop))
(Nconc EXCLUDE-LIST '(symbolp listp numberp typep progn prog1 dotimes dolist))
(Nconc EXCLUDE-LIST '(length first second third reverse subst sublis member))
(Nconc EXCLUDE-LIST '(+ - 1+ 1- * // \ = > < >= <=))
(Nconc EXCLUDE-LIST '(Plus minus difference times quotient greaterp lessp))
(Nconc EXCLUDE-LIST '(aref aset aloc array-leader store-array-leader))
(Nconc EXCLUDE-LIST '(when unless append nconc ldb dpb min max))
(Nconc EXCLUDE-LIST '(Read Open Close Print Princ terpri))

;; Set the READER macro #. so that it does NOT evaluate in READ.
#M(DEFSHARP /. (()) (READ))

;; THE VERY LAST THING AFTER LOADING EVERYTHING
;; Tell the user what he has.

(HELP)                                   
