;;; -*- Mode: LISP ; Base: 10. ; Package: user ; Fonts:CPTFONT,HL12B,HL12BI -*-

(DEFUN find-file (filename)
  (LET* ((PATHNAME (fs:parse-pathname filename))
         (DIRECTORY (SEND PATHNAME ':directory))
         (name (SEND PATHNAME ':name))
         (type (SEND PATHNAME ':type))
         (version (SEND PATHNAME ':version)))
    (LOOP for dir in (dc-directory)
          as directory-name = (directory-name dir)
          as printed-flag = NIL
          WHEN (OR (NOT (STRINGP DIRECTORY))
                   (STRING-SEARCH DIRECTORY directory-name))
          DO
          (LOOP for file in (read-directory-files dir)
                WHEN (OR (NOT (STRINGP name))
                         (STRING-SEARCH name (file-name file)))
                WHEN (OR (NOT (STRINGP type))
                         (STRING-SEARCH type (file-type file)))
                WHEN (OR (NOT (NUMBERP version))
                         (= version (file-version file)))
                DO
                (IF (NULL printed-flag)
                    (PROGN
                      (SETQ printed-flag T)
                      (FORMAT T "~2%~A;" directory-name)))
                (FORMAT T "~%~A.~A#~A"
                        (file-name file)(file-type file)(file-version file))))))

(DEFUN find-multiple-files ()
  (LOOP for dir in (dc-directories)
        as directory-name = (directory-name dir)
        as printed-flag = NIL
        DO
        (print-directory-multiple-versions dir)))

(DEFUN find-multiple-files-in-dir (DIRECTORY)
  (IF (NOT (TYPEP DIRECTORY 'FS:DIRECTORY))
      (SETQ DIRECTORY (lookup-directory DIRECTORY)))
  (LOOP for file in (read-directory-files DIRECTORY)
        with directory-results = NIL
        as file-entry = (ASSOC (file-name file) directory-results)
        as type-entry = (ASSOC (file-type file) (CDR file-entry))
        DO
        (COND ((AND (NULL file-entry)
                    (NULL type-entry))
               (PUSH (LIST (file-name file)
                           (LIST (file-type file) file))
                     directory-results))
              ((NULL type-entry)
               (PUSH (LIST (file-type file) file)
                     (CDR file-entry)))
              (t (PUSH file (CDR type-entry))))
        finally
        (RETURN directory-results)))

(DEFVAR type-versions
        '(("LISP" . 2)
          ("TEXT" . 2)
          ("INIT" . 2)
          ("RMAIL" . 2)
          ("QFASL" . 1)
          ("DRAW" . 2)))

(DEFUN print-directory-multiple-versions (DIRECTORY)
  (FORMAT t "~&~%Directory ~A~2%"
          (IF (TYPEP DIRECTORY ':STRING) DIRECTORY
            (directory-name DIRECTORY)))
  (LOOP for (filename . types) in (NREVERSE (find-multiple-files-in-dir DIRECTORY))
        DO
        (LOOP for (type . files) in types
              as versions = (CDR (ASSOC type type-versions))
              WHEN (OR (AND versions
                            (> (LENGTH files) versions))
                       (AND (NULL versions)
                            (> (LENGTH files) 1)))
              DO
              (LOOP for f in files
                    DO (FORMAT t "~%~A;~A.~A#~A"
                               (directory-name (file-directory f))
                               (file-name f)
                               (file-type f)
                               (file-version f))))))

(DEFUN find-directory-files (directory-name)
  (LET* ((directory-pathname
           (fs:parse-pathname (FORMAT nil "~A*.*" directory-name)))
         (directory-file-list (SEND directory-pathname ':DIRECTORY-LIST '(:SO  ?????))))
    (LOOP for file-entry in (CDR directory-file-list)
          with result = nil
          as PATHNAME = (FIRST file-entry)
          as name = (SEND PATHNAME ':name)
          as type = (SEND PATHNAME ':type)
          as STRING = (fs:make-pathname ':directory directory-name ':name name '    ??????)
          as SEARCH = (ASSOC STRING result)
          UNLESS SEARCH
          DO (PUSH (LIST STRING file-entry) result)
          else
          DO (SETF (SECOND SEARCH) file-entry)
          finally (RETURN result))))

(DEFUN find-all-single-files (&optional (dirs *))
  (LOOP for dir in dirs
        APPEND (find-directory-files dir)))

(DEFUN LMFS-LIST-DIRECTORIES (&REST DIRECTORIES)
  (OR DIRECTORIES (SETQ DIRECTORIES (TOP-LEVEL-DIRECTORIES)))
  (LOOP FOR DIRECTORY IN DIRECTORIES
	AS LOOKUP = (LOOKUP-DIRECTORY DIRECTORY T)
	WHEN LOOKUP DO (FORMAT T "~%~A" (DIRECTORY-NAME LOOKUP))
	ELSE DO (FORMAT T "~%~A No Such Directory." DIRECTORY)))

(DEFUN LMFS-LIST-FILES (&REST DIRECTORIES)
  (COND ((BOUNDP 'DISK-CONFIGURATION)
	 (OR DIRECTORIES (SETQ DIRECTORIES (TOP-LEVEL-DIRECTORIES)))
	 (LOOP FOR DIRECTORY IN DIRECTORIES
	       AS LOOKUP = (LOOKUP-DIRECTORY DIRECTORY T)
	       WHEN LOOKUP DO (FORMAT T "~%~A:" (DIRECTORY-NAME LOOKUP))
			      (DOLIST (FILE (DIRECTORY-FILES LOOKUP))
				(DBG-EXAMINE-SHOW-FILE FILE NIL))
			      (FORMAT T "~%~%")
	       ELSE DO (FORMAT T "~%~A:  No Such Directory.~%~%" DIRECTORY)))
	(T (FORMAT T "~%File system not mounted -- do (FS:BOOT-FILE-SYSTEM)"))))


; -*- Mode: LISP; Package: USER; Base: 10.; Fonts: CPTFONTB , HL12B , HL12BI -*-

(defconst *main-directory* "s5:>demo>")

(defun relocate-file (name)
  (string-append *main-directory* name))


(defmacro save-symbol (symbol file-name)
  `(lbin:dump-forms-to-file ,FILE-NAME '((setq ,symbol ',(SYMEVAL symbol)))))

(COMMENT DEFUN window-permit (flavor &optional priority quantum)
  "Find a FLAVOR window and set its deexposed-typeout-action to :PERMIT.
If the optional priority parameter is given, set the window's process priority.
If the optional quantum parameter is given, set the window's process quantum."
  (CHECK-ARG priority (OR (NULL priority) (NUMBERP priority)) "a number")
  (CHECK-ARG quantum (OR (NULL quantum) (AND (NUMBERP quantum) (PLUSP quantum)))
	     "a positive number")
  (LET ((w (OR (AND (EQ (TYPEP tv:selected-window) flavor) tv:selected-window)
	       (tv:find-window-of-flavor flavor))))
    (UNLESS w
      (SETQ w (make-instance flavor ':superior tv:default-screen))
      (SEND w ':activate))
    (SEND w ':set-deexposed-typeout-action ':permit)
    (WHEN priority (SEND (SEND w ':process) ':set-priority priority))
    (WHEN quantum (SEND (SEND w ':process) ':set-quantum quantum))))

(COMMENT window-permit 'tv:lisp-listener -2 15.)
(COMMENT window-permit 'tv:telnet)

;;; Make GRIND easer to type
(DEFF GRIND #'GRIND-TOP-LEVEL)
(GLOBALIZE 'GRIND)

(COMMENT DEFUN BACKUP ()
  "Backup my files to ai"
  (WITHOUT-MORE-PROCESSING STANDARD-OUTPUT
    (fs:balance-directories "lam2:nichols;*.lisp" "lam5:nichols;*.lisp" ':query-mode ':1->2))

    (COPY-FILE "lm:nichols;*.lisp#>" "v4:[dnichols.lispm]"))
;;    (copy-files-to-ai "lm2:oren;*.lisp" "<oren.lm>"))) ;source on lm2:bruce;env-backup.lisp


(COMMENT DEFUN LUNCH ()
  "Backup then do a garbage collect"
  (PRINC "  Don't get indigestion")
  (BACKUP)
  (SI:FULL-GC))

(DEFUN dribble ()
  "Dribble output to the editor"
  (DRIBBLE-START "v4:[dnichols]dribble.lis" 'EDITOR-P))

(DEFUN universal-string (form)
  (FORMAT nil "~a" form))

(DEFUN c (char)
  "Return a character name given a number"
  (FORMAT nil "~:c" char))

(COMMENT DEFUN td (&quote body)
  "Eval body & print return the execution time in seconds"
  (LET ((start (TIME:microsecond-time))
	end value)
    (SETQ value (MULTIPLE-VALUE-LIST (EVAL body)))
    (SETQ end ( TIME:microsecond-time))
    (FORMAT t "~%~6$ Seconds" (// (- end start) 1000000.0))
    (VALUES-LIST value)))

(DEFMACRO td (&body body)
  "Eval body & print return the execution time in seconds"
  `(LET (start end value)
     (WITHOUT-INTERRUPTS
       (SETQ start (TIME:microsecond-time)) 
       (SETQ value (MULTIPLE-VALUE-LIST ,@body))
       (SETQ end ( TIME:microsecond-time)))
    (FORMAT t "~%~6$ Seconds" (// (- end start) 1000000.0))
    (VALUES-LIST value)))

(defmacro string-if-error ((error-header) &body body)
  "If an error occurs while evaluating body, return the error-header string
 appended to the error handler message.  If no error occurs, return the result from body"
  `(CONDITION-CASE (ERROR)
       (progn . ,body)
     (ERROR (format nil "~@[~a~%~]~a" ,error-header (SEND ERROR ':report-string)))))

(DEFSUBST e (&rest parms)
  "Handy debugging function to get into the error handler printing parms"
  (CERROR ':yes nil nil "User forced error~{~%~s~}" parms))

(defun pause (&optional string &rest vars)
  "Pause displaying format string with /"Hit any key to continue/" appended"
  (tv:with-mouse-grabbed
    (setq string (format nil "~@[~a~%~]Hit any key to continue." string))
    (setq tv:who-line-mouse-grabbed-documentation
	  (lexpr-funcall #'format nil string vars))
    (send terminal-io ':any-tyi)))

(defun pause (&optional string &rest vars &aux (stream query-io) c)
  "Pause displaying format string with /"Hit any key to continue/" appended"
  (multiple-value-bind (cx cy) (send stream ':read-cursorpos)
    (send stream ':home-cursor)
    (send stream ':clear-eol)
    (setq string (format nil "~@[~a~%~]Hit any key to continue:" string))
    (lexpr-funcall #'format stream string vars)
    (setq c (send stream ':any-tyi))
    (send stream ':set-cursorpos cx cy))
  c)

(defmacro accept (value &optional string &rest vars)
  (if string
      `(setq ,value (acceptf ,value ,string . ,vars))
    `(setq ,value (acceptf ,value ,(format nil "Enter new ~a:" value)))))

(defun acceptf (default &optional string &rest vars &aux (stream query-io) result)
  "Read a value & return it.  Displays prompt string at top of window"
  (multiple-value-bind (cx cy) (send stream ':read-cursorpos)
    (send stream ':home-cursor)
    (send stream ':clear-eol)
    (setq string (format nil "~:[Enter value:~*~;~a~] (default ~s) " string string default))
    (setq result
	  (SEND stream ':rubout-handler
		`((:prompt ,(lexpr-funcall #'format nil string vars)))
		#'(lambda (stream default)
		    (LET ((ch (SEND stream ':tyi)))
		      (SEND stream ':untyi ch)
		      (IF (EQ ch #\return)
			  default
			(READ-FOR-TOP-LEVEL stream ""))))
		stream default))
    (send stream ':set-cursorpos cx cy))
  result)

(DEFUN string-trim-all (char-set STRING)
  "Remove all characters in CHAR-SET from STRING"
  (DO ((i 0 (1+ i))
       (j 0)
       (l (STRING-LENGTH STRING))
       (new-string (STRING-APPEND STRING))
       ch)
      ((>= i l) (ADJUST-ARRAY-SIZE new-string j))
    (WHEN (NOT (MEMQ (SETQ ch (AREF STRING i)) char-set))
      (ASET ch new-string j) 
      (INCF j))))

(COMMENT DEFUN write-options (&optional (ALIST tv:*all-user-option-alists*) (STREAM standard-output)
		      &AUX (ALPHABETIC-CASE-AFFECTS-STRING-COMPARISON T)
		      (SI:PRINT-READABLY T))
  "Write forms on STREAM to set all non-default variables on ALIST to their current values.
That is, for each variable on ALIST whose current value is not its default,
a LOGIN-SETQ form is output to STREAM which records the variable's current value.
ALIST may be one list, or a list of alists."
  (WHEN (NLISTP (CAAR alist)) (SETQ alist (LIST alist)))
  (DOLIST (alist alist)
    (WHEN (SYMBOLP alist)
      (FORMAT stream "~2%~
~%;********************************************************************************~
~%;~%;  ~a~%;~
~%;********************************************************************************"
	      (DOCUMENTATION alist))
      (SETQ alist (SYMEVAL alist)))
    (DO ((ALIST ALIST (CDR ALIST))
	 (OPTION) (VALUE) (form))
	((NULL ALIST))
    (SETQ OPTION (CAAR ALIST)
	  value (GET OPTION 'tv:DEFAULT-VALUE)
	  form (CDDAR alist))
    (FORMAT stream "~2%#| ~a ~%~VQ|#~%" (GET option ':documentation) form #'grind-top-level)
    (GRIND-TOP-LEVEL `(LOGIN-SETQ ,OPTION ,(IF (OR (NUMBERP VALUE) (MEMQ VALUE '(T NIL)))
						   VALUE `',VALUE))
			 95. STREAM))))


;===============================================================================

(defun get-line-counts (list-of-files)
  "Obtain a list of {<file name> <line count>} pairs with total-line-count
   appended to it."
  (let ((list-of-pairs (loop for file in list-of-files
			     collect `(,(funcall
					  (fs:merge-pathname-defaults file)
					  ':string-for-printing)
				       ,(count-lines-in-file file)))))
    (loop for pair in list-of-pairs
	  sum (cadr pair) into accumulation
	  finally (return (append list-of-pairs
				  `((total-line-count ,accumulation)))))))

;-------------------------------------------------------------------------------

(defun count-lines-in-file (input-file)
  "Find the number of lines in an input file."
  (let ((input-path (probef input-file)))	; Check existence of input
    (cond
      (input-path
       (with-open-file (input-stream input-path)
	 (do ((line (multiple-value-list (funcall input-stream ':line-in))
		    (multiple-value-list (funcall input-stream ':line-in)))
	      (line-no 1. (1+ line-no)))
	     ((cadr line)
	      (if (zerop (string-length (car line)))	; Line buffer empty
		  (setq line-no (1- line-no)))	; Decrement line number
	      line-no))))			; Return line number
      (t (format t				; No input file - Notify user
		 "~&Input file not found ==> ~A~&" input-file) 0.))))


(defun convert-names-and-types (sources &optional (name-pairs nil)
						  (type-pairs nil)
						  (reverse-p nil))
  "Produce a list of file pathname components with possible substitutions for
   the file name and file type components.  Substitutions for these components
   are found in association lists; if a particular component isn't in a list,
   then no substitution is made.  If reverse-p is non-nil, component
   association is performed on the second element of the pair."
  (if (not (or name-pairs type-pairs)) sources	; No translation
      (loop for (dir nam typ ver) in sources
	    for ty = (substring typ 1.)
	    for new-nam = (name-substitute nam name-pairs reverse-p)
	    for new-typ = (string-append "."
			    (name-substitute (substring typ 1.)
			      type-pairs reverse-p))
	    collect `(,dir ,new-nam ,new-typ ,ver))))

;-------------------------------------------------------------------------------

(defun name-substitute (name &optional (pair-list nil)
				       (reverse-p nil))
  "Perform a substitution for name from the association list, pair-list.  If
   reverse-p is non-nil, then association is performed on the second element of
   the pair."
  (let ((sub (if reverse-p
		 (cond ((car (rass 'samepnamep-car name pair-list))) (t name))
		 (cond ((cadr (ass 'samepnamep name pair-list))) (t name)))))
    (if (stringp sub) sub (get-pname sub))))

;-------------------------------------------------------------------------------

(defun samepnamep-car (name1 name2)
  "Adjusts for rass' tendency to return a list instead of an atom."
  (samepnamep name1 (car name2)))

;-------------------------------------------------------------------------------
(defun file-list (pathnames
		  &optional (files-only-p nil)
			    (subdirectories-p nil)
			    (directory-first-p nil)
			    (deleted-files-p nil))
  "This function will produce a list of file pathname components  The input
   parameter(s) to this function are one or more pathnames (including
   wildcard specifications).  The result is a list of 4-tuples, the string
   representation for directory, file name, file type, and version number.
   If files-only-p is non-nil, no directory file names will be included.
   If subdirectories-p is non-nil, the function is invoked recursively to
   include all of the subdirectories' files in the list.  If
   directory-first-p is non-nil, the directory name will precede the names
   of subdirectory files in the generated list. If deleted-files-p is non-nil,
   then only files marked for deletion will be included in the generated list."
  (loop for pathname in pathnames
	append
	(loop for (path . stuff)
	      in (let ((d-list (fs:directory-list
				 pathname ':sorted ':deleted ':noerror)))
		   (if (listp d-list) (cdr d-list)))
	      append
		(let* ((ht (string (funcall path ':host)))
		       (dr (let ((dstr ">"))
			     (loop for dname in (funcall path ':directory)
				   do (setq dstr (string-append dstr dname ">"))
				   finally (return dstr))))
		       (dl (get (locf stuff) 'deleted))
		       (de (or (and dl deleted-files-p)
			       (and (not dl) (not deleted-files-p))))
		       (htdr (string-append ht ":" dr))
		       (nm (string (funcall path ':name)))
		       (ty (string (funcall path ':type)))
		       (ve (format nil "~D" (funcall path ':version)))
		       (tuple `((,htdr ,nm ,(string-append "." ty)
				,(string-append "." ve))))
		       (dirtype (string-equal ty "DIRECTORY"))
		       (subnames (if (not (and subdirectories-p dirtype)) nil
				     (file-list
				       `(,(string-append htdr nm ">*.*.*")
					 files-only-p subdirectories-p
					 directory-first-p)))))
		  (if de `(,@(if (not directory-first-p) subnames)
			   ,@(if (and files-only-p dirtype) nil tuple)
			   ,@(if directory-first-p subnames)))))))



;===============================================================================

(defun grind-object (sexp
		     &optional (output-type t)
			       (destination standard-output))
  "Grind a lisp object into a selected destination:
         standard-output, a buffer, or a file"
  (selectq output-type
    ((standard-output t)			; Output to standard output
     (grind-top-level sexp 80. standard-output))
    ((file)					; Output to a file
     (with-open-file
       (output-stream destination ':out)
       (grind-top-level sexp 80. output-stream)))
    ((buffer)					; Output to an editor buffer
     (with-open-file
       (output-stream (string-append "ed-buffer:" destination) ':out)
       (grind-top-level sexp 80. output-stream)))
    (otherwise					; Output type error
     (format t "~&Error -- ~A invalid output type~&" output-type) 'error)))

;===============================================================================

(defun deletefiles (source-pathnames &optional (fil t))
  "This function will cause files selected by source-pathnames to be deleted."
  (let ((victims
	  (let ((source (if (or (null source-pathnames)
				(listp source-pathnames))
			    source-pathnames `(,source-pathnames))))
	    (loop for (dir nam typ ver) in (file-list source fil nil nil nil)
		  collect (string-append dir nam typ ver)))))
     (loop for victim in victims
	   do (deletef victim))))

;-------------------------------------------------------------------------------

(defun undeletefiles (source-pathnames &optional (fil t))
  "This function will cause files selected by source-pathnames to be undeleted."
  (let ((victims
	  (let ((source (if (or (null source-pathnames)
				(listp source-pathnames))
			    source-pathnames `(,source-pathnames))))
	    (loop for (dir nam typ ver) in (file-list source fil nil nil t)
		  collect (string-append dir nam typ ver)))))
    (loop for victim in victims
	  do (undeletef victim))))

;-------------------------------------------------------------------------------

(defun copyfiles (source-pathnames
		  &optional (destination-directory *main-directory*)
			    (version-reset-p nil))
  "This function will cause files selected by source-pathnames to be
   copied to the directory defined by destination-directory.  version-reset-p
   will cause all of the copied files to have version numbers of 1."
  (let ((io-pair
	  (let ((source (if (or (null source-pathnames)
				(listp source-pathnames))
			    source-pathnames `(,source-pathnames)))
		(destination destination-directory))
	    (loop for (dir nam typ ver) in (file-list source t t t)
		  for s-file = (string-append dir nam typ ver)
		  for d-file = (if version-reset-p
				   (string-append destination nam typ ".1")
				   (string-append destination nam typ ver))
		  collect `(,s-file ,d-file)))))
    (loop for (s-file d-file) in io-pair
	  do (copyf s-file d-file))))

;-------------------------------------------------------------------------------

(defun create-inferior-directory (&optional (dir *main-directory*)
					    (name "temp"))
  (send (fs:parse-pathname (string-append dir name ">")) ':create-directory))

;-------------------------------------------------------------------------------

(defun obliterate-directory (directory)
  "Erase this directory and all of its subdirectories from the disk."
  (let* ((all `(,(string-append directory "*.*.*")))
	 (subs (loop for (dir nam typ ver) in (file-list all nil nil t nil)
		     if (equal typ ".DIRECTORY")
		     collect (string-append dir nam ">")))
	 (total (if subs (loop for sub in subs
			       sum (obliterate-directory sub)) 0.)))
    (+ total (clean-out-directory directory))))

;-------------------------------------------------------------------------------

(defun clean-out-directory (directory)
  (let ((files (string-append directory "*.*.*")))
    (deletefiles files nil)
    (expunge-directory directory)))

;-------------------------------------------------------------------------------

(defun expunge-directory (directory)
  (let ((d-list (fs:directory-list directory ':sorted ':deleted ':noerror)))
    (if (listp d-list) (fs:expunge-directory directory ':error nil) 0.)))

;-------------------------------------------------------------------------------
fs:
(COMMENT DEFUN :disk-usage (&aux date size author)
  (FORMAT t "~%~16a ~5a  ~a  ~a ~a"
	  "Directory" "Files" "Last Creation Date" "Total Size" "Author")
  (DOLIST (f (READ-DIRECTORY-FILES (LOOKUP-DIRECTORY nil)))
    (SETQ date 0 size 0)
    (SETQ author (file-author (CAR (file-files f))))
    (DOLIST (file (file-files f))
      (SETQ date (MAX date (file-creation-date file)))
      (INCF size (map-npages (file-map file))))
    (FORMAT t "~%~16a ~5d  ~\time\  ~6d     ~a"
	    (file-name f) (LENGTH (file-files f)) date size
	    (IF (STRING-EQUAL (file-name f) author) "" author))))


(defun relocate-file (name)
  (string-append *main-directory* name))

(defun file-contents (output-path &rest infiles)
  "Scan some files and print car and cadr of the top level elements"
  (with-open-file (output-stream output-path ':out)
    (loop for new-path in infiles
	  for input-path = (probef new-path)
	  do (cond (input-path
		    (with-open-file (input-stream input-path)
		      (format output-stream
			      "~3&Reading from file ==> ~A~&" input-path)
		      (do ((sexp (read input-stream nil)
				 (read input-stream nil))
			   (counter 1. (1+ counter)))
			  ((null sexp))
			(cond ((atom sexp)
			       (format output-stream
				       "~2&~4D. Symbol ==> ~A"
				       counter sexp))
			      ((< (length sexp) 2.)
			       (format output-stream
				       "~2&~4D. ~A ==> "
				       counter (car sexp)))
			      (t
			       (format output-stream
				       "~2&~4D. ~A ==> ~A"
				       counter (car sexp) (cadr sexp)))))))
		   (t (format output-stream
			      "~3&File not found ==> ~A~&" new-path))))))

(DEFUN find-hogs ()
  (LOOP for dir in (fs:top-level-directories)
        DO (FORMAT t "~%~A:~15T~5D pages~30T~3D files"
                   (directory-name dir)
                   (LOOP for file in (read-directory-files dir)
                         sum (file-npages file) into sum
                         finally (RETURN (+ (map-npages (directory-map dir)) sum)))
                   (LENGTH (directory-files dir)))))

========================================

(defconst translation-table
	  '(("BASIC-FLAVOR" 		"BASFLA.LSP")
	    ("BUILD-REFERENCE-CARDS"	"BUILDRC.LSP")
	    ("COMMAND-LOOP"		"CMDLP.LSP")
	    ("CONFIGURATIONS"		"CONFIG.LSP")
	    ("CREATE-SIGNAL"		"CRSIG.LSP")
	    ("DATA-BASE"		"DB.LSP")
	    ("DEFINITIONS"		"DEFN.LSP")
	    ("FILE-STUFF"		"FILEST.LSP")
	    ("GRAPHICS"			"GRPH.LSP")
	    ("IMAGE-MOUSE"		"IMMOU.LSP")
	    ("IMAGE-SETUP"		"IMSET.LSP")
	    ("INTERFACE"		"INTER.LSP")
	    ("INTRODUCTION"		"INTRO.LSP")
	    ("MISC-TABLES"		"MISC.TBL")
	    ("PANES"			"PANES.LSP")
	    ("PANES-TABLE"		"PANES.TBL")
	    ("RADIANS-TO-DEGREES"	"RTOD.LSP")
	    ("RC-GENERIC-FUNCTIONS"	"RCFG.LSP")
	    ("RC-IMAGE"			"RCIM.LSP")
	    ("RC-MARINE"		"RCMAR.LSP")
	    ("RC-SDP"			"RCSDP.LSP")
	    ("RC-SIGNAL-INPUT"		"RCSIGIN.LSP")
	    ("RC-SIGNAL-OPERATIONS"	"RCSOP.LSP")
	    ("RC-SIW-MENU"		"RCSIWM.LSP")
	    ("RC-SPEECH"		"RCSPCH.LSP")
	    ("SESSION-MANAGER"		"SM.LSP")
	    ("SYSTEM"			"SYSTEM.LSP")
	    ("SYSTEM-HIERARCHY"		"SYSHI.LSP")
	    ("SYSTEM-HIERARCHY-TABLES"	"SYSHI.TBL")
	    ("TREE-EDITOR"		"TREED.LSP")
	    ("MARINE-1-MOUSE"		"MAR1MOU.LSP"	"WISP")
	    ("MARINE-AUXILIARY"		"MARAUX.LSP"	"WISP")
	    ("MARINE-COMPASS-DB"	"MARCDB.LSP"	"WISP")
	    ("MARINE-DEFINITIONS"	"MARDF.LSP"	"WISP")
	    ("MARINE-DESPIKE"	 	"MARDS.LSP"	"WISP")
	    ("MARINE-DESPIKE-LMS" 	"MARDSLM.LSP"	"WISP")
	    ("MARINE-DESPIKE-UTILS" 	"MARDSUT.LSP"	"WISP")
	    ("MARINE-DESPIKE-VARIABLES"	"MARDSVA.LSP"	"WISP")
	    ("MARINE-EDIT"	   	"MARED.LSP"	"WISP")
	    ("MARINE-PARAMETERS"	"MARPARM.LSP"	"WISP")
	    ("MARINE-QC"		"MARQC.LSP"	"WISP")
	    ("MARINE-QC2"		"MARQC2.LSP"	"WISP")
	    ("MARINE-READ-TAPE"		"MARRDTP.LSP"	"WISP")
	    ("MARINE-SMOOTH"		"MARSM.LSP"	"WISP")
	    ("MARINE-TYPEOUT"		"MARTYPE.LSP"	"WISP")
	    ("MARINE-ZOOM"		"MARZM.LSP"	"WISP")
	    ))

(defun copy-to-vax ()
  (let ((files (fs:directory-list "lm:siwb;*.lisp#>" ':sorted)))
    (loop for (file . properties) in files
	  do
	  (if (and file (y-or-n-p (send file ':string-for-printing)))
	      (let ((trans (assoc (send file ':name) translation-table)))
		 (if trans
		     (condition-case ()
			 (fs:copy-file file (string-append "v4:v4ud:[glicksman."
							   (cond ((third trans)) (t "siw"))
							   "]" (second trans)))
		         (fs:file-already-exists t))))))))

(defun files-size ()
  (let* ((files (fs:directory-list "lm:siwb;*.lisp#>" ':sorted))
	 (sum (loop for file-and-attributes in files
		    for file = (first file-and-attributes)
		    if (and file (y-or-n-p (send file ':string-for-printing)))
		    summing (get file-and-attributes ':length-in-bytes))))
    (format t "~&Length: ~D~%" sum)))

(defun print-files ()
  (let ((files (fs:directory-list "lm:siwb;*.lisp#>" ':sorted)))
    (format t "~&~18X Lispm Name ~32X Vax Name~2%")
    (loop for (file . properties) in files
	  do
	  (if file
	      (let ((trans (assoc (send file ':name) translation-table))
		    (vax-print-name ""))
		 (if trans
		     (setq vax-print-name (format nil "~11A    (~A)"
						  (second trans)
						  (cond ((third trans)) (t "SIW")))))
		 (format t "~&    ~45A      ~25A~%" (send file ':string-for-printing)
						  vax-print-name))))))

========================================

(DEFUN change-names ()
  (LET ((files (fs:directory-list "lam2:nichols;*.text#>")))
    (LOOP for file in files
          DO (PUTPROP (CADR file) "Dan The Man" ':author))))

(DEFUN get-name ()
  (LET ((file (CADR (fs:directory-list "lam2:nichols;lispm.init#49"))))
    (GET file ':author)))

(DEFUN put-name ()
  (LET ((file (CADR (fs:directory-list "lam2:nichols;lispm.init#49"))))
    (putprop file "DAN" ':author)))


;(DEFUN GET-PATHNAME
;       (&optional (allow-any t)
;	          (prompt    "Enter pathname")
;		  (default   (fs:merge-pathname-defaults "foo.picture")))
;  "This function gets a valid file pathname through a pop-up type-in window.
;The allow-any argument specifies if the pathname of a non-existant file should
;be allowed or not."
;  (DECLARE (RETURN-LIST pathname))
;  (DO (new-pathname STATUS) (nil)
;    (SETQ new-pathname (tv:get-line-from-keyboard
;			 (FORMAT nil "~A~%(Default=~A)" prompt default))
;	  default      (fs:merge-pathname-defaults new-pathname default)
;	  STATUS       (OR allow-any (PROBEF default)))
;    (COND ((NOT STATUS)
;	   (MULTIPLE-VALUE (new-pathname STATUS)
;	     (fs:complete-pathname default new-pathname "picture" nil))
;	   (COND ((EQ STATUS ':old)
;		  (RETURN (fs:parse-pathname new-pathname)))
;		 (t (SETQ default (fs:merge-pathname-defaults new-pathname
;							      default)))))
;	  (t (RETURN default)))
;    (FUNCALL tv:selected-window ':beep)))

;(DEFFLAVOR pop-up-editor-window ()
;	   (zwei:standalone-editor-window tv:box-label-mixin)
;  (:default-init-plist :*comtab* zwei:*standalone-comtab*
;		       :height   200.
;                       :label    (LIST ':string "Press  to exit."
;				       ':font   fonts:hl12b))
;  (:documentation :combination
;   "This is a stand-alone, pop-up editor window for editting arbitrary text strings.
;It has a boxed in label area so that it can be used to display some documentation
;text which will stand out from the text that is being edited."))


;(DEFVAR pop-up-editor-window (tv:make-window 'pop-up-editor-window)
;  "An instance of the pop up editor window that can be shared by everyone.
;This window is used to pop up a text editor for editting of an arbitrary text
;string.  It is global so we don't have to wait for its creation each time that we
;want to edit something.  To use this execute the function ZWEI:EDSTRING with
;this window as the second argument.")
*************
(DEFUN ADD-LOGICAL-PATHNAME-HOST (LOGICAL-HOST PHYSICAL-HOST TRANSLATIONS
				  &OPTIONAL DEFAULT-DEVICE
				  &AUX LOG OLD DEFDEV)
  "Define a logical host named LOGICAL-HOST, which translates to PHYSICAL-HOST.
TRANSLATIONS specifies the directory translations to use: each element looks like
 (logical-dir physical-dir), where logical-dir is just a string containing the
 directory name to translate, and physical-dir contains a directory and optionally
 a device, complete with delimiters as appropriate for PHYSICAL-HOST to parse.
An element in TRANSLATIONS that has NIL instead of a logical-dir specifies
the default device, to be used as the device when translating directories
that are not mentioned in TRANSLATIONS.
DEFAULT-DEVICE can be used to override the default device found in the translations."
  (IF (SETQ LOG (GET-PATHNAME-HOST LOGICAL-HOST T))
      (SETQ OLD T)
      (PUSH (SETQ LOG (MAKE-INSTANCE 'LOGICAL-HOST ':NAME LOGICAL-HOST))
	    *PATHNAME-HOST-LIST*))

  (SETQ PHYSICAL-HOST (OR (GET-PATHNAME-HOST PHYSICAL-HOST T) (SI:PARSE-HOST PHYSICAL-HOST)))
  ;; Here is a bit of a kludge for SI:SET-SITE.  If the physical host is not defined yet,
  ;; add it now.

  (OR (MEMQ PHYSICAL-HOST *PATHNAME-HOST-LIST*)
      (PUSH PHYSICAL-HOST *PATHNAME-HOST-LIST*))

  (FUNCALL LOG ':SET-HOST PHYSICAL-HOST)

  (IF TRANSLATIONS
      (FUNCALL LOG ':SET-TRANSLATIONS
	       (LOOP FOR (LOGICAL-DIRECTORY PHYSICAL-DIRECTORY) IN TRANSLATIONS
		     WITH DEVICE AND DIRECTORY
		     DO (LET ((PN (PARSE-PATHNAME PHYSICAL-DIRECTORY PHYSICAL-HOST)))

			  (SETQ DIRECTORY (PATHNAME-DIRECTORY PN))


			  (SETQ DEVICE (PATHNAME-DEVICE PN)))
		     WHEN (AND LOGICAL-DIRECTORY (MEMQ DIRECTORY '(NIL :UNSPECIFIC)))
		     DO (FERROR NIL
	"No directory specified in ~A, you probably forgot some delimiter characters."
				PHYSICAL-DIRECTORY)
		     ;; A translation for logical directory NIL specifies the default device.
		     WHEN (NULL LOGICAL-DIRECTORY)
		     DO (SETQ DEFDEV DEVICE)
		     ;; Default the default directory to the host's primary device.
		     WHEN (NULL DEFDEV)
		     DO (SETQ DEFDEV
			      (FUNCALL (SAMPLE-PATHNAME PHYSICAL-HOST) ':PRIMARY-DEVICE))
		     WHEN LOGICAL-DIRECTORY
		     COLLECT (MAKE-LOGICAL-PATHNAME-TRANSLATION
			       LOGICAL-DIRECTORY LOGICAL-DIRECTORY
			       PHYSICAL-DEVICE DEVICE
			       PHYSICAL-DIRECTORY DIRECTORY))))

  (FUNCALL LOG ':SET-DEFAULT-DEVICE (OR DEFAULT-DEVICE DEFDEV))

  LOG)

(COMMENT
(DEFUN CHANGE-LOGICAL-PATHNAME-HOST (LOGICAL-HOST PHYSICAL-HOST &AUX LOG)
  ;; Get the host instances for the logical and physical hosts.  They are both
  ;; pathname hosts, except that in the case of the physical host we may not
  ;; have learned that yet.
  (SETQ LOG (GET-PATHNAME-HOST LOGICAL-HOST))
  (SETQ PHYSICAL-HOST (OR (GET-PATHNAME-HOST PHYSICAL-HOST T) (SI:PARSE-HOST PHYSICAL-HOST)))
  (FUNCALL LOG ':SET-HOST PHYSICAL-HOST)
  ;; Here is a bit of a kludge for SI:SET-SITE.  If the physical host is not defined yet,
  ;; add it now.
  (OR (MEMQ PHYSICAL-HOST *PATHNAME-HOST-LIST*)
      (PUSH PHYSICAL-HOST *PATHNAME-HOST-LIST*))))

(DEFUN CHANGE-LOGICAL-PATHNAME-DIRECTORY (LOGICAL-HOST LOGICAL-DIRECTORY PHYSICAL-DIRECTORY
					  &AUX LOG PHYSICAL-HOST DEVICE DIRECTORY TRAN)
  (SETQ LOGICAL-DIRECTORY (STRING-UPCASE LOGICAL-DIRECTORY))
  (SETQ LOG (GET-PATHNAME-HOST LOGICAL-HOST))
  (SETQ PHYSICAL-HOST (FUNCALL LOG ':HOST))
  (LET ((PN (PARSE-PATHNAME PHYSICAL-DIRECTORY PHYSICAL-HOST)))
    (SETQ DIRECTORY (PATHNAME-DIRECTORY PN))
    (SETQ DEVICE (PATHNAME-DEVICE PN)))
  (AND (MEMQ DIRECTORY '(NIL :UNSPECIFIC))
       (FERROR NIL
  "No directory specified in ~A, you probably forgot some delimiter characters."
	       PHYSICAL-DIRECTORY))
  (IF (NULL (SETQ TRAN (ASSOC LOGICAL-DIRECTORY (FUNCALL LOG ':TRANSLATIONS))))
      (PUSH (MAKE-LOGICAL-PATHNAME-TRANSLATION
	      LOGICAL-DIRECTORY LOGICAL-DIRECTORY
	      PHYSICAL-DEVICE DEVICE
	      PHYSICAL-DIRECTORY DIRECTORY)
	    (FUNCALL LOG ':TRANSLATIONS))
      (SETF (TRANSLATION-PHYSICAL-DEVICE TRAN) DEVICE)
      (SETF (TRANSLATION-PHYSICAL-DIRECTORY TRAN) DIRECTORY)))
******************************
;;;diff
(DEFUN diff (&optional file1 file2)
  "Compare two files The output goes on the screen,
and also into a buffer named *Source Compare ...*."
  (LET (path1 path2 file-1 file-2 name-1 name-2)
    (COND (file1
           (COND (file2
                  (setq path1 (fs:parse-pathname file1)
                        path2 (fs:parse-pathname file2)))
                 (t
                  (SETQ path1 (fs:parse-pathname file1)
                        path2 path1)))         ;need version-1
           (SETQ file-1 (srccom:create-file file1)
                 name-1 (fs:parse-pathname file1)
                 file-2 (srccom:create-file file2)
                 name-2 (fs:parse-pathname file2)))
          (t
           (MULTIPLE-VALUE (file-1 name-1) (get-file "~%Compare" nil))
           (MULTIPLE-VALUE (file-2 name-2) (get-file (FORMAT nil "~%Compare ~A  with"
                                                             name-1)))))
    (UNWIND-PROTECT
        (PROGN
          (FORMAT t "*Source Compare ~A // ~A*" name-1 name-2)
          (SRCCOM:DESCRIBE-SRCCOM-SOURCES FILE-1 FILE-2 STANDARD-OUTPUT)
          (SRCCOM:SOURCE-COMPARE-FILES FILE-1 FILE-2 STANDARD-OUTPUT (SRCCOM:QUERY-TYPE)))
      (FORMAT T "~&Done.")
      (AND FILE-1 (FUNCALL (SRCCOM:FILE-STREAM FILE-1) ':CLOSE))
      (AND FILE-2 (FUNCALL (SRCCOM:FILE-STREAM FILE-2) ':CLOSE)))))


(DEFUN GET-FILE (PROMPT &OPTIONAL DEFAULT OLDEST-P)
  (DECLARE (RETURN-LIST FILE NAME))
  (LET ((PATHNAME (fs:parse-pathname (PROMPT-AND-READ ':pathname-or-nil (FORMAT nil "~A: " prompt "File")))))
    (FILE-RETRY-NEW-PATHNAME (PATHNAME fs:file-error)
      (VALUES (srccom:create-file PATHNAME) PATHNAME))))
                                                      
******************************
(DEFUN lsd ()
  (LET* ((DEFAULT (FUNCALL (DEFAULT-PATHNAME) ':NEW-PATHNAME
			   ':DIRECTORY ':WILD ':NAME ':UNSPECIFIC
			   ':TYPE ':UNSPECIFIC ':VERSION ':UNSPECIFIC))
	 (PATHNAME (READ-DEFAULTED-PATHNAME "List directories:" DEFAULT
					    ':UNSPECIFIC ':UNSPECIFIC))
	 (DIRS (FS:ALL-DIRECTORIES PATHNAME ':NOERROR)))
    (IF (ERRORP DIRS)
	(BARF "Error: ~A" DIRS)
      (SETQ DIRS (SORTCAR DIRS #'FS:PATHNAME-LESSP))
      (FUNCALL STANDARD-OUTPUT ':ITEM-LIST 'DIRECTORY
	       (LOOP FOR (PATHNAME) IN DIRS
		     COLLECT `(,(FUNCALL PATHNAME ':STRING-FOR-DIRECTORY)
			       . ,(FUNCALL PATHNAME ':NEW-PATHNAME ':NAME ':WILD
					   ':TYPE ':WILD ':VERSION ':WILD)))))))


******************************

(DEFUN MY-VIEW-FILE (PATHNAME &OPTIONAL DELETED-P)
  "/"View/" the text of file PATHNAME.  DELETED-P means allow deleted files.
This means let the user scroll with Space and Overprint.
A window overlying *WINDOW* is used."
  (WITH-OPEN-FILE (STREAM PATHNAME ':ERROR ':RETRY ':PRESERVE-DATES T ':DELETED DELETED-P)
    (FORMAT t "~%Viewing ~A" (FUNCALL STREAM ':TRUENAME))
    (my-VIEW-STREAM STREAM)))

(DEFUN my-VIEW-STREAM (STREAM)
  "/"View/" the text read from STREAM.
This means let the user scroll with Space and Overstrike.
A window overlying *WINDOW* is used."
  (LET ((window tv:selected-window))
    (FUNCALL (WINDOW-SHEET WINDOW) ':SET-LABEL "")
    (VIEW-WINDOW WINDOW STREAM)))

******************************
(DEFCOM COM-DELETE-FILE "Delete a file.
If wildcards are used, many files can be deleted." ()
  (LET ((PATHNAME (READ-DEFAULTED-PATHNAME "Delete file:" (PATHNAME-DEFAULTS))))
    (IF (SEND PATHNAME ':WILD-P)
	(LET ((DIR (CDR (FS:DIRECTORY-LIST PATHNAME))))
	  (FORMAT T "~&Files to be deleted:~%")
	  (MAPC *DIRECTORY-SINGLE-FILE-LISTER* DIR)
	  (WHEN (LET ((QUERY-IO STANDARD-OUTPUT))
		  (Y-OR-N-P "Delete them all? "))
	    (DOLIST (ELT DIR)
	      (CONDITION-CASE (ERROR)
		  (SEND (CAR ELT) ':DELETE)
		((FS:FILE-ERROR SYS:REMOTE-NETWORK-ERROR)
		 (FORMAT T "~&Deletion failure: ~A" ERROR))))
	    (FORMAT T "~&Done.~%")))
      (CONDITION-CASE (VALUE)
	  (DELETE-FILE PATHNAME)
	((FS:FILE-ERROR SYS:REMOTE-NETWORK-ERROR)
	 (BARF VALUE))
	(:NO-ERROR
	 (FORMAT QUERY-IO "~&~A deleted." (CAAR VALUE))))))
  DIS-NONE)

(DEFCOM COM-UNDELETE-FILE "Undelete a file.
If wildcards are used, many files can be undeleted." ()
  (LET ((PATHNAME (READ-DEFAULTED-PATHNAME "Undelete file:" (PATHNAME-DEFAULTS))))
    (IF (SEND PATHNAME ':WILD-P)
	(LET ((DIR (CDR (FS:DIRECTORY-LIST PATHNAME ':DELETED))))
	  (FORMAT T "~&Files to be undeleted:~%")
	  (MAPC *DIRECTORY-SINGLE-FILE-LISTER* DIR)
	  (WHEN (LET ((QUERY-IO STANDARD-OUTPUT))
		  (Y-OR-N-P "Undelete them all? "))
	    (DOLIST (ELT DIR)
	      (CONDITION-CASE (ERROR)
		  (SEND (CAR ELT) ':UNDELETE)
		((FS:FILE-ERROR SYS:REMOTE-NETWORK-ERROR)
		 (FORMAT T "~&Undeletion failure: ~A" ERROR))))
	    (FORMAT T "~&Done.~%")))
      (CONDITION-CASE (VALUE)
	  (UNDELETE-FILE PATHNAME)
	((FS:FILE-ERROR SYS:REMOTE-NETWORK-ERROR)
	 (BARF VALUE))
	(:NO-ERROR
	 (FORMAT QUERY-IO "~&~A undeleted." (CAAR VALUE))))))
  DIS-NONE)

(DEFCOM COM-RENAME-FILE "Rename a file.
If wildcards are used, many files can be renamed." ()
  (LET* ((PATHNAME (READ-DEFAULTED-PATHNAME "Rename file:" (PATHNAME-DEFAULTS)))
	 (TO-SPEC (READ-UNDEFAULTED-PATHNAME-STRING
		    (FORMAT NIL "Rename file ~A to:" PATHNAME)
		    PATHNAME))
	 BUFFERS-CONSIDERED)
    (DECLARE (SPECIAL BUFFERS-CONSIDERED))
    (IF (SEND PATHNAME ':WILD-P)
	(LET ((DIR (CDR (FS:DIRECTORY-LIST PATHNAME)))
	      (TO-PATHNAME (FS:MERGE-PATHNAMES TO-SPEC PATHNAME)))
	  (FORMAT T "~&Files to be renamed:~%")
	  (MAPC *DIRECTORY-SINGLE-FILE-LISTER* DIR)
	  (WHEN (LET ((QUERY-IO STANDARD-OUTPUT))
		  (Y-OR-N-P "Rename them all? "))
	    (DOLIST (ELT DIR)
	      (CONDITION-CASE (ERROR)
		  (SEND (CAR ELT) ':RENAME
			(SEND PATHNAME ':TRANSLATE-WILD-PATHNAME TO-PATHNAME (CAR ELT)))
		((FS:FILE-ERROR SYS:REMOTE-NETWORK-ERROR)
		 (FORMAT T "~&Rename failure: ~A" ERROR))
		(:NO-ERROR
		 (RENAME-FILE-1 PATHNAME
				(SEND PATHNAME ':TRANSLATE-WILD-PATHNAME
				      TO-PATHNAME (CAR ELT))))))
	    (FORMAT T "~&Done.~%")))
      (CONDITION-CASE (ERROR OLD-TRUENAME NEW-TRUENAME)
	  (RENAME-FILE PATHNAME TO-SPEC)
	((FS:FILE-ERROR SYS:REMOTE-NETWORK-ERROR)
	 (BARF ERROR))
	(:NO-ERROR
	 (FORMAT QUERY-IO "~&~A renamed~% to ~A." OLD-TRUENAME NEW-TRUENAME)
	 (RENAME-FILE-1 OLD-TRUENAME NEW-TRUENAME)))))
  DIS-NONE)

(DEFUN RENAME-FILE-1 (INPUT-PATHNAME OUTPUT-PATHNAME)
  (DECLARE (SPECIAL BUFFERS-CONSIDERED))
  ;; Offer to rename a buffer visiting this file, no specific version.
  ;; In order to avoid asking the same question for each file version renamed,
  ;; we record buffers that have been asked about and don't ask a second time.
  (LET ((BUF (FIND-FILE-BUFFER (SEND INPUT-PATHNAME ':NEW-VERSION ':NEWEST))))
    (WHEN (AND BUF (NOT (MEMQ BUF BUFFERS-CONSIDERED)))
      (PUSH BUF BUFFERS-CONSIDERED)
      (IF (FQUERY NIL "~&Rename buffer ~A as well? " BUF)
	  (SET-BUFFER-PATHNAME (SEND OUTPUT-PATHNAME ':NEW-VERSION ':NEWEST) BUF))))
  ;; Offer to rename a buffer visiting this version number specifically.
  (LET ((BUF (FIND-FILE-BUFFER INPUT-PATHNAME)))
    (WHEN (AND BUF (NOT (MEMQ BUF BUFFERS-CONSIDERED)))
      (PUSH BUF BUFFERS-CONSIDERED)
      (WHEN (FQUERY NIL "~&Rename buffer ~A as well? " BUF)
	(SET-BUFFER-PATHNAME OUTPUT-PATHNAME BUF)
	(WHEN (CONSP (BUFFER-FILE-ID BUF))
	  (SETF (BUFFER-FILE-ID BUF)
		(CONS OUTPUT-PATHNAME (CDR (BUFFER-FILE-ID BUF)))))))))

******************************
(DEFUN READ-DIRECTORY-NAME (PROMPT PATHNAME &OPTIONAL (WILDP ':WILD) &AUX TYPEIN)
  "Read a pathname to pass to FS:DIRECTORY-LIST.
Prompt with PROMPT, a string probably ending in a colon.
PATHNAME gives the defaults for host, device, directory.
WILDP gives the default used for the other components;
 normally :WILD, but could be NIL."
  (SETQ PATHNAME (FUNCALL PATHNAME ':NEW-PATHNAME ':NAME WILDP
						  ':TYPE WILDP
						  ':VERSION WILDP)
	PROMPT (FORMAT NIL "~A (Default is ~A)" PROMPT PATHNAME))
  (LET ((*READING-PATHNAME-DEFAULTS* PATHNAME)
	(*READING-PATHNAME-SPECIAL-TYPE* ':WILD)
	(*READING-PATHNAME-SPECIAL-VERSION* ':WILD)
	(*READING-PATHNAME-DIRECTION* ':READ)
	(*MINI-BUFFER-VALUE-HISTORY* *PATHNAME-ARGUMENT-HISTORY*))
    (MULTIPLE-VALUE-BIND (NIL NIL INTERVAL)
	(EDIT-IN-MINI-BUFFER *PATHNAME-READING-COMTAB* NIL NIL 
			     (LIST PROMPT '(:RIGHT-FLUSH " (Completion)")))
      (SETQ TYPEIN (STRING-INTERVAL INTERVAL))))
  (COND ((EQUAL TYPEIN "")
	 (PUSH-ON-HISTORY PATHNAME *PATHNAME-ARGUMENT-HISTORY*)
	 PATHNAME)
;	((NOT (DO ((I 0 (1+ I))
;		   (LEN (STRING-LENGTH TYPEIN))
;		   (CH))
;		  (( I LEN) NIL)
;		(SETQ CH (AREF TYPEIN I))
;		(OR (AND ( CH #/A) ( CH #/Z))
;		    (AND ( CH #/a) ( CH #/z))
;		    (AND ( CH #/0) ( CH #/9))
;		    (= CH #/-)
;		    (RETURN T))))
;	 ;;No funny characters, must be just a directory name
;	 (FUNCALL PATHNAME ':NEW-DIRECTORY TYPEIN))
	(T
	 (LET ((PATHNAME (FS:MERGE-PATHNAME-DEFAULTS TYPEIN PATHNAME ':WILD ':WILD)))
	   (PUSH-ON-HISTORY PATHNAME *PATHNAME-ARGUMENT-HISTORY*)
	   PATHNAME))))

(DEFUN READ-UNDEFAULTED-DIRECTORY-STRING (PROMPT PATHNAME &OPTIONAL (WILDP ':WILD))
  "Read a string specifying a pathname to pass to FS:DIRECTORY-LIST.
Prompt with PROMPT, a string probably ending in a colon.
PATHNAME gives the defaults for host, device, directory.
WILDP gives the default used for the other components;
 normally :WILD, but could be NIL.
These defaults are used only for completion."
  (SETQ PATHNAME (FUNCALL PATHNAME ':NEW-PATHNAME ':NAME WILDP
						  ':TYPE WILDP
						  ':VERSION WILDP)
	PROMPT (FORMAT NIL "~A (Default is ~A)" PROMPT PATHNAME))
  (LET ((*READING-PATHNAME-DEFAULTS* PATHNAME)
	(*READING-PATHNAME-SPECIAL-TYPE* ':WILD)
	(*READING-PATHNAME-SPECIAL-VERSION* ':WILD)
	(*READING-PATHNAME-DIRECTION* ':READ)
	(*MINI-BUFFER-VALUE-HISTORY* *PATHNAME-ARGUMENT-HISTORY*))
    (MULTIPLE-VALUE-BIND (NIL NIL INTERVAL)
	(EDIT-IN-MINI-BUFFER *PATHNAME-READING-COMTAB* NIL NIL 
			     (LIST PROMPT '(:RIGHT-FLUSH " (Completion)")))
      (LET ((STRING (STRING-INTERVAL INTERVAL)))
	(PUSH-ON-HISTORY STRING *PATHNAME-ARGUMENT-HISTORY*)
	STRING))))


(DEFUN VIEW-DIRECTORY (VIEWED-DIRECTORY)
  "/"View/" the text of the directory listing of VIEWED-DIRECTORY.
That is, let user scroll through it using Space and Overprint."
  (SETQ VIEWED-DIRECTORY (FS:MERGE-PATHNAME-DEFAULTS VIEWED-DIRECTORY *PATHNAME-DEFAULTS*))
  (FILE-RETRY-NEW-PATHNAME (VIEWED-DIRECTORY FS:FILE-ERROR)
    (WITH-OPEN-STREAM (STREAM (DIRECTORY-INPUT-STREAM VIEWED-DIRECTORY))
      (SETQ VIEWED-DIRECTORY (FUNCALL STREAM ':DIRECTORY-PATHNAME))
      (PROMPT-LINE "Viewing directory ~A" VIEWED-DIRECTORY)
      (VIEW-STREAM STREAM)))
  DIS-NONE)

(DEFUN DIRECTORY-INPUT-STREAM (DIRECTORY &AUX (DIR DIRECTORY))
  "Return a stream that reads a directory listing of pathname DIRECTORY.
Th stream supports only the :LINE-IN operation."
  (LET-CLOSED ((DIRECTORY DIR)
	       (DIRECTORY-LIST-STREAM NIL)
	       (REREAD-ENTRY))
    (COND ((ERRORP (SETQ DIRECTORY-LIST-STREAM
			 (FS:DIRECTORY-LIST-STREAM DIRECTORY)))
	   (BARF "Error: ~A"  DIRECTORY-LIST-STREAM)))
    (SETQ REREAD-ENTRY (FUNCALL DIRECTORY-LIST-STREAM ':ENTRY))
    (AND REREAD-ENTRY (NULL (CAR REREAD-ENTRY))
	 (SETQ DIRECTORY (GET REREAD-ENTRY ':PATHNAME)))
    #'DIRECTORY-INPUT-STREAM-IO))

;REREAD-ENTRY, if non-NIL, is an entry from the directory-list-stream
;that was peeked-ahead at.  The analogue of an UNTYI'd character.
(LOCAL-DECLARE ((SPECIAL DIRECTORY-LIST-STREAM DIRECTORY REREAD-ENTRY))
(DEFSELECT DIRECTORY-INPUT-STREAM-IO
  (:DIRECTORY-PATHNAME () DIRECTORY)
  (:CHARACTERS () T)
  (:LINE-IN (&OPTIONAL (LEADER 1) &AUX STRING TEM)
    (COND (DIRECTORY-LIST-STREAM
	   (COND ((OR DIRECTORY
		      (SETQ TEM (OR (PROG1 REREAD-ENTRY (SETQ REREAD-ENTRY NIL))
				    (FUNCALL DIRECTORY-LIST-STREAM ':ENTRY))))
		  (SETQ STRING (MAKE-ARRAY 80. ':TYPE 'ART-STRING
					       ':LEADER-LENGTH (IF (NUMBERP LEADER) LEADER 1)
					       ':LEADER-LIST '(0)))
		  (WITH-OUTPUT-TO-STRING (S STRING)
		    (IF TEM (FUNCALL *DIRECTORY-SINGLE-FILE-LISTER* TEM S)
		      (FUNCALL S ':STRING-OUT
			       (FUNCALL DIRECTORY ':STRING-FOR-PRINTING))
		      (SETQ DIRECTORY NIL)))
		  ;; Chop off any CRs
		  (COND ((= (AREF STRING (SETQ TEM (1- (ARRAY-LEADER STRING 0)))) #\CR)
			 (SETF (ARRAY-LEADER STRING 0) TEM)))
		  STRING)
		 (T (FUNCALL DIRECTORY-LIST-STREAM ':CLOSE)
		    (SETQ DIRECTORY-LIST-STREAM NIL)
		    (VALUES NIL T))))
	  (T (VALUES NIL T))))
  (:CLOSE (&OPTIONAL MODE)
    (COND ((AND DIRECTORY-LIST-STREAM
		(NOT (TYPEP DIRECTORY-LIST-STREAM 'FS:PATHNAME)))
	   (FUNCALL DIRECTORY-LIST-STREAM ':CLOSE MODE)
	   (SETQ DIRECTORY-LIST-STREAM NIL)
	   T)))) )
******************************
;;; Directory Listing stuff.

(DEFCOM COM-DISPLAY-DIRECTORY "Display current buffer's file's directory.
Use the directory listing function in the variable Directory Lister.
With an argument, accepts the name of a file to list." ()
  (LET ((PATHNAME (READ-DEFAULTED-WILD-PATHNAME "Display Directory:"
						(DEFAULT-PATHNAME)
						(NOT *NUMERIC-ARG-P*))))
    (FILE-RETRY-NEW-PATHNAME (PATHNAME FS:FILE-ERROR)
      (FUNCALL *DIRECTORY-LISTER* PATHNAME)))
  DIS-NONE)

(DEFUN READ-DEFAULTED-WILD-PATHNAME (PROMPT &OPTIONAL (DEFAULT (DEFAULT-PATHNAME))
				     DONT-READ-P)
  "Read and return a pathname, defaulting type and version to :WILD.
DEFAULT is a pathname that provides defaults for the other things.
DONT-READ-P means just return a default, don't read anything."
  (SETQ DEFAULT (FUNCALL DEFAULT ':NEW-PATHNAME ':TYPE ':WILD ':VERSION ':WILD))
  (OR DONT-READ-P
      (SETQ DEFAULT (READ-DEFAULTED-PATHNAME PROMPT DEFAULT ':WILD ':WILD)))
  DEFAULT)

(DEFUN MAYBE-DISPLAY-DIRECTORY (TYPE &OPTIONAL (PATHNAME (DEFAULT-PATHNAME)))
  "Do automatic directory display if the user wants it.
TYPE is :READ or :WRITE, saying what sort of file operation
has just been done.  PATHNAME specifies host, device, directory and name to list."
  (COND ((OR (AND (EQ TYPE ':READ) (MEMQ *AUTO-DIRECTORY-DISPLAY* '(:READ T)))
	     (AND (EQ TYPE ':WRITE) (MEMQ *AUTO-DIRECTORY-DISPLAY* '(:WRITE T))))
	 (FUNCALL *DIRECTORY-LISTER* (FUNCALL PATHNAME ':NEW-PATHNAME ':TYPE ':WILD
								      ':VERSION ':WILD)))))

;;; This is the default directory listing routine
(DEFUN DEFAULT-DIRECTORY-LISTER (PATHNAME)
  "Print a directory listing of PATHNAME in the default manner.
Uses the value of *DIRECTORY-SINGLE-FILE-LISTER* on each element of the directory-list."
  (WITH-OPEN-STREAM (STREAM (FS:DIRECTORY-LIST-STREAM PATHNAME))
    (LET ((NILENTRY (SEND STREAM ':ENTRY)))
      (IF (CAR NILENTRY)
	  (FERROR NIL "First entry returned by a directory-list stream is not for NIL"))
      ;; What directory did we actually read?
      (SETQ PATHNAME (OR (GET NILENTRY ':PATHNAME) PATHNAME))
      (FORMAT T "~&~A~%" PATHNAME)
      (FUNCALL *DIRECTORY-SINGLE-FILE-LISTER* NILENTRY)
      (DO-FOREVER
	(LET ((ENTRY (SEND STREAM ':ENTRY)))
	  (OR ENTRY (RETURN))
	  (FUNCALL *DIRECTORY-SINGLE-FILE-LISTER* ENTRY)))))
  (FORMAT T "Done.~%"))

;Note that *DIRECTORY-SINGLE-FILE-LISTER* is expected to output lines.

(DEFUN DEFAULT-LIST-ONE-FILE (FILE &OPTIONAL (STREAM STANDARD-OUTPUT) &AUX PATHNAME)
  (COND ((NULL (SETQ PATHNAME (CAR FILE)))
	 (COND ((GET FILE ':DISK-SPACE-DESCRIPTION)
		(FUNCALL STREAM ':LINE-OUT (GET FILE ':DISK-SPACE-DESCRIPTION)))
	       ((GET FILE ':PHYSICAL-VOLUME-FREE-BLOCKS)
		(DO ((FREE (GET FILE ':PHYSICAL-VOLUME-FREE-BLOCKS) (CDR FREE))
		     (FLAG T NIL))
		    ((NULL FREE) (FUNCALL STREAM ':TYO #\CR))
		 (FORMAT STREAM "~A #~A=~D" (IF FLAG "Free:" ",") (CAAR FREE) (CDAR FREE))))
	       (T
		(FUNCALL STREAM ':TYO #\CR))))
	((TYPEP STREAM 'INTERVAL-STREAM)
	 (LET ((STRING (CREATE-LINE 'ART-STRING 128. NIL)))
	   (DEFAULT-LIST-ONE-FILE FILE STRING)
	   (FUNCALL STREAM ':LINE-OUT STRING)))
	((OR (NULL STREAM) (STRINGP STREAM))
	 (LET ((STRING
		 (OR STREAM (MAKE-ARRAY 128. ':TYPE 'ART-STRING ':LEADER-LENGTH 1))))
	   (SETF (FILL-POINTER STRING) 0)
	   (ARRAY-INITIALIZE STRING #\SP 0 (ARRAY-LENGTH STRING))
	   (ARRAY-PUSH STRING (IF (GET FILE ':DELETED) #/D #\SP))
	   (ARRAY-PUSH STRING #\SP)
	   (STRING-NCONC STRING (OR (GET FILE ':PHYSICAL-VOLUME) ""))
	   (SETF (FILL-POINTER STRING)
		 (1+ (MAX 5 (FILL-POINTER STRING))))
	   (STRING-NCONC STRING (FUNCALL PATHNAME ':STRING-FOR-DIRED))
	   (ARRAY-PUSH STRING #\SP)
	   (SETF (FILL-POINTER STRING)
		 (MAX 20. (FILL-POINTER STRING)))
	   (LET ((LINK-TO (GET FILE ':LINK-TO)))
	     (IF LINK-TO
		 (PROGN (STRING-NCONC STRING "=> " LINK-TO " ")
			(SETF (FILL-POINTER STRING)
			      (MAX 40. (FILL-POINTER STRING))))
	       (LET ((LENGTH (GET FILE ':LENGTH-IN-BLOCKS)))
		 (SETF (FILL-POINTER STRING)
		       (MAX 23. (FILL-POINTER STRING)))
		 (COND ((NULL LENGTH)
			(STRING-NCONC STRING "     "))
		       ((> LENGTH 999.)
			(SETF (FILL-POINTER STRING)
			      (NUMBER-INTO-ARRAY STRING LENGTH 10.
						 (FILL-POINTER STRING) 4))
			(ARRAY-PUSH STRING #\SP))
		       (T
			(SETF (FILL-POINTER STRING)
			      (MAX 24. (FILL-POINTER STRING)))
			(SETF (FILL-POINTER STRING)
			      (NUMBER-INTO-ARRAY STRING LENGTH 10.
						 (FILL-POINTER STRING) 3))
			(ARRAY-PUSH STRING #\SP))))
	       (LET ((LENGTH (GET FILE ':LENGTH-IN-BYTES)))
		 (IF (GET FILE ':DIRECTORY)
		     (STRING-NCONC STRING "  DIRECTORY")
		   (WHEN LENGTH
		     (SETF (FILL-POINTER STRING)
			   (NUMBER-INTO-ARRAY STRING LENGTH 10.
					      (FILL-POINTER STRING) 6))
		     (ARRAY-PUSH STRING #/()
		     (SETF (FILL-POINTER STRING)
			   (NUMBER-INTO-ARRAY STRING (GET FILE ':BYTE-SIZE) 10.
					      (FILL-POINTER STRING)))
		     (ARRAY-PUSH STRING #/)))))
	       (SETF (FILL-POINTER STRING)
		     (MAX 39. (FILL-POINTER STRING)))
	       (ARRAY-PUSH STRING
			   (COND ((GET FILE ':OFFLINE) #/O)
				 ((GET FILE ':NOT-BACKED-UP) #/!)
				 (T #\SP)))))
	   (ARRAY-PUSH STRING (IF (GET FILE ':DONT-DELETE) #/@ #\SP))
	   (ARRAY-PUSH STRING (IF (GET FILE ':DONT-SUPERSEDE) #/# #\SP))
	   (ARRAY-PUSH STRING (IF (GET FILE ':DONT-REAP) #/$ #\SP))
	   (TIME-INTO-ARRAY STRING (GET FILE ':CREATION-DATE))
	   (LET* ((DATE-LAST-EXPUNGE (GET FILE ':DATE-LAST-EXPUNGE))
		  (REFERENCE-DATE (OR DATE-LAST-EXPUNGE (GET FILE ':REFERENCE-DATE))))
	     (WHEN REFERENCE-DATE
	       (STRING-NCONC STRING (IF DATE-LAST-EXPUNGE " X=" " ("))
	       (TIME-INTO-ARRAY STRING REFERENCE-DATE NIL)
	       (OR DATE-LAST-EXPUNGE (STRING-NCONC STRING ")"))))
	   (LET ((AUTHOR (GET FILE ':AUTHOR)))
	     (WHEN (AND AUTHOR (NOT (EQUAL AUTHOR (FUNCALL PATHNAME ':DIRECTORY))))
	       (SETF (FILL-POINTER STRING)
		     (MAX 74. (FILL-POINTER STRING)))
	       (STRING-NCONC STRING AUTHOR)))
	   (LET ((READER (GET FILE ':READER)))
	     (WHEN (AND READER (NOT (EQUAL READER (FUNCALL PATHNAME ':DIRECTORY))))
	       (SETF (FILL-POINTER STRING)
		     (MAX 84. (FILL-POINTER STRING)))
	       (STRING-NCONC STRING READER)))
	   STRING))
	(T (FORMAT STREAM "~C ~3A "
		   (IF (GET FILE ':DELETED) #/D #\SP)
		   (OR (GET FILE ':PHYSICAL-VOLUME) ""))
	   (IF (FUNCALL STREAM ':OPERATION-HANDLED-P ':ITEM)
	       (FUNCALL STREAM ':ITEM 'FILE PATHNAME "~A"
			(FUNCALL PATHNAME ':STRING-FOR-DIRED))
	       (FUNCALL STREAM ':STRING-OUT (FUNCALL PATHNAME ':STRING-FOR-DIRED)))
	   (FORMAT STREAM "~20T")
	   (LET ((LINK-TO (GET FILE ':LINK-TO)))
	     (IF LINK-TO
		 (FORMAT STREAM "=> ~A ~40T" LINK-TO)
	       (LET ((LENGTH (GET FILE ':LENGTH-IN-BLOCKS)))
		 (LET ((STANDARD-OUTPUT STREAM))
		   (FORMAT:TAB 23.))
		 (COND ((NULL LENGTH)
			(LET ((STANDARD-OUTPUT STREAM))
			  (FORMAT:TAB 28.)))
		       ((> LENGTH 999.)
			(FORMAT STREAM "~4D " LENGTH))
		       (T
			(LET ((STANDARD-OUTPUT STREAM))
			  (FORMAT:TAB 24.))
			(FORMAT STREAM "~3D " LENGTH))))
	       (LET ((LENGTH (GET FILE ':LENGTH-IN-BYTES)))
		 (IF (GET FILE ':DIRECTORY)
		     (PRINC "  DIRECTORY" STREAM)
		   (AND LENGTH
			(FORMAT STREAM "~6D(~D)" LENGTH (GET FILE ':BYTE-SIZE)))))
	       (FORMAT STREAM "~39T")
	       (FUNCALL STREAM ':TYO
			(COND ((GET FILE ':OFFLINE) #/O)
			      ((GET FILE ':NOT-BACKED-UP) #/!)
			      (T #\SP)))))
	   (FUNCALL STREAM ':TYO (IF (GET FILE ':DONT-DELETE) #/@ #\SP))
	   (FUNCALL STREAM ':TYO (IF (GET FILE ':DONT-SUPERSEDE) #/# #\SP))
	   (FUNCALL STREAM ':TYO (IF (GET FILE ':DONT-REAP) #/$ #\SP))
	   (LET ((CREATION-DATE (GET FILE ':CREATION-DATE)))
	     (IF CREATION-DATE
		 (MULTIPLE-VALUE-BIND (SECONDS MINUTES HOURS DAY MONTH YEAR)
		     (TIME:DECODE-UNIVERSAL-TIME CREATION-DATE)
		   (FORMAT STREAM "~2,'0D//~2,'0D//~2,'0D ~2,'0D:~2,'0D:~2,'0D"
			   MONTH DAY (MOD YEAR 100.) HOURS MINUTES SECONDS))
		 (FORMAT STREAM "~17X")))
	   (LET* ((DATE-LAST-EXPUNGE (GET FILE ':DATE-LAST-EXPUNGE))
		  (REFERENCE-DATE (OR DATE-LAST-EXPUNGE (GET FILE ':REFERENCE-DATE))))
	     (AND REFERENCE-DATE
		  (MULTIPLE-VALUE-BIND (NIL NIL NIL DAY MONTH YEAR)
		      (TIME:DECODE-UNIVERSAL-TIME REFERENCE-DATE)
		    (PRINC (IF DATE-LAST-EXPUNGE " X=" " (")
			   STREAM)
		    (FORMAT STREAM "~2,'0D//~2,'0D//~2,'0D" MONTH DAY (MOD YEAR 100.))
		    (OR DATE-LAST-EXPUNGE (PRINC ")" STREAM))))) 
	   (LET ((AUTHOR (GET FILE ':AUTHOR)))
	     (AND AUTHOR (NOT (EQUAL AUTHOR (FUNCALL PATHNAME ':DIRECTORY)))
		  (FORMAT STREAM "~74T~A" AUTHOR)))
	   (LET ((READER (GET FILE ':READER)))
	     (AND READER (NOT (EQUAL READER (FUNCALL PATHNAME ':DIRECTORY)))
		  (FORMAT STREAM "~84T~A" READER)))
	   (FUNCALL STREAM ':TYO #\CR))))
******************************
(DEFCOM COM-EXPUNGE-DIRECTORY "Expunge deleted files from a directory" ()
  (LET* ((DIRECTORY (READ-DIRECTORY-NAME "Expunge directory" (DEFAULT-PATHNAME)))
	 (RESULT (FS:EXPUNGE-DIRECTORY DIRECTORY ':ERROR ':RETRY)))
    (IF (ERRORP RESULT) (BARF "Cannot expunge ~A: ~A" DIRECTORY RESULT)
	(FORMAT QUERY-IO "~&~A: ~D block~:P freed" DIRECTORY RESULT)))
  DIS-NONE)

******************************
(DEFCOM COM-OCCUR "Display text lines that contain a given string.
With an argument, show the next n lines containing the string.  If
no argument is given, all lines are shown." ()
  (COM-LIST-MATCHING-LINES))

(DEFCOM COM-LIST-MATCHING-LINES "Display text lines that contain a given string.
With an argument, show the next n lines containing the string.  If
no argument is given, all lines are shown." ()
  (LET ((CNT (IF *NUMERIC-ARG-P* *NUMERIC-ARG* 7777777))
	KEY FUNCTION REVERSE-P BJ-P)
    (MULTIPLE-VALUE (FUNCTION KEY REVERSE-P BJ-P)
      (GET-EXTENDED-STRING-SEARCH-STRINGS NIL "Show lines containing:"
					  *STRING-SEARCH-SINGLE-LINE-COMTAB*))
    (DO ((BP (COND ((NOT BJ-P) (POINT))
		   ((NOT REVERSE-P) (INTERVAL-FIRST-BP *INTERVAL*))
		   (T (INTERVAL-LAST-BP *INTERVAL*))))
	 (I 0 (1+ I)))
	(( I CNT) NIL)
      (OR (SETQ BP (FUNCALL FUNCTION BP KEY REVERSE-P)) (RETURN NIL))
      (LET ((LINE (BP-LINE BP))
	    (INDEX (BP-INDEX BP)))
	(FUNCALL STANDARD-OUTPUT ':ITEM 'BP (CREATE-BP LINE INDEX)
		 "~A" (STRING-REMOVE-FONTS LINE)))
      (FUNCALL STANDARD-OUTPUT ':TYO #\CR)
      (OR (SETQ BP (BEG-LINE BP 1)) (RETURN NIL)))
    (FUNCALL STANDARD-OUTPUT ':LINE-OUT "Done."))
  DIS-NONE)

******************************
;;;==============================================================================

