;;;  -*- Mode:Zetalisp; Package:ZWEI; Base:8; Patch-file:T -*-
;;; 
;;;  First cut at Auto Save Mode, a minor mode that will save one or more of
;;;  your buffers every so many characters or after so much idle time. 
;;; 
;;;  Auto Save Interval is the number of characters, Auto Save Time Interval
;;;  is the number of idle minutes.  Auto Save All Buffers controls whether
;;;  all buffers in Auto Save Mode are saved, or just the current buffer. 
;;; 
;;;  Characters are only counted while in a buffer in Auto Save Mode;  this
;;;  may be a deficiency, but I haven't figured out a good way to overcome
;;;  it.  The equivalent to the Emacs variable Auto Save Default is to push
;;;  the symbol AUTO-SAVE-MODE onto the variable Initial Minor Modes -- doing
;;;  so will cause Auto Save Mode to be turned on in every new buffer.
;;;  Buffers are saved into their associated files -- there is no equivalent
;;;  to the Auto Save Filenames or Auto Save Visited File variables of Emacs.
;;;  There is also no equivalent to the Emacs variable Auto Save Max, which
;;;  controls the number of saved versions -- this Auto Save Mode just saves
;;;  the buffers and doesn't worry about how many versions there are.

(defvar *auto-save-character-count* 0)
(defvar *auto-save-timeout-process* nil)

(defvariable *auto-save-interval* 500. :fixnum
  "The count of characters between auto-saves when Auto Save Mode is on.
Auto-saves will also happen every Auto Save Time Interval minutes of idle time.")

(defvariable *auto-save-time-interval* 10. :fixnum
  "If the keyboard has been idle more than this many minutes, an auto-save will happen.
Auto-saves will also happen every Auto Save Interval characters when not idle.")

(defvariable *auto-save-all-buffers* nil :boolean
  "If true, auto-save all buffers that have Auto Save Mode set.  If nil (the default),
auto-save only the current buffer.")

(defminor com-auto-save-mode auto-save-mode "Save" 8.
	  "Minor mode that causes the current buffer to be saved at intervals."
	  ()
  (command-hook 'auto-save-hook *command-hook*)
  (progn (unless *auto-save-timeout-process*
	   (setq *auto-save-timeout-process*
		 (process-run-restartable-function '(:name "Auto Save Timeout Process" :priority -5)
						   'auto-save-timeout-process))))
  )

(defprop auto-save-hook 10 command-hook-priority)
(defun auto-save-hook (ignore)
   (incf *auto-save-character-count*)
   (when (> *auto-save-character-count* *auto-save-interval*)
     (auto-save-buffers)
     (setq *auto-save-character-count* 0)))

(defun auto-save-buffers ()
   (cond (*auto-save-all-buffers*
	  (auto-save-all-buffers))
	 ((buffer-needs-saving-p *interval*)
	  (format query-io "~&(Auto Save)")
	  (let ((*check-unbalanced-parentheses-when-saving* nil)
		(*discard-undo-info-on-saving* nil))
	    (save-buffer *interval*)))
	 (:else
	  nil)))

(defun auto-save-all-buffers ()
   (loop with buffers-to-be-saved
	 for buffer in *zmacs-buffer-list*
	 when (and (buffer-needs-saving-p buffer)
		   (assq 'auto-save-mode (send buffer :saved-mode-list)))
	 do (push buffer buffers-to-be-saved)
	 finally (when (or (not (null buffers-to-be-saved))
			   (word-abbrevs-need-saving-p t))
		   (format query-io "~&(Auto Saving)")
		   (loop for buffer in buffers-to-be-saved
			 doing (let ((*check-unbalanced-parentheses-when-saving* nil)
				     (*discard-undo-info-on-saving* nil))
				 (save-buffer buffer)))
		   (when (word-abbrevs-need-saving-p t)
		     (com-write-word-abbrev-file-internal))
		   (format query-io "~&(Auto Save Complete)"))))


(defun auto-save-timeout-process ()
   ;;  Sleep half the interval, then wake up and see if the keyboard's been
   ;;  idle for the full interval.
   (loop doing (progn (process-sleep (* *auto-save-time-interval* 30. 60.))
		      (when (> (time-difference (time) tv:kbd-last-activity-time)
			       (* *auto-save-time-interval* 60. 60.))
			(let ((w (if (typep tv:selected-window 'zmacs-window-pane)
				     tv:selected-window
				     (first *all-zmacs-windows*))))
			  (send w :force-kbd-input '(:execute auto-save-all-buffers)))))))

(set-comtab *standard-comtab* nil (make-command-alist '(com-auto-save-mode)))


;(defvar *saved-zmacs-buffer-list* nil)
;(defvar *auto-save-active* nil)			; Auto-save is on in some buffer.

;;;;
;;;;  From sys:zwei;comtab:
;;;;
;;;;  Increment the *auto-save-character-count* for every character if auto-save
;;;;  is on.  Do it here instead of with a command-hook because command-hooks are
;;;;  mode-specific and we want it to happen globally.

;(DEFUN PROCESS-COMMAND-CHAR (CH &AUX VALUE LINE INDEX)
;  "Process the character CH as an editor command.
;CH should be a keyboard character or mouse character, not a list."
;  (SETQ *LAST-COMMAND-CHAR* CH)

;   (when (and *auto-save-all-buffers*
;	      (cond ((eq *saved-zmacs-buffer-list* *zmacs-buffer-list*)
;		     *auto-save-active*)
;		    (:else
;		     (setq *saved-zmacs-buffer-list* *zmacs-buffer-list*)
;		     (setq *auto-save-active* (loop for b in *saved-zmacs-buffer-list*
;						    when (assq 'auto-save-mode (send b :saved-mode-list))
;						    return t
;						    finally (return nil))))))
;     (auto-save-hook nil))

;  ;; Look up the command in the table.
;  (LET ((*CURRENT-COMMAND* (COMMAND-LOOKUP *LAST-COMMAND-CHAR* *COMTAB*)))
;    ;; Execute the command.
;    (MULTIPLE-VALUE (VALUE LINE INDEX)
;      (COMMAND-EXECUTE *CURRENT-COMMAND* *LAST-COMMAND-CHAR* NIL *COMMAND-HOOK*))
;    ;; This command is creating the argument to a subsequent command.
;    (COND ((EQ VALUE ':ARGUMENT)
;	   VALUE)
;	  (T
;	   ;; If the mark is not being preserved, make it go away.
;	   (COND ((AND (NOT *MARK-STAYS*) (WINDOW-MARK-P *WINDOW*))
;		  (SETF (WINDOW-MARK-P *WINDOW*) NIL)
;		  (MUST-REDISPLAY *WINDOW* DIS-MARK-GOES)))
;	   ;; Report the returned value of the command to the window.
;	   (MUST-REDISPLAY *WINDOW* VALUE LINE INDEX)
;	   ;; Call the post-command hooks
;	   (DOLIST (HOOK *POST-COMMAND-HOOK*) (FUNCALL HOOK *LAST-COMMAND-CHAR*))))))
