;;; -*- Mode:Common-Lisp; Package:User; Fonts:(Cptfont Cptfontb Hl12bi); Base:10 -*-

;1;; *********************************************************************
;1;; Utility to build a zmacs file with embedded font characters allowing*
;1;; comments, documentation strings, and/or other string constants to be*
;1;; shown in different fonts.*
;1;;*
;1;; 10/15/86 Rhonda Alexander *
;1;;***********************************************************************

(DEFPARAMETER ESCAPE-CHAR #\\)

(DEFPARAMETER DEFAULT-FONT-STRING " Fonts:(Cptfont Cptfontb Hl12bi Tr10b); "
  "2Font indicators put in the mode line if no font indicators are present*")

(DEFPARAMETER DEBUG-FONTIFY NIL)

(DEFUN FONTIFY-FILE (SRC-PATHNAME &OPTIONAL DEST-PATHNAME (DO-COMMENTS T) (DO-DOC-STRINGS T) DO-OTHER-STRINGS)
  "2Insert ZMACS font indicators in SRC-PATHNAME, writing to DEST-PATHNAME (default = new version of SRC).
     DO-COMMENTS, DO-DOC-STRINGS, and DO-OTHER-STRINGS may be true or a font-number 
     to fontify lexical entities of these types*"
     (WITH-OPEN-FILE (INSTREAM SRC-PATHNAME)
        (WITH-OPEN-FILE (OUTSTREAM (OR DEST-PATHNAME SRC-PATHNAME)
				       :DIRECTION :OUTPUT :IF-EXISTS :NEW-VERSION)
	  (FONTIFY-HELPER INSTREAM OUTSTREAM DO-COMMENTS DO-DOC-STRINGS DO-OTHER-STRINGS))))


(DEFUN FONTIFY-HELPER (INSTREAM OUTSTREAM &OPTIONAL (DO-COMMENTS T) (DO-DOC-STRINGS T) DO-OTHER-STRINGS)
  
  (LET ((COMMENT-FONT (WHEN DO-COMMENTS (IF (NUMBERP DO-COMMENTS) DO-COMMENTS 1)))
	(DOC-FONT (WHEN DO-DOC-STRINGS (IF (NUMBERP DO-DOC-STRINGS) DO-DOC-STRINGS 2)))
	(STR-FONT (WHEN DO-OTHER-STRINGS (IF (NUMBERP DO-OTHER-STRINGS) DO-OTHER-STRINGS 3)))
	(NO-FONTS (NOT (OR DO-COMMENTS DO-DOC-STRINGS DO-OTHER-STRINGS)))
	SEEN-FIRST-LINE)
    
    (LOOP
      WITH LINE
      WITH CURSOR
      WITH CURSOR2
      WITH DOCSTRING
      WITH NEXT-EOL
      WITH GO-TO-NEXT-LINE
      WITH SPECIAL-CHAR
      WHILE (SETQ LINE (READ-LINE INSTREAM NIL)) DO
      
      ;1; MODE LINE*
      ;1; If ZMACS reads a file with embedded font characters and no font header, it permanently*
      1;; escapes all the epsilon characters  - insert font indicator if necessary.*
      (UNLESS SEEN-FIRST-LINE
	(SETQ SEEN-FIRST-LINE T)
	(IF (STRING-SEARCH "-*-" LINE)
	    (IF (STRING-SEARCH "Fonts" LINE)
		(WRITE-LINE LINE OUTSTREAM)
		;1; ELSE no fonts in mode line*
		(SETQ CURSOR (OR (POSITION #\; LINE :FROM-END T)
				 (POSITION #\sPACE LINE :FROM-END T)))
		(SEND OUTSTREAM :STRING-OUT LINE 0 (1+ CURSOR))
		(SEND OUTSTREAM :STRING-OUT DEFAULT-FONT-STRING)
		(SEND OUTSTREAM :STRING-OUT LINE (1+ CURSOR))
		(WRITE-LINE "" OUTSTREAM))
	    
	    ;1;; ELSE no mode line at all*
	    (SEND OUTSTREAM :STRING-OUT ";;; -*- ")
	    (SEND OUTSTREAM :STRING-OUT DEFAULT-FONT-STRING)
	    (WRITE-LINE " -*-" OUTSTREAM))
	(SETQ LINE (OR (READ-LINE INSTREAM NIL) "")))
      
      (SETQ CURSOR 0)
      (SETQ DOCSTRING 0)
      
      (IF (ZEROP (LENGTH LINE))
	  (WRITE-LINE "" OUTSTREAM)		;1GET BLANK LINES*
	  ;1; ELSE  Check for documentation strings    *
	  (WHEN (CHAR-EQUAL (AREF LINE 0) #\()
	    (MULTIPLE-VALUE-SETQ (LINE DOCSTRING)
				 (EXPAND-LINE INSTREAM LINE))))
      
      (LOOP WHILE (AND CURSOR (< CURSOR (LENGTH LINE))) DO
	    1;; may have created multi-line string*
	    (SETQ NEXT-EOL (NEXT-EOL LINE CURSOR))
	    (SETQ GO-TO-NEXT-LINE NIL)
	    
	    
	    (SETQ SPECIAL-CHAR 
		  (WHEN (SETQ CURSOR2 (STRING-SEARCH-SET '(#\" #\; #\epsilon) LINE CURSOR NEXT-EOL))
		    (AREF LINE CURSOR2)))
	    
	    (WHEN DEBUG-FONTIFY
	      (PRINT (FORMAT NIL "~A" LINE))
	      (PRINT (FORMAT NIL "CURSOR=~D CURSOR2=~D, DOCSTRING=~D NEXT-EOL=~D"
			     CURSOR CURSOR2 DOCSTRING NEXT-EOL)))
	    
	    
	    (COND 
	      
	      ;1; OLD FONT INDICATOR*
	      ((AND SPECIAL-CHAR
		    (CHAR-EQUAL SPECIAL-CHAR #\EPSILON)
		    (OR NO-FONTS
			(PARSE-INTEGER LINE :START (1+ CURSOR2) :END (+ 2 CURSOR2) :JUNK-ALLOWED T)))
	       ;1; discard the font indicator*	       
	       (OUTPUT OUTSTREAM LINE :START CURSOR :END CURSOR2)
	       (SETQ CURSOR (+ CURSOR2 2)))
	      
	      ;1; BEGINNING OF STRING*
	      ((AND SPECIAL-CHAR (CHAR-EQUAL SPECIAL-CHAR #\"))
	       (OUTPUT OUTSTREAM LINE :START CURSOR :END CURSOR2)
	       
	       (LET (STRING
		     START-FONT)
		 (MULTIPLE-VALUE-SETQ (STRING LINE CURSOR)
				      (SKIP-STRING INSTREAM LINE (1+ CURSOR2)))
		 
		 (COND
		   ((AND DOCSTRING (= CURSOR2 DOCSTRING))
		    (WHEN DO-DOC-STRINGS
		      (SETQ START-FONT DOC-FONT)))
		   (DO-OTHER-STRINGS
		    (SETQ START-FONT STR-FONT)))
		 
		 ;1; If already fontified, make sure it matches new font if specified*
		 (WHEN (AND START-FONT (CHAR-EQUAL (AREF STRING 0) #\EPSILON))
		   (SETF (AREF STRING 1) (+ #X30 START-FONT))
		   (SETF START-FONT NIL))
		 (OUTPUT OUTSTREAM STRING  :START-FONT START-FONT :END-FONT START-FONT
			 :PRE-CHAR #\" :POST-CHAR #\")))
	      
	      ;1; COMMENT*
	      ((AND SPECIAL-CHAR (CHAR-EQUAL SPECIAL-CHAR #\;))
	       (OUTPUT OUTSTREAM LINE :START CURSOR :END CURSOR2)
	       
	       ;1; Check for previous fontification (may span lines in which case we will*
	       ;1; separately fontify each line)*
	       (LET ((START-FONT COMMENT-FONT)
		     (END-FONT COMMENT-FONT)
;			      END-FONT-IX
		     (END-COMMENT-IX NEXT-EOL)
		     (EXISTING-FONT (CHAR-EQUAL (AREF LINE (1+ CURSOR2)) #\ePSILON)))
		 ;1; If already a font start, make sure it matches any new font*
		 (WHEN (AND START-FONT EXISTING-FONT)
		   (SETF (AREF LINE (+ 2 CURSOR2)) (+ #X30 START-FONT))
		   (SETF START-FONT NIL))
		 ;1; If closing epsilon exists, don't need to end the font.  Also, if*
		 ;1; there wasn't an opening epsilon, don't print the closing epsilon*
		 ;1; (this happens if this closes one started on a previous line)*
		 ;1; >> BETTER TO HAVE AN EXTRA CLOSING EPSILON*
;                           (WHEN (SETQ END-FONT-IX (POSITION #\ePSILON LINE :START (+ 2 CURSOR2) :END NEXT-EOL))
;                              (SETQ END-FONT NIL)
;			      (UNLESS (OR EXISTING-FONT START-FONT)
;				 (SETQ END-COMMENT-IX (- END-FONT-IX 2))))
		 
		 (OUTPUT OUTSTREAM LINE :START (1+ CURSOR2) :START-FONT START-FONT :END-FONT END-FONT
			 :PRE-CHAR #\; :EOL T :END END-COMMENT-IX)
		 (SETQ GO-TO-NEXT-LINE T)))
	      
	      (T
	       (OUTPUT OUTSTREAM LINE :START CURSOR :END NEXT-EOL :EOL T)
	       (SETQ GO-TO-NEXT-LINE T)))
	    
	    
	    (WHEN GO-TO-NEXT-LINE
	      (SETQ CURSOR (WHEN NEXT-EOL (1+ NEXT-EOL))))
	    
	    (WHEN (AND CURSOR (>= CURSOR (LENGTH LINE)))
	      (OUTPUT OUTSTREAM NIL :EOL T))))))
  

(DEFUN LINE-START (LINE)
   (TOKEN-START LINE))

(DEFUN TOKEN-START (LINE &OPTIONAL (START 0))
  (LOOP FOR I FROM START TO (1- (LENGTH LINE)) DO
     (UNLESS (MEMBER (AREF LINE I) '(#\SPACE #\RETURN #\TAB) :TEST 'CHAR-EQUAL)
	(RETURN I))))

(DEFUN LINE-END (LINE)
  (LOOP FOR I FROM (1- (LENGTH LINE)) DOWNTO 1 DO
     (UNLESS (CHAR-EQUAL #\SPACE (AREF LINE I))
	(RETURN I))))


(DEFUN NEXT-EOL (LINE &OPTIONAL (START 0))
  (LOOP FOR I FROM START TO (1- (LENGTH LINE)) DO
     (WHEN (CHAR-EQUAL (AREF LINE I) #\RETURN)
	(RETURN I))))
   

(DEFUN OUTPUT (STREAM STRING &KEY START-FONT END-FONT (START 0) END PRE-CHAR POST-CHAR EOL)
  (WHEN PRE-CHAR
     (SEND STREAM :TYO PRE-CHAR))
  (WHEN START-FONT
     (SEND STREAM :TYO #\ePSILON)
     (SEND STREAM :TYO (INT-CHAR (+ #X30 START-FONT))))

  (WHEN STRING
    (SEND STREAM :STRING-OUT STRING START END))

  (WHEN END-FONT
    (SEND STREAM :TYO #\ePSILON)
    (SEND STREAM :TYO #\*))

  (WHEN POST-CHAR
     (SEND STREAM :TYO POST-CHAR))

  (WHEN EOL
     (WRITE-LINE "" STREAM)))


(DEFUN SKIP-STRING (STREAM LINE START)
; (TV:NOTIFY NIL "SKIP STRING")
  (LET ((NEW-LINE LINE)
        STRING
	ESCAPE-MODE
	EOS)

     (LOOP UNTIL EOS DO
        (LOOP FOR I FROM START TO (1- (LENGTH NEW-LINE)) DO
           (COND
              (ESCAPE-MODE (SETQ ESCAPE-MODE NIL))
              ((CHAR-EQUAL (AREF NEW-LINE I) ESCAPE-CHAR)
	         (SETQ ESCAPE-MODE T))
	      ((CHAR-EQUAL (AREF NEW-LINE I) #\")
                 (SETQ STRING (STRING-APPEND (OR STRING "")
					     (IF STRING #\RETURN "")
					     (SUBSTRING NEW-LINE START I)))
	         (SETQ EOS I)
                 (RETURN)))
           FINALLY
	   (PROGN
             (SETQ STRING (STRING-APPEND (OR STRING "")
			                 (IF STRING #\RETURN "")
					 (SUBSTRING NEW-LINE START I)))
	     (SETQ NEW-LINE (READ-LINE STREAM))
	     (SETQ START 0))))

  (VALUES
     STRING					;1RESULTING STRING*
     NEW-LINE					;1CURRENT LINE*
     (1+ EOS))))				;1CURRENT CURSOR IN LINE*



(DEFUN EXPAND-LINE (STREAM LINE)
  ;1; Concatenate enough physical lines that we can use READ to locate a documentation string*

;   (TV:NOTIFY NIL "EXPAND-LINE")
   (LET ((NEW-LINE LINE)
         FRESH-LINE
         START 
         TOKEN
         DONE
	 DOC-STRING
	 )
       (LOOP UNTIL DONE DO
         (SETQ START 1)
;         (PRINT (FORMAT NIL "~A" NEW-LINE))
         (CONDITION-CASE (COND)
           (PROGN
             ;1; Skip DEFx, name, & parms or value*
             (DOTIMES (I 3)
               (MULTIPLE-VALUE-SETQ (TOKEN START)  (READ-FROM-STRING NEW-LINE T NIL :START START))
               (WHEN (ZEROP I)
                  (UNLESS (MEMBER TOKEN
				  '(DEFUN DEFMACRO DEFPARAMETER DEFCONST DEFCONSTANT DEFVAR DEFMETHOD
					  DEFSUBST DEFWHOPPER)
				  :TEST 'EQ)
                    (SETQ DONE T)
		    (RETURN))))

             (WHEN DONE (RETURN))

             ;1; get start of next token*
             (LOOP UNTIL DONE DO
                (IF (OR (NULL (SETQ START (TOKEN-START NEW-LINE START)))
			(CHAR-EQUAL (AREF NEW-LINE START) #\;))
		    ;1 Then get new line & set start to its beginning*
                    (IF (SETQ FRESH-LINE (READ-LINE STREAM NIL))
                       (PROGN
                          (SETQ START (1+ (LENGTH NEW-LINE)))
                          (SETQ NEW-LINE (STRING-APPEND NEW-LINE #\RETURN FRESH-LINE)))
                        ;1; ELSE we got an eof - dont loop*
                        (SETQ DONE T))
		    ;1; else*
                    (SELECTOR (AREF NEW-LINE START) CHAR-EQUAL
                       (#\epsilon (INCF START 2))
                       (#\" (SETQ DONE T) (SETQ DOC-STRING START))
                       (:OTHERWISE (SETQ DONE T))))))

         (SYS:PARSE-ERROR
            (SETQ DONE T))

         (SYS:READ-END-OF-FILE
             (IF (SETQ FRESH-LINE (READ-LINE STREAM NIL))
                (SETQ NEW-LINE (STRING-APPEND NEW-LINE #\RETURN FRESH-LINE))
                ;1; ELSE we got an eof - dont loop*
               (SETQ DONE T)))))

       (VALUES
          NEW-LINE
          DOC-STRING)))
