;;;; -*- Mode:Zetalisp; Package:user; Base:10; Fonts: MEDFNT,HL12B -*-

;1;;================================================================*
;1;; HISTORY:*
;1;;   Original - GJ*
;1;;   1/30/86 - RLA : Added daily and weekly alarms, check-alarms, *
;1;;                     save to appointments file, restart code*
;1;;   4/8/86 - PJW : User interface changed. Appointments file purged of stale*
;1;;                     appointments, updated with each set- .*
;1;; 7/18/86 - DNG : Fix so user doesn't have to be logged in at
;;;		midnight, and don't deliver message when a different
;;;		user is logged in.*
;1;;================================================================*
;1;;*
;1;; 1. To use, add this form to your login-init file:
;;;       (LOAD "<xfasl version of this file>")
;;;    This takes care of all process instantiation and initialization.
;;; 
;;; 2. Relevant functions [Pkg USER]:
;;;       a. (SET-ALARM) 
;;;             Sets and saves an alarm with associated reason.
;;;       b. (SET-WEEKLY-APPT)
;;;             Sets and saves a weekly alarm with associated reason. 
;;;       c. (CHECK-ALARMS)
;;;            Displays currently set alarms.  The currently set alarms include the REMAINING
;;;            alarms for today as well as NON-DAILY alarms for subsequent days.  There will
;;;            also be a **SPECIAL** entry for midnight, which sets alarms for the next day.
;;; 
;;; 3. Notes
;;;    a.  Turn the alarm off by hitting the TERM key (otherwise it beeps for 30 seconds).  It
;;;        then does a notify of the alarm reason.   (Hit RUBOUT after TERM to clear from
;;;        Terminal- state).
;;;    b.  If you don't like the default pathname "lm:<user-id>.alarm;appointments.lisp" (the directory will
;;;        be auto-created), you may precede the load with: 
;;;            (SETQ USER:APPOINTMENTS-FILE <pathname>)
;;;        This file can be edited, although it doesn't hurt to have old appointments in the file.
;;;        (the file just contains forms).
;;;    c.  Other variables of interest [Pkg USER]:
;;;          ALARM-TIME           - number of seconds alarm beeps (30) 
;;;          ALARM-EXPUNGE        - t to keep only 2 versions of appointments-file
;;;          ALARM-AUTO-DISPLAY   - t to call check-alarms after init and appointment file changes*

(DEFMACRO THEN (&BODY BODY)
  `(PROGN ,@BODY))

(DEFMACRO ELSE (&BODY BODY)
  `(PROGN ,@BODY))

1;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; The WAKEUP-LIST is made up of two entry lists.  The first entry is the universal time
;;; the alarm should go off and the second entry is the reason the person wants to be 
;;; notified.
;;;*
(DEFPARAMETER WAKEUP-LIST NIL "2List of times and reasons*")

1;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; ALARM-TIME is the number of seconds the alarm should ring before turing itself off.
;;;*
(DEFPARAMETER ALARM-TIME 30 "2Number of seconds the alarm should ring*")


1;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; ALARM-CLOCK waits for the wakeup list to get something on it "Alarm not set" and
;;; then waits for the time of the first element on the list to happen "Ticking".  When the
;;; specified time comes, the screen will blink and the beeper beep once every half second
;;; until the TERM key is struck or ALARM-TIME number of seconds pass.  If the TERM key
;;; is struck, a notify window pops up displaying the reason the user gave for wanting to
;;; have the alarm go off.
;;; *
(DEFUN ALARM-CLOCK (
		    &AUX msg)
  
  (DO-FOREVER
    ()
    
    (PROCESS-WAIT "Alarm not set"
		  #'(LAMBDA ()
		      USER:WAKEUP-LIST))
    
    (PROCESS-WAIT "Ticking"              
		  #'(LAMBDA ()  
                      (AND user:wakeup-list	   ;1this can happen during re-init*
		           (> (TIME:GET-UNIVERSAL-TIME) (CAAR USER:WAKEUP-LIST)))))
    
    (SETQ msg (SECOND (FIRST wakeup-list)))
    (IF (STRING-EQUAL msg "**SPECIAL**")
	(alarm-re-init t)
      ;1; else*
      (WHEN (EQUAL USER-ID '#,USER-ID)
	(DO ((ALARM-COUNT 0 (1+ ALARM-COUNT)))
	    ((OR TV:KBD-TERMINAL-HAPPENED 
		 (= ALARM-COUNT (* 2 ALARM-TIME)))
	     (TV:pop-up-format "~% ALARM  ~a~%~% ~A~%~%"
			       (TIME:print-universal-time (CAAR wakeup-list) nil) msg))
          (BEEP)
          (PROCESS-SLEEP 30)) )
      
      (SETF WAKEUP-LIST (CDR WAKEUP-LIST)))))

1;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; ALARM-CLOCK-PROCESS starts up the alarm clock process.
;;;*
(DEFVAR alarm-clock-process-id nil)	  

(DEFUN ALARM-CLOCK-PROCESS ()
   (UNLESS alarm-clock-process-id 
     (SETQ alarm-clock-process-id (PROCESS-RUN-FUNCTION '(:Name "Alarm clock"
							  :restart-after-boot t
							  :restart-after-reset T)
							'ALARM-CLOCK))))

1;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; ARM-ALARM accepts a text describing why the alarm should go off and the time it should
;;; go off.
;;;*
(DEFUN ARM-ALARM (REASON-FOR-ALARM UNIVERSAL-TIME)
  (IF (STRINGP universal-time) (SETF universal-time (TIME:parse-universal-time universal-time)))
  (IF (> UNIVERSAL-TIME (TIME:GET-UNIVERSAL-TIME))
      (THEN
	(IF (NOT WAKEUP-LIST)
	    (THEN
	      (SETF WAKEUP-LIST (LIST (LIST UNIVERSAL-TIME REASON-FOR-ALARM))))
	    (ELSE
	      (DO* ((INDEX 0 (1+ INDEX))
		    (ITEM (NTH INDEX WAKEUP-LIST) (NTH INDEX WAKEUP-LIST)))
		   ((OR (NOT ITEM)
			(< UNIVERSAL-TIME (CAR ITEM)))
		    (SETF WAKEUP-LIST
			  (APPEND (FIRSTN INDEX WAKEUP-LIST)
				  (LIST (LIST UNIVERSAL-TIME REASON-FOR-ALARM))
				  (NLEFT (- (LENGTH WAKEUP-LIST) INDEX) WAKEUP-LIST))))))))))
  

(DEFCONST days '("Mon" "Tue" "Wed" "Thu" "Fri" "Sat" "Sun"))

(DEFUN check-alarms (&aux format-list)
  (tv:pop-up-format "~%     CURRENT ALARMS~
                     ~%~
                     ~{~%~a~}~
                     ~%"
		    (DOLIST (event wakeup-list (NREVERSE format-list))
		      (IF (NOT (STRING-EQUAL (SECOND event) "**SPECIAL**"))
			  (SETF format-list
				(CONS 
				  (MULTIPLE-VALUE-BIND (sec min hour day mon year day-of-week)
				      (TIME:decode-universal-time (FIRST event))
				    (FORMAT nil "~A  ~2,48D-~2,48D-~2,48D  ~2,48D:~2,48D:~2,48D  ~A~%"
					    (NTH day-of-week days) mon day year hour min sec
					    (SECOND event)))
				  format-list)))))
  t)

(DEFUN arm-weekly-appt (reason weekdays hour &optional (minute 0)
                        &aux now current-weekday current-day)
  (PROCLAIM '(SPECIAL days))
  (SETQ now (MULTIPLE-VALUE-LIST (TIME:decode-universal-time (TIME:get-universal-time))))
  (SETQ current-weekday (SEVENTH now))
  (SETQ current-day (FOURTH now))
  (DOLIST (weekday weekdays t)
    (IF (EQUAL current-weekday (POSITION weekday days :test #'STRING-EQUAL))
	(arm-alarm reason 
		   (TIME:encode-universal-time 0 minute hour current-day (FIFTH now) (SIXTH now))))))

(DEFVAR appointments-file nil)
(DEFVAR alarm-expunge t)
(DEFVAR alarm-auto-display t)

(DEFUN alarm-cleanup (&optional (file appointments-file) (keep 2) &aux path current del result)
  (WHEN (AND alarm-expunge (SETQ path (PROBE-FILE file)))
    (SETQ current (SEND path :version))
    (LOOP
        while (> current keep) do
          (UNLESS (ERRORP (SEND (SETQ del (SEND path :new-version (- current keep))) :delete-and-expunge nil))
             (PUSH del result))
          (DECF current)))
  result)

(DEFVAR reason-for-alarm)
(DEFVAR universal-time)

(DEFUN Set-alarm ()
  (PROCLAIM '(SPECIAL reason-for-alarm universal-time))
  (SETF reason-for-alarm "                              ")
  (SETF universal-time (TIME:GET-UNIVERSAL-TIME))
  (tv:choose-variable-values
    '((reason-for-alarm "Alarm reason" :string)
      (universal-time "Alarm time"
		      :documentation "L: input new value from keyboard, R: edit this value" :date))
    :label "Set Alarm"
    :margin-choices '("Do it ()" ("Abort ()" (SIGNAL-CONDITION eh:abort-object))))
  (arm-alarm reason-for-alarm universal-time)
  (WITH-ZETALISP-ON 
    (WITH-OPEN-FILE
      (STREAM appointments-file :direction :output :if-exists :append :if-does-not-exist :create)
      (FORMAT stream "(arm-alarm ~S ~s)~%" reason-for-alarm (TIME:print-universal-time universal-time nil))))
  (alarm-cleanup) 
  (WHEN alarm-auto-display (check-alarms)))

(DEFVAR weekdays)
(DEFVAR hour)
(DEFVAR minute)

(DEFUN set-weekly-appt (&aux now (Mon "Mon") (Tue "Tue") (Wed "Wed")
			(Thu "Thu") (Fri "Fri") (Sat "Sat") (Sun "Sun"))
  (PROCLAIM '(SPECIAL reason-for-alarm weekdays hour minute))
  (SETF reason-for-alarm "                              ")
  (SETF now (MULTIPLE-VALUE-LIST (TIME:decode-universal-time (TIME:get-universal-time))))
  (SETF hour (THIRD now))
  (SETF minute (SECOND now))
  (SETF weekdays (LIST Mon Tue Wed Thu Fri))
  (tv:choose-variable-values
    `((reason-for-alarm "Alarm reason" :string)
      (weekdays "Weekdays" :set ,(LIST Mon Tue Wed Thu Fri Sat Sun))
      (hour "Alarm hour")
      (minute "Alarm minute"))
    :label "Set Weekly Appointment"
    :margin-choices '("Do it ()" ("Abort ()" (SIGNAL-CONDITION eh:abort-object))))
  (arm-weekly-appt reason-for-alarm weekdays hour minute)
  (WITH-ZETALISP-ON 
    (WITH-OPEN-FILE (STREAM appointments-file :direction :output :if-exists :append)
      (FORMAT stream "(arm-weekly-appt ~S '~S ~A ~A)~%" reason-for-alarm weekdays hour minute)))
  (alarm-cleanup)
  (WHEN alarm-auto-display (check-alarms)))


(DEFUN alarm-init (&aux dir)
   (alarm-clock-process)
   (UNLESS appointments-file
      (fs:create-directory (SETQ dir (FORMAT nil "lm:~A.alarm;" user-id)))
      (SETQ appointments-file  (STRING-APPEND dir "appointments.lisp")))
   (alarm-re-init))

(DEFUN alarm-re-init (&optional wait &aux now)
  (WHEN wait
    (PROCESS-SLEEP 60 "Alarm Re-init"))
  (SETQ wakeup-list nil)
 (LET (( user-id '#,user-id)) ; so user doesn't have to be logged in at midnight -- D.N.G. 6/24/86
  (WITH-ZETALISP-ON 
    (WITH-OPEN-FILE (in-stream appointments-file :direction :input :if-does-not-exist :create)
      (WITH-OPEN-FILE (out-stream appointments-file :direction :output :if-exists :new-version)
	(CONDITION-CASE ()
	    (DO ((object (READ in-stream) (READ in-stream)))
		(nil)
	      (COND ((EVAL object) (WRITE object :stream out-stream ) (TERPRI out-stream))))
	  (sys:end-of-file)))))
  (alarm-cleanup)
  (SETQ now (MULTIPLE-VALUE-LIST (TIME:decode-universal-time (TIME:get-universal-time))))
  (arm-alarm "**SPECIAL**" (TIME:encode-universal-time 59 59 23 (FOURTH now) (FIFTH now) (SIXTH now))))
 )


;1; Go ahead and set things up on load*
(IF alarm-clock-process-id (SEND alarm-clock-process-id :kill))
(SETQ alarm-clock-process-id nil)
(alarm-init)

