;;; -*- Mode: LISP; Package: USER; Base: 10; Fonts: MEDFNT, HL12B, HL12BI; -*-

;1; Functions to get information about flavors*
;1; LaMott Oren 1983*
;1;*

(GLOBALIZE 'APROPOS-FLAVOR)
(DEFUN APROPOS-FLAVOR (SUBSTRING &rest args)
  "2Find all flavors whose names contain a substring.
If PREDICATE is non-NIL, it is a function to be called with a symbol as arg; only
symbols for which the predicate returns non-NIL will be mentioned.  The :PACKAGE
argument defaults to the Global package.  The symbols are printed unless DONT-PRINT
is set.  A list of the symbols found is returned.*"
  (LET ((flavors (LEXPR-FUNCALL #'SUB-APROPOS SUBSTRING *all-flavor-names*
				:dont-print t args)))
    (WHEN (NOT (GET (MAKE-PLIST args) 'dont-print))
      (DOLIST (f flavors)
	(TERPRI)
	(PRINC (flavor-and-short-documentation-string f))))
    flavors))


(GLOBALIZE 'APROPOS-METHOD)
(DEFUN APROPOS-METHOD (SUBSTRING flavor-instance &rest args)
  "2Find all methods of a flavor whose names contain a substring.
If PREDICATE is non-NIL, it is a function to be called with a symbol as arg; only
symbols for which the predicate returns non-NIL will be mentioned.  The symbols are
printed unless DONT-PRINT is set.  A list of the symbols found is returned.*"
  (LET ((methods (LEXPR-FUNCALL #'SUB-APROPOS SUBSTRING
				(SEND flavor-instance ':which-operations)
				:dont-print t args)))
    (WHEN (NOT (GET (MAKE-PLIST args) ':dont-print))
      (DOLIST (m methods)
	(LET* ((FUNCTION (GET-HANDLER-FOR flavor-instance m))
	       (FUNCTION-NAME (FUNCTION-NAME function)))
	  (TERPRI)
	  (PRINC (function-and-short-documentation-string
		   FUNCTION (IF (listp function-name)
				(CDR function-name)
			      function-name))))))
    methods))

;1-----------------------------------------------------------------------------*



(GLOBALIZE 'SHOW-FLAVOR)
(DEFUNP SHOW-FLAVOR (name &optional (indent 0) (previous nil) &aux flavor)
  "2This function displays the hierarchy of the flavors used by NAME.
NAME is the flavor; It may be the name of a flavor, an instance of a flavor or the actual
flavor structure. INDENT and PREVIOUS are used internally for recursive calls.*"
  (WHEN (NULL (SETQ flavor (FLAVORP name)))
	  (FORMAT t "~%~S Is not a flavor or the name of one" name)
	  (RETURN))
  (TERPRI)
  (PRINC (flavor-and-short-documentation-string (si:flavor-name flavor) indent))
  (LET* ((children (si:flavor-depends-on flavor))
	 (prev (APPEND previous children)))
    (DOLIST (f children)
      (IF (OR (NULL previous) (NOT (MEMQ f previous)))
	  (SHOW-FLAVOR f (+ indent 2) prev)))))


(GLOBALIZE 'SHOW-FLAVOR-USERS)
(DEFUNP SHOW-FLAVOR-USERS (name &optional (indent 0) (previous nil) &aux flavor)
  "2This function displays the hierarchy of the flavors that use NAME.
NAME is the flavor; It may be the name of a flavor, an instance of a flavor or the actual
flavor structure. INDENT and PREVIOUS are used internally for recursive calls.*"
  (WHEN (NULL (SETQ flavor (FLAVORP name)))
    (FORMAT t "~%~S Is not a flavor or the name of one" name)
    (RETURN))
  (TERPRI)
  (PRINC (flavor-and-short-documentation-string (si:flavor-name flavor) indent))
  (LET* ((children (si:flavor-depended-on-by flavor))
	 (prev (APPEND previous children)))
    (DOLIST (f children)
      (IF (OR (NULL previous) (NOT (MEMQ f previous)))
	  (SHOW-FLAVOR-USERS f (+ indent 2) prev)))))


(GLOBALIZE 'SHOW-FLAVORS-THAT-USE)
(DEFUNP SHOW-FLAVORS-THAT-USE (flavor-or-instance &optional directly-p &aux flavor-name)
  "2List all the flavors that include the specified flavor.
This doesn't find uninstantiated flavors whose decendents (other than children) include
the specified flavor.*"
  (COND ((EQ (DATA-TYPE flavor-or-instance) 'dtp-instance)
	 (SETQ flavor-name (TYPEP flavor-or-instance)))
	((AND (SYMBOLP flavor-or-instance) (GET flavor-or-instance 'si:flavor)) 
	 (SETQ flavor-name flavor-or-instance))
	(t (FORMAT nil "~%~S is not a flavor or the name of one"
		   flavor-or-instance)
	   (RETURN)))
  (DOLIST (flavor *all-flavor-names*)
    (LET* ((f               (FLAVORP flavor))
	   (included-flavor (OR (AND (NOT directly-p) (MEMQ flavor-name (si:flavor-depends-on-all f)))
				(MEMQ flavor-name (si:flavor-depends-on f))
				(MEMQ flavor-name (si:flavor-includes f)))))
      (WHEN included-flavor
	(TERPRI)
	(PRINC (flavor-and-short-documentation-string flavor))))))


(GLOBALIZE 'SHOW-INCLUDED-FLAVORS)
(DEFUNP SHOW-INCLUDED-FLAVORS (instance  &optional (dont-use '(tv:superior))
					 (indent 2) (previous nil)
					 &aux flavor instance-variable name-list)
  "2Recursively print the instance variable name and flavor name of all the instance
variables whose DATA-TYPE is DTP-INSTANCE or whose DATA-TYPE is DTP-LIST and the
list elements are DTP-INSTANCE.  This is an easy way to print the hierarchy of window
panes and blinkers associated with a window. A special hack is included to ignore the
instance variable named TV:SUPERIOR. If this was included the program whould print all
the windows on the lisp machine.*"
  ;1;*
  ;1; Find the flavor or report error.*
  ;1;*
  (SETQ flavor (COND ((EQ (DATA-TYPE instance) 'dtp-instance)
		      (FLAVORP instance))
		     (t (FORMAT t "~%~S Is not a flavor" instance)
			(RETURN))))
  ;1;*
  ;1; Find all instance variables that are flavors.*
  ;1;*
  (DOLIST (symbol (si:flavor-all-instance-variables flavor))
    (SETQ instance-variable (GET-INSTANCE-VARIABLE instance symbol))
    (DOLIST (variable (IF (LISTP instance-variable) instance-variable
			(LIST instance-variable)))
      (COND ((NEQ (DATA-TYPE variable) 'dtp-instance) (RETURN))
	    ((MEMQ symbol dont-use) (RETURN))
	    ((MEMQ variable previous)
	     (NCONC (CDR (ASSQ variable name-list)) (LIST symbol)))
	    (t
	     (SETQ previous (APPEND previous (LIST variable)))
	     (SETQ name-list (NCONC name-list (LIST (LIST variable symbol))))))))
  ;1;*
  ;1; Print flavor list & do recursive call.*
  ;1;*
  (DOLIST (symbol name-list)
    (FORMAT t "~%~VX~A  ~s" indent (CDR symbol)
	    (si:flavor-name (GET (TYPEP (FIRST symbol)) 'si:flavor)))
    (SHOW-INCLUDED-FLAVORS (FIRST symbol) dont-use (+ indent 2) previous))
  instance)


(GLOBALIZE 'SHOW-METHODS)
(DEFUN SHOW-METHODS (name &optional (full-doc nil) (stream standard-output)
				    (remove-fonts? nil))
  "2List all the methods of a flavor along with arglist & short documentation.*"
  (PROG (flavor methods)
	(WHEN (NULL (SETQ flavor (FLAVORP name)))
	  (FORMAT t "~%~S Is not a flavor or the name of one" name)
	  (RETURN))
	(SETQ methods (flavor-method-function-specs flavor))
	(SETQ methods (SORT methods #'(lambda (x y)
					(ALPHALESSP (STRING (CAR (LAST x)))
						    (STRING (CAR (LAST y)))))))
	(DOLIST (method methods)
	  (TERPRI stream)
	  (PRINC (function-and-short-documentation-string
		   method (COND ((> (LENGTH method) 3.) (CDDR method))
				(t			(CADDR method)))
		   0. full-doc remove-fonts?)
		 stream)
	  (COND (full-doc (TERPRI stream))))))

;1-----------------------------------------------------------------------------*

(DEFUN flavor-and-short-documentation-string (flavor-name &optional (indent 0.))
  (LET* ((DOC (SECOND (MEMQ ':documentation
			    (si:flavor-plist (FLAVORP flavor-name)))))
	 (doc-string (OR (AND (LISTP doc) (SECOND DOC)) DOC))
	 (first-doc-string (EXTRACT-FIRST-LINE doc-string)))
    (STRING-APPEND (FORMAT nil "~vx~s~30t" indent flavor-name)
		   (IF first-doc-string
		       (FORMAT nil ":~a" first-doc-string) ""))))


(DEFUN FLAVOR-METHOD-FUNCTION-SPECS (FLAVOR &AUX METHODS)
  "2Return a list of function specs for all the methods (except combined) of FLAVOR.*"
  (DOLIST (MTE (si:FLAVOR-METHOD-TABLE FLAVOR))
    (DOLIST (METH (CDDDR MTE))
      (OR (EQ (SI:METH-METHOD-TYPE METH) ':COMBINED)
	  (NOT (SI:METH-DEFINEDP METH))
	  (PUSH (si:METH-FUNCTION-SPEC METH) METHODS))))
  METHODS)


(DEFUN function-and-short-documentation-string
       (function-spec &optional (name function-spec) (indent 0.) (full-doc nil)
				(remove-fonts? t))
  "2Return a string with the function name and short documentation.*"
  (LET* ((doc (COND (full-doc (DOCUMENTATION function-spec))
		    (t (EXTRACT-FIRST-LINE (DOCUMENTATION function-spec))))))
    (WHEN remove-fonts?
      (SETQ doc (remove-font-change-chars doc)))
    (MULTIPLE-VALUE-BIND (args returns) (ARGLIST function-spec)
      (WHEN (AND (LISTP function-spec) (EQ (FIRST function-spec) ':method) (LISTP (CDR args)))
	(SETQ args (REST3 args)))
      (STRING-APPEND (FORMAT nil "~vX~S ~:A --> ~{~A  ~}"
			     indent name args returns)
		     (IF DOC (FORMAT nil "~%~vx~2t~a" indent doc) "")))))



(DEFUN EXTRACT-FIRST-LINE (STRING)
  "2Truncate a string at the first carrage return.*"
  (IF (STRINGP STRING)
      (SUBSTRING STRING 0
		 (OR (STRING-SEARCH-CHAR #\return STRING)
		     (STRING-LENGTH STRING)))))

(DEFUN FLAVORP (name)
  "2Return the flavor structure of NAME. Returns nil if NAME is not a flavor
or an instance of a flavor.*"
  (COND ((AND (SYMBOLP name) (GET name 'si:flavor)))
	((EQ (NAMED-STRUCTURE-P name) 'si:flavor) name)
	((EQ (DATA-TYPE name) 'dtp-instance)
	 (GET (TYPEP name) 'si:flavor))))
