;;; -*- Mode:Zetalisp; Package:USER; Base:10; Fonts:(MEDFNT HL12B HL12BI) -*-

;1; Functions to find and load files that haven't already been loaded*
;1; LaMott Oren August 1985*

(DEFUN not-loaded-p (files &optional pkg)
  "2Returns the file pathname of FILE when the latest version is NOT loaded in package PKG.
FILE may be a single file or a list of files.  If a list, returns the list of files not loaded.*"
  ;1; When only one file, use probe*
  (COND ((ATOM files)
	 (LET* ((PATHNAME (fs:parse-pathname files))
		(loaded-id (si:get-file-loaded-id pathname pkg))
		(why :no-loaded-id)
		probe)
	   ;1 Add file type to pathname if needed*
	   (WHEN (AND loaded-id (MEMQ (SEND pathname :type) '(nil :unspecific)))
	     (SETQ pathname (SEND pathname :new-pathname :type (SEND (CAR loaded-id) :type))))
	   ;1 Probe the file to check version number and creation date*
	   (WHEN loaded-id
	     (SETQ probe (OPEN pathname ':direction nil ':error nil))
	     (WHEN (ERRORP probe) (FORMAT t "~%~a" probe)
		   (setq why :file-not-found)
		   (SETQ probe nil)))
	   (UNLESS (AND probe
			(SETQ why :neq-truenames)
			(EQ (CAR loaded-id) (SEND probe :truename))
			(SETQ why :neq-creation-dates)
			(= (CDR loaded-id) (SEND probe :creation-date)))
	     (values pathname why))))
	;1; If a list of files, call ourselves recursively for each file.*
	(t (LOOP for file in files
		 for pathname = (not-loaded-p file pkg)
		 when pathname collect pathname))))

(DEFUN loadf (file &rest options)
  "2Load FILE only if it isn't already loaded.*"
  (DECLARE (ARGLIST FILE &KEY &OPTIONAL PACKAGE VERBOSE SET-DEFAULT-PATHNAME
		    (IF-DOES-NOT-EXIST T) PRINT))
  (LET ((pkg (GET (LOCF options) :package)))
    (WHEN (not-loaded-p file pkg)
      (LEXPR-FUNCALL #'LOAD file options))))

(DEFUN load-menu (file-alist &optional &key
		  (label "Select the files you want loaded")
		  pathname-default
		  )
  "2Load those files in FILE-ALIST that the user selects from a menu.  Only those files that
haven't already been loaded are presented in the menu.  File-alist elements can be pathnames,
or (pathname menu documentation) where menu is the menu item string, and documentation is
the mouse documentation.*"
  (SETQ file-alist (COPYTREE file-alist))
  (WHEN pathname-default
    (SETQ pathname-default (fs:parse-pathname pathname-default)))
  ;1; Find those files not already loaded*
  (LET* ((file-list
	   (not-loaded-p
	     (LOOP for file in file-alist
		   with pathname
		   WHEN (LISTP file)
		   do (SETQ pathname (fs:merge-pathname-defaults
				       (CAR file)
				       pathname-default :xfasl :newest t))
		      (RPLACA file pathname)
		      collect pathname)))
	 ;1; Build a menu of the files not loaded*
	 (menu (LOOP for file in file-alist
		     with path-name and menu-item and doc
		     when (LISTP file)
		     do (SETQ path-name (FIRST file)
			      menu-item (SECOND file)
			      doc (THIRD file))
		        and when (MEMQ path-name file-list)
			collect `(,menu-item :value ,path-name :documentation
				  ,(or doc "Load this file"))
			else do nil
		    else collect `(,file :no-select t :font fonts:cptfontb)))
	 ;1; If there are any files not loaded, let the user choose them from the menu*
	 (load-list (AND file-list
#-elroy                  (tv:multiple-menu-choose menu label)
#+elroy                  (w:multiple-menu-choose menu :label label)
                         )))
    (LOOP for file in (nreverse load-list) do (LOAD file))))
