;;; -*- Mode:Common-LISP; Fonts:(CPTFONT HL12BI HL12B); Package:(REMIND :size 100); Base:10 -*-
;1;;*
;1;; Description:  A reminder tool for maintaining a list of reminders and notifying the *
;1;;                user when a reminder is due. See function REMINDERS and its*
;1;;                documentation string. This function presents a CVV interface for*
;1;;                examining and modifying the current reminder list.*
;1;;                *
;1;;                This reminder tool is similar in purpose to the ALARM-CLOCK*
;1;;                facility. Compared to ALARM-CLOCK, REMINDERS is somewhat*
;1;;                simpler (one function instead of three), is more mouse-oriented*
;1;;                (uses CVV and w:mouse-confirm pop-up windows), is a bit more*
;1;;                flexible for repeating reminders, and allows a wide array of*
;1;;                interesting notification sounds.*
;1;;*
;1;; How to Operate:*
;1;;                Load this file. Use function REMIND:REMINDERS.  This function is*
;1;;                also exported to the USER package.*
;1;;*
;1;;                Reminders are recorded in LM:REMIND;REMINDERS.LISP each time you*
;1;;                add/delete reminders.*
;1;;                *
;1;; History:      3/30/87  Kerry Kimbrough           Created*
;1;;                7/08/87  Dane Meyer                Added ABORT to CVV.*
;1;;                                                         Changed change-remind-order to ignore it*s1 args (aren't used).*
;1;;                                                         Moved defvar of *reminder-daemon* to top of file to prevent warnings.*
;1;;*              17/27/87  Dane meyer                Added Inhibit-reminders feature so reminders can be suppressed*
;1;;                                                           over some date range -- for times you won't be around.*


(DEFVAR *reminders-file* "lm:nichols;reminders.lisp#>" "2The saved reminders file*")
(DEFVAR *reminder-daemon* (MAKE-PROCESS "Reminder Daemon"))
(DEFVAR *reminders* nil "2The current reminder list.*")
(DEFVAR *reminder-beep-type* :beep "2The reminder beep type*")
(DEFVAR *inhibit-reminders-start* nil)  ; date when reminders start being inhibited.
(DEFVAR *inhibit-reminders-end* nil)    ; date when reminders stop being inhibited
(DEFVAR *inhibit-status-msg* "")         ; message to display when reminders are inhibited.


(DEFMACRO default-reminder ()
  `(LIST
     (GET-UNIVERSAL-TIME)				;1next reminder time*
     "Enter message for new reminder."			;1reminder message*
     1							;1occurrences left*
     1							;1repeat interval*
     :days						;1repeat units*
     ))

(DEFUN must-be-positive-number (new-value)
  "2Constraint function to ensure that reminder repeat interval is positive.*"
  (AND (NUMBERP new-value) (PLUSP new-value)))

(DEFUN refresh-reminders ()
  (SETQ *reminders*
	(REMOVE-IF #'(lambda (x) (AND (NUMBERP x) (NOT (PLUSP x))))
		   *reminders*
		   :key #'THIRD))
  (SORT *reminders* #'< :key #'FIRST)
  )

(DEFUN change-reminder-order (window variable old-value new-value)
  "2Called when reminder list changes to sort in order of next occurrence.*"
  (ignore window variable old-value new-value)
  (refresh-reminders)
  (SEND window :refresh)
  T)							;1return T for no redisplay*


(DEFUN add-reminder (&optional (reminder (default-reminder)))
  "2Add a new reminder and resort the reminder list.*"
  (PUSH reminder *reminders*)
  (refresh-reminders)
  T)							;1must return non-nil*

(DEFUN next-occurrence (reminder)
  "2Update the reminder, decrementing its number of occurrences left and
resetting its next reminder time. Return nil iff the number of occurrences is 0.*"
  (WHEN (OR (NULL (THIRD reminder)) (PLUSP (DECF (THIRD reminder))))
    (INCF (FIRST reminder)
	  (* (FOURTH reminder)
	     (CASE (FIFTH reminder)
	       (:minutes 60)				;1sec/minute*
	       (:hours   3600)				;1sec/hour*
	       (:days    86400)				;1sec/day*
	       (:weeks   604800))))))			;1sec/week*

(DEFUN save-reminders ()
  (WITH-OPEN-FILE (file  *reminders-file*
			:direction :output
			:if-exists :truncate
			:if-does-not-exist :create)
    (WRITE *reminders* :stream file)
    (FORMAT file " ")
    (WRITE *reminder-beep-type* :stream file)
    (WHEN (and *inhibit-reminders-start*
	       *inhibit-reminders-end*)
      (FORMAT file " ")
      (WRITE *inhibit-reminders-start* :stream file)
      (FORMAT file " ")
      (WRITE *inhibit-reminders-end* :stream file)))
  (reminders-cleanup))

(DEFUN reminders ()
  1"Examine and edit the current reminder list. 

The reminder system maintains a list of reminders, each of which can be set up
to reoccur at regular intervals. For example, you might have a reminder which
would warn you to go to lunch every day at noon. You can ask for a reminder to
reoccur for a fixed number of times only or to repeat indefinitely. You can
specify the interval between repetitions in minutes, hours, days, or weeks.

A copy of the reminder list is also maintained as a file (\"LM: REMIND;
REMINDERS.LISP\"). Thus, your repeating reminders are remembered, even if you
reboot."*

  ;1; Edit reminder list. If empty, ask first*
  (WHEN (OR *reminders*
	    (IF (w:mouse-confirm
		  "There are no reminders. Click here if you want to add one." "")
		(add-reminder)))
    
    ;1; Arrest the reminder daemon process*
    (PROCESS-DISABLE *reminder-daemon*)
    
    ;1; Repeat CVV until no additional functions are required*
    (DO (function)
	((NOT (SETQ function
		    (CATCH :before-repeat
		      (w:choose-variable-values
			`(""
			  (*reminders*
			    (0 "Next Reminder Time" 
			       :documentation "Change the time for the next occurrence of this reminder."
			       :date)
			    (1 "        Reminder Message        " 
			       :documentation "Change the message for this reminder."
			       :string)
			    (2 "Occurrences Left" 
			       :documentation 
			       "Change the remaining number of occurrences for this reminder (0 to remove, NIL to repeat forever)."
			       :fixnum-or-nil)
			    (3 "Repeat Interval"
			       :documentation "Change the interval between occurrences of this reminder."
			       :test must-be-positive-number
			       :fixnum)
			    (4 " Units "
			       :documentation "Change the interval units for this reminder."	1  *
			       :menu '(:Minutes :Hours :Days :Weeks)))
			  ""
			  ,*inhibit-status-msg*)
			:function       #'change-reminder-order
			:label          "Edit Reminder List"
			:margin-choices '("Do It"
					  ("Add Reminder"   (THROW :before-repeat :add-reminder))
					  ("Change Beep"    (THROW :before-repeat :change-beep))
					  ("Change Inhibit" (THROW :before-repeat :change-inhibit))
					  ("ABORT"          (THROW :before-repeat nil))
					  ))
		      
		      nil))))			;1exit :before-repeat loop*
      
      
      (CASE function
	;1; Add a new reminder to list*
	(:add-reminder
	 (add-reminder))
	
	;1; Change the reminder beep type*
	(:change-beep
	 (LET ((beep-types tv:*beeping-functions*))
	   (SETQ *reminder-beep-type*
		 (OR (w:menu-choose (SORT beep-types #'STRING<)
				    :label "Choose New Reminder Beep"
				    :default-item *reminder-beep-type*)
		     *reminder-beep-type*)))
	 (BEEP *reminder-beep-type*))

	(:change-inhibit
	 (inhibit-reminders))
	))
    
    ;1; Save current reminder list*
    (save-reminders)
    
    ;1; Restart the reminder daemon process*
    (SEND *reminder-daemon* :reset)
    (PROCESS-ENABLE *reminder-daemon*)))

(DEFUN inhibit-reminders ()
  "Indicate a date range to inhibit all reminders"
  
  (w:choose-variable-values
    '(""
      (*inhibit-reminders-start*
	:documentation "Date to start inhibitting reminders. NEVER to turn off all inhibitting."
	:date-or-never)
      (*inhibit-reminders-end*
	:documentation "Date to stop inhibitting reminders.  NEVER to turn off all inhibitting."
	:date-or-never)
      "")
    :label "Indicate dates to inibit"
    :margin-choices '(("Abort" (SIGNAL-CONDITION eh:*abort-object*)) "Do It"))

  (update-inhibit-status-msg))

(DEFUN update-inhibit-status-msg ()
  (IF (AND *inhibit-reminders-start*
	   *inhibit-reminders-end*)
      (SETQ *inhibit-status-msg*
	    (FORMAT nil "Reminders inhibited from ~a to ~a"
		    (time:print-universal-time *inhibit-reminders-start* nil)
		    (time:print-universal-time *inhibit-reminders-end* nil)))
    (SETQ  *inhibit-status-msg* ""
	   *inhibit-reminders-start* nil
	   *inhibit-reminders-end* nil)))

(DEFUN remind ()
  "2Wait for next reminder to be due and then display it.*"
  (DO (next-reminder-msg
       (next-reminder (CAR *reminders*) (CAR *reminders*)))

      ((NULL next-reminder))

    ;1; Wait for next reminder to come up.*
    (PROCESS-WAIT "Waiting for reminder"
		  #'(lambda (TIME) (> (GET-UNIVERSAL-TIME) time))
		  (FIRST next-reminder))
    
    ;1; Remove this reminder and add next occurrence of it, if necessary.*
    (POP  *reminders*)
    (COND ((next-occurrence next-reminder)
	   (add-reminder next-reminder)
	   (SETQ next-reminder-msg
		 (FORMAT nil "~%~%The next occurrence of this reminder will be at ~a."
			 (TIME:PRINT-UNIVERSAL-TIME (FIRST next-reminder) nil))))
	  (t
	   (SETQ next-reminder-msg "")))

    ;1; turn off inhibit-reminders if time has elapsed*
    (when (and *inhibit-reminders-end*
	       (> (get-universal-time)  *inhibit-reminders-end*))
      (setq *inhibit-reminders-start* nil
	    *inhibit-reminders-end* nil)
      (update-inhibit-status-msg))

    (save-reminders)

    (if (and *inhibit-reminders-start*
	     *inhibit-reminders-end*
	     (<  *inhibit-reminders-start* (time:get-universal-time) *inhibit-reminders-end*))
	(tv:notify nil *inhibit-status-msg*)

	;1; Else Display reminder message*
	(BEEP *reminder-beep-type*)
	(w:mouse-confirm (FORMAT nil "Reminder:       ~a"
				 (TIME:PRINT-UNIVERSAL-TIME (GET-UNIVERSAL-TIME) nil))
			 (FORMAT nil "~a~a"
				 (SECOND next-reminder)	;1display reminder message*
				 next-reminder-msg)	;1display time of next reminder*
			 fonts:tr12b
			 fonts:tr12b))
    ))




;1; Read in the reminder list file*
(UNLESS *reminders-file*
  (LET* (dir)
    (fs:create-directory (SETQ dir (FORMAT nil "lm:~A.remind;" user-id)))
    (SETQ *reminders-file*  (STRING-APPEND dir "reminders.lisp"))))
;(fs:create-directory "lm:remind;")
(WITH-OPEN-FILE (file "lm:remind;reminders.lisp#>"
			:direction :input
			:if-does-not-exist :create)
    (SETF *reminders* (READ file nil))
    (SETF *reminder-beep-type* (READ file nil :beep))
    (SETF *inhibit-reminders-start* (READ file nil nil))
    (SETF *inhibit-reminders-end*   (READ file nil nil))
    (update-inhibit-status-msg))

;1; Enable the reminder daemon process*
(DEFVAR *reminder-daemon* (MAKE-PROCESS "Reminder Daemon"))
(SEND *reminder-daemon*  :preset #'remind)
(PROCESS-ENABLE *reminder-daemon*)

;1; Make the reminders function available in USER:*
(EXPORT 'reminders)
(USE-PACKAGE 'remind 'user)

;1;;Cleanup stuff*
(DEFVAR reminders-expunge t)

(DEFUN reminders-cleanup (&optional (file *reminders-file*) (keep 2) &aux path current del result)
  (WHEN (AND reminders-expunge (SETQ path (PROBE-FILE file)))
    (SETQ current (SEND path :version))
    (LOOP
        while (> current keep) do
          (UNLESS (ERRORP (SEND (SETQ del (SEND path :new-version (- current keep))) :delete-and-expunge nil))
             (PUSH del result))
          (DECF current)))
  result)

