/*
 * disasm.c
 *
 * disassembler for macrocode
 */

#include "utils.h"

#include "types.h"
#include "memory.h"
#include <stdio.h>
#include <string.h>

#define OPT_UNKNOWN 0
#define OPT_MAIN    1
#define OPT_BRANCH  2
#define OPT_IMMED   3
#define OPT_MISC    4
#define OPT_AUX     5
#define OPT_AREFI   6
#define OPT_MODULE  7

struct instr {
    int type; /* OPT_xxx constants */
    char *name;
} instructions[0x80] = {
    /* 00 - 07 */
    {OPT_AUX},
    {OPT_MISC},
    {OPT_MODULE},
    {OPT_IMMED, "EQ-IMMED"},
    {OPT_IMMED, "=-IMMED"},
    {OPT_IMMED, ">-IMMED"},
    {OPT_IMMED, "<-IMMED"},
    {OPT_AREFI, "TEST-AREFI"},

    /* 08 - 0f */
    {OPT_MAIN, "TEST"},
    {OPT_MAIN, "TEST-CAR"},
    {OPT_MAIN, "TEST-CDR"},
    {OPT_MAIN, "TEST-CADR"},
    {OPT_MAIN, "TEST-CDDR"},
    {OPT_MAIN, "TEST-CAAR"},
    {OPT_MAIN, "TEST-MEMO"},
    {OPT_MAIN, "RETURN"},
    
    /* 10 - 17 */
    {OPT_MAIN,	"="},
    {OPT_MAIN,	">"},
    {OPT_MAIN,	"<"},
    {OPT_MAIN, "EQ"},
    {OPT_MAIN, "EQL"},
    {OPT_MAIN, "EQUAL"},
    {OPT_MAIN, "EQUALP"},
    {OPT_UNKNOWN},

    /* 18 - 1f */
    {OPT_MAIN, "NUMBERP"},
    {OPT_MAIN, "ARRAYP"},
    {OPT_MAIN, "LISTP"},
    {OPT_MAIN, "STRINGP"},
    {OPT_MAIN, "FIXNUMP"},
    {OPT_MAIN, "INTEGERP"},
    {OPT_MAIN, "PLUSP"},
    {OPT_MAIN, "MINUSP"},
    
    /* 20 - 27 */
    {OPT_UNKNOWN},
    {OPT_MISC},
    {OPT_MODULE},
    {OPT_IMMED,	"ADD-IMMED"},
    {OPT_IMMED, "LDB-IMMED"},
    {OPT_IMMED, "PUSH-NUMBER"}, /* ??? SSDN2 disagrees? */
    {OPT_IMMED, "PUSH-NEG-NUMBER"},
    {OPT_AREFI, "PUSH-AREFI"}, /* ??? SSDN2 says this is PUSH-NUMBER? */

    /* 28 - 2f */
    {OPT_MAIN, "PUSH"},
    {OPT_MAIN, "PUSH-CAR"},
    {OPT_MAIN, "PUSH-CDR"},
    {OPT_MAIN, "PUSH-CADR"},
    {OPT_MAIN, "PUSH-CDDR"},
    {OPT_MAIN, "PUSH-CADDR"},
    {OPT_MAIN, "PUSH-CONS"},
    {OPT_MAIN, "PUSH-GET"},
    
    /* 30 - 37 */
    {OPT_MAIN, "+"},
    {OPT_MAIN, "-"},
    {OPT_MAIN, "*"},
    {OPT_MAIN, "LOGAND"},
    {OPT_MAIN, "LOGXOR"},
    {OPT_MAIN, "1+"},
    {OPT_MAIN, "1-"},
    {OPT_MAIN, "PUSH-AR-1"},

    /* 38 - 3f */
    {OPT_IMMED, "PUSH-LONG-FEF"},
    {OPT_IMMED, "SELECT"},
    {OPT_IMMED, "DISPATCH"},
    {OPT_UNKNOWN},
    {OPT_UNKNOWN},
    {OPT_UNKNOWN},
    {OPT_MAIN, "LEXICAL-UNSHARE"}, /* ??? Doesn't specify OPT_xxx? */
    {OPT_MAIN, "LOCATE-LEXICAL-ENVIRONMENT"},
    
    /* 40 - 47 */
    {OPT_MAIN, "CALL-0-DEST-INDS"},
    {OPT_MAIN, "CALL-0-DEST-PUSH"},
    {OPT_MAIN, "CALL-0-DEST-RETURN"},
    {OPT_MAIN, "CALL-0-DEST-TAIL-REC"},
    {OPT_MAIN, "CALL-1-DEST-INDS"},
    {OPT_MAIN, "CALL-1-DEST-PUSH"},
    {OPT_MAIN, "CALL-1-DEST-RETURN"},
    {OPT_MAIN, "CALL-1-DEST-TAIL-REC"},

    /* 48 - 4f */
    {OPT_MAIN, "CALL-2-DEST-INDS"},
    {OPT_MAIN, "CALL-2-DEST-PUSH"},
    {OPT_MAIN, "CALL-2-DEST-RETURN"},
    {OPT_MAIN, "CALL-2-DEST-TAIL-REC"},
    {OPT_MAIN, "CALL-3-DEST-INDS"},
    {OPT_MAIN, "CALL-3-DEST-PUSH"},
    {OPT_MAIN, "CALL-3-DEST-RETURN"},
    {OPT_MAIN, "CALL-3-DEST-TAIL-REC"},

    /* 50 - 57 */
    {OPT_MAIN, "CALL-4-DEST-INDS"},
    {OPT_MAIN, "CALL-4-DEST-PUSH"},
    {OPT_MAIN, "CALL-4-DEST-RETURN"},
    {OPT_MAIN, "CALL-4-DEST-TAIL-REC"},
    {OPT_MAIN, "CALL-5-DEST-INDS"},
    {OPT_MAIN, "CALL-5-DEST-PUSH"},
    {OPT_MAIN, "CALL-5-DEST-RETURN"},
    {OPT_MAIN, "CALL-5-DEST-TAIL-REC"},

    /* 58 - 5f */
    {OPT_MAIN, "CALL-6-DEST-INDS"},
    {OPT_MAIN, "CALL-6-DEST-PUSH"},
    {OPT_MAIN, "CALL-6-DEST-RETURN"},
    {OPT_MAIN, "CALL-6-DEST-TAIL-REC"},
    {OPT_MAIN, "CALL-N-DEST-INDS"},
    {OPT_MAIN, "CALL-N-DEST-PUSH"},
    {OPT_MAIN, "CALL-N-DEST-RETURN"},
    {OPT_MAIN, "CALL-N-DEST-TAIL-REC"},
    
    /* 60 - 67 */
    {OPT_MAIN, "POP"},
    {OPT_MAIN, "MOVEM"},
    {OPT_MAIN, "SETE-CDR"},
    {OPT_MAIN, "SETE-CDDR"},
    {OPT_MAIN, "SETE-1+"},
    {OPT_MAIN, "SETE-1-"},
    {OPT_UNKNOWN},
    {OPT_MAIN, "PUSH-CDR-STORE-CAR-IF-CONS"},

    /* 68 - 6f */
    {OPT_MAIN, "PUSH-LOC"},
    {OPT_MAIN, "BIND-NIL"},
    {OPT_MAIN, "BIND-T"},
    {OPT_MAIN, "BIND-POP"},
    {OPT_MAIN, "BIND-CURRENT"},
    {OPT_MAIN, "SET-NIL"},
    {OPT_MAIN, "SET-T"},
    {OPT_MAIN, "SET-ZERO"},
    
    /* 70 - 77 */
    {OPT_BRANCH, "BR-NIL-ELSE-POP"},
    {OPT_BRANCH, "BR-NOT-NIL-ELSE-POP"},
    {OPT_BRANCH, "BR-NIL"},
    {OPT_BRANCH, "BR-NOT-NIL"},
    {OPT_BRANCH, "BR-ATOM"},
    {OPT_BRANCH, "BR-NOT-ATOM"},
    {OPT_BRANCH, "BR-ZEROP"},
    {OPT_BRANCH, "BR-NOT-ZEROP"},

    /* 78 - 7f */
    {OPT_BRANCH, "BR-SYMBOLP"},
    {OPT_BRANCH, "BR-NOT-SYMBOLP"},
    {OPT_BRANCH, "unused"},
    {OPT_BRANCH, "unused"},
    {OPT_BRANCH, "BR-NIL-LIKELY"},
    {OPT_BRANCH, "BR-NOT-NIL-LIKELY"},
    {OPT_BRANCH, "BR"},
    {OPT_BRANCH, "illegal"},
};

char *miscop_names[0x200] = { /* 0 - 0777 */
    /* 000 - 007 */
    NULL,
    NULL,
    "CAR",
    "CDR",
    "CAAR",
    "CADR",
    "CDAR",
    "CDDR",
    
    /* 010 - 017 */
    "CAAAR",
    "CAADR",
    "CADAR",
    "CADDR",
    "CDAAR",
    "CDADR",
    "CDDAR",
    "CDDDR",
    
    /* 020 - 027 */
    "CAAAAR",
    "CAAADR",
    "CAADAR",
    "CAADDR",
    "CADAAR",
    "CADADR",
    "CADDAR",
    "CADDDR",
    
    /* 030 - 037 */
    "CDAAAR",
    "CDAADR",
    "CDADAR",
    "CDADDR",
    "CDDAAR",
    "CDDADR",
    "CDDDAR",
    "CDDDDR",
    
    /* 040 - 047 */
    "PDL-WORD",
    "SHRINK-PDL-SAVE-TOP",
    NULL,
    NULL,
    NULL,
    NULL,
    "STACK-GROUP-RETURN",
    "STACK-GROUP-RESUME",
    
    /* 050 - 057 */
    "%P-DPB",
    "%P-DEPOSIT-FIELD",
    "%P-DPB-OFFSET",
    "%P-DEPOSIT-FIELD-OFFSET",
    "%PHYS-LOGLDB",
    "%PHYS-LOGDPB",
    NULL,
    "%GC-SCAV-RESET",
    
    /* 060 - 067 */
    NULL,
    NULL,
    "%P-STORE-CONTENTS",
    "%P-STORE-TAG-AND-POINTER",
    "%P-STORE-CDR-CODE",
    "%P-STORE-DATA-TYPE",
    "%P-STORE-POINTER",
    "%P-STORE-CONTENTS-OFFSET",
    
    /* 070 - 077 */
    "INTERNAL-GET-2",
    "GETL",
    "GET-LOCATION-OR-NIL",
    "INTERNAL-GET-3",
    NULL,
    NULL,
    "Predicate",
    "Not-Indicators",
    
    /* 100 - 107 */
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    
    /* 110 - 117 */
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    
    /* 120 - 127 */
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    
    /* 130 - 137 */
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    
    /* 140 - 147 */
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    
    /* 150 - 157 */
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    
    /* 160 - 167 */
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    
    /* 170 - 177 */
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    
    /* 200 - 207 */
    "BIND",
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    
    /* 210 - 217 */
    "%IO",
    "%ADD-INTERRUPT",
    NULL,
    NULL,
    "%MULTIBUS-WRITE-16",
    "%MULTIBUS-WRITE-8",
    "%MULTIBUS-WRITE-32",
    "%NUBUS-WRITE",
    
    /* 220 - 227 */
    "%NuBus-Write-8B",
    "%NuBus-Write-16B",
    NULL,
    "%blt-to-physical",
    "%blt-from-physical",
    "%IO-SPACE-WRITE",
    "%TEST&STORE-68K",
    "%TEST&SET-68K",
    
    /* 230 - 237 */
    "FLOATP",
    "LENGTH-GREATERP",
    "INTERNAL-CHAR-EQUAL",
    "%STRING-EQUAL",
    "ARRAY-HAS-LEADER-P",
    "GLOBAL:NLISTP",
    "NSYMBOLP",
    "FBOUNDP",
    
    /* 240 - 247 */
    "BOUNDP",
    "ARRAY-HAS-FILL-POINTER-P",
    NULL,
    "COMMON-LIST-LISTP",
    "VECTORP",
    "SIMPLE-VECTOR-P",
    "SIMPLE-ARRAY-P",
    "SIMPLE-STRING-P",
    
    /* 250 - 257 */
    "BIT-VECTOR-P",
    "SIMPLE-BIT-VECTOR-P",
    "TYPEP-STRUCTURE-OR-FLAVOR",
    "SMALL-FLOATP",
    "CHARACTERP",
    "ENDP",
    "Rationalp",
    "Ratiop",
    
    /* 260 - 267 */
    "Complexp",
    "EQ-T",
    "Single-Floatp",
    "Double-Floatp",
    NULL,
    NULL,
    NULL,
    NULL,
    
    /* 270 - 277 */
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    
    /* 300 - 307 */
    "RPLACA",
    "RPLACD",
    "SETCAR",
    "SETCDR",
    "CONSP-OR-POP",
    NULL,
    NULL,
    "SETELT",
    
    /* 310 - 317 */
    "SET",
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    
    /* 320 - 327 */
    "STORE-ARRAY-LEADER",
    "AS-1",
    "AS-2",
    "AS-3",
    "AS-1-FORCE",
    "GLOBAL:AS-2-REVERSE",
    NULL,
    NULL,
    
    /* 330 - 337 */
    "SET-ARRAY-LEADER",
    "SET-AR-1",
    "SET-AR-2",
    NULL,
    "SET-AR-3",
    "SET-AR-1-FORCE",
    NULL,
    NULL,
    
    /* 340 - 347 */
    "GLOBAL:ARRAY-PUSH",
    "VECTOR-PUSH",
    "COPY-ARRAY-CONTENTS",
    "COPY-ARRAY-CONTENTS-AND-LEADER",
    "COPY-ARRAY-PORTION",
    "BITBLT",
    "%BLT",
    "%BLT-TYPED",
    
    /* 350 - 357 */
    NULL,
    NULL,
    NULL,
    NULL,
    "SET-%INSTANCE-REF",
    "%DISPATCH-METHOD",
    "%CLASS-DESCRIPTION",
    NULL,
    
    /* 360 - 367 */
    "%CHANGE-PAGE-STATUS",
    NULL,
    "%DELETE-PHYSICAL-PAGE",
    "%PAGE-IN",
    "%LOAD-MEMORY-MAP",
    NULL,
    NULL,
    NULL,
    
    /* 370 - 377 */
    "%WRITE-INTERNAL-PROCESSOR-MEMORIES",
    "%PAGE-TRACE",
    "%RECORD-EVENT",
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    
    /* 400 - 407 */
    "%MAKE-STACK-LIST",
    "%MAKE-EXPLICIT-STACK-LIST",
    "%MAKE-EXPLICIT-STACK-LIST*",
    "%MAKE-STACK-LIST*",
    "%STACK-FRAME-POINTER",
    "SPECIAL-PDL-INDEX",
    "UNBIND-TO-INDEX-MOVE",
    NULL,
    
    /* 410 - 417 */
    "GLOBAL:NCONS",
    "GLOBAL:NCONS-IN-AREA",
    "CONS",
    "CONS-IN-AREA",
    "%MAKE-LIST",
    "%ALLOCATE-AND-INITIALIZE",
    "%ALLOCATE-AND-INITIALIZE-ARRAY",
    "%Ratio-Cons",
    
    /* 420 - 427 */
    "%Allocate-and-Initialize-Instance",
    "%MAKE-LIST*",
    "%ALLOCATE-AND-INITIALIZE-SYMBOL",
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    
    /* 430 - 437 */
    "%PHYSICAL-ADDRESS",
    "%MULTIBUS-READ-16",
    "%MULTIBUS-READ-8",
    "%MULTIBUS-READ-32",
    "%NUBUS-READ",
    "%NuBus-Read-8B",
    "%NuBus-Read-16B",
    "%NuBus-Read-32B",
    
    /* 440 - 447 */
    "%NuBus-Read-8B-Careful",
    "%IO-SPACE-READ",
    NULL,
    NULL,
    NULL,
    "%POINTER",
    "%MAKE-POINTER",
    "%MAKE-POINTER-OFFSET",
    
    /* 450 - 457 */
    "%DATA-TYPE",
    "%P-CDR-CODE",
    "%P-DATA-TYPE",
    "%P-POINTER",
    "%P-LDB",
    "%P-MASK-FIELD",
    "%P-CONTENTS-OFFSET",
    "%P-LDB-OFFSET",
    
    /* 460 - 467 */
    "%P-MASK-FIELD-OFFSET",
    "%P-CONTENTS-AS-LOCATIVE",
    "%P-CONTENTS-AS-LOCATIVE-OFFSET",
    "%POINTER-DIFFERENCE",
    "%STORE-CONDITIONAL",
    NULL,
    NULL,
    NULL,
    
    /* 470 - 477 */
    "%MICROSECOND-TIME",
    "%FIXNUM-MICROSECOND-TIME",
    "TIME-IN-60ths",
    NULL,
    "%PAGE-STATUS",
    "%COMPUTE-PAGE-HASH",
    "%FINDCORE",
    NULL,
    
    /* 500 - 507 */
    "%AREA-NUMBER",
    "%REGION-NUMBER",
    "%FIND-STRUCTURE-HEADER",
    "%STRUCTURE-BOXED-SIZE",
    "%STRUCTURE-TOTAL-SIZE",
    "%MAKE-REGION",
    "%FIND-STRUCTURE-LEADER",
    NULL,
    
    /* 510 - 517 */
    "GLOBAL:FIX",
    "SMALL-FLOAT",
    "INTERNAL-FLOAT",
    "ABS",
    "GLOBAL:MINUS",
    "HAULONG",
    "FLOAT-EXPONENT",
    "FLOAT-FRACTION",
    
    /* 520 - 527 */
    "LDB",
    "%LOGLDB",
    "MASK-FIELD",
    NULL,
    "DPB",
    "%LOGDPB",
    "DEPOSIT-FIELD",
    NULL,
    
    /* 530 - 537 */
    "LSH",
    "ASH",
    "ROT",
    "*BOOLE",
    "MAX",
    "MIN",
    "EXPT",
    NULL,
    
    /* 540 - 547 */
    "global:REMAINDER",
    "QUOTIENT",
    "LOGIOR",
    "GCD",
    "%DIV",
    "SCALE-FLOAT",
    "DOUBLE-FLOAT",
    NULL,
    
    /* 550 - 557 */
    "FLOOR-1",
    "CEILING-1",
    "TRUNCATE-1",
    "ROUND-1",
    "FLOOR-2",
    "CEILING-2",
    "TRUNCATE-2",
    "ROUND-2",
    
    /* 560 - 567 */
    "ZEROP",
    "FIXP",
    "EQUAL",
    "NOT",
    "ATOM",
    "NUMBERP",
    "PLUSP",
    "MINUSP",
    
    /* 570 - 577 */
    "<",
    ">",
    "=",
    "EQL",
    "EQ",
    "EQUALP",
    "LISTP",
    "SYMBOLP",
    
    /* 600 - 607 */
    "ARRAYP",
    "STRINGP",
    "FIXNUMP",
    "NAMED-STRUCTURE-P",
    NULL,
    NULL,
    NULL,
    NULL,
    
    /* 610 - 617 */
    "GLOBAL:ASSQ",
    "LAST",
    "LENGTH",
    "GLOBAL:MEMQ",
    "NTH",
    "NTHCDR",
    "GLOBAL:FIND-POSITION-IN-LIST",
    NULL,
    
    /* 620 - 627 */
    "CAR-SAFE",
    "CDR-SAFE",
    "CARCDR",
    NULL,
    NULL,
    "ELT",
    "COMMON-LISP-ELT",
    "SYMBOL-FUNCTION",
    
    /* 630 - 637 */
    NULL,
    "SYMBOL-NAME",
    "VALUE-CELL-LOCATION",
    "FUNCTION-CELL-LOCATION",
    "PROPERTY-CELL-LOCATION",
    "SYMBOL-PACKAGE",
    "SYMBOL-VALUE",
    "%EXTERNAL-VALUE-CELL",
    
    /* 640 - 647 */
    "ARRAY-LEADER",
    "GLOBAL:AR-1",
    "AR-2",
    "AR-3",
    "AP-LEADER",
    "AP-1",
    "AP-2",
    "AP-3",
    
    /* 650 - 657 */
    "GLOBAL:AR-2-REVERSE",
    "GLOBAL:AR-1-FORCE",
    "AP-1-FORCE",
    "G-L-P",
    "BIGNUM-TO-ARRAY",
    NULL,
    NULL,
    NULL,
    
    /* 660 - 667 */
    "ARRAY-LENGTH",
    NULL,
    "ARRAY-ACTIVE-LENGTH",
    "ARRAY-LEADER-LENGTH",
    "ARRAY-RANK",
    "ARRAY-DIMENSION",
    NULL,
    NULL,
    
    /* 670 - 677 */
    "COMMON-LISP-AR-1",
    "COMMON-LISP-AR-2",
    "COMMON-LISP-AR-3",
    "COMMON-LISP-AR-1-FORCE",
    NULL,
    NULL,
    NULL,
    NULL,
    
    /* 700 - 707 */
    "%SXHASH-STRING",
    "%STRING-SEARCH-CHAR",
    "%STRING-WIDTH",
    "INT-CHAR",
    "CHAR-INT",
    NULL,
    NULL,
    NULL,
    
    /* 710 - 717 */
    "LOCATE-IN-INSTANCE",
    "%GET-SELF-MAPPING-TABLE",
    "%INSTANCE-REF",
    "%INSTANCE-LOC",
    "%FUNCTION-INSIDE-SELF",
    NULL,
    NULL,
    "CLOSURE",
    
    /* 720 - 727 */
    "LOAD-FROM-HIGHER-CONTEXT",
    "LOCATE-IN-HIGHER-CONTEXT",
    "GET-LEXICAL-VALUE-CELL",
    "MAKE-LEXICAL-CLOSURE",
    "MAKE-EPHEMERAL-LEXICAL-CLOSURE",
    "PL-TRAIL1",
    "PL-BIND-VAR-TO-VAR",
    "PL-BIND-VAR-TO-TERM",
    
    /* 730 - 737 */
    "PL-VARP",
    "PL-LISTP",
    "PL-STRUCTP",
    "PL-ATOMP",
    "PL-ATOMICP",
    "PL-INITIALIZE",
    "PL-INITIALIZE1",
    "PL-DEREFERENCE",
    
    /* 740 - 747 */
    "PL-RESOLVE",
    "PL-Stack-Overflow-SaveCP",
    "PLL-FAIL1",
    "PL-UNIFY1",
    "%getlong",
    "%putlong",
    "%PUTWORD-16B",
    "%GETWORD-16B",
    
    /* 750 - 757 */
    "%BUFFER-CHAR-MAP",
    "%sum-array",
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    
    /* 760 - 767 */
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    
    /* 770 - 777 */
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
};

char *auxop_names[0x100] = { /* 0 - 0377 (0400 - 0777 handled elsewhere) */
    /* 000 - 007 */
    NULL,
    "BREAKPOINT",
    "%HALT",
    "%CRASH",
    NULL,
    NULL,
    NULL,
    NULL,
    
    /* 010 - 017 */
    "EXCHANGE",
    "%SPREAD",
    "%ASSURE-PDL-ROOM",
    "POP-M-FROM-UNDER-N",
    "POPJ",
    "%UNWIND-PROTECT-CLEANUP",
    "%USING-BINDING-INSTANCES",
    "UNBIND-TO-INDEX",
    
    /* 020 - 027 */
    NULL,
    NULL,
    "%SET-SELF-MAPPING-TABLE",
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    
    /* 030 - 037 */
    "STORE-IN-HIGHER-CONTEXT",
    "LEXICAL-UNSHARE-ALL",
    "%STORE-KEY-WORD-ARGS",
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    
    /* 040 - 047 */
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    
    /* 050 - 057 */
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    
    /* 060 - 067 */
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    
    /* 070 - 077 */
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    
    /* 100 - 107 */
    "COMPLEX-CALL-TO-INDS",
    "COMPLEX-CALL-TO-PUSH",
    "COMPLEX-CALL-TO-RETURN",
    "COMPLEX-CALL-TO-TAIL-REC",
    "APPLY-TO-INDS",
    "APPLY-TO-PUSH",
    "APPLY-TO-RETURN",
    "APPLY-TO-TAIL-REC",
    
    /* 110 - 117 */
    "APPLY-METHOD-TO-INDS",
    "APPLY-METHOD-TO-PUSH",
    "APPLY-METHOD-TO-RETURN",
    "APPLY-METHOD-TO-TAIL-REC",
    "LEXPR-FUNCALL-WITH-MAPPING-TABLE-TO-INDS",
    "LEXPR-FUNCALL-WITH-MAPPING-TABLE-TO-PUSH",
    "LEXPR-FUNCALL-WITH-MAPPING-TABLE-TO-RETURN",
    "LEXPR-FUNCALL-WITH-MAPPING-TABLE-TO-TAIL-REC",
    
    /* 120 - 127 */
    "Return-N",
    "Return-List",
    "Return-NIL",
    "Return-T",
    "%Open-Catch",
    "%Open-Catch-Multiple-Value",
    "%Open-Catch-Tail-Recursive",
    "%Open-Catch-MV-List",
    
    /* 130 - 137 */
    "%Throw",
    "%Throw-N",
    "%Unwind-Protect-Continue",
    "*Unwind-Stack",
    "%Close-Catch",
    "%Close-Catch-Unwind-Protect",
    "Return-Pred",
    "Return-Not-Inds",
    
    /* 140 - 147 */
    "%enable-nupi-locking",
    "%disable-nupi-locking",
    NULL,
    NULL,
    "%RETURN-PAGE-CLUSTER",
    "%SCRUB",
    "%CREATE-PHYSICAL-PAGE",
    "%DELETE-PHT-ENTRY",
    
    /* 150 - 157 */
    "%GC-FREE-REGION",
    "%GC-FLIP",
    "%GC-SCAVENGE",
    "%GC-CONS-WORK",
    "%FLUSH-EXTRA-PDL",
    "%IMPORT-OBJECT",
    "%DISK-RESTORE",
    "%DISK-SAVE",
    
    /* 160 - 167 */
    "Long-BR-NULL-ELSE-POP",
    "Long-BR-NOT-NULL-ELSE-POP",
    "Long-BR-NULL",
    "Long-BR-NOT-NULL",
    "Long-BR-ATOM",
    "Long-BR-NOT-ATOM",
    "Long-BR-ZEROP",
    "Long-BR-NOT-ZEROP",
    
    /* 170 - 177 */
    "Long-BR-SYMBOLP",
    "Long-BR-NOT_SYMBOLP",
    NULL,
    NULL,
    "Long-BR-NULL-LIKELY",
    "Long-BR-NOT-NULL-LIKELY",
    "Long-BR",
    "Long-PUSHJ",
    
    /* 200 - 207 */
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    
    /* 210 - 217 */
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    
    /* 220 - 227 */
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    
    /* 230 - 237 */
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    
    /* 240 - 247 */
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    
    /* 250 - 257 */
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    
    /* 260 - 267 */
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    
    /* 270 - 277 */
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    
    /* 300 - 307 */
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    
    /* 310 - 317 */
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    
    /* 320 - 327 */
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    
    /* 330 - 337 */
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    
    /* 340 - 347 */
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    
    /* 350 - 357 */
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    
    /* 360 - 367 */
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    
    /* 370 - 377 */
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
};

int handle_mainop(lisp_q fef, int address, u16 opcode, struct instr *instr)
{
    int base;
    int offset;

    base = (opcode >> 6) & 7;
    offset = opcode & 0x3f;

    printf("%04x: %04hx %s\t", address, opcode, instr->name);

    if (strlen(instr->name) <= 4) printf("\t");
    if (strlen(instr->name) <= 12) printf("\t");
    
    switch (base) {
    case 0:
	printf("FEF[%d]\t", offset);
	dump_dtp(memread(fef + offset));
	break;
	
    case 1:
	printf("FEF[%d]\t", offset + 0x40);
	dump_dtp(memread(fef + offset + 0x40));
	break;
	
    case 2:
	printf("FEF[%d]\t", offset + 0x80);
	dump_dtp(memread(fef + offset + 0x80));
	break;
	
    case 5:
	printf("local[%d]\t", offset);
	dump_local_name(fef, offset);
	break;
	
    case 6:
	printf("arg[%d]\t", offset);
	dump_arg_name(fef, offset);
	break;
	
    case 7:
	printf("PDL-POP\n");
	break;
	
    default:
	printf("%d[#o%o]\n", (opcode >> 6) & 7, opcode & 0x3f);
	break;
    }

    return 0;
}

int handle_branchop(lisp_q fef, int address, u16 opcode, struct instr *instr)
{
    int target;

    target = opcode & 0xff;
    if (opcode & 0x100) target |= 0xffffff00;
    target += address + 1;
    
    printf("%04x: %04hx %s", address, opcode, instr->name);
    if (strlen(instr->name) <= 4) printf("\t");
    if (strlen(instr->name) <= 12) printf("\t");
    printf("\t%04hx\n", target);

    return 0;
}

int handle_immedop(lisp_q fef, int address, u16 opcode, struct instr *instr)
{
    u16 value;

    value = opcode & 0x1ff;
    
    printf("%04x: %04hx %s", address, opcode, instr->name);
    if (strlen(instr->name) <= 4) printf("\t");
    if (strlen(instr->name) <= 12) printf("\t");
    printf("%x\n", value);

    return 0;
}

int handle_unknownop(lisp_q fef, int address, u16 opcode, struct instr *instr)
{
    printf("%04x: %04hx %s #o%o\n", address, opcode,
	   instr->name? instr->name: "unknown", (opcode >> 9));

    return 0;
}

int handle_miscop(lisp_q fef, int address, u16 opcode, struct instr *instr)
{
    u16 miscop;

    miscop = opcode & 0x1ff;

    if (miscop_names[miscop]) {
	printf("%04x: %04hx miscop #o%o (%s)\n",
	       address, opcode, miscop, miscop_names[miscop]);
    } else {
	printf("%04x: %04hx unknown miscop #o%o\n", address, opcode, miscop);
    }

    return 0;
}

int handle_auxop(lisp_q fef, int address, u16 opcode, struct instr *instr)
{
    u16 auxop;

    auxop = opcode & 0x1ff;

    /*
     * auxops 0400 - 0777 have extremely regular structure, so we
     * handle them specially.
     */
    
    if ((auxop < 0400) && (auxop_names[auxop])) {
	printf("%04x: %04hx auxop #o%o (%s)",
	       address, opcode, auxop, auxop_names[auxop]);
	if (!strncasecmp(auxop_names[auxop], "LONG-", 5)) {
	    lisp_q tmp;

	    tmp = memread(fef + ((address + 1) >> 1));
	    if (!(address & 1)) tmp >>= 16;
	    printf("\t%04lx\n", tmp & 0xffff);

	    return 1;
	} else {
	    printf("\n");
	}
    } else if ((auxop >= 0400) && (auxop < 0700)) {
	printf("%04x: %04hx auxop #o%o (%s-%d)\n",
	       address, opcode, auxop,
	       /* I'm being a bit too cute here, but it works. */
	       &"UNBIND\0\0Pop-PDL\0Return"[(auxop >> 3) & 030],
	       (auxop & 077) + ((auxop & 0700) != 0600));
    } else {
	printf("%04x: %04hx auxop #o%o\n", address, opcode, auxop);
    }

    return 0;
}

int handle_arefiop(lisp_q fef, int address, u16 opcode, struct instr *instr)
{
    printf("%04x: %04hx arefiop #o%o (%s-AREFI (type %d) %d)\n",
	   address, opcode, (opcode & 0x1ff),
	   (opcode & 0x4000) ? "PUSH" : "TEST",
	   (opcode & 0x01c0) >> 6,
	   (opcode & 0x003f));

    return 0;
}

int handle_moduleop(lisp_q fef, int address, u16 opcode, struct instr *instr)
{
    printf("%04x: %04hx moduleop #o%o:%o\n", address, opcode,
	   (opcode & 0x1f8) >> 3, opcode & 7);

    return 0;
}

int (*instr_handlers[8])(lisp_q, int, u16, struct instr *) = {
    handle_unknownop, /* OPT_UNKNOWN */
    handle_mainop,    /* OPT_MAIN    */
    handle_branchop,  /* OPT_BRANCH  */
    handle_immedop,   /* OPT_IMMED   */
    handle_miscop,    /* OPT_MISC    */
    handle_auxop,     /* OPT_AUX     */
    handle_arefiop,   /* OPT_AREFI   */
    handle_moduleop,  /* OPT_MODULE  */
};

int disassemble_instr(lisp_q fef, int address, u16 opcode)
{
    struct instr *cur_instr;

    cur_instr = &instructions[opcode >> 9];

    return instr_handlers[cur_instr->type](fef, address, opcode, cur_instr);
}

/* EOF */
