
	title "Debugger Cold Start"
;
; A Small Debugger for the 65802/816
;
; Author:	A. Kossow     (AED)
; Version:	X1.0	(June 1984)
;
; Commands are of the form:
;
; <hex adr><cmd> or <cmd>
;
; Valid operators are:
;
; lf	open next seq wrd or byt loc
; ^	open prev seq wrd or byt loc
; /	reopen wrd loc last opened
; \	reopen byt loc last opened
; <adr>/	open new word loc
; <adr>\	open new byte loc
;
;
; ;b		reset breakpoints
; ;nb		reset breakpoint (0-3)
; <adr>;b	set breakpoint
; <adr>;nb	set breakpoint   (0-3)
;
; r		display processor & breakpoint registers
; ;r		reset registers to initial state
;
; <adr>;m	display 16 memory locations starting at <adr>
;		<lf> displays the next 16 locations
;		<cr> returns to cmd decoder
;
; <adr>;g	start program execution at <adr>
;
; <val>;a	load accumulator with <val>
; <val>;p	load program counter with <val>
; <val>;s	load stack pointer with <val>
; <val>;x	load x register with <val>
; <val>;y	load y register with <val>
; <val>;z	load direct register with <val>
; <val>;f	load processor status reg with <val>
; <val>;d	load data bank register with <val>
; <val>;k	load program bank register with <val>
;
; '		display current instruction
; s		single step instruction mode
; p		proceed from breakpoint
; h		<help>  dump command summary
;
; Note:		When actually in the debugger,  interrupts
;		are disabled. As a result,  communications
;		with the terminal are not interrupt-driven
;	
	title	" "
;
;
; Debugger memory layout:
;
;		debugger variables
; dpr-> dirpag>	------------------
;		------------------ 4 byte firewall
; sp ->	dbgstk>	------------------
; (dbgr)	debugger stack (64 bytes)
;		------------------ 4 byte firewall
; sp ->	inisp>	------------------
; (user)	test pgm stack
;

dirpg	equ	1		; psect of direct page data
code	equ	0		; psect of code

byte	equ	1		; length of a byte
word	equ	2		; length of a word
long	equ	3		; length of a long adr
quad	equ	4		; length of a quadbyte (32 bits)
				;
	title	"Debugger Entry Point"
	psect	code
;
;	debugger cold entry point
;
dbg::
	sei		; disable interrupts
	clc
	xce		; go to native mode

	rep	#0x30	; 16 bit memory / index

	lda	##dbgstk
	tcs		; run from our own stack

	pea	##dirpag
	pld		; run from our own direct page

	bsl	dbgini	; do cold start initialization

;
;	set up debugger direct page
;
	stz	<0	; first location
	ldx	##dirpag
	ldy	##dirpag+1
	lda	##dirlen-2
	mvn	>0,>0	; clear it out

	sep	#0x20		; 8 bit mem		

	phk			; set debugger data bank=pgm bank 
	plb
	phk			; also save it for future reference
	pla
	sta	<dbgpbk
;
;	print up startup msg and go
;
	per	vermsg
	bsl	putstr
	brl	rstregs		; jmps to cmd loop after rsting regs

	title	"Command Decoder"
;
;	Main command loop
;	Get a chr from the keyboard and do something with it
;
cmdloop:
	rep	#0x30		; 16 bit mem/idx
	stz	<daccum
	stz	<daccum+2	; clear out digit accumulator
	stz	<digcnt		; no digits in the accumulator

	lda	##0xff
	sta	<bptnum		; no special bp in mind yet

	per	prompt
	bsl	putstr
;
;	enter here if you don't want to change the digit accumulator
;	or output a prompt
cmdl1:
	bsl	getchr
;
;	enter here with a chr ready in the accumulator
cmdl2:
	sep	#0x20		; 8 bit mem
;
;	Scan for a valid command character...
;
	cmp	#'\'	;'\'
	bne	$1
;
;	\ = open currently selected address
;	    accesses will be one byte at a time
;
	lda	#1
	sta	<onebyt
	bra	$2

$1:	cmp	#'/'		; '/'
	bne	$3
;
;	/ = open currently selected address
;	    accesses will be two bytes at a time
;

	stz	<onebyt

$2:	
	lda	<digcnt		; new address entered?
	beq	$21		; nope, use old one

	rep	#0x20		; 16 bit memory
	lda	<daccum		; update open address
	sta	<curadr
	lda	<daccum+2
	sta	<curadr+2
$21:
	bsl	crlf
	brl	opencmd		; open the memory location

$3:
	cmp	#0x0a
	bne	$4
;
;	\n = increment currently open address by 1 or 2
;	     increment stays within a 65k boundary
;
	rep	#0x20
	inc	<curadr		; bump curadr to next byte
	lda	<onebyt		; need to bump again for word ?
	ror	a
	bcs	$31		; branch if no
	inc	<curadr		; now curadr points to next word

$31:
	bsl	crlf
	brl	opencmd

$4:
	cmp	#'^'		; '^'
	bne	$5
;
;	^ = decrement currently open address and examine it
;	    decrement stays within 65k boundary
;
	rep	#0x20
	dec	<curadr
	lda	<onebyt
	ror	a
	bcs	$41
	dec	<curadr
$41
	bsl	crlf
	brl	opencmd

$5:
	cmp	#';'
	bne	$6
;
;	; - commands starting with semi-colon
;	    ;r ;b ;#b ;a ;x ;y ;f ;s
;
	brl	semicmd
$6:
	cmp	#'s'
	bne	$7
;
;	s - single step (once)
;
	brl	scmd

$7:
	cmp	#'p'
	bne	$8
;
;	p - proceed from breakpoint
;
	brl	pcmd

$8:
	cmp	#'h'
	bne	$9
;
;	h - print the help message
;
	per	hlpmsg
	bsl	putstr
	brl	cmdloop

$9:
	cmp	#'r'
	bne	$10

;
;	r - print registers
;
	brl	rcmd

$10:
	cmp	#0xd
	bne	$11
;
;	<cr> - ignored
;
	brl	cmdloop

$11:
	cmp	#0x27
	bne	$12
;
;	' - disassemble current location
;
	bsl	dasm
	brl	cmdloop

$12:
	cmp	#'m'
	bne	$13
;
;	m - memory dump of cur adr
;
	brl	mcmd

$13:
	cmp	#'l'
	bne	$xx
;
;	l - load ram from parallel link
;
	bsl	lcmd
	brl	cmdloop

;
;	check for numeric input
;
$xx:
	bsl	ishex
	bcs	badchr		; not a digit, don't know what
				; it is
	bsl	xtol
	sep	#0x20		; 8 bit mem
	brl	cmdl1		; try for more chrs
;
;	echo a '?' for unknown commands
;
badchr:
	lda	#'?'		; echo ? for unknown cmds
	bsl	putchr
	brl	cmdloop		; carry set if not

	title	"Examine a Location"
;
; open a location specifed by open adr
;
opencmd:
	rep	#0x30		; 16 bit index and memory
	pei	<curadr+2	; get value for data bank reg
	pei	<curadr		; get address of data
	lda	[<curadr]	; get the data
	pha			; save it 
	pea	##6		; push arg byte count
	lda	<onebyt
	ror	a		; display data byte or word?
	bcs	$1		; branch if byte
	per	opnwfmt		; push format string pointer
	bra	$2		; for word opens
$1:	
	per	opnbfmt		; push open byte fmt pointer
$2:
	bsl	printf		; print dbr,addr,data
;
;	initialize the digit accumulator for a new value to store
;
	stz	<digcnt		; clear out digit count	
	stz	<daccum
	stz	<daccum+2	; reset digit accumulator
$3:
	bsl	getchr
	sep	#0x20
	cmp	#0xd		; <cr> = store current val
	bne	$4
	bra	$8		; all updates done below
$4:
	cmp	#0xa		; <lf> = store and incr open adr
	bne	$5
	bra	$8		; all updates done below
$5:
	cmp	#'^'		; <^> = store and decr open adr
	bne	$6
	bra	$8		; all updates done below
$6:
	cmp	#0x27		; '  = disassemble this location
	bne	$7
	bsl	dasm		; output opcode,operands
	brl	cmdl1		; go get next char

$7:	bsl	ishex
	bcs	$10		; nope
	bsl	xtol
	bra	$3
;
;	update the currently open location
;	if there are any digits in the digit accumulator
$8:
	pha			; save the character
	lda	<digcnt
	beq	$9		; no digits, just exit

	lda	<onebyt		; update byte or word ?
	bne	$81		; branch if byte
	rep	#0x20		; word - set 16 bit mem
$81	lda	<daccum		; get data byte/word
	sta	[<curadr]	; store it 

$9	sep	#0x20		; restore 8 bit mem
	pla			; restore character
$10:
	brl	cmdl2		; finish decoding in the main loop

	title	"Semi-colon Command Decoder"
;
; process a command with a semi colon
;
semicmd:
	bsl	getchr

	cmp	#'b'		; b-breakpoint
	bne	$2
	brl	bcmd
$2:
	cmp	#'m'		; m-dump mem
	bne	$3
	brl	mcmd
$3:
	cmp	#'g'		; g-go
	bne	$4
	brl	gcmd

$4:
	cmp	#'a'		; a-load acc
	bne	$5
;
	ldx	<daccum
	stx	<sac
	brl	cmdloop

$5:
	cmp	#'s'		; s-load sp
	bne	$6
;
	ldx	<daccum
	stx	<ssp
	brl	cmdloop

$6:
	cmp	#'x'		; x-load ix
	bne	$7
;
	ldx	<daccum
	stx	<six
	brl	cmdloop

$7:
	cmp	#'y'		; y-load iy
	bne	$8
;
	ldx	<daccum
	stx	<siy
	brl	cmdloop

$8:
	cmp	#'z'		; z-load dpr
	bne	$9
;
	ldx	<daccum
	stx	<sdp
	brl	cmdloop

$9:
	cmp	#'f'		; f-load psw
	bne	$10
;
	lda	<daccum
	sta	<sps
	brl	cmdloop

$10:
	cmp	#'d'		; d-load dbr
	bne	$11
;
	lda	<daccum
	sta	<sdb
	brl	cmdloop

$11	cmp	#'k'		; k-load pbr
	bne	$12
;
	lda	<daccum
	sta	<spb
	brl	cmdloop

$12:	
	cmp	#'p'		; p-load pc
	bne	$13
;
	ldx	<daccum
	stx	<spc
	brl	cmdloop

$13:
	cmp	#'r'		; r-reset registers
	bne	sem14
;
;	reset all registers to intial state
;
rstregs:
	rep	#0x20
	lda	##inisp-4
	sta	<ssp
	lda	##inipc
	sta	<curadr		; default current adr is code base
	stz	<curadr+2
	stz	<onebyt
	inc	<onebyt		; default to 1 byte open
	sta	<spc
	stz	<sdp
	stz	<six
	stz	<siy
	stz	<sac
	stz	<sdb
	stz	<spb
	sep	#0x20
	lda	#inipsw
	sta	<sps		; 8 bit a, 8 bit x, intr off
	brl	cmdloop
;
;	allow one digit from 0 -> 3 for a breakpoint
sem14:
	sec
	sbc	#'0'
	cmp	#5
	bge	$xx	

	sta	<bptnum
	brl	semicmd		; get the rest of the ';' cmd
;
;	unknown cmd, print a ?
$xx:
	lda	#'?'		; unrecognized cmd
	bsl	putchr
	brl	cmdloop		; all done

	title	"GO command"
;
;	;g	start processor execution
;
gcmd:
	sep	#0x20		; 8 bit memory
	bsl	crlf

	lda	<digcnt
	beq	pcmd		; no new adr specifed, use old one

	ldx	<daccum
	stx	<spc		; update pc
	ldx	<daccum+2
	stx	<spb		; drops into p cmd....
;
;	proceed from breakpoint
;
pcmd:
;
;	restore the process's registers
;
	rep	#0x20	; 16 bit memory
	tsc
	sta	<dbgsp	; save a copy of our stack pointer
	tdc
	sta	<dbgdpr	; and our direct page reg

	ldx	<six
	ldy	<siy	; restore index regs

	lda	<ssp
	tcs		; stack references to his stack now...

	lda	<spc
	sta	2,s	; load the updated pc back on the stack

	sep	#0x20	; 8 bit memory
	lda	<sdb
	pha
	plb		; load data bank reg
	lda	<spb
	sta	4,s	; load the updated pbr onto the stack
	lda	<sps
	sta	1,s	; load the updated psw onto the stack

	rep	#0x30
	lda	<sac	; restore his acc.
	pei	<sdp	; restore his dir page.
	pld
;
;
;	the last thing we do is set the pgm ctr and psw
;	by doing an rti from his stack
;	(the pc was updated to the correct spot by the
;	 brk exception routine)
;
	rti			; away we go.....

	title	"Register Display"
;
;	r	display register list
;
rcmd:
;
;	display header
	per	regmsg
	bsl	putstr
;
;	r cmd alternate entry
;	(doesn't print header)
rcmd1:
	rep	#0x30		; 16 bit memory/ index
;	
;	display	pc		<pc>
	pei	<spb
	pei	<spc
;
;	display accumulator	<ac>
	pei	<sac
;
;	display ix		<ix>
	pei	<sdb
	pei	<six
;
;	display	iy		<iy>
	pei	<sdb
	pei	<siy
;
;	display dir pg reg	<dp>
	pei	<sdp
;
;	display	stk ptr		<sp>
	pei	<ssp
;
;	display flags		<ps>
	pei	<sps

	pea	##20		; # arg bytes to printf
	per	regfmt		; format string pointer
	bsl	printf
;
;	display breakpoints	(0-4)

	ldx	##0
	ldy	##5		; number of bkpts and ss bpt

bploop:
	lda	<bkptbl+2,x	; get high byte bp
	bmi	bpl1		; negative means bp is valid
	per	offmsg
	bsl	putstr		; just print '------ '
	bra	bpl2
bpl1:
	pha
	lda	<bkptbl,x	; get low byte bp
	pha
	pea	##4
	per	addfmt
	bsl	printf
bpl2:
	inx
	inx
	inx
	inx			; point to next bp
	dey
	bne	bploop

	brl	cmdloop

	title	"Set/Reset Breakpoints"
;
;	;b	set/reset breakpoints
;
bcmd:
	lda	<digcnt
	beq	rstbp		; no digits means reset bpt(s)
;
;	set a breakpoint
;
	lda	<bptnum
	bpl	setbp		; a specific bp was asked for, do that one
;
;
	lda	#0		; default bp is zero
setbp:
	rep	#0x30		; 16 bit idx/ mem
	and	##0x0003	;

	asl	a
	asl	a		; multiply index by 4
	tax			; put adr into x
	lda	<daccum
	sta	<bkptbl,x	; set 2 low bytes
	lda	<daccum+2
	sta	<bkptbl+2,x	; set 2 high bytes

	sep	#0x20		; 8 bit memory
	lda	#0xff
	sta	<bkptbl+3,x	; set flag to indicate busy

	lda	#0		; BRK instruction
	sta	[<daccum]	; try to set the breakpoint
	lda	[<daccum]	; see if the breakpoint got set
 	beq	setb1		; branch if probably it did
	per	bpserr		; couldn't set bp - PROM or NXM.
	bsl	putstr
setb1:
	brl 	cmdloop
;
;	reset a breakpoint
;
rstbp:
	lda	<bptnum
	bpl	clrbpt		; no bkpt num means reset all bps

	rep	#0x30		; 16 bit memory and idx
	stz	<bkptbl+0	
	stz	<bkptbl+2	; bkp 0
	stz	<bkptbl+4
	stz	<bkptbl+6	; bkp 1
	stz	<bkptbl+8
	stz	<bkptbl+10	; bkp 2
	stz	<bkptbl+12
	stz	<bkptbl+14	; bkp 3

	brl	cmdloop

clrbpt:
	rep	#0x30	; set 16 bit mem/ idx
	and	##3	; clear all but bp tbl idx

	asl	a	; mult index * 4 (size of bkpt)
	asl	a
	tax
	stz	<bkptbl,x
	stz	<bkptbl+2,x

	brl	cmdloop

	title	"Memory Dump"
;
;	;m	memory dump
;	dump memory starting at [<daccum]
;

mcmd:	
	lda	<digcnt		; new address entered ?
	beq	$0		; branch if no.

	ldx	<daccum		; move new addr to <curadr.
	stx	<curadr
	ldx	<daccum+2
	stx	<curadr+2
;
;	display address of mem to be dumped.
;
$0:
	pei	<curadr+2	; push dump bank.
	pei	<curadr		; push dump address.
	pea	##4		; push arg byte count.
	per	addfmt		; push ptr to format string.
	bsl	printf		; display bank,address.

	sep	#0x30		; set 8 bit M/X
	ldx	#16		; and a count of 16
	ldy	#0
$1:
	lda	[<curadr],y	; get a byte
	bsl	btox		; print in hex
	bsl	putsp		; a blank
	iny			; point to next byte
	dex			; decrement count
	bne	$1		; loop for all 16

	bsl	putsp		; output 2 spaces
	bsl	putsp
	ldx	#16		; reload count
	ldy	#0
$2:
	lda	[<curadr],y	; get the chr
	and	#0x7f		; mask parity bit
	cmp	#0x20		; check for printable
	bge	$3		; >= space ?
	lda	#'.'		; no, print dot
$3:	cmp	#0x7f		; del char ?
	blt	$4		; no
	lda	#'.'
$4:	bsl	putchr

	iny			; adv ptr
	dex			; dcr count
	bne	$2		; loop back while > 0

	rep	#0x30
	lda	<curadr		; get the adr
	clc
	adc	##16
	sta	<curadr		; move cur adr up by 16
	brl	cmdloop

	title	"Single Step"
;
; s - do single stepping
;
; this is painful on a 816 since each instruction
; must be disassembled to determine where to place
; the next break.
;
scmd:
	rep	#0x30		; 16 bit M/X
	lda	[<spc]		; get instruction at pc
	and	##0xff		; mask high byte
	asl	a		; convert to word offset
	tay			; put offset in y
	per	opctyp		; put base adr of tbl on stack
	lda	(1,s),y		; get entry for this opcode
	plx			; pop base off stack
;
;	the accumulator has the index to the opcode name
;	and the operand type
;
	and	##0xff		; mask off opname index
	tay			; offset into instruction length tbl
	per	opclen		; base of opcode length table
	lda	(1,s),y		; get the length of the instruction
	and	##0xff
	plx			; pop base off stack
	bit	##0x80		; check for IMM instructions
	beq	chk_bra		; branch if not IMM
;
;	calculate the length of an immediate instruction
;	based on the state of the M (or X) bits in the PSW
;
	sep	#0x20		; 8 bit memory
	bit	#0x40		; X or M?
	bne	$2		; chk X if set

	lda	<sps		; get saved PSW
	bit	#0x20		; M = 1?
	bne	$1		; yup, 8 bit immed
	lda	#3		; else 16 bit immed
	bra	$3

$1	lda	#2		; 8 bit immed
	bra	$3

$2	lda	<sps		; get saved PSW
	bit	#0x10		; X = 1?
	bne	$1		; yup, 8 bit immed
	lda	#3

$3	rep	#0x20		; reset 16 bit memory
	brl	set_ss		; set the new bp
;
;	check if the instruction is a branch
;
chk_bra:
	pha			; save offset
	sep	#0x20
	lda	[<spc]		; get opcode
	cmp	#0x82		; brl?
	bne	$1		; nope
;
;	brl - set up 16 bit offset
;
	rep	#0x20		; 16 bit memory
	ply			; throw away old offset
	ldy	##1	
	lda	[<spc],y	; get the offset for brl
	inc	a
	inc	a
	inc	a		; acc += 3
	brl	set_ss

;
;	check for the 8 bit branches, determine
;	whether it will be taken, and what the
;	offset is (YUK)
;

$1:
	cmp	#0x80		; bra (easy one)
	beq	$01

	cmp	#0x90		; bcc ?
	bne	$2
	lda	<sps		; get psw
	bit	#1		; carry clear?
	beq	$01		; yes.. branch
	brl	$nobr

$01:	brl	$br		; branch extension..

$2:	cmp	#0xb0		; bcs ?
	bne	$3
	lda	<sps
	bit	#1		; carry set?	
	bne	$br		; yes.. branch
	brl	$nobr

$3:	cmp	#0xf0		; beq ?
	bne	$4
	lda	<sps
	bit	#2		; zero set?
	bne	$br		; yes.. branch
	bra	$nobr

$4:	cmp	#0x30		; bmi ?
	bne	$5
	lda	<sps
	bit	#0x80		; neg set ?
	bne	$br		; yes.. branch
	bra	$nobr

$5:	cmp	#0xd0		; bne ?
	bne	$6
	lda	<sps
	bit	#2		; zero clear?
	beq	$br		; yes.. branch
	bra	$nobr

$6:	cmp	#0x10		; bpl ?
	bne	$7
	lda	<sps
	bit	#0x80		; neg clear ?
	beq	$br		; yes.. branch
	bra	$nobr

$7:	cmp	#0x50		; bvc ?
	bne	$8
	lda	<sps
	bit	#0x40		; overfl clear?
	beq	$br		; yes.. branch
	bra	$nobr

$8:	cmp	#0x70		; bvs ?
	bne	$9
	lda	<sps
	bit	#0x40		; overfl set ?
	bne	$br		; yes.. branch
	bra	$nobr
;
;	opcode wasn't a branch
;	make sure it isn't a (per/bra) (per/brl) (bsr/bsl)
;
$9:
	cmp	#0x62		; per?
	bne	$11		; nope...
;
;	check if the NEXT instruction is a brl/bra
;	if it is, this is really a bsl/bsr
;
	ldy	##3
	lda	[<spc],y	; get NEXT opcode
	cmp	#0x80		; bra?
	bne	$10
;
;	it WAS a bra.. so set the offset to 5
;
	rep	#0x20		; 16 bit memory
	plx			; throw away old offset
	lda	##5		; next instruction 5 bytes away
	bra	set_ss

$10:
	cmp	#0x82		;brl?
	bne	$11		;nope... use orig offset
;
;	it WAS a brl.. so set the offset to 6
	rep	#0x20		; 16 bit memory
	plx			; throw away old offset
	lda	##6		; next instruction 6 bytes away
	bra	set_ss

;
;	process offset
$11:	rep	#0x20		; 16 bit memory
	pla			; restore offset
	bra	set_ss
;
;	branch not taken, just add 2
;
$nobr:
	rep	#0x20		; 16 bit memory
	plx			; throw away old offset
	lda	##2
	bra	set_ss
;
;	branch taken, calculate a sign-extended offset
;
$br:
	ldy	##1
	lda	[<spc],y	; get the offset
	rep	#0x20		; reset 16 bit memory
	plx			; throw away old offset
	and	##0xff		; mask high byte
	bit	##0x80		; sign extend?
	beq	$br1		; nope
	ora	##0xff00
$br1:
	inc	a		; add size of branch to offset
	inc	a
;
;	add the current pc to the instruction offset
;	and use that as the spot where the next ss brkpt
;	is to go
;
;	Note: we currently do NOT follow branches or jmps
;
set_ss:
	clc
	adc	<spc		; add offset to current val
	sta	<bkptbl+16	; set ss adr to next instruction
	sep	#0x20		; 8 bit memory/ 16 bit idx
	lda	[<bkptbl+16]	; get the opcode at that adr
	sta	<bkpval+4	; save the opcode value
	lda	#0		; brk instruction
	sta	[<bkptbl+16]	; try to set the break
	lda	[<bkptbl+16]	; see if we did
	beq	$1		; branch if (probably) successful
	per	bpserr		; couldn't set bp, ROM or nxm
	bsl	putstr
$1:
	rep	#0x30
	brl	pcmd		; start it up with an implied 'p'

	title	"Binary Conversions"
;
; wtox - convert 16 bit accumulator to ascii
; convert 16 bit integer to ascii hex
; and output to aux port
;
wtox:
	php
	rep	#0x20	; set 16 bit memory
	pha		; save accum
	xba		; get high byte
	bsr	btox
	lda	1,s	; get low byte
	bsr	btox
	pla		; restore accumulator
	plp
	rts

;
; btox - convert 8 bit accumulator to ascii
;
; convert 8 bit integer to ascii hex
; and output to aux port
;
;
btox:
	php		; save memory mode
	rep	#0x20	; set to 16 bit memory
	pha		; save all 16 bits of accumulator
	ror	a	; get high nibble of low byte
	ror	a
	ror	a
	ror	a
	bsr	ntox
	lda	1,s	; get another copy of the accumulator
	bsr	ntox
	pla		; restore accumulator
	plp		; restore memory modes
	rts
	eject
;
; ntox()
; low four bits of accumulator to hex digit
;
; the contents of the accumulator are trashed
;
ntox:
	php
	rep	#0x30		; 16 bit memory and index
	phy
	and	##0xf		; clear all but low 4 bits
	tay
	per	hextbl		; push adr of hex table
	lda	(1,s),y		; get the ascii chr
	bsl	putchr
	ply			; subtract 2 from stack
	ply			; restore callers y
	plp			; restore memory modes
	rts

	title	"I/O Routines"
;
; putstr(string);
; output null terminated string to aux port
;
; Note:
;  You can't call this routine from a jsl instruction
;  as it expects the return address to be 2 bytes long
;
putstr:
	php		; save incoming context
	rep	#0x30	; 16 bit mem, idx
	phx
	lda	6,s	; get starting adr
	tax		; put it into x
	sep	#0x20	; 16 bit idx, 8 bit mem

	lda	0,x	; get the chr
	bne	$1	; print {null} if null string

	per	nulmsg
	plx		; set to point to {null} msg

$1:	lda	0,x	; fetch the chr, 0 -> done
	beq	$2

	bsl	putchr

	inx		; point to next chr
	bra	$1

$2:
	rep	#0x20	; 16 bit memory
	plx		; restore x

	lda	2,s	; get ret adr
	sta	4,s	; wipe out argument
	lda	0,s	; get psw to restore
	sta	2,s	; store psw at adr below new ret adr 3(sp)
	pla		; move stack up 2 bytes
	plp		; restore memory/idx modes
	rts		
	eject
;
;	crlf
;	output a cr/lf to the aux port
;
crlf:
	php		; save memory mode
	sep	#0x20	; set to 8 bits
	pha		; save accumulator
	lda	#0xd	; output <cr>
	bsl	putchr
	lda	#0xa	; and    <lf>
	bsl	putchr	; print chr
	pla		; restore accumulator
	plp		; restore psw
	rts		; exit
;
;	putsp()
;	output a single space
;
putsp:
	php
	sep	#0x20
	lda	#' '
	bsl	putchr
	plp
	rts

	title	"Hex to Binary Conversions"
;
;	test for a hex digit (0-9 & a-f)
;	carry set if it's not hex
;
ishex:
	cmp	#'0'
	blt	$2		; not hex if < 0
	cmp	#':'
	blt	$1		; is  hex if <= 9
	cmp	#'a'
	blt	$2		; not hex if < a
	cmp	#'g'
	bge	$2		; not hex if > f
$1:
	clc			; it's hex
	rts
$2:
	sec			; it's not hex
	rts
	eject
;
; convert ascii chr to hex digit
; digit stored in 32 bit digit accumulator "daccum"
;
xtol:
	php
	sep	#0x20
;
;	first convert ASCII hex to AED modified hex
;
	cmp	#0x61		; is it a-f?
	blt	$1		; nope
	sec
	sbc	#0x27		; convert a -> :
$1:
	rep	#0x30
;
;	convert modified hex to binary
;
	and	##0x0f
	pha			; save new low nibble
;
;	shift long <daccum 4 times
;
	lda	<daccum
$2:
	asl	a		; bit 15 -> carry, 0 -> bit 0
	rol	<daccum+2	; carry -> bit 16
	asl	a		; bit 15 -> carry, 0 -> bit 0
	rol	<daccum+2	; carry -> bit 16
	asl	a		; bit 15 -> carry, 0 -> bit 0
	rol	<daccum+2	; carry -> bit 16
	asl	a		; bit 15 -> carry, 0 -> bit 0
	rol	<daccum+2	; carry -> bit 16

	ora	1,s		; OR in new low nibble.
	sta	<daccum
	pla			; clean up stack

	sep	#0x20
	inc	<digcnt		; daccum has another valid digit
	plp
	rts
	end
