
;
; ACE interrupt service routines and queueing routines.
; Each ACE that uses interrupts is assumed to have
; an associated queue control block (qcb).  A pointer
; to the qcb is at a known location in the kernel RAM.
;
; A qcb looks like this :
;
;aceoff  ds 1 	; offset (from f80000) of the ACE.
;aceidx	 ds 1	; aceoff/8 (for indexing words related to Qcbs).
;intmsk	 ds 1	; interrupt enable mask (in Iemask).
;choked  ds 1	; bit 7 set if output is inhibited (XOFF received),
		; bit 6 set if input is inhibited (XOFF sent).
;iocntl  ds 1	; bit 7 set if XOFF/XON handshake,
		; bit 6 set if RTS/CTS.
		; bit 5 set means allow input buffer to overflow,
		; else discard input to a full buffer.
;icmask	 ds 1	; mask for zapping non-data bits in input
;qerror	 ds 1	; error reporting bits.
;qercnt	 ds 1	; error interrupt counter.
;iqptr   ds 3	; input buffer (FIFO) address and bank (0).
;iqhead  ds 1	; iqptr + iqhead is address of first char in buffer.
;iqtail  ds 1	; iqptr + iqtail is address of next free space.
;iqsize  ds 1	; size of the input buffer, in bytes.
;iqhigh  ds 1	; high water mark - choke sender when q this full.
;iqlow   ds 1	; low water mark - unchoke sender when q this full.
;oqptr   ds 3	; output buffer (FIFO) address and bank (0).
;oqhead  ds 1	; oqptr + oqhead is address of first char in buffer.
;oqtail  ds 1	; oqptr + oqtail is address of next free space.
;oqsize  ds 1	; size of the output buffer, in bytes.
;
; The queues work as follows : 
; Get next byte from buffer+head, then advance head.
; Put next byte at buffer+tail, then advance tail.  
; Queue is empty if head=tail, full if head=tail+1.
;
;
; Kernel globals :
; Irqdon - address of normal interrupt exit routine - 
;	   restores environment of interrupted task.
; Qcbac0-3 - pointers to qcbs.
; Iqcnt0-3 - negative byte count of ace input queues.
;
; Here on ACE interrupts.  On entry, dbr = 0 and mem/idx are
; 16 bit.  Each ACE vectors to a unique address where the
; pointer to its qcb is loaded and its interrupt enable mask
; is stacked.  Then code common to all the ACEs is entered. 
; This code disables further interrupts from the ACE and then 
; goes about servicing the interrupt.  The service routine does
; a cli as soon as is reasonable. This permits higher priority
; interrupts to be serviced without introducing re-entrancy 
; problems.  On exit the code re-enables interrupts for the
; ACE and jumps to irqdon.
;

XON	equ	0x11
XOFF	equ	0x13

;
; Offsets within an ACE of the ACE registers.
;
DLL	equ	0		; Divisor Latch Low (baud rate).
DLH	equ	2		; Divisor LAtch High.
IER	equ	2		; Interrupt Enable reg.
IIR	equ	4		; Interrupt Id reg.
LCR	equ	6		; Line Control reg.
MCR	equ	8		; Modem Control reg.
LSR	equ	10		; Line Status Reg.
MSR	equ	12		; Modem Status reg.

;
; ACE status/control bits.
;

RTS	equ	2		; in MCR
CTS	equ	16		; in MSR

;
; LSR status bits.
;

THRE	equ	32		; Transmitter empty
DR	equ	1		; Data ready
OE	equ	2		; Overrun error
FE	equ	8		; Framing error
PE	equ	4		; Parity error
BI	equ	16		; Break interrupt

ace0int:
	lda	Qcbac0		; get pointer to qcb
	bra	acecom		; branch to service routine
ace1int:
	lda	Qcbac1		; get pointer to qcb
	bra	acecom		; branch to service routine
ace2int:
	lda	Qcbac2		; get pointer to qcb
	bra	acecom		; branch to service routine
ace3int:
	lda	Qcbac3		; get pointer to qcb
	bra	acecom		; branch to service routine
;
; Common interrupt service routine for ACEs.
;

acecom:
	tcd			; Access qcb with direct addressing.

	sep	#0x30		; Set 8 bit mem/idx.
	lda	<intmsk		; Get intpt enable mask.
	trb	Iemask+1	; Disable interrupts from this ACE
				; (we'll do a CLI soon and don't 
				; want re-entrancy headaches).
	ldx	<aceoff		; Get offset this ACE.
	lda	>ACE,IIR,x	; Read ACE interrupt id register.

;
; After servicing an interrupt, branch back 
; here if this ACE has more interrupts pending.
;

more:
	cmp	#4		; Data ready interrupt ?
	bne	$1
	brl	recvint		; Yup.
$1:	cmp	#2		; Transmit holding reg empty intpt ?
	bne	$2
	brl	threint		; Yup.
$2:	cmp	#6		; Error or break interrupt ?
	bne	$3
	brl	acerror		; Yup.
$3:	brl	msrint		; No, must be modem status intpt.

;
; Here after servicing an interrupt.  See if more
; interrupts pending, service them if so.
;

done:
	cli			; recvint may not have done this.
	lda	>ACE,IIR,x	; Read ACE interrupt id register.
	bit	#1		; Interrupt pending ?
	bne	$1		; br if no.
	sei
	brl	more		; Yes, go service it.
$1:
	lda	<intmsk		; Get intpt enable mask.
	tsb	Iemask+1	; Re-enable intpts from this ACE.
	jmp	Irqdon		; Jump to normal interrupt exit.

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; Here on ACE error interrupt.	;
; Copy any set error bits to	;
; the low byte of <qerror.	;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

acerror
	lda	>ACE,LSR,x	; Read line status reg.
	cli
	and	#OE+PE+FE+BI	; Mask off non-error bits.
	tsb	<qerror		; Copy error bits for whoever cares.
	inc	<qercnt		; inc error count.
;
; Should probably disable lsr interrupt if count
; greater than some threshold
;	
	brl	done

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; Here on Modem status interrupt.	;
; See if CTS changed - that's the	;
; only modem status we care about.	;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

msrint:
	lda	>ACE,MSR,x	; Read modem status register.
	bit	#1		; CTS change interrupt ?
	bne	$1		; Yes, record new CTS value.
	brl	done		; No, ignore interrupt.
$1:
	and	#16		; Get current CTS signal value.
	bne	$2		; We are clear to send, unchoke 
				; output.
	lda	#128		; We got choked, set our
	tsb	<choked		; output inhibited bit
	brl	done		; Return from interrupt.
$2:
	cli
	brl	unchoke		; CTS came back, record it


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; Here on THRE interrupt.	;
; If output not choked and	;
; something to send, send it.	;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

threint:
	cli
	bit	<choked		; Output inhibited ?
	bpl	$1		; Branch if no.
	brl	done		; Yes, just return.
$1:
	brl	dqout		; Send next byte, if q not empty,
				; and jump to more.

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; Here on data ready interrupt.		;
; Read byte from ACE, put it in the	;
; input q, see if we need to choke or	;
; or unchoke the sender, etc.		;
; If it's the keyboard, and data is 	;
; 0xfe, set flag Setupk.		;
; If it's the keyboard, and data is	;
; 0xeb (control break), set beak in 	;
; host serial interface for 400 ms.	;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

recvint:

	lda	>ACE,0,x	; Get byte.
	and	<icmask		; Strip parity bits.
	bit	<intmsk		; talking to keyboard ?
	bpl	$10		; br if no, check control break.
	cmp	#0xfe		; yes, is it setup key ?
	bne	$5		; br if no.
	dec	Setupk		; yes, set flag and q the byte.
	bra	$10
$5:	cmp	#0xeb		; control break ?
	bne	$10
;
; control break hit at keyboard, set
; break bit for 400 milliseconds.
; NOTE - this timing is way wrong
; if the code is running in prom.
; Also we're assuming the host asynch
; is the one at 200020.
; The procedure used here is recommended in 
; the INS8250a data sheet.
;

$w1	lda	>ACE,0x20+LSR		; wait for host THRE.
	bit	#THRE
	beq	$w1
	lda	#0
	sta	>ACE,0x20		; send a null.
$w2	lda	>ACE,0x20+LSR		; wait for THRE.
	bit	#THRE
	beq	$w2
	lda	>ACE,0x20+LCR		; set break.
	ora	#0x40
	sta	>ACE,0x20+LCR

	rep	#0x10			; delay about 400 ms.
	ldy	##5
	ldx	##0
$loop:	dex
	bne	$loop		; 81.920 ms
	dey
	bne	$loop		; *5

	sep	#0x30
	lda	>ACE,0x20+LCR		; clear break.
	and	#0xff-0x40
	sta	>ACE,0x20+LCR
	brl	done			; rti
;
; Check for XOFF/XON, if pertinent.
;

$10:	bra	$2		; not yet pertinent.

	bit	<iocntl		; XOFF/XON if bit 7 set.
	bpl	$2		; Branch if no XON/XOFF.

	cmp	#XON
	bne	$1
	cli
	brl	unchoke		; Go enable output from outq
				; and jump to more.
$1:
	cmp	#XOFF
	bne	$2
	cli
	lda	#128		; We got choked, set our
	tsb	<choked		; output inhibited bit
	brl	done		; Return from interrupt.
$2:

;
; Not XOFF/XON, q the (stacked) byte if there's room.
;
	ldy	<iqtail		; Get pointer to end of q.
	phy			; Save it for later.
	iny			; Make pointer to new end.
	cpy	<iqsize		; Past end of buffer ?
	bne	$11		; Nope.
	ldy	#0		; Yes, wrap to start.
$11:	cpy	<iqhead		; Q full ? (head = tail+1)
	beq	$iqful		; Branch if yes.
;
; Put the byte in the q.
;
	sty	<iqtail		; Save new tail pointer.
	ply			; Get current tail pointer.
	sta	(<iqptr),y	; Put it in the q.

	ldx	<aceidx
	rep	#0x20
	dec	Iqcnt0,x
	sep	#0x20
	ldx	<aceoff
;
; See if we need to send XOFF or remove RTS.
;
	bit	<iocntl		; Using any handshakes ?
	bmi	$21		; Branch if yes.
	bvs	$21		; Ditto.
	brl	done		; No, we're done.
$21:
	bit	<choked		; Did we already shut 'em up ?
	bvc	$22		; Branch if no.
	brl	done		; Yes, we're done.
$22:
;
; See if # bytes queued is >= iqhigh.
; If yes, choke sender with xoff or
; -rts.
;
	sec
	lda	<iqtail
	sbc	<iqhead
	bcs	$23
	adc	<iqsize
$23:
	cmp	<iqhigh
	bcs	$24
	brl	done			
$24:
	lda	#64
	tsb	<choked
	bit	<iocntl
	bpl	$26
$25:
	lda	>ACE,LSR,x
	bit	#32
	beq	$25	
	lda	#XOFF
	sta	>ACE,0,x
	brl	done
$26:
	lda	>ACE,MCR,x
	and	#255-RTS
	sta	>ACE,MCR,x
	brl	done	
;
; Here if no more room in input q.  Either start a new q
; with this (stacked) byte or just chuck it.
;
	
$iqful:
	cli
	pha			; Save input byte.
	lda	<iocntl
	bit	#32
	bne	$newq
	pla			; Pop input byte.
	pla			; Pop old q tail.
	brl	done		; Return from interrupt.
$newq:

;
; Make an empty q and put the (stacked) byte in it.
; Remember at this point y is one past the old tail
; (on the stack).
;
	pla			; Get the byte.
	sty	<iqtail		; Save new tail.
	ply			; Get old tail.
	sty	<iqhead		; Make it the new head.
	sta	(<iqptr),y	; Q the byte.
	ldx	<aceidx		; set the byte count 
	rep	#0x20		; for this q to -1.
	lda	##0xffff
	sta	Iqcnt0,x
	sep	#0x20
	ldx	<aceoff
	brl	done		; That's it, return from intpt.



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; This routine jumped to when XON received or CTS	;
; deactivated.  Clear output inhibit flag in qcb 	;
; and restart output if possible.  Enter at dqout	;
; from THRE interrupt service routine.			; 
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

unchoke:
	lda	#128		; Clear our output inhibit flag.
	trb	<choked

	lda	>ACE,LSR,x	; Read line status register.
	bit	#THRE		; THR empty ?
	bne	$1		; Yes, go send something.
	brl	done		; No, can't send, we're done.
$1:				; Fall into send routine.

;
; Enter here from THRE service routine.
;

dqout:

;
; See if output q is empty.
;
	ldy	<oqhead		; Get index q head.
	cpy	<oqtail		; Head = tail ?
	bne	$1		; No, go send if possible.
	brl 	done		; Yup, q empty, we're done.
$1:
;
; Output q not empty, get next byte
; and send it, then update q pointer.
;

	lda	(<oqptr),y	; Get char to send.
	sta	>ACE,0,x	; Send it.
;
; Point oqhead at next char to send.
;
	iny			; Generate pointer to next byte.
	cpy	<oqsize		; Pointing past end of buffer ?
	bne	$2		; Not yet.
	ldy	#0		; Yes, wrap to beginning of buffer.
$2:	sty	<oqhead		; Save pointer to next byte.
	brl	done		; Thats it, go see if more intpts.



getkey:
	sec
	php
	rep	#0x20
	lda	>0,Iqcnt3
	bpl	$done
	lda	>0,Qcbac3
	pha
	jsl	>0,Getinq
	bcs	$done
	plp
	clc
	rtl
$done:
	plp
	rtl

gethst:
	sec
	php
	rep	#0x20
	lda	>0,Iqcnt2
	bpl	$done
	lda	>0,Qcbac2
	beq	$done
	pha
	jsl	>0,Getinq
	plp
	clc
	rtl
$done:	plp
	rtl
	
sndhst:
	php
	rep	#0x30
	pha
	lda	>0,Qcbac2
	beq	$done
	pha
	lda	3,s
	jsl	>0,Putoutq
$done:	pla
	plp
	rtl

break:
	rtl

;
; (De)Queueing routines.
; putoutq - put a byte into an output q.  Return with
;           carry set if q was full.
; getinq - get a byte from an input q.  Return with
;          carry set if q was empty.
; Enter these routines with pointer to qcb
; on the stack.
; Byte parameters are passed in the accumulator.
; All registers are preserved.  The qcb pointer is
; removed from the stack by these routines.
;

;
; Put a byte into an output q.
;  Example puts null terminated string in output
;  q for ace 1.
;
;	rep	#0x10		; 16 bit idx
;	sep	#0x20		; 8 bit mem
;	ldy	##0		; init index into string
;$loop:	lda	(<string),y	; get next string char
;	beq	$done		; null byte terminates string
;	iny			; point to next string char
;	ldx	>0,Qcbac1	; get address of qcb for ace 1
;$1:	phx			; push for call to Putoutq
;	jsl	>0,Putoutq	; q the char for output
;	bcc	$loop		; branch to do next string char
;	bra	$1		; carry set if q was full, try again
;$done:
;


putoutq:
	clc			; assume q not full
	php
	rep	#0x30
	phd
	pha
	lda	9,s		; Get pointer to qcb.
	tcd			; Make qcb the direct page.

	lda	7,s		; Move stack up to remove qcb ptr.
	sta	9,s
	lda	5,s
	sta	7,s
	pla
	sta	3,s
	phx
	phy
;
; Now stack looks like 
;
;		10,s 	ret addr,bank
;		9,s	psw
;		7,s 	a
;		5,s	d
;		3,s	x
;		1,s	y
;
; If the q is empty and output not choked and THR is
; empty, don't bother to q the byte, just send it,
; else q it.  If the first 2 conditions are met, but not
; the third, interrupts are disabled while the byte is
; being queued.
;

	sep	#0x30
	php			; Save IRQ status.
	ldx	<oqtail		; Get pointer to end of q.
	txy			; Save for later.
	cpx	<oqhead		; head = tail ?
	bne	$qit		; No, q not empty.
	bit	<choked		; Output allowed ?
	bmi	$qit		; No, q the byte.
;
; Send the byte, unless THR not empty.
;
	ldx	<aceoff		; Get ACE address.
	sei			; DISABLE INTERRUPTS
	lda	>ACE,LSR,x	; Read ACE line status reg.
	bit	#THRE		; Trans holding reg empty ?
	beq	$qit		; No, go q the byte.
	lda	7+1,s		; Yes, get the byte to send.
	sta	>ACE,0,x	; Send it.
	plp			; Restore IRQ status.
	bra	$done		; That's it, return.

;
; q the byte
;
	
$qit:
	tyx			; Get q tail index.
	inx			; Make pointer to new tail.
	cpx	<oqsize		; Past end of buffer ?
	bne	$1		; Nope.
	ldx	#0		; Yes, wrap to start.
$1:	cpx	<oqhead		; Q full ? (head = tail+1)
	bne	$2		; Branch if no.
	plp			; RESTORE IRQ status.
	lda	#1		; Yes, get mask to set carry.
	ora	9,s		; Or with callers psw.
	sta	9,s		; Save callers psw with carry set.
	bra	$done		; return
$2:

;
; Put the byte in the q
;
	lda	7+1,s		; Get the byte.
	sta	[<oqptr],y	; Put it in the q.
	stx	<oqtail		; Save new tail pointer.
	plp			; RESTORE IRQ status.

$done:
	rep	#0x30
	ply
	plx
	pld
	pla
	plp
	rtl

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; Get next byte from an input q.	;
; Call with 				;
;	push	qcbptr			;
;	jsl	>0,Getinq		;
; qcbptr is removed from stack before	;
; return.				;
; Increments Iqcnt for the qcb.		;
; All registers preserved.		;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
		
getinq:
	clc			; assume q not empty
	php
	rep	#0x30
	phd
	lda	7,s		; Get pointer to qcb.
	tcd			; Make qcb the direct page.

	lda	5,s		; Move stack up to remove qcb ptr.
	sta	7,s
	lda	3,s
	sta	5,s
	txa
	sta	3,s
	phy
;
; Now stack looks like 
;
;		8,s 	ret addr,bank
;		7,s	psw
;		5,s 	x
;		3,s	d
;		1,s	y

	sep	#0x30
	ldy	<iqhead		; See if q is empty.
	cpy	<iqtail
	bne	$1		; Branch if not empty.
	lda	#1		; Set carry in callers psw
	ora	7,s		; to indicate q empty error.
	sta	7,s
	bra	$done		; That's it, return.
$1:
	lda	[<iqptr],y	; Get byte at head of q.
	pha			; Save it for exit.

	iny			; Update <iqhead.
	cpy	<iqsize
	bne	$11
	ldy	#0
$11:
	sty	<iqhead
;
; Inc Iqcnt for this qcb.
;
	ldx	<aceidx
	php				; save m and i.
	rep	#0x20
	sei				; disable interrupts.
	lda	>0,Iqcnt0,x
	inc	a
	sta	>0,Iqcnt0,x
	plp				; cli sep 20.

	ldx	<aceoff
;
; If we choked sender, see if q now at low water mark.
; If it is, unchoke sender.
;
	bit	<choked			; Input choked ?
	bvc	$almost	;done		; No,return the dequeued byte
$2:
	sec
	lda	<iqtail			; See how full q is.
	sbc	<iqhead
	bcs	$3			; Negative value ?
	adc	<iqsize			; Add q size to correct.
$3:	cmp	<iqlow			; Down to low water mark ?
	bcs	$almost ;done		; No,return the dequeued byte
;
; Unchoke with XON or RTS, depending on the
; handshake indicated by ioctl.
;
	lda	#64			; Get 'input choked' mask.
	trb	<choked			; Clear input choked flag.
	bit	<iocntl			; See what handshake we're using.
	bpl	$rts			; Not XON, must be RTS.
;
; Send XON.  It would be better to leave the 
; choked flag set if output q is full - waiting
; in that event doesn't seem neccessary.
;
	lda	#XON
$xon:	phd				; push pointer to qcb
	jsl	>0,Putoutq		; put byte in output q
	bcs	$xon
	bra	$almost ;done		; return the dequeued byte
;
; Assert RTS.
;
$rts:	
	ldx	<aceoff			; Get ACE address.
	lda	>ACE,MCR,x		; Read modem control reg.
	ora	#RTS			; Set RTS bit.
	sta	>ACE,MCR,x		; Turn on RTS, and return.

$almost:
	pla				; Retrieve dequeued byte.
$done:
	rep	#0x30
	ply
	pld
	plx
	plp
	rtl
qend:
	end
