/*
 hwminit.c
 hwmhrst.c
 hwmtask.c
 hwmutil.c
 hwmptcl.c
 hwmintr.c
 hwmdlvr.c
 hwmscam.c
 hwmse2.c
 hwmdiag.c
*/

/***************************************************************************
*                                                                          *
* Copyright 1995,1996,1997,1998,1999 Adaptec, Inc.,  All Rights Reserved.  *
*                                                                          *
* This software contains the valuable trade secrets of Adaptec.  The       *
* software is protected under copyright laws as an unpublished work of     *
* Adaptec.  Notice is for informational purposes only and does not imply   *
* publication.  The user of this software may make copies of the software  *
* for use with parts manufactured by Adaptec or under license from Adaptec *
* and for no other use.                                                    *
*                                                                          *
***************************************************************************/

/***************************************************************************
*
*  Module Name:   HWMINIT.C
*
*  Description:
*                 Codes to initialize hardware management module
*
*  Owners:        ECX IC Firmware Team
*    
*  Notes:         NONE
*
*  Entry Point(s):
*                 SCSIHGetConfiguration
*                 SCSIHGetMemoryTable
*                 SCSIHApplyMemoryTable
*                 SCSIHInitialize
*                 SCSIHSetupHardware
*
***************************************************************************/

#define  SCSI_DATA_DEFINE

#include "scsi.h"
#include "hwm.h"

/*********************************************************************
*
*  SCSIHGetConfiguration
*
*     Get default configuration information
*
*  Return Value:  0 - configuration available
*                 others - failure
*                  
*  Parameters:    hhcb
*
*  Remarks:       THis routine requires scsiRegister valid in
*                 order to access the hardware. This routine
*                 may also access pci configuration space in 
*                 order to collect configuration information.
*
*********************************************************************/
int SCSIHGetConfiguration (SCSI_HHCB SCSI_HPTR hhcb)
{
   SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_UEXACT8 hcntrl;

   /* For Cardbus (APA-1480x), we need to turn off the Power Down mode */
   /* before we can access to any chip's registers.  However, we have  */
   /* clear the CHIPRST bit as well.  Therefore, we save the CHIPRST   */
   /* state in our HHCB for later reference. */
   hcntrl = OSD_INEXACT8(SCSI_AICREG(SCSI_HCNTRL));
   hhcb->SCSI_HF_chipReset = hcntrl & SCSI_CHIPRESET;

   /* Turn off Power Down mode and clear CHIPRST bit if and only if */
   /* the chip is in Power Down mode. */
   if (hcntrl & SCSI_POWRDN)
   {
      hcntrl &= ~(SCSI_POWRDN | SCSI_CHIPRESET);
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_HCNTRL), hcntrl);
   }

   /* set appropriate function pointers depending on mode of operation */
   /* the mode of operation is combination of firmware mode and compilation */
   /* mode (BIOS does not reference function pointer to save space) */
   SCSIhSetupEnvironment(hhcb);

   /* get hardware revision level and firmware (e.g. sequencer) */
   /* version number */
   /* check if chip reset bit asserted. psuse chip if reset was cleared */
   /* set initialization required flag if it was set */
   /* check if pause chip necessary */

   if (!(hcntrl & SCSI_PAUSEACK))
   {
      SCSIhPauseAndWait(hhcb);      
   }
   SCSIhPrepareConfig(hhcb);

   /* get configuration information which is dependent on firmware */
   /* mode of operation */
   SCSI_hGETCONFIGURATION(hhcb);

   /* get current hardware configuration */
   SCSIhGetHardwareConfiguration(hhcb);

   if (!(hcntrl & SCSI_PAUSEACK))
   {
      SCSI_hWRITE_HCNTRL(hhcb,scsiRegister,hcntrl);
   }

   OSD_SYNCHRONIZE_IOS(hhcb);

   return(0);
}

/*********************************************************************
*
*  SCSIHGetMemoryTable
*
*     This routine will collect memory requirement information and 
*     fill the memory table.
*
*  Return Value:  none
*                  
*  Parameters:    hhcb
*                 memoryTable
*
*  Remarks:       This routine may get called before/after 
*                 SCSIHGetConfiguration. It requires firmwareMode set
*                 with intended operating mode before calling
*                 this routine.
*
*                 This routine should not assume harwdare can be
*                 accessed. (e.g. scsiRegister may not be valid
*                 when this routine get called.
*                 
*********************************************************************/
#if !SCSI_DOWNSHIFT_MODE
void SCSIHGetMemoryTable (SCSI_UEXACT8 firmwareMode,
                          SCSI_UEXACT8 numberScbs,
                          SCSI_MEMORY_TABLE SCSI_HPTR memoryTable)
{
   /* assign memory type */                
   memoryTable->memory[SCSI_HM_QOUTPTRARRAY].memoryType = SCSI_MT_HPTR;
   memoryTable->memory[SCSI_HM_SCBBFRARRAY].memoryType = SCSI_MT_IPTR;
   memoryTable->memory[SCSI_HM_ACTIVEPTRARRAY].memoryType = SCSI_MT_HPTR;
   memoryTable->memory[SCSI_HM_SHARECONTROL].memoryType = SCSI_MT_HPTR;

   /* assign memory category */
   memoryTable->memory[SCSI_HM_QOUTPTRARRAY].memoryCategory = SCSI_MC_LOCKED;
   memoryTable->memory[SCSI_HM_SCBBFRARRAY].memoryCategory = SCSI_MC_LOCKED;
   memoryTable->memory[SCSI_HM_ACTIVEPTRARRAY].memoryCategory = SCSI_MC_UNLOCKED;
   memoryTable->memory[SCSI_HM_SHARECONTROL].memoryCategory = SCSI_MC_UNLOCKED;

   /* assign alignment */
   memoryTable->memory[SCSI_HM_QOUTPTRARRAY].memoryAlignment =
                  (SCSI_UEXACT8) 0xFF;
   memoryTable->memory[SCSI_HM_SCBBFRARRAY].memoryAlignment = 
                  (SCSI_UEXACT8)SCSI_hALIGNMENTSCBBUFFER(firmwareMode);
   memoryTable->memory[SCSI_HM_ACTIVEPTRARRAY].memoryAlignment =
                  (SCSI_UEXACT8)(sizeof(SCSI_HIOB SCSI_IPTR)-1);
   memoryTable->memory[SCSI_HM_SHARECONTROL].memoryAlignment = 0x03;

   /* assign granularity */                
   memoryTable->memory[SCSI_HM_QOUTPTRARRAY].granularity = 0;
   memoryTable->memory[SCSI_HM_SCBBFRARRAY].granularity =
                  SCSI_hSIZEOFSCBBUFFER(firmwareMode);
   memoryTable->memory[SCSI_HM_ACTIVEPTRARRAY].granularity = 0;
   memoryTable->memory[SCSI_HM_SHARECONTROL].granularity = 0;

   /* assign size and minimum size */
   memoryTable->memory[SCSI_HM_QOUTPTRARRAY].memorySize = 
      sizeof(SCSI_QOUTFIFO_NEW) * ((SCSI_UEXACT16) numberScbs + 1);
   if ((firmwareMode == SCSI_FMODE_SWAPPING64) ||
       (firmwareMode == SCSI_FMODE_SWAPPING32) ||
       (firmwareMode == SCSI_FMODE_SWAPPING_ADVANCED) ||
       (firmwareMode == SCSI_FMODE_SWAPPING_160M) || SCSI_SCBBFR_BUILTIN)
   {
      memoryTable->memory[SCSI_HM_SCBBFRARRAY].memorySize = 
                     numberScbs * SCSI_hSIZEOFSCBBUFFER(firmwareMode);
   }
   memoryTable->memory[SCSI_HM_ACTIVEPTRARRAY].memorySize = 
                  sizeof(SCSI_HIOB SCSI_IPTR) * numberScbs;

   /* assign minimum size must be matched */
   memoryTable->memory[SCSI_HM_QOUTPTRARRAY].minimumSize = 
                  memoryTable->memory[SCSI_HM_QOUTPTRARRAY].memorySize; 
   memoryTable->memory[SCSI_HM_SCBBFRARRAY].minimumSize = (SCSI_UEXACT16)
                  (SCSI_MINSCBS * SCSI_hSIZEOFSCBBUFFER(firmwareMode));
   memoryTable->memory[SCSI_HM_ACTIVEPTRARRAY].minimumSize =
                  memoryTable->memory[SCSI_HM_ACTIVEPTRARRAY].memorySize;
   memoryTable->memory[SCSI_HM_SHARECONTROL].minimumSize = 
                  memoryTable->memory[SCSI_HM_SHARECONTROL].memorySize;

#if SCSI_GROUP_DEVICES
   memoryTable->memory[SCSI_HM_SHARECONTROL].memorySize = sizeof(SCSI_SHARE);
#endif
}
#endif

/*********************************************************************
*
*  SCSIHApplyMemoryTable
*
*     This routine will apply memory pointers described in memory table.
*
*  Return Value:  0 - memory table applied
*                 1 - memory table not applicable
*                  
*  Parameters:    hhcb
*                 memoryTable
*
*  Remarks:       Memory pointers will be applied and saved in 
*                 associated hhcb. After this call the memory table 
*                 is not required (e.g. can be free) from now on.
*                 
*                 Memory pointers in memory table must be setup
*                 properly to satify the memory requirement 
*                 before this routine get called.
*
*********************************************************************/
#if !SCSI_DOWNSHIFT_MODE
SCSI_INT SCSIHApplyMemoryTable (SCSI_HHCB SCSI_HPTR hhcb,
                                SCSI_MEMORY_TABLE SCSI_HPTR memoryTable)
{
#if SCSI_SWAPPING_MODE || SCSI_SCBBFR_BUILTIN
   SCSI_UEXACT8 numberScbs;
#endif /* SCSI_SWAPPING_MODE || SCSI_SCBBFR_BUILTIN */
   SCSI_INT status = 0;
   
   /* set memory pointer for shared control for group devices */
#if SCSI_GROUP_DEVICES
   hhcb->shareControl = (SCSI_SHARE SCSI_HPTR)
            memoryTable->memory[SCSI_HM_SHARECONTROL].ptr.hPtr;
#endif   

   /* set memory pointer for queue out pointer array */
   hhcb->SCSI_QOUT_PTR_ARRAY = (void SCSI_HPTR)
            memoryTable->memory[SCSI_HM_QOUTPTRARRAY].ptr.hPtr;

   /* set memory pointer for active pointer array */
   hhcb->SCSI_ACTIVE_PTR_ARRAY = (SCSI_HIOB SCSI_IPTR SCSI_HPTR)
            memoryTable->memory[SCSI_HM_ACTIVEPTRARRAY].ptr.hPtr;

   /* set memory pointer for scb buffer array */
#if SCSI_SWAPPING_MODE || SCSI_SCBBFR_BUILTIN
   hhcb->SCSI_SCB_BUFFER.virtualAddress = (void SCSI_HPTR)
            memoryTable->memory[SCSI_HM_SCBBFRARRAY].ptr.hPtr;
   numberScbs = (SCSI_UEXACT8) (memoryTable->memory[SCSI_HM_SCBBFRARRAY].memorySize /
                                SCSI_hSIZEOFSCBBUFFER(hhcb->firmwareMode));
   if (numberScbs < hhcb->numberScbs)
   {
      if (numberScbs >= SCSI_MINSCBS)
      {
         hhcb->numberScbs = numberScbs;
      }
      else
      {
         status = 1;
      }
   }
#endif

   return(status);
}
#endif

/*********************************************************************
*
*  SCSIHInitialize
*
*     Initialize hardware based on configuration information 
*     described 
*
*  Return Value:  return status
*                 0 - initialization successful
*                  
*  Parameters:    hhcb
*
*  Remarks:       Usually this routine is called after SCSIHGetConfiguration,
*                 SCSIHGetMemoryTable and SCSIHSetMemoryTable. It requires 
*                 both configuration and memory available in order to 
*                 initialize hardware properly.
*                 
*********************************************************************/
int SCSIHInitialize (SCSI_HHCB SCSI_HPTR hhcb)
{
   SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
#if (SCSI_STANDARD_160M_MODE || SCSI_SWAPPING_160M_MODE)
   SCSI_UEXACT8 scsiSfunct;
   SCSI_UEXACT8 optionMode;
#endif /* (SCSI_STANDARD_160M_MODE || SCSI_SWAPPING_160M_MODE) */
#if (SCSI_AICBAYONET || ( SCSI_AIC78XX && (SCSI_EXCALIBUR_DAC_WORKAROUND == 2)))
   SCSI_UEXACT32 devConfig;
#endif

   SCSI_hWRITE_HCNTRL(hhcb,scsiRegister,SCSI_PAUSE);

   /* workaround for hanging of sequencer immediately after a pause */
   /* due to hardware stretch                                       */
   SCSI_hSEQSTRETCHWORKAROUND(hhcb);

   /* set MRDCEN bit to 0 */
#if SCSI_AICBAYONET
   if (hhcb->hardwareMode == SCSI_HMODE_AICBAYONET) 
   {  
      devConfig = OSD_READ_PCICFG_EXACT32(hhcb,SCSI_DEVCONFIG);
      OSD_WRITE_PCICFG_EXACT32(hhcb,SCSI_DEVCONFIG,devConfig & (~SCSI_MRDCEN));
   }  
#endif   
   
/* Workaround for Excalibur generating false DAC command after retrying  */
/* commands on PCI bus                                                   */
#if SCSI_AIC78XX   
#if SCSI_EXCALIBUR_DAC_WORKAROUND == 2
   /* set MRDCEN bit to 1 */
   if ((hhcb->deviceID & SCSI_ID_MASK) == SCSI_EXCALIBUR_BASE)
   {
      devConfig = OSD_READ_PCICFG_EXACT32(hhcb,SCSI_DEVCONFIG);
      OSD_WRITE_PCICFG_EXACT32(hhcb,SCSI_DEVCONFIG,devConfig | SCSI_MRDCEN);
   }
#endif /* SCSI_EXCALIBUR_DAC_WORKAROUND */
#endif /* SCSI_AIC78XX */

   /* enable access to external scb ram for Viking or Athena */
   /* and standard mode only */
   SCSI_hCONFIGURESCBRAM(hhcb);

   /* Moved to here from SCSIhInitializeHardware routine because the auto    */
   /* rate/offset features before calling SCSIhResetChannelHardware routine. */
   /* Set up some performance improvement options for Trident chip. */
   /* And these options must be set inorder for the standard and    */
   /* swapping 160m sequencer code to run.                          */
#if (SCSI_STANDARD_160M_MODE || SCSI_SWAPPING_160M_MODE)
   if (hhcb->hardwareMode == SCSI_HMODE_AICTRIDENT)
   {
      /* switch the chip to alternate mode for accessing optionmode reg. */
      scsiSfunct = OSD_INEXACT8(SCSI_AICREG(SCSI_SFUNCT));
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SFUNCT), (scsiSfunct | SCSI_ALT_MODE));
      optionMode = OSD_INEXACT8(SCSI_AICREG(SCSI_OPTIONMODE));

      /* Enable Auto Rate and Offset, Auto ACK, Bus Free Interrupt Revision, */
      /* Expected Phase Disable, SCSIDATL Image Register feature.            */
      /* ATN Management feature is not available yet.                        */
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_OPTIONMODE), optionMode |
          (SCSI_AUTORATEEN | SCSI_AUTOACKEN | SCSI_BUSFREEREV |
           SCSI_EXPPHASEDIS | SCSI_SCSIDATL_IMGEN));
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SFUNCT), scsiSfunct);
   }
#endif /* (SCSI_STANDARD_160M_MODE || SCSI_SWAPPING_160M_MODE) */

   /* download sequencer code and reset channel regardless initialization 
      flag is set or not. */
   /* setup sequencer for execution */
   if (SCSI_hSETUPSEQUENCER(hhcb))
      return(-1);

   /* reset channel hardware */
   SCSIhResetChannelHardware(hhcb);

#if SCSI_TARGET_OPERATION
   /* Initialize the node table ptrs
    * Note; don't put inside SCSI_hRESETSOFTWARE as
    * initializing the node table is done only once.
    */
   if (hhcb->SCSI_HF_targetMode)
   {
      SCSIhTargetInitializeNodeTbl(hhcb);
#if SCSI_MULTIPLEID_SUPPORT
      /* Initialize the DISABLE_ID field */
      hhcb->SCSI_HP_disableHiob = SCSI_NULL_HIOB;
#endif /* SCSI_MULTIPLEID_SUPPORT */
   }
#endif /* SCSI_TARGET_OPERATION */

   /* initialize hardware/data structures which are dependent on */
   /* mode of operation */
   SCSI_hRESETSOFTWARE(hhcb);

   /* initialize hardware */
   SCSIhInitializeHardware(hhcb);

   /* check if reset scsi necessary */
   if (hhcb->SCSI_HF_resetSCSI)
   {
      if (SCSIhResetScsi(hhcb))
      {
         return(-1);
      }

      /* Delay user specified ammount after scsi reset for slow devices to settle down. */
      SCSI_hRESETDELAY(hhcb, (hhcb->resetDelay*2));
   }
   else
   {
      /* Due to the Bayonet HW that needs to wait a total of 200ms for the   */
      /* scsi bus to stable after the chip reset we must wait for the scsi   */
      /* bus to become stable based on bit ENAB20 or ENAB40 that is set by   */
      /* the HW after the bus becomes stable */
      SCSI_hWAITFORBUSSTABLE(hhcb);
   }

   /* start to run sequencer */
   SCSIhStartToRunSequencer(hhcb);

   OSD_SYNCHRONIZE_IOS(hhcb);

   return(0);
}
                     
/*********************************************************************
*
*  SCSIhStandardConfigureScbRam
*
*     Configure scb ram
*
*  Return Value:  1 - external scb ram was enabled
*                 0 - external scb ram was not enabled
*                  
*  Parameters:    hhcb
*
*  Remarks:
*                  
*********************************************************************/
#if SCSI_STANDARD_MODE
int SCSIhStandardConfigureScbRam (SCSI_HHCB SCSI_HPTR hhcb)
{
   return(SCSIHEnableExtScbRam(hhcb));   
}
#endif

/*********************************************************************
*
*  SCSIhSetupEnvironment
*
*     Setup execution environment
*
*  Return Value:  none
*                  
*  Parameters:    hhcb
*
*  Remarks:
*                  
*********************************************************************/
void SCSIhSetupEnvironment (SCSI_HHCB SCSI_HPTR hhcb)
{
   /* setup hardware descriptor */
   SCSI_hSETUPHARDWAREDESC(hhcb);

   /* setup firmware descriptor */
   SCSI_hSETUPFIRMWAREDESC(hhcb);
}

/*********************************************************************
*
*  SCSIHSetupHardware
*
*     Setup hardware execution environment
*
*  Return Value:  none
*                  
*  Parameters:    hhcb
*
*  Remarks:
*                  
*********************************************************************/
void SCSIHSetupHardware (SCSI_HHCB SCSI_HPTR hhcb)
{
   /* setup hardware descriptor */
   SCSI_hSETUPHARDWAREDESC(hhcb);
}

/***************************************************************************
*
*  SCSIhPrepareConfig
*
*     This routine will prepare before collecting configuration information
*
*  Return Value:  none
*                  
*  Parameters:    hhcb
*
*  Remarks:       This routine should only be referenced at initialization
*                 time and it's part of initialization block.
*
***************************************************************************/
void SCSIhPrepareConfig (SCSI_HHCB SCSI_HPTR hhcb)
{
   SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_UEXACT32 subSysSubVendID;
   SCSI_UEXACT32 hostID;
   SCSI_UEXACT8 i;

   subSysSubVendID = OSDReadPciConfiguration(hhcb,SCSI_SUBSYS_SUBVEND);

   /* Check subVendor ID to turn on any OEM support flag */
   if(((SCSI_UEXACT16)subSysSubVendID & 0x0000FFFF) ==  SCSI_OEM1_SUBVID)
   {
      hhcb->OEMHardware = SCSI_OEMHARDWARE_OEM1;
   }
   else
   {
      /* this is standard hardware */
      hhcb->OEMHardware = SCSI_OEMHARDWARE_NONE;
   }
   
   if (hhcb->hardwareMode == SCSI_HMODE_AICBAYONET) /* bayonet class device */
   {
      hostID = OSDReadPciConfiguration(hhcb,SCSI_ID_REG);
      hhcb->deviceID = (SCSI_UEXACT16) (hostID >> 16);
      hhcb->SCSI_HP_aicBayonet = 1;

      /* type field is 0 or 1(host adapter). Get defaults from subsystem id */
      if (((hostID & 0x000F0000) == 0) || ((hostID & 0x000F0000) == 1)) 
      {
         hhcb->SubSystemIDValues.u16 =  (SCSI_UEXACT16) (subSysSubVendID >> 16);
      }
      /* type field must be 0x0f at this point since all others were */
      /* screened out at SCSIGetNextHostDeviceType used hardcoded defaults. */
      else 
      {
         hhcb->SCSI_SUBID_maxScsiRate = SCSI_ULTRA_2;
         hhcb->SCSI_SUBID_autoTerm = 1;   /* default to OFF, "0" means ON */
         hhcb->SCSI_SUBID_legacyConnector = 0; /* @@ appropriate default ?? */
         hhcb->SCSI_SUBID_seepromType = SCSI_NO_SEEPROM;
         hhcb->SCSI_SUBID_scsiChannels = SCSI_ONE_CHNL;
         hhcb->SCSI_SUBID_pciBusWidth = 0; /* @@ is 32 bit appropriate default ?? */ 
         hhcb->SCSI_SUBID_scsiBusWidth = 1; /* @@ is wide appropriate default ?? */ 
         /* if multi-function bit is turned on in device id */
         hhcb->SCSI_SUBID_multifunction = ((hostID & 0x00400000) == 0x00400000);
         hhcb->SCSI_SUBID_seDiff = 1;  /* @@ appropriate default ?? */
      }
   }
   /* trident class device */
   else if (hhcb->hardwareMode == SCSI_HMODE_AICTRIDENT)
   {
      hostID = OSDReadPciConfiguration(hhcb,SCSI_ID_REG);
      hhcb->deviceID = (SCSI_UEXACT16) (hostID >> 16);

      /* Disable disconnect for HDD on 1960 Disti board. */ 
      if (hhcb->deviceID == SCSI_19160_DISTI)
      {
         hhcb->SCSI_HP_disconnectOffForHDD = 1;
      } 

      /* type field is 0 or 1 (host adapter). Get defaults from subsystem id */
      if ((((hostID & 0x000F0000) == 0) || ((hostID & 0x000F0000) == 1)) && 
          (hhcb->OEMHardware != SCSI_OEMHARDWARE_OEM1))
      {
         hhcb->SubSystemIDValues.u16 =  (SCSI_UEXACT16) (subSysSubVendID >> 16);
      }
      /* type field must be 0x0f at this point since all others were */
      /* screened out at SCSIGetNextHostDeviceType used hardcoded defaults. */
      else 
      {
#if SCSI_PPR_ENABLE
         hhcb->SCSI_SUBID_maxScsiRate = SCSI_ULTRA_160;
#else
         hhcb->SCSI_SUBID_maxScsiRate = SCSI_ULTRA_2;
#endif
         hhcb->SCSI_SUBID_autoTerm = 1;   /* default to OFF, "0" means ON */
         hhcb->SCSI_SUBID_legacyConnector = 0; /* @@ appropriate default ?? */
         hhcb->SCSI_SUBID_seepromType = SCSI_NO_SEEPROM;
         hhcb->SCSI_SUBID_scsiChannels = SCSI_ONE_CHNL;
         hhcb->SCSI_SUBID_pciBusWidth = 0; /* @@ is 32 bit appropriate default ?? */ 
         hhcb->SCSI_SUBID_scsiBusWidth = 1; /* @@ is wide appropriate default ?? */ 
         /* if multi-function bit is turned on in device id */
         hhcb->SCSI_SUBID_multifunction = ((hostID & 0x00400000) == 0x00400000);
         hhcb->SCSI_SUBID_seDiff = 1;  /* @@ appropriate default ?? */
      }
   }

#ifdef SCSI_AIC78XX
   else /* legacy AIC-78xx device */
   {
      if (subSysSubVendID != 0)
      {
         hhcb->deviceID = (SCSI_UEXACT16) (subSysSubVendID >> 16);
         SCSIhGetCapability(hhcb);
      }
      else
      {
         /* must swap the bytes for non-subsystem and non-subvendor ID hardware */
         hhcb->deviceID = (SCSI_UEXACT16) OSD_INEXACT8(SCSI_AICREG(SCSI_DEVID0));
         hhcb->deviceID <<= 8;
         hhcb->deviceID |= (SCSI_UEXACT16) OSD_INEXACT8(SCSI_AICREG(SCSI_DEVID1));

         SCSIhGetCapability(hhcb);
      }
   }
#endif /* SCSI_AIC78XX */

   hhcb->firmwareVersion = (SCSI_UEXACT8) SCSI_hFIRMWARE_VERSION(hhcb);
   hhcb->softwareVersion = (SCSI_UEXACT8) SCSI_SOFTWARE_VERSION;
   hhcb->hardwareRevision = (SCSI_UEXACT8)
                        OSD_READ_PCICFG_EXACT32(hhcb, SCSI_DEV_REV_ID);

   /* set the delay time after a SCSI bus reset */
#if SCSI_TARGET_OPERATION
   if (hhcb->SCSI_HF_initiatorMode)
   {
#endif /* SCSI_TARGET_OPERATION */
      /* use initiator mode default */
      hhcb->resetDelay = SCSI_RESET_DELAY_DEFAULT;
#if SCSI_TARGET_OPERATION
   }
   else
   {
      /* normally no delay would be required when operating in 
       * target mode as the target is expected to respond within
       * 250ms after a reset
       */
      hhcb->resetDelay = 0;
   }
#endif /* SCSI_TARGET_OPERATION */
 
   /* check if chip reset bit asserted. Pause chip if reset was cleared */
   /* set initialization required flag if it was set */
   if (hhcb->SCSI_HF_chipReset)
   {
      /* Set InitNeeded will either use default or information */
      /* from SEEPROM */
      hhcb->SCSI_HP_initNeeded = 1;
      hhcb->SCSI_HF_resetSCSI = 1;

      for (i=0;i<16;i++)
      {
         SCSI_DEVICE_TABLE(hhcb)[i].SCSI_DF_resetSCSI = 1;
      }
   }
}

/* only the legacy need this routine */
#ifdef SCSI_AIC78XX
/***************************************************************************
*
*  SCSIhGetCapability
*
*     This routine will try to put in the information to match new ID 
*     definition for the old vendor ID 9004 devices
*
*  Return Value:  none
*                  
*  Parameters:    hhcb
*
*  Remarks:       This routine should only be referenced at initialization
*                 time and it's part of initialization block.
*
***************************************************************************/
void SCSIhGetCapability (SCSI_HHCB SCSI_HPTR hhcb)
{
   SCSI_UEXACT16 idValue;
   SCSI_HOST_ADDRESS SCSI_LPTR hostAddress;

   idValue = hhcb->deviceID & SCSI_ID_MASK;

   /* get ultraEnable configuration */
#if SCSI_4944IDFIX_ENABLE
   if (idValue == SCSI_LANCE_BASE || idValue == SCSI_DAGGER1_BASE ||
      hhcb->deviceID == 0x783B || /* 4944 ID fix */
      hhcb->deviceID == 0x78EC )  /* 4944 ID fix */
#else
   if (idValue == SCSI_LANCE_BASE || idValue == SCSI_DAGGER1_BASE )
#endif
   {   
      hhcb->SCSI_SUBID_maxScsiRate = SCSI_FAST;
   }
   else
   {   
      hhcb->SCSI_SUBID_maxScsiRate = SCSI_ULTRA;
   }

   /* default to C06C46 except raid hardware (Viking,Athena and ARO) */
#if SCSI_4944IDFIX_ENABLE
   if (hhcb->deviceID == SCSI_VIKING_LANCE ||
      hhcb->deviceID == SCSI_VIKING_KATANA ||
      hhcb->deviceID == SCSI_BAYONET_LANCE ||
      hhcb->deviceID == SCSI_BAYONET_KATANA ||
      hhcb->deviceID == SCSI_ARO_EXCALIBUR ||
      hhcb->deviceID == 0x783B || /* 4944 ID fix */
      hhcb->deviceID == 0x78EC )  /* 4944 ID fix */
#else
   if (hhcb->deviceID == SCSI_VIKING_LANCE ||
      hhcb->deviceID == SCSI_VIKING_KATANA ||
      hhcb->deviceID == SCSI_BAYONET_LANCE ||
      hhcb->deviceID == SCSI_BAYONET_KATANA ||
      hhcb->deviceID == SCSI_ARO_EXCALIBUR )
#endif
   {
      hhcb->SCSI_SUBID_seepromType = SCSI_SEE_TYPE1;
   }
   else
   {
      hhcb->SCSI_SUBID_seepromType = SCSI_SEE_TYPE0;
   }


   hhcb->indexWithinGroup = 0;
   hostAddress = OSD_GET_HOST_ADDRESS(hhcb);

   /* get number of SCSI channels information */
   switch(hhcb->deviceID)
   {
      case SCSI_VLIGHT_LANCE:
      case SCSI_VLIGHT_KATANA:
         hhcb->SCSI_SUBID_scsiChannels = SCSI_TWO_CHNL;
         if (hostAddress->pciAddress.deviceNumber != 4)
         {
            hhcb->indexWithinGroup = 1;
         }
         break;

      case SCSI_VIKING_LANCE:
      case SCSI_VIKING_KATANA:
         hhcb->SCSI_SUBID_scsiChannels = SCSI_THREE_CHNL;
         switch (hostAddress->pciAddress.deviceNumber)
         {
            case 1:
            case 4:
               /* based at the first section */
               break;

            case 2:
            case 8:
               /* based at the second section */
               hhcb->indexWithinGroup = 1;
               break;

            case 3:
            case 12:
               /* based at the third section */
               hhcb->indexWithinGroup = 2;
               break;
         }
         break;

      case SCSI_BAYONET_LANCE:
      case SCSI_BAYONET_KATANA:
#if SCSI_4944IDFIX_ENABLE
         case 0x78EC: /* 4944 ID fix */
         case 0x783B: /* 4944 ID fix */
#endif
         hhcb->SCSI_SUBID_scsiChannels = SCSI_FOUR_CHNL;
         break;

      default:
         hhcb->SCSI_SUBID_scsiChannels = SCSI_ONE_CHNL;
         break;
   }

   if((hhcb->deviceID & SCSI_ID_MASK) == SCSI_EXCALIBUR_BASE)
   {
      hhcb->SCSI_SUBID_scsiChannels = SCSI_TWO_CHNL;
   }

   /* Set default autoterm to be disable */
   hhcb->SCSI_SUBID_autoTerm = 1;

   /* Excalibur is multi-function device */
   if (idValue == SCSI_EXCALIBUR_BASE)
   {
      hhcb->SCSI_SUBID_multifunction = 1;
   }
   else
   {
      hhcb->SCSI_SUBID_multifunction = 0;
   }
   
   /* Set to no legacy connectors */
   hhcb->SCSI_SUBID_legacyConnector = 0;

   /* Setup differential device */
   switch (hhcb->deviceID)
   {
      case 0x7874:
      case 0x7875:
      case 0x7876:
      case 0x7884:
      case 0x7885:
         hhcb->SCSI_SUBID_seDiff = 1;
         break;

      default:
         hhcb->SCSI_SUBID_seDiff = 0;
         break;
   }
   
}
#endif /* SCSI_AIC78XX */

/***************************************************************************
*
*  SCSIhStandardGetConfig
*
*     Get configuration for standard 64 and standard 32 mode
*
*  Return Value:  none
*                  
*  Parameters:    hhcb
*
*  Remarks:       This routine should only be referenced at initialization
*                 time and it's part of initialization block.
*
***************************************************************************/
#if SCSI_STANDARD_MODE
void SCSIhStandardGetConfig (SCSI_HHCB SCSI_HPTR hhcb)
{
   hhcb->numberScbs = SCSIhGetMaxHardwareScbs(hhcb);
   hhcb->numberHWScbs = hhcb->numberScbs;
   if (hhcb->numberScbs == SCSI_MAXSCBS)
   {
      hhcb->SCSI_HF_multiTaskLun = 1;
   }

   /* get configuration common to all modes */
   SCSIhGetCommonConfig(hhcb);
}
#endif 

/*********************************************************************
*
*  SCSIhGetMaxHardwareScbs
*
*     Calculate Initialize hardware
*
*  Return Value:  number of hardware scbs available
*                  
*  Parameters:    hhcb
*
*  Remarks:       At return of this routine sequencer is guaranteed
*                 to be paused
*                  
*********************************************************************/
#if SCSI_STANDARD_MODE
int SCSIhGetMaxHardwareScbs (SCSI_HHCB SCSI_HPTR hhcb)
{
   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_UEXACT32 scbData = 0;
   SCSI_UEXACT8 numberScbs = SCSI_MAXSCBS;
   SCSI_UEXACT8 scbptr;
   SCSI_UEXACT8 blocks;
   int i;
   int wasEnabled;
   SCSI_UEXACT8 hcntrl;

   /* check if pause chip necessary */
   if (!((hcntrl = OSD_INEXACT8(SCSI_AICREG(SCSI_HCNTRL))) & SCSI_PAUSEACK))
   {
      SCSIhPauseAndWait(hhcb);      
   }

   /* enable external scb ram */
   wasEnabled = SCSIHEnableExtScbRam(hhcb);

   scbptr = OSD_INEXACT8(SCSI_AICREG(SCSI_SCBPTR)); /* Save current scb pointer         */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCBPTR), numberScbs);  /* Load scb pointer with FF         */

   for (i = 0; i < 4; i++)             
   {                                   /* Save current contents            */
      scbData |= ((SCSI_UEXACT32) OSD_INEXACT8(SCSI_AICREG(SCSI_SCB_BASE+i))) << (i * 8);
   }

   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+0),0xdd);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+1),0x66);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+2),0xaa);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+3),0x55);

   for (;; numberScbs >>= 1)                  /* Divide scb number by 2 and see if*/
   {                                   /*  data is mirrored                */
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCBPTR), numberScbs >> 1);
      if (OSD_INEXACT8(SCSI_AICREG(SCSI_SCB_BASE+0)) != 0xdd || OSD_INEXACT8(SCSI_AICREG(SCSI_SCB_BASE+1)) != 0x66 ||
         OSD_INEXACT8(SCSI_AICREG(SCSI_SCB_BASE+2)) != 0xaa || OSD_INEXACT8(SCSI_AICREG(SCSI_SCB_BASE+3)) != 0x55)
      {
         break;                        /* If data is not mirrored, num=max scb*/
      }
   }
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCBPTR), numberScbs);      /* Set scb pointer to max scb       */

   for (i = 0; i < 4; i++)             
   {                                   /* Save current contents            */
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+i),
                  (SCSI_UEXACT8)(scbData >> (i * 8)));
   }

   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCBPTR), scbptr);      /* restore current scb pointer      */
   if (numberScbs != SCSI_MAXSCBS)
   {
      ++numberScbs;
      if ((hhcb->hardwareMode == SCSI_HMODE_AICBAYONET) ||
          (hhcb->hardwareMode == SCSI_HMODE_AICTRIDENT))
      {
         blocks = numberScbs + 1;
         while (blocks)
         {
            numberScbs = blocks - 1;
            blocks &= (blocks - 1);
         }
      } 
   }

   /* restore enable/disable scb ram */
   if (!wasEnabled)
   {
      SCSIHDisableExtScbRam(hhcb);
   }
   
   if (!(hcntrl & SCSI_PAUSEACK))
   {
      SCSI_hWRITE_HCNTRL(hhcb,scsiRegister,hcntrl);
   }

   return(numberScbs);
}
#endif

/***************************************************************************
*
*  SCSIhNonStandardGetConfig
*
*     Get configuration for swapping and test modes
*
*  Return Value:  none
*                  
*  Parameters:    hhcb
*
*  Remarks:       This routine should only be referenced at initialization
*                 time and it's part of initialization block.
*
***************************************************************************/
#if SCSI_SWAPPING_MODE
void SCSIhNonStandardGetConfig (SCSI_HHCB SCSI_HPTR hhcb)
{
   hhcb->numberScbs = SCSI_MAXSCBS;

   /* get target mode configuration */
   SCSI_hTARGETGETCONFIG(hhcb);

   /* get configuration common to all modes */
   SCSIhGetCommonConfig(hhcb);
}
#endif

/***************************************************************************
*
*  SCSIhGetCommonConfiguration
*
*     Get configuration common to all modes
*
*  Return Value:  none
*                  
*  Parameters:    hhcb
*
*  Remarks:       This routine should only be referenced at initialization
*                 time and it's part of initialization block.
*
***************************************************************************/
void SCSIhGetCommonConfig (SCSI_HHCB SCSI_HPTR hhcb)
{
}

/***************************************************************************
*
*  SCSIhGetHardwareConfiguration
*
*     This routine will get harwdare configuration information which are
*     independent of operating mode.
*
*  Return Value:  none
*                  
*  Parameters:    hhcb
*
*  Remarks:       This routine should only be referenced at initialization
*                 time and it's part of initialization block.
*
***************************************************************************/
void SCSIhGetHardwareConfiguration (SCSI_HHCB SCSI_HPTR hhcb)
{
   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_UEXACT8 sxfrctl1;
   SCSI_UEXACT8 i;

   /* get configuration information which is independent of operating */
   /* mode. */
   /* get default configuration information */
   /* this includes IRQ number selection, threshold, host scsi id */
   /* parity enable/disable, differential board identification, */
   /* wide/narrow selection, fast20/synchronous/asynchronous selection, */
   /* disconnect enable/disable, termination selction and scam protocol */
   /* enable/disable */
   hhcb->busRelease = (SCSI_UEXACT8) (OSD_INEXACT8(SCSI_AICREG(SCSI_LATTIME)) & 0xfc);

   if (OSD_INEXACT8(SCSI_AICREG(SCSI_SBLKCTL)) & SCSI_SELWIDE)
   {
      hhcb->maxDevices = 16;
   }
   else
   {
      hhcb->maxDevices = 8;
   }

   /* By default, set the SCAM level to 0.  This will be covered some */
   /* SCSI boards do not have seeprom and want the SCAM to be disabled. */
   hhcb->SCSI_HF_scamLevel = 0;

   for (i = 0; i < hhcb->maxDevices; i++)
   {
      /* turn on synch/wide negotiation */
      if (OSD_INEXACT8(SCSI_AICREG(SCSI_SBLKCTL)) & SCSI_SELWIDE)
      {
         SCSI_DEVICE_TABLE(hhcb)[i].scsiOption = (SCSI_UEXACT8) 0x81; 
      }
      else
      {
         SCSI_DEVICE_TABLE(hhcb)[i].scsiOption = (SCSI_UEXACT8) 0x1; 
      }
      SCSI_DEVICE_TABLE(hhcb)[i].SCSI_DF_disconnectEnable = 1;

      /* By default, we should initiate wide and/or sync. xfer */
      /* as well as response to wide and/or sync. xfer. */
      SCSI_DEVICE_TABLE(hhcb)[i].SCSI_DF_setForSync = 1;
      SCSI_DEVICE_TABLE(hhcb)[i].SCSI_DF_setForWide = 1;

      /* Assume all targets are wide and sync. devices */
      SCSI_DEVICE_TABLE(hhcb)[i].SCSI_DF_syncDevice = 1;
      SCSI_DEVICE_TABLE(hhcb)[i].SCSI_DF_wideDevice = 1;

      /* Assume all targets are previously running in async. mode */
      SCSI_DEVICE_TABLE(hhcb)[i].SCSI_DF_wasSyncXfer = 0;
      SCSI_DEVICE_TABLE(hhcb)[i].SCSI_DF_wasWideXfer = 0;

#if SCSI_NEGOTIATION_PER_IOB
      /* Clear all 'forceNego' flags */
      SCSI_DEVICE_TABLE(hhcb)[i].SCSI_DNF_forceSync = 0;
      SCSI_DEVICE_TABLE(hhcb)[i].SCSI_DNF_forceAsync = 0;
      SCSI_DEVICE_TABLE(hhcb)[i].SCSI_DNF_forceWide = 0;
      SCSI_DEVICE_TABLE(hhcb)[i].SCSI_DNF_forceNarrow = 0;
#endif

      /* Set Host Managed by default */
      SCSI_DEVICE_TABLE(hhcb)[i].SCSI_DF_hostManaged = 1;
   }

   /* collect configuration which are id based */
   SCSIhGetIDBasedConfiguration(hhcb);

   /* Initialize negotiation context flag to default value */
   hhcb->SCSI_HP_respondToWideOrSynch = 0;
   
   /* Initialize threshold setting */ 
   hhcb->SCSI_HF_separateRWThresholdEnable = 0;

   /* Initialize updateTermLevel flag.  By default we do not need */
   /* to update termination power level. */
   hhcb->SCSI_HF_updateTermLevel = 0;

   if (hhcb->SCSI_HP_initNeeded)
   {
      /* use default configuration */
      hhcb->threshold = SCSI_hSETDATAFIFOTHRSHDEFAULT(hhcb);
      hhcb->hostScsiID = 7;
#if SCSI_PARITY_PER_IOB      
      /* Always turn off parity checking first for per iob parity checking */
      /* implementation. Parity will be turned on from sequencer base on   */
      /* iob's parityEnable flag                                           */ 
      hhcb->SCSI_HF_scsiParity = 0;
#else
      hhcb->SCSI_HF_scsiParity = 1;
#endif
      hhcb->SCSI_HF_selTimeout = 0;
      hhcb->SCSI_HF_cacheThEn = SCSI_hCACHETHENDEFAULT(hhcb);
      hhcb->SCSI_HF_expSupport = SCSI_hEXPACTIVEDEFAULT(hhcb);
   }
   else
   {
      /* use existing hardware configuration */
      hhcb->threshold = SCSI_hREADDATAFIFOTHRSH(hhcb);
      hhcb->hostScsiID = (OSD_INEXACT8(SCSI_AICREG(SCSI_hSCSIID(hhcb))) & SCSI_OID);
      sxfrctl1 = OSD_INEXACT8(SCSI_AICREG(SCSI_SXFRCTL1));
      hhcb->SCSI_HF_scsiParity = (sxfrctl1 & SCSI_ENSPCHK) ? 1 : 0;
      hhcb->SCSI_HF_selTimeout = (sxfrctl1 & SCSI_STIMESEL) >> 3;
      hhcb->SCSI_HF_cacheThEn = SCSI_hGETCACHETHEN(hhcb);
      /* turn on this bit depend on hardware type */
      hhcb->SCSI_HF_expSupport = SCSI_hEXPACTIVEDEFAULT(hhcb);
   }
}

/***************************************************************************
*
*  SCSIhGetIDBasedConfiguration
*
*     Collect configuration information which are id based
*
*  Return Value:  none
*                  
*  Parameters:    hhcb
*
*  Remarks:       This routine should only be referenced at initialization
*                 time and it's part of initialization block.
*
***************************************************************************/
void SCSIhGetIDBasedConfiguration (SCSI_HHCB SCSI_HPTR hhcb)
{
   SCSI_UEXACT16 idValue;
   int i;

   /* Generally CacheLineStreaming will be enabled by default   */
   /* This feature will be turned off if a specific hw does not */
   /* support it correctly.                                     */
   hhcb->SCSI_HF_cacheThEn           = 1; /* Enable CacheLineStreaming  */   
   hhcb->SCSI_HF_cacheThEnAdjustable = 1; /* CACHETHEN bit is by default adjustable */

   idValue = hhcb->deviceID & SCSI_ID_MASK;

   SCSI_hSETDEFAULTAUTOTERM(hhcb);

   /* auto termination mode adjusted by product */
   if(hhcb->hardwareMode == SCSI_HMODE_AIC78XX)
   {

#if SCSI_AUTO_TERMINATION

      switch (idValue)                       /* designate cable sensing   */
      {                                      /* auto termination schemes  */
         case SCSI_DAGGER1_BASE:             /*    755x  */
         case SCSI_DAGGER2_BASE:             /*    785x  */
         case SCSI_TALON_BASE:               /*    786x  */
#if SCSI_2930CVAR_SUPPORT
         case SCSI_TALON_2930CVAR:           /* 2930CVAR */
#endif            
         hhcb->SCSI_HP_autoTerminationMode
            = SCSI_AUTOTERM_MODE1_CABLE_SENSING;
         break;

         default:                            /* lance, katana, excalibur */
            hhcb->SCSI_HP_autoTerminationMode = SCSI_AUTOTERM_MODE0_CABLE_SENSING;
            break;
      }
#endif
      hhcb->SCSI_HF_separateRWThreshold = 0; /* AIC78XX do not support separate thresholds */

      if (!((idValue==SCSI_KATANA_BASE)&&(hhcb->hardwareRevision>=1)))
      {
         /* For Katana base hw, CacheLineStreaming only works on Revision B or higher */
         /* Disable this feature for other revision or non-katana hw */
         hhcb->SCSI_HF_cacheThEn           = 0;
         hhcb->SCSI_HF_cacheThEnAdjustable = 0;
      }
   }

   /* By default assume cache threshold and auto flush problems are not there */
   hhcb->SCSI_HF_autoFlushHWProblem = 0;

#if SCSI_AICBAYONET
   if (hhcb->hardwareMode == SCSI_HMODE_AICBAYONET)
   {
      /* If its bayonet hardware revision 0 then the cache threshold feature */
      /* doesn't work properly.  We are determining bayonet vs. other AIC's */
      /* by checking the multi-function bit */
      if (((hhcb->hardwareRevision & 0x000000FF) == 0) && 
         ((hhcb->deviceID & 0x0040) != 0x0040))
      {
         hhcb->SCSI_HF_cacheThEn           = 0; /* Disable CacheLineStreaming   */
         hhcb->SCSI_HF_cacheThEnAdjustable = 0; /* Mark it to be non-adjustable */

         hhcb->SCSI_HF_autoFlushHWProblem  = 1;
      }

      if (idValue == 0x0050)
      {
         /* Scimitar has problem with CACHETHEN=0,     */
         /* hence mark this feature as non adjustable. */
         hhcb->SCSI_HF_cacheThEnAdjustable = 0;
      }

      /* setup workaround for for Bayonet/Scimitar */
      /* the arbitor hung is need inly for 64 bits pci */
      hhcb->SCSI_HP_wwArbitorHung =
         ((OSDReadPciConfiguration(hhcb,SCSI_DEVCONFIG) & SCSI_64BITS_PCI) != 0);
   }
#endif /* SCSI_AICBAYONET */

#if (SCSI_AICBAYONET || SCSI_AICTRIDENT)
   /* set auto termination mode for Bayonet/Trident here due to the new */
   /* device id scheme of the Bayonet/Trident chip is hard to fit into  */
   /* the SCSI_ID_MASK. */
   if ((hhcb->hardwareMode == SCSI_HMODE_AICBAYONET) ||
       (hhcb->hardwareMode == SCSI_HMODE_AICTRIDENT))
   {
#if SCSI_AUTO_TERMINATION
      hhcb->SCSI_HP_autoTerminationMode = SCSI_AUTOTERM_MODE2_CABLE_SENSING;
#endif

      /* No expander active support by default */
      hhcb->SCSI_HF_expSupport = 0;

      /* Bayonet/Trident support separate thresholds */
      hhcb->SCSI_HF_separateRWThreshold = 1;
      
      for (i = 0; i < hhcb->maxDevices; i++)
      {
         /* Pull defaults from subsystem id */
         switch(hhcb->SCSI_SUBID_maxScsiRate)
         {
            case SCSI_ULTRA_2:
               SCSI_DEVICE_TABLE(hhcb)[i].bayScsiRate = (SCSI_UEXACT8) 0x13;
               break;
            case SCSI_ULTRA:
               SCSI_DEVICE_TABLE(hhcb)[i].bayScsiRate = (SCSI_UEXACT8) 0x15;
               break;
            case SCSI_ULTRA_160: /* FAST80:DE with CRC by default */
               SCSI_DEVICE_TABLE(hhcb)[i].bayScsiRate = (SCSI_UEXACT8) 0x42;
               break;
            case SCSI_FAST:
               SCSI_DEVICE_TABLE(hhcb)[i].bayScsiRate = (SCSI_UEXACT8) 0x18;
               break;
         }
         SCSI_DEVICE_TABLE(hhcb)[i].bayScsiOffset = (SCSI_UEXACT8) SCSI_BAYONET_OFFSET; 
         SCSI_DEVICE_TABLE(hhcb)[i].bayDefaultRate = SCSI_DEVICE_TABLE(hhcb)[i].bayScsiRate;
         SCSI_DEVICE_TABLE(hhcb)[i].bayDefaultOffset = SCSI_DEVICE_TABLE(hhcb)[i].bayScsiOffset;

#if SCSI_PPR_ENABLE
         if (hhcb->hardwareMode == SCSI_HMODE_AICTRIDENT)
         {
            /* Assume all targets support double-transition clocking. */
            SCSI_DEVICE_TABLE(hhcb)[i].SCSI_DF_dtcSupport = 1;
            /* and also support single-transition clocking */
            SCSI_DEVICE_TABLE(hhcb)[i].SCSI_DF_stcSupport = 1;
         }
#endif

      }
   }
#endif   /* SCSI_AICBAYONET || SCSI_AICTRIDENT */

   /* get ultraEnable configuration */
   for (i = 0; i < SCSI_MAXDEV; i++)
   {
      if (hhcb->SCSI_SUBID_maxScsiRate == SCSI_FAST)
      {   
         SCSI_DEVICE_TABLE(hhcb)[i].SCSI_DF_ultraEnable = 0;
      }
      else
      {   
         SCSI_DEVICE_TABLE(hhcb)[i].SCSI_DF_ultraEnable = 1;
      }
   }

   /* Need to save off this setting as it can be modified 
    * by NVRAM or EEPROM access.
    */
   hhcb->SCSI_HP_ultraCapable = SCSI_DEVICE_TABLE(hhcb)[0].SCSI_DF_ultraEnable;

   /* set special flags for Excalibur ,Excalibur Lite and Bayonet */
   if (((idValue == SCSI_EXCALIBUR_BASE) &&
        (hhcb->hardwareMode == SCSI_HMODE_AIC78XX)) ||
       (hhcb->hardwareMode == SCSI_HMODE_AICBAYONET) ||
       (hhcb->hardwareMode == SCSI_HMODE_AICTRIDENT))
   {
      hhcb->SCSI_HP_scratch1Exist = 1;
      hhcb->SCSI_HP_scbSize32BitExist = 1;
   }
   else
   {
      hhcb->SCSI_HP_scratch1Exist = 0;
      hhcb->SCSI_HP_scbSize32BitExist = 0;
   }
   
#if SCSI_AICTRIDENT
   /* Trident base host adapter require to use 4k NVRAM */
   if (hhcb->hardwareMode == SCSI_HMODE_AICTRIDENT)
   {
      hhcb->SCSI_SUBID_seepromType = SCSI_SEE_TYPE2;
   }
#endif /* SCSI_AICTRIDENT */

   /* default to C06C46 except raid hardware (Viking and Athena) */
   /* and Trident base host adapters                             */
   if (hhcb->SCSI_SUBID_seepromType == SCSI_SEE_TYPE1)
   {
      hhcb->SCSI_HP_se2Type = SCSI_EETYPE_C56C66;
   }
   else if (hhcb->SCSI_SUBID_seepromType == SCSI_SEE_TYPE2)
   {
      hhcb->SCSI_HP_se2Type = SCSI_EETYPE_C56C66;
   }
   else if (hhcb->SCSI_SUBID_seepromType == SCSI_SEE_TYPE0)
   {
      hhcb->SCSI_HP_se2Type = SCSI_EETYPE_C06C46;
   }

   /* setup id based workaround requirements */
#if SCSI_4944IDFIX_ENABLE
   if ((hhcb->hardwareMode == SCSI_HMODE_AIC78XX) &&
      (idValue == SCSI_LANCE_BASE || idValue == SCSI_DAGGER1_BASE ||
      idValue == SCSI_DAGGER2_BASE || idValue == SCSI_TALON_BASE ||
      hhcb->deviceID == 0x783B || /* 4944 ID fix */
      hhcb->deviceID == 0x78EC ))  /* 4944 ID fix */
#else
   if ((idValue == SCSI_LANCE_BASE || idValue == SCSI_DAGGER1_BASE ||
      idValue == SCSI_DAGGER2_BASE || idValue == SCSI_TALON_BASE ) && 
      (hhcb->hardwareMode == SCSI_HMODE_AIC78XX))
#endif
   {
      hhcb->SCSI_HP_wwWriteHCNTRL = 1;
      hhcb->SCSI_HP_wwReadINTSTAT = 1;
   }
}

/***************************************************************************
*
*  SCSIhStandard64SetupSequencer
*
*     Setup sequencer code to run standard 64 mode
*
*  Return Value:  return status
*                  
*  Parameters:    hhcb
*
*  Remarks:       This routine should only be referenced at initialization
*                 time and it's part of initialization block.
*
***************************************************************************/
#if SCSI_STANDARD64_MODE
int SCSIhStandard64SetupSequencer (SCSI_HHCB SCSI_HPTR hhcb)
{
   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;

   return(SCSIhLoadSequencer(scsiRegister,Seq_s64,sizeof(Seq_s64)));
}
#endif

/***************************************************************************
*
*  SCSIhSwapping32SetupSequencer
*
*     Setup sequencer code to run swapping 32 mode
*
*  Return Value:  return status
*                  
*  Parameters:    hhcb
*
*  Remarks:       This routine should only be referenced at initialization
*                 time and it's part of initialization block.
*
***************************************************************************/
#if SCSI_SWAPPING32_MODE
int SCSIhSwapping32SetupSequencer (SCSI_HHCB SCSI_HPTR hhcb)
{
   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;

   return(SCSIhLoadSequencer(scsiRegister,Seq_w32,sizeof(Seq_w32)));
}
#endif


/***************************************************************************
*
*  SCSIhSwapping64SetupSequencer
*
*     Setup sequencer code to run swapping 64 mode
*
*  Return Value:  return status
*                  
*  Parameters:    hhcb
*
*  Remarks:       This routine should only be referenced at initialization
*                 time and it's part of initialization block.
*
***************************************************************************/
#if SCSI_SWAPPING64_MODE
int SCSIhSwapping64SetupSequencer (SCSI_HHCB SCSI_HPTR hhcb)
{
   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;

   return(SCSIhLoadSequencer(scsiRegister,Seq_w64,sizeof(Seq_w64)));
}
#endif

/***************************************************************************
*
*  SCSIhStandardAdvSetupSequencer
*
*     Setup sequencer code to run standard advanced mode
*
*  Return Value:  return status
*                  
*  Parameters:    hhcb
*
*  Remarks:       This routine should only be referenced at initialization
*                 time and it's part of initialization block.
*
***************************************************************************/
#if SCSI_STANDARD_ADVANCED_MODE
int SCSIhStandardAdvSetupSequencer (SCSI_HHCB SCSI_HPTR hhcb)
{
   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;

   return(SCSIhLoadSequencer(scsiRegister,Seq_sAdv,sizeof(Seq_sAdv)));
}
#endif

/***************************************************************************
*
*  SCSIhSwappingAdvSetupSequencer
*
*     Setup sequencer code to run swapping advanced mode
*
*  Return Value:  return status
*                  
*  Parameters:    hhcb
*
*  Remarks:       This routine should only be referenced at initialization
*                 time and it's part of initialization block.
*
***************************************************************************/
#if SCSI_SWAPPING_ADVANCED_MODE
int SCSIhSwappingAdvSetupSequencer (SCSI_HHCB SCSI_HPTR hhcb)
{
   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;

   return(SCSIhLoadSequencer(scsiRegister,Seq_wAdv,sizeof(Seq_wAdv)));
}
#endif

/***************************************************************************
*
*  SCSIhStandard160mSetupSequencer
*
*     Setup sequencer code to run standard Ultra 160m mode
*
*  Return Value:  return status
*                  
*  Parameters:    hhcb
*
*  Remarks:       This routine should only be referenced at initialization
*                 time and it's part of initialization block.
*
***************************************************************************/
#if SCSI_STANDARD_160M_MODE
int SCSIhStandard160mSetupSequencer (SCSI_HHCB SCSI_HPTR hhcb)
{
   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;

   return(SCSIhLoadSequencer(scsiRegister,Seqs160m,sizeof(Seqs160m)));
}
#endif

/***************************************************************************
*
*  SCSIhSwapping160mSetupSequencer
*
*     Setup sequencer code to run swapping Ultra 160m mode
*
*  Return Value:  return status
*                  
*  Parameters:    hhcb
*
*  Remarks:       This routine should only be referenced at initialization
*                 time and it's part of initialization block.
*
***************************************************************************/
#if SCSI_SWAPPING_160M_MODE
int SCSIhSwapping160mSetupSequencer (SCSI_HHCB SCSI_HPTR hhcb)
{
   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;

   return(SCSIhLoadSequencer(scsiRegister,Seqw160m,sizeof(Seqw160m)));
}
#endif

/***************************************************************************
*
*  SCSIhInitializeHardware
*
*     This routine will initialize hardware configuration which is
*     required at initialization time
*
*  Return Value:  return status
*                  
*  Parameters:    host task set handle
*
*  Remarks:       This routine should only be referenced at initialization
*                 time and it's part of initialization block.
*
***************************************************************************/
void SCSIhInitializeHardware (SCSI_HHCB SCSI_HPTR hhcb)
{
   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
#if SCSI_AIC78XX   
   SCSI_UEXACT32 cache_size = 0; 
   SCSI_UEXACT32 command_reg;
   SCSI_UEXACT8 cache_mask = 0;
#endif /* SCSI_AIC78XX */
#if SCSI_TRIDENT_PROTO
   SCSI_UEXACT8 scsiSfunct;
#endif /* SCSI_TRIDENT_PROTO */
#if !SCSI_UPDATE_TERMINATION_ENABLE
   SCSI_UEXACT8 stpwenSetting;
#endif /* !SCSI_UPDATE_TERMINATION_ENABLE */
#if SCSI_PCI2_0_WORKAROUND
   SCSI_UEXACT32 cacheline_reg;
#endif /* SCSI_PCI2_0_WORKAROUND */
   SCSI_UEXACT32 devConfig;

   OSD_OUTEXACT8(SCSI_AICREG(SCSI_hSCSIID(hhcb)), hhcb->hostScsiID);

   /* auto flush is disabled for Bayonet because of hardware problem */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SBLKCTL),
      (OSD_INEXACT8(SCSI_AICREG(SCSI_SBLKCTL)) &
            ~(SCSI_DIAGLEDEN + SCSI_DIAGLEDON)) |
            (((hhcb->hardwareMode == SCSI_HMODE_AICBAYONET) &&
             (hhcb->SCSI_HF_autoFlushHWProblem)) ?
            SCSI_AUTOFLUSHDIS : 0));        /* Turn off led   */

   /*OSD_OUTEXACT8(SCSI_AICREG(SCSI_PCISTATUS), hhcb->threshold << 6)*/
   SCSI_hUPDATEDATAFIFOTHRSH(hhcb);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SXFRCTL0), SCSI_DFON);   /* Turn on digital filter */

   SCSI_hUPDATECACHETHEN(hhcb);

   if (hhcb->SCSI_HF_updateTermLevel)
   {
      /* From Trident based HA (AHA-3960D), the SCSI termination power level */
      /* must be set to the correct level (active low or high) based on the  */
      /* information stored in NVRAM (BIOS set this).  The reason is the     */
      /* board was designed to remove the inverter between the STPWLEVEL     */
      /* signal and the termination chip. */
      devConfig = OSD_READ_PCICFG_EXACT32(hhcb,SCSI_DEVCONFIG);
      if (hhcb->SCSI_HF_terminationLevel)
         devConfig |= SCSI_STPWLEVEL;
      else
         devConfig &= ~SCSI_STPWLEVEL;
      OSD_WRITE_PCICFG_EXACT32(hhcb,SCSI_DEVCONFIG, devConfig);
   }

#if SCSI_AUTO_TERMINATION
   /* do auto termination if user enables */
   SCSI_hPROCESSAUTOTERM(hhcb);
#endif

#if SCSI_UPDATE_TERMINATION_ENABLE
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SXFRCTL1), (SCSI_UEXACT8)
           ((hhcb->SCSI_HF_scsiParity ? SCSI_ENSPCHK : 0) +
            SCSI_ENSTIMER + SCSI_ACTNEGEN +    /* low byte termination get set here */
            (hhcb->SCSI_HF_selTimeout << 3) +
            (hhcb->SCSI_HF_terminationLow ? SCSI_STPWEN : 0)));

   /* update termination that controlled by external hardware logic */
   SCSI_hUPDATEEXTTERM(hhcb);
#else /* !SCSI_UPDATE_TERMINATION_ENABLE */
#if SCSI_CUSTOM_TERMINATION
#if (SCSI_AICBAYONET || SCSI_AICTRIDENT)
   if ((hhcb->hardwareMode == SCSI_HMODE_AICBAYONET) ||
       (hhcb->hardwareMode == SCSI_HMODE_AICTRIDENT))
   {
      if(hhcb->SCSI_HF_terminationLow || hhcb->SCSI_HF_terminationHigh)
      {
         stpwenSetting = 0;
      }
      else
      {
         stpwenSetting = SCSI_STPWEN;
      }

      /* Set the SCSI Termination Power Level to Active Low */
      /* This should be performed in the OSM code!! */
      devConfig  = OSD_READ_PCICFG_EXACT32(hhcb,SCSI_DEVCONFIG);
      devConfig |= SCSI_STPWLEVEL;
      OSD_WRITE_PCICFG_EXACT32(hhcb,SCSI_DEVCONFIG, devConfig);

      /* Set up SCSI_SXFRCTL1;  set the termination */
      stpwenSetting += (hhcb->SCSI_HF_scsiParity ? SCSI_ENSPCHK : 0) +
                        SCSI_ENSTIMER + SCSI_ACTNEGEN +
                        (hhcb->SCSI_HF_selTimeout << 3);

      /* The hardware seems to require us to write this high, then either */
      /* high or low (the way we want it) to act properly. */
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SXFRCTL1),(stpwenSetting | SCSI_STPWEN));
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SXFRCTL1),stpwenSetting);
   }   
#endif /* (SCSI_AICBAYONET || SCSI_AICTRIDENT) */
#else
   /* keep the current STPWEN setting */
   stpwenSetting = OSD_INEXACT8(SCSI_AICREG(SCSI_SXFRCTL1)) & SCSI_STPWEN;
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SXFRCTL1), (SCSI_UEXACT8)
           ((hhcb->SCSI_HF_scsiParity ? SCSI_ENSPCHK : 0) +
            SCSI_ENSTIMER + SCSI_ACTNEGEN +
            (hhcb->SCSI_HF_selTimeout << 3) +
            stpwenSetting));
#endif /* SCSI_CUSTOM_TERMINATION */
#endif /* SCSI_UPDATE_TERMINATION_ENABLE */

/* clear all Fast 20 registers */
SCSI_hCLEAR_ALL_FAST20_REG(hhcb);

#if SCSI_TRIDENT_PROTO
   if (hhcb->hardwareMode == SCSI_HMODE_AICTRIDENT)
   {
      scsiSfunct = OSD_INEXACT8(SCSI_AICREG(SCSI_SFUNCT));
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SFUNCT), (scsiSfunct | SCSI_ALT_MODE));
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_TRIDENT_IOPDNCTL),
         (OSD_INEXACT8(SCSI_AICREG(SCSI_TRIDENT_IOPDNCTL))|SCSI_DIS_FORCE_BC));
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SFUNCT), (scsiSfunct & ~SCSI_ALT_MODE));
   }
#endif /* SCSI_TRIDENT_PROTO */

/* Workaround for the problem of Excalibur B2/katana B/Talon B chips       */
/* Check for Cache Line Size of 4 DWDs (16 bytes and set to 0 if it is.    */
#if SCSI_AIC78XX   
   cache_size = OSDReadPciConfiguration(hhcb,SCSI_HDR_TYPE_REG);
   if ((hhcb->hardwareMode == SCSI_HMODE_AIC78XX) && 
      ((cache_size & SCSI_CDWDSIZE) == 4))
   {   
      OSDWritePciConfiguration(hhcb,SCSI_HDR_TYPE_REG,cache_size & (~SCSI_CDWDSIZE));   
      cache_size = 0;
   }
#endif /* SCSI_AIC78XX */

/* Workaround for Excalibur generating false DAC command after retrying  */
/* commands on PCI bus                                                   */
#if SCSI_AIC78XX   
#if SCSI_EXCALIBUR_DAC_WORKAROUND >= 1
   /* set Cache Line Size to 0 */
   if ((hhcb->deviceID & SCSI_ID_MASK) == SCSI_EXCALIBUR_BASE)
   {
      OSDWritePciConfiguration(hhcb,SCSI_HDR_TYPE_REG,cache_size & (~SCSI_CDWDSIZE));   
      cache_size = 0;
   }
#endif /* SCSI_EXCALIBUR_DAC_WORKAROUND */
#endif /* SCSI_AIC78XX */

/* Workaround for Excalibur B2 mwi data corruption problem on Orion C0 chipset*/
/* this workaround also covers the Lance problem with mwi and 1,2,3 bytes from*/
/* cache boundary.  Thus, all chips have workaround except katana and any     */
/* excalibur that is not rev 3 (excalibur B2).                                */   
#if SCSI_AIC78XX   
   if ((!cache_size) || (hhcb->hardwareMode == SCSI_HMODE_AIC78XX) &&
      ((((hhcb->deviceID & SCSI_ID_MASK) == SCSI_EXCALIBUR_BASE) && 
      hhcb->hardwareRevision > 3) || ((hhcb->deviceID & SCSI_ID_MASK) == SCSI_KATANA_BASE)))
   {
      cache_mask = 0;
   }
   else
   {
      if((cache_size & SCSI_CDWDSIZE) == 8)        /* 32 byte cache line size */
      {
         cache_mask = 0x1f;
      }
      else if((cache_size & SCSI_CDWDSIZE) == 16)  /* 64 byte cache line size */
      {
         cache_mask = 0x3f;
      }
      else if((cache_size & SCSI_CDWDSIZE) == 32)  /* 128 byte cache line size */
      {
         cache_mask = 0x7f;
      }
   }
   /* Turn mwi bit back on (bios turns it off for excalibur B2 (rev 3)).  BIOS */
   /* turns this bit off to fix the ORION C0 data corruption problem, however */
   /* this causes a performance problem for excalibur.  The sequencer workaround */
   /* fixes the data corruption problem and should bring performance almost back */
   /* to normal, so we can turn the MWICEN bit back on.                       */
   if ((((hhcb->deviceID & SCSI_ID_MASK) == SCSI_EXCALIBUR_BASE) && 
      (hhcb->hardwareMode == SCSI_HMODE_AIC78XX)) 
      && hhcb->hardwareRevision == 3)
   {
      command_reg = OSDReadPciConfiguration(hhcb,SCSI_STATUS_CMD_REG);
      OSDWritePciConfiguration(hhcb,SCSI_STATUS_CMD_REG,(command_reg | SCSI_MWRICEN));   
   }
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_hMWI_CACHE_MASK(hhcb)),cache_mask);
#endif /* SCSI_AIC78XX */

/* Workaround for Bayonet Rev A chip, set cache line size to 0 to fix the problem*/ 
/* that system hangs while Bayonet DMA from Mace DRAM.                           */    
#if SCSI_PCI2_0_WORKAROUND
   if((hhcb->deviceID  == SCSI_ARO_BAYONET) && 
      ((SCSI_UEXACT8)(OSDReadPciConfiguration(hhcb,SCSI_DEV_REV_ID)&0x000000FF)
      == SCSI_BAYONET_REV_A))
   {
      cacheline_reg = OSDReadPciConfiguration(hhcb,SCSI_HDR_TYPE_REG);
      OSDWritePciConfiguration(hhcb,SCSI_HDR_TYPE_REG,cacheline_reg & 0xffffff00);
   }
#endif /* SCSI_PCI2_0_WORKAROUND */

#if SCSI_SCAM_ENABLE                   /* and compiling with SCAM option   */
   if (hhcb->SCSI_HF_scamLevel)        /* if user enables scam operation   */
   {
      SCSI_UEXACT8   scamControl;
      SCSI_UEXACT8   scamLevelSupport;

      /* SCAM compile option set at level 2.  The default SCAM level */
      /* setting would be what ever hardware allows.                 */
      scamLevelSupport = SCSIhScamSupportCheck(scsiRegister);

      if (scamLevelSupport < hhcb->SCSI_HF_scamLevel)
         hhcb->SCSI_HF_scamLevel = scamLevelSupport;

      if (hhcb->SCSI_HF_scamLevel == 2)
      {
         scamControl = OSD_INEXACT8(SCSI_AICREG(SCSI_SCAMSTAT));
         OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCAMCTRL),
            (scamControl & ~SCSI_SCAMLVL) | hhcb->SCSI_HF_scamLevel);
      }
   }
#endif
}

/***************************************************************************
*
*  SCSIhStartToRunSequencer
*
*     Start to run sequencer
*
*  Return Value:  none
*                  
*  Parameters:    hhcb
*
*  Remarks:       This routine should only be referenced at initialization
*                 time and it's part of initialization block.
*
***************************************************************************/
void SCSIhStartToRunSequencer (SCSI_HHCB SCSI_HPTR hhcb)
{
   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;

#if SCSI_CHECK_PCI_ERROR
   /* we don't enable pci error checking for legacy devices */ 
   if(hhcb->hardwareMode != SCSI_HMODE_AIC78XX)
   {
      /* enable PCI error status interrupt and breakpoint status interrupt */
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SEQCTL),SCSI_FASTMODE + SCSI_BRKINTEN);
   }
   else
   {
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SEQCTL),SCSI_FASTMODE + SCSI_FAILDIS);
   }
#else
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SEQCTL),SCSI_FASTMODE + SCSI_FAILDIS);
#endif

   /* Entry points always low page  */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SEQADDR0), (SCSI_UEXACT8)SCSI_hIDLE_LOOP_ENTRY(hhcb));
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SEQADDR1), (SCSI_UEXACT8)SCSI_hIDLE_LOOP_ENTRY(hhcb) >> 2);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_BRKADDR1), SCSI_BRKDIS);

#if SCSI_DISABLE_INTERRUPTS   
   SCSI_hWRITEHCNTRL(scsiRegister, (SCSI_UEXACT8) 0);
#else
   SCSI_hWRITEHCNTRL(scsiRegister, (SCSI_UEXACT8) SCSI_INTEN);
#endif   
   /* Clear chipReset copy in HHCB as well */
   hhcb->SCSI_HF_chipReset = 0;
}

/***************************************************************************
*
*  SCSIhAIC78XXProcessAutoTerm
*
*     Process auto termination if user enables.
*
*  Return Value:  none
*                  
*  Parameters:    hhcb
*
*  Remarks:
*
***************************************************************************/
#if SCSI_AIC78XX && SCSI_AUTO_TERMINATION
void SCSIhAIC78XXProcessAutoTerm (SCSI_HHCB SCSI_HPTR hhcb)
{
   SCSI_UEXACT8 terminationSetting;

   if (hhcb->SCSI_HF_autoTermination)  /* if user enables autotermination */
   {
      terminationSetting = SCSIhAIC78XXAutoTermCable(hhcb); /* do autoterm */
      if (!(hhcb->deviceID == SCSI_2940AUW))
         hhcb->SCSI_HF_terminationLow =
            (terminationSetting & SCSI_TERMINATION_LOW) ? 1 : 0;
      hhcb->SCSI_HF_terminationHigh =
         (terminationSetting & SCSI_TERMINATION_HIGH) ? 1 : 0;
   }
}
#endif   /* SCSI_AIC78XX && SCSI_AUTO_TERMINATION */

/***************************************************************************
*
*  SCSIhAICBayonetProcessAutoTerm
*

*     Process auto termination if user enables.
*
*  Return Value:  none
*                  
*  Parameters:    hhcb
*
*  Remarks:
*
***************************************************************************/
#if (SCSI_AICBAYONET || SCSI_AICTRIDENT) && SCSI_AUTO_TERMINATION
void SCSIhAICBayonetProcessAutoTerm (SCSI_HHCB SCSI_HPTR hhcb)
{
   SCSI_UEXACT8 terminationSetting;

  /* if user enables primary autotermination */
   if (hhcb->SCSI_HF_autoTermination)
   {
      /* For the Bayonet, the HW autotermination logic detects */
      /* SCSI cable connections, decodes and provides the      */
      /* termination control bits for the software to read.    */
      /* Control bits are from BRDCTL register (1=ON, 0=OFF):  */
      /*    bit 6: Secondary Termination HIGH byte             */
      /*    bit 5: Secondary Termination LOW byte              */
      /*    bit 4: Primary Termination HIGH byte               */
      /*    bit 3: Primary Termination LOW byte                */
      terminationSetting = SCSIhCableSense(hhcb);
      hhcb->SCSI_HF_terminationLow =
         (terminationSetting & SCSI_TERMINATION_LOW) ? 1 : 0;
      hhcb->SCSI_HF_terminationHigh =
         (terminationSetting & SCSI_TERMINATION_HIGH) ? 1 : 0;

      /* if user enables secondary autotermination */
      if (hhcb->SCSI_HF_secondaryAutoTerm)
      {
         hhcb->SCSI_HF_secondaryTermLow =
            (terminationSetting & SCSI_SECONDARY_TERM_LOW) ? 1 : 0;
         hhcb->SCSI_HF_secondaryTermHigh =
            (terminationSetting & SCSI_SECONDARY_TERM_HIGH) ? 1 : 0;
      }
   }

   /* if user enables secondary autotermination */
   else if (hhcb->SCSI_HF_secondaryAutoTerm)
   {
      /* For the Bayonet, the HW autotermination logic detects */
      /* SCSI cable connections, decodes and provides the      */
      /* termination control bits for the software to read.    */
      /* Control bits are from BRDCTL register (1=ON, 0=OFF):  */
      /*    bit 6: Secondary Termination HIGH byte             */
      /*    bit 5: Secondary Termination LOW byte              */
      /*    bit 4: Primary Termination HIGH byte               */
      /*    bit 3: Primary Termination LOW byte                */
      terminationSetting = SCSIhCableSense(hhcb);
      hhcb->SCSI_HF_secondaryTermLow =
         (terminationSetting & SCSI_SECONDARY_TERM_LOW) ? 1 : 0;
      hhcb->SCSI_HF_secondaryTermHigh =
         (terminationSetting & SCSI_SECONDARY_TERM_HIGH) ? 1 : 0;
   }
}
#endif   /* (SCSI_AICBAYONET || SCSI_AICTRIDENT) && SCSI_AUTO_TERMINATION */

/***************************************************************************
*
*  SCSIhAIC78XXUpdateExtTerm
*
*     Update termination that controlled via external hardware logic.
*
*  Return Value:  none
*                  
*  Parameters:    hhcb
*
*  Remarks:       For non-Bayonet base HA, there is only the high byte
*                 termination will be updated.
*
***************************************************************************/
#if SCSI_AIC78XX
void SCSIhAIC78XXUpdateExtTerm (SCSI_HHCB SCSI_HPTR hhcb)
{
   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_HOST_ADDRESS SCSI_LPTR hostAddress;
   SCSI_UEXACT8 channelSelect;
   SCSI_UINT32 i = 0;
   
   /* high byte termination disable/enable only if it's wide bus */
   if (hhcb->maxDevices == 16)
   {
      /* process high byte termination */
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SEEPROM),SCSI_SEEMS);
      while ((!(OSD_INEXACT8(SCSI_AICREG(SCSI_SEEPROM)) & SCSI_SEERDY)) &&
             (i++ < 400000))
      {
         OSD_TIMER(hhcb);
      }

      /* Excalibur based boards (AHA-394xUxx) use BRDCS bit in BRDCTL register */
      /* to select either channel A (BRDCS = 0) or channel B (BRDCS = 1) */
      /* Get host address to access PCI device's function number */
      hostAddress = OSD_GET_HOST_ADDRESS(hhcb);
      if (((hhcb->deviceID & SCSI_ID_MASK) == SCSI_EXCALIBUR_BASE) &&
          (hostAddress->pciAddress.functionNumber == 0))
      {
         channelSelect = 0;            /* Select channel A */
      }
      else
      {
         channelSelect = SCSI_BRDCS;   /* Select channel B or use for other HAs */
      }
   
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SEEPROM),SCSI_SEEMS|SCSI_SEECS);
   
      if (hhcb->SCSI_HF_terminationHigh)
      {                                   /* enable high byte termination */
         OSD_OUTEXACT8(SCSI_AICREG(SCSI_BRDCTL),SCSI_BRDDAT6|SCSI_BRDSTB|channelSelect);
         OSD_OUTEXACT8(SCSI_AICREG(SCSI_BRDCTL),SCSI_BRDDAT6|channelSelect);        
      }
      else
      {                                   /* disable high byte termination */
         OSD_OUTEXACT8(SCSI_AICREG(SCSI_BRDCTL),SCSI_BRDSTB|channelSelect);
         OSD_OUTEXACT8(SCSI_AICREG(SCSI_BRDCTL),channelSelect);
      }

      OSD_OUTEXACT8(SCSI_AICREG(SCSI_BRDCTL),0);
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SEEPROM),0);
   }
}
#endif   /* SCSI_AIC78XX */

/***************************************************************************
*
*  SCSIhAICBayonetUpdateExtTerm
*
*     Update termination that controlled via external hardware logic.
*
*  Return Value:  none
*                  
*  Parameters:    hhcb
*
*  Remarks:       For Bayonet base HA, the primary high byte, secondary
*                 high and low byte termination will be updated.
*
***************************************************************************/
#if (SCSI_AICBAYONET || SCSI_AICTRIDENT)
void SCSIhAICBayonetUpdateExtTerm (SCSI_HHCB SCSI_HPTR hhcb)
{
   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_UEXACT8 scsiBrddat = 0;
   SCSI_UINT32 i = 0;
   /* primary high byte termination disable/enable only if it's wide bus */
   if ((hhcb->maxDevices == 16) && (hhcb->SCSI_HF_terminationHigh))
   {
      scsiBrddat |= SCSI_BAYONET_BRDDAT4;
   }

   if (hhcb->SCSI_HF_secondaryTermLow)
   {
      scsiBrddat |= SCSI_BAYONET_BRDDAT5;
   }

   if (hhcb->SCSI_HF_secondaryTermHigh)
   {
      scsiBrddat |= SCSI_BAYONET_BRDDAT6;
   }

   /* process primary high byte, secondary low and high byte termination */
   if (hhcb->hardwareMode == SCSI_HMODE_AICTRIDENT)
   {
      /* TridentII hardware need to assert both SEEMS and SEECS to arbitrate */
      /* for flex port. These two bits need to be asserted until chip finish */
      /* flex port accessing.                                                */
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SEEPROM),SCSI_SEEMS|SCSI_SEECS);
   }
   else
   {   
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SEEPROM),SCSI_SEEMS);
   }
  
   while ((!(OSD_INEXACT8(SCSI_AICREG(SCSI_SEEPROM)) & SCSI_SEERDY)) &&
          (i++ <= 400000))
   {
      OSD_TIMER(hhcb);
   }

   if (hhcb->hardwareMode == SCSI_HMODE_AICTRIDENT)
   {
      /* TridentII hardware need to assert both SEEMS and SEECS to arbitrate */
      /* for flex port. These two bits need to be asserted until chip finish */
      /* flex port accessing.                                                */
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SEEPROM),SCSI_SEEMS|SCSI_SEECS);
   }
   else
   {   
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SEEPROM),SCSI_SEEMS);
   }

   /* setting primary high byte, secondary low and high byte termination */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_BRDCTL),scsiBrddat);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_BRDCTL),scsiBrddat | SCSI_BAYONET_BRDSTB);

   OSD_OUTEXACT8(SCSI_AICREG(SCSI_BRDCTL), 0);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SEEPROM), 0); /* deassert SEEMS and SEECS */

}
#endif   /* SCSI_AICBAYONET || SCSI_AICTRIDENT */

/***************************************************************************
*                                                                          *
* Copyright 1995,1996,1997,1998,1999 Adaptec, Inc.,  All Rights Reserved.  *
*                                                                          *
* This software contains the valuable trade secrets of Adaptec.  The       *
* software is protected under copyright laws as an unpublished work of     *
* Adaptec.  Notice is for informational purposes only and does not imply   *
* publication.  The user of this software may make copies of the software  *
* for use with parts manufactured by Adaptec or under license from Adaptec *
* and for no other use.                                                    *
*                                                                          *
***************************************************************************/

/***************************************************************************
*
*  Module Name:   HWMHRST.C
*
*  Description:
*                 Codes to implement hardware reset module for hardware 
*                 management module
*
*  Owners:        ECX IC Firmware Team
*    
*  Notes:         NONE
*
***************************************************************************/

#include "scsi.h"
#include "hwm.h"

/*********************************************************************
*
*  SCSIhResetChannelHardware
*
*     Reset channel harwdare
*
*  Return Value:  none
*                  
*  Parameters:    hhcb
*
*  Remarks:       This routine can be called either at initialization
*                 or exceptional condition
*                  
*********************************************************************/
void SCSIhResetChannelHardware (SCSI_HHCB SCSI_HPTR hhcb)
{
   SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_UINT32 i = 0;          /* time out counter */
#if SCSI_STANDARD_ADVANCED_MODE || SCSI_SWAPPING_ADVANCED_MODE
   SCSI_UEXACT8 temp;
#endif /* SCSI_STANDARD_ADVANCED_MODE || SCSI_SWAPPING_ADVANCED_MODE */

   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSISEQ), 00);
#if SCSI_TRIDENT_PROTO
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_hSCSISEQ_COPY(hhcb)), 00);   /* TLUU - trident debug */
#endif /* SCSI_TRIDENT_PROTO */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_CLRSINT0), 0xff);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_CLRSINT1), 0xff);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SXFRCTL0),
       OSD_INEXACT8(SCSI_AICREG(SCSI_SXFRCTL0)) | (SCSI_CLRSTCNT|SCSI_CLRCHN));

   /* Make sure the bit bucket is turned off in case we are recovering from */
   /* data phase hang situation (especially during domain validation) */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SXFRCTL1),
       OSD_INEXACT8(SCSI_AICREG(SCSI_SXFRCTL1)) & ~(SCSI_BITBUCKET));

   /* workaround for Auto Rate/Offset issue where the scsi rate/offset */
   /* does not take affect immediately for current connection. */
   SCSI_hENABLEAUTORATEOPTION(hhcb);

   /* reset scsi rate/offset */
   SCSI_hRESETSCSIRATE(hhcb);

   /* Need to write 0 to DFCNTRL first in case HDMAENACK is still on */
   /* This is to fix a problem when we get 3rd party reset during    */
   /* data out phase with Excalibur chip and Bayonet chip            */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_DFCNTRL), 0);
   while ((OSD_INEXACT8(SCSI_AICREG(SCSI_DFCNTRL)) & SCSI_HDMAENACK) &&
          (i++ < 400000))
   {
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_DFCNTRL), 0);
      OSD_TIMER(hhcb);
     
   }
   SCSI_hRESETDFIFO(hhcb);

   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SIMODE1), SCSI_ENSCSIPERR + SCSI_ENSELTIMO + SCSI_ENSCSIRST);

#if SCSI_STANDARD_ADVANCED_MODE || SCSI_SWAPPING_ADVANCED_MODE
   if (hhcb->hardwareMode == SCSI_HMODE_AICBAYONET) /* bayonet class device */
   {
      i = 0; /* counter used so that we don't loop forever */
      /* The following code was added to fix the following hardware bug: */
      /* When the output of the [CD, MSG, IO] comparitor goes low        */
      /* indicating a mismatch between SCSISIG and the bus) a pulse is   */
      /* generated which is supposed to set PHASECHG, but sometimes      */
      /* doesn't.                                                        */
      /* The pulse only gets generated when the comparator output goes   */
      /* from high to low.                                               */     
      while ((!((OSD_INEXACT8(SCSI_AICREG(SCSI_SSTAT1)) & SCSI_PHASECHG))) &&
              (i++ < 400000))
      {
         /* Read SCSISIGI, write the same value - then write a different  */
         /* value to cause a phase change - sequencer requires phase      */
         /* change bit to be set.                                         */

         temp = OSD_INEXACT8(SCSI_AICREG(SCSI_SCSISIG));
  
         /* write top 3 bits as read (C/D, I/O, and MSG bits) */
         /* must ensure other bits are 0 */
         OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSISIG), (temp & 0xE0));
         /* now write different to force a phase change */
         OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSISIG), ((~temp) & 0xE0));
      }
    
      /* Now set SCSISIGO to 0 to match sequencer specification */
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSISIG), 0x00);

   }
 
   if (hhcb->hardwareMode == SCSI_HMODE_AICTRIDENT) /* trident class device */
   {
      /* Initialize SCSISIG to 0 in case the hardware is stuck in
       * selection out which has been seen to occur after a SCSI
       * bus reset. This is due to errors in the hardware selection
       * logic when aborting a selection due to a bus reset.
       */
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSISIG), 0x00);
   } 
#endif /* SCSI_STANDARD_ADVANCED_MODE || SCSI_SWAPPING_ADVANCED_MODE */

   SCSI_hENIOERR(hhcb);
   SCSI_hRESETCCCTL(hhcb);

#if SCSI_TARGET_OPERATION
   if (hhcb->SCSI_HF_targetMode)
   {
      /* Clear possible pending selection-in */
      /* This is only required for performance code */ 
      SCSIhTargetClearSelection(hhcb);
   }

#endif /* SCSI_TARGET_OPERATION */

   SCSI_hINITCCHHADDR(hhcb);
   SCSI_hRESETBTA(hhcb);
}

/*********************************************************************
*
*  SCSIhAbortChannel
*
*     Abort active hiobs associated with channel
*
*  Return Value:  none
*
*  Parameters:    hhcb
*
*  Remarks:
*
*********************************************************************/
void SCSIhAbortChannel (SCSI_HHCB SCSI_HPTR hhcb,
                        SCSI_UEXACT8 haStatus)
{
   SCSI_HIOB SCSI_IPTR hiob;
   SCSI_UEXACT8 i;

   for (i = 0; i < hhcb->numberScbs; i++)
   {
      if ((hiob = SCSI_ACTPTR[i]) != SCSI_NULL_HIOB)
      {
#if SCSI_TARGET_OPERATION  
         if (hiob->SCSI_IP_targetMode ||
             hiob->cmd == SCSI_CMD_ESTABLISH_CONNECTION)
         {
            if (SCSI_NEXUS_UNIT(hiob)->hhcb != hhcb)
            {
               continue; 
            }
         }    
         else
#endif /* SCSI_TARGET_OPERATION */
         if (SCSI_TARGET_UNIT(hiob)->hhcb != hhcb)
         {
            continue;
         }   

         hiob->stat = SCSI_SCB_ABORTED;
         hiob->haStat = haStatus;
         SCSI_ACTPTR[i] = SCSI_NULL_HIOB;
#if SCSI_TARGET_OPERATION
         if (hiob->SCSI_IP_targetMode)
         {
            SCSI_COMPLETE_SPECIAL_HIOB(hiob);
         }
         else
         {
            SCSI_COMPLETE_HIOB(hiob);
         }
#else
#if SCSI_DMA_SUPPORT
         if (hiob->cmd == SCSI_CMD_INITIATE_DMA_TASK)
         {
            SCSI_COMPLETE_SPECIAL_HIOB(hiob);
         }
         else
#endif /* SCSI_DMA_SUPPORT */
         { 
            SCSI_COMPLETE_HIOB(hiob);
         }
#endif /* SCSI_TARGET_OPERATION */    
      }
   }
}

/*********************************************************************
*
*  SCSIhStandard64ResetSoftware
*
*     Reset channel software
*
*  Return Value:  none
*                  
*  Parameters:    hhcb
*
*  Remarks:       This routine can be called either at initialization
*                 or exceptional condition
*                  
*********************************************************************/
#if SCSI_STANDARD64_MODE
void SCSIhStandard64ResetSoftware (SCSI_HHCB SCSI_HPTR hhcb)
{
   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_UEXACT16 disconnectDisable = 0;
   int i;
   SCSI_SCB_DESCRIPTOR SCSI_HPTR scbDescriptor;

   /* reset software common to all modes of operations */
   SCSIhCommonResetSoftware(hhcb);

   /* set disconnect option */
   for (i = 0; i < SCSI_MAXDEV; i++)
   {
      disconnectDisable |=
                  ((! SCSI_DEVICE_TABLE(hhcb)[i].SCSI_DF_disconnectEnable) << i);
   }
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_S64_DISCON_OPTION),disconnectDisable);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_S64_DISCON_OPTION+1),disconnectDisable >> 8);

   /* setup q_new_pointer */
   scbDescriptor = SCSI_GET_FREE_HEAD(hhcb);
   SCSI_hSETADDRESSSCRATCH(hhcb,SCSI_S64_Q_NEW_POINTER,scbDescriptor->scbBuffer.virtualAddress);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_S64_Q_NEXT_SCB),scbDescriptor->scbNumber);

   /* setup q_new_head */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_S64_Q_NEW_HEAD),(SCSI_UEXACT8)0);
   hhcb->SCSI_HP_qNewHead = 0;

   /* setup q_new_tail */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_S64_Q_NEW_TAIL),(SCSI_UEXACT8)0);

   /* setup q_done_base */
   SCSI_hSETADDRESSSCRATCH(hhcb,SCSI_S64_Q_DONE_BASE,SCSI_QOUT_PTR_ARRAY_NEW(hhcb));

   /* setup q_done_head */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_S64_Q_DONE_HEAD),(SCSI_UEXACT8)0xFF);
 
   /* setup q_done_element */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_S64_Q_DONE_ELEMENT),(SCSI_UEXACT8)0);
   hhcb->SCSI_HP_qDoneElement = 0;

   /* setup q_done_pass (both scratch and shadow for HIM */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_S64_Q_DONE_PASS),(SCSI_UEXACT8) 0xFF);
   hhcb->SCSI_HP_qDonePass = 0;

   /* setup q_exe_head */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_S64_Q_EXE_HEAD),(SCSI_UEXACT8)0xFF);
   
   /* setup done SCB queue in host */
   /* @@ */
   for (i = 0; i < (int) (((SCSI_UEXACT16) hhcb->numberScbs + 1) * 
                     sizeof(SCSI_QOUTFIFO_NEW)); i++)
   {
      *(((SCSI_UEXACT8 SCSI_HPTR) SCSI_QOUT_PTR_ARRAY_NEW(hhcb)) + i) = 
                              (SCSI_UEXACT8) 0xFF;
   }

   /* Invalidate waiting scb and active scb entry */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_S64_ACTIVE_SCB),SCSI_NULL_SCB);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_S64_WAITING_SCB),SCSI_NULL_SCB);

   OSD_OUTEXACT8(SCSI_AICREG(SCSI_S64_Q_MASK), (SCSI_UEXACT8)(hhcb->numberScbs + 1));
}
#endif


/*********************************************************************
*
*  SCSIhSwappingResetSoftware
*
*     Reset channel software for swapping32 mode
*
*  Return Value:  none
*                  
*  Parameters:    hhcb
*
*  Remarks:       This routine can be called either at initialization
*                 or exceptional condition
*                  
*********************************************************************/
#if SCSI_SWAPPING32_MODE + SCSI_SWAPPING64_MODE
void SCSIhSwappingResetSoftware (SCSI_HHCB SCSI_HPTR hhcb)
{
   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_UEXACT16 i, numberScbs = hhcb->numberScbs;
   SCSI_SCB_DESCRIPTOR SCSI_HPTR scbDescriptor;

   /* reset software common to all modes of operations */
   SCSIhCommonResetSoftware(hhcb);

   /* setup q_new_scb */ 
   scbDescriptor = SCSI_GET_FREE_HEAD(hhcb);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_W_Q_NEXT_SCB),scbDescriptor->scbNumber);

   /* setup q_new_head */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_W_Q_NEW_HEAD),(SCSI_UEXACT8)0);
   hhcb->SCSI_HP_qNewHead = 0;

   /* setup q_new_tail */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_W_Q_NEW_TAIL),(SCSI_UEXACT8)0);

   /* setup q_done_base */
   SCSI_hSETADDRESSSCRATCH(hhcb,SCSI_W_Q_DONE_BASE,SCSI_QOUT_PTR_ARRAY_NEW(hhcb));

   /* setup q_done_element */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_W_Q_DONE_ELEMENT),(SCSI_UEXACT8)0);
   hhcb->SCSI_HP_qDoneElement = 0;

   /* setup q_done_pass (both scratch and shadow for HIM */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_W_Q_DONE_PASS),(SCSI_UEXACT8) 0xFF);
   hhcb->SCSI_HP_qDonePass = 0;

   /* initialize pointer to scb pointer array for sequencer */
   SCSI_hSETADDRESSSCRATCH(hhcb,SCSI_W_SCB_PTR_SCRATCH,(hhcb)->SCSI_SCB_BUFFER.virtualAddress);

   /* setup done SCB queue in host */
   /* @@ */
   for (i = 0; i < (numberScbs + 1) * sizeof(SCSI_QOUTFIFO_NEW); i++)
   {
      *(((SCSI_UEXACT8 SCSI_HPTR) SCSI_QOUT_PTR_ARRAY_NEW(hhcb)) + i) = 
                     (SCSI_UEXACT8) 0xFF;
   }

   /* invalidate waiting scb and active scb entry */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_W_ACTIVE_SCB),SCSI_NULL_SCB);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_W_WAITING_SCB),SCSI_NULL_SCB);

   OSD_OUTEXACT8(SCSI_AICREG(SCSI_W_Q_MASK), (SCSI_UEXACT8)(numberScbs + 1));
}
#endif

/*********************************************************************
*
*  SCSIhStandardAdvResetSoftware
*
*     Reset channel software
*
*  Return Value:  none
*                  
*  Parameters:    hhcb
*
*  Remarks:       This routine can be called either at initialization
*                 or exceptional condition
*                  
*********************************************************************/
#if SCSI_STANDARD_ADVANCED_MODE
void SCSIhStandardAdvResetSoftware (SCSI_HHCB SCSI_HPTR hhcb)
{
   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_UEXACT16 disconnectDisable = 0;
   SCSI_UEXACT16 numberScbs = hhcb->numberScbs;
   int i;
   SCSI_SCB_DESCRIPTOR SCSI_HPTR scbDescriptor;
   SCSI_UEXACT32 tnScbs;
   SCSI_UEXACT8 tvalue;
   SCSI_BUS_ADDRESS busAddress;

   /* reset software common to all modes of operations */
   SCSIhCommonResetSoftware(hhcb);

   /* set disconnect option */
   for (i = 0; i < SCSI_MAXDEV; i++)
   {
      disconnectDisable |=
                  ((! SCSI_DEVICE_TABLE(hhcb)[i].SCSI_DF_disconnectEnable) << i);
   }

   /* setup q_new_pointer */
   scbDescriptor = SCSI_GET_FREE_HEAD(hhcb);
   SCSI_hSETADDRESSSCRATCH(hhcb,SCSI_SADV_Q_NEW_POINTER,scbDescriptor->scbBuffer.virtualAddress);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SADV_Q_NEXT_SCB),scbDescriptor->scbNumber);

   /* setup q_new_head */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_HNSCB_QOFF),(SCSI_UEXACT8)0);
   hhcb->SCSI_HP_qNewHead = 0;

   /* setup q_new_tail */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SNSCB_QOFF),(SCSI_UEXACT8)0);

   /* setup sequencer done SCB queue offset  */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SDSCB_QOFF),(SCSI_UEXACT8)0);

   /* setup q_done_base */
   SCSI_hSETADDRESSSCRATCH(hhcb,SCSI_SADV_Q_DONE_BASE,SCSI_QOUT_PTR_ARRAY_NEW(hhcb));

   /* Initialize Q_DONE_POINTER with lo 4 bytes of done queue address. The hi */
   /* 4 bytes will come from Q_DONE_BASE when the sequencer uses              */
   /* Q_DONE_POINTER so we only need to init. the lo 4 bytes. */
   /* Get bus address and load it to scratch ram */
   busAddress = OSD_GET_BUS_ADDRESS(hhcb,SCSI_MC_LOCKED,SCSI_QOUT_PTR_ARRAY_NEW(hhcb));
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SADV_Q_DONE_POINTER+0),((SCSI_QUADLET SCSI_SPTR)&busAddress)->u8.order0);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SADV_Q_DONE_POINTER+1),((SCSI_QUADLET SCSI_SPTR)&busAddress)->u8.order1);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SADV_Q_DONE_POINTER+2),((SCSI_QUADLET SCSI_SPTR)&busAddress)->u8.order2);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SADV_Q_DONE_POINTER+3),((SCSI_QUADLET SCSI_SPTR)&busAddress)->u8.order3);

   /* setup q_done_head */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SADV_Q_DONE_HEAD),(SCSI_UEXACT8)0xFF);
 
   /* setup q_done_element */
   hhcb->SCSI_HP_qDoneElement = 0;

   /* setup q_done_pass (both scratch and shadow for HIM */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SADV_Q_DONE_PASS),(SCSI_UEXACT8) 0x00);
 
   hhcb->SCSI_HP_qDonePass = 0;

   /* setup q_exe_head */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SADV_Q_EXE_HEAD),(SCSI_UEXACT8)0xFF);
   
   /* setup done SCB queue in host */
   /* @@ */
   for (i = 0; i < (int) (((SCSI_UEXACT16) hhcb->numberScbs + 1) * 
                     sizeof(SCSI_QOUTFIFO_NEW)); i++)
   {
      *(((SCSI_UEXACT8 SCSI_HPTR) SCSI_QOUT_PTR_ARRAY_NEW(hhcb)) + i) = 
                              (SCSI_UEXACT8) 0xFF;
   }

   /* Invalidate waiting scb and active scb entry */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SADV_ACTIVE_SCB),SCSI_NULL_SCB);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SADV_WAITING_SCB),SCSI_NULL_SCB);

   /* Program the appropriate encoded value for the SCSI_SCB_QSIZE bits */
   tnScbs = (SCSI_UEXACT32) numberScbs + 1;
   tvalue = OSD_INEXACT8(SCSI_AICREG(SCSI_SADV_Q_MASK))
            & ~((SCSI_UEXACT8)SCSI_SCB_QSIZE);
   while (tnScbs)
   {
      tvalue++;
      tnScbs /= 2;
   }
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SADV_Q_MASK),tvalue-3);

   /* Clear all the breakpoint bits */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SADV_ENT_PT_BITMAP),0);

   /* Set interrupt threshold timeout value.  Each sequencer's instruction */
   /* execution time is 50-100ns (10/20 MIPS).  We want to wait for        */
   /* approximately 5us in the sequencer idle loop before generating       */
   /* command complete interrupt.  There are 17 instructions repetitively  */
   /* executed in this idle loop if there is no I/O activity.  Take that   */
   /* into the account and pick the 50ns for each instruction.  We come up */
   /* with the interrupt threshold timeout value to be 6.                  */
   /* After running the benchmark, the value come out to be 24.            */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SADV_INTR_THRSH_TIMEOUT1),24);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SADV_INTR_THRSH_TIMEOUT2),24);

   /* setup workaround flag for arbitor hung (Bayone/Scimitar only) */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SADV_WW_FLAG),
      (hhcb->SCSI_HP_wwArbitorHung == 0) ? 0 : SCSI_SADV_ARB_HUNG);
}
#endif


/*********************************************************************
*
*  SCSIhSwappingAdvResetSoftware
*
*     Reset channel software for swapping advanced mode
*
*  Return Value:  none
*                  
*  Parameters:    hhcb
*
*  Remarks:       This routine can be called either at initialization
*                 or exceptional condition
*                  
*********************************************************************/
#if SCSI_SWAPPING_ADVANCED_MODE
void SCSIhSwappingAdvResetSoftware (SCSI_HHCB SCSI_HPTR hhcb)
{
   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_UEXACT16 i, numberScbs = hhcb->numberScbs;
   SCSI_SCB_DESCRIPTOR SCSI_HPTR scbDescriptor;
   SCSI_UEXACT32 tnScbs;
   SCSI_UEXACT8 tvalue;
   SCSI_BUS_ADDRESS busAddress;

   /* reset software common to all modes of operations */
   SCSIhCommonResetSoftware(hhcb);

#if SCSI_TARGET_OPERATION
   /* Initialize SCBPTR register to 0 - this is required
    * for the target mode sequencer - in case we reload 
    * without powering off system this is not protected by 
    * target mode. Revisit this!!
    */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCBPTR),(SCSI_UEXACT8)0);

   if (hhcb->SCSI_HF_targetMode)
   {
      /* setup q_est_scb */ 
      scbDescriptor = SCSI_GET_FREE_EST_HEAD(hhcb);
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_WADV_Q_EST_NEXT_SCB),scbDescriptor->scbNumber);

#if SCSI_MAILBOX_ENABLE
      /* Read value in sequencer portion (3-0) of the mailbox register
       * and set host portion(7-4) to same value.
       */
      hhcb->SCSI_HP_qEstHead = OSD_INEXACT8(SCSI_AICREG(SCSI_WADV_Q_EST_HEAD)) << 4;
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_WADV_Q_EST_HEAD),(SCSI_UEXACT8)(hhcb->SCSI_HP_qEstHead));
#else
      /* setup q_est_head */
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_WADV_HESCB_QOFF),(SCSI_UEXACT8)0);
      hhcb->SCSI_HP_qEstHead = 0;

      /* setup q_est_tail */
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_WADV_SESCB_QOFF),(SCSI_UEXACT8)0);
#endif /* SCSI_MAILBOX_ENABLE */
      
   }
#endif /* SCSI_TARGET_OPERATION */   
 
   /* setup q_new_scb */ 
   scbDescriptor = SCSI_GET_FREE_HEAD(hhcb);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_WADV_Q_NEXT_SCB),scbDescriptor->scbNumber);

   /* setup q_new_head */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_HNSCB_QOFF),(SCSI_UEXACT8)0);
   hhcb->SCSI_HP_qNewHead = 0;

   /* setup q_new_tail */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SNSCB_QOFF),(SCSI_UEXACT8)0);

   /* setup sequencer done SCB queue offset  */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SDSCB_QOFF),(SCSI_UEXACT8)0);

   /* setup q_done_base */
   SCSI_hSETADDRESSSCRATCH(hhcb,SCSI_WADV_Q_DONE_BASE,SCSI_QOUT_PTR_ARRAY_NEW(hhcb));

   /* Initialize Q_DONE_POINTER with lo 4 bytes of done queue address. The hi */
   /* 4 bytes will come from Q_DONE_BASE when the sequencer uses              */
   /* Q_DONE_POINTER so we only need to init. the lo 4 bytes. */
   /* Get bus address and load it to scratch ram */
   busAddress = OSD_GET_BUS_ADDRESS(hhcb,SCSI_MC_LOCKED,SCSI_QOUT_PTR_ARRAY_NEW(hhcb));
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_WADV_Q_DONE_POINTER+0),((SCSI_QUADLET SCSI_SPTR)&busAddress)->u8.order0);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_WADV_Q_DONE_POINTER+1),((SCSI_QUADLET SCSI_SPTR)&busAddress)->u8.order1);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_WADV_Q_DONE_POINTER+2),((SCSI_QUADLET SCSI_SPTR)&busAddress)->u8.order2);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_WADV_Q_DONE_POINTER+3),((SCSI_QUADLET SCSI_SPTR)&busAddress)->u8.order3);

   /* setup q_done_element */
   hhcb->SCSI_HP_qDoneElement = 0;

   /* setup q_done_pass (both scratch and shadow for HIM */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_WADV_Q_DONE_PASS),(SCSI_UEXACT8) 0x00);
   hhcb->SCSI_HP_qDonePass = 0;

   /* initialize pointer to scb pointer array for sequencer */
   SCSI_hSETADDRESSSCRATCH(hhcb,SCSI_WADV_SCB_PTR_SCRATCH,(hhcb)->SCSI_SCB_BUFFER.virtualAddress);

   /* setup done SCB queue in host */
   /* @@ */
   for (i = 0; i < (numberScbs + 1) * sizeof(SCSI_QOUTFIFO_NEW); i++)
   {
      *(((SCSI_UEXACT8 SCSI_HPTR) SCSI_QOUT_PTR_ARRAY_NEW(hhcb)) + i) = 
                     (SCSI_UEXACT8) 0xFF;
   }

   /* invalidate waiting scb and active scb entry */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_WADV_ACTIVE_SCB),SCSI_NULL_SCB);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_WADV_WAITING_SCB),SCSI_NULL_SCB);

   /* Program the appropriate encoded value for the SCB_QSIZE bits */
   tnScbs = (SCSI_UEXACT32) numberScbs + 1;
   tvalue = OSD_INEXACT8(SCSI_AICREG(SCSI_WADV_Q_MASK))
            & ~((SCSI_UEXACT8)SCSI_SCB_QSIZE);
   while (tnScbs)
   {
      tvalue++;
      tnScbs /= 2;
   }
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_WADV_Q_MASK),tvalue-3);

   /* Clear all the breakpoint bits */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_WADV_ENT_PT_BITMAP),0);

   /* If we've cleared the breakpoint bits then we should
    * complete any outstanding DISABLE_ID hiob.
    */
   SCSI_hTARGETDISABLEID(hhcb);

#if !SCSI_TARGET_OPERATION
   /* Set interrupt threshold timeout value.  Each sequencer's instruction */
   /* execution time is 50-100ns (10/20 MIPS).  We want to wait for        */
   /* approximately 5us in the sequencer idle loop before generating       */
   /* command complete interrupt.  There are 17 instructions repetitively  */
   /* executed in this idle loop if there is no I/O activity.  Take that   */
   /* into the account and pick the 50ns for each instruction.  We come up */
   /* with the interrupt threshold timeout value to be 6.                  */
   /* After running the benchmark, the value come out to be 24.            */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_WADV_INTR_THRSH_TIMEOUT1),24);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_WADV_INTR_THRSH_TIMEOUT2),24);

#endif /* !SCSI_TARGET_OPERATION */
}
#endif

/*********************************************************************
*
*  SCSIhStandard160mResetSoftware
*
*     Reset channel software for standard Ultra 160m mode
*
*  Return Value:  none
*                  
*  Parameters:    hhcb
*
*  Remarks:       This routine can be called either at initialization
*                 or exceptional condition
*                  
*********************************************************************/
#if SCSI_STANDARD_160M_MODE
void SCSIhStandard160mResetSoftware (SCSI_HHCB SCSI_HPTR hhcb)
{
   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_UEXACT16 disconnectDisable = 0;
   SCSI_UEXACT16 numberScbs = hhcb->numberScbs;
   SCSI_SCB_DESCRIPTOR SCSI_HPTR scbDescriptor;
   SCSI_UEXACT32 tnScbs;
   SCSI_UEXACT8 tvalue;
   int i;
   SCSI_BUS_ADDRESS busAddress;

   /* reset software common to all modes of operations */
   SCSIhCommonResetSoftware(hhcb);

   /* set disconnect option */
   for (i = 0; i < SCSI_MAXDEV; i++)
   {
      disconnectDisable |=
                  ((! SCSI_DEVICE_TABLE(hhcb)[i].SCSI_DF_disconnectEnable) << i);
   }

   /* setup q_new_pointer */
   scbDescriptor = SCSI_GET_FREE_HEAD(hhcb);
   SCSI_hSETADDRESSSCRATCH(hhcb,SCSI_S160M_Q_NEW_POINTER,scbDescriptor->scbBuffer.virtualAddress);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_S160M_Q_NEXT_SCB),scbDescriptor->scbNumber);

   /* setup q_new_head */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_HNSCB_QOFF),(SCSI_UEXACT8)0);
   hhcb->SCSI_HP_qNewHead = 0;

   /* setup q_new_tail */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SNSCB_QOFF),(SCSI_UEXACT8)0);

   /* setup sequencer done SCB queue offset  */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SDSCB_QOFF),(SCSI_UEXACT8)0);

   /* setup q_done_base */
   SCSI_hSETADDRESSSCRATCH(hhcb,SCSI_S160M_Q_DONE_BASE,SCSI_QOUT_PTR_ARRAY_NEW(hhcb));

   /* Initialize Q_DONE_POINTER with lo 4 bytes of done queue address. The hi */
   /* 4 bytes will come from Q_DONE_BASE when the sequencer uses              */
   /* Q_DONE_POINTER so we only need to init. the lo 4 bytes. */
   /* Get bus address and load it to scratch ram */
   busAddress = OSD_GET_BUS_ADDRESS(hhcb,SCSI_MC_LOCKED,SCSI_QOUT_PTR_ARRAY_NEW(hhcb));
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_S160M_Q_DONE_POINTER+0),((SCSI_QUADLET SCSI_SPTR)&busAddress)->u8.order0);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_S160M_Q_DONE_POINTER+1),((SCSI_QUADLET SCSI_SPTR)&busAddress)->u8.order1);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_S160M_Q_DONE_POINTER+2),((SCSI_QUADLET SCSI_SPTR)&busAddress)->u8.order2);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_S160M_Q_DONE_POINTER+3),((SCSI_QUADLET SCSI_SPTR)&busAddress)->u8.order3);

   /* setup q_done_head */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_S160M_Q_DONE_HEAD),(SCSI_UEXACT8)0xFF);
 
   /* setup q_done_element */
   hhcb->SCSI_HP_qDoneElement = 0;

   /* setup q_done_pass (both scratch and shadow for HIM */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_S160M_Q_DONE_PASS),(SCSI_UEXACT8) 0x00);
   hhcb->SCSI_HP_qDonePass = 0;

   /* setup q_exe_head */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_S160M_Q_EXE_HEAD),(SCSI_UEXACT8)0xFF);
   
   /* setup done SCB queue in host */
   for (i = 0; i < (int) (((SCSI_UEXACT16) hhcb->numberScbs + 1) * 
                     sizeof(SCSI_QOUTFIFO_NEW)); i++)
   {
      *(((SCSI_UEXACT8 SCSI_HPTR) SCSI_QOUT_PTR_ARRAY_NEW(hhcb)) + i) = 
                              (SCSI_UEXACT8) 0xFF;
   }

   /* Invalidate waiting scb and active scb entry */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_S160M_ACTIVE_SCB),SCSI_NULL_SCB);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_S160M_WAITING_SCB),SCSI_NULL_SCB);

   /* Program the appropriate encoded value for the SCSI_SCB_QSIZE bits */
   tnScbs = (SCSI_UEXACT32) numberScbs + 1;
   tvalue = OSD_INEXACT8(SCSI_AICREG(SCSI_S160M_Q_MASK))
            & ~((SCSI_UEXACT8)SCSI_SCB_QSIZE);
   while (tnScbs)
   {
      tvalue++;
      tnScbs /= 2;
   }
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_S160M_Q_MASK),tvalue-3);

   /* Clear all the breakpoint bits */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_S160M_ENT_PT_BITMAP),0);

   /* Set interrupt threshold timeout value.  Each sequencer's instruction */
   /* execution time is 50-100ns (10/20 MIPS).  We want to wait for        */
   /* approximately 5us in the sequencer idle loop before generating       */
   /* command complete interrupt.  There are 17 instructions repetitively  */
   /* executed in this idle loop if there is no I/O activity.  Take that   */
   /* into the account and pick the 50ns for each instruction.  We come up */
   /* with the interrupt threshold timeout value to be 6.                  */
   /* After running the benchmark, the value come out to be 24.            */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_S160M_INTR_THRSH_TIMEOUT1),24);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_S160M_INTR_THRSH_TIMEOUT2),24);

}
#endif /* SCSI_STANDARD_160M_MODE */

/*********************************************************************
*
*  SCSIhSwapping160mResetSoftware
*
*     Reset channel software for swapping Ultra 160m mode
*
*  Return Value:  none
*                  
*  Parameters:    hhcb
*
*  Remarks:       This routine can be called either at initialization
*                 or exceptional condition
*                  
*********************************************************************/
#if SCSI_SWAPPING_160M_MODE
void SCSIhSwapping160mResetSoftware (SCSI_HHCB SCSI_HPTR hhcb)
{
   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_UEXACT16 i, numberScbs = hhcb->numberScbs;
   SCSI_SCB_DESCRIPTOR SCSI_HPTR scbDescriptor;
   SCSI_UEXACT32 tnScbs;
   SCSI_UEXACT8 tvalue;
   SCSI_BUS_ADDRESS busAddress;

   /* reset software common to all modes of operations */
   SCSIhCommonResetSoftware(hhcb);

#if SCSI_TARGET_OPERATION
   /* Initialize SCBPTR register to 0 - this is required
    * for the target mode sequencer - in case we reload 
    * without powering off system this is not protected by 
    * target mode. Revisit this!!
    */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCBPTR),(SCSI_UEXACT8)0);

   if (hhcb->SCSI_HF_targetMode)
   {
      /* setup q_est_scb */ 
      scbDescriptor = SCSI_GET_FREE_EST_HEAD(hhcb);
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_W160M_Q_EST_NEXT_SCB),scbDescriptor->scbNumber);

      /* setup q_est_head */
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_W160M_HESCB_QOFF),(SCSI_UEXACT8)0);
      hhcb->SCSI_HP_qEstHead = 0;

      /* setup q_est_tail */
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_W160M_SESCB_QOFF),(SCSI_UEXACT8)0);

      /* Initialize TARGET_FLAG scratch register to 0 for sequencer */
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_W160M_TARGET_FLAG),(SCSI_UEXACT8)0);
   }
#endif /* SCSI_TARGET_OPERATION */

   /* setup q_new_scb */ 
   scbDescriptor = SCSI_GET_FREE_HEAD(hhcb);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_W160M_Q_NEXT_SCB),scbDescriptor->scbNumber);

   /* setup q_new_head */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_HNSCB_QOFF),(SCSI_UEXACT8)0);
   hhcb->SCSI_HP_qNewHead = 0;

   /* setup q_new_tail */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SNSCB_QOFF),(SCSI_UEXACT8)0);

   /* setup sequencer done SCB queue offset  */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SDSCB_QOFF),(SCSI_UEXACT8)0);

   /* setup q_done_base */
   SCSI_hSETADDRESSSCRATCH(hhcb,SCSI_W160M_Q_DONE_BASE,SCSI_QOUT_PTR_ARRAY_NEW(hhcb));

   /* Initialize Q_DONE_POINTER with lo 4 bytes of done queue address. The hi */
   /* 4 bytes will come from Q_DONE_BASE when the sequencer uses              */
   /* Q_DONE_POINTER so we only need to init. the lo 4 bytes. */
   /* Get bus address and load it to scratch ram */
   busAddress = OSD_GET_BUS_ADDRESS(hhcb,SCSI_MC_LOCKED,SCSI_QOUT_PTR_ARRAY_NEW(hhcb));
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_W160M_Q_DONE_POINTER+0),((SCSI_QUADLET SCSI_SPTR)&busAddress)->u8.order0);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_W160M_Q_DONE_POINTER+1),((SCSI_QUADLET SCSI_SPTR)&busAddress)->u8.order1);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_W160M_Q_DONE_POINTER+2),((SCSI_QUADLET SCSI_SPTR)&busAddress)->u8.order2);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_W160M_Q_DONE_POINTER+3),((SCSI_QUADLET SCSI_SPTR)&busAddress)->u8.order3);

   /* setup q_done_element */
   hhcb->SCSI_HP_qDoneElement = 0;

   /* setup q_done_pass (both scratch and shadow for HIM */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_W160M_Q_DONE_PASS),(SCSI_UEXACT8) 0x00);
   hhcb->SCSI_HP_qDonePass = 0;

   /* initialize pointer to scb pointer array for sequencer */
   SCSI_hSETADDRESSSCRATCH(hhcb,SCSI_W160M_SCB_PTR_SCRATCH,(hhcb)->SCSI_SCB_BUFFER.virtualAddress);

   /* setup done SCB queue in host */
   for (i = 0; i < (numberScbs + 1) * sizeof(SCSI_QOUTFIFO_NEW); i++)
   {
      *(((SCSI_UEXACT8 SCSI_HPTR) SCSI_QOUT_PTR_ARRAY_NEW(hhcb)) + i) = 
                     (SCSI_UEXACT8) 0xFF;
   }

   /* invalidate waiting scb and active scb entry */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_W160M_ACTIVE_SCB),SCSI_NULL_SCB);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_W160M_WAITING_SCB),SCSI_NULL_SCB);

   /* Program the appropriate encoded value for the SCB_QSIZE bits */
   tnScbs = (SCSI_UEXACT32) numberScbs + 1;
   tvalue = OSD_INEXACT8(SCSI_AICREG(SCSI_W160M_Q_MASK))
            & ~((SCSI_UEXACT8)SCSI_SCB_QSIZE);
   while (tnScbs)
   {
      tvalue++;
      tnScbs /= 2;
   }
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_W160M_Q_MASK),tvalue-3);

   /* Clear all the breakpoint bits */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_W160M_ENT_PT_BITMAP),0);

   /* Set interrupt threshold timeout value.  Each sequencer's instruction */
   /* execution time is 50-100ns (10/20 MIPS).  We want to wait for        */
   /* approximately 5us in the sequencer idle loop before generating       */
   /* command complete interrupt.  There are 17 instructions repetitively  */
   /* executed in this idle loop if there is no I/O activity.  Take that   */
   /* into the account and pick the 50ns for each instruction.  We come up */
   /* with the interrupt threshold timeout value to be 6.                  */
   /* After running the benchmark, the value come out to be 24.            */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_W160M_INTR_THRSH_TIMEOUT1),24);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_W160M_INTR_THRSH_TIMEOUT2),24);

}
#endif /* SCSI_SWAPPING_160M_MODE */

/*********************************************************************
*
*  SCSIhCommonResetSoftware
*
*     Reset software common to all modes of operations
*
*  Return Value:  none
*                  
*  Parameters:    hhcb
*
*  Remarks:       This routine can be called either at initialization
*                 or exceptional condition
*                  
*********************************************************************/
void SCSIhCommonResetSoftware (SCSI_HHCB SCSI_HPTR hhcb)
{
   SCSI_UEXACT8 i;

   /* set active pointer array to null */
   for (i = 0; i < hhcb->numberScbs; i++)
   {
      SCSI_ACTPTR[i] = SCSI_NULL_HIOB;
   }
   
   hhcb->numberSCBAborts = 0;
   
   /* set disconnect and transfer option */
   SCSIhCheckSyncNego(hhcb);

   /* make it ready to assign scb buffer (from scb buffer array) */
   SCSI_hSETUP_ASSIGN_SCB_BUFFER(hhcb);
}

/*********************************************************************
*
*  SCSIhSetAddressScratchBA32
*
*     Translate virtual address to 32 bits bus address and
*     set the 32 bits bus address to scratch ram
*
*  Return Value:  none
*                  
*  Parameters:    hhcb
*                 scratch address
*                 virtual address
*
*  Remarks:       This routine can be called either at initialization
*                 or exceptional condition
*                  
*********************************************************************/
#if OSD_BUS_ADDRESS_SIZE == 32
void SCSIhSetAddressScratchBA32 (SCSI_HHCB SCSI_HPTR hhcb,
                                 SCSI_UEXACT8 scratchAddress,
                                 void SCSI_HPTR virtualAddress)
{
   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_BUS_ADDRESS busAddress;

   /* get bus address */
   busAddress = OSD_GET_BUS_ADDRESS(hhcb,SCSI_MC_LOCKED,virtualAddress);

   /* load bus address to scratch ram */
   OSD_OUTEXACT8(SCSI_AICREG(scratchAddress+0),((SCSI_QUADLET SCSI_SPTR)&busAddress)->u8.order0);
   OSD_OUTEXACT8(SCSI_AICREG(scratchAddress+1),((SCSI_QUADLET SCSI_SPTR)&busAddress)->u8.order1);
   OSD_OUTEXACT8(SCSI_AICREG(scratchAddress+2),((SCSI_QUADLET SCSI_SPTR)&busAddress)->u8.order2);
   OSD_OUTEXACT8(SCSI_AICREG(scratchAddress+3),((SCSI_QUADLET SCSI_SPTR)&busAddress)->u8.order3);
}
#endif

/*********************************************************************
*
*  SCSIhSetAddressScratchBA64
*
*     Translate virtual address to 64 bits bus address and
*     set the 64 bits bus address to scratch ram
*
*  Return Value:  none
*                  
*  Parameters:    hhcb
*                 scratch address
*                 virtual address
*
*  Remarks:       This routine can be called either at initialization
*                 or exceptional condition
*                  
*********************************************************************/
#if OSD_BUS_ADDRESS_SIZE == 64
void SCSIhSetAddressScratchBA64 (SCSI_HHCB SCSI_HPTR hhcb,
                                 SCSI_UEXACT8 scratchAddress,
                                 void SCSI_HPTR virtualAddress)
{
   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_BUS_ADDRESS busAddress;

   /* get bus address */
   busAddress = OSD_GET_BUS_ADDRESS(hhcb,SCSI_MC_LOCKED,virtualAddress);

   /* load bus address to scratch ram */
   OSD_OUTEXACT8(SCSI_AICREG(scratchAddress+0),((SCSI_OCTLET SCSI_SPTR)&busAddress)->u8.order0);
   OSD_OUTEXACT8(SCSI_AICREG(scratchAddress+1),((SCSI_OCTLET SCSI_SPTR)&busAddress)->u8.order1);
   OSD_OUTEXACT8(SCSI_AICREG(scratchAddress+2),((SCSI_OCTLET SCSI_SPTR)&busAddress)->u8.order2);
   OSD_OUTEXACT8(SCSI_AICREG(scratchAddress+3),((SCSI_OCTLET SCSI_SPTR)&busAddress)->u8.order3);
   OSD_OUTEXACT8(SCSI_AICREG(scratchAddress+4),((SCSI_OCTLET SCSI_SPTR)&busAddress)->u8.order4);
   OSD_OUTEXACT8(SCSI_AICREG(scratchAddress+5),((SCSI_OCTLET SCSI_SPTR)&busAddress)->u8.order5);
   OSD_OUTEXACT8(SCSI_AICREG(scratchAddress+6),((SCSI_OCTLET SCSI_SPTR)&busAddress)->u8.order6);
   OSD_OUTEXACT8(SCSI_AICREG(scratchAddress+7),((SCSI_OCTLET SCSI_SPTR)&busAddress)->u8.order7);
}
#endif

/*********************************************************************
*
*  SCSIhStandardResetBTA
*
*     Reset standard busy target array
*
*  Return Value:  none
*                  
*  Parameters:    hhcb
*
*  Remarks:       This routine can be called either at initialization
*                 or exceptional condition
*                  
*********************************************************************/
#if SCSI_STANDARD_MODE
void SCSIhStandardResetBTA (SCSI_HHCB SCSI_HPTR hhcb)
{
   int i;
   SCSI_UEXACT8 n;

   /* check how big is the busy target array */
   /* note that for now n is based on the physical size of the SCB RAM */
   /* in the future if command chaining for the RAID environment is */
   /* implemented, then we will need to base n on the combination of */
   /* hhcb->numberHWScbs and SCSI_hTARG_LUN_MASK0(hhcb) which defines */
   /* how a shared SCB RAM is partitioned among multiple host adapters. */
   n = hhcb->numberHWScbs;

   for (i = 0; i <= n; i++)
   {
      SCSIhStandardIndexClearBusy(hhcb,(SCSI_UEXACT8)i);
   }
}
#endif


/*********************************************************************
*
*  SCSIhSwappingResetBTA
*
*     Reset busy target array for swapping mode
*
*  Return Value:  none
*                  
*  Parameters:    hhcb
*
*  Remarks:       This routine can be called either at initialization
*                 or exceptional condition
*                  
*********************************************************************/
#if SCSI_SWAPPING_MODE
void SCSIhSwappingResetBTA (SCSI_HHCB SCSI_HPTR hhcb)
{
   int i;

   for (i = 0; i < 16; i++)
   {
      SCSIhSwappingIndexClearBusy(hhcb,(SCSI_UEXACT8)i);
   }
}
#endif


/*********************************************************************
*
*  SCSIhStandard64TargetClearBusy
*
*  Clear target busy array for standard 64 mode
*
*  Return Value:  None
*                  
*  Parameters: hhcb
*              hiob
*
*  Remarks:                
*                  
*********************************************************************/
#if SCSI_STANDARD_MODE
void SCSIhStandardTargetClearBusy (SCSI_HHCB SCSI_HPTR hhcb,
                                   SCSI_HIOB SCSI_IPTR hiob)
{
   SCSI_UNIT_CONTROL SCSI_IPTR targetUnit = SCSI_TARGET_UNIT(hiob);
   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_UEXACT8 index;
   SCSI_UEXACT8 scbptr;

   scbptr = OSD_INEXACT8(SCSI_AICREG(SCSI_SCBPTR));         /* Save original SCBPTR */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCBPTR),hiob->scbNumber); /* and set new SCBPTR   */
   index = (SCSI_UEXACT8) targetUnit->scsiID;
   if (hhcb->SCSI_HF_multiTaskLun)
   {
      index += (SCSI_UEXACT8)(targetUnit->lunID * 16);
   }
   SCSIhStandardIndexClearBusy(hhcb,(SCSI_UEXACT8)index);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCBPTR), scbptr);      /* Restore original SCBPTR */
}
#endif


/*********************************************************************
*
*  SCSIhSwappingTargetClearBusy
*
*  Clear target busy array for swapping mode
*
*  Return Value:  None
*                  
*  Parameters: hhcb
*              hiob
*
*  Remarks:                
*                  
*********************************************************************/
#if SCSI_SWAPPING_MODE
void SCSIhSwappingTargetClearBusy (SCSI_HHCB SCSI_HPTR hhcb,
                                     SCSI_HIOB SCSI_IPTR hiob)
{
   SCSI_UNIT_CONTROL SCSI_IPTR targetUnit = SCSI_TARGET_UNIT(hiob);

   SCSIhSwappingIndexClearBusy(hhcb,(SCSI_UEXACT8)targetUnit->scsiID);
}
#endif


/*********************************************************************
*
*  SCSIhStandardIndexClearBusy
*
*  Clear indexed target busy array for standard 64 mode
*
*  Return Value:  none
*                  
*  Parameters: host task set handle
*              hiob
*
*  Remarks:                
*                  
*********************************************************************/
#if SCSI_STANDARD_MODE
void SCSIhStandardIndexClearBusy (SCSI_HHCB SCSI_HPTR hhcb,
                                  SCSI_UEXACT8 index)
{
   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;

   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCBPTR), index);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_hBTATABLE(hhcb)), SCSI_NULL_SCB);
}
#endif


/*********************************************************************
*
*  SCSIhSwappingIndexClearBusy
*
*  Clear indexed target busy array for swapping mode
*
*  Return Value:  none
*                  
*  Parameters: host task set handle
*              hiob
*
*  Remarks:                
*                  
*********************************************************************/
#if SCSI_SWAPPING_MODE
void SCSIhSwappingIndexClearBusy (SCSI_HHCB SCSI_HPTR hhcb,
                                    SCSI_UEXACT8 index)
{
   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;

   OSD_OUTEXACT8(SCSI_AICREG(SCSI_hBTATABLE(hhcb)+index),SCSI_NULL_SCB);
}
#endif


/*********************************************************************
*
*  SCSIhResetScsi
*
*     Reset SCSI bus
*
*  Return Value:  0 - if SCSI bus is good after the reset
*                 1 - otherwise
*                  
*  Parameters:    hhcb
*
*  Remarks:       This routine can be called either at initialization
*                 or exceptional condition
*                  
*********************************************************************/
HIM_UEXACT8 SCSIhResetScsi (SCSI_HHCB SCSI_HPTR hhcb)
{
   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_UEXACT8 timer0, timer1;
   SCSI_UEXACT8 simode0;
   SCSI_UEXACT8 intstat;
   SCSI_UINT32 j = 0;          /* time out counter */
   SCSI_UEXACT8 i;

   timer0 = OSD_INEXACT8(SCSI_AICREG(SCSI_hRESCNT_BASE(hhcb)+0));    /* save SCB16 and SCB17 */
   timer1 = OSD_INEXACT8(SCSI_AICREG(SCSI_hRESCNT_BASE(hhcb)+1));
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_hRESCNT_BASE(hhcb)+0), 0xff);      /* set timer to 64 msec */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_hRESCNT_BASE(hhcb)+1), 0x0);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SEQADDR0), (SCSI_UEXACT8) SCSI_hATN_TMR_ENTRY(hhcb) >> 2);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SEQADDR1), (SCSI_UEXACT8) SCSI_hATN_TMR_ENTRY(hhcb) >> 10);
   simode0 = OSD_INEXACT8(SCSI_AICREG(SCSI_SIMODE0));          /* Disable all SCSI */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SIMODE0),0);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SIMODE1),0);                 /* clear interrupt */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_CLRINT), SCSI_CLRSEQINT | SCSI_CLRSCSINT | SCSI_CLRCMDINT | SCSI_CLRBRKINT);

   /* Due to the Bayonet HW that needs to wait a total of 200ms for the scsi */
   /* bus to stable after the chip reset.  Before this time, it does not     */
   /* allow any scsi signals to send to scsi bus.  Therefore, we must wait   */
   /* for the scsi bus stable as well base on bit ENAB20 or ENAB40 that set  */
   /* by the HW when it confirmed that the scsi bus is stabled. */
   SCSI_hWAITFORBUSSTABLE(hhcb);

   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSISEQ), SCSI_SCSIRSTO);    /* assert RESET SCSI bus   */
   SCSIhUnPause(hhcb);                                         /* unpause sequencer */

   while(1)
   {
      if (OSD_INEXACT8(SCSI_AICREG(SCSI_HCNTRL)) & SCSI_PAUSEACK)
      {
         if (SCSI_hREAD_INTSTAT(hhcb,scsiRegister) &
            (SCSI_SCSIINT | SCSI_SEQINT))
         {
            break;
         }
      }
      if (j++ >= 400000)  /* time out counter check */
      {
         /* must make sure chip paused for Bayonet */
         if (!(OSD_INEXACT8(SCSI_AICREG(SCSI_HCNTRL)) & SCSI_PAUSEACK))
         {
            SCSIhPauseAndWait(hhcb);
         }
         break;
      }
      OSD_TIMER(hhcb);
   }

   SCSI_hSEQSTRETCHWORKAROUND(hhcb); /* sequencer stretch workaround */

   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSISEQ), 00);                  /* remove RESET SCSI bus   */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_CLRSINT1), SCSI_CLRSCSIRSTI);   /* Patch for unexpected */
                                                                  /* scsi interrupt       */
   /* Clear any possible pending scsi interrupt that may exists   */
   /* at this point.                                              */
   intstat = OSD_INEXACT8(SCSI_AICREG(SCSI_SSTAT1));
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_CLRSINT0), 0xff);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_CLRSINT1),
                 (intstat & ~(SCSI_PHASEMIS | SCSI_PHASECHG)));

   OSD_OUTEXACT8(SCSI_AICREG(SCSI_CLRINT), SCSI_CLRSEQINT | SCSI_CLRSCSINT);     /* clear interrupt */

   SCSIhDelayCount500us(hhcb, 2);      /* delay 1 millisec = 2 * 500 usec */

   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SIMODE0),simode0); /* Restore SIMODE0 and */
   SCSI_hENIOERR(hhcb);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SIMODE1), SCSI_ENSCSIPERR + SCSI_ENSELTIMO + SCSI_ENSCSIRST);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_hRESCNT_BASE(hhcb)+0), timer0);            /* restore SCB16 and */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_hRESCNT_BASE(hhcb)+1), timer1);                       /* SCB 17 */

   /* We need to clear skipScam flag here so that */
   /* the scam protocol can be invoked. */
   hhcb->SCSI_HF_skipScam = 0;

   /* Set here for the PAC to invoke suppress negotiation. */
   /* This to make sure the internal Bus Scan always talk async. to devices */
   hhcb->SCSI_HF_resetSCSI = 1;

   for (i=0;i<16;i++)
   {
      SCSI_DEVICE_TABLE(hhcb)[i].SCSI_DF_resetSCSI = 1;
   }

#if SCSI_TARGET_OPERATION   
   /* If  target mode enabled need to reset the holding
    * SCSI bus indicator or SCSIhTargetClearNexus will fail
    */ 
   if (hhcb->SCSI_HF_targetMode && hhcb->SCSI_HF_targetScsiBusHeld)
   {
      hhcb->SCSI_HF_targetScsiBusHeld = 0;
      hhcb->SCSI_HP_nexusHoldingBus->SCSI_XF_busHeld = 0;
   }
#endif /* SCSI_TARGET_OPERATION */

   if (!SCSIhBusWasDead(hhcb))
   {
      return(0);
   }
   else
   {
      return(1);
   }
}

/*********************************************************************
*
*  SCSIhScsiBusReset
*
*     Perform SCSI Bus Reset request
*
*  Return Value:  none
*
*  Parameters:    hhcb
*                 hiob
*
*  Remarks:
*
*********************************************************************/
void SCSIhScsiBusReset (SCSI_HHCB SCSI_HPTR hhcb,
                        SCSI_HIOB SCSI_IPTR hiob)
{
   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_UEXACT8 scbNumber;
   
   /* Pause the Sequencer */
   SCSIhPauseAndWait(hhcb);

   SCSI_hSTACKBUGFIX(hhcb);

   /* It is a request from the OSM to return the active target at the */
   /* time of reset.  We also check to make sure it is a valid one. */
   scbNumber = OSD_INEXACT8(SCSI_AICREG(SCSI_hACTIVE_SCB(hhcb)));
   if (scbNumber != SCSI_NULL_SCB)
   {
      /* Get active HIOB */
      SCSI_ACTIVE_HIOB(hiob) = (void SCSI_IPTR) SCSI_ACTPTR[scbNumber];
   }
   else
   {
      /* Indicate no active HIOB */
      SCSI_ACTIVE_HIOB(hiob) = (void SCSI_IPTR) SCSI_NULL_HIOB;
   }

   /* Issue SCSI bus reset */
   if (!SCSIhResetScsi(hhcb))
   {
      hiob->stat = SCSI_SCB_COMP;
   }
   else
   {
      hiob->stat = SCSI_SCB_ERR;
      hiob->haStat = SCSI_HOST_ABT_CHANNEL_FAILED;
   }

   /* Delay 2 seconds after scsi reset for slow devices to settle down. */
   SCSI_hRESETDELAY(hhcb, (hhcb->resetDelay*2)); /* remove??? */

   /* Reset channel hardware's parameters */
   SCSIhResetChannelHardware(hhcb);

   /* Block upper layers from issuing Establish Connection HIOBs */
   SCSI_hTARGETHOLDESTHIOBS(hhcb,1);

   /* Abort any active HIOB(s) */
   SCSIhAbortChannel(hhcb, SCSI_HOST_ABT_BUS_RST);

   /* Reset channel software's parameters */
   SCSI_hRESETSOFTWARE(hhcb);

   /* Unblock upper layers from issuing Establish Connection HIOBs */
   SCSI_hTARGETHOLDESTHIOBS(hhcb,0);

   /* Now, restart sequencer */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SEQADDR0), (SCSI_UEXACT8)SCSI_hIDLE_LOOP_ENTRY(hhcb) >> 2);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SEQADDR1), (SCSI_UEXACT8)SCSI_hIDLE_LOOP_ENTRY(hhcb) >> 10);

   /* UnPause the Sequencer */
   SCSIhUnPause(hhcb);
}

/*********************************************************************
*
*  SCSIhBusDeviceReset
*
*     Perform Bus Device Reset request
*
*  Return Value:  none
*
*  Parameters:    hhcb
*                 hiob
*
*  Remarks:       This routine will be called when the bus device reset
*                 message is ready to send to the target and during the
*                 sequencer interrupt CHIM to handle the AbortTarget.
*                 Therefore, the sequencer is already paused.
*                 
*                 All this routine does is to abort or post immediately
*                 HIOB(s) that has the same SCSI ID.
*
*********************************************************************/
void SCSIhBusDeviceReset (SCSI_HHCB SCSI_HPTR hhcb,
                          SCSI_HIOB SCSI_IPTR hiob)
{
   SCSI_HIOB SCSI_IPTR hiobAbort;
   SCSI_UEXACT8 i;

   /* Check all HIOBs in the active pointer array for matching */
   /* target SCSI ID. If so, abort each HIOB separately. */
   for (i = 0; i < hhcb->numberScbs; i++)
   {
      if (((hiobAbort = SCSI_ACTPTR[i]) != SCSI_NULL_HIOB) &&
          (hiobAbort != hiob))
      {
#if SCSI_TARGET_OPERATION  
         if (hiob->SCSI_IP_targetMode ||
             hiob->cmd == SCSI_CMD_ESTABLISH_CONNECTION)
         {
            /* Bus Device Reset has no effect on Target mode HIOBs */
            continue; 
         }    
         else
#endif /* SCSI_TARGET_OPERATION */
         if (SCSI_TARGET_UNIT(hiob)->hhcb != hhcb)
         {
            continue;
         }      

         if ((SCSI_TARGET_UNIT(hiobAbort)->hhcb == hhcb) &&
             (SCSI_TARGET_UNIT(hiobAbort)->scsiID == SCSI_TARGET_UNIT(hiob)->scsiID))
         {
            SCSI_hABORTHIOB(hhcb, hiobAbort, SCSI_HOST_ABT_TRG_RST);
         }
      }
   }
}

/*********************************************************************
*
*  SCSIhStandard64AbortHIOB
*
*     Abort the passed-in HIOB for Standard mode sequencer.
*
*  Return Value:  None
*
*  Parameters:    hhcb
*                 aborting hiob
*                 host adapter status
*
*  Remarks:       !!! The sequencer should be paused !!!
*                 !!! at entering this routine.      !!!
*
*                 This routine will check if the aborting HIOB is
*                 in Done queue or New queue.  And if not found,
*                 it will try to stop the sequencer by setting
*                 the break point at the sequencer's idle loop for
*                 further searching and abort process handling.
*
*********************************************************************/
#if SCSI_STANDARD64_MODE
void SCSIhStandard64AbortHIOB (SCSI_HHCB SCSI_HPTR hhcb,
                               SCSI_HIOB SCSI_IPTR hiob,
                               SCSI_UEXACT8 haStatus)
{
   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_UINT32 i = 0;          /* time out counter */
   SCSI_UEXACT8 scbptr;
   SCSI_UEXACT8 byteBuf;

   /* Make sure no SCB is in transit from the host to SCB ram */
   if (OSD_INEXACT8(SCSI_AICREG(SCSI_CCSCBCTL)) & SCSI_CCARREN)
   {
      while ((!(OSD_INEXACT8(SCSI_AICREG(SCSI_CCSCBCTL)) & SCSI_ARRDONE)) &&
             (i++ < 400000))
      {
         OSD_TIMER(hhcb);
      }
   }

   /* If the SCB we are trying to abort is the one which is getting DMAed, */
   /* set the 'aborted' bit in the SCB in the SCB array so that it will    */
   /* never be fired by the sequencer.                                     */
   if ((OSD_INEXACT8(SCSI_AICREG(SCSI_CCSCBCTL)) &
       (SCSI_CCSCBEN+SCSI_CCSCBDIR+SCSI_CCARREN)) ==
       (SCSI_CCSCBEN+SCSI_CCSCBDIR+SCSI_CCARREN))
   {
      if (OSD_INEXACT8(SCSI_AICREG(SCSI_CCSCBPTR)) == hiob->scbNumber)
      {
         /* Set abort status */
         hiob->stat = SCSI_SCB_ABORTED;
         hiob->haStat = haStatus;

         /* Set abort request is done */
         hiob->SCSI_IP_mgrStat = SCSI_MGR_DONE_ABT;

         /* Prepare scb to be aborted by setting 'aborted' bit in SCB ram */
         scbptr = OSD_INEXACT8(SCSI_AICREG(SCSI_SCBPTR));   /* Save SCBPTR */
         OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCBPTR), hiob->scbNumber);
         byteBuf = OSD_INEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
                       OSDoffsetof(SCSI_SCB_STANDARD64, chain_control)));
         OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
                       OSDoffsetof(SCSI_SCB_STANDARD64,
                       chain_control)), byteBuf | SCSI_S64_ABORTED);
         OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCBPTR), scbptr);   /* Restore SCBPTR */

         return;
      }
   }

   /* Check if aborting HIOB is in Done queue or New queue */
   if (!SCSIhStandardSearchDoneQ(hhcb, hiob) &&
       !SCSI_hSEARCHNEWQ(hhcb, hiob, haStatus))
   {
      /* If the abort request is due to bus device reset. */
      if (haStatus == SCSI_HOST_ABT_TRG_RST)
      {
         if (!SCSI_hSEARCHSEQDONEQ(hhcb, hiob, 1) &&
             !SCSI_hSEARCHEXEQ(hhcb, hiob, SCSI_HOST_ABT_TRG_RST, 1))
         {
            /* Aborting HIOB must be disconnecting, abort it anyway. */
            /* The target will not reconnect after bus device reset. */
            hiob->stat = SCSI_SCB_ABORTED;
            hiob->haStat = SCSI_HOST_ABT_TRG_RST;
            SCSIhTerminateCommand(hiob);
         }
      }
      else
      {
         /* The aborting HIOB is in SCB ram (either in execution queue or */
         /* has been executed).  In order to get further status of where */
         /* it's at we need to stop the sequencer and check */

         /* Make sure we're not overwritten the host adapter status */
         /* because it might has been set while we're checking it. */
         if (!hiob->haStat)
         {
            /* Set abort status */
            hiob->stat = SCSI_SCB_ABORTED;
            hiob->haStat = haStatus;
         }

         /* Set abort is in request for use in breakpoint interrupt */
         hiob->SCSI_IP_mgrStat = SCSI_MGR_ABORTINREQ;

         /* set non expander break */
         hhcb->SCSI_HP_nonExpBreak = 1;

         /* Set the breakpoint address at sequencer idle loop. */
         /* Since the sequencer program counter always be one  */
         /* instruction ahead, we need to add 4 to actual idle */
         /* loop value.  One sequencer instruction is 29 bits. */

         /* Set at idle_loop0 because the sequencer will jump to it */
         /* while waiting for any SCB to execute. */
         SCSI_hSETBREAKPOINT(hhcb, (SCSI_UEXACT8)ENTRY_IDLE_LOOP);
      }
   }  /* end of if not in Done queue and New queue */
}
#endif   /* SCSI_STANDARD64_MODE */

/*********************************************************************
*
*  SCSIhStandardAdvAbortHIOB
*
*     Abort the passed-in HIOB for Standard mode sequencer.
*
*  Return Value:  None
*
*  Parameters:    hhcb
*                 aborting hiob
*                 host adapter status
*
*  Remarks:       !!! The sequencer should be paused !!!
*                 !!! at entering this routine.      !!!
*
*                 This routine will check if the aborting HIOB is
*                 in Done queue or New queue.  And if not found,
*                 it will try to stop the sequencer by setting
*                 the break point at the sequencer's idle loop for
*                 further searching and abort process handling.
*
*********************************************************************/
#if SCSI_STANDARD_ADVANCED_MODE
void SCSIhStandardAdvAbortHIOB (SCSI_HHCB SCSI_HPTR hhcb,
                               SCSI_HIOB SCSI_IPTR hiob,
                               SCSI_UEXACT8 haStatus)
{
   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_UINT32 i = 0;          /* time out counter */
   SCSI_UEXACT8 scbptr;
   SCSI_UEXACT8 byteBuf;

   /* Make sure no SCB is in transit from the host to SCB ram */
   if (OSD_INEXACT8(SCSI_AICREG(SCSI_CCSCBCTL)) & SCSI_CCARREN)
   {
      while ((!(OSD_INEXACT8(SCSI_AICREG(SCSI_CCSCBCTL)) & SCSI_ARRDONE)) &&
             (i++ < 400000))
      {
         OSD_TIMER(hhcb);
      }   
   }
   /* If the SCB we are trying to abort is the one which is getting DMAed, */
   /* set the 'aborted' bit in the SCB in the SCB array so that it will    */
   /* never be fired by the sequencer.                                     */
   if ((OSD_INEXACT8(SCSI_AICREG(SCSI_CCSCBCTL)) &
       (SCSI_CCSCBEN+SCSI_CCSCBDIR+SCSI_CCARREN)) ==
       (SCSI_CCSCBEN+SCSI_CCSCBDIR+SCSI_CCARREN))
   {
      if (OSD_INEXACT8(SCSI_AICREG(SCSI_CCSCBPTR)) == hiob->scbNumber)
      {
         /* Set abort status */
         hiob->stat = SCSI_SCB_ABORTED;
         hiob->haStat = haStatus;

         /* Set abort request is done */
         hiob->SCSI_IP_mgrStat = SCSI_MGR_DONE_ABT;

         /* Prepare scb to be aborted by setting 'aborted' bit in SCB ram */
         scbptr = OSD_INEXACT8(SCSI_AICREG(SCSI_SCBPTR));   /* Save SCBPTR */
         OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCBPTR), hiob->scbNumber);
         byteBuf = OSD_INEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
                       OSDoffsetof(SCSI_SCB_STANDARD_ADVANCED, scontrol)));
         OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
               OSDoffsetof(SCSI_SCB_STANDARD_ADVANCED,
               scontrol)), byteBuf | SCSI_SADV_ABORTED);
         OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCBPTR), scbptr);   /* Restore SCBPTR */

         return;
      }
   }
   
   /* Check if aborting HIOB is in Done queue or New queue */
   if (!SCSIhStandardSearchDoneQ(hhcb, hiob) &&
       !SCSI_hSEARCHNEWQ(hhcb, hiob, haStatus))
   {
      /* If the abort request is due to bus device reset. */
      if (haStatus == SCSI_HOST_ABT_TRG_RST)
      {
         if (!SCSI_hSEARCHSEQDONEQ(hhcb, hiob, 1) &&
             !SCSI_hSEARCHEXEQ(hhcb, hiob, SCSI_HOST_ABT_TRG_RST, 1))
         {
            /* Aborting HIOB must be disconnecting, abort it anyway. */
            /* The target will not reconnect after bus device reset. */
            hiob->stat = SCSI_SCB_ABORTED;
            hiob->haStat = SCSI_HOST_ABT_TRG_RST;
            SCSIhTerminateCommand(hiob);
         }
      }
      else
      {
         /* The aborting HIOB is in SCB ram (either in execution queue or */
         /* has been executed).  In order to get further status of where */
         /* it's at we need to stop the sequencer and check */

         /* Make sure we're not overwritten the host adapter status */
         /* because it might has been set while we're checking it. */
         if (!hiob->haStat)
         {
            /* Set abort status */
            hiob->stat = SCSI_SCB_ABORTED;
            hiob->haStat = haStatus;
         }

         /* Set abort is in request for use in breakpoint interrupt */
         hiob->SCSI_IP_mgrStat = SCSI_MGR_ABORTINREQ;

         /* set non expander break */
         hhcb->SCSI_HP_nonExpBreak = 1;

         /* Set the breakpoint address at sequencer idle loop. */
         /* Since the sequencer program counter always be one  */
         /* instruction ahead, we need to add 4 to actual idle */
         /* loop value.  One sequencer instruction is 29 bits. */

         /* Set at idle_loop0 because the sequencer will jump to it */
         /* while waiting for any SCB to execute. */
         SCSI_hSETBREAKPOINT(hhcb, (SCSI_UEXACT8)ENTRY_IDLE_LOOP);
      }
   }  /* end of if not in Done queue and New queue */
}
#endif   /* SCSI_STANDARD_ADVANCED_MODE */

/*********************************************************************
*
*  SCSIhStandard160mAbortHIOB
*
*     Abort the passed-in HIOB for Standard mode sequencer.
*
*  Return Value:  None
*
*  Parameters:    hhcb
*                 aborting hiob
*                 host adapter status
*
*  Remarks:       !!! The sequencer should be paused !!!
*                 !!! at entering this routine.      !!!
*
*                 This routine will check if the aborting HIOB is
*                 in Done queue or New queue.  And if not found,
*                 it will try to stop the sequencer by setting
*                 the break point at the sequencer's idle loop for
*                 further searching and abort process handling.
*
*********************************************************************/
#if SCSI_STANDARD_160M_MODE
void SCSIhStandard160mAbortHIOB (SCSI_HHCB SCSI_HPTR hhcb,
                                 SCSI_HIOB SCSI_IPTR hiob,
                                 SCSI_UEXACT8 haStatus)
{
   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_UINT32 i = 0;            /* time out counter */
   SCSI_UEXACT8 scbptr;
   SCSI_UEXACT8 byteBuf;

   /* Make sure no SCB is in transit from the host to SCB ram */
   if (OSD_INEXACT8(SCSI_AICREG(SCSI_CCSCBCTL)) & SCSI_CCARREN)
   {
      while ((!(OSD_INEXACT8(SCSI_AICREG(SCSI_CCSCBCTL)) & SCSI_ARRDONE)) &&
             (i++ < 400000))
      {
         OSD_TIMER(hhcb);
      }
   }
   /* If the SCB we are trying to abort is the one which is getting DMAed, */
   /* set the 'aborted' bit in the SCB in the SCB array so that it will    */
   /* never be fired by the sequencer.                                     */
   if ((OSD_INEXACT8(SCSI_AICREG(SCSI_CCSCBCTL)) &
       (SCSI_CCSCBEN+SCSI_CCSCBDIR+SCSI_CCARREN)) ==
       (SCSI_CCSCBEN+SCSI_CCSCBDIR+SCSI_CCARREN))
   {
      if (OSD_INEXACT8(SCSI_AICREG(SCSI_CCSCBPTR)) == hiob->scbNumber)
      {
         /* Set abort status */
         hiob->stat = SCSI_SCB_ABORTED;
         hiob->haStat = haStatus;

         /* Set abort request is done */
         hiob->SCSI_IP_mgrStat = SCSI_MGR_DONE_ABT;

         /* Prepare scb to be aborted by setting 'aborted' bit in SCB ram */
         scbptr = OSD_INEXACT8(SCSI_AICREG(SCSI_SCBPTR));   /* Save SCBPTR */
         OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCBPTR), hiob->scbNumber);
         byteBuf = OSD_INEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
                       OSDoffsetof(SCSI_SCB_STANDARD_160M, scontrol)));
         OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
               OSDoffsetof(SCSI_SCB_STANDARD_160M, scontrol)),
               byteBuf | SCSI_S160M_ABORTED);
         OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCBPTR), scbptr);   /* Restore SCBPTR */

         return;
      }
   }
    /* Check if aborting HIOB is in Done queue or New queue */
   if (!SCSIhStandardSearchDoneQ(hhcb, hiob) &&
       !SCSI_hSEARCHNEWQ(hhcb, hiob, haStatus))
   {
      /* If the abort request is due to bus device reset. */
      if (haStatus == SCSI_HOST_ABT_TRG_RST)
      {
         if (!SCSI_hSEARCHSEQDONEQ(hhcb, hiob, 1) &&
             !SCSI_hSEARCHEXEQ(hhcb, hiob, SCSI_HOST_ABT_TRG_RST, 1))
         {
            /* Aborting HIOB must be disconnecting, abort it anyway. */
            /* The target will not reconnect after bus device reset. */
            hiob->stat = SCSI_SCB_ABORTED;
            hiob->haStat = SCSI_HOST_ABT_TRG_RST;
            SCSIhTerminateCommand(hiob);
         }
      }
      else
      {
         /* The aborting HIOB is in SCB ram (either in execution queue or */
         /* has been executed).  In order to get further status of where */
         /* it's at we need to stop the sequencer and check */

         /* Make sure we're not overwritten the host adapter status */
         /* because it might has been set while we're checking it. */
         if (!hiob->haStat)
         {
            /* Set abort status */
            hiob->stat = SCSI_SCB_ABORTED;
            hiob->haStat = haStatus;
         }

         /* Set abort is in request for use in breakpoint interrupt */
         hiob->SCSI_IP_mgrStat = SCSI_MGR_ABORTINREQ;

         /* set non expander break */
         hhcb->SCSI_HP_nonExpBreak = 1;

         /* Set the breakpoint address at sequencer idle loop. */
         /* Since the sequencer program counter always be one  */
         /* instruction ahead, we need to add 4 to actual idle */
         /* loop value.  One sequencer instruction is 29 bits. */

         /* Set at idle_loop0 because the sequencer will jump to it */
         /* while waiting for any SCB to execute. */
         SCSI_hSETBREAKPOINT(hhcb, (SCSI_UEXACT8)ENTRY_IDLE_LOOP);
      }
   }  /* end of if not in Done queue and New queue */
}
#endif   /* SCSI_STANDARD_160M_MODE */

/*********************************************************************
*
*  SCSIhSwappingAbortHIOB
*
*     Abort the passed-in HIOB for Swapping mode sequencer.
*
*  Return Value:  None
*
*  Parameters:    hhcb
*                 aborting hiob
*                 host adapter status
*
*  Remarks:       !!! The sequencer should be paused !!!
*                 !!! at entering this routine.      !!!
*
*                 This routine will check if the aborting HIOB is
*                 in Done queue, New queue or Next SCB array.  And
*                 if not found, it will try to stop the sequencer
*                 by setting the break point at the sequencer idle
*                 loop for further searching and abort process handling.
*
*********************************************************************/
#if SCSI_SWAPPING64_MODE + SCSI_SWAPPING32_MODE
void SCSIhSwappingAbortHIOB (SCSI_HHCB SCSI_HPTR hhcb,
                             SCSI_HIOB SCSI_IPTR hiob,
                             SCSI_UEXACT8 haStatus)
{

   /* Check if aborting HIOB is in Done queue, New queue, or Next SCB array. */
   if (!SCSIhStandardSearchDoneQ(hhcb, hiob) &&
       !SCSIhSwappingSearchNewQ(hhcb, hiob, haStatus))
   {
      /* If the abort request is due to bus device reset. */
      if (haStatus == SCSI_HOST_ABT_TRG_RST)
      {
         /* Aborting HIOB must be disconnecting, abort it anyway. */
         /* The target will not reconnect after bus device reset. */
         hiob->stat = SCSI_SCB_ABORTED;
         hiob->haStat = SCSI_HOST_ABT_TRG_RST;
         SCSIhTerminateCommand(hiob);
      }
      else
      {
         /* The aborting HIOB has been executed.  In order to get further */
         /* status of where it's at we need to stop the sequencer and check */

         /* Make sure we're not overwritten the host adapter status */
         /* because it might has been set while we're checking it. */
         if (!hiob->haStat)
         {
            /* Set abort status */
            hiob->stat = SCSI_SCB_ABORTED;
            hiob->haStat = haStatus;
         }

         /* Set abort is in request for use in breakpoint interrupt */
         hiob->SCSI_IP_mgrStat = SCSI_MGR_ABORTINREQ;

         /* set non expander break */
         hhcb->SCSI_HP_nonExpBreak = 1;

         /* Set the breakpoint address at sequencer idle loop entry. */
         SCSI_hSETBREAKPOINT(hhcb, (SCSI_UEXACT8)ENTRY_IDLE_LOOP);
      }
   }  /* end of if not in Done queue, New queue, and Next SCB array */
}
#endif   /* SCSI_SWAPPING64_MODE + SCSI_SWAPPING32_MODE */

/*********************************************************************
*
*  SCSIhSwappingAdvAbortHIOB
*
*     Abort the passed-in HIOB for Swapping mode sequencer.
*
*  Return Value:  None
*
*  Parameters:    hhcb
*                 aborting hiob
*                 host adapter status
*
*  Remarks:       !!! The sequencer should be paused !!!
*                 !!! at entering this routine.      !!!
*
*                 This routine will check if the aborting HIOB is
*                 in Done queue, New queue or Next SCB array.  And
*                 if not found, it will try to stop the sequencer
*                 by setting the break point at the sequencer idle
*                 loop for further searching and abort process handling.
*
*********************************************************************/
#if SCSI_SWAPPING_ADVANCED_MODE
void SCSIhSwappingAdvAbortHIOB (SCSI_HHCB SCSI_HPTR hhcb,
                                SCSI_HIOB SCSI_IPTR hiob,
                                SCSI_UEXACT8 haStatus)
{
   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_UINT32 i = 0;          /* time out counter */
   SCSI_UEXACT8 scbptr;
   SCSI_UEXACT8 byteBuf;

   /* Make sure no SCB is in transit from the host to SCB ram */
   if (OSD_INEXACT8(SCSI_AICREG(SCSI_CCSCBCTL)) & SCSI_CCARREN)
   {
      while ((!(OSD_INEXACT8(SCSI_AICREG(SCSI_CCSCBCTL)) & SCSI_ARRDONE)) &&
             (i++ < 400000))
      {
         OSD_TIMER(hhcb);
      }
   }

   /* If the SCB we are trying to abort is the one which is getting DMAed, */
   /* set the 'aborted' bit in the SCB in the internal SCB array so that   */
   /* it will never be fired by the sequencer.                             */
   if ((OSD_INEXACT8(SCSI_AICREG(SCSI_CCSCBCTL)) &
       (SCSI_CCSCBEN+SCSI_CCSCBDIR+SCSI_CCARREN)) ==
       (SCSI_CCSCBEN+SCSI_CCSCBDIR+SCSI_CCARREN))
   {
      /* If the SCB is getting DMAed before selection... */
      if (OSD_INEXACT8(SCSI_AICREG(SCSI_hQ_NEXT_SCB(hhcb))) == hiob->scbNumber)
      {
         /* Set abort status */
         hiob->stat = SCSI_SCB_ABORTED;
         hiob->haStat = haStatus;

         /* Set abort request is done */
         hiob->SCSI_IP_mgrStat = SCSI_MGR_DONE_ABT;

         /* Prepare scb to be aborted by setting 'aborted' bit in SCB ram */
         scbptr = OSD_INEXACT8(SCSI_AICREG(SCSI_SCBPTR));   /* Save SCBPTR */
         OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCBPTR),
                       OSD_INEXACT8(SCSI_AICREG(SCSI_CCSCBPTR)));
         byteBuf = OSD_INEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
                       OSDoffsetof(SCSI_SCB_SWAPPING_ADVANCED, scontrol)));
         OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
               OSDoffsetof(SCSI_SCB_SWAPPING_ADVANCED,
               scontrol)), byteBuf | SCSI_WADV_ABORTED);
         OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCBPTR), scbptr);   /* Restore SCBPTR */

         return;
      }
      /* If the SCB is getting DMAed after reselection... */
      else if (OSD_INEXACT8(SCSI_AICREG(SCSI_WADV_ACTIVE_SCB)) ==
               hiob->scbNumber)
      {
         /* Set abort status */
         hiob->stat = SCSI_SCB_ABORTED;
         hiob->haStat = haStatus;

         /* Set abort request is done */
         hiob->SCSI_IP_mgrStat = SCSI_MGR_DONE_ABT;

         /* Prepare scb to be aborted by setting 'aborted' bit in SCB ram */
         scbptr = OSD_INEXACT8(SCSI_AICREG(SCSI_SCBPTR));   /* Save SCBPTR */
         OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCBPTR),
                       OSD_INEXACT8(SCSI_AICREG(SCSI_CCSCBPTR)));
         byteBuf = OSD_INEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
                       OSDoffsetof(SCSI_SCB_SWAPPING_ADVANCED, scontrol)));
         OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
               OSDoffsetof(SCSI_SCB_SWAPPING_ADVANCED,
               scontrol)), byteBuf | SCSI_WADV_ABORTED);
         OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCBPTR), scbptr);   /* Restore SCBPTR */

         return;
      }
   }

   /* Check if aborting HIOB is in Done queue, New queue, or Next SCB array. */
   if (!SCSIhStandardSearchDoneQ(hhcb, hiob) &&
       !SCSIhSwappingSearchNewQ(hhcb, hiob, haStatus))
   {
      /* If the abort request is due to bus device reset. */
#if SCSI_TARGET_OPERATION 
      if (haStatus == SCSI_HOST_ABT_TRG_RST ||
          haStatus == SCSI_HOST_TARGET_RESET)
#else
      if (haStatus == SCSI_HOST_ABT_TRG_RST)
#endif /* SCSI_TARGET_OPERATION  */
      {
         /* Aborting HIOB must be disconnecting, abort it anyway. */
         /* The target will not reconnect after bus device reset. */
         hiob->stat = SCSI_SCB_ABORTED;
#if SCSI_TARGET_OPERATION 
         if (hiob->SCSI_IP_targetMode) 
         {
            hiob->haStat = SCSI_HOST_TARGET_RESET;      
            SCSIhTerminateTargetCommand(hiob); 
         }
         else
         {
            hiob->haStat = SCSI_HOST_ABT_TRG_RST;
            SCSIhTerminateCommand(hiob);
         }
#else
         hiob->haStat = SCSI_HOST_ABT_TRG_RST;
         SCSIhTerminateCommand(hiob);
#endif /* SCSI_TARGET_OPERATION  */
      }
      else
      {
         /* The aborting HIOB has been executed.  In order to get further */
         /* status of where it's at we need to stop the sequencer and check */

         /* Make sure we're not overwritten the host adapter status */
         /* because it might has been set while we're checking it. */
         if (!hiob->haStat)
         {
            /* Set abort status */
            hiob->stat = SCSI_SCB_ABORTED;
            hiob->haStat = haStatus;
         }

         /* Set abort is in request for use in breakpoint interrupt */
         hiob->SCSI_IP_mgrStat = SCSI_MGR_ABORTINREQ;

         /* set non expander break */
         hhcb->SCSI_HP_nonExpBreak = 1;

         /* Set the breakpoint address at sequencer idle loop entry. */
         SCSI_hSETBREAKPOINT(hhcb, (SCSI_UEXACT8)ENTRY_IDLE_LOOP);
      }
   }  /* end of if not in Done queue, New queue, and Next SCB array */
}
#endif   /* SCSI_SWAPPING_ADVANCED_MODE */

/*********************************************************************
*
*  SCSIhSwapping160mAbortHIOB
*
*     Abort the passed-in HIOB for Swapping mode sequencer.
*
*  Return Value:  None
*
*  Parameters:    hhcb
*                 aborting hiob
*                 host adapter status
*
*  Remarks:       !!! The sequencer should be paused !!!
*                 !!! at entering this routine.      !!!
*
*                 This routine will check if the aborting HIOB is
*                 in Done queue, New queue or Next SCB array.  And
*                 if not found, it will try to stop the sequencer
*                 by setting the break point at the sequencer idle
*                 loop for further searching and abort process handling.
*
*********************************************************************/
#if SCSI_SWAPPING_160M_MODE
void SCSIhSwapping160mAbortHIOB (SCSI_HHCB SCSI_HPTR hhcb,
                                 SCSI_HIOB SCSI_IPTR hiob,
                                 SCSI_UEXACT8 haStatus)
{
   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_UINT32 i = 0;            /* time out counter */
   SCSI_UEXACT8 scbptr;
   SCSI_UEXACT8 byteBuf;

   /* Make sure no SCB is in transit from the host to SCB ram */
   if (OSD_INEXACT8(SCSI_AICREG(SCSI_CCSCBCTL)) & SCSI_CCARREN)
   {
      while ((!(OSD_INEXACT8(SCSI_AICREG(SCSI_CCSCBCTL)) & SCSI_ARRDONE)) &&
             (i++ < 400000))
      {
         OSD_TIMER(hhcb);
      }
   }

   /* If the SCB we are trying to abort is the one which is getting DMAed, */
   /* set the 'aborted' bit in the SCB in the internal SCB array so that   */
   /* it will never be fired by the sequencer.                             */
   if ((OSD_INEXACT8(SCSI_AICREG(SCSI_CCSCBCTL)) &
       (SCSI_CCSCBEN+SCSI_CCSCBDIR+SCSI_CCARREN)) ==
       (SCSI_CCSCBEN+SCSI_CCSCBDIR+SCSI_CCARREN))
   {
      /* If the SCB is getting DMAed before selection... */
      if (OSD_INEXACT8(SCSI_AICREG(SCSI_W160M_Q_NEXT_SCB)) == hiob->scbNumber)
      {
         /* Set abort status */
         hiob->stat = SCSI_SCB_ABORTED;
         hiob->haStat = haStatus;

         /* Set abort request is done */
         hiob->SCSI_IP_mgrStat = SCSI_MGR_DONE_ABT;

         /* Prepare scb to be aborted by setting 'aborted' bit in SCB ram */
         scbptr = OSD_INEXACT8(SCSI_AICREG(SCSI_SCBPTR));   /* Save SCBPTR */
         OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCBPTR),
                       OSD_INEXACT8(SCSI_AICREG(SCSI_CCSCBPTR)));
         byteBuf = OSD_INEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
                       OSDoffsetof(SCSI_SCB_SWAPPING_160M, scontrol)));
         OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
               OSDoffsetof(SCSI_SCB_SWAPPING_160M, scontrol)),
               byteBuf | SCSI_W160M_ABORTED);
         OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCBPTR), scbptr);   /* Restore SCBPTR */
         SCSI_hUPDATE_ABORT_BIT_HOST_MEM(hhcb,hiob);
         
         return;
      }
      /* If the SCB is getting DMAed after reselection... */
      else if (OSD_INEXACT8(SCSI_AICREG(SCSI_W160M_ACTIVE_SCB)) ==
               hiob->scbNumber)
      {
         /* Set abort status */
         hiob->stat = SCSI_SCB_ABORTED;
         hiob->haStat = haStatus;

         /* Set abort request is done */
         hiob->SCSI_IP_mgrStat = SCSI_MGR_DONE_ABT;

         /* Prepare scb to be aborted by setting 'aborted' bit in SCB ram */
         scbptr = OSD_INEXACT8(SCSI_AICREG(SCSI_SCBPTR));   /* Save SCBPTR */
         OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCBPTR),
                       OSD_INEXACT8(SCSI_AICREG(SCSI_CCSCBPTR)));
         byteBuf = OSD_INEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
                       OSDoffsetof(SCSI_SCB_SWAPPING_160M, scontrol)));
         OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
               OSDoffsetof(SCSI_SCB_SWAPPING_160M, scontrol)),
               byteBuf | SCSI_W160M_ABORTED);
         OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCBPTR), scbptr);   /* Restore SCBPTR */

         SCSI_hUPDATE_ABORT_BIT_HOST_MEM(hhcb,hiob);
         
         

         return;
      }
   }

   /* Check if aborting HIOB is in Done queue, New queue, or Next SCB array. */
   if (!SCSIhStandardSearchDoneQ(hhcb, hiob) &&
       !SCSIhSwappingSearchNewQ(hhcb, hiob, haStatus))
   {
      /* If the abort request is due to bus device reset. */
#if SCSI_TARGET_OPERATION
      if (haStatus == SCSI_HOST_ABT_TRG_RST ||
          haStatus == SCSI_HOST_TARGET_RESET)
#else
      if (haStatus == SCSI_HOST_ABT_TRG_RST)
#endif /* SCSI_TARGET_OPERATION */
      {
         /* Aborting HIOB must be disconnecting, abort it anyway. */
         /* The target will not reconnect after bus device reset. */
         hiob->stat = SCSI_SCB_ABORTED;
#if SCSI_TARGET_OPERATION
         if (hiob->SCSI_IP_targetMode)
         {
            hiob->haStat = SCSI_HOST_TARGET_RESET;
            SCSIhTerminateTargetCommand(hiob);
         }
         else
         {
            hiob->haStat = SCSI_HOST_ABT_TRG_RST;
            SCSIhTerminateCommand(hiob);
         }
#else
         hiob->haStat = SCSI_HOST_ABT_TRG_RST;
         SCSIhTerminateCommand(hiob);
#endif /* SCSI_TARGET_OPERATION  */
      }
      else
      {
         /* The aborting HIOB has been executed.  In order to get further */
         /* status of where it's at we need to stop the sequencer and check */

         /* Make sure we're not overwritten the host adapter status */
         /* because it might has been set while we're checking it. */
         if (!hiob->haStat)
         {
            /* Set abort status */
            hiob->stat = SCSI_SCB_ABORTED;
            hiob->haStat = haStatus;
         }

         /* Set abort is in request for use in breakpoint interrupt */
         hiob->SCSI_IP_mgrStat = SCSI_MGR_ABORTINREQ;

         /* set non expander break */
         hhcb->SCSI_HP_nonExpBreak = 1;

         /* Set the breakpoint address at sequencer idle loop entry. */
         SCSI_hSETBREAKPOINT(hhcb, (SCSI_UEXACT8)ENTRY_IDLE_LOOP);
      }
   }  /* end of if not in Done queue, New queue, and Next SCB array */
}
#endif   /* SCSI_SWAPPING_160M_MODE */

/*********************************************************************
*
*  SCSIhStandardSearchSeqDoneQ
*
*     Search sequencer done queue for the aborting HIOB.
*
*  Return Value:  0 - aborting HIOB not found
*                 1 - aborting HIOB found
*
*  Parameters:    hhcb
*                 aborting hiob
*                 post HIOB back if found flag:
*                    1 - yes
*                    0 - no
*
*  Remarks:
*
*********************************************************************/
#if SCSI_STANDARD64_MODE
SCSI_UEXACT8 SCSIhStandardSearchSeqDoneQ (SCSI_HHCB SCSI_HPTR hhcb,
                                          SCSI_HIOB SCSI_IPTR hiob,
                                          SCSI_UEXACT8 postHiob)
{
   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_UEXACT8 currScb;
   SCSI_UEXACT8 prevScb;
   SCSI_UEXACT8 nextScb;
   SCSI_UEXACT8 scbptr;
   SCSI_UEXACT8 retStat = 0;

   /* Check if aborting HIOB is in sequencer done queue */
   scbptr = OSD_INEXACT8(SCSI_AICREG(SCSI_SCBPTR));   /* Save SCBPTR */

   /* Get done head pointer */
   prevScb = currScb = OSD_INEXACT8(SCSI_AICREG(SCSI_S64_Q_DONE_HEAD));

   while (currScb != SCSI_NULL_SCB)
   {
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCBPTR), currScb);  /* Current SCBPTR */

      if (currScb == hiob->scbNumber)
      {
         nextScb = OSD_INEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
                       OSDoffsetof(SCSI_SCB_STANDARD64, SCSI_S64_q_next)));

         /* Adjust sequencer done queue */
         if (currScb == prevScb)    /* Found at head of queue */
         {
            /* Update q_done_head with next scb */
            OSD_OUTEXACT8(SCSI_AICREG(SCSI_S64_Q_DONE_HEAD), nextScb);
         }
         else
         {
            /* Relink the sequencer done queue */
            OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCBPTR), prevScb);
            OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
                OSDoffsetof(SCSI_SCB_STANDARD64, SCSI_S64_q_next)), nextScb);
         }

         if (postHiob)     /* Post it back as request */
         {
            hiob->stat = 0;                        /* Clear HIOB's status */
            hiob->SCSI_IP_mgrStat = SCSI_MGR_DONE; /* Indicate HIOB is completed */

            /* Remove it from HWM and post it back to the upper layer */
            SCSIhTerminateCommand(hiob);
         }

         retStat = 1;
         break;
      }

      prevScb = currScb; 

      /* Next scb ram */
      currScb = OSD_INEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
                    OSDoffsetof(SCSI_SCB_STANDARD64, SCSI_S64_q_next)));
   }

   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCBPTR), scbptr);   /* Restore SCBPTR */

   return(retStat);
}
#endif   /* SCSI_STANDARD64_MODE */

/*********************************************************************
*
*  SCSIhStandardAdvSearchSeqDoneQ
*
*     Search sequencer done queue for the aborting HIOB.
*
*  Return Value:  0 - aborting HIOB not found
*                 1 - aborting HIOB found
*
*  Parameters:    hhcb
*                 aborting hiob
*                 post HIOB back if found flag:
*                    1 - yes
*                    0 - no
*
*  Remarks:
*
*********************************************************************/
#if SCSI_STANDARD_ADVANCED_MODE
SCSI_UEXACT8 SCSIhStandardAdvSearchSeqDoneQ (SCSI_HHCB SCSI_HPTR hhcb,
                                          SCSI_HIOB SCSI_IPTR hiob,
                                          SCSI_UEXACT8 postHiob)
{
   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_UEXACT8 currScb;
   SCSI_UEXACT8 prevScb;
   SCSI_UEXACT8 nextScb;
   SCSI_UEXACT8 scbptr;
   SCSI_UEXACT8 retStat = 0;

   /* Check if aborting HIOB is in sequencer done queue */
   scbptr = OSD_INEXACT8(SCSI_AICREG(SCSI_SCBPTR));   /* Save SCBPTR */

   /* Get done head pointer */
   prevScb = currScb = OSD_INEXACT8(SCSI_AICREG(SCSI_SADV_Q_DONE_HEAD));

   while (currScb != SCSI_NULL_SCB)
   {
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCBPTR), currScb);  /* Current SCBPTR */

      if (currScb == hiob->scbNumber)
      {
         nextScb = OSD_INEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
                       OSDoffsetof(SCSI_SCB_STANDARD_ADVANCED, SCSI_SADV_q_next)));

         /* Adjust sequencer done queue */
         if (currScb == prevScb)    /* Found at head of queue */
         {
            /* Update q_done_head with next scb */
            OSD_OUTEXACT8(SCSI_AICREG(SCSI_SADV_Q_DONE_HEAD), nextScb);
         }
         else
         {
            /* Relink the sequencer done queue */
            OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCBPTR), prevScb);
            OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
                OSDoffsetof(SCSI_SCB_STANDARD_ADVANCED, SCSI_SADV_q_next)), nextScb);
         }

         if (postHiob)     /* Post it back as request */
         {
            hiob->stat = 0;                        /* Clear HIOB's status */
            hiob->SCSI_IP_mgrStat = SCSI_MGR_DONE; /* Indicate HIOB is completed */

            /* Remove it from HWM and post it back to the upper layer */
            SCSIhTerminateCommand(hiob);
         }

         retStat = 1;
         break;
      }

      prevScb = currScb; 

      /* Next scb ram */
      currScb = OSD_INEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
                    OSDoffsetof(SCSI_SCB_STANDARD_ADVANCED, SCSI_SADV_q_next)));
   }

   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCBPTR), scbptr);   /* Restore SCBPTR */

   return(retStat);
}
#endif   /* SCSI_STANDARD_ADVANCED_MODE */

/*********************************************************************
*
*  SCSIhStandard160mSearchSeqDoneQ
*
*     Search sequencer done queue for the aborting HIOB.
*
*  Return Value:  0 - aborting HIOB not found
*                 1 - aborting HIOB found
*
*  Parameters:    hhcb
*                 aborting hiob
*                 post HIOB back if found flag:
*                    1 - yes
*                    0 - no
*
*  Remarks:
*
*********************************************************************/
#if SCSI_STANDARD_160M_MODE
SCSI_UEXACT8 SCSIhStandard160mSearchSeqDoneQ (SCSI_HHCB SCSI_HPTR hhcb,
                                              SCSI_HIOB SCSI_IPTR hiob,
                                              SCSI_UEXACT8 postHiob)
{
   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_UEXACT8 currScb;
   SCSI_UEXACT8 prevScb;
   SCSI_UEXACT8 nextScb;
   SCSI_UEXACT8 scbptr;
   SCSI_UEXACT8 retStat = 0;

   /* Check if aborting HIOB is in sequencer done queue */
   scbptr = OSD_INEXACT8(SCSI_AICREG(SCSI_SCBPTR));   /* Save SCBPTR */

   /* Get done head pointer */
   prevScb = currScb = OSD_INEXACT8(SCSI_AICREG(SCSI_S160M_Q_DONE_HEAD));

   while (currScb != SCSI_NULL_SCB)
   {
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCBPTR), currScb);  /* Current SCBPTR */

      if (currScb == hiob->scbNumber)
      {
         nextScb = OSD_INEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
                       OSDoffsetof(SCSI_SCB_STANDARD_160M, SCSI_S160M_q_next)));

         /* Adjust sequencer done queue */
         if (currScb == prevScb)    /* Found at head of queue */
         {
            /* Update q_done_head with next scb */
            OSD_OUTEXACT8(SCSI_AICREG(SCSI_S160M_Q_DONE_HEAD), nextScb);
         }
         else
         {
            /* Relink the sequencer done queue */
            OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCBPTR), prevScb);
            OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
                OSDoffsetof(SCSI_SCB_STANDARD_160M, SCSI_S160M_q_next)), nextScb);
         }

         if (postHiob)     /* Post it back as request */
         {
            hiob->stat = 0;                        /* Clear HIOB's status */
            hiob->SCSI_IP_mgrStat = SCSI_MGR_DONE; /* Indicate HIOB is completed */

            /* Remove it from HWM and post it back to the upper layer */
            SCSIhTerminateCommand(hiob);
         }

         retStat = 1;
         break;
      }

      prevScb = currScb; 

      /* Next scb ram */
      currScb = OSD_INEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
                    OSDoffsetof(SCSI_SCB_STANDARD_160M, SCSI_S160M_q_next)));
   }

   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCBPTR), scbptr);   /* Restore SCBPTR */

   return(retStat);
}
#endif   /* SCSI_STANDARD_160M_MODE */

/*********************************************************************
*
*  SCSIhStandardSearchDoneQ
*
*     Search Done queue for the aborting HIOB for both standard and
*     swapping modes.
*
*  Return Value:  0 - aborting HIOB not found
*                 1 - aborting HIOB found
*
*  Parameters:    hhcb
*                 aborting hiob
*
*  Remarks:
*
*********************************************************************/
#if SCSI_NEW_MODE
SCSI_UEXACT8 SCSIhStandardSearchDoneQ (SCSI_HHCB SCSI_HPTR hhcb,
                                       SCSI_HIOB SCSI_IPTR hiob)
{
   SCSI_QOUTFIFO_NEW SCSI_HPTR qOutPtr;
   SCSI_QOUTFIFO_NEW SCSI_HPTR qOutPrevPtr;
   SCSI_UEXACT8 scbNumber;
   SCSI_UEXACT8 qDoneElement;
   SCSI_UEXACT8 qPassCnt;
   SCSI_UEXACT8 qDonePass;
   SCSI_UEXACT8 retStat = 0;

   /* Check if aborting HIOB is in Done queue */
   qDoneElement = hhcb->SCSI_HP_qDoneElement;
   qOutPtr = SCSI_QOUT_PTR_ARRAY_NEW(hhcb) + qDoneElement;

   /* Invalidate the whole QOUTFIFO */
   SCSI_INVALIDATE_CACHE((SCSI_UEXACT8 SCSI_HPTR)SCSI_QOUT_PTR_ARRAY_NEW(hhcb),
        (hhcb->numberScbs + 1) * sizeof(SCSI_QOUTFIFO_NEW));

   SCSI_GET_LITTLE_ENDIAN8(hhcb, &qPassCnt, qOutPtr,
        OSDoffsetof(SCSI_QOUTFIFO_NEW, quePassCnt));

   qDonePass = hhcb->SCSI_HP_qDonePass;

   /* Keep on searching until Pass Count and Done Pass value */
   /* are different or the aborting HIOB found. */
   while (qPassCnt == qDonePass)
   {
      /* Found valid entry, get scb number */
      SCSI_GET_LITTLE_ENDIAN8(hhcb, &scbNumber, qOutPtr,
           OSDoffsetof(SCSI_QOUTFIFO_NEW, scbNumber));

      /* Check if scb number matches with aborting HIOB */
      if (scbNumber == hiob->scbNumber)
      {
         /* Mending Done queue by moving all done SCBs up one location */
         /* Need to keep correct scb number and q_pass_cnt value */
         while (qDoneElement != hhcb->SCSI_HP_qDoneElement)
         {
            qOutPtr = SCSI_QOUT_PTR_ARRAY_NEW(hhcb) + qDoneElement;
            if (--qDoneElement == 0xff)
            {
               qDoneElement = hhcb->numberScbs;
            }

            qOutPrevPtr = SCSI_QOUT_PTR_ARRAY_NEW(hhcb) + qDoneElement;

            SCSI_GET_LITTLE_ENDIAN8(hhcb, &scbNumber, qOutPrevPtr,
                 OSDoffsetof(SCSI_QOUTFIFO_NEW, scbNumber));
            SCSI_PUT_LITTLE_ENDIAN8(hhcb, qOutPtr, OSDoffsetof(SCSI_QOUTFIFO_NEW,
                 scbNumber), scbNumber);

            /* Don't need to get q_pass_cnt value. Its value */
            /* get change only when the queue is rollover. */
            SCSI_PUT_LITTLE_ENDIAN8(hhcb, qOutPtr, OSDoffsetof(SCSI_QOUTFIFO_NEW,
                 quePassCnt), qPassCnt);

            /* Decrement q_pass_cnt if the qDoneElement rollover */
            if (qDoneElement == hhcb->numberScbs)
            {
               --qPassCnt;
            }
         }

         /* Update qDoneElement */
         if (++hhcb->SCSI_HP_qDoneElement == (SCSI_UEXACT8)(hhcb->numberScbs + 1))
         {
            hhcb->SCSI_HP_qDoneElement = 0;
            ++hhcb->SCSI_HP_qDonePass;
         }

         /* Make sure HIOB's status is clear. Just in case, */
         /* the completed HIOB is found during active abort. */
         hiob->stat = 0;

         /*  Set the aborting HIOB was finished prior the abort requested. */
         hiob->SCSI_IP_mgrStat = SCSI_MGR_DONE;

         /* Remove it from HWM and post it back to the upper layer */
         SCSI_hTERMINATECOMMAND(hiob);

         /* Decrement the number of outstanding entries in Done queue */
         --hhcb->SCSI_HP_qoutcnt;

         retStat = 1;
         break;
      }

      /* Increment qDoneElement */
      if (++qDoneElement == (SCSI_UEXACT8)(hhcb->numberScbs + 1))
      {
         qDoneElement = 0;
         ++qDonePass;
      }

      qOutPtr = SCSI_QOUT_PTR_ARRAY_NEW(hhcb) + qDoneElement;

      SCSI_GET_LITTLE_ENDIAN8(hhcb, &qPassCnt, qOutPtr,
           OSDoffsetof(SCSI_QOUTFIFO_NEW, quePassCnt));
   }

   /* Flush the whole QOUTFIFO */
   SCSI_FLUSH_CACHE((SCSI_UEXACT8 SCSI_HPTR)SCSI_QOUT_PTR_ARRAY_NEW(hhcb),
        (hhcb->numberScbs + 1) * sizeof(SCSI_QOUTFIFO_NEW));

   return(retStat);
}
#endif   /* SCSI_NEW_MODE */

/*********************************************************************
*
*  SCSIhStandardSearchNewQ
*
*     Search New queue for the aborting HIOB.
*
*  Return Value:  0 - aborting HIOB not found
*                 1 - aborting HIOB found
*
*  Parameters:    hhcb
*                 aborting hiob
*                 host adapter status
*
*  Remarks:
*
*********************************************************************/
#if SCSI_STANDARD_MODE
SCSI_UEXACT8 SCSIhStandardSearchNewQ (SCSI_HHCB SCSI_HPTR hhcb,
                                      SCSI_HIOB SCSI_IPTR hiob,
                                      SCSI_UEXACT8 haStatus)
{
   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_UEXACT8 qNextScb;
   SCSI_UEXACT8 scbCurr;
   SCSI_UEXACT8 scbPrev;
   SCSI_UEXACT8 scbNext;
   SCSI_HIOB SCSI_IPTR hiobCurr;
   SCSI_HIOB SCSI_IPTR hiobPrev;
   SCSI_BUS_ADDRESS scbNextbusAddress;
   SCSI_UEXACT8 retStat = 0;

   /* Get q_next_scb value from scratch ram */
   qNextScb = OSD_INEXACT8(SCSI_AICREG(SCSI_hQ_NEXT_SCB(hhcb)));
   scbCurr = scbPrev = qNextScb;

   /* Check if aborting HIOB is in New queue */
   while (SCSI_ACTPTR[scbCurr] != SCSI_NULL_HIOB)
   {
      /* Search for matching SCB number */
      if (scbCurr == hiob->scbNumber)
      {
         /* Set abort status */
         hiob->stat = SCSI_SCB_ABORTED;
         hiob->haStat = haStatus;

         /* Set abort request is done */
         hiob->SCSI_IP_mgrStat = SCSI_MGR_DONE_ABT;

         /* We won't remove the aborting HIOB if it lies at the tail of */
         /* the New queue or next to the tail 'cause sequencer might    */
         /* accessing it. If it is a bus_device_reset, then post back   */
         /* the HIOB.                                                   */
         if (scbCurr == qNextScb || scbPrev == qNextScb)
         {
		 
            SCSI_hUPDATE_ABORT_BIT_HOST_MEM(hhcb,hiob);
         
		 }
         else
         {
            /* Decrement and update qNewHead into scratch ram */
            (SCSI_UEXACT8)hhcb->SCSI_HP_qNewHead--;
            OSD_OUTEXACT8(SCSI_AICREG(SCSI_hQ_NEW_HEAD(hhcb)), hhcb->SCSI_HP_qNewHead);

            /* Remove aborted HIOB from New queue and mend the queue */

            /* Obtain current HIOB in Active pointer array */ 
            hiobCurr = SCSI_ACTPTR[scbCurr];
            scbNext = SCSI_hOBTAIN_NEXT_SCB_NUM(hhcb,hiobCurr);
            scbNextbusAddress = SCSI_hOBTAIN_NEXT_SCB_ADDRESS(hhcb,hiobCurr);

            /* Obtain previous HIOB in Active pointer array */ 
            hiobPrev = SCSI_ACTPTR[scbPrev];
            SCSI_hUPDATE_NEXT_SCB_NUM(hhcb,hiobPrev,scbNext);
            SCSI_hUPDATE_NEXT_SCB_ADDRESS(hhcb,hiobPrev,scbNextbusAddress);

            /* Post aborted HIOB back to the upper layer */
            SCSIhTerminateCommand(hiob);
         }

         retStat = 1;
         break;
      }

      /* Need to get next scb from current scb */

      scbPrev = scbCurr;
      
      /* Obtain current HIOB in Active pointer array */ 
      hiobCurr = SCSI_ACTPTR[scbCurr];
      scbCurr = SCSI_hOBTAIN_NEXT_SCB_NUM(hhcb,hiobCurr);
   }

   return(retStat);
}
#endif   /* SCSI_STANDARD_MODE */

/*********************************************************************
*
*  SCSIhSwappingSearchNewQ
*
*     Search New queue for the aborting HIOB.
*
*  Return Value:  0 - aborting HIOB not found
*                 1 - aborting HIOB found
*
*  Parameters:    hhcb
*                 aborting hiob
*                 host adapter status
*
*  Remarks:
*
*********************************************************************/
#if SCSI_SWAPPING_MODE
SCSI_UEXACT8 SCSIhSwappingSearchNewQ (SCSI_HHCB SCSI_HPTR hhcb,
                                      SCSI_HIOB SCSI_IPTR hiob,
                                      SCSI_UEXACT8 haStatus)
{
   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_UEXACT8 qNewTail;
   SCSI_UEXACT8 qNextScb;
   SCSI_UEXACT8 scbCurr;
   SCSI_UEXACT8 scbPrev;
   SCSI_UEXACT8 scbNext;
   SCSI_HIOB SCSI_IPTR hiobCurr;
   SCSI_HIOB SCSI_IPTR hiobPrev;
   SCSI_UEXACT8 retStat = 0;

   /* Get q_new_tail value */
   qNewTail = SCSI_hGETQNEWTAIL(hhcb);

   /* Get q_next_scb value from scratch ram */
   qNextScb = OSD_INEXACT8(SCSI_AICREG(SCSI_hQ_NEXT_SCB(hhcb)));
   scbCurr = scbPrev = qNextScb;


   /* Check if aborting HIOB is in New queue */
   while (qNewTail != hhcb->SCSI_HP_qNewHead)
   {

      /* Search for matching SCB number */
      if (scbCurr == hiob->scbNumber)
      {
         /* Set abort status */
         hiob->stat = SCSI_SCB_ABORTED;
         hiob->haStat = haStatus;

         /* Set abort request is done */
         hiob->SCSI_IP_mgrStat = SCSI_MGR_DONE_ABT;

         /* We won't remove the aborting HIOB if it lies at the tail of */
         /* the New queue or next to the tail 'cause sequencer might    */
         /* accessing it. If it is a bus_device_reset, then post back   */
         /* the HIOB.                                                   */
         if (scbCurr == qNextScb || scbPrev == qNextScb)
         {
            SCSI_hUPDATE_ABORT_BIT_HOST_MEM(hhcb,hiob);
         }
         else
         {
            /* Decrement and update qNewHead into scratch ram */
            (SCSI_UEXACT8)hhcb->SCSI_HP_qNewHead--;
            OSD_OUTEXACT8(SCSI_AICREG(SCSI_hQ_NEW_HEAD(hhcb)), hhcb->SCSI_HP_qNewHead);

            /* Remove aborted HIOB from New queue and mend the queue */

            /* Obtain current HIOB in Active pointer array */ 
            hiobCurr = SCSI_ACTPTR[scbCurr];
            scbNext = SCSI_hOBTAIN_NEXT_SCB_NUM(hhcb,hiobCurr);

            /* Obtain previous HIOB in Active pointer array */ 
            hiobPrev = SCSI_ACTPTR[scbPrev];
            SCSI_hUPDATE_NEXT_SCB_NUM(hhcb,hiobPrev,scbNext);

            /* Post aborted HIOB back to the upper layer */
            SCSI_hTERMINATECOMMAND(hiob);
         }

         retStat = 1;
         break;
      }

      /* Need to get next scb from current scb */

      scbPrev = scbCurr;
      
      /* Obtain current HIOB in Active pointer array */ 
      hiobCurr = SCSI_ACTPTR[scbCurr];
      scbCurr = SCSI_hOBTAIN_NEXT_SCB_NUM(hhcb,hiobCurr);
      
      (SCSI_UEXACT8)qNewTail++;
   }

   return(retStat);
}
#endif   /* SCSI_SWAPPING_MODE */

/*********************************************************************
*
*  SCSIhStandard64ObtainNextScbNum
*
*     Get next SCB number in the SCB for Standard64 mode
*
*  Return Value:  next scb number
*                  
*  Parameters:    hhcb
*                 hiob
*
*  Remarks:
*
*********************************************************************/
#if SCSI_STANDARD64_MODE
SCSI_UEXACT8 SCSIhStandard64ObtainNextScbNum (SCSI_HHCB SCSI_HPTR hhcb,
                                              SCSI_HIOB SCSI_IPTR hiob)
{
   SCSI_UEXACT8 nextScb;
   SCSI_SCB_STANDARD64 SCSI_IPTR scbBuffer =
      (SCSI_SCB_STANDARD64 SCSI_IPTR )hiob->scbDescriptor->scbBuffer.virtualAddress;

   SCSI_GET_LITTLE_ENDIAN8(hhcb, &nextScb, scbBuffer,
      OSDoffsetof(SCSI_SCB_STANDARD64, SCSI_S64_array_site_next));

   return (nextScb);
}
#endif

/*********************************************************************
*
*  SCSIhSwapping32ObtainNextScbNum
*
*     Get next SCB number in SCB for Swapping32 mode  
*
*  Return Value:  next scb number
*                  
*  Parameters:    hhcb
*                 hiob
*
*  Remarks:
*
*********************************************************************/
#if SCSI_SWAPPING32_MODE
SCSI_UEXACT8 SCSIhSwapping32ObtainNextScbNum (SCSI_HHCB SCSI_HPTR hhcb,
                                              SCSI_HIOB SCSI_IPTR hiob)
{
   SCSI_UEXACT8 nextScb;
   SCSI_SCB_SWAPPING32 SCSI_IPTR scbBuffer =
      (SCSI_SCB_SWAPPING32 SCSI_IPTR )hiob->scbDescriptor->scbBuffer.virtualAddress;

   SCSI_GET_LITTLE_ENDIAN8(hhcb, &nextScb, scbBuffer,
      OSDoffsetof(SCSI_SCB_SWAPPING32, next_SCB));

   return (nextScb);
}
#endif

/*********************************************************************
*
*  SCSIhSwapping64ObtainNextScbNum
*
*     Get next SCB number in the SCB for Swapping64 mode
*     
*  Return Value:  next scb number
*                  
*  Parameters:    hhcb
*                 hiob
*
*  Remarks:
*
*********************************************************************/
#if SCSI_SWAPPING64_MODE
SCSI_UEXACT8 SCSIhSwapping64ObtainNextScbNum (SCSI_HHCB SCSI_HPTR hhcb,
                                              SCSI_HIOB SCSI_IPTR hiob)
{
   SCSI_UEXACT8 nextScb;
   SCSI_SCB_SWAPPING64 SCSI_IPTR scbBuffer =
      (SCSI_SCB_SWAPPING64 SCSI_IPTR )hiob->scbDescriptor->scbBuffer.virtualAddress;

   SCSI_GET_LITTLE_ENDIAN8(hhcb, &nextScb, scbBuffer,
      OSDoffsetof(SCSI_SCB_SWAPPING64, next_SCB));

   return (nextScb);
}
#endif

/*********************************************************************
*
*  SCSIhStandardAdvObtainNextScbNum
*
*     Get next SCB number in the SCB for Advance Standard mode 
*
*  Return Value:  next scb number
*                  
*  Parameters:    hhcb
*                 hiob
*
*  Remarks:
*
*********************************************************************/
#if SCSI_STANDARD_ADVANCED_MODE
SCSI_UEXACT8 SCSIhStandardAdvObtainNextScbNum (SCSI_HHCB SCSI_HPTR hhcb,
                                                 SCSI_HIOB SCSI_IPTR hiob)
{
   SCSI_UEXACT8 nextScb;
   SCSI_SCB_STANDARD_ADVANCED SCSI_IPTR scbBuffer =
      (SCSI_SCB_STANDARD_ADVANCED SCSI_IPTR )hiob->scbDescriptor->scbBuffer.virtualAddress;

   SCSI_GET_LITTLE_ENDIAN8(hhcb, &nextScb, scbBuffer,
      OSDoffsetof(SCSI_SCB_STANDARD_ADVANCED, SCSI_SADV_array_site_next));

   return (nextScb);
}
#endif

/*********************************************************************
*
*  SCSIhSwappingAdvObtainNextScbNum
*
*     Get next SCB number in the SCB for Advance Swapping mode 
*     
*  Return Value:  next scb number
*                  
*  Parameters:    hhcb
*                 hiob
*
*  Remarks:
*
*********************************************************************/
#if SCSI_SWAPPING_ADVANCED_MODE
SCSI_UEXACT8 SCSIhSwappingAdvObtainNextScbNum (SCSI_HHCB SCSI_HPTR hhcb,
                                               SCSI_HIOB SCSI_IPTR hiob)
{
   SCSI_UEXACT8 nextScb;
   SCSI_SCB_SWAPPING_ADVANCED SCSI_IPTR scbBuffer =
      (SCSI_SCB_SWAPPING_ADVANCED SCSI_IPTR )hiob->scbDescriptor->scbBuffer.virtualAddress;

   SCSI_GET_LITTLE_ENDIAN8(hhcb, &nextScb, scbBuffer,
      OSDoffsetof(SCSI_SCB_SWAPPING_ADVANCED, next_SCB));

   return (nextScb);
}
#endif

/*********************************************************************
*
*  SCSIhStandard160mObtainNextScbNum
*
*     Get next SCB number in the SCB for Standard 160m mode 
*
*  Return Value:  next scb number
*                  
*  Parameters:    hhcb
*                 hiob
*
*  Remarks:
*
*********************************************************************/
#if SCSI_STANDARD_160M_MODE
SCSI_UEXACT8 SCSIhStandard160mObtainNextScbNum (SCSI_HHCB SCSI_HPTR hhcb,
                                                SCSI_HIOB SCSI_IPTR hiob)
{
   SCSI_UEXACT8 nextScb;
   SCSI_SCB_STANDARD_160M SCSI_IPTR scbBuffer =
      (SCSI_SCB_STANDARD_160M SCSI_IPTR) hiob->scbDescriptor->scbBuffer.virtualAddress;

   SCSI_GET_LITTLE_ENDIAN8(hhcb, &nextScb, scbBuffer,
        OSDoffsetof(SCSI_SCB_STANDARD_160M, SCSI_S160M_array_site_next));

   return (nextScb);
}
#endif

/*********************************************************************
*
*  SCSIhSwapping160mObtainNextScbNum
*
*     Get next SCB number in the SCB for Swapping 160m mode 
*     
*  Return Value:  next scb number
*                  
*  Parameters:    hhcb
*                 hiob
*
*  Remarks:
*
*********************************************************************/
#if SCSI_SWAPPING_160M_MODE
SCSI_UEXACT8 SCSIhSwapping160mObtainNextScbNum (SCSI_HHCB SCSI_HPTR hhcb,
                                                SCSI_HIOB SCSI_IPTR hiob)
{
   SCSI_UEXACT8 nextScb;
   SCSI_SCB_SWAPPING_160M SCSI_IPTR scbBuffer =
      (SCSI_SCB_SWAPPING_160M SCSI_IPTR) hiob->scbDescriptor->scbBuffer.virtualAddress;

   SCSI_GET_LITTLE_ENDIAN8(hhcb, &nextScb, scbBuffer,
        OSDoffsetof(SCSI_SCB_SWAPPING_160M, next_SCB));

   return (nextScb);
}
#endif

/*********************************************************************
*
*  SCSIhStandard64UpdateNextScbNum
*
*     Modify next SCB number in the SCB for Standard64 mode
*
*  Return Value:  none
*                  
*  Parameters:    hhcb
*                 hiob
*                 scbNumber
*
*  Remarks:
*
*********************************************************************/
#if SCSI_STANDARD64_MODE
void SCSIhStandard64UpdateNextScbNum (SCSI_HHCB SCSI_HPTR hhcb,
                                      SCSI_HIOB SCSI_IPTR hiob,
                                      SCSI_UEXACT8 scbNumber)
{
   SCSI_SCB_STANDARD64 SCSI_IPTR scbBuffer =
      (SCSI_SCB_STANDARD64 SCSI_IPTR )hiob->scbDescriptor->scbBuffer.virtualAddress;

   SCSI_PUT_LITTLE_ENDIAN8(hhcb, scbBuffer,
      OSDoffsetof(SCSI_SCB_STANDARD64, SCSI_S64_array_site_next), scbNumber);

   /* Flush the cache */
   SCSI_FLUSH_CACHE(scbBuffer, sizeof(SCSI_SCB_STANDARD64));
}
#endif

/*********************************************************************
*
*  SCSIhSwapping32UpdateNextScbNum
*
*     Modify next SCB number in SCB for Swapping32 mode
*
*  Return Value:  none
*                  
*  Parameters:    hhcb
*                 hiob
*                 scbNumber
*
*  Remarks:
*
*********************************************************************/
#if SCSI_SWAPPING32_MODE
void SCSIhSwapping32UpdateNextScbNum (SCSI_HHCB SCSI_HPTR hhcb,
                                      SCSI_HIOB SCSI_IPTR hiob,
                                      SCSI_UEXACT8 scbNumber)
{
   SCSI_SCB_SWAPPING32 SCSI_IPTR scbBuffer =
      (SCSI_SCB_SWAPPING32 SCSI_IPTR )hiob->scbDescriptor->scbBuffer.virtualAddress;

   SCSI_PUT_LITTLE_ENDIAN8(hhcb, scbBuffer,
      OSDoffsetof(SCSI_SCB_SWAPPING32, next_SCB), scbNumber);

   /* Flush the cache */
   SCSI_FLUSH_CACHE(scbBuffer, sizeof(SCSI_SCB_SWAPPING32));
}
#endif

/*********************************************************************
*
*  SCSIhSwapping64UpdateNextScbNum
*
*     Modify next SCB number in the SCB for Swapping64 mode
*
*  Return Value:  none
*                  
*  Parameters:    hhcb
*                 hiob
*                 scbNumber
*
*  Remarks:
*
*********************************************************************/
#if SCSI_SWAPPING64_MODE
void SCSIhSwapping64UpdateNextScbNum (SCSI_HHCB SCSI_HPTR hhcb,
                                         SCSI_HIOB SCSI_IPTR hiob,
                                         SCSI_UEXACT8 scbNumber)
{
   SCSI_SCB_SWAPPING64 SCSI_IPTR scbBuffer =
      (SCSI_SCB_SWAPPING64 SCSI_IPTR )hiob->scbDescriptor->scbBuffer.virtualAddress;

   SCSI_PUT_LITTLE_ENDIAN8(hhcb, scbBuffer,
      OSDoffsetof(SCSI_SCB_SWAPPING64, next_SCB), scbNumber);

   /* Flush the cache */
   SCSI_FLUSH_CACHE(scbBuffer, sizeof(SCSI_SCB_SWAPPING64));
}
#endif

/*********************************************************************
*
*   SCSIhStandardAdvUpdateNextScbNum
*
*     Modify next SCB number in the SCB for Advance Standard mode
*
*  Return Value:  none
*                  
*  Parameters:    hhcb
*                 hiob
*                 scbNumber
*
*  Remarks:
*
*********************************************************************/
#if SCSI_STANDARD_ADVANCED_MODE
void SCSIhStandardAdvUpdateNextScbNum (SCSI_HHCB SCSI_HPTR hhcb,
                                       SCSI_HIOB SCSI_IPTR hiob,
                                       SCSI_UEXACT8 scbNumber)
{
   SCSI_SCB_STANDARD_ADVANCED SCSI_IPTR scbBuffer =
      (SCSI_SCB_STANDARD_ADVANCED SCSI_IPTR )hiob->scbDescriptor->scbBuffer.virtualAddress;

   SCSI_PUT_LITTLE_ENDIAN8(hhcb,scbBuffer,
      OSDoffsetof(SCSI_SCB_STANDARD_ADVANCED, SCSI_SADV_array_site_next), scbNumber);

   /* Flush the cache */
   SCSI_FLUSH_CACHE(scbBuffer, sizeof(SCSI_SCB_STANDARD_ADVANCED));
}
#endif

/*********************************************************************
*
*  SCSIhSwappingAdvUpdateNextScbNum
*
*     Modify next SCB number in the SCB for Advance Swapping mode
*
*  Return Value:  none
*                  
*  Parameters:    hhcb
*                 hiob
*                 scbNumber
*
*  Remarks:
*
*********************************************************************/
#if SCSI_SWAPPING_ADVANCED_MODE
void SCSIhSwappingAdvUpdateNextScbNum (SCSI_HHCB SCSI_HPTR hhcb,
                                       SCSI_HIOB SCSI_IPTR hiob,
                                       SCSI_UEXACT8 scbNumber)
{
   SCSI_SCB_SWAPPING_ADVANCED SCSI_IPTR scbBuffer =
      (SCSI_SCB_SWAPPING_ADVANCED SCSI_IPTR )hiob->scbDescriptor->scbBuffer.virtualAddress;

   SCSI_PUT_LITTLE_ENDIAN8(hhcb, scbBuffer,
      OSDoffsetof(SCSI_SCB_SWAPPING_ADVANCED, next_SCB), scbNumber);

   /* Flush the cache */
   SCSI_FLUSH_CACHE(scbBuffer, sizeof(SCSI_SCB_SWAPPING_ADVANCED));
}
#endif

/*********************************************************************
*
*   SCSIhStandard160mUpdateNextScbNum
*
*     Modify next SCB number in the SCB for Standard 160m mode
*
*  Return Value:  none
*                  
*  Parameters:    hhcb
*                 hiob
*                 scbNumber
*
*  Remarks:
*
*********************************************************************/
#if SCSI_STANDARD_160M_MODE
void SCSIhStandard160mUpdateNextScbNum (SCSI_HHCB SCSI_HPTR hhcb,
                                        SCSI_HIOB SCSI_IPTR hiob,
                                        SCSI_UEXACT8 scbNumber)
{
   SCSI_SCB_STANDARD_160M SCSI_IPTR scbBuffer =
      (SCSI_SCB_STANDARD_160M SCSI_IPTR) hiob->scbDescriptor->scbBuffer.virtualAddress;

   SCSI_PUT_LITTLE_ENDIAN8(hhcb,scbBuffer,
        OSDoffsetof(SCSI_SCB_STANDARD_160M, SCSI_S160M_array_site_next),
        scbNumber);

   /* Flush the cache */
   SCSI_FLUSH_CACHE(scbBuffer, sizeof(SCSI_SCB_STANDARD_160M));
}
#endif

/*********************************************************************
*
*  SCSIhSwapping160mUpdateNextScbNum
*
*     Modify next SCB number in the SCB for Swapping 160m mode
*
*  Return Value:  none
*                  
*  Parameters:    hhcb
*                 hiob
*                 scbNumber
*
*  Remarks:
*
*********************************************************************/
#if SCSI_SWAPPING_160M_MODE
void SCSIhSwapping160mUpdateNextScbNum (SCSI_HHCB SCSI_HPTR hhcb,
                                        SCSI_HIOB SCSI_IPTR hiob,
                                        SCSI_UEXACT8 scbNumber)
{
   SCSI_SCB_SWAPPING_160M SCSI_IPTR scbBuffer =
      (SCSI_SCB_SWAPPING_160M SCSI_IPTR) hiob->scbDescriptor->scbBuffer.virtualAddress;

   SCSI_PUT_LITTLE_ENDIAN8(hhcb, scbBuffer,
        OSDoffsetof(SCSI_SCB_SWAPPING_160M, next_SCB), scbNumber);

   /* Flush the cache */
   SCSI_FLUSH_CACHE(scbBuffer, sizeof(SCSI_SCB_SWAPPING_160M));
}
#endif

/*********************************************************************
*
*  SCSIhStandard64ObtainNextScbAddress
*
*     Get next SCB address in the SCB for Standard64 mode
*
*  Return Value:  next scb address
*                  
*  Parameters:    hhcb
*                 hiob
*
*  Remarks:
*
*********************************************************************/
#if SCSI_STANDARD64_MODE
SCSI_BUS_ADDRESS SCSIhStandard64ObtainNextScbAddress (SCSI_HHCB SCSI_HPTR hhcb,
                                                      SCSI_HIOB SCSI_IPTR hiob)
{
   SCSI_BUS_ADDRESS nextScbBusAddress;
   SCSI_SCB_STANDARD64 SCSI_IPTR scbBuffer =
      (SCSI_SCB_STANDARD64 SCSI_IPTR )hiob->scbDescriptor->scbBuffer.virtualAddress;

   SCSI_hGETBUSADDRESS(hhcb, &nextScbBusAddress,
      scbBuffer, OSDoffsetof(SCSI_SCB_STANDARD64, next_SCB_addr0));

   return (nextScbBusAddress);
}
#endif

/*********************************************************************
*
*  SCSIhStandardAdvObtainNextScbAddress
*
*     Get next SCB address in the SCB for Advance Standard mode 
*
*  Return Value:  next scb address
*                  
*  Parameters:    hhcb
*                 hiob
*
*  Remarks:
*
*********************************************************************/
#if SCSI_STANDARD_ADVANCED_MODE
SCSI_BUS_ADDRESS SCSIhStandardAdvObtainNextScbAddress (SCSI_HHCB SCSI_HPTR hhcb,
                                                       SCSI_HIOB SCSI_IPTR hiob)
{
   SCSI_BUS_ADDRESS nextScbBusAddress;
   SCSI_SCB_STANDARD_ADVANCED SCSI_IPTR scbBuffer =
      (SCSI_SCB_STANDARD_ADVANCED SCSI_IPTR )hiob->scbDescriptor->scbBuffer.virtualAddress;

   SCSI_hGETBUSADDRESS(hhcb, &nextScbBusAddress,
      scbBuffer, OSDoffsetof(SCSI_SCB_STANDARD_ADVANCED, next_SCB_addr0));

   return (nextScbBusAddress);
}
#endif

/*********************************************************************
*
*  SCSIhStandard160mObtainNextScbAddress
*
*     Get next SCB address in the SCB for Standard 160m mode 
*
*  Return Value:  next scb address
*                  
*  Parameters:    hhcb
*                 hiob
*
*  Remarks:
*
*********************************************************************/
#if SCSI_STANDARD_160M_MODE
SCSI_BUS_ADDRESS SCSIhStandard160mObtainNextScbAddress (SCSI_HHCB SCSI_HPTR hhcb,
                                                        SCSI_HIOB SCSI_IPTR hiob)
{
   SCSI_BUS_ADDRESS nextScbBusAddress;
   SCSI_SCB_STANDARD_160M SCSI_IPTR scbBuffer =
      (SCSI_SCB_STANDARD_160M SCSI_IPTR) hiob->scbDescriptor->scbBuffer.virtualAddress;

   SCSI_hGETBUSADDRESS(hhcb, &nextScbBusAddress, scbBuffer,
        OSDoffsetof(SCSI_SCB_STANDARD_160M, next_SCB_addr0));

   return (nextScbBusAddress);
}
#endif

/*********************************************************************
*
*  SCSIhStandard64UpdateNextScbAddress
*
*     Modify next SCB address in the SCB for Standard64 mode
*
*  Return Value:  none
*                  
*  Parameters:    hhcb
*                 hiob
*                 ScbBusAddress
*
*  Remarks:
*
*********************************************************************/
#if SCSI_STANDARD64_MODE
void SCSIhStandard64UpdateNextScbAddress (SCSI_HHCB SCSI_HPTR hhcb,
                                          SCSI_HIOB SCSI_IPTR hiob,
                                          SCSI_BUS_ADDRESS ScbBusAddress)
{
   SCSI_SCB_STANDARD64 SCSI_IPTR scbBuffer =
      (SCSI_SCB_STANDARD64 SCSI_IPTR )hiob->scbDescriptor->scbBuffer.virtualAddress;

   SCSI_hPUTBUSADDRESS(hhcb, scbBuffer,
      OSDoffsetof(SCSI_SCB_STANDARD64, next_SCB_addr0), ScbBusAddress);

   /* Flush the cache */
   SCSI_FLUSH_CACHE(scbBuffer, sizeof(SCSI_SCB_STANDARD64));
}
#endif

/*********************************************************************
*
*   SCSIhStandardAdvUpdateNextScbAddress
*
*     Modify next SCB address in the SCB for Advance Standard mode 
*
*  Return Value:  none
*                  
*  Parameters:    hhcb
*                 hiob
*                 ScbBusAddress
*
*  Remarks:
*
*********************************************************************/
#if SCSI_STANDARD_ADVANCED_MODE
void SCSIhStandardAdvUpdateNextScbAddress (SCSI_HHCB SCSI_HPTR hhcb,
                                           SCSI_HIOB SCSI_IPTR hiob,
                                           SCSI_BUS_ADDRESS ScbBusAddress)
{
   SCSI_SCB_STANDARD_ADVANCED SCSI_IPTR scbBuffer =
      (SCSI_SCB_STANDARD_ADVANCED SCSI_IPTR )hiob->scbDescriptor->scbBuffer.virtualAddress;

   SCSI_hPUTBUSADDRESS(hhcb, scbBuffer,
      OSDoffsetof(SCSI_SCB_STANDARD_ADVANCED, next_SCB_addr0), ScbBusAddress);

   /* Flush the cache */
   SCSI_FLUSH_CACHE(scbBuffer, sizeof(SCSI_SCB_STANDARD_ADVANCED));
}
#endif

/*********************************************************************
*
*   SCSIhStandard160mUpdateNextScbAddress
*
*     Modify next SCB address in the SCB for Standard 160m mode 
*
*  Return Value:  none
*                  
*  Parameters:    hhcb
*                 hiob
*                 ScbBusAddress
*
*  Remarks:
*
*********************************************************************/
#if SCSI_STANDARD_160M_MODE
void SCSIhStandard160mUpdateNextScbAddress (SCSI_HHCB SCSI_HPTR hhcb,
                                            SCSI_HIOB SCSI_IPTR hiob,
                                            SCSI_BUS_ADDRESS ScbBusAddress)
{
   SCSI_SCB_STANDARD_160M SCSI_IPTR scbBuffer =
      (SCSI_SCB_STANDARD_160M SCSI_IPTR) hiob->scbDescriptor->scbBuffer.virtualAddress;

   SCSI_hPUTBUSADDRESS(hhcb, scbBuffer,
        OSDoffsetof(SCSI_SCB_STANDARD_160M, next_SCB_addr0), ScbBusAddress);

   /* Flush the cache */
   SCSI_FLUSH_CACHE(scbBuffer, sizeof(SCSI_SCB_STANDARD_160M));
}
#endif

/*********************************************************************
*
*  SCSIhStandard64SearchExeQ
*
*     Search Execution queue for the aborting HIOB.
*
*  Return Value:  0 - aborting HIOB not found
*                 1 - aborting HIOB found
*
*  Parameters:    hhcb
*                 aborting hiob
*                 host adapter status
*                 post HIOB back if found flag:
*                    1 - yes
*                    0 - no
*
*  Remarks:
*
*********************************************************************/
#if SCSI_STANDARD64_MODE
SCSI_UEXACT8 SCSIhStandard64SearchExeQ (SCSI_HHCB SCSI_HPTR hhcb,
                                        SCSI_HIOB SCSI_IPTR hiob,
                                        SCSI_UEXACT8 haStatus,
                                        SCSI_UEXACT8 postHiob)
{
   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_UEXACT8 scbptr;
   SCSI_UEXACT8 scbNumber;
   SCSI_UEXACT8 prevScb;
   SCSI_UEXACT8 nextScb;
   SCSI_UEXACT8 retStat = 0;

   scbptr = OSD_INEXACT8(SCSI_AICREG(SCSI_SCBPTR));   /* Save SCBPTR */

   /* Check if aborting HIOB is in Execution queue */
   prevScb = scbNumber = OSD_INEXACT8(SCSI_AICREG(SCSI_S64_Q_EXE_HEAD));

   while (scbNumber != SCSI_NULL_SCB)
   {
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCBPTR), scbNumber);

      if (scbNumber == hiob->scbNumber)
      {
         /* Set abort status */
         hiob->stat = SCSI_SCB_ABORTED;
         hiob->haStat = haStatus;

         /* Set abort request is done */
         hiob->SCSI_IP_mgrStat = SCSI_MGR_DONE_ABT;

         /* Remove the SCB from execution queue */
         nextScb = OSD_INEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
                       OSDoffsetof(SCSI_SCB_STANDARD64, SCSI_S64_q_next)));
                        
         /* Mending the execution queue */
         if (scbNumber == prevScb)
         {
            /* Removed HIOB at the queue head */
            OSD_OUTEXACT8(SCSI_AICREG(SCSI_S64_Q_EXE_HEAD), nextScb);
         }
         else
         {
            OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCBPTR), prevScb);
            OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
                OSDoffsetof(SCSI_SCB_STANDARD64, SCSI_S64_q_next)), nextScb);

            if (scbNumber == OSD_INEXACT8(SCSI_AICREG(SCSI_S64_Q_EXE_TAIL)))
            {
               /* Removed HIOB at the queue tail */
               OSD_OUTEXACT8(SCSI_AICREG(SCSI_S64_Q_EXE_TAIL), prevScb);
            }
         }

         if (postHiob)
         {
            /* Post it back to upper layer as instructed */
            SCSIhTerminateCommand(hiob);
         }

         retStat = 1;
         break;
      }

      /* Keep previous SCB */
      prevScb = scbNumber;
      
      /* Advance to next SCB in Execution queue */
      scbNumber = OSD_INEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
                      OSDoffsetof(SCSI_SCB_STANDARD64, SCSI_S64_q_next)));
   }

   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCBPTR), scbptr);   /* Restore SCBPTR */

   return(retStat);
}
#endif   /* SCSI_STANDARD64_MODE */

/*********************************************************************
*
*  SCSIhSwappingSearchExeQ
*
*     Search Execution queue for the aborting HIOB.
*
*  Return Value:  0 - aborting HIOB not found
*                 1 - aborting HIOB found
*
*  Parameters:    hhcb
*                 aborting hiob
*                 host adapter status
*                 post HIOB back if found flag:
*                    1 - yes
*                    0 - no
*
*  Remarks:
*
*********************************************************************/
#if SCSI_SWAPPING_MODE
SCSI_UEXACT8 SCSIhSwappingSearchExeQ (SCSI_HHCB SCSI_HPTR hhcb,
                                        SCSI_HIOB SCSI_IPTR hiob,
                                        SCSI_UEXACT8 haStatus,
                                        SCSI_UEXACT8 postHiob)
{
   return(0);
}
#endif

/*********************************************************************
*
*  SCSIhStandardAdvSearchExeQ
*
*     Search Execution queue for the aborting HIOB.
*
*  Return Value:  0 - aborting HIOB not found
*                 1 - aborting HIOB found
*
*  Parameters:    hhcb
*                 aborting hiob
*                 host adapter status
*                 post HIOB back if found flag:
*                    1 - yes
*                    0 - no
*
*  Remarks:
*
*********************************************************************/
#if SCSI_STANDARD_ADVANCED_MODE
SCSI_UEXACT8 SCSIhStandardAdvSearchExeQ (SCSI_HHCB SCSI_HPTR hhcb,
                                        SCSI_HIOB SCSI_IPTR hiob,
                                        SCSI_UEXACT8 haStatus,
                                        SCSI_UEXACT8 postHiob)
{
   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_UEXACT8 scbptr;
   SCSI_UEXACT8 scbNumber;
   SCSI_UEXACT8 prevScb;
   SCSI_UEXACT8 nextScb;
   SCSI_UEXACT8 retStat = 0;

   scbptr = OSD_INEXACT8(SCSI_AICREG(SCSI_SCBPTR));   /* Save SCBPTR */

   /* Check if aborting HIOB is in Execution queue */
   prevScb = scbNumber = OSD_INEXACT8(SCSI_AICREG(SCSI_SADV_Q_EXE_HEAD));

   while (scbNumber != SCSI_NULL_SCB)
   {
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCBPTR), scbNumber);

      if (scbNumber == hiob->scbNumber)
      {
         /* Set abort status */
         hiob->stat = SCSI_SCB_ABORTED;
         hiob->haStat = haStatus;

         /* Set abort request is done */
         hiob->SCSI_IP_mgrStat = SCSI_MGR_DONE_ABT;

         /* Remove the SCB from execution queue */
         nextScb = OSD_INEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
                       OSDoffsetof(SCSI_SCB_STANDARD_ADVANCED, SCSI_SADV_q_next)));
                        
         /* Mending the execution queue */
         if (scbNumber == prevScb)
         {
            /* Removed HIOB at the queue head */
            OSD_OUTEXACT8(SCSI_AICREG(SCSI_SADV_Q_EXE_HEAD), nextScb);
         }
         else
         {
            OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCBPTR), prevScb);
            OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
                OSDoffsetof(SCSI_SCB_STANDARD_ADVANCED, SCSI_SADV_q_next)), nextScb);

            if (scbNumber == OSD_INEXACT8(SCSI_AICREG(SCSI_SADV_Q_EXE_TAIL)))
            {
               /* Removed HIOB at the queue tail */
               OSD_OUTEXACT8(SCSI_AICREG(SCSI_SADV_Q_EXE_TAIL), prevScb);
            }
         }

         if (postHiob)
         {
            /* Post it back to upper layer as instructed */
            SCSIhTerminateCommand(hiob);
         }

         retStat = 1;
         break;
      }

      /* Keep previous SCB */
      prevScb = scbNumber;
      
      /* Advance to next SCB in Execution queue */
      scbNumber = OSD_INEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
                      OSDoffsetof(SCSI_SCB_STANDARD_ADVANCED, SCSI_SADV_q_next)));
   }

   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCBPTR), scbptr);   /* Restore SCBPTR */

   return(retStat);
}
#endif   /* SCSI_STANDARD_ADVANCED_MODE */

/*********************************************************************
*
*  SCSIhStandard160mSearchExeQ
*
*     Search Execution queue for the aborting HIOB.
*
*  Return Value:  0 - aborting HIOB not found
*                 1 - aborting HIOB found
*
*  Parameters:    hhcb
*                 aborting hiob
*                 host adapter status
*                 post HIOB back if found flag:
*                    1 - yes
*                    0 - no
*
*  Remarks:
*
*********************************************************************/
#if SCSI_STANDARD_160M_MODE
SCSI_UEXACT8 SCSIhStandard160mSearchExeQ (SCSI_HHCB SCSI_HPTR hhcb,
                                          SCSI_HIOB SCSI_IPTR hiob,
                                          SCSI_UEXACT8 haStatus,
                                          SCSI_UEXACT8 postHiob)
{
   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_UEXACT8 scbptr;
   SCSI_UEXACT8 scbNumber;
   SCSI_UEXACT8 prevScb;
   SCSI_UEXACT8 nextScb;
   SCSI_UEXACT8 retStat = 0;

   scbptr = OSD_INEXACT8(SCSI_AICREG(SCSI_SCBPTR));   /* Save SCBPTR */

   /* Check if aborting HIOB is in Execution queue */
   prevScb = scbNumber = OSD_INEXACT8(SCSI_AICREG(SCSI_S160M_Q_EXE_HEAD));

   while (scbNumber != SCSI_NULL_SCB)
   {
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCBPTR), scbNumber);

      if (scbNumber == hiob->scbNumber)
      {
         /* Set abort status */
         hiob->stat = SCSI_SCB_ABORTED;
         hiob->haStat = haStatus;

         /* Set abort request is done */
         hiob->SCSI_IP_mgrStat = SCSI_MGR_DONE_ABT;

         /* Remove the SCB from execution queue */
         nextScb = OSD_INEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
                       OSDoffsetof(SCSI_SCB_STANDARD_160M, SCSI_S160M_q_next)));
                        
         /* Mending the execution queue */
         if (scbNumber == prevScb)
         {
            /* Removed HIOB at the queue head */
            OSD_OUTEXACT8(SCSI_AICREG(SCSI_S160M_Q_EXE_HEAD), nextScb);
         }
         else
         {
            OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCBPTR), prevScb);
            OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
                OSDoffsetof(SCSI_SCB_STANDARD_160M, SCSI_S160M_q_next)), nextScb);

            if (scbNumber == OSD_INEXACT8(SCSI_AICREG(SCSI_S160M_Q_EXE_TAIL)))
            {
               /* Removed HIOB at the queue tail */
               OSD_OUTEXACT8(SCSI_AICREG(SCSI_S160M_Q_EXE_TAIL), prevScb);
            }
         }

         if (postHiob)
         {
            /* Post it back to upper layer as instructed */
            SCSIhTerminateCommand(hiob);
         }

         retStat = 1;
         break;
      }

      /* Keep previous SCB */
      prevScb = scbNumber;
      
      /* Advance to next SCB in Execution queue */
      scbNumber = OSD_INEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
                      OSDoffsetof(SCSI_SCB_STANDARD_160M, SCSI_S160M_q_next)));
   }

   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCBPTR), scbptr);   /* Restore SCBPTR */

   return(retStat);
}
#endif   /* SCSI_STANDARD_160M_MODE */

/*********************************************************************
*
*  SCSIhStandard64UpdateAbortBitHostMem routine -
*
*  Update the chain control field in host memory for  
*  standard 64 mode.
*
*  Return Value:  none.
*                  
*  Parameters:    hhcb
*                 hiob
*
*  Remarks:                
*                  
*********************************************************************/
#if SCSI_STANDARD64_MODE
void SCSIhStandard64UpdateAbortBitHostMem (SCSI_HHCB SCSI_HPTR hhcb,
                                           SCSI_HIOB SCSI_IPTR hiob)
{
   SCSI_SCB_STANDARD64 SCSI_HPTR scbS64;
   SCSI_UEXACT8 byteBuf;

   /* Prepare scb to be aborted via 'aborted' bit of chain control */
   scbS64 = (SCSI_SCB_STANDARD64 SCSI_HPTR) hiob->scbDescriptor->scbBuffer.virtualAddress;

   SCSI_GET_LITTLE_ENDIAN8(hhcb, &byteBuf, scbS64,
           OSDoffsetof(SCSI_SCB_STANDARD64, chain_control));
   SCSI_PUT_LITTLE_ENDIAN8(hhcb, scbS64, OSDoffsetof(SCSI_SCB_STANDARD64,
        chain_control), byteBuf | SCSI_S64_ABORTED);

   /* Flush the cache */
   SCSI_FLUSH_CACHE(scbS64, sizeof(SCSI_SCB_STANDARD64));
}
#endif

/*********************************************************************
*
*  SCSIhStandardAdvUpdateAbortBitHostMem routine -
*
*  Update the chain control field in host memory for  
*  standard Advanced mode.
*
*  Return Value:  none.
*                  
*  Parameters:    hhcb
*                 hiob
*
*  Remarks:                
*                  
*********************************************************************/
#if SCSI_STANDARD_ADVANCED_MODE
void SCSIhStandardAdvUpdateAbortBitHostMem (SCSI_HHCB SCSI_HPTR hhcb,
                                            SCSI_HIOB SCSI_IPTR hiob)
{
   SCSI_SCB_STANDARD_ADVANCED SCSI_HPTR scbSAdv;
   SCSI_UEXACT8 byteBuf;

   /* Prepare scb to be aborted via 'aborted' bit of chain control */
   scbSAdv = (SCSI_SCB_STANDARD_ADVANCED SCSI_HPTR) hiob->scbDescriptor->scbBuffer.virtualAddress;
   SCSI_GET_LITTLE_ENDIAN8(hhcb, &byteBuf, scbSAdv,
           OSDoffsetof(SCSI_SCB_STANDARD_ADVANCED, scontrol));
   SCSI_PUT_LITTLE_ENDIAN8(hhcb, scbSAdv, OSDoffsetof(SCSI_SCB_STANDARD_ADVANCED,
        scontrol), byteBuf | SCSI_SADV_ABORTED);
   /* Flush the cache */
   SCSI_FLUSH_CACHE(scbSAdv, sizeof(SCSI_SCB_STANDARD_ADVANCED));
}
#endif

/*********************************************************************
*
*  SCSIhStandard160mUpdateAbortBitHostMem routine -
*
*  Update the chain control field in host memory for
*  standard 160m mode
*
*  Return Value:  none.
*                  
*  Parameters:    hhcb
*                 hiob
*
*  Remarks:                
*                  
*********************************************************************/
#if SCSI_STANDARD_160M_MODE
void SCSIhStandard160mUpdateAbortBitHostMem (SCSI_HHCB SCSI_HPTR hhcb,
                                             SCSI_HIOB SCSI_IPTR hiob)
{
   SCSI_SCB_STANDARD_160M SCSI_HPTR scbS160m;
   SCSI_UEXACT8 byteBuf;

   /* Prepare scb to be aborted via 'aborted' bit of chain control */
   scbS160m = (SCSI_SCB_STANDARD_160M SCSI_HPTR) hiob->scbDescriptor->scbBuffer.virtualAddress;
   SCSI_GET_LITTLE_ENDIAN8(hhcb, &byteBuf, scbS160m,
        OSDoffsetof(SCSI_SCB_STANDARD_160M, scontrol));
   SCSI_PUT_LITTLE_ENDIAN8(hhcb, scbS160m,
        OSDoffsetof(SCSI_SCB_STANDARD_160M, scontrol),
        byteBuf | SCSI_S160M_ABORTED);

   /* Flush the cache */
   SCSI_FLUSH_CACHE(scbS160m, sizeof(SCSI_SCB_STANDARD_160M));
}
#endif

/*********************************************************************
*
*  SCSIhSwapping32UpdateAbortBitHostMem routine -
*
*  Update the chain control field in host memory for  
*  swapping 32 mode.
*
*  Return Value:  none.
*                  
*  Parameters:    hhcb
*                 hiob
*
*  Remarks:                
*                  
*********************************************************************/
#if SCSI_SWAPPING32_MODE
void SCSIhSwapping32UpdateAbortBitHostMem (SCSI_HHCB SCSI_HPTR hhcb,
                                           SCSI_HIOB SCSI_IPTR hiob)
{
   SCSI_SCB_SWAPPING32 SCSI_HPTR scbW32;
   SCSI_UEXACT8 byteBuf;

   /* Prepare scb to be aborted via 'aborted' bit of chain control */
   scbW32 = (SCSI_SCB_SWAPPING32 SCSI_HPTR) hiob->scbDescriptor->scbBuffer.virtualAddress;

   SCSI_GET_LITTLE_ENDIAN8(hhcb, &byteBuf, scbW32,
           OSDoffsetof(SCSI_SCB_SWAPPING32, chain_control));
   SCSI_PUT_LITTLE_ENDIAN8(hhcb, scbW32, OSDoffsetof(SCSI_SCB_SWAPPING32,
        chain_control), byteBuf | SCSI_W32_ABORTED);

   /* Flush the cache */
   SCSI_FLUSH_CACHE(scbW32, sizeof(SCSI_SCB_SWAPPING32));
}
#endif

/*********************************************************************
*
*  SCSIhSwapping64UpdateAbortBitHostMem routine -
*
*  Update the chain control field in host memory for  
*  swapping 64 mode.
*
*  Return Value:  none.
*                  
*  Parameters:    hhcb
*                 hiob
*
*  Remarks:                
*                  
*********************************************************************/
#if SCSI_SWAPPING64_MODE
void SCSIhSwapping64UpdateAbortBitHostMem (SCSI_HHCB SCSI_HPTR hhcb,
                                           SCSI_HIOB SCSI_IPTR hiob)
{
   SCSI_SCB_SWAPPING64 SCSI_HPTR scbW64;
   SCSI_UEXACT8 byteBuf;

   /* Prepare scb to be aborted via 'aborted' bit of chain control */
   scbW64 = (SCSI_SCB_SWAPPING64 SCSI_HPTR) hiob->scbDescriptor->scbBuffer.virtualAddress;

   SCSI_GET_LITTLE_ENDIAN8(hhcb, &byteBuf, scbW64,
           OSDoffsetof(SCSI_SCB_SWAPPING64, chain_control));
   SCSI_PUT_LITTLE_ENDIAN8(hhcb, scbW64, OSDoffsetof(SCSI_SCB_SWAPPING64,
        chain_control), byteBuf | SCSI_W64_ABORTED);

   /* Flush the cache */
   SCSI_FLUSH_CACHE(scbW64, sizeof(SCSI_SCB_SWAPPING64));
}
#endif

/*********************************************************************
*
*  SCSIhSwappingAdvUpdateAbortBitHostMem routine -
*
*  Update the chain control field in host memory for  
*  swapping Advanced mode.
*
*  Return Value:  none.
*                  
*  Parameters:    hhcb
*                 hiob
*
*  Remarks:                
*                  
*********************************************************************/
#if SCSI_SWAPPING_ADVANCED_MODE
void SCSIhSwappingAdvUpdateAbortBitHostMem (SCSI_HHCB SCSI_HPTR hhcb,
                                            SCSI_HIOB SCSI_IPTR hiob)
{
   SCSI_SCB_SWAPPING_ADVANCED SCSI_HPTR scbWAdv;
   SCSI_UEXACT8 byteBuf;

   /* Prepare scb to be aborted via 'aborted' bit of scontrol */
   scbWAdv = (SCSI_SCB_SWAPPING_ADVANCED SCSI_HPTR) hiob->scbDescriptor->scbBuffer.virtualAddress;
   SCSI_GET_LITTLE_ENDIAN8(hhcb, &byteBuf, scbWAdv,
           OSDoffsetof(SCSI_SCB_SWAPPING_ADVANCED, scontrol));
   SCSI_PUT_LITTLE_ENDIAN8(hhcb, scbWAdv, OSDoffsetof(SCSI_SCB_SWAPPING_ADVANCED,
        scontrol), byteBuf | SCSI_WADV_ABORTED);

   /* Flush the cache */
   SCSI_FLUSH_CACHE(scbWAdv, sizeof(SCSI_SCB_SWAPPING_ADVANCED));
}
#endif

/*********************************************************************
*
*  SCSIhSwapping160mUpdateAbortBitHostMem routine -
*
*  Update the chain control field in host memory for  
*  swapping 160m mode.
*
*  Return Value:  none.
*                  
*  Parameters:    hhcb
*                 hiob
*
*  Remarks:                
*                  
*********************************************************************/
#if SCSI_SWAPPING_160M_MODE
void SCSIhSwapping160mUpdateAbortBitHostMem (SCSI_HHCB SCSI_HPTR hhcb,
                                             SCSI_HIOB SCSI_IPTR hiob)
{

   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_SCB_SWAPPING_160M SCSI_HPTR scbW160m;
   SCSI_UEXACT8 byteBuf;

   /* Prepare scb to be aborted via 'aborted' bit of scontrol */
   scbW160m = (SCSI_SCB_SWAPPING_160M SCSI_HPTR) hiob->scbDescriptor->scbBuffer.virtualAddress;
   SCSI_GET_LITTLE_ENDIAN8(hhcb, &byteBuf, scbW160m,
        OSDoffsetof(SCSI_SCB_SWAPPING_160M, scontrol));
   SCSI_PUT_LITTLE_ENDIAN8(hhcb, scbW160m,
        OSDoffsetof(SCSI_SCB_SWAPPING_160M, scontrol),
        byteBuf | SCSI_W160M_ABORTED);

   /* Flush the cache */
   SCSI_FLUSH_CACHE(scbW160m, sizeof(SCSI_SCB_SWAPPING_160M));

   OSD_OUTEXACT8(SCSI_AICREG(SCSI_W160M_CHECK_ABORT), 1);
   
   hhcb->numberSCBAborts++;
   
}
#endif

/*********************************************************************
*
*  SCSIhStandard64SetupBDR
*
*     Setup Bus Device Reset for Standard mode sequencer.
*
*  Return Value:  none
*
*  Parameters:    hhcb
*                 hiob
*
*  Remarks:       Setup Bus Device Reset directly in SCB ram.
*                 Enqueue SCB to the head of excecution queue.
*
*********************************************************************/
#if SCSI_STANDARD64_MODE
void SCSIhStandard64SetupBDR (SCSI_HHCB SCSI_HPTR hhcb,
                              SCSI_HIOB SCSI_IPTR hiob)
{

   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;

   SCSI_UNIT_CONTROL SCSI_UPTR targetUnit = SCSI_TARGET_UNIT(hiob);
   SCSI_UEXACT8 scbptr;
   SCSI_UEXACT8 hcntrl;
   SCSI_INT i;

   /* Save the HCNTRL value */
   hcntrl = OSD_INEXACT8(SCSI_AICREG(SCSI_HCNTRL));

   /* Pause the chip if necessary */
   if (!(hcntrl & SCSI_PAUSEACK))
   {
      SCSIhPauseAndWait(hhcb);
   }

   scbptr = OSD_INEXACT8(SCSI_AICREG(SCSI_SCBPTR));   /* Save SCBPTR */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCBPTR), hiob->scbNumber);
   
   /* Clear SCB ram */
   for (i = 0; i < 40; i++)
   {
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+i), 0);
   }

   /* Setup special function (SCB00) - Bit 3 of the SCB control flag */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+OSDoffsetof(SCSI_SCB_STANDARD64,
       scontrol)), SCSI_S64_SPECFUN);

   /* Setup target id/lun (SCB01 & SCB039) */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+OSDoffsetof(SCSI_SCB_STANDARD64,starget)),
                   (SCSI_UEXACT8)targetUnit->scsiID); /* SCB01  */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+OSDoffsetof(SCSI_SCB_STANDARD64,slun)),
                   (SCSI_UEXACT8)targetUnit->lunID); /* SCB39  */

   /* Setup no data length (SCB02) */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+OSDoffsetof(SCSI_SCB_STANDARD64,
       scdb_length)), SCSI_S64_NODATA);

   /* Setup special opcode - message to target (SCB03) */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+OSDoffsetof(SCSI_SCB_STANDARD64,
       SCSI_S64_special_opcode)), SCSI_S64_MSGTOTARG);

   /* Setup special info - Bus Device Reset message (SCB04) */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+OSDoffsetof(SCSI_SCB_STANDARD64,
       SCSI_S64_special_info)), SCSI_MSG0C);

   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCBPTR), scbptr);   /* Restore SCBPTR */

   /* Put BDR HIOB into active pointer array */
   SCSI_ACTPTR[hiob->scbNumber] = hiob;

   /* Enqueue BDR HIOB at head of execution queue. */
   /* To avoid CHIM and Sequencer might access to the same location */
   /* in the execution queue, we have to stop the sequencer at */
   /* the idle loop before enqueue the HIOB. */

   /* Set BDR HIOB state needs to be enqueued */
   hiob->SCSI_IP_mgrStat = SCSI_MGR_ENQUE_BDR;

   /* set non expander break */
   hhcb->SCSI_HP_nonExpBreak = 1;

   /* Set the breakpoint address at sequencer idle loop. Since the sequencer */
   /* program counter always be one instruction ahead, we need to add 4 to */
   /* the actual idle loop value.  One sequencer instruction is 29 bits. */

   /* Set at idle_loop0 because the sequencer will jump to it */
   /* while waiting for any SCB to execute. */
   SCSI_hSETBREAKPOINT(hhcb, (SCSI_UEXACT8)ENTRY_IDLE_LOOP);

   /* Restore HCNTRL if necessary */
   if (!(hcntrl & SCSI_PAUSEACK))
   {
      SCSI_hWRITE_HCNTRL(hhcb, scsiRegister, hcntrl);
   }
}
#endif   /* SCSI_STANDARD64_MODE */

/*********************************************************************
*
*  SCSIhSwapping32SetupBDR
*
*     Setup Bus Device Reset for Swapping mode sequencer.
*
*  Return Value:  none
*                  
*  Parameters:    host task set handle
*                 hiob
*
*  Remarks:       Setup Bus Device Reset directly in SCB ram.
*                 Enqueue SCB to the head of excecution queue.
*                  
*********************************************************************/
#if SCSI_SWAPPING32_MODE
void SCSIhSwapping32SetupBDR (SCSI_HHCB SCSI_HPTR hhcb,
                              SCSI_HIOB SCSI_IPTR hiob)
{
   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_UNIT_CONTROL SCSI_IPTR targetUnit = SCSI_TARGET_UNIT(hiob);
   SCSI_SCB_SWAPPING32 SCSI_HPTR scbW32;
   SCSI_UEXACT8 hcntrl;

   /* Save the HCNTRL value */
   hcntrl = OSD_INEXACT8(SCSI_AICREG(SCSI_HCNTRL));

   /* Pause the chip if necessary */
   if (!(hcntrl & SCSI_PAUSEACK))
   {
      SCSIhPauseAndWait(hhcb);
   }

   /* assign scb buffer when necessary */

   scbW32 = (SCSI_SCB_SWAPPING32 SCSI_HPTR) hiob->scbDescriptor->scbBuffer.virtualAddress;

   /* Clear swapping SCB buffer */
   OSDmemset(scbW32, 0, sizeof(SCSI_SCB_SWAPPING32));

   /* Setup special function (SCB00) - Bit 3 of the General Flags Byte */
   SCSI_PUT_LITTLE_ENDIAN8(hhcb, scbW32, OSDoffsetof(SCSI_SCB_SWAPPING32,
            scontrol), SCSI_W32_SPECFUN);

   /* Setup target id (SCB01) */
   SCSI_PUT_LITTLE_ENDIAN8(hhcb, scbW32, OSDoffsetof(SCSI_SCB_SWAPPING32,
            starget), targetUnit->scsiID);

   /* Setup target lun (SCB02) */
   SCSI_PUT_LITTLE_ENDIAN8(hhcb, scbW32, OSDoffsetof(SCSI_SCB_SWAPPING32,
            slun), targetUnit->lunID);

   /* Setup no data length (SCB03) */
   SCSI_PUT_LITTLE_ENDIAN8(hhcb, scbW32, OSDoffsetof(SCSI_SCB_SWAPPING32,
            scdb_length), SCSI_W32_NODATA);

   /* Setup special opcode - message to target (SCB04) */
   SCSI_PUT_LITTLE_ENDIAN8(hhcb, scbW32, OSDoffsetof(SCSI_SCB_SWAPPING32,
            SCSI_W32_special_opcode), SCSI_W32_MSGTOTARG);

   /* Setup special info - Bus Device Reset message (SCB05) */
   SCSI_PUT_LITTLE_ENDIAN8(hhcb, scbW32, OSDoffsetof(SCSI_SCB_SWAPPING32,
            SCSI_W32_special_info), SCSI_MSG0C);
   
   /* Setup the next_SCB with the next scb number */
   SCSI_PUT_LITTLE_ENDIAN8(hhcb, scbW32,
       OSDoffsetof(SCSI_SCB_SWAPPING32,next_SCB),
         hiob->scbDescriptor->queueNext->scbNumber);

   /* Put BDR HIOB into active pointer array */
   SCSI_ACTPTR[hiob->scbNumber] = hiob;

   /* Flush the cache */
   SCSI_FLUSH_CACHE(scbW32, sizeof(SCSI_SCB_SWAPPING32));

   /* Enqueue BDR HIOB at head of new queue. */
   /* To avoid CHIM and Sequencer might access to the same location */
   /* in the new queue, we have to stop the sequencer */
   /* at the idle loop before enqueue the HIOB. */

   /* Set BDR HIOB state needs to be enqueued */
   hiob->SCSI_IP_mgrStat = SCSI_MGR_ENQUE_BDR;

   /* set non expander break */
   hhcb->SCSI_HP_nonExpBreak = 1;

   /* Set the breakpoint address at sequencer idle loop entry. Since the */
   /* sequencer program counter always be one instruction ahead, we need */
   /* to add 4 to actual idle loop entry value.  One sequencer instruction */
   /* is 29 bits. */
   SCSI_hSETBREAKPOINT(hhcb, (SCSI_UEXACT8)ENTRY_IDLE_LOOP);

   /* Restore HCNTRL if necessary */
   if (!(hcntrl & SCSI_PAUSEACK))
   {
      SCSI_hWRITE_HCNTRL(hhcb, scsiRegister, hcntrl);
   }
}
#endif   /* SCSI_SWAPPING32_MODE */


/*********************************************************************
*
*  SCSIhSwapping64SetupBDR
*
*     Setup Bus Device Reset for Swapping mode sequencer.
*
*  Return Value:  none
*                  
*  Parameters:    host task set handle
*                 hiob
*
*  Remarks:       Setup Bus Device Reset directly in SCB ram.
*                 Enqueue SCB to the head of excecution queue.
*                  
*********************************************************************/
#if SCSI_SWAPPING64_MODE
void SCSIhSwapping64SetupBDR (SCSI_HHCB SCSI_HPTR hhcb,
                              SCSI_HIOB SCSI_IPTR hiob)
{
   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_UNIT_CONTROL SCSI_IPTR targetUnit = SCSI_TARGET_UNIT(hiob);
   SCSI_SCB_SWAPPING64 SCSI_HPTR scbW64;
   SCSI_UEXACT8 hcntrl;

   /* Save the HCNTRL value */
   hcntrl = OSD_INEXACT8(SCSI_AICREG(SCSI_HCNTRL));

   /* Pause the chip if necessary */
   if (!(hcntrl & SCSI_PAUSEACK))
   {
      SCSIhPauseAndWait(hhcb);
   }

   /* assign scb buffer when necessary */

   scbW64 = (SCSI_SCB_SWAPPING64 SCSI_HPTR) hiob->scbDescriptor->scbBuffer.virtualAddress;

   /* Clear swapping SCB buffer */
   OSDmemset(scbW64, 0, sizeof(SCSI_SCB_SWAPPING64));

   /* Setup special function (SCB00) - Bit 3 of the General Flags Byte */
   SCSI_PUT_LITTLE_ENDIAN8(hhcb, scbW64, OSDoffsetof(SCSI_SCB_SWAPPING64,
            scontrol), SCSI_W64_SPECFUN);

   /* Setup target id (SCB01) */
   SCSI_PUT_LITTLE_ENDIAN8(hhcb, scbW64, OSDoffsetof(SCSI_SCB_SWAPPING64,
            starget), targetUnit->scsiID);

   /* Setup target lun (SCB02) */
   SCSI_PUT_LITTLE_ENDIAN8(hhcb, scbW64, OSDoffsetof(SCSI_SCB_SWAPPING64,
            slun), targetUnit->lunID);

   /* Setup no data length (SCB03) */
   SCSI_PUT_LITTLE_ENDIAN8(hhcb, scbW64, OSDoffsetof(SCSI_SCB_SWAPPING64,
            scdb_length), SCSI_W64_NODATA);

   /* Setup special opcode - message to target (SCB04) */
   SCSI_PUT_LITTLE_ENDIAN8(hhcb, scbW64, OSDoffsetof(SCSI_SCB_SWAPPING64,
            SCSI_W64_special_opcode), SCSI_W64_MSGTOTARG);

   /* Setup special info - Bus Device Reset message (SCB05) */
   SCSI_PUT_LITTLE_ENDIAN8(hhcb, scbW64, OSDoffsetof(SCSI_SCB_SWAPPING64,
            SCSI_W64_special_info), SCSI_MSG0C);
   
   /* Setup the next_SCB with the next scb number */
   SCSI_PUT_LITTLE_ENDIAN8(hhcb, scbW64,
       OSDoffsetof(SCSI_SCB_SWAPPING64,next_SCB),
         hiob->scbDescriptor->queueNext->scbNumber);

   /* Put BDR HIOB into active pointer array */
   SCSI_ACTPTR[hiob->scbNumber] = hiob;

   /* Flush the cache */
   SCSI_FLUSH_CACHE(scbW64, sizeof(SCSI_SCB_SWAPPING64));

   /* Enqueue BDR HIOB at head of new queue. */
   /* To avoid CHIM and Sequencer might access to the same location */
   /* in the new queue, we have to stop the sequencer */
   /* at the idle loop before enqueue the HIOB. */

   /* Set BDR HIOB state needs to be enqueued */
   hiob->SCSI_IP_mgrStat = SCSI_MGR_ENQUE_BDR;

   /* set non expander break */
   hhcb->SCSI_HP_nonExpBreak = 1;

   /* Set the breakpoint address at sequencer idle loop entry. Since the */
   /* sequencer program counter always be one instruction ahead, we need */
   /* to add 4 to actual idle loop entry value.  One sequencer instruction */
   /* is 29 bits. */
   SCSI_hSETBREAKPOINT(hhcb, (SCSI_UEXACT8)ENTRY_IDLE_LOOP);

   /* Restore HCNTRL if necessary */
   if (!(hcntrl & SCSI_PAUSEACK))
   {
      SCSI_hWRITE_HCNTRL(hhcb, scsiRegister, hcntrl);
   }
}
#endif   /* SCSI_SWAPPING64_MODE */

/*********************************************************************
*
*  SCSIhStandardAdvSetupBDR
*
*     Setup Bus Device Reset for Standard mode sequencer.
*
*  Return Value:  none
*
*  Parameters:    hhcb
*                 hiob
*
*  Remarks:       Setup Bus Device Reset directly in SCB ram.
*                 Enqueue SCB to the head of excecution queue.
*
*********************************************************************/
#if SCSI_STANDARD_ADVANCED_MODE
void SCSIhStandardAdvSetupBDR (SCSI_HHCB SCSI_HPTR hhcb,
                               SCSI_HIOB SCSI_IPTR hiob)
{
   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_UNIT_CONTROL SCSI_UPTR targetUnit = SCSI_TARGET_UNIT(hiob);
   SCSI_UEXACT8 scbptr;
   SCSI_UEXACT8 hcntrl;
   SCSI_INT i;

   SCSI_UEXACT8 starget=0;

   /* The enhanced sequencer will put the HA scsi ID into the hi nybble of */
   /* starget after the SCB is in the SCB RAM.  Since this routine will PIO */
   /* the scb directly to the RAM, we must do the same. */
   starget = hhcb->hostScsiID;
   starget <<= 4;
   starget |= (SCSI_UEXACT8)targetUnit->scsiID;

   /* Save the HCNTRL value */
   hcntrl = OSD_INEXACT8(SCSI_AICREG(SCSI_HCNTRL));

   /* Pause the chip if necessary */
   if (!(hcntrl & SCSI_PAUSEACK))
   {
      SCSIhPauseAndWait(hhcb);
   }

   scbptr = OSD_INEXACT8(SCSI_AICREG(SCSI_SCBPTR));   /* Save SCBPTR */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCBPTR), hiob->scbNumber);
   
   /* Clear SCB ram */
   for (i = 0; i < 40; i++)
   {
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+i), 0);
   }

   /* Setup special function - Bit 3 of the SCB control flag */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+OSDoffsetof(SCSI_SCB_STANDARD_ADVANCED,
       scontrol)), SCSI_SADV_SPECFUN);

   /* Setup target id/lun */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
         OSDoffsetof(SCSI_SCB_STANDARD_ADVANCED,starget)),
         (SCSI_UEXACT8)starget);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
                     OSDoffsetof(SCSI_SCB_STANDARD_ADVANCED,slun)),
                     (SCSI_UEXACT8)targetUnit->lunID);

   /* Setup no data length */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+OSDoffsetof(SCSI_SCB_STANDARD_ADVANCED,
       scdb_length)), SCSI_SADV_NODATA);

   /* Setup special opcode - message to target  */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+OSDoffsetof(SCSI_SCB_STANDARD_ADVANCED,
       SCSI_SADV_special_opcode)), SCSI_SADV_MSGTOTARG);

   /* Setup special info - Bus Device Reset message */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+OSDoffsetof(SCSI_SCB_STANDARD_ADVANCED,
       SCSI_SADV_special_info)), SCSI_MSG0C);

   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCBPTR), scbptr);   /* Restore SCBPTR */

   /* Put BDR HIOB into active pointer array */
   SCSI_ACTPTR[hiob->scbNumber] = hiob;

   /* Enqueue BDR HIOB at head of execution queue. */
   /* To avoid CHIM and Sequencer might access to the same location */
   /* in the execution queue, we have to stop the sequencer at */
   /* the idle loop before enqueue the HIOB. */

   /* Set BDR HIOB state needs to be enqueued */
   hiob->SCSI_IP_mgrStat = SCSI_MGR_ENQUE_BDR;

   /* set non expanser break */
   hhcb->SCSI_HP_nonExpBreak = 1;

   /* Set the breakpoint address at sequencer idle loop. Since the sequencer */
   /* program counter always be one instruction ahead, we need to add 4 to */
   /* the actual idle loop value.  One sequencer instruction is 29 bits. */

   /* Set at idle_loop0 because the sequencer will jump to it */
   /* while waiting for any SCB to execute. */
   SCSI_hSETBREAKPOINT(hhcb, (SCSI_UEXACT8)ENTRY_IDLE_LOOP);

   /* Restore HCNTRL if necessary */
   if (!(hcntrl & SCSI_PAUSEACK))
   {
      SCSI_hWRITE_HCNTRL(hhcb, scsiRegister, hcntrl);
   }
}
#endif   /* SCSI_STANDARD_ADVANCED_MODE */

/*********************************************************************
*
*  SCSIhSwappingAdvSetupBDR
*
*     Setup Bus Device Reset for Swapping Advanced mode sequencer.
*
*  Return Value:  none
*                  
*  Parameters:    host task set handle
*                 hiob
*
*  Remarks:       Setup Bus Device Reset directly in SCB ram.
*                 Enqueue SCB to the head of excecution queue.
*                  
*********************************************************************/
#if SCSI_SWAPPING_ADVANCED_MODE
void SCSIhSwappingAdvSetupBDR (SCSI_HHCB SCSI_HPTR hhcb,
                               SCSI_HIOB SCSI_IPTR hiob)
{
   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_UNIT_CONTROL SCSI_IPTR targetUnit = SCSI_TARGET_UNIT(hiob);
   SCSI_SCB_SWAPPING_ADVANCED SCSI_HPTR scbWAdv;
   SCSI_UEXACT8 hcntrl;

   /* Save the HCNTRL value */
   hcntrl = OSD_INEXACT8(SCSI_AICREG(SCSI_HCNTRL));

   /* Pause the chip if necessary */
   if (!(hcntrl & SCSI_PAUSEACK))
   {
      SCSIhPauseAndWait(hhcb);
   }

   scbWAdv = (SCSI_SCB_SWAPPING_ADVANCED SCSI_HPTR) hiob->scbDescriptor->scbBuffer.virtualAddress;

   /* Clear swapping SCB buffer */
   OSDmemset(scbWAdv, 0, sizeof(SCSI_SCB_SWAPPING_ADVANCED));

   /* Setup special function - Bit 3 of the General Flags Byte */
   SCSI_PUT_LITTLE_ENDIAN8(hhcb, scbWAdv, OSDoffsetof(SCSI_SCB_SWAPPING_ADVANCED,
            scontrol), SCSI_WADV_SPECFUN);

   /* Setup target id */
   SCSI_PUT_LITTLE_ENDIAN8(hhcb, scbWAdv, OSDoffsetof(SCSI_SCB_SWAPPING_ADVANCED,
            starget), targetUnit->scsiID);

   /* Setup target lun */
   SCSI_PUT_LITTLE_ENDIAN8(hhcb, scbWAdv, OSDoffsetof(SCSI_SCB_SWAPPING_ADVANCED,
            slun), targetUnit->lunID);

   /* Setup no data length */
   SCSI_PUT_LITTLE_ENDIAN8(hhcb, scbWAdv, OSDoffsetof(SCSI_SCB_SWAPPING_ADVANCED,
            scdb_length), SCSI_WADV_NODATA);

   /* Setup special opcode - message to target */
   SCSI_PUT_LITTLE_ENDIAN8(hhcb, scbWAdv, OSDoffsetof(SCSI_SCB_SWAPPING_ADVANCED,
            SCSI_WADV_special_opcode), SCSI_WADV_MSGTOTARG);

   /* Setup special info - Bus Device Reset message */
   SCSI_PUT_LITTLE_ENDIAN8(hhcb, scbWAdv, OSDoffsetof(SCSI_SCB_SWAPPING_ADVANCED,
            SCSI_WADV_special_info), SCSI_MSG0C);
   
   /* Setup the next_SCB with the next scb number */
   SCSI_PUT_LITTLE_ENDIAN8(hhcb, scbWAdv,
       OSDoffsetof(SCSI_SCB_SWAPPING_ADVANCED,next_SCB),
         hiob->scbDescriptor->queueNext->scbNumber);

   /* Put BDR HIOB into active pointer array */
   SCSI_ACTPTR[hiob->scbNumber] = hiob;

   /* Flush the cache */
   SCSI_FLUSH_CACHE(scbWAdv, sizeof(SCSI_SCB_SWAPPING_ADVANCED));

   /* Enqueue BDR HIOB at head of new queue. */
   /* To avoid CHIM and Sequencer might access to the same location */
   /* in the new queue, we have to stop the sequencer */
   /* at the idle loop before enqueue the HIOB. */

   /* Set BDR HIOB state needs to be enqueued */
   hiob->SCSI_IP_mgrStat = SCSI_MGR_ENQUE_BDR;

   /* set non expander break */
   hhcb->SCSI_HP_nonExpBreak = 1;

   /* Set the breakpoint address at sequencer idle loop entry. */
   SCSI_hSETBREAKPOINT(hhcb, (SCSI_UEXACT8)ENTRY_IDLE_LOOP);

   /* Restore HCNTRL if necessary */
   if (!(hcntrl & SCSI_PAUSEACK))
   {
      SCSI_hWRITE_HCNTRL(hhcb, scsiRegister, hcntrl);
   }
}
#endif   /* SCSI_SWAPPING_ADVANCED_MODE */

/*********************************************************************
*
*  SCSIhStandard160mSetupBDR
*
*     Setup Bus Device Reset for Standard mode sequencer.
*
*  Return Value:  none
*
*  Parameters:    hhcb
*                 hiob
*
*  Remarks:       Setup Bus Device Reset directly in SCB ram.
*                 Enqueue SCB to the head of excecution queue.
*
*********************************************************************/
#if SCSI_STANDARD_160M_MODE
void SCSIhStandard160mSetupBDR (SCSI_HHCB SCSI_HPTR hhcb,
                                SCSI_HIOB SCSI_IPTR hiob)
{
   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_UNIT_CONTROL SCSI_UPTR targetUnit = SCSI_TARGET_UNIT(hiob);
   SCSI_UEXACT8 scbptr;
   SCSI_UEXACT8 hcntrl;
   SCSI_INT i;

   SCSI_UEXACT8 starget = 0;

   /* The sequencer will put the HA scsi ID into the hi nibble of starget */
   /* after the SCB is in the SCB RAM.  Since this routine will PIO       */
   /* the scb directly to the RAM, we must do the same. */
   starget = hhcb->hostScsiID;
   starget <<= 4;
   starget |= (SCSI_UEXACT8)targetUnit->scsiID;

   /* Save the HCNTRL value */
   hcntrl = OSD_INEXACT8(SCSI_AICREG(SCSI_HCNTRL));

   /* Pause the chip if necessary */
   if (!(hcntrl & SCSI_PAUSEACK))
   {
      SCSIhPauseAndWait(hhcb);
   }

   scbptr = OSD_INEXACT8(SCSI_AICREG(SCSI_SCBPTR));   /* Save SCBPTR */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCBPTR), hiob->scbNumber);
   
   /* Clear SCB ram */
   for (i = 0; i < 40; i++)
   {
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+i), 0);
   }

   /* Setup special function - Bit 3 of the SCB control flag */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
       OSDoffsetof(SCSI_SCB_STANDARD_160M, scontrol)), SCSI_S160M_SPECFUN);

   /* Setup target id/lun */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
       OSDoffsetof(SCSI_SCB_STANDARD_160M, starget)), (SCSI_UEXACT8)starget);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
       OSDoffsetof(SCSI_SCB_STANDARD_160M, slun)),
       (SCSI_UEXACT8)targetUnit->lunID);

   /* Setup no data length */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
       OSDoffsetof(SCSI_SCB_STANDARD_160M, scdb_length)), SCSI_S160M_NODATA);

   /* Setup special opcode - message to target  */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
       OSDoffsetof(SCSI_SCB_STANDARD_160M, SCSI_S160M_special_opcode)),
       SCSI_S160M_MSGTOTARG);

   /* Setup special info - Bus Device Reset message */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
       OSDoffsetof(SCSI_SCB_STANDARD_160M, SCSI_S160M_special_info)),
       SCSI_MSG0C);

   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCBPTR), scbptr);   /* Restore SCBPTR */

   /* Put BDR HIOB into active pointer array */
   SCSI_ACTPTR[hiob->scbNumber] = hiob;

   /* Enqueue BDR HIOB at head of execution queue. */
   /* To avoid CHIM and Sequencer might access to the same location */
   /* in the execution queue, we have to stop the sequencer at */
   /* the idle loop before enqueue the HIOB. */

   /* Set BDR HIOB state needs to be enqueued */
   hiob->SCSI_IP_mgrStat = SCSI_MGR_ENQUE_BDR;

   /* set non expanser break */
   hhcb->SCSI_HP_nonExpBreak = 1;

   /* Set the breakpoint address at sequencer idle loop. Since the sequencer */
   /* program counter always be one instruction ahead, we need to add 4 to */
   /* the actual idle loop value.  One sequencer instruction is 29 bits. */

   /* Set at idle_loop0 because the sequencer will jump to it */
   /* while waiting for any SCB to execute. */
   SCSI_hSETBREAKPOINT(hhcb, (SCSI_UEXACT8)ENTRY_IDLE_LOOP);

   /* Restore HCNTRL if necessary */
   if (!(hcntrl & SCSI_PAUSEACK))
   {
      SCSI_hWRITE_HCNTRL(hhcb, scsiRegister, hcntrl);
   }
}
#endif   /* SCSI_STANDARD_160M_MODE */

/*********************************************************************
*
*  SCSIhSwapping160mSetupBDR
*
*     Setup Bus Device Reset for Swapping 160m mode sequencer.
*
*  Return Value:  none
*                  
*  Parameters:    host task set handle
*                 hiob
*
*  Remarks:       Setup Bus Device Reset directly in SCB ram.
*                 Enqueue SCB to the head of excecution queue.
*                  
*********************************************************************/
#if SCSI_SWAPPING_160M_MODE
void SCSIhSwapping160mSetupBDR (SCSI_HHCB SCSI_HPTR hhcb,
                                SCSI_HIOB SCSI_IPTR hiob)
{
   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_UNIT_CONTROL SCSI_IPTR targetUnit = SCSI_TARGET_UNIT(hiob);
   SCSI_SCB_SWAPPING_160M SCSI_HPTR scbW160m;
   SCSI_UEXACT8 hcntrl;

   /* Save the HCNTRL value */
   hcntrl = OSD_INEXACT8(SCSI_AICREG(SCSI_HCNTRL));

   /* Pause the chip if necessary */
   if (!(hcntrl & SCSI_PAUSEACK))
   {
      SCSIhPauseAndWait(hhcb);
   }

   scbW160m = (SCSI_SCB_SWAPPING_160M SCSI_HPTR) hiob->scbDescriptor->scbBuffer.virtualAddress;

   /* Clear swapping SCB buffer */
   OSDmemset(scbW160m, 0, sizeof(SCSI_SCB_SWAPPING_160M));

   /* Setup special function - Bit 3 of the General Flags Byte */
   SCSI_PUT_LITTLE_ENDIAN8(hhcb, scbW160m,
        OSDoffsetof(SCSI_SCB_SWAPPING_160M, scontrol), SCSI_W160M_SPECFUN);

   /* Setup target id */
   SCSI_PUT_LITTLE_ENDIAN8(hhcb, scbW160m,
        OSDoffsetof(SCSI_SCB_SWAPPING_160M, starget), targetUnit->scsiID);

   /* Setup target lun */
   SCSI_PUT_LITTLE_ENDIAN8(hhcb, scbW160m,
        OSDoffsetof(SCSI_SCB_SWAPPING_160M, slun), targetUnit->lunID);

   /* Setup no data length */
   SCSI_PUT_LITTLE_ENDIAN8(hhcb, scbW160m,
        OSDoffsetof(SCSI_SCB_SWAPPING_160M, scdb_length), SCSI_W160M_NODATA);

   /* Setup special opcode - message to target */
   SCSI_PUT_LITTLE_ENDIAN8(hhcb, scbW160m,
        OSDoffsetof(SCSI_SCB_SWAPPING_160M, SCSI_W160M_special_opcode),
        SCSI_W160M_MSGTOTARG);

   /* Setup special info - Bus Device Reset message */
   SCSI_PUT_LITTLE_ENDIAN8(hhcb, scbW160m,
        OSDoffsetof(SCSI_SCB_SWAPPING_160M, SCSI_W160M_special_info),
        SCSI_MSG0C);
   
   /* Setup the next_SCB with the next scb number */
   SCSI_PUT_LITTLE_ENDIAN8(hhcb, scbW160m,
        OSDoffsetof(SCSI_SCB_SWAPPING_160M, next_SCB),
        hiob->scbDescriptor->queueNext->scbNumber);

   /* Put BDR HIOB into active pointer array */
   SCSI_ACTPTR[hiob->scbNumber] = hiob;

   /* Flush the cache */
   SCSI_FLUSH_CACHE(scbW160m, sizeof(SCSI_SCB_SWAPPING_160M));

   /* Enqueue BDR HIOB at head of new queue. */
   /* To avoid CHIM and Sequencer might access to the same location */
   /* in the new queue, we have to stop the sequencer */
   /* at the idle loop before enqueue the HIOB. */

   /* Set BDR HIOB state needs to be enqueued */
   hiob->SCSI_IP_mgrStat = SCSI_MGR_ENQUE_BDR;

   /* set non expander break */
   hhcb->SCSI_HP_nonExpBreak = 1;

   /* Set the breakpoint address at sequencer idle loop entry. */
   SCSI_hSETBREAKPOINT(hhcb, (SCSI_UEXACT8)ENTRY_IDLE_LOOP);

   /* Restore HCNTRL if necessary */
   if (!(hcntrl & SCSI_PAUSEACK))
   {
      SCSI_hWRITE_HCNTRL(hhcb, scsiRegister, hcntrl);
   }
}
#endif   /* SCSI_SWAPPING_160M_MODE */

/*********************************************************************
*
*  SCSIhStandard64StackBugFix
*
*     Workaround for two stack problem after reset during data out phase
*
*  Return Value:  none
*
*  Parameters:    hhcb
*
*  Remarks:       !!! The sequencer should be paused !!!
*                 !!! at entering this routine.      !!!
*                  
*********************************************************************/
#if SCSI_STANDARD64_MODE
void SCSIhStandard64StackBugFix (SCSI_HHCB SCSI_HPTR hhcb)
{
   SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_UINT16 i;
   SCSI_UEXACT8 stackcontent;
   SCSI_UEXACT8 stackcontent0,stackcontent1,stackcontent2,stackcontent3;
   SCSI_UEXACT8 stackcontent4,stackcontent5,stackcontent6,stackcontent7;
   SCSI_BOOLEAN stackwrong = SCSI_FALSE;

   /* There is a problem when we get 3rd party reset during data out phase  */
   /* for Excalibur in standard 64 mode.  We will get sequencer stack       */
   /* pointer misaligned and cause sequencer stack corrupted later on when  */
   /* we try to restart sequencer.  The cause is that we get 3rd party      */
   /* reset in the middle when sequencer try to move return address from    */
   /* stack to scratch or from scratch to stack.  Since return address is   */
   /* two bytes long and when we get 3rd party reset just after sequencer   */
   /* finish transferring one byte and sequencer get paused we will get     */
   /* stack pointer misaligned.                                             */
   /* The following will work around this problem by checking sequencer     */
   /* stack content to see if we get sequencer stack pointer misaligned and */
   /* then read one more byte from sequencer stack which will make          */
   /* sequencer stack pointer aligned.                                      */
   if (hhcb->firmwareMode == SCSI_FMODE_STANDARD64)
   {
      for (i=0;i<8;i++)
      {
         stackcontent = OSD_INEXACT8(SCSI_AICREG(SCSI_STACK));
         if (i % 2 == 1)
         {
            if (stackcontent >= 0x02)
            {
               stackwrong = SCSI_TRUE;
            }
            switch (i)
            {
               case 1:
                  stackcontent1 = stackcontent;
                  break;
               case 3:
                  stackcontent3 = stackcontent;
                  break;
               case 5:
                  stackcontent5 = stackcontent;
                  break;
               case 7:
                  stackcontent7 = stackcontent;
                  break;
               default:
                  break;
            }
         }
         else if (i % 2 == 0)
         {
            switch (i)
            {
               case 0:
                  stackcontent0 = stackcontent;
                  break;
               case 2:
                  stackcontent2 = stackcontent;
                  break;
               case 4:
                  stackcontent4 = stackcontent;
                  break;
               case 6:
                  stackcontent6 = stackcontent;
                  break;
               default:
                  break;
            }
         }
      }

      if (stackwrong == SCSI_TRUE)
      {
         OSD_INEXACT8(SCSI_AICREG(SCSI_STACK));
      }
   
      /* There is a problem when we get 3rd party reset during data out     */
      /* phase for Excalibur in standard 64 mode.  We will get sequencer    */
      /* stack pointer lock up and can't write to SEQADDR0 before we write  */
      /* a value to SEQADDR1 register.                                      */
      /* The following will work around this problem by checking to see     */
      /* if we get stack pointer lock up and then write 0 to SEQADDR1       */
      /* register.  This write 0 to SEQADDR1 register is a dummy write.     */
      if ((stackcontent0 == stackcontent2) &&
          (stackcontent0 == stackcontent4) &&
          (stackcontent0 == stackcontent6) &&
          (stackcontent1 == stackcontent3) &&
          (stackcontent1 == stackcontent5) &&
          (stackcontent1 == stackcontent7))
      {
         OSD_OUTEXACT8(SCSI_AICREG(SCSI_SEQADDR1), 0x00);
      }
   }

}
#endif   /* SCSI_STANDARD64_MODE */

/*********************************************************************
*
*  SCSIhSwapping32FreezeHWQueue
*
*     Freeze the hardware queue for the given target id (which is in the
*     given hiob), by removing all the SCBs for this target from the relevant
*     hardware queues.
*
*  Return Value:  None
*
*  Parameters:    hhcb
*                 Failed hiob
*
*  Remarks:       !!! The sequencer is assumed to be paused and also !!!
*                 !!! assumed to be sitting in the error path        !!!
*
*                 This routine pulls out all the SCBs in the hardware layer
*                 and sends them back to the RSM layer with a special status
*                 (SCSI_SCB_FROZEN) so that the RSM layer can queue them back
*                 to its device queue.
*
*********************************************************************/
#if SCSI_SWAPPING32_MODE
void SCSIhSwapping32FreezeHWQueue (SCSI_HHCB SCSI_HPTR hhcb,
                                   SCSI_HIOB SCSI_IPTR hiob)
{
   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_UNIT_CONTROL SCSI_IPTR targetUnit = SCSI_TARGET_UNIT(hiob);
   SCSI_UEXACT8 qNewTail;
   SCSI_UEXACT8 qNextScb;
   SCSI_UEXACT8 scbCurr;
   SCSI_UEXACT8 scbPrev;
   SCSI_UEXACT8 scbNext;
   SCSI_HIOB SCSI_IPTR hiobCurr;
   SCSI_HIOB SCSI_IPTR hiobPrev;
   SCSI_UEXACT8 scsiId;
   SCSI_UEXACT8 scsiseq;

   /* Generate asynchronous event to the RSM layer to notify that the */
   /* hardware layer is going to start freezing of its queues for the */
   /* failed hiob.                                                    */
   SCSI_ASYNC_EVENT_COMMAND(hhcb,hiob,SCSI_AE_FREEZEONERROR_START);

   /* Get the target id */
   scsiId = (SCSI_UEXACT8) targetUnit->scsiID;

   /* The following logic posts back the SCBs for this target in a          */
   /* chronological order, starting from the SCB which is about to get fired*/
   /* by the sequencer.                                                     */

   /* If a selection is in progress for this target, disable the selection  */
   /* process and post back this SCB to the upper layer.                    */
   if (((scsiseq = (OSD_INEXACT8(SCSI_AICREG(SCSI_SCSISEQ)))) & SCSI_ENSELO))
   {
       hiobCurr = SCSI_ACTPTR[OSD_INEXACT8(SCSI_AICREG(SCSI_W32_WAITING_SCB))];
       if ((SCSI_UEXACT8)(SCSI_TARGET_UNIT(hiobCurr)->scsiID) == scsiId)
       {
          /* Disable any outstanding selection */
          OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSISEQ),
             scsiseq & ~(SCSI_TEMODEO+SCSI_ENSELO+SCSI_ENAUTOATNO+SCSI_ENAUTOATNI));
          /* To flush the cache */
          scsiseq = OSD_INEXACT8(SCSI_AICREG(SCSI_SCSISEQ));

          /* free the associated entry in active pointer array */
          SCSI_ACTPTR[hiobCurr->scbNumber] = SCSI_NULL_HIOB;

          /* Put proper status in HIOB, so that rsm layer will queue it back */
          /* into the device queue.                                          */
          hiobCurr->stat = SCSI_SCB_FROZEN;

          /* Post back this HIOB */
          if (hiobCurr->cmd == SCSI_CMD_INITIATE_TASK)
          {
             SCSI_COMPLETE_HIOB(hiobCurr);
          }
          else
          {
             SCSI_COMPLETE_SPECIAL_HIOB(hiobCurr);
          }
       }
   }

   /* Post back all the SCBs for the failed target from the New Queue.     */

   /* Get q_new_tail value */
   qNewTail = SCSI_hGETQNEWTAIL(hhcb);

   /* Get q_next_scb value from scratch ram */
   qNextScb = OSD_INEXACT8(SCSI_AICREG(SCSI_hQ_NEXT_SCB(hhcb)));
   scbCurr = scbPrev = qNextScb;

   /* Check if any SCB for this target is in New queue */
   while (qNewTail != hhcb->SCSI_HP_qNewHead)
   {
      /* Search for matching target id */
      hiobCurr = SCSI_ACTPTR[scbCurr];
      if ((SCSI_UEXACT8)(SCSI_TARGET_UNIT(hiobCurr)->scsiID) == scsiId)
      {
         /* Decrement and update qNewHead into scratch ram */
         (SCSI_UEXACT8)hhcb->SCSI_HP_qNewHead--;
         OSD_OUTEXACT8(SCSI_AICREG(SCSI_hQ_NEW_HEAD(hhcb)), hhcb->SCSI_HP_qNewHead);

         /* Remove aborted HIOB from New queue and mend the queue */

         scbNext = SCSI_hOBTAIN_NEXT_SCB_NUM(hhcb,hiobCurr);

         if (scbCurr == qNextScb)
         {
            qNextScb = scbPrev = scbNext;
            OSD_OUTEXACT8(SCSI_AICREG(SCSI_hQ_NEXT_SCB(hhcb)),qNextScb);
         }
         else
         {
            /* Obtain previous HIOB in Active pointer array */ 
            hiobPrev = SCSI_ACTPTR[scbPrev];
            SCSI_hUPDATE_NEXT_SCB_NUM(hhcb,hiobPrev,scbNext);
         }

         /* free the associated entry in active pointer array */
         SCSI_ACTPTR[hiobCurr->scbNumber] = SCSI_NULL_HIOB;

         /* Put proper status in HIOB, so that rsm layer will queue it back */
         /* into the device queue.                                          */
         hiobCurr->stat = SCSI_SCB_FROZEN;

         /* Post back this HIOB */
         if (hiobCurr->cmd == SCSI_CMD_INITIATE_TASK)
         {
            SCSI_COMPLETE_HIOB(hiobCurr);
         }
         else
         {
            SCSI_COMPLETE_SPECIAL_HIOB(hiobCurr);
         }

         scbCurr = scbNext;
      }
      else
      {
         /* Need to get next scb from current scb */
         scbPrev = scbCurr;
      
         /* Obtain current HIOB in Active pointer array */ 
         scbCurr = SCSI_hOBTAIN_NEXT_SCB_NUM(hhcb,hiobCurr);

         (SCSI_UEXACT8)qNewTail++;
      }
   }

   /* Generate asynchronous event to the RSM layer to notify that the      */
   /* hardware layer has finished freezing its queues for the failed hiob. */
   SCSI_ASYNC_EVENT_COMMAND(hhcb,hiob,SCSI_AE_FREEZEONERROR_END);
}
#endif   /* SCSI_SWAPPING32_MODE */

/*********************************************************************
*
*  SCSIhStandardAdvFreezeHWQueue
*
*     Freeze the hardware queue for the given target id (which is in the
*     given hiob), by removing all the SCBs for this target from the relevant
*     hardware queues.
*
*  Return Value:  None
*
*  Parameters:    hhcb
*                 Failed hiob
*
*  Remarks:       !!! The sequencer is assumed to be paused and also !!!
*                 !!! assumed to be sitting in the error path        !!!
*
*                 This routine pulls out all the SCBs in the hardware layer
*                 and sends them back to the RSM layer with a special status
*                 (SCSI_SCB_FROZEN) so that the RSM layer can queue them back
*                 to its device queue.
*
*********************************************************************/
#if SCSI_STANDARD_ADVANCED_MODE
void SCSIhStandardAdvFreezeHWQueue (SCSI_HHCB SCSI_HPTR hhcb,
                                    SCSI_HIOB SCSI_IPTR hiob)
{
   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_UNIT_CONTROL SCSI_IPTR targetUnit = SCSI_TARGET_UNIT(hiob);
   SCSI_UEXACT8 qNextScb;
   SCSI_UEXACT8 scbCurr;
   SCSI_UEXACT8 scbPrev;
   SCSI_UEXACT8 scbNext;
   SCSI_HIOB SCSI_IPTR hiobCurr;
   SCSI_HIOB SCSI_IPTR hiobPrev;
   SCSI_UEXACT8 scsiId;
   SCSI_UEXACT8 scsiseq;
   SCSI_UEXACT8 scbptr;
   SCSI_BUS_ADDRESS scbNextbusAddress;

   /* Generate asynchronous event to the RSM layer to notify that the */
   /* hardware layer is going to start freezing of its queues for the */
   /* failed hiob.                                                    */
   SCSI_ASYNC_EVENT_COMMAND(hhcb,hiob,SCSI_AE_FREEZEONERROR_START);

   /* Get the target id */
   scsiId = (SCSI_UEXACT8) targetUnit->scsiID;

   /* The following logic posts back the SCBs for this target in a          */
   /* chronological order, starting from the SCB which is about to get fired*/
   /* by the sequencer.                                                     */

   /* If a selection is in progress for this target, disable the selection  */
   /* process and post back this SCB to the upper layer.                    */
   if (((scsiseq = (OSD_INEXACT8(SCSI_AICREG(SCSI_SCSISEQ)))) & SCSI_ENSELO))
   {
       hiobCurr = SCSI_ACTPTR[OSD_INEXACT8(SCSI_AICREG(SCSI_SADV_WAITING_SCB))];
       if ((SCSI_UEXACT8)(SCSI_TARGET_UNIT(hiobCurr)->scsiID) == scsiId)
       {
          /* Disable any outstanding selection */
          OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSISEQ),
             scsiseq & ~(SCSI_TEMODEO+SCSI_ENSELO+SCSI_ENAUTOATNO+SCSI_ENAUTOATNI));
          /* To flush the cache */
          scsiseq = OSD_INEXACT8(SCSI_AICREG(SCSI_SCSISEQ));

          /* free the associated entry in active pointer array */
          SCSI_ACTPTR[hiobCurr->scbNumber] = SCSI_NULL_HIOB;

          /* Put proper status in HIOB, so that rsm layer will queue it back */
          /* into the device queue.                                          */
          hiobCurr->stat = SCSI_SCB_FROZEN;

          /* Post back this HIOB */
          if (hiobCurr->cmd == SCSI_CMD_INITIATE_TASK)
          {
             SCSI_COMPLETE_HIOB(hiobCurr);
          }
          else
          {
             SCSI_COMPLETE_SPECIAL_HIOB(hiobCurr);
          }
       }
   }

   /* Post back all the SCBs for the failed target from the Exe Queue.     */

   scbptr = OSD_INEXACT8(SCSI_AICREG(SCSI_SCBPTR));   /* Save SCBPTR */

   /* Check if aborting HIOB is in Execution queue */
   scbPrev = scbCurr = OSD_INEXACT8(SCSI_AICREG(SCSI_SADV_Q_EXE_HEAD));

   while (scbCurr != SCSI_NULL_SCB)
   {
      hiobCurr = SCSI_ACTPTR[scbCurr];

      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCBPTR), scbCurr);

      /* Search for matching target id */
      if ((SCSI_UEXACT8)(SCSI_TARGET_UNIT(hiobCurr)->scsiID) == scsiId)
      {
         /* Remove the SCB from execution queue */
         scbNext = OSD_INEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
                    OSDoffsetof(SCSI_SCB_STANDARD_ADVANCED, SCSI_SADV_q_next)));
                        
         /* Mending the execution queue */
         if (scbCurr == scbPrev)
         {
            scbPrev = scbNext;
            /* Removed HIOB at the queue head */
            OSD_OUTEXACT8(SCSI_AICREG(SCSI_SADV_Q_EXE_HEAD), scbNext);
         }
         else
         {
            OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCBPTR), scbPrev);
            OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
              OSDoffsetof(SCSI_SCB_STANDARD_ADVANCED,SCSI_SADV_q_next)),scbNext);

            if (scbCurr == OSD_INEXACT8(SCSI_AICREG(SCSI_SADV_Q_EXE_TAIL)))
            {
               /* Removed HIOB at the queue tail */
               OSD_OUTEXACT8(SCSI_AICREG(SCSI_SADV_Q_EXE_TAIL), scbPrev);
            }
         }

         /* free the associated entry in active pointer array */
         SCSI_ACTPTR[hiobCurr->scbNumber] = SCSI_NULL_HIOB;

         /* Put proper status in HIOB, so that rsm layer will queue it back */
         /* into the device queue.                                          */
         hiobCurr->stat = SCSI_SCB_FROZEN;

         /* Post back this HIOB */
         if (hiobCurr->cmd == SCSI_CMD_INITIATE_TASK)
         {
            SCSI_COMPLETE_HIOB(hiobCurr);
         }
         else
         {
            SCSI_COMPLETE_SPECIAL_HIOB(hiobCurr);
         }

         scbCurr = scbNext;
      }
      else
      {
         /* Keep previous SCB */
         scbPrev = scbCurr;
      
         /* Advance to next SCB in Execution queue */
         scbCurr = OSD_INEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
                   OSDoffsetof(SCSI_SCB_STANDARD_ADVANCED, SCSI_SADV_q_next)));
      }
   }

   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCBPTR), scbptr);   /* Restore SCBPTR */

   /* If an SCB for this target is getting DMAed, just disable the DMA        */
   /* (this hiob need not be posted back, as it will be done in the next step */
   if ((OSD_INEXACT8(SCSI_AICREG(SCSI_CCSCBCTL)) &
       (SCSI_CCSCBEN+SCSI_CCSCBDIR+SCSI_CCARREN)) ==
       (SCSI_CCSCBEN+SCSI_CCSCBDIR+SCSI_CCARREN))
   {
       /* Take the SCB from the head of the new queue and check if it is for  */
       /* the failed target.                                                  */
       hiobCurr = SCSI_ACTPTR[OSD_INEXACT8(SCSI_AICREG(SCSI_hQ_NEXT_SCB(hhcb)))];
       if ((SCSI_UEXACT8)(SCSI_TARGET_UNIT(hiobCurr)->scsiID) == scsiId)
       {
          /* Disable DMA */
          OSD_OUTEXACT8(SCSI_AICREG(SCSI_CCSCBCTL),0);

          /* Since the sequencer has already incremented the SNSCB_QOFF, it */
          /* should be decremented as it will be adjusted during posting    */
          /* back in the logic below. */
          OSD_OUTEXACT8(SCSI_AICREG(SCSI_SNSCB_QOFF),
                        OSD_INEXACT8(SCSI_AICREG(SCSI_SNSCB_QOFF))-1);
       }

       else
       {
          /* Obtain the next SCB in the new queue and check if it is for the */
          /* failed target.  We have to disable DMA even in this case because*/
          /* after the DMA the q_new_pointer and array_site_next variables   */
          /* will be modified by the sequencer to point to this SCB; but this*/
          /* SCB will be removed while we are searching the new queue down   */
          /* in the logic. */
          if ((scbCurr = SCSI_hOBTAIN_NEXT_SCB_NUM(hhcb,hiobCurr)) !=
              SCSI_NULL_SCB)
          {
             hiobCurr = SCSI_ACTPTR[scbCurr];
             if ((hiobCurr != SCSI_NULL_HIOB) &&
                 ((SCSI_UEXACT8)(SCSI_TARGET_UNIT(hiobCurr)->scsiID) == scsiId))
             {
                /* Disable DMA */
                OSD_OUTEXACT8(SCSI_AICREG(SCSI_CCSCBCTL),0);

               /* Since the sequencer has already incremented the SNSCB_QOFF, */
               /* it should be decremented as it will be adjusted during      */
               /* posting back in the logic below. */
               OSD_OUTEXACT8(SCSI_AICREG(SCSI_SNSCB_QOFF),
                             OSD_INEXACT8(SCSI_AICREG(SCSI_SNSCB_QOFF))-1);
             }
          }
       }
   }

   /* Post back all the SCBs for the failed target from the New Queue.     */

   /* Get q_next_scb value from scratch ram */
   qNextScb = OSD_INEXACT8(SCSI_AICREG(SCSI_hQ_NEXT_SCB(hhcb)));
   scbCurr = scbPrev = qNextScb;

   /* Check if any SCB for this target is in New queue */
   while ((hiobCurr = SCSI_ACTPTR[scbCurr]) != SCSI_NULL_HIOB)
   {
      /* Search for matching target id */
      if ((SCSI_UEXACT8)(SCSI_TARGET_UNIT(hiobCurr)->scsiID) == scsiId)
      {
         /* Decrement and update qNewHead into scratch ram */
         (SCSI_UEXACT8)hhcb->SCSI_HP_qNewHead--;
         OSD_OUTEXACT8(SCSI_AICREG(SCSI_hQ_NEW_HEAD(hhcb)),
                       hhcb->SCSI_HP_qNewHead);

         /* Remove aborted HIOB from New queue and mend the queue */

         scbNext = SCSI_hOBTAIN_NEXT_SCB_NUM(hhcb,hiobCurr);
         scbNextbusAddress = SCSI_hOBTAIN_NEXT_SCB_ADDRESS(hhcb,hiobCurr);

         if (scbCurr == qNextScb)
         {
            qNextScb = scbPrev = scbNext;
            OSD_OUTEXACT8(SCSI_AICREG(SCSI_hQ_NEXT_SCB(hhcb)),qNextScb);
            SCSI_hSETADDRESSSCRATCH(hhcb,SCSI_SADV_Q_NEW_POINTER,
               hiobCurr->scbDescriptor->queueNext->scbBuffer.virtualAddress);
         }
         else
         {
            /* Obtain previous HIOB in Active pointer array */ 
            hiobPrev = SCSI_ACTPTR[scbPrev];
            SCSI_hUPDATE_NEXT_SCB_NUM(hhcb,hiobPrev,scbNext);
            SCSI_hUPDATE_NEXT_SCB_ADDRESS(hhcb,hiobPrev,scbNextbusAddress);
         }

         /* free the associated entry in active pointer array */
         SCSI_ACTPTR[hiobCurr->scbNumber] = SCSI_NULL_HIOB;

         /* Put proper status in HIOB, so that rsm layer will queue it back */
         /* into the device queue.                                          */
         hiobCurr->stat = SCSI_SCB_FROZEN;

         /* Post back this HIOB */
         if (hiobCurr->cmd == SCSI_CMD_INITIATE_TASK)
         {
            SCSI_COMPLETE_HIOB(hiobCurr);
         }
         else
         {
            SCSI_COMPLETE_SPECIAL_HIOB(hiobCurr);
         }

         scbCurr = scbNext;
      }
      else
      {
         /* Need to get next scb from current scb */
         scbPrev = scbCurr;
      
         /* Obtain current HIOB in Active pointer array */ 
         scbCurr = SCSI_hOBTAIN_NEXT_SCB_NUM(hhcb,hiobCurr);
      }
   }

   /* Generate asynchronous event to the RSM layer to notify that the      */
   /* hardware layer has finished freezing its queues for the failed hiob. */
   SCSI_ASYNC_EVENT_COMMAND(hhcb,hiob,SCSI_AE_FREEZEONERROR_END);
}
#endif   /* SCSI_STANDARD_ADVANCED_MODE */

/*********************************************************************
*
*  SCSIhSwappingAdvFreezeHWQueue
*
*     Freeze the hardware queue for the given target id (which is in the
*     given hiob), by removing all the SCBs for this target from the relevant
*     hardware queues.
*
*  Return Value:  None
*
*  Parameters:    hhcb
*                 Failed hiob
*
*  Remarks:       !!! The sequencer is assumed to be paused and also !!!
*                 !!! assumed to be sitting in the error path        !!!
*
*                 This routine pulls out all the SCBs in the hardware layer
*                 and sends them back to the RSM layer with a special status
*                 (SCSI_SCB_FROZEN) so that the RSM layer can queue them back
*                 to its device queue.
*
*********************************************************************/
#if SCSI_SWAPPING_ADVANCED_MODE
void SCSIhSwappingAdvFreezeHWQueue (SCSI_HHCB SCSI_HPTR hhcb,
                                    SCSI_HIOB SCSI_IPTR hiob)
{
   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_UNIT_CONTROL SCSI_IPTR targetUnit = SCSI_TARGET_UNIT(hiob);
   SCSI_UEXACT8 qNewTail;
   SCSI_UEXACT8 qNextScb;
   SCSI_UEXACT8 scbCurr;
   SCSI_UEXACT8 scbPrev;
   SCSI_UEXACT8 scbNext;
   SCSI_HIOB SCSI_IPTR hiobCurr;
   SCSI_HIOB SCSI_IPTR hiobPrev;
   SCSI_UEXACT8 scsiId;
   SCSI_UEXACT8 scsiseq;

   /* Generate asynchronous event to the RSM layer to notify that the */
   /* hardware layer is going to start freezing of its queues for the */
   /* failed hiob.                                                    */
   SCSI_ASYNC_EVENT_COMMAND(hhcb,hiob,SCSI_AE_FREEZEONERROR_START);

   /* Get the target id */
   scsiId = (SCSI_UEXACT8) targetUnit->scsiID;

   /* The following logic posts back the SCBs for this target in a          */
   /* chronological order, starting from the SCB which is about to get fired*/
   /* by the sequencer.                                                     */

   /* If a selection is in progress for this target, disable the selection  */
   /* process and post back this SCB to the upper layer.                    */
   if (((scsiseq = (OSD_INEXACT8(SCSI_AICREG(SCSI_SCSISEQ)))) & SCSI_ENSELO))
   {
       hiobCurr = SCSI_ACTPTR[OSD_INEXACT8(SCSI_AICREG(SCSI_WADV_WAITING_SCB))];
       if ((SCSI_UEXACT8)(SCSI_TARGET_UNIT(hiobCurr)->scsiID) == scsiId)
       {
          /* Disable any outstanding selection */
          OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSISEQ),
             scsiseq & ~(SCSI_TEMODEO+SCSI_ENSELO+SCSI_ENAUTOATNO+SCSI_ENAUTOATNI));
          /* To flush the cache */
          scsiseq = OSD_INEXACT8(SCSI_AICREG(SCSI_SCSISEQ));

          /* free the associated entry in active pointer array */
          SCSI_ACTPTR[hiobCurr->scbNumber] = SCSI_NULL_HIOB;

          /* Put proper status in HIOB, so that rsm layer will queue it back */
          /* into the device queue.                                          */
          hiobCurr->stat = SCSI_SCB_FROZEN;

          /* Post back this HIOB */
          if (hiobCurr->cmd == SCSI_CMD_INITIATE_TASK)
          {
             SCSI_COMPLETE_HIOB(hiobCurr);
          }
          else
          {
             SCSI_COMPLETE_SPECIAL_HIOB(hiobCurr);
          }
       }
   }

   /* If an SCB for this target is getting DMAed, just disable the DMA        */
   /* (this hiob need not be posted back, as it will be done in the next step */
   if ((OSD_INEXACT8(SCSI_AICREG(SCSI_CCSCBCTL)) &
       (SCSI_CCSCBEN+SCSI_CCSCBDIR+SCSI_CCARREN)) ==
       (SCSI_CCSCBEN+SCSI_CCSCBDIR+SCSI_CCARREN))
   {
       /* Take the SCB from the head of the new queue and check if it is for  */
       /* the failed target.                                                  */
       hiobCurr = SCSI_ACTPTR[OSD_INEXACT8(SCSI_AICREG(SCSI_hQ_NEXT_SCB(hhcb)))];
       if ((SCSI_UEXACT8)(SCSI_TARGET_UNIT(hiobCurr)->scsiID) == scsiId)
       {
          /* Disable DMA */
          OSD_OUTEXACT8(SCSI_AICREG(SCSI_CCSCBCTL),0);
       }
   }

   /* Post back all the SCBs for the failed target from the New Queue.     */

   /* Get q_new_tail value */
   qNewTail = SCSI_hGETQNEWTAIL(hhcb);

   /* Get q_next_scb value from scratch ram */
   qNextScb = OSD_INEXACT8(SCSI_AICREG(SCSI_hQ_NEXT_SCB(hhcb)));
   scbCurr = scbPrev = qNextScb;

   /* Check if any SCB for this target is in New queue */
   while (qNewTail != hhcb->SCSI_HP_qNewHead)
   {
      /* Search for matching target id */
      hiobCurr = SCSI_ACTPTR[scbCurr];
      if ((SCSI_UEXACT8)(SCSI_TARGET_UNIT(hiobCurr)->scsiID) == scsiId)
      {
         /* Decrement and update qNewHead into scratch ram */
         (SCSI_UEXACT8)hhcb->SCSI_HP_qNewHead--;
         OSD_OUTEXACT8(SCSI_AICREG(SCSI_hQ_NEW_HEAD(hhcb)), hhcb->SCSI_HP_qNewHead);

         /* Remove aborted HIOB from New queue and mend the queue */

         scbNext = SCSI_hOBTAIN_NEXT_SCB_NUM(hhcb,hiobCurr);

         if (scbCurr == qNextScb)
         {
            qNextScb = scbPrev = scbNext;
            OSD_OUTEXACT8(SCSI_AICREG(SCSI_hQ_NEXT_SCB(hhcb)),qNextScb);
         }
         else
         {
            /* Obtain previous HIOB in Active pointer array */ 
            hiobPrev = SCSI_ACTPTR[scbPrev];
            SCSI_hUPDATE_NEXT_SCB_NUM(hhcb,hiobPrev,scbNext);
         }

         /* free the associated entry in active pointer array */
         SCSI_ACTPTR[hiobCurr->scbNumber] = SCSI_NULL_HIOB;

         /* Put proper status in HIOB, so that rsm layer will queue it back */
         /* into the device queue.                                          */
         hiobCurr->stat = SCSI_SCB_FROZEN;

         /* Post back this HIOB */
         if (hiobCurr->cmd == SCSI_CMD_INITIATE_TASK)
         {
            SCSI_COMPLETE_HIOB(hiobCurr);
         }
         else
         {
            SCSI_COMPLETE_SPECIAL_HIOB(hiobCurr);
         }

         scbCurr = scbNext;
      }
      else
      {
         /* Need to get next scb from current scb */
         scbPrev = scbCurr;
      
         /* Obtain current HIOB in Active pointer array */ 
         scbCurr = SCSI_hOBTAIN_NEXT_SCB_NUM(hhcb,hiobCurr);

         (SCSI_UEXACT8)qNewTail++;
      }
   }

   /* Generate asynchronous event to the RSM layer to notify that the      */
   /* hardware layer has finished freezing its queues for the failed hiob. */
   SCSI_ASYNC_EVENT_COMMAND(hhcb,hiob,SCSI_AE_FREEZEONERROR_END);
}
#endif   /* SCSI_SWAPPING_ADVANCED_MODE */

/*********************************************************************
*
*  SCSIhStandard160mFreezeHWQueue
*
*     Freeze the hardware queue for the given target id (which is in the
*     given hiob), by removing all the SCBs for this target from the relevant
*     hardware queues.
*
*  Return Value:  None
*
*  Parameters:    hhcb
*                 Failed hiob
*
*  Remarks:       !!! The sequencer is assumed to be paused and also !!!
*                 !!! assumed to be sitting in the error path        !!!
*
*                 This routine pulls out all the SCBs in the hardware layer
*                 and sends them back to the RSM layer with a special status
*                 (SCSI_SCB_FROZEN) so that the RSM layer can queue them back
*                 to its device queue.
*
*********************************************************************/
#if SCSI_STANDARD_160M_MODE
void SCSIhStandard160mFreezeHWQueue (SCSI_HHCB SCSI_HPTR hhcb,
                                     SCSI_HIOB SCSI_IPTR hiob)
{
   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_UNIT_CONTROL SCSI_IPTR targetUnit = SCSI_TARGET_UNIT(hiob);
   SCSI_UEXACT8 qNextScb;
   SCSI_UEXACT8 scbCurr;
   SCSI_UEXACT8 scbPrev;
   SCSI_UEXACT8 scbNext;
   SCSI_HIOB SCSI_IPTR hiobCurr;
   SCSI_HIOB SCSI_IPTR hiobPrev;
   SCSI_UEXACT8 scsiId;
   SCSI_UEXACT8 scsiseq;
   SCSI_UEXACT8 scbptr;
   SCSI_BUS_ADDRESS scbNextbusAddress;

   /* Generate asynchronous event to the RSM layer to notify that the */
   /* hardware layer is going to start freezing of its queues for the */
   /* failed hiob.                                                    */
   SCSI_ASYNC_EVENT_COMMAND(hhcb,hiob,SCSI_AE_FREEZEONERROR_START);

   /* Get the target id */
   scsiId = (SCSI_UEXACT8) targetUnit->scsiID;

   /* The following logic posts back the SCBs for this target in a          */
   /* chronological order, starting from the SCB which is about to get fired*/
   /* by the sequencer.                                                     */

   /* If a selection is in progress for this target, disable the selection  */
   /* process and post back this SCB to the upper layer.                    */
   if (((scsiseq = (OSD_INEXACT8(SCSI_AICREG(SCSI_SCSISEQ)))) & SCSI_ENSELO))
   {
       hiobCurr = SCSI_ACTPTR[OSD_INEXACT8(SCSI_AICREG(SCSI_S160M_WAITING_SCB))];
       if ((SCSI_UEXACT8)(SCSI_TARGET_UNIT(hiobCurr)->scsiID) == scsiId)
       {
          /* Disable any outstanding selection */
          OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSISEQ),
             scsiseq & ~(SCSI_TEMODEO+SCSI_ENSELO+SCSI_ENAUTOATNO+SCSI_ENAUTOATNI));
          /* To flush the cache */
          scsiseq = OSD_INEXACT8(SCSI_AICREG(SCSI_SCSISEQ));

          /* free the associated entry in active pointer array */
          SCSI_ACTPTR[hiobCurr->scbNumber] = SCSI_NULL_HIOB;

          /* Put proper status in HIOB, so that rsm layer will queue it back */
          /* into the device queue.                                          */
          hiobCurr->stat = SCSI_SCB_FROZEN;

          /* Post back this HIOB */
          if (hiobCurr->cmd == SCSI_CMD_INITIATE_TASK)
          {
             SCSI_COMPLETE_HIOB(hiobCurr);
          }
          else
          {
             SCSI_COMPLETE_SPECIAL_HIOB(hiobCurr);
          }
       }
   }

   /* Post back all the SCBs for the failed target from the Exe Queue.     */

   scbptr = OSD_INEXACT8(SCSI_AICREG(SCSI_SCBPTR));   /* Save SCBPTR */

   /* Check if aborting HIOB is in Execution queue */
   scbPrev = scbCurr = OSD_INEXACT8(SCSI_AICREG(SCSI_S160M_Q_EXE_HEAD));

   while (scbCurr != SCSI_NULL_SCB)
   {
      hiobCurr = SCSI_ACTPTR[scbCurr];

      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCBPTR), scbCurr);

      /* Search for matching target id */
      if ((SCSI_UEXACT8)(SCSI_TARGET_UNIT(hiobCurr)->scsiID) == scsiId)
      {
         /* Remove the SCB from execution queue */
         scbNext = OSD_INEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
                    OSDoffsetof(SCSI_SCB_STANDARD_160M, SCSI_S160M_q_next)));
                        
         /* Mending the execution queue */
         if (scbCurr == scbPrev)
         {
            scbPrev = scbNext;
            /* Removed HIOB at the queue head */
            OSD_OUTEXACT8(SCSI_AICREG(SCSI_S160M_Q_EXE_HEAD), scbNext);
         }
         else
         {
            OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCBPTR), scbPrev);
            OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
                OSDoffsetof(SCSI_SCB_STANDARD_160M,SCSI_S160M_q_next)),scbNext);

            if (scbCurr == OSD_INEXACT8(SCSI_AICREG(SCSI_S160M_Q_EXE_TAIL)))
            {
               /* Removed HIOB at the queue tail */
               OSD_OUTEXACT8(SCSI_AICREG(SCSI_S160M_Q_EXE_TAIL), scbPrev);
            }
         }

         /* free the associated entry in active pointer array */
         SCSI_ACTPTR[hiobCurr->scbNumber] = SCSI_NULL_HIOB;

         /* Put proper status in HIOB, so that rsm layer will queue it back */
         /* into the device queue.                                          */
         hiobCurr->stat = SCSI_SCB_FROZEN;

         /* Post back this HIOB */
         if (hiobCurr->cmd == SCSI_CMD_INITIATE_TASK)
         {
            SCSI_COMPLETE_HIOB(hiobCurr);
         }
         else
         {
            SCSI_COMPLETE_SPECIAL_HIOB(hiobCurr);
         }

         scbCurr = scbNext;
      }
      else
      {
         /* Keep previous SCB */
         scbPrev = scbCurr;
      
         /* Advance to next SCB in Execution queue */
         scbCurr = OSD_INEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
                   OSDoffsetof(SCSI_SCB_STANDARD_160M, SCSI_S160M_q_next)));
      }
   }

   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCBPTR), scbptr);   /* Restore SCBPTR */

   /* If an SCB for this target is getting DMAed, just disable the DMA        */
   /* (this hiob need not be posted back, as it will be done in the next step */
   if ((OSD_INEXACT8(SCSI_AICREG(SCSI_CCSCBCTL)) &
       (SCSI_CCSCBEN+SCSI_CCSCBDIR+SCSI_CCARREN)) ==
       (SCSI_CCSCBEN+SCSI_CCSCBDIR+SCSI_CCARREN))
   {
       /* Take the SCB from the head of the new queue and check if it is for  */
       /* the failed target.                                                  */
       hiobCurr = SCSI_ACTPTR[OSD_INEXACT8(SCSI_AICREG(SCSI_S160M_Q_NEXT_SCB))];
       if ((SCSI_UEXACT8)(SCSI_TARGET_UNIT(hiobCurr)->scsiID) == scsiId)
       {
          /* Disable DMA */
          OSD_OUTEXACT8(SCSI_AICREG(SCSI_CCSCBCTL),0);

          /* Since the sequencer has already incremented the SNSCB_QOFF, it */
          /* should be decremented as it will be adjusted during posting    */
          /* back in the logic below. */
          OSD_OUTEXACT8(SCSI_AICREG(SCSI_SNSCB_QOFF),
                        OSD_INEXACT8(SCSI_AICREG(SCSI_SNSCB_QOFF))-1);
       }

       else
       {
          /* Obtain the next SCB in the new queue and check if it is for the */
          /* failed target.  We have to disable DMA even in this case because*/
          /* after the DMA the q_new_pointer and array_site_next variables   */
          /* will be modified by the sequencer to point to this SCB; but this*/
          /* SCB will be removed while we are searching the new queue down   */
          /* in the logic. */
          if ((scbCurr = SCSIhStandard160mObtainNextScbNum(hhcb,hiobCurr)) !=
              SCSI_NULL_SCB)
          {
             hiobCurr = SCSI_ACTPTR[scbCurr];
             if ((hiobCurr != SCSI_NULL_HIOB) &&
                 ((SCSI_UEXACT8)(SCSI_TARGET_UNIT(hiobCurr)->scsiID) == scsiId))
             {
                /* Disable DMA */
                OSD_OUTEXACT8(SCSI_AICREG(SCSI_CCSCBCTL),0);

               /* Since the sequencer has already incremented the SNSCB_QOFF, */
               /* it should be decremented as it will be adjusted during      */
               /* posting back in the logic below. */
               OSD_OUTEXACT8(SCSI_AICREG(SCSI_SNSCB_QOFF),
                             OSD_INEXACT8(SCSI_AICREG(SCSI_SNSCB_QOFF))-1);
             }
          }
       }
   }

   /* Post back all the SCBs for the failed target from the New Queue.     */

   /* Get q_next_scb value from scratch ram */
   qNextScb = OSD_INEXACT8(SCSI_AICREG(SCSI_S160M_Q_NEXT_SCB));
   scbCurr = scbPrev = qNextScb;

   /* Check if any SCB for this target is in New queue */
   while ((hiobCurr = SCSI_ACTPTR[scbCurr]) != SCSI_NULL_HIOB)
   {
      /* Search for matching target id */
      if ((SCSI_UEXACT8)(SCSI_TARGET_UNIT(hiobCurr)->scsiID) == scsiId)
      {
         /* Decrement and update qNewHead into scratch ram */
         (SCSI_UEXACT8)hhcb->SCSI_HP_qNewHead--;
         OSD_OUTEXACT8(SCSI_AICREG(SCSI_S160M_Q_NEW_HEAD),
                       hhcb->SCSI_HP_qNewHead);

         /* Remove aborted HIOB from New queue and mend the queue */

         scbNext = SCSIhStandard160mObtainNextScbNum(hhcb,hiobCurr);
         scbNextbusAddress = SCSIhStandard160mObtainNextScbAddress(hhcb,hiobCurr);

         if (scbCurr == qNextScb)
         {
            qNextScb = scbPrev = scbNext;
            OSD_OUTEXACT8(SCSI_AICREG(SCSI_S160M_Q_NEXT_SCB),qNextScb);
            SCSI_hSETADDRESSSCRATCH(hhcb,SCSI_S160M_Q_NEW_POINTER,
               hiobCurr->scbDescriptor->queueNext->scbBuffer.virtualAddress);
         }
         else
         {
            /* Obtain previous HIOB in Active pointer array */ 
            hiobPrev = SCSI_ACTPTR[scbPrev];
            SCSIhStandard160mUpdateNextScbNum(hhcb,hiobPrev,scbNext);
            SCSIhStandard160mUpdateNextScbAddress(hhcb,hiobPrev,scbNextbusAddress);
         }

         /* free the associated entry in active pointer array */
         SCSI_ACTPTR[hiobCurr->scbNumber] = SCSI_NULL_HIOB;

         /* Put proper status in HIOB, so that rsm layer will queue it back */
         /* into the device queue.                                          */
         hiobCurr->stat = SCSI_SCB_FROZEN;

         /* Post back this HIOB */
         if (hiobCurr->cmd == SCSI_CMD_INITIATE_TASK)
         {
            SCSI_COMPLETE_HIOB(hiobCurr);
         }
         else
         {
            SCSI_COMPLETE_SPECIAL_HIOB(hiobCurr);
         }

         scbCurr = scbNext;
      }
      else
      {
         /* Need to get next scb from current scb */
         scbPrev = scbCurr;
      
         /* Obtain current HIOB in Active pointer array */ 
         scbCurr = SCSIhStandard160mObtainNextScbNum(hhcb,hiobCurr);
      }
   }

   /* Generate asynchronous event to the RSM layer to notify that the      */
   /* hardware layer has finished freezing its queues for the failed hiob. */
   SCSI_ASYNC_EVENT_COMMAND(hhcb,hiob,SCSI_AE_FREEZEONERROR_END);
}
#endif   /* SCSI_STANDARD_160M_MODE */

/*********************************************************************
*
*  SCSIhSwapping160mFreezeHWQueue
*
*     Freeze the hardware queue for the given target id (which is in the
*     given hiob), by removing all the SCBs for this target from the relevant
*     hardware queues.
*
*  Return Value:  None
*
*  Parameters:    hhcb
*                 Failed hiob
*
*  Remarks:       !!! The sequencer is assumed to be paused and also !!!
*                 !!! assumed to be sitting in the error path        !!!
*
*                 This routine pulls out all the SCBs in the hardware layer
*                 and sends them back to the RSM layer with a special status
*                 (SCSI_SCB_FROZEN) so that the RSM layer can queue them back
*                 to its device queue.
*
*********************************************************************/
#if SCSI_SWAPPING_160M_MODE
void SCSIhSwapping160mFreezeHWQueue (SCSI_HHCB SCSI_HPTR hhcb,
                                    SCSI_HIOB SCSI_IPTR hiob)
{
   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_UNIT_CONTROL SCSI_IPTR targetUnit = SCSI_TARGET_UNIT(hiob);
   SCSI_UEXACT8 qNewTail;
   SCSI_UEXACT8 qNextScb;
   SCSI_UEXACT8 scbCurr;
   SCSI_UEXACT8 scbPrev;
   SCSI_UEXACT8 scbNext;
   SCSI_HIOB SCSI_IPTR hiobCurr;
   SCSI_HIOB SCSI_IPTR hiobPrev;
   SCSI_UEXACT8 scsiId;
   SCSI_UEXACT8 scsiseq;

   /* Generate asynchronous event to the RSM layer to notify that the */
   /* hardware layer is going to start freezing of its queues for the */
   /* failed hiob.                                                    */
   SCSI_ASYNC_EVENT_COMMAND(hhcb,hiob,SCSI_AE_FREEZEONERROR_START);

   /* Get the target id */
   scsiId = (SCSI_UEXACT8) targetUnit->scsiID;

   /* The following logic posts back the SCBs for this target in a          */
   /* chronological order, starting from the SCB which is about to get fired*/
   /* by the sequencer.                                                     */

   /* If a selection is in progress for this target, disable the selection  */
   /* process and post back this SCB to the upper layer.                    */
   if (((scsiseq = (OSD_INEXACT8(SCSI_AICREG(SCSI_SCSISEQ)))) & SCSI_ENSELO))
   {
       hiobCurr = SCSI_ACTPTR[OSD_INEXACT8(SCSI_AICREG(SCSI_W160M_WAITING_SCB))];
       if ((SCSI_UEXACT8)(SCSI_TARGET_UNIT(hiobCurr)->scsiID) == scsiId)
       {
          /* Disable any outstanding selection */
          OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSISEQ),
             scsiseq & ~(SCSI_TEMODEO+SCSI_ENSELO+SCSI_ENAUTOATNO+SCSI_ENAUTOATNI));
          /* To flush the cache */
          scsiseq = OSD_INEXACT8(SCSI_AICREG(SCSI_SCSISEQ));

          /* free the associated entry in active pointer array */
          SCSI_ACTPTR[hiobCurr->scbNumber] = SCSI_NULL_HIOB;

          /* Put proper status in HIOB, so that rsm layer will queue it back */
          /* into the device queue.                                          */
          hiobCurr->stat = SCSI_SCB_FROZEN;

          /* Post back this HIOB */
          if (hiobCurr->cmd == SCSI_CMD_INITIATE_TASK)
          {
             SCSI_COMPLETE_HIOB(hiobCurr);
          }
          else
          {
             SCSI_COMPLETE_SPECIAL_HIOB(hiobCurr);
          }
       }
   }

   /* If an SCB for this target is getting DMAed, just disable the DMA        */
   /* (this hiob need not be posted back, as it will be done in the next step */
   if ((OSD_INEXACT8(SCSI_AICREG(SCSI_CCSCBCTL)) &
       (SCSI_CCSCBEN+SCSI_CCSCBDIR+SCSI_CCARREN)) ==
       (SCSI_CCSCBEN+SCSI_CCSCBDIR+SCSI_CCARREN))
   {
       /* Take the SCB from the head of the new queue and check if it is for  */
       /* the failed target.                                                  */
       hiobCurr = SCSI_ACTPTR[OSD_INEXACT8(SCSI_AICREG(SCSI_W160M_Q_NEXT_SCB))];
       if ((SCSI_UEXACT8)(SCSI_TARGET_UNIT(hiobCurr)->scsiID) == scsiId)
       {
          /* Disable DMA */
          OSD_OUTEXACT8(SCSI_AICREG(SCSI_CCSCBCTL),0);
       }
   }

   /* Post back all the SCBs for the failed target from the New Queue.     */

   /* Get q_new_tail value */
   qNewTail = SCSI_hGETQNEWTAIL(hhcb);

   /* Get q_next_scb value from scratch ram */
   qNextScb = OSD_INEXACT8(SCSI_AICREG(SCSI_W160M_Q_NEXT_SCB));
   scbCurr = scbPrev = qNextScb;

   /* Check if any SCB for this target is in New queue */
   while (qNewTail != hhcb->SCSI_HP_qNewHead)
   {
      /* Search for matching target id */
      hiobCurr = SCSI_ACTPTR[scbCurr];
      if ((SCSI_UEXACT8)(SCSI_TARGET_UNIT(hiobCurr)->scsiID) == scsiId)
      {
         /* Decrement and update qNewHead into scratch ram */
         (SCSI_UEXACT8)hhcb->SCSI_HP_qNewHead--;
         OSD_OUTEXACT8(SCSI_AICREG(SCSI_W160M_Q_NEW_HEAD), hhcb->SCSI_HP_qNewHead);

         /* Remove aborted HIOB from New queue and mend the queue */

         scbNext = SCSIhSwapping160mObtainNextScbNum(hhcb,hiobCurr);

         if (scbCurr == qNextScb)
         {
            qNextScb = scbPrev = scbNext;
            OSD_OUTEXACT8(SCSI_AICREG(SCSI_W160M_Q_NEXT_SCB),qNextScb);
         }
         else
         {
            /* Obtain previous HIOB in Active pointer array */ 
            hiobPrev = SCSI_ACTPTR[scbPrev];
            SCSIhSwapping160mUpdateNextScbNum(hhcb,hiobPrev,scbNext);
         }

         /* free the associated entry in active pointer array */
         SCSI_ACTPTR[hiobCurr->scbNumber] = SCSI_NULL_HIOB;

         /* Put proper status in HIOB, so that rsm layer will queue it back */
         /* into the device queue.                                          */
         hiobCurr->stat = SCSI_SCB_FROZEN;

         /* Post back this HIOB */
         if (hiobCurr->cmd == SCSI_CMD_INITIATE_TASK)
         {
            SCSI_COMPLETE_HIOB(hiobCurr);
         }
         else
         {
            SCSI_COMPLETE_SPECIAL_HIOB(hiobCurr);
         }

         scbCurr = scbNext;
      }
      else
      {
         /* Need to get next scb from current scb */
         scbPrev = scbCurr;
      
         /* Obtain current HIOB in Active pointer array */ 
         scbCurr = SCSIhSwapping160mObtainNextScbNum(hhcb,hiobCurr);

         (SCSI_UEXACT8)qNewTail++;
      }
   }

   /* Generate asynchronous event to the RSM layer to notify that the      */
   /* hardware layer has finished freezing its queues for the failed hiob. */
   SCSI_ASYNC_EVENT_COMMAND(hhcb,hiob,SCSI_AE_FREEZEONERROR_END);
}
#endif   /* SCSI_SWAPPING_160M_MODE */

/*$Header:   Y:/source/chimscsi/src/himscsi/hwmscsi/HWMTASK.CV_   1.56.2.3   24 Mar 1998 17:16:42   NGUY2129  $*/
/***************************************************************************
*                                                                          *
* Copyright 1995,1996,1997,1998,1999 Adaptec, Inc.,  All Rights Reserved.  *
*                                                                          *
* This software contains the valuable trade secrets of Adaptec.  The       *
* software is protected under copyright laws as an unpublished work of     *
* Adaptec.  Notice is for informational purposes only and does not imply   *
* publication.  The user of this software may make copies of the software  *
* for use with parts manufactured by Adaptec or under license from Adaptec *
* and for no other use.                                                    *
*                                                                          *
***************************************************************************/

/***************************************************************************
*
*  Module Name:   HWMTASK.C
*
*  Description:
*                 Codes to implement task handler for hardware management 
*                 module
*
*  Owners:        ECX IC Firmware Team
*    
*  Notes:         NONE
*
*  Entry Point(s):
*                 SCSIHSetUnitHandle
*                 SCSIHFreeUnitHandle
*                 SCSIHDisableIRQ
*                 SCSIHEnableIRQ
*                 SCSIHPollIRQ
*                 SCSIHQueueHIOB
*                 SCSIHQueueSpecialHIOB
*                 SCSIHSaveState
*                 SCSIHRestoreState
*                 SCSIHPowerManagement
*                 SCSIHApplyDeviceTable
*                 SCSIHCheckHostIdle
*                 SCSIHCheckDeviceIdle
*                 SCSIHSuppressNegotiation
*                 SCSIHForceNegotiation
*                 SCSIHGetHWInformation
*                 SCSIHPutHWInformation
*
***************************************************************************/

#include "scsi.h"
#include "hwm.h"

/*%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%*/
/* Modules as interface to hardware management layer                      */ 
/*%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%*/

/*********************************************************************
*
*  SCSIHSetUnitHandle
*
*     Validate unit handle for specified device control block
*
*  Return Value:  none
*                  
*  Parameters:    hhcb
*                 target unit control block
*                 scsi id
*                 logical unit number
*
*  Remarks:       Validated device handle is used to indicate HIOB
*                 destination. After protocol automatic configuration
*                 OSD may examine device table (embedded in SCSI_HHCB)
*                 and find out the device/unit it interested in. This
*                 routine is for OSD to acquire a unit handle which
*                 is required fir building SCSI_HIOB. After protocol
*                 automatic configuration all unit handles become
*                 inlidated automatically.
*                  
*********************************************************************/
void SCSIHSetUnitHandle (SCSI_HHCB SCSI_HPTR hhcb,
                                     SCSI_UNIT_CONTROL SCSI_UPTR targetUnit,
                                     SCSI_UEXACT8 scsiID,
                                     SCSI_UEXACT8 lunID)
{
   /* associate target unit id with particular unit */
   targetUnit->hhcb = hhcb;
   targetUnit->scsiID = scsiID;
   targetUnit->lunID = lunID;
#if SCSI_DMA_SUPPORT
   if (scsiID == SCSI_DMA_ID)
   {
      /* the deviceTable entry is not used for DMA I/Os */
      /* however lets just initialize it to the hostScsiID - just in case */  
      targetUnit->deviceTable = &SCSI_DEVICE_TABLE(hhcb)[hhcb->hostScsiID];
   }
   else
#endif /* SCSI_DMA_SUPPORT */
   {
      targetUnit->deviceTable = &SCSI_DEVICE_TABLE(hhcb)[scsiID];
   }
}

/*********************************************************************
*
*  SCSIHFreeUnitHandle
*
*     Invalidate specified unit handle
*
*  Return Value:  none
*                  
*  Parameters:    unit handle
*
*  Remarks:       After invalidated the unit handle should not
*                 be used in HIOB. Whenever there is a validated unit
*                 handle which is not in OSD's interest it can
*                 invalidate unit handle through this routine.
*                  
*********************************************************************/
void SCSIHFreeUnitHandle (SCSI_UNIT_HANDLE unitHandle)
{
   /* nothing need to be done for now */
}

/*********************************************************************
*
*  SCSIHDisableIRQ
*
*     Disable device hardware interrupt
*
*  Return Value:  None
*                  
*  Parameters:    hhcb
*
*  Remarks:
*                  
*********************************************************************/
void SCSIHDisableIRQ (SCSI_HHCB SCSI_HPTR hhcb)
{
   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;

   /* program hardware to disable interrupt */
   SCSI_hWRITE_HCNTRL(hhcb,scsiRegister,
                  OSD_INEXACT8(SCSI_AICREG(SCSI_HCNTRL)) & ~SCSI_INTEN);

   OSD_SYNCHRONIZE_IOS(hhcb);
}

/*********************************************************************
*
*  SCSIHEnableIRQ
*
*     Enable device hardware interrupt
*
*  Return Value:  None
*                  
*  Parameters:    hhcb
*
*  Remarks:
*                  
*********************************************************************/
void SCSIHEnableIRQ (SCSI_HHCB SCSI_HPTR hhcb)
{
   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;

   /* program hardware to enable interrupt */
   SCSI_hWRITE_HCNTRL(hhcb,scsiRegister,
                     OSD_INEXACT8(SCSI_AICREG(SCSI_HCNTRL)) | SCSI_INTEN);

   OSD_SYNCHRONIZE_IOS(hhcb);
}

/*********************************************************************
*
*  SCSIHPollIRQ
*
*     Poll if interrupt pending for associated device hardware
*
*  Return Value:  None
*                  
*  Parameters:    hhcb
*
*  Remarks:
*                  
*********************************************************************/
int SCSIHPollIRQ (SCSI_HHCB SCSI_HPTR hhcb)
{
   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;

   /* poll hardware interrupt status */
   return(SCSI_hREAD_INTSTAT_FAST(hhcb,scsiRegister) &
         (SCSI_BRKINT | SCSI_SCSIINT | SCSI_CMDCMPLT | SCSI_SEQINT));
}

/*********************************************************************
*
*  SCSIHQueueHIOB 
*
*     Queue iob to sequencer/hardware queue
*
*  Return Value:  none
*                  
*  Parameters:    hiob
*
*  Remarks:
*                  
*********************************************************************/
/* @@@@ do later - additional inline code needed to do this #if !SCSI_STREAMLINE_QIOPATH */
void SCSIHQueueHIOB (SCSI_HIOB SCSI_IPTR hiob)
{
   SCSI_HHCB SCSI_HPTR hhcb = SCSI_TARGET_UNIT(hiob)->hhcb;

   /* fresh the hiob status and working area */
   SCSI_hFRESH_HIOB(hiob);

   /* setup active pointer */
   SCSI_ACTPTR[hiob->scbNumber] = hiob;

   /* assign scb buffer when necessary */
   SCSI_hASSIGN_SCB_BUFFER(hhcb,hiob);

   /* deliver it to sequencer */
   SCSI_hDELIVERSCB(hhcb, hiob);

   OSD_SYNCHRONIZE_IOS(hhcb);
}
/* @@@@ do later - additional inline code needed to do this #endif */ /* !SCSI_STREAMLINE_QIOPATH */

/*********************************************************************
*
*  SCSIHQueueSpecialHIOB
*
*     Queue special iob to head of sequencer/hardware queue
*
*  Return Value:  none
*                  
*  Parameters:    hiob
*
*  Remarks:       This routine will process all special request
*                 through HIOB.
*                  
*********************************************************************/
void SCSIHQueueSpecialHIOB (SCSI_HIOB SCSI_IPTR hiob)
{
   SCSI_HHCB SCSI_HPTR hhcb;

   /* fresh the hiob status and working area */
   SCSI_hFRESH_HIOB(hiob);

   switch(hiob->cmd)
   {
      case SCSI_CMD_ABORT_TASK:
         hhcb = SCSI_TARGET_UNIT(SCSI_RELATED_HIOB(hiob))->hhcb;
         SCSIhAbortTask(hiob);
         break;

      case SCSI_CMD_ABORT_TASK_SET:
         hhcb = SCSI_TARGET_UNIT(hiob)->hhcb;
         SCSIhAbortTaskSet(hiob);
         break;

      case SCSI_CMD_RESET_BUS:
         /* Perform Ssci_Bus_Reset request */
         hhcb = SCSI_INITIATOR_UNIT(hiob);
         SCSIhScsiBusReset(hhcb,hiob);

         /* Post reset bus done */
         SCSI_COMPLETE_SPECIAL_HIOB(hiob);
         break;

      case SCSI_CMD_RESET_TARGET:
         /* Perform Bus_Device_Reset request */
         hhcb = SCSI_TARGET_UNIT(hiob)->hhcb;

         /* Setup SCB to issue Bus_Device_Reset message. */
         /* All HIOBs will be aborted when the sequencer informed the CHIM */
         /* that the BDR message (0x0C) is ready to send to the target. */
         /* The SCSIhAbortTarget() will be called. */
         SCSI_hSETUPTARGETRESET(hhcb, hiob);
         break;

      case SCSI_CMD_RESET_BOARD:
      case SCSI_CMD_RESET_HARDWARE:
         hhcb = SCSI_INITIATOR_UNIT(hiob);
         break;

      case SCSI_CMD_PROTO_AUTO_CFG:
         hhcb = SCSI_INITIATOR_UNIT(hiob);

         SCSI_hTARGETMODEENABLE(hhcb);

         if (!hhcb->SCSI_HF_skipScam)
         {
            SCSI_hPROTOCOLAUTOCONFIG(hhcb);

            /* For scam level 2 only, do not invoke SCAM protocol on */
            /* the subsequent protocol_auto_configs until there is a */
            /* reset.  Due to the mismatch of SCSI transfer modes:   */
            /* asynchronous (initiator) versus synchronous (target). */
            hhcb->SCSI_HF_skipScam = (hhcb->SCSI_HF_scamLevel == 2);
         }

         hiob->stat = SCSI_SCB_COMP;
         SCSI_COMPLETE_SPECIAL_HIOB(hiob);
         break;

#if SCSI_TARGET_OPERATION     
      case SCSI_CMD_ESTABLISH_CONNECTION:
         /* Should never come here */
         hhcb = SCSI_NEXUS_UNIT(hiob)->hhcb;
         
         /* setup active pointer */
         SCSI_ACTPTR[hiob->scbNumber] = hiob;

         /* Set target HIOB indicator */
         SCSI_hTARGETMODE_HIOB(hiob);

         /* assign scb buffer when necessary */
         SCSI_hASSIGN_SCB_BUFFER(hhcb,hiob);
   
         /* Deliver Establish SCB */
         SCSI_hTARGETDELIVERESTSCB(hhcb, hiob);
         break; 

      case SCSI_CMD_REESTABLISH_INTERMEDIATE:
         hhcb = SCSI_NEXUS_UNIT(hiob)->hhcb;
         SCSIhTargetSendHiob(hhcb,hiob); 
         break; 

      case SCSI_CMD_REESTABLISH_AND_COMPLETE:
         hhcb = SCSI_NEXUS_UNIT(hiob)->hhcb;
         if (SCSI_NEXUS_UNIT(hiob)->SCSI_XP_taskManagementFunction)
         {  
             SCSI_hFRESH_HIOB(hiob);         
             if ((SCSI_NEXUS_UNIT(hiob)->SCSI_XP_deviceResetInProgress) &&
                 (!hhcb->SCSI_HF_initiatorMode) &&
                 (hhcb->SCSI_HF_targetNumIDs <= 1))            
             {  /* if target reset then we should re-initialize the
                 * hardware as long as initiator mode is not enabled
                 * and we're active as multiple Ids for this adapter
                 */
                SCSIhTargetResetTarget(hhcb);
             }      
             else
             {
                SCSIhTargetScsiBusFree(hhcb);
                /* Sequencer is idling -- start it working again */
                /* This routine probably needs modified to 
                 * not unpause sequencer unless it's already paused.
                 * Revisit.
                 */
                SCSIhTargetRedirectSequencer(hhcb); 
             }
             /* complete the request */
             SCSIhTerminateTargetCommand(hiob);
         }       
         else
         {
            SCSIhTargetSendHiob(hhcb,hiob); 
         }
         break; 
         
     case SCSI_CMD_ABORT_NEXUS:
         hhcb = SCSI_NEXUS_UNIT(hiob)->hhcb;
         SCSIhTargetAbortNexusSet(hiob);
         break;
         
#if SCSI_MULTIPLEID_SUPPORT
      case SCSI_CMD_ENABLE_ID:
         hhcb = SCSI_INITIATOR_UNIT(hiob);
         SCSIhTargetEnableId(hiob);
         break;

      case SCSI_CMD_DISABLE_ID:
         hhcb = SCSI_INITIATOR_UNIT(hiob); 
         SCSIhTargetSetupDisableId(hiob);
         break; 
#endif /* SCSI_MULTIPLEID_SUPPORT */
#endif  /* SCSI_TARGET_OPERATION */    

      default:
         return;
   }

   /* flush the register access */
   OSD_SYNCHRONIZE_IOS(hhcb);
}

/*********************************************************************
*
*  SCSIHSaveState
*
*     Save current hardware/sequencer state 
*
*  Return Value:  none
*                  
*  Parameters:    hhcb
*                 state
*
*  Remarks:       This routine will save whatever current harwdare
*                 into the state structure. It's caller's responbility
*                 to make sure state saving is necessary.
*                  
*********************************************************************/
#if SCSI_SAVE_RESTORE_STATE
void SCSIHSaveState (SCSI_HHCB SCSI_HPTR hhcb,
                     SCSI_STATE SCSI_HPTR state)
{
   SCSI_UINT16 i;
   SCSI_UEXACT8 scbsize;
   SCSI_UEXACT8 hcntrl;
   SCSI_UEXACT8 scbptr;
   SCSI_UEXACT8 seqctl;
   
#if POWER_MANAGEMENT_SAVE_RESTORE   
   SCSI_UEXACT8  firmware_mode;
#endif

#if SCSI_AICTRIDENT
   SCSI_UEXACT8 scsiSfunct;
#endif /* SCSI_AICTRIDENT */
   SCSI_UEXACT32 value;
   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;


   /* save the HCNTRL value */
   hcntrl = OSD_INEXACT8(SCSI_AICREG(SCSI_HCNTRL));

#if POWER_MANAGEMENT_SAVE_RESTORE   
   state->hcntrl = hcntrl;
#endif   

   /* pause the chip if necessary */
   if (!(hcntrl & SCSI_PAUSEACK))
   {
      state->SCSI_SF_chipPause = 0;
      SCSIhPauseAndWait(hhcb);
   }
   else
   {
      state->SCSI_SF_chipPause = 1;
   }

#if !POWER_MANAGEMENT_SAVE_RESTORE   
   
   if (!(hcntrl & SCSI_INTEN))
   {
      state->SCSI_SF_interruptEnable = 0;
   }
   else
   {
      state->SCSI_SF_interruptEnable = 1;
   }
#endif

#if SCSI_AIC78XX
   if (hhcb->hardwareMode == SCSI_HMODE_AIC78XX)
   {
   /* Save the HADDR into state structure for HP problem of switching between */ 
   /* DOS and Win95 */

      for (i = 0; i < 4; i++)
      {
         state->hAddr[i] = OSD_INEXACT8(SCSI_AICREG((SCSI_UEXACT8)(SCSI_HADDR0+i)));
      }
   }
#endif /* SCSI_AIC78xx */

#if SCSI_TARGET_OPERATION  
   /* Save state of the enable selection for target mode */
   if (OSD_INEXACT8(SCSI_AICREG(SCSI_SCSISEQ)) & SCSI_ENSELI) 
   {
      state->SCSI_SF_selectionEnable = 1;
   }
   else
   {
      state->SCSI_SF_selectionEnable = 0;
   }
#endif  /* SCSI_TARGET_OPERATION */   
   /* Save upper and lower scratch ram areas into the state structure   */
   /*    Some chips have two scratch ranges and some have one.          */
   /*    (e.g. Lance/Katana has 1 scratch range and excalibur has two)  */
   for (i = 0; i < SCSI_SCRATCH0_SIZE; i++)
   {
      state->scratch0[i] = 
            OSD_INEXACT8(SCSI_AICREG((SCSI_UEXACT8)(SCSI_SCRATCH0+i)));
   }
   if (hhcb->SCSI_HP_scratch1Exist)
   {
      for (i = 0; i <  SCSI_SCRATCH1_SIZE; i++)
      {
         state->scratch1[i] = 
               OSD_INEXACT8(SCSI_AICREG((SCSI_UEXACT8)(SCSI_SCRATCH1+i)));
      }
   }

#if (SCSI_AICBAYONET || SCSI_AICTRIDENT)
   if ((hhcb->hardwareMode == SCSI_HMODE_AICBAYONET) ||
       (hhcb->hardwareMode == SCSI_HMODE_AICTRIDENT))
   {
      /* save the newhead/tail registers */   
      state->qNewHead = OSD_INEXACT8(SCSI_AICREG(SCSI_hQ_NEW_HEAD(hhcb)));
      state->qNewTail = OSD_INEXACT8(SCSI_AICREG(SCSI_hQ_NEW_TAIL(hhcb)));
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_hQ_NEW_TAIL(hhcb)), state->qNewTail);
   }
#endif

#if SCSI_AICTRIDENT
   if (hhcb->hardwareMode == SCSI_HMODE_AICTRIDENT)
   {
      /* save the optionmode register */
      scsiSfunct = OSD_INEXACT8(SCSI_AICREG(SCSI_SFUNCT));
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SFUNCT), (scsiSfunct | SCSI_ALT_MODE));
      state->optionMode = OSD_INEXACT8(SCSI_AICREG(SCSI_OPTIONMODE));
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SFUNCT), scsiSfunct);
   }
#endif /* SCSI_AICTRIDENT */

   /* save the hostSCSIID */
   state->hostScsiId = (OSD_INEXACT8(SCSI_AICREG(SCSI_hSCSIID(hhcb))) & 0x0f); 

   seqctl = OSD_INEXACT8(SCSI_AICREG(SCSI_SEQCTL));   /* Save SEQCTL */
   
#if POWER_MANAGEMENT_SAVE_RESTORE   
   
   state->hdr_type = OSD_READ_PCICFG_EXACT32(hhcb,SCSI_HDR_TYPE_REG);
   state->status_cmd_register = OSD_READ_PCICFG_EXACT32(hhcb,SCSI_STATUS_CMD_REG);
   state->scsiseq =  OSD_INEXACT8(SCSI_AICREG(SCSI_SCSISEQ));
   state->sxfrctl0 = OSD_INEXACT8(SCSI_AICREG(SCSI_SXFRCTL0));
   state->sxfrctl1 = OSD_INEXACT8(SCSI_AICREG(SCSI_SXFRCTL1));
   state->simode0 = OSD_INEXACT8(SCSI_AICREG(SCSI_SIMODE0));
   state->simode1 = OSD_INEXACT8(SCSI_AICREG(SCSI_SIMODE1));
   state->sblkctl = OSD_INEXACT8(SCSI_AICREG(SCSI_SBLKCTL));
   state->pcistatus = SCSI_hREADDATAFIFOTHRSH(hhcb);
   state->dfcntrl = OSD_INEXACT8(SCSI_AICREG(SCSI_DFCNTRL));
   state->scamctrl = OSD_INEXACT8(SCSI_AICREG(SCSI_SCAMCTRL));
   state->scsisig = OSD_INEXACT8(SCSI_AICREG(SCSI_SCSISIG));

#if (SCSI_STANDARD_MODE || SCSI_SWAPPING_ADVANCED_MODE || SCSI_SWAPPING_160M_MODE)

   firmware_mode =  hhcb->firmwareMode;

   if (firmware_mode == SCSI_FMODE_STANDARD64 ||
      firmware_mode == SCSI_FMODE_STANDARD32 ||
      firmware_mode == SCSI_FMODE_STANDARD_ADVANCED ||
      firmware_mode == SCSI_FMODE_STANDARD_160M ||
      firmware_mode == SCSI_FMODE_SWAPPING_160M ||
      firmware_mode == SCSI_FMODE_SWAPPING_ADVANCED)

   {

      state->cchhaddr0 =  OSD_INEXACT8(SCSI_AICREG(SCSI_CCHHADDR0));
      state->cchhaddr1 =  OSD_INEXACT8(SCSI_AICREG(SCSI_CCHHADDR1));
      state->cchhaddr2 =  OSD_INEXACT8(SCSI_AICREG(SCSI_CCHHADDR2));
      state->cchhaddr3 =  OSD_INEXACT8(SCSI_AICREG(SCSI_CCHHADDR3));
      state->ccscbctl  =  OSD_INEXACT8(SCSI_AICREG(SCSI_CCSCBCTL));
      state->ccsgctl = OSD_INEXACT8(SCSI_AICREG(SCSI_CCSGCTL));
   }
#endif
#endif   

   /* Save the sequencer ram into the state structure */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SEQCTL), SCSI_FAILDIS + SCSI_FASTMODE + SCSI_PERRORDIS);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SEQCTL), SCSI_FAILDIS + SCSI_FASTMODE + 
         SCSI_PERRORDIS + SCSI_LOADRAM);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SEQADDR0), 00);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SEQADDR1), 00);

   if (!hhcb->SCSI_HF_dontSaveSeqInState)
   {
      /* only save the sequencer if requested */ 
      for (i = 0; i < SCSI_hSEQCODE_SIZE(hhcb); i++)
      {
         state->seqRam[i] = OSD_INEXACT8(SCSI_AICREG(SCSI_SEQRAM));
      }
   }

   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SEQCTL),SCSI_FAILDIS + SCSI_FASTMODE + SCSI_PERRORDIS);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SEQCTL), seqctl);   /* Restore SEQCTL */

   /* Save SCBRAMSEL bit in DEVCONFIG*/
   value = SCSI_hGETSCBRAMCTRLS(hhcb);

#if POWER_MANAGEMENT_SAVE_RESTORE   
   state->devconfig = value;
#endif   

   if (value & SCSI_hSCBRAMSEL(hhcb)) state->SCSI_SF_scbRamSel = 1;
   else state->SCSI_SF_scbRamSel = 0;

   if (value & SCSI_hSCBSIZE32(hhcb)) state->SCSI_SF_scbSize32 = 1;
   else state->SCSI_SF_scbSize32 = 0;

   /* if state->SCSI_SF_scbRamSel=0 and state->SCSI_SF_scbSize32=0, it means */
   /* we are running standard mode, so we need to save busy target array.    */
   if(!state->SCSI_SF_scbRamSel && !state->SCSI_SF_scbSize32)
   {
      scbptr = OSD_INEXACT8(SCSI_AICREG(SCSI_SCBPTR));   /* Save curr scb ptr */
      for(i = 0; i < (SCSI_MAXSCBS+1); i++)
      {
         OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCBPTR), i);
         state->Bta[i] = OSD_INEXACT8(SCSI_AICREG(SCSI_hBTATABLE(hhcb)));
      }
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCBPTR), scbptr); /* restore cur scb ptr */
   }

   /* If SCSI_SCBSIZE32 bit in the DEVCONFIG register exists            */
   /* then a zero in this bit means 64 byte scb page otherwise 32 byte  */
   if (hhcb->SCSI_HP_scbSize32BitExist && (!state->SCSI_SF_scbSize32))
   {
      scbsize = 64;
   }
   else
   {
      scbsize = 32;
   }

   /* Originally we only save scb #2 information.  To fix a NetWare        */
   /* installation problem we need to also save scb #0 information.        */
   /* During NetWare environment, especially installation process,         */
   /* the NetWare driver needs to switch to real mode driver, ASPI8U2.SYS  */
   /* (or ASPI8DOS.SYS) to access CD-ROM and switch back to protected mode */
   /* NetWare driver a lot.  For optima sequencer inside HIM/ASPI8U2.SYS   */
   /* driver, we use sdiscon(bit 2) of scontrol byte inside scb #0 to tell */
   /* if it is a scb #0 for BIOS or scb #1 for ASPI8U2.SYS during          */
   /* reselection.  Since CHIM inside NetWare driver will also use scb #0  */
   /* for sending commands, the sdiscon bitinside scb #0 will get          */
   /* overwritten.  Save scb #0 information here fixes the problem.        */
   /* Save scb #2 and #0 (internal) */
   SCSI_hPUTSCBRAMCTRLS(hhcb,value|((SCSI_UEXACT32)SCSI_hSCBRAMSEL(hhcb))); /* Internal scbs */
   scbptr = OSD_INEXACT8(SCSI_AICREG(SCSI_SCBPTR));   /* Save curr scb ptr */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCBPTR), 2);        /* Ld scb ptr with 2 */
   for (i=0;i<scbsize;i++)
   {                 
      state->scb2[i] = OSD_INEXACT8(SCSI_AICREG(SCSI_SCB_BASE+i));
   }
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCBPTR), 0);        /* Ld scb ptr with 0 */
   for (i=0;i<scbsize;i++)
   {                 
      state->scb0[i] = OSD_INEXACT8(SCSI_AICREG(SCSI_SCB_BASE+i));
   }
   /* restore cur scb ptr */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCBPTR), scbptr);
   /* restore SCB RAM CONTROL bits */
   SCSI_hPUTSCBRAMCTRLS(hhcb,value);

#if !POWER_MANAGEMENT_SAVE_RESTORE   

   if ((((hhcb->deviceID & SCSI_ID_MASK) == SCSI_EXCALIBUR_BASE) && 
      (hhcb->hardwareMode == SCSI_HMODE_AIC78XX)) 
      && hhcb->hardwareRevision == 3)
   {
      if (OSDReadPciConfiguration(hhcb,SCSI_STATUS_CMD_REG) & SCSI_MWRICEN)
      {
         state->SCSI_SF_mwiBit = 1;
      }
      else
      {
         state->SCSI_SF_mwiBit = 0;
      }
   }
#endif
   /* Reset Sequencer Start address */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SEQADDR0), 
         (SCSI_UEXACT8)SCSI_hIDLE_LOOP_ENTRY(hhcb) >> 2);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SEQADDR1), 
         (SCSI_UEXACT8)SCSI_hIDLE_LOOP_ENTRY(hhcb) >> 10);

   /* restore HCNTRL if necessary */
   if (!(hcntrl & SCSI_PAUSEACK))
   {
      SCSI_hWRITE_HCNTRL(hhcb,scsiRegister,hcntrl);
   }
   
   OSD_SYNCHRONIZE_IOS(hhcb);
}
#endif

/*********************************************************************
*
*  SCSIHRestoreState
*
*     Apply state structure to restore hardware/sequencer state 
*
*  Return Value:  none
*                  
*  Parameters:    hhcb
*                 state
*
*  Remarks:       This routine will restore state specified by
*                 state structure to harwdare/sequencer.
*                 It's caller's responbility to make sure state 
*                 saving is necessary.
*                  
*********************************************************************/
#if SCSI_SAVE_RESTORE_STATE
void SCSIHRestoreState (SCSI_HHCB SCSI_HPTR hhcb,
                        SCSI_STATE SCSI_HPTR state)
{
   SCSI_UINT16 i;
   SCSI_UEXACT8 scbsize;
   SCSI_UEXACT8 scbptr;
#if SCSI_AICTRIDENT
   SCSI_UEXACT8 scsiSfunct;
#endif /* SCSI_AICTRIDENT */

   SCSI_UEXACT8 hcntrl;
#if POWER_MANAGEMENT_SAVE_RESTORE   
   SCSI_UEXACT8  firmware_mode;
   SCSI_UEXACT8 j;
#else 
   SCSI_UEXACT32 command_reg;
#endif
   SCSI_UEXACT32 value, tempvalue;
   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;

   /* save the HCNTRL value */
   hcntrl = OSD_INEXACT8(SCSI_AICREG(SCSI_HCNTRL));

   /* Pause sequencer */
   SCSIhPauseAndWait(hhcb);

   /* For Cardbus (APA-1480x) we need to turn off the Power Down mode */
   /* before we can access to any chip's registers.                   */
   
   if (hcntrl & SCSI_POWRDN)
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_HCNTRL),(hcntrl & ~(SCSI_POWRDN | SCSI_CHIPRESET)));   
       
#if SCSI_AIC78XX  
   if (hhcb->hardwareMode == SCSI_HMODE_AIC78XX)
   {   
   /* Restore the HADDR from state structure for HP problem of switching between */ 
   /* DOS and Win95 */
      for (i = 0; i < 4; i++)
      {
         OSD_OUTEXACT8(SCSI_AICREG((SCSI_UEXACT8)(SCSI_HADDR0+i)), state->hAddr[i]);
      }
   }
#endif /* SCSI_AIC78xx */

   /* For restoring real mode driver, update the xfer option with the value */
   /* currently in device table                                             */
   if(state->SCSI_SF_optMode)
   {
      for(i=0; i < hhcb->maxDevices; i++)
      {
         state->scratch0[i] =
            (SCSI_UEXACT8)(SCSI_hGETXFEROPTION(hhcb,(SCSI_UEXACT8)i)&0x00ff);

         /* For U2 hardware, scsi offset is in scratch 1 */
         if (hhcb->hardwareMode == SCSI_HMODE_AICBAYONET)
         {
            state->scratch1[i] =
               (SCSI_UEXACT8)(SCSI_hGETXFEROPTION(hhcb,(SCSI_UEXACT8)i) >> 8);
         }
         else if (hhcb->hardwareMode == SCSI_HMODE_AICTRIDENT)
         {
#if SCSI_OEM1_SUPPORT
            /* Special for OEM1 support.  We cannot pass 0x8F over   */
            /* to the real mode because the real mode code does not  */
            /* support negotiation.  Instead, we will copy xfer rate */
            /* and offset in the auto scsi rate table.               */
            if ((state->scratch0[i] == SCSI_NEEDNEGO) 
                && (hhcb->OEMHardware==SCSI_OEMHARDWARE_OEM1))
            {
               /* get scsi rate from hardware table */
               OSD_OUTEXACT8(SCSI_AICREG(SCSI_AUTORATE_PNTR), (SCSI_RATE_DATA | i));
               state->scratch0[i] = OSD_INEXACT8(SCSI_AICREG(SCSI_AUTORATE_DATA));

               /* get scsi offset from hardware table */
               OSD_OUTEXACT8(SCSI_AICREG(SCSI_AUTORATE_PNTR), i);
               state->scratch1[i] = OSD_INEXACT8(SCSI_AICREG(SCSI_AUTORATE_DATA));
            }
            else
#endif /* SCSI_OEM1_SUPPORT */
            {
               state->scratch1[i] =
                  (SCSI_UEXACT8)(SCSI_hGETXFEROPTION(hhcb,(SCSI_UEXACT8)i) >> 8);
            }
         }        
      }
   }

   /* Restore upper and lower scratch ram areas from the state structure   */
   /*    Some chips have two scratch ranges and some have one.             */
   /*    (e.g. Lance/Katana has 1 scratch range and excalibur has two)     */
   for (i = 0; i < SCSI_SCRATCH0_SIZE; i++)
   {
      OSD_OUTEXACT8(SCSI_AICREG((SCSI_UEXACT8)(SCSI_SCRATCH0+i)),
            state->scratch0[i]);
   }
   if (hhcb->SCSI_HP_scratch1Exist)
   {
      for (i = 0; i < SCSI_SCRATCH1_SIZE; i++)
      {
         OSD_OUTEXACT8(SCSI_AICREG((SCSI_UEXACT8)(SCSI_SCRATCH1+i)),
               state->scratch1[i]);
      }
   } 

#if (SCSI_AICBAYONET || SCSI_AICTRIDENT)
   if ((hhcb->hardwareMode == SCSI_HMODE_AICBAYONET) ||
       (hhcb->hardwareMode == SCSI_HMODE_AICTRIDENT))
   {
      /* restore the newhead/tail registers */   
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_hQ_NEW_HEAD(hhcb)), state->qNewHead);
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_hQ_NEW_TAIL(hhcb)), state->qNewTail);
   }
#endif /* (SCSI_AICBAYONET || SCSI_AICTRIDENT) */

#if SCSI_AICTRIDENT
   if (hhcb->hardwareMode == SCSI_HMODE_AICTRIDENT)
   {
      /* restore optionmode register */
      scsiSfunct = OSD_INEXACT8(SCSI_AICREG(SCSI_SFUNCT));
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SFUNCT), (scsiSfunct | SCSI_ALT_MODE));
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_OPTIONMODE), state->optionMode);
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SFUNCT), scsiSfunct);
   }
#endif /* SCSI_AICTRIDENT */

   /* Restore the hostSCSIID */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_hSCSIID(hhcb)),state->hostScsiId );
   
   if (hhcb->SCSI_HF_dontSaveSeqInState)
   {
      /* Restore sequencer from initialization array */
      SCSI_hSETUPSEQUENCER(hhcb);
   }
   else
   {
      /* Restore the sequencer ram from the state structure */
      SCSIhLoadSequencer(scsiRegister,state->seqRam,(int)SCSI_hSEQCODE_SIZE(hhcb));
   } 


#if POWER_MANAGEMENT_SAVE_RESTORE   
   value = state->devconfig;
#else
   /* get DEVCONFIG reg */
   value = SCSI_hGETSCBRAMCTRLS(hhcb);
#endif

   /* Restore the SCSI_SCBSIZE32 bit. If this bit in the DEVCONFIG register  */
   /* exists then zero in this bit means 64 byte scb page otherwise 32 byte  */
   if (!state->SCSI_SF_scbSize32)
   {
      value &= ~((SCSI_UEXACT32) SCSI_hSCBSIZE32(hhcb));
      if (hhcb->SCSI_HP_scbSize32BitExist)
      {
         scbsize = 64;
      }
      else
      {
         scbsize = 32;
      }
   }
   else
   {
      value |= (SCSI_UEXACT32) SCSI_hSCBSIZE32(hhcb);
      scbsize = 32;
   }

   /* Restore scb #2 and #0 (internal) */
   tempvalue = (value | ((SCSI_UEXACT32)SCSI_hSCBRAMSEL(hhcb)));
   SCSI_hPUTSCBRAMCTRLS(hhcb,tempvalue); /* Internal scbs */
   scbptr = OSD_INEXACT8(SCSI_AICREG(SCSI_SCBPTR));   /* Save curr scb ptr  */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCBPTR), 2);        /* Load scb ptr with 2*/
   for (i=0;i<scbsize;i++)                            /* Restore scb #2     */
   {                 
      OSD_OUTEXACT8(SCSI_AICREG((SCSI_UEXACT8)(SCSI_SCB_BASE+i)),
            state->scb2[i]);
   }
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCBPTR), 0);        /* Load scb ptr with 0*/
   for (i=0;i<scbsize;i++)                            /* Restore scb #0     */
   {                 
      OSD_OUTEXACT8(SCSI_AICREG((SCSI_UEXACT8)(SCSI_SCB_BASE+i)),
            state->scb0[i]);
   }
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCBPTR), scbptr);   /* restore cur scb ptr*/
   
   /* Restore SCBRAMSEL bit */
   if (state->SCSI_SF_scbRamSel == 1)
   {
      value |= (SCSI_UEXACT32) SCSI_hSCBRAMSEL(hhcb);
   }
   else
   {
      value &= ~((SCSI_UEXACT32) SCSI_hSCBRAMSEL(hhcb));
   }   

   /* restore SCB RAM CONTROL bits */
   SCSI_hPUTSCBRAMCTRLS(hhcb,value);

   /* if state->SCSI_SF_scbRamSel=0 and state->SCSI_SF_scbSize32=0, it means */
   /* we are running standard mode, so we need to restore busy target array. */
   if(!state->SCSI_SF_scbRamSel && !state->SCSI_SF_scbSize32)
   {
      scbptr = OSD_INEXACT8(SCSI_AICREG(SCSI_SCBPTR));   /* Save curr scb ptr */
      for(i = 0; i < (SCSI_MAXSCBS+1); i++)
      {
         OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCBPTR), i);
         OSD_OUTEXACT8(SCSI_AICREG(SCSI_hBTATABLE(hhcb)), state->Bta[i]);
      }
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCBPTR), scbptr); /* restore cur scb ptr */
   }


#if POWER_MANAGEMENT_SAVE_RESTORE   

   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSISEQ), state->scsiseq);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SXFRCTL0), state->sxfrctl0);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SXFRCTL1), state->sxfrctl1);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SIMODE0), state->simode0);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SIMODE1), state->simode1);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SBLKCTL), state->sblkctl);
   SCSI_hUPDATEDATAFIFOTHRSH(hhcb);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_DFCNTRL), state->dfcntrl);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCAMCTRL), state->scamctrl);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSISIG), state->scsisig);
   OSD_WRITE_PCICFG_EXACT32(hhcb,SCSI_STATUS_CMD_REG, state->status_cmd_register);
   OSD_WRITE_PCICFG_EXACT32(hhcb,SCSI_HDR_TYPE_REG, state->hdr_type);
   
   /* Restore CACHETHEN bit */
   if(hhcb->SCSI_HF_cacheThEn)
   {
      /* Enable cache line streaming */
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_COMMAND),
         (OSD_INEXACT8(SCSI_AICREG(SCSI_COMMAND)) | SCSI_CACHETHEN)); 
   }
   else
   {
      /* Disable cache line streaming */
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_COMMAND),
         (OSD_INEXACT8(SCSI_AICREG(SCSI_COMMAND)) & ~SCSI_CACHETHEN)); 
   }
   
   

#if (SCSI_STANDARD_MODE || SCSI_SWAPPING_ADVANCED_MODE || SCSI_SWAPPING_160M_MODE)

   firmware_mode =  hhcb->firmwareMode;

   if (firmware_mode == SCSI_FMODE_STANDARD64 ||
      firmware_mode == SCSI_FMODE_STANDARD32 ||
      firmware_mode == SCSI_FMODE_STANDARD_ADVANCED ||
      firmware_mode == SCSI_FMODE_STANDARD_160M ||
      firmware_mode == SCSI_FMODE_SWAPPING_160M ||
      firmware_mode == SCSI_FMODE_SWAPPING_ADVANCED)
   {
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_CCHHADDR0), state->cchhaddr0);
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_CCHHADDR1), state->cchhaddr1);
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_CCHHADDR2), state->cchhaddr2);
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_CCHHADDR3), state->cchhaddr3);
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_CCSCBCTL), state->ccscbctl);
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_CCSGCTL), state->ccsgctl);
   }
#endif

   /* clear this for good measure */
   
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_CLRSINT0), 0xff);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_CLRSINT1), 0xff);


   /* force negotiation */
   
   for (j = 0; j < hhcb->maxDevices; j++)   
   {
         /*  Set negotiation needed */
         SCSI_hXFEROPTASSIGN(hhcb,j,SCSI_NEEDNEGO);

   }
#if SCSI_UPDATE_TERMINATION_ENABLE
   /* update termination that controlled by external hardware logic */
   SCSI_hUPDATEEXTTERM(hhcb);
#endif
#else   

   if ((((hhcb->deviceID & SCSI_ID_MASK) == SCSI_EXCALIBUR_BASE) && 
      (hhcb->hardwareMode == SCSI_HMODE_AIC78XX)) 
      && hhcb->hardwareRevision == 3)
   {
     command_reg = OSDReadPciConfiguration(hhcb,SCSI_STATUS_CMD_REG);
      if (state->SCSI_SF_mwiBit)
      {
         command_reg |= SCSI_MWRICEN;
      }
      else
      {
         command_reg &= ~SCSI_MWRICEN;
      }
      OSDWritePciConfiguration(hhcb,SCSI_STATUS_CMD_REG,command_reg);   
   }
#endif

#if POWER_MANAGEMENT_SAVE_RESTORE
      OSDWritePciConfiguration(hhcb,SCSI_STATUS_CMD_REG,state->status_cmd_register);   
#endif
   
   /* Reset Sequencer Start address */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SEQADDR0), 
         (SCSI_UEXACT8)SCSI_hIDLE_LOOP_ENTRY(hhcb) >> 2);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SEQADDR1), 
         (SCSI_UEXACT8)SCSI_hIDLE_LOOP_ENTRY(hhcb) >> 10);

#if SCSI_TARGET_OPERATION  
   /* restore the selection enable */
   if (state->SCSI_SF_selectionEnable)
   {
      /* selection-in enable */
      SCSIhTargetModeEnable(hhcb);
   }
   else
   {
      /* reset selection-in enable */
      SCSIHTargetModeDisable(hhcb);
   }

#endif  /* SCSI_TARGET_OPERATION */ 

#if !POWER_MANAGEMENT_SAVE_RESTORE

    
   /* Restore pause bit and inten bit of HCNTRL */
   if (state->SCSI_SF_interruptEnable)
   {
      hcntrl |= SCSI_INTEN;
   }
   else
   {
      hcntrl &= ~SCSI_INTEN;
   }
   if (state->SCSI_SF_chipPause)
   {
      hcntrl |= SCSI_PAUSEACK;
   }
   else
   {
      hcntrl &= ~SCSI_PAUSEACK;
   }

   /* a case where disabled BIOS sets this bit */
   
   hcntrl &= ~SCSI_CHIPRESET;

   SCSI_hWRITE_HCNTRL(hhcb,scsiRegister,hcntrl);

#else
   SCSI_hWRITE_HCNTRL(hhcb,scsiRegister,state->hcntrl);
#endif

   OSD_SYNCHRONIZE_IOS(hhcb);
}
#endif

/*********************************************************************
*
*  SCSIHPowerManagement
*
*     Execute power management function
*
*  Return Value:  0 - function execute successfully
*                 1 - function not supported
*                 others, to be defined
*                 
*                  
*  Parameters:    hhcb
*                 powerMode
*
*  Remarks:       There should be no active hiob when this routine
*                 get called.
*                  
*********************************************************************/
SCSI_UINT SCSIHPowerManagement (SCSI_HHCB SCSI_HPTR hhcb,
                                SCSI_UINT powerMode)
{
   if (!SCSIHCheckHostIdle(hhcb))
   {
      /* indicate host device busy (no power manageemnt allowed) */
      return(2);
   }
   else
   {
      /* this function is not supported at the moment */
      return(1);
   }
}

/*********************************************************************
*
*  SCSIHApplyDeviceTable
*
*     Apply new device table which may has been modified by OSM
*
*  Return Value:  0 - device table updated successfully
*                 others, failure
*                  
*  Parameters:    hhcb
*
*  Remarks:       Usually this routine is called after protocol
*                 automatic configuration. The device table may
*                 have been updated by OSD in the process of 
*                 protocol automatic configuration. After the
*                 protocol automatic configuration OSD may have 
*                 to update fields like disconnectEnable and
*                 scsiOption because scsi id has been assigned to
*                 different target/host devices (if scam protocol
*                 is enabled).
*                  
*********************************************************************/
#if defined(SCSI_INTERNAL)
int SCSIHApplyDeviceTable (SCSI_HHCB SCSI_HPTR hhcb)
{
   return(0);
}
#endif /* defined(SCSI_INTERNAL) */
                          
/*********************************************************************
*
*  SCSIHCheckHostIdle
*
*     Get the Check host device is idle
*
*  Return Value:  1 - host device is idle
*                 0 - host is busy 
*                  
*  Parameters:    hhcb
*
*  Remarks:       This routine will actually handle interrupt event
*                 though it was cleared by SCSIHFrontEndISR.
*
*********************************************************************/
int SCSIHCheckHostIdle (SCSI_HHCB SCSI_HPTR hhcb)
{
   SCSI_UEXACT8 i;

   /* check if there is any outstanding request in active */
   /* pointer array */
   for (i = 0; i < hhcb->numberScbs; i++)
   {
      if (SCSI_ACTPTR[i] != SCSI_NULL_HIOB)
      {
         return(0);
      }
   }

   return(1);
}

/*********************************************************************
*
*  SCSIHCheckDeviceIdle
*
*     Check device is idle
*
*  Return Value:  1 - device is idle
*                 0 - device is busy 
*                  
*  Parameters:    hhcb
*                 targetID
*
*  Remarks:       
*
*********************************************************************/
int SCSIHCheckDeviceIdle (SCSI_HHCB SCSI_HPTR hhcb,
                          SCSI_UEXACT8 targetID)
{
   SCSI_UEXACT8 i;
   SCSI_HIOB SCSI_IPTR hiobNew;

   /* check if there is any outstanding request in active */
   /* pointer array */
   for (i = 0; i < hhcb->numberScbs; i++)
   {
      if ((hiobNew = SCSI_ACTPTR[i]) != SCSI_NULL_HIOB)
      {
#if SCSI_TARGET_OPERATION
         if (hiobNew->SCSI_IP_targetMode)
         {
            /* target mode hiobs do not affect idless of
             * device - so skip over them.
             */
            continue;
         }
#endif /* SCSI_TARGET_OPERATION */

         if (((SCSI_UNIT_CONTROL SCSI_UPTR) 
               SCSI_TARGET_UNIT(hiobNew))->scsiID == targetID)
         {
            return(0);
         }
      }
   }

   return(1);
}

/*********************************************************************
*
*  SCSIhAbortTask
*
*     Abort task associated
*
*  Return Value:  none
*                  
*  Parameters:    hiob
*
*  Remarks:       There is no guarantee the aborted hiob is still
*                 in hardware management layer.
*
*********************************************************************/
void SCSIhAbortTask (SCSI_HIOB SCSI_IPTR hiob)
{
   SCSI_HIOB SCSI_IPTR hiobAbort = SCSI_RELATED_HIOB(hiob);
   SCSI_HHCB SCSI_HPTR hhcb = SCSI_TARGET_UNIT(hiobAbort)->hhcb;
   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_UEXACT8 hcntrl;
   SCSI_UEXACT8 i;

   /* Save the HCNTRL value */
   hcntrl = OSD_INEXACT8(SCSI_AICREG(SCSI_HCNTRL));

   /* Pause the chip if necessary */
   if (!(hcntrl & SCSI_PAUSEACK))
   {
      SCSIhPauseAndWait(hhcb);
   }

   /* Check all HIOBs in the active pointer array */
   /* for the match of aborting HIOB. */
   for (i = 0; i < hhcb->numberScbs; i++)
   {
      if (hiobAbort == SCSI_ACTPTR[i])
      {
         /* If aborting HIOB found in Done queue then the HIOB */
         /* with SCSI_CMD_ABORT_TASK command code returns with */
         /* a status of SCSI_HOST_ABT_NOT_FND. */
         /* The sequencer must be paused before call this macro */
         SCSI_hABORTHIOB(hhcb, hiobAbort, SCSI_HOST_ABT_HOST);

         /* Check if found in Done queue */
         if (hiobAbort->SCSI_IP_mgrStat == SCSI_MGR_DONE)
         {
            hiob->stat = SCSI_SCB_ERR;
            hiob->haStat = SCSI_HOST_ABT_CMDDONE;
         }

         else if (hiobAbort->SCSI_IP_mgrStat == SCSI_MGR_DONE_ABT)
         {
            hiob->stat = SCSI_SCB_COMP;
         }

         else if (hiobAbort->SCSI_IP_mgrStat == SCSI_MGR_ABORTINREQ)
         {
            hiob->stat = SCSI_SCB_PENDING;
            hiob->haStat = SCSI_HOST_ABT_STARTED;
         }
         break;
      }
   }

   if (i == hhcb->numberScbs)
   {
      hiob->stat = SCSI_SCB_ERR;
      hiob->haStat = SCSI_HOST_ABT_NOT_FND;
   }

   /* post back special HIOB to upper layer code */
   SCSI_COMPLETE_SPECIAL_HIOB(hiob);

   /* Restore HCNTRL if necessary */
   if (!(hcntrl & SCSI_PAUSEACK))
   {
      SCSI_hWRITE_HCNTRL(hhcb, scsiRegister, hcntrl);
   }
}

/*********************************************************************
*
*  SCSIhAbortTaskSet
*
*     Abort task set associated
*
*  Return Value:  none
*                  
*  Parameters:    hiob
*
*  Remarks:       There is no guarantee the aborted hiob is still
*                 in hardware management layer.
*
*********************************************************************/
void SCSIhAbortTaskSet (SCSI_HIOB SCSI_IPTR hiob)
{
   SCSI_HHCB SCSI_HPTR hhcb = SCSI_TARGET_UNIT(hiob)->hhcb;
   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_HIOB SCSI_IPTR hiobAbort;
   SCSI_UEXACT8 scsiID = SCSI_TARGET_UNIT(hiob)->scsiID;
   SCSI_UEXACT8 hcntrl;
   SCSI_UEXACT8 i;

   /* Save the HCNTRL value */
   hcntrl = OSD_INEXACT8(SCSI_AICREG(SCSI_HCNTRL));

   /* Pause the chip if necessary */
   if (!(hcntrl & SCSI_PAUSEACK))
   {
      SCSIhPauseAndWait(hhcb);
   }

   /* Check all HIOBs in the active pointer array for the match */
   /* of target SCSI ID. If so, abort each HIOB separately. */
   for (i = 0; i < hhcb->numberScbs; i++)
   {
      if ((hiobAbort = SCSI_ACTPTR[i]) != SCSI_NULL_HIOB)
      {
         if ((SCSI_TARGET_UNIT(hiobAbort)->hhcb == hhcb) &&
             (SCSI_TARGET_UNIT(hiobAbort)->scsiID == scsiID))
         {
            /* The sequencer must be paused before call this macro */
            SCSI_hABORTHIOB(hhcb, hiobAbort, SCSI_HOST_ABT_HOST);
         }
      }
   }

   hiob->stat = SCSI_SCB_COMP;

   /* post back special HIOB to upper layer code */
   SCSI_COMPLETE_SPECIAL_HIOB(hiob);

   /* Restore HCNTRL if necessary */
   if (!(hcntrl & SCSI_PAUSEACK))
   {
      SCSI_hWRITE_HCNTRL(hhcb, scsiRegister, hcntrl);
   }
}

/*********************************************************************
*
*  SCSIHSuppressNegotiation
*
*     Suppress wide/sync negotiation initiation
*
*  Return Value:  none
*                  
*  Parameters:    hiob
*
*  Remarks:       Suppress negotiation will be applied only if the
*                 HIM is in a state ready to initiate negotiation.
*
*                 This routine should only be called when device
*                 is idle
*
*********************************************************************/
void SCSIHSuppressNegotiation (SCSI_UNIT_CONTROL SCSI_UPTR targetUnit)
{
#if SCSI_AICBAYONET
   SCSI_REGISTER scsiRegister = targetUnit->hhcb->scsiRegister;
   SCSI_UEXACT8 hcntrl;
#endif /* SCSI_AICBAYONET */

   /* device must be idle */
   if (SCSIHCheckDeviceIdle(targetUnit->hhcb,targetUnit->scsiID))
   {
#if SCSI_AICBAYONET
      /* For Bayonet, the chip should be paused before accessing */
      /* scratch RAM or SCB array. */
      if ((targetUnit->hhcb->hardwareMode == SCSI_HMODE_AICBAYONET) &&
         (!((hcntrl = OSD_INEXACT8(SCSI_AICREG(SCSI_HCNTRL))) & SCSI_PAUSEACK)))
      {
         SCSIhPauseAndWait(targetUnit->hhcb);
      }
#endif /* SCSI_AICBAYONET */

      if (SCSI_hGETXFEROPTION(targetUnit->hhcb,targetUnit->scsiID) ==
         SCSI_NEEDNEGO)
      {
         /* turn off the negotiation initiation */
         SCSI_hXFEROPTASSIGN(targetUnit->hhcb,targetUnit->scsiID,0x00);
      }

      /* Set the SCSI RATE to defaults if SCSI_LOOPBACK_OPERATION = 1 */
      SCSI_hSETXFERRATE(targetUnit->hhcb,targetUnit->scsiID);

#if SCSI_AICBAYONET
      if ((targetUnit->hhcb->hardwareMode == SCSI_HMODE_AICBAYONET) &&
          (!(hcntrl & SCSI_PAUSEACK)))
      {
         OSD_OUTEXACT8(SCSI_AICREG(SCSI_HCNTRL), hcntrl);
      }
#endif /* SCSI_AICBAYONET */

      /* Set suppress negotiation flag */
      targetUnit->deviceTable->SCSI_DF_suppressNego = 1;
   }
}

/*********************************************************************
*
*  SCSIHForceNegotiation
*
*     Force wide/sync negotiation initiation
*
*  Return Value:  none
*                  
*  Parameters:    hiob
*
*  Remarks:       This routine should only be called when device
*                 is idle
*
*********************************************************************/
void SCSIHForceNegotiation (SCSI_UNIT_CONTROL SCSI_UPTR targetUnit)
{
#if SCSI_AICBAYONET
   SCSI_REGISTER scsiRegister = targetUnit->hhcb->scsiRegister;
   SCSI_UEXACT8 hcntrl;
#endif /* SCSI_AICBAYONET */
   SCSI_UEXACT8 uexact8Value;

   /* device must be idle */
   if (SCSIHCheckDeviceIdle(targetUnit->hhcb,targetUnit->scsiID))
   {
      /* turn on the negotiation initiation */
      if (targetUnit->deviceTable->scsiOption & (SCSI_WIDE_XFER |
         SCSI_SYNC_XFER))
      {
         uexact8Value = SCSI_NEEDNEGO;
      }
      else
      {
         uexact8Value = 0;
      }

#if SCSI_AICBAYONET
      /* For Bayonet, the chip should be paused before accessing scratch RAM */
      /* or SCB array */
      if ((targetUnit->hhcb->hardwareMode == SCSI_HMODE_AICBAYONET) &&
         (!((hcntrl = OSD_INEXACT8(SCSI_AICREG(SCSI_HCNTRL))) & SCSI_PAUSEACK)))
      {
         SCSIhPauseAndWait(targetUnit->hhcb);
      }
#endif /* SCSI_AICBAYONET */

      SCSI_hXFEROPTASSIGN(targetUnit->hhcb,targetUnit->scsiID,uexact8Value);

      /* Set the SCSI RATE to defaults if SCSI_LOOPBACK_OPERATION = 1 */
      SCSI_hSETXFERRATE(targetUnit->hhcb,targetUnit->scsiID);

#if SCSI_AICBAYONET
      if ((targetUnit->hhcb->hardwareMode == SCSI_HMODE_AICBAYONET) &&
          (!(hcntrl & SCSI_PAUSEACK)))
      {
         OSD_OUTEXACT8(SCSI_AICREG(SCSI_HCNTRL), hcntrl);
      }
#endif /* SCSI_AICBAYONET */

      /* Clear suppress negotiation flag */
      targetUnit->deviceTable->SCSI_DF_suppressNego = 0;
   }
}

/*********************************************************************
*
*  SCSIHGetHWInformation
*
*     Updates the SCSI_HW_INFORMATION structure with information from
*  the HA hardware.  An example usage is to get adapter/target profile
*  information for the CHIM interface.
*
*  Return Value:  none.
*                  
*  Parameters:    hwInfo      - SCSI_HW_INFORMATION structure.
*                 hhcb        - HHCB structure for the adapter.
*
*  Remarks:       None.
*                 
*********************************************************************/
#if SCSI_PROFILE_INFORMATION
void SCSIHGetHWInformation (SCSI_HW_INFORMATION SCSI_LPTR hwInfo, 
                            SCSI_HHCB SCSI_HPTR hhcb)                  
{
   SCSI_UEXACT8   i, j;
   SCSI_UEXACT8   busNumber, deviceNumber;
   SCSI_UEXACT32  baseAddress;
   SCSI_HOST_ADDRESS SCSI_LPTR hostAddress;
   register SCSI_REGISTER  scsiRegister = hhcb->scsiRegister;
   SCSI_UEXACT8 hcntrl;
   SCSI_UEXACT8 sblkctl; 

   /* save the HCNTRL value */
   hcntrl = OSD_INEXACT8(SCSI_AICREG(SCSI_HCNTRL));

   /* pause the chip if necessary */
   if (!(hcntrl & SCSI_PAUSEACK))
   {
      SCSIhPauseAndWait(hhcb);
   }

   sblkctl = OSD_INEXACT8(SCSI_AICREG(SCSI_SBLKCTL));

   /* Calculate WWID */
   hwInfo->WWID[0] = 
         (sblkctl & SCSI_SELWIDE)
         ? SCSI_UPTO_0F : SCSI_UPTO_07;

   hwInfo->WWID[0] |= SCSI_SNA;
   hwInfo->WWID[0] |= SCSI_VALID_BUT_UNASSIGN;

   /* set the current ID entry */
   hwInfo->WWID[1] |= hhcb->hostScsiID;

   /* set vendor identification to ADAPTEC */
   hwInfo->WWID[2] = 'A';
   hwInfo->WWID[3] = 'D';
   hwInfo->WWID[4] = 'A';
   hwInfo->WWID[5] = 'P';
   hwInfo->WWID[6] = 'T';
   hwInfo->WWID[7] = 'E';
   hwInfo->WWID[8] = 'C';
   hwInfo->WWID[9] = ' ';
 
   /* Generating the remaining portion of the WWID.               */
   hwInfo->WWID[10] = 'A';
   hwInfo->WWID[11] = 'I';
   hwInfo->WWID[12] = 'C';

   hwInfo->WWID[13] =
      (SCSI_UEXACT8)((hhcb->deviceID & 0xf000) >> 12) + '0';
   hwInfo->WWID[14] =
      (SCSI_UEXACT8)((hhcb->deviceID & 0x0f00) >> 8) + '0';
   hwInfo->WWID[15] =
      (SCSI_UEXACT8)((hhcb->deviceID & 0x00f0) >> 4) + '0';
   hwInfo->WWID[16] =
      (SCSI_UEXACT8)(hhcb->deviceID & 0x000f) + '0';

   /* PCI bus number is scanned from 0 to 255, so the boot/primary */
   /* host adapter will have lowest bus number.  But the id string */
   /* should start with higher value i.e. from 255 to 0 */
   hostAddress = OSD_GET_HOST_ADDRESS(hhcb);
   busNumber = (SCSI_UEXACT8)(0xff - hostAddress->pciAddress.busNumber);

   hwInfo->WWID[19] = (SCSI_UEXACT8)((busNumber % 10) + '0');
   busNumber /= 10;
   hwInfo->WWID[18] = (SCSI_UEXACT8)((busNumber % 10) + '0');
   hwInfo->WWID[17] = (SCSI_UEXACT8)((busNumber / 10) + '0');

   deviceNumber = (SCSI_UEXACT8)(0x1f - hostAddress->pciAddress.deviceNumber);
   hwInfo->WWID[21] = (SCSI_UEXACT8)((deviceNumber % 10) + '0');
   hwInfo->WWID[20] = (SCSI_UEXACT8)((deviceNumber / 10) + '0');

   /* Get base address from configuration space */
   baseAddress = OSDReadPciConfiguration(hhcb,SCSI_BASE_ADDR_REG) & 0xFFFFFFFE;

   for (i = 0; i < 8; i++)
   {
      j = ((SCSI_UEXACT8)(baseAddress >> (28 - i * 4))) & 0x0f;

      /* convert to ASCII character */
      hwInfo->WWID[22 + i] = (j >= 9) ? (j - 10) + 'A' : j + '0';
   }

   hwInfo->WWID[30] = hwInfo->WWID[31] = ' ';

   hwInfo->hostScsiID = hhcb->hostScsiID;
   hwInfo->SCSI_PF_scsiParity = hhcb->SCSI_HF_scsiParity;
   hwInfo->SCSI_PF_selTimeout = hhcb->SCSI_HF_selTimeout;

   hwInfo->SCSI_PF_separateRWThreshold = hhcb->SCSI_HF_separateRWThreshold;
   hwInfo->SCSI_PF_separateRWThresholdEnable = hhcb->SCSI_HF_separateRWThresholdEnable;

   if (hhcb->hardwareMode == SCSI_HMODE_AIC78XX)
   {
      hwInfo->threshold = hhcb->threshold; 
   }
   else
   {
      if (hwInfo->SCSI_PF_separateRWThresholdEnable)
      { 
         hwInfo->readThreshold = hhcb->threshold & 0x0F;
         hwInfo->writeThreshold = hhcb->threshold >> 4;
      }
      else 
      {
         /* assume both 4 bit fields are the same */
        hwInfo->threshold = hhcb->threshold & 0x0F; 
      }   
   }
      
   if (sblkctl & SCSI_SELWIDE)
   {
      hwInfo->SCSI_PF_wideSupport = 1;
   }
   else
   {
      hwInfo->SCSI_PF_wideSupport = 0;
   }

   for (i = 0; i < SCSI_MAXDEV; i++)
   {
      hwInfo->targetInfo[i].xferOption = SCSI_hGETXFEROPTION(hhcb,i);
      hwInfo->targetInfo[i].SCSI_TF_ultraEnable = SCSI_hGETFAST20REG(hhcb,i);
   }

   /* used by target mode operation */
   hwInfo->SCSI_PF_ultraCapable = hhcb->SCSI_HP_ultraCapable;

   SCSI_hGETPROFILEPARAMETERS(hwInfo, hhcb);

   SCSI_hGETTARGETIDMASK(hwinfo,hhcb);

   if ((hwInfo->intrThreshold = SCSI_hGETINTRTHRESHOLD(hhcb)) != (SCSI_UEXACT8)0xFF)
   {
      hwInfo->intrThresholdSupport = 1;
   }
   else
   {
      hwInfo->intrThresholdSupport = 0;
      hwInfo->intrThreshold        = 0;  /* To be consistent, we'll set     */ 
                                         /* intrThreshold to 0 when the f/w */
                                         /* does not support this feature.  */ 
   }

   hwInfo->SCSI_PF_cacheThEn = hhcb->SCSI_HF_cacheThEn;
   hwInfo->SCSI_PF_cacheThEnAdjustable = hhcb->SCSI_HF_cacheThEnAdjustable;   

   /* restore HCNTRL if necessary */
   if (!(hcntrl & SCSI_PAUSEACK))
   {
      SCSI_hWRITE_HCNTRL(hhcb,scsiRegister,hcntrl);
   }
}
#endif /* SCSI_PROFILE_INFORMATION */

/*********************************************************************
*
*  SCSIHPutHWInformation
*
*     Updates the the hardware with information from the SCSI_HW_INFORMATION
*  structure.  An example usage is to update adapter/target profile
*  information for the CHIM interface.
*
*  Return Value:  none.
*                  
*  Parameters:    hwInfo      - SCSI_HW_INFORMATION structure.
*                 hhcb        - HHCB structure for the adapter.
*
*  Remarks:       None.
*                 
*********************************************************************/
#if SCSI_PROFILE_INFORMATION
void SCSIHPutHWInformation (SCSI_HW_INFORMATION SCSI_LPTR hwInfo, 
                            SCSI_HHCB SCSI_HPTR hhcb)                  
{
   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_UEXACT8 sxfrctl1;
   SCSI_UEXACT8 hcntrl;

   /* save the HCNTRL value */
   hcntrl = OSD_INEXACT8(SCSI_AICREG(SCSI_HCNTRL));

   /* pause the chip if necessary */
   if (!(hcntrl & SCSI_PAUSEACK))
   {
      SCSIhPauseAndWait(hhcb);
   }

   OSD_OUTEXACT8(SCSI_AICREG(SCSI_hSCSIID(hhcb)), hwInfo->hostScsiID);
   hhcb->hostScsiID = hwInfo->hostScsiID;

   if (hhcb->hardwareMode == SCSI_HMODE_AIC78XX)
   {
      hhcb->threshold = hwInfo->threshold;
   }
   else
   {
      /* Update hhcb separate threshold status. */ 
      hhcb->SCSI_HF_separateRWThresholdEnable = hwInfo->SCSI_PF_separateRWThresholdEnable;
 
      if (hwInfo->SCSI_PF_separateRWThresholdEnable)
      { 
        hhcb->threshold = hwInfo->writeThreshold;
        hhcb->threshold = (hhcb->threshold << 4)| hwInfo->readThreshold; 
      }
      else 
      {
         /* Both read and write threshold values are the same */ 
        hhcb->threshold = hwInfo->threshold & 0x0F;  /* use lower 4 bits */ 
        hhcb->threshold = (hhcb->threshold << 4)| hwInfo->threshold;
      }   
   }

   /* Update hardware register */ 
   SCSI_hUPDATEDATAFIFOTHRSH(hhcb);
   
   /* Update CacheLineStreaming feature */
   hhcb->SCSI_HF_cacheThEn = hwInfo->SCSI_PF_cacheThEn;

   if(hhcb->SCSI_HF_cacheThEn)
   {
      /* Enable cache line streaming */
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_COMMAND),
         (OSD_INEXACT8(SCSI_AICREG(SCSI_COMMAND)) | SCSI_CACHETHEN)); 
   }
   else
   {
      /* Disable cache line streaming */
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_COMMAND),
         (OSD_INEXACT8(SCSI_AICREG(SCSI_COMMAND)) & ~SCSI_CACHETHEN)); 
   }

   /* Update scsi parity error checking and selection timeout */
   sxfrctl1 = OSD_INEXACT8(SCSI_AICREG(SCSI_SXFRCTL1));
   sxfrctl1 = (SCSI_UEXACT8)((sxfrctl1 & ~SCSI_ENSPCHK) |
                             (hwInfo->SCSI_PF_scsiParity << 5));
   sxfrctl1 = (SCSI_UEXACT8)((sxfrctl1 & ~(SCSI_STIMESEL1 + SCSI_STIMESEL0)) |
                             (hwInfo->SCSI_PF_selTimeout << 3));
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SXFRCTL1), sxfrctl1);
   hhcb->SCSI_HF_scsiParity = hwInfo->SCSI_PF_scsiParity;
   hhcb->SCSI_HF_selTimeout = hwInfo->SCSI_PF_selTimeout;

   /* load target id registers */
   SCSI_hLOADTARGETIDS(hhcb); 

   SCSI_hPUTPROFILEPARAMETERS(hwInfo, hhcb);

   SCSI_hSETINTRTHRESHOLD(hhcb, hwInfo->intrThreshold);

   /* restore HCNTRL if necessary */
   if (!(hcntrl & SCSI_PAUSEACK))
   {
      SCSI_hWRITE_HCNTRL(hhcb,scsiRegister,hcntrl);
   }
}
#endif /* SCSI_PROFILE_INFORMATION */

/*********************************************************************
*
*  SCSIHEnableExpStatus
*
*     Enable expander status examination
*
*  Return Value:  none
*                  
*  Parameters:    hhcb
*
*  Remarks:       * This routine should only be called when device
*                   is idle.
*                 * This routine will enable the break point
*                   for bus expander status examination.
*                 * After this call all following requests will cause
*                   the device table updated per bus expander status
*                    examined from bus.
*
*********************************************************************/
#if (SCSI_AICBAYONET || SCSI_AICTRIDENT)
void SCSIHEnableExpStatus (SCSI_HHCB SCSI_HPTR hhcb)
{
   if (hhcb->SCSI_HF_expSupport)
   {
      /* assert expander status request */
      hhcb->SCSI_HP_expRequest = 1;

      /* enable break point interrupt for expander status examination */
      /* pause the chip */
      SCSIhPauseAndWait(hhcb);
      
      /* enable and program the break address */
      SCSI_hSETBREAKPOINT(hhcb, (SCSI_UEXACT8)ENTRY_EXPANDER_BREAK);

      /* unpause the chip */
      SCSIhUnPause(hhcb);
   }
}
#endif

/*********************************************************************
*
*  SCSIHDisableExpStatus
*
*     Disable expander status examination
*
*  Return Value:  none
*                  
*  Parameters:    hhcb
*
*  Remarks:       This routine should only be called when device
*                 is idle
*                 This routine will disable the break point
*                 for bus expander status examination
*
*********************************************************************/
#if (SCSI_AICBAYONET || SCSI_AICTRIDENT)
void SCSIHDisableExpStatus (SCSI_HHCB SCSI_HPTR hhcb)
{

   if (hhcb->SCSI_HF_expSupport)
   {
      /* clear expander status request */
      hhcb->SCSI_HP_expRequest = 0;

      /* disable break point interrupt */
      /* pause the chip */
      SCSIhPauseAndWait(hhcb);

      /* disable break point */
      SCSI_hCLEARBREAKPOINT(hhcb, (SCSI_UEXACT8)ENTRY_EXPANDER_BREAK);

      /* unpause the chip */
      SCSIhUnPause(hhcb);
   }
}
#endif

#if SCSI_PAC_NSX_REPEATER
/*********************************************************************
*
*  SCSIHSuppressNegoAcitveTarget
*
*     Suppress wide/sync negotiation initiation
*     while the target is not in the idle state.
*
*  Return Value:  none
*                  
*  Parameters:    targetUnit
*
*  Remarks:       Suppress negotiation will be applied only if the
*                 HIM is in a state ready to initiate negotiation.
*
*********************************************************************/
void SCSIHSuppressNegoActiveTarget (SCSI_UNIT_CONTROL SCSI_UPTR targetUnit)
{
   SCSI_REGISTER scsiRegister = targetUnit->hhcb->scsiRegister;
   SCSI_HHCB SCSI_HPTR hhcb = targetUnit->hhcb;
   SCSI_UEXACT8 hcntrl;

   /* The sequencer should be paused because the set negotiation routine */
   /* will touch delivery queue index. */
   if (!((hcntrl = OSD_INEXACT8(SCSI_AICREG(SCSI_HCNTRL))) & SCSI_PAUSEACK))
   {
      SCSIhPauseAndWait(hhcb);
   }

   if (SCSI_hGETXFEROPTION(hhcb,targetUnit->scsiID) == SCSI_NEEDNEGO)
   {
      /* turn off the negotiation initiation only to those IOs that */
      /* have not sent to the target yet. */
      SCSI_hCHKCONDXFERASSIGN(hhcb,targetUnit->scsiID,0x00);
   }

   if (!(hcntrl & SCSI_PAUSEACK))
   {
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_HCNTRL), hcntrl);
   }

   /* Set suppress negotiation flag */
   targetUnit->deviceTable->SCSI_DF_suppressNego = 1;
}
#endif /* SCSI_PAC_NSX_REPEATER */

#if SCSI_PAC_NSX_REPEATER
/*********************************************************************
*
*  SCSIHForceNegoActiveTarget
*
*     Force wide/sync negotiation initiation to a target while
*     it's still in busy state.  
*
*  Return Value:  none
*                  
*  Parameters:    hhcb
*
*  Remarks:
*
*********************************************************************/
void SCSIHForceNegoActiveTarget (SCSI_HHCB SCSI_HPTR hhcb)
{
   SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_UEXACT8 hcntrl;
   SCSI_UEXACT8 i;

   /* The sequencer should be paused because the set negotiation routine */
   /* will touch delivery queue index. */
   if (!((hcntrl = OSD_INEXACT8(SCSI_AICREG(SCSI_HCNTRL))) & SCSI_PAUSEACK))
   {
      SCSIhPauseAndWait(hhcb);
   }

   /* Re-Initialize sync/wide negotiation parameters depending on */
   /* whether SuppressNego flag is set                            */
   for (i = 0; i < hhcb->maxDevices; i++)   
   {
      if (SCSI_DEVICE_TABLE(hhcb)[i].scsiOption &
          (SCSI_WIDE_XFER | SCSI_SYNC_XFER))
      {
         /*  Set negotiation needed */
         SCSI_hCHKCONDXFERASSIGN(hhcb,i,SCSI_NEEDNEGO);
      }
      else
      {
         SCSI_hCHKCONDXFERASSIGN(hhcb,i,0x00);
      }         

      /* Clear suppress negotiation flag */
      SCSI_DEVICE_TABLE(hhcb)[i].SCSI_DF_suppressNego = 0;
   }

   if (!(hcntrl & SCSI_PAUSEACK))
   {
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_HCNTRL), hcntrl);
   }
}
#endif /* SCSI_PAC_NSX_REPEATER */

/***************************************************************************
*                                                                          *
* Copyright 1995,1996,1997,1998,1999 Adaptec, Inc.,  All Rights Reserved.  *
*                                                                          *
* This software contains the valuable trade secrets of Adaptec.  The       *
* software is protected under copyright laws as an unpublished work of     *
* Adaptec.  Notice is for informational purposes only and does not imply   *
* publication.  The user of this software may make copies of the software  *
* for use with parts manufactured by Adaptec or under license from Adaptec *
* and for no other use.                                                    *
*                                                                          *
***************************************************************************/

/***************************************************************************
*
*  Module Name:   HWMUTIL.C
*
*  Description:
*                 Codes to implement utility for hardware management module
*
*  Owners:        ECX IC Firmware Team
*    
*  Notes:         Utilities defined in this file should be generic and
*                 primitive.
*
*  Entry Point(s):
*                 SCSIHEnableExtScbRam
*                 SCSIHDisableExtScbRam
*                 SCSIHExtScbRAMAvail
*                 SCSIHReadScbRAM
*                 SCSIHWriteScbRAM
*                 SCSIHHardwareInResetState
*                 SCSIHReadScratchRam
*                 SCSIHWriteScratchRam
*                 SCSIHReadHADDR
*
***************************************************************************/

#include "scsi.h"
#include "hwm.h"

/*********************************************************************
*
*  SCSIhPauseAndWait
*
*     Pause and wait for pauseack
*
*  Return Value:  none
*
*  Parameters:    register access
*
*  Remarks:
*                  
*********************************************************************/
void SCSIhPauseAndWait (SCSI_HHCB SCSI_HPTR hhcb)
{
   SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_UINT32 i = 0;
      
   SCSI_hWRITE_HCNTRL(hhcb,scsiRegister,
                  OSD_INEXACT8(SCSI_AICREG(SCSI_HCNTRL)) | SCSI_PAUSE);
   while ((!(OSD_INEXACT8(SCSI_AICREG(SCSI_HCNTRL)) & SCSI_PAUSEACK)) &&
          (i++ < 400000))
   {
      OSD_TIMER(hhcb);
   }    

   /* workaround for hanging of sequencer immediately after a pause */
   /* due to hardware stretch                                       */
   SCSI_hSEQSTRETCHWORKAROUND(hhcb);
}

/*********************************************************************
*
*  SCSIhUnPause
*
*     UnPause chip
*
*  Return Value:  none
*
*  Parameters:    hhcb
*
*  Remarks:
*                  
*********************************************************************/
void SCSIhUnPause (SCSI_HHCB SCSI_HPTR hhcb)
{
   SCSI_REGISTER scsiRegister = hhcb->scsiRegister;

   SCSI_hWRITE_HCNTRL(hhcb,scsiRegister,
               OSD_INEXACT8(SCSI_AICREG(SCSI_HCNTRL)) & ~SCSI_PAUSE);
}

/*********************************************************************
*
*  SCSIHEnableExtScbRam
*
*     Enable external scb ram for standard mode
*
*  Return Value:  1 - external scb ram was enabled
*                 0 - external scb ram was disabled
*
*  Parameters:    hhcb
*
*  Remarks:       This routine should be included only if scb chainning
*                 is enabled.
*                  
*********************************************************************/
#if SCSI_BIOS_ASPI8DOS + SCSI_STANDARD_MODE + defined(SCSI_INTERNAL)
int SCSIHEnableExtScbRam (SCSI_HHCB SCSI_HPTR hhcb)
{
   int retValue = 0;
   SCSI_UEXACT32 value;
   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_UEXACT8 hcntrl;

   /* check if pause chip necessary */
   if (!((hcntrl = OSD_INEXACT8(SCSI_AICREG(SCSI_HCNTRL))) & SCSI_PAUSEACK))
   {
   
      SCSIhPauseAndWait(hhcb);      
   
   }

   /* read current register value */
   value = SCSI_hGETSCBRAMCTRLS(hhcb);

   /* check if ram present */
   if (value & SCSI_hRAMPSM(hhcb))
   {
      retValue = !(value & SCSI_hSCBRAMSEL(hhcb));
      if (!retValue)
      {
         /* turn on external scb ram */
         value &= ~((SCSI_UEXACT32) SCSI_hSCBRAMSEL(hhcb));
         SCSI_hPUTSCBRAMCTRLS(hhcb,value);
      }

      /* select ram bank based on PCI function number*/
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCBBADDR), 
            OSD_GET_HOST_ADDRESS(hhcb)->pciAddress.functionNumber);         
   }

   if (!(hcntrl & SCSI_PAUSEACK))
   {
      SCSI_hWRITE_HCNTRL(hhcb,scsiRegister,hcntrl);
   }

   return(retValue);
}
#endif /* SCSI_BIOS_ASPI8DOS + SCSI_STANDARD_MODE + defined(SCSI_INTERNAL) */

/*********************************************************************
*
*  SCSIHDisableExtScbRam
*
*     Disable external scb ram
*
*  Return Value:  1 - external scb ram was enabled
*                 0 - external scb ram was disabled
*
*  Parameters:    hhcb
*
*  Remarks:       This routine should be included only if scb chainning
*                 is enabled.
*                  
*********************************************************************/
#if SCSI_BIOS_ASPI8DOS + SCSI_STANDARD_MODE + defined(SCSI_INTERNAL)
int SCSIHDisableExtScbRam (SCSI_HHCB SCSI_HPTR hhcb)
{
   int retValue = 0;
   SCSI_UEXACT32 value;
   SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_UEXACT8 hcntrl;

   /* check if pause chip necessary */
   if (!((hcntrl = OSD_INEXACT8(SCSI_AICREG(SCSI_HCNTRL))) & SCSI_PAUSEACK))
   {
      SCSIhPauseAndWait(hhcb);      
   }

   /* read current register value */
   value = SCSI_hGETSCBRAMCTRLS(hhcb);

   /* check if ram present */
   if (value & SCSI_hRAMPSM(hhcb))
   {
      retValue = !(value & SCSI_hSCBRAMSEL(hhcb));
      if (retValue)
      {
         /* turn off external scb ram */
         value |= (SCSI_UEXACT32) SCSI_hSCBRAMSEL(hhcb);
         SCSI_hPUTSCBRAMCTRLS(hhcb,value);
      }
   }
   if (!(hcntrl & SCSI_PAUSEACK))
   {
      SCSI_hWRITE_HCNTRL(hhcb,scsiRegister,hcntrl);
   }
   return(retValue);
}
#endif /* SCSI_BIOS_ASPI8DOS + SCSI_STANDARD_MODE + defined(SCSI_INTERNAL) */

/*********************************************************************
*
*  SCSIhLoadSequencer
*
*     Load sequencer code
*
*  Return Value:  0 - sequencer code loaded and verified
*                 others - sequencer code verification failed
*
*  Parameters:    scsiRegister
*                 address of sequencer code to be loaded
*                 size of sequencer code 
*
*  Remarks:       none
*                  
*********************************************************************/
int SCSIhLoadSequencer (SCSI_REGISTER scsiRegister,
                        SCSI_UEXACT8 SCSI_LPTR seqCode,
                        int seqSize)
{
   int cnt;
   SCSI_UEXACT8 seqctl;

   seqctl = OSD_INEXACT8(SCSI_AICREG(SCSI_SEQCTL));   /* Save SEQCTL */

   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SEQCTL), SCSI_FAILDIS + SCSI_FASTMODE + SCSI_PERRORDIS + SCSI_LOADRAM);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SEQADDR0), 00);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SEQADDR1), 00);
   for (cnt = 0; cnt < seqSize; cnt++)
   {
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SEQRAM), seqCode[cnt]);
   }
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SEQCTL), SCSI_FAILDIS + SCSI_FASTMODE + SCSI_PERRORDIS);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SEQCTL), SCSI_FAILDIS + SCSI_FASTMODE + SCSI_PERRORDIS + SCSI_LOADRAM);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SEQADDR0), 00);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SEQADDR1), 00);
   for (cnt = 0; cnt < (int) seqSize; cnt++)
   {
      if (OSD_INEXACT8(SCSI_AICREG(SCSI_SEQRAM)) != seqCode[cnt])
      {
        return(-1);
      }
   }
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SEQCTL), SCSI_FAILDIS + SCSI_FASTMODE + SCSI_PERRORDIS);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SEQCTL), seqctl);   /* Restore SEQCTL */

   return(0);
}

/*********************************************************************
*
*  SCSIhWait4Req
*
*     Wait for target to assert REQ.
*
*  Return Value:  current SCSI bus phase
*                 -1 - error indication
*
*  Parameters     hhcb
*
*  Remarks:       bypasses sequencer
*
*********************************************************************/
SCSI_UEXACT8 SCSIhWait4Req (SCSI_HHCB SCSI_HPTR hhcb)
{
   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_UEXACT8 stat;
   SCSI_UEXACT8 phase;
   SCSI_UINT32 i = 0;
   SCSI_UINT32 count;

   /* The usual 0x800000 count has been reduced to 0x80000, as 0x800000 turns */
   /* out to be more than 2 minutes for this outer loop in Jalapeno board. */
   count = 0x80000;
   
   while(1)
   {

      /* Bus is stuck and we cannot clear error */
      if (count-- == 0)
      {
         return((SCSI_UEXACT8)-1);
      }

      while ((OSD_INEXACT8(SCSI_AICREG(SCSI_SCSISIG)) & SCSI_ACKI) &&
             (i++ < 400000))
      {
         OSD_TIMER(hhcb);
      }

      i = 0;
      while (((stat = OSD_INEXACT8(SCSI_AICREG(SCSI_SSTAT1))) & SCSI_REQINIT) == 0) 
      {
         if (   (stat & (SCSI_BUSFREE | SCSI_SCSIRSTI))
             || (i++ >= 400000))           /* time out check */
         {
            return((SCSI_UEXACT8)-1);
         }
         OSD_TIMER(hhcb);
      }
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_CLRSINT1), SCSI_CLRSCSIPERR);
      phase = (SCSI_UEXACT8)(OSD_INEXACT8(SCSI_AICREG(SCSI_SCSISIG)) & SCSI_BUSPHASE);
      if ((phase & SCSI_IOI) &&
          (phase != SCSI_DIPHASE) &&
          (OSD_INEXACT8(SCSI_AICREG(SCSI_SSTAT1)) & SCSI_SCSIPERR))
      {
         OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSISIG), phase);
         OSD_INEXACT8(SCSI_AICREG(SCSI_SCSIDATL));
      }
      else
         break;
   }
   return(phase);
}

/*********************************************************************
*
*  SCSIhDelayCount500us
*
*     Delay multiple of 500usec
*
*  Return Value:  none
*
*  Parameters     hhcb
*                 count of 500usec
*
*  Remarks:       bypasses sequencer
*
*********************************************************************/
void SCSIhDelayCount500us (SCSI_HHCB SCSI_HPTR hhcb,
                           SCSI_UEXACT32 count)
{
   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_UEXACT8 timer0, timer1;
   SCSI_UEXACT16 msec;
   SCSI_UEXACT32 i;
   SCSI_UEXACT8 hcntrl;

   if (!count)
      return;                    /* use this for deskewing */

   if (!((hcntrl = OSD_INEXACT8(SCSI_AICREG(SCSI_HCNTRL))) & SCSI_PAUSEACK))
   {
      SCSIhPauseAndWait(hhcb);      
   }

   msec = SCSI_hDELAYFACTOR(hhcb);        /* setup delay for 500 msec */
   timer0 = OSD_INEXACT8(SCSI_AICREG(SCSI_hRESCNT_BASE(hhcb)+0));         /* save SCB16 and SCB17 */
   timer1 = OSD_INEXACT8(SCSI_AICREG(SCSI_hRESCNT_BASE(hhcb)+1));
   for (i = 0; i != count ; i++)
   {
      SCSIhPauseAndWait(hhcb);
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_hRESCNT_BASE(hhcb)+0), (SCSI_UEXACT8)msec);          /* set timer to 64 usec */
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_hRESCNT_BASE(hhcb)+1), (SCSI_UEXACT8)(msec >> 8));
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SEQADDR0), (SCSI_UEXACT8) SCSI_hATN_TMR_ENTRY(hhcb) >> 2);
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SEQADDR1), (SCSI_UEXACT8) SCSI_hATN_TMR_ENTRY(hhcb) >> 10);
      
      SCSIhUnPause(hhcb);                 /* unpause sequencer */

      while(1)
      {
         if (OSD_INEXACT8(SCSI_AICREG(SCSI_HCNTRL)) & SCSI_PAUSEACK)
         {
            if (SCSI_hREAD_INTSTAT(hhcb,scsiRegister) &
               (SCSI_SCSIINT | SCSI_SEQINT))
            {
               break;
            }
         }
      }

      SCSI_hSEQSTRETCHWORKAROUND(hhcb);
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_CLRSINT1), SCSI_CLRSCSIRSTI);
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_CLRINT),
         SCSI_CLRSCSINT | SCSI_CLRSEQINT);
   }
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_hRESCNT_BASE(hhcb)+0), timer0);              /* restore SCB16 & SCB17 */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_hRESCNT_BASE(hhcb)+1), timer1);

   if (!(hcntrl & SCSI_PAUSEACK))
   {
      SCSI_hWRITE_HCNTRL(hhcb,scsiRegister,hcntrl);
   }
}

/*********************************************************************
*
*  SCSIhSetupAssignScbBuffer
*
*     Setup for assign scb buffer from builtin scb buffer pool
*
*  Return Value:  none
*
*  Parameters     hhcb
*                 hiob
*
*  Remarks:       
*
*********************************************************************/
#if SCSI_SCBBFR_BUILTIN || SCSI_SWAPPING_MODE
void SCSIhSetupAssignScbBuffer (SCSI_HHCB SCSI_HPTR hhcb)
{
   hhcb->SCSI_SCB_BUFFER.busAddress = OSD_GET_BUS_ADDRESS(hhcb,
               SCSI_MC_LOCKED,hhcb->SCSI_SCB_BUFFER.virtualAddress);
   hhcb->SCSI_SCB_BUFFER.bufferSize =
      SCSI_hSIZEOFSCBBUFFER(hhcb->firmwareMode) * hhcb->numberScbs;
}
#endif
/*********************************************************************
*
*  SCSIhSwappingAssignScbBuffer
*
*     Assign scb buffer from builtin scb buffer pool
*
*  Return Value:  none
*
*  Parameters     hhcb
*                 hiob
*
*  Remarks:       
*
*********************************************************************/
#if !SCSI_RESOURCE_MANAGEMENT
#if SCSI_SCBBFR_BUILTIN && SCSI_SWAPPING_MODE
void SCSIhSwappingAssignScbBuffer (SCSI_HHCB SCSI_HPTR hhcb,
                           SCSI_HIOB SCSI_IPTR hiob)
{
   SCSI_UEXACT8 SCSI_HPTR virtual_Address;

   virtual_Address = hhcb->SCSI_SCB_BUFFER.virtualAddress;
   hiob->scbDescriptor->scbBuffer.virtualAddress = 
       &(virtual_Address[hiob->scbDescriptor->scbNumber * 
            SCSI_hSIZEOFSCBBUFFER(hhcb->firmwareMode)]);

   hiob->scbDescriptor->scbBuffer.busAddress = hhcb->SCSI_SCB_BUFFER.busAddress;
         
   OSD_ADJUST_BUS_ADDRESS(hhcb,&(hiob->scbDescriptor->scbBuffer.busAddress),
         ((SCSI_UINT)hiob->scbDescriptor->scbNumber * SCSI_hSIZEOFSCBBUFFER(hhcb->firmwareMode)));

   hiob->scbDescriptor->scbBuffer.bufferSize = SCSI_hSIZEOFSCBBUFFER(hhcb->firmwareMode);
}
#endif /* SCSI_SCBBFR_BUILTIN && SCSI_SWAPPING_MODE */
#endif /* !SCSI_RESOURCE_MANAGEMENT */

/*********************************************************************
*
*  SCSIhStandardAssignScbBuffer
*
*     Assign scb buffer from builtin scb buffer pool
*
*  Return Value:  none
*
*  Parameters     hhcb
*                 hiob
*
*  Remarks:       
*
*********************************************************************/
#if !SCSI_RESOURCE_MANAGEMENT
#if SCSI_SCBBFR_BUILTIN && SCSI_STANDARD_MODE 
void SCSIhStandardAssignScbBuffer (SCSI_HHCB SCSI_HPTR hhcb,
                           SCSI_HIOB SCSI_IPTR hiob)
{
   SCSI_UEXACT8 SCSI_HPTR virtual_Address;

   virtual_Address = hhcb->SCSI_SCB_BUFFER.virtualAddress;
   hiob->scbDescriptor->scbBuffer.virtualAddress = 
       &(virtual_Address[hiob->scbDescriptor->scbNumber * 
            SCSI_hSIZEOFSCBBUFFER(hhcb->firmwareMode)]);

   hiob->scbDescriptor->scbBuffer.busAddress = hhcb->SCSI_SCB_BUFFER.busAddress;
         
   OSD_ADJUST_BUS_ADDRESS(hhcb,&(hiob->scbDescriptor->scbBuffer.busAddress),
         ((SCSI_UINT)hiob->scbDescriptor->scbNumber * SCSI_hSIZEOFSCBBUFFER(hhcb->firmwareMode)));

   hiob->scbDescriptor->scbBuffer.bufferSize = SCSI_hSIZEOFSCBBUFFER(hhcb->firmwareMode);

   /* Need to assign scb buffer for queueNext scb */
   hiob->scbDescriptor->queueNext->scbBuffer.busAddress =
      hhcb->SCSI_SCB_BUFFER.busAddress;
         
   OSD_ADJUST_BUS_ADDRESS(hhcb,&(hiob->scbDescriptor->queueNext->scbBuffer.busAddress),
         ((SCSI_UINT)hiob->scbDescriptor->queueNext->scbNumber * SCSI_hSIZEOFSCBBUFFER(hhcb->firmwareMode)));

   hiob->scbDescriptor->queueNext->scbBuffer.bufferSize = SCSI_hSIZEOFSCBBUFFER(hhcb->firmwareMode);
   
}
#endif /* SCSI_SCBBFR_BUILTIN && SCSI_STANDARD_MODE */
#endif /* !SCSI_RESOURCE_MANAGEMENT */

/*********************************************************************
*
*  SCSIHExtScbRAMAvail
*
*     Check if external scb RAM hardware available
*
*  Return Value:  1 - available
*                 0 - not available
*
*  Parameters     hhcb
*
*  Remarks:       This routine is designed help OSD to configure
*                 operating modes based on whether external harwdare
*                 scb RAM available or not. Usually the standard 32/64
*                 mode should be used when external scb ram is available.
*                 The swapping 32/64 mode should be used when there is
*                 no external scb ram hardware. It's up to the OSD
*                 to deside which modes get used even if it's against
*                 rules mentioned above.
*
*                 This routine requires access to PCI configuration
*                 space.
*
*********************************************************************/
#if SCSI_STANDARD_MODE || defined(SCSI_INTERNAL)
SCSI_INT SCSIHExtScbRAMAvail (SCSI_HHCB SCSI_HPTR hhcb)
{
   SCSI_INT value;
   SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_UEXACT8 hcntrl;

   /* check if pause chip necessary */
   if (!((hcntrl = OSD_INEXACT8(SCSI_AICREG(SCSI_HCNTRL))) & SCSI_PAUSEACK))
   {
      SCSIhPauseAndWait(hhcb);      
   }
   /* check if external ram present */
   value = ((SCSI_hGETSCBRAMCTRLS(hhcb) & SCSI_hRAMPSM(hhcb)) != 0);
   if (!(hcntrl & SCSI_PAUSEACK))
   {
      SCSI_hWRITE_HCNTRL(hhcb,scsiRegister,hcntrl);
   }
   return(value);
}
#endif /* SCSI_STANDARD_MODE || defined(SCSI_INTERNAL) */

/*********************************************************************
*
*  SCSIHReadScbRAM
*
*     Read from scb RAM hardware
*
*  Return Value:  none
*
*  Parameters     hhcb
*                 data buffer to be filled
*                 scb number
*                 offset within specified scb
*                 length of bytes to be read
*
*  Remarks:       This routine is designed for OSD to access scb
*                 hardware memory. OSD must have special knowledge
*                 about information stored in scb ram. Usually this
*                 routine is required only for software which is
*                 compatible with ADAPTEC BIOS.
*
*********************************************************************/
#if SCSI_ACCESS_RAM
void SCSIHReadScbRAM (SCSI_HHCB SCSI_HPTR hhcb,
                      SCSI_UEXACT8 SCSI_SPTR dataBuffer,
                      SCSI_UEXACT8 scbNumber,
                      SCSI_UEXACT8 scbOffset,
                      SCSI_UEXACT8 byteLength)
{
   SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_UEXACT8 hcntrl;
   SCSI_UEXACT8 scbptr;
   SCSI_UEXACT8 i;

   /* save the HCNTRL value */
   hcntrl = OSD_INEXACT8(SCSI_AICREG(SCSI_HCNTRL));

   /* pause the chip if necessary */
   if (!(hcntrl & SCSI_PAUSEACK))
   {
      SCSIhPauseAndWait(hhcb);
   }

   /* save SCBPTR value */
   scbptr = OSD_INEXACT8(SCSI_AICREG(SCSI_SCBPTR));

   /* set scbNumber to SCBPTR */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCBPTR),scbNumber);

   /* read from scb RAM */
   for (i = 0; i < byteLength; i++)
   {
      dataBuffer[i] = OSD_INEXACT8(SCSI_AICREG(SCSI_SCB_BASE+scbOffset+i));
   }

   /* restore SCBPTR */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCBPTR),scbptr);

   /* restore HCNTRL if necessary */
   if (!(hcntrl & SCSI_PAUSEACK))
   {
      SCSI_hWRITE_HCNTRL(hhcb,scsiRegister,hcntrl);
   }
}
#endif

/*********************************************************************
*
*  SCSIHWriteScbRAM
*
*     Write to scb RAM hardware
*
*  Return Value:  none
*
*  Parameters     hhcb
*                 scb number
*                 offset within specified scb
*                 length of bytes to be read
*                 data buffer to be filled
*
*  Remarks:       This routine is designed for OSD to access scb
*                 hardware memory. OSD must have special knowledge
*                 about information stored in scb ram. Usually this
*                 routine is required only for software which is
*                 compatible with ADAPTEC BIOS.
*
*********************************************************************/
#if SCSI_ACCESS_RAM
void SCSIHWriteScbRAM (SCSI_HHCB SCSI_HPTR hhcb,
                       SCSI_UEXACT8 scbNumber,
                       SCSI_UEXACT8 scbOffset,
                       SCSI_UEXACT8 byteLength,
                       SCSI_UEXACT8 SCSI_SPTR dataBuffer)
{
   SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_UEXACT8 hcntrl;
   SCSI_UEXACT8 scbptr;
   SCSI_UEXACT8 i;

   /* save the HCNTRL value */
   hcntrl = OSD_INEXACT8(SCSI_AICREG(SCSI_HCNTRL));

   /* pause the chip if necessary */
   if (!(hcntrl & SCSI_PAUSEACK))
   {
      SCSIhPauseAndWait(hhcb);
   }

   /* save SCBPTR value */
   scbptr = OSD_INEXACT8(SCSI_AICREG(SCSI_SCBPTR));

   /* set scbNumber to SCBPTR */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCBPTR),scbNumber);

   /* write to scb RAM */
   for (i = 0; i < byteLength; i++)
   {
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+scbOffset+i),dataBuffer[i]);
   }

   /* restore SCBPTR */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCBPTR),scbptr);

   /* restore HCNTRL if necessary */
   if (!(hcntrl & SCSI_PAUSEACK))
   {
      SCSI_hWRITE_HCNTRL(hhcb,scsiRegister,hcntrl);
   }
}
#endif

/*********************************************************************
*
*  SCSIHHardwareInResetState
*
*     Check if harwdare is in chip reset state
*
*  Return Value:  1, chip is in reset state
*                 0, chim is not in reset state
*
*  Parameters     hhcb
*
*  Remarks:       This routine is designed for upper layer to check
*                 if hardware has ever been initialized. It's typically
*                 used internallly by upper layers implemented by
*                 Adaptec.
*
*********************************************************************/
#if SCSI_ACCESS_RAM
SCSI_INT SCSIHHardwareInResetState (SCSI_HHCB SCSI_HPTR hhcb)
{
   SCSI_REGISTER scsiRegister = hhcb->scsiRegister;

   return((OSD_INEXACT8(SCSI_AICREG(SCSI_HCNTRL)) & SCSI_CHIPRESET) != 0);
}
#endif

/*********************************************************************
*
*  SCSIHReadScratchRam
*
*     Read from scratch ram hardware
*
*  Return Value:  none
*
*  Parameters     hhcb
*                 dataBuffer
*                 offset
*                 length
*
*  Remarks:       This routine is designed for upper layer to check
*                 if hardware has ever been initialized. It's typically
*                 used internallly by upper layers implemented by
*                 Adaptec.
*
*********************************************************************/
void SCSIHReadScratchRam (SCSI_HHCB SCSI_HPTR hhcb,
                          SCSI_UEXACT8 SCSI_SPTR dataBuffer,
                          int offset,
                          int length)
{
   SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_UEXACT8 hcntrl;
   int i;
   SCSI_UEXACT8 scratchAddress;

   /* save the HCNTRL value */
   hcntrl = OSD_INEXACT8(SCSI_AICREG(SCSI_HCNTRL));

   /* pause the chip if necessary */
   if (!(hcntrl & SCSI_PAUSEACK))
   {
      SCSIhPauseAndWait(hhcb);
   }

   if (offset < SCSI_SCRATCH0_SIZE)
   {
      scratchAddress = SCSI_SCRATCH0 + offset;
   }
   else
   {
      scratchAddress = SCSI_SCRATCH1 + (offset - SCSI_SCRATCH0_SIZE);
   }

   /* read from scratch RAM */
   for (i = 0; i < length; i++)
   {
      dataBuffer[i] = OSD_INEXACT8(SCSI_AICREG(scratchAddress+i));
   }

   /* restore HCNTRL if necessary */
   if (!(hcntrl & SCSI_PAUSEACK))
   {
      SCSI_hWRITE_HCNTRL(hhcb,scsiRegister,hcntrl);
   }
}

/*********************************************************************
*
*  SCSIHWriteScratchRam
*
*     Write to scratch ram hardware
*
*  Return Value:  none
*
*  Parameters     hhcb
*                 offset
*                 length
*                 dataBuffer
*
*  Remarks:       This routine is designed for upper layer to check
*                 if hardware has ever been initialized. It's typically
*                 used internallly by upper layers implemented by
*                 Adaptec.
*
*********************************************************************/
#if SCSI_STANDARD64_MODE || defined(SCSI_INTERNAL) 
void SCSIHWriteScratchRam (SCSI_HHCB SCSI_HPTR hhcb,
                           int offset,
                           int length,
                           SCSI_UEXACT8 SCSI_SPTR dataBuffer)
{
   SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_UEXACT8 hcntrl;
   SCSI_UEXACT8 scratchAddress;
   int i;

   /* save the HCNTRL value */
   hcntrl = OSD_INEXACT8(SCSI_AICREG(SCSI_HCNTRL));

   /* pause the chip if necessary */
   if (!(hcntrl & SCSI_PAUSEACK))
   {
      SCSIhPauseAndWait(hhcb);
   }

   if (offset < SCSI_SCRATCH0_SIZE)
   {
      scratchAddress = SCSI_SCRATCH0 + offset;
   }
   else
   {
      scratchAddress = SCSI_SCRATCH1 + (offset - SCSI_SCRATCH0_SIZE);
   }

   /* write to scratch RAM */
   for (i = 0; i < length; i++)
   {
      OSD_OUTEXACT8(SCSI_AICREG(scratchAddress+i),dataBuffer[i]);
   }

   /* restore HCNTRL if necessary */
   if (!(hcntrl & SCSI_PAUSEACK))
   {
      SCSI_hWRITE_HCNTRL(hhcb,scsiRegister,hcntrl);
   }
}
#endif /* SCSI_STANDARD64_MODE || defined(SCSI_INTERNAL) */

/*********************************************************************
*
*  SCSIhAIC78XXAutoTermCable
*
*  Do the cable-sensing autotermination based upon whether the internal
*  and external cable is installed.
*
*  Return Value:  termination setting
*
*  Parameters:    hhcb
*
*  Activation:
*
*  Remarks:       The termination setting is return and it is up to
*                 caller to update internal flag, SEEPROM, or board
*                 logic control.
*
*********************************************************************/
#if SCSI_AIC78XX && SCSI_AUTO_TERMINATION
SCSI_UEXACT8 SCSIhAIC78XXAutoTermCable (SCSI_HHCB SCSI_HPTR hhcb)
{
   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_UEXACT8 cableStat;
   SCSI_UEXACT8 terminationSetting = 0;

   cableStat = SCSIhCableSense(hhcb);  /* read cable info */
   
   if (hhcb->deviceID == SCSI_2940AUW)
   {
   
      if ((cableStat & SCSI_CABLE_EXT) && (cableStat & SCSI_CABLE_INT68))

         terminationSetting &= ~SCSI_TERMINATION_HIGH;
      else
         terminationSetting |= SCSI_TERMINATION_HIGH;

   }
   else
   {
      /* need to change HA termination control according to cable info. */
      /* if eeprom setting is incorrect, eeprom would be adjusted for   */
      /* backward compatibility.                                        */
      if (OSD_INEXACT8(SCSI_AICREG(SCSI_SBLKCTL)) & SCSI_SELWIDE)   /* wide HA */
      {
         /* determining the low byte termination control */
         if (((cableStat & SCSI_CABLE_EXT) && ((cableStat & SCSI_CABLE_INT50) ^ (cableStat & SCSI_CABLE_INT68)))
            || (!(cableStat & SCSI_CABLE_EXT) &&  (cableStat & SCSI_CABLE_INT50) && (cableStat & SCSI_CABLE_INT68)))
         {                             /* need to turn OFF low termination */
            terminationSetting &= ~SCSI_TERMINATION_LOW;
         }
         else                          /* need to turn ON low termination */
         {
            terminationSetting |= SCSI_TERMINATION_LOW;
         }

         /* determining the high byte termination control */
         if ((cableStat & SCSI_CABLE_EXT) && !(cableStat & SCSI_CABLE_INT50) && (cableStat & SCSI_CABLE_INT68))
         {                             /* need to turn OFF high termination */
            terminationSetting &= ~SCSI_TERMINATION_HIGH;
         }
         else                          /* need to turn ON high termination */
         {
            terminationSetting |= SCSI_TERMINATION_HIGH;
         }
#if SCSI_2930UW_SUPPORT
      
         if (hhcb->deviceID == SCSI_KATANA_2930UW)
      
         {
            terminationSetting |= SCSI_TERMINATION_HIGH;
         }
#endif
      }
      else                                   /* narrow HA */
      {
         if ((cableStat & SCSI_CABLE_EXT) && (cableStat & SCSI_CABLE_INT50))
         {
            terminationSetting &= ~SCSI_TERMINATION_LOW;
         }
         else                 /* only one or no cable connected */
         {
            terminationSetting |= SCSI_TERMINATION_LOW;
         }
      }
   }

   return(terminationSetting);
}
#endif

/*********************************************************************
*
*  SCSIhCableSense
*
*     Read the internal/external cable configuration status
*
*  Return Value:  cableStat
*                  
*  Parameters:    hhcb
*
*  Activation:
*
*  Remarks:       The only auto termination scheme assumed is
*                 cable sensing.
*                  
*********************************************************************/
#if SCSI_AUTO_TERMINATION
SCSI_UEXACT8 SCSIhCableSense (SCSI_HHCB SCSI_HPTR hhcb)
{
   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_HOST_ADDRESS SCSI_LPTR hostAddress;
   SCSI_UEXACT8 cableStat = 0;
   SCSI_UEXACT8 regValueLow;
   SCSI_UEXACT8 regValueHigh;
   SCSI_UEXACT8 channelSelect;
   SCSI_UEXACT8 seectl;

   seectl = OSD_INEXACT8(SCSI_AICREG(SCSI_SEEPROM));        /* save SEECTL */
   if (hhcb->hardwareMode == SCSI_HMODE_AICTRIDENT)
   {
      /* TridentII hardware need to assert both SEEMS and SEECS to arbitrate */
      /* for flex port. These two bits need to be asserted until chip finish */
      /* flex port accessing.                                                */
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SEEPROM),SCSI_SEEMS|SCSI_SEECS);
   }
   else
   {   
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SEEPROM),SCSI_SEEMS);
   }
   
   /* check external arbitration enable bit status */
   SCSI_hCHECKEXTARBACKENBIT(hhcb);

   /* Excalibur based boards (AHA-394xUxx) use BRDCS bit in BRDCTL register */
   /* to select either channel A (BRDCS = 0) or channel B (BRDCS = 1) */
   /* Get host address to access PCI device's function number */
   hostAddress = OSD_GET_HOST_ADDRESS(hhcb);
   if (((hhcb->deviceID & SCSI_ID_MASK) == SCSI_EXCALIBUR_BASE) &&
       (hostAddress->pciAddress.functionNumber == 0))
   {
      channelSelect = 0;            /* Select channel A */
   }
   else
   {
      channelSelect = SCSI_BRDCS;   /* Select channel B or use for other HAs */
   }
   
   switch (hhcb->SCSI_HP_autoTerminationMode)
   {
      case SCSI_AUTOTERM_MODE0_CABLE_SENSING:
         
         OSD_OUTEXACT8(SCSI_AICREG(SCSI_SEEPROM), SCSI_SEEMS | SCSI_SEECS);

         /* BRDDAT[7:5] of BRDCTL register               */
         /*                                              */
         /* bit 5  -  external MUXSEL                    */
         /*                                              */
         /* BANK 0:                                      */
         /* bit 7  -  internal 68-pin connector sense    */
         /* bit 6  -  internal 50-pin connector sense    */
         /*                                              */
         /* BANK 1:                                      */
         /* bit 7  -  flash installation indicator       */
         /* bit 6  -  external connector sense           */

         /* write 0 to bit 5 MUXSEL to select BANK 0  */
         OSD_OUTEXACT8(SCSI_AICREG(SCSI_BRDCTL), SCSI_BRDSTB | channelSelect);
         OSD_OUTEXACT8(SCSI_AICREG(SCSI_BRDCTL), channelSelect);
         OSD_OUTEXACT8(SCSI_AICREG(SCSI_BRDCTL), 0x00);

         /* read BANK 0 of MUX through BRDCTL port */
         OSD_OUTEXACT8(SCSI_AICREG(SCSI_BRDCTL), SCSI_BRDRW | channelSelect);
         regValueLow = OSD_INEXACT8(SCSI_AICREG(SCSI_BRDCTL));
         OSD_OUTEXACT8(SCSI_AICREG(SCSI_BRDCTL), 0x00);

         /* write 1 to bit 5 MUXSEL to select BANK 1  */
         OSD_OUTEXACT8(SCSI_AICREG(SCSI_BRDCTL),
            SCSI_BRDDAT5 | SCSI_BRDSTB | channelSelect);
         OSD_OUTEXACT8(SCSI_AICREG(SCSI_BRDCTL), SCSI_BRDDAT5 | channelSelect);
         OSD_OUTEXACT8(SCSI_AICREG(SCSI_BRDCTL), SCSI_BRDDAT5);

         /* read BANK 1 of MUX through BRDCTL port */
         OSD_OUTEXACT8(SCSI_AICREG(SCSI_BRDCTL), SCSI_BRDRW | channelSelect);
         regValueHigh = OSD_INEXACT8(SCSI_AICREG(SCSI_BRDCTL));
         OSD_OUTEXACT8(SCSI_AICREG(SCSI_BRDCTL), 0x00);
         OSD_OUTEXACT8(SCSI_AICREG(SCSI_SEEPROM), 0x00); /* deassert SEEMS and SEECS */

         if (OSD_INEXACT8(SCSI_AICREG(SCSI_SBLKCTL)) & SCSI_SELWIDE)   /* for 2940 UW */
         {
            if (!(regValueLow & 0x80))    /* a zero indicates cable installed */
               cableStat |= SCSI_CABLE_INT68;
   
            if (!(regValueLow & 0x40) && !(hhcb->deviceID == SCSI_2940AUW))
               cableStat |= SCSI_CABLE_INT50;

            if (!(regValueHigh & 0x40))
               cableStat |= SCSI_CABLE_EXT;
         }
         else
         {
            if (!(regValueLow & 0x80))    /* a zero indicates cable installed */
               cableStat |= SCSI_CABLE_EXT;

            if (!(regValueLow & 0x40))
               cableStat |= SCSI_CABLE_INT50;
         }

         break;

      case SCSI_AUTOTERM_MODE1_CABLE_SENSING:      /* DAGGER plus and   */
                                                   /*  TALON based      */   
         /* BRDDAT[6:5] of BRDCTL register               */
         /*                                              */
         /* bit 6  -  external connector sense           */
         /* bit 5  -  internal 50-pin connector sense    */
         regValueLow = OSD_INEXACT8(SCSI_AICREG(SCSI_SPIOCAP));
         regValueLow &= ~SCSI_SOFTCMDEN;        /* Disable soft commands */
         regValueLow |= SCSI_EXT_BRDCTL;        /* Enable external board port */

         /* write it out */
         OSD_OUTEXACT8(SCSI_AICREG(SCSI_SPIOCAP), regValueLow);

         /* enable board r/w and chip select */
         regValueLow = OSD_INEXACT8(SCSI_AICREG(SCSI_BRDCTL));
         regValueLow |= SCSI_BRDRW | SCSI_BRDCS;

         SCSIhDelayCount500us(hhcb, 0x00);   /* wait for deskew delay */

         OSD_OUTEXACT8(SCSI_AICREG(SCSI_BRDCTL), regValueLow);

         regValueLow = OSD_INEXACT8(SCSI_AICREG(SCSI_SEEPROM));
         OSD_OUTEXACT8(SCSI_AICREG(SCSI_SEEPROM), regValueLow | SCSI_SEEMS);

         SCSIhDelayCount500us(hhcb, 0x00);   /* wait for deskew delay */

         OSD_OUTEXACT8(SCSI_AICREG(SCSI_SEEPROM), regValueLow &= ~SCSI_SEEMS);

         regValueLow = OSD_INEXACT8(SCSI_AICREG(SCSI_BRDCTL));

         if (!(regValueLow & 0x20))       /* zero indicates cable installed */
            cableStat |= SCSI_CABLE_INT50;

         if (!(regValueLow & 0x40))
            cableStat |= SCSI_CABLE_EXT;
         
         break;

#if (SCSI_AICBAYONET || SCSI_AICTRIDENT)
      case SCSI_AUTOTERM_MODE2_CABLE_SENSING:      /* Bayonet based   */
         if (hhcb->hardwareMode == SCSI_HMODE_AICTRIDENT)
         {
            /* TridentII hardware need to assert both SEEMS and SEECS to arbitrate */
            /* for flex port. These two bits need to be asserted until chip finish */
            /* flex port accessing.                                                */
            OSD_OUTEXACT8(SCSI_AICREG(SCSI_SEEPROM),SCSI_SEEMS|SCSI_SEECS);
         }
         else
         {   
            OSD_OUTEXACT8(SCSI_AICREG(SCSI_SEEPROM),SCSI_SEEMS);
         }

         /* BAYONET based - BRDCTL register           */
         /*    bit 7: flash/Eprom status              */
         /*    bit 6: Secondary Termination HIGH byte */
         /*    bit 5: Secondary Termination LOW byte  */
         /*    bit 4: Primary Termination HIGH byte   */
         /*    bit 3: Primary Termination LOW byte    */
         /*    bit 2: Reserved (always read 1)        */

         /* read BRDCTL */
         OSD_OUTEXACT8(SCSI_AICREG(SCSI_BRDCTL), SCSI_BAYONET_BRDRW);
         cableStat = OSD_INEXACT8(SCSI_AICREG(SCSI_BRDCTL));

         OSD_OUTEXACT8(SCSI_AICREG(SCSI_BRDCTL), 0x00);  /* deassert BRDRW */
         OSD_OUTEXACT8(SCSI_AICREG(SCSI_SEEPROM), 0x00); /* deassert SEEMS and SEECS */

         break;
#endif /* SCSI_AICBAYONET || SCSI_AICTRIDENT */
   }
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SEEPROM), seectl);     /* restore SEECTL */
   return(cableStat);
}
#endif

/*********************************************************************
*
*  SCSIhWriteHCNTRL
*
*     Implement required for write to HCNTRL register
*
*  Return Value:  none
*                  
*  Parameters:    hhcb
*                 value to be written
*
*  Remarks:
*                  
*********************************************************************/
#if SCSI_AIC787X + SCSI_AIC785X
void SCSIhWriteHCNTRL (SCSI_HHCB SCSI_HPTR hhcb,
                       SCSI_UEXACT8 value)
{
   SCSI_REGISTER scsiRegister = hhcb->scsiRegister;

   if (hhcb->SCSI_HP_wwWriteHCNTRL)
   {
      SCSIhWWWriteHCNTRL(scsiRegister,value);
   }
   else
   {
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_HCNTRL), value);
   }
}
#endif

/*********************************************************************
*
*  SCSIhWWWriteHCNTRL
*
*     Implement workaround required for write to HCNTRL register
*
*  Return Value:  none
*                  
*  Parameters:    hhcb
*                 value to be written
*
*  Remarks:
*                  
*********************************************************************/
#if SCSI_AIC787X + SCSI_AIC785X
void SCSIhWWWriteHCNTRL (SCSI_REGISTER scsiRegister,
                         SCSI_UEXACT8 value)
{
   SCSI_UEXACT8 hcntrlValue;

   if (!(value & SCSI_PAUSE))                           /* pause chip, just*/
   {                                                    /* do the output.  */
      hcntrlValue = OSD_INEXACT8(SCSI_AICREG(SCSI_HCNTRL));            
      if (!(hcntrlValue & SCSI_PAUSEACK))               /* If chip is not  */
      {                                                 /* paused, pause   */
         OSD_OUTEXACT8(SCSI_AICREG(SCSI_HCNTRL), hcntrlValue | SCSI_PAUSE); /* the chip first. */
         while (!(OSD_INEXACT8(SCSI_AICREG(SCSI_HCNTRL)) & SCSI_PAUSEACK)); 
      }                                                 /* If the chip is  */
      if (OSD_INEXACT8(SCSI_AICREG(SCSI_INTSTAT)) & SCSI_ANYPAUSE)          /* paused due to an */
      {                                                 /* interrupt, make */
         value |= SCSI_PAUSE;                           /* sure we turn the*/
      }                                                 /* pause bit on.   */
   }

   OSD_OUTEXACT8(SCSI_AICREG(SCSI_HCNTRL), value);
}
#endif

/*********************************************************************
*
*  SCSIhReadINTSTAT
*
*     Implement woraround required for read from INTSTAT register
*
*  Return Value:  value from register
*                  
*  Parameters:    hhcb
*
*  Remarks:
*                  
*********************************************************************/
#if SCSI_AIC787X + SCSI_AIC785X
SCSI_UEXACT8 SCSIhReadINTSTAT (SCSI_HHCB SCSI_HPTR hhcb)
{
   SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_UEXACT8 value;

   if (hhcb->SCSI_HP_wwReadINTSTAT)
   {
      value = SCSIhWWReadINTSTAT(scsiRegister);
   }
   else
   {                                                    /* Already paused  */
      value = OSD_INEXACT8(SCSI_AICREG(SCSI_INTSTAT));  /* just read it    */
   }

   return(value);
}
#endif

/*********************************************************************
*
*  SCSIhWWReadINTSTAT
*
*     Implement read from INTSTAT register
*
*  Return Value:  value from register
*                  
*  Parameters:    hhcb
*
*  Remarks:
*                  
*********************************************************************/
#if SCSI_AIC787X + SCSI_AIC785X
SCSI_UEXACT8 SCSIhWWReadINTSTAT (SCSI_REGISTER scsiRegister)
{
   SCSI_UEXACT8 hcntrlValue;
   SCSI_UEXACT8 value;

   hcntrlValue = OSD_INEXACT8(SCSI_AICREG(SCSI_HCNTRL));/* If output will  */
   if (!(hcntrlValue & SCSI_PAUSE))                     /* pause chip, just*/
   {                                                    /* do the output.  */
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_HCNTRL), hcntrlValue | SCSI_PAUSE);    /* pause the chip  */
      while (!(OSD_INEXACT8(SCSI_AICREG(SCSI_HCNTRL)) & SCSI_PAUSEACK));    /* first.          */
      if ((value = OSD_INEXACT8(SCSI_AICREG(SCSI_INTSTAT))) & SCSI_ANYPAUSE)/* paused due to an*/
      {                                                 /* interrupt, make */
         hcntrlValue |= SCSI_PAUSE;                     /* sure we turn the*/
      }                                                 /* pause bit on.   */
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_HCNTRL), hcntrlValue); /* Restore HCNTRL  */
   }
   else
   {                                                    /* Already paused  */
      value = OSD_INEXACT8(SCSI_AICREG(SCSI_INTSTAT));  /* just read it    */
   }

   return(value);
}
#endif

/*********************************************************************
*
*  SCSIHReadHADDR
*
*     Read the HADDR register
*
*  Return Value:  
*                  
*  Parameters:    hhcb
*                 dataBuffer
*                 
*
*  Remarks:
*                  
*********************************************************************/
#if SCSI_AIC78XX || defined(SCSI_INTERNAL)
void SCSIHReadHADDR (SCSI_HHCB SCSI_HPTR hhcb,
                     SCSI_UEXACT8 SCSI_SPTR dataBuffer)
{
   SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_UEXACT8 hcntrl;

   /* save the HCNTRL value */
   hcntrl = OSD_INEXACT8(SCSI_AICREG(SCSI_HCNTRL));
   
   /* pause the chip if necessary */
   if (!(hcntrl & SCSI_PAUSEACK))
   {
      SCSIhPauseAndWait(hhcb);
   }

   dataBuffer[0] = OSD_INEXACT8(SCSI_AICREG(SCSI_HADDR0));
   dataBuffer[1] = OSD_INEXACT8(SCSI_AICREG(SCSI_HADDR1));
   dataBuffer[2] = OSD_INEXACT8(SCSI_AICREG(SCSI_HADDR2));
   dataBuffer[3] = OSD_INEXACT8(SCSI_AICREG(SCSI_HADDR3));

   /* restore HCNTRL if necessary */
   if (!(hcntrl & SCSI_PAUSEACK))
   {
      SCSI_hWRITE_HCNTRL(hhcb,scsiRegister,hcntrl);
   }
}
#endif /* SCSI_AIC78XX || defined(SCSI_INTERNAL) */

/*********************************************************************
*
*  SCSIhSetAddressScbBA32
*
*     Translate virtual address to 32 bits bus address and
*     set the 32 bits bus address to scb ram
*
*  Return Value:  none
*                  
*  Parameters:    hhcb
*                 scb address
*                 virtual address
*
*  Remarks:       This routine is mostly called during building a
*                 request sense commnand.  The SCB pointer register
*                 should point to the correct SCB ram.
*                  
*********************************************************************/
#if OSD_BUS_ADDRESS_SIZE == 32 && SCSI_STANDARD_MODE
void SCSIhSetAddressScbBA32 (SCSI_HHCB SCSI_HPTR hhcb,
                             SCSI_UEXACT8 scbAddress,
                             void SCSI_HPTR virtualAddress)
{
   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_BUS_ADDRESS busAddress;

   /* get bus address */
   busAddress = OSD_GET_BUS_ADDRESS(hhcb,SCSI_MC_LOCKED,virtualAddress);

   /* load bus address to scb ram */
   OSD_OUTEXACT8(SCSI_AICREG(scbAddress+0),((SCSI_QUADLET SCSI_IPTR)&busAddress)->u8.order0);
   OSD_OUTEXACT8(SCSI_AICREG(scbAddress+1),((SCSI_QUADLET SCSI_IPTR)&busAddress)->u8.order1);
   OSD_OUTEXACT8(SCSI_AICREG(scbAddress+2),((SCSI_QUADLET SCSI_IPTR)&busAddress)->u8.order2);
   OSD_OUTEXACT8(SCSI_AICREG(scbAddress+3),((SCSI_QUADLET SCSI_IPTR)&busAddress)->u8.order3);
}
#endif /* OSD_BUS_ADDRESS_SIZE == 32 && SCSI_STANDARD_MODE */

/*********************************************************************
*
*  SCSIhSetAddressScbBA64
*
*     Translate virtual address to 64 bits bus address and
*     set the 64 bits bus address to scb ram
*
*  Return Value:  none
*                  
*  Parameters:    hhcb
*                 scb address
*                 virtual address
*
*  Remarks:       This routine is mostly called during building a
*                 request sense commnand.  The SCB pointer register
*                 should point to the correct SCB ram.
*                  
*********************************************************************/
#if OSD_BUS_ADDRESS_SIZE == 64 && SCSI_STANDARD_MODE
void SCSIhSetAddressScbBA64 (SCSI_HHCB SCSI_HPTR hhcb,
                             SCSI_UEXACT8 scbAddress,
                             void SCSI_HPTR virtualAddress)
{
   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_BUS_ADDRESS busAddress;

   /* get bus address */
   busAddress = OSD_GET_BUS_ADDRESS(hhcb,SCSI_MC_LOCKED,virtualAddress);

   /* load bus address to scb ram */
   OSD_OUTEXACT8(SCSI_AICREG(scbAddress+0),((SCSI_OCTLET SCSI_IPTR)&busAddress)->u8.order0);
   OSD_OUTEXACT8(SCSI_AICREG(scbAddress+1),((SCSI_OCTLET SCSI_IPTR)&busAddress)->u8.order1);
   OSD_OUTEXACT8(SCSI_AICREG(scbAddress+2),((SCSI_OCTLET SCSI_IPTR)&busAddress)->u8.order2);
   OSD_OUTEXACT8(SCSI_AICREG(scbAddress+3),((SCSI_OCTLET SCSI_IPTR)&busAddress)->u8.order3);
   OSD_OUTEXACT8(SCSI_AICREG(scbAddress+4),((SCSI_OCTLET SCSI_IPTR)&busAddress)->u8.order4);
   OSD_OUTEXACT8(SCSI_AICREG(scbAddress+5),((SCSI_OCTLET SCSI_IPTR)&busAddress)->u8.order5);
   OSD_OUTEXACT8(SCSI_AICREG(scbAddress+6),((SCSI_OCTLET SCSI_IPTR)&busAddress)->u8.order6);
   OSD_OUTEXACT8(SCSI_AICREG(scbAddress+7),((SCSI_OCTLET SCSI_IPTR)&busAddress)->u8.order7);
}
#endif /* OSD_BUS_ADDRESS_SIZE == 64 && SCSI_STANDARD_MODE */


/*********************************************************************
*
*  SCSIhGetAddressScbBA32
*
*     Translate virtual address to 32 bits bus address and
*     set the 32 bits bus address to scb ram
*
*  Return Value:  none
*                  
*  Parameters:    hhcb
*                 scb address
*                 virtual address
*
*  Remarks:       This routine is mostly called during a residual 
*                 length calculation.  The SCB pointer register
*                 should point to the correct SCB ram.
*                  
*********************************************************************/
#if OSD_BUS_ADDRESS_SIZE == 32
SCSI_BUS_ADDRESS SCSIhGetAddressScbBA32 (SCSI_HHCB SCSI_HPTR hhcb,
                                         SCSI_UEXACT8 scbAddress)
{
   return((SCSI_BUS_ADDRESS)SCSIhGetEntity32FromScb(hhcb,scbAddress));
}
#endif

/*********************************************************************
*
*  SCSIhGetEntity32FromScb
*
*     Get a 32 bit entity from scb ram
*
*  Return Value:  none
*                  
*  Parameters:    hhcb
*                 scb address
*                 virtual address
*
*  Remarks:       This routine is mostly called during a residual 
*                 length calculation.
*                  
*********************************************************************/
SCSI_UEXACT32 SCSIhGetEntity32FromScb (SCSI_HHCB SCSI_HPTR hhcb,
                                       SCSI_UEXACT8 scbAddress)
{
   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_UEXACT32 entity;

   /* load bus address to scb ram */
   ((SCSI_QUADLET SCSI_IPTR)&entity)->u8.order0 = 
         OSD_INEXACT8(SCSI_AICREG(scbAddress+0));
   ((SCSI_QUADLET SCSI_IPTR)&entity)->u8.order1 = 
         OSD_INEXACT8(SCSI_AICREG(scbAddress+1));
   ((SCSI_QUADLET SCSI_IPTR)&entity)->u8.order2 = 
         OSD_INEXACT8(SCSI_AICREG(scbAddress+2));
   ((SCSI_QUADLET SCSI_IPTR)&entity)->u8.order3 = 
         OSD_INEXACT8(SCSI_AICREG(scbAddress+3));

   return(entity);
}

/*********************************************************************
*
*  SCSIhGetAddressScbBA64
*
*     Translate virtual address to 64 bits bus address and
*     set the 64 bits bus address to scb ram
*
*  Return Value:  none
*                  
*  Parameters:    hhcb
*                 scb address
*                 virtual address
*
*  Remarks:       This routine is mostly called during a residual 
*                 length calculation.  The SCB pointer register
*                 should point to the correct SCB ram.
*                  
*********************************************************************/
#if OSD_BUS_ADDRESS_SIZE == 64
SCSI_BUS_ADDRESS SCSIhGetAddressScbBA64 (SCSI_HHCB SCSI_HPTR hhcb,
                                         SCSI_UEXACT8 scbAddress)
{
   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_BUS_ADDRESS busAddress;

   /* load bus address to scb ram */
   ((SCSI_OCTLET SCSI_IPTR)&busAddress)->u8.order0 = OSD_INEXACT8(SCSI_AICREG(scbAddress+0));
   ((SCSI_OCTLET SCSI_IPTR)&busAddress)->u8.order1 = OSD_INEXACT8(SCSI_AICREG(scbAddress+1));
   ((SCSI_OCTLET SCSI_IPTR)&busAddress)->u8.order2 = OSD_INEXACT8(SCSI_AICREG(scbAddress+2));
   ((SCSI_OCTLET SCSI_IPTR)&busAddress)->u8.order3 = OSD_INEXACT8(SCSI_AICREG(scbAddress+3));
   ((SCSI_OCTLET SCSI_IPTR)&busAddress)->u8.order4 = OSD_INEXACT8(SCSI_AICREG(scbAddress+4));
   ((SCSI_OCTLET SCSI_IPTR)&busAddress)->u8.order5 = OSD_INEXACT8(SCSI_AICREG(scbAddress+5));
   ((SCSI_OCTLET SCSI_IPTR)&busAddress)->u8.order6 = OSD_INEXACT8(SCSI_AICREG(scbAddress+6));
   ((SCSI_OCTLET SCSI_IPTR)&busAddress)->u8.order7 = OSD_INEXACT8(SCSI_AICREG(scbAddress+7));

   return(busAddress);
}
#endif

/*********************************************************************
*
*  SCSIhCompareBusAddress
*
*     Compare two bus addresses.
*
*  Return Value:  none
*                  
*  Parameters:    busAddress1
*                 busAddress2
*
*  Remarks:       This routine compares two busAddresses byte by byte.
*                  
*********************************************************************/
SCSI_UINT8 SCSIhCompareBusAddress (SCSI_BUS_ADDRESS busAddress1,
                                   SCSI_BUS_ADDRESS busAddress2)
{
#if OSD_BUS_ADDRESS_SIZE == 32
   /* Compare on the 32-bit bus address */
   if ((((SCSI_QUADLET SCSI_LPTR) &busAddress1)->u16.order0 != 
        ((SCSI_QUADLET SCSI_LPTR) &busAddress2)->u16.order0) ||
       (((SCSI_QUADLET SCSI_LPTR) &busAddress1)->u16.order1 != 
        ((SCSI_QUADLET SCSI_LPTR) &busAddress2)->u16.order1))
   {
      return(1);
   }
#else
   /* Compare on the 64-bit bus address */
   if ((((SCSI_OCTLET SCSI_LPTR) &busAddress1)->u32.order0 !=
        ((SCSI_OCTLET SCSI_LPTR) &busAddress2)->u32.order0) ||
       (((SCSI_OCTLET SCSI_LPTR) &busAddress1)->u32.order1 !=
        ((SCSI_OCTLET SCSI_LPTR) &busAddress2)->u32.order1))
   {
      return(1);
   }
#endif
   return(0);
}

/*********************************************************************
*
*  SCSIhAIC78XXLoadScsiRate
*
*     Load the SCSIRATE register
*
*  Return Value:  none
*
*  Parameters:    hhcb
*                 targetID
*                 value
*
*  Remarks:
*                  
*********************************************************************/
#if SCSI_AIC78XX
void SCSIhAIC78XXLoadScsiRate (SCSI_HHCB SCSI_HPTR hhcb,
                               SCSI_UEXACT8 targetID,
                               SCSI_UEXACT16 value)
{
   SCSI_REGISTER scsiRegister = hhcb->scsiRegister;

   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSIRATE),(SCSI_UEXACT8)(value));
}
#endif /* SCSI_AIC78XX */

/*********************************************************************
*
*  SCSIhAICBayonetLoadScsiRate
*
*     Load the SCSIRATE register
*
*  Return Value:  none
*
*  Parameters:    hhcb
*                 targetID
*                 value
*
*  Remarks:
*                  
*********************************************************************/
#if SCSI_AICBAYONET
void SCSIhAICBayonetLoadScsiRate (SCSI_HHCB SCSI_HPTR hhcb,
                                  SCSI_UEXACT8 targetID,
                                  SCSI_UEXACT16 value)
{
   SCSI_REGISTER scsiRegister = hhcb->scsiRegister;

   OSD_OUTEXACT8(SCSI_AICREG(SCSI_BAYONET_SCSIRATE),
         (SCSI_UEXACT8)(value & 0x00ff));
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_BAYONET_SCSIOFFSET),
         (SCSI_UEXACT8)((value >> 8) & 0x00ff));
}
#endif /* SCSI_AICBAYONET */

#if SCSI_AICTRIDENT
/*********************************************************************
*
*  SCSIhAICTridentLoadScsiRate
*
*     Load the auto SCSI RATE/OFFSET table
*
*  Return Value:  none
*
*  Parameters:    hhcb
*                 targetID
*                 value
*
*  Remarks:
*                  
*********************************************************************/
#if (SCSI_STANDARD_160M_MODE || SCSI_SWAPPING_160M_MODE)
void SCSIhAICTridentLoadScsiRate (SCSI_HHCB SCSI_HPTR hhcb,
                                  SCSI_UEXACT8 targetID,
                                  SCSI_UEXACT16 value)
{
   SCSI_REGISTER scsiRegister = hhcb->scsiRegister;

   /* load scsi rate */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_AUTORATE_PNTR), (SCSI_RATE_DATA | targetID));
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_AUTORATE_DATA),
          (SCSI_UEXACT8)(value & 0x00ff));

   /* load scsi offset */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_AUTORATE_PNTR), targetID);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_AUTORATE_DATA),
          (SCSI_UEXACT8)((value >> 8) & 0x00ff));
}
#endif /* (SCSI_STANDARD_160M_MODE || SCSI_SWAPPING_160M_MODE) */

/*********************************************************************
*
*  SCSIhAICTridentLoadScsiRate
*
*     Load the SCSIRATE and SCSIOFFSET registers
*
*  Return Value:  none
*
*  Parameters:    hhcb
*                 targetID
*                 value
*
*  Remarks:
*                  
*********************************************************************/
#if !(SCSI_STANDARD_160M_MODE || SCSI_SWAPPING_160M_MODE)
void SCSIhAICTridentLoadScsiRate (SCSI_HHCB SCSI_HPTR hhcb,
                                  SCSI_UEXACT8 targetID,
                                  SCSI_UEXACT16 value)
{
   SCSI_REGISTER scsiRegister = hhcb->scsiRegister;

   OSD_OUTEXACT8(SCSI_AICREG(SCSI_TRIDENT_SCSIRATE),
       (SCSI_UEXACT8)(value & 0x00ff));
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_TRIDENT_SCSIOFFSET),
       (SCSI_UEXACT8)((value >> 8) & 0x00ff));
}
#endif /* !(SCSI_STANDARD_160M_MODE || SCSI_SWAPPING_160M_MODE) */

#endif /* SCSI_AICTRIDENT */

/*********************************************************************
*
*  SCSIhAIC78XXResetScsiRate
*
*     Reset the SCSI RATE and OFFSET
*
*  Return Value:  none
*
*  Parameters:    hhcb
*
*  Remarks:
*                  
*********************************************************************/
#if SCSI_AIC78XX
void SCSIhAIC78XXResetScsiRate (SCSI_HHCB SCSI_HPTR hhcb)
{
   SCSI_REGISTER scsiRegister = hhcb->scsiRegister;

   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSIRATE),0);
}
#endif /* SCSI_AIC78XX */

/*********************************************************************
*
*  SCSIhAICBayonetResetScsiRate
*
*     Reset the SCSI RATE and OFFSET
*
*  Return Value:  none
*
*  Parameters:    hhcb
*
*  Remarks:
*                  
*********************************************************************/
#if SCSI_AICBAYONET
void SCSIhAICBayonetResetScsiRate (SCSI_HHCB SCSI_HPTR hhcb)
{
   SCSI_REGISTER scsiRegister = hhcb->scsiRegister;

   OSD_OUTEXACT8(SCSI_AICREG(SCSI_BAYONET_SCSIRATE),0);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_BAYONET_SCSIOFFSET),0);
}
#endif /* SCSI_AICBAYONET */

#if SCSI_AICTRIDENT
/*********************************************************************
*
*  SCSIhAICTridentResetScsiRate
*
*     Reset the SCSI RATE/OFFSET table
*
*  Return Value:  none
*
*  Parameters:    hhcb
*
*  Remarks:
*                  
*********************************************************************/
#if (SCSI_STANDARD_160M_MODE || SCSI_SWAPPING_160M_MODE)
void SCSIhAICTridentResetScsiRate (SCSI_HHCB SCSI_HPTR hhcb)
{
   SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_UEXACT8 i;

   for (i = 0; i < hhcb->maxDevices; i++)
   {
      /* load scsi rate */
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_AUTORATE_PNTR), (SCSI_RATE_DATA | i));
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_AUTORATE_DATA), 0x00);

      /* load scsi offset */
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_AUTORATE_PNTR), i);
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_AUTORATE_DATA), 0x00);
   }
}
#endif /* (SCSI_STANDARD_160M_MODE || SCSI_SWAPPING_160M_MODE) */

/*********************************************************************
*
*  SCSIhAICTridentResetScsiRate
*
*     Reset the SCSI RATE and OFFSET
*
*  Return Value:  none
*
*  Parameters:    hhcb
*
*  Remarks:
*                  
*********************************************************************/
#if !(SCSI_STANDARD_160M_MODE || SCSI_SWAPPING_160M_MODE)
void SCSIhAICTridentResetScsiRate (SCSI_HHCB SCSI_HPTR hhcb)
{
   SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_TRIDENT_SCSIRATE),0);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_TRIDENT_SCSIOFFSET),0);
}
#endif /* !(SCSI_STANDARD_160M_MODE || SCSI_SWAPPING_160M_MODE) */

#endif /* SCSI_AICTRIDENT */

#if SCSI_AICTRIDENT
/*********************************************************************
*
*  SCSIhAICTridentClearScsiRate
*
*     Clear the auto SCSI RATE/OFFSET table
*
*  Return Value:  none
*
*  Parameters:    hhcb
*
*  Remarks:       This function do nothing to the auto SCSI rate/
*                 offset table.  This function normally invoked
*                 during run-time especially at the error exception
*                 handling e.g. Check Condition status.  The auto
*                 rate/offset table is used by the hardware on every
*                 SCSI connection.  If we cleared scsi rate/offset
*                 for a target in this table, then hardware will run
*                 with the incorrect scsi rate/offset transfer if the
*                 same target ID with different LUN reselected us.
*                  
*********************************************************************/
#if (SCSI_STANDARD_160M_MODE || SCSI_SWAPPING_160M_MODE)
void SCSIhAICTridentClearScsiRate (SCSI_HHCB SCSI_HPTR hhcb)
{
   return;
}
#endif /* (SCSI_STANDARD_160M_MODE || SCSI_SWAPPING_160M_MODE) */

/*********************************************************************
*
*  SCSIhAICTridentClearScsiRate
*
*     Clear the SCSI RATE/OFFSET
*
*  Return Value:  none
*
*  Parameters:    hhcb
*
*  Remarks:
*                  
*********************************************************************/
#if !(SCSI_STANDARD_160M_MODE || SCSI_SWAPPING_160M_MODE)
void SCSIhAICTridentClearScsiRate (SCSI_HHCB SCSI_HPTR hhcb)
{
   SCSI_REGISTER scsiRegister = hhcb->scsiRegister;

   OSD_OUTEXACT8(SCSI_AICREG(SCSI_TRIDENT_SCSIRATE),0);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_TRIDENT_SCSIOFFSET),0);
}
#endif /* !(SCSI_STANDARD_160M_MODE || SCSI_SWAPPING_160M_MODE) */
#endif /* SCSI_AICTRIDENT */

/*********************************************************************
*
*  SCSIhAIC78XXResetDFifo
*
*     Reset the Data Fifo
*
*  Return Value:  none
*
*  Parameters:    hhcb
*
*  Remarks:
*                  
*********************************************************************/
#if SCSI_AIC78XX
void SCSIhAIC78XXResetDFifo (SCSI_HHCB SCSI_HPTR hhcb)
{
   SCSI_REGISTER scsiRegister = hhcb->scsiRegister;

   OSD_OUTEXACT8(SCSI_AICREG(SCSI_DFCNTRL), SCSI_FIFORESET);
}
#endif /* SCSI_AIC78XX */

/*********************************************************************
*
*  SCSIhAICBayonetResetDFifo
*
*     Reset the Data Fifo
*
*  Return Value:  none
*
*  Parameters:    hhcb
*
*  Remarks:
*                  
*********************************************************************/
#if (SCSI_AICBAYONET || SCSI_AICTRIDENT)
void SCSIhAICBayonetResetDFifo (SCSI_HHCB SCSI_HPTR hhcb)
{
   SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_UEXACT8 value;

   /* No fiforeset for bayonet @dm bay */
   value = OSD_INEXACT8(SCSI_AICREG(SCSI_SXFRCTL0));
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SXFRCTL0), value | (SCSI_CLRSTCNT+SCSI_CLRCHN));
}
#endif

/*********************************************************************
*
*  SCSIhAIC78XXEnableIOErr
*
*     Enable IOERR status to generate an interrupt
*
*  Return Value:  none
*
*  Parameters:    hhcb
*
*  Remarks:
*                  
*********************************************************************/
#if SCSI_AIC78XX
void SCSIhAIC78XXEnableIOErr (SCSI_HHCB SCSI_HPTR hhcb)
{
   return;
}
#endif

/*********************************************************************
*
*  SCSIhAICBayonetEnableIOErr
*
*     Enable IOERR status to generate an interrupt
*
*  Return Value:  none
*
*  Parameters:    hhcb
*
*  Remarks:
*                  
*********************************************************************/
#if (SCSI_AICBAYONET || SCSI_AICTRIDENT)
void SCSIhAICBayonetEnableIOErr (SCSI_HHCB SCSI_HPTR hhcb)
{
   SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_UEXACT8 value;

   value = OSD_INEXACT8(SCSI_AICREG(SCSI_SIMODE0));
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SIMODE0), value | SCSI_BAYONET_ENIOERR);
}
#endif

/*********************************************************************
*
*  SCSIhAIC78XXGetScbRamCtrls
*
*     Get SCB RAM control bits (SCBRAMSEL,RAMPSM, and SCBSIZE32) from the
*     DEVCONFIG register in the PCI Configuration space
*
*  Return Value:  32 bit value
*
*  Parameters:    hhcb
*
*  Remarks:
*                  
*********************************************************************/
#if SCSI_AIC78XX
SCSI_UEXACT32 SCSIhAIC78XXGetScbRamCtrls (SCSI_HHCB SCSI_HPTR hhcb)
{
   return(OSD_READ_PCICFG_EXACT32(hhcb,SCSI_DEVCONFIG));
}
#endif

/*********************************************************************
*
*  SCSIhAICBayonetGetScbRamCtrls
*
*     Get SCB RAM control bits (SCBRAMSEL,RAMPSM, and SCBSIZE32) from the
*     DSCOMMAND0 register
*
*  Return Value:  32 bit value
*
*  Parameters:    hhcb
*
*  Remarks:
*                  
*********************************************************************/
#if (SCSI_AICBAYONET || SCSI_AICTRIDENT)
SCSI_UEXACT32 SCSIhAICBayonetGetScbRamCtrls (SCSI_HHCB SCSI_HPTR hhcb)
{
   SCSI_REGISTER scsiRegister = hhcb->scsiRegister;

   return((SCSI_UEXACT32)OSD_INEXACT8(SCSI_AICREG(SCSI_BAYONET_COMMAND0)));
}
#endif

/*********************************************************************
*
*  SCSIhAIC78XXPutScbRamCtrls
*
*     Put SCB RAM control bits (SCBRAMSEL,RAMPSM, and SCBSIZE32) into the
*     DEVCONFIG register in the PCI Configuration space
*
*  Return Value:  None
*
*  Parameters:    hhcb
*
*  Remarks:
*                  
*********************************************************************/
#if SCSI_AIC78XX
void SCSIhAIC78XXPutScbRamCtrls (SCSI_HHCB SCSI_HPTR hhcb,
                                 SCSI_UEXACT32 value)
{
   OSD_WRITE_PCICFG_EXACT32(hhcb,SCSI_DEVCONFIG,value);
}
#endif

/*********************************************************************
*
*  SCSIhAICBayonetPutScbRamCtrls
*
*     Put SCB RAM control bits (SCBRAMSEL,RAMPSM, and SCBSIZE32) into the
*     DSCOMMAND0 register
*
*  Return Value:  None
*
*  Parameters:    hhcb
*
*  Remarks:
*                  
*********************************************************************/
#if (SCSI_AICBAYONET || SCSI_AICTRIDENT)
void SCSIhAICBayonetPutScbRamCtrls (SCSI_HHCB SCSI_HPTR hhcb,
                                    SCSI_UEXACT32 value)
{
   SCSI_REGISTER scsiRegister = hhcb->scsiRegister;

   OSD_OUTEXACT8(SCSI_AICREG(SCSI_BAYONET_COMMAND0), (SCSI_UEXACT8)value);
}
#endif


/*********************************************************************
*
*  SCSIhGetCacheThEn
*
*     Get the CacheThEn status
*
*  Return Value:  None
*
*  Parameters:    hhcb
*
*  Remarks:
*                  
*********************************************************************/
SCSI_UEXACT8 SCSIhGetCacheThEn (SCSI_HHCB SCSI_HPTR hhcb)
{
   SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_UEXACT8 hcntrl, ret_val = 0;

   /* check if pause chip necessary */
   if (!((hcntrl = OSD_INEXACT8(SCSI_AICREG(SCSI_HCNTRL))) & SCSI_PAUSEACK))
   {
      SCSIhPauseAndWait(hhcb);      
   }

   if ((OSD_INEXACT8(SCSI_AICREG(SCSI_COMMAND)) & SCSI_CACHETHEN))
   {
      ret_val = 1;
   }
   else
   {
      ret_val = 0;
   }

   if (!(hcntrl & SCSI_PAUSEACK))
   {
      SCSI_hWRITE_HCNTRL(hhcb,scsiRegister,hcntrl);
   }

   return (ret_val);
}


/*********************************************************************
*
*  SCSIhAIC78XXUpdateCacheThEn
*
*     Update default CACHETEN bit approppriately
*
*  Return Value:  None
*
*  Parameters:    hhcb
*
*  Remarks:
*                  
*********************************************************************/
#if SCSI_AIC78XX
void SCSIhAIC78XXUpdateCacheThEn (SCSI_HHCB SCSI_HPTR hhcb)
{
   SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_UEXACT8  hcntrl;

   /* check if pause chip necessary */
   if (!((hcntrl = OSD_INEXACT8(SCSI_AICREG(SCSI_HCNTRL))) & SCSI_PAUSEACK))
   {
      SCSIhPauseAndWait(hhcb);      
   }

   if (((hhcb->deviceID&SCSI_ID_MASK)==SCSI_KATANA_BASE)&&(hhcb->hardwareRevision>=1))
   {
      /* For Katana base hw, CacheLineStreaming only works on Revision B or higher */
      hhcb->SCSI_HF_cacheThEn           = 1;
      hhcb->SCSI_HF_cacheThEnAdjustable = 1;
      /* Enable cache line streaming */

      OSD_OUTEXACT8(SCSI_AICREG(SCSI_COMMAND),
         (OSD_INEXACT8(SCSI_AICREG(SCSI_COMMAND)) | SCSI_CACHETHEN)); 
   }
   else
   {
      /* Disable this feature for other revision or non-katana hw */
      hhcb->SCSI_HF_cacheThEn           = 0;

      hhcb->SCSI_HF_cacheThEnAdjustable = 0;
      /* DIsable cache line streaming */
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_COMMAND),
         (OSD_INEXACT8(SCSI_AICREG(SCSI_COMMAND)) & ~SCSI_CACHETHEN)); 
   }

   if (!(hcntrl & SCSI_PAUSEACK))
   {
      SCSI_hWRITE_HCNTRL(hhcb,scsiRegister,hcntrl);
   }
}
#endif

/*********************************************************************
*
*  SCSIhAICBayonetUpdateCacheThEn
*
*     Update default CACHETEN bit approppriately
*
*  Return Value:  None
*
*  Parameters:    hhcb
*
*  Remarks:
*                  
*********************************************************************/
#if SCSI_AICBAYONET
void SCSIhAICBayonetUpdateCacheThEn (SCSI_HHCB SCSI_HPTR hhcb)
{
   SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_UEXACT8  hcntrl;

   /* check if pause chip necessary */
   if (!((hcntrl = OSD_INEXACT8(SCSI_AICREG(SCSI_HCNTRL))) & SCSI_PAUSEACK))
   {
      SCSIhPauseAndWait(hhcb);      
   }

   /* Generally CacheLineStreaming will be enabled by default   */ 
   /* This feature will be turned off if a specific hw does not */
   /* support it correctly.                                     */
   hhcb->SCSI_HF_cacheThEn           = 1;
   hhcb->SCSI_HF_cacheThEnAdjustable = 1;

   if (((hhcb->deviceID&SCSI_ID_MASK)==0x0010)&&(hhcb->hardwareRevision==0))
   {
      /* Bayonet Rev A does support cache threshold properly */
      hhcb->SCSI_HF_cacheThEn           = 0;
      hhcb->SCSI_HF_cacheThEnAdjustable = 0;
   }

   if ((hhcb->deviceID&SCSI_ID_MASK)==0x0050)
   {
      /* Scimitar has problem with CACHETHEN=0,     */
      /* hence mark this feature as non adjustable. */
      hhcb->SCSI_HF_cacheThEnAdjustable = 0;
   }

   if(hhcb->SCSI_HF_cacheThEn)
   {
      /* Enable cache line streaming */
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_COMMAND),
         (OSD_INEXACT8(SCSI_AICREG(SCSI_COMMAND)) | SCSI_CACHETHEN)); 
   }
   else
   {
      /* Disable cache line streaming */
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_COMMAND),
         (OSD_INEXACT8(SCSI_AICREG(SCSI_COMMAND)) & ~SCSI_CACHETHEN)); 
   }

   if (!(hcntrl & SCSI_PAUSEACK))
   {
      SCSI_hWRITE_HCNTRL(hhcb,scsiRegister,hcntrl);
   }
}
#endif 


/*********************************************************************
*
*  SCSIhAICTridentUpdateCacheThEn
*
*     Update default CACHETEN bit approppriately
*
*  Return Value:  None
*
*  Parameters:    hhcb
*
*  Remarks:
*                  
*********************************************************************/
#if SCSI_AICTRIDENT
void SCSIhAICTridentUpdateCacheThEn (SCSI_HHCB SCSI_HPTR hhcb)
{
   SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_UEXACT8  hcntrl;

   /* check if pause chip necessary */
   if (!((hcntrl = OSD_INEXACT8(SCSI_AICREG(SCSI_HCNTRL))) & SCSI_PAUSEACK))
   {
      SCSIhPauseAndWait(hhcb);      
   }

   /* Generally CacheLineStreaming will be enabled by default   */ 
   /* This feature will be turned off if a specific hw does not */
   /* support it correctly.                                     */
   hhcb->SCSI_HF_cacheThEn           = 1;
   hhcb->SCSI_HF_cacheThEnAdjustable = 1;

   if(hhcb->SCSI_HF_cacheThEn)
   {
      /* Enable cache line streaming */
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_COMMAND),
         (OSD_INEXACT8(SCSI_AICREG(SCSI_COMMAND)) | SCSI_CACHETHEN)); 
   }
   else
   {
      /* Disable cache line streaming */
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_COMMAND),
         (OSD_INEXACT8(SCSI_AICREG(SCSI_COMMAND)) & ~SCSI_CACHETHEN)); 
   }

   if (!(hcntrl & SCSI_PAUSEACK))
   {
      SCSI_hWRITE_HCNTRL(hhcb,scsiRegister,hcntrl);
   }
}
#endif 

/*********************************************************************
*
*  SCSIhStandardResetCCCtl
*
*     Reset Command Channel SCB Control and S/G Control registers
*
*  Return Value:  None
*
*  Parameters:    hhcb
*
*  Remarks:
*                  
*********************************************************************/
#if (SCSI_STANDARD_MODE || SCSI_SWAPPING_ADVANCED_MODE || SCSI_SWAPPING_160M_MODE)
void SCSIhStandardResetCCCtl (SCSI_HHCB SCSI_HPTR hhcb)
{
   SCSI_REGISTER scsiRegister = hhcb->scsiRegister;

   OSD_OUTEXACT8(SCSI_AICREG(SCSI_CCSCBCTL),(SCSI_UEXACT8) 0);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_CCSGCTL),(SCSI_UEXACT8) 0);
}
#endif 

/*********************************************************************
*
*  SCSIhStandardInitCCHHAddr
*
*     Initialize Command Channel Host Address
*
*  Return Value:  None
*
*  Parameters:    hhcb
*
*  Remarks:
*                  
*********************************************************************/
#if (SCSI_STANDARD_MODE || SCSI_SWAPPING_ADVANCED_MODE || SCSI_SWAPPING_160M_MODE)
void SCSIhStandardInitCCHHAddr (SCSI_HHCB SCSI_HPTR hhcb)
{
   SCSI_REGISTER scsiRegister = hhcb->scsiRegister;

   OSD_OUTEXACT8(SCSI_AICREG(SCSI_CCHHADDR0),(SCSI_UEXACT8) 0);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_CCHHADDR1),(SCSI_UEXACT8) 0);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_CCHHADDR2),(SCSI_UEXACT8) 0);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_CCHHADDR3),(SCSI_UEXACT8) 0);
}
#endif 

/*********************************************************************
*
*  SCSIhAIC78XXCheckExtArbAckEnBit
*
*     Check External Arbitration enable bit
*
*  Return Value:  None
*
*  Parameters:    hhcb
*
*  Remarks:       0: Arbitration bit enabled
*                 1: Arbitration bit disabled
*                  
*********************************************************************/
#if SCSI_AIC78XX
void SCSIhAIC78XXCheckExtArbAckEnBit (SCSI_HHCB SCSI_HPTR hhcb)
{
   SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_UINT32 i = 0;          /* time out counter */

   while ((!(OSD_INEXACT8(SCSI_AICREG(SCSI_SEEPROM)) & SCSI_SEERDY)) &&
          (i++ < 400000))
   {
      OSD_TIMER(hhcb);  
   }
}
#endif /* SCSI_AIC78XX */
   
/*********************************************************************
*
*  SCSIhAICBayonetCheckExtArbAckEnBit
*
*     Check External Arbitration enable bit
*
*  Return Value:  None
*
*  Parameters:    hhcb
*
*  Remarks:       0: Arbitration bit disabled
*                 1: Arbitration bit enabled 
*********************************************************************/
#if (SCSI_AICBAYONET || SCSI_AICTRIDENT)
void SCSIhAICBayonetCheckExtArbAckEnBit (SCSI_HHCB SCSI_HPTR hhcb)
{
   SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_UINT32 i = 0;          /* time out counter */

   while ((!(OSD_INEXACT8(SCSI_AICREG(SCSI_SEEPROM)) & SCSI_SEERDY)) &&
          (i++ < 400000))
   {
      OSD_TIMER(hhcb);
   }
}   
#endif /* SCSI_AICBAYONET || SCSI_AICTRIDENT */

/*********************************************************************
*
*  SCSIhAIC78XXGetSpeed
*
*     Calculates the Speed in tenths of Mtransfers/second.
*
*  Return Value:  Speed.
*                  
*  Parameters:    scsiRate    - format of SCSIRATE register
*                 ultraSpeed  - 0 if non-ultra, 1 if ultra
*
*  Remarks:       For example a value of 100 returned means 10 
*                 Mtransfers/second.
*                 
*********************************************************************/
#if SCSI_AIC78XX
SCSI_UEXACT16 SCSIhAIC78XXGetSpeed (SCSI_UEXACT16 scsiRate,
                                    SCSI_UEXACT8 ultraSpeed)
{
   SCSI_UEXACT16 speed = 36;
   
   if (ultraSpeed)
   {
      switch(scsiRate & SCSI_SYNC_RATE)
      {
         case 0x00:
            speed = 200;
            break;
         case 0x10:
            speed = 160;
            break;
         case 0x20:
            speed = 133;
            break;
      }
   }
   else
   {
      switch(scsiRate & SCSI_SYNC_RATE)
      {
         case 0x00:
            speed = 100;
            break;
         case 0x10:
            speed = 80;
            break;
         case 0x20:
            speed = 67;
            break;
         case 0x30:
            speed = 57;
            break;
         case 0x40:
            speed = 50;
            break;
         case 0x50:
            speed = 44;
            break;
         case 0x60:
            speed = 40;
            break;
         case 0x70:
            speed = 36;
            break;
      }
   }
   return(speed);
}
#endif /* SCSI_AIC78XX */

/*********************************************************************
*
*  SCSIhAICBayonetGetSpeed
*
*     Calculates the Speed in tenths of Mtransfers/second.
*
*  Return Value:  Speed.
*                  
*  Parameters:    scsiRate    - format of SCSIRATE register
*                 ultraSpeed  - 0 if non-ultra, 1 if ultra
*
*  Remarks:       For example a value of 100 returned means 10 
*                 Mtransfers/second.
*                 
*********************************************************************/
#if SCSI_AICBAYONET
SCSI_UEXACT16 SCSIhAICBayonetGetSpeed (SCSI_UEXACT16 scsiRate,
                                       SCSI_UEXACT8 ultraSpeed)
{
   SCSI_UEXACT16 speed = 50;
   
   switch(scsiRate & SCSI_BAYONET_XFERRATE)
   {
      case 0x13:
         speed = 400;
         break;
      case 0x14:
         speed = 267;
         break;
      case 0x15:
         speed = 200;
         break;
      case 0x16:
         speed = 160;
         break;
      case 0x17:
         speed = 133;
         break;
      case 0x18:
         speed = 100;
         break;
      case 0x19:
         speed = 80;
         break;
      case 0x1A:
         speed = 67;
         break;
      case 0x1B:
         speed = 57;
         break;
      case 0x1C:
         speed = 50;
         break;
   }
   return(speed);
}
#endif /* SCSI_AICBAYONET */

/*********************************************************************
*
*  SCSIhAICTridentGetSpeed
*
*     Calculates the Speed in tenths of Mtransfers/second.
*
*  Return Value:  Speed.
*                  
*  Parameters:    scsiRate    - format of SCSIRATE register
*                 ultraSpeed  - 0 if non-ultra, 1 if ultra
*
*  Remarks:       For example a value of 100 returned means 10 
*                 Mtransfers/second.
*                 
*********************************************************************/
#if SCSI_AICTRIDENT
SCSI_UEXACT16 SCSIhAICTridentGetSpeed (SCSI_UEXACT16 scsiRate,
                                       SCSI_UEXACT8 ultraSpeed)
{
   SCSI_UEXACT16 speed = 50;
   
#if SCSI_PPR_ENABLE
   if ((scsiRate & SCSI_ENABLE_CRC) && !(scsiRate & SCSI_SINGLE_EDGE))
   {
      speed = 100;
      switch(scsiRate & SCSI_TRIDENT_XFERRATE)
      {
         case 0x02:
            speed = 800;
            break;
         case 0x03:
            speed = 400;
            break;
         case 0x04:
            speed = 267;
            break;
         case 0x05:
            speed = 200;
            break;
         case 0x06:
            speed = 160;
            break;
         case 0x07:
            speed = 133;
            break;
         case 0x08:
            speed = 100;
            break;
      }
   }
   else
   {
      switch(scsiRate & SCSI_TRIDENT_XFERRATE)
      {
         case 0x03:
            speed = 400;
            break;
         case 0x04:
            speed = 267;
            break;
         case 0x05:
            speed = 200;
            break;
         case 0x06:
            speed = 160;
            break;
         case 0x07:
            speed = 133;
            break;
         case 0x08:
            speed = 100;
            break;
         case 0x09:
            speed = 80;
            break;
         case 0x0A:
            speed = 67;
            break;
         case 0x0B:
            speed = 57;
            break;
         case 0x0C:
            speed = 50;
            break;
      }
   }
#else /* !SCSI_PPR_ENABLE */
   switch(scsiRate & SCSI_TRIDENT_XFERRATE)
   {
      case 0x03:
         speed = 400;
         break;
      case 0x04:
         speed = 267;
         break;
      case 0x05:
         speed = 200;
         break;
      case 0x06:
         speed = 160;
         break;
      case 0x07:
         speed = 133;
         break;
      case 0x08:
         speed = 100;
         break;
      case 0x09:
         speed = 80;
         break;
      case 0x0A:
         speed = 67;
         break;
      case 0x0B:
         speed = 57;
         break;
      case 0x0C:
         speed = 50;
         break;
   }
#endif /* SCSI_PPR_ENABLE */
   return(speed);
}
#endif /* SCSI_AICTRIDENT */

/*********************************************************************
*
*  SCSIhAIC78XXCalcScsiOption
*
*     Calculates the scsi synch negotiation option in the form of the
*  SCSIRATE register.
*
*  Return Value:  scsioption - in form of SCSIRATE register
*
*  Parameters:    speed      - Speed in tenths of Mtransfers/sec
*                 offset     - SCSI offset
*                 width      - Number of bits in data bus
*                 dtcEnable  - Dual Transition Clocking enable 
*
*  Remarks:       None.
*
*********************************************************************/
#if SCSI_AIC78XX
SCSI_UEXACT16 SCSIhAIC78XXCalcScsiOption (SCSI_UEXACT16 speed,
                                          SCSI_UEXACT8 offset,
                                          SCSI_UEXACT8 width,
                                          SCSI_UEXACT8 dtcEnable)
{
   SCSI_UEXACT8 scsiRate = 0;

   /* Calculate Period */
   if (speed >= 200)
      scsiRate = 0x00;
   else if (speed < 200 && speed >= 160)
      scsiRate = 0x10;
   else if (speed < 160 && speed >= 133)
      scsiRate = 0x20;
   else if (speed < 133 && speed >= 100)
      scsiRate = 0x00;
   else if (speed < 100 && speed >= 80)
      scsiRate = 0x10;
   else if (speed < 80 && speed >= 67)
      scsiRate = 0x20;
   else if (speed < 67 && speed >= 57)
      scsiRate = 0x30;
   else if (speed < 57 && speed >= 50)
      scsiRate = 0x40;
   else if (speed < 50 && speed >= 44)
      scsiRate = 0x50;
   else if (speed < 44 && speed >= 40)
      scsiRate = 0x60;
   else if (speed < 40 && speed >= 36)
      scsiRate = 0x70;

   /* Calculate Offset */

   /* As the actual offset value will be calculated at the time of       */
   /* negotiation, we need to just indicate whether sync or async xfer   */
   /* in Offset variable.                                                */
   if (offset != 0) offset = SCSI_SYNC_XFER;
   /* Put the offset in the appropriate place in ScsiRate variable.      */
   scsiRate |= (offset & SCSI_SOFS);

   /* Calculate Width */
   if (width == 16) scsiRate |= SCSI_WIDEXFER;

   return((SCSI_UEXACT16)scsiRate);
}
#endif /* SCSI_AIC78XX */

/*********************************************************************
*
*  SCSIhAICBayonetCalcScsiOption
*
*     Calculates the scsi synch negotiation option in the form of the
*  SCSIRATE+SCSIOFFSET registers.
*
*  Return Value:  scsioption - in form of SCSIRATE register
*
*  Parameters:    speed      - Speed in tenths of Mtransfers/sec
*                 offset     - SCSI offset
*                 width      - Number of bits in data bus
*                 dtcEnable  - Dual Transition Clocking enable 
*
*  Remarks:       None.
*
*********************************************************************/
#if SCSI_AICBAYONET
SCSI_UEXACT16 SCSIhAICBayonetCalcScsiOption (SCSI_UEXACT16 speed,
                                             SCSI_UEXACT8 offset,
                                             SCSI_UEXACT8 width,
                                             SCSI_UEXACT8 dtcEnable)
{
   SCSI_UEXACT16 scsiRate = 0;

   /* Calculate Period */
   if (speed >= 400)
      scsiRate = 0x13;
   else if (speed < 400 && speed >= 267)
      scsiRate = 0x14;
   else if (speed < 267 && speed >= 200)
      scsiRate = 0x15;
   else if (speed < 200 && speed >= 160)
      scsiRate = 0x16;
   else if (speed < 160 && speed >= 133)
      scsiRate = 0x17;
   else if (speed < 133 && speed >= 100)
      scsiRate = 0x18;
   else if (speed < 100 && speed >= 80)
      scsiRate = 0x19;
   else if (speed < 80 && speed >= 67)
      scsiRate = 0x1A;
   else if (speed < 67 && speed >= 57)
      scsiRate = 0x1B;
   else if (speed < 57 && speed >= 50)
      scsiRate = 0x1C;

   /* Calculate Offset */
   scsiRate |= (SCSI_UEXACT16)(offset & SCSI_BAYONET_OFFSET) << 8;

   /* Calculate Width */
   if (width == 16) scsiRate |= SCSI_WIDEXFER;

   return(scsiRate);
}
#endif /* SCSI_AICBAYONET */

/*********************************************************************
*
*  SCSIhAICTridentCalcScsiOption
*
*     Calculates the scsi synch negotiation option in the form of the
*  SCSIRATE+SCSIOFFSET registers.
*
*  Return Value:  scsioption - in form of SCSIRATE register
*
*  Parameters:    speed      - Speed in tenths of Mtransfers/sec
*                 offset     - SCSI offset
*                 width      - Number of bits in data bus
*                 dtcEnable  - Dual Transition Clocking enable 
*
*  Remarks:       None.
*
*********************************************************************/
#if SCSI_AICTRIDENT
SCSI_UEXACT16 SCSIhAICTridentCalcScsiOption (SCSI_UEXACT16 speed,
                                             SCSI_UEXACT8 offset,
                                             SCSI_UEXACT8 width,
                                             SCSI_UEXACT8 dtcEnable)
{
   SCSI_UEXACT16 scsiRate = 0;

#if SCSI_PPR_ENABLE
   /* Calculate Period */
   if (dtcEnable)
   {
      if (speed >= 800)
         scsiRate = 0x42;
      else if (speed >= 400)
         scsiRate = 0x43;
      else if (speed < 400 && speed >= 267)
         scsiRate = 0x44;
      else if (speed < 267 && speed >= 200)
         scsiRate = 0x45;
      else if (speed < 200 && speed >= 160)
         scsiRate = 0x46;
      else if (speed < 160 && speed >= 133)
         scsiRate = 0x47;
      else if (speed < 133 && speed >= 100)
         scsiRate = 0x48;
   }
   else
   {
      /* Calculate Period */
      if (speed >= 400)
         scsiRate = 0x13;
      else if (speed < 400 && speed >= 267)
         scsiRate = 0x14;
      else if (speed < 267 && speed >= 200)
         scsiRate = 0x15;
      else if (speed < 200 && speed >= 160)
         scsiRate = 0x16;
      else if (speed < 160 && speed >= 133)
         scsiRate = 0x17;
      else if (speed < 133 && speed >= 100)
         scsiRate = 0x18;
      else if (speed < 100 && speed >= 80)
         scsiRate = 0x19;
      else if (speed < 80 && speed >= 67)
         scsiRate = 0x1A;
      else if (speed < 67 && speed >= 57)
         scsiRate = 0x1B;
      else if (speed < 57 && speed >= 50)
         scsiRate = 0x1C;
   }
#else /* !SCSI_PPR_ENABLE */
   /* Calculate Period */
   if (speed >= 400)
      scsiRate = 0x13;
   else if (speed < 400 && speed >= 267)
      scsiRate = 0x14;
   else if (speed < 267 && speed >= 200)
      scsiRate = 0x15;
   else if (speed < 200 && speed >= 160)
      scsiRate = 0x16;
   else if (speed < 160 && speed >= 133)
      scsiRate = 0x17;
   else if (speed < 133 && speed >= 100)
      scsiRate = 0x18;
   else if (speed < 100 && speed >= 80)
      scsiRate = 0x19;
   else if (speed < 80 && speed >= 67)
      scsiRate = 0x1A;
   else if (speed < 67 && speed >= 57)
      scsiRate = 0x1B;
   else if (speed < 57 && speed >= 50)
      scsiRate = 0x1C;
#endif /* SCSI_PPR_ENABLE */

   /* Calculate Offset */
   scsiRate |= (SCSI_UEXACT16)(offset & SCSI_BAYONET_OFFSET) << 8;

   /* Calculate Width */
   if (width == 16) scsiRate |= SCSI_WIDEXFER;

   return(scsiRate);
}
#endif /* SCSI_AICTRIDENT */

/*********************************************************************
*
*  SCSIhAIC78XXGetProfileParameters
*
*     Gets the profile parameters from the hardware layer parameters.
*
*  Return Value:  None
*
*  Parameters:    hwInfo      - pointer to hw information structure
*                 hhcb        - pointer to hhcb structure
*
*  Remarks:       None.
*
*********************************************************************/
#if SCSI_AIC78XX
void SCSIhAIC78XXGetProfileParameters (SCSI_HW_INFORMATION SCSI_LPTR hwInfo, 
                                       SCSI_HHCB SCSI_HPTR hhcb)
{
   SCSI_DEVICE SCSI_DPTR deviceTable;
   SCSI_UEXACT8 scsiOption, i;
   SCSI_UEXACT16 currentXferOption;
   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   
   /* Gather all the xfer option profile parameters (Current, Default, and  */
   /* Max) for all the targets, in the hw information structure.            */
   for (i = 0; i < SCSI_MAXDEV; i++)
   {
      currentXferOption = SCSI_hGETXFEROPTION(hhcb,i);
      deviceTable = &SCSI_DEVICE_TABLE(hhcb)[i];
      scsiOption = deviceTable->scsiOption;

      if (deviceTable->SCSI_DF_ultraEnable)
      {
         hwInfo->targetInfo[i].SCSIMaxSpeed = 200;
      }
      else
      {
         hwInfo->targetInfo[i].SCSIMaxSpeed = 100;
      }

      hwInfo->targetInfo[i].SCSIDefaultSpeed = SCSIhAIC78XXGetSpeed(scsiOption, 
               (SCSI_UEXACT8)deviceTable->SCSI_DF_ultraEnable);

      hwInfo->targetInfo[i].SCSICurrentSpeed =
               SCSIhAIC78XXGetSpeed(currentXferOption,
                                    (SCSI_UEXACT8)SCSI_hGETFAST20REG(hhcb,i));
        
      hwInfo->targetInfo[i].SCSICurrentOffset =
               (SCSI_UEXACT8)(SCSI_SOFS & currentXferOption);
         
      hwInfo->targetInfo[i].SCSIDefaultOffset = 0;      
      if (scsiOption & SCSI_WIDE_XFER)
      {
         if (SCSI_SOFS & scsiOption)
         {
            hwInfo->targetInfo[i].SCSIDefaultOffset = SCSI_WIDE_OFFSET;      
         }
      }
      else
      {
         if (SCSI_SOFS & scsiOption)
         {
            hwInfo->targetInfo[i].SCSIDefaultOffset = SCSI_NARROW_OFFSET;      
         }
      }

      if (currentXferOption & SCSI_WIDE_XFER)
      {
         hwInfo->targetInfo[i].SCSIMaxOffset = SCSI_WIDE_OFFSET;
      }
      else
      {
         hwInfo->targetInfo[i].SCSIMaxOffset = SCSI_NARROW_OFFSET;
      }

      if (OSD_INEXACT8(SCSI_AICREG(SCSI_SBLKCTL)) & SCSI_SELWIDE)
      {
         hwInfo->targetInfo[i].SCSIMaxWidth = 16;
      }
      else
      {
         hwInfo->targetInfo[i].SCSIMaxWidth = 8;
      }
   
      if (scsiOption & SCSI_WIDE_XFER)
      {
         hwInfo->targetInfo[i].SCSIDefaultWidth = 16;
      }
      else
      {
         hwInfo->targetInfo[i].SCSIDefaultWidth = 8;
      }
   
      if (currentXferOption & SCSI_WIDE_XFER)
      {
         hwInfo->targetInfo[i].SCSICurrentWidth = 16;
      }
      else
      {
         hwInfo->targetInfo[i].SCSICurrentWidth = 8;
      }

      hwInfo->targetInfo[i].SCSI_TF_scsiOptChanged = 0;

      hwInfo->targetInfo[i].SCSICurrentProtocolOption=SCSI_NO_PROTOCOL_OPTION;
      hwInfo->targetInfo[i].SCSIDefaultProtocolOption=SCSI_NO_PROTOCOL_OPTION;
   }

   /* Get the minimum speed for synchronous xfer */
   hwInfo->minimumSyncSpeed = 36;

   /* Transceiver mode is undefined for these AICs */
   hwInfo->SCSI_PF_transceiverMode = SCSI_UNKNOWN_MODE;

}
#endif /* SCSI_AIC78XX */

/*********************************************************************
*
*  SCSIhAICBayonetGetProfileParameters
*
*     Gets the profile parameters from the hardware layer parameters.
*
*  Return Value:  None
*
*  Parameters:    hwInfo      - pointer to hw information structure
*                 hhcb        - pointer to hhcb structure
*
*  Remarks:       Assumes sequencer is paused for register access.
*
*********************************************************************/
#if SCSI_AICBAYONET
void SCSIhAICBayonetGetProfileParameters (SCSI_HW_INFORMATION SCSI_LPTR hwInfo, 
                                          SCSI_HHCB SCSI_HPTR hhcb)
{
   SCSI_DEVICE SCSI_DPTR deviceTable;
   SCSI_UEXACT8 scsiOption, i;
   SCSI_UEXACT16 currentXferOption;
   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_UEXACT8 sblkctl = OSD_INEXACT8(SCSI_AICREG(SCSI_SBLKCTL));
   
   /* Gather all the xfer option profile parameters (Current, Default, and  */
   /* Max) for all the targets, in the hw information structure.            */
   for (i = 0; i < SCSI_MAXDEV; i++)
   {
      currentXferOption = SCSI_hGETXFEROPTION(hhcb,i);
      deviceTable = &SCSI_DEVICE_TABLE(hhcb)[i];
      scsiOption = deviceTable->scsiOption;

      if((deviceTable->bayScsiRate & SCSI_BAYONET_XFERRATE) < 0x15)
      {
         hwInfo->targetInfo[i].SCSIMaxSpeed = 400;
      }
      else if((deviceTable->bayScsiRate & SCSI_BAYONET_XFERRATE) < 0x18)
      {
         hwInfo->targetInfo[i].SCSIMaxSpeed = 200;
      }
      else
      {
         hwInfo->targetInfo[i].SCSIMaxSpeed = 100;
      }

      hwInfo->targetInfo[i].SCSIDefaultSpeed =
               SCSIhAICBayonetGetSpeed(deviceTable->bayScsiRate, 0);

      hwInfo->targetInfo[i].SCSICurrentSpeed =
               SCSIhAICBayonetGetSpeed(currentXferOption, 0);
        
      hwInfo->targetInfo[i].SCSICurrentOffset =
               (SCSI_UEXACT8)((currentXferOption >> 8) & SCSI_BAYONET_OFFSET);
         
      hwInfo->targetInfo[i].SCSIDefaultOffset = deviceTable->bayScsiOffset;

      hwInfo->targetInfo[i].SCSIMaxOffset = SCSI_BAYONET_OFFSET;

      if (sblkctl & SCSI_SELWIDE)
      {
         hwInfo->targetInfo[i].SCSIMaxWidth = 16;
      }
      else
      {
         hwInfo->targetInfo[i].SCSIMaxWidth = 8;
      }
   
      if (scsiOption & SCSI_WIDE_XFER)
      {
         hwInfo->targetInfo[i].SCSIDefaultWidth = 16;
      }
      else
      {
         hwInfo->targetInfo[i].SCSIDefaultWidth = 8;
      }
   
      if (currentXferOption & SCSI_WIDE_XFER)
      {
         hwInfo->targetInfo[i].SCSICurrentWidth = 16;
      }
      else
      {
         hwInfo->targetInfo[i].SCSICurrentWidth = 8;
      }

      hwInfo->targetInfo[i].SCSI_TF_scsiOptChanged = 0;
      
      hwInfo->targetInfo[i].SCSICurrentProtocolOption = SCSI_NO_PROTOCOL_OPTION;
      hwInfo->targetInfo[i].SCSIDefaultProtocolOption = SCSI_NO_PROTOCOL_OPTION;
   }

   /* Get the minimum speed for synchronous xfer */
   hwInfo->minimumSyncSpeed = 50;
   
   /* Determine the transceiver mode */
   hwInfo->SCSI_PF_transceiverMode = SCSI_UNKNOWN_MODE;
   if (sblkctl & SCSI_BAYONET_XCVR)
   {
      if ((sblkctl & SCSI_BAYONET_ENAB20) && (!(sblkctl & SCSI_BAYONET_ENAB40)))
      {
         /* HVD mode */ 
         hwInfo->SCSI_PF_transceiverMode = SCSI_HVD_MODE;
      }
   }   
   else
   {
      if ((sblkctl & SCSI_BAYONET_ENAB40) &&
          (!(sblkctl & SCSI_BAYONET_ENAB20))) 
      {
         /* LVD mode */
         hwInfo->SCSI_PF_transceiverMode = SCSI_LVD_MODE;
      }
      else
      if ((!(sblkctl & SCSI_BAYONET_ENAB40)) &&
          (sblkctl & SCSI_BAYONET_ENAB20)) 
      {
         /* SE mode */
         hwInfo->SCSI_PF_transceiverMode = SCSI_SE_MODE;
      } 
   } 
    
}
#endif /* SCSI_AICBAYONET */

/*********************************************************************
*
*  SCSIhAICTridentGetProfileParameters
*
*     Gets the profile parameters from the hardware layer parameters.
*
*  Return Value:  None
*
*  Parameters:    hwInfo      - pointer to hw information structure
*                 hhcb        - pointer to hhcb structure
*
*  Remarks:       None.
*
*********************************************************************/
#if SCSI_AICTRIDENT
void SCSIhAICTridentGetProfileParameters (SCSI_HW_INFORMATION SCSI_LPTR hwInfo, 
                                          SCSI_HHCB SCSI_HPTR hhcb)
{
   SCSI_DEVICE SCSI_DPTR deviceTable;
   SCSI_UEXACT8 scsiOption, i;
   SCSI_UEXACT16 currentXferOption;
   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_UEXACT8 sblkctl = OSD_INEXACT8(SCSI_AICREG(SCSI_SBLKCTL));
   
   /* Gather all the xfer option profile parameters (Current, Default, and  */
   /* Max) for all the targets, in the hw information structure.            */
   for (i = 0; i < SCSI_MAXDEV; i++)
   {
      currentXferOption = SCSI_hGETXFEROPTION(hhcb,i);
      deviceTable = &SCSI_DEVICE_TABLE(hhcb)[i];
      scsiOption = deviceTable->scsiOption;

      /* Running Dual-Edge with CRC transfer mode */
      if ((deviceTable->bayScsiRate & SCSI_ENABLE_CRC) &&
          !(deviceTable->bayScsiRate & SCSI_SINGLE_EDGE))
      {
         hwInfo->targetInfo[i].SCSI_TF_dtcEnable = 1;

         /* Get the minimum speed for synchronous xfer */
         hwInfo->minimumSyncSpeed = 100;
         
         hwInfo->targetInfo[i].SCSIDefaultProtocolOption = SCSI_DT_DATA_WITH_CRC;         
      }
      else
      {
         hwInfo->targetInfo[i].SCSI_TF_dtcEnable = 0;

         /* Get the minimum speed for synchronous xfer */
         hwInfo->minimumSyncSpeed = 50;
         
         hwInfo->targetInfo[i].SCSIDefaultProtocolOption = SCSI_ST_DATA;         
      }

      /*  Retrieve current protocol option */
      if ((currentXferOption & SCSI_ENABLE_CRC) &&
          !(currentXferOption & SCSI_SINGLE_EDGE))
      {
         hwInfo->targetInfo[i].SCSICurrentProtocolOption = SCSI_DT_DATA_WITH_CRC;
      }
      else
      {
         hwInfo->targetInfo[i].SCSICurrentProtocolOption = SCSI_ST_DATA;
      }

      /* Get max. speed for dual or single edge transfer mode */
      if ((deviceTable->bayScsiRate & SCSI_TRIDENT_XFERRATE) < 0x03)
      {
         hwInfo->targetInfo[i].SCSIMaxSpeed = 800;
      }
      else if ((deviceTable->bayScsiRate & SCSI_TRIDENT_XFERRATE) < 0x05)
      {
         hwInfo->targetInfo[i].SCSIMaxSpeed = 400;
      }
      else if ((deviceTable->bayScsiRate & SCSI_TRIDENT_XFERRATE) < 0x08)
      {
         hwInfo->targetInfo[i].SCSIMaxSpeed = 200;
      }
      else
      {
         hwInfo->targetInfo[i].SCSIMaxSpeed = 100;
      }

      hwInfo->targetInfo[i].SCSIDefaultSpeed =
               SCSIhAICTridentGetSpeed(deviceTable->bayScsiRate, 0);

      hwInfo->targetInfo[i].SCSICurrentSpeed =
               SCSIhAICTridentGetSpeed(currentXferOption, 0);
        
      hwInfo->targetInfo[i].SCSICurrentOffset =
               (SCSI_UEXACT8)((currentXferOption >> 8) & SCSI_BAYONET_OFFSET);
         
      hwInfo->targetInfo[i].SCSIDefaultOffset = deviceTable->bayScsiOffset;

      hwInfo->targetInfo[i].SCSIMaxOffset = SCSI_BAYONET_OFFSET;

      if (OSD_INEXACT8(SCSI_AICREG(SCSI_SBLKCTL)) & SCSI_SELWIDE)
      {
         hwInfo->targetInfo[i].SCSIMaxWidth = 16;
      }
      else
      {
         hwInfo->targetInfo[i].SCSIMaxWidth = 8;
      }
   
      if (scsiOption & SCSI_WIDE_XFER)
      {
         hwInfo->targetInfo[i].SCSIDefaultWidth = 16;
      }
      else
      {
         hwInfo->targetInfo[i].SCSIDefaultWidth = 8;
      }
   
      if (currentXferOption & SCSI_WIDE_XFER)
      {
         hwInfo->targetInfo[i].SCSICurrentWidth = 16;
      }
      else
      {
         hwInfo->targetInfo[i].SCSICurrentWidth = 8;
      }

      hwInfo->targetInfo[i].SCSI_TF_scsiOptChanged = 0;
   }

   /* Determine the transceiver mode */
   hwInfo->SCSI_PF_transceiverMode = SCSI_UNKNOWN_MODE;

   if (sblkctl & SCSI_BAYONET_XCVR)
   {
      if ((sblkctl & SCSI_BAYONET_ENAB20) && (!(sblkctl & SCSI_BAYONET_ENAB40)))
      {
         /* HVD mode */ 
         hwInfo->SCSI_PF_transceiverMode = SCSI_HVD_MODE;
      }
   }   
   else
   {
      if ((sblkctl & SCSI_BAYONET_ENAB40) &&
          (!(sblkctl & SCSI_BAYONET_ENAB20))) 
      {
         /* LVD mode */
         hwInfo->SCSI_PF_transceiverMode = SCSI_LVD_MODE;
      }
      else
      if ((!(sblkctl & SCSI_BAYONET_ENAB40)) &&
          (sblkctl & SCSI_BAYONET_ENAB20)) 
      {
         /* SE mode */
         hwInfo->SCSI_PF_transceiverMode = SCSI_SE_MODE;
      } 
   } 
}
#endif /* SCSI_AICTRIDENT */

/*********************************************************************
*
*  SCSIhAIC78XXPutProfileParameters
*
*     Incorporates the changes in profile parameters into the hardware
*  layer parameters.
*
*  Return Value:  None
*
*  Parameters:    hwInfo      - pointer to hw information structure
*                 hhcb        - pointer to hhcb structure
*
*  Remarks:       None.
*
*********************************************************************/
#if SCSI_AIC78XX
void SCSIhAIC78XXPutProfileParameters (SCSI_HW_INFORMATION SCSI_LPTR hwInfo, 
                                       SCSI_HHCB SCSI_HPTR hhcb)
{
   SCSI_DEVICE SCSI_DPTR deviceTable;
   SCSI_UEXACT16 speed;
   SCSI_UEXACT8 offset;
   SCSI_UEXACT8 width;
   SCSI_UEXACT8 i;
#if !SCSI_NEGOTIATION_PER_IOB
   SCSI_UEXACT8 uexact8Value;
#endif /* !SCSI_NEGOTIATION_PER_IOB */
   
   /* For all the targets, incorporate the xfer option paramters from the  */
   /* hw information structure.                                            */
   for (i = 0; i < SCSI_MAXDEV; i++)
   {
      if (hwInfo->targetInfo[i].SCSI_TF_scsiOptChanged)
      {
         deviceTable = &SCSI_DEVICE_TABLE(hhcb)[i];
         speed = hwInfo->targetInfo[i].SCSIDefaultSpeed;
         offset = hwInfo->targetInfo[i].SCSIDefaultOffset;
         width = hwInfo->targetInfo[i].SCSIDefaultWidth;

         deviceTable->scsiOption =
               (SCSI_UEXACT8)SCSIhAIC78XXCalcScsiOption(speed,offset,width,0);
#if SCSI_NEGOTIATION_PER_IOB
         if (offset)
         {
            deviceTable->origOffset = offset;
         }
#endif

         deviceTable->SCSI_DF_ultraEnable = (speed >= 133) ? 1 : 0;

#if !SCSI_NEGOTIATION_PER_IOB
         /* Force negotiation if device is idle */
         if (SCSIHCheckDeviceIdle(hhcb,i))
         {
            /* turn on the negotiation initiation */
            if (deviceTable->scsiOption & (SCSI_WIDE_XFER | SCSI_SYNC_XFER))
            {
               uexact8Value = SCSI_NEEDNEGO;
            }
            else
            {
               uexact8Value = 0;
            }

            SCSI_hXFEROPTASSIGN(hhcb,i,uexact8Value);
         }
#endif /* !SCSI_NEGOTIATION_PER_IOB */
      }
   }
}
#endif /* SCSI_AIC78XX */

/*********************************************************************
*
*  SCSIhAICBayonetPutProfileParameters
*
*     Incorporates the changes in profile parameters into the hardware
*  layer parameters.
*
*  Return Value:  None
*
*  Parameters:    hwInfo      - pointer to hw information structure
*                 hhcb        - pointer to hhcb structure
*
*  Remarks:       None.
*
*********************************************************************/
#if SCSI_AICBAYONET
void SCSIhAICBayonetPutProfileParameters (SCSI_HW_INFORMATION SCSI_LPTR hwInfo, 
                                          SCSI_HHCB SCSI_HPTR hhcb)
{
   SCSI_DEVICE SCSI_DPTR deviceTable;
   SCSI_UEXACT16 speed, defaultXferOption;
   SCSI_UEXACT8 offset;
   SCSI_UEXACT8 width;
   SCSI_UEXACT8 i;
   SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
#if !SCSI_NEGOTIATION_PER_IOB
   SCSI_UEXACT8 hcntrl, uexact8Value;
#endif /* !SCSI_NEGOTIATION_PER_IOB */
   
   /* For all the targets, incorporate the xfer option paramters from the  */
   /* hw information structure.                                            */
   for (i = 0; i < SCSI_MAXDEV; i++)
   {
      if (hwInfo->targetInfo[i].SCSI_TF_scsiOptChanged)
      {
         deviceTable = &SCSI_DEVICE_TABLE(hhcb)[i];
         speed = hwInfo->targetInfo[i].SCSIDefaultSpeed;
         offset = hwInfo->targetInfo[i].SCSIDefaultOffset;
         width = hwInfo->targetInfo[i].SCSIDefaultWidth;

         defaultXferOption = SCSIhAICBayonetCalcScsiOption(speed,offset,width,0);
         deviceTable->bayScsiRate = (SCSI_UEXACT8)(defaultXferOption);
         deviceTable->bayScsiOffset = (SCSI_UEXACT8)(defaultXferOption >> 8);
         deviceTable->bayDefaultRate = deviceTable->bayScsiRate;
         deviceTable->bayDefaultOffset = deviceTable->bayScsiOffset;
#if SCSI_NEGOTIATION_PER_IOB
         if (offset)
         {
            deviceTable->origOffset = offset;
         }
#endif

         deviceTable->scsiOption = 0;
         if (deviceTable->bayScsiRate & SCSI_WIDE_XFER)
            deviceTable->scsiOption |= SCSI_WIDE_XFER;
         if (deviceTable->bayScsiOffset & SCSI_BAYONET_OFFSET)
            deviceTable->scsiOption |= SCSI_SYNC_XFER;

#if !SCSI_NEGOTIATION_PER_IOB
         /* Force negotiation if device is idle */
         if (SCSIHCheckDeviceIdle(hhcb,i))
         {
            /* turn on the negotiation initiation */
            if (deviceTable->scsiOption & (SCSI_WIDE_XFER | SCSI_SYNC_XFER))
            {
               uexact8Value = SCSI_NEEDNEGO;
            }
            else
            {
               uexact8Value = 0;
            }

            /* For Bayonet, the chip should be paused before accessing  */
            /* scratch RAM or SCB array                                 */
            if (!((hcntrl = OSD_INEXACT8(SCSI_AICREG(SCSI_HCNTRL))) & SCSI_PAUSEACK))
            {
               SCSIhPauseAndWait(hhcb);
            }

            SCSI_hXFEROPTASSIGN(hhcb,i,uexact8Value);

            if (!(hcntrl & SCSI_PAUSEACK))
            {
               OSD_OUTEXACT8(SCSI_AICREG(SCSI_HCNTRL), hcntrl);
            }
         }
#endif /* !SCSI_NEGOTIATION_PER_IOB */
      }
   }
}
#endif /* SCSI_AICBAYONET */

/*********************************************************************
*
*  SCSIhAICTridentPutProfileParameters
*
*     Incorporates the changes in profile parameters into the hardware
*  layer parameters.
*
*  Return Value:  None
*
*  Parameters:    hwInfo      - pointer to hw information structure
*                 hhcb        - pointer to hhcb structure
*
*  Remarks:       None.
*
*********************************************************************/
#if SCSI_AICTRIDENT
void SCSIhAICTridentPutProfileParameters (SCSI_HW_INFORMATION SCSI_LPTR hwInfo, 
                                          SCSI_HHCB SCSI_HPTR hhcb)
{
   SCSI_DEVICE SCSI_DPTR deviceTable;
   SCSI_UEXACT16 speed, defaultXferOption;
   SCSI_UEXACT8 offset;
   SCSI_UEXACT8 width;
   SCSI_UEXACT8 i;
#if !SCSI_NEGOTIATION_PER_IOB
   SCSI_UEXACT8 uexact8Value;
#endif /* !SCSI_NEGOTIATION_PER_IOB */
   
   /* For all the targets, incorporate the xfer option paramters from the  */
   /* hw information structure.                                            */
   for (i = 0; i < SCSI_MAXDEV; i++)
   {
      if (hwInfo->targetInfo[i].SCSI_TF_scsiOptChanged)
      {
         deviceTable = &SCSI_DEVICE_TABLE(hhcb)[i];
         speed = hwInfo->targetInfo[i].SCSIDefaultSpeed;
         offset = hwInfo->targetInfo[i].SCSIDefaultOffset;
         width = hwInfo->targetInfo[i].SCSIDefaultWidth;

         /* @TL - need to revise, temporary workaround until the CHIM   */
         /* interface changes to support the Dual-Edge option in target */
         /* profile. For now, we're assuming Dual-Edge transfer enable  */
         /* if the default speed >= 800.  Workaround is for NT driver.  */
         if (speed >= 800)
         {
            hwInfo->targetInfo[i].SCSI_TF_dtcEnable = 1;
         }
         else
         {
            hwInfo->targetInfo[i].SCSI_TF_dtcEnable = 0;
         }

         defaultXferOption = SCSIhAICTridentCalcScsiOption(speed,offset,width,
                        (SCSI_UEXACT8)hwInfo->targetInfo[i].SCSI_TF_dtcEnable);
         deviceTable->bayScsiRate = (SCSI_UEXACT8)(defaultXferOption);
         deviceTable->bayScsiOffset = (SCSI_UEXACT8)(defaultXferOption >> 8);
         deviceTable->bayDefaultRate = deviceTable->bayScsiRate;
         deviceTable->bayDefaultOffset = deviceTable->bayScsiOffset;
#if SCSI_NEGOTIATION_PER_IOB
         if (offset)
         {
            deviceTable->origOffset = offset;
         }
#endif

         deviceTable->scsiOption = 0;
         if (deviceTable->bayScsiRate & SCSI_WIDE_XFER)
            deviceTable->scsiOption |= SCSI_WIDE_XFER;
         if (deviceTable->bayScsiOffset & SCSI_BAYONET_OFFSET)
            deviceTable->scsiOption |= SCSI_SYNC_XFER;

#if !SCSI_NEGOTIATION_PER_IOB
         /* Force negotiation if device is idle */
         if (SCSIHCheckDeviceIdle(hhcb,i))
         {
            /* turn on the negotiation initiation */
            if (deviceTable->scsiOption & (SCSI_WIDE_XFER | SCSI_SYNC_XFER))
            {
               uexact8Value = SCSI_NEEDNEGO;
            }
            else
            {
               uexact8Value = 0;
            }

            SCSI_hXFEROPTASSIGN(hhcb,i,uexact8Value);
         }
#endif /* !SCSI_NEGOTIATION_PER_IOB */
      }
   }
}
#endif /* SCSI_AICTRIDENT */

/*********************************************************************
*
*  SCSIhAIC78XXGetQNewTail
*
*     Gets the q_new_tail from the scratch ram.
*
*  Return Value:  None
*
*  Parameters:    hhcb        - pointer to hhcb structure
*
*  Remarks:       None.
*
*********************************************************************/
#if SCSI_AIC78XX
SCSI_UEXACT8 SCSIhAIC78XXGetQNewTail (SCSI_HHCB SCSI_HPTR hhcb)
{
   SCSI_REGISTER scsiRegister = hhcb->scsiRegister;

   return(OSD_INEXACT8(SCSI_AICREG(SCSI_hQ_NEW_TAIL(hhcb)))); 
}
#endif /* SCSI_AIC78XX */

/*********************************************************************
*
*  SCSIhAICBayonetGetQNewTail
*
*     Gets the q_new_tail from the SNSCB_QOFF register.
*
*  Return Value:  None
*
*  Parameters:    hhcb        - pointer to hhcb structure
*
*  Remarks:       None.
*
*********************************************************************/
#if (SCSI_AICBAYONET || SCSI_AICTRIDENT)
SCSI_UEXACT8 SCSIhAICBayonetGetQNewTail (SCSI_HHCB SCSI_HPTR hhcb)
{
   SCSI_REGISTER scsiRegister = hhcb->scsiRegister;

   SCSI_UEXACT8 qNewTail;

   qNewTail = OSD_INEXACT8(SCSI_AICREG(SCSI_hQ_NEW_TAIL(hhcb))); 

   /* As the SNSCB_QOFF register auto increments on a read, we need to write */
   /* back what we read (this will adjust the SCB_AVAIL and ROLLOVER bits in */
   /* the QOFF_CTLSTA register automatically, if needed).                    */
      
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_hQ_NEW_TAIL(hhcb)), qNewTail);


   return(qNewTail);

}
#endif /* SCSI_AICBAYONET || SCSI_AICTRIDENT */

/*********************************************************************
*
*  SCSIhAICBayonetWaitForBusStable
*
*     Wait for the SCSI bus stable by checking ENAB20 and ENAB40 bit.
*
*  Return Value:  None
*
*  Parameters:    hhcb        - pointer to hhcb structure
*
*  Remarks:       None.
*
*********************************************************************/
#if (SCSI_AICBAYONET || SCSI_AICTRIDENT)
void SCSIhAICBayonetWaitForBusStable (SCSI_HHCB SCSI_HPTR hhcb)
{
   SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_UINT32 count;                        /* time out counter */
   struct
   {
      SCSI_UEXACT8 preValue:1;
      SCSI_UEXACT8 curValue:1;
      SCSI_UEXACT8 Reserved:6;
   } clkout;

   /* Due to the Bayonet HW that needs to wait a total of 200ms for the scsi */
   /* bus to stable after the chip reset.  Before this time, it does not */
   /* allow any scsi signals to send to scsi bus.  Therefore, we must wait */
   /* for the scsi bus stable as well base on bit ENAB20 or ENAB40 that set */
   /* by the HW when it confirmed that the scsi bus is stabled. */

   /* CLKOUT provides a 102.40us period from a 40MHz clock-in source. */
   /* The hardware will wait for a total of 200ms before it turn on the */
   /* ENAB20 or ENAB40.  So, HIM will use at least 400ms for time-out to */
   /* cover some delay might happen. */

   /* count will be decremented by 1 at every time the clock out bit changed */
   /* its value.  Thus for every one CLKOUT period, the cnt will be */
   /* decremented by 2. */
   count = 8000;

   /* Get the current value of clkout to start with */
   clkout.preValue =
            (OSD_INEXACT8(SCSI_AICREG(SCSI_TARGIDIN)) & SCSI_CLKOUT) ? 1 : 0;

   while (1)
   {
      if (OSD_INEXACT8(SCSI_AICREG(SCSI_SBLKCTL)) & 
                 (SCSI_BAYONET_ENAB20 | SCSI_BAYONET_ENAB40))
      {
         break;
      }

      clkout.curValue =
            (OSD_INEXACT8(SCSI_AICREG(SCSI_TARGIDIN)) & SCSI_CLKOUT) ? 1 : 0;

      /* Check if current clock out value changes since last one */
      if (clkout.curValue ^ clkout.preValue)
      {
         /* Update new clkout value */
         clkout.preValue = clkout.curValue;
         if (!(--count))
         {
            /* time-out has reached */
            break;
         }
      }
   }
}
#endif /* SCSI_AICBAYONET || SCSI_AICTRIDENT */

/*********************************************************************
*
*  SCSIhSwapping32SetBreakPoint
*
*     Sets breakpoint at the given entry point.
*
*  Return Value:  None
*
*  Parameters:    hhcb
*                 entryBitMap
*
*  Remarks:
*                  
*********************************************************************/
#if SCSI_SWAPPING32_MODE
void SCSIhSwapping32SetBreakPoint (SCSI_HHCB SCSI_HPTR hhcb,
                                   SCSI_UEXACT8 entryBitMap)
{
   SCSI_REGISTER scsiRegister = hhcb->scsiRegister;

   /* Set breakpoint interrupt enable bit */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SEQCTL),
        OSD_INEXACT8(SCSI_AICREG(SCSI_SEQCTL)) | SCSI_BRKINTEN);

   /* Since the sequencer program counter always be one instruction */
   /* ahead, we need to add 4 to actual idle loop entry value. */
   /* One sequencer instruction is 29 bits. */
   if (entryBitMap & ENTRY_IDLE_LOOP)
   {
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_BRKADDR0),
                    (SCSI_UEXACT8)((SCSI_W32_IDLE_LOOP_ENTRY + 4) >> 2));
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_BRKADDR1),
                    (SCSI_UEXACT8)((SCSI_W32_IDLE_LOOP_ENTRY + 4) >> 10));
   }
}
#endif 

/*********************************************************************
*
*  SCSIhSwapping64SetBreakPoint
*
*     Sets breakpoint at the given entry point.
*
*  Return Value:  None
*
*  Parameters:    hhcb
*                 entryBitMap
*
*  Remarks:
*                  
*********************************************************************/
#if SCSI_SWAPPING64_MODE
void SCSIhSwapping64SetBreakPoint (SCSI_HHCB SCSI_HPTR hhcb,
                                   SCSI_UEXACT8 entryBitMap)
{
   SCSI_REGISTER scsiRegister = hhcb->scsiRegister;

   /* Set breakpoint interrupt enable bit */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SEQCTL),
        OSD_INEXACT8(SCSI_AICREG(SCSI_SEQCTL)) | SCSI_BRKINTEN);

   /* Since the sequencer program counter always be one instruction */
   /* ahead, we need to add 4 to actual idle loop entry value. */
   /* One sequencer instruction is 29 bits. */
   if (entryBitMap & ENTRY_IDLE_LOOP)
   {
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_BRKADDR0),
                    (SCSI_UEXACT8)((SCSI_W64_IDLE_LOOP_ENTRY + 4) >> 2));
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_BRKADDR1),
                    (SCSI_UEXACT8)((SCSI_W64_IDLE_LOOP_ENTRY + 4) >> 10));
   }
}
#endif 

/*********************************************************************
*
*  SCSIhSwappingAdvSetBreakPoint
*
*     Sets breakpoint at the given entry point.
*
*  Return Value:  None
*
*  Parameters:    hhcb
*                 entryBitMap
*
*  Remarks:
*                  
*********************************************************************/
#if SCSI_SWAPPING_ADVANCED_MODE
void SCSIhSwappingAdvSetBreakPoint (SCSI_HHCB SCSI_HPTR hhcb,
                                   SCSI_UEXACT8 entryBitMap)
{

   SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_UEXACT8 brk_cnt;

   if (entryBitMap & ENTRY_IDLE_LOOP)
   {
      brk_cnt = OSD_INEXACT8(SCSI_AICREG(SCSI_WADV_ENT_PT_BITMAP));
      brk_cnt |= (SCSI_UEXACT8)ENTRY_IDLE_LOOP;
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_WADV_ENT_PT_BITMAP),brk_cnt);
   }
   else if (entryBitMap & ENTRY_EXPANDER_BREAK)
   {
      brk_cnt = OSD_INEXACT8(SCSI_AICREG(SCSI_WADV_ENT_PT_BITMAP));
      brk_cnt |= (SCSI_UEXACT8)ENTRY_EXPANDER_BREAK;
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_WADV_ENT_PT_BITMAP),brk_cnt);
   }

}
#endif 

/*********************************************************************
*
*  SCSIhSwapping160mSetBreakPoint
*
*     Sets breakpoint at the given entry point.
*
*  Return Value:  None
*
*  Parameters:    hhcb
*                 entryBitMap
*
*  Remarks:
*                  
*********************************************************************/
#if SCSI_SWAPPING_160M_MODE
void SCSIhSwapping160mSetBreakPoint (SCSI_HHCB SCSI_HPTR hhcb,
                                     SCSI_UEXACT8 entryBitMap)
{
   SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_UEXACT8 brk_cnt;

   if (entryBitMap & ENTRY_IDLE_LOOP)
   {
      brk_cnt = OSD_INEXACT8(SCSI_AICREG(SCSI_W160M_ENT_PT_BITMAP));
      brk_cnt |= (SCSI_UEXACT8)ENTRY_IDLE_LOOP;
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_W160M_ENT_PT_BITMAP),brk_cnt);
   }
   else if (entryBitMap & ENTRY_EXPANDER_BREAK)
   {
      brk_cnt = OSD_INEXACT8(SCSI_AICREG(SCSI_W160M_ENT_PT_BITMAP));
      brk_cnt |= (SCSI_UEXACT8)ENTRY_EXPANDER_BREAK;
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_W160M_ENT_PT_BITMAP),brk_cnt);
   }

}
#endif 

/*********************************************************************
*
*  SCSIhStandard64SetBreakPoint
*
*     Sets breakpoint at the given entry point.
*
*  Return Value:  None
*
*  Parameters:    hhcb
*                 entryBitMap
*
*  Remarks:
*                  
*********************************************************************/
#if SCSI_STANDARD64_MODE
void SCSIhStandard64SetBreakPoint (SCSI_HHCB SCSI_HPTR hhcb,
                                   SCSI_UEXACT8 entryBitMap)
{
   SCSI_REGISTER scsiRegister = hhcb->scsiRegister;

   /* Set breakpoint interrupt enable bit */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SEQCTL),
        OSD_INEXACT8(SCSI_AICREG(SCSI_SEQCTL)) | SCSI_BRKINTEN);

   /* Since the sequencer program counter always be one instruction */
   /* ahead, we need to add 4 to actual idle loop entry value. */
   /* One sequencer instruction is 29 bits. */
   if (entryBitMap & ENTRY_IDLE_LOOP)
   {
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_BRKADDR0),
                    (SCSI_UEXACT8)((SCSI_S64_IDLE_LOOP0 + 4) >> 2));
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_BRKADDR1),
                    (SCSI_UEXACT8)((SCSI_S64_IDLE_LOOP0 + 4) >> 10));
   }
}
#endif 

/*********************************************************************
*
*  SCSIhStandardAdvSetBreakPoint
*
*     Sets breakpoint at the given entry point.
*
*  Return Value:  None
*
*  Parameters:    hhcb
*                 entryBitMap
*
*  Remarks:
*                  
*********************************************************************/
#if SCSI_STANDARD_ADVANCED_MODE
void SCSIhStandardAdvSetBreakPoint (SCSI_HHCB SCSI_HPTR hhcb,
                                    SCSI_UEXACT8 entryBitMap)
{
   SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_UEXACT8 brk_cnt;

   if (entryBitMap & ENTRY_IDLE_LOOP)
   {
      brk_cnt = OSD_INEXACT8(SCSI_AICREG(SCSI_SADV_ENT_PT_BITMAP));
      brk_cnt |= (SCSI_UEXACT8)ENTRY_IDLE_LOOP;
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SADV_ENT_PT_BITMAP),brk_cnt);
   }
   else if (entryBitMap & ENTRY_EXPANDER_BREAK)
   {
      brk_cnt = OSD_INEXACT8(SCSI_AICREG(SCSI_SADV_ENT_PT_BITMAP));
      brk_cnt |= (SCSI_UEXACT8)ENTRY_EXPANDER_BREAK;
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SADV_ENT_PT_BITMAP),brk_cnt);
   }
}
#endif 

/*********************************************************************
*
*  SCSIhStandard160mSetBreakPoint
*
*     Sets breakpoint at the given entry point.
*
*  Return Value:  None
*
*  Parameters:    hhcb
*                 entryBitMap
*
*  Remarks:
*                  
*********************************************************************/
#if SCSI_STANDARD_160M_MODE
void SCSIhStandard160mSetBreakPoint (SCSI_HHCB SCSI_HPTR hhcb,
                                     SCSI_UEXACT8 entryBitMap)
{
   SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_UEXACT8 brk_cnt;

   if (entryBitMap & ENTRY_IDLE_LOOP)
   {
      brk_cnt = OSD_INEXACT8(SCSI_AICREG(SCSI_S160M_ENT_PT_BITMAP));
      brk_cnt |= (SCSI_UEXACT8)ENTRY_IDLE_LOOP;
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_S160M_ENT_PT_BITMAP),brk_cnt);
   }
   else if (entryBitMap & ENTRY_EXPANDER_BREAK)
   {
      brk_cnt = OSD_INEXACT8(SCSI_AICREG(SCSI_S160M_ENT_PT_BITMAP));
      brk_cnt |= (SCSI_UEXACT8)ENTRY_EXPANDER_BREAK;
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_S160M_ENT_PT_BITMAP),brk_cnt);
   }
}
#endif 

/*********************************************************************
*
*  SCSIhSwapping32ClearBreakPoint
*
*     Clears breakpoint at the given entry point.
*
*  Return Value:  None
*                 entryBitMap
*
*  Parameters:    hhcb
*
*  Remarks:
*                  
*********************************************************************/
#if SCSI_SWAPPING32_MODE
void SCSIhSwapping32ClearBreakPoint (SCSI_HHCB SCSI_HPTR hhcb,
                                     SCSI_UEXACT8 entryBitMap)
{
   SCSI_REGISTER scsiRegister = hhcb->scsiRegister;

   /* Clear break point interrupt by setting break disable bit */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_BRKADDR1), SCSI_BRKDIS);
}
#endif 

/*********************************************************************
*
*  SCSIhSwapping64ClearBreakPoint
*
*     Clears breakpoint at the given entry point.
*
*  Return Value:  None
*
*  Parameters:    hhcb
*                 entryBitMap
*
*  Remarks:
*                  
*********************************************************************/
#if SCSI_SWAPPING64_MODE
void SCSIhSwapping64ClearBreakPoint (SCSI_HHCB SCSI_HPTR hhcb,
                                     SCSI_UEXACT8 entryBitMap)
{
   SCSI_REGISTER scsiRegister = hhcb->scsiRegister;

   /* Clear break point interrupt by setting break disable bit */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_BRKADDR1), SCSI_BRKDIS);
}
#endif 

/*********************************************************************
*
*  SCSIhSwappingAdvClearBreakPoint
*
*     Clears breakpoint at the given entry point.
*
*  Return Value:  None
*                 entryBitMap
*
*  Parameters:    hhcb
*
*  Remarks:
*                  
*********************************************************************/
#if SCSI_SWAPPING_ADVANCED_MODE
void SCSIhSwappingAdvClearBreakPoint (SCSI_HHCB SCSI_HPTR hhcb,
                                      SCSI_UEXACT8 entryBitMap)
{
   SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_UEXACT8 brk_cnt;

   if (entryBitMap & ENTRY_IDLE_LOOP)
   {
      brk_cnt = OSD_INEXACT8(SCSI_AICREG(SCSI_WADV_ENT_PT_BITMAP));
      brk_cnt &= ~(SCSI_UEXACT8)ENTRY_IDLE_LOOP;
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_WADV_ENT_PT_BITMAP),brk_cnt);
   }
   else if (entryBitMap & ENTRY_EXPANDER_BREAK)
   {
      brk_cnt = OSD_INEXACT8(SCSI_AICREG(SCSI_WADV_ENT_PT_BITMAP));
      brk_cnt &= ~(SCSI_UEXACT8)ENTRY_EXPANDER_BREAK;
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_WADV_ENT_PT_BITMAP),brk_cnt);
   }

}
#endif 

/*********************************************************************
*
*  SCSIhSwapping160mClearBreakPoint
*
*     Clears breakpoint at the given entry point.
*
*  Return Value:  None
*
*  Parameters:    hhcb
*                 entryBitMap
*
*  Remarks:
*                  
*********************************************************************/
#if SCSI_SWAPPING_160M_MODE
void SCSIhSwapping160mClearBreakPoint (SCSI_HHCB SCSI_HPTR hhcb,
                                       SCSI_UEXACT8 entryBitMap)
{
   SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_UEXACT8 brk_cnt;

   if (entryBitMap & ENTRY_IDLE_LOOP)
   {
      brk_cnt = OSD_INEXACT8(SCSI_AICREG(SCSI_W160M_ENT_PT_BITMAP));
      brk_cnt &= ~(SCSI_UEXACT8)ENTRY_IDLE_LOOP;
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_W160M_ENT_PT_BITMAP),brk_cnt);
   }
   else if (entryBitMap & ENTRY_EXPANDER_BREAK)
   {
      brk_cnt = OSD_INEXACT8(SCSI_AICREG(SCSI_W160M_ENT_PT_BITMAP));
      brk_cnt &= ~(SCSI_UEXACT8)ENTRY_EXPANDER_BREAK;
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_W160M_ENT_PT_BITMAP),brk_cnt);
   }

}
#endif 

/*********************************************************************
*
*  SCSIhStandard64ClearBreakPoint
*
*     Clears breakpoint at the given entry point.
*
*  Return Value:  None
*
*  Parameters:    hhcb
*                 entryBitMap
*
*  Remarks:
*                  
*********************************************************************/
#if SCSI_STANDARD64_MODE
void SCSIhStandard64ClearBreakPoint (SCSI_HHCB SCSI_HPTR hhcb,
                                     SCSI_UEXACT8 entryBitMap)
{
   SCSI_REGISTER scsiRegister = hhcb->scsiRegister;

   /* Clear break point interrupt by setting break disable bit */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_BRKADDR1), SCSI_BRKDIS);
}
#endif 

/*********************************************************************
*
*  SCSIhStandardAdvClearBreakPoint
*
*     Clears breakpoint at the given entry point.
*
*  Return Value:  None
*
*  Parameters:    hhcb
*                 entryBitMap
*
*  Remarks:
*                  
*********************************************************************/
#if SCSI_STANDARD_ADVANCED_MODE
void SCSIhStandardAdvClearBreakPoint (SCSI_HHCB SCSI_HPTR hhcb,
                                      SCSI_UEXACT8 entryBitMap)
{
   SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_UEXACT8 brk_cnt;

   if (entryBitMap & ENTRY_IDLE_LOOP)
   {
      brk_cnt = OSD_INEXACT8(SCSI_AICREG(SCSI_SADV_ENT_PT_BITMAP));
      brk_cnt &= ~(SCSI_UEXACT8)ENTRY_IDLE_LOOP;
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SADV_ENT_PT_BITMAP),brk_cnt);
   }
   else if (entryBitMap & ENTRY_EXPANDER_BREAK)
   {
      brk_cnt = OSD_INEXACT8(SCSI_AICREG(SCSI_SADV_ENT_PT_BITMAP));
      brk_cnt &= ~(SCSI_UEXACT8)ENTRY_EXPANDER_BREAK;
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SADV_ENT_PT_BITMAP),brk_cnt);
   }
}
#endif 

/*********************************************************************
*
*  SCSIhStandard160mClearBreakPoint
*
*     Clears breakpoint at the given entry point.
*
*  Return Value:  None
*
*  Parameters:    hhcb
*                 entryBitMap
*
*  Remarks:
*                  
*********************************************************************/
#if SCSI_STANDARD_160M_MODE
void SCSIhStandard160mClearBreakPoint (SCSI_HHCB SCSI_HPTR hhcb,
                                       SCSI_UEXACT8 entryBitMap)
{
   SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_UEXACT8 brk_cnt;

   if (entryBitMap & ENTRY_IDLE_LOOP)
   {
      brk_cnt = OSD_INEXACT8(SCSI_AICREG(SCSI_S160M_ENT_PT_BITMAP));
      brk_cnt &= ~(SCSI_UEXACT8)ENTRY_IDLE_LOOP;
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_S160M_ENT_PT_BITMAP),brk_cnt);
   }
   else if (entryBitMap & ENTRY_EXPANDER_BREAK)
   {
      brk_cnt = OSD_INEXACT8(SCSI_AICREG(SCSI_S160M_ENT_PT_BITMAP));
      brk_cnt &= ~(SCSI_UEXACT8)ENTRY_EXPANDER_BREAK;
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_S160M_ENT_PT_BITMAP),brk_cnt);
   }
}
#endif 

/*********************************************************************
*
*  SCSIhAICSeqStretchWorkaround
*
*     Clear the CIOBUS stretch signal by reading a register that will
*  set this signal and deassert it.  Without this workaround, if the 
*  chip is paused while accessing scb ram then accesses to certain
*  registers will hang the system (infinite retries by host).
*
*  Return Value:  None
*
*  Parameters:    hhcb        - pointer to hhcb structure
*
*  Remarks:       None.
*
*********************************************************************/
#if SCSI_AICBAYONET
void SCSIhAICBayonetSeqStretchWorkaround (SCSI_HHCB SCSI_HPTR hhcb)
{
   SCSI_REGISTER scsiRegister = hhcb->scsiRegister;

   OSD_INEXACT8(SCSI_AICREG(SCSI_CCSCBCTL));
}
#endif /* SCSI_AICBAYONET */

/*********************************************************************
*
*  SCSIhStandardAdvSetIntrThreshold
*
*     Sets the interrupt threshold in the scratch register
*
*  Return Value:  None
*
*  Parameters:    hhcb
*
*  Remarks:
*                  
*********************************************************************/
#if SCSI_STANDARD_ADVANCED_MODE
void SCSIhStandardAdvSetIntrThreshold (SCSI_HHCB SCSI_HPTR hhcb,
                                       SCSI_UEXACT8 threshold)
{
   SCSI_REGISTER scsiRegister = hhcb->scsiRegister;

   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SADV_INTR_THRSH_TIMEOUT1), threshold);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SADV_INTR_THRSH_TIMEOUT2), threshold);
}
#endif 

/*********************************************************************
*
*  SCSIhSwappingAdvSetIntrThreshold
*
*     Sets the interrupt threshold in the scratch register
*
*  Return Value:  None
*
*  Parameters:    hhcb
*
*  Remarks:
*                  
*********************************************************************/
#if SCSI_SWAPPING_ADVANCED_MODE
void SCSIhSwappingAdvSetIntrThreshold (SCSI_HHCB SCSI_HPTR hhcb,
                                       SCSI_UEXACT8 threshold)
{
   SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
#if !SCSI_TARGET_OPERATION
/* not supported in target mode sequencer yet */ 
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_WADV_INTR_THRSH_TIMEOUT1), threshold);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_WADV_INTR_THRSH_TIMEOUT2), threshold);
#endif /* !SCSI_TARGET_OPERATION */
}
#endif 

/*********************************************************************
*
*  SCSIhStandard160mSetIntrThreshold
*
*     Sets the interrupt threshold in the scratch register
*
*  Return Value:  None
*
*  Parameters:    hhcb
*
*  Remarks:
*                  
*********************************************************************/
#if SCSI_STANDARD_160M_MODE
void SCSIhStandard160mSetIntrThreshold (SCSI_HHCB SCSI_HPTR hhcb,
                                        SCSI_UEXACT8 threshold)
{
   SCSI_REGISTER scsiRegister = hhcb->scsiRegister;

   OSD_OUTEXACT8(SCSI_AICREG(SCSI_S160M_INTR_THRSH_TIMEOUT1), threshold);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_S160M_INTR_THRSH_TIMEOUT2), threshold);
}
#endif 

/*********************************************************************
*
*  SCSIhSwapping160mSetIntrThreshold
*
*     Sets the interrupt threshold in the scratch register
*
*  Return Value:  None
*
*  Parameters:    hhcb
*
*  Remarks:
*                  
*********************************************************************/
#if SCSI_SWAPPING_160M_MODE
void SCSIhSwapping160mSetIntrThreshold (SCSI_HHCB SCSI_HPTR hhcb,
                                        SCSI_UEXACT8 threshold)
{
   SCSI_REGISTER scsiRegister = hhcb->scsiRegister;

   OSD_OUTEXACT8(SCSI_AICREG(SCSI_W160M_INTR_THRSH_TIMEOUT1), threshold);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_W160M_INTR_THRSH_TIMEOUT2), threshold);
}
#endif 

/*********************************************************************
*
*  SCSIhStandard64GetIntrThreshold
*
*     this mode does not support interrupt batching
*
*  Return Value:  None
*
*  Parameters:    hhcb
*
*  Remarks:
*                  
*********************************************************************/
#if SCSI_STANDARD64_MODE
SCSI_UEXACT8 SCSIhStandard64GetIntrThreshold (SCSI_HHCB SCSI_HPTR hhcb)
{
   return ((SCSI_UEXACT8)0xFF);
}
#endif 

/*********************************************************************
*
*  SCSIhStandardAdvGetIntrThreshold
*
*     Gets the interrupt threshold from the scratch register
*
*  Return Value:  None
*
*  Parameters:    hhcb
*
*  Remarks:
*                  
*********************************************************************/
#if SCSI_STANDARD_ADVANCED_MODE
SCSI_UEXACT8 SCSIhStandardAdvGetIntrThreshold (SCSI_HHCB SCSI_HPTR hhcb)
{
   SCSI_REGISTER scsiRegister = hhcb->scsiRegister;

   return (OSD_INEXACT8(SCSI_AICREG(SCSI_SADV_INTR_THRSH_TIMEOUT1)) & 0x7F);
}
#endif 

/*********************************************************************
*
*  SCSIhStandard160mGetIntrThreshold
*
*     Gets the interrupt threshold from the scratch register
*
*  Return Value:  None
*
*  Parameters:    hhcb
*
*  Remarks:
*                  
*********************************************************************/
#if SCSI_STANDARD_160M_MODE
SCSI_UEXACT8 SCSIhStandard160mGetIntrThreshold (SCSI_HHCB SCSI_HPTR hhcb)
{
   SCSI_REGISTER scsiRegister = hhcb->scsiRegister;

   return (OSD_INEXACT8(SCSI_AICREG(SCSI_S160M_INTR_THRSH_TIMEOUT1)) & 0x7F);
}
#endif 

/*********************************************************************
*
*  SCSIhSwappingGetIntrThreshold
*
*     this mode does not support interrupt batching
*
*  Return Value:  None
*
*  Parameters:    hhcb
*
*  Remarks:
*                  
*********************************************************************/
#if SCSI_SWAPPING64_MODE + SCSI_SWAPPING32_MODE
SCSI_UEXACT8 SCSIhSwappingGetIntrThreshold (SCSI_HHCB SCSI_HPTR hhcb)
{
   return ((SCSI_UEXACT8)0xFF);
}
#endif 

/*********************************************************************
*
*  SCSIhSwappingAdvGetIntrThreshold
*
*     Gets the interrupt threshold from the scratch register
*
*  Return Value:  None
*
*  Parameters:    hhcb
*
*  Remarks:
*                  
*********************************************************************/
#if SCSI_SWAPPING_ADVANCED_MODE
SCSI_UEXACT8 SCSIhSwappingAdvGetIntrThreshold (SCSI_HHCB SCSI_HPTR hhcb)
{
   SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
#if SCSI_TARGET_OPERATION
/* not supported in sequencer yet */
   return ((SCSI_UEXACT8)0xFF);
#else 
   return (OSD_INEXACT8(SCSI_AICREG(SCSI_WADV_INTR_THRSH_TIMEOUT1)) & 0x7F);
#endif /* SCSI_TARGET_OPERATION */
}
#endif 

/*********************************************************************
*
*  SCSIhSwapping160mGetIntrThreshold
*
*     Gets the interrupt threshold from the scratch register
*
*  Return Value:  None
*
*  Parameters:    hhcb
*
*  Remarks:
*                  
*********************************************************************/
#if SCSI_SWAPPING_160M_MODE
SCSI_UEXACT8 SCSIhSwapping160mGetIntrThreshold (SCSI_HHCB SCSI_HPTR hhcb)
{
   SCSI_REGISTER scsiRegister = hhcb->scsiRegister;

   return (OSD_INEXACT8(SCSI_AICREG(SCSI_W160M_INTR_THRSH_TIMEOUT1)) & 0x7F);
}
#endif 

/*********************************************************************
*
*  SCSIhEnableAutoRateOption
*
*     Enable automatic SCSI rate/offset feature
*
*  Return Value:  None
*
*  Parameters:    hhcb
*
*  Remarks:
*                  
*********************************************************************/
#if (SCSI_STANDARD_160M_MODE || SCSI_SWAPPING_160M_MODE)
void SCSIhEnableAutoRateOption (SCSI_HHCB SCSI_HPTR hhcb)
{
   SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_UEXACT8 scsiSfunct;

   /* Switch the chip to alternate mode for accessing optionmode reg. */
   scsiSfunct = OSD_INEXACT8(SCSI_AICREG(SCSI_SFUNCT));
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SFUNCT), (scsiSfunct | SCSI_ALT_MODE));

   /* Enable Auto Rate and Offset feature */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_OPTIONMODE),
       (OSD_INEXACT8(SCSI_AICREG(SCSI_OPTIONMODE)) | SCSI_AUTORATEEN));
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SFUNCT), scsiSfunct);

   /* Indicate auto rate/offset feature enable to sequencer */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_hAUTO_RATE_ENABLE(hhcb)), 0x01);
}
#endif /* (SCSI_STANDARD_160M_MODE || SCSI_SWAPPING_160M_MODE) */

/*********************************************************************
*
*  SCSIhDisableAutoRateOption
*
*     Disable automatic SCSI rate/offset feature
*
*  Return Value:  None
*
*  Parameters:    hhcb
*                 xferOption
*
*  Remarks:       The traditional or non-auto scsi rate/offset
*                 registers will be updated with passed-in xfer
*                 option value at the end of the function.
*
*********************************************************************/
#if (SCSI_STANDARD_160M_MODE || SCSI_SWAPPING_160M_MODE)
void SCSIhDisableAutoRateOption (SCSI_HHCB SCSI_HPTR hhcb,
                           SCSI_UEXACT16 xferOption)
{
   SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_UEXACT8 scsiSfunct;

   /* Switch the chip to alternate mode for accessing optionmode reg. */
   scsiSfunct = OSD_INEXACT8(SCSI_AICREG(SCSI_SFUNCT));
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SFUNCT), (scsiSfunct | SCSI_ALT_MODE));

   /* Disable Auto Rate and Offset feature */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_OPTIONMODE),
       (OSD_INEXACT8(SCSI_AICREG(SCSI_OPTIONMODE)) & ~SCSI_AUTORATEEN));
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SFUNCT), scsiSfunct);

   /* Indicate auto rate/offset feature disable to sequencer */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_hAUTO_RATE_ENABLE(hhcb)), 0x00);

   OSD_OUTEXACT8(SCSI_AICREG(SCSI_TRIDENT_SCSIRATE),
       (SCSI_UEXACT8)(xferOption & 0x00ff));
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_TRIDENT_SCSIOFFSET),
       (SCSI_UEXACT8)((xferOption >> 8) & 0x00ff));
}
#endif /* (SCSI_STANDARD_160M_MODE || SCSI_SWAPPING_160M_MODE) */

/*********************************************************************
*
*  SCSIHCheckSigSCB2
*
*     Check to see whether signature of SCB is valid or not
*
*  Return Value:  1 if valid
*                 0 if not valid
*                  
*  Parameters:    hhcb
*                 nvmDataPtr
*
*  Remarks:
*
*********************************************************************/
SCSI_UINT8 SCSIHCheckSigSCB2 (SCSI_HHCB SCSI_HPTR hhcb,
                              SCSI_UEXACT8 SCSI_SPTR dataBuffer)
{
   SCSI_INT i;
   SCSI_UEXACT8 scbptr;
   SCSI_UINT8 return_value;
   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   
   /* SCSI_NVM_LAYOUT nvmData;  */

   /* set to page 2, and check signature on SCB2 */
   SCSIhPauseAndWait(hhcb);

   /* save SCBPTR value */
   scbptr = OSD_INEXACT8(SCSI_AICREG(SCSI_SCBPTR));

   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCBPTR), 0x02);
   if (   (OSD_INEXACT8(SCSI_AICREG(SCSI_SCB_BASE+0)) == 'A')
       && (OSD_INEXACT8(SCSI_AICREG(SCSI_SCB_BASE+1)) == 'D')
       && (OSD_INEXACT8(SCSI_AICREG(SCSI_SCB_BASE+2)) == 'P')
       && (OSD_INEXACT8(SCSI_AICREG(SCSI_SCB_BASE+3)) == 'T') )
   {
      /* copy 64 bytes from scratch ram (20h-5Fh) to nvmdata */
      for (i=0; i <= 63; i++)
      {
         *((SCSI_UEXACT8 SCSI_SPTR) (dataBuffer + i )) = OSD_INEXACT8(SCSI_AICREG(SCSI_SCRATCH0 + i));
      }
      /* NVRAM offset 0x1C is placed in SCB2 offset 04 */
      *((SCSI_UEXACT8 SCSI_SPTR) (dataBuffer + 0x1C )) = OSD_INEXACT8(SCSI_AICREG(SCSI_SCB_BASE+4));
      return_value = 1;
   }
   else
      return_value = 0; 

   /* restore SCBPTR */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCBPTR),scbptr);
   SCSIhUnPause(hhcb);
   return(return_value);
}
#if CURRENT_SENSING
/*********************************************************************
*
*  SCSIhGetCurrentSensing routine -
*
*     Read Termination Current Sensing status
*
*  Return Value:  0 - Termination Okay
*                 1 - Over Terminated
*                 2 - Under Terminated
*                 3 - Invalid Channel
*                  
*  Parameters:    hhcb
*
*  Remarks:       none
*                  
*********************************************************************/
SCSI_UEXACT8 SCSIhGetCurrentSensing (SCSI_HHCB SCSI_HPTR hhcb,
                                     SCSI_UEXACT8 channel_selection)
{
   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_UEXACT8 seeprom_value;
   SCSI_UEXACT8 current_sensing = 0;
   SCSI_UEXACT8 i;

   SCSIhPauseAndWait(hhcb);      
      
   seeprom_value = OSD_INEXACT8(SCSI_AICREG(SCSI_SEEPROM));
                                                        
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SEEPROM),SCSI_SEEMS|SCSI_SEECS );     
     
   while(!(OSD_INEXACT8(SCSI_AICREG(SCSI_SEEPROM)) & SCSI_SEERDY))
      ;

   /* write mode */

   if (hhcb->SCSI_HF_terminationHigh) 
   {   
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_BRDCTL),SCSI_BRDSTB|SCSI_BRDCS|SCSI_BRDDAT7|SCSI_BRDDAT6); 
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_BRDCTL),SCSI_BRDCS|SCSI_BRDDAT7|SCSI_BRDDAT6);
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_BRDCTL),SCSI_BRDDAT7|SCSI_BRDDAT6);
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_BRDCTL),SCSI_BRDDAT7|SCSI_BRDDAT6); 
   }
   else
   {
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_BRDCTL),SCSI_BRDSTB|SCSI_BRDCS|SCSI_BRDDAT7); 
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_BRDCTL),SCSI_BRDCS|SCSI_BRDDAT7);
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_BRDCTL),SCSI_BRDDAT7);            
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_BRDCTL),SCSI_BRDDAT7);  
   }

   /* write channel selection */
   
   for (i=0; i<2; i++)
   {
      channel_selection >>= i;     

      if (hhcb->hardwareMode == SCSI_HMODE_AICTRIDENT)
      {
         /* For TridentII hardware, both SEEMS and SEECS need to be asserted */
         /* in order to keep the flex port.                                  */
         OSD_OUTEXACT8(SCSI_AICREG(SCSI_SEEPROM),SCSI_SEEMS|SCSI_SEECS);
      }
      else
      {   
         OSD_OUTEXACT8(SCSI_AICREG(SCSI_SEEPROM),SCSI_SEEMS);
      }
      
      if (channel_selection & 0x01)          
        
         OSD_OUTEXACT8(SCSI_AICREG(SCSI_SEEPROM),SCSI_SEEMS|SCSI_SEECS|SCSI_SEEDO);       
      
      else
     
         OSD_OUTEXACT8(SCSI_AICREG(SCSI_SEEPROM),SCSI_SEEMS|SCSI_SEECS);       
   }   

   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SEEPROM),SCSI_SEEMS|SCSI_SEECS);       

         
   /* read mode */
   

   if (hhcb->SCSI_HF_terminationHigh) 

      OSD_OUTEXACT8(SCSI_AICREG(SCSI_BRDCTL),SCSI_BRDRW|SCSI_BRDCS|SCSI_BRDDAT6);  /* read mode */
   
   else
   
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_BRDCTL),SCSI_BRDRW|SCSI_BRDCS);  
   
   /* invalid channel */
   
   if (OSD_INEXACT8(SCSI_AICREG(SCSI_SEEPROM)) & SCSI_SEEDI)
      
      current_sensing = 3;
   else
   {  
      if (hhcb->hardwareMode != SCSI_HMODE_AICTRIDENT)
      {
         /* don't release SEECS to prevent Trident chip de-assert EXTARBACK */
         OSD_OUTEXACT8(SCSI_AICREG(SCSI_SEEPROM),SCSI_SEEMS);
      }

      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SEEPROM),SCSI_SEEMS|SCSI_SEECS);    
      
           
      if (OSD_INEXACT8(SCSI_AICREG(SCSI_SEEPROM)) & SCSI_SEEDI)
         /* over terminated */
         current_sensing = 1;
     
      else
      {
         if (hhcb->hardwareMode != SCSI_HMODE_AICTRIDENT)
         {
         /* don't release SEECS to prevent Trident chip de-assert EXTARBACK */
            OSD_OUTEXACT8(SCSI_AICREG(SCSI_SEEPROM),SCSI_SEEMS);
         }

         OSD_OUTEXACT8(SCSI_AICREG(SCSI_SEEPROM),SCSI_SEEMS|SCSI_SEECS);    
      
         if (OSD_INEXACT8(SCSI_AICREG(SCSI_SEEPROM)) & SCSI_SEEDI)
            /* under terminated */
            current_sensing = 2;
         else
            /* termination okay */
            current_sensing = 0;
      }
   }

   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SEEPROM),SCSI_SEEMS|SCSI_SEECS);    

   if (hhcb->SCSI_HF_terminationHigh) 
   {
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_BRDCTL),SCSI_BRDSTB|SCSI_BRDCS|SCSI_BRDDAT6); /* strobe MD7 to 0 */
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_BRDCTL),SCSI_BRDCS|SCSI_BRDDAT6); 
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_BRDCTL),SCSI_BRDDAT6); 
   }
   else
   {
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_BRDCTL),SCSI_BRDSTB|SCSI_BRDCS); /* strobe MD7 to 0 */
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_BRDCTL),SCSI_BRDCS); 
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_BRDCTL),0); 
   }

   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SEEPROM),seeprom_value);    

   SCSIhUnPause(hhcb); 

   return(current_sensing);
}


/*********************************************************************
*
*  SCSIhGetCurrentSensing160 routine -
*
*     Read Termination Current Sensing status
*
*  Return Value:  0 - Termination Okay
*                 1 - Over Terminated
*                 2 - Under Terminated
*                 3 - Invalid Channel
*                  
*  Parameters:    hhcb
*
*  Remarks:       none
*                  
*********************************************************************/
SCSI_UEXACT8 SCSIhGetCurrentSensing160 (SCSI_HHCB SCSI_HPTR hhcb,
                                     SCSI_UEXACT8 channel_selection)
{
   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_UEXACT8 seeprom_value;
   SCSI_UEXACT8 current_sensing = 0;
   SCSI_UEXACT8 i;
   SCSI_UEXACT8 hcntrl;
   SCSI_UEXACT8 terminationsetting;
   SCSI_UEXACT8 value;
   
   /* check if pause chip necessary */
   if (!((hcntrl = OSD_INEXACT8(SCSI_AICREG(SCSI_HCNTRL))) & SCSI_PAUSEACK))
   {
      SCSIhPauseAndWait(hhcb);      
   }
      
   /* save SEECTL contents */
   
   seeprom_value = OSD_INEXACT8(SCSI_AICREG(SCSI_SEEPROM));
                                                        
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SEEPROM),SCSI_SEEMS|SCSI_SEECS );     
   
   /* save termination settings */
   
   if (hhcb->SCSI_HF_terminationHigh) 
      terminationsetting = SCSI_BAYONET_BRDDAT4;
   if (hhcb->SCSI_HF_secondarytermhigh)
      terminationsetting |= SCSI_BAYONET_BRDDAT6;
   if (hhcb->SCSI_HF_secondarytermlow)
      terminationsetting |= SCSI_BAYONET_BRDDAT5;	  
   
	    
   while(!(OSD_INEXACT8(SCSI_AICREG(SCSI_SEEPROM)) & SCSI_SEERDY))
      ;

   
   /* activate current sensing */
   
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_BRDCTL),SCSI_BAYONET_BRDDAT7 | terminationsetting); 

   OSD_OUTEXACT8(SCSI_AICREG(SCSI_BRDCTL),SCSI_BAYONET_BRDDAT7 | SCSI_BAYONET_BRDSTB | terminationsetting); 
   
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_BRDCTL),SCSI_BAYONET_BRDDAT7 | terminationsetting); 
   
   /* channel selection */
   
   switch(channel_selection)
   {

      case 0:

         OSD_OUTEXACT8(SCSI_AICREG(SCSI_BRDCTL),SCSI_BAYONET_BRDDAT7); 
         OSD_OUTEXACT8(SCSI_AICREG(SCSI_BRDCTL),SCSI_BAYONET_BRDDAT7 | SCSI_BAYONET_BRDSTB); 
         OSD_OUTEXACT8(SCSI_AICREG(SCSI_BRDCTL),SCSI_BAYONET_BRDDAT7); 
         break;


      case 1:

         OSD_OUTEXACT8(SCSI_AICREG(SCSI_BRDCTL),SCSI_BAYONET_BRDDAT7 | SCSI_BAYONET_BRDDAT5); 
         OSD_OUTEXACT8(SCSI_AICREG(SCSI_BRDCTL),SCSI_BAYONET_BRDDAT7 | SCSI_BAYONET_BRDDAT5 | SCSI_BAYONET_BRDSTB); 
         OSD_OUTEXACT8(SCSI_AICREG(SCSI_BRDCTL),SCSI_BAYONET_BRDDAT7 | SCSI_BAYONET_BRDDAT5); 
         break;



      case 2:

         OSD_OUTEXACT8(SCSI_AICREG(SCSI_BRDCTL),SCSI_BAYONET_BRDDAT7 | SCSI_BAYONET_BRDDAT6); 
         OSD_OUTEXACT8(SCSI_AICREG(SCSI_BRDCTL),SCSI_BAYONET_BRDDAT7 | SCSI_BAYONET_BRDDAT6 | SCSI_BAYONET_BRDSTB); 
         OSD_OUTEXACT8(SCSI_AICREG(SCSI_BRDCTL),SCSI_BAYONET_BRDDAT7 | SCSI_BAYONET_BRDDAT6); 
         break;


      case 3:

         OSD_OUTEXACT8(SCSI_AICREG(SCSI_BRDCTL),SCSI_BAYONET_BRDDAT7 | SCSI_BAYONET_BRDDAT6 | SCSI_BAYONET_BRDDAT5); 

         OSD_OUTEXACT8(SCSI_AICREG(SCSI_BRDCTL),SCSI_BAYONET_BRDDAT7 | SCSI_BAYONET_BRDDAT6 | SCSI_BAYONET_BRDDAT5 | SCSI_BAYONET_BRDSTB); 
         OSD_OUTEXACT8(SCSI_AICREG(SCSI_BRDCTL),SCSI_BAYONET_BRDDAT7 | SCSI_BAYONET_BRDDAT6 | SCSI_BAYONET_BRDDAT5); 
         break;

      default:
	     break;
   }		 

   /* read mode */
   
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_BRDCTL),SCSI_BAYONET_BRDRW);  
   
   value = INBYTE(AIC7870[BRDCTL]);

   if ((value & CURRENT_SENSING_UNDER_BIT) && (value & CURRENT_SENSING_OVER_BIT))
   
      /* invalid channel */
      current_sensing = 3;
   else
      /* under terminated */
      if (value & CURRENT_SENSING_UNDER_BIT)
	      current_sensing = 2;
      else
	      /* over terminated */
		  if (value & CURRENT_SENSING_OVER_BIT)
    	     /* over terminated */
	         current_sensing = 1;
          else
		     /* termination okay */
		     current_sensing = 0;
   		  		                  

      
   /* deactivate current sensing */
   
  
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_BRDCTL),0); 

   OSD_OUTEXACT8(SCSI_AICREG(SCSI_BRDCTL),SCSI_BAYONET_BRDSTB); 

   OSD_OUTEXACT8(SCSI_AICREG(SCSI_BRDCTL),0); 
 
   /* restore seectl contents */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SEEPROM),seeprom_value);    

   /* unpause if originally unpaused */
   
   if (!(hcntrl & SCSI_PAUSEACK))
   {
      SCSI_hWRITE_HCNTRL(hhcb,scsiRegister,hcntrl);
   }


   return(current_sensing);
}
#endif
#if SCSI_NEGOTIATION_PER_IOB
/*********************************************************************
*
*  SCSIhAIC78XXChangeXferOption
*
*     Incorporates the changes in xfer option parameters into the hardware
*     layer parameters.  Note that this routine assumes that Device for which
*     the xfer option needs to be changed is idle.
*
*  Return Value:  None
*
*  Parameters:    hiob
*
*  Remarks:       None.
*
*********************************************************************/
#if SCSI_AIC78XX
void SCSIhAIC78XXChangeXferOption (SCSI_HIOB SCSI_IPTR hiob)
{
   SCSI_UNIT_CONTROL SCSI_IPTR targetUnit = SCSI_TARGET_UNIT(hiob);
   SCSI_HHCB SCSI_HPTR hhcb = targetUnit->hhcb;
   SCSI_TARGET_CONTROL SCSI_HPTR targetControl = targetUnit->targetControl;
   SCSI_DEVICE SCSI_DPTR deviceTable;
   SCSI_UEXACT16 speed;
   SCSI_UEXACT8 offset;
   SCSI_UEXACT8 width;
   SCSI_UEXACT16 currentXferOption;
   
   /* For the given target, incorporate the incoming xfer option paramters. */
   /* The xfer option parameters come through HIOB. */

   deviceTable = &SCSI_DEVICE_TABLE(hhcb)[targetUnit->scsiID];

   /* Get the current xfer option parameters */
   currentXferOption = SCSI_hGETXFEROPTION(hhcb,targetUnit->scsiID);
   offset = (SCSI_UEXACT8)(SCSI_SOFS & currentXferOption);
   if (currentXferOption & SCSI_WIDE_XFER)
   {
      width = 16;
   }
   else
   {
      width = 8;
   }
   speed = SCSIhAIC78XXGetSpeed(deviceTable->scsiOption, 
            (SCSI_UEXACT8)deviceTable->SCSI_DF_ultraEnable);
   
   /* Change the width according the force bits */
   if ((hiob->SCSI_IF_forceWide) || (hiob->SCSI_IF_forceNarrow))
   {
      if (hiob->SCSI_IF_forceWide)
      {
         width = 16;
      }
      else
      {
         width = 8;
      }
   }

   /* Adjust the 'setForWide' parameter according to the resultant width */
   if (width == 16)
   {
      deviceTable->SCSI_DF_setForWide = 1;  /* Set setForWide flag */
   }
   else
   {
      deviceTable->SCSI_DF_setForWide = 0;  /* Clear setForWide flag */

      /* If the device's running in wide mode, we need to clear       */
      /* the setForWide flag and set the width to 16 so that the wide */
      /* xfer bit will be set when the new scsiOption get calucated.  */
      /* This way when the next IOB for this target start execute,    */
      /* CHIM will intiate wide negotiation. By then, it will clear   */
      /* the wide xfer bit because of the setForWide was clear.       */
      if (currentXferOption & SCSI_WIDEXFER)
      {
         width = 16;
      }
   }

   /* Do nothing if the speed and offset are the same */
   if ((hiob->SCSI_IF_forceSync) || (hiob->SCSI_IF_forceAsync))
   {
      if ((hiob->SCSI_IF_forceSync))
      {
         offset = deviceTable->origOffset; 
      }
      else
      {
         offset = 0;
      }
   }

   /* If user wants a non-zero offset and speed above async range, */
   /* this means do sync neg. with hardware default offset */
   if (offset)
   {
      deviceTable->SCSI_DF_setForSync = 1;   /* Set setForSync flag */
   }
   else
   {
      deviceTable->SCSI_DF_setForSync = 0;   /* Clear setForSync flag */
      /* Since the default offset is zero and/or speed is less than   */
      /* 36, which means do async. xfer, we need to clear the         */
      /* setForSync flag and set the offset to the current offset.    */
      /* This way when the next IOB for this target start execute,    */
      /* CHIM will intiate sync. negotiation if the current offset is */
      /* nonzero. By then, it will set the offset to zero because of  */
      /* the setForSync was clear.                                    */
      offset = (SCSI_UEXACT8)(SCSI_SOFS & currentXferOption);
   }

   /* Incorporate the xfer option changes into the deviceTable and kick start */
   /* the negotiation. */
   deviceTable->scsiOption = (SCSI_UEXACT8)SCSIhAIC78XXCalcScsiOption(speed,offset,width, 0);

   deviceTable->SCSI_DF_ultraEnable = (speed >= 133) ? 1 : 0;

   /* Save 'forceNego' flags into the deviceTable */
   deviceTable->SCSI_DNF_forceSync = hiob->SCSI_IF_forceSync;
   deviceTable->SCSI_DNF_forceAsync = hiob->SCSI_IF_forceAsync;
   deviceTable->SCSI_DNF_forceWide = hiob->SCSI_IF_forceWide;
   deviceTable->SCSI_DNF_forceNarrow = hiob->SCSI_IF_forceNarrow;

   /* Clear sync/wide xfer flags as it will be reassigned below */
   deviceTable->scsiOption &= ~SCSI_SYNC_XFER;
   deviceTable->scsiOption &= ~SCSI_WIDE_XFER;

   /* Reassign scsiOption */
   SCSIhChangeNegotiation(targetUnit);
}
#endif /* SCSI_AIC78XX */

/*********************************************************************
*
*  SCSIhAICBayonetChangeXferOption
*
*     Incorporates the changes in xfer option parameters into the hardware
*     layer parameters.  Note that this routine assumes that Device for which
*     the xfer option needs to be changed is idle.
*
*  Return Value:  None
*
*  Parameters:    hiob
*
*  Remarks:       None.
*
*********************************************************************/
#if SCSI_AICBAYONET
void SCSIhAICBayonetChangeXferOption (SCSI_HIOB SCSI_IPTR hiob)
{
   SCSI_UNIT_CONTROL SCSI_IPTR targetUnit = SCSI_TARGET_UNIT(hiob);
   SCSI_HHCB SCSI_HPTR hhcb = targetUnit->hhcb;
   SCSI_TARGET_CONTROL SCSI_HPTR targetControl = targetUnit->targetControl;
   SCSI_DEVICE SCSI_DPTR deviceTable;
   SCSI_UEXACT16 speed;
   SCSI_UEXACT8 offset;
   SCSI_UEXACT8 width;
   SCSI_UEXACT16 currentXferOption;
   SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   
   /* For the given target, incorporate the incoming xfer option paramters. */
   /* The xfer option parameters come through HIOB. */

   deviceTable = &SCSI_DEVICE_TABLE(hhcb)[targetUnit->scsiID];

   /* Get the current xfer option parameters */
   currentXferOption = SCSI_hGETXFEROPTION(hhcb,targetUnit->scsiID);
   offset = (SCSI_UEXACT8)((currentXferOption >> 8) & SCSI_BAYONET_OFFSET);
   if (currentXferOption & SCSI_WIDE_XFER)
   {
      width = 16;
   }
   else
   {
      width = 8;
   }
   speed = SCSIhAICBayonetGetSpeed(deviceTable->bayScsiRate, 0);

   /* Change the width according the force bits */
   if ((hiob->SCSI_IF_forceWide) || (hiob->SCSI_IF_forceNarrow))
   {
      if (hiob->SCSI_IF_forceWide)
      {
         width = 16;
      }
      else
      {
         width = 8;
      }
   }

   /* Adjust the 'setForWide' parameter according to the resultant width */
   if (width == 16)
   {
      deviceTable->SCSI_DF_setForWide = 1;  /* Set setForWide flag */
   }
   else
   {
      deviceTable->SCSI_DF_setForWide = 0;  /* Clear setForWide flag */

      /* If the device's running in wide mode, we need to clear       */
      /* the setForWide flag and set the width to 16 so that the wide */
      /* xfer bit will be set when the new scsiOption get calucated.  */
      /* This way when the next IOB for this target start execute,    */
      /* CHIM will intiate wide negotiation. By then, it will clear   */
      /* the wide xfer bit because of the setForWide was clear.       */
      if (currentXferOption & SCSI_WIDEXFER)
      {
         width = 16;
      }
   }

   /* Do nothing if the speed and offset are the same */
   if ((hiob->SCSI_IF_forceSync) || (hiob->SCSI_IF_forceAsync))
   {
      if ((hiob->SCSI_IF_forceSync))
      {
         offset = deviceTable->origOffset; 
      }
      else
      {
         offset = 0;
      }
   }

   /* If user wants a non-zero offset and speed above async range, */
   /* this means do sync neg. with hardware default offset */
   if (offset)
   {
      deviceTable->SCSI_DF_setForSync = 1;   /* Set setForSync flag */
   }
   else
   {
      deviceTable->SCSI_DF_setForSync = 0;   /* Clear setForSync flag */
      /* Since the default offset is zero and/or speed is less than   */
      /* 36, which means do async. xfer, we need to clear the         */
      /* setForSync flag and set the offset to the current offset.    */
      /* This way when the next IOB for this target start execute,    */
      /* CHIM will intiate sync. negotiation if the current offset is */
      /* nonzero. By then, it will set the offset to zero because of  */
      /* the setForSync was clear.                                    */
      offset = (SCSI_UEXACT8)((currentXferOption >> 8) & SCSI_BAYONET_OFFSET);
   }

   /* Incorporate the xfer option changes into the deviceTable and kick start */
   /* the negotiation. */
   currentXferOption = SCSIhAICBayonetCalcScsiOption(speed,offset,width, 0);
   deviceTable->bayScsiRate = (SCSI_UEXACT8)(currentXferOption);
   deviceTable->bayScsiOffset = (SCSI_UEXACT8)(currentXferOption >> 8);

   /* Save 'forceNego' flags into the deviceTable */
   deviceTable->SCSI_DNF_forceSync = hiob->SCSI_IF_forceSync;
   deviceTable->SCSI_DNF_forceAsync = hiob->SCSI_IF_forceAsync;
   deviceTable->SCSI_DNF_forceWide = hiob->SCSI_IF_forceWide;
   deviceTable->SCSI_DNF_forceNarrow = hiob->SCSI_IF_forceNarrow;

   /* Clear scsiOption before assigning new value to it */
   deviceTable->scsiOption = 0;

   /* Reassign scsiOption */
   SCSIhChangeNegotiation(targetUnit);
}
#endif /* SCSI_AICBAYONET */

/*********************************************************************
*
*  SCSIhAICTridentChangeXferOption
*
*     Incorporates the changes in xfer option parameters into the hardware
*     layer parameters.  Note that this routine assumes that Device for which
*     the xfer option needs to be changed is idle.
*
*  Return Value:  None
*
*  Parameters:    hiob
*
*  Remarks:       None.
*
*********************************************************************/
#if SCSI_AICTRIDENT
void SCSIhAICTridentChangeXferOption (SCSI_HIOB SCSI_IPTR hiob)
{
   SCSI_UNIT_CONTROL SCSI_IPTR targetUnit = SCSI_TARGET_UNIT(hiob);
   SCSI_HHCB SCSI_HPTR hhcb = targetUnit->hhcb;
   SCSI_TARGET_CONTROL SCSI_HPTR targetControl = targetUnit->targetControl;
   SCSI_DEVICE SCSI_DPTR deviceTable;
   SCSI_UEXACT16 speed;
   SCSI_UEXACT8 offset;
   SCSI_UEXACT8 width;
   SCSI_UEXACT16 currentXferOption;
   SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   
   /* For the given target, incorporate the incoming xfer option paramters. */
   /* The xfer option parameters come through HIOB. */

   deviceTable = &SCSI_DEVICE_TABLE(hhcb)[targetUnit->scsiID];

   /* Get the current xfer option parameters */
   currentXferOption = SCSI_hGETXFEROPTION(hhcb,targetUnit->scsiID);
   offset = (SCSI_UEXACT8)((currentXferOption >> 8) & SCSI_BAYONET_OFFSET);
   if (currentXferOption & SCSI_WIDE_XFER)
   {
      width = 16;
   }
   else
   {
      width = 8;
   }
   speed = SCSIhAICTridentGetSpeed(deviceTable->bayScsiRate, 0);

   /* Change the width according the force bits */
   if ((hiob->SCSI_IF_forceWide) || (hiob->SCSI_IF_forceNarrow))
   {
      if (hiob->SCSI_IF_forceWide)
      {
         width = 16;
      }
      else
      {
         width = 8;
      }
   }

   /* Adjust the 'setForWide' parameter according to the resultant width */
   if (width == 16)
   {
      deviceTable->SCSI_DF_setForWide = 1;  /* Set setForWide flag */
   }
   else
   {
      deviceTable->SCSI_DF_setForWide = 0;  /* Clear setForWide flag */

      /* If the device's running in wide mode, we need to clear       */
      /* the setForWide flag and set the width to 16 so that the wide */
      /* xfer bit will be set when the new scsiOption get calucated.  */
      /* This way when the next IOB for this target start execute,    */
      /* CHIM will intiate wide negotiation. By then, it will clear   */
      /* the wide xfer bit because of the setForWide was clear.       */
      if (currentXferOption & SCSI_WIDEXFER)
      {
         width = 16;
      }
   }

   /* Do nothing if the speed and offset are the same */
   if ((hiob->SCSI_IF_forceSync) || (hiob->SCSI_IF_forceAsync))
   {
      if ((hiob->SCSI_IF_forceSync))
      {
         offset = deviceTable->origOffset; 
      }
      else
      {
         offset = 0;
      }
   }

   /* If user wants a non-zero offset and speed above async range, */
   /* this means do sync neg. with hardware default offset */
   if (offset)
   {
      deviceTable->SCSI_DF_setForSync = 1;   /* Set setForSync flag */
   }
   else
   {
      deviceTable->SCSI_DF_setForSync = 0;   /* Clear setForSync flag */
      /* Since the default offset is zero and/or speed is less than   */
      /* 36, which means do async. xfer, we need to clear the         */
      /* setForSync flag and set the offset to the current offset.    */
      /* This way when the next IOB for this target start execute,    */
      /* CHIM will intiate sync. negotiation if the current offset is */
      /* nonzero. By then, it will set the offset to zero because of  */
      /* the setForSync was clear.                                    */
      offset = (SCSI_UEXACT8)((currentXferOption >> 8) & SCSI_BAYONET_OFFSET);
   }

   /* Incorporate the xfer option changes into the deviceTable and kick start */
   /* the negotiation. */
   if(speed >= 800)
   {
      currentXferOption = SCSIhAICTridentCalcScsiOption(speed,offset,width, 1);
   }
   else
   {
      currentXferOption = SCSIhAICTridentCalcScsiOption(speed,offset,width, 0);
   }
   deviceTable->bayScsiRate = (SCSI_UEXACT8)(currentXferOption);
   deviceTable->bayScsiOffset = (SCSI_UEXACT8)(currentXferOption >> 8);

   /* Save 'forceNego' flags into the deviceTable */
   deviceTable->SCSI_DNF_forceSync = hiob->SCSI_IF_forceSync;
   deviceTable->SCSI_DNF_forceAsync = hiob->SCSI_IF_forceAsync;
   deviceTable->SCSI_DNF_forceWide = hiob->SCSI_IF_forceWide;
   deviceTable->SCSI_DNF_forceNarrow = hiob->SCSI_IF_forceNarrow;

   /* Clear scsiOption before assigning new value to it */
   deviceTable->scsiOption = 0;

   /* Reassign scsiOption */
   SCSIhChangeNegotiation(targetUnit);
}
#endif /* SCSI_AICTRIDENT */

/*******************************************************************************
*
*  SCSIhChangeNegotiation
*
*     Changes scsiOption field to reflect the negotiation changes from the
*     deviceTable 'negoFlags' fields.
*
*  Return Value:  None
*
*  Parameters:    targetUnit - pointer to target unit structure
*
*  Remarks:       None.
*
*******************************************************************************/
void SCSIhChangeNegotiation (SCSI_UNIT_CONTROL SCSI_UPTR targetUnit)
{
   SCSI_HHCB SCSI_HPTR hhcb = targetUnit->hhcb;
   SCSI_DEVICE SCSI_DPTR deviceTable;
   SCSI_REGISTER scsiRegister = targetUnit->hhcb->scsiRegister;
#if (SCSI_AICBAYONET || SCSI_AICTRIDENT)
   SCSI_UEXACT8 hcntrl;
#endif

   deviceTable = &SCSI_DEVICE_TABLE(hhcb)[targetUnit->scsiID];

#if (SCSI_AICBAYONET || SCSI_AICTRIDENT)
   /* For Bayonet&Tirdent, the chip should be paused before accessing  */
   /* scratch RAM or SCB array                                         */
   if (!((hcntrl = OSD_INEXACT8(SCSI_AICREG(SCSI_HCNTRL))) & SCSI_PAUSEACK))
   {
      SCSIhPauseAndWait(hhcb);
   }
#endif

   /* Turn on sync negotiation */
   if ((deviceTable->SCSI_DNF_forceSync) || (deviceTable->SCSI_DNF_forceAsync))
   {
      deviceTable->scsiOption |= SCSI_SYNC_XFER;
   }

   /* Turn on wide/narrow negotiation if the adapter supports wide bus */
   if (OSD_INEXACT8(SCSI_AICREG(SCSI_SBLKCTL)) & SCSI_SELWIDE)
   {
      if ((deviceTable->SCSI_DNF_forceWide) ||
          (deviceTable->SCSI_DNF_forceNarrow))
      {
         deviceTable->scsiOption |= (SCSI_WIDE_XFER | SCSI_SYNC_XFER);
      }
   }

   /* Turn on narrow negotiation only if the adapter supports only narrow bus */
   else
   {
      if (deviceTable->SCSI_DNF_forceNarrow)
      {
         deviceTable->scsiOption |= (SCSI_WIDE_XFER | SCSI_SYNC_XFER);
      }
   }

   /* turn on the negotiation initiation */
   SCSI_hXFEROPTASSIGN(hhcb,targetUnit->scsiID,SCSI_NEEDNEGO);

#if (SCSI_AICBAYONET || SCSI_AICTRIDENT)
   if (!(hcntrl & SCSI_PAUSEACK))
   {
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_HCNTRL), hcntrl);
   }
#endif
}
#endif /* SCSI_NEGOTIATION_PER_IOB */

#if SCSI_PAC_NSX_REPEATER
/*********************************************************************
*
*  SCSIHSingleEndBus
*
*  Return Value:  0 - SCSI Bus is not Single Ended bus
*                 1 - SCSI Bus is Single Ended bus
*
*  Parameters:    hhcb
*
*  Remarks:       !!ATTENTION!! Need to put in Hardware Desciptior.
*
*********************************************************************/
SCSI_UEXACT8 SCSIHSingleEndBus (SCSI_HHCB SCSI_HPTR hhcb)
{
   SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   
   /* For AIC78XX HW mode, it is always SE bus.  For Bayonet and Trident HWs */
   /* Scsi bus is currently in SE mode if ENAB20 bit of SBLKCTL register set.*/
   if ((hhcb->hardwareMode == SCSI_HMODE_AIC78XX) ||
       (OSD_INEXACT8(SCSI_AICREG(SCSI_SBLKCTL)) & SCSI_BAYONET_ENAB20)) 
   {
      return(1);
   }

   return(0);
}
#endif /* SCSI_PAC_NSX_REPEATER */

/***************************************************************************
*                                                                          *
* Copyright 1995,1996,1997,1998,1999 Adaptec, Inc.,  All Rights Reserved.  *
*                                                                          *
* This software contains the valuable trade secrets of Adaptec.  The       *
* software is protected under copyright laws as an unpublished work of     *
* Adaptec.  Notice is for informational purposes only and does not imply   *
* publication.  The user of this software may make copies of the software  *
* for use with parts manufactured by Adaptec or under license from Adaptec *
* and for no other use.                                                    *
*                                                                          *
***************************************************************************/

/***************************************************************************
*
*  Module Name:   HWMPTCL.C
*
*  Description:
*                 Codes to implement protocol specific for  hardware 
*                 management module
*
*  Owners:        ECX IC Firmware Team
*    
*  Notes:         NONE
*
***************************************************************************/

#include "scsi.h"
#include "hwm.h"

/*%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%*/
/* Modules internal to hardware management layer                          */
/*%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%*/

/*********************************************************************
*
*  SCSIhNegotiate
*
*     Handle negotiation needed interrupt
*
*  Return Value:  none
*                  
*  Parameters:    hhcb
*                 hiob
*
*  Remarks:
*
*********************************************************************/
void SCSIhNegotiate (SCSI_HHCB SCSI_HPTR hhcb,
                     SCSI_HIOB SCSI_IPTR hiob)
{
   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_UNIT_CONTROL SCSI_UPTR unitHandle = SCSI_TARGET_UNIT(hiob);
   SCSI_DEVICE SCSI_DPTR deviceTable = unitHandle->deviceTable;
   SCSI_UEXACT8 scsiID = (SCSI_UEXACT8) unitHandle->scsiID;
#if SCSI_PPR_ENABLE
   SCSI_UEXACT8 phase;
#endif

   /* set reentrant address for sequencer */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SEQADDR0),
                        (SCSI_UEXACT8) SCSI_hSIOSTR3_ENTRY(hhcb) >> 2);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SEQADDR1),
                        (SCSI_UEXACT8) SCSI_hSIOSTR3_ENTRY(hhcb) >> 10);

   /* must be message out to initiate negotiation */
   if ((OSD_INEXACT8(SCSI_AICREG(SCSI_SCSISIG)) & SCSI_BUSPHASE) !=
      SCSI_MOPHASE)
   {
      /* default to async/narrow mode */
      SCSI_hXFEROPTASSIGN(hhcb,scsiID,0x00);
      SCSI_hCLEARFAST20REG(hhcb,scsiID);
      return;
   }      

   switch (hiob->SCSI_IP_negoState)
   {
      case SCSI_RESPONSE_WIDE:
         /* send messages built to target */
         /* xfer_option already updated during handling extended msg in. */
         SCSIhSendMessage(hhcb,hiob,4);
         hiob->SCSI_IP_negoState = SCSI_NOT_NEGOTIATING;
         break;
         
      case SCSI_RESPONSE_SYNC:
         /* send messages built to target */
         /* xfer_option already updated during handling extended msg in. */
         SCSIhSendMessage(hhcb,hiob,5);
         hiob->SCSI_IP_negoState = SCSI_NOT_NEGOTIATING;
         break;

#if SCSI_PPR_ENABLE
      case SCSI_RESPONSE_PPR:
         /* send messages built to target */
         /* xfer_option already updated during handling extended msg in. */
         SCSIhSendMessage(hhcb,hiob,8);
         hiob->SCSI_IP_negoState = SCSI_NOT_NEGOTIATING;
         break;
#endif
         
      default:       /* must be a trully sync. nego. needed */
         /* default to async/narrow mode */
         SCSI_hXFEROPTASSIGN(hhcb,scsiID,0x00);
         SCSI_hCLEARFAST20REG(hhcb,scsiID);

#if SCSI_PPR_ENABLE
         /* For Trident chip, we'll use Parallel Protocol Request msg to */
         /* negotiate for wide/narrow and/or sync. transfer only on the  */
         /* dual-transition clocking. */
         if ((hhcb->hardwareMode == SCSI_HMODE_AICTRIDENT) &&
             (deviceTable->bayScsiRate & SCSI_ENABLE_CRC) &&
             !(deviceTable->bayScsiRate & SCSI_SINGLE_EDGE) &&
             deviceTable->SCSI_DF_dtcSupport)
         {
            deviceTable->SCSI_DF_negotiateDtc = 1;

            /* must initiate PPR message */
            SCSIhInitiatePPR(hhcb,hiob);

            /* check msg reject */
            if ((phase = SCSIhWait4Req(hhcb)) == SCSI_MIPHASE)
            {
               /* if target responds with message reject */
               if (OSD_INEXACT8(SCSI_AICREG(SCSI_SCSIBUSL)) == SCSI_MSG07)
               {
                  if (deviceTable->scsiOption & (SCSI_WIDE_XFER | SCSI_SYNC_XFER))
                  {
                     /* assert ATN and ack the last msg byte received */
                     OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSISIG), SCSI_MIPHASE | SCSI_ATNO);
                     OSD_INEXACT8(SCSI_AICREG(SCSI_SCSIDATL));
               
                     /* match the phase */                           
                     if (SCSIhWait4Req(hhcb) == (SCSI_UEXACT8)-1)
                     {
                        SCSIhBadSeq(hhcb);
                        return;
                     }

                     OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSISIG),
                       OSD_INEXACT8(SCSI_AICREG(SCSI_SCSISIG)) & SCSI_BUSPHASE);

                     /* Target gives us a fault support or CHIM runs with the */
                     /* default setting.  We need to clear the target's       */
                     /* DTC support so that later the SCSI xfer rate will be  */
                     /* properly caculated */
                     deviceTable->SCSI_DF_negotiateDtc = 0;
                  
                     /* continue with normal wide/synchronous negotiation */
                     /* procedure */
                     SCSIhNegotiateWideOrSync(hhcb,hiob);
                  }
                  else
                  {
                     /* Ack the last msg byte received. */
                     /* Handle the case where a device is wide but does not */
                     /* support sync. or sync. disabled via target profile. */
                     OSD_INEXACT8(SCSI_AICREG(SCSI_SCSIDATL));
                     
                     /* Done with negotiation */
                     hiob->SCSI_IP_negoState = SCSI_NOT_NEGOTIATING;
                  }
               }
            }
            else if (phase == (SCSI_UEXACT8)-1)
            {
               SCSIhBadSeq(hhcb);
               return;
            }
         }
         else
         {
            /* use the normal wide and sync. negotiation message */
            SCSIhNegotiateWideOrSync(hhcb,hiob);
         }
#else  /* !SCSI_PPR_ENABLE */
         /* use the normal wide and sync. negotiation message */
         SCSIhNegotiateWideOrSync(hhcb,hiob);
#endif /* SCSI_PPR_ENABLE */

         break;
   }  /* end of switch (hiob->SCSI_IP_negoState) */
}

/*********************************************************************
*
*  SCSIhNegotiateWideOrSync
*
*     Negotiate for Wide and/or Sync. transfer
*
*  Return Value:  none
*                  
*  Parameters:    hhcb
*                 hiob
*
*  Remarks:
*
*********************************************************************/
void SCSIhNegotiateWideOrSync (SCSI_HHCB SCSI_HPTR hhcb,
                               SCSI_HIOB SCSI_IPTR hiob)
{
   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_UNIT_CONTROL SCSI_UPTR unitHandle = SCSI_TARGET_UNIT(hiob);
   SCSI_DEVICE SCSI_DPTR deviceTable = unitHandle->deviceTable;
   SCSI_UEXACT8 phase;

   if (deviceTable->scsiOption & SCSI_WIDE_XFER)
   {
      /* must initiate wide negotiation first */
      SCSIhInitiateWide(hhcb,hiob);

      /* check msg reject */
      if ((phase = SCSIhWait4Req(hhcb)) == SCSI_MIPHASE)
      {
         /* if target responds with message reject */
         if (OSD_INEXACT8(SCSI_AICREG(SCSI_SCSIBUSL)) == SCSI_MSG07)
         {
            if (deviceTable->scsiOption & SCSI_SYNC_XFER)
            {
               /* assert ATN and ack the last msg byte received */
               OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSISIG), SCSI_MIPHASE | SCSI_ATNO);
               OSD_INEXACT8(SCSI_AICREG(SCSI_SCSIDATL));
               
               /* match the phase */                           
               if (SCSIhWait4Req(hhcb) == (SCSI_UEXACT8)-1)
               {
                  SCSIhBadSeq(hhcb);
                  return;
               }

               OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSISIG),
                   OSD_INEXACT8(SCSI_AICREG(SCSI_SCSISIG)) & SCSI_BUSPHASE);
                  
               /* continue with initiating synchronous negotiation procedure */
               SCSIhInitiateSync(hhcb,hiob,
                   SCSI_hGETNARROWSCSIOFFSET(hhcb,hiob));
            }
            else
            {
               /* Ack the last msg byte received. */
               /* Handle the case where a device is wide but does not */
               /* support sync. or sync. disabled via target profile. */
               OSD_INEXACT8(SCSI_AICREG(SCSI_SCSIDATL));
                     
               /* Done with negotiation */
               hiob->SCSI_IP_negoState = SCSI_NOT_NEGOTIATING;
            }
         }
      }
      else if (phase == (SCSI_UEXACT8)-1)
      {
         SCSIhBadSeq(hhcb);
         return;
      }
   }
   else
   {
#if SCSI_NEGOTIATION_PER_IOB
      if(deviceTable->SCSI_DF_wasWideXfer)
      {
         /* initiate synchronous with wide offset */
         SCSIhInitiateSync(hhcb,hiob,SCSI_hGETWIDESCSIOFFSET(hhcb,hiob));
         /* restore wide xfer bit in scsiOption that was turn off previously */
         deviceTable->scsiOption |= SCSI_WIDE_XFER;
      }
      else
#endif            
      {
         /* initiate synchronous first only if wide is not on */
         SCSIhInitiateSync(hhcb,hiob,SCSI_hGETNARROWSCSIOFFSET(hhcb,hiob));
      }
   }
}

/*********************************************************************
*
*  SCSIhExtMsgi
*
*     Handle extended message from target interrupt
*
*  Return Value:  none
*                  
*  Parameters:    hhcb
*                 hiob
*
*  Remarks:
*
*********************************************************************/
void SCSIhExtMsgi (SCSI_HHCB SCSI_HPTR hhcb,
                   SCSI_HIOB SCSI_IPTR hiob)
{
   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_UEXACT8 count;

   /* get length field of extended message */ 
   count = OSD_INEXACT8(SCSI_AICREG(SCSI_hPASS_TO_DRIVER(hhcb)));

   /* collect remainning extended messages */
   if (((SCSI_UEXACT8)SCSIhReceiveMessage(hhcb,hiob,count)) != count)
   {
      /* something wrong */
      /* Indicate Ack is needed for the last message byte, which is used */
      /* when CHIM acknowledges the last byte */
      hiob->SCSI_IP_ackLastMsg = SCSI_ACK_NEEDED;
      return;
   }

   /* Modify Data Pointer Message */
   if (hiob->SCSI_IP_workArea[0] == SCSI_MSGMDP)
   {
      SCSI_hMODIFYDATAPTR(hhcb,hiob);
   }
   /* Assume Synch or Wide Negotiation Message */
   else
   {
      /* check current negotiation state */
      switch(hiob->SCSI_IP_negoState)
      {
         case SCSI_INITIATE_WIDE:
            /* just initiated wide, must be response from target */
            if (hiob->SCSI_IP_workArea[0] == SCSI_MSGWIDE)
            {
               SCSIhVerifyWideResponse(hhcb,hiob);
            }
            break;

         case SCSI_INITIATE_SYNC:
            /* just initiated sync, must be response from target */
            if (hiob->SCSI_IP_workArea[0] == SCSI_MSGSYNC)
            {
               SCSIhVerifySyncResponse(hhcb,hiob);
            }
            break;

#if SCSI_PPR_ENABLE
         case SCSI_INITIATE_PPR:
            /* just initiated Parallel Protocol Request, must be response */
            /* from target */
            if (hiob->SCSI_IP_workArea[0] == SCSI_MSGPPR)
            {
               SCSIhVerifyPPRResponse(hhcb,hiob);
            }
            break;
#endif

         default:
            /* must be ext messages initiated from target */
            if (hiob->SCSI_IP_workArea[0] == SCSI_MSGWIDE)
            {
               /* respond to wide negotiation */
               SCSIhRespondToWide(hhcb,hiob);
            }
            else if (hiob->SCSI_IP_workArea[0] == SCSI_MSGSYNC)
            {
               /* respond to sync negotiation */
               SCSIhRespondToSync(hhcb,hiob);
            }

#if SCSI_PPR_ENABLE
            else if (hiob->SCSI_IP_workArea[0] == SCSI_MSGPPR)
            {
               /* respond to parallel protocol request */
               SCSIhRespondToPPR(hhcb,hiob);
            }
#endif
            break;
      }  /* end of switch */
   }
}

/*********************************************************************
*
*  SCSIhInitiatePPR
*
*     Initiate Parallel Protocol Request message process
*
*  Return Value:  none
*                  
*  Parameters:    hhcb
*                 hiob
*
*  Remarks:
*
*********************************************************************/
#if SCSI_PPR_ENABLE
void SCSIhInitiatePPR (SCSI_HHCB SCSI_HPTR hhcb,
                       SCSI_HIOB SCSI_IPTR hiob)
{
   SCSI_DEVICE SCSI_DPTR deviceTable = SCSI_TARGET_UNIT(hiob)->deviceTable;
   SCSI_UEXACT8 offset = 0;      /* assume async. xfer */
   SCSI_UEXACT8 width = 0;       /* assume narrow mode */

   /* indicate initiate parallel protocol request state */
   hiob->SCSI_IP_negoState = SCSI_INITIATE_PPR;

   /* build parallel protocol request message based on configuration */
   hiob->SCSI_IP_workArea[0] = SCSI_MSG01;
   hiob->SCSI_IP_workArea[1] = 6;
   hiob->SCSI_IP_workArea[2] = SCSI_MSGPPR;
   hiob->SCSI_IP_workArea[3] = SCSI_hGETXFERPERIOD(hhcb,hiob);
   hiob->SCSI_IP_workArea[4] = 0;                     /* reserved byte */

   /* target supports sync */
   if (deviceTable->scsiOption & SCSI_SYNC_XFER)
   {
      /* if currently set to run synchronous xfer */
      if (deviceTable->SCSI_DF_setForSync)
      {
         offset = SCSI_hGETNARROWSCSIOFFSET(hhcb,hiob);
      }
      else
      {
         offset = 0;                                  /* Asynchronous mode */
         SCSI_hCLEARSCSIOFFSET(hhcb,hiob);            /* Clear scsi offset */
      }
   }

   hiob->SCSI_IP_workArea[5] = offset;

   /* target supports wide */
   if (deviceTable->scsiOption & SCSI_WIDE_XFER)
   {
      /* if currently set to run wide mode */
      if (deviceTable->SCSI_DF_setForWide)
      {
         width = SCSI_WIDE_WIDTH;   /* Wide bus */
      }
      else
      {
         width = 0;                 /* Narrow bus */

         /* Clear wide xfer bit in scsiOption */
         deviceTable->scsiOption &= ~SCSI_WIDEXFER;
      }
   }

   hiob->SCSI_IP_workArea[6] = width;

   /* setup Protocol Options field: bit 0-3 of byte 7 */
   if ((width == 0) || (offset == 0))
   {
      hiob->SCSI_IP_workArea[7] = 0;   /* ST DATA IN and ST DATA OUT */
      if (hiob->SCSI_IP_workArea[3] <= 0x09)
         hiob->SCSI_IP_workArea[3] = 0x0A;   /* max. speed is 80MB/s in ST mode */
   }
   else if (deviceTable->SCSI_DF_negotiateDtc)
   {
      hiob->SCSI_IP_workArea[7] = 2;   /* DT DATA IN and DT DATA OUT with CRC */
   }
   else
   {
      hiob->SCSI_IP_workArea[7] = 0;   /* ST DATA IN and ST DATA OUT */
      if (hiob->SCSI_IP_workArea[3] <= 0x09)
         hiob->SCSI_IP_workArea[3] = 0x0A;   /* max. speed is 80MB/s in ST mode */
   }

   /* send these 8 bytes extended msg over to target */
   SCSIhSendMessage(hhcb,hiob,8);
}
#endif /* SCSI_PPR_ENABLE */

/*********************************************************************
*
*  SCSIhVerifyPPRResponse
*
*     Accept parallel protocol request response from target
*
*  Return Value:  none
*                  
*  Parameters:    hhcb
*                 hiob
*
*  Remarks:
*
*********************************************************************/
#if SCSI_PPR_ENABLE
void SCSIhVerifyPPRResponse (SCSI_HHCB SCSI_HPTR hhcb,
                             SCSI_HIOB SCSI_IPTR hiob)
{
   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_UNIT_CONTROL SCSI_UPTR unitHandle = SCSI_TARGET_UNIT(hiob);
   SCSI_DEVICE SCSI_DPTR deviceTable = unitHandle->deviceTable;
   SCSI_UEXACT8 scsiID = (SCSI_UEXACT8) unitHandle->scsiID;
   SCSI_UEXACT16 xfer_option;
   SCSI_UEXACT8 offset;
   SCSI_UEXACT8 phase;

   xfer_option = SCSI_hGETXFEROPTION(hhcb,scsiID);

   /* configure hardware based on protocol option response from target. */
   /* only low 4 bits are use for protocol option value */
   switch (hiob->SCSI_IP_workArea[5] & 0x0F)
   {
      case 0:
         deviceTable->SCSI_DF_negotiateDtc = 0;
         break;
         
      case 2:
         deviceTable->SCSI_DF_negotiateDtc = 1;
         break;
      
      default:
         /* for other cases, temporary run in ST mode */
         deviceTable->SCSI_DF_negotiateDtc = 0;
         break;
   }

   /* configure hardware based on wide response from target */
   if (hiob->SCSI_IP_workArea[4])
   {
      xfer_option |= SCSI_WIDE_XFER;
      offset = SCSI_hGETWIDESCSIOFFSET(hhcb, hiob);
   }
   else
   {
      xfer_option &= ~SCSI_WIDE_XFER;
      offset = SCSI_hGETNARROWSCSIOFFSET(hhcb, hiob);
      deviceTable->SCSI_DF_negotiateDtc = 0; /* DT DATA xfer does not run on */
                                             /* narrow bus. */
   }

   /* configure hardware based on xfer rate and offset response from target */
   if ((hiob->SCSI_IP_workArea[2] == 0) ||
       (hiob->SCSI_IP_workArea[1] <= SCSI_hMAXSYNCPERIOD(hhcb,scsiID)))
   {
      xfer_option &= SCSI_WIDE_XFER;
      xfer_option |= SCSI_hCALCSCSIRATE(hhcb,scsiID,hiob->SCSI_IP_workArea[1],
                                        hiob->SCSI_IP_workArea[3]);

      /* workaround for Auto Rate/Offset issue where the scsi rate/offset */
      /* does not take affect immediately for current connection. */
      /* Auto Rate/Offset might be previously disabled. */
      SCSI_hENABLEAUTORATEOPTION(hhcb);

      SCSI_hXFEROPTASSIGN(hhcb,scsiID,xfer_option);
      
      /* workaround for Auto Rate/Offset issue where the scsi rate/offset */
      /* does not take affect immediately for current connection. */
      SCSI_hDISABLEAUTORATEOPTION(hhcb,xfer_option);

      /* Indicate Ack is needed for the last message byte, which is used */
      /* when CHIM acknowledges the last byte */
      hiob->SCSI_IP_ackLastMsg = SCSI_ACK_NEEDED;
   }
   else
   {
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSISIG), SCSI_MIPHASE | SCSI_ATNO);
      do {
         OSD_INEXACT8(SCSI_AICREG(SCSI_SCSIDATL));
         phase = SCSIhWait4Req(hhcb);
      } while (phase == SCSI_MIPHASE);
      if (phase != SCSI_MOPHASE)
      {
         SCSIhBadSeq(hhcb);
      }
      else
      {
         OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSISIG), SCSI_MOPHASE);
         OSD_OUTEXACT8(SCSI_AICREG(SCSI_CLRSINT1), SCSI_CLRATNO);
         OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSIDATL), SCSI_MSG07);
         if (SCSIhWait4Req(hhcb) == (SCSI_UEXACT8)-1)
         {
            SCSIhBadSeq(hhcb);
            return;
         }
         /* set reentrant address for sequencer */
         OSD_OUTEXACT8(SCSI_AICREG(SCSI_SEQADDR0),
               (SCSI_UEXACT8)SCSI_hSIOSTR3_ENTRY(hhcb) >> 2);
         OSD_OUTEXACT8(SCSI_AICREG(SCSI_SEQADDR1),
               (SCSI_UEXACT8)SCSI_hSIOSTR3_ENTRY(hhcb) >> 10);
      }
   }

   /* done with negotiation state */
   hiob->SCSI_IP_negoState = SCSI_NOT_NEGOTIATING;
}
#endif /* SCSI_PPR_ENABLE */

/*********************************************************************
*
*  SCSIhInitiateWide
*
*     Initiate wide negotiation process
*
*  Return Value:  none
*                  
*  Parameters:    hhcb
*                 hiob
*
*  Remarks:
*
*********************************************************************/
void SCSIhInitiateWide (SCSI_HHCB SCSI_HPTR hhcb,
                        SCSI_HIOB SCSI_IPTR hiob)
{
   SCSI_DEVICE SCSI_DPTR deviceTable = SCSI_TARGET_UNIT(hiob)->deviceTable;

   /* indicate initiate wide nego state */
   hiob->SCSI_IP_negoState = SCSI_INITIATE_WIDE;

   /* build wide negotiation message based on configuration */
   hiob->SCSI_IP_workArea[0] = SCSI_MSG01;
   hiob->SCSI_IP_workArea[1] = 2;
   hiob->SCSI_IP_workArea[2] = SCSI_MSGWIDE;

   /* If currently set to run wide mode */
   if (deviceTable->SCSI_DF_setForWide)
   {
      hiob->SCSI_IP_workArea[3] = SCSI_WIDE_WIDTH;    /* Wide bus */
   }
   else
   {
      hiob->SCSI_IP_workArea[3] = 0;                  /* Narrow bus */

      /* Clear wide xfer bit in scsiOption */
      deviceTable->scsiOption &= ~SCSI_WIDEXFER;
   }

   /* send these 4 bytes extended msg over to target */
   SCSIhSendMessage(hhcb,hiob,4);
}

/*********************************************************************
*
*  SCSIhVerifyWideResponse
*
*     Accept wide response from target
*
*  Return Value:  none
*                  
*  Parameters:    hhcb
*                 hiob
*
*  Remarks:
*
*********************************************************************/
void SCSIhVerifyWideResponse (SCSI_HHCB SCSI_HPTR hhcb,
                              SCSI_HIOB SCSI_IPTR hiob)
{
   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_UEXACT16 xfer_option;
   SCSI_UEXACT8 offset;
   SCSI_UNIT_CONTROL SCSI_UPTR unitHandle = SCSI_TARGET_UNIT(hiob);
   SCSI_DEVICE SCSI_DPTR deviceTable = unitHandle->deviceTable;
   SCSI_UEXACT8 scsiID = (SCSI_UEXACT8) unitHandle->scsiID;

   xfer_option = SCSI_hGETXFEROPTION(hhcb,scsiID);

   /* set reentrant address for sequencer */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SEQADDR0),
                        (SCSI_UEXACT8) SCSI_hSIOSTR3_ENTRY(hhcb) >> 2);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SEQADDR1),
                        (SCSI_UEXACT8) SCSI_hSIOSTR3_ENTRY(hhcb) >> 10);

   /* configure hardware based on wide response from target */
   if (hiob->SCSI_IP_workArea[1])
   {
      xfer_option |= SCSI_WIDE_XFER;
      offset = SCSI_hGETWIDESCSIOFFSET(hhcb, hiob);
#if SCSI_NEGOTIATION_PER_IOB
      deviceTable->SCSI_DF_wasWideXfer = 1;
#endif
   }
   else
   {
      xfer_option &= ~SCSI_WIDE_XFER;
      offset = SCSI_hGETNARROWSCSIOFFSET(hhcb, hiob);
#if SCSI_NEGOTIATION_PER_IOB
      deviceTable->SCSI_DF_wasWideXfer = 0;
#endif
   }

   /* workaround for Auto Rate/Offset issue where the scsi rate/offset */
   /* does not take affect immediately for current connection. */
   /* Auto Rate/Offset might be previously disabled. */
   SCSI_hENABLEAUTORATEOPTION(hhcb);

   SCSI_hXFEROPTASSIGN(hhcb,scsiID,xfer_option);

   /* workaround for Auto Rate/Offset issue where the scsi rate/offset */
   /* does not take affect immediately for current connection. */
   SCSI_hDISABLEAUTORATEOPTION(hhcb,xfer_option);

   SCSI_hSETNEGOTIATEDSTATE(hhcb,scsiID,1);

   if (deviceTable->scsiOption & SCSI_SYNC_XFER)
   {
      /* assert ATN and ack the last msg byte received */
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSISIG), SCSI_MIPHASE | SCSI_ATNO);
      OSD_INEXACT8(SCSI_AICREG(SCSI_SCSIDATL));

      /* match the phase */                              
      if (SCSIhWait4Req(hhcb) == (SCSI_UEXACT8)-1)
      {
         SCSIhBadSeq(hhcb);
         return;
      }

      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSISIG),
            OSD_INEXACT8(SCSI_AICREG(SCSI_SCSISIG)) & SCSI_BUSPHASE);

      /* continue with initiating synchronous negotiation procedure */
      SCSIhInitiateSync(hhcb,hiob,offset);
   }
   else
   {
      /* Ack the last msg byte received. */
      /* Handle the case where a device is wide but does not */
      /* support sync. or sync. disabled via target profile. */
      OSD_INEXACT8(SCSI_AICREG(SCSI_SCSIDATL));

      /* done with negotiation */
      hiob->SCSI_IP_negoState = SCSI_NOT_NEGOTIATING;
   }
}

/*********************************************************************
*
*  SCSIhInitiateSync
*
*     Initiate synchronous negotiation process
*
*  Return Value:  none
*                  
*  Parameters:    hhcb
*                 hiob
*                 number of offset
*
*  Remarks:
*
*********************************************************************/
void SCSIhInitiateSync (SCSI_HHCB SCSI_HPTR hhcb,
                        SCSI_HIOB SCSI_IPTR hiob,
                        SCSI_UEXACT8 offset)
{
   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_UNIT_CONTROL SCSI_UPTR unitHandle = SCSI_TARGET_UNIT(hiob);
   SCSI_DEVICE SCSI_DPTR deviceTable = unitHandle->deviceTable;
   SCSI_UEXACT8 phase;

   /* set current negotiation state */
   hiob->SCSI_IP_negoState = SCSI_INITIATE_SYNC;

   /* build synchronous negotiation message */   
   hiob->SCSI_IP_workArea[0] = SCSI_MSG01;
   hiob->SCSI_IP_workArea[1] = 3;
   hiob->SCSI_IP_workArea[2] = SCSI_MSGSYNC;

   hiob->SCSI_IP_workArea[3] = SCSI_hGETXFERPERIOD(hhcb,hiob);

   /* If currently set to run asynchronous xfer */
   if (!deviceTable->SCSI_DF_setForSync)
   {
      offset = 0x00;                      /* Asynchronous mode */
      SCSI_hCLEARSCSIOFFSET(hhcb,hiob);   /* Clear scsi offset */
   }

   hiob->SCSI_IP_workArea[4] = offset;

   /* send messages bytes over to target */
   SCSIhSendMessage(hhcb,hiob,5);
   
   /* Handle message reject */
   if ((phase = SCSIhWait4Req(hhcb)) == SCSI_MIPHASE)
   {
      /* if target responds with message reject */
      if (OSD_INEXACT8(SCSI_AICREG(SCSI_SCSIBUSL)) == SCSI_MSG07)
      {
         /* match the phase and ack the last msg byte received */
         OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSISIG), SCSI_MIPHASE);
         OSD_INEXACT8(SCSI_AICREG(SCSI_SCSIDATL));

         /* match the phase */                           
         if (SCSIhWait4Req(hhcb) == (SCSI_UEXACT8)-1)
         {
            SCSIhBadSeq(hhcb);
            return;
         }

         OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSISIG),
               OSD_INEXACT8(SCSI_AICREG(SCSI_SCSISIG)) & SCSI_BUSPHASE);
      }
   }
   else if (phase == (SCSI_UEXACT8)-1)
   {
      SCSIhBadSeq(hhcb);
      return;
   }
}

/*********************************************************************
*
*  SCSIhVerifySyncResponse
*
*     Accept sync response from target
*
*  Return Value:  none
*                  
*  Parameters:    hhcb
*                 hiob
*
*  Remarks:
*
*********************************************************************/
void SCSIhVerifySyncResponse (SCSI_HHCB SCSI_HPTR hhcb,
                              SCSI_HIOB SCSI_IPTR hiob)
{
   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_UNIT_CONTROL SCSI_UPTR unitHandle = SCSI_TARGET_UNIT(hiob);
   SCSI_UEXACT8 scsiID = (SCSI_UEXACT8) unitHandle->scsiID, phase;
   SCSI_DEVICE SCSI_DPTR deviceTable = unitHandle->deviceTable;
   SCSI_UEXACT16 xfer_option;

   if ((hiob->SCSI_IP_workArea[2] == 0) ||
       (hiob->SCSI_IP_workArea[1] <= SCSI_hMAXSYNCPERIOD(hhcb,scsiID)))
   {
      xfer_option = SCSI_hGETXFEROPTION(hhcb,scsiID);
   
      xfer_option |= SCSI_hCALCSCSIRATE(hhcb,scsiID,hiob->SCSI_IP_workArea[1],
                                        hiob->SCSI_IP_workArea[2]);

#if SCSI_NEGOTIATION_PER_IOB
      if(deviceTable->SCSI_DF_wasWideXfer)
      {
         xfer_option |= SCSI_WIDE_XFER;
      }
#endif

      /* workaround for Auto Rate/Offset issue where the scsi rate/offset */
      /* does not take affect immediately for current connection. */
      /* Auto Rate/Offset might be previously disabled. */
      SCSI_hENABLEAUTORATEOPTION(hhcb);

      SCSI_hXFEROPTASSIGN(hhcb,scsiID,xfer_option);

      /* workaround for Auto Rate/Offset issue where the scsi rate/offset */
      /* does not take affect immediately for current connection. */
      SCSI_hDISABLEAUTORATEOPTION(hhcb,xfer_option);

      SCSI_hSETNEGOTIATEDSTATE(hhcb,scsiID,1);

      /* record in Scratch RAM */
      SCSI_hLOGFAST20MAP(hhcb,scsiID,hiob->SCSI_IP_workArea[1]);

      /* Indicate Ack is needed for the last message byte, which is used */
      /* when CHIM acknowledges the last byte */
      hiob->SCSI_IP_ackLastMsg = SCSI_ACK_NEEDED;
   }
   else
   {
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSISIG), SCSI_MIPHASE | SCSI_ATNO);
      do {
         OSD_INEXACT8(SCSI_AICREG(SCSI_SCSIDATL));
         phase = SCSIhWait4Req(hhcb);
      } while (phase == SCSI_MIPHASE);
      if (phase != SCSI_MOPHASE)
      {
         SCSIhBadSeq(hhcb);
      }
      else
      {
         OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSISIG), SCSI_MOPHASE);
         OSD_OUTEXACT8(SCSI_AICREG(SCSI_CLRSINT1), SCSI_CLRATNO);
         OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSIDATL), SCSI_MSG07);
         if (SCSIhWait4Req(hhcb) == (SCSI_UEXACT8)-1)
         {
            SCSIhBadSeq(hhcb);
            return;
         }
         /* set reentrant address for sequencer */
         OSD_OUTEXACT8(SCSI_AICREG(SCSI_SEQADDR0),
               (SCSI_UEXACT8)SCSI_hSIOSTR3_ENTRY(hhcb) >> 2);
         OSD_OUTEXACT8(SCSI_AICREG(SCSI_SEQADDR1),
               (SCSI_UEXACT8)SCSI_hSIOSTR3_ENTRY(hhcb) >> 10);
      }
   }

   /* done with negotiation state */
   hiob->SCSI_IP_negoState = SCSI_NOT_NEGOTIATING;
}

/*********************************************************************
*
*  SCSIhRespondToPPR
*
*     Respond to parallel protocol request initiated from target
*
*  Return Value:  none
*                  
*  Parameters:    hhcb
*                 hiob
*
*  Remarks:       CHIM will response to a target PPR negotiation
*                 based on the scsiOption setting.  CHIM will
*                 response to run in narrow and async mode if
*                 the current negotiation was suppressed.
*
*********************************************************************/
#if SCSI_PPR_ENABLE
void SCSIhRespondToPPR (SCSI_HHCB SCSI_HPTR hhcb,
                        SCSI_HIOB SCSI_IPTR hiob)
{
   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_UNIT_CONTROL SCSI_UPTR unitHandle = SCSI_TARGET_UNIT(hiob);
   SCSI_DEVICE SCSI_DPTR deviceTable = unitHandle->deviceTable;
   SCSI_UEXACT8 scsiID = (SCSI_UEXACT8) unitHandle->scsiID;
   SCSI_UEXACT8 width = 0;
   SCSI_UEXACT8 period;
   SCSI_UEXACT8 offset;
   SCSI_UEXACT8 protocolOptions;
   SCSI_UEXACT16 xfer_option;

   /* configure hardware based on protocol option initiated from target */
   /* only low 4 bits are use for protocol option value */
   switch (hiob->SCSI_IP_workArea[5] & 0x0F)
   {
      case 0:
         protocolOptions = 0;
         deviceTable->SCSI_DF_negotiateDtc = 0;
         break;
         
      case 2:
         /* Note: when target profile adjustment supported
          *       we need to add code here for TRIDENT
          */ 
         if (hhcb->hardwareMode == SCSI_HMODE_AICTRIDENT)
         {
            /* Only respond with protocol value of 2 for
             * Trident.
             */
            protocolOptions = 2;
            deviceTable->SCSI_DF_negotiateDtc = 1;
         }
         else
         {
            /* Respond with ST DATA IN & OUT */
            protocolOptions = 0;
            deviceTable->SCSI_DF_negotiateDtc = 0;
         }
         break;
      
      default:
         /* for other cases, temporary run in ST mode */
         protocolOptions = 0;
         deviceTable->SCSI_DF_negotiateDtc = 0;
         break;
   }

   /* We should run in narrow mode if negotiation is currently suppressed */
   /* or SCSI_DF_setForWide flag set to 0. */
   if (unitHandle->deviceTable->SCSI_DF_suppressNego ||
       (!unitHandle->deviceTable->SCSI_DF_setForWide))
   {
      xfer_option = width = 0;      /* Narrow bus */

      /* If currently set to run narrow mode. */
      /* We check the flag here again to update scsiOption. */
      if (!unitHandle->deviceTable->SCSI_DF_setForWide)
      {
         /* Clear wide xfer bit in scsiOption */
         unitHandle->deviceTable->scsiOption &= ~SCSI_WIDEXFER;
      }
   }
   else
   {
      /* check the hardware capability and scsiOption setting */
      if ((OSD_INEXACT8(SCSI_AICREG(SCSI_SBLKCTL)) & SCSI_SELWIDE) &&
          (unitHandle->deviceTable->scsiOption & SCSI_WIDEXFER))
      {
         width = SCSI_WIDE_WIDTH;
      }
      else
      {
         width = 0;
      }

      /* coordinate with whatever target initiated */ 
      if (hiob->SCSI_IP_workArea[4] < width)
      {
         width = hiob->SCSI_IP_workArea[4];
      }

      /* configure to hardware */
      if (width == SCSI_WIDE_WIDTH)
      {
         /* set wide transfer and turn on the wide negotiation option */
         /* this is necessary in order to have force negotiation */
         /* working properly */
         xfer_option = SCSI_WIDEXFER;
         unitHandle->deviceTable->scsiOption |= SCSI_WIDEXFER;
      }
      else
      {
         xfer_option = 0;
      }
   }

   if (width == 0)
   {
      protocolOptions = 0;
      deviceTable->SCSI_DF_negotiateDtc = 0;
   }

   /* get the initiate period value */
   period = SCSI_hGETXFERPERIOD(hhcb,hiob);

   /* match the period and offset with msg initiated from target     */
   if (hiob->SCSI_IP_workArea[3])   /* if offset is non-zero, target */
   {                                /* will do synchronous xfer      */
      if (xfer_option)              /* wide xfer was preset was done */
      {
         offset = SCSI_hGETWIDESCSIOFFSET(hhcb,hiob);
      }
      else
      {
         offset = SCSI_hGETNARROWSCSIOFFSET(hhcb,hiob);
      }

      if (hiob->SCSI_IP_workArea[3] < offset)
      {
         offset = hiob->SCSI_IP_workArea[3];
      }

      /* if speed is slower than supported, go async */
      if (hiob->SCSI_IP_workArea[1] > SCSI_hMAXSYNCPERIOD(hhcb,scsiID))
      {                                         /* synchronous xfer, set */
         offset = 0;                            /* to asynchronous xfer  */
      }
      else if (hiob->SCSI_IP_workArea[1] > period) /* the smaller, the faster */
      {
         period = hiob->SCSI_IP_workArea[1];    /* take the slower of the two */
      }
   }
   else
   {
      offset = 0;
   }

   /* We should run in async. mode if negotiation is currently suppressed */
   /* or scsiOption set to run async. mode. */
   if ((unitHandle->deviceTable->SCSI_DF_suppressNego) ||
       !(unitHandle->deviceTable->scsiOption & SCSI_SYNC_XFER))
   {
      offset = 0;                            /* Asynchronous mode */
   }

   /* If currently set to run asynchronous xfer */
   if (!deviceTable->SCSI_DF_setForSync)
   {
      offset = 0;                         /* Asynchronous mode */
      SCSI_hCLEARSCSIOFFSET(hhcb,hiob);   /* Clear scsi offset */
   }

   if (offset == 0)
   {
      protocolOptions = 0;
   }

   /* configure hardware based on the result of calculation */
   xfer_option |= SCSI_hCALCSCSIRATE(hhcb,scsiID,period,offset);

   hhcb->SCSI_HP_respondToWideOrSynch = 1;
   SCSI_hSETNEGOTIATEDSTATE(hhcb,scsiID,1);

   /* workaround for Auto Rate/Offset issue where the scsi rate/offset */
   /* does not take affect immediately for current connection. */
   /* Auto Rate/Offset might be previously disabled. */
   SCSI_hENABLEAUTORATEOPTION(hhcb);

   SCSI_hXFEROPTASSIGN(hhcb,scsiID,xfer_option);

   /* workaround for Auto Rate/Offset issue where the scsi rate/offset */
   /* does not take affect immediately for current connection. */
   SCSI_hDISABLEAUTORATEOPTION(hhcb,xfer_option);

   /* build PPR negotiation message */
   hiob->SCSI_IP_workArea[0] = SCSI_MSG01;
   hiob->SCSI_IP_workArea[1] = 6;
   hiob->SCSI_IP_workArea[2] = SCSI_MSGPPR;
   hiob->SCSI_IP_workArea[3] = period;
   hiob->SCSI_IP_workArea[4] = 0;            /* reserved byte */
   hiob->SCSI_IP_workArea[5] = offset;
   hiob->SCSI_IP_workArea[6] = width;
   hiob->SCSI_IP_workArea[7] = protocolOptions;

   /* Handle the case where an narrow device that initiates wide negotiation */
   /* and an async. device that initiates sync. negotiation. */
   deviceTable->SCSI_DF_wasWideXfer = (width != 0);
   deviceTable->SCSI_DF_wasSyncXfer = (SCSI_UEXACT16)
                           SCSI_hWASSYNCXFER(hhcb,offset);

   /* assert ATN, message will be sent at: */
   /* 1. handle message out routine if the target initiated PPR nego. */
   /*    before the command phase. */
   /* 2. negotiate routine if target initiated PPR nego. right before */
   /*    data phase. */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSISIG), SCSI_MIPHASE | SCSI_ATNO);

   hiob->SCSI_IP_negoState = SCSI_RESPONSE_PPR;

   /* Indicate Ack is needed for the last message byte, which is used */
   /* when CHIM acknowledges the last byte */
   hiob->SCSI_IP_ackLastMsg = SCSI_ACK_NEEDED;
}
#endif /* SCSI_PPR_ENABLE */

/*********************************************************************
*
*  SCSIhRespondToWide
*
*     Respond to wide negotiation initiated from target
*
*  Return Value:  none
*                  
*  Parameters:    hhcb
*                 hiob
*
*  Remarks:       CHIM will response to a target wide negotiation
*                 based on the scsiOption's wide_xfer bit.  CHIM will
*                 response to run in narrow mode if the current
*                 negotiation was suppressed.
*
*********************************************************************/
void SCSIhRespondToWide (SCSI_HHCB SCSI_HPTR hhcb,
                         SCSI_HIOB SCSI_IPTR hiob)
{
   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_UNIT_CONTROL SCSI_UPTR unitHandle = SCSI_TARGET_UNIT(hiob);
   SCSI_UEXACT8 width = 0;
   SCSI_UEXACT8 scsiID = (SCSI_UEXACT8) unitHandle->scsiID;
   SCSI_UEXACT16 xfer_option;

   /* We should run in narrow mode if negotiation is currently suppressed */
   /* or SCSI_DF_setForWide flag set to 0. */
   if (unitHandle->deviceTable->SCSI_DF_suppressNego ||
       (!unitHandle->deviceTable->SCSI_DF_setForWide))
   {
      xfer_option = width = 0;      /* Narrow bus */

      /* If currently set to run narrow mode. */
      /* We check the flag here again to update scsiOption. */
      if (!unitHandle->deviceTable->SCSI_DF_setForWide)
      {
         /* Clear wide xfer bit in scsiOption */
         unitHandle->deviceTable->scsiOption &= ~SCSI_WIDEXFER;
      }
   }
   else
   {
      /* check the hardware capability and scsiOption setting */
      if ((OSD_INEXACT8(SCSI_AICREG(SCSI_SBLKCTL)) & SCSI_SELWIDE) &&
          (unitHandle->deviceTable->scsiOption & SCSI_WIDEXFER))
      {
         width = SCSI_WIDE_WIDTH;
      }
      else
      {
         width = 0;
      }

      /* coordinate with whatever target initiated */ 
      if (hiob->SCSI_IP_workArea[1] < width)
      {
         width = hiob->SCSI_IP_workArea[1];
      }

      /* configure to hardware */
      if (width == SCSI_WIDE_WIDTH)
      {
         /* set wide transfer and turn on the wide negotiation option */
         /* this is necessary in order to have force negotiation */
         /* working properly */
         xfer_option = SCSI_WIDEXFER;
         unitHandle->deviceTable->scsiOption |= SCSI_WIDEXFER;
      }
      else
      {
         xfer_option = 0;
      }
   }

   hhcb->SCSI_HP_respondToWideOrSynch = 1;

   SCSI_hSETNEGOTIATEDSTATE(hhcb,scsiID,1);

   /* workaround for Auto Rate/Offset issue where the scsi rate/offset */
   /* does not take affect immediately for current connection. */
   /* Auto Rate/Offset might be previously disabled. */
   SCSI_hENABLEAUTORATEOPTION(hhcb);

   SCSI_hXFEROPTASSIGN(hhcb,scsiID,xfer_option);

   /* workaround for Auto Rate/Offset issue where the scsi rate/offset */
   /* does not take affect immediately for current connection. */
   SCSI_hDISABLEAUTORATEOPTION(hhcb,xfer_option);

   /* build wide negotiation message */
   hiob->SCSI_IP_workArea[0] = SCSI_MSG01;
   hiob->SCSI_IP_workArea[1] = 2;
   hiob->SCSI_IP_workArea[2] = SCSI_MSGWIDE;
   hiob->SCSI_IP_workArea[3] = width;

   /* Handle the case where an narrow device that initiates wide nego. */
   unitHandle->deviceTable->SCSI_DF_wasWideXfer = (width != 0);

   /* assert ATN, message will be sent at: */
   /* 1. handle message out routine if the target initiated wide nego. */
   /*    before the command phase. */
   /* 2. negotiate routine if target initiated wide nego. right before */
   /*    data phase. */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSISIG), SCSI_MIPHASE | SCSI_ATNO);

   hiob->SCSI_IP_negoState = SCSI_RESPONSE_WIDE;

   /* Indicate Ack is needed for the last message byte, which is used */
   /* when CHIM acknowledges the last byte */
   hiob->SCSI_IP_ackLastMsg = SCSI_ACK_NEEDED;
}

/*********************************************************************
*
*  SCSIhRespondToSync
*
*     Respond to sync negotiation initiated from target
*
*  Return Value:  none
*                  
*  Parameters:    hhcb
*                 hiob
*
*  Remarks:       CHIM will response to a target sync negotiation
*                 based on the scsiOption's sync_xfer bit.  The
*                 max. xfer rate will be in scsiOption for AIC-78xx
*                 hardware and bayScsiRate for AIC-Bayonet and
*                 AIC-Trident.  CHIM will response to run in async
*                 mode if the current negotiation was suppressed.
*
*********************************************************************/
void SCSIhRespondToSync (SCSI_HHCB SCSI_HPTR hhcb,
                         SCSI_HIOB SCSI_IPTR hiob)
{
   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_UNIT_CONTROL SCSI_UPTR unitHandle = SCSI_TARGET_UNIT(hiob);
   SCSI_DEVICE SCSI_DPTR deviceTable = unitHandle->deviceTable;
   SCSI_UEXACT8 scsiID = (SCSI_UEXACT8) unitHandle->scsiID;
   SCSI_UEXACT16 xfer_option;
   SCSI_UEXACT8 period;
   SCSI_UEXACT8 offset;

   /* get the initiate period value */
   period = SCSI_hGETXFERPERIOD(hhcb,hiob);

   /* Get current wide/narrow configuration */
   xfer_option = SCSI_hGETXFEROPTION(hhcb,scsiID);
   if (xfer_option == SCSI_NEEDNEGO)       /* if no nego ever took place    */
   {
      xfer_option = 0;
   }
   else                             /* otherwise, isolate WIDE bit   */
   {
      xfer_option &= SCSI_WIDEXFER;
   }

   /* match the period and offset with msg initiated from target */
   if (hiob->SCSI_IP_workArea[2])   /* if offset is non-zero, tgt */
   {                                /* will do synchronous xfer   */
      if (xfer_option)              /* wide xfer was preset was done */
      {
         offset = SCSI_hGETWIDESCSIOFFSET(hhcb,hiob);
      }
      else
      {
         offset = SCSI_hGETNARROWSCSIOFFSET(hhcb,hiob);
      }

      if (hiob->SCSI_IP_workArea[2] < offset)
      {
         offset = hiob->SCSI_IP_workArea[2];
      }

      /* if speed is slower than supported, go async */
      if (hiob->SCSI_IP_workArea[1] > SCSI_hMAXSYNCPERIOD(hhcb,scsiID))
      {                                         /* synchronous xfer, set */
         offset = 0;                            /* to asynchronous xfer  */
      }
      else if (hiob->SCSI_IP_workArea[1] > period) /* the smaller, the faster */
      {
         period = hiob->SCSI_IP_workArea[1];    /* take the slower of the two */
      }
   }
   else
   {
      offset = 0;
   }

   /* We should run in async. mode if negotiation is currently suppressed */
   /* or scsiOption set to run async. mode. */
   if ((unitHandle->deviceTable->SCSI_DF_suppressNego) ||
       !(unitHandle->deviceTable->scsiOption & SCSI_SYNC_XFER))
   {
      offset = 0;                            /* Asynchronous mode */
   }

   /* If currently set to run asynchronous xfer */
   if (!deviceTable->SCSI_DF_setForSync)
   {
      offset = 0;                         /* Asynchronous mode */
      SCSI_hCLEARSCSIOFFSET(hhcb,hiob);   /* Clear scsi offset */
   }

   /* configure hardware based on the result of calculation */
   xfer_option |= SCSI_hCALCSCSIRATE(hhcb,scsiID,period,offset);
   hhcb->SCSI_HP_respondToWideOrSynch = 1;
   SCSI_hSETNEGOTIATEDSTATE(hhcb,scsiID,1);

   /* workaround for Auto Rate/Offset issue where the scsi rate/offset */
   /* does not take affect immediately for current connection. */
   /* Auto Rate/Offset might be previously disabled. */
   SCSI_hENABLEAUTORATEOPTION(hhcb);

   SCSI_hXFEROPTASSIGN(hhcb,scsiID,xfer_option);

   /* workaround for Auto Rate/Offset issue where the scsi rate/offset */
   /* does not take affect immediately for current connection. */
   SCSI_hDISABLEAUTORATEOPTION(hhcb,xfer_option);

   SCSI_hLOGFAST20MAP(hhcb,scsiID,period);

   /* build response messages */
   hiob->SCSI_IP_workArea[0] = SCSI_MSG01;
   hiob->SCSI_IP_workArea[1] = 3;
   hiob->SCSI_IP_workArea[2] = SCSI_MSGSYNC;
   hiob->SCSI_IP_workArea[3] = period;
   hiob->SCSI_IP_workArea[4] = offset;

   /* Handle the case where an async. device that initiates sync. nego. */
   deviceTable->SCSI_DF_wasSyncXfer = (SCSI_UEXACT16)
                           SCSI_hWASSYNCXFER(hhcb,offset);

   /* assert ATN, message will be sent at: */
   /* 1. handle message out routine if the target initiated sync. nego. */
   /*    before the command phase. */
   /* 2. negotiate routine if target initiated sync. nego. right before */
   /*    data phase. */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSISIG), SCSI_MIPHASE | SCSI_ATNO);

   hiob->SCSI_IP_negoState = SCSI_RESPONSE_SYNC;

   /* Indicate Ack is needed for the last message byte, which is used */
   /* when CHIM acknowledges the last byte */
   hiob->SCSI_IP_ackLastMsg = SCSI_ACK_NEEDED;
}

/*********************************************************************
*
*  SCSIhSendMessage
*
*     Send message out to target
*
*  Return Value:  none
*                  
*  Parameters:    hhcb
*                 hiob
*                 count of message to be sent
*
*  Remarks:
*
*********************************************************************/
void SCSIhSendMessage (SCSI_HHCB SCSI_HPTR hhcb,
                       SCSI_HIOB SCSI_IPTR hiob,
                       int count)
{
   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_UEXACT8 savCnt0,savCnt1,savCnt2;
   SCSI_UINT32 j = 0;
   int i;
   SCSI_UEXACT8 phase;

   /* we must be in message out phase to send message */
   if (SCSIhWait4Req(hhcb) != SCSI_MOPHASE)
   {
      return ;
   }

   savCnt0 = OSD_INEXACT8(SCSI_AICREG(SCSI_STCNT0));        /* save STCNT here */
   savCnt1 = OSD_INEXACT8(SCSI_AICREG(SCSI_STCNT1));        /* save STCNT here */
   savCnt2 = OSD_INEXACT8(SCSI_AICREG(SCSI_STCNT2));        /* save STCNT here */

   /* Transfer all but the last byte of the extended message */
   for (i = 0; i < count-1; i++)
   {
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSIDATL), hiob->SCSI_IP_workArea[i]);
      if ((phase = SCSIhWait4Req(hhcb)) != SCSI_MOPHASE)
      {
         if (phase == (SCSI_UEXACT8)-1)
         {
            SCSIhBadSeq(hhcb);
            return;
         }

         OSD_OUTEXACT8(SCSI_AICREG(SCSI_CLRSINT1), SCSI_CLRATNO);
      
         OSD_OUTEXACT8(SCSI_AICREG(SCSI_STCNT0),savCnt0);      /* restore STCNT */
         OSD_OUTEXACT8(SCSI_AICREG(SCSI_STCNT1),savCnt1);      /* restore STCNT */
         OSD_OUTEXACT8(SCSI_AICREG(SCSI_STCNT2),savCnt2);      /* restore STCNT */
         return;
       }
   }

   /* deassserted ATN and transfer the last byte */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_CLRSINT1), SCSI_CLRATNO);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSIDATL), hiob->SCSI_IP_workArea[i]);

   /* wait until ACK get dropped */
   while ((OSD_INEXACT8(SCSI_AICREG(SCSI_SCSISIG))&SCSI_ACKI) && (j++ < 400000))
   {
      OSD_TIMER(hhcb);
   }

   OSD_OUTEXACT8(SCSI_AICREG(SCSI_STCNT0),savCnt0);            /* restore STCNT */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_STCNT1),savCnt1);            /* restore STCNT */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_STCNT2),savCnt2);            /* restore STCNT */
}

/*********************************************************************
*
*  SCSIhReceiveMessage
*
*     Receive message from target
*
*  Return Value:  number of bytes actually received
*                  
*  Parameters:    hhcb
*                 hiob
*                 count of message to be received
*
*  Remarks:
*
*********************************************************************/
int SCSIhReceiveMessage (SCSI_HHCB SCSI_HPTR hhcb,
                         SCSI_HIOB SCSI_IPTR hiob,
                         int count)
{
   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_UEXACT8 sxfrctl0;
   SCSI_UEXACT8 sxfrctl1;
   SCSI_UEXACT8 phase;
   int i;

   for (i = 0; i < count; i++)
   {
      if ((phase = SCSIhWait4Req(hhcb)) != SCSI_MIPHASE)
      {
         if (phase == (SCSI_UEXACT8)-1)
         {
            SCSIhBadSeq(hhcb);
            break;
         }

         if ((OSD_INEXACT8(SCSI_AICREG(SCSI_SCSISIG)) &
            (SCSI_BUSPHASE | SCSI_ATNI)) == (SCSI_MOPHASE | SCSI_ATNI))
         {
            /* must be parity error, clear it */
            OSD_OUTEXACT8(SCSI_AICREG(SCSI_CLRSINT1), SCSI_CLRATNO);
            OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSISIG), SCSI_MOPHASE);
            sxfrctl0 = OSD_INEXACT8(SCSI_AICREG(SCSI_SXFRCTL0));
            sxfrctl1 = OSD_INEXACT8(SCSI_AICREG(SCSI_SXFRCTL1));
            OSD_OUTEXACT8(SCSI_AICREG(SCSI_SXFRCTL1), sxfrctl1 &
               ~SCSI_ENSPCHK);
            OSD_OUTEXACT8(SCSI_AICREG(SCSI_SXFRCTL1), sxfrctl1 |
               SCSI_ENSPCHK);
            OSD_OUTEXACT8(SCSI_AICREG(SCSI_CLRINT), SCSI_CLRSCSINT);
            OSD_OUTEXACT8(SCSI_AICREG(SCSI_SXFRCTL0), sxfrctl0 &
               ~SCSI_SPIOEN);  /* Place message parity */
            OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSIDATL), SCSI_MSG09);        /* error on bus without */
            OSD_OUTEXACT8(SCSI_AICREG(SCSI_SXFRCTL0), sxfrctl0 | SCSI_SPIOEN);   /* an ack.              */
         }

         break;
      }
      else
      {
         /* collect the message bytes */
         hiob->SCSI_IP_workArea[i] =
                              OSD_INEXACT8(SCSI_AICREG(SCSI_SCSIBUSL));

         /* ack the byte except the last one so that we can assert */
         /* ATN just in case */
         if (i != count - 1)
         {
            OSD_INEXACT8(SCSI_AICREG(SCSI_SCSIDATL));
         }
      }
   }

   return(i);
}

/*********************************************************************
*
*  SCSIhHandleMsgOut
*
*     Handle message out interrupt from sequencer
*
*  Return Value:  none
*                  
*  Parameters:    hhcb
*                 hiob
*
*  Remarks:
*
*********************************************************************/
void SCSIhHandleMsgOut (SCSI_HHCB SCSI_HPTR hhcb,
                        SCSI_HIOB SCSI_IPTR hiob)
{
   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_UEXACT8 sxfrctl1;
   SCSI_UINT32 count = (SCSI_UINT32)0x800000;

   /* match meessage out phase */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSISIG), SCSI_MOPHASE);

   if (OSD_INEXACT8(SCSI_AICREG(SCSI_SCSISIG)) & SCSI_ATNI)
   {
      switch(hiob->SCSI_IP_negoState)
      {
         case SCSI_RESPONSE_WIDE:
            /* send response to target */
            /* xfer_option already updated during handling extended msg in. */
            SCSIhSendMessage(hhcb,hiob,4);
            hiob->SCSI_IP_negoState = SCSI_NOT_NEGOTIATING;
            break;
      
         case SCSI_RESPONSE_SYNC:
            /* send messages built to target */
            /* xfer_option already updated during handling extended msg in. */
            SCSIhSendMessage(hhcb,hiob,5);
            hiob->SCSI_IP_negoState = SCSI_NOT_NEGOTIATING;
            break;

#if SCSI_PPR_ENABLE
         case SCSI_RESPONSE_PPR:
            /* send response to target */
            /* xfer_option already updated during handling extended msg in. */
            SCSIhSendMessage(hhcb,hiob,8);
            hiob->SCSI_IP_negoState = SCSI_NOT_NEGOTIATING;
            break;
#endif

         default:
            /* must be parity error etc */
            sxfrctl1 = OSD_INEXACT8(SCSI_AICREG(SCSI_SXFRCTL1));  /* Turn off parity checking to*/
            OSD_OUTEXACT8(SCSI_AICREG(SCSI_SXFRCTL1), sxfrctl1 &
                  ~SCSI_ENSPCHK);                                 /* clear any residual error.  */
            OSD_OUTEXACT8(SCSI_AICREG(SCSI_SXFRCTL1), sxfrctl1 |
                  SCSI_ENSPCHK);    /* Turn it back on explicitly */
                                    /* because it may have been   */
                                    /* cleared in 'Ph_ParityError'. */
                                    /* (It had to been previously */
                                    /* set or we wouldn't have    */
                                    /* gotten here.)              */
            OSD_OUTEXACT8(SCSI_AICREG(SCSI_CLRSINT1),
                           SCSI_CLRSCSIPERR | SCSI_CLRATNO);
            OSD_OUTEXACT8(SCSI_AICREG(SCSI_CLRINT), SCSI_CLRSCSINT);

#if SCSI_STANDARD_ADVANCED_MODE || SCSI_SWAPPING_ADVANCED_MODE
            if((hhcb->firmwareMode == SCSI_FMODE_STANDARD_ADVANCED) ||
               (hhcb->firmwareMode == SCSI_FMODE_SWAPPING_ADVANCED) )
            {
               if(hiob->SCSI_IP_workArea[SCSI_MAX_WORKAREA-2] & SCSI_PHASEMIS)
               {
                  OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSISIG), SCSI_MOPHASE);
               }
               else
               {
                  OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSISIG), SCSI_MIPHASE);
               }
            }
#endif /* SCSI_STANDARD_ADVANCED_MODE || SCSI_SWAPPING_ADVANCED_MODE */
            OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSIDATL),
                           hiob->SCSI_IP_workArea[SCSI_MAX_WORKAREA-1]);
            break;
      }
   }
   else
   {
      /* just ignore it */
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSIDATL), SCSI_MSG08);
   }

   while ((OSD_INEXACT8(SCSI_AICREG(SCSI_SCSISIG)) & SCSI_ACKI) && (--count))
      ;
}

/*********************************************************************
*
*  SCSIhHandleMsgIn
*
*     Handle message unknown to sequencer
*
*  Return Value:  none
*                  
*  Parameters:    hhcb
*                 hiob
*
*  Remarks:
*
*********************************************************************/
void SCSIhHandleMsgIn (SCSI_HHCB SCSI_HPTR hhcb,
                       SCSI_HIOB SCSI_IPTR hiob)
{
   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_UEXACT8 rejectedMsg;
   SCSI_UEXACT8 phase;
   SCSI_UEXACT8 ignoreXfer;

   if ((OSD_INEXACT8(SCSI_AICREG(SCSI_SCSISIG)) & SCSI_ATNI) == 0)
   {
      switch (OSD_INEXACT8(SCSI_AICREG(SCSI_SCSIBUSL)))  /* reading without ACK */
      {
         case SCSI_MSG01:
            OSD_INEXACT8(SCSI_AICREG(SCSI_SCSIDATL));            /* ACK MSG01 message byte */
            /* collect extended messages count byte*/
            if (((SCSI_UEXACT8)SCSIhReceiveMessage(hhcb,hiob,1)) != 1)
            {
               /* something wrong */
               return;
            }
            OSD_INEXACT8(SCSI_AICREG(SCSI_SCSIDATL));            /* ACK "number bytes" message byte */
            OSD_OUTEXACT8(SCSI_AICREG(SCSI_hPASS_TO_DRIVER(hhcb)), 
               hiob->SCSI_IP_workArea[0]);   /* put number of bytes into pass to driver */
                                             /* assume ack is not needed */
            hiob->SCSI_IP_ackLastMsg = SCSI_ACK_NOT_NEEDED;
                                             /* and call SCSIhExtMsgi */
            SCSIhExtMsgi(hhcb,hiob);
            if (hiob->SCSI_IP_ackLastMsg == SCSI_ACK_NEEDED)
            {
               OSD_INEXACT8(SCSI_AICREG(SCSI_SCSIDATL));         /* ACK last message byte */
            }
            return;
         case SCSI_MSG07:
            /* Get rejected msg */
            rejectedMsg = OSD_INEXACT8(SCSI_AICREG(SCSI_hSCSIDATL_IMAGE(hhcb)));

            /* If msg Identify or tag type, abort */
            if (rejectedMsg & (SCSI_MSGID | SCSI_MSGTAG))
            {
               OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSISIG), SCSI_MIPHASE | SCSI_ATNO);
               OSD_INEXACT8(SCSI_AICREG(SCSI_SCSIDATL));
               SCSIhAbortTarget(hhcb,hiob);
               return;
            }
            break;

         case SCSI_MSG23:                      /* ignore wide residue */
            /* To handle the IGNORE WIDE RESIDUE message:
               1. ACK the 0x23h message.
               2. read w/o ACK the 2nd message byte.
               3. Increment STCNT0-3 to back up due to the extra bad byte(s).
               4. Increment rescnt field of SCB by the reduced number.
               5. Read SHADDR0-3 registers, decrement by the reduced number,
                  and write to HADDR0-3 which will shine thru to SHADDR0-3.
               6. ACK the 2nd message byte.(Done outside of the switch).
               7. Unpause the sequencer.(Done by PH_IntHandler() when return).
            */
            OSD_INEXACT8(SCSI_AICREG(SCSI_SCSIDATL));            /* ACK MSG23 message byte */
            if ((phase = SCSIhWait4Req(hhcb)) == (SCSI_UEXACT8)-1)
            {
               SCSIhBadSeq(hhcb);
               return;
            }
            ignoreXfer = OSD_INEXACT8(SCSI_AICREG(SCSI_SCSIBUSL));  /* rd 2nd byte, no ACK */
            if (ignoreXfer)
            {                          /* do nothing if zero */
               SCSI_hIGNOREWIDERESCALC(hhcb,hiob,ignoreXfer);
            }
            break;

#if SCSI_PAC_NSX_REPEATER || SCSI_NSX_REPEATER 
         case SCSI_MSG00:                      /* command complete */
            /* special nsx repeater communication */
            /* need to obtain pointers to data buffer */
            SCSIhNsxRepeaterCommunication(hhcb,hiob);                 
            break;        
#endif /* SCSI_PAC_NSX_REPEATER || SCSI_NSX_REPEATER */
             
         default:
            OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSISIG), SCSI_MIPHASE |
               SCSI_ATNO);
            do {
               OSD_INEXACT8(SCSI_AICREG(SCSI_SCSIDATL));
               phase = SCSIhWait4Req(hhcb);
            } while (phase == SCSI_MIPHASE);
            if (phase != SCSI_MOPHASE)
            {
               SCSIhBadSeq(hhcb);
               return;
            }
            OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSISIG), SCSI_MOPHASE);
            OSD_OUTEXACT8(SCSI_AICREG(SCSI_CLRSINT1), SCSI_CLRATNO);
            OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSIDATL), SCSI_MSG07);
            return;
      }                          /* end of switch statement */
   }
   OSD_INEXACT8(SCSI_AICREG(SCSI_SCSIDATL));    /* Drive ACK active to release SCSI bus */
}

/*********************************************************************
*
*  SCSIhAIC78XXCalcScsiRate routine - 
*
*  Set synchronous transfer rate based on negotiation for AIC 78xx.
*
*  Return Value:  synchronous transfer rate
*                  
*  Parameters:    hhcb
*                 targetID
*                 period
*                 offset
*
*  Remarks:
*                  
*********************************************************************/
#if SCSI_AIC78XX
SCSI_UEXACT16 SCSIhAIC78XXCalcScsiRate (SCSI_HHCB SCSI_HPTR hhcb,
                                        SCSI_UEXACT8 targetID,
                                        SCSI_UEXACT8 period, 
                                        SCSI_UEXACT8 offset)
{
   SCSI_UEXACT16 syncRate;

   if (period == 12)          /* double speed checking */
      syncRate = 0x00;
   else if (period <= 16)
      syncRate = 0x10;
   else if (period <= 20)
      syncRate = 0x20;
   else if (period <= 25)     /* single speed checking */
      syncRate = 0x00;
   else if (period <= 31)
      syncRate = 0x10;
   else if (period <= 37)
      syncRate = 0x20;
   else if (period <= 43)
      syncRate = 0x30;
   else if (period <= 50)
      syncRate = 0x40;
   else if (period <= 56)
      syncRate = 0x50;
   else if (period <= 62)
      syncRate = 0x60;
   else
      syncRate = 0x70;

   syncRate |= offset;
   syncRate &= 0x00ff;

   return(syncRate);
}
#endif /* SCSI_AIC78XX */

/*********************************************************************
*
*  SCSIhAICBayonetCalcScsiRate routine -  
*
*  Set synchronous transfer rate based on negotiation for Bayonet AIC's
*
*  Return Value:  synchronous transfer rate
*                  
*  Parameters:    hhcb
*                 targetID
*                 period
*                 offset
*
*  Remarks:
*                  
*********************************************************************/
#if SCSI_AICBAYONET
SCSI_UEXACT16 SCSIhAICBayonetCalcScsiRate (SCSI_HHCB SCSI_HPTR hhcb,
                                           SCSI_UEXACT8 targetID,
                                           SCSI_UEXACT8 period,
                                           SCSI_UEXACT8 offset)
{
   SCSI_UEXACT16 syncRate;
   SCSI_UEXACT16 temp = 0;

   if (period == 10)             /* msg 0x0A - 40MB/s */
      syncRate = 0x13;           
   else if (period == 11)        /* msg 0x0B - 33 or 26.6 MB/s */
      syncRate = 0x14;
   else if (period == 12)        /* msg 0x0C - 20MB/s */
      syncRate = 0x15;
   else if (period <= 16)
      syncRate = 0x16;
   else if (period <= 19)
      syncRate = 0x17;
   else if (period <= 25)        /* 10MB/s */
      syncRate = 0x18;
   else if (period <= 31)
      syncRate = 0x19;
   else if (period <= 37)
      syncRate = 0x1A;
   else if (period <= 43)
      syncRate = 0x1B;
   else
      syncRate = 0x1C;           /* 5.0MB/S */

   temp |= (SCSI_UEXACT16)((SCSI_UEXACT16)offset << 8);
   temp &= 0xff00;
   syncRate |= temp;

   return(syncRate);
}
#endif /* SCSI_AICBAYONET */

/*********************************************************************
*
*  SCSIhAICTridentCalcScsiRate routine -  
*
*  Set synchronous transfer rate based on negotiation for Trident AIC's
*
*  Return Value:  synchronous transfer rate
*                  
*  Parameters:    hhcb
*                 targetID
*                 period
*                 offset
*
*  Remarks:
*                  
*********************************************************************/
#if SCSI_AICTRIDENT
SCSI_UEXACT16 SCSIhAICTridentCalcScsiRate (SCSI_HHCB SCSI_HPTR hhcb,
                                           SCSI_UEXACT8 targetID,
                                           SCSI_UEXACT8 period,
                                           SCSI_UEXACT8 offset)
{
   SCSI_UEXACT16 syncRate;
   SCSI_UEXACT16 temp = 0;

#if SCSI_PPR_ENABLE
   /* Check for dual-transition clocking with CRC transfer are supporting */
   /* by both initiator and target. */
   if ((hhcb->deviceTable[targetID].bayScsiRate & SCSI_ENABLE_CRC) &&
       (!(hhcb->deviceTable[targetID].bayScsiRate & SCSI_SINGLE_EDGE)) &&
       hhcb->deviceTable[targetID].SCSI_DF_negotiateDtc)
   {
      /* calculate DTC w/ CRC xfer rate */
      if (period == 9)              /* msg 0x09 - 80MB/s */
         syncRate = 0x42;
      else if (period == 10)        /* msg 0x0A - 40MB/s */
         syncRate = 0x43;           
      else if (period == 11)        /* msg 0x0B - 33 or 26.6 MB/s */
         syncRate = 0x44;
      else if (period == 12)        /* msg 0x0C - 20MB/s */
         syncRate = 0x45;
      else if (period <= 16)
         syncRate = 0x46;
      else if (period <= 19)
         syncRate = 0x47;
      else
         syncRate = 0x48;           /* 10MB/s */
   }
   else
   {
      /* calculate STC xfer rate */
      if (period == 10)             /* msg 0x0A - 40MB/s */
         syncRate = 0x13;           
      else if (period == 11)        /* msg 0x0B - 33 or 26.6 MB/s */
         syncRate = 0x14;
      else if (period == 12)        /* msg 0x0C - 20MB/s */
         syncRate = 0x15;
      else if (period <= 16)
         syncRate = 0x16;
      else if (period <= 19)
         syncRate = 0x17;
      else if (period <= 25)        /* 10MB/s */
         syncRate = 0x18;
      else if (period <= 31)
         syncRate = 0x19;
      else if (period <= 37)
         syncRate = 0x1A;
      else if (period <= 43)
         syncRate = 0x1B;
      else
         syncRate = 0x1C;           /* 5.0MB/S */
   }
#else /* !SCSI_PPR_ENABLE */
   /* calculate STC xfer rate */
   if (period == 10)             /* msg 0x0A - 40MB/s */
      syncRate = 0x13;           
   else if (period == 11)        /* msg 0x0B - 33 or 26.6 MB/s */
      syncRate = 0x14;
   else if (period == 12)        /* msg 0x0C - 20MB/s */
      syncRate = 0x15;
   else if (period <= 16)
      syncRate = 0x16;
   else if (period <= 19)
      syncRate = 0x17;
   else if (period <= 25)        /* 10MB/s */
      syncRate = 0x18;
   else if (period <= 31)
      syncRate = 0x19;
   else if (period <= 37)
      syncRate = 0x1A;
   else if (period <= 43)
      syncRate = 0x1B;
   else
      syncRate = 0x1C;           /* 5.0MB/S */
#endif /* SCSI_PPR_ENABLE */

   temp |= (SCSI_UEXACT16)((SCSI_UEXACT16)offset << 8);
   temp &= 0xff00;
   syncRate |= temp;

   return(syncRate);
}
#endif /* SCSI_AICTRIDENT */

/*********************************************************************
*
*  SCSIhAIC78XXGetXferPeriod
*
*     Returns xfer period for synchronous negotiation process for
*  for AIC 78xx devices.
*
*  Return Value:  xfer period
*                  
*  Parameters:    hhcb
*                 hiob
*
*  Remarks:
*
*********************************************************************/
#if SCSI_AIC78XX
SCSI_UEXACT8 SCSIhAIC78XXGetXferPeriod(SCSI_HHCB SCSI_HPTR hhcb,
                                        SCSI_HIOB SCSI_IPTR hiob)
{
   SCSI_UNIT_CONTROL SCSI_UPTR unitHandle = SCSI_TARGET_UNIT(hiob);
   SCSI_DEVICE SCSI_DPTR deviceTable = unitHandle->deviceTable;
   SCSI_UEXACT8 ScsiOption = deviceTable->scsiOption;
   SCSI_UEXACT8 maxRate;

   /* For fast20 support per device */
   if (deviceTable->SCSI_DF_ultraEnable)
   {
      /* ScsiOption register bit 4-6:                  */
      /* 000 - 20MB/s      period:  50ns     12(12.5)  */
      /* 001 - 16MB/s               62.5ns   16(15.6)  */
      /* 010 - 13.4MB/s             75ns     19(18.75) */
      /* 011 - (NOT SUPPORTED), default to 13.4MB/s    */
      /* 100 - 10MB/s               100ns    25(25)    */
      switch ((SCSI_UEXACT8)((ScsiOption & SCSI_SYNC_RATE) >> 4))
      {
         case 0x00:       /* 20MB/s */
            maxRate = 12;
            break;

         case 0x01:       /* 16MB/s */
            maxRate = 16;
            break;

         case 0x02:       /* 13.3MB/s */
         case 0x03:       /* not supported, default to next higher speed */
            maxRate = 19;
            break;

         case 0x04:       /* 10MB/s */
         default:         /* or any other xfer rate setting   */
                          /* not supported under fast20 mode  */
            maxRate = 25;
            break;
      }
   }
   else
   {
      /* ScsiOption register bit 4-6:                  */
      /* 000 - 10MB/s      period:  100ns    25(25)    */
      /* 001 - 8.0MB/s              125ns    31(31.25) */
      /* 010 - 6.7MB/s              150ns    37(37.5)  */
      /* 011 - 5.7MB/s              175ns    43(43.75) */
      /* 100 - 5.0MB/s              200ns    50(50)    */
      /* 101 - 4.4MB/s              225ns    56(56.25) */
      /* 110 - 4.0MB/s              250ns    62(62.5)  */
      /* 111 - 3.6MB/s              275ns    68(68.75) */
      maxRate = ((ScsiOption & SCSI_SYNC_RATE) >> 4) * 6;
      maxRate += (SCSI_UEXACT8) 25;

      if (ScsiOption & SCSI_SXFR2)
         ++maxRate;
   }

   return(maxRate);
}
#endif /* SCSI_AIC78XX */

/*********************************************************************
*
*  SCSIhAICBayonetGetXferPeriod
*
*     Returns xfer period for synchronous negotiation process for a
*  Bayonet device.
*
*  Return Value:  xfer period
*                  
*  Parameters:    hhcb
*                 hiob
*
*  Remarks:
*
*********************************************************************/
#if SCSI_AICBAYONET
SCSI_UEXACT8 SCSIhAICBayonetGetXferPeriod(SCSI_HHCB SCSI_HPTR hhcb,
                                           SCSI_HIOB SCSI_IPTR hiob)
{
   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_DEVICE SCSI_DPTR deviceTable;
   SCSI_UEXACT8 maxRate;

   /* ScsiOption register bit 0-6:                           */
   /* 001_0011 - 40.0MB/s     period:  25ns        6(6.25)   */
   /* 001_0100 - 26.67MB/s             37.5ns      9(9.375)  */
   /*            (or 33.0MB/s)         (or 30ns)   (or 7.5)  */
   /* 001_0101 - 20.0MB/s              50ns        12(12.5)  */
   /* 001_0110 - 16.0MB/s              62.5ns      16(15.6)  */
   /* 001_0111 - 13.33MB/s             75ns        19(18.75) */
   /* 001_1000 - 10.0MB/s              100ns       25(25)    */
   /* 001_1001 - 8.0MB/s               125ns       31(31.25) */
   /* 001_1010 - 6.7MB/s               150ns       37(37.5)  */
   /* 001_1011 - 5.7MB/s               175ns       43(43.75) */
   /* 001_1100 - 5.0MB/s               200ns       50(50)    */

   SCSI_hGETDEVICETABLEPTR(hiob,deviceTable); 

   if (((deviceTable->bayScsiRate & SCSI_BAYONET_XFERRATE) < 0x13) ||
       ((deviceTable->bayScsiRate & SCSI_BAYONET_XFERRATE) > 0x1C))
   {
      /* xfer rate is invalid; so, default the max rate to 25 (as in 78XX) */
      maxRate = 25;
   }
   /* xfer rate is at 10MB/s or below */
   else if (deviceTable->bayScsiRate & SCSI_BAYONET_XFERRATE3)
   {
      maxRate = (SCSI_UEXACT8)(((deviceTable->bayScsiRate & 0x07) * 6) + 25);
      if (deviceTable->bayScsiRate & SCSI_BAYONET_XFERRATE2)
      {
         ++maxRate;
      }
   }
   else     /* xfer rate is above 10MB/s */
   {
      maxRate = (SCSI_UEXACT8)(((deviceTable->bayScsiRate & 0x07) - 1) * 3);

      /* If the transfer rate is 16 or 13.33 MB/s (bayScsiRate is0x16 or 0x17) */
      /* then the formula produces a value that is wrong by one.  So adjust it */
      if ((deviceTable->bayScsiRate &
           (SCSI_BAYONET_XFERRATE2 | SCSI_BAYONET_XFERRATE1)) ==
          (SCSI_BAYONET_XFERRATE2 | SCSI_BAYONET_XFERRATE1))
      {
         ++maxRate;
      }

      /* SCSI transfer rate is 40 MB/s or 26.67/33 MB/s so cannot use a */
      /* formula to calculate the message */
      if ((deviceTable->bayScsiRate & SCSI_BAYONET_XFERRATE) == 0x13)
      {
         maxRate = 0x0A;
      }
      else if ((deviceTable->bayScsiRate & SCSI_BAYONET_XFERRATE) == 0x14)
      {
         maxRate = 0x0B;
      }

      /* SCSI transfer rate will be at 20MB/s or less if the Expander */
      /* Active bit of SSTAT2 is set or the scsi bus is currently in  */
      /* single-end bus, ENAB20 bit of SBLKCTL register. The smaller  */ 
      /* maxRate is the faster speed. */
      if (((OSD_INEXACT8(SCSI_AICREG(SCSI_SSTAT2)) & SCSI_BAYONET_EXP_ACTIVE) ||
            (OSD_INEXACT8(SCSI_AICREG(SCSI_SBLKCTL)) & SCSI_BAYONET_ENAB20)) && 
            (maxRate < 12))
      {
         maxRate = 12;
      }
   }   

   return(maxRate);
}
#endif /* SCSI_AICBAYONET */

/*********************************************************************
*
*  SCSIhAICTridentGetXferPeriod
*
*     Returns xfer period for synchronous negotiation process for a
*  Trident device.
*
*  Return Value:  xfer period
*                  
*  Parameters:    hhcb
*                 hiob
*
*  Remarks:
*
*********************************************************************/
#if SCSI_AICTRIDENT
SCSI_UEXACT8 SCSIhAICTridentGetXferPeriod (SCSI_HHCB SCSI_HPTR hhcb,
                                           SCSI_HIOB SCSI_IPTR hiob)
{
   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_DEVICE SCSI_DPTR deviceTable;
   SCSI_UEXACT8 maxRate;
   SCSI_UEXACT8 bayScsiRate;

   SCSI_hGETDEVICETABLEPTR(hiob,deviceTable);
   bayScsiRate = deviceTable->bayScsiRate;

#if SCSI_PPR_ENABLE
   /* we will run in dual-edge with CRC transfer mode only if */
   /* following two conditions satisfy:                       */
   /*   1. Scsi rate CRC (bit 6) = 1 and DT/ST (bit 4) = 0    */
   /*   2. Target supports DTC and CRC from Inquiry Data.     */
   if ((bayScsiRate & SCSI_ENABLE_CRC) &&
       (!(bayScsiRate & SCSI_SINGLE_EDGE)) &&
       deviceTable->SCSI_DF_negotiateDtc)
   {
      /* ScsiOption register bit 0-6:                           */
      /* 100_0010 - 80.0MB/s  DT w/ CRC      period:  12.5ns    */
      /* 100_0011 - 40.0MB/s  DT w/ CRC               25ns      */   
      /* 100_0100 - 26.67MB/s DT w/ CRC               37.5ns    */
      /* 100_0101 - 20.0MB/s  DT w/ CRC               50ns      */
      /* 100_0110 - 16.0MB/s  DT w/ CRC               62.5ns    */
      /* 100_0111 - 13.33MB/s DT w/ CRC               75ns      */
      /* 100_1000 - 10.0MB/s  DT w/ CRC               100ns     */
      /* it is a question whether we should support any xfer rates below 10MB/s? */
      /* 001_1001 - 8.0MB/s   DT w/ CRC               125ns     */
      /* 001_1010 - 6.7MB/s   DT w/ CRC               150ns     */
      /* 001_1011 - 5.7MB/s   DT w/ CRC               175ns     */
      /* 001_1100 - 5.0MB/s   DT w/ CRC               200ns     */

      if (((bayScsiRate & SCSI_BAYONET_XFERRATE) < 0x42) ||
          ((bayScsiRate & SCSI_BAYONET_XFERRATE) >= 0x48))
      {
         /* xfer rate is invalid; so, default the max rate to 25 or 10MB/s */
         maxRate = 25;
      }
      else     /* xfer rate is above 10MB/s */
      {
         maxRate = (SCSI_UEXACT8)(((bayScsiRate & 0x07) - 1) * 3);

         /* If the transfer rate is 16 or 13.33 MB/s (bayScsiRate is 0x16 or */
         /* 0x17) then the formula produces a value that is wrong by one.    */
         /* So adjust it  */
         if ((bayScsiRate & (SCSI_BAYONET_XFERRATE2 | SCSI_BAYONET_XFERRATE1)) ==
             (SCSI_BAYONET_XFERRATE2 | SCSI_BAYONET_XFERRATE1))
         {
            ++maxRate;
         }

         /* SCSI transfer rate is 80MB/s 40MB/s or 26.67/33MB/s so cannot */
         /* use a formula to calculate the message */
         if ((bayScsiRate & SCSI_BAYONET_XFERRATE) == 0x42)
         {
            maxRate = 0x09;
         }
         if ((bayScsiRate & SCSI_BAYONET_XFERRATE) == 0x43)
         {
            maxRate = 0x0A;
         }
         else if ((bayScsiRate & SCSI_BAYONET_XFERRATE) == 0x44)
         {
            maxRate = 0x0B;
         }

         /* SCSI transfer rate will be at 20MB/s or less if the Expander */
         /* Active bit of SSTAT2 is set or the scsi bus is currently in  */
         /* single-end bus, ENAB20 bit of SBLKCTL register. The smaller  */ 
         /* maxRate is the faster speed. */
         if (((OSD_INEXACT8(SCSI_AICREG(SCSI_SSTAT2)) & SCSI_BAYONET_EXP_ACTIVE) ||
             (OSD_INEXACT8(SCSI_AICREG(SCSI_SBLKCTL)) & SCSI_BAYONET_ENAB20)) && 
             (maxRate < 12))
         {
            maxRate = 12;
            deviceTable->SCSI_DF_negotiateDtc = 0;
         }
      }   
   }
   else
   {
      /* Adjust bayScsiRate so that it falls back to the speed lesser than */
      /* Ultra 160. */
      bayScsiRate &= ~SCSI_BAYONET_XFERRATE6;   /* clear CRC bit */
      bayScsiRate |= SCSI_BAYONET_XFERRATE4;    /* set ST bit */
      if ((bayScsiRate & SCSI_BAYONET_XFERRATE) < 0x13)
      {
         bayScsiRate = 0x13;
      }

      /* ScsiOption register bit 0-6:                           */
      /* 001_0011 - 40.0MB/s     period:  25ns        6(6.25)   */
      /* 001_0100 - 26.67MB/s             37.5ns      9(9.375)  */
      /*            (or 33.0MB/s)         (or 30ns)   (or 7.5)  */
      /* 001_0101 - 20.0MB/s              50ns        12(12.5)  */
      /* 001_0110 - 16.0MB/s              62.5ns      16(15.6)  */
      /* 001_0111 - 13.33MB/s             75ns        19(18.75) */
      /* 001_1000 - 10.0MB/s              100ns       25(25)    */
      /* 001_1001 - 8.0MB/s               125ns       31(31.25) */
      /* 001_1010 - 6.7MB/s               150ns       37(37.5)  */
      /* 001_1011 - 5.7MB/s               175ns       43(43.75) */
      /* 001_1100 - 5.0MB/s               200ns       50(50)    */

      if (((bayScsiRate & SCSI_BAYONET_XFERRATE) < 0x13) ||
          ((bayScsiRate & SCSI_BAYONET_XFERRATE) > 0x1C))
      {
         /* xfer rate is invalid; so, default the max rate to 25 (as in 78XX) */
         maxRate = 25;
      }
      /* xfer rate is at 10MB/s or below */
      else if (bayScsiRate & SCSI_BAYONET_XFERRATE3)
      {
         maxRate = (SCSI_UEXACT8)(((bayScsiRate & 0x07) * 6) + 25);
         if (bayScsiRate & SCSI_BAYONET_XFERRATE2)
         {
            ++maxRate;
         }
      }
      else     /* xfer rate is above 10MB/s */
      {
         maxRate = (SCSI_UEXACT8)(((bayScsiRate & 0x07) - 1) * 3);

         /* If the transfer rate is 16 or 13.33 MB/s (bayScsiRate is0x16 or */
         /* 0x17) then the formula produces a value that is wrong by one.   */
         /* So adjust it */
         if ((bayScsiRate & (SCSI_BAYONET_XFERRATE2 | SCSI_BAYONET_XFERRATE1)) ==
             (SCSI_BAYONET_XFERRATE2 | SCSI_BAYONET_XFERRATE1))
         {
            ++maxRate;
         }

         /* SCSI transfer rate is 40 MB/s or 26.67/33 MB/s so cannot use a */
         /* formula to calculate the message */
         if ((bayScsiRate & SCSI_BAYONET_XFERRATE) == 0x13)
         {
            maxRate = 0x0A;
         }
         else if ((bayScsiRate & SCSI_BAYONET_XFERRATE) == 0x14)
         {
            maxRate = 0x0B;
         }

         /* SCSI transfer rate will be at 20MB/s or less if the Expander */
         /* Active bit of SSTAT2 is set or the scsi bus is currently in  */
         /* single-end bus, ENAB20 bit of SBLKCTL register. The smaller  */ 
         /* maxRate is the faster speed. */
         if (((OSD_INEXACT8(SCSI_AICREG(SCSI_SSTAT2)) & SCSI_BAYONET_EXP_ACTIVE) ||
             (OSD_INEXACT8(SCSI_AICREG(SCSI_SBLKCTL)) & SCSI_BAYONET_ENAB20)) && 
             (maxRate < 12))
         {
            maxRate = 12;
         }
      }   
   }

#else /* !SCSI_PPR_ENABLE */

   /* ScsiOption register bit 0-6:                           */
   /* 001_0011 - 40.0MB/s     period:  25ns        6(6.25)   */
   /* 001_0100 - 26.67MB/s             37.5ns      9(9.375)  */
   /*            (or 33.0MB/s)         (or 30ns)   (or 7.5)  */
   /* 001_0101 - 20.0MB/s              50ns        12(12.5)  */
   /* 001_0110 - 16.0MB/s              62.5ns      16(15.6)  */
   /* 001_0111 - 13.33MB/s             75ns        19(18.75) */
   /* 001_1000 - 10.0MB/s              100ns       25(25)    */
   /* 001_1001 - 8.0MB/s               125ns       31(31.25) */
   /* 001_1010 - 6.7MB/s               150ns       37(37.5)  */
   /* 001_1011 - 5.7MB/s               175ns       43(43.75) */
   /* 001_1100 - 5.0MB/s               200ns       50(50)    */

   if (((bayScsiRate & SCSI_BAYONET_XFERRATE) < 0x13) ||
       ((bayScsiRate & SCSI_BAYONET_XFERRATE) > 0x1C))
   {
      /* xfer rate is invalid; so, default the max rate to 25 (as in 78XX) */
      maxRate = 25;
   }
   /* xfer rate is at 10MB/s or below */
   else if (bayScsiRate & SCSI_BAYONET_XFERRATE3)
   {
      maxRate = (SCSI_UEXACT8)(((bayScsiRate & 0x07) * 6) + 25);
      if (bayScsiRate & SCSI_BAYONET_XFERRATE2)
      {
         ++maxRate;
      }
   }
   else     /* xfer rate is above 10MB/s */
   {
      maxRate = (SCSI_UEXACT8)(((bayScsiRate & 0x07) - 1) * 3);

      /* If the transfer rate is 16 or 13.33 MB/s (bayScsiRate is0x16 or */
      /* 0x17) then the formula produces a value that is wrong by one.   */
      /* So adjust it */
      if ((bayScsiRate & (SCSI_BAYONET_XFERRATE2 | SCSI_BAYONET_XFERRATE1)) ==
          (SCSI_BAYONET_XFERRATE2 | SCSI_BAYONET_XFERRATE1))
      {
         ++maxRate;
      }

      /* SCSI transfer rate is 40 MB/s or 26.67/33 MB/s so cannot use a */
      /* formula to calculate the message */
      if ((bayScsiRate & SCSI_BAYONET_XFERRATE) == 0x13)
      {
         maxRate = 0x0A;
      }
      else if ((bayScsiRate & SCSI_BAYONET_XFERRATE) == 0x14)
      {
         maxRate = 0x0B;
      }

      /* SCSI transfer rate will be at 20MB/s or less if the Expander */
      /* Active bit of SSTAT2 is set or the scsi bus is currently in  */
      /* single-end bus, ENAB20 bit of SBLKCTL register. The smaller  */ 
      /* maxRate is the faster speed. */
      if (((OSD_INEXACT8(SCSI_AICREG(SCSI_SSTAT2)) & SCSI_BAYONET_EXP_ACTIVE) ||
          (OSD_INEXACT8(SCSI_AICREG(SCSI_SBLKCTL)) & SCSI_BAYONET_ENAB20)) && 
          (maxRate < 12))
      {
         maxRate = 12;
      }
   }   
#endif /* SCSI_PPR_ENABLE */

   return(maxRate);
}
#endif /* SCSI_AICTRIDENT */

/*********************************************************************
*
*  SCSIhAIC78XXMaxSyncPeriod
*
*     Returns max. xfer period to go synchronous xfer
*  for AIC 78xx devices.
*
*  Return Value:  max xfer period
*                  
*  Parameters:    hhcb
*                 targetID
*
*  Remarks:
*
*********************************************************************/
#if SCSI_AIC78XX
SCSI_UEXACT16 SCSIhAIC78XXMaxSyncPeriod (SCSI_HHCB SCSI_HPTR hhcb,
                                         SCSI_UEXACT8 targetID)
{
   return ((SCSI_UEXACT16)SCSI_AIC78XX_MAX_SYNC_PERIOD);
}
#endif /* SCSI_AIC78XX */

/*********************************************************************
*
*  SCSIhAICBayonetMaxSyncPeriod
*
*     Returns max. xfer period to go synchronous xfer
*  for AIC Bayonet devices.
*
*  Return Value:  max xfer period
*                  
*  Parameters:    hhcb
*                 targetID
*
*  Remarks:
*
*********************************************************************/
#if SCSI_AICBAYONET
SCSI_UEXACT16 SCSIhAICBayonetMaxSyncPeriod (SCSI_HHCB SCSI_HPTR hhcb,
                                            SCSI_UEXACT8 targetID)
{
   return ((SCSI_UEXACT16)SCSI_BAYONET_MAX_SYNC_PERIOD);
}
#endif /* SCSI_AICBAYONET */

/*********************************************************************
*
*  SCSIhAICTridentMaxSyncPeriod
*
*     Returns max. xfer period to go synchronous xfer
*  for AIC Trident devices.
*
*  Return Value:  max xfer period
*
*  Parameters:    hhcb
*                 targetID
*  Remarks:
*
*********************************************************************/
#if SCSI_AICTRIDENT
SCSI_UEXACT16 SCSIhAICTridentMaxSyncPeriod (SCSI_HHCB SCSI_HPTR hhcb,
                                            SCSI_UEXACT8 targetID)
{
#if SCSI_PPR_ENABLE
   if ((hhcb->deviceTable[targetID].bayScsiRate & SCSI_ENABLE_CRC) &&
       (!(hhcb->deviceTable[targetID].bayScsiRate & SCSI_SINGLE_EDGE)) &&
       hhcb->deviceTable[targetID].SCSI_DF_negotiateDtc)
   {
      /* max sync period to go sync in DT mode */
      return ((SCSI_UEXACT16)SCSI_TRIDENT_MAX_DTSYNC_PERIOD);
   }
   else
   {
      /* max sync period to go sync in ST mode */
      return ((SCSI_UEXACT16)SCSI_TRIDENT_MAX_STSYNC_PERIOD);
   }
#else /* !SCSI_PPR_ENABLE */
   return ((SCSI_UEXACT16)SCSI_TRIDENT_MAX_STSYNC_PERIOD);
#endif /* SCSI_PPR_ENABLE */
}
#endif /* SCSI_AICTRIDENT */

/*********************************************************************
*  
*  SCSIhStandardGetFast20Reg -
*  
*  Get Fast20 option from the fast20 map in scratch ram.
*
*  Return Value:  void
*             
*  Parameters:    hhcb
*                 targetID
*
*  Remarks:
*                 
*********************************************************************/
#if SCSI_STANDARD32_MODE + SCSI_STANDARD64_MODE
SCSI_UEXACT8 SCSIhStandardGetFast20Reg (SCSI_HHCB SCSI_HPTR hhcb,
                                        SCSI_UEXACT8 targetID)
{
   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_UEXACT8 idMask, fast20Map;

   if (targetID <= 7)
   {
      /* clear the fast20 device map */
      idMask = (SCSI_UEXACT8)(1 << targetID);
      fast20Map = OSD_INEXACT8(SCSI_AICREG(SCSI_hFAST20_LOW(hhcb)));
      fast20Map &= idMask;
   }
   else
   {
      /* clear the fast20 device map */
      idMask = (SCSI_UEXACT8)(1 << (targetID-8));
      fast20Map = OSD_INEXACT8(SCSI_AICREG(SCSI_hFAST20_HIGH(hhcb)));
      fast20Map &= idMask;
   }
   if (fast20Map) return(1);
   else return(0);
}
#endif

/*********************************************************************
*  
*  SCSIhStandardAdvGetFast20Reg -
*  
*  This function is not needed for standard advanced mode
*
*  Return Value:  0
*             
*  Parameters:    hhcb
*                 targetID
*
*  Remarks:
*                 
*********************************************************************/
#if (SCSI_STANDARD_ADVANCED_MODE || SCSI_STANDARD_160M_MODE)
SCSI_UEXACT8 SCSIhStandardAdvGetFast20Reg (SCSI_HHCB SCSI_HPTR hhcb,
                                           SCSI_UEXACT8 targetID)
{
   return(0);
}
#endif


/*********************************************************************
*  
*  SCSIhSwappingGetFast20Reg -
*  
*  Get Fast20 option from the fast20 map in the device table.
*
*  Return Value:  void
*             
*  Parameters:    hhcb
*                 targetID
*
*  Remarks:
*                 
*********************************************************************/
#if SCSI_SWAPPING32_MODE + SCSI_SWAPPING64_MODE
SCSI_UEXACT8 SCSIhSwappingGetFast20Reg (SCSI_HHCB SCSI_HPTR hhcb,
                                        SCSI_UEXACT8 targetID)
{
   return((SCSI_UEXACT8) SCSI_DEVICE_TABLE(hhcb)[targetID].SCSI_DF_fast20EnableW);
}
#endif

/*********************************************************************
*  
*  SCSIhSwappingAdvGetFast20Reg -
*  
*  This function is not needed for swapping advanced mode
*
*  Return Value:  0
*             
*  Parameters:    hhcb
*                 targetID
*
*  Remarks:
*                 
*********************************************************************/
#if (SCSI_SWAPPING_ADVANCED_MODE || SCSI_SWAPPING_160M_MODE)
SCSI_UEXACT8 SCSIhSwappingAdvGetFast20Reg (SCSI_HHCB SCSI_HPTR hhcb,
                                           SCSI_UEXACT8 targetID)
{
   return(0);
}
#endif


/*********************************************************************
*  
*  SCSIhStandardClearFast20Reg -
*  
*  Initializing control register that related to Fast20 to non-fast20
*  mode.
*
*  Return Value:  void
*             
*  Parameters:    hhcb
*                 targetID
*
*  Remarks:
*                 
*********************************************************************/
#if SCSI_STANDARD64_MODE + SCSI_STANDARD32_MODE
void SCSIhStandardClearFast20Reg (SCSI_HHCB SCSI_HPTR hhcb,
                                  SCSI_UEXACT8 targetID)
{
   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_UEXACT8 idMask, fast20Map, sxfrctl0;

   if (targetID <= 7)
   {
      /* clear the fast20 device map */
      idMask = (SCSI_UEXACT8)(1 << targetID);
      fast20Map = OSD_INEXACT8(SCSI_AICREG(SCSI_hFAST20_LOW(hhcb)));
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_hFAST20_LOW(hhcb)), (SCSI_UEXACT8)(fast20Map) & ~idMask);
   }
   else
   {
      /* clear the fast20 device map */
      idMask = (SCSI_UEXACT8)(1 << (targetID-8));
      fast20Map = OSD_INEXACT8(SCSI_AICREG(SCSI_hFAST20_HIGH(hhcb)));
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_hFAST20_HIGH(hhcb)), (SCSI_UEXACT8)(fast20Map & ~idMask));
   }
   sxfrctl0 = OSD_INEXACT8(SCSI_AICREG(SCSI_SXFRCTL0));
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SXFRCTL0), (SCSI_UEXACT8)(sxfrctl0 & ~SCSI_FAST20));
}
#endif


/*********************************************************************
*  
*  SCSIhSwappingClearFast20Reg -
*  
*  Initializing control register that related to Fast20 to non-fast20
*  mode.
*
*  Return Value:  void
*             
*  Parameters:    hhcb
*                 targetID
*
*  Remarks:
*                 
*********************************************************************/
#if SCSI_SWAPPING64_MODE + SCSI_SWAPPING32_MODE
void SCSIhSwappingClearFast20Reg (SCSI_HHCB SCSI_HPTR hhcb,
                                    SCSI_UEXACT8 targetID)
{
   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_HIOB SCSI_IPTR hiob;
   SCSI_UEXACT8 sxfrctl0;
   SCSI_UEXACT8 scbNumber;

   SCSI_DEVICE_TABLE(hhcb)[targetID].SCSI_DF_fast20EnableW = 0;

   /* change the fast20 option for the current hiob in SCB RAM */
   SCSI_hUPDATE_FAST20_HW(hhcb,0);

   /* For all SCBs in ACTPTR array, update fast20 if the same target ID */
   for (scbNumber = 0; scbNumber < hhcb->numberScbs; scbNumber++)
   {
      if ((hiob = SCSI_ACTPTR[scbNumber]) != SCSI_NULL_HIOB)
      {
         if (targetID == SCSI_TARGET_UNIT(hiob)->scsiID)
         {
             /* clear fast 20 bit */
            SCSI_hUPDATE_FAST20_HOST_MEM(hhcb,hiob,0);
         }
      }
   }
   sxfrctl0 = OSD_INEXACT8(SCSI_AICREG(SCSI_SXFRCTL0));
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SXFRCTL0), (SCSI_UEXACT8)(sxfrctl0 & ~SCSI_FAST20));
}
#endif


/*********************************************************************
*  
*  SCSIhStandardLogFast20Map -
*  
*  Log into scratch RAM locations the fast20 device map
*
*  Return Value:  void
*             
*  Parameters:    hhcb
*                 targetID
*                 synchronous period
*
*  Remarks:    This routine assumes the two internal registers of the
*              sequencer were set to zero upon power on/reset.
*                 
*********************************************************************/
#if SCSI_STANDARD64_MODE + SCSI_STANDARD32_MODE
void SCSIhStandardLogFast20Map (SCSI_HHCB SCSI_HPTR hhcb,
                                SCSI_UEXACT8 targetID,
                                SCSI_UEXACT8 period)
{
   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_UEXACT8 fast20Map, idMask, sxfrctl0;

   sxfrctl0 = OSD_INEXACT8(SCSI_AICREG(SCSI_SXFRCTL0));
   if (targetID <= 7)
   {
      idMask = (SCSI_UEXACT8)(1 << targetID);
      fast20Map = OSD_INEXACT8(SCSI_AICREG(SCSI_hFAST20_LOW(hhcb)));

      if (period < SCSI_FAST20_THRESHOLD)
      {
         fast20Map |= idMask;      /* set the fast20 device map */
         sxfrctl0 |= SCSI_FAST20;
      }
      else
      {
         fast20Map &= ~idMask;     /* clear the fast20 device map */
         sxfrctl0 &= ~SCSI_FAST20;
      }
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_hFAST20_LOW(hhcb)), fast20Map);
   }
   else
   {
      idMask = (SCSI_UEXACT8)(1 << (targetID-8));
      fast20Map = OSD_INEXACT8(SCSI_AICREG(SCSI_hFAST20_HIGH(hhcb)));

      if (period < SCSI_FAST20_THRESHOLD)
      {
         fast20Map |= idMask;      /* set the fast20 device map */
         sxfrctl0 |= SCSI_FAST20;
      }
      else
      {
         fast20Map &= ~idMask;     /* clear the fast20 device map */
         sxfrctl0 &= ~SCSI_FAST20;
      }
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_hFAST20_HIGH(hhcb)), fast20Map);
   }
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SXFRCTL0), sxfrctl0);
}
#endif


/*********************************************************************
*  
*  SCSIhSwappingLogFast20Map -
*  
*  Log into scratch RAM locations the fast20 device map
*
*  Return Value:  void
*             
*  Parameters:    hhcb
*                 targetID
*                 synchronous period
*
*  Remarks:    This routine assumes the two internal registers of the
*              sequencer were set to zero upon power on/reset.
*                 
*********************************************************************/
#if SCSI_SWAPPING64_MODE + SCSI_SWAPPING32_MODE
void SCSIhSwappingLogFast20Map (SCSI_HHCB SCSI_HPTR hhcb,
                                  SCSI_UEXACT8 targetID,
                                  SCSI_UEXACT8 period)
{
   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_HIOB SCSI_IPTR hiob;
   SCSI_UEXACT8 sxfrctl0;
   SCSI_UEXACT8 scbNumber;

   sxfrctl0 = OSD_INEXACT8(SCSI_AICREG(SCSI_SXFRCTL0));

   if (period < SCSI_FAST20_THRESHOLD)
   {
      SCSI_DEVICE_TABLE(hhcb)[targetID].SCSI_DF_fast20EnableW = 1; /* set the fast20 device map */
      sxfrctl0 |= SCSI_FAST20;

      /* change the fast20 option for the current hiob in SCB RAM */
      SCSI_hUPDATE_FAST20_HW(hhcb,1);

      /* For all SCBs in ACTPTR array, update fast20 if the same target ID */
      for (scbNumber = 0; scbNumber < hhcb->numberScbs; scbNumber++)
      {
         if ((hiob = SCSI_ACTPTR[scbNumber]) != SCSI_NULL_HIOB)
         {
            if (targetID == SCSI_TARGET_UNIT(hiob)->scsiID)
            {
                /* set fast 20 bit */
               SCSI_hUPDATE_FAST20_HOST_MEM(hhcb,hiob,1);
            }
         }
      }
   }
   else
   {
      SCSI_DEVICE_TABLE(hhcb)[targetID].SCSI_DF_fast20EnableW = 0;     /* clear the fast20 device map */
      sxfrctl0 &= ~SCSI_FAST20;

      /* change the fast20 option for the current hiob in SCB RAM */
      SCSI_hUPDATE_FAST20_HW(hhcb,0);

      /* For all SCBs in ACTPTR array, update fast20 if the same target ID */
      for (scbNumber = 0; scbNumber < hhcb->numberScbs; scbNumber++)
      {
         if ((hiob = SCSI_ACTPTR[scbNumber]) != SCSI_NULL_HIOB)
         {
            if (targetID == SCSI_TARGET_UNIT(hiob)->scsiID)
            {
                /* clear fast 20 bit */
               SCSI_hUPDATE_FAST20_HOST_MEM(hhcb,hiob,0);
            }
         }
      }
   }
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SXFRCTL0), sxfrctl0);
}
#endif



/*********************************************************************
*  
*  SCSIhSwapping32UpdateFast20HostMem
*  
*  Update SCB in host memory with fast 20 bit.
*
*  Return Value:  void
*             
*  Parameters:    hhcb
*                 hiob
*                 flag - (0 - clear fast 20 bit, 1 - set fast 20 bit)
*
*  Remarks:    This routine will clear or set the fast 20 bit for SCB
*              in host mem based on value of flag parameter.
*                 
*********************************************************************/
#if SCSI_SWAPPING32_MODE
void SCSIhSwapping32UpdateFast20HostMem (SCSI_HHCB SCSI_HPTR hhcb,
                                  SCSI_HIOB SCSI_IPTR hiob,
                                  SCSI_UEXACT8 flag)
{
   SCSI_SCB_SWAPPING32 SCSI_HPTR scbW32;
   SCSI_UEXACT8 scontrol;

   scbW32 = (SCSI_SCB_SWAPPING32 SCSI_HPTR) hiob->scbDescriptor->scbBuffer.virtualAddress;

   /* Invalidate the scb buffer cache */
   SCSI_INVALIDATE_CACHE(scbW32, sizeof(SCSI_SCB_SWAPPING32));

   /* Get the scontrol byte */
   SCSI_GET_LITTLE_ENDIAN8(hhcb, &scontrol, scbW32,
        OSDoffsetof(SCSI_SCB_SWAPPING32, scontrol));

   if (flag)
   {
      scontrol |= SCSI_W32_FAST20;
   }
   else
   {
      scontrol &= ~SCSI_W32_FAST20;
   }

   /* Update xfer option to the scb buffer */
   SCSI_PUT_LITTLE_ENDIAN8(hhcb, scbW32, OSDoffsetof(SCSI_SCB_SWAPPING32,
        scontrol), scontrol);

   /* Flush the cache */
   SCSI_FLUSH_CACHE(scbW32 +
      (OSDoffsetof(SCSI_SCB_SWAPPING32,scontrol) / OSD_DMA_SWAP_WIDTH) *
      OSD_DMA_SWAP_WIDTH,OSD_DMA_SWAP_WIDTH);
}
#endif



/*********************************************************************
*  
*  SCSIhSwapping64UpdateFast20HostMem
*  
*  Update SCB in host memory with fast 20 bit.
*
*  Return Value:  void
*             
*  Parameters:    hhcb
*                 hiob
*                 flag - (0 - clear fast 20 bit, 1 - set fast 20 bit)
*
*  Remarks:    This routine will clear or set the fast 20 bit for SCB
*              in host mem based on value of flag parameter.
*                 
*********************************************************************/
#if SCSI_SWAPPING64_MODE
void SCSIhSwapping64UpdateFast20HostMem (SCSI_HHCB SCSI_HPTR hhcb,
                                  SCSI_HIOB SCSI_IPTR hiob,
                                  SCSI_UEXACT8 flag)
{
   SCSI_SCB_SWAPPING64 SCSI_HPTR scbW64;
   SCSI_UEXACT8 scontrol;

   scbW64 = (SCSI_SCB_SWAPPING64 SCSI_HPTR) hiob->scbDescriptor->scbBuffer.virtualAddress;

   /* Invalidate the scb buffer cache */
   SCSI_INVALIDATE_CACHE(scbW64, sizeof(SCSI_SCB_SWAPPING64));

   /* Get the scontrol byte */
   SCSI_GET_LITTLE_ENDIAN8(hhcb, &scontrol, scbW64,
        OSDoffsetof(SCSI_SCB_SWAPPING64, scontrol));

   if (flag)
   {
      scontrol |= SCSI_W_FAST20;
   }
   else
   {
      scontrol &= ~SCSI_W_FAST20;
   }

   /* Update xfer option to the scb buffer */
   SCSI_PUT_LITTLE_ENDIAN8(hhcb, scbW64, OSDoffsetof(SCSI_SCB_SWAPPING64,
        scontrol), scontrol);

   /* Flush the cache */
   SCSI_FLUSH_CACHE(scbW64 +
      (OSDoffsetof(SCSI_SCB_SWAPPING64,scontrol) / OSD_DMA_SWAP_WIDTH) *
      OSD_DMA_SWAP_WIDTH,OSD_DMA_SWAP_WIDTH);
}
#endif

/*********************************************************************
*  
*  SCSIhSwapping32UpdateFast20HW
*  
*  Update SCB in SCB RAM with fast 20 bit.
*
*  Return Value:  void
*             
*  Parameters:    hhcb
*                 flag - (0 - clear fast 20 bit, 1 - set fast 20 bit)
*
*  Remarks:    This routine will clear or set the fast 20 bit for SCB
*              in SCB RAM based on value of flag parameter.
*                 
*********************************************************************/
#if SCSI_SWAPPING32_MODE
void SCSIhSwapping32UpdateFast20HW (SCSI_HHCB SCSI_HPTR hhcb,
                                        SCSI_UEXACT8 flag)
{
   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_UEXACT8 fast20;

   /* change the fast20 option for the current hiob in SCB RAM */
   fast20 = OSD_INEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
                OSDoffsetof(SCSI_SCB_SWAPPING32,scontrol)));
   if (flag)
   {
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+OSDoffsetof(SCSI_SCB_SWAPPING32,
          scontrol)), (fast20 | SCSI_W32_FAST20));
   }
   else
   {
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+OSDoffsetof(SCSI_SCB_SWAPPING32,
          scontrol)), (fast20 & ~SCSI_W32_FAST20));
   }
}
#endif


/*********************************************************************
*  
*  SCSIhSwapping64UpdateFast20HW
*  
*  Update SCB in SCB RAM with fast 20 bit.
*
*  Return Value:  void
*             
*  Parameters:    hhcb
*                 flag - (0 - clear fast 20 bit, 1 - set fast 20 bit)
*
*  Remarks:    This routine will clear or set the fast 20 bit for SCB
*              in SCB RAM based on value of flag parameter.
*                 
*********************************************************************/
#if SCSI_SWAPPING64_MODE
void SCSIhSwapping64UpdateFast20HW (SCSI_HHCB SCSI_HPTR hhcb,
                                        SCSI_UEXACT8 flag)
{
   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_UEXACT8 fast20;

   /* change the fast20 option for the current hiob in SCB RAM */
   fast20 = OSD_INEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
                OSDoffsetof(SCSI_SCB_SWAPPING64,scontrol)));
   if (flag)
   {
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+OSDoffsetof(SCSI_SCB_SWAPPING64,
          scontrol)), (fast20 | SCSI_W_FAST20));
   }
   else
   {
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+OSDoffsetof(SCSI_SCB_SWAPPING64,
          scontrol)), (fast20 & ~SCSI_W_FAST20));
   }
}
#endif

/*********************************************************************
*
*  SCSIhCheckSyncNego routine -
*
*  Readjust the synchronous negotiation parameters based
*
*  Return Value:  None
*                  
*  Parameters:    hhcb
*
*  Remarks:                
*                  
*********************************************************************/
void SCSIhCheckSyncNego (SCSI_HHCB SCSI_HPTR hhcb)
{
   SCSI_UEXACT8 i;
#if SCSI_TARGET_OPERATION && SCSI_MULTIPLEID_SUPPORT
   SCSI_UEXACT8 j;
#endif /* SCSI_TARGET_OPERATION && SCSI_MULTIPLEID_SUPPORT */

   /* Re-Initialize sync/wide negotiation parameters depending on */
   /* whether SuppressNego flag is set                            */
   for (i = 0; i < hhcb->maxDevices; i++)   
   {
#if SCSI_NEGOTIATION_PER_IOB
      /* Unless forced through IOB we should never negotiate; so, assume */
      /* async/narrow */
      SCSI_hXFEROPTASSIGN(hhcb,i,0x00);
#else /* SCSI_NEGOTIATION_PER_IOB */
      if (SCSI_DEVICE_TABLE(hhcb)[i].scsiOption &
          (SCSI_WIDE_XFER | SCSI_SYNC_XFER)
#if SCSI_TARGET_OPERATION
          /* if target mode enabled ONLY then don't set 
           * SCSI_NEEDNEGO unless target initiated negotiation
           * is enabled - our current rates should be 
           * async and narrow.
           */
          && (hhcb->SCSI_HF_initiatorMode ||
              (hhcb->SCSI_HF_targetMode &&
               hhcb->SCSI_HF_targetInitNegotiation))
#endif /* SCSI_TARGET_OPERATION */
         )
      {
         /*  Set negotiation needed */
#if SCSI_TARGET_OPERATION && SCSI_MULTIPLEID_SUPPORT
         /* Need to set negotiation for all possible IDs */
         for (j = 0; j < hhcb->maxDevices; j++)   
         {
            SCSI_hMULTIPLEIDXFEROPTASSIGN(hhcb,i,j,SCSI_NEEDNEGO);
         } 
#else
         /*  Set negotiation needed */
         SCSI_hXFEROPTASSIGN(hhcb,i,SCSI_NEEDNEGO);
#endif /* SCSI_TARGET_OPERATION && SCSI_MULTIPLEID_SUPPORT */
      }
      else
      {
#if SCSI_TARGET_OPERATION && SCSI_MULTIPLEID_SUPPORT
         /* Need to set negotiation for all possible IDs */
         for (j = 0; j < hhcb->maxDevices; j++)   
         {
            SCSI_hMULTIPLEIDXFEROPTASSIGN(hhcb,i,j,0x00);
         } 
#else
         SCSI_hXFEROPTASSIGN(hhcb,i,0x00);
#endif /* SCSI_TARGET_OPERATION && SCSI_MULTIPLEID_SUPPORT */
      }         

      /* Set the SCSI RATE to defaults if SCSI_LOOPBACK_OPERATION = 1 */
      SCSI_hSETXFERRATE(hhcb,i);

#endif /* SCSI_NEGOTIATION_PER_IOB */
   }
}

/*********************************************************************
*
*  SCSIhStandardXferOptAssign routine -
*
*  Assign the transfer option fields for negotiation for standard/test mode
*
*  Return Value:  None
*                  
*  Parameters:    hhcb
*                 target ID / option index
*                 value to be assigned
*
*  Remarks:                
*                  
*********************************************************************/
#if SCSI_STANDARD64_MODE
void SCSIhStandardXferOptAssign (SCSI_HHCB SCSI_HPTR hhcb,
                                 SCSI_UEXACT8 index,
                                 SCSI_UEXACT16 value)
{
   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;

   OSD_OUTEXACT8(SCSI_AICREG(SCSI_hXFER_OPTION(hhcb) + index), (SCSI_UEXACT8) value);

   /* Need to update scsirate register as well because a sync. */
   /* negotiation may occur right before data phase. And the sequencer */
   /* will not load this new xfer_option to scsirate register. */
   /* However, loading of SCSIRATE should only be done if responding */
   /* to Synch or Wide negotiation.  Also 0x8f should not be programmed */
   /* to the SCSIRATE register. */
   if (hhcb->SCSI_HP_respondToWideOrSynch)
   {
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSIRATE),
            ((SCSI_DOUBLET SCSI_LPTR)&value)->u8.order0);
      hhcb->SCSI_HP_respondToWideOrSynch = 0;
   }
}
#endif

/*********************************************************************
*
*  SCSIhHostXferOptAssign routine -
*
*  Assign the transfer option fields for negotiation for modes with
*  xfer option residing in host memory
*
*  Return Value:  None
*                  
*  Parameters:    hhcb
*                 target ID / option index
*                 value to be assigned
*
*  Remarks:
*
*********************************************************************/
#if (SCSI_SWAPPING_MODE || SCSI_STANDARD_ADVANCED_MODE || SCSI_STANDARD_160M_MODE)
void SCSIhHostXferOptAssign (SCSI_HHCB SCSI_HPTR hhcb,
                             SCSI_UEXACT8 targetID,
                             SCSI_UEXACT16 value)
{
   SCSI_HIOB SCSI_IPTR hiob;
   SCSI_UEXACT8 scbNumber;

#if SCSI_TRIDENT_PROTO
   /* workaround to fix the trident issue of switching from DT data xfer */
   /* back to async. data xfer. */
   /* 'value' consists: hi-byte is offset, lo-byte is xfer rate */
   if ((hhcb->hardwareMode == SCSI_HMODE_AICTRIDENT) &&
       ((value & 0xFF00) == 0) &&            /* check for offset 0 */
       ((value & 0x00FF) != 0x008F))         /* check for xfer rate not 0x8F */
   {
      value |= 0x0010;  /* turn on bit 4 (ST bit) */
   }
#endif /* SCSI_TRIDENT_PROTO */

   /* Assign the transfer option fields that common to both */
   /* "with the check condition" or "without" */
   SCSI_hCOMMON_XFER_ASSIGN(hhcb, targetID, value);
   
   /* For all SCBs in ACTPTR array, update xfer_option if the same target ID */
   for (scbNumber = 0; scbNumber < hhcb->numberScbs; scbNumber++)
   {
      if ((hiob = SCSI_ACTPTR[scbNumber]) != SCSI_NULL_HIOB)
      {
#if SCSI_TARGET_OPERATION
         if (hiob->cmd == SCSI_CMD_ESTABLISH_CONNECTION)
         {
            /* no xfer options associated with Establish Connection
             * SCBs.
             */
            continue;
         }   
         if (hiob->SCSI_IP_targetMode)
         {
            /* obtain ID from Nexus */
            if (targetID == SCSI_NEXUS_UNIT(hiob)->scsiID)
            {
               SCSI_hUPDATE_XFER_OPTION_HOST_MEM(hhcb,hiob,value);
            }
         } 
         else
#endif /* SCSI_TARGET_OPERATION */
         if (targetID == SCSI_TARGET_UNIT(hiob)->scsiID)
         {
            SCSI_hUPDATE_XFER_OPTION_HOST_MEM(hhcb,hiob,value);
         }
      }
   }
}
#endif
/*********************************************************************
*
*  SCSIhMultipleHostIdXferOptAssign routine -
*
*  Assign the transfer option fields for negotiation for modes with
*  xfer option residing in host memory
*
*  Return Value:  None
*                  
*  Parameters:    hhcb
*                 target ID / option index
*                 hostID / selectedID
*                 value to be assigned
*
*  Remarks:
*
*********************************************************************/
#if SCSI_TARGET_OPERATION && SCSI_MULTIPLEID_SUPPORT
#if SCSI_SWAPPING_MODE || SCSI_STANDARD_ADVANCED_MODE
void SCSIhMultipleHostIdXferOptAssign (SCSI_HHCB SCSI_HPTR hhcb,
                                       SCSI_UEXACT8 targetID,
                                       SCSI_UEXACT8 hostID,
                                       SCSI_UEXACT16 value)
{
   SCSI_HIOB SCSI_IPTR hiob;
   SCSI_UEXACT8 scbNumber;

   /* Assign the transfer option fields that common to both */
   /* "with the check condition" or "without" */
   SCSI_hMULTIID_COMMON_XFER_ASSIGN(hhcb, targetID, hostID, value);
   
   /* For all SCBs in ACTPTR array, update xfer_option if the same target ID */
   for (scbNumber = 0; scbNumber < hhcb->numberScbs; scbNumber++)
   {
      if ((hiob = SCSI_ACTPTR[scbNumber]) != SCSI_NULL_HIOB)
      {
         if (hiob->cmd == SCSI_CMD_ESTABLISH_CONNECTION)
         {
            /* no xfer options associated with Establish Connection
             * SCBs.
             */
            continue;
         }   
         if (hiob->SCSI_IP_targetMode)
         {
            /* obtain IDs from Nexus */
            if ((targetID == SCSI_NEXUS_UNIT(hiob)->scsiID) &&
                (hostID == SCSI_NEXUS_UNIT(hiob)->selectedID))
            {
               SCSI_hUPDATE_XFER_OPTION_HOST_MEM(hhcb,hiob,value);
            }
         } 
         else
         if (targetID == SCSI_TARGET_UNIT(hiob)->scsiID)
         {
            SCSI_hUPDATE_XFER_OPTION_HOST_MEM(hhcb,hiob,value);
         }
      }
   }

}
#endif
#endif /* SCSI_TARGET_OPERATION && SCSI_MULTIPLEID_SUPPORT */

/*********************************************************************
*
*  SCSIhChkCondXferAssign routine -
*
*  Assign the transfer option fields for negotiation for modes with xfer
*  option in host memory during the check condition only.
*
*  Return Value:  None
*                  
*  Parameters:    hhcb
*                 target ID / option index
*                 value to be assigned
*
*  Remarks:       This routine will handle the issue when running
*                 multiple commands to multiple LUNs of the same target,
*                 one LUN reports a check condition and the other LUN
*                 reconnects after bus free.
*
*                 Previous way: when received a check condition, the
*                 CHIM will force a renegotiation (wide/sync.) by 
*                 writing 0x8F (SCSI_NEEDNEGO) to xfer_option of all
*                 SCBs in active pointer array that has the same target
*                 ID.  Therefore, when target reselects with different
*                 LUN from the one had check condition, the sequencer
*                 will load new xfer rate which is 0x8F to SCSIRATE
*                 register.  And this an invalid setting.
*
*                 New way: when received a check condition, the CHIM
*                 will force a renogatiation (wide/sync.) to all SCBs
*                 in the New queue that has the same target ID and also
*                 the one is currently in next scb array if the entry
*                 is valid.
*                  
*********************************************************************/
#if (SCSI_SWAPPING_MODE || SCSI_STANDARD_ADVANCED_MODE || SCSI_STANDARD_160M_MODE)
void SCSIhChkCondXferAssign (SCSI_HHCB SCSI_HPTR hhcb,
                             SCSI_UEXACT8 targetID,
                             SCSI_UEXACT16 value)
{
   SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_HIOB SCSI_IPTR hiob;
   SCSI_UEXACT8 scbNumber;
   SCSI_UEXACT8 qNewTail;
   SCSI_SCB_DESCRIPTOR SCSI_HPTR scbDescriptor;
   
   SCSI_hCOMMON_XFER_ASSIGN(hhcb, targetID, value);
  
  /* deleted logic that previously referenced next scb array... no more scb array */
   
   /* Get q_new_tail value */
   qNewTail = SCSI_hGETQNEWTAIL(hhcb);
    
   /* Obtain scbnumber */
   scbNumber = OSD_INEXACT8(SCSI_AICREG(SCSI_hQ_NEXT_SCB(hhcb)));
     
   /* Obtain HIOB in Active pointer array */ 
   hiob = SCSI_ACTPTR[scbNumber];
   
   /* Have the search completed? */
   while ((qNewTail != hhcb->SCSI_HP_qNewHead) &&
          (hiob != SCSI_NULL_HIOB))
   {
#if SCSI_TARGET_OPERATION
      if (hiob->cmd == SCSI_CMD_ESTABLISH_CONNECTION)
      {
         /* No xfer options associated with Establish Connection
          * SCBs. Note for swapping advanced mode there should 
          * not be Establish Connection SCBs in the qNew.
          * However, this is not true for standard advanced mode. 
          */
         ;
      }   
      if (hiob->SCSI_IP_targetMode)
      {
         /* obtain ID from Nexus */
         if (targetID == SCSI_NEXUS_UNIT(hiob)->scsiID)
         {
            SCSI_hUPDATE_XFER_OPTION_HOST_MEM(hhcb,hiob,value);
         }
      } 
      else
#endif /* SCSI_TARGET_OPERATION */       
      if (targetID == SCSI_TARGET_UNIT(hiob)->scsiID)
      {
         SCSI_hUPDATE_XFER_OPTION_HOST_MEM(hhcb,hiob,value);
      }
      
       /* .....Obtain the ScbNumber */   
      scbDescriptor=hiob->scbDescriptor->queueNext;
      scbNumber = scbDescriptor->scbNumber;
      
      /* ... obtain HIOB in Active pointer array */
      hiob = SCSI_ACTPTR[scbNumber];

      /* Check if we need to wrap around the queue */
      (SCSI_UEXACT8)qNewTail++;
   }
}

#endif

/*********************************************************************
*
*  SCSIhSwapping32CommonXferAssign routine -
*
*  Assign the common transfer option fields for negotiation for
*  swapping 32 mode
*
*  Return Value:  None
*                  
*  Parameters:    hhcb
*                 target ID / option index
*                 value to be assigned
*
*  Remarks:
*
*********************************************************************/
#if SCSI_SWAPPING32_MODE
void SCSIhSwapping32CommonXferAssign (SCSI_HHCB SCSI_HPTR hhcb,
                                      SCSI_UEXACT8 targetID,
                                      SCSI_UEXACT16 value)
{
   SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_UEXACT8 scbptr;
   SCSI_UEXACT8 hcntrl;

   SCSI_DEVICE_TABLE(hhcb)[targetID].xferOptionHost = value;

   /* check if pause chip necessary */
   if (!((hcntrl = OSD_INEXACT8(SCSI_AICREG(SCSI_HCNTRL))) & SCSI_PAUSEACK))
   {
      SCSIhPauseAndWait(hhcb);      
   }

   scbptr = OSD_INEXACT8(SCSI_AICREG(SCSI_SCBPTR));   /* Save current SCBPTR */

   /* Map ScbPtr to SCB 1 i.e. selection scb */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCBPTR), 0x01);

   /* Check if the current SCB has the same target ID */
   if (targetID == OSD_INEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
                       OSDoffsetof(SCSI_SCB_SWAPPING32,starget))))
   {
      /* change the xfer_option for the current hiob in SCB RAM */
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+OSDoffsetof(SCSI_SCB_SWAPPING32,
          xfer_option)), ((SCSI_DOUBLET SCSI_LPTR)&value)->u8.order0);
   }

   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCBPTR), scbptr);   /* Restore original SCBPTR */

   /* Sync. negotiation may occur right before data phase and */
   /* sequencer does not load this new xfer_option to scsirate register. */
   /* However, loading of SCSIRATE should only be done if responding */
   /* to Synch or Wide negotiation.  Also 0x8f should not be programmed */
   /* to the SCSIRATE register. */
   if (hhcb->SCSI_HP_respondToWideOrSynch)
   {
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSIRATE), 
            ((SCSI_DOUBLET SCSI_LPTR)&value)->u8.order0);
      hhcb->SCSI_HP_respondToWideOrSynch = 0;
   }

   if (!(hcntrl & SCSI_PAUSEACK))
   {
      SCSI_hWRITE_HCNTRL(hhcb,scsiRegister,hcntrl);
   }
}
#endif

/*********************************************************************
*
*  SCSIhSwapping64CommonXferAssign routine -
*
*  Assign the common transfer option fields for negotiation for
*  swapping 64 mode
*
*  Return Value:  None
*                  
*  Parameters:    hhcb
*                 target ID / option index
*                 value to be assigned
*
*  Remarks:
*
*********************************************************************/
#if SCSI_SWAPPING64_MODE
void SCSIhSwapping64CommonXferAssign (SCSI_HHCB SCSI_HPTR hhcb,
                                      SCSI_UEXACT8 targetID,
                                      SCSI_UEXACT16 value)
{
   SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_UEXACT8 scbptr;
   SCSI_UEXACT8 hcntrl;

   SCSI_DEVICE_TABLE(hhcb)[targetID].xferOptionHost = value;

   /* check if pause chip necessary */
   if (!((hcntrl = OSD_INEXACT8(SCSI_AICREG(SCSI_HCNTRL))) & SCSI_PAUSEACK))
   {
      SCSIhPauseAndWait(hhcb);      
   }

   scbptr = OSD_INEXACT8(SCSI_AICREG(SCSI_SCBPTR));   /* Save current SCBPTR */

   /* Map ScbPtr to SCB 1 i.e. selection scb */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCBPTR), 0x01);

   /* Check if the current SCB has the same target ID */
   if (targetID == OSD_INEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
                       OSDoffsetof(SCSI_SCB_SWAPPING64,starget))))
   {
      /* change the xfer_option for the current hiob in SCB RAM */
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+OSDoffsetof(SCSI_SCB_SWAPPING64,
          xfer_option)), ((SCSI_DOUBLET SCSI_LPTR)&value)->u8.order0);
   }

   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCBPTR), scbptr);   /* Restore original SCBPTR */

   /* Sync. negotiation may occur right before data phase and */
   /* sequencer does not load this new xfer_option to scsirate register. */
   /* However, loading of SCSIRATE should only be done if responding */
   /* to Synch or Wide negotiation.  Also 0x8f should not be programmed */
   /* to the SCSIRATE register. */
   if (hhcb->SCSI_HP_respondToWideOrSynch)
   {
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSIRATE), 
            ((SCSI_DOUBLET SCSI_LPTR)&value)->u8.order0);
      hhcb->SCSI_HP_respondToWideOrSynch = 0;
   }

   if (!(hcntrl & SCSI_PAUSEACK))
   {
      SCSI_hWRITE_HCNTRL(hhcb,scsiRegister,hcntrl);
   }
}
#endif

/*********************************************************************
*
*  SCSIhStandardAdvCommonXferAssign routine -
*
*  Assign the common transfer option fields for negotiation for
*  standard advanced mode
*
*  Return Value:  None
*                  
*  Parameters:    hhcb
*                 target ID / option index
*                 value to be assigned
*
*  Remarks:
*
*********************************************************************/
#if SCSI_STANDARD_ADVANCED_MODE
void SCSIhStandardAdvCommonXferAssign (SCSI_HHCB SCSI_HPTR hhcb,
                                      SCSI_UEXACT8 targetID,
                                      SCSI_UEXACT16 value)
{
   SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   

   SCSI_DEVICE_TABLE(hhcb)[targetID].xferOptionHost = value; 

   /* Check if the current SCB has the same target ID.  Must check low nybble */
   /* only since Sequencer uses high nybble for other purposes. */
   if (targetID == (OSD_INEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
                       OSDoffsetof(SCSI_SCB_STANDARD_ADVANCED,starget))) & 0x0F))
   {
      /* change the xfer_option for the current hiob in SCB RAM */
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
         OSDoffsetof(SCSI_SCB_STANDARD_ADVANCED,
         xfer_option0)), ((SCSI_DOUBLET SCSI_LPTR)&value)->u8.order0);
      /* change the xfer_option for the current hiob in SCB RAM */
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
         OSDoffsetof(SCSI_SCB_STANDARD_ADVANCED,
         xfer_option1)), ((SCSI_DOUBLET SCSI_LPTR)&value)->u8.order1);
   }

   
   /* Sync. negotiation may occur right before data phase and */
   /* sequencer does not load this new xfer_option to scsirate register. */
   /* However, loading of SCSIRATE should only be done if responding */
   /* to Synch or Wide negotiation.  Also 0x8f should not be programmed */
   /* to the SCSIRATE register. */
   if (hhcb->SCSI_HP_respondToWideOrSynch)
   {
      SCSI_hLOADSCSIRATE(hhcb,targetID,value);
      hhcb->SCSI_HP_respondToWideOrSynch = 0;
   }
}
#endif

/*********************************************************************
*
*  SCSIhSwappingAdvUpdateScbRamXferOption
*
*     Update the xfer option field of active scb if  
*     the SCSI ID to which the SCB is directed matches
*     the targetID parameter. 
*
*  Return Value:  none
*                  
*  Parameters:    hhcb
*                 targetID
*                 value to be assigned
* 
*  Remarks: 
*     
*                  
*********************************************************************/
#if SCSI_SWAPPING_ADVANCED_MODE
void SCSIhSwappingAdvUpdateScbRamXferOption (SCSI_HHCB SCSI_HPTR hhcb,
                                             SCSI_UEXACT8 targetID,
                                             SCSI_UEXACT16 value)
{
   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;

   /* Check if the current SCB has the same target ID (must check low nibble*/
   /* only as sequencer uses high nibble for other purposes                 */
   if (targetID == (OSD_INEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
                       OSDoffsetof(SCSI_SCB_SWAPPING_ADVANCED,starget)))&0x0F))
   {
      /* change the xfer_option for the current hiob in SCB RAM */
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
         OSDoffsetof(SCSI_SCB_SWAPPING_ADVANCED,
         xfer_option0)), ((SCSI_DOUBLET SCSI_LPTR)&value)->u8.order0);
      /* change the xfer_option for the current hiob in SCB RAM */
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
         OSDoffsetof(SCSI_SCB_SWAPPING_ADVANCED,
         xfer_option1)), ((SCSI_DOUBLET SCSI_LPTR)&value)->u8.order1);
   }
}
#endif /* SCSI_SWAPPING_ADVANCED_MODE */

/*********************************************************************
*
*  SCSIhSwappingAdvCommonXferAssign routine -
*
*  Assign the common transfer option fields for negotiation for
*  standard advanced mode
*
*  Return Value:  None
*                  
*  Parameters:    hhcb
*                 target ID / option index
*                 value to be assigned
*
*  Remarks:
*
*********************************************************************/
#if SCSI_SWAPPING_ADVANCED_MODE
#if !SCSI_MULTIPLEID_SUPPORT
void SCSIhSwappingAdvCommonXferAssign (SCSI_HHCB SCSI_HPTR hhcb,
                                       SCSI_UEXACT8 targetID,
                                       SCSI_UEXACT16 value)
{
   SCSI_DEVICE_TABLE(hhcb)[targetID].xferOptionHost = value;

   /* At this point, we need to get the active SCB which is already pointed */
   /* by the SCBPTR; So, we do not have to save and restore SCBPTR.         */

#if SCSI_TARGET_OPERATION
   if (hhcb->SCSI_HF_targetMode)
   {
      SCSIhTargetSwappingAdvUpdateScbRamXferOption(hhcb,targetID,hhcb->hostScsiID,value);
   }
   else
#endif /* SCSI_TARGET_OPERATION */
   {
      SCSIhSwappingAdvUpdateScbRamXferOption(hhcb,targetID,value);
   } 

   /* Sync. negotiation may occur right before data phase and */
   /* sequencer does not load this new xfer_option to scsirate register. */
   /* However, loading of SCSIRATE should only be done if responding */
   /* to Synch or Wide negotiation.  Also 0x8f should not be programmed */
   /* to the SCSIRATE register. */
   if (hhcb->SCSI_HP_respondToWideOrSynch)
   {
      SCSI_hLOADSCSIRATE(hhcb,targetID,value);
      hhcb->SCSI_HP_respondToWideOrSynch = 0;
   }
}
#endif /* !SCSI_MULTIPLEID_SUPPORT */
#endif /* SCSI_SWAPPING_ADVANCED_MODE */

/*********************************************************************
*
*  SCSIhMultipleHostIdCommonXferAssign routine -
*
*  Assign the common transfer option fields for negotiation for
*  swapping advanced mode when multiple ID support
*
*  Return Value:  None
*                  
*  Parameters:    hhcb
*                 target ID / option index
*                 host ID / our ID
*                 value to be assigned
*
*  Remarks:
*
*********************************************************************/
#if SCSI_SWAPPING_ADVANCED_MODE 
#if SCSI_TARGET_OPERATION && SCSI_MULTIPLEID_SUPPORT
void SCSIhMultipleHostIdCommonXferAssign (SCSI_HHCB SCSI_HPTR hhcb,
                                          SCSI_UEXACT8 targetID,
                                          SCSI_UEXACT8 hostID,
                                          SCSI_UEXACT16 value)
{
   SCSI_DEVICE_TABLE(hhcb)[targetID].xferOptionHost[hostID] = value;

   /* At this point, we need to get the active SCB which is already pointed */
   /* by the SCBPTR; So, we do not have to save and restore SCBPTR          */
   if (hhcb->SCSI_HF_targetMode)
   {
      SCSIhTargetSwappingAdvUpdateScbRamXferOption(hhcb,targetID,hostID,value);
   }
   else
   {
      SCSIhSwappingAdvUpdateScbRamXferOption(hhcb,targetID,value);
   } 
 
   /* Sync. negotiation may occur right before data phase and */
   /* sequencer does not load this new xfer_option to scsirate register. */
   /* However, loading of SCSIRATE should only be done if responding */
   /* to Synch or Wide negotiation.  Also 0x8f should not be programmed */
   /* to the SCSIRATE register. */
   if (hhcb->SCSI_HP_respondToWideOrSynch)
   {
      SCSI_hLOADSCSIRATE(hhcb,targetID,value);
      hhcb->SCSI_HP_respondToWideOrSynch = 0;
   }
}
#endif
#endif /* SCSI_TARGET_OPERATION && SCSI_MULTIPLEID_SUPPORT */

/*********************************************************************
*
*  SCSIhStandard160mCommonXferAssign routine -
*
*  Assign the common transfer option fields for negotiation for
*  standard 160m mode
*
*  Return Value:  None
*                  
*  Parameters:    hhcb
*                 target ID / option index
*                 value to be assigned
*
*  Remarks:
*
*********************************************************************/
#if SCSI_STANDARD_160M_MODE
void SCSIhStandard160mCommonXferAssign (SCSI_HHCB SCSI_HPTR hhcb,
                                        SCSI_UEXACT8 targetID,
                                        SCSI_UEXACT16 value)
{
   SCSI_DEVICE_TABLE(hhcb)[targetID].xferOptionHost = value;

   /* update auto scsi rate/offset per targetID if nego. is not needed */
   if (((SCSI_UEXACT8)(value & 0x00ff)) != SCSI_NEEDNEGO)
   {
      SCSI_hLOADSCSIRATE(hhcb,targetID,value);
   }
}
#endif

/*********************************************************************
*
*  SCSIhSwapping160mCommonXferAssign routine -
*
*  Assign the common transfer option fields for negotiation for
*  standard 160m mode
*
*  Return Value:  None
*                  
*  Parameters:    hhcb
*                 target ID / option index
*                 value to be assigned
*
*  Remarks:
*
*********************************************************************/
#if SCSI_SWAPPING_160M_MODE
void SCSIhSwapping160mCommonXferAssign (SCSI_HHCB SCSI_HPTR hhcb,
                                        SCSI_UEXACT8 targetID,
                                        SCSI_UEXACT16 value)
{
   SCSI_DEVICE_TABLE(hhcb)[targetID].xferOptionHost = value;

   /* update auto scsi rate/offset per targetID if nego. is not needed */
   if (((SCSI_UEXACT8)(value & 0x00ff)) != SCSI_NEEDNEGO)
   {
      SCSI_hLOADSCSIRATE(hhcb,targetID,value);
   }
}
#endif

/*********************************************************************
*
*  SCSIhSwapping32UpdateXferOptionHostMem routine -
*
*  Update the transfer option fields in host memory for negotiation for 
*  swapping 32 mode.
*
*  Return Value:  none.
*                  
*  Parameters:    hiob
*                 value
*
*  Remarks:                
*                  
*********************************************************************/
#if SCSI_SWAPPING32_MODE
void SCSIhSwapping32UpdateXferOptionHostMem (SCSI_HHCB SCSI_HPTR hhcb,
                                             SCSI_HIOB SCSI_IPTR hiob,
                                             SCSI_UEXACT16 value)
{
   SCSI_SCB_SWAPPING32 SCSI_HPTR scbW32;

   /* Update xfer option to the scb buffer */
   scbW32 = (SCSI_SCB_SWAPPING32 SCSI_HPTR) hiob->scbDescriptor->scbBuffer.virtualAddress;
   SCSI_PUT_LITTLE_ENDIAN8(hhcb, scbW32, OSDoffsetof(SCSI_SCB_SWAPPING32,
        xfer_option), ((SCSI_DOUBLET SCSI_LPTR)&value)->u8.order0);

   /* Flush the cache */
   SCSI_FLUSH_CACHE(scbW32, sizeof(SCSI_SCB_SWAPPING32));
}
#endif

/*********************************************************************
*
*  SCSIhSwapping64UpdateXferOptionHostMem routine -
*
*  Update the transfer option fields in host memory for negotiation for 
*  swapping 64 mode.
*
*  Return Value:  none.
*                  
*  Parameters:    hiob
*                 value
*
*  Remarks:                
*                  
*********************************************************************/
#if SCSI_SWAPPING64_MODE
void SCSIhSwapping64UpdateXferOptionHostMem (SCSI_HHCB SCSI_HPTR hhcb,
                                             SCSI_HIOB SCSI_IPTR hiob,
                                             SCSI_UEXACT16 value)
{
   SCSI_SCB_SWAPPING64 SCSI_HPTR scbW64;

   /* Update xfer option to the scb buffer */
   scbW64 = (SCSI_SCB_SWAPPING64 SCSI_HPTR) hiob->scbDescriptor->scbBuffer.virtualAddress;
   SCSI_PUT_LITTLE_ENDIAN8(hhcb, scbW64, OSDoffsetof(SCSI_SCB_SWAPPING64,
        xfer_option), ((SCSI_DOUBLET SCSI_LPTR)&value)->u8.order0);

   /* Flush the cache */
   SCSI_FLUSH_CACHE(scbW64, sizeof(SCSI_SCB_SWAPPING64));
}
#endif

/*********************************************************************
*
*  SCSIhStandardAdvUpdateXferOptionHostMem routine -
*
*  Update the transfer option fields in host memory for negotiation for 
*  standard advanced mode.
*
*  Return Value:  none.
*                  
*  Parameters:    hiob
*                 value
*
*  Remarks:                
*                  
*********************************************************************/
#if SCSI_STANDARD_ADVANCED_MODE
void SCSIhStandardAdvUpdateXferOptionHostMem (SCSI_HHCB SCSI_HPTR hhcb,
                                             SCSI_HIOB SCSI_IPTR hiob,
                                             SCSI_UEXACT16 value)
{
   SCSI_SCB_STANDARD_ADVANCED SCSI_HPTR scbSAdv;

   /* Update xfer option to the scb buffer */
   scbSAdv = (SCSI_SCB_STANDARD_ADVANCED SCSI_HPTR) hiob->scbDescriptor->scbBuffer.virtualAddress;
   SCSI_PUT_LITTLE_ENDIAN16(hhcb, scbSAdv, OSDoffsetof(SCSI_SCB_STANDARD_ADVANCED,
        xfer_option0), value);

   /* Flush the cache */
   SCSI_FLUSH_CACHE(scbSAdv, sizeof(SCSI_SCB_STANDARD_ADVANCED));
}
#endif

/*********************************************************************
*
*  SCSIhSwappingAdvUpdateXferOptionHostMem routine -
*
*  Update the transfer option fields in host memory for negotiation for 
*  swapping advanced mode.
*
*  Return Value:  none.
*                  
*  Parameters:    hiob
*                 value
*
*  Remarks:                
*                  
*********************************************************************/
#if SCSI_SWAPPING_ADVANCED_MODE
void SCSIhSwappingAdvUpdateXferOptionHostMem (SCSI_HHCB SCSI_HPTR hhcb,
                                             SCSI_HIOB SCSI_IPTR hiob,
                                             SCSI_UEXACT16 value)
{
   SCSI_SCB_SWAPPING_ADVANCED SCSI_HPTR scbWAdv;

   /* Update xfer option to the scb buffer */
   scbWAdv = (SCSI_SCB_SWAPPING_ADVANCED SCSI_HPTR) hiob->scbDescriptor->scbBuffer.virtualAddress;
   SCSI_PUT_LITTLE_ENDIAN16(hhcb, scbWAdv, OSDoffsetof(SCSI_SCB_SWAPPING_ADVANCED,
        xfer_option0), value);

   /* Flush the cache */
   SCSI_FLUSH_CACHE(scbWAdv, sizeof(SCSI_SCB_SWAPPING_ADVANCED));
}
#endif

/*********************************************************************
*
*  SCSIhStandard160mUpdateXferOptionHostMem routine -
*
*  Update the transfer option fields in host memory for negotiation for 
*  standard 160m mode.
*
*  Return Value:  none.
*                  
*  Parameters:    hhcb
*                 hiob
*                 value
*
*  Remarks:                
*                  
*********************************************************************/
#if SCSI_STANDARD_160M_MODE
void SCSIhStandard160mUpdateXferOptionHostMem (SCSI_HHCB SCSI_HPTR hhcb,
                                               SCSI_HIOB SCSI_IPTR hiob,
                                               SCSI_UEXACT16 value)
{
   SCSI_SCB_STANDARD_160M SCSI_IPTR scbS160m =
      (SCSI_SCB_STANDARD_160M SCSI_IPTR)hiob->scbDescriptor->scbBuffer.virtualAddress;

   if (hiob->SCSI_IF_tagEnable)
   {
      if (((SCSI_UEXACT8)(value & 0x00ff)) == SCSI_NEEDNEGO)
      {
         /* patch for tagged queue cmd and sync negotiation needed */
         SCSI_PUT_LITTLE_ENDIAN8(hhcb,scbS160m,
              OSDoffsetof(SCSI_SCB_STANDARD_160M,atn_length),0x04);
      }
      else
      {
         /* patch for tagged queue cmd and sync negotiation not needed */
         SCSI_PUT_LITTLE_ENDIAN8(hhcb,scbS160m,
              OSDoffsetof(SCSI_SCB_STANDARD_160M,atn_length),0x03);
      }
   }
   else
   {
      if (((SCSI_UEXACT8)(value & 0x00ff)) == SCSI_NEEDNEGO)
      {
         /* patch for non-tagged queue cmd and sync negotiation needed */
         SCSI_PUT_LITTLE_ENDIAN8(hhcb,scbS160m,
              OSDoffsetof(SCSI_SCB_STANDARD_160M,atn_length),0x02);
      }
      else
      {
         /* patch for non-tagged queue cmd and sync negotiation not needed */
         SCSI_PUT_LITTLE_ENDIAN8(hhcb,scbS160m,
              OSDoffsetof(SCSI_SCB_STANDARD_160M,atn_length),0x01);
      }
   }

   /* Flush the cache */
   SCSI_FLUSH_CACHE(scbS160m, sizeof(SCSI_SCB_STANDARD_160M));
}
#endif

/*********************************************************************
*
*  SCSIhSwapping160mUpdateXferOptionHostMem routine -
*
*  Update the transfer option fields in host memory for negotiation for 
*  swapping 160m mode.
*
*  Return Value:  none.
*                  
*  Parameters:    hiob
*                 value
*
*  Remarks:                
*                  
*********************************************************************/
#if SCSI_SWAPPING_160M_MODE
void SCSIhSwapping160mUpdateXferOptionHostMem (SCSI_HHCB SCSI_HPTR hhcb,
                                               SCSI_HIOB SCSI_IPTR hiob,
                                               SCSI_UEXACT16 value)
{
   SCSI_SCB_SWAPPING_160M SCSI_HPTR scbW160m =
      (SCSI_SCB_SWAPPING_160M SCSI_HPTR)hiob->scbDescriptor->scbBuffer.virtualAddress;

   if (hiob->SCSI_IF_tagEnable)
   {
      if (((SCSI_UEXACT8)(value & 0x00ff)) == SCSI_NEEDNEGO)
      {
         /* patch for tagged queue cmd and sync negotiation needed */
         SCSI_PUT_LITTLE_ENDIAN8(hhcb,scbW160m,
              OSDoffsetof(SCSI_SCB_SWAPPING_160M,atn_length),0x04);
      }
      else
      {
         /* patch for tagged queue cmd and sync negotiation not needed */
         SCSI_PUT_LITTLE_ENDIAN8(hhcb,scbW160m,
              OSDoffsetof(SCSI_SCB_SWAPPING_160M,atn_length),0x03);
      }
   }
   else
   {
      if (((SCSI_UEXACT8)(value & 0x00ff)) == SCSI_NEEDNEGO)
      {
         /* patch for non-tagged queue cmd and sync negotiation needed */
         SCSI_PUT_LITTLE_ENDIAN8(hhcb,scbW160m,
              OSDoffsetof(SCSI_SCB_SWAPPING_160M,atn_length),0x02);
      }
      else
      {
         /* patch for non-tagged queue cmd and sync negotiation not needed */
         SCSI_PUT_LITTLE_ENDIAN8(hhcb,scbW160m,
              OSDoffsetof(SCSI_SCB_SWAPPING_160M,atn_length),0x01);
      }
   }

   /* Flush the cache */
   SCSI_FLUSH_CACHE(scbW160m, sizeof(SCSI_SCB_SWAPPING_160M));
}
#endif

/*********************************************************************
*
*  SCSIhStandardGetOption routine -
*
*  Get the transfer option fields for negotiation for standard/test mode
*
*  Return Value:  option value
*                  
*  Parameters:    hhcb
*                 option index
*
*  Remarks:                
*                  
*********************************************************************/
#if SCSI_STANDARD64_MODE
SCSI_UEXACT16 SCSIhStandardGetOption (SCSI_HHCB SCSI_HPTR hhcb,
                                     SCSI_UEXACT8 index)
{
   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;

   return((SCSI_UEXACT16)(0x00ff & 
         (OSD_INEXACT8(SCSI_AICREG(SCSI_hXFER_OPTION(hhcb) + index)))));
}
#endif

/*********************************************************************
*
*  SCSIhHostGetOption routine -
*
*  Get the transfer option fields for negotiation for swapping mode
*
*  Return Value:  option value
*                  
*  Parameters:    hhcb
*                 option index
*
*  Remarks:                
*                  
*********************************************************************/
#if (SCSI_SWAPPING_MODE || SCSI_STANDARD_ADVANCED_MODE || SCSI_STANDARD_160M_MODE)
#if !SCSI_MULTIPLEID_SUPPORT
SCSI_UEXACT16 SCSIhHostGetOption (SCSI_HHCB SCSI_HPTR hhcb,
                                  SCSI_UEXACT8 index)
{
   return(SCSI_DEVICE_TABLE(hhcb)[index].xferOptionHost);
}
#endif /* !SCSI_MULTIPLEID_SUPPORT */
#endif /* (SCSI_SWAPPING_MODE || SCSI_STANDARD_ADVANCED_MODE || SCSI_STANDARD_160M_MODE) */

/*********************************************************************
*
*  SCSIhMultipleHostIdGetOption routine -
*
*  Get the current transfer option fields for negotiation for swapping mode
*
*  Return Value:  option value
*                  
*  Parameters:    hhcb
*                 option index
*                 our host ID for this operation 
*
*  Remarks:                
*                  
*********************************************************************/
#if SCSI_TARGET_OPERATION && SCSI_MULTIPLEID_SUPPORT
#if (SCSI_SWAPPING_MODE || SCSI_STANDARD_ADVANCED_MODE || SCSI_STANDARD_160M_MODE)
SCSI_UEXACT16 SCSIhMultipleHostIdGetOption (SCSI_HHCB SCSI_HPTR hhcb,
                                  SCSI_UEXACT8 index,
                                  SCSI_UEXACT8 hostID)
{
   return(SCSI_DEVICE_TABLE(hhcb)[index].xferOptionHost[hostID]);
}
#endif /* (SCSI_SWAPPING_MODE || SCSI_STANDARD_ADVANCED_MODE || SCSI_STANDARD_160M_MODE) */
#endif /* SCSI_TARGET_OPERATION && SCSI_MULTIPLEID_SUPPORT */

/*********************************************************************
*
*  SCSIhStandard64ModifyDataPtr routine -
*
*  Handle Modify Data Pointer message for Standard 64 mode.
*
*  Return Value:  none
*                  
*  Parameters:    hhcb
*                 hiob
*
*  Remarks:                
*                  
*********************************************************************/
#if SCSI_STANDARD64_MODE
void SCSIhStandard64ModifyDataPtr (SCSI_HHCB SCSI_HPTR hhcb,
                                   SCSI_HIOB SCSI_IPTR hiob)
{

   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_UEXACT8 phase;


   /* Reject the modify data pointer message */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSISIG), SCSI_MIPHASE | SCSI_ATNO);
   do {
      OSD_INEXACT8(SCSI_AICREG(SCSI_SCSIDATL));
      phase = SCSIhWait4Req(hhcb);
   } while (phase == SCSI_MIPHASE);
   if (phase != SCSI_MOPHASE)
   {
      SCSIhBadSeq(hhcb);
   }
   else
   {
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSISIG), SCSI_MOPHASE);
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_CLRSINT1), SCSI_CLRATNO);
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSIDATL), SCSI_MSG07);
      if (SCSIhWait4Req(hhcb) == (SCSI_UEXACT8)-1)
      {
         SCSIhBadSeq(hhcb);
         return;
      }

      /* Reset Sequencer Start address at SIO215 */
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SEQADDR0),
                      (SCSI_UEXACT8)(SCSI_hSIO215(hhcb) >> 2));
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SEQADDR1), 
                      (SCSI_UEXACT8)(SCSI_hSIO215(hhcb) >> 10));

   }
}

#endif   /* SCSI_STANDARD64_MODE */

/*********************************************************************
*
*  SCSIhSwapping32ModifyDataPtr routine -
*
*  Handle Modify Data Pointer message for Swapping 32 mode.
*
*  Return Value:  none
*                  
*  Parameters:    hhcb
*                 hiob
*
*  Remarks:                
*                  
*********************************************************************/
#if SCSI_SWAPPING32_MODE
void SCSIhSwapping32ModifyDataPtr (SCSI_HHCB SCSI_HPTR hhcb,
                                   SCSI_HIOB SCSI_IPTR hiob)
{

   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_UEXACT8 phase;


   /* Reject the modify data pointer message */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSISIG), SCSI_MIPHASE | SCSI_ATNO);
   do {
      OSD_INEXACT8(SCSI_AICREG(SCSI_SCSIDATL));
      phase = SCSIhWait4Req(hhcb);
   } while (phase == SCSI_MIPHASE);
   if (phase != SCSI_MOPHASE)
   {
      SCSIhBadSeq(hhcb);
   }
   else
   {
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSISIG), SCSI_MOPHASE);
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_CLRSINT1), SCSI_CLRATNO);
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSIDATL), SCSI_MSG07);
      if (SCSIhWait4Req(hhcb) == (SCSI_UEXACT8)-1)
      {
         SCSIhBadSeq(hhcb);
         return;
      }

      /* Reset Sequencer Start address at SIO215 */
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SEQADDR0),
                      (SCSI_UEXACT8)(SCSI_hSIO215(hhcb) >> 2));
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SEQADDR1), 
                      (SCSI_UEXACT8)(SCSI_hSIO215(hhcb) >> 10));

   }
}


#endif   /* SCSI_SWAPPING32_MODE */


/*********************************************************************
*
*  SCSIhSwapping64ModifyDataPtr routine -
*
*  Handle Modify Data Pointer message for Swapping 32 mode.
*
*  Return Value:  none
*                  
*  Parameters:    hhcb
*                 hiob
*
*  Remarks:                
*                  
*********************************************************************/
#if SCSI_SWAPPING64_MODE
void SCSIhSwapping64ModifyDataPtr (SCSI_HHCB SCSI_HPTR hhcb,
                                   SCSI_HIOB SCSI_IPTR hiob)
{

   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_UEXACT8 phase;


   /* Reject the modify data pointer message */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSISIG), SCSI_MIPHASE | SCSI_ATNO);
   do {
      OSD_INEXACT8(SCSI_AICREG(SCSI_SCSIDATL));
      phase = SCSIhWait4Req(hhcb);
   } while (phase == SCSI_MIPHASE);
   if (phase != SCSI_MOPHASE)
   {
      SCSIhBadSeq(hhcb);
   }
   else
   {
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSISIG), SCSI_MOPHASE);
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_CLRSINT1), SCSI_CLRATNO);
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSIDATL), SCSI_MSG07);
      if (SCSIhWait4Req(hhcb) == (SCSI_UEXACT8)-1)
      {
         SCSIhBadSeq(hhcb);
         return;
      }

       /* Reset Sequencer Start address at SIO215 */
       OSD_OUTEXACT8(SCSI_AICREG(SCSI_SEQADDR0),
                      (SCSI_UEXACT8)(SCSI_hSIO215(hhcb) >> 2));
       OSD_OUTEXACT8(SCSI_AICREG(SCSI_SEQADDR1), 
                      (SCSI_UEXACT8)(SCSI_hSIO215(hhcb) >> 10));

   }
}


#endif   /* SCSI_SWAPPING64_MODE */

/*********************************************************************
*
*  SCSIhStandardAdvModifyDataPtr routine -
*
*  Handle Modify Data Pointer message for Standard Adv mode.
*
*  Return Value:  none
*                  
*  Parameters:    hhcb
*                 hiob
*
*  Remarks:                
*                  
*********************************************************************/
#if SCSI_STANDARD_ADVANCED_MODE
void SCSIhStandardAdvModifyDataPtr (SCSI_HHCB SCSI_HPTR hhcb,
                                    SCSI_HIOB SCSI_IPTR hiob)
{
   SCSI_UEXACT8 phase;
   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;

   /* Reject the modify data pointer message */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSISIG), SCSI_MIPHASE | SCSI_ATNO);
   do {
      OSD_INEXACT8(SCSI_AICREG(SCSI_SCSIDATL));
      phase = SCSIhWait4Req(hhcb);
   } while (phase == SCSI_MIPHASE);
   if (phase != SCSI_MOPHASE)
   {
      SCSIhBadSeq(hhcb);
   }
   else
   {
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSISIG), SCSI_MOPHASE);
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_CLRSINT1), SCSI_CLRATNO);
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSIDATL), SCSI_MSG07);
      if (SCSIhWait4Req(hhcb) == (SCSI_UEXACT8)-1)
      {
         SCSIhBadSeq(hhcb);
         return;
      }
      /* set reentrant address for sequencer */
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SEQADDR0),
            (SCSI_UEXACT8)SCSI_hSIOSTR3_ENTRY(hhcb) >> 2);
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SEQADDR1),
            (SCSI_UEXACT8)SCSI_hSIOSTR3_ENTRY(hhcb) >> 10);
   }
}
#endif   /* SCSI_STANDARD_ADVANCED_MODE */

/*********************************************************************
*
*  SCSIhSwappingAdvModifyDataPtr routine -
*
*  Handle Modify Data Pointer message for Swapping 32 mode.
*
*  Return Value:  none
*                  
*  Parameters:    hhcb
*                 hiob
*
*  Remarks:                
*                  
*********************************************************************/
#if SCSI_SWAPPING_ADVANCED_MODE
void SCSIhSwappingAdvModifyDataPtr (SCSI_HHCB SCSI_HPTR hhcb,
                                    SCSI_HIOB SCSI_IPTR hiob)
{
   SCSI_UEXACT8 phase;
   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;

   /* Reject the modify data pointer message */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSISIG), SCSI_MIPHASE | SCSI_ATNO);
   do {
      OSD_INEXACT8(SCSI_AICREG(SCSI_SCSIDATL));
      phase = SCSIhWait4Req(hhcb);
   } while (phase == SCSI_MIPHASE);
   if (phase != SCSI_MOPHASE)
   {
      SCSIhBadSeq(hhcb);
   }
   else
   {
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSISIG), SCSI_MOPHASE);
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_CLRSINT1), SCSI_CLRATNO);
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSIDATL), SCSI_MSG07);
      if (SCSIhWait4Req(hhcb) == (SCSI_UEXACT8)-1)
      {
         SCSIhBadSeq(hhcb);
         return;
      }
      /* set reentrant address for sequencer */
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SEQADDR0),
            (SCSI_UEXACT8)SCSI_hSIOSTR3_ENTRY(hhcb) >> 2);
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SEQADDR1),
            (SCSI_UEXACT8)SCSI_hSIOSTR3_ENTRY(hhcb) >> 10);
   }
}
#endif   /* SCSI_SWAPPING_ADVANCED_MODE */

/*********************************************************************
*
*  SCSIhStandard160mModifyDataPtr routine -
*
*  Handle Modify Data Pointer message for Standard 160m mode.
*
*  Return Value:  none
*                  
*  Parameters:    hhcb
*                 hiob
*
*  Remarks:                
*                  
*********************************************************************/
#if SCSI_STANDARD_160M_MODE
void SCSIhStandard160mModifyDataPtr (SCSI_HHCB SCSI_HPTR hhcb,
                                     SCSI_HIOB SCSI_IPTR hiob)
{

   SCSI_UEXACT8 phase;
   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;

   /* Reject the modify data pointer message */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSISIG), SCSI_MIPHASE | SCSI_ATNO);
   do {
      OSD_INEXACT8(SCSI_AICREG(SCSI_SCSIDATL));
      phase = SCSIhWait4Req(hhcb);
   } while (phase == SCSI_MIPHASE);
   if (phase != SCSI_MOPHASE)
   {
      SCSIhBadSeq(hhcb);
   }
   else
   {
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSISIG), SCSI_MOPHASE);
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_CLRSINT1), SCSI_CLRATNO);
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSIDATL), SCSI_MSG07);
      if (SCSIhWait4Req(hhcb) == (SCSI_UEXACT8)-1)
      {
         SCSIhBadSeq(hhcb);
         return;
      }
      /* set reentrant address for sequencer */
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SEQADDR0),
            (SCSI_UEXACT8)(SCSI_S160M_SIOSTR3_ENTRY >> 2));
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SEQADDR1),
            (SCSI_UEXACT8)(SCSI_S160M_SIOSTR3_ENTRY >> 10));
   }
}

#endif   /* SCSI_STANDARD_160M_MODE */

/*********************************************************************
*
*  SCSIhSwapping160mModifyDataPtr routine -
*
*  Handle Modify Data Pointer message for Swapping 160m mode.
*
*  Return Value:  none
*                  
*  Parameters:    hhcb
*                 hiob
*
*  Remarks:                
*                  
*********************************************************************/
#if SCSI_SWAPPING_160M_MODE
void SCSIhSwapping160mModifyDataPtr (SCSI_HHCB SCSI_HPTR hhcb,
                                     SCSI_HIOB SCSI_IPTR hiob)
{
   SCSI_UEXACT8 phase;
   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;

   /* Reject the modify data pointer message */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSISIG), SCSI_MIPHASE | SCSI_ATNO);
   do {
      OSD_INEXACT8(SCSI_AICREG(SCSI_SCSIDATL));
      phase = SCSIhWait4Req(hhcb);
   } while (phase == SCSI_MIPHASE);
   if (phase != SCSI_MOPHASE)
   {
      SCSIhBadSeq(hhcb);
   }
   else
   {
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSISIG), SCSI_MOPHASE);
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_CLRSINT1), SCSI_CLRATNO);
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSIDATL), SCSI_MSG07);
      if (SCSIhWait4Req(hhcb) == (SCSI_UEXACT8)-1)
      {
         SCSIhBadSeq(hhcb);
         return;
      }
      /* set reentrant address for sequencer */
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SEQADDR0),
            (SCSI_UEXACT8)(SCSI_W160M_SIOSTR3_ENTRY >> 2));
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SEQADDR1),
            (SCSI_UEXACT8)(SCSI_W160M_SIOSTR3_ENTRY >> 10));
   }
}
#endif   /* SCSI_SWAPPING_160M_MODE */

/*********************************************************************
*
*  SCSIhAIC78XXGetScsiWideOffset routine -
*
*  Get wide scsi offset
*
*  Return Value:  wide scsi offset
*                  
*  Parameters:    hhcb
*                 hiob
*
*  Remarks:                
*                  
*********************************************************************/
#if SCSI_AIC78XX
SCSI_UEXACT8 SCSIhAIC78XXGetWideScsiOffset (SCSI_HIOB SCSI_IPTR hiob)
{
   return((SCSI_UEXACT8) SCSI_WIDE_OFFSET);
}
#endif /* SCSI_AIC78XX */

/*********************************************************************
*
*  SCSIhAIC78XXGetScsiNarrowOffset routine -
*
*  Get wide scsi offset
*
*  Return Value:  wide scsi offset
*                  
*  Parameters:    hhcb
*                 hiob
*
*  Remarks:                
*                  
*********************************************************************/
#if SCSI_AIC78XX
SCSI_UEXACT8 SCSIhAIC78XXGetNarrowScsiOffset (SCSI_HIOB SCSI_IPTR hiob)
{
   return((SCSI_UEXACT8) SCSI_NARROW_OFFSET);
}
#endif /* SCSI_AIC78XX */

/*********************************************************************
*
*  SCSIhAICBayonetGetScsiOffset routine -
*
*  Get wide scsi offset
*
*  Return Value:  wide scsi offset
*                  
*  Parameters:    hiob
*
*  Remarks:                
*                  
*********************************************************************/
#if (SCSI_AICBAYONET || SCSI_AICTRIDENT)
SCSI_UEXACT8 SCSIhAICBayonetGetScsiOffset (SCSI_HIOB SCSI_IPTR hiob)
{
   SCSI_DEVICE SCSI_DPTR deviceTable;

   SCSI_hGETDEVICETABLEPTR(hiob,deviceTable);

   return((SCSI_UEXACT8)(deviceTable->bayScsiOffset));
}
#endif /* (SCSI_AICBAYONET || SCSI_AICTRIDENT) */

/*********************************************************************
*
*  SCSIhAIC78XXClearScsiOffset routine -
*
*  Clear scsi offset
*
*  Return Value:  None
*                  
*  Parameters:    hiob
*
*  Remarks:                
*                  
*********************************************************************/
#if SCSI_AIC78XX
void SCSIhAIC78XXClearScsiOffset (SCSI_HIOB SCSI_IPTR hiob)
{
   SCSI_DEVICE SCSI_DPTR deviceTable = SCSI_TARGET_UNIT(hiob)->deviceTable;

   deviceTable->scsiOption &= ~SCSI_SOFS;
}
#endif /* SCSI_AIC78XX */

/*********************************************************************
*
*  SCSIhAICBayonetClearScsiOffset routine -
*
*  Clear scsi offset
*
*  Return Value:  None
*                  
*  Parameters:    hiob
*
*  Remarks:                
*                  
*********************************************************************/
#if (SCSI_AICBAYONET || SCSI_AICTRIDENT)
void SCSIhAICBayonetClearScsiOffset (SCSI_HIOB SCSI_IPTR hiob)
{
   SCSI_DEVICE SCSI_DPTR deviceTable;

   SCSI_hGETDEVICETABLEPTR(hiob,deviceTable);

   /* Must clear scsiOption so that we're not going to initiate negotiation */
   deviceTable->scsiOption &= ~SCSI_SOFS;

   /* Clear actual scsi offset for Bayonet */
   deviceTable->bayScsiOffset &= ~SCSI_BAYONET_OFFSET;
}
#endif /* (SCSI_AICBAYONET || SCSI_AICTRIDENT) */

/*********************************************************************
*
*  SCSIhAIC78XXWasSyncXfer routine -
*
*  Check if current scsi offset is sync. offset.
*
*  Return Value:  0 - Running Asynchronous xfer
*                 1 - Running Synchronous xfer
*                  
*  Parameters:    offset
*
*  Remarks:                
*                  
*********************************************************************/
#if SCSI_AIC78XX
SCSI_UEXACT8 SCSIhAIC78XXWasSyncXfer (SCSI_UEXACT8 offset)
{
   return((offset & SCSI_SOFS) != 0);
}
#endif /* SCSI_AIC78XX */

/*********************************************************************
*
*  SCSIhAICBayonetWasSyncXfer routine -
*
*  Check if current scsi offset is sync. offset.
*
*  Return Value:  0 - Running Asynchronous xfer
*                 1 - Running Synchronous xfer
*                  
*  Parameters:    offset
*
*  Remarks:                
*                  
*********************************************************************/
#if (SCSI_AICBAYONET || SCSI_AICTRIDENT)
SCSI_UEXACT8 SCSIhAICBayonetWasSyncXfer (SCSI_UEXACT8 offset)
{
   return((offset & SCSI_BAYONET_OFFSET) != 0);
}
#endif /* (SCSI_AICBAYONET || SCSI_AICTRIDENT) */

/***************************************************************************
*
*  SCSIhAIC78XXSetDataFifoThrshDefault
*
*     Set Data Fifo threshold default.
*
*  Return Value:  threshold
*                  
*  Parameters:    hhcb
*
*  Remarks:       For non-Bayonet base HA.
*
***************************************************************************/
#if SCSI_AIC78XX
SCSI_UEXACT8 SCSIhAIC78XXSetDataFifoThrshDefault (SCSI_HHCB SCSI_HPTR hhcb)
{
   SCSI_UEXACT8 threshold;
   
   threshold = ((SCSI_DFTHRSH0 + SCSI_DFTHRSH1) >> 6);
   
   return(threshold);
}
#endif   /* SCSI_AIC78XX */

/***************************************************************************
*
*  SCSIhAIC78XXReadDataFifoThrsh
*
*     Read Data Fifo threshold. 
*
*  Return Value: threshold 
*                  
*  Parameters:    hhcb
*
*  Remarks:       For non-Bayonet base HA.
*
***************************************************************************/
#if SCSI_AIC78XX
SCSI_UEXACT8 SCSIhAIC78XXReadDataFifoThrsh (SCSI_HHCB SCSI_HPTR hhcb)
{
   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_UEXACT8 threshold;
   
   threshold = (OSD_INEXACT8(SCSI_AICREG(SCSI_PCISTATUS)) >> 6);
   
   return(threshold);
}
#endif   /* SCSI_AIC78XX */

/***************************************************************************
*
*  SCSIhAIC78XXUpdateDataFifoThrsh
*
*     Update Data Fifo threshold. 
*
*  Return Value: none 
*                  
*  Parameters:    hhcb
*
*  Remarks:       For non-Bayonet base HA.
*
***************************************************************************/
#if SCSI_AIC78XX
void SCSIhAIC78XXUpdateDataFifoThrsh (SCSI_HHCB SCSI_HPTR hhcb)
{
   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_PCISTATUS), hhcb->threshold << 6);
   
}
#endif   /* SCSI_AIC78XX */

/***************************************************************************
*
*  SCSIhAICBayonetSetDataFifoThrshDefault
*
*     Set Data Fifo threshold default.
*
*  Return Value:  threshold
*                  
*  Parameters:    hhcb
*
*  Remarks:       For Bayonet base HA.
*
***************************************************************************/
#if (SCSI_AICBAYONET || SCSI_AICTRIDENT)
SCSI_UEXACT8 SCSIhAICBayonetSetDataFifoThrshDefault (SCSI_HHCB SCSI_HPTR hhcb)
{
   SCSI_UEXACT8 threshold;
   
   threshold = (SCSI_BAYONET_WR_DFTHRSH + SCSI_BAYONET_RD_DFTHRSH);
   
   return(threshold);
}
#endif   /* (SCSI_AICBAYONET || SCSI_AICTRIDENT) */

/***************************************************************************
*
*  SCSIhAICBayonetReadDataFifoThrsh
*
*     Read Data Fifo threshold. 
*
*  Return Value: threshold 
*                  
*  Parameters:    hhcb
*
*  Remarks:       For Bayonet base HA.
*
***************************************************************************/
#if (SCSI_AICBAYONET || SCSI_AICTRIDENT)
SCSI_UEXACT8 SCSIhAICBayonetReadDataFifoThrsh (SCSI_HHCB SCSI_HPTR hhcb)
{
   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_UEXACT8 threshold;
   
   threshold = OSD_INEXACT8(SCSI_AICREG(SCSI_BAYONET_FIFO_THRSH));
   
   return(threshold);
}
#endif   /* (SCSI_AICBAYONET || SCSI_AICTRIDENT) */

/***************************************************************************
*
*  SCSIhAICBayonetUpdateDataFifoThrsh
*
*     Update Data Fifo threshold. 
*
*  Return Value: none 
*                  
*  Parameters:    hhcb
*
*  Remarks:       For Bayonet base HA.
*
***************************************************************************/
#if (SCSI_AICBAYONET || SCSI_AICTRIDENT)
void SCSIhAICBayonetUpdateDataFifoThrsh (SCSI_HHCB SCSI_HPTR hhcb)
{
   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_BAYONET_FIFO_THRSH), hhcb->threshold);
   
}
#endif /* (SCSI_AICBAYONET || SCSI_AICTRIDENT) */

/***************************************************************************
*
*  SCSIhSwappingClearAllFast20Reg
*
*     Clear all fast 20 registers.
*
*  Return Value:  none
*                  
*  Parameters:    hhcb
*
*  Remarks:       For AIC78XX base HA.
*
***************************************************************************/
#if SCSI_SWAPPING32_MODE || SCSI_SWAPPING64_MODE
void SCSIhSwappingClearAllFast20Reg (SCSI_HHCB SCSI_HPTR hhcb)
{
   SCSI_UEXACT16  i;
      
      
   for (i = 0; i < SCSI_MAXDEV; i++)
   {
      SCSI_DEVICE_TABLE(hhcb)[i].SCSI_DF_fast20EnableW = 0; 
   }
}     
#endif   /* SCSI_AIC78XX */

/***************************************************************************
*
*  SCSIhStandard64ClearAllFast20Reg
*
*     Clear all fast 20 registers.
*
*  Return Value:  none
*                  
*  Parameters:    hhcb
*
*  Remarks:       For AIC78XX base HA.
*
***************************************************************************/
#if SCSI_STANDARD64_MODE 
void SCSIhStandard64ClearAllFast20Reg (SCSI_HHCB SCSI_HPTR hhcb)
{
   SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_hFAST20_LOW(hhcb)),0);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_hFAST20_HIGH(hhcb)),0);
   
}     
#endif   /* SCSI_STANDARD64_MODE */

/***************************************************************************
*
*  SCSIhAIC78XXIgnoreWideResCalc
*
*     Ignore Wide Residue calculation for AIC78XX based HA.
*
*  Return Value:  None
*                  
*  Parameters:    hhcb
*                 hiob
*                 number of byte to ignore
*
*  Remarks:
*
***************************************************************************/
#if SCSI_AIC78XX
void SCSIhAIC78XXIgnoreWideResCalc(SCSI_HHCB SCSI_HPTR hhcb,
                                   SCSI_HIOB SCSI_IPTR hiob,
                                   SCSI_UEXACT8 ignoreXfer)
{
   SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_QUADLET resCnt;
   SCSI_QUADLET regValue;

   /* To handle the IGNORE WIDE RESIDUE message:
      1. Increment STCNT0-3 to back up due to the extra bad byte(s).
      2. Increment rescnt field of SCB by the reduced number.
      3. Read SHADDR0-3 registers, decrement by the reduced number,
         and write to HADDR0-3 which will shine thru to SHADDR0-3.
   */
   regValue.u8.order0 = OSD_INEXACT8(SCSI_AICREG(SCSI_STCNT0));
   regValue.u8.order1 = OSD_INEXACT8(SCSI_AICREG(SCSI_STCNT1));
   regValue.u8.order2 = OSD_INEXACT8(SCSI_AICREG(SCSI_STCNT2));
   regValue.u8.order3 = 0;
   *((SCSI_UEXACT32 *) &regValue) += (SCSI_UEXACT32)ignoreXfer;      /* restore the ACK */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_STCNT0), regValue.u8.order0);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_STCNT1), regValue.u8.order1);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_STCNT2), regValue.u8.order2);

   resCnt.u8.order0 = OSD_INEXACT8(SCSI_AICREG(SCSI_hRESCNT_BASE(hhcb)+0));
   resCnt.u8.order1 = OSD_INEXACT8(SCSI_AICREG(SCSI_hRESCNT_BASE(hhcb)+1));
   resCnt.u8.order2 = OSD_INEXACT8(SCSI_AICREG(SCSI_hRESCNT_BASE(hhcb)+2));
   resCnt.u8.order3 = OSD_INEXACT8(SCSI_AICREG(SCSI_hRESCNT_BASE(hhcb)+3));
   hiob->residualLength += (SCSI_UEXACT32)ignoreXfer;
   hiob->residualLength = (hiob->residualLength & 0x0fff)
                           | (*((SCSI_UEXACT32 SCSI_SPTR) &resCnt) & 0xf000);

   regValue.u8.order0 = OSD_INEXACT8(SCSI_AICREG(SCSI_SHADDR0));
   regValue.u8.order1 = OSD_INEXACT8(SCSI_AICREG(SCSI_SHADDR1));
   regValue.u8.order2 = OSD_INEXACT8(SCSI_AICREG(SCSI_SHADDR2));
   regValue.u8.order3 = OSD_INEXACT8(SCSI_AICREG(SCSI_SHADDR3));
   *((SCSI_UEXACT32 *) &regValue) -= (SCSI_UEXACT32)ignoreXfer;
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_HADDR0), regValue.u8.order0);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_HADDR1), regValue.u8.order1);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_HADDR2), regValue.u8.order2);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_HADDR3), regValue.u8.order3);
}
#endif /* SCSI_AIC78XX */

/***************************************************************************
*
*  SCSIhAICBayonetIgnoreWideResCalc
*
*     Ignore Wide Residue calculation for Bayonet based HA.
*
*  Return Value:  None
*                  
*  Parameters:    hhcb
*                 hiob
*                 number of byte to ignore
*
*  Remarks:
*
***************************************************************************/
#if (SCSI_AICBAYONET || SCSI_AICTRIDENT)
void SCSIhAICBayonetIgnoreWideResCalc(SCSI_HHCB SCSI_HPTR hhcb,
                                      SCSI_HIOB SCSI_IPTR hiob,
                                      SCSI_UEXACT8 ignoreXfer)
{
   SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_QUADLET resCnt;
   SCSI_QUADLET regValue;

   /* To handle the IGNORE WIDE RESIDUE message:
      1. Increment SDATA_RESIDUE0-3 to back up due to the extra bad byte(s)
         and write to embbeded SG_ELEMENT_SCB LENGTH0-2 which will be loaded
         into HCNT and STCNT when data xfer resume.
      2. Increment rescnt field of SCB by the reduced number.
      3. Read SHADDR0-3 registers, decrement by the reduced number,
         and write to embbeded SG_ELEMENT_SCB ADDRESS0-3 which will be loaded
         into HADDRESS0-3 and SHADDRESS0-3 when data xfer resume.
   */
   regValue.u8.order0 = OSD_INEXACT8(SCSI_AICREG(SCSI_hRESCNT_BASE(hhcb)+0));
   regValue.u8.order1 = OSD_INEXACT8(SCSI_AICREG(SCSI_hRESCNT_BASE(hhcb)+1));
   regValue.u8.order2 = OSD_INEXACT8(SCSI_AICREG(SCSI_hRESCNT_BASE(hhcb)+2));
   regValue.u8.order3 = 0;
   *((SCSI_UEXACT32 *) &regValue) += (SCSI_UEXACT32)ignoreXfer;      /* restore the ACK */
   SCSI_hUPDATESGLENGTH(hhcb,regValue);

   resCnt.u8.order0 = OSD_INEXACT8(SCSI_AICREG(SCSI_hRESCNT_BASE(hhcb)+0));
   resCnt.u8.order1 = OSD_INEXACT8(SCSI_AICREG(SCSI_hRESCNT_BASE(hhcb)+1));
   resCnt.u8.order2 = OSD_INEXACT8(SCSI_AICREG(SCSI_hRESCNT_BASE(hhcb)+2));
   resCnt.u8.order3 = OSD_INEXACT8(SCSI_AICREG(SCSI_hRESCNT_BASE(hhcb)+3));
   hiob->residualLength += (SCSI_UEXACT32)ignoreXfer;
   hiob->residualLength = (hiob->residualLength & 0x0fff)
                           | (*((SCSI_UEXACT32 SCSI_SPTR) &resCnt) & 0xf000);

   regValue.u8.order0 = OSD_INEXACT8(SCSI_AICREG(SCSI_SHADDR0));
   regValue.u8.order1 = OSD_INEXACT8(SCSI_AICREG(SCSI_SHADDR1));
   regValue.u8.order2 = OSD_INEXACT8(SCSI_AICREG(SCSI_SHADDR2));
   regValue.u8.order3 = OSD_INEXACT8(SCSI_AICREG(SCSI_SHADDR3));
   *((SCSI_UEXACT32 *) &regValue) -= (SCSI_UEXACT32)ignoreXfer;
   SCSI_hUPDATESGADDRESS(hhcb,regValue);
}
#endif /* (SCSI_AICBAYONET || SCSI_AICTRIDENT) */

/***************************************************************************
*
*  SCSIhStandardAdvUpdateSGLength
*
*     Update SCB slength0 - slength2.
*
*  Return Value:  None
*                  
*  Parameters:    hhcb
*                 regValue
*
*  Remarks:
*
***************************************************************************/
#if SCSI_STANDARD_ADVANCED_MODE
void SCSIhStandardAdvUpdateSGLength (SCSI_HHCB SCSI_HPTR hhcb,
                                     SCSI_QUADLET regValue)
{
   SCSI_REGISTER scsiRegister = hhcb->scsiRegister;

   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
       OSDoffsetof(SCSI_SCB_STANDARD_ADVANCED,slength0)), regValue.u8.order0);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
       OSDoffsetof(SCSI_SCB_STANDARD_ADVANCED,slength1)), regValue.u8.order1);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
       OSDoffsetof(SCSI_SCB_STANDARD_ADVANCED,slength2)), regValue.u8.order2);
}
#endif /* SCSI_STANDARD_ADVANCED_MODE */

/***************************************************************************
*
*  SCSIhSwappingAdvUpdateSGLength
*
*     Update SCB slength0 - slength2.
*
*  Return Value:  None
*                  
*  Parameters:    hhcb
*                 regValue
*
*  Remarks:
*
***************************************************************************/
#if SCSI_SWAPPING_ADVANCED_MODE
void SCSIhSwappingAdvUpdateSGLength (SCSI_HHCB SCSI_HPTR hhcb,
                                     SCSI_QUADLET regValue)
{
   SCSI_REGISTER scsiRegister = hhcb->scsiRegister;

   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
       OSDoffsetof(SCSI_SCB_SWAPPING_ADVANCED,slength0)), regValue.u8.order0);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
       OSDoffsetof(SCSI_SCB_SWAPPING_ADVANCED,slength1)), regValue.u8.order1);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
       OSDoffsetof(SCSI_SCB_SWAPPING_ADVANCED,slength2)), regValue.u8.order2);
}
#endif /* SCSI_SWAPPING_ADVANCED_MODE */

/***************************************************************************
*
*  SCSIhStandard160mUpdateSGLength
*
*     Update SCB slength0 - slength2.
*
*  Return Value:  None
*                  
*  Parameters:    hhcb
*                 regValue
*
*  Remarks:
*
***************************************************************************/
#if SCSI_STANDARD_160M_MODE
void SCSIhStandard160mUpdateSGLength (SCSI_HHCB SCSI_HPTR hhcb,
                                      SCSI_QUADLET regValue)
{
   SCSI_REGISTER scsiRegister = hhcb->scsiRegister;

   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
       OSDoffsetof(SCSI_SCB_STANDARD_160M,slength0)), regValue.u8.order0);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
       OSDoffsetof(SCSI_SCB_STANDARD_160M,slength1)), regValue.u8.order1);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
       OSDoffsetof(SCSI_SCB_STANDARD_160M,slength2)), regValue.u8.order2);
}
#endif /* SCSI_STANDARD_160M_MODE */

/***************************************************************************
*
*  SCSIhSwapping160mUpdateSGLength
*
*     Update SCB slength0 - slength2.
*
*  Return Value:  None
*                  
*  Parameters:    hhcb
*                 regValue
*
*  Remarks:
*
***************************************************************************/
#if SCSI_SWAPPING_160M_MODE
void SCSIhSwapping160mUpdateSGLength (SCSI_HHCB SCSI_HPTR hhcb,
                                      SCSI_QUADLET regValue)
{
   SCSI_REGISTER scsiRegister = hhcb->scsiRegister;

   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
       OSDoffsetof(SCSI_SCB_SWAPPING_160M,slength0)), regValue.u8.order0);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
       OSDoffsetof(SCSI_SCB_SWAPPING_160M,slength1)), regValue.u8.order1);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
       OSDoffsetof(SCSI_SCB_SWAPPING_160M,slength2)), regValue.u8.order2);
}
#endif /* SCSI_SWAPPING_160M_MODE */

/***************************************************************************
*
*  SCSIhStandardAdvUpdateSGAddress
*
*     Update SCB saddress0 - slength3.
*
*  Return Value:  None
*                  
*  Parameters:    hhcb
*                 regValue
*
*  Remarks:
*
***************************************************************************/
#if SCSI_STANDARD_ADVANCED_MODE
void SCSIhStandardAdvUpdateSGAddress (SCSI_HHCB SCSI_HPTR hhcb,
                                      SCSI_QUADLET regValue)
{
   SCSI_REGISTER scsiRegister = hhcb->scsiRegister;

   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
       OSDoffsetof(SCSI_SCB_STANDARD_ADVANCED,saddress0)), regValue.u8.order0);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
       OSDoffsetof(SCSI_SCB_STANDARD_ADVANCED,saddress1)), regValue.u8.order1);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
       OSDoffsetof(SCSI_SCB_STANDARD_ADVANCED,saddress2)), regValue.u8.order2);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
       OSDoffsetof(SCSI_SCB_STANDARD_ADVANCED,saddress3)), regValue.u8.order3);
}
#endif /* SCSI_STANDARD_ADVANCED_MODE */

/***************************************************************************
*
*  SCSIhSwappingAdvUpdateSGAddress
*
*     Update SCB saddress0 - saddress3.
*
*  Return Value:  None
*                  
*  Parameters:    hhcb
*                 regValue
*
*  Remarks:
*
***************************************************************************/
#if SCSI_SWAPPING_ADVANCED_MODE
void SCSIhSwappingAdvUpdateSGAddress (SCSI_HHCB SCSI_HPTR hhcb,
                                      SCSI_QUADLET regValue)
{
   SCSI_REGISTER scsiRegister = hhcb->scsiRegister;

   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
       OSDoffsetof(SCSI_SCB_SWAPPING_ADVANCED,saddress0)), regValue.u8.order0);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
       OSDoffsetof(SCSI_SCB_SWAPPING_ADVANCED,saddress1)), regValue.u8.order1);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
       OSDoffsetof(SCSI_SCB_SWAPPING_ADVANCED,saddress2)), regValue.u8.order2);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
       OSDoffsetof(SCSI_SCB_SWAPPING_ADVANCED,saddress3)), regValue.u8.order3);
}
#endif /* SCSI_SWAPPING_ADVANCED_MODE */

/***************************************************************************
*
*  SCSIhStandard160mUpdateSGAddress
*
*     Update SCB saddress0 - slength3.
*
*  Return Value:  None
*                  
*  Parameters:    hhcb
*                 regValue
*
*  Remarks:
*
***************************************************************************/
#if SCSI_STANDARD_160M_MODE
void SCSIhStandard160mUpdateSGAddress (SCSI_HHCB SCSI_HPTR hhcb,
                                       SCSI_QUADLET regValue)
{
   SCSI_REGISTER scsiRegister = hhcb->scsiRegister;

   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
       OSDoffsetof(SCSI_SCB_STANDARD_160M,saddress0)), regValue.u8.order0);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
       OSDoffsetof(SCSI_SCB_STANDARD_160M,saddress1)), regValue.u8.order1);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
       OSDoffsetof(SCSI_SCB_STANDARD_160M,saddress2)), regValue.u8.order2);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
       OSDoffsetof(SCSI_SCB_STANDARD_160M,saddress3)), regValue.u8.order3);
}
#endif /* SCSI_STANDARD_160M_MODE */

/***************************************************************************
*
*  SCSIhSwapping160mUpdateSGAddress
*
*     Update SCB saddress0 - saddress3.
*
*  Return Value:  None
*                  
*  Parameters:    hhcb
*                 regValue
*
*  Remarks:
*
***************************************************************************/
#if SCSI_SWAPPING_160M_MODE
void SCSIhSwapping160mUpdateSGAddress (SCSI_HHCB SCSI_HPTR hhcb,
                                       SCSI_QUADLET regValue)
{
   SCSI_REGISTER scsiRegister = hhcb->scsiRegister;

   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
       OSDoffsetof(SCSI_SCB_SWAPPING_160M,saddress0)), regValue.u8.order0);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
       OSDoffsetof(SCSI_SCB_SWAPPING_160M,saddress1)), regValue.u8.order1);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
       OSDoffsetof(SCSI_SCB_SWAPPING_160M,saddress2)), regValue.u8.order2);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
       OSDoffsetof(SCSI_SCB_SWAPPING_160M,saddress3)), regValue.u8.order3);
}
#endif /* SCSI_SWAPPING_160M_MODE */

/*********************************************************************
*
*  SCSIhSetXferRate routine -
*
*  Set the transfer rates to the default value.
*
*  Return Value:  none
*                  
*  Parameters:    hhcb
*                 scsiID
*
*  Remarks: To be implemented!!!               
*                  
*********************************************************************/
#if SCSI_LOOPBACK_OPERATION
void SCSIhSetXferRate (SCSI_HHCB SCSI_HPTR hhcb,
                       SCSI_UEXACT8 scsiID)
{
   SCSI_DEVICE SCSI_DPTR deviceTable = &SCSI_DEVICE_TABLE(hhcb)[scsiID];
   SCSI_UEXACT8 period;
   SCSI_UEXACT16 scsiOption = deviceTable->scsiOption;
   SCSI_UEXACT8 offset;
   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
      
   if (OSD_INEXACT8(SCSI_AICREG(SCSI_SBLKCTL)) & SCSI_SELWIDE)
   {
      offset = SCSI_WIDE_OFFSET;
      deviceTable->scsiOption |= SCSI_WIDEXFER; 
   }
   else
   {
      offset = SCSI_NARROW_OFFSET;
      deviceTable->scsiOption &= ~SCSI_WIDEXFER;
   }
   
   if (scsiOption & SCSI_SYNC_MODE)
   {
      /* reset bit 0 */
      deviceTable->scsiOption &= ~SCSI_SYNC_MODE;
   } 
      
   /* For fast20 support per device */
   if (deviceTable->SCSI_DF_ultraEnable)
   {
      /* ScsiOption register bit 4-6:                  */
      /* 000 - 20MB/s      period:  50ns     12(12.5)  */
      /* 001 - 16MB/s               62.5ns   16(15.6)  */
      /* 010 - 13.4MB/s             75ns     19(18.75) */
      /* 011 - (NOT SUPPORTED), default to 13.4MB/s    */
      /* 100 - 10MB/s               100ns    25(25)    */
      period = (scsiOption & SCSI_SYNC_RATE) >> (4 - 2);
      period += (SCSI_UEXACT8) 12;
   }
   else
   {
      /* ScsiOption register bit 4-6:                  */
      /* 000 - 10MB/s      period:  100ns    25(25)    */
      /* 001 - 8.0MB/s              125ns    31(31.25) */
      /* 010 - 6.7MB/s              150ns    37(37.5)  */
      /* 011 - 5.7MB/s              175ns    43(43.75) */
      /* 100 - 5.0MB/s              200ns    50(50)    */
      /* 101 - 4.4MB/s              225ns    56(56.25) */
      /* 110 - 4.0MB/s              250ns    62(62.5)  */
      /* 111 - 3.6MB/s              275ns    68(68.75) */
      period = ((scsiOption & SCSI_SYNC_RATE) >> 4) * 6;
      period += (SCSI_UEXACT8) 25;

      if (scsiOption & SCSI_SXFR2)
         ++period;
   }

   /* If currently set to run asynchronous xfer */
   if (!deviceTable->SCSI_DF_setForSync)
   {
      offset = 0x00;                         /* Asynchronous mode */
      deviceTable->scsiOption &= ~SCSI_SOFS; /* Clear offset in scsiOption */
   }
   else
   {
      deviceTable->scsiOption |= offset;
   }

   /* assign xferOption */
   SCSI_hXFEROPTASSIGN(hhcb,scsiID,deviceTable->scsiOption);

   /* record in Scratch RAM */
   SCSI_hLOGFAST20MAP(hhcb,scsiID,period);

}
#endif /* SCSI_LOOPBACK_OPERATION */
/*********************************************************************
*
*  SCSIhNsxRepeaterCommunication
*
*     Perform special NSX repeater Communication
*
*  Return Value:  none
*                  
*  Parameters:    hhcb
*                 hiob
*
*  Remarks:
*
*********************************************************************/
#if SCSI_PAC_NSX_REPEATER || SCSI_NSX_REPEATER
void SCSIhNsxRepeaterCommunication (SCSI_HHCB SCSI_HPTR hhcb,
                                    SCSI_HIOB SCSI_IPTR hiob)
{
   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_BUFFER_DESCRIPTOR SCSI_LPTR nsxBuffer = SCSI_GET_NSX_BUFFER(hiob);
   SCSI_UEXACT8 i, temp;
   SCSI_UINT32 bytesRead = 0;
   SCSI_UEXACT32 bufferSize = nsxBuffer->bufferSize;
   SCSI_UEXACT32 bufferIndex = 0;
   SCSI_UEXACT8 SCSI_SPTR nsxBufferPtr;
   SCSI_UINT8 assertATN;  /* used for ATN assert/de-assert request */
   SCSI_UEXACT8 status = 0; /* used for status */
   SCSI_UINT16 count = 0;
   SCSI_UINT32 j;
   SCSI_UINT32 timeOutValue = 400000; /* 5 microsecs * 400000 = 2 secs */ 
 
   /* ack the message manually */
   /* ensure spioen is disabled */
   i = OSD_INEXACT8(SCSI_AICREG(SCSI_SXFRCTL0));
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SXFRCTL0), (SCSI_UEXACT8)(i & ~SCSI_SPIOEN));
            
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSISIG),
                 OSD_INEXACT8(SCSI_AICREG(SCSI_SCSISIG)) | SCSI_ACKO); 
                  
   /* wait until REQ deasserted */
   j = 0;
   while ((OSD_INEXACT8(SCSI_AICREG(SCSI_SCSISIG)) & SCSI_REQO) &&
          (j++ < timeOutValue))
   {
      OSD_TIMER(hhcb);  
   }
 
   if (j >= timeOutValue)
   {
      /* timed out waiting on REQ deassert */
      status = SCSI_NSX_PROTOCOL_ERROR;
      SCSI_RETURN_NSX_STATUS(hiob,status,count);
      SCSIhBadSeq(hhcb);
      return;
   }

   /* we can go ahead with the protocol */
   /* Must delay a minimum of bus settle delay before starting NSX */
   /* communication protocol. */            
   OSD_TIMER(hhcb);

   /* Then 4 ATN signal transitions to obtain data */
   assertATN = 1;  
      
   for (i = 0; i < 4; i++)
   {  
      /* Obtain data on fourth signal transition */
      if (assertATN)
      {
         /* Assert ATN */
         OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSISIG),
                       OSD_INEXACT8(SCSI_AICREG(SCSI_SCSISIG)) | SCSI_ATNO);
         assertATN = 0;
      }
      else
      {
         /* de-assert ATN */
         OSD_OUTEXACT8(SCSI_AICREG(SCSI_CLRSINT1), SCSI_CLRATNO);
         assertATN = 1;
      }
 
      /* Delay */
      OSD_TIMER(hhcb);
   }

   /* Initialize buffer pointer */
   nsxBufferPtr = (SCSI_UEXACT8 SCSI_SPTR)nsxBuffer->virtualAddress;
      
   /* Now let's see if we have data we expect 55h AAh */
   while (1)
   {  
      temp = OSD_INEXACT8(SCSI_AICREG(SCSI_SCSIBUSL));
      bytesRead++;

      if (temp == 0x55)
      {
         if (bufferIndex < bufferSize)
         {
            nsxBufferPtr[bufferIndex] = temp;
            bufferIndex++;           
         }

         /* Assert ATN */
         OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSISIG),
                       OSD_INEXACT8(SCSI_AICREG(SCSI_SCSISIG)) | SCSI_ATNO);
         assertATN = 0;
       
         /* delay */
         OSD_TIMER(hhcb);
         temp = OSD_INEXACT8(SCSI_AICREG(SCSI_SCSIBUSL));
         bytesRead++;

         if (temp == 0xAA)
         {
            /* we've definately got valid data packet */
            if (bufferIndex < bufferSize)
            {
               nsxBufferPtr[bufferIndex] = temp;
               bufferIndex++;
            }
        
            /* collect the data - six bytes */
            for (i = 0; i < 6 ; i++)
            {
               if (assertATN)
               {
                  /* Assert ATN */
                  OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSISIG),
                                OSD_INEXACT8(SCSI_AICREG(SCSI_SCSISIG)) | SCSI_ATNO);
                  assertATN = 0;
               }
               else
               { 
                  /* de assert ATN */
                  OSD_OUTEXACT8(SCSI_AICREG(SCSI_CLRSINT1), SCSI_CLRATNO);
                  assertATN = 1;
               }
             
               /* delay */
               OSD_TIMER(hhcb);           

               if (bufferIndex < bufferSize)
               {
                  nsxBufferPtr[bufferIndex] = OSD_INEXACT8(SCSI_AICREG(SCSI_SCSIBUSL));
                  bufferIndex++;
               }
               else
               { 
                  /* trash data */
                  temp = OSD_INEXACT8(SCSI_AICREG(SCSI_SCSIBUSL));
               }
               bytesRead++;
         
            } /* end for loop */
 
            /* Now final de-assert which takes us to next repeater maybe? */
            OSD_OUTEXACT8(SCSI_AICREG(SCSI_CLRSINT1), SCSI_CLRATNO);
            assertATN = 1;

            /* Delay */
            OSD_TIMER(hhcb);
      
         } /* end = 0xAA */
         else
         {
            bytesRead--; /* adjust as invalid */
            break;
         }  
      } /* end = 0x55 */
      else
      {
         bytesRead--; /* adjust as invalid */ 
         break;
      }
 
   }  /* end while(1) */
        

   /* make sure ATN is de-asserted */
   if (!assertATN)
   {
      /* de assert ATN */
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_CLRSINT1), SCSI_CLRATNO);
   }

   if (bytesRead > bufferSize) 
   {
      /* overrun */
      status = SCSI_NSX_OVERRUN;
   }
   else
   {
      status = SCSI_NSX_GOOD; /* good */ 
   }

   count = (SCSI_UINT16)bufferIndex;

   SCSI_RETURN_NSX_STATUS(hiob,status,count);
}
#endif /* SCSI_PAC_NSX_REPEATER || SCSI_NSX_REPEATER */

/***************************************************************************
*                                                                          *
* Copyright 1995,1996,1997,1998,1999 Adaptec, Inc.,  All Rights Reserved.  *
*                                                                          *
* This software contains the valuable trade secrets of Adaptec.  The       *
* software is protected under copyright laws as an unpublished work of     *
* Adaptec.  Notice is for informational purposes only and does not imply   *
* publication.  The user of this software may make copies of the software  *
* for use with parts manufactured by Adaptec or under license from Adaptec *
* and for no other use.                                                    *
*                                                                          *
***************************************************************************/

/***************************************************************************
*
*  Module Name:   HWMINTR.C
*
*  Description:
*                 Codes to implement interrupt dispatcher for hardware 
*                 management module
*
*  Owners:        ECX IC Firmware Team
*    
*  Notes:         NONE
*
*  Entry Point(s):
*                 SCSIHFrontEndISR
*                 SCSIHBackEndISR
*
***************************************************************************/

#include "scsi.h"
#include "hwm.h"

/*********************************************************************
*
*  SCSIHFrontEndISR
*
*     Front end processing of ISR
*
*  Return Value:  interrupt cleared (or was pending)
*                  
*  Parameters:    hhcb
*
*  Remarks:       This routine will just save the interrupt status and
*                 then clear the interrupt was pending. It's OSM's
*                 responsibility to schedule the execution of SCSIHBackEndISR
*                 at proper time.
*
*********************************************************************/
#if !SCSI_STREAMLINE_QIOPATH
int SCSIHFrontEndISR (SCSI_HHCB SCSI_HPTR hhcb)
{
   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_UEXACT8 intstat;
   SCSI_UEXACT8 qoutcnt;
#if SCSI_CHECK_PCI_ERROR
   SCSI_UEXACT8 error;
#endif /* #if SCSI_CHECK_PCI_ERROR */
   int i;

   /* check if there is any command complete interrupt */
   for (i = 0;; i++)
   {
      qoutcnt = SCSI_hQOUTCNT(hhcb);
      if (hhcb->SCSI_HP_qoutcnt != qoutcnt)
      {
         /* remember the interrupt status (command complete) */
         hhcb->SCSI_HP_qoutcnt = qoutcnt;
         hhcb->SCSI_HP_intstat |= (intstat = SCSI_CMDCMPLT);

         /* clear the interrupt and assume this is the only */
         /* interrupt we have so far */
         OSD_OUTEXACT8(SCSI_AICREG(SCSI_CLRINT),SCSI_CLRCMDINT);
         break;
      }
      else
      {
         /* just read interrupt status from hardware */
         intstat = SCSI_hREAD_INTSTAT(hhcb,scsiRegister);

         /* For Cardbus (APA-1480x), the instat register value is 0xFF */
         /* when the card is hot removed from the bus.  For this case, */
         /* CHIM returns with no interrupt pending.  This value is not */
         /* going to happen on the scsi host adapter. */
         if (intstat == 0xFF)
         {
            hhcb->SCSI_HP_intstat = intstat = 0;
            break;
         }

         if (intstat & SCSI_CMDCMPLT)
         {
            if (i)
            {
               /* clear the processed but not cleared */
               /* command complete interrupt */
               OSD_OUTEXACT8(SCSI_AICREG(SCSI_CLRINT),SCSI_CLRCMDINT);
            }
            else
            {
               /* go back to check if it's processed but not cleared */
               /* command complete interrupt */
               continue;
            }
         }

#if SCSI_CHECK_PCI_ERROR
         /* only check pci error for now */
         if((hhcb->hardwareMode != SCSI_HMODE_AIC78XX) && 
           (error = (OSD_INEXACT8(SCSI_AICREG(SCSI_ERROR)) & SCSI_PCIERRSTAT)))
         {
            /* svae error status */
            hhcb->SCSI_HP_error = error;
            hhcb->SCSI_HP_hcntrl = OSD_INEXACT8(SCSI_AICREG(SCSI_HCNTRL));
            
            /* disable hardware interrupt with pause */
            SCSI_hWRITE_HCNTRL(hhcb,scsiRegister,SCSI_PAUSE);
         }
#endif /* #if SCSI_CHECK_PCI_ERROR */

         if (intstat & (SCSI_BRKINT | SCSI_SCSIINT | SCSI_SEQINT))
         {
            if (intstat & SCSI_SEQINT)
            {
               /* keep sequencer interrupt code */
               hhcb->SCSI_HP_intstat |= (intstat & SCSI_INTCODE);
            }

            /* keep interrupt status */
            hhcb->SCSI_HP_intstat |= (intstat & (SCSI_BRKINT | SCSI_SCSIINT | SCSI_CMDCMPLT | SCSI_SEQINT));
            hhcb->SCSI_HP_hcntrl = OSD_INEXACT8(SCSI_AICREG(SCSI_HCNTRL));

            /* disable hardware interrupt with pause */
            SCSI_hWRITE_HCNTRL(hhcb,scsiRegister,SCSI_PAUSE);

            /* workaround for hanging of sequencer immediately after a pause */
            /* due to hardware stretch                                       */
            SCSI_hSEQSTRETCHWORKAROUND(hhcb);
         }

         /* ignore the upper nibble */
         /* there is no point to record sparrious interrupt */
         intstat &= (SCSI_BRKINT | SCSI_SCSIINT | SCSI_CMDCMPLT | SCSI_SEQINT);
         break;
      }
   }

   OSD_SYNCHRONIZE_IOS(hhcb);

#if SCSI_CHECK_PCI_ERROR
   return(intstat|error);
#else
   return(intstat);
#endif /* #if SCSI_CHECK_PCI_ERROR */
}
#endif /* !SCSI_STREAMLINE_QIOPATH */

/*********************************************************************
*
*  SCSIHBackEndISR
*
*     Back end processing of ISR
*
*  Return Value:  none
*                  
*  Parameters:    hhcb
*
*  Remarks:       This routine will actually handle interrupt event
*                 though it was cleared by SCSIHFrontEndISR.
*
*********************************************************************/
#if !SCSI_STREAMLINE_QIOPATH
void SCSIHBackEndISR (SCSI_HHCB SCSI_HPTR hhcb)
{
   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_HIOB SCSI_IPTR hiob;
   SCSI_UEXACT8 intstat;
   SCSI_UEXACT8 sstat1;
   SCSI_UEXACT8 scbNumber;
   SCSI_UEXACT8 intcode;
#if SCSI_CHECK_PCI_ERROR
   SCSI_UEXACT8 error;
#endif /* #if SCSI_CHECK_PCI_ERROR */
#if SCSI_TARGET_OPERATION  
   SCSI_UEXACT8 sstat0; 
#endif  /* SCSI_TARGET_OPERATION */   
   SCSI_UEXACT8 exceptionCount;

   intstat = hhcb->SCSI_HP_intstat;
   hhcb->SCSI_HP_intstat = 0;

#if SCSI_CHECK_PCI_ERROR
   error = hhcb->SCSI_HP_error;
   hhcb->SCSI_HP_error = 0;
#endif /* #if SCSI_CHECK_PCI_ERROR */

   if (intstat & SCSI_CMDCMPLT)
   {
      /* handle command complete interrupt */
      SCSIhCmdComplete(hhcb);
   }

   /* check if there was any exceptional interrupt */
#if SCSI_CHECK_PCI_ERROR
   if(error)
   {
      SCSIhClearPCIError(hhcb);
      /* unpause chip (restore interrupt enable bit) */
      SCSI_hWRITE_HCNTRL(hhcb,scsiRegister,(SCSI_UEXACT8)(hhcb->SCSI_HP_hcntrl & SCSI_INTEN));
   }
#endif /* #if SCSI_CHECK_PCI_ERROR */

   if (intstat & (SCSI_BRKINT | SCSI_SCSIINT | SCSI_SEQINT))
   {
      exceptionCount = 0;

      while ((intstat = SCSI_hREAD_INTSTAT(hhcb,scsiRegister)) &
         (SCSI_BRKINT | SCSI_SCSIINT | SCSI_SEQINT))
      {
         /* If the number of exceptions exceeds 10, then assume that it is */
         /* spurious and recover from it. */
         exceptionCount++;
         if (exceptionCount > 10)
         {
            exceptionCount = 0;
            SCSIhBadSeq(hhcb);
            break;
         }

         /* For Cardbus (APA-1480x), the instat register value is 0xFF */
         /* when the card is hot removed from the bus.  For this case, */
         /* CHIM must break out the loop.  This value is not going to  */
         /* happen on the scsi host adapter. */
         if (intstat == 0xFF)
         {
            break;
         }

         SCSI_hSEQSTRETCHWORKAROUND(hhcb); /* sequencer stretch workaround */
         if ((sstat1 = OSD_INEXACT8(SCSI_AICREG(SCSI_SSTAT1))) & SCSI_SCSIRSTI)
         {
            /* scsi reset interrupt */
            SCSIhIntSrst(hhcb);
#if SCSI_TARGET_OPERATION  
            /* Need to revisit this change. SCSI_CLRSCSIRSTI is cleared in
             * SCSIhIntSrst.
             */
            /* clear SCSI interrupt */
            OSD_OUTEXACT8(SCSI_AICREG(SCSI_CLRSINT1),SCSI_CLRSCSIRSTI);
#endif /* SCSI_TARGET_OPERATION */
            scbNumber = SCSI_NULL_SCB;
            continue;
         }
#if (SCSI_AICBAYONET || SCSI_AICTRIDENT)
         else if (OSD_INEXACT8(SCSI_AICREG(SCSI_SSTAT0)) & SCSI_BAYONET_IOERR)
         {
            /* io error interrupt */
            SCSIhIntIOError(hhcb);
            scbNumber = SCSI_NULL_SCB;
            continue;
         }
#endif /* SCSI_AICBAYONET || SCSI_AICTRIDENT */
         else
         {
            if (sstat1 & SCSI_SELTO)      /* handle SCSI selection timout */
            {
               scbNumber = OSD_INEXACT8(SCSI_AICREG(SCSI_hWAITING_SCB(hhcb)));
            }
            else
            {
               scbNumber = OSD_INEXACT8(SCSI_AICREG(SCSI_hACTIVE_SCB(hhcb)));
            }
         }

         if (scbNumber == SCSI_NULL_SCB)
         {
            hiob = SCSI_NULL_HIOB;
         }
         else
         {
            hiob = SCSI_ACTPTR[scbNumber];
         }

         if (intstat & SCSI_SEQINT)      
         {
            /* Process sequencer interrupt */
            intstat &= ~SCSI_SEQINT;
            intcode = intstat & SCSI_INTCODE;
            if ((scbNumber == SCSI_NULL_SCB) || 
                ((hiob == SCSI_NULL_HIOB) && 
                 ((intcode != SCSI_CDB_XFER_PROBLEM) &&
                  (intcode != SCSI_AUTO_RATE_OFFSET))))
            {
               /* If hiob is SCSI_NULL_HIOB then don't enter any routine     */
               /* which doesn't have protection on hiob parameter.           */
               /* Abort only if it is not a break point interrupt at idle    */
               /* loop, as an scb can be null in this case                   */
               if (intcode != SCSI_IDLE_LOOP_BREAK)
               {
                  intcode = SCSI_ABORT_TARGET;
               }
            }

            /* process all the sequencer interrupts */
            switch (intcode)
            {
               case SCSI_DATA_OVERRUN:         /* data overrun/underrun */
                  SCSIhCheckLength(hhcb,hiob);
                  break;

               case SCSI_CDB_XFER_PROBLEM:     /* cdb bad transfer */
                  SCSIhCdbAbort(hhcb);
                  break;

               case SCSI_HANDLE_MSG_OUT:       /* send msg out */
                  SCSIhHandleMsgOut(hhcb,hiob);
                  break;

               case SCSI_SYNC_NEGO_NEEDED:
                  SCSI_hNEGOTIATE(hhcb,hiob);
                  break;

               case SCSI_CHECK_CONDX:
                  SCSIhCheckCondition(hhcb,hiob);
                  break;

               case SCSI_PHASE_ERROR:
                  SCSIhBadSeq(hhcb);
                  break;

               case SCSI_EXTENDED_MSG:
                  SCSIhExtMsgi(hhcb,hiob);
                  break;

               case SCSI_UNKNOWN_MSG:
                  SCSIhHandleMsgIn(hhcb,hiob);
                  break;

               case SCSI_ABORT_TARGET:
                  SCSIhAbortTarget(hhcb,hiob);
                  break;

               case SCSI_NO_ID_MSG:
                  SCSIhAbortTarget(hhcb,hiob);
                  break;

               case SCSI_IDLE_LOOP_BREAK:
               case SCSI_EXPANDER_BREAK:
                  SCSIhBreakInterrupt(hhcb,hiob);
                  break;

#if SCSI_TARGET_OPERATION  
               case SCSI_TARGET_BUS_HELD:
                  SCSIhTargetScsiBusHeld(hhcb,hiob);
                  break;

               case SCSI_TARGET_SPECIAL_FUNC: 
                  SCSIhTargetHandleSpecialFunction(hhcb,hiob);
                  break;
#endif  /* SCSI_TARGET_OPERATION */           

               case SCSI_AUTO_RATE_OFFSET:
                  /* workaround for Auto Rate/Offset issue where */
                  /* the scsi rate/offset does not take affect   */
                  /* immediately for current connection. */
                  SCSI_hENABLEAUTORATEOPTION(hhcb);
                  break;

               default:
                  /* if we obtain an unknown sequencer interrupt
                   * assume something wrong
                   */
                  SCSIhBadSeq(hhcb);
            }                    /* end of sequencer interrupt handling */

            /* clear SEQ interrupt */
            OSD_OUTEXACT8(SCSI_AICREG(SCSI_CLRINT),SCSI_CLRSEQINT);
         }

         /* process scsi interrupt */
         else if (intstat & SCSI_SCSIINT)
         {
            intstat &= ~SCSI_SCSIINT;

#if SCSI_SCAM_ENABLE == 2
            /* SCAM selection interrupt handled here:                   */
            /*                                                          */
            /* Must check SCAM selection before selection timeout since */
            /* selection timeout will be set during SCAM selection.     */
            if ((OSD_INEXACT8(SCSI_AICREG(SCSI_SCAMSTAT)) & SCSI_SCAMSELD) &&
                (hhcb->SCSI_HF_scamLevel == 2))
            {
               /* Need to clear selection timeout */
               OSD_OUTEXACT8(SCSI_AICREG(SCSI_CLRSINT1), SCSI_CLRSELTIMO);
               SCSIhScamSelection(hhcb,hiob);
            }
#if SCSI_TARGET_OPERATION  
            else if (((sstat0 = OSD_INEXACT8(SCSI_AICREG(SCSI_SSTAT0)))& SCSI_TARGET) &&
                     (sstat0 & SCSI_SELDI) && 
                     (OSD_INEXACT8(SCSI_AICREG(SCSI_SIMODE0)) & SCSI_ENSELDI))
            {    
                SCSIhTargetIntSelIn(hhcb);
            }
            else if ((sstat0 & SCSI_TARGET) && (sstat1 & SCSI_ATNTARG) &&
                     (OSD_INEXACT8(SCSI_AICREG(SCSI_SIMODE1)) & SCSI_ENATNTARG))
            {
                SCSIhTargetATNIntr(hhcb,hiob);
            }
            else if (hiob != SCSI_NULL_HIOB && hiob->SCSI_IP_targetMode)
            {
                if ((sstat1 & SCSI_SELTO))
                { 
                   SCSIhTargetIntSelto(hhcb,hiob);  /* selection time out */
                }
                else if (sstat1 & SCSI_SCSIPERR)
                {
                   SCSIhTargetParityError(hhcb,hiob); /* parity error */
                } 
            }
#endif /* SCSI_TARGET_OPERATION */

            /* Selection timeout must be checked before bus free since */
            /* bus free status is set due to a selection timeout. */
            else if (sstat1 & SCSI_SELTO)
               SCSIhIntSelto(hhcb,hiob);        /* selection time out */
#else
#if SCSI_TARGET_OPERATION  
            if (((sstat0=OSD_INEXACT8(SCSI_AICREG(SCSI_SSTAT0)))& SCSI_TARGET) &&
                (sstat0 & SCSI_SELDI) && 
                (OSD_INEXACT8(SCSI_AICREG(SCSI_SIMODE0)) & SCSI_ENSELDI)) 
            {    
               SCSIhTargetIntSelIn(hhcb);
            }
            else if ((sstat0 & SCSI_TARGET) && (sstat1 & SCSI_ATNTARG) &&
                     (OSD_INEXACT8(SCSI_AICREG(SCSI_SIMODE1)) & SCSI_ENATNTARG))
            {
               SCSIhTargetATNIntr(hhcb,hiob);
            } 
            else if (hiob != SCSI_NULL_HIOB && hiob->SCSI_IP_targetMode)
            {
                if ((sstat1 & SCSI_SELTO))
                   SCSIhTargetIntSelto(hhcb,hiob);  /* selection time out */
                else if (sstat1 & SCSI_SCSIPERR)
                   SCSIhTargetParityError(hhcb,hiob); /* parity error */
            }
            else 
#endif /* SCSI_TARGET_OPERATION */ 
            if      (sstat1 & SCSI_SELTO)
               SCSIhIntSelto(hhcb,hiob);        /* selection time out */

#endif   /* (#ifdef   SCSI_SCAM_ENABLE) */

            else if (sstat1 & SCSI_BUSFREE)
               SCSIhIntFree(hhcb,hiob); /* unexpected BUSFREE interrupt */

            else if (sstat1 & SCSI_SCSIPERR)
               SCSIhParityError(hhcb,hiob);     /* parity error */
            else if (sstat1 & SCSI_PHASEMIS)
            {
               OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSISIG),
                  (SCSI_UEXACT8)(OSD_INEXACT8(SCSI_AICREG(SCSI_SCSISIG)) & SCSI_BUSPHASE));
               SCSIhBadSeq(hhcb);
            }
            else if (sstat1 & SCSI_PHASECHG)
            {
               OSD_OUTEXACT8(SCSI_AICREG(SCSI_CLRSINT1),SCSI_CLRPHASECHG);
            }

            /* clear SCSI interrupt */
            OSD_OUTEXACT8(SCSI_AICREG(SCSI_CLRINT),SCSI_CLRSCSINT);
         }

         /* process sequencer breakpoint */
         else if (intstat & SCSI_BRKINT)
         {
            intstat &= ~SCSI_BRKINT;

            SCSIhBreakInterrupt(hhcb,hiob);          /* just for debug  */

            /* clear SCSI interrupt */
            OSD_OUTEXACT8(SCSI_AICREG(SCSI_CLRINT),SCSI_CLRBRKINT);
         }

         /* unpause chip (restore interrupt enable bit) */
         SCSI_hWRITE_HCNTRL(hhcb,scsiRegister,(SCSI_UEXACT8)(hhcb->SCSI_HP_hcntrl & SCSI_INTEN));
      }
   }

   OSD_SYNCHRONIZE_IOS(hhcb);
}
#endif /* !SCSI_STREAMLINE_QIOPATH */

/*%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%*/
/* Modules internal to hardware management layer                          */
/*%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%*/

/*********************************************************************
*
*  SCSIhCmdComplete
*
*     Handle command complete interrupt from sequencer
*
*  Return Value:  number of complete scb processed at all
*                  
*  Parameters:    hhcb
*
*  Remarks:       This routine should be referenced by interrupt dispatcher.
*                 At entrance of interrupt dispatcher it should always
*                 assume the interrupt pending is command complete.
*                 Without verifying the source of interrupt this routine
*                 must be called blindly. The return status of this routine
*                 indicates if there was a command complete pending or not.
*                 The interrupt dispatcher should check the return status
*                 of this routine. If the returned status is not zero then
*                 interrupt dispatcher should return control to resource
*                 management layer (with return statement). If the returned
*                 status is zero then interrupt dispatcher should verify
*                 if there is any error/exception interrupt pending and
*                 process it until all pending error/exception has been
*                 processed.
*
*                 If there was any command complete pending it will be
*                 cleared by this routine.
*
*********************************************************************/
#if !SCSI_STREAMLINE_QIOPATH
int SCSIhCmdComplete (SCSI_HHCB SCSI_HPTR hhcb)
{
   int retValue = 0;
   SCSI_HIOB SCSI_IPTR hiob;
   SCSI_UEXACT8 scbNumber;
   register SCSI_REGISTER scsiRegister;
   SCSI_UEXACT8 hcntrl;

   while (1)
   {
      /* service all command complete outstanding */
      while ((scbNumber = (SCSI_UEXACT8) SCSI_hRETRIEVESCB(hhcb)) != SCSI_NULL_SCB)
      {
         /* get associated hiob */
         hiob = SCSI_ACTPTR[scbNumber];

         /* If there is no associated hiob, assume it as fatal */
         if (hiob == SCSI_NULL_HIOB)
         {
            /* Something is wrong with the device or sequencer */
            /* if there is no hiob reference */
            /* save the HCNTRL value */
            hcntrl = OSD_INEXACT8(SCSI_AICREG(SCSI_HCNTRL));

            /* pause the chip if necessary */
            if (!(hcntrl & SCSI_PAUSEACK))
            {
               SCSIhPauseAndWait(hhcb);
            }

            SCSIhBadSeq(hhcb);

            /* restore HCNTRL if necessary */
            if (!(hcntrl & SCSI_PAUSEACK))
            {
               SCSI_hWRITE_HCNTRL(hhcb,scsiRegister,hcntrl);
            }

            break;
         }

         /* Due to the active abort implementation, a HIOB to be aborted */
         /* might completed before the active abort message has ever */
         /* started.  Therefore, we will not post back any HIOB that */
         /* associated with the abort already in progress status. */
         if (hiob->SCSI_IP_mgrStat != SCSI_MGR_ABORTINPROG)
         {
            /* indicate command complete was active */
            ++retValue;

            /* terminate the command */
            SCSI_hTERMINATECOMMAND(hiob);
         }

         --hhcb->SCSI_HP_qoutcnt;

#if SCSI_BACKENDISR_OUTSIDE_INTRCONTEXT
         /* There is a problem when the BackEndISR() was called outside    */
         /* interrupt context.  The problem would occur when the sequencer */
         /* post more completed SCBs in the done queue between the time    */
         /* the FrontEndISR() executed and the last completed SCB          */
         /* (accounted by HIM in FrontEndISR()) was post back to upper     */
         /* layer in this routine.  CHIM will continues to post back all   */
         /* new completed SCBs and cleared the CMDCMPLT bit.  However, the */
         /* hardware interrupt is still pending out there.  Then when the  */
         /* pending interrupt interrupted, the FrontEndISR() will report   */
         /* back a value of 0 to the OSM saying this interrupt wasn't for  */
         /* us.  This will create a spurious interrupt and for some OSes   */
         /* like OS/2, will disable that interrupt in the PIC until the    */
         /* next system boot. */
         if (!hhcb->SCSI_HP_qoutcnt)
         {
            return(retValue);
         }
#endif
      }

      if (hhcb->SCSI_HP_qoutcnt)
      {
         /* we have serviced more command complete than we should */
         /* better clear the clear command interrupt */
         scsiRegister = hhcb->scsiRegister;
         OSD_OUTEXACT8(SCSI_AICREG(SCSI_CLRINT),SCSI_CMDCMPLT);
         hhcb->SCSI_HP_qoutcnt = 0;
      }
      else
      {
         break;
      }
   }

   return(retValue);
}
#endif /* !SCSI_STREAMLINE_QIOPATH */

/*********************************************************************
*
*  SCSIhIntSrst routine -
*
*     Handle cases when another device resets scsi bus
*
*  Return Value:  None
*                  
*  Parameters:    hhcb
*
*  Remarks:       SCAM and AUTOTERMINATION operation would have to
*                 re-initiated SCB chain would have to be cleaned up.
*                  
*********************************************************************/
void SCSIhIntSrst (SCSI_HHCB SCSI_HPTR hhcb)
{
   SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_UINT32 j = 0;          /* time out counter */
   SCSI_UINT16 i, event = SCSI_AE_3PTY_RESET;

   /* convert resetDelay from msec to 500usec and add 1 to ensure that we 
    * enter loop at least once
    */
   SCSI_UINT32 timerCount = ((SCSI_UINT32)hhcb->resetDelay * 2) + 1;
 
   /* wait until reset deasserted */
   while ((OSD_INEXACT8(SCSI_AICREG(SCSI_SSTAT1)) & SCSI_SCSIRSTI) &&
          (j++ < 400000))
   {
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_CLRSINT1), SCSI_CLRSCSIRSTI);
      OSD_TIMER(hhcb);
   }

   SCSI_ASYNC_EVENT(hhcb, SCSI_AE_OSMFREEZE);      /* AEN call to RSM layer */

#if SCSI_TARGET_OPERATION
   /* If  target mode enabled need to reset the holding
    * SCSI bus indicator or SCSIhTargetClearNexus will fail
    */ 
   if (hhcb->SCSI_HF_targetMode)
   {    
      if (hhcb->SCSI_HF_targetScsiBusHeld)
      {
         hhcb->SCSI_HF_targetScsiBusHeld = 0;
         hhcb->SCSI_HP_nexusHoldingBus->SCSI_XF_busHeld = 0;
      }

      /* Block upper layers from issuing Establish Connection HIOBs */
      SCSI_hTARGETHOLDESTHIOBS(hhcb,1);
   }
#endif /* SCSI_TARGET_OPERATION */

   SCSIhResetChannelHardware(hhcb);                /* reset channel hardware*/
   SCSIhAbortChannel(hhcb, SCSI_HOST_ABT_3RD_RST); /* abort all active cmds */
   SCSI_hRESETSOFTWARE(hhcb);                      /* reset channel software*/
   if (SCSIhBusWasDead(hhcb))                   /* check if bus is really*/ 
   {                                            /* dead, then do reset   */
      if (SCSIhResetScsi(hhcb))
      {
         event = SCSI_AE_IO_CHANNEL_FAILED;
      }
   }

   SCSI_hSTACKBUGFIX(hhcb);

   /* We need to clear skipScam flag here so that */
   /* the scam protocol can be invoked. */
   hhcb->SCSI_HF_skipScam = 0;
   
   /* Set here for the PAC to invoke suppress negotiation. */
   /* This to make sure the internal Bus Scan always talk async. to devices */
   hhcb->SCSI_HF_resetSCSI = 1;

   for (i=0;i<16;i++)
   {
      SCSI_DEVICE_TABLE(hhcb)[i].SCSI_DF_resetSCSI = 1;
   }

   /* clear interrupt and delay */
   for (j = 0; j < timerCount; j++)
   {
      if ((SCSI_hREAD_INTSTAT(hhcb,scsiRegister)) &
          (SCSI_BRKINT | SCSI_SCSIINT | SCSI_SEQINT))
      {
         SCSI_hSEQSTRETCHWORKAROUND(hhcb); /* sequencer stretch workaround */
         SCSIhResetChannelHardware(hhcb);    /* reset channel hardware */
         OSD_OUTEXACT8(SCSI_AICREG(SCSI_CLRSINT1), SCSI_CLRSCSIRSTI);
         OSD_OUTEXACT8(SCSI_AICREG(SCSI_CLRINT),
               SCSI_CLRSCSINT | SCSI_CLRSEQINT);
      }

      SCSI_hRESETDELAY(hhcb, 1);
   }   

   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SEQADDR0), 
         (SCSI_UEXACT8)SCSI_hIDLE_LOOP_ENTRY(hhcb) >> 2);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SEQADDR1), 
         (SCSI_UEXACT8)SCSI_hIDLE_LOOP_ENTRY(hhcb) >> 10);

   /* Unblock upper layers from issuing Establish Connection HIOBs */
   SCSI_hTARGETHOLDESTHIOBS(hhcb,0);

   /* unpause chip (restore interrupt enable bit) */
   SCSI_hWRITE_HCNTRL(hhcb,scsiRegister,
         (SCSI_UEXACT8)(hhcb->SCSI_HP_hcntrl & SCSI_INTEN));

   OSD_SYNCHRONIZE_IOS(hhcb);

   /* Async. Event to OSM */
   SCSI_ASYNC_EVENT(hhcb, event);

   SCSI_ASYNC_EVENT(hhcb, SCSI_AE_OSMUNFREEZE); /* AEN call back to RSM layer */

}

/*********************************************************************
*
*  SCSIhIntIOError routine -
*
*     Handle cases when the I/O operating mode is changed
*
*  Return Value:  None
*                  
*  Parameters:    hhcb
*
*  Remarks:       SCAM and AUTOTERMINATION operation would have to
*                 re-initiated SCB chain would have to be cleaned up.
*                  
*********************************************************************/
void SCSIhIntIOError (SCSI_HHCB SCSI_HPTR hhcb)
{
   SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_UINT32 i;
   SCSI_UINT16 event = SCSI_AE_IOERROR;

   /* convert resetDelay from msec to 500usec and add 1 to ensure that we 
    * enter loop at least once
    */   SCSI_UINT32 timerCount = ((SCSI_UINT32)hhcb->resetDelay * 2) +1; 

   OSD_OUTEXACT8(SCSI_AICREG(SCSI_CLRSINT0), SCSI_BAYONET_CLRIOERR);

   SCSI_ASYNC_EVENT(hhcb, SCSI_AE_OSMFREEZE);      /* AEN call to RSM layer */

#if SCSI_TARGET_OPERATION
   if (hhcb->SCSI_HF_initiatorMode)
   {
      /* We need to reset SCSI bus after ioerror happens */
      /* This is a violation of SPI-2 specification - we */
      /* will only do this if initiator mode enabled.    */ 
      if (SCSIhResetScsi(hhcb))
      {
         event = SCSI_AE_IO_CHANNEL_FAILED;
      }
   }
#else
   /* We need to reset SCSI bus after ioerror happens */
   /* This is a violation of SPI-2 specification - we */
   /* will only do this if initiator mode enabled.    */ 
   if (SCSIhResetScsi(hhcb))
   {
      event = SCSI_AE_IO_CHANNEL_FAILED;
   }
#endif /* SCSI_TARGET_OPERATION */

   SCSIhResetChannelHardware(hhcb);                /* reset channel hardware*/
    /* Block upper layers from issuing Establish Connection HIOBs */
   SCSI_hTARGETHOLDESTHIOBS(hhcb,1);
   SCSIhAbortChannel(hhcb, SCSI_HOST_ABT_IOERR);   /* abort all active cmds */
   SCSI_hRESETSOFTWARE(hhcb);                      /* reset channel software*/

   /* We need to clear skipScam flag here so that */
   /* the scam protocol can be invoked. */
   hhcb->SCSI_HF_skipScam = 0;
   
   /* Set here for the PAC to invoke suppress negotiation. */
   /* This to make sure the internal Bus Scan always talk async. to devices */
   hhcb->SCSI_HF_resetSCSI = 1;

   /* clear interrupt and delay */
   for (i=0;i<timerCount;i++)
   {
      if ((SCSI_hREAD_INTSTAT(hhcb,scsiRegister)) &
          (SCSI_BRKINT | SCSI_SCSIINT | SCSI_SEQINT))
      {
         SCSIhResetChannelHardware(hhcb);    /* reset channel hardware */
         OSD_OUTEXACT8(SCSI_AICREG(SCSI_CLRINT),
               SCSI_CLRSCSINT | SCSI_CLRSEQINT);
      }
      SCSI_hRESETDELAY(hhcb, 1);
   }   

   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SEQADDR0), 
         (SCSI_UEXACT8)SCSI_hIDLE_LOOP_ENTRY(hhcb) >> 2);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SEQADDR1), 
         (SCSI_UEXACT8)SCSI_hIDLE_LOOP_ENTRY(hhcb) >> 10);

   /* Unblock upper layers from issuing Establish Connection HIOBs */
   SCSI_hTARGETHOLDESTHIOBS(hhcb,0);   

   /* unpause chip (restore interrupt enable bit) */
   SCSI_hWRITE_HCNTRL(hhcb,scsiRegister,
         (SCSI_UEXACT8)(hhcb->SCSI_HP_hcntrl & SCSI_INTEN));

   OSD_SYNCHRONIZE_IOS(hhcb);

   /* Async. Event to OSM */
   SCSI_ASYNC_EVENT(hhcb, event);

   SCSI_ASYNC_EVENT(hhcb, SCSI_AE_OSMUNFREEZE); /* AEN call back to RSM layer */

}

/*********************************************************************
*
*  SCSIhCheckLength
*
*     Check for underrun/overrun conditions following exception
*     condition occuring during data transfer
*
*  Return Value:  None
*                  
*  Parameters:    hhcb
*                 hiob
*
*  Remarks:                
*                  
*********************************************************************/
void SCSIhCheckLength (SCSI_HHCB SCSI_HPTR hhcb,
                       SCSI_HIOB SCSI_IPTR hiob)
{
   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_UEXACT8 phase;
   SCSI_UEXACT8 ubyteWork;
   SCSI_UEXACT8 stat;
   SCSI_UINT32 i = 0;          /* time out counter */
   SCSI_UEXACT32 residualLength;

   if (hiob->SCSI_IF_freezeOnError &&
       !(SCSI_TARGET_UNIT(hiob)->targetControl->freezeMap &
         SCSI_FREEZE_HIPRIEXISTS))
   {
      SCSI_hFREEZEHWQUEUE(hhcb,hiob);
   }
   
   /* underrun if SCSI bus at STATUS phase */
   /* For Trident, data overrun might occur even though SCSI bus is at */
   /* STATUS phase because of auto-ACK feature.  If all overrun data   */
   /* already acked in the data fifo, the target then might switch to  */
   /* STATUS phase.  We need to further qualify the underrun condition */
   /* with checking for the fifo empty.                                */
   phase = (SCSI_UEXACT8)(OSD_INEXACT8(SCSI_AICREG(SCSI_SCSISIG)) & SCSI_BUSPHASE);
   if ((phase == SCSI_STPHASE) &&
       (OSD_INEXACT8(SCSI_AICREG(SCSI_DFSTATUS)) & SCSI_FIFOEMP))
   {
      /* do nothing if underrun error suppressed */
      if (hiob->SCSI_IF_noUnderrun)
      {
         return;
      }
      else
      {
         residualLength = SCSI_hRESIDUECALC(hhcb,hiob);

         if (hiob->trgStatus == SCSI_UNIT_CHECK)
         {
            hiob->snsResidual = (SCSI_UEXACT8) residualLength;
            return ;
         }
         else
         {
            hiob->residualLength = residualLength;
         }
      }
   }
   else if ((phase & SCSI_CDO) == 0)   /* overrun if SCSI bus at Data phase */
   {
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSISIG), phase);
      /* need to set the correct direction bit before turn on BITBUCKET mode */
      ubyteWork = OSD_INEXACT8(SCSI_AICREG(SCSI_DFCNTRL));
      if ((phase & SCSI_IOO) == 0)     /* Data Out, set direction bit */
      {
         OSD_OUTEXACT8(SCSI_AICREG(SCSI_DFCNTRL), ubyteWork | SCSI_DIRECTION);
      }
      else                             /* Data In, clear direction bit */
      {
         OSD_OUTEXACT8(SCSI_AICREG(SCSI_DFCNTRL), ubyteWork & ~SCSI_DIRECTION);
      }
      ubyteWork = OSD_INEXACT8(SCSI_AICREG(SCSI_SXFRCTL1));
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SXFRCTL1), ubyteWork | SCSI_BITBUCKET);
      while ((((stat = OSD_INEXACT8(SCSI_AICREG(SCSI_SSTAT1))) & SCSI_PHASEMIS) == 0) && (i++ < 400000))
      {
         
         if (   (stat & (SCSI_BUSFREE | SCSI_SCSIRSTI)))
            break;
         OSD_TIMER(hhcb);
      }

      if (i > 400000)
      {
         SCSIhBadSeq(hhcb);
         return;
      }

      /* scb should contain zero residual value */
      hiob->residualLength = 0;      
               
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SXFRCTL1), ubyteWork);
   }
   else
   {
      /* For Trident, the auto-ACK feature might causes the CHIM to detect    */
      /* other SCSI phases than DATA-IN or DATA-OUT in the data overrun case. */
      hiob->residualLength = 0;      
   }      

   hiob->haStat = SCSI_HOST_DU_DO;       /* indicate overrun/underrun status */
}

/*********************************************************************
*
*  SCSIhCdbAbort
*
*     Send SCSI abort msg to selected target
*
*  Return Value:  None
*                  
*  Parameters:    hhcb
*                 
*  Remarks:       limited implementation, at present
*                  
*********************************************************************/
void SCSIhCdbAbort (SCSI_HHCB SCSI_HPTR hhcb)
{
   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_UEXACT8 phase;

   if ((OSD_INEXACT8(SCSI_AICREG(SCSI_SCSISIG)) & SCSI_BUSPHASE) != SCSI_MIPHASE)
   {
      SCSIhBadSeq(hhcb);
      return;
   }

   if (OSD_INEXACT8(SCSI_AICREG(SCSI_SCSIBUSL)) == SCSI_MSG03)
   {
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSISIG), SCSI_MIPHASE);

#if (SCSI_STANDARD_ADVANCED_MODE || SCSI_SWAPPING_ADVANCED_MODE ||\
     SCSI_STANDARD_160M_MODE || SCSI_SWAPPING_160M_MODE)
      if ((hhcb->firmwareMode == SCSI_FMODE_STANDARD_ADVANCED) ||
          (hhcb->firmwareMode == SCSI_FMODE_SWAPPING_ADVANCED) ||
          (hhcb->firmwareMode == SCSI_FMODE_STANDARD_160M) ||
          (hhcb->firmwareMode == SCSI_FMODE_SWAPPING_160M))
      {
         OSD_OUTEXACT8(SCSI_AICREG(SCSI_SXFRCTL0),
             (OSD_INEXACT8(SCSI_AICREG(SCSI_SXFRCTL0)) | (SCSI_CLRSTCNT | SCSI_CLRCHN)));
      }
#endif /* (SCSI_STANDARD_ADVANCED_MODE || SCSI_SWAPPING_ADVANCED_MODE || */
       /*  SCSI_STANDARD_160M_MODE || SCSI_SWAPPING_160M_MODE) */
 
      OSD_INEXACT8(SCSI_AICREG(SCSI_SCSIDATL));
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SEQADDR0), (SCSI_UEXACT8)SCSI_hSIOSTR3_ENTRY(hhcb) >> 2);
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SEQADDR1), (SCSI_UEXACT8)SCSI_hSIOSTR3_ENTRY(hhcb) >> 10);
   }
   else
   {
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSISIG), SCSI_MIPHASE | SCSI_ATNO);
      do
      {
         OSD_INEXACT8(SCSI_AICREG(SCSI_SCSIDATL));
         phase = SCSIhWait4Req(hhcb);
      } while (phase == SCSI_MIPHASE);

      if (phase != SCSI_MOPHASE)
      {
         SCSIhBadSeq(hhcb);
         return;
      }

      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSISIG), SCSI_MOPHASE);
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_CLRSINT1), SCSI_CLRATNO);
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSIDATL), SCSI_MSG07);
      if (SCSIhWait4Req(hhcb) == (SCSI_UEXACT8)-1)
      {
         SCSIhBadSeq(hhcb);
         return;
      }
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SEQADDR0), (SCSI_UEXACT8)SCSI_hSIO204_ENTRY(hhcb) >> 2);
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SEQADDR1), (SCSI_UEXACT8)SCSI_hSIO204_ENTRY(hhcb) >> 10);
   }
}
               
/*********************************************************************
*
*  SCSIhCheckCondition
*
*     Handle response to target check condition
*
*  Return Value:  None
*                  
*  Parameters:    hhcb
*                 hiob
*
*  Remarks:                
*                  
*********************************************************************/
void SCSIhCheckCondition (SCSI_HHCB SCSI_HPTR hhcb,
                          SCSI_HIOB SCSI_IPTR hiob)
{
   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_UNIT_CONTROL SCSI_UPTR unitHandle = SCSI_TARGET_UNIT(hiob);
   SCSI_UEXACT8 status;

#if (SCSI_STANDARD_ADVANCED_MODE || SCSI_SWAPPING_ADVANCED_MODE ||\
     SCSI_STANDARD_160M_MODE || SCSI_SWAPPING_160M_MODE)

   SCSI_UINT32 i = 0;          /* time out counter */
#endif /* (SCSI_STANDARD_ADVANCED_MODE || SCSI_SWAPPING_ADVANCED_MODE || */
       /*  SCSI_STANDARD_160M_MODE || SCSI_SWAPPING_160M_MODE) */

   if (hiob->SCSI_IP_mgrStat == SCSI_MGR_AUTOSENSE)
   {
      hiob->haStat = SCSI_HOST_SNS_FAIL;
      status = 0;
   }
   else
   {
      status = OSD_INEXACT8(SCSI_AICREG(SCSI_hPASS_TO_DRIVER(hhcb)));
      hiob->trgStatus = status;         /* record UNIT CHECK condition */
   }

   if (((hiob->SCSI_IF_freezeOnError) ||
       (status == SCSI_UNIT_BUSY) ||
       (status == SCSI_UNIT_QUEFULL)) &&
       !(SCSI_TARGET_UNIT(hiob)->targetControl->freezeMap &
         SCSI_FREEZE_HIPRIEXISTS))
   {
      SCSI_hFREEZEHWQUEUE(hhcb,hiob);
   }
   
   if ((status == SCSI_UNIT_CHECK) && hiob->SCSI_IF_autoSense)
   {
      hiob->SCSI_IP_negoState = 0;
      SCSI_hREQUESTSENSE(hhcb,hiob);   /* issue REQUEST SENSE command */
   }
   else
   {
      /* clear target busy must be done here also */
      /* if there is a queue full, it will be handle in RSM layer during posting. */
      SCSI_hTARGETCLEARBUSY(hhcb,hiob);
      SCSIhTerminateCommand(hiob);
   }

   /* Reset synchronous/wide negotiation only for CHECK CONDITION */
   /* reset sync/wide as long as configured to do so   */
   /* even if it's negotiated without sync/wide        */
   if (status == SCSI_UNIT_CHECK)
   {
#if SCSI_NEGOTIATION_PER_IOB
      /* If negotiation is forced, only then initiate both sync and wide */
      if (hiob->SCSI_IF_forceReqSenseNego)
      {
         SCSIhChangeNegotiation(unitHandle);
      }
#else /* SCSI_NEGOTIATION_PER_IOB */
      if (!(unitHandle->deviceTable->SCSI_DF_suppressNego))
      {
         if (unitHandle->deviceTable->scsiOption & (SCSI_WIDE_XFER | SCSI_SYNC_XFER))
         {
#if SCSI_DOMAIN_VALIDATION
            /* We should get the request sense data; so, we should make sure */
            /* we negotiate async/narrow during domain validation */
            if (hiob->SCSI_IF_dvIOB)
            {
               unitHandle->deviceTable->SCSI_DF_setForWide = 0;
               unitHandle->deviceTable->SCSI_DF_setForSync = 0;
            }
#endif /* SCSI_DOMAIN_VALIDATION */
            SCSI_hCHKCONDXFERASSIGN(hhcb,unitHandle->scsiID,SCSI_NEEDNEGO);

            /* Set the SCSI RATE to defaults if SCSI_LOOPBACK_OPERATION = 1 */
            SCSI_hSETXFERRATE(hhcb,unitHandle->scsiID);
         }
      }
#endif /* SCSI_NEGOTIATION_PER_IOB */
   }
   
#if (SCSI_STANDARD_ADVANCED_MODE || SCSI_SWAPPING_ADVANCED_MODE ||\
     SCSI_STANDARD_160M_MODE || SCSI_SWAPPING_160M_MODE)

   if ((hhcb->firmwareMode == SCSI_FMODE_STANDARD_ADVANCED) ||
       (hhcb->firmwareMode == SCSI_FMODE_SWAPPING_ADVANCED) ||
       (hhcb->firmwareMode == SCSI_FMODE_STANDARD_160M) ||
       (hhcb->firmwareMode == SCSI_FMODE_SWAPPING_160M))
   
          OSD_OUTEXACT8(SCSI_AICREG(SCSI_DFCNTRL), 00);

#endif /* (SCSI_STANDARD_ADVANCED_MODE || SCSI_SWAPPING_ADVANCED_MODE || */
       /*  SCSI_STANDARD_160M_MODE || SCSI_SWAPPING_160M_MODE) */

   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SXFRCTL0), 
      ((OSD_INEXACT8(SCSI_AICREG(SCSI_SXFRCTL0)) | SCSI_CLRCHN) & ~SCSI_FAST20)); 
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SIMODE1), OSD_INEXACT8(SCSI_AICREG(SCSI_SIMODE1)) & ~SCSI_ENBUSFREE);

   /* workaround for Auto Rate/Offset issue where the scsi rate/offset */
   /* does not take affect immediately for current connection. */
   SCSI_hENABLEAUTORATEOPTION(hhcb);

#if SCSI_PAC_NSX_REPEATER || SCSI_NSX_REPEATER
   if (hiob->SCSI_IF_nsxCommunication)
   {
      /* manual ack of command complete msg for NSX repeater IOBs */
       OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSISIG),0);
   }
   else
#endif /* SCSI_PAC_NSX_REPEATER || SCSI_NSX_REPEATER */
   { 
      OSD_INEXACT8(SCSI_AICREG(SCSI_SCSIDATL)); /* acked command complete msg */
   } 

#if (SCSI_STANDARD_ADVANCED_MODE || SCSI_SWAPPING_ADVANCED_MODE ||\
     SCSI_STANDARD_160M_MODE || SCSI_SWAPPING_160M_MODE)
   
   if ((hhcb->firmwareMode == SCSI_FMODE_STANDARD_ADVANCED) ||
       (hhcb->firmwareMode == SCSI_FMODE_SWAPPING_ADVANCED) ||
       (hhcb->firmwareMode == SCSI_FMODE_STANDARD_160M) ||
       (hhcb->firmwareMode == SCSI_FMODE_SWAPPING_160M))
   
   {
      /* Make sure 'seldo' bit turns off after SCSI bus goes bus-free */
      while ((OSD_INEXACT8(SCSI_AICREG(SCSI_SSTAT0)) & SCSI_SELDO) &&
             (i++ < 400000))
      {
         OSD_TIMER(hhcb);
      }
   }
   
#endif /* (SCSI_STANDARD_ADVANCED_MODE || SCSI_SWAPPING_ADVANCED_MODE || */
       /*  SCSI_STANDARD_160M_MODE || SCSI_SWAPPING_160M_MODE) */

   SCSI_hCLEARSCSIRATE(hhcb);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SEQADDR0), (SCSI_UEXACT8)SCSI_hIDLE_LOOP_ENTRY(hhcb) >> 2);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SEQADDR1), (SCSI_UEXACT8)SCSI_hIDLE_LOOP_ENTRY(hhcb) >> 10);
}

/*********************************************************************
*
*  SCSIhBadSeq
*
*     Terminate SCSI command sequence because sequence that is illegal,
*     or if we just can't handle it.
*
*  Return Value:  None
*                  
*  Parameters:    hhcb
*
*  Remarks:                
*                  
*********************************************************************/
void SCSIhBadSeq (SCSI_HHCB SCSI_HPTR hhcb)
{
   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_UINT16 event = SCSI_AE_HA_RESET;

   /* Don't reset/abort if there is a SCSI bus free interrupt pending */   
   if (!((SCSI_hREAD_INTSTAT(hhcb,scsiRegister) & SCSI_SCSIINT)
         && (OSD_INEXACT8(SCSI_AICREG(SCSI_SSTAT1)) & SCSI_BUSFREE)))
   {
      SCSI_ASYNC_EVENT(hhcb, SCSI_AE_OSMFREEZE);      /* AEN call back to RSM layer */

#if SCSI_TARGET_OPERATION
      if (hhcb->SCSI_HF_initiatorMode)
      {
         /* Only reset SCSI bus if operating in initiator mode */
         if (SCSIhResetScsi(hhcb))
         {
            event = SCSI_AE_IO_CHANNEL_FAILED;
         }

         /* Delay resetDelay msecs after scsi reset for slow devices to settle down. */
         SCSI_hRESETDELAY(hhcb, (hhcb->resetDelay * 2));
      }
#else
      if (SCSIhResetScsi(hhcb))
      {
         event = SCSI_AE_IO_CHANNEL_FAILED;
      }

      /* Delay resetDelay msecs after scsi reset for slow devices to settle down. */
      SCSI_hRESETDELAY(hhcb, (hhcb->resetDelay * 2));
#endif /* SCSI_TARGET_OPERATION */

      SCSIhResetChannelHardware(hhcb);
      /* Block upper layers from issuing Establish Connection HIOBs */
      SCSI_hTARGETHOLDESTHIOBS(hhcb,1);
      SCSIhAbortChannel(hhcb, SCSI_HOST_PHASE_ERR);
      SCSI_hRESETSOFTWARE(hhcb);
      /* Unblock upper layers from issuing Establish Connection HIOBs */
      SCSI_hTARGETHOLDESTHIOBS(hhcb,0);

      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SEQADDR0), (SCSI_UEXACT8)SCSI_hIDLE_LOOP_ENTRY(hhcb) >> 2);
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SEQADDR1), (SCSI_UEXACT8)SCSI_hIDLE_LOOP_ENTRY(hhcb) >> 10);

      /* clear interrupt */
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_CLRINT),SCSI_CLRSCSINT | SCSI_CLRSEQINT);

      /* unpause chip (restore interrupt enable bit) */
      SCSI_hWRITE_HCNTRL(hhcb,scsiRegister,(SCSI_UEXACT8)(hhcb->SCSI_HP_hcntrl & SCSI_INTEN));

      OSD_SYNCHRONIZE_IOS(hhcb);

      /* AEN to RSM - HA initiated Reset */
      SCSI_ASYNC_EVENT(hhcb, event);
     
      SCSI_ASYNC_EVENT(hhcb, SCSI_AE_OSMUNFREEZE);      /* AEN call back to RSM layer */
   }
   else
   {
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SEQADDR0), (SCSI_UEXACT8)SCSI_hIDLE_LOOP_ENTRY(hhcb) >> 2);
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SEQADDR1), (SCSI_UEXACT8)SCSI_hIDLE_LOOP_ENTRY(hhcb) >> 10);
   }
}

/*********************************************************************
*
*  SCSIhAbortTarget
*
*     Abort current target
*
*  Return Value:  None
*             
*  Parameters:    hhcb
*                 hiob
*
*  Remarks:   
*             
*********************************************************************/
void SCSIhAbortTarget (SCSI_HHCB SCSI_HPTR hhcb,
                       SCSI_HIOB SCSI_IPTR hiob)
{
   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   
   if ((SCSI_hREAD_INTSTAT(hhcb,scsiRegister) & SCSI_INTCODE) == SCSI_NO_ID_MSG)
   {
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSISIG), SCSI_MIPHASE | SCSI_ATNO);
      OSD_INEXACT8(SCSI_AICREG(SCSI_SCSIDATL));
   }

   if (SCSIhWait4Req(hhcb) == SCSI_MOPHASE)
   {
      /* SCB ptr valid? */
      if (OSD_INEXACT8(SCSI_AICREG(SCSI_hACTIVE_SCB(hhcb))) != SCSI_NULL_SCB)
      {
         if (hiob == SCSI_NULL_HIOB)
         {
            SCSIhBadSeq(hhcb);
            return;
         }
         if (hiob->SCSI_IP_mgrStat == SCSI_MGR_ABORTINPROG)
         {
            /* For standard 64 and standard adv modes only, check if the     */
            /* target is reselected while active abort is in execution queue */
            /* or selecting a target.  Then we need to remove it. */
            SCSI_hREMOVEACTIVEABORT(hhcb, hiob);

            hiob->SCSI_IP_mgrStat = SCSI_MGR_ABORTED;
            return;
         }
         else if (hiob->SCSI_IP_mgrStat == SCSI_MGR_BDR)
         {
            /* Abort any related HIOB(s) in active pointer array */
            SCSIhBusDeviceReset(hhcb, hiob);
            return;
         }
         /* if we had set the aborted bit in host memory during searching    */
         /* of new queue, then we would have set the mgrStat to MGR_DONE_ABT;*/
         /* in this case we do not have to do anything.  We simply return    */
         else if (hiob->SCSI_IP_mgrStat == SCSI_MGR_DONE_ABT)
         {
            return;
         }
      }
   }
   SCSIhBadSeq(hhcb);
}

/*********************************************************************
*
*  SCSIhScamSelection
*
*     Handle SCAM selection interrupt generated from either another
*     SCAM devices or the host adapter itself.
*
*  Return Value:  none
*                  
*  Parameters:    hhcb
*                 hiob
*
*  Activation:    SCSIHBackEndISR
*                  
*  Remarks:       SCAM level 2 chip will generate SCAM selection
*                 interrupt eventhough the SCAM protocol was
*                 originated from the chip itself.
*                  
*********************************************************************/
#if SCSI_SCAM_ENABLE == 2
void SCSIhScamSelection (SCSI_HHCB SCSI_HPTR hhcb,
                         SCSI_HIOB SCSI_IPTR hiob)
{
   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_UEXACT8   regValue;
   SCSI_UINT16 event = SCSI_AE_SCAM_SELD;

   regValue = OSD_INEXACT8(SCSI_AICREG(SCSI_SCAMSTAT)) | SCSI_CLRSCAMSELD;

   /* check if SCAM protocol was initiated by the host adapter */
   if (regValue & SCSI_SCAMSELOSTA)
   {
      /* clear SCAM Selection Enable Out and SCAM Selection Done bits */
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCAMCTRL), (regValue & ~SCSI_ENSCAMSELO));
   }

   /* if the HA is a subordinate initiator and SCAM selection occurs, it   */
   /* would simply clear interrupt bit and return.                         */
   else if (hhcb->SCSI_HP_scamSubordinateInit)
   {
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCAMCTRL), regValue);
   }
   else         /* other SCAM devices initiate SCAM protocol */
   {
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCAMCTRL), regValue);  /* clear SCAM Selection Done */

      hhcb->SCSI_HP_scamSelectDetected = 1;

      SCSI_ASYNC_EVENT(hhcb, SCSI_AE_OSMFREEZE);      /* AEN call back to RSM layer */

      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSISIG), 0x00);
      SCSIhDelayCount500us (hhcb, 1);
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_CLRSINT1), SCSI_CLRBUSFREE);

      /* The purpose of SCSI bus reset is: */
      /*    1. To send all SCAM devices back to their monitor state */
      /*    2. To reset all nexus agreement between initiator and */
      /*       targets i.e. transfer mode, etc... */
      /*    3. To terminate the nexus with the disconnecting devices */
      if (SCSIhResetScsi(hhcb))
      {
         event = SCSI_AE_IO_CHANNEL_FAILED;
      }

      /* Delay resetDelay msec after scsi reset for slow devices to settle down. */
      SCSI_hRESETDELAY(hhcb, (hhcb->resetDelay * 2));

      /* Disarm any outstanding selections */
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSISEQ), 0x00);

      /* Block upper layers from issuing Establish Connection HIOBs */
      SCSI_hTARGETHOLDESTHIOBS(hhcb,1);

      SCSIhResetChannelHardware(hhcb);             /* reset channel hardware */
      SCSIhAbortChannel(hhcb, SCSI_HOST_ABT_HA);   /* abort all active commands */
      SCSI_hRESETSOFTWARE(hhcb);                   /* reset channel software */

      /* Unblock upper layers from issuing Establish Connection HIOBs */
      SCSI_hTARGETHOLDESTHIOBS(hhcb,0);

      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SEQADDR0), (SCSI_UEXACT8)SCSI_hIDLE_LOOP_ENTRY(hhcb) >> 2);
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SEQADDR1), (SCSI_UEXACT8)SCSI_hIDLE_LOOP_ENTRY(hhcb) >> 10);

      /* clear interrupt */
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_CLRINT),SCSI_CLRSCSINT | SCSI_CLRSEQINT);

      /* unpause chip (restore interrupt enable bit) */
      SCSI_hWRITE_HCNTRL(hhcb,scsiRegister,(SCSI_UEXACT8)(hhcb->SCSI_HP_hcntrl & SCSI_INTEN));

      OSD_SYNCHRONIZE_IOS(hhcb);

      /* Async. Event to OSM so it will invoke SCAM protocol */
      SCSI_ASYNC_EVENT(hhcb, event);

      SCSI_ASYNC_EVENT(hhcb, SCSI_AE_OSMUNFREEZE);      /* AEN call back to RSM layer */
   }
}

#endif      /* of (#if  SCSI_SCAM_ENABLE == 2)  */

/*********************************************************************
*
*  SCSIhIntSelto
*
*     Handle SCSI selection timeout
*
*  Return Value:  None
*                  
*  Parameters:    hhcb
*                 hiob
*
*  Remarks:                
*                  
*********************************************************************/
void SCSIhIntSelto (SCSI_HHCB SCSI_HPTR hhcb,
                    SCSI_HIOB SCSI_IPTR hiob)
{
   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_UNIT_CONTROL SCSI_UPTR unitHandle;

   if (hiob != SCSI_NULL_HIOB)
   { 
      unitHandle = SCSI_TARGET_UNIT(hiob);

      if (hiob->SCSI_IF_freezeOnError &&
          !(SCSI_TARGET_UNIT(hiob)->targetControl->freezeMap &
            SCSI_FREEZE_HIPRIEXISTS))
      {
         SCSI_hFREEZEHWQUEUE(hhcb,hiob);
      }
   
#if !SCSI_NEGOTIATION_PER_IOB
      /* Force negotiation as the sync and wide values are no more valid */
      if (unitHandle->deviceTable->scsiOption & (SCSI_WIDE_XFER | SCSI_SYNC_XFER))
      {
         SCSI_hXFEROPTASSIGN(hhcb,unitHandle->scsiID,SCSI_NEEDNEGO);
      }
#endif /* !SCSI_NEGOTIATION_PER_IOB */
   }
   
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSISEQ),
       (OSD_INEXACT8(SCSI_AICREG(SCSI_SCSISEQ)) & ~(SCSI_ENSELO + SCSI_ENAUTOATNO)));
#if SCSI_TRIDENT_PROTO
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_hSCSISEQ_COPY(hhcb)),
       OSD_INEXACT8(SCSI_AICREG(SCSI_SCSISEQ)));   /* TLUU - trident debug */
#endif /* SCSI_TRIDENT_PROTO */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_CLRSINT1), SCSI_CLRSELTIMO + SCSI_CLRBUSFREE);

   /* turn off LED for Non-Legacy hardware  */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_CLRSINT0), SCSI_CLRSELINGO);

   if (hiob != SCSI_NULL_HIOB)
   {
      SCSI_hTARGETCLEARBUSY(hhcb,hiob);
      hiob->haStat = SCSI_HOST_SEL_TO;
      SCSIhTerminateCommand(hiob);
   }
   else
   {
      SCSIhBadSeq(hhcb);
   }

   return;
}

/*********************************************************************
*
*  SCSIhIntFree
*
*     Acknowledge and clear SCSI Bus Free interrupt
*
*  Return Value:  None
*                  
*  Parameters:    hhcb
*                 hiob
*
*  Remarks:                
*                  
*********************************************************************/
void SCSIhIntFree (SCSI_HHCB SCSI_HPTR hhcb,
                   SCSI_HIOB SCSI_IPTR hiob)
{
   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_UINT16 count = 0xFFFF;

   /* workaround for Excallibur and maybe Bayone */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_DFCNTRL),0);
   while(count)
   {
   /* make sure HDMAEN deassert before doing FIFORESET, we see bug in Excalibur */
      if((OSD_INEXACT8(SCSI_AICREG(SCSI_DFCNTRL)) & SCSI_HDMAENACK))
         count--;
      else
         break;
   }

   /* Reset DMA & SCSI transfer logic */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_DFCNTRL),SCSI_FIFORESET);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SXFRCTL0),(OSD_INEXACT8(SCSI_AICREG(SCSI_SXFRCTL0)) 
         | (SCSI_CLRSTCNT | SCSI_CLRCHN | SCSI_SPIOEN) & ~SCSI_FAST20));

   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SIMODE1), OSD_INEXACT8(SCSI_AICREG(SCSI_SIMODE1)) & ~SCSI_ENBUSFREE);
   SCSI_hCLEARSCSIRATE(hhcb);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_CLRSINT1), SCSI_CLRBUSFREE);

   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SEQADDR0), (SCSI_UEXACT8)(SCSI_hIDLE_LOOP_ENTRY(hhcb) >> 2));
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SEQADDR1), (SCSI_UEXACT8)(SCSI_hIDLE_LOOP_ENTRY(hhcb) >> 10));

   if (hiob != SCSI_NULL_HIOB)
   {
#if (SCSI_STANDARD_160M_MODE || SCSI_SWAPPING_160M_MODE)
      if (hhcb->hardwareMode == SCSI_HMODE_AICTRIDENT)
      {
         /* Handle the case where the bus free occurred after a target */
         /* selection completed and right before the sequencer had a   */
         /* chance to disable selection.  We should allow the target   */
         /* to be selected again.                                      */
         if ((OSD_INEXACT8(SCSI_AICREG(SCSI_hWAITING_SCB(hhcb))) ==
              OSD_INEXACT8(SCSI_AICREG(SCSI_hACTIVE_SCB(hhcb)))) &&
             (OSD_INEXACT8(SCSI_AICREG(SCSI_SCSISEQ)) & SCSI_ENSELO))
         {
            return;
         }
      }
#endif /* (SCSI_STANDARD_160M_MODE || SCSI_SWAPPING_160M_MODE) */

      if (hiob->SCSI_IF_freezeOnError &&
          !(SCSI_TARGET_UNIT(hiob)->targetControl->freezeMap &
            SCSI_FREEZE_HIPRIEXISTS))
      {
         SCSI_hFREEZEHWQUEUE(hhcb,hiob);
      }
   
      SCSI_hTARGETCLEARBUSY(hhcb,hiob);
      hiob->haStat = SCSI_HOST_BUS_FREE;
      SCSIhTerminateCommand(hiob);

      /* workaround for Auto Rate/Offset issue where the scsi rate/offset */
      /* does not take affect immediately for current connection. */
      SCSI_hENABLEAUTORATEOPTION(hhcb);
   }
   return;
}

/*********************************************************************
*
*  SCSIhParityError
*
*     Handle SCSI parity errors
*
*  Return Value:  none
*                  
*  Parameters:    hhcb
*                 hiob
*
*  Remarks:                
*                  
*********************************************************************/
void SCSIhParityError (SCSI_HHCB SCSI_HPTR hhcb,
                       SCSI_HIOB SCSI_IPTR hiob)
{
   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_UEXACT8 ubyteWork;
#if !(SCSI_NEGOTIATION_PER_IOB)
   SCSI_UNIT_CONTROL SCSI_UPTR unitHandle;
#endif
#if SCSI_AICTRIDENT
   SCSI_UEXACT8 sstat2;
#endif

   if (hiob == SCSI_NULL_HIOB)
   {
      ubyteWork = OSD_INEXACT8(SCSI_AICREG(SCSI_SXFRCTL1));
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SXFRCTL1), ubyteWork & ~SCSI_ENSPCHK);
/* don't turn on parity if per iob parity checking. we need parity off in bus free*/
#if !SCSI_PARITY_PER_IOB
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SXFRCTL1), ubyteWork | SCSI_ENSPCHK);
#endif
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_CLRSINT1), SCSI_CLRSCSIPERR | SCSI_CLRATNO);
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_CLRINT), SCSI_CLRSCSINT);

      SCSIhBadSeq(hhcb);
   }
   else
   {
      if (hiob->SCSI_IF_freezeOnError &&
          !(SCSI_TARGET_UNIT(hiob)->targetControl->freezeMap &
            SCSI_FREEZE_HIPRIEXISTS))
      {
         SCSI_hFREEZEHWQUEUE(hhcb,hiob);
      }
   
      /* Initiator Detected Error message */
      hiob->SCSI_IP_workArea[SCSI_MAX_WORKAREA-1] = SCSI_MSG05;
      if ((OSD_INEXACT8(SCSI_AICREG(SCSI_SCSISIG)) & SCSI_BUSPHASE) == SCSI_MIPHASE)
      {
         /* If parity error detected during msg in phase */
         /* then the initiator will send out the Message Parity Error message */
         hiob->SCSI_IP_workArea[SCSI_MAX_WORKAREA-1] = SCSI_MSG09;
      }

#if SCSI_STANDARD_ADVANCED_MODE || SCSI_SWAPPING_ADVANCED_MODE
      if((hhcb->firmwareMode == SCSI_FMODE_STANDARD_ADVANCED) ||
         (hhcb->firmwareMode == SCSI_FMODE_SWAPPING_ADVANCED) )
      {
         hiob->SCSI_IP_workArea[SCSI_MAX_WORKAREA-2] = 
               OSD_INEXACT8(SCSI_AICREG(SCSI_SSTAT1)) & SCSI_PHASEMIS;
      }
#endif /* SCSI_STANDARD_ADVANCED_MODE  || SCSI_SWAPPING_ADVANCED_MODE */
#if SCSI_AICTRIDENT
      if(hhcb->hardwareMode == SCSI_HMODE_AICTRIDENT)
      {
         sstat2 = OSD_INEXACT8(SCSI_AICREG(SCSI_SSTAT2));
         switch(sstat2&0x0F)
         {
            case SCSI_TRIDENT_CRCVALERR:
            case SCSI_TRIDENT_CRCREQERR:
               /* Do nothing */
               break;
            case SCSI_TRIDENT_CRCENDERR:
            case SCSI_TRIDENT_DUAL_EDGE_ERR:
               /* need to send abort task message to target */
               if ((hiob->SCSI_IF_tagEnable)  && (!hiob->SCSI_IF_disallowDisconnect))
               {
                  /* Abort tagged command */
                  hiob->SCSI_IP_workArea[SCSI_MAX_WORKAREA-1] = SCSI_MSG0D;
               }
               else
               {
                  /* Abort non-tagged command */
                  hiob->SCSI_IP_workArea[SCSI_MAX_WORKAREA-1] = SCSI_MSG06;
               }
               break;
            default:
               break;

         }
      }
#endif

      if (SCSIhWait4Req(hhcb) == (SCSI_UEXACT8)-1)
      {
         SCSIhBadSeq(hhcb);
         return;
      }

#if !(SCSI_NEGOTIATION_PER_IOB)
      unitHandle = SCSI_TARGET_UNIT(hiob);
      /* need to re-negotiate with this target.    */ 
      if (unitHandle->deviceTable->scsiOption & (SCSI_WIDE_XFER | SCSI_SYNC_XFER))
      {
         SCSI_hCHKCONDXFERASSIGN(hhcb,unitHandle->scsiID,SCSI_NEEDNEGO);

         /* Set the SCSI RATE to defaults if SCSI_LOOPBACK_OPERATION = 1 */
         SCSI_hSETXFERRATE(hhcb,unitHandle->scsiID);
      }
#endif /* !SCSI_NEGOTIATION_PER_IOB */
   
      /* Turn parity checking off. */
      /* It will be turned back on in message out phase. */
      ubyteWork = OSD_INEXACT8(SCSI_AICREG(SCSI_SXFRCTL1));
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SXFRCTL1), ubyteWork & ~SCSI_ENSPCHK);

      hiob->haStat = SCSI_HOST_DETECTED_ERR;
   }
}

/*********************************************************************
*
*  SCSIhStandard64RequestSense
*
*     Perform request sense for standard 64 firmware operating mode
*
*  Return Value:  None
*                  
*  Parameters:    hhcb
*                 hiob
*
*  Remarks:                
*                  
*********************************************************************/
#if SCSI_STANDARD64_MODE
void SCSIhStandard64RequestSense (SCSI_HHCB SCSI_HPTR hhcb,
                                  SCSI_HIOB SCSI_IPTR hiob)
{
   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_UNIT_CONTROL SCSI_IPTR targetUnit = SCSI_TARGET_UNIT(hiob);
   int i;

   /* set auto sense state */
   hiob->SCSI_IP_mgrStat = SCSI_MGR_AUTOSENSE;

   /* fresh SCB ram */
   for (i = 0; i < 40; i++)
   {
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+i),0);
   }

   /* setup cdb contecnts */

   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+OSDoffsetof(SCSI_SCB_STANDARD64,scontrol)), 0x00); /* SCB01  */

   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+OSDoffsetof(SCSI_SCB_STANDARD64,starget)),
                   (SCSI_UEXACT8)targetUnit->scsiID); /* SCB01  */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+OSDoffsetof(SCSI_SCB_STANDARD64,slun)),
                   (SCSI_UEXACT8)targetUnit->lunID); /* SCB39  */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+OSDoffsetof(SCSI_SCB_STANDARD64,scdb_length)), 0x06 | SCSI_S64_ONESGSEG);  /* SCB02    */

   /* setup CDB in scb */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+OSDoffsetof(SCSI_SCB_STANDARD64,SCSI_S64_scdb0)),0x03);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+OSDoffsetof(SCSI_SCB_STANDARD64,SCSI_S64_scdb1)),
                     (SCSI_UEXACT8) (targetUnit->lunID << 5));
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+OSDoffsetof(SCSI_SCB_STANDARD64,SCSI_S64_scdb2)),0);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+OSDoffsetof(SCSI_SCB_STANDARD64,SCSI_S64_scdb3)),0);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+OSDoffsetof(SCSI_SCB_STANDARD64,SCSI_S64_scdb4)),
                     (SCSI_UEXACT8)hiob->snsLength);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+OSDoffsetof(SCSI_SCB_STANDARD64,SCSI_S64_scdb5)),0);
   
   /* do NOT have to setup s/g list pointer in scb */

   /* setup address of the first s/g element in scb */
   SCSI_hSETADDRESSSCB(hhcb,(SCSI_UEXACT8)(SCSI_SCB_BASE+
                       OSDoffsetof(SCSI_SCB_STANDARD64,saddress0)),hiob->snsBuffer);

   /* setup count of the first s/g element in scb */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+OSDoffsetof(SCSI_SCB_STANDARD64,slength0)),
           hiob->snsLength);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+OSDoffsetof(SCSI_SCB_STANDARD64,slength1)),
           0);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+OSDoffsetof(SCSI_SCB_STANDARD64,slength2)),
           0);
   
   /* clear associated busy target array */
   SCSIhStandardTargetClearBusy(hhcb,hiob);

   /* put it to head of execution queue */
   SCSIhStandard64QHead(hhcb,hiob);
}
#endif

/*********************************************************************
*
*  SCSIhSwapping32RequestSense
*
*     Perform request sense for swapping 32 firmware operating mode
*
*  Return Value:  None
*                  
*  Parameters:    hhcb
*                 hiob
*
*  Remarks:                
*                  
*********************************************************************/
#if SCSI_SWAPPING32_MODE
void SCSIhSwapping32RequestSense (SCSI_HHCB SCSI_HPTR hhcb,
                                  SCSI_HIOB SCSI_IPTR hiob)
{
   SCSI_UNIT_CONTROL SCSI_IPTR targetUnit = SCSI_TARGET_UNIT(hiob);
   SCSI_SCB_SWAPPING32 SCSI_HPTR scbW32;
   SCSI_BUS_ADDRESS busAddress;
   SCSI_UEXACT32 snsLength = (SCSI_UEXACT32) hiob->snsLength;
   int i;

   scbW32 = (SCSI_SCB_SWAPPING32 SCSI_HPTR) hiob->scbDescriptor->scbBuffer.virtualAddress;

   /* set auto sense state */
   hiob->SCSI_IP_mgrStat = SCSI_MGR_AUTOSENSE;

   /* setup scb buffer for request sense */
   /* setup 6 bytes CDB of request sense command in scb */
   SCSI_PUT_LITTLE_ENDIAN8(hhcb,scbW32,OSDoffsetof(SCSI_SCB_SWAPPING32,
            scdb_length), 0x06 | SCSI_W32_ONESGSEG);

   SCSI_PUT_LITTLE_ENDIAN8(hhcb,scbW32,OSDoffsetof(SCSI_SCB_SWAPPING32,
            SCSI_W32_scdb0),0x03);
   SCSI_PUT_LITTLE_ENDIAN8(hhcb,scbW32,OSDoffsetof(SCSI_SCB_SWAPPING32,
            SCSI_W32_scdb1),targetUnit->lunID << 5);
   SCSI_PUT_LITTLE_ENDIAN8(hhcb,scbW32,OSDoffsetof(SCSI_SCB_SWAPPING32,
            SCSI_W32_scdb2),0x00);
   SCSI_PUT_LITTLE_ENDIAN8(hhcb,scbW32,OSDoffsetof(SCSI_SCB_SWAPPING32,
            SCSI_W32_scdb3),0x00);
   SCSI_PUT_LITTLE_ENDIAN8(hhcb,scbW32,OSDoffsetof(SCSI_SCB_SWAPPING32,
            SCSI_W32_scdb4),hiob->snsLength);
   SCSI_PUT_LITTLE_ENDIAN8(hhcb,scbW32,OSDoffsetof(SCSI_SCB_SWAPPING32,
            SCSI_W32_scdb5),0x00);

   /* Setup to NOT Accept Modified Data Pointer within SEQUENCER */
   /* Must disable disconnect for request sense */
   /* Must disable tagged queuing for request sense */
   
   SCSI_PUT_LITTLE_ENDIAN8(hhcb,scbW32,OSDoffsetof(SCSI_SCB_SWAPPING32,
            scontrol),0x00);
   
   /* setup segment address and segment length */
   busAddress = OSD_GET_BUS_ADDRESS(hhcb,SCSI_MC_LOCKED,hiob->snsBuffer);
   SCSI_hPUTBUSADDRESS(hhcb,scbW32,OSDoffsetof(SCSI_SCB_SWAPPING32,
         saddress0),busAddress);

   SCSI_PUT_LITTLE_ENDIAN24(hhcb,scbW32,OSDoffsetof(SCSI_SCB_SWAPPING32,
         slength0),snsLength);

   /* clear the remainning CDB buffer */
   for (i = 6; i < 12; i++)
   {
      SCSI_PUT_LITTLE_ENDIAN8(hhcb,scbW32,
           OSDoffsetof(SCSI_SCB_SWAPPING32,SCSI_W32_scdb0)+i,0);
   }

   /* enque at head of sequencer queue */
   SCSIhSwapping32QHead(hhcb,hiob); 

   /* clear associated busy target array */
   SCSIhSwappingTargetClearBusy(hhcb,hiob); 

   SCSI_FLUSH_CACHE(scbW32,sizeof(SCSI_SCB_SWAPPING32));
}
#endif


/*********************************************************************
*
*  SCSIhSwapping64RequestSense
*
*     Perform request sense for swapping 64 firmware operating mode
*
*  Return Value:  None
*                  
*  Parameters:    hhcb
*                 hiob
*
*  Remarks:                
*                  
*********************************************************************/
#if SCSI_SWAPPING64_MODE
void SCSIhSwapping64RequestSense (SCSI_HHCB SCSI_HPTR hhcb,
                                  SCSI_HIOB SCSI_IPTR hiob)
{
   SCSI_UNIT_CONTROL SCSI_IPTR targetUnit = SCSI_TARGET_UNIT(hiob);
   SCSI_SCB_SWAPPING64 SCSI_HPTR scbW64;
   SCSI_BUS_ADDRESS busAddress;
   SCSI_UEXACT32 snsLength = (SCSI_UEXACT32) hiob->snsLength;
   int i;

   scbW64 = (SCSI_SCB_SWAPPING64 SCSI_HPTR) hiob->scbDescriptor->scbBuffer.virtualAddress;

   /* set auto sense state */
   hiob->SCSI_IP_mgrStat = SCSI_MGR_AUTOSENSE;

   /* setup scb buffer for request sense */
   /* setup 6 bytes CDB of request sense command in scb */
   SCSI_PUT_LITTLE_ENDIAN8(hhcb,scbW64,OSDoffsetof(SCSI_SCB_SWAPPING64,
            scdb_length), 0x06 | SCSI_W64_ONESGSEG);

   SCSI_PUT_LITTLE_ENDIAN8(hhcb,scbW64,OSDoffsetof(SCSI_SCB_SWAPPING64,
            SCSI_W64_scdb0),0x03);
   SCSI_PUT_LITTLE_ENDIAN8(hhcb,scbW64,OSDoffsetof(SCSI_SCB_SWAPPING64,
            SCSI_W64_scdb1),targetUnit->lunID << 5);
   SCSI_PUT_LITTLE_ENDIAN8(hhcb,scbW64,OSDoffsetof(SCSI_SCB_SWAPPING64,
            SCSI_W64_scdb2),0x00);
   SCSI_PUT_LITTLE_ENDIAN8(hhcb,scbW64,OSDoffsetof(SCSI_SCB_SWAPPING64,
            SCSI_W64_scdb3),0x00);
   SCSI_PUT_LITTLE_ENDIAN8(hhcb,scbW64,OSDoffsetof(SCSI_SCB_SWAPPING64,
            SCSI_W64_scdb4),hiob->snsLength);
   SCSI_PUT_LITTLE_ENDIAN8(hhcb,scbW64,OSDoffsetof(SCSI_SCB_SWAPPING64,
            SCSI_W64_scdb5),0x00);

   /* Setup to NOT Accept Modified Data Pointer within SEQUENCER */
   /* Must disable disconnect for request sense */
   /* Must disable tagged queuing for request sense */

   SCSI_PUT_LITTLE_ENDIAN8(hhcb,scbW64,OSDoffsetof(SCSI_SCB_SWAPPING64,
            scontrol),0x00);

   /* setup segment address and segment length */
   busAddress = OSD_GET_BUS_ADDRESS(hhcb,SCSI_MC_LOCKED,hiob->snsBuffer);
   SCSI_hPUTBUSADDRESS(hhcb,scbW64,OSDoffsetof(SCSI_SCB_SWAPPING64,
         SCSI_W64_saddress0),busAddress);

   SCSI_PUT_LITTLE_ENDIAN24(hhcb,scbW64,OSDoffsetof(SCSI_SCB_SWAPPING64,
         SCSI_W64_slength0),snsLength);

   /* clear the remainning CDB buffer */
   for (i = 6; i < 12; i++)
   {
      SCSI_PUT_LITTLE_ENDIAN8(hhcb,scbW64,
           OSDoffsetof(SCSI_SCB_SWAPPING64,SCSI_W64_scdb0)+i,0);
   }

   /* enque at head of sequencer queue */
   SCSIhSwapping64QHead(hhcb,hiob); 

   /* clear associated busy target array */
   SCSIhSwappingTargetClearBusy(hhcb,hiob); 

   SCSI_FLUSH_CACHE(scbW64,sizeof(SCSI_SCB_SWAPPING64));
}
#endif

/*********************************************************************
*
*  SCSIhStandardAdvRequestSense
*
*     Perform request sense for standard advanced firmware operating mode
*
*  Return Value:  None
*                  
*  Parameters:    hhcb
*                 hiob
*
*  Remarks:                
*                  
*********************************************************************/
#if SCSI_STANDARD_ADVANCED_MODE
void SCSIhStandardAdvRequestSense (SCSI_HHCB SCSI_HPTR hhcb,
                                  SCSI_HIOB SCSI_IPTR hiob)
{
   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_UNIT_CONTROL SCSI_IPTR targetUnit = SCSI_TARGET_UNIT(hiob);
   int i;
   SCSI_UEXACT8 starget=0;

   /* The enhanced sequencer will put the HA scsi ID into the hi nybble of */
   /* starget after the SCB is in the SCB RAM.  Since this routine will PIO */
   /* the scb directly to the RAM, we must do the same. */
   starget = hhcb->hostScsiID;
   starget <<= 4;
   starget |= (SCSI_UEXACT8)targetUnit->scsiID;

   /* set auto sense state */
   hiob->SCSI_IP_mgrStat = SCSI_MGR_AUTOSENSE;

   /* fresh SCB ram */
   for (i = 0; i < 40; i++)
   {
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+i),0);
   }

   /* setup cdb contecnts */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
         OSDoffsetof(SCSI_SCB_STANDARD_ADVANCED,scontrol)), 
         0x00);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
         OSDoffsetof(SCSI_SCB_STANDARD_ADVANCED,starget)),
         (SCSI_UEXACT8)starget);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
         OSDoffsetof(SCSI_SCB_STANDARD_ADVANCED,slun)),
         (SCSI_UEXACT8)targetUnit->lunID);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
         OSDoffsetof(SCSI_SCB_STANDARD_ADVANCED,scdb_length)),0x06);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
         OSDoffsetof(SCSI_SCB_STANDARD_ADVANCED,sg_cache_SCB)), 
         SCSI_SADV_ONESGSEG);

   /* setup CDB in scb */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
         OSDoffsetof(SCSI_SCB_STANDARD_ADVANCED,SCSI_SADV_scdb0)),0x03);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
         OSDoffsetof(SCSI_SCB_STANDARD_ADVANCED,SCSI_SADV_scdb1)),
         (SCSI_UEXACT8) (targetUnit->lunID << 5));
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
         OSDoffsetof(SCSI_SCB_STANDARD_ADVANCED,SCSI_SADV_scdb2)),0);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
         OSDoffsetof(SCSI_SCB_STANDARD_ADVANCED,SCSI_SADV_scdb3)),0);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
         OSDoffsetof(SCSI_SCB_STANDARD_ADVANCED,SCSI_SADV_scdb4)),
         (SCSI_UEXACT8)hiob->snsLength);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
         OSDoffsetof(SCSI_SCB_STANDARD_ADVANCED,SCSI_SADV_scdb5)),0);
   
   /* do NOT have to setup s/g list pointer in scb */

   /* setup address of the first s/g element in scb */
   SCSI_hSETADDRESSSCB(hhcb,(SCSI_UEXACT8)(SCSI_SCB_BASE+
         OSDoffsetof(SCSI_SCB_STANDARD_ADVANCED,saddress0)),hiob->snsBuffer);

   /* setup count of the first s/g element in scb */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
         OSDoffsetof(SCSI_SCB_STANDARD_ADVANCED,slength0)),
         hiob->snsLength);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
         OSDoffsetof(SCSI_SCB_STANDARD_ADVANCED,slength1)),0);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
         OSDoffsetof(SCSI_SCB_STANDARD_ADVANCED,slength2)),0);
   
   /* clear associated busy target array */
   SCSIhStandardTargetClearBusy(hhcb,hiob);

   /* put it to head of execution queue */
   SCSIhStandardAdvQHead(hhcb,hiob);
}
#endif

/*********************************************************************
*
*  SCSIhSwappingAdvRequestSense
*
*     Perform request sense for swapping advanced firmware operating mode
*
*  Return Value:  None
*                  
*  Parameters:    hhcb
*                 hiob
*
*  Remarks:                
*                  
*********************************************************************/
#if SCSI_SWAPPING_ADVANCED_MODE
void SCSIhSwappingAdvRequestSense (SCSI_HHCB SCSI_HPTR hhcb,
                                  SCSI_HIOB SCSI_IPTR hiob)
{
   SCSI_UNIT_CONTROL SCSI_IPTR targetUnit = SCSI_TARGET_UNIT(hiob);
   SCSI_SCB_SWAPPING_ADVANCED SCSI_HPTR scbWAdv;
   SCSI_BUS_ADDRESS busAddress;
   SCSI_UEXACT32 snsLength = (SCSI_UEXACT32) hiob->snsLength;
   int i;
   SCSI_UEXACT8 byteBuf;

   scbWAdv = (SCSI_SCB_SWAPPING_ADVANCED SCSI_HPTR) hiob->scbDescriptor->scbBuffer.virtualAddress;

   /* The enhanced sequencer will put the HA scsi ID into the hi nibble of */
   /* starget after the SCB is in the SCB RAM.  We need to mask out the hi */
   /* nibble of starget for the request sense command */
   SCSI_GET_LITTLE_ENDIAN8(hhcb, &byteBuf, scbWAdv,
            OSDoffsetof(SCSI_SCB_SWAPPING_ADVANCED, starget));
   byteBuf &= 0x0F;
   SCSI_PUT_LITTLE_ENDIAN8(hhcb,scbWAdv,OSDoffsetof(SCSI_SCB_SWAPPING_ADVANCED,
            starget), byteBuf);

   /* set auto sense state */
   hiob->SCSI_IP_mgrStat = SCSI_MGR_AUTOSENSE;

   /* setup scb buffer for request sense */
   /* setup 6 bytes CDB of request sense command in scb */
   SCSI_PUT_LITTLE_ENDIAN8(hhcb,scbWAdv,OSDoffsetof(SCSI_SCB_SWAPPING_ADVANCED,
            scdb_length), 0x06);
   SCSI_PUT_LITTLE_ENDIAN8(hhcb,scbWAdv,OSDoffsetof(SCSI_SCB_SWAPPING_ADVANCED,
            sg_cache_SCB), SCSI_WADV_ONESGSEG);

   SCSI_PUT_LITTLE_ENDIAN8(hhcb,scbWAdv,OSDoffsetof(SCSI_SCB_SWAPPING_ADVANCED,
            SCSI_WADV_scdb0),0x03);
   SCSI_PUT_LITTLE_ENDIAN8(hhcb,scbWAdv,OSDoffsetof(SCSI_SCB_SWAPPING_ADVANCED,
            SCSI_WADV_scdb1),targetUnit->lunID << 5);
   SCSI_PUT_LITTLE_ENDIAN8(hhcb,scbWAdv,OSDoffsetof(SCSI_SCB_SWAPPING_ADVANCED,
            SCSI_WADV_scdb2),0x00);
   SCSI_PUT_LITTLE_ENDIAN8(hhcb,scbWAdv,OSDoffsetof(SCSI_SCB_SWAPPING_ADVANCED,
            SCSI_WADV_scdb3),0x00);
   SCSI_PUT_LITTLE_ENDIAN8(hhcb,scbWAdv,OSDoffsetof(SCSI_SCB_SWAPPING_ADVANCED,
            SCSI_WADV_scdb4),hiob->snsLength);
   SCSI_PUT_LITTLE_ENDIAN8(hhcb,scbWAdv,OSDoffsetof(SCSI_SCB_SWAPPING_ADVANCED,
            SCSI_WADV_scdb5),0x00);

   /* Setup to NOT Accept Modified Data Pointer within SEQUENCER */
   /* Must disable disconnect for request sense */
   /* Must disable tagged queuing for request sense */
   SCSI_GET_LITTLE_ENDIAN8(hhcb, &byteBuf, scbWAdv,
            OSDoffsetof(SCSI_SCB_SWAPPING_ADVANCED, slun));
   byteBuf &= ~SCSI_WADV_DISCENB;
   SCSI_PUT_LITTLE_ENDIAN8(hhcb,scbWAdv,OSDoffsetof(SCSI_SCB_SWAPPING_ADVANCED,
            slun), byteBuf);
   SCSI_PUT_LITTLE_ENDIAN8(hhcb,scbWAdv,OSDoffsetof(SCSI_SCB_SWAPPING_ADVANCED,
            scontrol),0x00);

   /* setup segment address and segment length */
   busAddress = OSD_GET_BUS_ADDRESS(hhcb,SCSI_MC_LOCKED,hiob->snsBuffer);
   SCSI_hPUTBUSADDRESS(hhcb,scbWAdv,OSDoffsetof(SCSI_SCB_SWAPPING_ADVANCED,
         saddress0),busAddress);

   SCSI_PUT_LITTLE_ENDIAN24(hhcb,scbWAdv,OSDoffsetof(SCSI_SCB_SWAPPING_ADVANCED,
         slength0),snsLength);

   /* clear the remainning CDB buffer */
   for (i = 6; i < 12; i++)
   {
      SCSI_PUT_LITTLE_ENDIAN8(hhcb,scbWAdv,
           OSDoffsetof(SCSI_SCB_SWAPPING_ADVANCED,SCSI_WADV_scdb0)+i,0);
   }

   /* enqueue at head of sequencer queue */
   SCSIhSwappingAdvQHead(hhcb,hiob); 

   /* clear associated busy target array */
   SCSIhSwappingTargetClearBusy(hhcb,hiob); 

   SCSI_FLUSH_CACHE(scbWAdv,sizeof(SCSI_SCB_SWAPPING_ADVANCED));
}
#endif

/*********************************************************************
*
*  SCSIhStandard160mRequestSense
*
*     Perform request sense for standard 160m firmware operating mode
*
*  Return Value:  None
*                  
*  Parameters:    hhcb
*                 hiob
*
*  Remarks:                
*                  
*********************************************************************/
#if SCSI_STANDARD_160M_MODE
void SCSIhStandard160mRequestSense (SCSI_HHCB SCSI_HPTR hhcb,
                                    SCSI_HIOB SCSI_IPTR hiob)
{
   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_UNIT_CONTROL SCSI_IPTR targetUnit = SCSI_TARGET_UNIT(hiob);
   SCSI_UEXACT8 starget = 0;
   int i;

   /* The sequencer will put the HA scsi ID into the hi nibble of starget */
   /* after the SCB is in the SCB RAM.  Since this routine will PIO       */
   /* the scb directly to the RAM, we must do the same. */
   starget = hhcb->hostScsiID;
   starget <<= 4;
   starget |= (SCSI_UEXACT8)targetUnit->scsiID;

   /* set auto sense state */
   hiob->SCSI_IP_mgrStat = SCSI_MGR_AUTOSENSE;

   /* fresh SCB ram */
   for (i = 0; i < 40; i++)
   {
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+i),0);
   }

   /* setup cdb contecnts */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
       OSDoffsetof(SCSI_SCB_STANDARD_160M,scontrol)), 0);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
       OSDoffsetof(SCSI_SCB_STANDARD_160M,starget)), (SCSI_UEXACT8)starget);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
       OSDoffsetof(SCSI_SCB_STANDARD_160M,slun)),
       (SCSI_UEXACT8)targetUnit->lunID);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
       OSDoffsetof(SCSI_SCB_STANDARD_160M,scdb_length)),0x06);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
       OSDoffsetof(SCSI_SCB_STANDARD_160M,sg_cache_SCB)), SCSI_S160M_ONESGSEG);

   /* setup CDB in scb */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
       OSDoffsetof(SCSI_SCB_STANDARD_160M,SCSI_S160M_scdb0)),0x03);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
       OSDoffsetof(SCSI_SCB_STANDARD_160M,SCSI_S160M_scdb1)),
       (SCSI_UEXACT8) (targetUnit->lunID << 5));
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
       OSDoffsetof(SCSI_SCB_STANDARD_160M,SCSI_S160M_scdb2)),0);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
       OSDoffsetof(SCSI_SCB_STANDARD_160M,SCSI_S160M_scdb3)),0);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
       OSDoffsetof(SCSI_SCB_STANDARD_160M,SCSI_S160M_scdb4)),
       (SCSI_UEXACT8)hiob->snsLength);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
       OSDoffsetof(SCSI_SCB_STANDARD_160M,SCSI_S160M_scdb5)),0);
   
   /* do NOT have to setup s/g list pointer in scb */

   /* setup address of the first s/g element in scb */
   SCSI_hSETADDRESSSCB(hhcb,(SCSI_UEXACT8)(SCSI_SCB_BASE+
        OSDoffsetof(SCSI_SCB_STANDARD_160M,saddress0)),hiob->snsBuffer);

   /* setup count of the first s/g element in scb */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
       OSDoffsetof(SCSI_SCB_STANDARD_160M,slength0)), hiob->snsLength);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
       OSDoffsetof(SCSI_SCB_STANDARD_160M,slength1)),0);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
       OSDoffsetof(SCSI_SCB_STANDARD_160M,slength2)),0);
   
   /* clear associated busy target array */
   SCSIhStandardTargetClearBusy(hhcb,hiob);

   /* put it to head of execution queue */
   SCSIhStandard160mQHead(hhcb,hiob);
}
#endif /* SCSI_STANDARD_160M_MODE */

/*********************************************************************
*
*  SCSIhSwapping160mRequestSense
*
*     Perform request sense for swapping 160m firmware operating mode
*
*  Return Value:  None
*                  
*  Parameters:    hhcb
*                 hiob
*
*  Remarks:                
*                  
*********************************************************************/
#if SCSI_SWAPPING_160M_MODE
void SCSIhSwapping160mRequestSense (SCSI_HHCB SCSI_HPTR hhcb,
                                    SCSI_HIOB SCSI_IPTR hiob)
{
   SCSI_UNIT_CONTROL SCSI_IPTR targetUnit = SCSI_TARGET_UNIT(hiob);
   SCSI_SCB_SWAPPING_160M SCSI_HPTR scbW160m;
   SCSI_BUS_ADDRESS busAddress;
   SCSI_UEXACT32 snsLength = (SCSI_UEXACT32) hiob->snsLength;
   int i;
   SCSI_UEXACT8 byteBuf;

   scbW160m = (SCSI_SCB_SWAPPING_160M SCSI_HPTR) hiob->scbDescriptor->scbBuffer.virtualAddress;

   /* The sequencer will put the HA scsi ID into the hi nibble of starget */
   /* after the SCB is in the SCB RAM.  We need to mask out the hi nibble */
   /* of starget for the request sense command. */
   SCSI_GET_LITTLE_ENDIAN8(hhcb,&byteBuf,scbW160m,
        OSDoffsetof(SCSI_SCB_SWAPPING_160M,starget));
   byteBuf &= 0x0F;
   SCSI_PUT_LITTLE_ENDIAN8(hhcb,scbW160m,
        OSDoffsetof(SCSI_SCB_SWAPPING_160M,starget),byteBuf);

   /* set auto sense state */
   hiob->SCSI_IP_mgrStat = SCSI_MGR_AUTOSENSE;

   /* setup scb buffer for request sense */
   /* setup 6 bytes CDB of request sense command in scb */
   SCSI_PUT_LITTLE_ENDIAN8(hhcb,scbW160m,
        OSDoffsetof(SCSI_SCB_SWAPPING_160M,scdb_length),0x06);
   SCSI_PUT_LITTLE_ENDIAN8(hhcb,scbW160m,
        OSDoffsetof(SCSI_SCB_SWAPPING_160M,sg_cache_SCB),SCSI_W160M_ONESGSEG);

   SCSI_PUT_LITTLE_ENDIAN8(hhcb,scbW160m,
        OSDoffsetof(SCSI_SCB_SWAPPING_160M,SCSI_W160M_scdb0),0x03);
   SCSI_PUT_LITTLE_ENDIAN8(hhcb,scbW160m,
        OSDoffsetof(SCSI_SCB_SWAPPING_160M,SCSI_W160M_scdb1),
        targetUnit->lunID << 5);
   SCSI_PUT_LITTLE_ENDIAN8(hhcb,scbW160m,
        OSDoffsetof(SCSI_SCB_SWAPPING_160M,SCSI_W160M_scdb2),0x00);
   SCSI_PUT_LITTLE_ENDIAN8(hhcb,scbW160m,
        OSDoffsetof(SCSI_SCB_SWAPPING_160M,SCSI_W160M_scdb3),0x00);
   SCSI_PUT_LITTLE_ENDIAN8(hhcb,scbW160m,
        OSDoffsetof(SCSI_SCB_SWAPPING_160M,SCSI_W160M_scdb4),hiob->snsLength);
   SCSI_PUT_LITTLE_ENDIAN8(hhcb,scbW160m,
        OSDoffsetof(SCSI_SCB_SWAPPING_160M,SCSI_W160M_scdb5),0x00);

   /* Setup to NOT Accept Modified Data Pointer within SEQUENCER */
   /* Must disable disconnect for request sense */
   /* Must disable tagged queuing for request sense */
   SCSI_GET_LITTLE_ENDIAN8(hhcb,&byteBuf,scbW160m,
        OSDoffsetof(SCSI_SCB_SWAPPING_160M,slun));
   byteBuf &= ~SCSI_W160M_DISCENB;
   SCSI_PUT_LITTLE_ENDIAN8(hhcb,scbW160m,
        OSDoffsetof(SCSI_SCB_SWAPPING_160M,slun),byteBuf);
   SCSI_PUT_LITTLE_ENDIAN8(hhcb,scbW160m,
        OSDoffsetof(SCSI_SCB_SWAPPING_160M,scontrol),0x00);

   /* setup segment address and segment length */
   busAddress = OSD_GET_BUS_ADDRESS(hhcb,SCSI_MC_LOCKED,hiob->snsBuffer);
   SCSI_hPUTBUSADDRESS(hhcb,scbW160m,
        OSDoffsetof(SCSI_SCB_SWAPPING_160M,saddress0),busAddress);

   SCSI_PUT_LITTLE_ENDIAN24(hhcb,scbW160m,
        OSDoffsetof(SCSI_SCB_SWAPPING_160M,slength0),snsLength);

   /* clear the remainning CDB buffer */
   for (i = 6; i < 12; i++)
   {
      SCSI_PUT_LITTLE_ENDIAN8(hhcb,scbW160m,
           OSDoffsetof(SCSI_SCB_SWAPPING_160M,SCSI_W160M_scdb0)+i,0);
   }

   /* enque at head of sequencer queue */
   SCSIhSwapping160mQHead(hhcb,hiob); 

   /* clear associated busy target array */
   SCSIhSwappingTargetClearBusy(hhcb,hiob); 

   SCSI_FLUSH_CACHE(scbW160m,sizeof(SCSI_SCB_SWAPPING_160M));
}
#endif /* SCSI_SWAPPING_160M_MODE */

/*********************************************************************
*
*  SCSIhTerminateCommand
*
*     Terminate the specified hiob
*
*  Return Value:  None
*                  
*  Parameters:    hiob
*
*  Remarks:                
*                  
*********************************************************************/
void SCSIhTerminateCommand (SCSI_HIOB SCSI_IPTR hiob)
{
   SCSI_UNIT_CONTROL SCSI_IPTR targetUnit = SCSI_TARGET_UNIT(hiob);
   SCSI_HHCB SCSI_HPTR hhcb = targetUnit->hhcb;
   SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
#if !SCSI_NEGOTIATION_PER_IOB
   SCSI_UEXACT8 hcntrl;
#endif /* !SCSI_NEGOTIATION_PER_IOB */

   /* free the associated entry in active pointer array */
   SCSI_ACTPTR[hiob->scbNumber] = SCSI_NULL_HIOB;

   /* make sure proper status get passed back */
   SCSIhSetStat(hiob);

   if (hiob->cmd == SCSI_CMD_INITIATE_TASK)
   {
      /* post it back to upper layer code */
      SCSI_COMPLETE_HIOB(hiob);
   }
   else
   {
      /* Check and set if scsi negotiation is needed for */
      /* a particular target after the bus device reset executed */
      if (hiob->cmd == SCSI_CMD_RESET_TARGET)
      {
#if !SCSI_NEGOTIATION_PER_IOB
         /* save the HCNTRL value */
         hcntrl = OSD_INEXACT8(SCSI_AICREG(SCSI_HCNTRL));

         /* pause the chip if necessary */
         if (!(hcntrl & SCSI_PAUSEACK))
         {
            SCSIhPauseAndWait(hhcb);
         }

         if (targetUnit->deviceTable->scsiOption & (SCSI_WIDE_XFER | SCSI_SYNC_XFER))
         {
            /*  Set negotiation needed */
            SCSI_hXFEROPTASSIGN(hhcb, targetUnit->scsiID, SCSI_NEEDNEGO);
         }
         else
         {
            SCSI_hXFEROPTASSIGN(hhcb, targetUnit->scsiID, 0x00);
         }

         /* Set the SCSI RATE to defaults if SCSI_LOOPBACK_OPERATION = 1 */
         SCSI_hSETXFERRATE(hhcb,targetUnit->scsiID);

         /* restore HCNTRL if necessary */
         if (!(hcntrl & SCSI_PAUSEACK))
         {
            SCSI_hWRITE_HCNTRL(hhcb,scsiRegister,hcntrl);
         }
#endif /* !SCSI_NEGOTIATION_PER_IOB */
      }

#if  SCSI_SWAPPING_160M_MODE
     if (hhcb->hardwareMode == SCSI_HMODE_AICTRIDENT)
     {
        SCSI_SCB_SWAPPING_160M SCSI_HPTR scbW160m;
        SCSI_UEXACT8 byteBuf;

        /* Prepare scb to be aborted via 'aborted' bit of scontrol */
        scbW160m = (SCSI_SCB_SWAPPING_160M SCSI_HPTR) hiob->scbDescriptor->scbBuffer.virtualAddress;
        SCSI_GET_LITTLE_ENDIAN8(hhcb, &byteBuf, scbW160m,
           OSDoffsetof(SCSI_SCB_SWAPPING_160M, scontrol));
        if (byteBuf & SCSI_W160M_ABORTED)
        {
           hhcb->numberSCBAborts--;
           if (hhcb->numberSCBAborts == 0)
              OSD_OUTEXACT8(SCSI_AICREG(SCSI_W160M_CHECK_ABORT), 0);

        }
		
        /* Flush the cache */
        SCSI_FLUSH_CACHE(scbW160m, sizeof(SCSI_SCB_SWAPPING_160M));
     }
#endif
      /* post back special HIOB to upper layer code */
      SCSI_COMPLETE_SPECIAL_HIOB(hiob);
   }
}

/*********************************************************************
*
*  SCSIhSetStat
*
*     Set status value to MgrStat
*
*  Return Value:  None
*                  
*  Parameters:    hiob
*
*  Remarks:
*                  
*********************************************************************/
void SCSIhSetStat (SCSI_HIOB SCSI_IPTR hiob)
{
   if (hiob->haStat || ((hiob->trgStatus != SCSI_UNIT_GOOD) &&
                        (hiob->trgStatus != SCSI_UNIT_MET) &&
                        (hiob->trgStatus != SCSI_UNIT_INTERMED) &&
                        (hiob->trgStatus != SCSI_UNIT_INTMED_GD)))
   {
      if (hiob->stat != SCSI_SCB_ABORTED)
      {
         /* The HIOB is really completed.  But the haStat was set during */
         /* active abort process.  And again check for the trgStatus. */
         if (((hiob->haStat == SCSI_HOST_ABT_HOST) ||
              (hiob->haStat == SCSI_HOST_ABT_TRG_RST)) &&
             ((hiob->trgStatus == SCSI_UNIT_GOOD) ||
              (hiob->trgStatus == SCSI_UNIT_MET) ||
              (hiob->trgStatus == SCSI_UNIT_INTERMED) ||
              (hiob->trgStatus == SCSI_UNIT_INTMED_GD)))
         {
            hiob->stat = SCSI_SCB_COMP;   /* HIOB completed without error */
         }
         else
         {
            hiob->stat = SCSI_SCB_ERR;    /* HIOB completed with error */
         }
      }

      else
      {
         /* Check for the terminate HIOB due to the selection time-out, */
         /* unexpected busfree, or other target errors: queue full etc. */
         if ((hiob->haStat != SCSI_HOST_ABT_HOST) &&
             (hiob->haStat != SCSI_HOST_ABT_TRG_RST))
         {
            hiob->stat = SCSI_SCB_ERR;    /* HIOB completed with error */
         }
         else if ((hiob->trgStatus != SCSI_UNIT_GOOD) &&
                  (hiob->trgStatus != SCSI_UNIT_MET) &&
                  (hiob->trgStatus != SCSI_UNIT_INTERMED) &&
                  (hiob->trgStatus != SCSI_UNIT_INTMED_GD))
         {
            /* Need to clear haStat because the error is the trgStatus */
            hiob->haStat = 0;
            hiob->stat = SCSI_SCB_ERR;    /* HIOB completed with error */
         }

      }
   }
   else
   {
      hiob->stat = SCSI_SCB_COMP;           /* Update status */
   }
}

/*********************************************************************
*
*  SCSIhBreakInterrupt
*
*     Trap for break interrupt
*
*  Return Value:  None
*                  
*  Parameters:    hhcb
*                 hiob
*
*  Remarks:       This routine will search through active pointer array
*                 for any HIOB need to be aborted or enqueue to head
*                 of the queue if it's Bus_Device_Reset HIOB.
*                  
*********************************************************************/
void SCSIhBreakInterrupt (SCSI_HHCB SCSI_HPTR hhcb,
                           SCSI_HIOB SCSI_IPTR hiobActive)
{
   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_HIOB SCSI_IPTR hiob;
   SCSI_DEVICE SCSI_DPTR deviceTable; 
   SCSI_UINT i;
#if SCSI_TARGET_OPERATION 
   SCSI_HHCB SCSI_HPTR aborthhcb;
#endif  /* SCSI_TARGET_OPERATION */   

   if (hhcb->SCSI_HP_expRequest)
   {
      if (hhcb->SCSI_HP_nonExpBreak)
      {
         /* Clear the idle loop break point interrupt */
         SCSI_hCLEARBREAKPOINT(hhcb, (SCSI_UEXACT8)ENTRY_IDLE_LOOP);

         /* have to restore exp status break */
         SCSI_hSETBREAKPOINT(hhcb, (SCSI_UEXACT8)ENTRY_EXPANDER_BREAK);
      }
      else
      {
         if (hiobActive == SCSI_NULL_HIOB)
         {
            /* Clear the expander active break point interrupt */
            SCSI_hCLEARBREAKPOINT(hhcb, (SCSI_UEXACT8)ENTRY_EXPANDER_BREAK);

            SCSIhBadSeq(hhcb);
            return;
         }
         /* examine the expander status */
         deviceTable = SCSI_TARGET_UNIT(hiobActive)->deviceTable;
         deviceTable->SCSI_DF_behindExp = ((OSD_INEXACT8(SCSI_AICREG(SCSI_SSTAT2)) & SCSI_BAYONET_EXP_ACTIVE) != 0);

         /* Clear the expander active break point interrupt */
         SCSI_hCLEARBREAKPOINT(hhcb, (SCSI_UEXACT8)ENTRY_EXPANDER_BREAK);
      }
   }
   else
   {
      /* Clear the idle loop break point interrupt */
      SCSI_hCLEARBREAKPOINT(hhcb, (SCSI_UEXACT8)ENTRY_IDLE_LOOP);
   }

   if (hhcb->SCSI_HP_nonExpBreak)
   {
      /* clear non exp break flag */
      hhcb->SCSI_HP_nonExpBreak = 0;

      /* check if target mode DISABLE_ID is active */
      SCSI_hTARGETDISABLEID(hhcb);

      /* Search all HIOB in active pointer array */
      for (i = 0; i < hhcb->numberScbs; i++)
      {
         if ((hiob = SCSI_ACTPTR[i]) != SCSI_NULL_HIOB)
         {
            /* Make sure the request is for the current host adapter */
#if SCSI_TARGET_OPERATION       
            aborthhcb = 
               (hiob->SCSI_IP_targetMode)? SCSI_NEXUS_UNIT(hiob)->hhcb : SCSI_TARGET_UNIT(hiob)->hhcb;
            if (aborthhcb == hhcb)
#else
            if (SCSI_TARGET_UNIT(hiob)->hhcb == hhcb)
#endif  /* SCSI_TARGET_OPERATION */   
            {
               if (hiob->SCSI_IP_mgrStat == SCSI_MGR_ABORTINREQ)
               {
                  SCSI_hACTIVEABORT(hhcb, hiob);
               }
               else if (hiob->SCSI_IP_mgrStat == SCSI_MGR_ENQUE_BDR)
               {
                  SCSI_hENQUEHEADBDR(hhcb, hiob);
   
                  /* Set Bus_Device_Reset state */
                  hiob->SCSI_IP_mgrStat = SCSI_MGR_BDR;
               }
            }
         }
      }

   }
   
}

/*********************************************************************
*
*  SCSIhStandard64ActiveAbort
*
*     Abort the active HIOB for Standard mode sequencer.
*
*  Return Value:  None
*
*  Parameters:    hhcb
*                 aborting hiob
*
*  Remarks:       For the case of the target's disconnection, it might
*                 reconnected before the abort message HIOB has ever
*                 started or done on SCSI selection.  This issues is
*                 addressed in two places:
*                 1. In SCSIhAbortTarget routine will try to remove
*                    the HIOB from the internal sequencer Done queue.
*                 2. In SCSIhCmdComplete routine to not post it back.
*
*********************************************************************/
#if SCSI_STANDARD64_MODE
void SCSIhStandard64ActiveAbort (SCSI_HHCB SCSI_HPTR hhcb,
                                 SCSI_HIOB SCSI_IPTR hiob)
{
   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_UEXACT8 scbptr;
   SCSI_UEXACT8 byteBuf;

   /* If the SCB to be aborted is currently running, just let it finish */
   if ((hiob->scbNumber == OSD_INEXACT8(SCSI_AICREG(SCSI_S64_ACTIVE_SCB))) &&
       ((OSD_INEXACT8(SCSI_AICREG(SCSI_CCSGCTL)) & SCSI_CCSGEN) ||
        (OSD_INEXACT8(SCSI_AICREG(SCSI_CCSCBCTL)) & (SCSI_CCARREN | SCSI_CCSCBEN)) ||
        (OSD_INEXACT8(SCSI_AICREG(SCSI_DFCNTRL)) & (SCSI_HDMAEN | SCSI_SDMAEN | SCSI_SCSIEN))))
   {
      /* Need to clear stat because the command will be completed normally */
      hiob->stat = 0;
      hiob->SCSI_IP_mgrStat = SCSI_MGR_DONE;
   }
   /* Check if aborting HIOB is currently waiting. */
   /* The waiting SCB can be either in: */
   /*   1. The process of selecting the target or */
   /*   2. Reselection occurred while selecting the target */
   else if ((OSD_INEXACT8(SCSI_AICREG(SCSI_SCSISEQ)) & SCSI_ENSELO) &&
            (hiob->scbNumber == OSD_INEXACT8(SCSI_AICREG(SCSI_S64_WAITING_SCB))))
   {
      /* Set aborted status is done. */
      /* The stat and haStat are set before breakpoint interrupt. */
      hiob->SCSI_IP_mgrStat = SCSI_MGR_DONE_ABT;

      /* Prepare scb to be aborted by setting 'aborted' bit in SCB ram */
      scbptr = OSD_INEXACT8(SCSI_AICREG(SCSI_SCBPTR));   /* Save SCBPTR */
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCBPTR), hiob->scbNumber);
      byteBuf = OSD_INEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
                    OSDoffsetof(SCSI_SCB_STANDARD64, chain_control)));
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+OSDoffsetof(SCSI_SCB_STANDARD64,
          chain_control)), byteBuf | SCSI_S64_ABORTED);
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCBPTR), scbptr);   /* Restore SCBPTR */
   }

   /* Check if the aborting HIOB is currently at: */
   /*   1. Execution queue - waiting to be executed */
   /*   2. Completion - either in Done queue or Sequencer Done queue */
   /*   3. Disconnection - in SCB ram */

   /* The aborting HIOB will be posted back in these routines if it found */
   else if (!SCSIhStandard64SearchExeQ(hhcb, hiob, hiob->haStat, 1) &&
            !SCSIhStandardSearchSeqDoneQ(hhcb, hiob, 1) &&
            !SCSIhStandardSearchDoneQ(hhcb, hiob))
   {
      /* If the aborting HIOB has not been completed yet, we will  */
      /* issue the abort message to the target just in case the    */
      /* target disconnected and never come back which will result */
      /* in selection time out.  On the other hand, the command    */
      /* will be executed and abort message will be sent to target.*/

      /* Prepare scb to be aborted via special function */
      scbptr = OSD_INEXACT8(SCSI_AICREG(SCSI_SCBPTR));   /* Save SCBPTR */
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCBPTR), hiob->scbNumber);

      /* Prepare scb to be aborted by setting 'aborted' bit in SCB ram */
      byteBuf = OSD_INEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
                 OSDoffsetof(SCSI_SCB_STANDARD64, chain_control)));
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+OSDoffsetof(SCSI_SCB_STANDARD64,
          chain_control)), byteBuf | SCSI_S64_ABORTED);

      /* Setup special function bit - Bit 3 of the scontrol flag */
      byteBuf = OSD_INEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
                    OSDoffsetof(SCSI_SCB_STANDARD64, scontrol)));
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+OSDoffsetof(SCSI_SCB_STANDARD64,
          scontrol)), byteBuf | SCSI_S64_SPECFUN);

      /* Setup special opcode - msg_to_targ */
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+OSDoffsetof(SCSI_SCB_STANDARD64,
          SCSI_S64_special_opcode)), SCSI_S64_MSGTOTARG);

      /* Setup message byte - special_info */
     
      if ((hiob->SCSI_IF_tagEnable)  && (!hiob->SCSI_IF_disallowDisconnect))
      {
         /* Abort tagged command */
         OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+OSDoffsetof(SCSI_SCB_STANDARD64,
             SCSI_S64_special_info)), SCSI_MSG0D);
      }
      else
      {
         /* Abort non-tagged command */
         OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+OSDoffsetof(SCSI_SCB_STANDARD64,
             SCSI_S64_special_info)), SCSI_MSG06);
      }

      /* Enqueue at head of the execution queue */
      SCSIhStandard64QHead(hhcb, hiob);

      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCBPTR), scbptr);   /* Restore SCBPTR */

      /* Indicate abort is in progress */
      hiob->SCSI_IP_mgrStat = SCSI_MGR_ABORTINPROG;
   }  /* end of else if not in both Done queues */
}
#endif   /* SCSI_STANDARD64_MODE */

/*********************************************************************
*
*  SCSIhSwapping32ActiveAbort
*
*     Abort the active HIOB for Swapping mode sequencer.
*
*  Return Value:  None
*
*  Parameters:    hhcb
*                 aborting hiob
*
*  Remarks:       For the case of the target's disconnection, it might
*                 reconnected before the abort message HIOB has ever
*                 started or done on SCSI selection.  This issues is
*                 addressed in SCSIhCmdComplete routine to not post
*                 back the HIOB.
*
*********************************************************************/
#if SCSI_SWAPPING32_MODE
void SCSIhSwapping32ActiveAbort (SCSI_HHCB SCSI_HPTR hhcb,
                                 SCSI_HIOB SCSI_IPTR hiob)
{
   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_SCB_SWAPPING32 SCSI_HPTR scbW32;
   SCSI_UEXACT8 scbptr;
   SCSI_UEXACT8 byteBuf;

   /* Check if aborting HIOB is currently waiting. */
   /* The waiting SCB can be either in: */
   /*   1. The process of selecting the target or */
   /*   2. Reselection occurred while selecting the target */
   if ((OSD_INEXACT8(SCSI_AICREG(SCSI_SCSISEQ)) & SCSI_ENSELO) &&
       (hiob->scbNumber == OSD_INEXACT8(SCSI_AICREG(SCSI_W32_WAITING_SCB))))
   {
      scbptr = OSD_INEXACT8(SCSI_AICREG(SCSI_SCBPTR));   /* Save SCBPTR */

      /* Set aborted bit directly to SCB #1, i.e. selection SCB */
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCBPTR), 0x01);

      byteBuf = OSD_INEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
                    OSDoffsetof(SCSI_SCB_SWAPPING32, chain_control)));
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+OSDoffsetof(SCSI_SCB_SWAPPING32,
          chain_control)), byteBuf | SCSI_W32_ABORTED);

      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCBPTR), scbptr);   /* Restore SCBPTR */

      hiob->SCSI_IP_mgrStat = SCSI_MGR_DONE_ABT;
   }

   /* The aborting HIOB state is currently at: */
   /*   1. Completion - in Done queue */
   /*   2. Disconnection - move back to SCB array in host memory */

   /* The aborting HIOB will be post back if it found in Done queue */
   else if (!SCSIhStandardSearchDoneQ(hhcb, hiob))
   {
      /* If the aborting HIOB has not completed yet, we will issue */
      /* the abort message to the target just in case the target   */
      /* disconnected and never come back which will result in     */
      /* selection time out.  On the other hand, the command will  */
      /* be executed and abort message will be sent to target.     */

      scbW32 = (SCSI_SCB_SWAPPING32 SCSI_HPTR) hiob->scbDescriptor->scbBuffer.virtualAddress;

      /* Prepare scb to be aborted by setting "aborted" bit in SCB buffer */
      SCSI_GET_LITTLE_ENDIAN8(hhcb, &byteBuf, scbW32,
           OSDoffsetof(SCSI_SCB_SWAPPING32, chain_control));
      SCSI_PUT_LITTLE_ENDIAN8(hhcb, scbW32, OSDoffsetof(SCSI_SCB_SWAPPING32,
           chain_control), byteBuf | SCSI_W32_ABORTED);

      /* Setup special function (SCB00) - Bit 3 of scontrol flag */
      SCSI_GET_LITTLE_ENDIAN8(hhcb, &byteBuf, scbW32,
           OSDoffsetof(SCSI_SCB_SWAPPING32, scontrol));
      SCSI_PUT_LITTLE_ENDIAN8(hhcb, scbW32, OSDoffsetof(SCSI_SCB_SWAPPING32,
           scontrol), byteBuf | SCSI_W32_SPECFUN);

      /* Setup special opcode - msg_to_targ */
      SCSI_PUT_LITTLE_ENDIAN8(hhcb, scbW32, OSDoffsetof(SCSI_SCB_SWAPPING32,
           SCSI_W32_special_opcode), SCSI_W32_MSGTOTARG);

      /* Setup message byte - special_info */
      
      if ((hiob->SCSI_IF_tagEnable)  && (!hiob->SCSI_IF_disallowDisconnect))
      
      {
         /* Abort tagged command */
         SCSI_PUT_LITTLE_ENDIAN8(hhcb, scbW32, OSDoffsetof(SCSI_SCB_SWAPPING32,
              SCSI_W32_special_info), SCSI_MSG0D);
      }
      else
      {
         /* Abort non-tagged command */
         SCSI_PUT_LITTLE_ENDIAN8(hhcb, scbW32, OSDoffsetof(SCSI_SCB_SWAPPING32,
              SCSI_W32_special_info), SCSI_MSG06);
      }

      /* Enqueue at head of the new queue */
      SCSIhSwapping32QHead(hhcb, hiob); 

      /* Flush the cache */
      SCSI_FLUSH_CACHE(scbW32, sizeof(SCSI_SCB_SWAPPING32));

      /* Indicate abort is in progress */
      hiob->SCSI_IP_mgrStat = SCSI_MGR_ABORTINPROG;
   }
}
#endif   /* SCSI_SWAPPING32_MODE */


/*********************************************************************
*
*  SCSIhSwapping64ActiveAbort
*
*     Abort the active HIOB for Swapping mode sequencer.
*
*  Return Value:  None
*
*  Parameters:    hhcb
*                 aborting hiob
*
*  Remarks:       For the case of the target's disconnection, it might
*                 reconnected before the abort message HIOB has ever
*                 started or done on SCSI selection.  This issues is
*                 addressed in SCSIhCmdComplete routine to not post
*                 back the HIOB.
*
*********************************************************************/
#if SCSI_SWAPPING64_MODE
void SCSIhSwapping64ActiveAbort (SCSI_HHCB SCSI_HPTR hhcb,
                                 SCSI_HIOB SCSI_IPTR hiob)
{
   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_SCB_SWAPPING64 SCSI_HPTR scbW64;
   SCSI_UEXACT8 scbptr;
   SCSI_UEXACT8 byteBuf;

   /* Check if aborting HIOB is currently waiting. */
   /* The waiting SCB can be either in: */
   /*   1. The process of selecting the target or */
   /*   2. Reselection occurred while selecting the target */
   if ((OSD_INEXACT8(SCSI_AICREG(SCSI_SCSISEQ)) & SCSI_ENSELO) &&
       (hiob->scbNumber == OSD_INEXACT8(SCSI_AICREG(SCSI_W_WAITING_SCB))))
   {
      scbptr = OSD_INEXACT8(SCSI_AICREG(SCSI_SCBPTR));   /* Save SCBPTR */

      /* Set aborted bit directly to SCB #1, i.e. selection SCB */
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCBPTR), 0x01);

      byteBuf = OSD_INEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
                    OSDoffsetof(SCSI_SCB_SWAPPING64, chain_control)));
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+OSDoffsetof(SCSI_SCB_SWAPPING64,
          chain_control)), byteBuf | SCSI_W64_ABORTED);

      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCBPTR), scbptr);   /* Restore SCBPTR */

      hiob->SCSI_IP_mgrStat = SCSI_MGR_DONE_ABT;
   }

   /* The aborting HIOB state is currently at: */
   /*   1. Completion - in Done queue */
   /*   2. Disconnection - move back to SCB array in host memory */

   /* The aborting HIOB will be post back if it found in Done queue */
   else if (!SCSIhStandardSearchDoneQ(hhcb, hiob))
   {
      /* If the aborting HIOB has not completed yet, we will issue */
      /* the abort message to the target just in case the target   */
      /* disconnected and never come back which will result in     */
      /* selection time out.  On the other hand, the command will  */
      /* be executed and abort message will be sent to target.     */

      scbW64 = (SCSI_SCB_SWAPPING64 SCSI_HPTR) hiob->scbDescriptor->scbBuffer.virtualAddress;

      /* Prepare scb to be aborted by setting "aborted" bit in SCB buffer */
      SCSI_GET_LITTLE_ENDIAN8(hhcb, &byteBuf, scbW64,
           OSDoffsetof(SCSI_SCB_SWAPPING64, chain_control));
      SCSI_PUT_LITTLE_ENDIAN8(hhcb, scbW64, OSDoffsetof(SCSI_SCB_SWAPPING64,
           chain_control), byteBuf | SCSI_W64_ABORTED);

      /* Setup special function (SCB00) - Bit 3 of scontrol flag */
      SCSI_GET_LITTLE_ENDIAN8(hhcb, &byteBuf, scbW64,
           OSDoffsetof(SCSI_SCB_SWAPPING64, scontrol));
      SCSI_PUT_LITTLE_ENDIAN8(hhcb, scbW64, OSDoffsetof(SCSI_SCB_SWAPPING64,
           scontrol), byteBuf | SCSI_W64_SPECFUN);

      /* Setup special opcode - msg_to_targ */
      SCSI_PUT_LITTLE_ENDIAN8(hhcb, scbW64, OSDoffsetof(SCSI_SCB_SWAPPING64,
           SCSI_W64_special_opcode), SCSI_W64_MSGTOTARG);

      /* Setup message byte - special_info */
      
      if ((hiob->SCSI_IF_tagEnable)  && (!hiob->SCSI_IF_disallowDisconnect))
      {
         /* Abort tagged command */
         SCSI_PUT_LITTLE_ENDIAN8(hhcb, scbW64, OSDoffsetof(SCSI_SCB_SWAPPING64,
              SCSI_W64_special_info), SCSI_MSG0D);
      }
      else
      {
         /* Abort non-tagged command */
         SCSI_PUT_LITTLE_ENDIAN8(hhcb, scbW64, OSDoffsetof(SCSI_SCB_SWAPPING64,
              SCSI_W64_special_info), SCSI_MSG06);
      }

      /* Enqueue at head of the new queue */
      SCSIhSwapping64QHead(hhcb, hiob); 

      /* Flush the cache */
      SCSI_FLUSH_CACHE(scbW64, sizeof(SCSI_SCB_SWAPPING64));

      /* Indicate abort is in progress */
      hiob->SCSI_IP_mgrStat = SCSI_MGR_ABORTINPROG;
   }
}
#endif   /* SCSI_SWAPPING64_MODE */

/*********************************************************************
*
*  SCSIhStandardAdvActiveAbort
*
*     Abort the active HIOB for Standard mode sequencer.
*
*  Return Value:  None
*
*  Parameters:    hhcb
*                 aborting hiob
*
*  Remarks:       For the case of the target's disconnection, it might
*                 reconnected before the abort message HIOB has ever
*                 started or done on SCSI selection.  This issues is
*                 addressed in two places:
*                 1. In SCSIhAbortTarget routine will try to remove
*                    the HIOB from the internal sequencer Done queue.
*                 2. In SCSIhCmdComplete routine to not post it back.
*
*********************************************************************/
#if SCSI_STANDARD_ADVANCED_MODE
void SCSIhStandardAdvActiveAbort (SCSI_HHCB SCSI_HPTR hhcb,
                                 SCSI_HIOB SCSI_IPTR hiob)
{
   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_UEXACT8 scbptr;
   SCSI_UEXACT8 byteBuf;

   /* If the SCB to be aborted is currently running, just let it finish */
   if ((hiob->scbNumber == OSD_INEXACT8(SCSI_AICREG(SCSI_SADV_ACTIVE_SCB))) &&
       ((OSD_INEXACT8(SCSI_AICREG(SCSI_CCSGCTL)) & SCSI_CCSGEN) ||
        (OSD_INEXACT8(SCSI_AICREG(SCSI_CCSCBCTL)) & (SCSI_CCARREN | SCSI_CCSCBEN)) ||
        (OSD_INEXACT8(SCSI_AICREG(SCSI_DFCNTRL)) & (SCSI_HDMAEN | SCSI_SDMAEN | SCSI_SCSIEN))))
   {
      /* Need to clear stat because the command will be completed normally */
      hiob->stat = 0;
      hiob->SCSI_IP_mgrStat = SCSI_MGR_DONE;
   }
   /* Check if aborting HIOB is currently waiting. */
   /* The waiting SCB can be either in: */
   /*   1. The process of selecting the target or */
   /*   2. Reselection occurred while selecting the target */
   else if ((OSD_INEXACT8(SCSI_AICREG(SCSI_SCSISEQ)) & SCSI_ENSELO) &&
            (hiob->scbNumber == OSD_INEXACT8(SCSI_AICREG(SCSI_SADV_WAITING_SCB))))
   {
      /* Set aborted status is done. */
      /* The stat and haStat are set before breakpoint interrupt. */
      hiob->SCSI_IP_mgrStat = SCSI_MGR_DONE_ABT;

      /* Prepare scb to be aborted by setting 'aborted' bit in SCB ram */
      scbptr = OSD_INEXACT8(SCSI_AICREG(SCSI_SCBPTR));   /* Save SCBPTR */
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCBPTR), hiob->scbNumber);
      byteBuf = OSD_INEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
                    OSDoffsetof(SCSI_SCB_STANDARD_ADVANCED, scontrol)));
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
            OSDoffsetof(SCSI_SCB_STANDARD_ADVANCED,
            scontrol)), byteBuf | SCSI_SADV_ABORTED);
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCBPTR), scbptr);   /* Restore SCBPTR */
   }

   /* Check if the aborting HIOB is currently at: */
   /*   1. Execution queue - waiting to be executed */
   /*   2. Completion - either in Done queue or Sequencer Done queue */
   /*   3. Disconnection - in SCB ram */

   /* The aborting HIOB will be posted back in these routines if it found */
   else if (!SCSIhStandardAdvSearchExeQ(hhcb, hiob, hiob->haStat, 1) &&
            !SCSIhStandardAdvSearchSeqDoneQ(hhcb, hiob, 1) &&
            !SCSIhStandardSearchDoneQ(hhcb, hiob))
   {
      /* If the aborting HIOB has not been completed yet, we will  */
      /* issue the abort message to the target just in case the    */
      /* target disconnected and never come back which will result */
      /* in selection time out.  On the other hand, the command    */
      /* will be executed and abort message will be sent to target.*/

      /* Prepare scb to be aborted via special function */
      scbptr = OSD_INEXACT8(SCSI_AICREG(SCSI_SCBPTR));   /* Save SCBPTR */
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCBPTR), hiob->scbNumber);

      /* Prepare scb to be aborted by setting 'aborted' bit in SCB ram */
      byteBuf = OSD_INEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
                 OSDoffsetof(SCSI_SCB_STANDARD_ADVANCED, scontrol)));
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
            OSDoffsetof(SCSI_SCB_STANDARD_ADVANCED,
            scontrol)), byteBuf | SCSI_SADV_ABORTED);

      /* Setup special function bit - Bit 3 of the scontrol flag */
      byteBuf = OSD_INEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
                    OSDoffsetof(SCSI_SCB_STANDARD_ADVANCED, scontrol)));
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
            OSDoffsetof(SCSI_SCB_STANDARD_ADVANCED,
            scontrol)), byteBuf | SCSI_SADV_SPECFUN);

      /* Setup special opcode - msg_to_targ */
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
            OSDoffsetof(SCSI_SCB_STANDARD_ADVANCED,
            SCSI_SADV_special_opcode)), SCSI_SADV_MSGTOTARG);

      /* Setup message byte - special_info */
      
      if ((hiob->SCSI_IF_tagEnable)  && (!hiob->SCSI_IF_disallowDisconnect))
      {
         /* Abort tagged command */
         OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
               OSDoffsetof(SCSI_SCB_STANDARD_ADVANCED,
               SCSI_SADV_special_info)), SCSI_MSG0D);
      }
      else
      {
         /* Abort non-tagged command */
         OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
               OSDoffsetof(SCSI_SCB_STANDARD_ADVANCED,
               SCSI_SADV_special_info)), SCSI_MSG06);
      }

      /* Enqueue at head of the execution queue */
      SCSIhStandardAdvQHead(hhcb, hiob);

      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCBPTR), scbptr);   /* Restore SCBPTR */

      /* Indicate abort is in progress */
      hiob->SCSI_IP_mgrStat = SCSI_MGR_ABORTINPROG;
   }  /* end of else if not in both Done queues */
}
#endif   /* SCSI_STANDARD_ADVANCED_MODE */

/*********************************************************************
*
*  SCSIhSwappingAdvActiveAbort
*
*     Abort the active HIOB for Swapping mode sequencer.
*
*  Return Value:  None
*
*  Parameters:    hhcb
*                 aborting hiob
*
*  Remarks:       For the case of the target's disconnection, it might
*                 reconnected before the abort message HIOB has ever
*                 started or done on SCSI selection.  This issues is
*                 addressed in SCSIhCmdComplete routine to not post
*                 back the HIOB.
*
*********************************************************************/
#if SCSI_SWAPPING_ADVANCED_MODE
void SCSIhSwappingAdvActiveAbort (SCSI_HHCB SCSI_HPTR hhcb,
                                 SCSI_HIOB SCSI_IPTR hiob)
{
   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_SCB_SWAPPING_ADVANCED SCSI_HPTR scbWAdv;
   SCSI_UEXACT8 scbptr;
   SCSI_UEXACT8 byteBuf;

   /* If the SCB to be aborted is currently running, just let it finish */
   if ((hiob->scbNumber == OSD_INEXACT8(SCSI_AICREG(SCSI_WADV_ACTIVE_SCB))) &&
       ((OSD_INEXACT8(SCSI_AICREG(SCSI_CCSGCTL)) & SCSI_CCSGEN) ||
        (OSD_INEXACT8(SCSI_AICREG(SCSI_CCSCBCTL)) & (SCSI_CCARREN | SCSI_CCSCBEN)) ||
        (OSD_INEXACT8(SCSI_AICREG(SCSI_DFCNTRL)) & (SCSI_HDMAEN | SCSI_SDMAEN | SCSI_SCSIEN))))
   {
      /* Need to clear stat because the command will be completed normally */
      hiob->stat = 0;
      hiob->SCSI_IP_mgrStat = SCSI_MGR_DONE;
   }
   /* Check if aborting HIOB is currently waiting. */
   /* The waiting SCB can be either in: */
   /*   1. The process of selecting the target or */
   /*   2. Reselection occurred while selecting the target */
#if SCSI_TRIDENT_PROTO
   else if ((OSD_INEXACT8(SCSI_AICREG(SCSI_hSCSISEQ_COPY(hhcb))) & SCSI_ENSELO) &&
       (hiob->scbNumber == OSD_INEXACT8(SCSI_AICREG(SCSI_WADV_WAITING_SCB))))
#else /* SCSI_TRIDENT_PROTO */
   else if ((OSD_INEXACT8(SCSI_AICREG(SCSI_SCSISEQ)) & SCSI_ENSELO) &&
       (hiob->scbNumber == OSD_INEXACT8(SCSI_AICREG(SCSI_WADV_WAITING_SCB))))
#endif /* SCSI_TRIDENT_PROTO */
   {
      scbptr = OSD_INEXACT8(SCSI_AICREG(SCSI_SCBPTR));   /* Save SCBPTR */

      /* As the SCBPTR points to the active SCB during this period and could */
      /* be pointing to either SCB#0 or SCB#1, just toggle the SCBPTR to get */
      /* the waiting SCB to be aborted                                       */ 
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCBPTR), scbptr^(SCSI_UEXACT8)1);

      byteBuf = OSD_INEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
                    OSDoffsetof(SCSI_SCB_SWAPPING_ADVANCED, scontrol)));
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+OSDoffsetof(SCSI_SCB_SWAPPING_ADVANCED,
          scontrol)), byteBuf | SCSI_WADV_ABORTED);

      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCBPTR), scbptr);   /* Restore SCBPTR */

      hiob->SCSI_IP_mgrStat = SCSI_MGR_DONE_ABT;
   }

   /* If the SCB we are trying to abort is the one which is getting DMAed, */
   /* set the 'aborted' bit in the SCB in the internal SCB array so that   */
   /* it will be aborted.                                                  */
   else if ((OSD_INEXACT8(SCSI_AICREG(SCSI_CCSCBCTL)) &
       (SCSI_CCSCBEN+SCSI_CCSCBDIR+SCSI_CCARREN)) ==
       (SCSI_CCSCBEN+SCSI_CCSCBDIR+SCSI_CCARREN))
   {
      /* If the SCB is getting DMAed after reselection... */
      if (OSD_INEXACT8(SCSI_AICREG(SCSI_WADV_ACTIVE_SCB)) == hiob->scbNumber)
      {
         /* Prepare scb to be aborted by setting 'aborted' bit in SCB ram */
         scbptr = OSD_INEXACT8(SCSI_AICREG(SCSI_SCBPTR));   /* Save SCBPTR */
         OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCBPTR),
                       OSD_INEXACT8(SCSI_AICREG(SCSI_CCSCBPTR)));
         byteBuf = OSD_INEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
                       OSDoffsetof(SCSI_SCB_SWAPPING_ADVANCED, scontrol)));
         OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
               OSDoffsetof(SCSI_SCB_SWAPPING_ADVANCED,
               scontrol)), byteBuf | SCSI_WADV_ABORTED);
         OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCBPTR), scbptr);   /* Restore SCBPTR */
      }

      /* Set abort request is done */
      hiob->SCSI_IP_mgrStat = SCSI_MGR_DONE_ABT;
   }

   /* The aborting HIOB state is currently at: */
   /*   1. Completion - in Done queue */
   /*   2. Disconnection - move back to SCB array in host memory */

   /* The aborting HIOB will be post back if it found in Done queue */
   else if (!SCSIhStandardSearchDoneQ(hhcb, hiob))
   {
      /* If the aborting HIOB has not completed yet, we will issue */
      /* the abort message to the target just in case the target   */
      /* disconnected and never come back which will result in     */
      /* selection time out.  On the other hand, the command will  */
      /* be executed and abort message will be sent to target.     */

      scbWAdv = (SCSI_SCB_SWAPPING_ADVANCED SCSI_HPTR) hiob->scbDescriptor->scbBuffer.virtualAddress;

      /* Prepare scb to be aborted by setting "aborted" bit in SCB buffer */
      SCSI_GET_LITTLE_ENDIAN8(hhcb, &byteBuf, scbWAdv,
           OSDoffsetof(SCSI_SCB_SWAPPING_ADVANCED, scontrol));
      SCSI_PUT_LITTLE_ENDIAN8(hhcb, scbWAdv, OSDoffsetof(SCSI_SCB_SWAPPING_ADVANCED,
           scontrol), byteBuf | SCSI_WADV_ABORTED);

      /* Setup special function (SCB00) - Bit 3 of scontrol flag */
      SCSI_GET_LITTLE_ENDIAN8(hhcb, &byteBuf, scbWAdv,
           OSDoffsetof(SCSI_SCB_SWAPPING_ADVANCED, scontrol));
      SCSI_PUT_LITTLE_ENDIAN8(hhcb, scbWAdv, OSDoffsetof(SCSI_SCB_SWAPPING_ADVANCED,
           scontrol), byteBuf | SCSI_WADV_SPECFUN);

      /* Setup special opcode - msg_to_targ */
      SCSI_PUT_LITTLE_ENDIAN8(hhcb, scbWAdv, OSDoffsetof(SCSI_SCB_SWAPPING_ADVANCED,
           SCSI_WADV_special_opcode), SCSI_WADV_MSGTOTARG);

      /* Setup message byte - special_info */
      
      if ((hiob->SCSI_IF_tagEnable)  && (!hiob->SCSI_IF_disallowDisconnect))
      {
         /* Abort tagged command */
         SCSI_PUT_LITTLE_ENDIAN8(hhcb, scbWAdv, 
               OSDoffsetof(SCSI_SCB_SWAPPING_ADVANCED,
               SCSI_WADV_special_info), SCSI_MSG0D);
      }
      else
      {
         /* Abort non-tagged command */
         SCSI_PUT_LITTLE_ENDIAN8(hhcb, scbWAdv, 
               OSDoffsetof(SCSI_SCB_SWAPPING_ADVANCED,
               SCSI_WADV_special_info), SCSI_MSG06);
      }

      /* Enqueue at head of the new queue */
      SCSIhSwappingAdvQHead(hhcb, hiob); 

      /* Flush the cache */
      SCSI_FLUSH_CACHE(scbWAdv, sizeof(SCSI_SCB_SWAPPING_ADVANCED));

      /* Indicate abort is in progress */
      hiob->SCSI_IP_mgrStat = SCSI_MGR_ABORTINPROG;
   }
}
#endif   /* SCSI_SWAPPING_ADVANCED_MODE */

/*********************************************************************
*
*  SCSIhStandard160mActiveAbort
*
*     Abort the active HIOB for Standard 160m mode sequencer.
*
*  Return Value:  None
*
*  Parameters:    hhcb
*                 aborting hiob
*
*  Remarks:       For the case of the target's disconnection, it might
*                 reconnected before the abort message HIOB has ever
*                 started or done on SCSI selection.  This issues is
*                 addressed in two places:
*                 1. In SCSIhAbortTarget routine will try to remove
*                    the HIOB from the internal sequencer Done queue.
*                 2. In SCSIhCmdComplete routine to not post it back.
*
*********************************************************************/
#if SCSI_STANDARD_160M_MODE
void SCSIhStandard160mActiveAbort (SCSI_HHCB SCSI_HPTR hhcb,
                                   SCSI_HIOB SCSI_IPTR hiob)
{
   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_UEXACT8 scbptr;
   SCSI_UEXACT8 byteBuf;

   /* If the SCB to be aborted is currently running, just let it finish */
   if ((hiob->scbNumber == OSD_INEXACT8(SCSI_AICREG(SCSI_S160M_ACTIVE_SCB))) &&
       ((OSD_INEXACT8(SCSI_AICREG(SCSI_CCSGCTL)) & SCSI_CCSGEN) ||
        (OSD_INEXACT8(SCSI_AICREG(SCSI_CCSCBCTL)) & (SCSI_CCARREN | SCSI_CCSCBEN)) ||
        (OSD_INEXACT8(SCSI_AICREG(SCSI_DFCNTRL)) & (SCSI_HDMAEN | SCSI_SDMAEN | SCSI_SCSIEN))))
   {
      /* Need to clear stat because the command will be completed normally */
      hiob->stat = 0;
      hiob->SCSI_IP_mgrStat = SCSI_MGR_DONE;
   }
   /* Check if aborting HIOB is currently waiting. */
   /* The waiting SCB can be either in: */
   /*   1. The process of selecting the target or */
   /*   2. Reselection occurred while selecting the target */
   else if ((OSD_INEXACT8(SCSI_AICREG(SCSI_SCSISEQ)) & SCSI_ENSELO) &&
            (hiob->scbNumber == OSD_INEXACT8(SCSI_AICREG(SCSI_S160M_WAITING_SCB))))
   {
      /* Set aborted status is done. */
      /* The stat and haStat are set before breakpoint interrupt. */
      hiob->SCSI_IP_mgrStat = SCSI_MGR_DONE_ABT;

      /* Prepare scb to be aborted by setting 'aborted' bit in SCB ram */
      scbptr = OSD_INEXACT8(SCSI_AICREG(SCSI_SCBPTR));   /* Save SCBPTR */
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCBPTR), hiob->scbNumber);
      byteBuf = OSD_INEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
                    OSDoffsetof(SCSI_SCB_STANDARD_160M, scontrol)));
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
          OSDoffsetof(SCSI_SCB_STANDARD_160M,scontrol)),
          byteBuf | SCSI_S160M_ABORTED);
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCBPTR), scbptr);   /* Restore SCBPTR */
   }

   /* Check if the aborting HIOB is currently at: */
   /*   1. Execution queue - waiting to be executed */
   /*   2. Completion - either in Done queue or Sequencer Done queue */
   /*   3. Disconnection - in SCB ram */

   /* The aborting HIOB will be posted back in these routines if it found */
   else if (!SCSIhStandard160mSearchExeQ(hhcb, hiob, hiob->haStat, 1) &&
            !SCSIhStandard160mSearchSeqDoneQ(hhcb, hiob, 1) &&
            !SCSIhStandardSearchDoneQ(hhcb, hiob))
   {
      /* If the aborting HIOB has not been completed yet, we will  */
      /* issue the abort message to the target just in case the    */
      /* target disconnected and never come back which will result */
      /* in selection time out.  On the other hand, the command    */
      /* will be executed and abort message will be sent to target.*/

      /* Prepare scb to be aborted via special function */
      scbptr = OSD_INEXACT8(SCSI_AICREG(SCSI_SCBPTR));   /* Save SCBPTR */
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCBPTR), hiob->scbNumber);

      /* Prepare scb to be aborted by setting 'aborted' bit in SCB ram */
      byteBuf = OSD_INEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
                    OSDoffsetof(SCSI_SCB_STANDARD_160M, scontrol)));
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
          OSDoffsetof(SCSI_SCB_STANDARD_160M,scontrol)),
          byteBuf | SCSI_S160M_ABORTED);

      /* Setup special function bit - Bit 3 of the scontrol flag */
      byteBuf = OSD_INEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
                    OSDoffsetof(SCSI_SCB_STANDARD_160M, scontrol)));
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
          OSDoffsetof(SCSI_SCB_STANDARD_160M,scontrol)),
          byteBuf | SCSI_S160M_SPECFUN);

      /* Setup special opcode - msg_to_targ */
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
          OSDoffsetof(SCSI_SCB_STANDARD_160M,SCSI_S160M_special_opcode)),
          SCSI_S160M_MSGTOTARG);

      /* Setup message byte - special_info */
      
      if ((hiob->SCSI_IF_tagEnable)  && (!hiob->SCSI_IF_disallowDisconnect))
      {
         /* Abort tagged command */
         OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
             OSDoffsetof(SCSI_SCB_STANDARD_160M,SCSI_S160M_special_info)),
             SCSI_MSG0D);
      }
      else
      {
         /* Abort non-tagged command */
         OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
             OSDoffsetof(SCSI_SCB_STANDARD_160M,SCSI_S160M_special_info)),
             SCSI_MSG06);
      }

      /* Enqueue at head of the execution queue */
      SCSIhStandard160mQHead(hhcb, hiob);

      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCBPTR), scbptr);   /* Restore SCBPTR */

      /* Indicate abort is in progress */
      hiob->SCSI_IP_mgrStat = SCSI_MGR_ABORTINPROG;
   }  /* end of else if not in both Done queues */
}
#endif   /* SCSI_STANDARD_160M_MODE */

/*********************************************************************
*
*  SCSIhSwapping160mActiveAbort
*
*     Abort the active HIOB for Swapping 160m mode sequencer.
*
*  Return Value:  None
*
*  Parameters:    hhcb
*                 aborting hiob
*
*  Remarks:       For the case of the target's disconnection, it might
*                 reconnected before the abort message HIOB has ever
*                 started or done on SCSI selection.  This issues is
*                 addressed in SCSIhCmdComplete routine to not post
*                 back the HIOB.
*
*********************************************************************/
#if SCSI_SWAPPING_160M_MODE
void SCSIhSwapping160mActiveAbort (SCSI_HHCB SCSI_HPTR hhcb,
                                   SCSI_HIOB SCSI_IPTR hiob)
{
   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_SCB_SWAPPING_160M SCSI_HPTR scbW160m;
   SCSI_UEXACT8 scbptr;
   SCSI_UEXACT8 byteBuf;

   /* If the SCB to be aborted is currently running, just let it finish */
   if ((hiob->scbNumber == OSD_INEXACT8(SCSI_AICREG(SCSI_W160M_ACTIVE_SCB))) &&
       ((OSD_INEXACT8(SCSI_AICREG(SCSI_CCSGCTL)) & SCSI_CCSGEN) ||
        (OSD_INEXACT8(SCSI_AICREG(SCSI_CCSCBCTL)) & (SCSI_CCARREN | SCSI_CCSCBEN)) ||
        (OSD_INEXACT8(SCSI_AICREG(SCSI_DFCNTRL)) & (SCSI_HDMAEN | SCSI_SDMAEN | SCSI_SCSIEN))))
   {
      /* Need to clear stat because the command will be completed normally */
      hiob->stat = 0;
      hiob->SCSI_IP_mgrStat = SCSI_MGR_DONE;
   }
   /* Check if aborting HIOB is currently waiting. */
   /* The waiting SCB can be either in: */
   /*   1. The process of selecting the target or */
   /*   2. Reselection occurred while selecting the target */
   else if ((OSD_INEXACT8(SCSI_AICREG(SCSI_SCSISEQ)) & SCSI_ENSELO) &&
       (hiob->scbNumber == OSD_INEXACT8(SCSI_AICREG(SCSI_W160M_WAITING_SCB))))
   {
      scbptr = OSD_INEXACT8(SCSI_AICREG(SCSI_SCBPTR));   /* Save SCBPTR */

      /* As the SCBPTR points to the active SCB during this period and could */
      /* be pointing to either SCB#0 or SCB#1, just toggle the SCBPTR to get */
      /* the waiting SCB to be aborted                                       */ 
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCBPTR), scbptr^(SCSI_UEXACT8)1);

      byteBuf = OSD_INEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
                    OSDoffsetof(SCSI_SCB_SWAPPING_160M, scontrol)));
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
          OSDoffsetof(SCSI_SCB_SWAPPING_160M,scontrol)),
          byteBuf | SCSI_W160M_ABORTED);

      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCBPTR), scbptr);   /* Restore SCBPTR */

      SCSI_hUPDATE_ABORT_BIT_HOST_MEM(hhcb,hiob);

      

      hiob->SCSI_IP_mgrStat = SCSI_MGR_DONE_ABT;
   }

   /* If the SCB we are trying to abort is the one which is getting DMAed, */
   /* set the 'aborted' bit in the SCB in the internal SCB array so that   */
   /* it will be aborted.                                                  */
   else if ((OSD_INEXACT8(SCSI_AICREG(SCSI_CCSCBCTL)) &
       (SCSI_CCSCBEN+SCSI_CCSCBDIR+SCSI_CCARREN)) ==
       (SCSI_CCSCBEN+SCSI_CCSCBDIR+SCSI_CCARREN))
   {
      /* If the SCB is getting DMAed after reselection... */
      if (OSD_INEXACT8(SCSI_AICREG(SCSI_W160M_ACTIVE_SCB)) == hiob->scbNumber)
      {
         /* Prepare scb to be aborted by setting 'aborted' bit in SCB ram */
         scbptr = OSD_INEXACT8(SCSI_AICREG(SCSI_SCBPTR));   /* Save SCBPTR */
         OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCBPTR),
                       OSD_INEXACT8(SCSI_AICREG(SCSI_CCSCBPTR)));
         byteBuf = OSD_INEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
                       OSDoffsetof(SCSI_SCB_SWAPPING_160M,scontrol)));
         OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
             OSDoffsetof(SCSI_SCB_SWAPPING_160M,scontrol)),
             byteBuf | SCSI_W160M_ABORTED);
         OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCBPTR), scbptr);   /* Restore SCBPTR */
      }

      SCSI_hUPDATE_ABORT_BIT_HOST_MEM(hhcb,hiob);

      

      /* Set abort request is done */
      hiob->SCSI_IP_mgrStat = SCSI_MGR_DONE_ABT;
   }

   /* The aborting HIOB state is currently at: */
   /*   1. Completion - in Done queue */
   /*   2. Disconnection - move back to SCB array in host memory */

   /* The aborting HIOB will be post back if it found in Done queue */
   else if (!SCSIhStandardSearchDoneQ(hhcb, hiob))
   {
      /* If the aborting HIOB has not completed yet, we will issue */
      /* the abort message to the target just in case the target   */
      /* disconnected and never come back which will result in     */
      /* selection time out.  On the other hand, the command will  */
      /* be executed and abort message will be sent to target.     */

      scbW160m = (SCSI_SCB_SWAPPING_160M SCSI_HPTR) hiob->scbDescriptor->scbBuffer.virtualAddress;

      /* Prepare scb to be aborted by setting "aborted" bit in SCB buffer */
      SCSI_GET_LITTLE_ENDIAN8(hhcb, &byteBuf, scbW160m,
           OSDoffsetof(SCSI_SCB_SWAPPING_160M, scontrol));
      SCSI_PUT_LITTLE_ENDIAN8(hhcb, scbW160m,
           OSDoffsetof(SCSI_SCB_SWAPPING_160M, scontrol),
           byteBuf | SCSI_W160M_ABORTED);

      OSD_OUTEXACT8(SCSI_AICREG(SCSI_W160M_CHECK_ABORT), 1);   /* flag the sequencer to check for aborts */

      hhcb->numberSCBAborts++;
        
      /* Setup special function (SCB00) - Bit 3 of scontrol flag */
      SCSI_GET_LITTLE_ENDIAN8(hhcb, &byteBuf, scbW160m,
           OSDoffsetof(SCSI_SCB_SWAPPING_160M, scontrol));
      SCSI_PUT_LITTLE_ENDIAN8(hhcb, scbW160m,
           OSDoffsetof(SCSI_SCB_SWAPPING_160M, scontrol),
           byteBuf | SCSI_W160M_SPECFUN);

      /* Setup special opcode - msg_to_targ */
      SCSI_PUT_LITTLE_ENDIAN8(hhcb, scbW160m,
           OSDoffsetof(SCSI_SCB_SWAPPING_160M, SCSI_W160M_special_opcode),
           SCSI_W160M_MSGTOTARG);

      /* Setup message byte - special_info */
      
      if ((hiob->SCSI_IF_tagEnable)  && (!hiob->SCSI_IF_disallowDisconnect))
      {
         /* Abort tagged command */
         SCSI_PUT_LITTLE_ENDIAN8(hhcb, scbW160m, 
              OSDoffsetof(SCSI_SCB_SWAPPING_160M, SCSI_W160M_special_info),
              SCSI_MSG0D);
      }
      else
      {
         /* Abort non-tagged command */
         SCSI_PUT_LITTLE_ENDIAN8(hhcb, scbW160m, 
              OSDoffsetof(SCSI_SCB_SWAPPING_160M, SCSI_W160M_special_info),
              SCSI_MSG06);
      }

      /* Enqueue at head of the new queue */
      SCSIhSwapping160mQHead(hhcb, hiob); 

      /* Flush the cache */
      SCSI_FLUSH_CACHE(scbW160m, sizeof(SCSI_SCB_SWAPPING_160M));

      /* Indicate abort is in progress */
      hiob->SCSI_IP_mgrStat = SCSI_MGR_ABORTINPROG;
   }
}
#endif   /* SCSI_SWAPPING_160M_MODE */

/*********************************************************************
*
*  SCSIhSwappingRemoveActiveAbort
*
*     Remove active abort from new queue or from the state of
*     selecting the target.
*
*  Return Value:  None
*
*  Parameters:    hhcb
*                 hiob
*
*  Remarks:
*
*********************************************************************/
#if SCSI_SWAPPING_MODE
void SCSIhSwappingRemoveActiveAbort (SCSI_HHCB SCSI_HPTR hhcb,
                                       SCSI_HIOB SCSI_IPTR hiob)
{
   SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_UEXACT8 scsiseq;

#if SCSI_TRIDENT_PROTO
   scsiseq = OSD_INEXACT8(SCSI_AICREG(SCSI_hSCSISEQ_COPY(hhcb)));
#else /* SCSI_TRIDENT_PROTO */
   scsiseq = OSD_INEXACT8(SCSI_AICREG(SCSI_SCSISEQ));
#endif /* SCSI_TRIDENT_PROTO */
   
   /* If the active abort SCB is curretly selecting the target */
   if ((scsiseq & SCSI_ENSELO) &&
       (OSD_INEXACT8(SCSI_AICREG(SCSI_hWAITING_SCB(hhcb))) == 
        OSD_INEXACT8(SCSI_AICREG(SCSI_hACTIVE_SCB(hhcb)))))
   {
      hiob->SCSI_IP_mgrStat = SCSI_MGR_DONE;

      /* Disable any outstanding selection */
      /* Reselection occurred while the active abort SCB is selecting. */
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSISEQ),
          scsiseq & ~(SCSI_TEMODEO+SCSI_ENSELO+SCSI_ENAUTOATNO+SCSI_ENAUTOATNI));
      scsiseq = OSD_INEXACT8(SCSI_AICREG(SCSI_SCSISEQ)); /* To flush the cache */
#if SCSI_TRIDENT_PROTO
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_hSCSISEQ_COPY(hhcb)),scsiseq);
      scsiseq = OSD_INEXACT8(SCSI_AICREG(SCSI_hSCSISEQ_COPY(hhcb))); /* To flush the cache */
#endif /* SCSI_TRIDENT_PROTO */
   }
   
   /* Remove the active abort SCB if found in the new queue */
   else if (SCSIhSwappingRemoveAbortHIOBFromNewQ(hhcb, hiob, hiob->haStat))
   {
      hiob->SCSI_IP_mgrStat = SCSI_MGR_DONE;
   }
}
#endif   /* SCSI_SWAPPING_MODE */

/*********************************************************************
*
*  SCSIhStandardRemoveActiveAbort
*
*     Remove active abort from execution queue or from the state of
*     selecting the target.
*
*  Return Value:  None
*
*  Parameters:    hhcb
*                 hiob
*
*  Remarks:
*
*********************************************************************/
#if SCSI_STANDARD_MODE
void SCSIhStandardRemoveActiveAbort (SCSI_HHCB SCSI_HPTR hhcb,
                                       SCSI_HIOB SCSI_IPTR hiob)
{
   SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_UEXACT8 scsiseq;

   scsiseq = OSD_INEXACT8(SCSI_AICREG(SCSI_SCSISEQ));
   
   /* If the active abort SCB is curretly selecting the target */
   if ((scsiseq & SCSI_ENSELO) &&
       (OSD_INEXACT8(SCSI_AICREG(SCSI_hWAITING_SCB(hhcb))) == 
        OSD_INEXACT8(SCSI_AICREG(SCSI_hACTIVE_SCB(hhcb)))))
   {
      hiob->SCSI_IP_mgrStat = SCSI_MGR_DONE;

      /* Disable any outstanding selection */
      /* Reselection occurred while the active abort SCB is selecting. */
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSISEQ),
          scsiseq & ~(SCSI_TEMODEO+SCSI_ENSELO+SCSI_ENAUTOATNO+SCSI_ENAUTOATNI));
      scsiseq = OSD_INEXACT8(SCSI_AICREG(SCSI_SCSISEQ)); /* To flush the cache */
   }
   
   /* Remove the active abort SCB if found in execution queue */
   else if (SCSI_hSEARCHEXEQ(hhcb, hiob, hiob->haStat, 0))
   {
      hiob->SCSI_IP_mgrStat = SCSI_MGR_DONE;
   }
}
#endif   /* SCSI_STANDARD_MODE */

/*********************************************************************
*
*  SCSIhStandard64ResidueCalc
*
*     Calculate residual length.
*
*  Return Value:  None
*                  
*  Parameters:    hhcb
*                 hiob
*
*  Remarks:                
*                  
*********************************************************************/
#if SCSI_STANDARD64_MODE
SCSI_UEXACT32 SCSIhStandard64ResidueCalc (SCSI_HHCB SCSI_HPTR hhcb,
                                          SCSI_HIOB SCSI_IPTR hiob)
{
   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_BUS_ADDRESS busAddress;
   void SCSI_LPTR virtualAddress;
   SCSI_BUS_ADDRESS sgPointerSCB;
   SCSI_UEXACT32 sgLen;
   SCSI_BUFFER_DESCRIPTOR SCSI_LPTR sgList;
   SCSI_UEXACT8 sgCacheWork=0;
   SCSI_UEXACT32 residueWork=0;
   SCSI_UEXACT8 mwiResidual;
   SCSI_UINT32 count;          /* time out counter */
   int i=0;
   int currentSGindex;  
     
   /* Return if no data */
   if(OSD_INEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
      OSDoffsetof(SCSI_SCB_STANDARD64,scdb_length))) & SCSI_S64_NODATA)
   {
      return(0);
   }
   
   /* Put embedded s/g element length into residueWork. */
   /* If CDB length in SCSI_S64_CDBLENMSK is >0, transfer was not started.    */
   /*    Need to initialize residueWork with embedded s/g element,            */
   /*    Current s/g pointer is from sg_pointer_SCB                           */
   /* Else                                                                    */
   /*    SCSI_S64_sdata_residue0-3 are used to intitialize residueWork        */
   /*    Current s/g pointer is from sg_pointer_work                          */
   if(OSD_INEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
      OSDoffsetof(SCSI_SCB_STANDARD64,scdb_length))) & SCSI_S64_CDBLENMSK)
   {
      /* Get current segment length                                           */
      /*    Its ok to use SCSIhGetAddressBA32 cause we know length is 3 bytes */
      /*    regardless of Bus Address size                                    */
      residueWork = (SCSI_UEXACT32)((SCSI_UEXACT32)SCSIhGetEntity32FromScb(hhcb,
                    (SCSI_UEXACT8)(SCSI_SCB_BASE+
                    OSDoffsetof(SCSI_SCB_STANDARD64,slength0))) & 0x00FFFFFF);

      /* Get s/g pointer from SCB - this is pointer to s/g cache line.     */
      /* Actual s/g element pointer is index (s/g cache work) + s/g cache  */
      /* line pointer.                                                     */
      sgPointerSCB = SCSI_hGETADDRESSSCB(hhcb,(SCSI_UEXACT8)(SCSI_SCB_BASE+
                     OSDoffsetof(SCSI_SCB_STANDARD64,SCSI_S64_sg_pointer_SCB0)));

      sgCacheWork = OSD_INEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
         OSDoffsetof(SCSI_SCB_STANDARD64,SCSI_S64_sg_cache_SCB)));

      if(OSD_INEXACT8(SCSI_AICREG(SCSI_SCB_BASE+                            
         OSDoffsetof(SCSI_SCB_STANDARD64,scdb_length))) & SCSI_S64_ONESGSEG) 
      {
         return(residueWork);
      }
   
   }
   else
   {
      /* Get current segment length                                           */
      /*    Its ok to use SCSIhGetEntity32FromScb cause we know sdata_residue is  */
      /*    4 bytes regardless of Bus Address size                            */
      residueWork = (SCSI_UEXACT32)(SCSIhGetEntity32FromScb(hhcb,
                    (SCSI_UEXACT8)(SCSI_SCB_BASE+
                    OSDoffsetof(SCSI_SCB_STANDARD64,SCSI_S64_sdata_residue0)))
                    & 0x00FFFFFF);

      /* ww_ins mwi - Need to add mwiResidual bytes to length of the current */
      /* segment if mwi correction has been done to the current s/g element. */
      /* mwi workaround will break an element into to elements (element A with  */
      /* length = length - mwiResidual and element B with length of mwiResidual.*/
      /* Therefore, if the mwi correction has been done mwiResidual bytes need */
      /* to be added */
      if(mwiResidual = OSD_INEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
         OSDoffsetof(SCSI_SCB_STANDARD64,SCSI_S64_mwiResidual))))
      {
         residueWork += (SCSI_UEXACT32) mwiResidual;
      }

      /* Get s/g pointer from SCB - this is pointer to s/g cache line.     */
      /* Actual s/g element pointer is index (s/g cache work) + s/g cache  */
      /* line pointer.                                                     */
#if OSD_BUS_ADDRESS_SIZE == 64
      ((SCSI_OCTLET SCSI_IPTR)&sgPointerSCB)->u32.order0 = 
         (SCSI_UEXACT32) SCSIhGetEntity32FromScb(hhcb,(SCSI_UEXACT8)(SCSI_SCB_BASE+
         OSDoffsetof(SCSI_SCB_STANDARD64,SCSI_S64_sg_pointer_work0)));
      ((SCSI_OCTLET SCSI_IPTR)&sgPointerSCB)->u32.order1 = 
         (SCSI_UEXACT32) SCSIhGetEntity32FromScb(hhcb,(SCSI_UEXACT8)(SCSI_SCB_BASE+
         OSDoffsetof(SCSI_SCB_STANDARD64,SCSI_S64_sg_pointer_SCB4)));
#else
      sgPointerSCB = SCSIhGetAddressScbBA32(hhcb,(SCSI_UEXACT8)(SCSI_SCB_BASE+
         OSDoffsetof(SCSI_SCB_STANDARD64,SCSI_S64_sg_pointer_work0)));
#endif
   
      /* Get s/g cache work */
      sgCacheWork = OSD_INEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
         OSDoffsetof(SCSI_SCB_STANDARD64,SCSI_S64_sg_cache_work)));

      if (OSD_INEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
          OSDoffsetof(SCSI_SCB_STANDARD64,SCSI_S64_sg_flags_work)))
          & SCSI_S64_ONESGSEG)
      {
         return(residueWork);
      }

   }
   
   /* Get the S/G List (its a buffer descriptor) */
   sgList = OSD_GET_SG_LIST(hiob);
   busAddress = sgList->busAddress;
   virtualAddress = sgList->virtualAddress;

   /* Must walk down s/g list until at the same point sequencer */
   /* or time out counter expires.                              */                     
   count=(SCSI_UINT32)0x800000;
   while(1)
   {
      if ((!SCSIhCompareBusAddress(sgPointerSCB,busAddress))|| (--count==0))
      {
         break;
      }
      else 
      {
         OSD_ADJUST_BUS_ADDRESS(hhcb,&busAddress,2*sizeof(SCSI_BUS_ADDRESS));
         i++;
      }
   }

   /* S/G element is i + index into s/g cache line pointer */
   i += sgCacheWork/((SCSI_UEXACT8)(2*sizeof(SCSI_BUS_ADDRESS)));

   /* sgPointerSCB point to next sg element, need to adjust it to get */
   /* current index. i should always not be equal to 0, though. */
   if (i==0)
   {
      currentSGindex = i;
   }
   else
   {
      currentSGindex = i-1;
   }

   /* If current s/g element is last one then don't walk down list */
   SCSI_GET_LITTLE_ENDIAN32(hhcb,&sgLen,virtualAddress,
       (currentSGindex*2*sizeof(SCSI_BUS_ADDRESS))
       +sizeof(SCSI_BUS_ADDRESS));
   if  (sgLen & (SCSI_UEXACT32)0x80000000)
   {
      return(residueWork);
   }

   /* Walk down remainder of s/g list and sum the length portions of the s/g             */
   /* elements. When the delimiter is encountered, or timeout counter expires then exit. */
   count=(SCSI_UINT32)0x800000;    
   while(1)
   {
      SCSI_GET_LITTLE_ENDIAN32(hhcb,&sgLen,virtualAddress,
          (i*2*sizeof(SCSI_BUS_ADDRESS))+sizeof(SCSI_BUS_ADDRESS));

      residueWork += (sgLen & 0x00FFFFFF);
      /* to handle one segment and zero in length passed in */
      if ((sgLen & (SCSI_UEXACT32)0x80000000)|| (--count ==0))
      {
         break;
      }
      i++;
   }
   return(residueWork);
}
#endif

/*********************************************************************
*
*  SCSIhSwapping32ResidueCalc
*
*     Calculate residual length.
*
*  Return Value:  None
*                  
*  Parameters:    hhcb
*                 hiob
*
*  Remarks:                
*                  
*********************************************************************/
#if SCSI_SWAPPING32_MODE
SCSI_UEXACT32 SCSIhSwapping32ResidueCalc (SCSI_HHCB SCSI_HPTR hhcb,
                                          SCSI_HIOB SCSI_IPTR hiob)
{
   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_BUS_ADDRESS busAddress;
   void SCSI_LPTR virtualAddress;
   SCSI_BUS_ADDRESS sgPointerSCB;
   SCSI_UEXACT32 sgLen;
   SCSI_BUFFER_DESCRIPTOR SCSI_LPTR sgList;
   SCSI_UEXACT32 residueWork=0;
   SCSI_UEXACT8 mwiResidual;
   SCSI_UINT32 count;          /* time out counter */
   int i=0;
   int currentSGindex;
   
   /* Return if no data */
   if(OSD_INEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
      OSDoffsetof(SCSI_SCB_SWAPPING32,scdb_length))) & SCSI_W32_NODATA)
   {
      return(0);
   }
   
   /* Put embedded s/g element length into residueWork. */
   /* If CDB length in SCSI_W32_CDBLENMSK is >0, transfer was not started.    */
   /*    Need to initialize residueWork with embedded s/g element,            */
   /*    Current s/g pointer is from sg_pointer_SCB                           */
   /* Else                                                                    */
   /*    SCSI_W32_sdata_residue0-3 are used to intitialize residueWork        */
   /*    Current s/g pointer is from sg_pointer_work                          */
   if(OSD_INEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
      OSDoffsetof(SCSI_SCB_SWAPPING32,scdb_length))) & SCSI_W32_CDBLENMSK)
   {
      /* Get current segment length                                           */
      /*    Its ok to use SCSIhGetEntity32FromScb cause we know length is 3 bytes */
      /*    regardless of Bus Address size                                    */
      residueWork = (SCSI_UEXACT32)((SCSI_UEXACT32)SCSIhGetEntity32FromScb(hhcb,
                    (SCSI_UEXACT8)(SCSI_SCB_BASE+
                    OSDoffsetof(SCSI_SCB_SWAPPING32,slength0))) & 0x00FFFFFF);

      /* Get s/g pointer from SCB */
      sgPointerSCB = SCSI_hGETADDRESSSCB(hhcb,(SCSI_UEXACT8)(SCSI_SCB_BASE+
                     OSDoffsetof(SCSI_SCB_SWAPPING32,SCSI_W32_sg_pointer_SCB0)));

      if(OSD_INEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
         OSDoffsetof(SCSI_SCB_SWAPPING32,scdb_length))) & SCSI_W32_ONESGSEG)
      {
         return(residueWork);
      }
   
   }
   else
   {
      /* Get current segment length                                           */
      /*    Its ok to use SCSIhGetEntity32FromScb cause we know sdata_residue is  */
      /*    4 bytes regardless of Bus Address size                            */
      residueWork = (SCSI_UEXACT32)(SCSIhGetEntity32FromScb(hhcb,
                    (SCSI_UEXACT8)(SCSI_SCB_BASE+
                    OSDoffsetof(SCSI_SCB_SWAPPING32,SCSI_W32_sdata_residue0)))
                    & 0x00FFFFFF);

      /* ww_ins mwi - Need to add mwiResidual bytes to length of the current */
      /* segment if mwi correction has been done to the current s/g element. */
      /* mwi workaround will break an element into to elements (element A with  */
      /* length = length - mwiResidual and element B with length of mwiResidual.*/
      /* Therefore, if the mwi correction has been done mwiResidual bytes need */
      /* to be added */
      if(mwiResidual = OSD_INEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
         OSDoffsetof(SCSI_SCB_SWAPPING32,SCSI_W32_mwiResidual))))
      {
         residueWork += (SCSI_UEXACT32) mwiResidual;
      }

      /* Get s/g pointer from SCB */
      sgPointerSCB = SCSIhGetAddressScbBA32(hhcb,(SCSI_UEXACT8)(SCSI_SCB_BASE+
         OSDoffsetof(SCSI_SCB_SWAPPING32,SCSI_W32_sg_pointer_work0)));
   
      if(OSD_INEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
         OSDoffsetof(SCSI_SCB_SWAPPING32,SCSI_W32_sg_flags_work)))
         & SCSI_W32_ONESGSEG)
      {
         return(residueWork);
      }
   
   }
  
   /* Get the S/G List (its a buffer descriptor) */
   sgList = OSD_GET_SG_LIST(hiob);
   busAddress = sgList->busAddress;
   virtualAddress = sgList->virtualAddress;

   /* Must walk down s/g list until at the same point sequencer or  */
   /* time out counter expires.                                     */
   count=(SCSI_UINT32)0x800000;
   while(1)
   {
      if ((!SCSIhCompareBusAddress(sgPointerSCB,busAddress))||(--count==0))
      {
         break;
      }
      else 
      {
         OSD_ADJUST_BUS_ADDRESS(hhcb,&busAddress,2*sizeof(SCSI_BUS_ADDRESS));
         i++;
      }
   }

   /* sgPointerSCB point to next sg element, need to adjust it to get */
   /* current index. i should always not be equal to 0, though. */
   if (i==0)
   {
      currentSGindex = i;
   }
   else
   {
      currentSGindex = i-1;
   }

   /* If current s/g element is last one then don't walk down list */
   SCSI_GET_LITTLE_ENDIAN32(hhcb,&sgLen,virtualAddress,
       (currentSGindex*2*sizeof(SCSI_BUS_ADDRESS))
       +sizeof(SCSI_BUS_ADDRESS));

   if (sgLen & (SCSI_UEXACT32)0x80000000)
   {
      return(residueWork);
   }

   /* Walk down remainder of s/g list and sum the length portions of the s/g    */
   /* elements. When delimiter is encountered, or time out counter expires exit */
   count=(SCSI_UINT32)0x800000;
   while(1)
   {
      SCSI_GET_LITTLE_ENDIAN32(hhcb,&sgLen,virtualAddress,
         (i*2*sizeof(SCSI_BUS_ADDRESS))+sizeof(SCSI_BUS_ADDRESS));

      residueWork += (sgLen & 0x00FFFFFF);
      /* to handle one segment and zero in length passed in */
      if ((sgLen & (SCSI_UEXACT32)0x80000000)|| (--count==0))
      {
         break;
      }
      i++;
   }
   return(residueWork);
}
#endif

/*********************************************************************
*
*  SCSIhSwapping64ResidueCalc
*
*     Calculate residual length.
*
*  Return Value:  None
*                  
*  Parameters:    hhcb
*                 hiob
*
*  Remarks:                
*                  
*********************************************************************/
#if SCSI_SWAPPING64_MODE
SCSI_UEXACT32 SCSIhSwapping64ResidueCalc (SCSI_HHCB SCSI_HPTR hhcb,
                                          SCSI_HIOB SCSI_IPTR hiob)
{
   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_BUS_ADDRESS busAddress;
   void SCSI_LPTR virtualAddress;
   SCSI_BUS_ADDRESS sgPointerSCB;
   SCSI_UEXACT32 sgLen;
   SCSI_BUFFER_DESCRIPTOR SCSI_LPTR sgList;
   SCSI_UEXACT32 residueWork=0;
   int currentSGindex;
   SCSI_UEXACT8 mwiResidual;
   SCSI_SCB_SWAPPING64 SCSI_HPTR scbW64;
   SCSI_UINT32 count;          /* time out counter */
   int i=0;
   
   /* Return if no data */
   if(OSD_INEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
      OSDoffsetof(SCSI_SCB_SWAPPING64,scdb_length))) & SCSI_W64_NODATA)
   {
      return(0);
   }
   
   /* Put embedded s/g element length into residueWork. */
   /* If CDB length in SCSI_W64_CDBLENMSK is >0, transfer was not started.    */
   /*    Need to initialize residueWork with embedded s/g element,            */
   /*    Current s/g pointer is from sg_pointer_SCB                           */
   /* Else                                                                    */
   /*    SCSI_W32_sdata_residue0-3 are used to intitialize residueWork        */
   /*    Current s/g pointer is from sg_pointer_work                          */
   if(OSD_INEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
      OSDoffsetof(SCSI_SCB_SWAPPING64,scdb_length))) & SCSI_W64_CDBLENMSK)
   {
      /* Get current segment length                                           */
      /*    Its ok to use SCSIhGetAddressBA32 cause we know length is 3 bytes */
      /*    regardless of Bus Address size                                    */
      residueWork = (SCSI_UEXACT32)((SCSI_UEXACT32)SCSIhGetEntity32FromScb(hhcb,
                    (SCSI_UEXACT8)(SCSI_SCB_BASE+
                    OSDoffsetof(SCSI_SCB_SWAPPING64,SCSI_W64_slength0))) & 0x00FFFFFF);

      /* Get s/g pointer from SCB */
  
      scbW64 = (SCSI_SCB_SWAPPING64 SCSI_HPTR) hiob->scbDescriptor->scbBuffer.virtualAddress;

      /* Invalidate the cache */
      SCSI_INVALIDATE_CACHE(scbW64, sizeof(SCSI_SCB_SWAPPING64));

      SCSI_GET_LITTLE_ENDIAN64(hhcb, &sgPointerSCB, scbW64, OSDoffsetof(SCSI_SCB_SWAPPING64,
           SCSI_W64_sg_pointer_SCB0));

      if(OSD_INEXACT8(SCSI_AICREG(SCSI_SCB_BASE+                              
         OSDoffsetof(SCSI_SCB_SWAPPING64,scdb_length))) & SCSI_W64_ONESGSEG)  
      {
         return(residueWork);
      }
   
   }
   else
   {
      /* Get current segment length                                           */
      /*    Its ok to use SCSIhGetAddressBA32 cause we know sdata_residue is  */
      /*    4 bytes regardless of Bus Address size                            */
      residueWork = (SCSI_UEXACT32)(SCSIhGetEntity32FromScb(hhcb,
                    (SCSI_UEXACT8)(SCSI_SCB_BASE+
                    OSDoffsetof(SCSI_SCB_SWAPPING64,SCSI_W64_sdata_residue0)))
                    & 0x00FFFFFF);

      /* ww_ins mwi - Need to add mwiResidual bytes to length of the current */
      /* segment if mwi correction has been done to the current s/g element. */
      /* mwi workaround will break an element into to elements (element A with  */
      /* length = length - mwiResidual and element B with length of mwiResidual.*/
      /* Therefore, if the mwi correction has been done mwiResidual bytes need */
      /* to be added */                                                
      if(mwiResidual = OSD_INEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
         OSDoffsetof(SCSI_SCB_SWAPPING64,SCSI_W64_mwiResidual))))
      {
         residueWork += (SCSI_UEXACT32) mwiResidual;
      }

      /* Get s/g pointer from SCB */
      sgPointerSCB = SCSI_hGETADDRESSSCB(hhcb,(SCSI_UEXACT8)(SCSI_SCB_BASE+
                     OSDoffsetof(SCSI_SCB_SWAPPING64,SCSI_W64_sg_pointer_work0)));

      if(OSD_INEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
         OSDoffsetof(SCSI_SCB_SWAPPING64,SCSI_W64_sg_flags_work)))
         & SCSI_W64_ONESGSEG)                                        
      {
         return(residueWork);
      }
   
   }
  
   /* Get the S/G List (its a buffer descriptor) */
   sgList = OSD_GET_SG_LIST(hiob);
   busAddress = sgList->busAddress;
   virtualAddress = sgList->virtualAddress;
   
   /* Must walk down s/g list until at the same point sequencer or  */
   /* time out counter expires.                                     */
   count=(SCSI_UINT32)0x800000;
   while(1)
   {
      if ((!SCSIhCompareBusAddress(sgPointerSCB,busAddress))||(--count==0))
      {
         break;
      }
      else 
      {
         OSD_ADJUST_BUS_ADDRESS(hhcb,&busAddress,2*sizeof(SCSI_BUS_ADDRESS));
         i++;
      }
   }

   /* sgPointerSCB point to next sg element, need to adjust it to get */
   /* current index. i should always not be equal to 0, though. */
   if (i==0)
   {
      currentSGindex = i;
   }
   else
   {
      currentSGindex = i-1;
   }

   /* If current s/g element is last one then don't walk down list */
   SCSI_GET_LITTLE_ENDIAN32(hhcb,&sgLen,virtualAddress,
       (currentSGindex*2*sizeof(SCSI_BUS_ADDRESS))
       +sizeof(SCSI_BUS_ADDRESS));

   if (sgLen & (SCSI_UEXACT32)0x80000000)
   {
      return(residueWork);
   }
  
   /* Walk down remainder of s/g list and sum the length portions of the s/g    */
   /* elements. When delimiter is encountered, or time out counter expires exit */
   count=(SCSI_UINT32)0x800000;
   while(1)
   {
      SCSI_GET_LITTLE_ENDIAN32(hhcb,&sgLen,virtualAddress,
          (i*2*sizeof(SCSI_BUS_ADDRESS))+sizeof(SCSI_BUS_ADDRESS));

      residueWork += (sgLen & 0x00FFFFFF);
      /* to handle one segment and zero in length passed in */
      if ((sgLen & (SCSI_UEXACT32)0x80000000)|| (--count==0))
      {
         break;
      }
      i++;
   }
   return(residueWork);
}
#endif

/*********************************************************************
*
*  SCSIhStandardAdvResidueCalc
*
*     Calculate residual length.
*
*  Return Value:  None
*                  
*  Parameters:    hhcb
*                 hiob
*
*  Remarks:                
*                  
*********************************************************************/
#if SCSI_STANDARD_ADVANCED_MODE
SCSI_UEXACT32 SCSIhStandardAdvResidueCalc (SCSI_HHCB SCSI_HPTR hhcb,
                                          SCSI_HIOB SCSI_IPTR hiob)
{
   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_BUS_ADDRESS busAddress;
   void SCSI_LPTR virtualAddress;
   SCSI_BUS_ADDRESS sgPointerSCB;
   SCSI_UEXACT32 sgLen;
   SCSI_BUFFER_DESCRIPTOR SCSI_LPTR sgList;
   SCSI_UEXACT8 sgCacheWork=0;
   SCSI_UEXACT32 residueWork=0;
   SCSI_UINT32 count;          /* time out counter */
   int i=0;
   int currentSGindex;  
     
   /* Return if no data */
   if(OSD_INEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
      OSDoffsetof(SCSI_SCB_STANDARD_ADVANCED,sg_cache_SCB))) & SCSI_SADV_NODATA)
   {
      return(0);
   }
   
   /* Put embedded s/g element length into residueWork. */
   /* If CDB length in SCSI_SADV_CDBLENMSK is >0, transfer was not started.   */
   /*    Need to initialize residueWork with embedded s/g element,            */
   /*    Current s/g pointer is from sg_pointer_SCB                           */
   /* Else                                                                    */
   /*    SCSI_SADV_sdata_residue0-3 are used to intitialize residueWork       */
   /*    Current s/g pointer is from sg_pointer_work                          */
   if(OSD_INEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
      OSDoffsetof(SCSI_SCB_STANDARD_ADVANCED,scdb_length))) & SCSI_SADV_CDBLENMSK)
   {
      /* Get current segment length                                           */
      /*    Its ok to use SCSIhGetAddressBA32 cause we know length is 3 bytes */
      /*    regardless of Bus Address size                                    */
      residueWork = (SCSI_UEXACT32)((SCSI_UEXACT32)SCSIhGetEntity32FromScb(hhcb,
                    (SCSI_UEXACT8)(SCSI_SCB_BASE+
                    OSDoffsetof(SCSI_SCB_STANDARD_ADVANCED,slength0))) & 0x00FFFFFF);

      /* Get s/g pointer from SCB - this is pointer to s/g cache line.     */
      /* Actual s/g element pointer is index (s/g cache work) + s/g cache  */
      /* line pointer.                                                     */
      sgPointerSCB = SCSI_hGETADDRESSSCB(hhcb,(SCSI_UEXACT8)(SCSI_SCB_BASE+
                     OSDoffsetof(SCSI_SCB_STANDARD_ADVANCED,SCSI_SADV_sg_pointer_SCB0)));

      sgCacheWork = OSD_INEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
         OSDoffsetof(SCSI_SCB_STANDARD_ADVANCED,sg_cache_SCB)));

      if(OSD_INEXACT8(SCSI_AICREG(SCSI_SCB_BASE+                            
         OSDoffsetof(SCSI_SCB_STANDARD_ADVANCED,sg_cache_SCB))) & SCSI_SADV_ONESGSEG) 
      {
         return(residueWork);
      }

   }
   else
   {
      /* Get current segment length                                           */
      /* Its ok to use SCSIhGetEntity32FromScb cause we know sdata_residue is */
      /*    4 bytes regardless of Bus Address size                            */
      residueWork = (SCSI_UEXACT32)(SCSIhGetEntity32FromScb(hhcb,
                    (SCSI_UEXACT8)(SCSI_SCB_BASE+
                    OSDoffsetof(SCSI_SCB_STANDARD_ADVANCED,SCSI_SADV_sdata_residue0)))
                    & 0x00FFFFFF);

      /* Get s/g pointer from SCB - this is pointer to s/g cache line.     */
      /* Actual s/g element pointer is index (s/g cache work) + s/g cache  */
      /* line pointer.                                                     */
#if OSD_BUS_ADDRESS_SIZE == 64
      ((SCSI_OCTLET SCSI_IPTR)&sgPointerSCB)->u32.order0 = 
         (SCSI_UEXACT32) SCSIhGetEntity32FromScb(hhcb,(SCSI_UEXACT8)(SCSI_SCB_BASE+
         OSDoffsetof(SCSI_SCB_STANDARD_ADVANCED,SCSI_SADV_sg_pointer_work0)));
      ((SCSI_OCTLET SCSI_IPTR)&sgPointerSCB)->u32.order1 = 
         (SCSI_UEXACT32) SCSIhGetEntity32FromScb(hhcb,(SCSI_UEXACT8)(SCSI_SCB_BASE+
         OSDoffsetof(SCSI_SCB_STANDARD_ADVANCED,SCSI_SADV_sg_pointer_SCB4)));
#else
      sgPointerSCB = SCSIhGetAddressScbBA32(hhcb,(SCSI_UEXACT8)(SCSI_SCB_BASE+
         OSDoffsetof(SCSI_SCB_STANDARD_ADVANCED,SCSI_SADV_sg_pointer_work0)));
#endif
   
      /* Get s/g cache work */
      sgCacheWork = OSD_INEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
         OSDoffsetof(SCSI_SCB_STANDARD_ADVANCED,SCSI_SADV_sg_cache_work)));

      if (sgCacheWork & SCSI_SADV_ONESGSEG)
      {
         return(residueWork);
      }
  
   }
   
   /* Get the S/G List (its a buffer descriptor) */
   sgList = OSD_GET_SG_LIST(hiob);
   busAddress = sgList->busAddress;
   virtualAddress = sgList->virtualAddress;

   /* Must walk down s/g list until at the same point sequencer */
   /* or time out counter expires.                              */                     
   count=(SCSI_UINT32)0x800000;
   while(1)
   {
      if ((!SCSIhCompareBusAddress(sgPointerSCB,busAddress))|| (--count==0))
      {
         break;
      }
      else 
      {
         OSD_ADJUST_BUS_ADDRESS(hhcb,&busAddress,2*sizeof(SCSI_BUS_ADDRESS));
         i++;
      }
   }

   /* S/G element is i + index into s/g cache line pointer */
   i += sgCacheWork/((SCSI_UEXACT8)(2*sizeof(SCSI_BUS_ADDRESS)));

   /* sgPointerSCB point to next sg element, need to adjust it to get */
   /* current index. i should always not be equal to 0, though. */
   if (i==0)
   {
      currentSGindex = i;
   }
   else
   {
      currentSGindex = i-1;
   }

   /* If current s/g element is last one then don't walk down list */
   SCSI_GET_LITTLE_ENDIAN32(hhcb,&sgLen,virtualAddress,
       (currentSGindex*2*sizeof(SCSI_BUS_ADDRESS))
       +sizeof(SCSI_BUS_ADDRESS));
   if  (sgLen & (SCSI_UEXACT32)0x80000000)
   {
      return(residueWork);
   }

   /* Walk down remainder of s/g list and sum the length portions of the s/g             */
   /* elements. When the delimiter is encountered, or timeout counter expires then exit. */
   count=(SCSI_UINT32)0x800000;    
   while(1)
   {
      SCSI_GET_LITTLE_ENDIAN32(hhcb,&sgLen,virtualAddress,
          (i*2*sizeof(SCSI_BUS_ADDRESS))+sizeof(SCSI_BUS_ADDRESS));

      residueWork += (sgLen & 0x00FFFFFF);
      /* to handle one segment and zero in length passed in */
      if ((sgLen & (SCSI_UEXACT32)0x80000000)|| (--count ==0))
      {
         break;
      }
      i++;
   }
   return(residueWork);
}
#endif

/*********************************************************************
*
*  SCSIhSwappingAdvResidueCalc
*
*     Calculate residual length.
*
*  Return Value:  None
*                  
*  Parameters:    hhcb
*                 hiob
*
*  Remarks:                
*                  
*********************************************************************/
#if SCSI_SWAPPING_ADVANCED_MODE
SCSI_UEXACT32 SCSIhSwappingAdvResidueCalc (SCSI_HHCB SCSI_HPTR hhcb,
                                          SCSI_HIOB SCSI_IPTR hiob)
{
   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_BUS_ADDRESS busAddress;
   void SCSI_LPTR virtualAddress;
   SCSI_BUS_ADDRESS sgPointerSCB;
   SCSI_UEXACT32 sgLen;
   SCSI_BUFFER_DESCRIPTOR SCSI_LPTR sgList;
   SCSI_UEXACT8 sgCacheWork=0;
   SCSI_UEXACT32 residueWork=0;
   SCSI_UINT32 count;          /* time out counter */
   int i=0;
   int currentSGindex;  
     
   /* Return if no data */
   if(OSD_INEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
      OSDoffsetof(SCSI_SCB_SWAPPING_ADVANCED,sg_cache_SCB))) & SCSI_WADV_NODATA)
   {
      return(0);
   }
   
   /* Put embedded s/g element length into residueWork. */
   /* If CDB length in SCSI_WADV_CDBLENMSK is >0, transfer was not started.   */
   /*    Need to initialize residueWork with embedded s/g element,            */
   /*    Current s/g pointer is from sg_pointer_SCB                           */
   /* Else                                                                    */
   /*    SCSI_WADV_sdata_residue0-3 are used to intitialize residueWork       */
   /*    Current s/g pointer is from sg_pointer_work                          */
   if(OSD_INEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
      OSDoffsetof(SCSI_SCB_SWAPPING_ADVANCED,scdb_length))) & SCSI_WADV_CDBLENMSK)
   {
      /* Get current segment length                                           */
      /*    Its ok to use SCSIhGetAddressBA32 cause we know length is 3 bytes */
      /*    regardless of Bus Address size                                    */
      residueWork = (SCSI_UEXACT32)((SCSI_UEXACT32)SCSIhGetEntity32FromScb(hhcb,
                    (SCSI_UEXACT8)(SCSI_SCB_BASE+
                    OSDoffsetof(SCSI_SCB_SWAPPING_ADVANCED,slength0))) & 0x00FFFFFF);

      /* Get s/g pointer from SCB - this is pointer to s/g cache line.     */
      /* Actual s/g element pointer is index (s/g cache work) + s/g cache  */
      /* line pointer.                                                     */
      sgPointerSCB = SCSI_hGETADDRESSSCB(hhcb,(SCSI_UEXACT8)(SCSI_SCB_BASE+
                     OSDoffsetof(SCSI_SCB_SWAPPING_ADVANCED,SCSI_WADV_sg_pointer_SCB0)));

      sgCacheWork = OSD_INEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
         OSDoffsetof(SCSI_SCB_SWAPPING_ADVANCED,sg_cache_SCB)));

      if(OSD_INEXACT8(SCSI_AICREG(SCSI_SCB_BASE+                            
         OSDoffsetof(SCSI_SCB_SWAPPING_ADVANCED,sg_cache_SCB))) & SCSI_WADV_ONESGSEG) 
      {
         return(residueWork);
      }

   }
   else
   {
      /* Get current segment length                                           */
      /* Its ok to use SCSIhGetEntity32FromScb cause we know sdata_residue is */
      /*    4 bytes regardless of Bus Address size                            */
      residueWork = (SCSI_UEXACT32)(SCSIhGetEntity32FromScb(hhcb,
                    (SCSI_UEXACT8)(SCSI_SCB_BASE+
                    OSDoffsetof(SCSI_SCB_SWAPPING_ADVANCED,SCSI_WADV_sdata_residue0)))
                    & 0x00FFFFFF);

      /* Get s/g pointer from SCB - this is pointer to s/g cache line.     */
      /* Actual s/g element pointer is index (s/g cache work) + s/g cache  */
      /* line pointer.                                                     */
#if OSD_BUS_ADDRESS_SIZE == 64
      ((SCSI_OCTLET SCSI_IPTR)&sgPointerSCB)->u32.order0 = 
         (SCSI_UEXACT32) SCSIhGetEntity32FromScb(hhcb,(SCSI_UEXACT8)(SCSI_SCB_BASE+
         OSDoffsetof(SCSI_SCB_SWAPPING_ADVANCED,SCSI_WADV_sg_pointer_work0)));
      ((SCSI_OCTLET SCSI_IPTR)&sgPointerSCB)->u32.order1 = 
         (SCSI_UEXACT32) SCSIhGetEntity32FromScb(hhcb,(SCSI_UEXACT8)(SCSI_SCB_BASE+
         OSDoffsetof(SCSI_SCB_SWAPPING_ADVANCED,SCSI_WADV_sg_pointer_SCB4)));
#else
      sgPointerSCB = SCSIhGetAddressScbBA32(hhcb,(SCSI_UEXACT8)(SCSI_SCB_BASE+
         OSDoffsetof(SCSI_SCB_SWAPPING_ADVANCED,SCSI_WADV_sg_pointer_work0)));
#endif
   
      /* Get s/g cache work */
      sgCacheWork = OSD_INEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
         OSDoffsetof(SCSI_SCB_SWAPPING_ADVANCED,SCSI_WADV_sg_cache_work)));

      if (sgCacheWork & SCSI_WADV_ONESGSEG)
      {
         return(residueWork);
      }
  
   }
   
   /* Get the S/G List (its a buffer descriptor) */
   sgList = OSD_GET_SG_LIST(hiob);
   busAddress = sgList->busAddress;
   virtualAddress = sgList->virtualAddress;

   /* Must walk down s/g list until at the same point sequencer */
   /* or time out counter expires.                              */                     
   count=(SCSI_UINT32)0x800000;
   while(1)
   {
      if ((!SCSIhCompareBusAddress(sgPointerSCB,busAddress))|| (--count==0))
      {
         break;
      }
      else 
      {
         OSD_ADJUST_BUS_ADDRESS(hhcb,&busAddress,2*sizeof(SCSI_BUS_ADDRESS));
         i++;
      }
   }

   /* S/G element is i + index into s/g cache line pointer */
   i += sgCacheWork/((SCSI_UEXACT8)(2*sizeof(SCSI_BUS_ADDRESS)));

   /* sgPointerSCB point to next sg element, need to adjust it to get */
   /* current index. i should always not be equal to 0, though. */
   if (i==0)
   {
      currentSGindex = i;
   }
   else
   {
      currentSGindex = i-1;
   }

   /* If current s/g element is last one then don't walk down list */
   SCSI_GET_LITTLE_ENDIAN32(hhcb,&sgLen,virtualAddress,
       (currentSGindex*2*sizeof(SCSI_BUS_ADDRESS))
       +sizeof(SCSI_BUS_ADDRESS));
   if  (sgLen & (SCSI_UEXACT32)0x80000000)
   {
      return(residueWork);
   }

   /* Walk down remainder of s/g list and sum the length portions of the s/g             */
   /* elements. When the delimiter is encountered, or timeout counter expires then exit. */
   count=(SCSI_UINT32)0x800000;    
   while(1)
   {
      SCSI_GET_LITTLE_ENDIAN32(hhcb,&sgLen,virtualAddress,
          (i*2*sizeof(SCSI_BUS_ADDRESS))+sizeof(SCSI_BUS_ADDRESS));

      residueWork += (sgLen & 0x00FFFFFF);
      /* to handle one segment and zero in length passed in */
      if ((sgLen & (SCSI_UEXACT32)0x80000000)|| (--count ==0))
      {
         break;
      }
      i++;
   }
   return(residueWork);
}
#endif

/*********************************************************************
*
*  SCSIhStandard160mResidueCalc
*
*     Calculate residual length.
*
*  Return Value:  None
*                  
*  Parameters:    hhcb
*                 hiob
*
*  Remarks:                
*                  
*********************************************************************/
#if SCSI_STANDARD_160M_MODE
SCSI_UEXACT32 SCSIhStandard160mResidueCalc (SCSI_HHCB SCSI_HPTR hhcb,
                                            SCSI_HIOB SCSI_IPTR hiob)
{
   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_BUS_ADDRESS busAddress;
   void SCSI_LPTR virtualAddress;
   SCSI_BUS_ADDRESS sgPointerSCB;
   SCSI_UEXACT32 sgLen;
   SCSI_BUFFER_DESCRIPTOR SCSI_LPTR sgList;
   SCSI_UEXACT8 sgCacheWork = 0;
   SCSI_UEXACT32 residueWork = 0;
   SCSI_UINT32 count;          /* time out counter */
   int i=0;
   int currentSGindex;  
     
   /* Return if no data */
   if(OSD_INEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
      OSDoffsetof(SCSI_SCB_STANDARD_160M,sg_cache_SCB))) & SCSI_S160M_NODATA)
   {
      return(0);
   }
   
   /* Put embedded s/g element length into residueWork. */
   /* If CDB length in SCSI_S160M_CDBLENMSK is >0, transfer was not started.  */
   /*    Need to initialize residueWork with embedded s/g element,            */
   /*    Current s/g pointer is from sg_pointer_SCB                           */
   /* Else                                                                    */
   /*    SCSI_S160M_sdata_residue0-3 are used to intitialize residueWork      */
   /*    Current s/g pointer is from sg_pointer_work                          */
   if(OSD_INEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
      OSDoffsetof(SCSI_SCB_STANDARD_160M,scdb_length))) & SCSI_S160M_CDBLENMSK)
   {
      /* Get current segment length                                           */
      /*    Its ok to use SCSIhGetAddressBA32 cause we know length is 3 bytes */
      /*    regardless of Bus Address size                                    */
      residueWork = (SCSI_UEXACT32)((SCSI_UEXACT32)SCSIhGetEntity32FromScb(hhcb,
                    (SCSI_UEXACT8)(SCSI_SCB_BASE+
                    OSDoffsetof(SCSI_SCB_STANDARD_160M,slength0))) & 0x00FFFFFF);

      /* Get s/g pointer from SCB - this is pointer to s/g cache line.     */
      /* Actual s/g element pointer is index (s/g cache work) + s/g cache  */
      /* line pointer.                                                     */
      sgPointerSCB = SCSI_hGETADDRESSSCB(hhcb,(SCSI_UEXACT8)(SCSI_SCB_BASE+
                     OSDoffsetof(SCSI_SCB_STANDARD_160M,SCSI_S160M_sg_pointer_SCB0)));

      sgCacheWork = OSD_INEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
         OSDoffsetof(SCSI_SCB_STANDARD_160M,sg_cache_SCB)));

      if(OSD_INEXACT8(SCSI_AICREG(SCSI_SCB_BASE+                            
         OSDoffsetof(SCSI_SCB_STANDARD_160M,sg_cache_SCB))) & SCSI_S160M_ONESGSEG) 
      {
         return(residueWork);
      }

   }
   else
   {
      /* Get current segment length                                           */
      /* Its ok to use SCSIhGetEntity32FromScb cause we know sdata_residue is */
      /*    4 bytes regardless of Bus Address size                            */
      residueWork = (SCSI_UEXACT32)(SCSIhGetEntity32FromScb(hhcb,
                    (SCSI_UEXACT8)(SCSI_SCB_BASE+
                    OSDoffsetof(SCSI_SCB_STANDARD_160M,SCSI_S160M_sdata_residue0)))
                    & 0x00FFFFFF);

      /* Get s/g pointer from SCB - this is pointer to s/g cache line.     */
      /* Actual s/g element pointer is index (s/g cache work) + s/g cache  */
      /* line pointer.                                                     */
#if OSD_BUS_ADDRESS_SIZE == 64
      ((SCSI_OCTLET SCSI_IPTR)&sgPointerSCB)->u32.order0 = 
         (SCSI_UEXACT32) SCSIhGetEntity32FromScb(hhcb,(SCSI_UEXACT8)(SCSI_SCB_BASE+
         OSDoffsetof(SCSI_SCB_STANDARD_160M,SCSI_S160M_sg_pointer_work0)));
      ((SCSI_OCTLET SCSI_IPTR)&sgPointerSCB)->u32.order1 = 
         (SCSI_UEXACT32) SCSIhGetEntity32FromScb(hhcb,(SCSI_UEXACT8)(SCSI_SCB_BASE+
         OSDoffsetof(SCSI_SCB_STANDARD_160M,SCSI_S160M_sg_pointer_SCB4)));
#else
      sgPointerSCB = SCSIhGetAddressScbBA32(hhcb,(SCSI_UEXACT8)(SCSI_SCB_BASE+
         OSDoffsetof(SCSI_SCB_STANDARD_160M,SCSI_S160M_sg_pointer_work0)));
#endif
   
      /* Get s/g cache work */
      sgCacheWork = OSD_INEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
         OSDoffsetof(SCSI_SCB_STANDARD_160M,SCSI_S160M_sg_cache_work)));

      if (sgCacheWork & SCSI_S160M_ONESGSEG)
      {
         return(residueWork);
      }
  
   }
   
   /* Get the S/G List (its a buffer descriptor) */
   sgList = OSD_GET_SG_LIST(hiob);
   busAddress = sgList->busAddress;
   virtualAddress = sgList->virtualAddress;

   /* Must walk down s/g list until at the same point sequencer */
   /* or time out counter expires.                              */                     
   count=(SCSI_UINT32)0x800000;
   while(1)
   {
      if ((!SCSIhCompareBusAddress(sgPointerSCB,busAddress)) || (--count == 0))
      {
         break;
      }
      else 
      {
         OSD_ADJUST_BUS_ADDRESS(hhcb,&busAddress,2*sizeof(SCSI_BUS_ADDRESS));
         i++;
      }
   }

   /* S/G element is i + index into s/g cache line pointer */
   i += sgCacheWork/((SCSI_UEXACT8)(2*sizeof(SCSI_BUS_ADDRESS)));

   /* sgPointerSCB point to next sg element, need to adjust it to get */
   /* current index. i should always not be equal to 0, though. */
   if (i==0)
   {
      currentSGindex = i;
   }
   else
   {
      currentSGindex = i-1;
   }

   /* If current s/g element is last one then don't walk down list */
   SCSI_GET_LITTLE_ENDIAN32(hhcb,&sgLen,virtualAddress,
        (currentSGindex*2*sizeof(SCSI_BUS_ADDRESS))+sizeof(SCSI_BUS_ADDRESS));
   if  (sgLen & (SCSI_UEXACT32)0x80000000)
   {
      return(residueWork);
   }

   /* Walk down remainder of s/g list and sum the length portions of the s/g  */
   /* elements. When the delimiter is encountered, or timeout counter expires */
   /* then exit. */
   count=(SCSI_UINT32)0x800000;    
   while(1)
   {
      SCSI_GET_LITTLE_ENDIAN32(hhcb,&sgLen,virtualAddress,
           (i*2*sizeof(SCSI_BUS_ADDRESS))+sizeof(SCSI_BUS_ADDRESS));

      residueWork += (sgLen & 0x00FFFFFF);
      /* to handle one segment and zero in length passed in */
      if ((sgLen & (SCSI_UEXACT32)0x80000000) || (--count == 0))
      {
         break;
      }
      i++;
   }
   return(residueWork);
}
#endif /* SCSI_STANDARD_160M_MODE */

/*********************************************************************
*
*  SCSIhSwapping160mResidueCalc
*
*     Calculate residual length.
*
*  Return Value:  None
*                  
*  Parameters:    hhcb
*                 hiob
*
*  Remarks:                
*                  
*********************************************************************/
#if SCSI_SWAPPING_160M_MODE
SCSI_UEXACT32 SCSIhSwapping160mResidueCalc (SCSI_HHCB SCSI_HPTR hhcb,
                                            SCSI_HIOB SCSI_IPTR hiob)
{
   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_BUS_ADDRESS busAddress;
   void SCSI_LPTR virtualAddress;
   SCSI_BUS_ADDRESS sgPointerSCB;
   SCSI_UEXACT32 sgLen;
   SCSI_BUFFER_DESCRIPTOR SCSI_LPTR sgList;
   SCSI_UEXACT8 sgCacheWork = 0;
   SCSI_UEXACT32 residueWork = 0;
   SCSI_UINT32 count;          /* time out counter */
   int i=0;
   int currentSGindex;  
     
   /* Return if no data */
   if(OSD_INEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
      OSDoffsetof(SCSI_SCB_SWAPPING_160M,sg_cache_SCB))) & SCSI_W160M_NODATA)
   {
      return(0);
   }
   
   /* Put embedded s/g element length into residueWork. */
   /* If CDB length in SCSI_W160M_CDBLENMSK is >0, transfer was not started.  */
   /*    Need to initialize residueWork with embedded s/g element,            */
   /*    Current s/g pointer is from sg_pointer_SCB                           */
   /* Else                                                                    */
   /*    SCSI_W160M_sdata_residue0-3 are used to intitialize residueWork      */
   /*    Current s/g pointer is from sg_pointer_work                          */
   if(OSD_INEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
      OSDoffsetof(SCSI_SCB_SWAPPING_160M,scdb_length))) & SCSI_W160M_CDBLENMSK)
   {
      /* Get current segment length                                           */
      /*    Its ok to use SCSIhGetAddressBA32 cause we know length is 3 bytes */
      /*    regardless of Bus Address size                                    */
      residueWork = (SCSI_UEXACT32)((SCSI_UEXACT32)SCSIhGetEntity32FromScb(hhcb,
                    (SCSI_UEXACT8)(SCSI_SCB_BASE+
                    OSDoffsetof(SCSI_SCB_SWAPPING_160M,slength0))) & 0x00FFFFFF);

      /* Get s/g pointer from SCB - this is pointer to s/g cache line.     */
      /* Actual s/g element pointer is index (s/g cache work) + s/g cache  */
      /* line pointer.                                                     */
      sgPointerSCB = SCSI_hGETADDRESSSCB(hhcb,(SCSI_UEXACT8)(SCSI_SCB_BASE+
                     OSDoffsetof(SCSI_SCB_SWAPPING_160M,SCSI_W160M_sg_pointer_SCB0)));

      sgCacheWork = OSD_INEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
         OSDoffsetof(SCSI_SCB_SWAPPING_160M,sg_cache_SCB)));

      if(OSD_INEXACT8(SCSI_AICREG(SCSI_SCB_BASE+                            
         OSDoffsetof(SCSI_SCB_SWAPPING_160M,sg_cache_SCB))) & SCSI_W160M_ONESGSEG) 
      {
         return(residueWork);
      }

   }
   else
   {
      /* Get current segment length                                           */
      /* Its ok to use SCSIhGetEntity32FromScb cause we know sdata_residue is */
      /*    4 bytes regardless of Bus Address size                            */
      residueWork = (SCSI_UEXACT32)(SCSIhGetEntity32FromScb(hhcb,
                    (SCSI_UEXACT8)(SCSI_SCB_BASE+
                    OSDoffsetof(SCSI_SCB_SWAPPING_160M,SCSI_W160M_sdata_residue0)))
                    & 0x00FFFFFF);

      /* Get s/g pointer from SCB - this is pointer to s/g cache line.     */
      /* Actual s/g element pointer is index (s/g cache work) + s/g cache  */
      /* line pointer.                                                     */
#if OSD_BUS_ADDRESS_SIZE == 64
      ((SCSI_OCTLET SCSI_IPTR)&sgPointerSCB)->u32.order0 = 
         (SCSI_UEXACT32) SCSIhGetEntity32FromScb(hhcb,(SCSI_UEXACT8)(SCSI_SCB_BASE+
         OSDoffsetof(SCSI_SCB_SWAPPING_160M,SCSI_W160M_sg_pointer_work0)));
      ((SCSI_OCTLET SCSI_IPTR)&sgPointerSCB)->u32.order1 = 
         (SCSI_UEXACT32) SCSIhGetEntity32FromScb(hhcb,(SCSI_UEXACT8)(SCSI_SCB_BASE+
         OSDoffsetof(SCSI_SCB_SWAPPING_160M,SCSI_W160M_sg_pointer_SCB4)));
#else
      sgPointerSCB = SCSIhGetAddressScbBA32(hhcb,(SCSI_UEXACT8)(SCSI_SCB_BASE+
         OSDoffsetof(SCSI_SCB_SWAPPING_160M,SCSI_W160M_sg_pointer_work0)));
#endif
   
      /* Get s/g cache work */
      sgCacheWork = OSD_INEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
         OSDoffsetof(SCSI_SCB_SWAPPING_160M,SCSI_W160M_sg_cache_work)));

      if (sgCacheWork & SCSI_W160M_ONESGSEG)
      {
         return(residueWork);
      }
  
   }
   
   /* Get the S/G List (its a buffer descriptor) */
   sgList = OSD_GET_SG_LIST(hiob);
   busAddress = sgList->busAddress;
   virtualAddress = sgList->virtualAddress;

   /* Must walk down s/g list until at the same point sequencer */
   /* or time out counter expires.                              */                     
   count=(SCSI_UINT32)0x800000;
   while(1)
   {
      if ((!SCSIhCompareBusAddress(sgPointerSCB,busAddress)) || (--count == 0))
      {
         break;
      }
      else 
      {
         OSD_ADJUST_BUS_ADDRESS(hhcb,&busAddress,2*sizeof(SCSI_BUS_ADDRESS));
         i++;
      }
   }

   /* S/G element is i + index into s/g cache line pointer */
   i += sgCacheWork/((SCSI_UEXACT8)(2*sizeof(SCSI_BUS_ADDRESS)));

   /* sgPointerSCB point to next sg element, need to adjust it to get */
   /* current index. i should always not be equal to 0, though. */
   if (i==0)
   {
      currentSGindex = i;
   }
   else
   {
      currentSGindex = i-1;
   }

   /* If current s/g element is last one then don't walk down list */
   SCSI_GET_LITTLE_ENDIAN32(hhcb,&sgLen,virtualAddress,
        (currentSGindex*2*sizeof(SCSI_BUS_ADDRESS))+sizeof(SCSI_BUS_ADDRESS));
   if  (sgLen & (SCSI_UEXACT32)0x80000000)
   {
      return(residueWork);
   }

   /* Walk down remainder of s/g list and sum the length portions of the s/g  */
   /* elements. When the delimiter is encountered, or timeout counter expires */
   /* then exit. */
   count=(SCSI_UINT32)0x800000;    
   while(1)
   {
      SCSI_GET_LITTLE_ENDIAN32(hhcb,&sgLen,virtualAddress,
           (i*2*sizeof(SCSI_BUS_ADDRESS))+sizeof(SCSI_BUS_ADDRESS));

      residueWork += (sgLen & 0x00FFFFFF);
      /* to handle one segment and zero in length passed in */
      if ((sgLen & (SCSI_UEXACT32)0x80000000) || (--count == 0))
      {
         break;
      }
      i++;
   }
   return(residueWork);
}
#endif /* SCSI_SWAPPING_160M_MODE */

/*********************************************************************
*
*  SCSIhSwappingRemoveHIOBfromNewQ
*
*     Search and remove the aborting HIOB in the New queue.
*
*  Return Value:  0 - aborting HIOB not found
*                 1 - aborting HIOB found
*
*  Parameters:    hhcb
*                 aborting hiob
*                 host adapter status
*
*  Remarks:
*
*********************************************************************/
#if SCSI_SWAPPING_MODE
SCSI_UEXACT8 SCSIhSwappingRemoveAbortHIOBFromNewQ (SCSI_HHCB SCSI_HPTR hhcb,
                                                   SCSI_HIOB SCSI_IPTR hiob,
                                                   SCSI_UEXACT8 haStatus)
{
   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_UEXACT8 qNewTail;
   SCSI_UEXACT8 qNextScb;
   SCSI_UEXACT8 ScbCurr;
   SCSI_UEXACT8 ScbPrev;
   SCSI_UEXACT8 ScbNew;
   SCSI_HIOB SCSI_IPTR hiobCurr;
   SCSI_HIOB SCSI_IPTR hiobPrev;
   SCSI_UEXACT8 retStat = 0;

   /* Get q_new_tail value */
   qNewTail = SCSI_hGETQNEWTAIL(hhcb);

   /* Get q_next_scb value from scratch ram */
   qNextScb = OSD_INEXACT8(SCSI_AICREG(SCSI_hQ_NEXT_SCB(hhcb)));
   ScbCurr = ScbPrev = qNextScb;

   /* Check if aborting HIOB is in New queue */
   while (qNewTail != hhcb->SCSI_HP_qNewHead)
   {
      /* Search for matching SCB number */
      if (ScbCurr == hiob->scbNumber)
      {
         /* Set abort status */
         hiob->stat = SCSI_SCB_ABORTED;
         hiob->haStat = haStatus;

         /* Decrement and update qNewHead into scratch ram */
         (SCSI_UEXACT8)hhcb->SCSI_HP_qNewHead--;
         OSD_OUTEXACT8(SCSI_AICREG(SCSI_hQ_NEW_HEAD(hhcb)), hhcb->SCSI_HP_qNewHead);

         /* Obtain current HIOB in Active pointer array */ 
         hiobCurr = SCSI_ACTPTR[ScbCurr];
         ScbNew = SCSI_hOBTAIN_NEXT_SCB_NUM(hhcb,hiobCurr);

         if (ScbCurr == qNextScb)
         {
            /* Remove abort HIOB from the tail of the New queue */
            OSD_OUTEXACT8(SCSI_AICREG(SCSI_hQ_NEXT_SCB(hhcb)), ScbNew);
         }
         else
         {
            /* Remove abort HIOB from New queue and mend the queue */
            hiobPrev = SCSI_ACTPTR[ScbPrev];
            SCSI_hUPDATE_NEXT_SCB_NUM(hhcb,hiobPrev,ScbNew);
         }

         retStat = 1;
         break;
      }

      /* Need to get next scb from current scb */

      ScbPrev = ScbCurr;
      
      /* Obtain current HIOB in Active pointer array */ 
      hiobCurr = SCSI_ACTPTR[ScbCurr];
      ScbCurr = SCSI_hOBTAIN_NEXT_SCB_NUM(hhcb,hiobCurr);
      
      (SCSI_UEXACT8)qNewTail++;
   }

   return(retStat);
}
#endif /* SCSI_SWAPPING_MODE */

/*********************************************************************
*
*  SCSIhBusWasDead
*
*     Check if scsi bus still alive.
*
*  Return Value:  0 : scsi bus is alive
*                 1 : scsi bus is dead
*                  
*  Parameters:    hhcb
*
*  Remarks:                
*                  
*********************************************************************/
SCSI_UEXACT8 SCSIhBusWasDead (SCSI_HHCB SCSI_HPTR hhcb)
{
   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_UEXACT8 scsisig;
   SCSI_UEXACT32 timer = (SCSI_UINT32)0x8000;
   SCSI_UEXACT8 hcntrl, status;

   /* check if pause chip necessary */
   if (!((hcntrl = OSD_INEXACT8(SCSI_AICREG(SCSI_HCNTRL))) & SCSI_PAUSEACK))
   {
      SCSIhPauseAndWait(hhcb);      
   }

   /* Check if any of the control signals change; if so, bus is alive, else */
   /* it is dead */
   if ((scsisig = OSD_INEXACT8(SCSI_AICREG(SCSI_SCSISIG))))
   {
      while(timer--)
      {
         SCSIhDelayCount500us(hhcb, 1);
         if (scsisig != OSD_INEXACT8(SCSI_AICREG(SCSI_SCSISIG)))
         {
            status = 0;  /* scsi bus is alive */
            break;
         }
      }
      if (!timer)
      {
         status = 1; /* scsi bus is dead */
      }
   }
   /* Check if RST signal goes off; if so, bus is alive, else it is dead */
   else if ((OSD_INEXACT8(SCSI_AICREG(SCSI_SSTAT1))) & SCSI_SCSIRSTI)
   {
      timer = (SCSI_UINT32)0x8000;
      while(timer--)
      {
         SCSIhDelayCount500us(hhcb, 1);
         if (!((OSD_INEXACT8(SCSI_AICREG(SCSI_SSTAT1))) & SCSI_SCSIRSTI))
         {
            status = 0;  /* scsi bus is alive */
            break;
         }
      }
      if (!timer)
      {
         status = 1; /* scsi bus is dead */
      }
   }
   /* Bus is quiet; so assume it is alive */
   else
   {
      status = 0;  /* scsi bus is alive */
   }

   if (!(hcntrl & SCSI_PAUSEACK))
   {
      SCSI_hWRITE_HCNTRL(hhcb,scsiRegister,hcntrl);
   }

   return(status);
}

#if SCSI_CHECK_PCI_ERROR
/*********************************************************************
*
*  SCSIhClearPCIError
*
*     This routine clears all PCI errors through Bayonet back door mechanism.
*     A proper AE will be sent to OSM
*
*  Return Value:  None
*                  
*  Parameters:    hhcb
*
*  Remarks: In most case, SCSI bus might be hung, we let OSM to decide what to
*           do.               
*                  
*********************************************************************/
void SCSIhClearPCIError(SCSI_HHCB SCSI_HPTR hhcb)
{
   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_UEXACT32 pcierrstat;
   SCSI_UEXACT8 sfunct;

   /* Reset DMA & SCSI transfer logic */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_DFCNTRL),0);
   
   /* clear data channel */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SXFRCTL0),(OSD_INEXACT8(SCSI_AICREG(SCSI_SXFRCTL0)) 
    | (SCSI_CLRSTCNT | SCSI_CLRCHN | SCSI_SPIOEN) & ~SCSI_FAST20));

   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SIMODE1), OSD_INEXACT8(SCSI_AICREG(SCSI_SIMODE1)) & ~SCSI_ENBUSFREE);
   SCSI_hCLEARSCSIRATE(hhcb);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_CLRSINT1), SCSI_CLRBUSFREE);

   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SEQADDR0), (SCSI_UEXACT8)(SCSI_hIDLE_LOOP_ENTRY(hhcb) >> 2));
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SEQADDR1), (SCSI_UEXACT8)(SCSI_hIDLE_LOOP_ENTRY(hhcb) >> 10));

   /* save SFUNCT reg */
   sfunct = OSD_INEXACT8(SCSI_AICREG(SCSI_SFUNCT)); 
   
   /* enabled config space register */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SFUNCT), SCSI_CS_ON);
   
   /* clear all pci error bits of status register in config space */
   pcierrstat = OSD_INEXACT8(SCSI_AICREG(SCSI_STATUS_HI_REG));
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_STATUS_HI_REG), (SCSI_UEXACT8)pcierrstat);

   /* disabled config space register */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SFUNCT), sfunct);

   pcierrstat <<= 24;

   /* AEN to RSM - HA received PCI error */
   if(pcierrstat & SCSI_RTA)
      SCSI_ASYNC_EVENT(hhcb, SCSI_AE_PCI_RTA);
   else
      SCSI_ASYNC_EVENT(hhcb, SCSI_AE_PCI_ERROR);
}
#endif

/***************************************************************************
*                                                                          *
* Copyright 1995,1996,1997,1998,1999 Adaptec, Inc.,  All Rights Reserved.  *
*                                                                          *
* This software contains the valuable trade secrets of Adaptec.  The       *
* software is protected under copyright laws as an unpublished work of     *
* Adaptec.  Notice is for informational purposes only and does not imply   *
* publication.  The user of this software may make copies of the software  *
* for use with parts manufactured by Adaptec or under license from Adaptec *
* and for no other use.                                                    *
*                                                                          *
***************************************************************************/

/***************************************************************************
*
*  Module Name:   HWMDLVR.C
*
*  Description:
*                 Codes to implement delivery mechanism for  hardware 
*                 management module
*
*  Owners:        ECX IC Firmware Team
*    
*  Notes:         NONE
*
***************************************************************************/

#include "scsi.h"
#include "hwm.h"

/*********************************************************************
*
*  SCSIhDeliverScb
*
*     Deliver scb for standard mode
*
*  Return Value:  none
*                  
*  Parameters:    hhcb
*                 hiob
*
*  Remarks:
*                  
*********************************************************************/
/* @@@@ do later - additional inline code needed to do this #if !SCSI_STREAMLINE_QIOPATH */
void SCSIhDeliverScb (SCSI_HHCB SCSI_HPTR hhcb,
                      SCSI_HIOB SCSI_IPTR hiob)
{
   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;

   /* setup scb buffer */
#if SCSI_TARGET_OPERATION 
   if (hiob->SCSI_IP_targetMode)
   {                     
      /* setup scb buffer */
      OSD_BUILD_TARGET_SCB(hiob);
      /* Determine if an ignore wide message is required */ 
      if (hhcb->SCSI_HF_targetIgnoreWideResidMsg &&
          (SCSI_DATA_IOLENGTH(hiob) & 0x1))  /* odd data transfer length? */
      {
         SCSI_hTARGETSETIGNOREWIDEMSG(hhcb,hiob);
      }
   }   
   else
   {
     OSD_BUILD_SCB(hiob);
   }   
#else 

   OSD_BUILD_SCB(hiob);
#endif  /* SCSI_TARGET_OPERATION */  

   SCSI_hPATCH_XFER_OPT(hhcb,hiob);

   /* bump q new head */
   (SCSI_UEXACT8)hhcb->SCSI_HP_qNewHead++;

   OSD_OUTEXACT8(SCSI_AICREG(SCSI_hQ_NEW_HEAD(hhcb)),hhcb->SCSI_HP_qNewHead);
}      
/* @@@@ do later - additional inline code needed to do this #endif */ /* !SCSI_STREAMLINE_QIOPATH */

/*********************************************************************
*
*  SCSIhSwapping32PatchXferOpt
*
*     Patch xfer option and fast 20 into scb for swapping32 mode sequencer
*
*  Return Value:  none
*                  
*  Parameters:    hhcb
*                 hiob
*
*  Remarks:
*                  
*********************************************************************/
#if SCSI_SWAPPING32_MODE
void SCSIhSwapping32PatchXferOpt (SCSI_HHCB SCSI_HPTR hhcb,
                                  SCSI_HIOB SCSI_IPTR hiob)
{
   SCSI_DEVICE SCSI_DPTR deviceTable = SCSI_TARGET_UNIT(hiob)->deviceTable;
   SCSI_UEXACT8 uexact8Value;
   SCSI_SCB_SWAPPING32 SCSI_IPTR scbBuffer =
      (SCSI_SCB_SWAPPING32 SCSI_IPTR )hiob->scbDescriptor->scbBuffer.virtualAddress;

   /* patch the transfer rate */
   SCSI_PUT_LITTLE_ENDIAN8(hhcb,scbBuffer,
         OSDoffsetof(SCSI_SCB_SWAPPING32,xfer_option),
         deviceTable->xferOptionHost);

   /* patch the fast flag */
   if (deviceTable->SCSI_DF_fast20EnableW)
   {
      SCSI_GET_LITTLE_ENDIAN8(hhcb,&uexact8Value,scbBuffer,
         OSDoffsetof(SCSI_SCB_SWAPPING32,scontrol));
      uexact8Value |= SCSI_W_FAST20;
      SCSI_PUT_LITTLE_ENDIAN8(hhcb,scbBuffer,
         OSDoffsetof(SCSI_SCB_SWAPPING32,scontrol),uexact8Value);

      SCSI_FLUSH_CACHE(((SCSI_UEXACT8 SCSI_HPTR)scbBuffer) +
         (OSDoffsetof(SCSI_SCB_SWAPPING32,scontrol) / OSD_DMA_SWAP_WIDTH) *
         OSD_DMA_SWAP_WIDTH,OSD_DMA_SWAP_WIDTH);
   }

}
#endif

/*********************************************************************
*
*  SCSIhSwapping64PatchXferOpt
*
*     Patch xfer option and fast 20 into scb for swapping64 mode sequencer
*
*  Return Value:  none
*                  
*  Parameters:    hhcb
*                 hiob
*
*  Remarks:
*                  
*********************************************************************/
#if SCSI_SWAPPING64_MODE
void SCSIhSwapping64PatchXferOpt (SCSI_HHCB SCSI_HPTR hhcb,
                                  SCSI_HIOB SCSI_IPTR hiob)
{
   SCSI_DEVICE SCSI_DPTR deviceTable = SCSI_TARGET_UNIT(hiob)->deviceTable;
   SCSI_UEXACT8 uexact8Value;
   SCSI_SCB_SWAPPING64 SCSI_IPTR scbBuffer =
      (SCSI_SCB_SWAPPING64 SCSI_IPTR )hiob->scbDescriptor->scbBuffer.virtualAddress;

   /* patch the transfer rate */
   SCSI_PUT_LITTLE_ENDIAN8(hhcb,scbBuffer,
         OSDoffsetof(SCSI_SCB_SWAPPING64,xfer_option),
         deviceTable->xferOptionHost);

   /* patch the fast flag */
   if (deviceTable->SCSI_DF_fast20EnableW)
   {
      SCSI_GET_LITTLE_ENDIAN8(hhcb,&uexact8Value,scbBuffer,
         OSDoffsetof(SCSI_SCB_SWAPPING64,scontrol));
      uexact8Value |= SCSI_W_FAST20;
      SCSI_PUT_LITTLE_ENDIAN8(hhcb,scbBuffer,
         OSDoffsetof(SCSI_SCB_SWAPPING64,scontrol),uexact8Value);

      SCSI_FLUSH_CACHE(((SCSI_UEXACT8 SCSI_HPTR)scbBuffer) +
         (OSDoffsetof(SCSI_SCB_SWAPPING64,scontrol) / OSD_DMA_SWAP_WIDTH) *
         OSD_DMA_SWAP_WIDTH,OSD_DMA_SWAP_WIDTH);
   }

}
#endif

/*********************************************************************

*  SCSIhStandardAdvPatchXferOpt
*
*     Patch xfer option into scb for standard adv mode sequencer
*
*  Return Value:  none
*                  
*  Parameters:    hhcb
*                 hiob
*
*  Remarks:
*                  
*********************************************************************/
#if SCSI_STANDARD_ADVANCED_MODE
void SCSIhStandardAdvPatchXferOpt (SCSI_HHCB SCSI_HPTR hhcb,
                                   SCSI_HIOB SCSI_IPTR hiob)
{
   SCSI_DEVICE SCSI_DPTR deviceTable = SCSI_TARGET_UNIT(hiob)->deviceTable;
   SCSI_SCB_STANDARD_ADVANCED SCSI_IPTR scbBuffer =
      (SCSI_SCB_STANDARD_ADVANCED SCSI_IPTR )hiob->scbDescriptor->scbBuffer.virtualAddress;

   /* patch the transfer rate */
   SCSI_PUT_LITTLE_ENDIAN16(hhcb,scbBuffer,
         OSDoffsetof(SCSI_SCB_STANDARD_ADVANCED,xfer_option0),
         deviceTable->xferOptionHost);

}
#endif /* SCSI_STANDARD_ADVANCED_MODE */

/*********************************************************************

*  SCSIhSwappingAdvPatchXferOpt
*
*     Patch xfer option into scb for swapping adv mode sequencer
*
*  Return Value:  none
*                  
*  Parameters:    hhcb
*                 hiob
*
*  Remarks:
*                  
*********************************************************************/
#if SCSI_SWAPPING_ADVANCED_MODE
void SCSIhSwappingAdvPatchXferOpt (SCSI_HHCB SCSI_HPTR hhcb,
                                   SCSI_HIOB SCSI_IPTR hiob)
{
   SCSI_DEVICE SCSI_DPTR deviceTable;
   SCSI_SCB_SWAPPING_ADVANCED SCSI_IPTR scbBuffer =
      (SCSI_SCB_SWAPPING_ADVANCED SCSI_IPTR )hiob->scbDescriptor->scbBuffer.virtualAddress;
#if SCSI_TARGET_OPERATION
   SCSI_NEXUS SCSI_XPTR nexusHandle;
   SCSI_UEXACT16 xferOption;
#endif  /* SCSI_TARGET_OPERATION */

#if SCSI_TARGET_OPERATION
   if (hiob->SCSI_IP_targetMode)
   {
      nexusHandle = SCSI_NEXUS_UNIT(hiob);
      deviceTable = SCSI_GET_NODE(nexusHandle)->deviceTable;

#if SCSI_MULTIPLEID_SUPPORT
      xferOption = SCSI_hGETDEVICETABLEXFEROPT(deviceTable,nexusHandle->selectedID);
#else
      xferOption = SCSI_hGETDEVICETABLEXFEROPT(deviceTable,0);
#endif /* SCSI_MULTIPLEID_SUPPORT */      
   }
   else
   {
      deviceTable = SCSI_TARGET_UNIT(hiob)->deviceTable;
      xferOption = SCSI_hGETDEVICETABLEXFEROPT(deviceTable,hhcb->hostScsiID);
   }

   /* patch the transfer rate */
   SCSI_PUT_LITTLE_ENDIAN16(hhcb,scbBuffer,
         OSDoffsetof(SCSI_SCB_SWAPPING_ADVANCED,xfer_option0),
         xferOption);   

#else
   deviceTable = SCSI_TARGET_UNIT(hiob)->deviceTable;
   /* patch the transfer rate */
   SCSI_PUT_LITTLE_ENDIAN16(hhcb,scbBuffer,
         OSDoffsetof(SCSI_SCB_SWAPPING_ADVANCED,xfer_option0),
         deviceTable->xferOptionHost);
#endif /* SCSI_TARGET_OPERATION */

}
#endif /* SCSI_SWAPPING_ADVANCED_MODE */

/*********************************************************************

*  SCSIhStandard160mPatchXferOpt
*
*     Patch xfer option into scb for standard 160m mode sequencer
*
*  Return Value:  none
*                  
*  Parameters:    hhcb
*                 hiob
*
*  Remarks:
*                  
*********************************************************************/
#if SCSI_STANDARD_160M_MODE
void SCSIhStandard160mPatchXferOpt (SCSI_HHCB SCSI_HPTR hhcb,
                                    SCSI_HIOB SCSI_IPTR hiob)
{
   SCSI_DEVICE SCSI_DPTR deviceTable = SCSI_TARGET_UNIT(hiob)->deviceTable;
   SCSI_SCB_STANDARD_160M SCSI_IPTR scbBuffer =
      (SCSI_SCB_STANDARD_160M SCSI_IPTR )hiob->scbDescriptor->scbBuffer.virtualAddress;

   if (hiob->SCSI_IF_tagEnable)
   {
      if (((SCSI_UEXACT8)deviceTable->xferOptionHost) == SCSI_NEEDNEGO)
      {
         /* patch for tagged queue cmd and sync negotiation needed */
         SCSI_PUT_LITTLE_ENDIAN8(hhcb,scbBuffer,
              OSDoffsetof(SCSI_SCB_STANDARD_160M,atn_length),0x04);
      }
      else
      {
         /* patch for tagged queue cmd and sync negotiation not needed */
         SCSI_PUT_LITTLE_ENDIAN8(hhcb,scbBuffer,
              OSDoffsetof(SCSI_SCB_STANDARD_160M,atn_length),0x03);
      }
   }
   else
   {
      if (((SCSI_UEXACT8)deviceTable->xferOptionHost) == SCSI_NEEDNEGO)
      {
         /* patch for non-tagged queue cmd and sync negotiation needed */
         SCSI_PUT_LITTLE_ENDIAN8(hhcb,scbBuffer,
              OSDoffsetof(SCSI_SCB_STANDARD_160M,atn_length),0x02);
      }
      else
      {
         /* patch for non-tagged queue cmd and sync negotiation not needed */
         SCSI_PUT_LITTLE_ENDIAN8(hhcb,scbBuffer,
              OSDoffsetof(SCSI_SCB_STANDARD_160M,atn_length),0x01);
      }
   }
}
#endif /* SCSI_STANDARD_160M_MODE */

/*********************************************************************

*  SCSIhSwapping160mPatchXferOpt
*
*     Patch xfer option into scb for swapping 160m mode sequencer
*
*  Return Value:  none
*                  
*  Parameters:    hhcb
*                 hiob
*
*  Remarks:
*                  
*********************************************************************/
#if SCSI_SWAPPING_160M_MODE
void SCSIhSwapping160mPatchXferOpt (SCSI_HHCB SCSI_HPTR hhcb,
                                    SCSI_HIOB SCSI_IPTR hiob)
{
   SCSI_DEVICE SCSI_DPTR deviceTable;
   SCSI_SCB_SWAPPING_160M SCSI_IPTR scbBuffer =
      (SCSI_SCB_SWAPPING_160M SCSI_IPTR )hiob->scbDescriptor->scbBuffer.virtualAddress;

#if SCSI_TARGET_OPERATION && SCSI_MULTIPLEID_SUPPORT
   SCSI_NEXUS SCSI_XPTR nexusHandle;
   SCSI_UEXACT16 xferOption;
#endif /* SCSI_TARGET_OPERATION && SCSI_MULTIPLEID_SUPPORT */

#if SCSI_TARGET_OPERATION && SCSI_MULTIPLEID_SUPPORT
   if (hiob->SCSI_IP_targetMode)
   {
      nexusHandle = SCSI_NEXUS_UNIT(hiob);
      deviceTable = SCSI_GET_NODE(nexusHandle)->deviceTable;

      xferOption = SCSI_hGETDEVICETABLEXFEROPT(deviceTable,nexusHandle->selectedID);
   }
   else
   {
      deviceTable = SCSI_TARGET_UNIT(hiob)->deviceTable;
      xferOption = SCSI_hGETDEVICETABLEXFEROPT(deviceTable,hhcb->hostScsiID);
   }

   /* patch the transfer rate */
   SCSI_PUT_LITTLE_ENDIAN16(hhcb,scbBuffer,
         OSDoffsetof(SCSI_SCB_SWAPPING_160M,xfer_option0),
         xferOption);

#else /* SCSI_TARGET_OPERATION && SCSI_MULTIPLEID_SUPPORT */
   deviceTable = SCSI_TARGET_UNIT(hiob)->deviceTable;

   if (hiob->SCSI_IF_tagEnable)
   {
      if (((SCSI_UEXACT8)deviceTable->xferOptionHost) == SCSI_NEEDNEGO)
      {
         /* patch for tagged queue cmd and sync negotiation needed */
         SCSI_PUT_LITTLE_ENDIAN8(hhcb,scbBuffer,
              OSDoffsetof(SCSI_SCB_SWAPPING_160M,atn_length),0x04);
      }
      else
      {
         /* patch for tagged queue cmd and sync negotiation not needed */
         SCSI_PUT_LITTLE_ENDIAN8(hhcb,scbBuffer,
              OSDoffsetof(SCSI_SCB_SWAPPING_160M,atn_length),0x03);
      }
   }
   else
   {
      if (((SCSI_UEXACT8)deviceTable->xferOptionHost) == SCSI_NEEDNEGO)
      {
         /* patch for non-tagged queue cmd and sync negotiation needed */
         SCSI_PUT_LITTLE_ENDIAN8(hhcb,scbBuffer,
              OSDoffsetof(SCSI_SCB_SWAPPING_160M,atn_length),0x02);
      }
      else
      {
         /* patch for non-tagged queue cmd and sync negotiation not needed */
         SCSI_PUT_LITTLE_ENDIAN8(hhcb,scbBuffer,
              OSDoffsetof(SCSI_SCB_SWAPPING_160M,atn_length),0x01);
      }
   }
#endif /* SCSI_TARGET_OPERATION && SCSI_MULTIPLEID_SUPPORT */

}
#endif /* SCSI_SWAPPING_160M_MODE */

/*********************************************************************
*
*  SCSIhStandard64QHead
*
*     Enqueue at head of execution queue for standard 64 mode sequencer
*
*  Return Value:  none
*                  
*  Parameters:    hhcb
*                 hiob
*
*  Remarks:       sequencer must be paused when this routine
*                 get referenced.
*                  
*********************************************************************/
#if SCSI_STANDARD64_MODE
void SCSIhStandard64QHead (SCSI_HHCB SCSI_HPTR hhcb,
                           SCSI_HIOB SCSI_IPTR hiob)
{
   SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_HIOB SCSI_IPTR hiobSelecting;
   SCSI_UEXACT8 scbNumber;
   SCSI_UEXACT8 nextScb;
   SCSI_UEXACT8 scbptr;
   SCSI_UEXACT8 scsiseq;

   scbptr = OSD_INEXACT8(SCSI_AICREG(SCSI_SCBPTR));   /* Save SCBPTR */

   /* To handle the check condition received during the reselection */
   /* at one HIOB while a selection is starting from another HIOB. */
   /* Both HIOBs are for the same targert. */ 
   if (((scsiseq = OSD_INEXACT8(SCSI_AICREG(SCSI_SCSISEQ))) & SCSI_ENSELO) &&
       (hiob->trgStatus == SCSI_UNIT_CHECK))
   {
      scbNumber = OSD_INEXACT8(SCSI_AICREG(SCSI_S64_WAITING_SCB));

      /* Get the HIOB that currently do the selection and check if */
      /* its target ID matches with the one of the current HIOB. */
      hiobSelecting = SCSI_ACTPTR[scbNumber];
      if ((SCSI_TARGET_UNIT(hiobSelecting)->hhcb == hhcb) &&
          (SCSI_TARGET_UNIT(hiobSelecting)->scsiID == SCSI_TARGET_UNIT(hiob)->scsiID))
      {
         /* Put it back to head of execution queue */
         nextScb = OSD_INEXACT8(SCSI_AICREG(SCSI_S64_Q_EXE_HEAD));
         OSD_OUTEXACT8(SCSI_AICREG(SCSI_S64_Q_EXE_HEAD), scbNumber);

         if (nextScb == SCSI_NULL_SCB)
         {
            OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+OSDoffsetof(SCSI_SCB_STANDARD64,
                SCSI_S64_q_next)), SCSI_NULL_SCB);
            OSD_OUTEXACT8(SCSI_AICREG(SCSI_S64_Q_EXE_TAIL), scbNumber);
         }
         else
         {
            OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+OSDoffsetof(SCSI_SCB_STANDARD64,
            SCSI_S64_q_next)), nextScb);
         }
         
         /* Disable any outstanding selection */
         OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSISEQ),
             scsiseq & ~(SCSI_TEMODEO+SCSI_ENSELO+SCSI_ENAUTOATNO+SCSI_ENAUTOATNI));
         scsiseq = OSD_INEXACT8(SCSI_AICREG(SCSI_SCSISEQ)); /* To flush the cache */
      }
   }
   
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCBPTR), hiob->scbNumber);

   /* Put it to head of execution queue */
   nextScb = OSD_INEXACT8(SCSI_AICREG(SCSI_S64_Q_EXE_HEAD));
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_S64_Q_EXE_HEAD), hiob->scbNumber);

   if (nextScb == SCSI_NULL_SCB)
   {
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+OSDoffsetof(SCSI_SCB_STANDARD64,
          SCSI_S64_q_next)), SCSI_NULL_SCB);
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_S64_Q_EXE_TAIL), hiob->scbNumber);
   }
   else
   {
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+OSDoffsetof(SCSI_SCB_STANDARD64,
      SCSI_S64_q_next)), nextScb);
   }

   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCBPTR), scbptr);   /* Restore SCBPTR */
}
#endif

/*********************************************************************
*
*  SCSIhSwapping32QHead
*
*     Enqueue at head for swapping 32 mode sequencer
*
*  Return Value:  none
*                  
*  Parameters:    hhcb
*                 hiob
*
*  Remarks:       sequencer must be paused when this routine
*                 get referenced.
*                  
*********************************************************************/
#if SCSI_SWAPPING32_MODE
void SCSIhSwapping32QHead (SCSI_HHCB SCSI_HPTR hhcb,
                           SCSI_HIOB SCSI_IPTR hiob)
{
   SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_HIOB SCSI_IPTR hiobSelecting;
   SCSI_UEXACT8 saveScb;
   SCSI_UEXACT8 qIndex;
   SCSI_UEXACT8 scbNumber;
   SCSI_UEXACT8 scsiseq;
  
   /* save scb number for the head of queue from scratch RAM */
   saveScb = OSD_INEXACT8(SCSI_AICREG(SCSI_hQ_NEXT_SCB(hhcb)));

   /* To handle the check condition received during the reselection */
   /* at one HIOB while a selection is starting from another HIOB. */
   /* Both HIOBs are for the same target. */
   if (((scsiseq = OSD_INEXACT8(SCSI_AICREG(SCSI_SCSISEQ))) & SCSI_ENSELO) &&
       (hiob->trgStatus == SCSI_UNIT_CHECK))
   {
      /* save scb number of waiting scb */
      scbNumber =OSD_INEXACT8(SCSI_AICREG(SCSI_hWAITING_SCB(hhcb))); 

      /* Get the HIOB that currently do the selection and check if */
      /* its target ID matches with the one of the current HIOB. */
      hiobSelecting = SCSI_ACTPTR[scbNumber];
      if ((SCSI_TARGET_UNIT(hiobSelecting)->hhcb == hhcb) &&
          (SCSI_TARGET_UNIT(hiobSelecting)->scsiID == SCSI_TARGET_UNIT(hiob)->scsiID))
      {
          /* decrement queue to add entry */    
          qIndex = OSD_INEXACT8(SCSI_AICREG(SCSI_hQ_NEW_TAIL(hhcb)));
          OSD_OUTEXACT8(SCSI_AICREG(SCSI_hQ_NEW_TAIL(hhcb)),--qIndex);
          
          /* updating scb number in the scb for selecting hiob */                
          SCSI_hUPDATE_NEXT_SCB_NUM(hhcb,hiobSelecting,saveScb);
          
          saveScb=scbNumber;
         
         /* Disable any outstanding selection */
         OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSISEQ),
             scsiseq & ~(SCSI_TEMODEO+SCSI_ENSELO+SCSI_ENAUTOATNO+SCSI_ENAUTOATNI));
         scsiseq = OSD_INEXACT8(SCSI_AICREG(SCSI_SCSISEQ)); /* To flush the cache */
      }
   }

   /* update scb number in scb, only if Q_NEXT_SCB in scratch points to a  */
   /* valid hiob (if it is not valid then saveScb and hiob->scbNumber will */
   /* be equal, in which case we should not update next scb number in scb  */ 
   if (saveScb != hiob->scbNumber)
   {
      SCSI_hUPDATE_NEXT_SCB_NUM(hhcb,hiob,saveScb);
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_hQ_NEXT_SCB(hhcb)),hiob->scbNumber);
      /* decrement the tail of the queue to add entry */
      qIndex = OSD_INEXACT8(SCSI_AICREG(SCSI_hQ_NEW_TAIL(hhcb)));
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_hQ_NEW_TAIL(hhcb)), --qIndex);
   }
}
#endif /* SCSI_SWAPPING32_MODE */

/*********************************************************************
*
*  SCSIhSwapping64QHead
*
*     Enqueue at head for swapping 64 mode sequencer
*
*  Return Value:  none
*                  
*  Parameters:    hhcb
*                 hiob
*
*  Remarks:       sequencer must be paused when this routine
*                 get referenced.
*                  
*********************************************************************/
#if SCSI_SWAPPING64_MODE
void SCSIhSwapping64QHead (SCSI_HHCB SCSI_HPTR hhcb,
                           SCSI_HIOB SCSI_IPTR hiob)
{
   SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_HIOB SCSI_IPTR hiobSelecting;
   SCSI_UEXACT8 saveScb;
   SCSI_UEXACT8 qIndex;
   SCSI_UEXACT8 scbNumber;
   SCSI_UEXACT8 scsiseq;
  
   /* save scb number for the head of queue from scratch RAM */
   saveScb = OSD_INEXACT8(SCSI_AICREG(SCSI_hQ_NEXT_SCB(hhcb)));

   /* To handle the check condition received during the reselection */
   /* at one HIOB while a selection is starting from another HIOB. */
   /* Both HIOBs are for the same target. */
   if (((scsiseq = OSD_INEXACT8(SCSI_AICREG(SCSI_SCSISEQ))) & SCSI_ENSELO) &&
       (hiob->trgStatus == SCSI_UNIT_CHECK))
   {
      /* save scb number of waiting scb */
      scbNumber =OSD_INEXACT8(SCSI_AICREG(SCSI_hWAITING_SCB(hhcb))); 

      /* Get the HIOB that currently do the selection and check if */
      /* its target ID matches with the one of the current HIOB. */
      hiobSelecting = SCSI_ACTPTR[scbNumber];
      if ((SCSI_TARGET_UNIT(hiobSelecting)->hhcb == hhcb) &&
          (SCSI_TARGET_UNIT(hiobSelecting)->scsiID == SCSI_TARGET_UNIT(hiob)->scsiID))
      {
          /* decrement queue to add entry */    
          qIndex = OSD_INEXACT8(SCSI_AICREG(SCSI_hQ_NEW_TAIL(hhcb)));
          OSD_OUTEXACT8(SCSI_AICREG(SCSI_hQ_NEW_TAIL(hhcb)),--qIndex);
          
          /* updating scb number in the scb for selecting hiob */                
          SCSI_hUPDATE_NEXT_SCB_NUM(hhcb,hiobSelecting,saveScb);
          
          saveScb=scbNumber;
         
         /* Disable any outstanding selection */
         OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSISEQ),
             scsiseq & ~(SCSI_TEMODEO+SCSI_ENSELO+SCSI_ENAUTOATNO+SCSI_ENAUTOATNI));
         scsiseq = OSD_INEXACT8(SCSI_AICREG(SCSI_SCSISEQ)); /* To flush the cache */
      }
   }

   /* update scb number in scb, only if Q_NEXT_SCB in scratch points to a  */
   /* valid hiob (if it is not valid then saveScb and hiob->scbNumber will */
   /* be equal, in which case we should not update next scb number in scb  */ 
   if (saveScb != hiob->scbNumber)
   {
      SCSI_hUPDATE_NEXT_SCB_NUM(hhcb,hiob,saveScb);
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_hQ_NEXT_SCB(hhcb)),hiob->scbNumber);
      /* decrement the tail of the queue to add entry */
      qIndex = OSD_INEXACT8(SCSI_AICREG(SCSI_hQ_NEW_TAIL(hhcb)));
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_hQ_NEW_TAIL(hhcb)), --qIndex);
   }
}
#endif /* SCSI_SWAPPING64_MODE */

/*********************************************************************
*
*  SCSIhSwappingAdvQHead
*
*     Enqueue at head for swapping advanced mode sequencer
*
*  Return Value:  none
*                  
*  Parameters:    hhcb
*                 hiob
*
*  Remarks:       sequencer must be paused when this routine
*                 get referenced.
*                  
*********************************************************************/
#if SCSI_SWAPPING_ADVANCED_MODE
void SCSIhSwappingAdvQHead (SCSI_HHCB SCSI_HPTR hhcb,
                            SCSI_HIOB SCSI_IPTR hiob)
{
   SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_HIOB SCSI_IPTR hiobSelecting;
   SCSI_UEXACT8 saveScb, newScb;
   SCSI_UEXACT8 qIndex;
   SCSI_UEXACT8 scbNumber;
   SCSI_UEXACT8 scsiseq;
   SCSI_UEXACT8 scbptr;
   SCSI_UINT32 i = 0;  

   /* save scb number for the head of queue from scratch RAM */
   saveScb = OSD_INEXACT8(SCSI_AICREG(SCSI_hQ_NEXT_SCB(hhcb)));

   /* To handle the check condition received during the reselection */
   /* at one HIOB while a selection is starting from another HIOB. */
   /* Both HIOBs are for the same target. */
#if SCSI_TARGET_OPERATION
   if (!hiob->SCSI_IP_targetMode &&
       ((scsiseq = OSD_INEXACT8(SCSI_AICREG(SCSI_SCSISEQ))) & SCSI_ENSELO) &&
       (hiob->trgStatus == SCSI_UNIT_CHECK))
#else
#if SCSI_TRIDENT_PROTO
   if (((scsiseq = OSD_INEXACT8(SCSI_AICREG(SCSI_hSCSISEQ_COPY(hhcb)))) & SCSI_ENSELO) &&
       (hiob->trgStatus == SCSI_UNIT_CHECK))
#else /* SCSI_TRIDENT_PROTO */
   if (((scsiseq = OSD_INEXACT8(SCSI_AICREG(SCSI_SCSISEQ))) & SCSI_ENSELO) &&
       (hiob->trgStatus == SCSI_UNIT_CHECK))
#endif /* SCSI_TRIDENT_PROTO */
#endif /* SCSI_TARGET_OPERATION */
   {
      /* save scb number of waiting scb */
      scbNumber = OSD_INEXACT8(SCSI_AICREG(SCSI_hWAITING_SCB(hhcb))); 

      /* Get the HIOB that currently do the selection and check if */
      /* its target ID matches with the one of the current HIOB. */
      hiobSelecting = SCSI_ACTPTR[scbNumber];
      if ((SCSI_TARGET_UNIT(hiobSelecting)->hhcb == hhcb) &&
          (SCSI_TARGET_UNIT(hiobSelecting)->scsiID == SCSI_TARGET_UNIT(hiob)->scsiID))
      {
          /* decrement queue to add entry */    
          qIndex = OSD_INEXACT8(SCSI_AICREG(SCSI_hQ_NEW_TAIL(hhcb)));
          OSD_OUTEXACT8(SCSI_AICREG(SCSI_hQ_NEW_TAIL(hhcb)),--qIndex);
          
          /* updating scb number in the scb for selecting hiob */                
          SCSI_hUPDATE_NEXT_SCB_NUM(hhcb,hiobSelecting,saveScb);
          
          saveScb=scbNumber;
         
         /* Disable any outstanding selection */
         OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSISEQ),
             scsiseq & ~(SCSI_TEMODEO+SCSI_ENSELO+SCSI_ENAUTOATNO+SCSI_ENAUTOATNI));
         scsiseq = OSD_INEXACT8(SCSI_AICREG(SCSI_SCSISEQ)); /* To flush the cache */
#if SCSI_TRIDENT_PROTO
         OSD_OUTEXACT8(SCSI_AICREG(SCSI_hSCSISEQ_COPY(hhcb)), scsiseq);
         scsiseq = OSD_INEXACT8(SCSI_AICREG(SCSI_hSCSISEQ_COPY(hhcb))); /* To flush the cache */
#endif /* SCSI_TRIDENT_PROTO */
      }
   }

   /* Check if there is any SCB dma going on from host memory to scb array */
   if ((OSD_INEXACT8(SCSI_AICREG(SCSI_CCSCBCTL)) & SCSI_CCARREN) &&
       (OSD_INEXACT8(SCSI_AICREG(SCSI_CCSCBCTL)) & SCSI_CCSCBDIR))
   {

      /* If so, wait till the 'intransit SCB' completes */
      while ((!(OSD_INEXACT8(SCSI_AICREG(SCSI_CCSCBCTL)) & SCSI_ARRDONE)) &&
             (i++ < 400000))
      {
         OSD_TIMER (hhcb);     
      }

      scbptr = OSD_INEXACT8(SCSI_AICREG(SCSI_SCBPTR));   /* Save SCBPTR */
 
      /* Check if the SCB which just got into the SCB array is for a fresh */
      /* selection (and not for a reselection, as the SCB could be */
      /* transferred even this case) */
      if (scbptr != OSD_INEXACT8(SCSI_AICREG(SCSI_CCSCBPTR)))
      {
         /* newScb = next SCB sitting inside the 'intransit SCB' */
         newScb = SCSI_hOBTAIN_NEXT_SCB_NUM(hhcb,SCSI_ACTPTR[saveScb]);

         /* Modify the next_SCB of the 'intransit SCB' with the one which is */
         /* supposed to go to head... */
         /* in the host... */
         SCSI_hUPDATE_NEXT_SCB_NUM(hhcb,SCSI_ACTPTR[saveScb],hiob->scbNumber);
      
         /* as well as in the internal SCB array */
         OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCBPTR),
                       OSD_INEXACT8(SCSI_AICREG(SCSI_CCSCBPTR)));

         OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
                       OSDoffsetof(SCSI_SCB_SWAPPING_ADVANCED,next_SCB)),
                       hiob->scbNumber);

         OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCBPTR), scbptr);   /* Restore SCBPTR */

         /* We need to update (and advance queue tail) only if newScb is not */
         /* equal to the SCB which we need to put in the head */
         if (newScb != hiob->scbNumber)
         {
            SCSI_hUPDATE_NEXT_SCB_NUM(hhcb,hiob,newScb);
            /* decrement the tail of the queue to add entry */
            qIndex = OSD_INEXACT8(SCSI_AICREG(SCSI_hQ_NEW_TAIL(hhcb)));
            OSD_OUTEXACT8(SCSI_AICREG(SCSI_hQ_NEW_TAIL(hhcb)), --qIndex);
         }

         return;
      }
   }

   /* update scb number in scb, only if Q_NEXT_SCB in scratch points to a  */
   /* valid hiob (if it is not valid then saveScb and hiob->scbNumber will */
   /* be equal, in which case we should not update next scb number in scb  */ 
   if (saveScb != hiob->scbNumber)
   {
      SCSI_hUPDATE_NEXT_SCB_NUM(hhcb,hiob,saveScb);
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_hQ_NEXT_SCB(hhcb)),hiob->scbNumber);
      /* decrement the tail of the queue to add entry */
      qIndex = OSD_INEXACT8(SCSI_AICREG(SCSI_hQ_NEW_TAIL(hhcb)));
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_hQ_NEW_TAIL(hhcb)), --qIndex);
   }
}
#endif

/*********************************************************************
*
*  SCSIhStandardAdvQHead
*
*     Enqueue at head of execution queue for standard advanced mode 
*     sequencer
*
*  Return Value:  none
*                  
*  Parameters:    hhcb
*                 hiob
*
*  Remarks:       sequencer must be paused when this routine
*                 get referenced.
*                  
*********************************************************************/
#if SCSI_STANDARD_ADVANCED_MODE
void SCSIhStandardAdvQHead (SCSI_HHCB SCSI_HPTR hhcb,
                           SCSI_HIOB SCSI_IPTR hiob)
{
   SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_HIOB SCSI_IPTR hiobSelecting;
   SCSI_UEXACT8 scbNumber;
   SCSI_UEXACT8 nextScb;
   SCSI_UEXACT8 scbptr;
   SCSI_UEXACT8 scsiseq;

   scbptr = OSD_INEXACT8(SCSI_AICREG(SCSI_SCBPTR));   /* Save SCBPTR */

   /* To handle the check condition received during the reselection */
   /* at one HIOB while a selection is starting from another HIOB. */
   /* Both HIOBs are for the same targert. */ 
   if (((scsiseq = OSD_INEXACT8(SCSI_AICREG(SCSI_SCSISEQ))) & SCSI_ENSELO) &&
       (hiob->trgStatus == SCSI_UNIT_CHECK))
   {
      scbNumber = OSD_INEXACT8(SCSI_AICREG(SCSI_SADV_WAITING_SCB));

      /* Get the HIOB that currently do the selection and check if */
      /* its target ID matches with the one of the current HIOB. */
      hiobSelecting = SCSI_ACTPTR[scbNumber];
      if ((SCSI_TARGET_UNIT(hiobSelecting)->hhcb == hhcb) &&
          (SCSI_TARGET_UNIT(hiobSelecting)->scsiID == SCSI_TARGET_UNIT(hiob)->scsiID))
      {
         /* Put it back to head of execution queue */
         nextScb = OSD_INEXACT8(SCSI_AICREG(SCSI_SADV_Q_EXE_HEAD));
         OSD_OUTEXACT8(SCSI_AICREG(SCSI_SADV_Q_EXE_HEAD), scbNumber);

         if (nextScb == SCSI_NULL_SCB)
         {
            OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
                  OSDoffsetof(SCSI_SCB_STANDARD_ADVANCED,
                  SCSI_SADV_q_next)), SCSI_NULL_SCB);
            OSD_OUTEXACT8(SCSI_AICREG(SCSI_SADV_Q_EXE_TAIL), scbNumber);
         }
         else
         {
            OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
                  OSDoffsetof(SCSI_SCB_STANDARD_ADVANCED,
                  SCSI_SADV_q_next)), nextScb);
         }
         
         /* Disable any outstanding selection */
         OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSISEQ),
             scsiseq & ~(SCSI_TEMODEO+SCSI_ENSELO+SCSI_ENAUTOATNO+SCSI_ENAUTOATNI));
         scsiseq = OSD_INEXACT8(SCSI_AICREG(SCSI_SCSISEQ)); /* To flush the cache */
      }
   }
   
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCBPTR), hiob->scbNumber);

   /* Put it to head of execution queue */
   nextScb = OSD_INEXACT8(SCSI_AICREG(SCSI_SADV_Q_EXE_HEAD));
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SADV_Q_EXE_HEAD), hiob->scbNumber);

   if (nextScb == SCSI_NULL_SCB)
   {
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
            OSDoffsetof(SCSI_SCB_STANDARD_ADVANCED,
            SCSI_SADV_q_next)), SCSI_NULL_SCB);
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SADV_Q_EXE_TAIL), hiob->scbNumber);
   }
   else
   {
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
            OSDoffsetof(SCSI_SCB_STANDARD_ADVANCED,
            SCSI_SADV_q_next)), nextScb);
   }

   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCBPTR), scbptr);   /* Restore SCBPTR */
}
#endif

/*********************************************************************
*
*  SCSIhSwapping160mQHead
*
*     Enqueue at head for swapping 160m mode sequencer
*
*  Return Value:  none
*                  
*  Parameters:    hhcb
*                 hiob
*
*  Remarks:       sequencer must be paused when this routine
*                 get referenced.
*                  
*********************************************************************/
#if SCSI_SWAPPING_160M_MODE
void SCSIhSwapping160mQHead (SCSI_HHCB SCSI_HPTR hhcb,
                             SCSI_HIOB SCSI_IPTR hiob)
{
   SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_HIOB SCSI_IPTR hiobSelecting;
   SCSI_UEXACT8 saveScb, newScb;
   SCSI_UEXACT8 qIndex;
   SCSI_UEXACT8 scbNumber;
   SCSI_UEXACT8 scsiseq;
   SCSI_UEXACT8 scbptr;
   SCSI_UINT32 i = 0;
  
   /* save scb number for the head of queue from scratch RAM */
   saveScb = OSD_INEXACT8(SCSI_AICREG(SCSI_W160M_Q_NEXT_SCB));

   /* To handle the check condition received during the reselection */
   /* at one HIOB while a selection is starting from another HIOB. */
   /* Both HIOBs are for the same target. */
   if (((scsiseq = OSD_INEXACT8(SCSI_AICREG(SCSI_SCSISEQ))) & SCSI_ENSELO) &&
#if SCSI_TARGET_OPERATION
       (!hiob->SCSI_IP_targetMode) &&
#endif /* SCSI_TARGET_OPERATION */
       (hiob->trgStatus == SCSI_UNIT_CHECK))
   {
      /* save scb number of waiting scb */
      scbNumber = OSD_INEXACT8(SCSI_AICREG(SCSI_W160M_WAITING_SCB)); 

      /* Get the HIOB that currently do the selection and check if */
      /* its target ID matches with the one of the current HIOB. */
      hiobSelecting = SCSI_ACTPTR[scbNumber];
      if ((SCSI_TARGET_UNIT(hiobSelecting)->hhcb == hhcb) &&
          (SCSI_TARGET_UNIT(hiobSelecting)->scsiID == SCSI_TARGET_UNIT(hiob)->scsiID))
      {
          /* decrement queue to add entry */    
          qIndex = OSD_INEXACT8(SCSI_AICREG(SCSI_W160M_Q_NEW_TAIL));
          OSD_OUTEXACT8(SCSI_AICREG(SCSI_W160M_Q_NEW_TAIL),--qIndex);
          
          /* updating scb number in the scb for selecting hiob */                
          SCSI_hUPDATE_NEXT_SCB_NUM(hhcb,hiobSelecting,saveScb);
          
          saveScb = scbNumber;
         
         /* Disable any outstanding selection */
         OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSISEQ),
             scsiseq & ~(SCSI_TEMODEO+SCSI_ENSELO+SCSI_ENAUTOATNO+SCSI_ENAUTOATNI));
         scsiseq = OSD_INEXACT8(SCSI_AICREG(SCSI_SCSISEQ)); /* To flush the cache */
      }
   }

   /* Check if there is any SCB dma going on from host memory to scb array */
   if ((OSD_INEXACT8(SCSI_AICREG(SCSI_CCSCBCTL)) & SCSI_CCARREN) &&
       (OSD_INEXACT8(SCSI_AICREG(SCSI_CCSCBCTL)) & SCSI_CCSCBDIR))
   {
      /* If so, wait till the 'intransit SCB' completes */
      while ((!(OSD_INEXACT8(SCSI_AICREG(SCSI_CCSCBCTL)) & SCSI_ARRDONE)) &&
             (i++ < 400000))
      {
         OSD_TIMER (hhcb);     
      }

      scbptr = OSD_INEXACT8(SCSI_AICREG(SCSI_SCBPTR));   /* Save SCBPTR */
 
      /* Check if the SCB which just got into the SCB array is for a fresh */
      /* selection (and not for a reselection, as the SCB could be */
      /* transferred even this case) */
      if (scbptr != OSD_INEXACT8(SCSI_AICREG(SCSI_CCSCBPTR)))
      {
         /* newScb = next SCB sitting inside the 'intransit SCB' */
         newScb = SCSI_hOBTAIN_NEXT_SCB_NUM(hhcb,SCSI_ACTPTR[saveScb]);

         /* Modify the next_SCB of the 'intransit SCB' with the one which is */
         /* supposed to go to head... */
         /* in the host... */
         SCSI_hUPDATE_NEXT_SCB_NUM(hhcb,SCSI_ACTPTR[saveScb],hiob->scbNumber);
      
         /* as well as in the internal SCB array */
         OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCBPTR),
             OSD_INEXACT8(SCSI_AICREG(SCSI_CCSCBPTR)));

         OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
             OSDoffsetof(SCSI_SCB_SWAPPING_160M,next_SCB)),hiob->scbNumber);

         OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCBPTR), scbptr);   /* Restore SCBPTR */

         /* We need to update (and advance queue tail) only if newScb is not */
         /* equal to the SCB which we need to put in the head */
         if (newScb != hiob->scbNumber)
         {
            SCSI_hUPDATE_NEXT_SCB_NUM(hhcb,hiob,newScb);
            /* decrement the tail of the queue to add entry */
            qIndex = OSD_INEXACT8(SCSI_AICREG(SCSI_W160M_Q_NEW_TAIL));
            OSD_OUTEXACT8(SCSI_AICREG(SCSI_W160M_Q_NEW_TAIL), --qIndex);
         }

         return;
      }
   }

   /* update scb number in scb, only if Q_NEXT_SCB in scratch points to a  */
   /* valid hiob (if it is not valid then saveScb and hiob->scbNumber will */
   /* be equal, in which case we should not update next scb number in scb  */ 
   if (saveScb != hiob->scbNumber)
   {
      SCSI_hUPDATE_NEXT_SCB_NUM(hhcb,hiob,saveScb);
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_W160M_Q_NEXT_SCB),hiob->scbNumber);
      /* decrement the tail of the queue to add entry */
      qIndex = OSD_INEXACT8(SCSI_AICREG(SCSI_W160M_Q_NEW_TAIL));
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_W160M_Q_NEW_TAIL), --qIndex);
   }
}
#endif /* SCSI_SWAPPING_160M_MODE */

/*********************************************************************
*
*  SCSIhStandard160mQHead
*
*     Enqueue at head of execution queue for standard 160m mode 
*     sequencer
*
*  Return Value:  none
*                  
*  Parameters:    hhcb
*                 hiob
*
*  Remarks:       sequencer must be paused when this routine
*                 get referenced.
*                  
*********************************************************************/
#if SCSI_STANDARD_160M_MODE
void SCSIhStandard160mQHead (SCSI_HHCB SCSI_HPTR hhcb,
                             SCSI_HIOB SCSI_IPTR hiob)
{
   SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_HIOB SCSI_IPTR hiobSelecting;
   SCSI_UEXACT8 scbNumber;
   SCSI_UEXACT8 nextScb;
   SCSI_UEXACT8 scbptr;
   SCSI_UEXACT8 scsiseq;

   scbptr = OSD_INEXACT8(SCSI_AICREG(SCSI_SCBPTR));   /* Save SCBPTR */

   /* To handle the check condition received during the reselection */
   /* at one HIOB while a selection is starting from another HIOB. */
   /* Both HIOBs are for the same targert. */ 
   if (((scsiseq = OSD_INEXACT8(SCSI_AICREG(SCSI_SCSISEQ))) & SCSI_ENSELO) &&
       (hiob->trgStatus == SCSI_UNIT_CHECK))
   {
      scbNumber = OSD_INEXACT8(SCSI_AICREG(SCSI_S160M_WAITING_SCB));

      /* Get the HIOB that currently do the selection and check if */
      /* its target ID matches with the one of the current HIOB. */
      hiobSelecting = SCSI_ACTPTR[scbNumber];
      if ((SCSI_TARGET_UNIT(hiobSelecting)->hhcb == hhcb) &&
          (SCSI_TARGET_UNIT(hiobSelecting)->scsiID == SCSI_TARGET_UNIT(hiob)->scsiID))
      {
         /* Put it back to head of execution queue */
         nextScb = OSD_INEXACT8(SCSI_AICREG(SCSI_S160M_Q_EXE_HEAD));
         OSD_OUTEXACT8(SCSI_AICREG(SCSI_S160M_Q_EXE_HEAD), scbNumber);

         if (nextScb == SCSI_NULL_SCB)
         {
            OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
                OSDoffsetof(SCSI_SCB_STANDARD_160M,SCSI_S160M_q_next)),
                SCSI_NULL_SCB);
            OSD_OUTEXACT8(SCSI_AICREG(SCSI_S160M_Q_EXE_TAIL), scbNumber);
         }
         else
         {
            OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
                OSDoffsetof(SCSI_SCB_STANDARD_160M,SCSI_S160M_q_next)),
                nextScb);
         }
         
         /* Disable any outstanding selection */
         OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSISEQ),
             scsiseq & ~(SCSI_TEMODEO+SCSI_ENSELO+SCSI_ENAUTOATNO+SCSI_ENAUTOATNI));
         scsiseq = OSD_INEXACT8(SCSI_AICREG(SCSI_SCSISEQ)); /* To flush the cache */
      }
   }
   
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCBPTR), hiob->scbNumber);

   /* Put it to head of execution queue */
   nextScb = OSD_INEXACT8(SCSI_AICREG(SCSI_S160M_Q_EXE_HEAD));
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_S160M_Q_EXE_HEAD), hiob->scbNumber);

   if (nextScb == SCSI_NULL_SCB)
   {
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
          OSDoffsetof(SCSI_SCB_STANDARD_160M,SCSI_S160M_q_next)),
          SCSI_NULL_SCB);
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_S160M_Q_EXE_TAIL), hiob->scbNumber);
   }
   else
   {
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCB_BASE+
          OSDoffsetof(SCSI_SCB_STANDARD_160M,SCSI_S160M_q_next)), nextScb);
   }

   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCBPTR), scbptr);   /* Restore SCBPTR */
}
#endif /* SCSI_STANDARD_160M_MODE */

/*********************************************************************
*
*  SCSIhStandardRetrieveScb
*
*     Retrieve scb for standard and swapping mode
*
*  Return Value:  scb number retrieved
*                  
*  Parameters:    hhcb
*
*  Remarks:
*                  
*********************************************************************/
/* @@@@ do later - additional inline code needed to do this #if !SCSI_STREAMLINE_QIOPATH */
#if SCSI_NEW_MODE
SCSI_UEXACT8 SCSIhStandardRetrieveScb (SCSI_HHCB SCSI_HPTR hhcb)
{
   SCSI_QOUTFIFO_NEW SCSI_HPTR qOutPtr =
                SCSI_QOUT_PTR_ARRAY_NEW(hhcb) + hhcb->SCSI_HP_qDoneElement;
   SCSI_UEXACT8 scbNumber;
   SCSI_UEXACT8 qPassCnt;

   /* invalidate cache */
   SCSI_INVALIDATE_CACHE(((SCSI_UEXACT8 SCSI_HPTR)SCSI_QOUT_PTR_ARRAY_NEW(hhcb)) +
      ((hhcb->SCSI_HP_qDoneElement * sizeof(SCSI_QOUTFIFO_NEW)) /
      OSD_DMA_SWAP_WIDTH) * OSD_DMA_SWAP_WIDTH, OSD_DMA_SWAP_WIDTH);

   /* check if there is any valid entry */
   SCSI_GET_LITTLE_ENDIAN8(hhcb,&qPassCnt,qOutPtr,
      OSDoffsetof(SCSI_QOUTFIFO_NEW,quePassCnt));
   if (qPassCnt == hhcb->SCSI_HP_qDonePass)
   {
      /* found valid entry, get scb number and update */
      SCSI_GET_LITTLE_ENDIAN8(hhcb,&scbNumber,qOutPtr,
         OSDoffsetof(SCSI_QOUTFIFO_NEW,scbNumber));

      if (++hhcb->SCSI_HP_qDoneElement == (SCSI_UEXACT8) (hhcb->numberScbs + 1))
      {
         hhcb->SCSI_HP_qDoneElement = 0;
         ++hhcb->SCSI_HP_qDonePass;                  
      }
   }
   else
   {
      /* set scb number to null */
      scbNumber = SCSI_NULL_SCB;
   }

   return(scbNumber);
}
#endif
/* @@@@ do later - additional inline code needed to do this #endif */ /* !SCSI_STREAMLINE_QIOPATH */

/*********************************************************************
*
*  SCSIhStandardQoutcnt
*
*     Check if the QOUTFIFO is empty or not for
*     standard and swapping mode 
*
*  Return Value:  number of entries not checked in QOUTFIFO
*                  
*  Parameters:    hhcb
*
*  Remarks:
*                  
*********************************************************************/
/* @@@@ do later - additional inline code needed to do this #if !SCSI_STREAMLINE_QIOPATH */
#if SCSI_NEW_MODE
SCSI_UEXACT8 SCSIhStandardQoutcnt (SCSI_HHCB SCSI_HPTR hhcb)
{
   SCSI_UEXACT8 n = 0;
   SCSI_UEXACT8 qDonePass = hhcb->SCSI_HP_qDonePass;
   SCSI_UEXACT8 qDoneElement = hhcb->SCSI_HP_qDoneElement;
   SCSI_QOUTFIFO_NEW SCSI_HPTR qOutPtr = SCSI_QOUT_PTR_ARRAY_NEW(hhcb) +
                                          qDoneElement;
                                    
   SCSI_UEXACT8 quePassCnt;

   while(1)
   {
      /* invalidate cache */
      SCSI_INVALIDATE_CACHE(((SCSI_UEXACT8 SCSI_HPTR)SCSI_QOUT_PTR_ARRAY_NEW(hhcb)) +
         ((qDoneElement * sizeof(SCSI_QOUTFIFO_NEW)) /
         OSD_DMA_SWAP_WIDTH) * OSD_DMA_SWAP_WIDTH, OSD_DMA_SWAP_WIDTH);

      /* check if there is any valid entry */
      SCSI_GET_LITTLE_ENDIAN8(hhcb,&quePassCnt,qOutPtr,
         OSDoffsetof(SCSI_QOUTFIFO_NEW,quePassCnt));

      if (quePassCnt == qDonePass)
      {
         /* bump the number of outstanding entries in qoutfifo */
         ++n;

         /* increment index */
         if (++qDoneElement != (SCSI_UEXACT8) (hhcb->numberScbs + 1))
         {
            ++qOutPtr;
         }
         else
         {
            /* wrap around */
            ++qDonePass;
            qDoneElement = 0;
            qOutPtr = SCSI_QOUT_PTR_ARRAY_NEW(hhcb);
         }
      }
      else
      {
         break;
      }
   }

   return(n);
}
#endif
/* @@@@ do later - additional inline code needed to do this #endif */ /* !SCSI_STREAMLINE_QIOPATH */

/*********************************************************************
*
*  SCSIhStandard64QHeadBDR
*
*     Enqueue BDR scb at head of execution queue for standard 64 mode sequencer
*
*  Return Value:  none
*                  
*  Parameters:    hhcb
*                 hiob
*
*  Remarks:       sequencer must be paused when this routine
*                 get referenced.
*                  
*********************************************************************/
#if SCSI_STANDARD64_MODE
void SCSIhStandard64QHeadBDR (SCSI_HHCB SCSI_HPTR hhcb,
                              SCSI_HIOB SCSI_IPTR hiob)
{
   SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_UEXACT8 ScbCurr, ScbBDR, ScbNext;
   SCSI_HIOB SCSI_IPTR hiobCurr;
   SCSI_BUS_ADDRESS ScbBDRbusAddress;
   void SCSI_IPTR   ScbBDRvirtualAddress;
  
   /* As BDR scb is a new scb (unlike the case of abort where the scb which */
   /* has to aborted has already been fired), before putting into head of   */
   /* exe queue the BDR scb should be deleted from the new queue if present.*/

   /* Get the next scb of the BDR scb */
   ScbBDR = hiob->scbDescriptor->queueNext->scbNumber;
   ScbBDRbusAddress = hiob->scbDescriptor->queueNext->scbBuffer.busAddress;
   ScbBDRvirtualAddress = hiob->scbDescriptor->queueNext->scbBuffer.virtualAddress;

   /* Get q_next_scb value from scratch ram */
   ScbCurr = OSD_INEXACT8(SCSI_AICREG(SCSI_hQ_NEXT_SCB(hhcb)));

   /* If q_next_scb is already pointing to the BDR scb, update q_next_scb to */
   /* point to the next scb of the BDR scb. */
   if (ScbCurr == hiob->scbNumber)
   {
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_hQ_NEXT_SCB(hhcb)),ScbBDR);
      SCSI_hSETADDRESSSCB(hhcb,SCSI_S64_Q_NEW_POINTER,ScbBDRvirtualAddress);
   }

   /* Else, go through the new queue and remove the BDR scb if it is sitting */
   /* there. */
   else
   {
      ScbNext = ScbCurr;
      while (SCSI_ACTPTR[ScbCurr] != SCSI_NULL_HIOB)
      {
         hiobCurr = SCSI_ACTPTR[ScbCurr];
         ScbNext = SCSI_hOBTAIN_NEXT_SCB_NUM(hhcb,hiobCurr);
         if (ScbNext == hiob->scbNumber)
         {
            hiobCurr = SCSI_ACTPTR[ScbCurr];
            SCSI_hUPDATE_NEXT_SCB_NUM(hhcb,hiobCurr,ScbBDR);
            SCSI_hUPDATE_NEXT_SCB_ADDRESS(hhcb,hiobCurr,ScbBDRbusAddress);

            break;
         }
         ScbCurr = ScbNext;
      }
   }

   /* Put the BDR scb in the head of the exe queue */
   SCSIhStandard64QHead(hhcb, hiob);
}
#endif

/*********************************************************************
*
*  SCSIhSwapping32QHeadBDR
*
*     Enqueue BDR scb at head for swapping mode sequencer
*
*  Return Value:  none
*                  
*  Parameters:    hhcb
*                 hiob
*
*  Remarks:       sequencer must be paused when this routine
*                 get referenced.
*                  
*********************************************************************/
#if SCSI_SWAPPING32_MODE
void SCSIhSwapping32QHeadBDR (SCSI_HHCB SCSI_HPTR hhcb,
                              SCSI_HIOB SCSI_IPTR hiob)
{
   SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_UEXACT8 qNewTail;
   SCSI_UEXACT8 ScbCurr, ScbBDR, ScbNext;
   SCSI_HIOB SCSI_IPTR hiobCurr;
  
   /* As BDR scb is a new scb (unlike the case of abort where the scb which */
   /* has to aborted has already been fired), before putting into head of   */
   /* new queue the new queue should be adjusted to make the last element   */
   /* not point to the BDR scb; instead it should be pointed to the next scb*/
   /* of the BDR scb.                                                       */

   /* Get q_new_tail value */
   qNewTail = SCSI_hGETQNEWTAIL(hhcb);

   /* Get the next scb of the BDR scb */
   ScbBDR = hiob->scbDescriptor->queueNext->scbNumber;

   /* Get q_next_scb value from scratch ram */
   ScbCurr = OSD_INEXACT8(SCSI_AICREG(SCSI_hQ_NEXT_SCB(hhcb)));

   /* If q_next_scb is already pointing to the BDR scb, just send it to the */
   /* sequencer.                                                            */
   if (ScbCurr == hiob->scbNumber)
   {
      /* bump q new head */
      (SCSI_UEXACT8)hhcb->SCSI_HP_qNewHead++;
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_hQ_NEW_HEAD(hhcb)),hhcb->SCSI_HP_qNewHead);
   }

   /* Else, go through the new queue and remove the BDR scb if it is sitting */
   /* there and put the BDR scb in the head of the new queue.                */
   else
   {
      ScbNext = ScbCurr;
      while (qNewTail != hhcb->SCSI_HP_qNewHead)
      {
         ScbCurr = ScbNext;
         hiobCurr = SCSI_ACTPTR[ScbCurr];
         ScbNext = SCSI_hOBTAIN_NEXT_SCB_NUM(hhcb,hiobCurr);
         if (ScbNext == hiob->scbNumber)
         {
            hiobCurr = SCSI_ACTPTR[ScbCurr];
            SCSI_hUPDATE_NEXT_SCB_NUM(hhcb,hiobCurr,ScbBDR);

            break;
         }
         qNewTail++;
      }

      /* Put the BDR scb in the head of the new queue */
      SCSIhSwapping32QHead(hhcb, hiob);
   }
}
#endif /* SCSI_SWAPPING32_MODE */

/*********************************************************************
*
*  SCSIhSwapping64QHeadBDR
*
*     Enqueue BDR scb at head for swapping mode sequencer
*
*  Return Value:  none
*                  
*  Parameters:    hhcb
*                 hiob
*
*  Remarks:       sequencer must be paused when this routine
*                 get referenced.
*                  
*********************************************************************/
#if SCSI_SWAPPING64_MODE
void SCSIhSwapping64QHeadBDR (SCSI_HHCB SCSI_HPTR hhcb,
                              SCSI_HIOB SCSI_IPTR hiob)
{
   SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_UEXACT8 qNewTail;
   SCSI_UEXACT8 ScbCurr, ScbBDR, ScbNext;
   SCSI_HIOB SCSI_IPTR hiobCurr;
  
   /* As BDR scb is a new scb (unlike the case of abort where the scb which */
   /* has to aborted has already been fired), before putting into head of   */
   /* new queue the new queue should be adjusted to make the last element   */
   /* not point to the BDR scb; instead it should be pointed to the next scb*/
   /* of the BDR scb.                                                       */

   /* Get q_new_tail value */
   qNewTail = SCSI_hGETQNEWTAIL(hhcb);

   /* Get the next scb of the BDR scb */
   ScbBDR = hiob->scbDescriptor->queueNext->scbNumber;

   /* Get q_next_scb value from scratch ram */
   ScbCurr = OSD_INEXACT8(SCSI_AICREG(SCSI_hQ_NEXT_SCB(hhcb)));

   /* If q_next_scb is already pointing to the BDR scb, just send it to the */
   /* sequencer.                                                            */
   if (ScbCurr == hiob->scbNumber)
   {
      /* bump q new head */
      (SCSI_UEXACT8)hhcb->SCSI_HP_qNewHead++;
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_hQ_NEW_HEAD(hhcb)),hhcb->SCSI_HP_qNewHead);
   }

   /* Else, go through the new queue and remove the BDR scb if it is sitting */
   /* there and put the BDR scb in the head of the new queue.                */
   else
   {
      ScbNext = ScbCurr;
      while (qNewTail != hhcb->SCSI_HP_qNewHead)
      {
         ScbCurr = ScbNext;
         hiobCurr = SCSI_ACTPTR[ScbCurr];
         ScbNext = SCSI_hOBTAIN_NEXT_SCB_NUM(hhcb,hiobCurr);
         if (ScbNext == hiob->scbNumber)
         {
            hiobCurr = SCSI_ACTPTR[ScbCurr];
            SCSI_hUPDATE_NEXT_SCB_NUM(hhcb,hiobCurr,ScbBDR);

            break;
         }
         qNewTail++;
      }

      /* Put the BDR scb in the head of the new queue */
      SCSIhSwapping64QHead(hhcb, hiob);
   }
}
#endif /* SCSI_SWAPPING64_MODE */

/*********************************************************************
*
*  SCSIhStandardAdvQHeadBDR
*
*     Enqueue BDR scb at head of execution queue for standard advanced mode 
*     sequencer
*
*  Return Value:  none
*                  
*  Parameters:    hhcb
*                 hiob
*
*  Remarks:       sequencer must be paused when this routine
*                 get referenced.
*                  
*********************************************************************/
#if SCSI_STANDARD_ADVANCED_MODE
void SCSIhStandardAdvQHeadBDR (SCSI_HHCB SCSI_HPTR hhcb,
                               SCSI_HIOB SCSI_IPTR hiob)
{
   SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_UEXACT8 ScbCurr, ScbBDR, ScbNext;
   SCSI_HIOB SCSI_IPTR hiobCurr;
   SCSI_BUS_ADDRESS ScbBDRbusAddress;
   void SCSI_IPTR   ScbBDRvirtualAddress;

   /* As BDR scb is a new scb (unlike the case of abort where the scb which */
   /* has to aborted has already been fired), before putting into head of   */
   /* exe queue the BDR scb should be deleted from the new queue if present.*/

   /* Get the next scb of the BDR scb */
   ScbBDR = hiob->scbDescriptor->queueNext->scbNumber;
   ScbBDRbusAddress = hiob->scbDescriptor->queueNext->scbBuffer.busAddress;
   ScbBDRvirtualAddress = hiob->scbDescriptor->queueNext->scbBuffer.virtualAddress;

   /* Get q_next_scb value from scratch ram */
   ScbCurr = OSD_INEXACT8(SCSI_AICREG(SCSI_hQ_NEXT_SCB(hhcb)));

   /* If q_next_scb is already pointing to the BDR scb, update q_next_scb to */
   /* point to the next scb of the BDR scb. */
   if (ScbCurr == hiob->scbNumber)
   {
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_hQ_NEXT_SCB(hhcb)),ScbBDR);
      SCSI_hSETADDRESSSCB(hhcb,SCSI_SADV_Q_NEW_POINTER,ScbBDRvirtualAddress);
   }

   /* Else, go through the new queue and remove the BDR scb if it is sitting */
   /* there. */
   else
   {
      ScbNext = ScbCurr;
      while (SCSI_ACTPTR[ScbCurr] != SCSI_NULL_HIOB)
      {
         hiobCurr = SCSI_ACTPTR[ScbCurr];
         ScbNext = SCSI_hOBTAIN_NEXT_SCB_NUM(hhcb,hiobCurr);
         if (ScbNext == hiob->scbNumber)
         {
            hiobCurr = SCSI_ACTPTR[ScbCurr];
            SCSI_hUPDATE_NEXT_SCB_NUM(hhcb,hiobCurr,ScbBDR);
            SCSI_hUPDATE_NEXT_SCB_ADDRESS(hhcb,hiobCurr,ScbBDRbusAddress);

            break;
         }
         ScbCurr = ScbNext;
      }
   }

   /* Put the BDR scb in the head of the exe queue */
   SCSIhStandardAdvQHead(hhcb, hiob);
}
#endif

/*********************************************************************
*
*  SCSIhSwappingAdvQHeadBDR
*
*     Enqueue the BusDeviceReset SCB at head for swapping mode sequencer
*
*  Return Value:  none
*                  
*  Parameters:    hhcb
*                 hiob
*
*  Remarks:       sequencer must be paused when this routine
*                 get referenced.
*                  
*********************************************************************/
#if SCSI_SWAPPING_ADVANCED_MODE
void SCSIhSwappingAdvQHeadBDR (SCSI_HHCB SCSI_HPTR hhcb,
                               SCSI_HIOB SCSI_IPTR hiob)
{
   SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_UEXACT8 qNewTail;
   SCSI_UEXACT8 ScbCurr, ScbBDR, ScbNext;
   SCSI_HIOB SCSI_IPTR hiobCurr;
  
   /* As BDR scb is a new scb (unlike the case of abort where the scb which */
   /* has to aborted has already been fired), before putting into head of   */
   /* new queue the new queue should be adjusted to make the last element   */
   /* not point to the BDR scb; instead it should be pointed to the next scb*/
   /* of the BDR scb.                                                       */

   /* Get q_new_tail value */
   qNewTail = SCSI_hGETQNEWTAIL(hhcb);

   /* Get the next scb of the BDR scb */
   ScbBDR = hiob->scbDescriptor->queueNext->scbNumber;

   /* Get q_next_scb value from scratch ram */
   ScbCurr = OSD_INEXACT8(SCSI_AICREG(SCSI_hQ_NEXT_SCB(hhcb)));

   /* If q_next_scb is already pointing to the BDR scb, just send it to the */
   /* sequencer.                                                            */
   if (ScbCurr == hiob->scbNumber)
   {
      /* bump q new head */
      (SCSI_UEXACT8)hhcb->SCSI_HP_qNewHead++;
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_hQ_NEW_HEAD(hhcb)),hhcb->SCSI_HP_qNewHead);
   }

   /* Else, go through the new queue and remove the BDR scb if it is sitting */
   /* there and put the BDR scb in the head of the new queue.                */
   else
   {
      ScbNext = ScbCurr;
      while (qNewTail != hhcb->SCSI_HP_qNewHead)
      {
         ScbCurr = ScbNext;
         hiobCurr = SCSI_ACTPTR[ScbCurr];
         ScbNext = SCSI_hOBTAIN_NEXT_SCB_NUM(hhcb,hiobCurr);
         if (ScbNext == hiob->scbNumber)
         {
            hiobCurr = SCSI_ACTPTR[ScbCurr];
            SCSI_hUPDATE_NEXT_SCB_NUM(hhcb,hiobCurr,ScbBDR);

            break;
         }
         qNewTail++;
      }

      /* Put the BDR scb in the head of the new queue */
      SCSIhSwappingAdvQHead(hhcb, hiob);
   }
}
#endif

/*********************************************************************
*
*  SCSIhStandard160mQHeadBDR
*
*     Enqueue BusDeviceReset SCB at head of execution queue for
*     standard 160m mode sequencer
*
*  Return Value:  none
*                  
*  Parameters:    hhcb
*                 hiob
*
*  Remarks:       sequencer must be paused when this routine
*                 get referenced.
*                  
*********************************************************************/
#if SCSI_STANDARD_160M_MODE
void SCSIhStandard160mQHeadBDR (SCSI_HHCB SCSI_HPTR hhcb,
                                SCSI_HIOB SCSI_IPTR hiob)
{
   SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_UEXACT8 ScbCurr, ScbBDR, ScbNext;
   SCSI_HIOB SCSI_IPTR hiobCurr;
   SCSI_BUS_ADDRESS ScbBDRbusAddress;
   void SCSI_IPTR   ScbBDRvirtualAddress;

   /* As BDR scb is a new scb (unlike the case of abort where the scb which */
   /* has to aborted has already been fired), before putting into head of   */
   /* exe queue the BDR scb should be deleted from the new queue if present.*/

   /* Get the next scb of the BDR scb */
   ScbBDR = hiob->scbDescriptor->queueNext->scbNumber;
   ScbBDRbusAddress = hiob->scbDescriptor->queueNext->scbBuffer.busAddress;
   ScbBDRvirtualAddress = hiob->scbDescriptor->queueNext->scbBuffer.virtualAddress;

   /* Get q_next_scb value from scratch ram */
   ScbCurr = OSD_INEXACT8(SCSI_AICREG(SCSI_S160M_Q_NEXT_SCB));

   /* If q_next_scb is already pointing to the BDR scb, update q_next_scb to */
   /* point to the next scb of the BDR scb. */
   if (ScbCurr == hiob->scbNumber)
   {
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_S160M_Q_NEXT_SCB),ScbBDR);
      SCSI_hSETADDRESSSCB(hhcb,SCSI_S160M_Q_NEW_POINTER,ScbBDRvirtualAddress);
   }

   /* Else, go through the new queue and remove the BDR scb if it is sitting */
   /* there. */
   else
   {
      ScbNext = ScbCurr;
      while (SCSI_ACTPTR[ScbCurr] != SCSI_NULL_HIOB)
      {
         hiobCurr = SCSI_ACTPTR[ScbCurr];
         ScbNext = SCSI_hOBTAIN_NEXT_SCB_NUM(hhcb,hiobCurr);
         if (ScbNext == hiob->scbNumber)
         {
            hiobCurr = SCSI_ACTPTR[ScbCurr];
            SCSI_hUPDATE_NEXT_SCB_NUM(hhcb,hiobCurr,ScbBDR);
            SCSI_hUPDATE_NEXT_SCB_ADDRESS(hhcb,hiobCurr,ScbBDRbusAddress);

            break;
         }
         ScbCurr = ScbNext;
      }
   }

   /* Put the BDR scb in the head of the exe queue */
   SCSIhStandard160mQHead(hhcb, hiob);
}
#endif /* SCSI_STANDARD_160M_MODE */

/*********************************************************************
*
*  SCSIhSwapping160mQHeadBDR
*
*     Enqueue BusDeviceReset SCB at head of new queue for swapping
*     160m mode sequencer
*
*  Return Value:  none
*                  
*  Parameters:    hhcb
*                 hiob
*
*  Remarks:       sequencer must be paused when this routine
*                 get referenced.
*                  
*********************************************************************/
#if SCSI_SWAPPING_160M_MODE
void SCSIhSwapping160mQHeadBDR (SCSI_HHCB SCSI_HPTR hhcb,
                                SCSI_HIOB SCSI_IPTR hiob)
{
   SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_UEXACT8 qNewTail;
   SCSI_UEXACT8 ScbCurr, ScbBDR, ScbNext;
   SCSI_HIOB SCSI_IPTR hiobCurr;
  
   /* As BDR scb is a new scb (unlike the case of abort where the scb which */
   /* has to aborted has already been fired), before putting into head of   */
   /* new queue the new queue should be adjusted to make the last element   */
   /* not point to the BDR scb; instead it should be pointed to the next scb*/
   /* of the BDR scb.                                                       */

   /* Get q_new_tail value */
   qNewTail = SCSI_hGETQNEWTAIL(hhcb);

   /* Get the next scb of the BDR scb */
   ScbBDR = hiob->scbDescriptor->queueNext->scbNumber;

   /* Get q_next_scb value from scratch ram */
   ScbCurr = OSD_INEXACT8(SCSI_AICREG(SCSI_W160M_Q_NEXT_SCB));

   /* If q_next_scb is already pointing to the BDR scb, just send it to the */
   /* sequencer.                                                            */
   if (ScbCurr == hiob->scbNumber)
   {
      /* bump q new head */
      (SCSI_UEXACT8)hhcb->SCSI_HP_qNewHead++;
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_W160M_Q_NEW_HEAD),hhcb->SCSI_HP_qNewHead);
   }

   /* Else, go through the new queue and remove the BDR scb if it is sitting */
   /* there and put the BDR scb in the head of the new queue.                */
   else
   {
      ScbNext = ScbCurr;
      while (qNewTail != hhcb->SCSI_HP_qNewHead)
      {
         ScbCurr = ScbNext;
         hiobCurr = SCSI_ACTPTR[ScbCurr];
         ScbNext = SCSI_hOBTAIN_NEXT_SCB_NUM(hhcb,hiobCurr);
         if (ScbNext == hiob->scbNumber)
         {
            hiobCurr = SCSI_ACTPTR[ScbCurr];
            SCSI_hUPDATE_NEXT_SCB_NUM(hhcb,hiobCurr,ScbBDR);

            break;
         }
         qNewTail++;
      }

      /* Put the BDR scb in the head of the new queue */
      SCSIhSwapping160mQHead(hhcb, hiob);
   }
}
#endif /* SCSI_SWAPPING_160M_MODE */

/*$Header:   Y:/source/chimscsi/src/himscsi/hwmscsi/HWMSCAM.CV_   1.24.14.1   14 Nov 1997 21:31:10   LUU  $*/
/***************************************************************************
*                                                                          *
* Copyright 1995,1996,1997,1998,1999 Adaptec, Inc.,  All Rights Reserved.  *
*                                                                          *
* This software contains the valuable trade secrets of Adaptec.  The       *
* software is protected under copyright laws as an unpublished work of     *
* Adaptec.  Notice is for informational purposes only and does not imply   *
* publication.  The user of this software may make copies of the software  *
* for use with parts manufactured by Adaptec or under license from Adaptec *
* and for no other use.                                                    *
*                                                                          *
***************************************************************************/

/***************************************************************************
*
*  Module Name:   HWMSCAM.C
*
*  Description:
*                 Codes to implement scam/non scam protocol
*
*  Owners:        ECX IC Firmware Team
*    
*  Notes:         NONE
*
*  Entry Point(s):
*
***************************************************************************/

#include "scsi.h"
#include "hwm.h"

#define  MAX_GLITCHES   16

/*********************************************************************
*
*  SCSIhScamLevel1
*
*  Process level 1 SCAM protocol
*
*  Return Value:  0x00  -  if SCAM operation completed without any error
*                 0xff  -  failure encountered during SCAM operation
*                  
*  Parameters:    hhcb
*
*  Remarks:       After scam protocol get exercised the device table
*                 should be updated with new device existance map,
*                 device world wide id.
*                  
*********************************************************************/
#if SCSI_SCAM_ENABLE >= 1 
SCSI_UEXACT8   SCSIhScamLevel1 (SCSI_HHCB SCSI_HPTR hhcb)
{
   SCSI_UEXACT8   scamStatus;
   SCSI_UEXACT8   hcntrl, hostScsiID;     /* host scsi id */
   register SCSI_REGISTER  scsiRegister;

   scsiRegister = hhcb->scsiRegister;

   hostScsiID = hhcb->hostScsiID & 0x0f;

   /* make sure power is not down, interrupt off, and sequencer paused */
   hcntrl = OSD_INEXACT8(SCSI_AICREG(SCSI_HCNTRL));
   SCSI_hWRITE_HCNTRL(hhcb,scsiRegister,(SCSI_UEXACT8)(hcntrl & ~(SCSI_INTEN | SCSI_POWRDN)));
   SCSIhPauseAndWait(hhcb);

   SCSIhScamIdStringInit(hhcb);

   /* wait until bus free before we can start SCAM initiation */
   if (SCSIhScamWaitForRelease(  hhcb,
                                 SCSI_SCSISIG,
                                 SCSI_BSYI|SCSI_SELI))
      return(SCSI_SCAM_FAILURE);

   /* reset scsi bus after power-on */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSISEQ), SCSI_SCSIRSTO); /* reset SCSI bus */
   SCSIhDelayCount500us(hhcb, 0x01);                  /* hold for 500us */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSISEQ), 0x00);    /* deassert reset line */
   SCSIhDelayCount500us(hhcb, 4000);   /* 2 seconds delay for slow device(s) */

   /* reset transfer option after SCAM scsi reset */
   SCSIhCheckSyncNego(hhcb);
   
   if (SCSIhScamLevel1Initiate(hhcb))
      return(SCSI_SCAM_FAILURE);

   if (SCSIhScamDetectDIC(hhcb))    /* if DIC function detected on SCSI bus */
   {
      /* set Dominance Preference Code to 00b - SCAM level 1 */
      SCSI_DEVICE_TABLE(hhcb)[hostScsiID].scamWWID[0] &= ~SCSI_PRIORI_CODE;

      /* participate the DIC function and becomes dominant or subordinate */
      hhcb->SCSI_HP_scamDominantInit = SCSIhScamIsolation(hhcb);
   }
   else                 /* no DIC, definitely dominant */
      hhcb->SCSI_HP_scamDominantInit = 1;

   /* NOTE: the return ID contains only the SCAM device ID map    */
   scamStatus = (hhcb->SCSI_HP_scamDominantInit)
               ? SCSIhScamDominantInit(hhcb)
               : SCSIhScamSubordinateInit(hhcb);

   SCSIhResetChannelHardware(hhcb);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SEQADDR0), (SCSI_UEXACT8)SCSI_hIDLE_LOOP_ENTRY(hhcb));
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SEQADDR1), (SCSI_UEXACT8)SCSI_hIDLE_LOOP_ENTRY(hhcb) >> 2);

   SCSI_hWRITE_HCNTRL(hhcb,scsiRegister,hcntrl);      /* restore */

   return(scamStatus);
}
#endif

/*********************************************************************
*
*  SCSIhScamLevel1Initiate
*
*  Initiate the SCAM protocol.  At the end, SEL, BUSY, CD, and IO
*  are asserted.
*
*  Return Value:  0x00  -  if SCAM operation completed without any error
*                 0xff  -  failure encountered during SCAM operation
*                  
*  Parameters:    hhcb
*
*  Activation:    SCSIhScamLevel1
*
*  Remarks:       This routine does the front end sequence on SCSI
*                 signals for the scam operations.
*                  
*********************************************************************/
#if SCSI_SCAM_ENABLE >= 1 
SCSI_UEXACT8   SCSIhScamLevel1Initiate (SCSI_HHCB SCSI_HPTR hhcb)
{
   SCSI_UEXACT8   hostScsiID;                /* host scsi id */
   SCSI_UEXACT8   regValue;
   register SCSI_REGISTER  scsiRegister;

   scsiRegister = hhcb->scsiRegister;

   hostScsiID = hhcb->hostScsiID & 0x0f;

   regValue = OSD_INEXACT8(SCSI_AICREG(SCSI_SXFRCTL0));
   regValue |= SCSI_CLRSTCNT + SCSI_CLRCHN + SCSI_SCAMEN;
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SXFRCTL0), regValue);  

   /* turn on active negation */
   regValue = OSD_INEXACT8(SCSI_AICREG(SCSI_SXFRCTL1));
   regValue |= SCSI_ENSTIMER + SCSI_ACTNEGEN;
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SXFRCTL1), regValue);  

   /* Grab the bus by winning the arbitration */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSIDATL), (0x01 << hostScsiID));
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSISIG), SCSI_BSYO); /* do the arbitration */
   SCSIhDelayCount500us(hhcb, 1);      /* arbitration delay - 2.4us min */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSISIG), (SCSI_BSYO | SCSI_SELO)); /* assert both BSY & SEL */
   SCSIhDelayCount500us(hhcb, 0x00);   /* wait for deskew delay */
   
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSIDATL), 0x00);   /* release SCSI bus */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSISIG), (SCSI_BSYO | SCSI_MSGO | SCSI_SELO));   /* Assert MSG+BSY+SEL */
   SCSIhDelayCount500us(hhcb, 0x00);   /* wait for deskew delay */

   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSISIG), (SCSI_SELO | SCSI_MSGO)); /* assert MSG & SEL, release BUSY */
   SCSIhDelayCount500us(hhcb, 500);    /* wait 250ms */

   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSISIG), SCSI_SELO);  /* release MSG */
   if (SCSIhScamWaitForRelease(  hhcb,             /* wait for MSG rls */
                                 SCSI_SCSISIG,
                                 SCSI_MSGO))
      return(SCSI_SCAM_FAILURE);
      

   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSISIG), (SCSI_BSYO | SCSI_SELO)); /* assert BUSY */

   SCSIhDelayCount500us(hhcb, 0x00);   /* wait for deskew delay */

   /* assert CD and IO */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSISIG), (SCSI_BSYO | SCSI_SELO | SCSI_CDO | SCSI_IOO));
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSIDATL), (SCSI_DB6 | SCSI_DB7));  /* assert DB6 & DB7 */

   SCSIhDelayCount500us(hhcb, 0x00);   /* wait for deskew delay */

   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSISIG), (SCSI_BSYO | SCSI_CDO | SCSI_IOO));  /* release SEL */
   if (SCSIhScamWaitForRelease(  hhcb,          /* wait for SEL rls */
                                 SCSI_SCSISIG,
                                 SCSI_SELO))
      return(SCSI_SCAM_FAILURE);
                           
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSIDATL), SCSI_DB7);          /* de-assert DB6 */
   if (SCSIhScamWaitForRelease(  hhcb,             /* wait for DB6 rls */
                                 SCSI_SCSIBUSL,
                                 SCSI_DB6))
      return(SCSI_SCAM_FAILURE);
      
   /* assert SEL */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSISIG), (SCSI_BSYO | SCSI_SELO | SCSI_CDO | SCSI_IOO));

   /* By now, the initiation of the SCAM protocol has been completed   */
   /* with SEL & DB7 asserted. we are ready to go into TRANSFER CYCLES */

   SCSIhDelayCount500us(hhcb, 0x01);   /* wait 500us delays */

   return(0x00);
}
#endif

/*********************************************************************
*
*  SCSIhScamDetectDIC
*
*  Check for Dominant Initiator Contention function.
*
*  Return Value:  0 - DIC did not detected
*                 1 - DIC detected
*                  
*  Parameters:    hhcb
*
*  Activation:    SCSIhScamLevel1
*                  
*  Remarks:                
*                  
*********************************************************************/
#if SCSI_SCAM_ENABLE >= 1 
SCSI_UEXACT8 SCSIhScamDetectDIC (SCSI_HHCB SCSI_HPTR hhcb)
{
   SCSI_UEXACT8   busValue, retValue;

   retValue = 0;

   /* send SYNC pattern */
   busValue = SCSIhScamDataOut(hhcb, SCSI_SYNC_PATTERN);

   if (busValue == SCSI_SYNC_PATTERN)
   {
      /* send CONFIGURATION PROCESS COMPLETE function code. */
      /* If the returned bus value is expected then the HA  */
      /* is the only initiator on the SCSI bus. */
      busValue = SCSIhScamDataOut(hhcb, SCSI_CONFIG_DONE);    /* CONFIG COMPLETE */

      /* Dominant Initiator Contention */      
      if (busValue == SCSI_DOMINIT_CONT)
         retValue = 1;
   }

   else if (busValue == SCSI_DOMINIT_CONT) /* Dominant Initiator Contention */
   {
      retValue = 1;
   }

   return(retValue);
}
#endif

/*********************************************************************
*
*  SCSIhScamLevel2
*
*     Exercise scam level 2 protocol to configuration devices on scsi bus
*
*  Return Value:  0x00  -  if SCAM operation completed without any error
*                 0xff  -  failure encountered during SCAM operation
*                  
*  Parameters:    hhcb
*
*  Remarks:       After scam protocol get exercised the device table
*                 should be updated with new device existance map,
*                 device world wide id.
*
*                 This is the start point of scam level 2 operation.
*                 SCAM target devices are recorded in scamDevice field
*                 of SCSI_DEVICE.
*                  
*********************************************************************/
#if SCSI_SCAM_ENABLE == 2 
SCSI_UEXACT8   SCSIhScamLevel2 (SCSI_HHCB SCSI_HPTR hhcb)
{
   SCSI_UEXACT8   scamStatus;
   SCSI_UEXACT8   hcntrl, hostScsiID;                /* host scsi id */
   register SCSI_REGISTER  scsiRegister;

   scsiRegister = hhcb->scsiRegister;

   hostScsiID = hhcb->hostScsiID & 0x0f;

   /* make sure power is not down, interrupt on, and sequencer paused */
   hcntrl = OSD_INEXACT8(SCSI_AICREG(SCSI_HCNTRL));
   SCSI_hWRITE_HCNTRL(hhcb,scsiRegister,(SCSI_UEXACT8)((hcntrl | SCSI_INTEN) & ~SCSI_POWRDN));
   SCSIhPauseAndWait(hhcb);

   /* Initialize Host Adapter World Wide ID */
   SCSIhScamIdStringInit(hhcb);

   if (SCSIhScamLevel2Initiate(hhcb))
      return(SCSI_SCAM_FAILURE);

   /* need to terminate SCAM protocol before scan for Non-Scam devices */
   if (hhcb->SCSI_HP_scamDominantInit)
   {
      SCSIhScamDataOut(hhcb, SCSI_SYNC_PATTERN);   /* SYNC pattern */
      SCSIhScamDataOut(hhcb, SCSI_CONFIG_DONE);    /* CONFIG COMPLETE */
   }

   /* here is the common portion of code between level 1 and level 2 scam */
   scamStatus = (hhcb->SCSI_HP_scamDominantInit)
               ? SCSIhScamDominantInit(hhcb)
               : SCSIhScamSubordinateInit(hhcb);

   SCSIhResetChannelHardware(hhcb);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SEQADDR0), (SCSI_UEXACT8)SCSI_hIDLE_LOOP_ENTRY(hhcb));
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SEQADDR1), (SCSI_UEXACT8)SCSI_hIDLE_LOOP_ENTRY(hhcb) >> 2);

   SCSI_hWRITE_HCNTRL(hhcb,scsiRegister,hcntrl);      /* restore */

   return(scamStatus);
}
#endif

/*********************************************************************
*
*  SCSIhScamIdStringInit
*
*     Initialize Host Adapter World Wide ID
*
*  Return Value:  none
*                  
*  Parameters:    hhcb
*
*  Remarks:       Host Adapter ID is generated only of scam-level 2
*                 is supported.
*                  
*                 Algorithm to generate the WWID:
*                 ProductID + PCIBusNumber + PCIDevNumber + BaseAddr
*                 (4 bytes)   (3 bytes)      (2 bytes)      (8 bytes)
*********************************************************************/
#if SCSI_SCAM_ENABLE >= 1
void SCSIhScamIdStringInit (SCSI_HHCB SCSI_HPTR hhcb)
{
   SCSI_UINT8     busNumber, deviceNumber;
   SCSI_UEXACT8   i, j;
   SCSI_UEXACT8   hostScsiID;                /* host scsi id */
   SCSI_UEXACT32  baseAddress;
   SCSI_HOST_ADDRESS SCSI_LPTR hostAddress;
   register SCSI_REGISTER  scsiRegister;

   scsiRegister = hhcb->scsiRegister;

   hostScsiID = hhcb->hostScsiID & 0x0f;

   /* initialize the Host Adapter WWID as follows:                   */
   /*       priority code     -                                      */
   /*       maximum ID code   -  set to 7 even for the wide HAs.     */
   /*       SNA               -  set to 1                            */

   SCSI_DEVICE_TABLE(hhcb)[hostScsiID].scamWWID[0]  = 
      (OSD_INEXACT8(SCSI_AICREG(SCSI_SBLKCTL)) & SCSI_SELWIDE)
         ? SCSI_UPTO_0F : SCSI_UPTO_07;

   SCSI_DEVICE_TABLE(hhcb)[hostScsiID].scamWWID[0] |= SCSI_SNA;
   SCSI_DEVICE_TABLE(hhcb)[hostScsiID].scamWWID[0] |= SCSI_VALID_BUT_UNASSIGN;

   /* set the current ID entry */
   SCSI_DEVICE_TABLE(hhcb)[hostScsiID].scamWWID[1] |= hostScsiID;

   /* set vendor identification to ADAPTEC */
   SCSI_DEVICE_TABLE(hhcb)[hostScsiID].scamWWID[2] = 'A';
   SCSI_DEVICE_TABLE(hhcb)[hostScsiID].scamWWID[3] = 'D';
   SCSI_DEVICE_TABLE(hhcb)[hostScsiID].scamWWID[4] = 'A';
   SCSI_DEVICE_TABLE(hhcb)[hostScsiID].scamWWID[5] = 'P';
   SCSI_DEVICE_TABLE(hhcb)[hostScsiID].scamWWID[6] = 'T';
   SCSI_DEVICE_TABLE(hhcb)[hostScsiID].scamWWID[7] = 'E';
   SCSI_DEVICE_TABLE(hhcb)[hostScsiID].scamWWID[8] = 'C';
   SCSI_DEVICE_TABLE(hhcb)[hostScsiID].scamWWID[9] = ' ';
 
   /* Generating the remaining portion of the WWID.               */
   SCSI_DEVICE_TABLE(hhcb)[hostScsiID].scamWWID[10] = 'A';
   SCSI_DEVICE_TABLE(hhcb)[hostScsiID].scamWWID[11] = 'I';
   SCSI_DEVICE_TABLE(hhcb)[hostScsiID].scamWWID[12] = 'C';

   SCSI_DEVICE_TABLE(hhcb)[hostScsiID].scamWWID[13] =
      (SCSI_UEXACT8)((hhcb->deviceID & 0xf000) >> 12) + '0';
   SCSI_DEVICE_TABLE(hhcb)[hostScsiID].scamWWID[14] =
      (SCSI_UEXACT8)((hhcb->deviceID & 0x0f00) >> 8) + '0';
   SCSI_DEVICE_TABLE(hhcb)[hostScsiID].scamWWID[15] =
      (SCSI_UEXACT8)((hhcb->deviceID & 0x00f0) >> 4) + '0';
   SCSI_DEVICE_TABLE(hhcb)[hostScsiID].scamWWID[16] =
      (SCSI_UEXACT8)(hhcb->deviceID & 0x000f) + '0';

   /* PCI bus number is scanned from 0 to 255, so the boot/primary */
   /* host adapter will have lowest bus number.  But the id string */
   /* should start with higher value i.e. from 255 to 0 */
   hostAddress = OSD_GET_HOST_ADDRESS(hhcb);
   busNumber = 0xff - hostAddress->pciAddress.busNumber;

   SCSI_DEVICE_TABLE(hhcb)[hostScsiID].scamWWID[19] = (SCSI_UEXACT8)((busNumber % 10) + '0');
   busNumber /= 10;
   SCSI_DEVICE_TABLE(hhcb)[hostScsiID].scamWWID[18] = (SCSI_UEXACT8)((busNumber % 10) + '0');
   SCSI_DEVICE_TABLE(hhcb)[hostScsiID].scamWWID[17] = (SCSI_UEXACT8)((busNumber / 10) + '0');

   deviceNumber = 0x1f - hostAddress->pciAddress.deviceNumber;
   SCSI_DEVICE_TABLE(hhcb)[hostScsiID].scamWWID[21] = (SCSI_UEXACT8)((deviceNumber % 10) + '0');
   SCSI_DEVICE_TABLE(hhcb)[hostScsiID].scamWWID[20] = (SCSI_UEXACT8)((deviceNumber / 10) + '0');

   /* Get base address from configuration space */
   baseAddress = OSDReadPciConfiguration(hhcb,SCSI_BASE_ADDR_REG) & 0xFFFFFFFE;

   for (i = 0; i < 8; i++)
   {
      j = ((SCSI_UEXACT8)(baseAddress >> (28 - i * 4))) & 0x0f;

      /* convert to ASCII character */
      SCSI_DEVICE_TABLE(hhcb)[hostScsiID].scamWWID[22 + i] = (j >= 9) ? (j - 10) + 'A' : j + '0';
   }

   SCSI_DEVICE_TABLE(hhcb)[hostScsiID].scamWWID[30] =
      SCSI_DEVICE_TABLE(hhcb)[hostScsiID].scamWWID[31] = ' ';
}
#endif

/*********************************************************************
*
*  SCSIhScamLevel2Initiate
*
*     To start the level 2 SCAM operation   
*
*  Return Value:  0x00  -  if SCAM operation completed without any error
*                 0xff  -  failure encountered during SCAM operation
*                  
*  Parameters:    hhcb
*
*  Remarks:       This routine does the DOMINANT INITIATOR CONTENTION
*                 and ISOLATION part of the SCAM operation.
*
*********************************************************************/
#if SCSI_SCAM_ENABLE == 2 
SCSI_UEXACT8   SCSIhScamLevel2Initiate (SCSI_HHCB SCSI_HPTR hhcb)
{
   SCSI_UEXACT8 regValue, hostScsiID;
   register SCSI_REGISTER  scsiRegister;

   scsiRegister = hhcb->scsiRegister;

   hostScsiID = hhcb->hostScsiID & 0x0f;

   if (hhcb->SCSI_HP_scamSelectDetected)
      hhcb->SCSI_HP_scamSelectDetected = 0;  /* clr SCAM selection flag */
   else                                      /* OSM started SCAM protocol */
   {
      /* wait until bus free before we can initiate SCAM protocol. */
      /* Detect BUSFREE bit won't help here */
      if (SCSIhScamWaitForRelease(  hhcb,    /* wait for BUS FREE */
                                    SCSI_SCSISIG,
                                    SCSI_BSYI|SCSI_SELI))
         return(SCSI_SCAM_FAILURE);

   }

   /* initiate SCAM level 2 protocol */
   regValue = OSD_INEXACT8(SCSI_AICREG(SCSI_SCAMCTRL));

   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCAMCTRL), regValue | SCSI_ENSCAMSELO);

   if (SCSIhScamWaitForAssertion(hhcb,
                                 SCSI_SCAMSTAT,
                                 SCSI_SCAMSELD))
      return(SCSI_SCAM_FAILURE);

   /* turn off SCAM selection enable bit and clear SCAM selection done bit */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCAMCTRL),
      (regValue & ~SCSI_ENSCAMSELO) | SCSI_CLRSCAMSELD);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_CLRINT), SCSI_CLRSCSINT);   /* clrear SCSIINT */

   SCSIhDelayCount500us (hhcb, 1);  /* waits 500us delay */

   /* set Dominance Preference Code */
   if (hhcb->SCSI_HP_scamDominantInit)
      /* Dominant Initiator from last SCAM invocation */
      SCSI_DEVICE_TABLE(hhcb)[hostScsiID].scamWWID[0] |= SCSI_DOM_SCAM2_INIT;
   else                       /* just SCAM level 2 initiator */
   {
      SCSI_DEVICE_TABLE(hhcb)[hostScsiID].scamWWID[0] &= 0x3f;
      SCSI_DEVICE_TABLE(hhcb)[hostScsiID].scamWWID[0] |= SCSI_SCAM2_INIT;
   }

   SCSIhScamIssueDIC(hhcb);   /* issue Dominant Initiator Contention */

   if (hhcb->SCSI_HP_scamDominantInit = SCSIhScamIsolation(hhcb))
   {
      /* update Dominance Preference Code to 11b */
      SCSI_DEVICE_TABLE(hhcb)[hostScsiID].scamWWID[0] |= SCSI_DOM_SCAM2_INIT;
   }
   return(0x00);
}
#endif

/*********************************************************************
*
*  SCSIhScamIssueDIC
*
*  Initiate Dominant Initiator Contention function.
*
*  Return Value:  none
*                  
*  Parameters:    hhcb
*
*  Activation:    SCSIhScamLevel2Initiate
*
*  Remarks:
*
*********************************************************************/
#if SCSI_SCAM_ENABLE == 2 
void SCSIhScamIssueDIC (SCSI_HHCB SCSI_HPTR hhcb)
{
   SCSIhScamDataOut(hhcb, SCSI_SYNC_PATTERN);   /* SYNC pattern */
   SCSIhScamDataOut(hhcb, SCSI_DOMINIT_CONT);   /* CONFIG COMPLETE */
}
#endif

/*********************************************************************
*
*  SCSIhScamIsolation
*
*  Broadcast initiator's numerically identification string
*
*  Return Value:  0 - Became Subordinate SCAM initiator for DIC
*                     Lost isolation stage (suboridnate SCAM initiator) 
*
*                 1 - Became Dominant SCAM initiator for DIC
*                     Won isolation stage (subordinate SCAM initiator)
*
*  Return Value:  none
*                  
*  Parameters:    hhcb
*
*  Activation:    SCSIhScamLevel1 
*
*  Remarks:       This routine is activated under the followning
*                 conditions:
*
*                 LEVLE 1 -
*                 After detection of DOMINANT INITIATOR CONTENTION
*                 function from another SCAM LEVEL 2 initiator.
*                 
*                 LEVEL 2 DOMINANT INITIATOR -
*
*                 LEVEL 2 SUBORDINATE INITIATOR -
*
*********************************************************************/
#if SCSI_SCAM_ENABLE >= 1 
SCSI_UEXACT8 SCSIhScamIsolation (SCSI_HHCB SCSI_HPTR hhcb)
{
   SCSI_UEXACT8   busValue, numBytes, numBits, IDByte, IDBits;
   SCSI_UEXACT8   hostScsiID;                     /* host scsi id */

   hostScsiID = hhcb->hostScsiID & 0x0f;

   /* start the isolation stage - */
   for (numBytes=0; numBytes<SCSI_SIZEWWID; numBytes++)
   {
      IDByte = SCSI_DEVICE_TABLE(hhcb)[hostScsiID].scamWWID[numBytes];

      for (numBits=0; numBits<=7; numBits++)
      {
         IDBits = (IDByte & 0x80) ? SCSI_DB1 : SCSI_DB0;

         busValue = SCSIhScamDataOut(hhcb, IDBits);

         /* compare own initiator's id string vs. others */
         /* back off if the id_bits value is lower       */
         if (((SCSI_UEXACT8)(IDBits ^ busValue)) > IDBits)
            return(0);

         IDByte <<= 1;                 /* send next bit */
      }
   }

   /* end of the identification string */
   busValue = SCSIhScamDataOut(hhcb, 0x00);

   return(1);
}
#endif

/*********************************************************************
*
*  SCSIhScamDominantInit
*
*  
*  Parameters:    hhcb
*
*  Remarks:       This routine does the ID assignment part of the
*                 dominant initiator scam level 2 protocol.
*
*********************************************************************/
#if SCSI_SCAM_ENABLE >= 1
SCSI_UEXACT8 SCSIhScamDominantInit (SCSI_HHCB SCSI_HPTR hhcb)
{
   SCSI_UEXACT8   scamStatus=0;
   SCSI_UEXACT16  nonScamDeviceMap, deviceMap;
   SCSI_UEXACT32  deviceMapStat;
   register SCSI_REGISTER  scsiRegister;

   scsiRegister = hhcb->scsiRegister;

   /* find non-scam devices and record in bit position of DeviceMap */
   deviceMap = SCSIhScamFindNonScamDevices(hhcb);

   deviceMap |= ((SCSI_UEXACT16)0x01 << (hhcb->hostScsiID & 0x0f));
   nonScamDeviceMap = deviceMap;       /* record scam-tolerant and HA map */

   /* if SCAM selection detected, return to OSM immediately */
   /* so that the OSM will invoke SCAM protocol again       */
   if (hhcb->SCSI_HP_scamSelectDetected)
   {
      deviceMap = 0xffff;
   }
   else
   {
      /* For intolerant SCAM devices, scsi bus reset will bring them  */
      /* back to state where they will repsond to the second SCAM */
      /* initiation/selection from the initiator.  If there wasn't a scsi */
      /* bus reset, some intolerant SCAM devices will lock up scsi bus. */
      /* While others will not participate in SCAM selection eventhough */
      /* they have not been assigned an SCSI ID.  And they do not respond */
      /* to the normal SCSI command neither. */
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSISEQ), SCSI_SCSIRSTO); /* reset SCSI bus */
      SCSIhDelayCount500us(hhcb, 0x01);                  /* hold for 500us */
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSISEQ), 0x00);    /* deassert reset line */
      SCSIhDelayCount500us(hhcb, 4000);   /* 2 seconds delay for slow device(s) */

      /* we need to initiate SCAM protocol after scanning for Non-Scam */
      /* devices to start the isolation and id assignment and also */
      /* take SCAM devices back to their assignable id state */
#if SCSI_SCAM_ENABLE == 2
      if (hhcb->SCSI_HF_scamLevel == 2)    /* SCAM level 2 chip */
         SCSIhScamLevel2Initiate(hhcb);
      else
#endif
         scamStatus = SCSIhScamLevel1Initiate(hhcb);

      /* assign ids to all possible scam drives.  Assignment failure */
      /* would mean the end of assigning all devices.                */
      while (1)
      {
         deviceMapStat = SCSIhScamAssignID(hhcb, deviceMap);

         /* update deviceMap */
         deviceMap =(SCSI_UEXACT16)((deviceMapStat & SCSI_SCAM_DEVICE_MASK) >> 16);
                         /* end or failure of assignment */
         
         if ((SCSI_UEXACT16)(deviceMapStat & SCSI_SCAM_STATUS_MASK) == SCSI_ASSIGN_DONE)
         {
            SCSIhScamDataOut(hhcb, SCSI_SYNC_PATTERN);   /* SYNC pattern */
            SCSIhScamDataOut(hhcb, SCSI_CONFIG_DONE);    /* CONFIG COMPLETE */
            break;
         }
      }

      deviceMap &= ~nonScamDeviceMap;     /* remove HA and scam-tolerant devices */

      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSISEQ), 0x00);
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSIDATL), 0x00);
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSISIG), 0x00);
      
      SCSIhDelayCount500us (hhcb, 1);     /* deskew delay for SCSI bus */

      /* wait for bus free */
      if (SCSIhScamWaitForAssertion(hhcb,
                                    SCSI_SSTAT1,
                                    SCSI_BUSFREE))
         return(SCSI_SCAM_FAILURE);

      /* clear bus free and SCSI interrupt bit */
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_CLRSINT1), SCSI_CLRBUSFREE);
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_CLRINT), SCSI_CLRSCSINT);

      SCSIhDelayCount500us (hhcb, 1);     /* deskew delay for SCSI bus */
   }                 /* end of else SCAM selection not detected */

   return(scamStatus);
}
#endif

/*********************************************************************
*
*  SCSIhScamAssignID
*
*  Assign SCAM id to scam targets
*
*  Return Value:  32-bit  - MSW contains the latest device map
*                           LSW contains the status
*                             ASSIGN_INRPOG - assignment still in progress
*                             ASSIGN_DONE - assignment completed
*                             ASSIGN_FAIL - SCAM operation failed
*  Parameters:    hhcb
*                 deviceMap   -  to record scam devices in the setup
*
*  Remarks:       The ID assigned ID is dependent upon three factors:
*
*                    1. The maximum allowed ID based upon byte 0 in type code
*                    2. If the device is wide or narrow.
*                    3. The assigned ID shall never exceed 0x0f.  This is a
*                       limitation of the current SCAM code.  It is subjected
*                       to future review.
*
*                 ID preservation is implemented here as follows:
*                 ( TO BE FILLED IN )
*
*********************************************************************/
#if SCSI_SCAM_ENABLE >= 1
#define  CYCLE_FAIL        0xff
#define  CYCLE_START       0x00
#define  CYCLE_COMPLETE    0x01
#define  CYCLE_INPROGRESS  0x02

SCSI_UEXACT32 SCSIhScamAssignID (SCSI_HHCB SCSI_HPTR hhcb,
                                 SCSI_UEXACT16 deviceMap)
{
   SCSI_UEXACT8   numBytes, numBits, targetWWID[SCSI_SIZEWWID], busValue;
   SCSI_UEXACT8   cycleDone, parity, assignedID;
   SCSI_UEXACT16  bitPosition;
   SCSI_UEXACT32  deviceMapStat;   /* MSW - Dev Map returned; LSW - Scam Status */

   cycleDone = CYCLE_START;

   /* do the transfer cycle - send SYNC pattern */
   SCSIhScamDataOut(hhcb, SCSI_SYNC_PATTERN);   /* SYNC pattern */
   SCSIhScamDataOut(hhcb, SCSI_ASSIGN);         /* CONFIG COMPLETE */

   /* start the isolation stage - */
   /* if ID preservation would be done, this is where to capture  */
   /* the device type code and ID.                                */
   for ( numBytes=0; numBytes<=31; numBytes++)
   {
      /* initialize type code bytes to zero and others to space */
      targetWWID[numBytes] = (numBytes <= 1) ? 0x00 : ' ';

      for (numBits=0; numBits<=7; numBits++)
      {
         busValue = SCSIhScamDataIn(hhcb);   /* read scsi bus */

         if (busValue == SCSI_SCAM_FAILURE)
         {
            cycleDone = CYCLE_FAIL;             /* indicate failure */
            break;
         }

         else if ((busValue & 0x1f) == SCSI_ISOLATE_TERMINATE)
         {
            if (   (numBits == 0x00)      /* return scanning */
                && (numBytes == 0x00))
               cycleDone = CYCLE_COMPLETE;      /* SCAM completed */
            else
               cycleDone = CYCLE_INPROGRESS;    /* target has less than 32 bytes */
            break;
         }

         targetWWID[numBytes] <<= 1;
         targetWWID[numBytes] |= (SCSI_UEXACT8)((busValue>>1) & 0x01);
      }                          /* end of bit transfer */

      if (cycleDone != CYCLE_START)       /* for any reason */
         break;
   }

   if (cycleDone == CYCLE_FAIL)
   {
      deviceMapStat  = (SCSI_UEXACT32)deviceMap << 16;   /* whatever we have */
      deviceMapStat |= (SCSI_UEXACT32)SCSI_ASSIGN_FAIL;
      return(deviceMapStat);   /* return device map & stat info */
   }
   else if (cycleDone == CYCLE_COMPLETE)
   {
      deviceMapStat  = (SCSI_UEXACT32)deviceMap << 16;
      deviceMapStat |= (SCSI_UEXACT32)SCSI_ASSIGN_DONE;
      return(deviceMapStat);   /* return device map & stat info */
   }

   /* end of isolation stage reached */
   /* see if the default is available else will get an id assigned */
   /* returned value:   bit 15-8   =  ID value of assigned ID  */
   /*                   bit  7-0   =  ID value of requested ID */
   assignedID = (SCSI_UEXACT8)((SCSIhScamChooseID( hhcb,
                                                   deviceMap,
                                                   targetWWID[0], (SCSI_UEXACT8)
                                                   (targetWWID[1] & 0x1f)) >> 8) & 0x00ff);

   if (assignedID != 0xff)         /* if ID successfully chosen */
   {
      assignedID &= 0x0f;          /* assigned maximum is 0x0f */
      bitPosition = (SCSI_UEXACT16)(0x0001<<assignedID);  /* find bit position of assigned */
      deviceMap |= bitPosition;    /* update the Scam dev map info */

      /* generate the parity info of the assigned ID */
      parity = 0;
      if (!(assignedID & 0x01))
         parity++;
      if (!(assignedID & 0x02))
         parity++;
      if (!(assignedID & 0x04))
         parity++;

      /* send the first quintet - assign ID command */
      if (assignedID < 0x08)    /* id between 0x00 and 0x07 */
         SCSIhScamDataOut(hhcb, SCSI_ASSIGN_00ID);
      else                       /* id between 0x08 and 0x0f */
         SCSIhScamDataOut(hhcb, SCSI_ASSIGN_01ID);

      /* send the 2nd quintet - assigned target ID with parity information */
      SCSIhScamDataOut(hhcb, 
                      (SCSI_UEXACT8)((assignedID & 0x07) | (parity << 3)));
      
      deviceMapStat  = (SCSI_UEXACT32)(deviceMap) << 16;
      deviceMapStat |= (SCSI_UEXACT32)SCSI_ASSIGN_INPROG;

      /* record world-wide ID of target device in device table */
      /* also mark the device as a scam device                 */
      SCSI_DEVICE_TABLE(hhcb)[assignedID].SCSI_DF_scamDevice = 1;
      OSDmemcpy((SCSI_UEXACT8 SCSI_SPTR)(SCSI_DEVICE_TABLE(hhcb)[assignedID].scamWWID),
                (SCSI_UEXACT8 SCSI_SPTR)(targetWWID),
                (SCSI_UINT)SCSI_SIZEWWID);
   }

   else                       /* failed to assign ID */
   {
      deviceMapStat  = (SCSI_UEXACT32)(deviceMap) << 16;
      deviceMapStat |= (SCSI_UEXACT32)SCSI_ASSIGN_FAIL;
   }

   return(deviceMapStat);   /* return device map & stat info */
}
#endif

/*********************************************************************
*
*  SCSIhScamChooseID
*
*  Checks to see if it can assign the current/default ID.  If not,
*  this routine wil return with an alternate ID, if one can be found.
*
*  Return Value:  Default/current ID in LSB;
*                 and the assigned ID in MSB.
*                 0xffff if no alternate ID can be found.
*                  
*  Parameters:    base address of AIC-7870
*                 deviceMap from previous selection operation on ID's
*                 Type code info(2 bytes) of the device
*
*  Remarks:
*
*********************************************************************/
#if SCSI_SCAM_ENABLE >= 1 
SCSI_UEXACT16 SCSIhScamChooseID (SCSI_HHCB SCSI_HPTR hhcb,
                                 SCSI_UEXACT16  deviceMap,
                                 SCSI_UEXACT8   typeCode,
                                 SCSI_UEXACT8   defaultID)
{
   SCSI_UEXACT16  retValue=0;
   SCSI_UEXACT16  bitPosition;
   SCSI_UEXACT8   currentID, nextID, hostScsiID;
   register SCSI_REGISTER  scsiRegister;

   scsiRegister = hhcb->scsiRegister;

   hostScsiID = hhcb->hostScsiID & 0x0f;

   switch (typeCode & SCSI_ID_VALID)
   {
      case  SCSI_NOT_VALID:
         /* this is the case when the device does not have a preferred  */
         /* ID and the ID field type code has no significance           */
         switch (typeCode & SCSI_MAX_ID_CODE)
         {
            case  SCSI_UPTO_0F:
            case  SCSI_UPTO_1F:
               if (OSD_INEXACT8(SCSI_AICREG(SCSI_SBLKCTL)) & SCSI_SELWIDE)
                  currentID = 0x0f;
               else
                  currentID = 0x07;   /* ID higher than 0x0f is not supported */
               break;

            case  SCSI_UPTO_07:
            default:
               currentID = 0x07;
               break;
         }
         break;

      case  SCSI_VALID_BUT_UNASSIGN:
         /* This the case when the device has a preferred ID and it is  */
         /* up to the SCAM protocol to assign give it the preferrence,  */
         /* if it is not already assigned, or an alternate, if it is.   */
         currentID = defaultID;      /* current ID */
         break;

      case  SCSI_VALID_AND_ASSIGN:
         /* this is the case when the device has already been assigned  */
         /* an ID by the SCAM protocol, as reflected in the ID field.   */
         currentID = defaultID;      /* current ID */
         break;

      default:
         currentID = defaultID;      /* current ID */
   }

   /* start with the requested bit position and scan downward until */
   /* an empty slot is captured.                                    */

   bitPosition = 0x0001<<currentID;  /* record ID in bit */
   nextID = currentID;               /* back it up */

   do
   {
      if (!(deviceMap & bitPosition))              /* chk if available */
      {                                            /* yes, it is */
         /* Just in case the user might install a wide SCAM device on the */
         /* narrow cable, thus HIM should not assign this device to an ID */
         /* greater than 7.  This should be documented so that technical  */
         /* support can suggest the user to turn off "Initiate Wide       */
         /* Negotiation" option in SCSISelect if he/she can not change or */
         /* did not know how to change the default SCSI ID.               */
         if ((nextID > 0x07) &&
           ((SCSI_DEVICE_TABLE(hhcb)[nextID].scsiOption & SCSI_WIDE_XFER) == 0))
         {
            nextID--;
            bitPosition >>= 1;            /* try the next position */
         }
         else
         {   
            retValue =  (SCSI_UEXACT16)(nextID) << 8; /* record assigned MSW */
            retValue |= (SCSI_UEXACT16)(defaultID);   /* requested LSW */
            break;
         }
      }
      else
      {
         nextID--;
         bitPosition >>= 1;           /* try the next position */
      }
      
      if (nextID == 0xff)             /* end of the downward scan */
      {
         if (   (OSD_INEXACT8(SCSI_AICREG(SCSI_SBLKCTL)) & SCSI_SELWIDE)
             && (   ((typeCode & SCSI_MAX_ID_CODE) == SCSI_UPTO_0F)
                 || ((typeCode & SCSI_MAX_ID_CODE) == SCSI_UPTO_1F)))
            nextID = 0x0f;          /* start with 0x0f if wide and allowed */
         else
            nextID = 0x07;          /* start with ID 0x07 otherwise */

         bitPosition = 0x0001 << nextID;
      }

      if (nextID == currentID)      /* wrapped around to the start */
         retValue = 0xffff;         /* can not assign - error! */

   } while (nextID != currentID);

   return(retValue);

}
#endif

/*********************************************************************
*
*  SCSIhScamFindNonScamDevices
*
*  Find non scam devices on the bus
*
*  Return Value:  UWORD of which the bit position indicates either
*                 the existence of scam-tolerant devices
*                 (not including the ID of Host Adapter).
*                             - OR -
*                 0xFFFF -    Failure during the operation
*                 
*                 0x???? -    HA detected SCAM selection
*                  
*  Parameters:    config_ptr - pointer to configuration structure 
*
*  Activation:    SCSIhNonScamScan
*
*  Remarks:
*
*********************************************************************/
#if SCSI_SCAM_ENABLE >= 1
SCSI_UEXACT16 SCSIhScamFindNonScamDevices (SCSI_HHCB SCSI_HPTR hhcb)
{
   SCSI_UEXACT16  nonScamDeviceMap;
   SCSI_UEXACT8   targetID, regValue;
   register SCSI_REGISTER  scsiRegister;

   scsiRegister = hhcb->scsiRegister;

   nonScamDeviceMap  = 0;

   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSISIG),0x00); /* clr register */

   /* wait for SCSI Bus Free */
   if (SCSIhScamWaitForRelease(  hhcb,
                                 SCSI_SCSISIG,
                                 SCSI_BSYI|SCSI_SELI))
      return(SCSI_SCAM_FAILURE);

   /* clear SCSI interrupt and all SCSI status  */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_CLRSINT1), ~SCSI_PHASEMIS);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_CLRINT), SCSI_CLRSCSINT);

   /* if SCAM selection is detected, return to OSM right away */
   if (   (hhcb->SCSI_HF_scamLevel == 2)
       && (hhcb->SCSI_HP_scamSelectDetected))
   {
      return(0xffff);         /* another scam level 2 initiator starting */
   }

   /* we'll issue normal SCSI operations so need to disable SCAM */
   regValue = OSD_INEXACT8(SCSI_AICREG(SCSI_SXFRCTL0));
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SXFRCTL0), (regValue & ~SCSI_SCAMEN));

   /* search through the maximum allowable device range */
   for ( targetID = 0;
         targetID < hhcb->maxDevices;
         targetID++) 
   {
      if (targetID != hhcb->hostScsiID)   /* select on non-HA id's */
      {
         if (SCSIhScamSelectDevice(hhcb, targetID))            /* try to select tgt */
         {
            nonScamDeviceMap |= (SCSI_UEXACT16)(0x0001 << targetID);   /* device found */
            SCSI_DEVICE_TABLE(hhcb)[targetID].SCSI_DF_scamTolerant = 1;
         }
      }

     SCSIhDelayCount500us(hhcb, 0);       /* wait 5ms */

      if (   (hhcb->SCSI_HF_scamLevel == 2)
          && (hhcb->SCSI_HP_scamSelectDetected))   /* quit current SCAM */
      {                                            /* proccess and return */
         nonScamDeviceMap = 0xffff;                /* status if SCAM selection */
         break;                                    /* detected */
      }
   }

   /* restore original reg. value */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SXFRCTL0), regValue);

   return(nonScamDeviceMap);
}
#endif

/*********************************************************************
*
*  SCSIhScamSelectDevice
*
*
*  Return Value:  0 -   SCAM-tolerant device does not exist at the
*                       specified device ID location.
*
*                 1 -   Found SCAM-tolerant device at the specified
*                       device ID location.
*  Parameters:    hhcb
*                 deviceID
*
*  Remarks:       If the legacy or scam-tolerant target device exists
*                 at the inquired ID location, the follow-up INQUIRY
*                 command will fill the WWID entry in hhcb.
*
*********************************************************************/
#if SCSI_SCAM_ENABLE >= 1
SCSI_UEXACT8 SCSIhScamSelectDevice (SCSI_HHCB SCSI_HPTR hhcb,
                                    SCSI_UEXACT8 deviceID)
{
   SCSI_UEXACT8   index, regValue, retValue, targetID;
   register SCSI_REGISTER  scsiRegister;

   scsiRegister = hhcb->scsiRegister;

   /* clear all interrupts */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SIMODE0),0x00);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SIMODE1),0x00);

   regValue = OSD_INEXACT8(SCSI_AICREG(SCSI_SXFRCTL0));
   regValue |= SCSI_CLRSTCNT + SCSI_SPIOEN + SCSI_CLRCHN;
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SXFRCTL0), regValue);

   /* load target ID and host ID */
   targetID = deviceID;
   deviceID <<= 4;
   deviceID &= 0xf0;
   deviceID |= hhcb->hostScsiID & 0x0f;
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_hSCSIID(hhcb)), deviceID);
   
   /* expecting message out phase */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSISIG), SCSI_MOPHASE);

   regValue = OSD_INEXACT8(SCSI_AICREG(SCSI_SXFRCTL1));
   regValue |= SCSI_ENSTIMER;
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SXFRCTL1), regValue);

   /* set the data transfer as ASYNCHRONOUS */
   SCSI_hLOADSCSIRATE(hhcb,targetID,0x00);

   /* enable selection out and auto attention out */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSISEQ),
      (SCSI_ENSELO+SCSI_ENAUTOATNO+SCSI_ENAUTOATNP));
   
   /* wait for selection timeout.                     */
   /* SCAM-tolerance device must respond within 2ms.  */
   for (index = 0; index < 4; index++)    /* each delay unit is 500us */
   {
      regValue = OSD_INEXACT8(SCSI_AICREG(SCSI_SSTAT0));
      if (regValue & SCSI_SELDO)
         break;

      SCSIhDelayCount500us(hhcb, 1);      /* wait 500 usec */
   }

   if (!(regValue & SCSI_SELDO))          /* No scam tolerant device at ID */
      retValue = 0;
   else                                   /* SCAM Tolerant device responded */
   {
      SCSIhScamInquiry(hhcb, SCSI_SIZEWWID); /* INQUIRY to finish up */
      retValue = 1;
   }

   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSISEQ), 0x00);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSISIG), 0x00);

   SCSIhDelayCount500us (hhcb, 1);        /* wait 500 usec */

   if (SCSIhScamWaitForAssertion(hhcb,    /* wait for bus free */
                                 SCSI_SSTAT1,
                                 SCSI_BUSFREE))
      return(SCSI_SCAM_FAILURE);

   /* clear bus free and SCSIINT bit */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_CLRSINT1), SCSI_CLRBUSFREE);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_CLRINT), SCSI_CLRSCSINT);
   
   return(retValue);
}
#endif

/*********************************************************************
*
*  SCSIhScamInquiry
*
*  Finish the scsi selection through the use of INQUIRY command
*
*  Return Value:  0x00  -  INQUIRY command successfully completed
*                 0xff  -  INQUIRY command failed due to timeout on SCSI
*                  
*  Parameters:    hhcb
*                 sizeBuffer    -   size of buffer area
*
*  Activation:    SCSIhScamSelectDevice
*
*  Remarks:       The return from INQUIRY command is not used to
*                 generate WWID and is not saved.
*
*********************************************************************/
#if SCSI_SCAM_ENABLE >= 1
SCSI_UEXACT8   SCSIhScamInquiry (SCSI_HHCB SCSI_HPTR hhcb,
                                 SCSI_UEXACT8 sizeBuffer)
{
   SCSI_UEXACT8 phase, expectedPhase;
   SCSI_UEXACT8 count, retry, cmdDone;
   register SCSI_REGISTER scsiRegister;

   scsiRegister = hhcb->scsiRegister;
   expectedPhase = SCSI_MOPHASE;
   cmdDone = retry = count = 0;

   /* wait for REQ to assert */
   if ((phase = SCSIhWait4Req(hhcb)) == SCSI_ERRPHASE)
   {
      return(SCSI_SCAM_FAILURE);
   }

   while (1)
   {
      /* check if phase mismatch occurs */
      if (OSD_INEXACT8(SCSI_AICREG(SCSI_SSTAT1)) & SCSI_PHASEMIS)
      {
         count = 0;                                         /* reset counter */
         OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSISIG), phase);   /* clear phase mismatched */

         switch (phase)
         {
            /* reject any message in from target when phase mismatched, */
            /* it might be the target initiates sync. or wide nego.     */
            case SCSI_MIPHASE:
               /* handle restore pointers message */
               /* usually result from parity error */
               if (OSD_INEXACT8(SCSI_AICREG(SCSI_SCSIBUSL)) == SCSI_MSG03)
               {
                  OSD_INEXACT8(SCSI_AICREG(SCSI_SCSIDATL)); /* ack the byte */

                  /* wait for REQ to assert */
                  if ((phase = SCSIhWait4Req(hhcb)) == SCSI_ERRPHASE)
                     return(SCSI_SCAM_FAILURE);

                  /* other phases consider to be mismatched */
                  if ((phase == SCSI_CMDPHASE) ||
                      (phase == SCSI_DIPHASE) ||
                      (phase == SCSI_STPHASE))
                  {
                     OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSISIG),
                        (expectedPhase = phase));
                  }
                  break;
               }

               /* assert ATN signal */
               OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSISIG), SCSI_MIPHASE | SCSI_ATNO);

               /* ack MSG IN bytes until phase change to MSG OUT */
               do
               {
                  OSD_INEXACT8(SCSI_AICREG(SCSI_SCSIDATL));

                  /* wait for REQ to assert */
                  if ((phase = SCSIhWait4Req(hhcb)) == SCSI_ERRPHASE)
                     return(SCSI_SCAM_FAILURE);

               } while (phase == SCSI_MIPHASE);

               if (phase != SCSI_MOPHASE)
               {
                  SCSIhBadSeq(hhcb);
                  return(SCSI_SCAM_FAILURE);
               }

               /* clear phase mismatched if any */
               OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSISIG), SCSI_MOPHASE);
               OSD_OUTEXACT8(SCSI_AICREG(SCSI_CLRSINT1), SCSI_CLRATNO); /* deasserted ATN */
               OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSIDATL), SCSI_MSG07);   /* REJECT message */

               /* wait for REQ to assert */
               if ((phase = SCSIhWait4Req(hhcb)) == SCSI_ERRPHASE)
                  return(SCSI_SCAM_FAILURE);

               OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSISIG), expectedPhase);

               ++retry;
               break;

            case SCSI_MOPHASE:
               /* it must be parity error */
               switch (expectedPhase)
               {
                  case SCSI_CMDPHASE:
                     /* parity error detected on message out phase */
                     /* assert ATN signal to re-send the id message */
                     OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSISIG), SCSI_MOPHASE | SCSI_ATNO);
                     expectedPhase = SCSI_MOPHASE;

                     /* clear parity error */
                     OSD_OUTEXACT8(SCSI_AICREG(SCSI_CLRSINT1), SCSI_CLRSCSIPERR);
                     break;

                  case SCSI_DIPHASE:
                  case SCSI_STPHASE:
                     /* parity error detected on data in or status phase */
                     /* clear parity error and deassert ATN */
                     OSD_OUTEXACT8(SCSI_AICREG(SCSI_CLRSINT1),
                                   SCSI_CLRSCSIPERR | SCSI_CLRATNO);

                     /* response with Initiator Detected Error message */
                     OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSIDATL), SCSI_MSG05);

                     /* wait for REQ to assert, target should response */
                     /* with Restore Pointers message */
                     if ((phase = SCSIhWait4Req(hhcb)) == SCSI_ERRPHASE)
                        return(SCSI_SCAM_FAILURE);
                     break;
   
                  case SCSI_MIPHASE:
                     /* parity error detected on message in phase */
                     /* clear parity error and deassert ATN */
                     OSD_OUTEXACT8(SCSI_AICREG(SCSI_CLRSINT1),
                                   SCSI_CLRSCSIPERR | SCSI_CLRATNO);

                     /* response with Message Parity Error message so that */
                     /* the target will re-send the commplete message */
                     OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSIDATL), SCSI_MSG09);

                     /* wait for REQ to assert */
                     if ((phase = SCSIhWait4Req(hhcb)) == SCSI_ERRPHASE)
                        return(SCSI_SCAM_FAILURE);

                     /* set message in phase as exptected phase in SCSISIG */
                     OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSISIG), SCSI_MIPHASE);
                     break;

                  default:
                     SCSIhBadSeq(hhcb);         /* other phases consider error */
                     return(SCSI_SCAM_FAILURE); /* so stop SCSI process immediately */
               }

               break;

            default:
               SCSIhBadSeq(hhcb);         /* other phases consider error */
               return(SCSI_SCAM_FAILURE); /* so stop SCSI process immediately */
         }
      }

      else
      {
         switch (phase)
         {
            case SCSI_MOPHASE:
               /* beginning of MESSAGE OUT phase */
               OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSISEQ),
                  (OSD_INEXACT8(SCSI_AICREG(SCSI_SCSISEQ)))
                        & ~(SCSI_ENSELO|SCSI_ENAUTOATNO));

               OSD_OUTEXACT8(SCSI_AICREG(SCSI_CLRSINT1), SCSI_CLRATNO); /* deasserted ATN */
               OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSIDATL), SCSI_MSGID);   /* IDENTIFY msg, no disconnection */

               /* wait for REQ to assert */
               if ((phase = SCSIhWait4Req(hhcb)) == SCSI_ERRPHASE)
                  return(SCSI_SCAM_FAILURE);

               OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSISIG),
                  (expectedPhase = SCSI_CMDPHASE));   /* expecting command phase */
               break;

            case SCSI_CMDPHASE:
               if (count == 0)
                  OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSIDATL), 0x12);
               else if (count == 4)
                  OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSIDATL), sizeBuffer);
               else
                  OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSIDATL), 0x00);
                  
               /* wait for REQ to assert */
               if ((phase = SCSIhWait4Req(hhcb)) == SCSI_ERRPHASE)
                  return(SCSI_SCAM_FAILURE);

               if (count++ >= 5)
               {
                  OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSISIG),
                     (expectedPhase = SCSI_DIPHASE)); /* expecting data in phase */
                  count = 0;
               }
               break;

            case SCSI_DIPHASE:            /* read INQUIRY data byte */
               OSD_INEXACT8(SCSI_AICREG(SCSI_SCSIDATL));    /* to ACK */

               /* wait for REQ to assert */
               if ((phase = SCSIhWait4Req(hhcb)) == SCSI_ERRPHASE)
                  return(SCSI_SCAM_FAILURE);

               if (count++ >= (SCSI_UEXACT8)(sizeBuffer-1)) /* fill the buffer */
               {
                  OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSISIG),
                     (expectedPhase = SCSI_STPHASE)); /* expecting status phase */
                  count = 0;
               }
               break;

            case SCSI_STPHASE:
               OSD_INEXACT8(SCSI_AICREG(SCSI_SCSIDATL)); /* read status byte */

               /* wait for REQ to assert */
               if ((phase = SCSIhWait4Req(hhcb)) == SCSI_ERRPHASE)
                  return(SCSI_SCAM_FAILURE);

               /* expecting message in phase for command complete */
               OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSISIG),
                  (expectedPhase = SCSI_MIPHASE));
               break;

            case SCSI_MIPHASE:                           /* command complete */
               OSD_INEXACT8(SCSI_AICREG(SCSI_SCSIDATL)); /* read message in byte */

               /* wait for ACK to deassert */
               if (SCSIhScamWaitForRelease(hhcb, SCSI_SCSISIG, SCSI_ACKI))
                  return(SCSI_SCAM_FAILURE);

               cmdDone = 1;
               break;

            default:
               SCSIhBadSeq(hhcb);         /* other phases consider error */
               return(SCSI_SCAM_FAILURE); /* so stop SCSI process immediately */
         }  /* end of switch (phase) */
      }     /* end of else phase mismatch does not occur */

      /* return if timeout or command completed */
      if ((retry >= 3) || cmdDone)
         break;
   }
   return(0x00);
}
#endif

/*********************************************************************
*
*  SCSIhScamSubordinateInit
*
*     Respond to device ID assignment opeartion as a subordinate
*     initiator under scam level 2 dominant initiator protocol.
*
*  Return Value:  none
*                  
*  Parameters:    hhcb
*
*  Remarks:
*
*********************************************************************/
#if SCSI_SCAM_ENABLE >= 1
SCSI_UEXACT8   SCSIhScamSubordinateInit (SCSI_HHCB SCSI_HPTR hhcb)
{
   SCSI_UEXACT8   scamStatus;
   SCSI_UEXACT8   busValue;
   SCSI_UEXACT8   hostScsiID;                     /* host scsi id */
   SCSI_UEXACT16  deviceMap;

   register SCSI_REGISTER  scsiRegister;

   scsiRegister = hhcb->scsiRegister;

   hostScsiID = hhcb->hostScsiID & 0x0f;
   scamStatus = 0x00;

   /* become subordinate initiator so release C/D signal */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSISIG), (SCSI_BSYO | SCSI_SELO | SCSI_IOO));

   hhcb->SCSI_HP_scamSubordinateInit = 1;
   hhcb->SCSI_HP_scsi3rdPartyReset = 0;

   /*
      we need to think the way of quitting this loop when 3rd party reset
      occurred - TL
   */
   while (1)
   {
      if (hhcb->SCSI_HP_scsi3rdPartyReset)
         break;

      /* Synchronization pattern */
      if ((busValue = SCSIhScamDataIn(hhcb)) == SCSI_SYNC_PATTERN)
      {
         /* Isolate & Assign function */
         if ((busValue = SCSIhScamDataIn(hhcb)) == SCSI_ASSIGN)
         {
            if (SCSI_DEVICE_TABLE(hhcb)[hostScsiID].scamWWID[0] & SCSI_VALID_BUT_UNASSIGN)
            {
               if (SCSIhScamIsolation(hhcb))    /* if we won the isolation */
                  SCSIhScamSubordinateIDAssign(hhcb);
            }
         }
         else if (busValue == SCSI_DOMINIT_CONT)  /* Dominant Init Contention */
         {
            SCSIhScamIsolation(hhcb);
         }
         else if (busValue == SCSI_CONFIG_DONE)   /* Config Process Complete */
         {
            /* Got id assinged */
            if (SCSI_DEVICE_TABLE(hhcb)[hostScsiID].scamWWID[0] & SCSI_VALID_AND_ASSIGN)
            {
               break;
            }
            else
            {
               /* set id field to valid but unassign */
               SCSI_DEVICE_TABLE(hhcb)[hostScsiID].scamWWID[0] &= ~SCSI_ID_VALID;
               SCSI_DEVICE_TABLE(hhcb)[hostScsiID].scamWWID[0] |= SCSI_VALID_BUT_UNASSIGN;

               /* clear SubordinateInit flag */
               hhcb->SCSI_HP_scamSubordinateInit = 0;

               if (hhcb->SCSI_HF_scamLevel == 2)
               {
                  while (!hhcb->SCSI_HP_scamSelectDetected)
                  {
                     if (hhcb->SCSI_HP_scsi3rdPartyReset)
                        break;
                  }

                  hhcb->SCSI_HP_scamSelectDetected = 0;
               }
               else
               {
                  /* waiting for SCAM protocol */
                  while (   (OSD_INEXACT8(SCSI_AICREG(SCSI_SCSISIG))
                              != (SCSI_BSYI | SCSI_SELI | SCSI_CDI | SCSI_IOI))
                         && (OSD_INEXACT8(SCSI_AICREG(SCSI_SCSIDATL)) != SCSI_DB7))
                  {
                     if (hhcb->SCSI_HP_scsi3rdPartyReset)
                        break;
                  }

                  /* assert BUSY, SEL, IO and DB7 */
                  OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSISIG), (SCSI_BSYO | SCSI_SELO | SCSI_IOO));
                  OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSIDATL), SCSI_DB7);
               }

               hhcb->SCSI_HP_scamSubordinateInit = 1;
            }
         }  /* else if the configuration process complete */
      }
   }

   deviceMap = 0;          /* Inform OSM that HA is subordinate */
   
   return(scamStatus);
}
#endif

/*********************************************************************
*
*  SCSIhScamSubordinateIDAssign
*
*  To receive an assigned id from the dominant SCAM initiator
*
*  Return Value:  0x00-0x0f:  the ID assigned to HA by another
*                             dominant initiator.
*                 (others):   error in ID assignment
*                  
*  Parameters:    hhcb
*
*  Remarks:
*
*********************************************************************/
#if SCSI_SCAM_ENABLE >= 1
SCSI_UEXACT8 SCSIhScamSubordinateIDAssign (SCSI_HHCB SCSI_HPTR hhcb)
{
   SCSI_UEXACT8   functionQuintet, IDQuintet, regValue, parity;
   SCSI_UEXACT8   hostScsiID, maxAssignableID;
   register SCSI_REGISTER  scsiRegister;

   scsiRegister = hhcb->scsiRegister;

   hostScsiID = hhcb->hostScsiID & 0x0f;

   functionQuintet = SCSIhScamDataIn(hhcb);     /* 1st quintet */
   IDQuintet = SCSIhScamDataIn(hhcb);           /* 2nd quintet */

   /* The check bits(DB3-4) should contain the total number of 0's */
   parity = 3;
   if (IDQuintet & 0x01)
      --parity;
   if (IDQuintet & 0x02)
      --parity;
   if (IDQuintet & 0x04)
      --parity;

   /* assigned id is valid if parity bits are correct */
   if (((SCSI_UEXACT8)(IDQuintet >> 3)) == parity)
   {
      hostScsiID = IDQuintet & 0x07;

      switch (functionQuintet)
      {
         case SCSI_ASSIGN_11ID:
            hostScsiID |= 0x18;
            break;

         case SCSI_ASSIGN_10ID:
            hostScsiID |= 0x10;
            break;

         case SCSI_ASSIGN_01ID:
            hostScsiID |= 0x08;
            break;

         case SCSI_ASSIGN_00ID:
         default:   
            break;
      }

      if (OSD_INEXACT8(SCSI_AICREG(SCSI_SBLKCTL)) & SCSI_SELWIDE)
         maxAssignableID = 0x07;
      else
         maxAssignableID = 0x0f;

      if (hostScsiID <= maxAssignableID)
      {
         hhcb->hostScsiID = hostScsiID;
         regValue = OSD_INEXACT8(SCSI_AICREG(SCSI_hSCSIID(hhcb)));
         regValue &= ~SCSI_OID;
         OSD_OUTEXACT8(SCSI_AICREG(SCSI_hSCSIID(hhcb)), (regValue | hostScsiID));

         SCSI_DEVICE_TABLE(hhcb)[hostScsiID].scamWWID[0] &= ~SCSI_ID_VALID;
         SCSI_DEVICE_TABLE(hhcb)[hostScsiID].scamWWID[0] |= SCSI_VALID_AND_ASSIGN;

         /* asynchronous event call to report host adapter ID change */
         /*    OSMEvent SCSI_ASYNC_EVENT(hhcb, SCSI_AE_HAID_CHANGE); don't need to do this */
         /*    now if the proposal for OSMEvents for Profile changes is */
         /*    approved, then this will need to be implemented */
      }
      else
         hostScsiID = 0xff;      /* error */
   }
   else
      hostScsiID = 0xff;         /* error */

   return(hostScsiID);
}
#endif

/*********************************************************************
*
*  SCSIhScamSupportCheck
*
*  Return Value:  The level of SCAM compliance the hardware supports.
*                 2  -  indicates that hardware supports scam level 2
*                 1  -                                              1
*                 0  -  does not support any scam operation
*
*  Parameters:    scsiRegister
*
*  Remarks:       This routine returns only the SCAM support level that
*                 the hardware supports.  Such information has nothing
*                 to do with the run-time level of SCAM operation.
*
*********************************************************************/
#if SCSI_SCAM_ENABLE
SCSI_UEXACT8 SCSIhScamSupportCheck (register SCSI_REGISTER scsiRegister)
{
   SCSI_UEXACT8   retValue = 0x00;  /* initialize and no SCAM supported */

#if SCSI_SCAM_ENABLE >= 1
   SCSI_UEXACT8   regValue, originalRegValue;

   /* Check to see if SCAM is implemented:                              */
   regValue = OSD_INEXACT8(SCSI_AICREG(SCSI_SXFRCTL0));
   if (!(regValue & SCSI_SCAMEN))
   {
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SXFRCTL0), regValue | SCSI_SCAMEN);
      if (!(OSD_INEXACT8(SCSI_AICREG(SCSI_SXFRCTL0)) & SCSI_SCAMEN))
      {
         OSD_OUTEXACT8(SCSI_AICREG(SCSI_SXFRCTL0), regValue);
         return(0x00);
      }
      else
         OSD_OUTEXACT8(SCSI_AICREG(SCSI_SXFRCTL0), regValue);
   }

   /* To check the SCAM support level of the hardware:                  */
   /*                                                                   */
   /* The SCAMVL bits(SCAMLVL1 & SCAMLVL2) and ALTSTIM bit in SCAMCTRL  */
   /* (offset reg. 1A) are set to 2 and 1.  The register is then read   */
   /* back to see if ALTSTIM is set.  If so, the controller supports    */
   /* SCAM level 2, otherwise, only SCAM level 1 is supported.          */

   originalRegValue = regValue
      = OSD_INEXACT8(SCSI_AICREG(SCSI_SCAMSTAT));  /* save setting */

   if (regValue)     /* the level 1 chip may power on with non-zero value  */
   {                 /* need to make sure this is truly a scam level  1    */
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCAMCTRL), 0x00);

      if (regValue = OSD_INEXACT8(SCSI_AICREG(SCSI_SCAMSTAT)))
      {
         retValue = 1;     /* only SCAM level 1 is possible */
      }
   }

   if (!regValue)          /* possible scam level 2 controller present */
   {
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCAMCTRL), (SCSI_SCAMLVL2|SCSI_ALTSTIM));

      if (OSD_INEXACT8(SCSI_AICREG(SCSI_SCAMSTAT)) & SCSI_ALTSTIMSTA)
      {
         retValue = 2;
      }
      else
         retValue = 1;
   }

   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCAMCTRL), originalRegValue);  /* restore */
#endif                  /* of (SCSI_SCAM_ENABLE >= 1) */

   return (retValue);       /* supports no scam operation */
}
#endif /* SCSI_SCAM_ENABLE */

/*********************************************************************
*
*  SCSIhScamDataOut
*
*  Return Value:  data from the SCSI bus for Dominant Initiator Contention
*                 and subordinate SCAM initiator participate in isolation
*                  
*  Parameters:    hhcb
*                 data
*
*  Remarks:
*
*********************************************************************/
#if SCSI_SCAM_ENABLE >= 1
SCSI_UEXACT8 SCSIhScamDataOut (SCSI_HHCB SCSI_HPTR hhcb,
                               SCSI_UEXACT8 data)
{
   SCSI_UEXACT8   retValue, busValue;
   register SCSI_REGISTER  scsiRegister;

   scsiRegister = hhcb->scsiRegister;

   busValue = 0;
   data &= 0x1f;

   data |= (SCSI_DB5 | SCSI_DB7);
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSIDATL), data);   /* assert DB5 and DB7 */
   data &= ~SCSI_DB7;                                 /* rls DB7, keep DB5 */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSIDATL), data);

   if (SCSIhScamWaitForRelease(  hhcb,             /* wait for DB7 rls */
                                 SCSI_SCSIBUSL,
                                 SCSI_DB7))
      return(SCSI_SCAM_FAILURE);

   retValue = busValue = OSD_INEXACT8(SCSI_AICREG(SCSI_SCSIBUSL)) & 0x1f;

   busValue |=  SCSI_DB5 + SCSI_DB6;               /* assert DB5 and DB6 */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSIDATL), busValue);
   busValue &= ~SCSI_DB5;                          /* drop db5, keep DB6 */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSIDATL), busValue);

   if (SCSIhScamWaitForRelease(  hhcb,             /* wait for DB5 rls */
                                 SCSI_SCSIBUSL,
                                 SCSI_DB5))
     return(SCSI_SCAM_FAILURE);

   busValue = SCSI_DB6 + SCSI_DB7;                 /* assert DB6 and DB7 */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSIDATL), busValue);
   busValue &= ~SCSI_DB6;                          /* drop db6, keep DB7 */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSIDATL), busValue);

   if (SCSIhScamWaitForRelease(  hhcb,             /* wait for DB6 rls */
                                 SCSI_SCSIBUSL,
                                 SCSI_DB6))
      return(SCSI_SCAM_FAILURE);

   else
      return(retValue & 0x1f);
}
#endif

/*********************************************************************
*
*  SCSIhScamDataIn
*
*  Transfer info in from SCAM Targets under TRANSFER CYCLE protocol
*  of SCAM.
*
*  Return Value:  data on the SCSI BUS if bit 7 is reset
*                 error if bit 7 is set
*                  
*  Parameters:    hhcb
*
*  Remarks:
*
*********************************************************************/
#if SCSI_SCAM_ENABLE >= 1
SCSI_UEXACT8 SCSIhScamDataIn (SCSI_HHCB SCSI_HPTR hhcb)
{
   SCSI_UEXACT8   i, busValue, returnValue;
   register SCSI_REGISTER  scsiRegister;

   scsiRegister = hhcb->scsiRegister;

   busValue = i = 0;                      /* initialize */
   returnValue = SCSI_SCAM_FAILURE;       /* initialized at failure */

   if (SCSIhScamWaitForAssertion(hhcb,    /* wait for DB7 to be asserted */
                                 SCSI_SCSIBUSL,
                                 SCSI_DB7))
      return(SCSI_SCAM_FAILURE);

   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSIDATL), SCSI_DB5);  /* assert DB5 */

   if (SCSIhScamWaitForRelease(  hhcb,             /* wait for DB7 rls */
                                 SCSI_SCSIBUSL,
                                 SCSI_DB7))
      return(SCSI_SCAM_FAILURE);

   for (i = 0; i < 16; i++)               /* wait till bus is settled */
      busValue = OSD_INEXACT8(SCSI_AICREG(SCSI_SCSIBUSL));

   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSIDATL), SCSI_DB6);  /* assert DB6, drop DB5 */

   if (SCSIhScamWaitForRelease(  hhcb,             /* wait DB5 rls */
                                 SCSI_SCSIBUSL,
                                 SCSI_DB5))
      return(SCSI_SCAM_FAILURE);

   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SCSIDATL), SCSI_DB7);  /* keep DB7, drop DB6 */

   if (SCSIhScamWaitForRelease(  hhcb,             /* Wait DB6 rls */
                                 SCSI_SCSIBUSL,
                                 SCSI_DB6))
      return(SCSI_SCAM_FAILURE);

   else
      return(busValue & 0x1f);
}
#endif

/*********************************************************************
*
*  SCSIhScamWaitForRelease
*
*  Wait for the release of the specified signal at the specified port
*
*  Return Value:  0x00  -  no error in waiting for release
*                 0xff  -  timeout in waitinf for release
*                  
*  Parameters:    scsiRegister
*                 scsi port address
*                 scsi signal
*
*  Activation:
*
*  Remarks:
*
*********************************************************************/
#if SCSI_SCAM_ENABLE >= 1
SCSI_UEXACT8   SCSIhScamWaitForRelease (SCSI_HHCB SCSI_HPTR hhcb,
                                        SCSI_UEXACT8 scsiPort,
                                        SCSI_UEXACT8 signal)
{
   SCSI_UEXACT8   maxGlitches;            /* maximum of glitches */
   SCSI_UEXACT8   returnValue;
   SCSI_UEXACT8   count;                  /* time-out counter */
   SCSI_UEXACT16  index;
   register SCSI_REGISTER  scsiRegister;

   scsiRegister = hhcb->scsiRegister;

   returnValue = SCSI_SCAM_FAILURE;             /* initialize as failure */

   for ( index = 0; index < 1000; index++)      /* maximum wait - 500ms */
   {
      if ((OSD_INEXACT8(SCSI_AICREG(scsiPort)) & signal) == 0) /* released? */
      {
         maxGlitches = MAX_GLITCHES;
         count = 0x40;

         while ((maxGlitches != 0) && (--count))
         {
            /* deglitch the signal by reading it a few more times */
            if ((OSD_INEXACT8(SCSI_AICREG(scsiPort)) & signal) == 0)
               --maxGlitches;
         }
         if (!maxGlitches) /* Is signal really de-asserted? */
         {
            returnValue = 0x00;
            break;
         }
      }
      else
         SCSIhDelayCount500us(hhcb, 1);      /* wait 500 usec */
   }
   return(returnValue);
}
#endif

/*********************************************************************
*
*  SCSIhScamWaitForAssertion
*
*  Wait for assertion of the specified signal at the specified port
*
*  Return Value:  0x00  -  no error in waiting for assertion
*                 0xff  -  timeout in waiting for assertion
*                  
*  Parameters:    scsiRegister
*                 scsi port address
*                 scsi signal
*
*  Activation:
*
*  Remarks:
*
*********************************************************************/
#if SCSI_SCAM_ENABLE >= 1
SCSI_UEXACT8   SCSIhScamWaitForAssertion (SCSI_HHCB SCSI_HPTR hhcb,
                                          SCSI_UEXACT8 scsiPort,
                                          SCSI_UEXACT8 signal)

{
   SCSI_UEXACT8   maxGlitches;            /* maximum of glitches */
   SCSI_UEXACT8   returnValue;
   SCSI_UEXACT8   count;                  /* time out counter */
   SCSI_UEXACT16  index;
   register SCSI_REGISTER  scsiRegister;

   scsiRegister = hhcb->scsiRegister;

   returnValue = SCSI_SCAM_FAILURE;             /* initialize as failure */

   for ( index = 0; index < 1000; index++)      /* maximum wait - 500ms */
   {
      if (OSD_INEXACT8(SCSI_AICREG(scsiPort)) & signal)    /* asserted ? */
      {
         maxGlitches = MAX_GLITCHES;
         count = 0x40;

         while ((maxGlitches != 0) && (--count))
         {
            /* deglitch the signal by reading it a few more times */
            if (OSD_INEXACT8(SCSI_AICREG(scsiPort)) & signal)
               --maxGlitches;
         }
         if (!maxGlitches) /* Is signal really asserted? */
         {
            returnValue = 0x00;
            break;
         }
      }
      else
         SCSIhDelayCount500us(hhcb, 1);      /* wait 500 usec */
   }
   return(returnValue);
 }
#endif

/***************************************************************************
*                                                                          *
* Copyright 1995,1996,1997,1998 Adaptec, Inc.,  All Rights Reserved.       *
*                                                                          *
* This software contains the valuable trade secrets of Adaptec.  The       *
* software is protected under copyright laws as an unpublished work of     *
* Adaptec.  Notice is for informational purposes only and does not imply   *
* publication.  The user of this software may make copies of the software  *
* for use with parts manufactured by Adaptec or under license from Adaptec *
* and for no other use.                                                    *
*                                                                          *
***************************************************************************/

/***************************************************************************
*
*  Module Name:   HWMSE2.C
*
*  Description:
*                 Codes access SEEPROM attached to host device
*
*  Owners:        ECX IC Firmware Team
*    
*  Notes:         It's up to the OSD to determine if the code should
*                 stay resident after initialization.
*
*  Entry Point(s):
*                 SCSIHSizeSEEPROM
*                 SCSIHReadSEEPROM
*                 SCSIHWriteSEEPROM
*
***************************************************************************/

#include "scsi.h"
#include "hwm.h"

/*********************************************************************
*
*  SCSIHSizeSEEPROM
*
*     Get size of SEEPROM supported
*
*  Return Value:  size of SEEPROM in words (16 bits)
*                  
*  Parameters:    hhcb
*
*  Remarks:       This routine return the size of SEEPROM supported
*                 by host device. There is no guaranteee that the SEEPROM
*                 is attached to the specified host device.
*                  
*********************************************************************/
#if SCSI_SEEPROM_ACCESS
SCSI_INT SCSIHSizeSEEPROM (SCSI_HHCB SCSI_HPTR hhcb)
{
   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_INT size = 0;
   SCSI_UEXACT8 hcntrl;

   /* check if pause chip necessary */
   if (!((hcntrl = OSD_INEXACT8(SCSI_AICREG(SCSI_HCNTRL))) & SCSI_PAUSEACK))
   {
      SCSIhPauseAndWait(hhcb);      
   }

   /* Not just like the legacy hardware, SEECS bit in the SEECTL register   */
   /* of Bayonet chip cannot be written and then read back to check for     */
   /* the written value to be the same until the memory (or flex) port is   */
   /* already completed the arbitration for use. */
   if (hhcb->hardwareMode == SCSI_HMODE_AIC78XX)
   {
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SEEPROM),SCSI_SEECS);
      if (OSD_INEXACT8(SCSI_AICREG(SCSI_SEEPROM)) & SCSI_SEECS)
      {
         if (hhcb->SCSI_HP_se2Type == SCSI_EETYPE_C06C46)
         {
            size = 64;
         }
         else if (hhcb->SCSI_HP_se2Type == SCSI_EETYPE_C56C66)
         {
            size = 128;
         }
      }
   }
   else
   {
      if (hhcb->SCSI_HP_se2Type == SCSI_EETYPE_C06C46)
      {
         size = 64;
      }
      else if (hhcb->SCSI_HP_se2Type == SCSI_EETYPE_C56C66)
      {
         if(hhcb->SCSI_SUBID_seepromType == SCSI_SEE_TYPE2)
         {
            size = 256;
         }
         else
         {
            size = 128;
         }
      }
   }
   
   if (!(hcntrl & SCSI_PAUSEACK))
   {
      SCSI_hWRITE_HCNTRL(hhcb,scsiRegister,hcntrl);
   }

   return(size);
}
#endif

/*********************************************************************
*
*  SCSIHReadSEEPROM
*
*     Read from SEEPROM attached to host device
*
*  Return Value:  status
*                 0 - read successfully
*                  
*  Parameters:    hhcb
*                 dataBuffer to be filled 
*                 offset (in words) relative to the beginning of SEEPROM
*                 length (in words) of information to be read from SEEPROM
*
*  Remarks:       This routine assumes no understanding of how 
*                 information get layed out on SEEPROM. It's caller's
*                 responsibility to know what to read.
*                 Each word is 16 bits (2 bytes).
*                  
*********************************************************************/
int SCSIHReadSEEPROM (SCSI_HHCB SCSI_HPTR hhcb,
                      SCSI_UEXACT8 SCSI_SPTR dataBuffer,
                      SCSI_SINT16 wordAddress,
                      SCSI_SINT16 wordLength)
{
#if SCSI_SEEPROM_ACCESS
   register SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_INT i;
   SCSI_UEXACT8 hcntrl;

#if SCSI_AICTRIDENT
   if (hhcb->hardwareMode == SCSI_HMODE_AICTRIDENT)
   {
      /* Use Trident's hardware state machine to access SEEPORM */
      return(SCSIhTridentReadSEEPROM(hhcb, dataBuffer, wordAddress, wordLength));
   /* If the state machine does not work, we do not try the old way for now. */
   }
#endif

   /* check if pause chip necessary */
   if (!((hcntrl = OSD_INEXACT8(SCSI_AICREG(SCSI_HCNTRL))) & SCSI_PAUSEACK))
   {
      SCSIhPauseAndWait(hhcb);      
   }

   /* Make sure EEPROM interface available.  This is necessary because chip */
   /* like Dagger which does not have regsiters SEEPROM and BRDCTL defined  */
   /* may get hang. */
   /* Not just like the legacy hardware, SEECS bit in the SEECTL register   */
   /* of Bayonet chip cannot be written and then read back to check for     */
   /* the written value to be the same until the memory (or flex) port is   */
   /* already completed the arbitration for use. */
   if (hhcb->hardwareMode == SCSI_HMODE_AIC78XX)
   {
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SEEPROM),SCSI_SEECS);
      if (!(OSD_INEXACT8(SCSI_AICREG(SCSI_SEEPROM)) & SCSI_SEECS))
      {
         return(1);
      }
   }

   /* Enable EEPROM access */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SEEPROM), SCSI_SEEMS);
   while (! (OSD_INEXACT8(SCSI_AICREG(SCSI_SEEPROM)) & SCSI_SEERDY));

   /* for each EEPROM register */
   for (i = 0; i < wordLength; i++)
   {
      /* read from eeprom register */
      *((SCSI_UEXACT16 SCSI_SPTR) (dataBuffer + i * 2)) =
            SCSIhReadE2Register((SCSI_UINT16)(wordAddress + i),
                                scsiRegister, (SCSI_UINT)hhcb->SCSI_HP_se2Type);
   }

   /* Disable EEPROM access */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SEEPROM), 0);

   if (!(hcntrl & SCSI_PAUSEACK))
   {
      SCSI_hWRITE_HCNTRL(hhcb,scsiRegister,hcntrl);
   }
   return(0);
#else
   return(1);
#endif
}

/*********************************************************************
*
*  SCSIHWriteSEEPROM
*
*     Write to SEEPROM attached to host device
*
*  Return Value:  status
*                 0 - write successfully
*                  
*  Parameters:    hhcb
*                 offset (in words) relative to the beginning of SEEPROM
*                 length (in words) of information to written to SEEPROM
*                 dataBuffer
*
*  Remarks:       This routine assumes no understanding of how 
*                 information get layed out on SEEPROM. It's caller's
*                 responsibility to know what will be written.
*                  
*********************************************************************/
#if SCSI_SEEPROM_ACCESS
int SCSIHWriteSEEPROM (SCSI_HHCB SCSI_HPTR hhcb,
                       SCSI_SINT16 wordAddress,
                       SCSI_SINT16 wordLength,
                       SCSI_UEXACT8 SCSI_SPTR dataBuffer)
{
   SCSI_REGISTER scsiRegister = hhcb->scsiRegister;
   SCSI_INT count;
   SCSI_INT i;
   SCSI_UINT16 addressEWEN;
   SCSI_UINT16 addressERASE;
   SCSI_UINT16 addressEWDS;
   SCSI_UEXACT8 hcntrl;

#if SCSI_AICTRIDENT
   if (hhcb->hardwareMode == SCSI_HMODE_AICTRIDENT)
   {
      /* Use Trident's hardware state machine to access SEEPORM */
      return(SCSIhTridentWriteSEEPROM(hhcb, dataBuffer, wordAddress, wordLength));
   /* If the state machine does not work, we don't try the old way for now. */
   }
#endif

   /* check if pause chip necessary */
   if (!((hcntrl = OSD_INEXACT8(SCSI_AICREG(SCSI_HCNTRL))) & SCSI_PAUSEACK))
   {
      SCSIhPauseAndWait(hhcb);      
   }

   /* Make sure EEPROM interface available.  This is necessary because chip */
   /* like Dagger which does not have regsiters SEEPROM and BRDCTL defined  */
   /* may get hang. */
   /* Not just like the legacy hardware, SEECS bit in the SEECTL register   */
   /* of Bayonet chip cannot be written and then read back to check for     */
   /* the written value to be the same until the memory (or flex) port is   */
   /* already completed the arbitration for use. */
   if (hhcb->hardwareMode == SCSI_HMODE_AIC78XX)
   {
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SEEPROM),SCSI_SEECS);
      if (!(OSD_INEXACT8(SCSI_AICREG(SCSI_SEEPROM)) & SCSI_SEECS))
      {
         return(1);
      }
   }

   /* setup EEPROM address needed based on type */
   if (hhcb->SCSI_HP_se2Type == SCSI_EETYPE_C06C46)
   {                                   /* Setup for NM93C06 or NM93C46 */
      addressEWEN = 0x0030;                  /* OR with read command */
      addressERASE = (SCSI_UINT16)(0x00C0 | wordAddress);
      addressEWDS = 0x0000;
      count = 8;                             /* Setup loop count */
   }
   else
   {                                   /* Setup for NM93C56 or NM93C66 */
      addressEWEN = 0x00C0;                  /* OR with read command */
      addressERASE = (SCSI_UINT16)(0x0300 | wordAddress);
      addressEWDS = 0x0000;
      count = 10;                            /* Setup loop count */
   }

   /* Enable EEPROM access */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SEEPROM), SCSI_SEEMS);
   while ( !(OSD_INEXACT8(SCSI_AICREG(SCSI_SEEPROM)) & SCSI_SEERDY) );

   /* enable EEPROM programming */
   SCSIhSendE2Address(addressEWEN, count, scsiRegister);

   /* Drop CS to make sure we don't drop the first erase command */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SEEPROM), SCSI_SEEMS);

   /* copy (write) to EEPROM register */
   for (i = 0; i < wordLength; i++)
   {
      /* erase EEPROM register */
      SCSIhSendE2Address((SCSI_UINT16)(addressERASE + i), count, scsiRegister);

      /* drop CS */
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SEEPROM), SCSI_SEEMS);
      SCSIhWait2usec(SCSI_SEEMS, scsiRegister);

      /* raise CS */
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SEEPROM), SCSI_SEEMS | SCSI_SEECS);
      SCSIhWait2usec(SCSI_SEEMS | SCSI_SEECS, scsiRegister);

      /* wait for done status */
      while(!(OSD_INEXACT8(SCSI_AICREG(SCSI_SEEPROM)) & SCSI_SEEDI))
         ;

      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SEEPROM), SCSI_SEEMS);
      SCSIhWait2usec(SCSI_SEEMS, scsiRegister);

      /* write EEPROM register */
      SCSIhWriteE2Register(((SCSI_UINT16)wordAddress + i),
         *((SCSI_UEXACT16 SCSI_SPTR) (dataBuffer + i * 2)),
                     scsiRegister, hhcb->SCSI_HP_se2Type);
   }

   /* disable EEPROM programming */
   SCSIhSendE2Address(addressEWDS, count, scsiRegister);

   /* Disable EEPROM access */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SEEPROM), 0);

   if (!(hcntrl & SCSI_PAUSEACK))
   {
      SCSI_hWRITE_HCNTRL(hhcb,scsiRegister,hcntrl);
   }

   return(0);
}
#endif

/*********************************************************************
*
*  SCSIhReadE2Register -
*
*     Read from EEPROM register word
*
*  Return Value:  register value (word)
*
*  Parameters:    EEPROM register address
*                 base address of AIC-7870
*                 seeprom type
*
*  Activation:    read_eeprom
*
*  Remarks:
*
*********************************************************************/
#if SCSI_SEEPROM_ACCESS
SCSI_UEXACT16 SCSIhReadE2Register (SCSI_UINT16 wordAddress,
                                   SCSI_REGISTER scsiRegister,
                                   SCSI_UINT se2Type)
{
   SCSI_UEXACT16 retval = 0;
   SCSI_INT i;
   SCSI_INT count;

   /* send EEPROM register address with op code for read */
   if (se2Type == SCSI_EETYPE_C06C46)
   {                                   /* Setup for NM93C06 or NM93C46 */
      wordAddress |= 0x0080;                       /* OR with read command */
      count = 8;                                   /* Setup loop count */
   }
   else
   {                                   /* Setup for NM93C56 or NM93C66 */
      wordAddress |= 0x0200;                       /* OR with read command */
      count = 10;                                  /* Setup loop count */
   }
   SCSIhSendE2Address(wordAddress, count, scsiRegister);

   /* get value for the register */
   for ( i = 0; i < 16; i++ )
   {
      retval <<= 1;
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SEEPROM), SCSI_SEEMS | SCSI_SEECS | SCSI_SEECK);
      SCSIhWait2usec(SCSI_SEEMS | SCSI_SEECS | SCSI_SEECK, scsiRegister);
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SEEPROM), SCSI_SEEMS | SCSI_SEECS);
      if ( OSD_INEXACT8(SCSI_AICREG(SCSI_SEEPROM)) & SCSI_SEEDI )
         retval |= 1;
      SCSIhWait2usec(SCSI_SEEMS | SCSI_SEECS, scsiRegister);
   }

   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SEEPROM), SCSI_SEEMS);
   SCSIhWait2usec(SCSI_SEEMS, scsiRegister);
   return( retval );
}
#endif

/*********************************************************************
*
*  SCSIhWriteE2Register -
*
*     Read from EEPROM register word
*
*  Return Value:  none
*
*  Parameters:    EEPROM register address
*                 value to be written
*                 base address of AIC-7870
*                 eeprom type
*
*  Activation:    read_eeprom
*
*  Remarks:
*
*********************************************************************/
#if SCSI_SEEPROM_ACCESS
void SCSIhWriteE2Register (SCSI_UINT16 wordAddress,
                           SCSI_UINT16 value,
                           SCSI_REGISTER scsiRegister,
                           int se2Type)
{
   SCSI_UEXACT8 se2Value;
   SCSI_INT i;
   SCSI_INT count;

   /* send EEPROM register address with op code for write */
   if (se2Type == SCSI_EETYPE_C06C46)
   {                                   /* Setup for NM93C06 or NM93C46 */
      wordAddress |= 0x0040;                       /* OR with read command */
      count = 8;                                   /* Setup loop count */
   }
   else
   {                                   /* Setup for NM93C56 or NM93C66 */
      wordAddress |= 0x0100;                       /* OR with read command */
      count = 10;                                  /* Setup loop count */
   }
   SCSIhSendE2Address(wordAddress, count, scsiRegister);

   /* set value for the register */
   for ( i = 0; i < 16; i++ )
   {
      se2Value = SCSI_SEEMS + SCSI_SEECS + ((value & (SCSI_UINT16)0x8000) ? SCSI_SEEDO : 0);
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SEEPROM), se2Value);
      SCSIhWait2usec(se2Value, scsiRegister);
      OSD_OUTEXACT8(SCSI_AICREG(SCSI_SEEPROM), (SCSI_UEXACT8)(se2Value |= SCSI_SEECK));
      SCSIhWait2usec(se2Value, scsiRegister);
      value <<= 1;
   }

   /* drop the clock */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SEEPROM), SCSI_SEEMS | SCSI_SEECS);
   SCSIhWait2usec(SCSI_SEEMS | SCSI_SEECS, scsiRegister);

   /* drop CS */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SEEPROM), SCSI_SEEMS);
   SCSIhWait2usec(SCSI_SEEMS, scsiRegister);

   /* raise CS */
   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SEEPROM), SCSI_SEEMS | SCSI_SEECS);
   SCSIhWait2usec(SCSI_SEEMS | SCSI_SEECS, scsiRegister);

   /* wait for done status */
   while(!(OSD_INEXACT8(SCSI_AICREG(SCSI_SEEPROM)) & SCSI_SEEDI))
      ;

   OSD_OUTEXACT8(SCSI_AICREG(SCSI_SEEPROM), SCSI_SEEMS);
   SCSI