/*
    i2c-temp-mon.c - i2c-bus driver, char device interface  

    Copyright (C) 1995-97 Simon G. Vogl
    Copyright (C) 1998-99 Frodo Looijaard <frodol@dds.nl>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

/* Note that this is a complete rewrite of Simon Vogl's i2c-temp-mon module.
   But I have used so much of his original code and ideas that it seems
   only fair to recognize him as co-author -- Frodo */

/* The I2C_RDWR ioctl code is written by Kolja Waschk <waschk@telos.de> */

/* $Id: i2c-temp-mon.c,v 1.3 2000/12/05 10:48:05 stig Exp $ */

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/fs.h>
#include <linux/malloc.h>
#include <linux/version.h>

/* If you want debugging uncomment: */
/* #define DEBUG */

#include <linux/init.h>
#include <asm/uaccess.h>

#include <linux/i2c-new.h>
//#include <linux/i2c-temp-mon.h>

#ifdef MODULE
extern int init_module(void);
extern int cleanup_module(void);
#endif /* def MODULE */

/* struct file_operations changed too often in the 2.1 series for nice code */

static int i2c_temp_attach_adapter(struct i2c_adapter *adap);
static int i2c_temp_detach_client(struct i2c_client *client);
static int i2c_temp_command(struct i2c_client *client, unsigned int cmd,
                           void *arg);

#ifdef MODULE
static
#else
extern
#endif
       int __init i2c_temp_init(void);
static int i2c_temp_cleanup(void);

#define I2CDEV_ADAPS_MAX I2C_ADAP_MAX
static struct i2c_adapter *i2c_temp_adaps[I2CDEV_ADAPS_MAX];
static struct i2c_adapter *i2c_temp_first;

static struct i2c_driver i2c_temp_driver = {
	/* name */		"i2c-temp driver",
	/* id */		I2C_DRIVERID_I2CDEV,
	/* flags */		I2C_DF_DUMMY,
	/* attach_adapter */	i2c_temp_attach_adapter,
	/* detach_client */	i2c_temp_detach_client,
	/* command */		i2c_temp_command,
	/* inc_use */		NULL,
	/* dec_use */		NULL,
};

static int i2c_temp_initialized=0;
static struct timer_list i2c_temp_timer;
static int lm75_addrs_arr[5]={0,};
static int lm75_high=0, lm75_low=0, lm75_intv=0, lm75_print=0;
static char *lm75_addrs=NULL;
static void i2c_temp_watch( unsigned long );

void i2c_temp_sleep(long timeout)
{
        init_timer(&i2c_temp_timer);
        i2c_temp_timer.expires = jiffies + timeout;
        i2c_temp_timer.data = 0UL;
        i2c_temp_timer.function = i2c_temp_watch;

        add_timer(&i2c_temp_timer);
}

static void i2c_temp_watch( unsigned long blob )
{
	int i, res=0;
	union i2c_smbus_data temp;

	for (i=0 ; lm75_addrs_arr[i]; i++) {
		res += i2c_smbus_xfer(i2c_temp_first,lm75_addrs_arr[i], 0,
		      I2C_SMBUS_READ, 0, I2C_SMBUS_WORD_DATA,&temp);
		if (lm75_print)
			printk("unit%d:%dC\n", i, temp.word &0xff);
		if ((temp.word & 0xff) > lm75_high) {
			printk("********************************************************\n");
			printk("CPU temp. excced the limit, shutdown procedure initiated\n");
			printk("Limit %dC, CPU temp %dC\n", lm75_high, temp.word&0xff);
			printk("To stop shutdown procedure run \"/sbin/shutdown -c\"\n");
			printk("********************************************************\n");
			kill_proc(1, SIGPWR, 1);
		}
	}
	i2c_temp_sleep(lm75_intv * HZ);
}

int i2c_temp_attach_adapter(struct i2c_adapter *adap)
{
	int i;

	if ((i = i2c_adapter_id(adap)) < 0) {
		printk("i2c-temp-mon.o: Unknown adapter ?!?\n");
		return -ENODEV;
	}
	if (i >= I2CDEV_ADAPS_MAX) {
		printk("i2c-temp-mon.o: Adapter number too large?!? (%d)\n",i);
		return -ENODEV;
	}

	if (! i2c_temp_adaps[i]) {
		i2c_temp_adaps[i] = adap;
		printk("i2c-temp-mon.o: Registered '%s' as minor %d\n",adap->name,i);
	} else {
		i2c_temp_adaps[i] = NULL;
#ifdef DEBUG
		printk("i2c-temp-mon.o: Adapter unregistered: %s\n",adap->name);
#endif
	}
	if (i == 0) 
		i2c_temp_first = adap;

	return 0;
}

int i2c_temp_detach_client(struct i2c_client *client)
{
	return 0;
}

static int i2c_temp_command(struct i2c_client *client, unsigned int cmd,
                           void *arg)
{
	return -1;
}

static int atoi(char *v)
{
	int r=0;

	while (*v) 
		r = r*10 + *v++ - '0';
	return r;
}

int detect_lm75_addrs(void)
{
	int addr=0x48; /* possible start address */
	int i=0;

	printk("Detecting lm75s\n");
	printk("To force addresses, try lm75_addrs=74:78, lm75_addrs=72 etc\n");

	for (addr=0x48; addr < 0x50; addr++) {
                if (i2c_smbus_xfer(i2c_temp_first, addr,0,0,0,I2C_SMBUS_QUICK,NULL) >= 0) {
			printk("LM75 detected address=%d\n", addr);
			lm75_addrs_arr[i++] = addr;
		}
	}
	return i;
}
		
void parse_lm75_addrs(char *lm75_addrs)
{
	char *s, *p;
	char bk[10];
	int i=0;

	strncpy(bk, lm75_addrs, 10);

	s = p = bk;

	while (*p && i<2) {
		if (*p==':') {
			*p = 0;
			lm75_addrs_arr[i] = atoi(s);
printk("addr %d %x %s\n", i, lm75_addrs_arr[i], s);
			s = ++p;
			i++;
		} else 
			p++;
	}
	lm75_addrs_arr[i] = atoi(s);
printk("addr %d %x %s\n", i, lm75_addrs_arr[i], s);
}

int i2c_temp_lm75_setup(void)
{
	int res=0, i;
	union i2c_smbus_data temp;

	for (i=0 ; lm75_addrs_arr[i]; i++) {
		temp.word = 0;
		res  = i2c_smbus_xfer(i2c_temp_first, lm75_addrs_arr[i], 0,
		      I2C_SMBUS_WRITE, 1, I2C_SMBUS_WORD_DATA,&temp);
		temp.word = lm75_low;
		res += i2c_smbus_xfer(i2c_temp_first, lm75_addrs_arr[i], 0,
		      I2C_SMBUS_WRITE, 2, I2C_SMBUS_WORD_DATA,&temp);
		temp.word = lm75_high;
		res += i2c_smbus_xfer(i2c_temp_first, lm75_addrs_arr[i], 0,
		      I2C_SMBUS_WRITE, 3, I2C_SMBUS_WORD_DATA,&temp);
	}
	return res;
}
	
int __init i2c_temp_init(void)
{
	int res;

	printk("i2c-temp-mon.o: i2c temp. monitor driver module\n");

	i2c_temp_initialized = 0;
	if ((res = i2c_add_driver(&i2c_temp_driver))) {
		printk("i2c-temp-mon.o: Driver registration failed, module not inserted.\n");
		i2c_temp_cleanup();
		return res;
	}

	if (!lm75_addrs) {
		if (!detect_lm75_addrs())
			return -1;
	} else 
		parse_lm75_addrs(lm75_addrs);

	if (!lm75_intv) lm75_intv = 60;
	if (!lm75_high) lm75_high = 60;
	if (!lm75_low) lm75_low = 45;

	i2c_temp_lm75_setup();

	i2c_temp_watch( 0UL );		/* arg added to make timer type match */

	i2c_temp_initialized ++;
	return 0;
}

int i2c_temp_cleanup(void)
{
	int res;

	if (i2c_temp_initialized) {
		if ((res = i2c_del_driver(&i2c_temp_driver))) {
			printk("i2c-temp-mon.o: Driver deregistration failed, "
			       "module not removed.\n");
			return res;
		}
	}
        del_timer(&i2c_temp_timer);
	return 0;
}

EXPORT_NO_SYMBOLS;

#ifdef MODULE

MODULE_AUTHOR("Soohoon Lee <soohoon.lee@alpha-processor.com>");
MODULE_DESCRIPTION("I2C temp. monitor");

MODULE_PARM(lm75_addrs, "s");
MODULE_PARM(lm75_high, "i");
MODULE_PARM(lm75_low, "i");
MODULE_PARM(lm75_intv, "i");
MODULE_PARM(lm75_print, "i");

int init_module(void)
{
	return i2c_temp_init();
}

int cleanup_module(void)
{
	return i2c_temp_cleanup();
}

#endif /* def MODULE */

