/*---------------------------------------------------------------------
 *        [ Copyright (c) 1999 Alpha Processor Inc.] - Unpublished Work
 *          All rights reserved
 * 
 *    This file contains source code written by Alpha Processor, Inc.
 *    It may not be used without express written permission. The
 *    expression of the information contained herein is protected under
 *    federal copyright laws as an unpublished work and all copying
 *    without permission is prohibited and may be subject to criminal
 *    and civil penalties. Alpha Processor, Inc.  assumes no
 *    responsibility for errors, omissions, or damages caused by the use
 *    of these programs or from use of the information contained herein.
 *  
 *-------------------------------------------------------------------*/
/* Alpha Diagnostics Asset Info Manager */
/* Begun by Stig Telfer, API NetWorks, 21 February 2001 */


#undef TRACE_ENABLE

#include "lib.h"
#include "uilib.h"

#include "platform.h"
#include "i2c.h"
#include "i2c/pcf-eprom.h"

#include "asset.h"


/*----------------------------------------------------------------------*/
/* Private widgetry */


/* Dump an EEPROM's entire contents as a big block of printable ASCII */
#define ROW_BYTES 64
static void eeprom_dump( const eeprom_t *E )
{
    int i, j;
    char c;

    for ( i=0; i<EEPROM_MAXLEN; i+=ROW_BYTES )
    {
	for( j=0; j<ROW_BYTES; j++ )
	{
	    c = E->data[i+j];
	    printf_dbm( "%c", isprint( c ) ? c : '.' );
	}
	printf_dbm( "\r" );
    }
}


/*----------------------------------------------------------------------*/
/* Given an I2C device descriptor, fill in the details about an EEPROM */
/* The returned data is allocated from the heap so may be free'd */
eeprom_t *eeprom_scan( eeprom_t *E )
{
    DBM_STATUS sval;

    if ( E == NULL )
	return NULL;

    /* Can we reach this EEPROM in the current config? */
    sval = plat_i2c_select( E->I );
    if ( sval != STATUS_SUCCESS )
    {
	mobo_alertf( "EEPROM Access Error!",
		"This EEPROM, '%s' is either not present\r"
		"Or it is defective", E->I->desc );
	return E;
    }

    /* Read in the data from the EEPROM */
    sval = pcfeprom_read( E->I->addr, 0, EEPROM_MAXLEN, E->data );

    if ( sval == STATUS_FAILURE )
    {
	mobo_alertf( "EEPROM read error!",
	    "This EEPROM, '%s' may be defective", E->I->desc );
    }

    return E;
}


/* Given EEPROM descriptor, allow the user to view the entire contents of
 * the EEPROM */

DBM_STATUS eeprom_examine( eeprom_t *E )
{
    Point P;
    DBM_STATUS sval;

    /* Can we reach this EEPROM in the current config? */
    sval = plat_i2c_select( E->I );

    /* EEPROM absent, eg on removable module? */
    if ( sval == STATUS_WARNING )
    {
	mobo_alertf( "EEPROM Access Error!",
		"This EEPROM, '%s' is not present\r"
		"in the current hardware configuration", E->I->desc );
	return STATUS_FAILURE;
    }

    /* EEPROM unreachable due to f*cked hardware? */
    if ( sval == STATUS_FAILURE )
    {
	mobo_alertf( "EEPROM Access Error!",
		"This EEPROM, '%s' appears to be defective", E->I->desc );
	return STATUS_FAILURE;
    }


    /* Otherwise we're happy to display contents */
    mobo_box( r_app, E->I->desc );
    P.x = r_app.x + 2, P.y = r_app.y + 2;
    mobo_goto( P );
    eeprom_dump( E );

 
    /* Could there be some kind of validation eg checksumming applied here? */
    return STATUS_SUCCESS;
}


/* Given EEPROM descriptor, allow the user to edit the contents of the EEPROM */
/* Return value depends on whether the contents were written back OK */

DBM_STATUS eeprom_edit( eeprom_t *E )
{
    static const Point orig = { 2, 4 }, new = { 2, 12 };
    static const Point oblk = { 4, 6 }, nblk = { 4, 14 };
    Point current;
    int key, ptr;
    DBM_STATUS sval;
    int retries;

    /* Draw the user interface */
    mobo_box( r_scrn, "Edit Asset Information EEPROM Contents" );

    /* Is this EEPROM actually available? */
    if ( plat_i2c_select( E->I ) != STATUS_SUCCESS )
    {
	mobo_alertf( "Access Error",
		"Diags is unable to access the selected EEPROM" );
	return STATUS_FAILURE;
    }

    printf_dbm( "EEPROM: %s", E->I->desc );

    mobo_goto( orig );
    printf_dbm( "Original Contents:" );
    mobo_goto( oblk );
    eeprom_dump( E );

    mobo_goto( new );
    printf_dbm( "New Contents:" );
    mobo_goto( nblk );
    eeprom_dump( E );

    mobo_goto( p_help );
    printf_dbm( "Use cursor keys to navigate, press RETURN to finish editing" );
    
    /* Now for the editing cycle */
    ptr = 0;
    do {
	current.x = nblk.x + ptr % ROW_BYTES;
	current.y = nblk.y + ptr / ROW_BYTES;
	mobo_goto( current );
	key = mobo_key( 0 );

	/* Check for quit request */
	if ( key == '\r' || key == '\n' )
		break;

	

	/* Check for cursor keys as ESC-[-A...ESC-[-D */
	if ( key == '\x1B' )		/* ESC */
	{
	    key = mobo_key( 1 );

	    /* Check for backspace (either ESC-0x7E or ESC-0x7F) */
	    if ( key == 0x7E || key == 0x7F )
	    {
		if( ptr > 0 )
		    ptr--;
		continue;
	    }

	    /* If we're a cursor key, we need a ']' next */
	    if ( key != '[' )
		continue;

	    key = mobo_key( 1 );
	    switch( key )
	    {
		case 'A':		/* up */
		    if( ptr >= ROW_BYTES )
			ptr -= ROW_BYTES;
		    continue;

		case 'B':		/* down */
		    if ( EEPROM_MAXLEN - ptr > ROW_BYTES )
			ptr += ROW_BYTES;
		    continue;

		case 'C':		/* right */
		    if ( EEPROM_MAXLEN - ptr > 1 )
			ptr++;
		    continue;

		case 'D':		/* left */
		    if( ptr > 0 )
			ptr--;
		    continue;

		default:		/* anything else */
		    continue;
	    }
	}

	E->data[ ptr ] = key;
	ptr++;
	if ( ptr > EEPROM_MAXLEN - 1 ) 
	    ptr = EEPROM_MAXLEN - 1;

	printf_dbm( "%c", key );

    } while( TRUE );


    /* Write back the contents of the EEPROM */
    mobo_box( r_error, "Writing to EEPROM" );
    printf_dbm( "Writing data back to '%s'\r"
		"Please wait...", E->I->desc );

    for ( ptr = 0; ptr < EEPROM_MAXLEN; ptr++ )
    {
	do {
	    retries = 0;
	    sval = pcfeprom_write( E->I->addr, ptr, 1, &E->data[ptr] );
	    if ( sval == STATUS_FAILURE )
	    {
		TRACE( "Write attempt %d addr %d failed, retrying...\n",
			retries, ptr );
		retries++;
		msleep( 50 );
	    }
	} while( retries < 4 && sval == STATUS_FAILURE );
	
	if ( sval == STATUS_FAILURE )
	{
	    mobo_alertf( "EEPROM Write Error!",
		"Could not write to EEPROM '%s'", E->I->desc );
	    return STATUS_FAILURE;
	}

    }

    /* Read back the contents of the EEPROM */
    mobo_box( r_error, "Verifying EEPROM" );
    printf_dbm( "Reading data back from '%s'\r"
		"Please wait...", E->I->desc );
    E = eeprom_scan( E );

    return STATUS_SUCCESS;
}

