/*---------------------------------------------------------------------
 *        [ Copyright (c) 1999 Alpha Processor Inc.] - Unpublished Work
 *          All rights reserved
 * 
 *    This file contains source code written by Alpha Processor, Inc.
 *    It may not be used without express written permission. The
 *    expression of the information contained herein is protected under
 *    federal copyright laws as an unpublished work and all copying
 *    without permission is prohibited and may be subject to criminal
 *    and civil penalties. Alpha Processor, Inc.  assumes no
 *    responsibility for errors, omissions, or damages caused by the use
 *    of these programs or from use of the information contained herein.
 *  
 *-------------------------------------------------------------------*/
/* Linux loader from the manufacturing diags. */
/* Begun by Stig Telfer, Alpha Processor Inc, 17 April 1999 */
/* this code borrows heavily from milo and the kernel itself */


#undef TRACE_ENABLE	/* define this to get debug info about booting */

#include "lib.h"

#include "uilib.h"
#include "specifics.h"
#include "platform.h"
#include "northbridge.h"
#include "osf.h"
#include "hwrpb.h"
#include "cmos_rtc.h"
#include "floppy.h"
#include "rom.h"
#include "nvram.h"
#include "smp.h"			/* oh yes :-) */
#include "ledcodes.h"

#include "cpu/ev6.h"

/* Page table entry (PTE) management */
typedef struct { unsigned long pte; } pte_t;
typedef struct { unsigned long pgprot; } pgprot_t;

#define pte_val(x)      ((x).pte)
#define pgprot_val(x)   ((x).pgprot)

#define __pte(x)        ((pte_t) { (x) } )
#define __pgprot(x)     ((pgprot_t) { (x) } )

/*
 * Conversion functions:  convert a page and protection to a page entry,
 * and a page entry and page directory to the page they refer to.
 */

/* extern */ inline pte_t mk_pte(unsigned long page, pgprot_t pgprot)
{
    pte_t pte;
    pte_val(pte) =
	((page - LINUX_SUPER) << (32 - PAGE_SHIFT)) | pgprot_val(pgprot);
    return pte;
}

#define _PFN_MASK       0xFFFFFFFF00000000

/* extern */ inline unsigned long pte_page(pte_t pte)
{
    return LINUX_SUPER + ((pte_val(pte) & _PFN_MASK) >> (32 - PAGE_SHIFT));
}

/*
 * OSF/1 PAL-code-imposed page table bits
 */
#define _PAGE_VALID     0x0001
#define _PAGE_FOR       0x0002	/* used for page protection (fault on read) */
#define _PAGE_FOW       0x0004	/* used for page protection (fault on write) */
#define _PAGE_FOE       0x0008	/* used for page protection (fault on exec) */
#define _PAGE_ASM       0x0010
#define _PAGE_KRE       0x0100	/* xxx - see below on the "accessed" bit */
#define _PAGE_URE       0x0200	/* xxx */
#define _PAGE_KWE       0x1000	/* used to do the dirty bit in software */
#define _PAGE_UWE       0x2000	/* used to do the dirty bit in software */



/* set the correct superpage bits for DBM.  Note: Linux currently doesn't 
 * use this superpage mode, so we need to be careful about which superpage
 * we're talking about during handover - FFFF8 or FFFFFC... */

/* Vital addresses */
#define DFLT_INITRD_BASE	(16UL<<20)	/* off the top of my head... */

/* These definitions should only be accessed via KRNLADDR below */
#define K_BASE			0x0
#define K_SWAPPER_PGD		0x0
#define K_INIT_STACK		0x2000
#define K_EMPTY_PGT		0x4000
#define K_EMPTY_PGE		0x8000
#define K_ZERO_PGE		0xA000
#define		ZPG_INITRD_BASE	0x100		/* Initrd start address ptr */
#define		ZPG_INITRD_SIZE	0x108		/* Initrd length in bytes */
#define K_START_ADDR		0x10000

/* Raw physical address with no superpage (of any variety) */
#define KRNLADDR(rgn)	( (krnlbase-K_START_ADDR) | (rgn) )

#define DBM_PGTBL	(DBM_SUPER | KRNLADDR(K_SWAPPER_PGD) )
#define DBM_KERNEL	(DBM_SUPER | KRNLADDR(K_START_ADDR) )
#define DBM_ZERO	(DBM_SUPER | KRNLADDR(K_ZERO_PGE) )
#define DBM_HWRPB	(DBM_SUPER | HWRPB_DATA)

#define LINUX_PGTBL	(LINUX_SUPER | KRNLADDR(K_SWAPPER_PGD) )
#define LINUX_STACK	(LINUX_SUPER | KRNLADDR(K_INIT_STACK) )
#define LINUX_KERNEL	(LINUX_SUPER | KRNLADDR(K_START_ADDR) )

/* Virtual address of HWRPB used by kernel */
#define INIT_HWRPB 	0x10000000UL

/* The InitRD can go more or less anywhere (depending a little on its size) */
#define DBM_DFLT_INITRD_START	( DBM_SUPER | DFLT_INITRD_BASE )

/* As of 2.4 krnlbase needs to be flexible, store it here */
static uint64 palbase = DIAGS_OSFPAL;
static uint64 krnlbase = (uint64)NULL;
static uint64 initrdbase = DBM_DFLT_INITRD_START, initrdlen;

String boot_device="/dev/sda2";		/* in the absence of anything better */
String boot_file="vmlinux.gz";		/* in the absence of anything better */


/* Can't tell beforehand how much space we'll need; guess a max kernel size */
#define MAX_KERNEL_SIZE 	(4UL<<20)


/* These are used to refer to what is being downloaded, and are tested for 
 * during post-processing (so beware calling ld_from_medium() with anything
 * different).
 */
#define KERNELSTR	"kernel"
#define	PALCODESTR	"PALcode"
#define INITRDSTR	"InitRD"



int elf_unpack( char *src, char **dest );
int uncompress_kernel(void *src, void *dest );


#ifdef TRACE_ENABLE
/* validation of image when loaded into memory. */
static unsigned checksum( unsigned *rgn, unsigned bytes )
{
    unsigned total = 0;
    unsigned i;

    for ( i=0; i < bytes/4; i++ )
    {
	total = compute_checksuml( rgn[i], total );
    }
    return total;
}
#endif



/*----------------------------------------------------------------------*/
/* Obtain executable images, ie for kernel and PALcode */


static DBM_STATUS ld_floppy_file( String iname, size_t *isize, char *dst )
{
    printf_dbm("Enter %s filename [or quit]: ", iname );
    mobo_input(cbuf, sizeof(cbuf));
    if (strcmp( cbuf, "quit" ) == 0)		return STATUS_FAILURE;
    printf_dbm("...");
    *isize = LoadAFile(cbuf, dst);
    printf_dbm("\n");
    if (*isize == -1)				return STATUS_FAILURE;

    boot_device="/dev/fd0";
    return STATUS_SUCCESS;
}

static DBM_STATUS ld_xmodem_file( String iname, size_t *isize, char *dst )
{
    io_dev com;
    String com_s;

    /* find the most likely port we'll be downloading over */
    switch( getty() ) {
	default:
	case GRP_COM1:	
		com = DEV_COM1, com_s = "COM1", boot_device = "/dev/ttyS0";
		break;
	case GRP_COM2:	
		com = DEV_COM2, com_s = "COM2", boot_device = "/dev/ttyS1";
		break;
	case GRP_COM3:	
		com = DEV_COM3, com_s = "COM3", boot_device = "/dev/ttyS2";
		break;
	case GRP_COM4:	
		com = DEV_COM4, com_s = "COM4", boot_device = "/dev/ttyS3";
		break;
	case GRP_SROM:	
		com = DEV_SROM, com_s = "SROM", boot_device = "/dev/srom0";
		break;
    }
    printf_dbm("Press any key to start %s download (via %s): ", iname, com_s );
    mobo_key( 0 );

    *isize = XReceive( dst, com );
    printf_dbm("\n");
    if ( *isize <= 0 )				return STATUS_FAILURE;

    boot_device="/dev/ttyS0";
    return STATUS_SUCCESS;
}

static DBM_STATUS ld_rom_image( String iname, FWID I, size_t *isize, char *dst )
{
    rom_t *R;

    /* Here we try our best to quickly find the matching FWID.
     * The first place to look is ROM-in-RAM.  Then a quick scan of the ROM.
     * If it doesn't show up on a quick search we build a full index of the ROM.
     * Then we use the usual lookup routine to find a matching R (if any)
     */

    R = rom_search( I, rom_ram_map );
    if ( R != NULL )
    {
	printf_dbm( "Found '%s' image in ROM-in-RAM\n", iname );
    }
    else 
    {
	printf_dbm("Scanning ROM for image...\n" );
	if ( !rom_mapped )
	{
	    R = rom_quickscan( I );
	    if ( R == NULL )
	    {
		rom_index();
		R = rom_phys_map;
	    }
	} else  R = rom_phys_map;


	/* R points to a linked list:
	 * - either just the matching entry, which makes lookup trivial
	 * - or the entire ROM index (an exhaustive search was needed)
	 */
	R = rom_search( I, R );
	if ( R == NULL ) {
	    printf_dbm("Image %s not found in ROM!\n", iname );
	    return STATUS_FAILURE;
	}
    }

    printf_dbm("Loading %s... ", iname);
    if ( rom_load( R, dst ) != dst )
    {
	printf_dbm("failed!\n");
	return STATUS_FAILURE;
    }

    *isize = R->rsize;
    printf_dbm("loaded (%dKb)\n", *isize >> 10 );

    boot_device = "/dev/rom", boot_file = "vmlinuz.rom";
    return STATUS_SUCCESS;
}


/* Main image-loading function, calls the above for selected media */
/* Set choice non-zero to skip user interaction (typically 'r' for ROM) */
/* This function returns the final locations and sizes of kernel and palcode */

static DBM_STATUS ld_from_medium( String name, FWID romid,
			char **addr, size_t *size, char choice )
{
    DBM_STATUS sval;
    char *dld_buf;		/* buffer into which image is downloaded */
    char *dld_base;		/* any small adjustment eg for ROM header */
    char *proc_buf;		/* buffer into which image is post-processed */
    int gzsize;
    int elfsize;
    BOOLEAN expect_elf, expect_gzip;
    romheader_t *H;

    /* Here we apply a little sanity from the calling arguments */
    expect_elf = FALSE, expect_gzip = FALSE;

    if ( strcmp( name, KERNELSTR ) == 0 )
	expect_elf = TRUE, expect_gzip = TRUE;

    if ( strcmp( name, INITRDSTR ) == 0 )
	expect_gzip = TRUE;


    /* Allocate buffer for download.  Later we'll try to realloc this to the
     * correct size. */
    dld_buf = malloc( MAX_KERNEL_SIZE );
    if ( dld_buf == NULL )
    {
	printf_dbm( "Insufficient heap space to handle download!\n" );
	return STATUS_FAILURE;
    }
    memset( dld_buf, 0, MAX_KERNEL_SIZE );		/* DEBUG PARANOIA */

    TRACE( "Download buffer at 0x%016lX\n", dld_buf );


    /* 
     * Fetch kernel from selected medium 
     */

    switch ( tolower(choice) )
    {
    case 'x':					/* Xmodem Download */
	TRACE("Loading Xmodem\n" );
	sval = ld_xmodem_file( name, size, dld_buf );
	break;


    case 'f':					/* load from DOS floppy */
	TRACE("Loading Floppy\n" );
	sval = FileDirectory("\\", 0 );
	if ( sval == STATUS_FAILURE )
		break;

	sval = ld_floppy_file( name, size, dld_buf );
	break;


     case 'r':					/* load image from ROM */
	TRACE("Loading ROM\n" );
	sval = ld_rom_image( name, romid, size, dld_buf );
	break;


    default:
	TRACE("Huh?  Don't understand option '%c' (0x%02X)\n", choice, choice );
        sval = STATUS_FAILURE;
	break;
    }
    if ( sval == STATUS_FAILURE )
		return sval;

    TRACE( "Checksum of %s after load: 0x%X (%d bytes)\n",
		    name, checksum( (unsigned *)dld_buf, *size ), *size );

    /* Check we haven't overspilled our buffer */
    if ( *size > MAX_KERNEL_SIZE )
    {
	mobo_logf( LOG_CRIT "%s buffer overflowed (%dKB, max %dKB)!\n",
		name, *size >> 10, MAX_KERNEL_SIZE >> 10 );
	printf_dbm( "This %s is too big (%dKB) to be handled by diags\n",
		name, *size >> 10 );
	return STATUS_FAILURE;
    }


    /* Reduce the heap memory allocation to only what is necessary */
    dld_buf = realloc( dld_buf, *size );
    if ( dld_buf == NULL )
    {
	printf_dbm( "Couldn't process, not enough heap memory!\n" );
	return STATUS_FAILURE;
    }
    TRACE( "Download buffer realloc'ed to 0x%016lX\n", dld_buf );

    TRACE( "Checksum of %s after load: 0x%X (%d bytes)\n",
		    name, checksum( (unsigned *)dld_buf, *size ), *size );


    /*
     * Prepare to post-process image
     */
    H = (romheader_t *)dld_buf;
    if ( rom_hdr_valid( H ) )
    {
	printf_dbm( "Processing ROM header for image of type '%s'\n",
		rom_I2name( H->romh.V1.fw_id ) );

	/* Currently we just skip over the header to get the body */
	dld_base = dld_buf + H->romh.V0.hsize;

    } else dld_base = dld_buf;		/* otherwise carry on as normal */


    proc_buf = malloc( MAX_KERNEL_SIZE );
    if ( proc_buf == NULL )
    {
	printf_dbm( "Insufficient heap space to handle post-process!\n" );
	return STATUS_FAILURE;
    }
    memset( proc_buf, 0, MAX_KERNEL_SIZE );		/* DEBUG PARANOIA */


    /* check for gzip compression - returns new size if successful */
    gzsize = uncompress_kernel( (void *)dld_base, (void *)proc_buf );
    if ( gzsize != -1)
    {
	/* Things are strange when the dld object was unexpectedly gzipped */
	if ( !expect_gzip )
	{
	    printf_dbm( "Image %s was unexpectedly gzipped, "
			"this may not be what you want\n", name );
	}

	/* image was gzipped and after processing is now this size */
	*size = gzsize;	
	TRACE( "Image uncompressed to 0x%X (%d bytes)\n", proc_buf, *size );
    } else {
	/* image wasn't gzipped, copy it over anyway */
	TRACE( "Copying uncompressed image to 0x%X (%d bytes)\n",
		proc_buf, *size );
	memcpy( proc_buf, dld_base, *size );
    }

    /* Check for ELF file format.  Note: kernel must be this, PALcode is not. */
    elfsize = elf_unpack( proc_buf, addr );

    if ( elfsize != -1 )
    {
	/* Things are strange when the dld unexpectedly has ELF structure */
	if ( !expect_elf )
	{
	    printf_dbm( "Image %s was unexpectedly an ELF file, "
			"this may not be what you want\n", name );
	}

	/* File was an ELF image, we get the PHDR size in return */
	*size = elfsize;
	TRACE( "ELF program unpacked to 0x%X (%d bytes)\n", *addr, *size );

    } else {

	/* Things are strange when the dld unexpectedly lacks ELF structure */
	if ( expect_elf )
	{
	    printf_dbm( "Image %s was not an ELF file, "
			"this may not be what you want\n", name );
	}

	/* not an ELF file, copy it to destination ourselves, isize unchanged */
	TRACE( "Copying raw data image to 0x%X (%d bytes)\n", *addr, *size );
	memcpy( *addr, proc_buf, *size );	
    }

    free( dld_buf );
    free( proc_buf );
    return STATUS_SUCCESS;
}


/*----------------------------------------------------------------------*/
/* Memory setup, in preparation for the jump to full virtual addressing */
/* The memory description used by the kernel is built by the HWRPB.
 * Page tables, etc. are built here */


/* PTBR: use the same pfn that Linux will use later (SWAPPER_PGD (0x300000)).
 * This is the first page in the kernel.  It saves us
 * allocating a page and then telling Linux that it isn't available.
 */

static unsigned long ptbr;
static unsigned long vptbr = 0;



/*  Given all of the information, build a valid PTE.  */

static pte_t build_pte(unsigned long pfn, pgprot_t protection)
{
    pte_t pte;

    pte_val(pte) = pte_val( mk_pte( (unsigned long)PFN2ADDR(pfn, LINUX_SUPER ), protection ) );
    pte_val(pte) = pte_val(pte) | _PAGE_VALID;

    return pte;
}


/*
 *  Given a VA, make a set of page table entries for it (if they
 *  don't already exist).
 *
 *  Args:
 *
 *  va            virtual address to add page table entries for.
 *  vpfn          the PFN that the virtual address is to be mapped to.
 *  protection    the protection to set the L3 page table entry to (all L1 and
 *                L2 pages are marked as Kernel read/write).
 */

static unsigned int num_l2_pages = 0;
static unsigned int num_l3_pages = 0;

void add_VA(unsigned long va, unsigned int vpfn, unsigned int protection)
{
    unsigned long l1, l2, l3;
    unsigned long pfn, newpfn;
    unsigned long pa;
    pte_t pte;

/*
 *  Dismantle the VA.
 */

    l1 = 0UL;
    l2 = (va & 0xff800000UL) >> 20;
    l3 = (va & 0x007fe000UL) >> 10;

/*
 *  Now build a set of entries for it (if they do not already
 *  exist).  In the next blocks of code, pfn is always the pfn of
 *  the current entry and newpfn is the pfn that is referenced from
 *  it.
 */

/*
 *  L1.  The L1 PT PFN is held in the global integer ptbr.
 */
    pa = (unsigned long)PFN2ADDR( ptbr, DBM_SUPER ) + l1;
    if (ReadQ(pa) == 0UL) {
	newpfn = page_alloc( 1, ALLOCATED_PAGE );	/* for L2 */
	num_l2_pages++;
	memset( PFN2ADDR( newpfn, DBM_SUPER ), 0, PAGE_SIZE );
	pte = build_pte(newpfn, __pgprot(_PAGE_KWE | _PAGE_KRE));
	WriteQ(pa, pte_val(pte));
	TRACE( "L1 added: paddr 0x%016lx, value 0x%016lx (ptbr pfn %d)\n", 
			pa, pte_val(pte), ptbr );
    } else {
	pte_val(pte) = ReadQ(pa);
    }

    /* pte_page gives krnl kseg */
    pfn = ADDR2PFN( pte_page(pte), LINUX_SUPER );

/*
 *  L2, pfn inherited from L1 entry (which may have been allocated
 *  above).
 */
    pa = (unsigned long)PFN2ADDR( pfn, DBM_SUPER ) + l2;
    if (ReadQ(pa) == 0UL) {
	num_l3_pages++;
	newpfn = page_alloc( 1, ALLOCATED_PAGE );
	memset( PFN2ADDR( newpfn, DBM_SUPER ), 0, PAGE_SIZE );
	pte = build_pte(newpfn, __pgprot(_PAGE_KWE | _PAGE_KRE));
	WriteQ(pa, pte_val(pte));
	TRACE( "L2 added: paddr 0x%016lx, value 0x%016lx\n", 
			pa, pte_val(pte) );
    } else {
	pte_val(pte) = ReadQ(pa);
    }

    /* pte_page gives krnl kseg */
    pfn = ADDR2PFN( pte_page((pte_t) pte), LINUX_SUPER );


/*
 *  L3, pfn inherited from L2 entry above.  vpfn contains the page frame
 *  number of the real thing whose PTEs we have just invented above.
 */
    pa = (unsigned long)PFN2ADDR( pfn, DBM_SUPER ) + l3;
    if (ReadQ(pa) == 0UL) {
	pte = build_pte(vpfn, __pgprot(protection));
	WriteQ(pa, pte_val(pte));
	TRACE( "L3 added: paddr 0x%016lx, value 0x%016lx\n", 
			pa, pte_val(pte) );
    } else
	pte_val(pte) = ReadQ(pa);

}


/*
 * Build the memory management data structures (eg page tables)
 */
static unsigned long raw_pcb_addr;

static void init_virtual_mem()
{
    pte_t pte;
    HWPCB_t *boot_pcb;

    ptbr = ADDR2PFN( LINUX_PGTBL, LINUX_SUPER );
    TRACE( "1) ptbr pfn = %d, krnlbase = 0x%X\n", ptbr, krnlbase );

    /* zap the L1 page table */
    memset( PFN2ADDR( ptbr, DBM_SUPER ), 0, PAGE_SIZE );

    /*
     *  The virtual page table base pointer is used as a shortcut to
     *  get to the L3 entry for any given faulting address.  It is a
     *  special entry in the L1 page table.  This part is important.
     *  Linux 1.2 sets up the vptbr so that the *last* entry in the
     *  L1 PT contains the L1 pfn.  I honour that here.  Change this
     *  at your peril.
     */

    vptbr = 0xfffffffe00000000UL;
    pte = build_pte(ptbr, __pgprot(_PAGE_KWE | _PAGE_KRE));
    WriteQ( (unsigned long)PFN2ADDR( ptbr, DBM_SUPER ) + (sizeof(pte) * 0x3ff),
		 pte_val(pte));

    TRACE( "2) ptbr pfn = %d, krnlbase = 0x%X\n", ptbr, krnlbase );

    /* Add the HWRPB into the virtual address map (make_HWRPB_virtual) */
    add_VA( (unsigned long) INIT_HWRPB,			/* virtual address */
		ADDR2PFN( RAW_HWRPB, 0 ), 		/* phys. page no */
		(_PAGE_KWE | _PAGE_KRE));

    TRACE( "3) ptbr pfn = %d, krnlbase = 0x%X\n", ptbr, krnlbase );

    TRACE( "Added HWRPB address translation 0x%lx->0x%lx\n",
	       RAW_HWRPB, INIT_HWRPB);

    /* PCB (Process Control Block) initialisation.  This sits in the top of the 
     * perCPU structure, so I use the primary's perCPU structure
     * so that it matches well.
     */ 

    boot_pcb = hwrpb_gethwpcb( 0 );

    TRACE( "Boot pcb is at 0x%X (getting ptbr value of %d)\n", boot_pcb, ptbr );

    /* priveliged context block (PCB) is in the arch handbook, section II-B */
    boot_pcb->ksp    = LINUX_STACK + PAGE_SIZE;
    boot_pcb->usp    = 0UL;
    boot_pcb->ptbr   = ptbr;		/* physical page of L1 pg tbl */
    boot_pcb->pcc    = 0UL;
    boot_pcb->asn    = 0UL;
    boot_pcb->unique = 0UL;
    boot_pcb->flags  = 0UL;		/* perf mon, floating pt. disabled */
    boot_pcb->res1   = boot_pcb->res2 = 0UL;	/* reserved */


    /* PALcode likes this and the Linux kernel doesn't object */
    raw_pcb_addr = (unsigned long)boot_pcb & ~DBM_SUPER;

    TRACE( "Boot pcb raw address is 0x%X\n", raw_pcb_addr );
}


/*----------------------------------------------------------------------*/
/* Pass boot arguments to the kernel via the "zero page" */

static void init_zeropage( String krnl_args )
{
    String InitRD;
    char *dbm_zero = (char *)DBM_ZERO;
    uint64 *zp_initrdbase = (uint64 *)(DBM_ZERO + ZPG_INITRD_BASE);
    uint64 *zp_initrdlen  = (uint64 *)(DBM_ZERO + ZPG_INITRD_SIZE);

    memset((void *) dbm_zero, 0, PAGE_SIZE);

    sprintf_dbm( dbm_zero, "bootdevice=%s filename=%s %s",
		boot_device, boot_file, krnl_args );

    /*
     * Add any extra boot params concerning DBLX whistles and bells from NVRAM
     */

    /* If we want InitRD we need to add load_ramdisk=1 */
    InitRD = nvenv_lookup( KEY_DBLX_INITRD );
    if ( InitRD != NULL )
    {
	strcat( dbm_zero, " load_ramdisk=1" );
    }


    /* Add information for the kernel about the InitRD (if loaded) */
    if( initrdbase != 0 )
    {
	*zp_initrdbase = (initrdbase & ~DBM_SUPER) | LINUX_SUPER;
	*zp_initrdlen = initrdlen;
	TRACE( "Added info for an initrd at: \n"
		"    addrs: 0x%016lX, 0x%016lX,\n"
		"    data:  0x%016lX, %d\n",
		zp_initrdbase, zp_initrdlen, *zp_initrdbase, *zp_initrdlen );
    }

    /* should we add a page table entry for it, or mark it as occupied? */
}


/*----------------------------------------------------------------------*/
/* swap_to_palcode - make the jump into the kernel, via PALcode */

void swap_to_palcode(void)
{
#ifdef TRACE_ENABLE
    HWPCB_t *H = (HWPCB_t *)(DBM_SUPER | raw_pcb_addr);
#endif

    int rval;

    printf_dbm( "Swapping to OSF PALcode at 0x%X and kernel at 0x%X...\n\n", 
		RAW_OSFPAL, LINUX_KERNEL );

#ifdef TRACE_ENABLE
    printf_dbm( "DEBUG: instructions are:\r"
		"\t0x%X\r"
		"\t0x%X\r"
		"\t0x%X\r"
		"\t0x%X\r",
		*(unsigned *)DIAGS_OSFPAL,
		*(unsigned *)(DIAGS_OSFPAL + 4),
		*(unsigned *)(DIAGS_OSFPAL + 8),
		*(unsigned *)(DIAGS_OSFPAL + 12) );
	
    TRACE( "SWAP PAL PCB: at address 0x%X\n"
	   "\tKSP:  0x%016lX\n"
	   "\tUSP:  0x%016lX\n"
	   "\tPTBR: 0x%016lX\n"
	   "\tPCC:  0x%016lX\n"
	   "\tASN:  0x%016lX\n"
	   "\tuniq: 0x%016lX\n"
	   "\tFLAGS:0x%016lX\n",
	H, H->ksp, H->usp, H->ptbr, H->pcc, H->asn, H->unique, H->flags );
	


#endif			/* TRACE_ENABLE */



    outled( LED_STARTING_KERNEL );	/* trigger bus analyser */
    *diags_do_decompress = TRUE;	/* set state for re-entry */

#ifdef CONFIG_SHARK
    /* Shark multiplexes serial ports between SROM and COM, we need to make 
     * sure that the switch is in the direction of COM ports before the kernel
     * is entered.
     */

    shark_serial_select( SHARK_COM ); 

    /* Switch off the yellow activity LED as per spec */
    set_yellow_light( FALSE );
#endif

    rval = swppal( RAW_OSFPAL, LINUX_KERNEL, raw_pcb_addr, vptbr);

    /* if we returned, things went badly wrong */
    *diags_do_decompress = FALSE;		/* world was saved after all */
    printf_dbm("returned, with error code %d\n", rval);
}


/*----------------------------------------------------------------------*/
/* SMP Secondary CPUs polling loop code */
/* the secondaries will enter here and wait for the kernel to signal their 
 * start.  The kernel does this like so:
 *
 * 1) Writes an address for secondaries to enter (__smp_callin) into the 
 * HWRPB field CPU_restart.
 *
 * 2) sends a command to the secondary CPUs by writing "START" to the IPC 
 * field of their percpu structures using a message communication protocol.
 * 
 * 3) clears the BIP flag (bit 0) for this percpu struct
 * 
 * 4) Secondary CPU acknowledges the command back to the kernel by re-setting
 * the BIP bit.
 * 
 * In this function we must not use stack or heap memory after we have
 * indicated that we are ready to do so.
 */

extern void secondary_polling_loop( unsigned long, void **,
			unsigned long, unsigned long, unsigned long * );

static DBM_STATUS smp_wait_for_kernel( int a, char **b )
{
    /* First the primary must return and do the proper thing */
    if ( smp_primary() )	return STATUS_SUCCESS;

    /* Comin' up... */
    percpu_setflags( smp_phys_id(), PERCPU_BIP );
    smp_sync();					/* synchronise here. */

    TRACE( "LINUX: CPU %d awaiting start signal from primary\n",
		smp_phys_id() );

    hwrpb_await_signal(	RAW_OSFPAL, vptbr );

    return STATUS_FAILURE;			/* should never reach here */
}


static DBM_STATUS go_kernel( size_t klen, size_t plen, String krnl_args )
{
    size_t roundklen, roundplen;
    String Bobbins;
    uint64 bootmem_base, bootmem_base_pfn, bootmem_size, bootmem_size_pfn;

    /*--------------------------------------------------------------------*/
    /* Do we want to reserve bootmem? */

    Bobbins = nvenv_lookup( KEY_ENABLE_BOOTMEM );
    if ( Bobbins != NULL && strcmp( Bobbins, EB_ON ) == 0 )
    {
	bootmem_size = atoi( nvenv_lookup( KEY_BOOTMEM_ALLOC ) );
	bootmem_size <<= 20;		/* alloc size in bytes */
	bootmem_size_pfn = PFNCOUNT( bootmem_size );

	/* Reserve from the top of memory */
	bootmem_base = DBM_SUPER | (primary_impure->MEM_SIZE - bootmem_size);
	bootmem_base_pfn = ADDR2PFN( bootmem_base, DBM_SUPER );

	if ( bootmem_size_pfn > 0UL )
	{
	    TRACE( "Marking & clearing %d bootmem pages\n", bootmem_size_pfn );
	    page_mark_range( bootmem_base_pfn, bootmem_size_pfn, BOOTMEM_PAGE );
	    memset( (void *)bootmem_base, 0, bootmem_size );
	}
    }

#ifdef TRACE_ENABLE
    hwrpb_dump();
#endif

    /*--------------------------------------------------------------------*/
    /* We now have the right images loaded into the right places */
    /* setup memory for handover to the kernel, ie page tables, allocations */

    /* We're officially booting */
    percpu_setflags( smp_phys_id(), PERCPU_BIP );
    init_virtual_mem();

    /* kernel and PALcode can be marked in as being allocated */
    /* note - round up to the nearest page */

    /* round up if necessary.  We (safely) assume page size is a power of 2 */
    if ( klen & (PAGE_SIZE-1) )	roundklen = (klen | (PAGE_SIZE-1)) + 1;
    else			roundklen = klen;
    if ( plen & (PAGE_SIZE-1) )	roundplen = (plen | (PAGE_SIZE-1)) + 1;
    else			roundplen = plen;


    /* Reserve PALcode from kernel's memory pool */
    page_mark_range( ADDR2PFN( palbase, DBM_SUPER ),
			roundplen / PAGE_SIZE, ALLOCATED_PAGE );


    /*--------------------------------------------------------------------*/
    /* pass arguments on to the kernel via HWRPB and zeropage */

    init_zeropage( krnl_args );

    /*--------------------------------------------------------------------*/
    /* Additional hardware initialisations and SMP */
    /* at this point we may have an uninitialised screen - set up local VGA */

    if ( VgaDeviceDetected )
	if ( io_dev_init( DEV_VGA ) != STATUS_SUCCESS )
	    mobo_logf( LOG_WARN "local VGA not found or not working...\n");

    /* Drain interrupts and aborts */
    mobo_logf( LOG_WARN "draining any aborts caused...\n");
    plat_intsetup( 1 );
    plat_intclr();
    wrmces( MCES_M_SCE | MCES_M_PCE );

    swpipl( 0 );		/* paranoia in case the above fails */
    usleep( 1000 );
    swpipl( 7 );


    TRACE( "Calling in secondary CPUs\n" );
    smp_exec( smp_wait_for_kernel );		/* prepare any secondaries */
    TRACE( "Syncing with secondary CPUs\n" );
    smp_sync();			/* wait until they aren't using stack or heap */


    /*--------------------------------------------------------------------*/
    /* Enter the PALcode with virtual addressing enabled! */

    swap_to_palcode();


    /* Something went badly wrong if we're back here */
    printf_dbm("Well that was a good experiment...\n");
    mobo_key( 0 );
    return STATUS_FAILURE;
}



/*----------------------------------------------------------------------*/
/* Main routine */

/* Procedure at present:
 * 1) download a palcode image and put it at 2MB (DFLT_PALCODE_BASE )
 *    currently, this is possible either by DOS floppy, xmodem or from ROM
 * 2) download a kernel image and put it at 3MB (START_ADDR)
 *    currently, this is possible either by DOS floppy, xmodem or from ROM
 * 3) build the memory structure that is expected by Linux (ie page tables)
 *    and the HWRPB (Hardware Reset Parameter Block)
 * 4) set up all the devices, including PCI devs, vga, dev_config fixups
 * 5) jump to the new palcode, with virtual addressing enabled, and the 
 *    kernel at the other side.
 */

extern volatile char *_end;

#define NOSMPFLAG	"-nosmp"

DBM_STATUS go_linux( int argc, char *argv[] )
{
    size_t klen, plen;
    DBM_STATUS sval;
    String def_action = NULL;
    char krnl_args[64];
    String nvramflags;
    char choice = 0;
    String dblx_initrd;
    BOOLEAN skipsmp = FALSE;
    int i;

    mobo_cls();
    printf_dbm("DBLX Linux Loader\n");


    /*--------------------------------------------------------------------*/
    /* Process command-line args */

    if ( argc > 1 )
    {
	for( i=1; i<argc; i++ )
	{
	    /* Match recognised arguments */
	    if ( strcmp( argv[i], NOSMPFLAG ) == 0 )
	    {
		skipsmp = TRUE;
		continue;
	    }

	    /* If we get here, we got something that didn't match an arg */
	    mobo_alertf( "Usage",
		    "The command usage is:\r"
		    "%s [" NOSMPFLAG "]", argv[0] );
	    return STATUS_FAILURE;
	}
    }

#ifdef CONFIG_SMP
    /* Do we start the secondary CPU's here? */

    if ( (smp_online_count > 1) || skipsmp )		/* already done */
    {
	printf_dbm( "Skipping SMP startup\n" );
    } else {
	plat_smpstart( primary_impure->PAL_BASE );
    }

#endif		/* CONFIG_SMP */

    /*--------------------------------------------------------------------*/
    /* Obtain images of kernel and PALcode */
    /* Check for pre-determined action, otherwise prompt the user */

    sval = nvenv_init( );

    /* Doing this should be OK even without NVRAM ENV */
    def_action = nvenv_lookup( KEY_BOOTDEF_DEV );

    if ( def_action != NULL )
    {
	if( strcmp( def_action, BD_DBLX ) == 0 )
	    choice = 'r';

	if( strcmp( def_action, BD_COM1 ) == 0 )
	    choice = 'x';
    }

    /* Did the caller specify a download mechanism? */
    if ( choice == 0 )
    {
	/* download the Kernel and PALcode images to their starting places */
	printf_dbm( "Select medium for download\n"
		    "\t- press x for Xmodem serial transfer\n"
		    "\t- press f for FAT floppy disk\n"
		    "\t- press r for embedded ROM image\n");
	choice = mobo_key(0);
    }

    sval = ld_from_medium( KERNELSTR, FW_LINUX, (char **)&krnlbase,
			&klen, choice );
    if ( sval != STATUS_SUCCESS )
    {
	mobo_logf( LOG_CRIT "LINUX: " KERNELSTR " load was not successful\n" );
	return sval;
    }

    sval = ld_from_medium( PALCODESTR, FW_OSFPAL, (char **)&palbase, 
			&plen, choice );
    if ( sval != STATUS_SUCCESS )
    {
	mobo_logf( LOG_CRIT "LINUX: " PALCODESTR " load was not successful\n" );
	return sval;
    }

    dblx_initrd = nvenv_lookup( KEY_DBLX_INITRD );
    if ( (dblx_initrd != NULL) && (strcmp( dblx_initrd, DB_ON ) == 0) )
    {
	/* We have been requested to load an initrd */
	/* If the load fails, we can still hopefully press on */
	sval = ld_from_medium( INITRDSTR, FW_INITRD, (char **)&initrdbase,
			&initrdlen, choice );
	if ( sval == STATUS_FAILURE )
	{
	    mobo_logf( LOG_WARN
		"DBLX: load of InitRD was not successful, "
		"attempting to continue...\n" );
	    initrdbase = initrdlen = 0UL;
	}

    } else initrdbase = initrdlen = 0UL;


    nvramflags = nvenv_lookup( KEY_BOOT_OSFLAGS );
    if ( nvramflags != NULL )
    {
	strncpy( krnl_args, nvramflags, sizeof( krnl_args ) );
    } else {
	printf_dbm("Enter kernel arguments: " );
	mobo_input( krnl_args, sizeof( krnl_args ) );
	printf_dbm("\n");
    }


    nvenv_write( KEY_BOOTED_OSFLAGS, krnl_args ); 
    return go_kernel( klen, plen, krnl_args );
}


/* Linux has already booted, lets take the previously used environment and 
 * extract from it the kernel details, then get back onto the chase */

DBM_STATUS reboot_linux( void )
{
    String krnl_args;
    String dblx_initrd;
    size_t klen, plen;
    DBM_STATUS sval;

    TRACE( "Running\n" );

    krnl_args = nvenv_lookup( KEY_BOOTED_OSFLAGS );
    if ( krnl_args == NULL )
    {
	mobo_logf( LOG_CRIT "no previous environment to reboot kernel!\n" );
	return STATUS_FAILURE;
    }

    TRACE( "Retrieved last boot args of %s\n", krnl_args );

    /* Load kernel and palcode from ROM (note, since we're explicitly
     * rebooting, this is a requirement). */

    TRACE( "Loading kernel from ROM\n" );

    /* Here we reload kernel but not PALcode -
     * FIXME: diags still running on OSF PALcode at this point. */

    sval = ld_from_medium( KERNELSTR, FW_LINUX, (char **)&krnlbase, &klen, 'r');
    if ( sval != STATUS_SUCCESS )
    {
	mobo_logf( LOG_CRIT "REBOOT: Kernel reload from ROM failed!\n" );
	return sval;
    }


#if 1
    TRACE( "Skipping load of PALcode from rom\n" );
    plen = 32<<10;
#else
    TRACE( "Loading PALcode from rom\n" );

    sval = ld_rom_image( "PALcode", FW_OSFPAL, &plen, (char *)LINUX_PAL );
    if ( sval != STATUS_SUCCESS )
    {
	mobo_logf( LOG_CRIT "REBOOT: PALcode reload from ROM failed!\n" );
	return sval;
    }
#endif


    dblx_initrd = nvenv_lookup( KEY_DBLX_INITRD );
    if ( (dblx_initrd != NULL) && (strcmp( dblx_initrd, DB_ON ) == 0) )
    {
	/* We have been requested to load an initrd */
	/* If the load fails, we can still hopefully press on */
	TRACE( "Loading InitRD from ROM\n" );
	sval = ld_from_medium( INITRDSTR, FW_INITRD, (char **)&initrdbase,
			&initrdlen, 'r' );
	if ( sval == STATUS_FAILURE )
	{
	    mobo_logf( LOG_WARN
		"DBLX: load of InitRD was not successful, "
		"attempting to continue...\n" );
	    initrdbase = initrdlen = 0UL;
	}

    } else initrdbase = initrdlen = 0UL;


    return go_kernel( klen, plen, krnl_args );
}


